---
icon: material-symbols:bolt-rounded
---

# H3

> H3 class is the core of server.

You can create a new H3 app instance using `new H3()`:

```js
import { H3 } from "h3";

const app = new H3({
  /* optional config */
});
```

## `H3` Methods

### `H3.request`

A [fetch](https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API)-compatible function allowing to fetch app routes.

- Input can be a relative path, [URL](https://developer.mozilla.org/en-US/docs/Web/API/URL), or [Request](https://developer.mozilla.org/en-US/docs/Web/API/Request).
- Returned value is a [Response](https://developer.mozilla.org/en-US/docs/Web/API/Response) promise.

```ts
const response = await app.request("/");
console.log(response, await response.text());
```

### `H3.fetch`

Similar to `H3.request` but only accepts one `(req: Request)` argument for cross runtime compatibility.

### `H3.on`

Register route handler for specific HTTP method.

```js
const app = new H3().on("GET", "/", () => "OK");
```

:read-more{to="/guide/basics/routing" title="Routing"}

### `H3.[method]`

Register route handler for specific HTTP method (shortcut for `app.on(method, ...)`).

```js
const app = new H3().get("/", () => "OK");
```

### `H3.all`

Register route handler for all HTTP methods.

```js
const app = new H3().all("/", () => "OK");
```

### `H3.use`

Register a global [middleware](/guide/basics/middleware).

```js
const app = new H3()
  .use((event) => {
    console.log(`request: ${event.req.url}`);
  })
  .all("/", () => "OK");
```

:read-more{to="/guide/basics/middleware" title="Middleware"}

### `H3.register`

Register a H3 plugin to extend app.

:read-more{to="/guide/advanced/plugins" title="Plugins"}

### `H3.handler`

An H3 [event handler](/guide/basics/handler) useful to compose multiple H3 app instances.

**Example:** Nested apps.

```js
import { H3, serve, redirect, withBase } from "h3";

const nestedApp = new H3().get("/test", () => "/test (sub app)");

const app = new H3()
  .get("/", (event) => redirect(event, "/api/test"))
  .all("/api/**", withBase("/api", nestedApp.handler));

serve(app);
```

### `H3.mount`

Using `.mount` method, you can register a sub-app with prefix.

:read-more{to="/guide/basics/nested-apps" title="Nested Apps"}

## `H3` Options

You can pass global app configuration when initializing an app.

Supported options:

- `debug`: Displays debugging stack traces in HTTP responses (potentially dangerous for production!).
- `silent`: When enabled, console errors for unhandled exceptions will not be displayed.
- `plugins`: (see [plugins](/guide/advanced/plugins) for more information)

> [!IMPORTANT]
> Enabling `debug` option, sends important stuff like stack traces in error responses. Only enable during development.

### Global Hooks

When initializing an H3 app, you can register global hooks:

- `onError`
- `onRequest`
- `onResponse`

These hooks are called for every request and can be used to add global logic to your app such as logging, error handling, etc.

```js
const app = new H3({
  onRequest: (event) => {
    console.log("Request:", event.req.url);
  },
  onResponse: (response, event) => {
    console.log("Response:", event.path, response.status);
  },
  onError: (error, event) => {
    console.error(error);
  },
});
```

> [!IMPORTANT]
> Global hooks only run from main H3 app and **not** sub-apps. Use [middleware](/guide/basics/middleware) for more flexibility.

## `H3` Properties

### `H3.config`

Global H3 instance config.
