---
icon: wpf:key-security
---

# Security

> H3 security utilities.

## Authentication

<!-- automd:jsdocs src="../../src/utils/auth.ts" -->

### `basicAuth(opts)`

Create a basic authentication middleware.

**Example:**

```ts
import { H3, serve, basicAuth } from "h3";
const auth = basicAuth({ password: "test" });
app.get("/", (event) => `Hello ${event.context.basicAuth?.username}!`, [auth]);
serve(app, { port: 3000 });
```

### `requireBasicAuth(event, opts)`

Apply basic authentication for current request.

**Example:**

```ts
import { defineHandler, requireBasicAuth } from "h3";
export default defineHandler(async (event) => {
  await requireBasicAuth(event, { password: "test" });
  return `Hello, ${event.context.basicAuth.username}!`;
});
```

<!-- /automd -->

## Session

<!-- automd:jsdocs src="../../src/utils/session.ts" -->

### `clearSession(event, config)`

Clear the session data for the current request.

### `getSession(event, config)`

Get the session for the current request.

### `sealSession(event, config)`

Encrypt and sign the session data for the current request.

### `unsealSession(_event, config, sealed)`

Decrypt and verify the session data for the current request.

### `updateSession(event, config, update?)`

Update the session data for the current request.

### `useSession(event, config)`

Create a session manager for the current request.

<!-- /automd -->

## Fingerprint

<!-- automd:jsdocs src="../../src/utils/fingerprint.ts" -->

### `getRequestFingerprint(event, opts)`

Get a unique fingerprint for the incoming request.

<!-- /automd -->

## CORS

<!-- automd:jsdocs src="../../src/utils/cors.ts" -->

### `appendCorsHeaders(event, options)`

Append CORS headers to the response.

### `appendCorsPreflightHeaders(event, options)`

Append CORS preflight headers to the response.

### `handleCors(event, options)`

Handle CORS for the incoming request.

If the incoming request is a CORS preflight request, it will append the CORS preflight headers and send a 204 response.

If return value is not `false`, the request is handled and no further action is needed.

**Example:**

```ts
const app = new H3();
const router = createRouter();
router.use("/", async (event) => {
  const corsRes = handleCors(event, {
    origin: "*",
    preflight: {
      statusCode: 204,
    },
    methods: "*",
  });
  if (corsRes !== false) {
    return corsRes;
  }
  // Your code here
});
```

### `isCorsOriginAllowed(origin, options)`

Check if the origin is allowed.

### `isPreflightRequest(event)`

Check if the incoming request is a CORS preflight request.

<!-- /automd -->
