// Copyright 2024
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package controller

import (
	"context"
	"crypto/sha256"
	"encoding/hex"
	"errors"
	"fmt"
	"strings"
	"time"

	helmcontrollerv2 "github.com/fluxcd/helm-controller/api/v2"
	fluxmeta "github.com/fluxcd/pkg/apis/meta"
	fluxconditions "github.com/fluxcd/pkg/runtime/conditions"
	sourcev1 "github.com/fluxcd/source-controller/api/v1"
	"golang.org/x/sync/errgroup"
	"helm.sh/helm/v3/pkg/action"
	"helm.sh/helm/v3/pkg/chart"
	corev1 "k8s.io/api/core/v1"
	apiextv1 "k8s.io/apiextensions-apiserver/pkg/apis/apiextensions/v1"
	apierrors "k8s.io/apimachinery/pkg/api/errors"
	apimeta "k8s.io/apimachinery/pkg/api/meta"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/util/json"
	utilrand "k8s.io/apimachinery/pkg/util/rand"
	clusterapiv1 "sigs.k8s.io/cluster-api/api/core/v1beta2"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	crcache "sigs.k8s.io/controller-runtime/pkg/cache"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/event"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/source"
	"sigs.k8s.io/yaml"

	kcmv1 "github.com/K0rdent/kcm/api/v1beta1"
	"github.com/K0rdent/kcm/internal/helm"
	"github.com/K0rdent/kcm/internal/metrics"
	"github.com/K0rdent/kcm/internal/providerinterface"
	"github.com/K0rdent/kcm/internal/record"
	"github.com/K0rdent/kcm/internal/serviceset"
	authutil "github.com/K0rdent/kcm/internal/util/auth"
	conditionsutil "github.com/K0rdent/kcm/internal/util/conditions"
	kubeutil "github.com/K0rdent/kcm/internal/util/kube"
	labelsutil "github.com/K0rdent/kcm/internal/util/labels"
	ratelimitutil "github.com/K0rdent/kcm/internal/util/ratelimit"
	schemeutil "github.com/K0rdent/kcm/internal/util/scheme"
	validationutil "github.com/K0rdent/kcm/internal/util/validation"
)

var (
	errClusterNotFound         = errors.New("cluster is not found")
	errClusterTemplateNotFound = errors.New("cluster template is not found")

	errClusterDeploymentSpecUpdated = errors.New("cluster deployment spec updated")
	errIPAMNotReady                 = errors.New("IPAM not ready")
	errInvalidIPAMClaimRef          = errors.New("invalid IPAM claim ref")
)

type helmActor interface {
	DownloadChartFromArtifact(ctx context.Context, artifact *fluxmeta.Artifact) (*chart.Chart, error)
	InitializeConfiguration(clusterDeployment *kcmv1.ClusterDeployment, log action.DebugLog) (*action.Configuration, error)
	EnsureReleaseWithValues(ctx context.Context, actionConfig *action.Configuration, hcChart *chart.Chart, clusterDeployment *kcmv1.ClusterDeployment) error
}

// ClusterDeploymentReconciler reconciles a ClusterDeployment object
type ClusterDeploymentReconciler struct {
	MgmtClient client.Client
	helmActor
	SystemNamespace        string
	GlobalRegistry         string
	GlobalK0sURL           string
	K0sURLCertSecretName   string // Name of a Secret with K0s Download URL Root CA with ca.crt key
	RegistryCertSecretName string // Name of a Secret with Registry Root CA with ca.crt key

	DefaultHelmTimeout time.Duration
	defaultRequeueTime time.Duration

	IsDisabledValidationWH bool // is webhook disabled set via the controller flags
}

type (
	clusterScope struct {
		cd        *kcmv1.ClusterDeployment
		cred      *kcmv1.Credential
		region    *kcmv1.Region
		kine      *kineConfig
		auth      *authConfig
		rgnClient client.Client
	}

	authConfig struct {
		clAuth         *kcmv1.ClusterAuthentication
		authConfigHash string
	}

	kineConfig struct {
		dataSource        *kcmv1.DataSource
		clusterDataSource *kcmv1.ClusterDataSource
	}
)

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
func (r *ClusterDeploymentReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	l := ctrl.LoggerFrom(ctx)
	l.Info("Reconciling ClusterDeployment")

	clusterDeployment := &kcmv1.ClusterDeployment{}
	if err := r.MgmtClient.Get(ctx, req.NamespacedName, clusterDeployment); err != nil {
		if apierrors.IsNotFound(err) {
			l.Info("ClusterDeployment not found, ignoring since object must be deleted")
			return ctrl.Result{}, nil
		}

		l.Error(err, "Failed to get ClusterDeployment")
		return ctrl.Result{}, err
	}

	management := &kcmv1.Management{}
	if err := r.MgmtClient.Get(ctx, client.ObjectKey{Name: kcmv1.ManagementName}, management); err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to get Management: %w", err)
	}

	scope, err := r.getClusterScope(ctx, clusterDeployment)
	if err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to get cluster scope: %w", err)
	}

	if !clusterDeployment.DeletionTimestamp.IsZero() {
		l.Info("Deleting ClusterDeployment")
		return r.reconcileDelete(ctx, management, scope)
	}

	if !management.DeletionTimestamp.IsZero() {
		l.Info("Management is being deleted, skipping ClusterDeployment reconciliation")
		return ctrl.Result{}, nil
	}

	return r.reconcileUpdate(ctx, scope)
}

func (r *ClusterDeploymentReconciler) getClusterScope(ctx context.Context, cd *kcmv1.ClusterDeployment) (*clusterScope, error) {
	l := ctrl.LoggerFrom(ctx)

	cred := &kcmv1.Credential{}
	credKey := client.ObjectKey{Namespace: cd.Namespace, Name: cd.Spec.Credential}
	if err := r.MgmtClient.Get(ctx, credKey, cred); err != nil {
		err = fmt.Errorf("failed to get Credential %s: %w", credKey, err)
		if r.setCondition(cd, kcmv1.CredentialReadyCondition, err) {
			r.warnf(cd, "CredentialError", err.Error())
		}
		return nil, fmt.Errorf("failed to get Credential %s: %w", credKey, err)
	}

	scope := &clusterScope{
		cd:        cd,
		cred:      cred,
		rgnClient: r.MgmtClient,
	}

	if cd.Spec.DataSource != "" {
		// NOTE: cld does not watch DataSource resource at this moment due to the current implementation:
		// if one creates a DataSource, triggering a CLD does not make much sense without the reference
		// if the reference is set, it means that a CLD should be triggered
		ds, dsKey := new(kcmv1.DataSource), client.ObjectKey{Namespace: cd.Namespace, Name: cd.Spec.DataSource}
		if err := r.MgmtClient.Get(ctx, dsKey, ds); err != nil {
			err = fmt.Errorf("failed to get DataSource %s: %w", dsKey, err)
			if r.setCondition(cd, kcmv1.DataSourceReadyCondition, err) {
				r.warnf(cd, "DataSourceError", err.Error())
			}

			return nil, err
		}

		_ = r.setCondition(cd, kcmv1.DataSourceReadyCondition, nil)
		scope.kine = &kineConfig{dataSource: ds}
	}

	if cd.Spec.ClusterAuth != "" {
		clAuth := &kcmv1.ClusterAuthentication{}
		clAuthKey := client.ObjectKey{Namespace: cd.Namespace, Name: cd.Spec.ClusterAuth}
		if err := r.MgmtClient.Get(ctx, clAuthKey, clAuth); err != nil {
			err = fmt.Errorf("failed to get ClusterAuthentication %s: %w", clAuthKey, err)
			if r.setCondition(cd, kcmv1.ClusterAuthenticationReadyCondition, err) {
				r.warnf(cd, "ClusterAuthenticationError", err.Error())
			}
			return nil, err
		}

		if r.IsDisabledValidationWH {
			if err := validationutil.ValidateClusterAuthentication(ctx, r.MgmtClient, clAuth); err != nil {
				if r.setCondition(cd, kcmv1.ClusterAuthenticationReadyCondition, err) {
					r.warnf(cd, "ClusterAuthenticationError", err.Error())
				}
				l.Error(err, fmt.Sprintf("ClusterAuthentication %s is invalid: %s, will not retrigger this error", clAuthKey, err))
				return &clusterScope{}, nil
			}
		}

		r.setCondition(cd, kcmv1.ClusterAuthenticationReadyCondition, nil)
		scope.auth = &authConfig{
			clAuth: clAuth,
		}
	}

	if cred.Spec.Region != "" {
		var err error
		rgn := &kcmv1.Region{}
		if err := r.MgmtClient.Get(ctx, client.ObjectKey{Name: cred.Spec.Region}, rgn); err != nil {
			return nil, fmt.Errorf("failed to get %s region: %w", cred.Spec.Region, err)
		}
		if scope.rgnClient, _, err = kubeutil.GetRegionalClient(ctx, r.MgmtClient, r.SystemNamespace, rgn, schemeutil.GetRegionalScheme); err != nil {
			return nil, fmt.Errorf("failed to get client for %s region: %w", cred.Spec.Region, err)
		}
		scope.region = rgn
	}

	return scope, nil
}

func (r *ClusterDeploymentReconciler) reconcileUpdate(ctx context.Context, scope *clusterScope) (_ ctrl.Result, err error) {
	l := ctrl.LoggerFrom(ctx)

	cd := scope.cd

	if controllerutil.AddFinalizer(cd, kcmv1.ClusterDeploymentFinalizer) {
		if err := r.MgmtClient.Update(ctx, cd); err != nil {
			return ctrl.Result{}, fmt.Errorf("failed to update clusterDeployment %s/%s: %w", cd.Namespace, cd.Name, err)
		}
		return ctrl.Result{}, nil
	}

	if updated, err := labelsutil.AddKCMComponentLabel(ctx, r.MgmtClient, cd); updated || err != nil {
		if err != nil {
			l.Error(err, "adding component label")
		}
		return ctrl.Result{}, err
	}

	clusterTpl := &kcmv1.ClusterTemplate{}
	defer func() {
		err = errors.Join(err, r.updateStatus(ctx, cd, clusterTpl))
	}()

	if scope.region != nil {
		if _, ok := scope.region.Annotations[kcmv1.RegionPauseAnnotation]; ok {
			l.Info("Region is paused for this ClusterDeployment, skipping reconciliation")
			if apimeta.SetStatusCondition(&cd.Status.Conditions, metav1.Condition{
				Type:               kcmv1.PausedCondition,
				Status:             metav1.ConditionTrue,
				Reason:             kcmv1.PausedReason,
				Message:            fmt.Sprintf("Related Region %s is paused", scope.region.Name),
				ObservedGeneration: cd.Generation,
			}) {
				r.eventf(cd, "RegionPaused", "Region %s is paused, skipping reconciliation", scope.region.Name)
			}
			return ctrl.Result{}, nil
		}

		if apimeta.SetStatusCondition(&cd.Status.Conditions, metav1.Condition{
			Type:               kcmv1.PausedCondition,
			Status:             metav1.ConditionFalse,
			Reason:             kcmv1.NotPausedReason,
			Message:            "",
			ObservedGeneration: cd.Generation,
		}) {
			r.eventf(cd, "RegionUnpaused", "Region %s has been unpaused", scope.region.Name)
		}
	}

	if err := r.handleCertificateSecrets(ctx, scope.rgnClient, cd); err != nil {
		l.Error(err, "failed to handle certificate secrets")
		return ctrl.Result{}, err
	}

	if err := r.MgmtClient.Get(ctx, client.ObjectKey{Name: cd.Spec.Template, Namespace: cd.Namespace}, clusterTpl); err != nil {
		err = fmt.Errorf("failed to get ClusterTemplate %s/%s: %w", cd.Namespace, cd.Spec.Template, err)
		if r.setCondition(cd, kcmv1.TemplateReadyCondition, err) {
			r.warnf(cd, "ClusterTemplateError", err.Error())
		}

		if r.IsDisabledValidationWH {
			l.Error(err, "failed to get ClusterTemplate, will not retrigger")
			return ctrl.Result{}, nil // no retrigger
		}

		l.Error(err, "failed to get ClusterTemplate")
		return ctrl.Result{}, err
	}

	ipamEnabled := cd.Spec.IPAMClaim.ClusterIPAMClaimRef != "" || cd.Spec.IPAMClaim.ClusterIPAMClaimSpec != nil
	if ipamEnabled {
		// we need to wait until IPAM is bound before processing ClusterDeployment, otherwise we will
		// create a cluster which does not use allocated addresses.
		ipamErr := r.processClusterIPAM(ctx, cd)
		// in case IPAM is not ready yet, need to requeue cluster deployment
		if errors.Is(ipamErr, errIPAMNotReady) {
			return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
		}
		// in case cluster deployment spec was updated, the object will be requeued,
		// hence no need to requeue here
		if errors.Is(ipamErr, errClusterDeploymentSpecUpdated) {
			return ctrl.Result{}, nil
		}
		// in case cluster deployment object refers IPAM claim which refers different cluster deployment,
		// we need to stop reconciliation until cluster deployment's IPAM definition is	fixed: for instance
		// by adding explicit IPAM configuration which will lead to IPAM claim object creation with proper
		// cluster reference.
		if errors.Is(ipamErr, errInvalidIPAMClaimRef) {
			return ctrl.Result{}, nil
		}
		// in case other errors occurred, return an error
		if ipamErr != nil {
			return ctrl.Result{}, fmt.Errorf("failed to process cluster IPAM: %w", ipamErr)
		}
	}

	clusterRes, clusterErr := r.updateCluster(ctx, clusterTpl, scope)
	servicesErr := r.updateServices(ctx, cd)

	if err := errors.Join(clusterErr, servicesErr); err != nil {
		return ctrl.Result{}, err
	}

	if !clusterRes.IsZero() {
		return clusterRes, nil
	}

	return ctrl.Result{}, nil
}

func (r *ClusterDeploymentReconciler) updateCluster(
	ctx context.Context,
	clusterTpl *kcmv1.ClusterTemplate,
	scope *clusterScope,
) (ctrl.Result, error) {
	if clusterTpl == nil {
		return ctrl.Result{}, errors.New("cluster template cannot be nil")
	}

	l := ctrl.LoggerFrom(ctx)

	cd := scope.cd
	r.initClusterConditions(cd)

	if !clusterTpl.Status.Valid {
		return r.setInvalidTemplateCondition(ctx, clusterTpl, cd)
	}
	r.setCondition(cd, kcmv1.TemplateReadyCondition, nil)
	// template is ok, propagate data from it
	cd.Status.KubernetesVersion = clusterTpl.Status.KubernetesVersion

	r.setStatusRegion(scope)

	if r.IsDisabledValidationWH {
		if err := r.validateDeployOnce(ctx, clusterTpl, cd); err != nil {
			l.Error(err, "failed to validate ClusterDeployment, will not retrigger this error")
			return ctrl.Result{}, err
		}
	}

	if err := r.validateConfig(ctx, cd, clusterTpl); err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to validate ClusterDeployment configuration: %w", err)
	}

	if !r.IsDisabledValidationWH {
		if !scope.cred.Status.Ready {
			if r.setCondition(cd, kcmv1.CredentialReadyCondition, fmt.Errorf("the Credential %s is not ready", client.ObjectKeyFromObject(scope.cred))) {
				r.warnf(cd, "CredentialNotReady", "Credential %s/%s is not ready", cd.Namespace, cd.Spec.Credential)
			}
		} else {
			r.setCondition(cd, kcmv1.CredentialReadyCondition, nil)
		}
	}

	if cd.Spec.DryRun {
		r.eventf(cd, "DryRunEnabled", "DryRun mode is enabled. Remove spec.dryRun to proceed with the deployment")
		return ctrl.Result{}, nil
	}

	if err := r.ensureAuthConfigSecret(ctx, scope); err != nil {
		err = fmt.Errorf("failed to create or update AuthenticationConfiguration secret: %w", err)
		r.warnf(cd, "AuthConfigSecretError", err.Error())
		return ctrl.Result{}, err
	}

	requeueDataSource, err := r.ensureDataSourceReferences(ctx, scope)
	if err != nil {
		err = fmt.Errorf("failed to ensure DataSource %s references: %w", cd.Spec.DataSource, err)
		if r.setCondition(cd, kcmv1.ClusterDataSourceReadyCondition, err) {
			r.warnf(cd, "ClusterDataSourceError", err.Error())
		}
		return ctrl.Result{}, err
	}
	if requeueDataSource {
		err = fmt.Errorf("cross-referenced ClusterDataSource %s is not yet ready", client.ObjectKeyFromObject(cd))
		if r.setCondition(cd, kcmv1.ClusterDataSourceReadyCondition, err) {
			r.warnf(cd, "ClusterDataSourceNotReady", err.Error())
		}
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	}
	_ = r.setCondition(cd, kcmv1.ClusterDataSourceReadyCondition, nil)

	if err := r.fillHelmValues(scope); err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to fill helm values: %w", err)
	}

	if err := r.copyRegionalKubeConfigSecret(ctx, scope); err != nil {
		r.warnf(cd, "KubeConfigSecretCopyFailed", "Failed to copy regional KubeConfig secret: %v", err)
		return ctrl.Result{}, fmt.Errorf("failed to copy regional KubeConfig secret: %w", err)
	}

	hrReconcileOpts, err := r.getHelmReleaseReconcileOpts(clusterTpl, scope)
	if err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to get HelmRelease reconcile opts: %w", err)
	}

	// Now create the CAPI cluster by helm releasing the helm chart associated with the cluster template.
	capiClusterKey := client.ObjectKeyFromObject(cd)
	hr, operation, err := helm.ReconcileHelmRelease(ctx, r.MgmtClient, capiClusterKey.Name, capiClusterKey.Namespace, hrReconcileOpts)
	if err != nil {
		err = fmt.Errorf("failed to reconcile HelmRelease: %w", err)
		if r.setCondition(cd, kcmv1.HelmReleaseReadyCondition, err) {
			r.warnf(cd, "HelmReleaseReconcileFailed", err.Error())
		}
		return ctrl.Result{}, err
	}
	if operation == controllerutil.OperationResultCreated {
		r.eventf(cd, "HelmReleaseCreated", "Successfully created HelmRelease %s/%s", cd.Namespace, cd.Name)
	}
	if operation == controllerutil.OperationResultUpdated {
		r.eventf(cd, "HelmReleaseUpdated", "Successfully updated HelmRelease %s/%s", cd.Namespace, cd.Name)
	}

	hrReadyCondition := fluxconditions.Get(hr, fluxmeta.ReadyCondition)
	if hrReadyCondition != nil {
		if apimeta.SetStatusCondition(cd.GetConditions(), metav1.Condition{
			Type:    kcmv1.HelmReleaseReadyCondition,
			Status:  hrReadyCondition.Status,
			Reason:  hrReadyCondition.Reason,
			Message: hrReadyCondition.Message,
		}) {
			r.eventf(cd, "HelmReleaseIsReady", "HelmRelease %s/%s is ready", cd.Namespace, cd.Name)
		}
	}

	requeue, err := r.aggregateCapiConditions(ctx, scope)
	if err != nil {
		if requeue {
			return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, err
		}

		return ctrl.Result{}, err
	}

	if requeue || !fluxconditions.IsReady(hr) {
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	}

	return ctrl.Result{}, nil
}

func (r *ClusterDeploymentReconciler) setInvalidTemplateCondition(ctx context.Context, clusterTpl *kcmv1.ClusterTemplate, cd *kcmv1.ClusterDeployment) (ctrl.Result, error) {
	l := ctrl.LoggerFrom(ctx)

	errMsg := fmt.Sprintf("ClusterTemplate %s is not marked as valid", client.ObjectKeyFromObject(clusterTpl))
	if clusterTpl.Status.ValidationError != "" {
		errMsg += ": " + clusterTpl.Status.ValidationError
	}

	err := errors.New(errMsg)
	if r.setCondition(cd, kcmv1.TemplateReadyCondition, err) {
		r.warnf(cd, "InvalidClusterTemplate", errMsg)
	}

	if r.IsDisabledValidationWH {
		l.Error(err, "template is not valid, will not retrigger this error")
		return ctrl.Result{}, nil
	}

	return ctrl.Result{}, err
}

func (r *ClusterDeploymentReconciler) validateDeployOnce(ctx context.Context, clusterTpl *kcmv1.ClusterTemplate, cd *kcmv1.ClusterDeployment) error {
	l := ctrl.LoggerFrom(ctx).WithName("validator")

	l.Info("Validating ClusterTemplate required providers")
	ctErr := validationutil.ClusterTemplateProviders(ctx, r.MgmtClient, clusterTpl, cd)
	if ctErr != nil {
		ctErr = fmt.Errorf("failed to validate ClusterTemplate required providers: %w", ctErr)
	}

	l.Info("Validating ClusterTemplate K8s compatibility")
	compErr := validationutil.ClusterTemplateK8sCompatibility(ctx, r.MgmtClient, clusterTpl, cd)
	if compErr != nil {
		ctErr = errors.Join(ctErr, fmt.Errorf("failed to validate ClusterTemplate K8s compatibility: %w", compErr))
	}
	r.setCondition(cd, kcmv1.TemplateReadyCondition, ctErr)

	l.Info("Validating Credential")
	var credErr error
	if credErr = validationutil.ClusterDeployCredential(ctx, r.MgmtClient, r.SystemNamespace, cd, clusterTpl); credErr != nil {
		credErr = fmt.Errorf("failed to validate Credential: %w", credErr)
	}
	r.setCondition(cd, kcmv1.CredentialReadyCondition, credErr)

	return errors.Join(ctErr, credErr)
}

func (*ClusterDeploymentReconciler) setStatusRegion(scope *clusterScope) {
	if scope.region != nil {
		scope.cd.Status.Region = scope.region.Name
	}
	if scope.region == nil && scope.cd.Status.Region != "" {
		scope.cd.Status.Region = ""
	}
}

// copyRegionalKubeConfigSecret copies the Regional cluster kubeconfig to the namespace of the ClusterDeployment
// when the kubeconfig secret reference is defined in the Region.
// To deploy the cluster template into the Regional cluster, its kubeconfig must be present
// in the same namespace as the ClusterDeployment.
func (r *ClusterDeploymentReconciler) copyRegionalKubeConfigSecret(ctx context.Context, scope *clusterScope) error {
	cd := scope.cd

	// 1. if the cld is in the system ns, then nothing to copy.
	// 2. if kubeconfig is nil, then we still have we copy the secret, and
	// that means that the clusterdeployment reference is given,
	// and the secret <regional-cld-namespace>.<regional-cld-name>-kubeconfig is to be copied
	// and renamed to <regional-cld-name>-kubeconfig.
	// Basically, this is the reverse logic of the Region reconciler,
	// that copies the regional source cld kubeconfig to the system namespace, renaming it along the copying.
	// 3. if clusterdeployment is nil, then kubeconfig reference should be copied as is
	if scope.cred.Spec.Region == "" || scope.region == nil || cd.Namespace == r.SystemNamespace {
		return nil
	}

	kubeconfigRef, err := kubeutil.GetRegionalKubeconfigSecretRef(scope.region)
	if err != nil {
		return fmt.Errorf("failed to get kubeconfig secret reference: %w", err)
	}

	nameOverride := ""
	if cldRef := scope.region.Spec.ClusterDeployment; cldRef != nil {
		nameOverride = kubeconfigRef.Name
		// mutate the original ref to <regional-cld-namespace>.<regional-cld-name>-kubeconfig
		kubeconfigRef.Name = cldRef.Namespace + "." + kubeconfigRef.Name
	}

	return kubeutil.CopySecret(
		ctx,
		r.MgmtClient,
		r.MgmtClient,
		client.ObjectKey{Namespace: r.SystemNamespace, Name: kubeconfigRef.Name},
		cd.Namespace,
		nameOverride,
		cd,
		nil,
	)
}

func (r *ClusterDeploymentReconciler) getHelmReleaseReconcileOpts(
	clusterTpl *kcmv1.ClusterTemplate,
	scope *clusterScope,
) (helm.ReconcileHelmReleaseOpts, error) {
	cd := scope.cd
	hrReconcileOpts := helm.ReconcileHelmReleaseOpts{
		Values: cd.Spec.Config,
		OwnerReference: &metav1.OwnerReference{
			APIVersion: kcmv1.GroupVersion.String(),
			Kind:       kcmv1.ClusterDeploymentKind,
			Name:       cd.Name,
			UID:        cd.UID,
		},
		ChartRef: clusterTpl.Status.ChartRef,
		Timeout:  r.DefaultHelmTimeout,
	}

	if clusterTpl.Spec.Helm.ChartSpec != nil {
		hrReconcileOpts.ReconcileInterval = &clusterTpl.Spec.Helm.ChartSpec.Interval.Duration
	}

	if scope.region != nil {
		kubeconfigRef, err := kubeutil.GetRegionalKubeconfigSecretRef(scope.region)
		if err != nil {
			return helm.ReconcileHelmReleaseOpts{}, fmt.Errorf("failed to get kubeconfig secret reference for %s region: %w", scope.region.Name, err)
		}

		hrReconcileOpts.KubeConfigRef = kubeconfigRef
	}

	return hrReconcileOpts, nil
}

func (r *ClusterDeploymentReconciler) ensureDataSourceReferences(ctx context.Context, scope *clusterScope) (requeue bool, err error) {
	if scope.kine == nil || scope.kine.dataSource == nil || scope.cd.Spec.DataSource == "" {
		return false, nil
	}

	cd := scope.cd

	clusterdatasource, cdsKey := new(kcmv1.ClusterDataSource), client.ObjectKey{Name: cd.Name, Namespace: cd.Namespace}
	err = r.MgmtClient.Get(ctx, cdsKey, clusterdatasource)
	if err == nil { // if NO error
		if clusterdatasource.Spec.DataSource != cd.Spec.DataSource {
			clusterdatasource.Spec.DataSource = cd.Spec.DataSource
			if err := r.MgmtClient.Update(ctx, clusterdatasource); err != nil {
				return false, fmt.Errorf("failed to update ClusterDataSource %s spec: %w", cdsKey, err)
			}

			return true, nil // updated the CDS, wait for the object to be reconciled
		}
		scope.kine.clusterDataSource = clusterdatasource

		return r.ensureClusterDataSourceRegionalSecrets(ctx, clusterdatasource, scope)
	}

	if !apierrors.IsNotFound(err) {
		return false, fmt.Errorf("failed to get ClusterDataSource %s: %w", cdsKey, err)
	}

	const randomSuffixLen = 5
	randStr := utilrand.String(randomSuffixLen)

	clusterdatasource = &kcmv1.ClusterDataSource{
		ObjectMeta: metav1.ObjectMeta{
			Name:      cd.Name,
			Namespace: cd.Namespace,
			Labels: map[string]string{
				kcmv1.KCMManagedLabelKey:        kcmv1.KCMManagedLabelValue,          // managed by us
				kcmv1.GenericComponentNameLabel: kcmv1.GenericComponentLabelValueKCM, // to be included in a backup
			},
			Finalizers: []string{kcmv1.ClusterDataSourceFinalizer},
		},
		Spec: kcmv1.ClusterDataSourceSpec{
			Schema:     strings.ReplaceAll(fmt.Sprintf("%s_%s_%s", cd.Namespace, cd.Name, randStr), "-", "_"),
			DataSource: cd.Spec.DataSource,
		},
	}

	if err := controllerutil.SetControllerReference(cd, clusterdatasource, r.MgmtClient.Scheme()); err != nil {
		return false, fmt.Errorf("failed to set owner references onto the ClusterDataSource %s: %w", cdsKey, err)
	}

	if err := r.MgmtClient.Create(ctx, clusterdatasource); client.IgnoreAlreadyExists(err) != nil {
		return false, fmt.Errorf("failed to create ClusterDataSource %s: %w", cdsKey, err)
	}

	return true, nil // created the CDS, wait for the object to be reconciled
}

func (r *ClusterDeploymentReconciler) ensureClusterDataSourceRegionalSecrets(ctx context.Context, cds *kcmv1.ClusterDataSource, scope *clusterScope) (requeue bool, err error) {
	if !cds.Status.Ready || cds.Status.Error != "" || cds.Status.ObservedGeneration != cds.Generation { // sanity check in case it is ready but has an error or not yet synced
		ctrl.LoggerFrom(ctx).WithValues(
			"ClusterDataSource", client.ObjectKeyFromObject(cds).String(),
			"observed generation", cds.Status.ObservedGeneration,
			"generation", cds.Generation,
			"ready", cds.Status.Ready,
			"error", cds.Status.Error,
		).Info("ClusterDataSource is not yet ready, or has not consumed the latest generation, or has an error, and cannot be proceeded")

		return true, nil // wait until the object is actually ready
	}

	// check whether we have to copy secrets to the region set
	if scope.cred.Spec.Region == "" || scope.region == nil {
		return false, nil
	}

	cd := scope.cd

	if err := kubeutil.EnsureNamespace(ctx, scope.rgnClient, cd.Namespace); err != nil {
		return false, fmt.Errorf("failed to ensure namespace %s: %w", cd.Namespace, err)
	}

	owner, err := r.getPartialCapiCluster(ctx, scope.rgnClient, cd)
	if err != nil {
		return false, fmt.Errorf("failed to retrieve the owner: %w", err)
	}

	for _, secretName := range []string{cds.Status.CASecret, cds.Status.KineDataSourceSecret} {
		if secretName == "" {
			continue
		}

		secretKey := client.ObjectKey{Name: secretName, Namespace: cd.Namespace}
		regionalSecret := &corev1.Secret{
			ObjectMeta: metav1.ObjectMeta{
				Name:      secretKey.Name,
				Namespace: secretKey.Namespace,
			},
		}

		op, err := controllerutil.CreateOrUpdate(ctx, scope.rgnClient, regionalSecret, func() error {
			mgmtSecret := new(corev1.Secret)
			if err := r.MgmtClient.Get(ctx, secretKey, mgmtSecret); err != nil {
				return fmt.Errorf("failed to get Secret %s on the mgmt cluster: %w", secretKey, err)
			}

			if owner != nil {
				if err := controllerutil.SetOwnerReference(owner, regionalSecret, scope.rgnClient.Scheme()); err != nil {
					return fmt.Errorf("failed to add OwnerReference on Secret %s: %w", secretKey, err)
				}
			}

			regionalSecret.Data = mgmtSecret.Data

			return nil
		})
		if err != nil {
			return false, fmt.Errorf("failed to create or update Secret %s in the region %s: %w", secretKey, scope.cred.Spec.Region, err)
		}

		if op == controllerutil.OperationResultCreated {
			r.eventf(cd, "KineSecretCreated", "Successfully created Secret %s/%s for kine data source", cd.Namespace, secretName)
		}
		if op == controllerutil.OperationResultUpdated {
			r.eventf(cd, "KineSecretUpdated", "Successfully updated Secret %s/%s for kine data source", cd.Namespace, secretName)
		}
	}
	return false, nil // we had both CDS and its secret data
}

// getPartialCapiCluster retrieves a CAPI Cluster as the partial metadata object.
// The method returns nil if the Cluster object is not found (nil, nil),
// or an error if the retrieval operation fails for any other reason.
func (*ClusterDeploymentReconciler) getPartialCapiCluster(ctx context.Context, cl client.Client, cd *kcmv1.ClusterDeployment) (client.Object, error) {
	cluster := new(metav1.PartialObjectMetadata)
	cluster.SetGroupVersionKind(clusterapiv1.GroupVersion.WithKind(clusterapiv1.ClusterKind))

	err := cl.Get(ctx, client.ObjectKeyFromObject(cd), cluster)
	if client.IgnoreNotFound(err) != nil {
		return nil, fmt.Errorf("failed to get Cluster object %s: %w", client.ObjectKeyFromObject(cd), err)
	}

	if err != nil { // NotFound error
		return nil, nil //nolint:nilerr,nilnil // to avoid 3-return-param signature
	}

	return cluster, nil
}

const authConfigSecretKey = "config" // fixed name of the auth Secret key, used in the Secret and helm values
func (r *ClusterDeploymentReconciler) ensureAuthConfigSecret(ctx context.Context, scope *clusterScope) error {
	if scope.auth == nil || scope.auth.clAuth == nil || scope.auth.clAuth.Spec.AuthenticationConfiguration == nil {
		return nil
	}
	cd := scope.cd
	clAuth := scope.auth.clAuth

	authConf, err := authutil.GetAuthenticationConfiguration(ctx, r.MgmtClient, clAuth)
	if err != nil {
		return fmt.Errorf("failed to get AuthenticationConfiguration: %w", err)
	}

	// TODO (Kshatrix,eromanova): if the referenced CA Secret object has been updated,
	// an accidental event on CLD could update mounts in the k0smotroncontrollerplane triggering upgrade of machines
	data, err := yaml.Marshal(authConf)
	if err != nil {
		return fmt.Errorf("failed to marshal AuthenticationConfiguration from ClusterAuthentication: %w", err)
	}

	hash := sha256.Sum256(data)
	scope.auth.authConfigHash = hex.EncodeToString(hash[:4])

	owner, err := r.getPartialCapiCluster(ctx, scope.rgnClient, cd)
	if err != nil {
		return fmt.Errorf("failed to retrieve the owner: %w", err)
	}
	ownerObjectExists := owner != nil

	secretName := r.getAuthConfigSecretName(cd.Name)
	authConfigSecret := &corev1.Secret{
		ObjectMeta: metav1.ObjectMeta{
			Name:      secretName,
			Namespace: cd.Namespace,
		},
	}

	operation, err := controllerutil.CreateOrUpdate(ctx, scope.rgnClient, authConfigSecret, func() error {
		if authConfigSecret.Data == nil {
			authConfigSecret.Data = make(map[string][]byte)
		}

		authConfigSecret.Data = map[string][]byte{
			authConfigSecretKey: data,
		}

		if ownerObjectExists {
			if err := controllerutil.SetOwnerReference(owner, authConfigSecret, scope.rgnClient.Scheme()); err != nil {
				return fmt.Errorf("failed to add OwnerReference on Secret %s: %w", client.ObjectKeyFromObject(authConfigSecret), err)
			}
		}

		return nil
	})
	if err != nil {
		return fmt.Errorf("failed to create or update Secret with the AuthenticationConfiguration: %w", err)
	}

	if operation == controllerutil.OperationResultCreated {
		r.eventf(cd, "AuthConfigSecretCreated", "Successfully created Secret with the AuthenticationConfiguration %s/%s", cd.Namespace, secretName)
	}
	if operation == controllerutil.OperationResultUpdated {
		r.eventf(cd, "AuthConfigSecretUpdated", "Successfully updated Secret with the AuthenticationConfiguration %s/%s", cd.Namespace, secretName)
	}

	return nil
}

func (r *ClusterDeploymentReconciler) fillHelmValues(scope *clusterScope) error {
	cd := scope.cd
	cred := scope.cred
	if err := cd.AddHelmValues(func(values map[string]any) error {
		r.fillClusterAuthenticationValues(scope, values)
		r.fillDataSourceValues(scope, values)

		values["clusterIdentity"] = map[string]any{
			"apiVersion": cred.Spec.IdentityRef.APIVersion,
			"kind":       cred.Spec.IdentityRef.Kind,
			"name":       cred.Spec.IdentityRef.Name,
			"namespace":  cred.Spec.IdentityRef.Namespace,
		}

		global := map[string]any{
			"registry":           r.GlobalRegistry,
			"k0sURL":             r.GlobalK0sURL,
			"registryCertSecret": r.RegistryCertSecretName,
			"k0sURLCertSecret":   r.K0sURLCertSecretName,
		}
		for _, v := range global {
			if v != "" {
				values["global"] = global
				break
			}
		}

		if _, ok := values["clusterLabels"]; !ok {
			// Use the ManagedCluster's own labels if not defined.
			values["clusterLabels"] = cd.GetObjectMeta().GetLabels()
		}

		return nil
	}); err != nil {
		return fmt.Errorf("failed to add helm values for the ClusterDeployment %s/%s: %w", cd.Namespace, cd.Name, err)
	}

	return nil
}

// fillClusterAuthenticationValues passes the Authentication configuration values to all the ClusterDeployments if
// the [github.com/K0rdent/kcm/api/v1beta1.ClusterAuthentication] was provided in the
// [github.com/K0rdent/kcm/api/v1beta1.ClusterDeployment] spec.
//
// Example:
//
//	auth:
//	  configWithAnon: true
//	  configSecret:
//	    name: auth-config-secret
//	    key: config
//	    hash: 7ed534
func (r *ClusterDeploymentReconciler) fillClusterAuthenticationValues(scope *clusterScope, values map[string]any) {
	if scope.auth == nil || scope.auth.clAuth == nil || scope.auth.clAuth.Spec.AuthenticationConfiguration == nil {
		return
	}

	val := map[string]any{
		"configSecret": map[string]any{
			"name": r.getAuthConfigSecretName(scope.cd.Name),
			"key":  authConfigSecretKey,
			"hash": scope.auth.authConfigHash,
		},
	}

	if scope.auth.clAuth.Spec.AuthenticationConfiguration.Anonymous != nil {
		val["configWithAnon"] = true // this will remove --anonymous-auth flag from the kube-apiserver, if applicable for a template
	}

	values["auth"] = val
}

func (*ClusterDeploymentReconciler) getAuthConfigSecretName(cdName string) string {
	return cdName + "-auth-config"
}

// fillDataSourceValues passes the ClusterDataSource secrets to all the ClusterDeployments if
// the [github.com/K0rdent/kcm/api/v1beta1.DataSource] was provided in the
// [github.com/K0rdent/kcm/api/v1beta1.ClusterDeployment] spec and
// [github.com/K0rdent/kcm/api/v1beta1.ClusterDataSource] has been reconciled.
//
// Example:
//
//	dataSource:
//	  kineDataSourceSecretName: "<cld-based-name-generated-by-cds-controller>"
//	  caSecret:
//	    name: "<cds-controller-generated-secret-name>"
//	    key: "<user-provided-secret-key>"
func (*ClusterDeploymentReconciler) fillDataSourceValues(scope *clusterScope, values map[string]any) {
	if scope.kine == nil || scope.kine.dataSource == nil || scope.kine.clusterDataSource == nil || scope.cd.Spec.DataSource == "" {
		return
	}

	val := map[string]any{
		"kineDataSourceSecretName": scope.kine.clusterDataSource.Status.KineDataSourceSecret,
	}

	if scope.kine.dataSource.Spec.CertificateAuthority != nil {
		val["caSecret"] = map[string]any{
			"name": scope.kine.clusterDataSource.Status.CASecret,
			"key":  scope.kine.dataSource.Spec.CertificateAuthority.Key,
		}
	}

	values["dataSource"] = val
}

func (r *ClusterDeploymentReconciler) validateConfig(ctx context.Context, cd *kcmv1.ClusterDeployment, clusterTpl *kcmv1.ClusterTemplate) error {
	helmChartArtifact, err := r.getSourceArtifact(ctx, clusterTpl.Status.ChartRef)
	if err != nil {
		err = fmt.Errorf("failed to get HelmChart Artifact: %w", err)
		if r.setCondition(cd, kcmv1.HelmChartReadyCondition, err) {
			r.warnf(cd, "InvalidSource", err.Error())
		}
		return err
	}

	l := ctrl.LoggerFrom(ctx)
	l.Info("Downloading Helm chart")
	hcChart, err := r.DownloadChartFromArtifact(ctx, helmChartArtifact)
	if err != nil {
		err = fmt.Errorf("failed to download HelmChart from Artifact %s: %w", helmChartArtifact.URL, err)
		if r.setCondition(cd, kcmv1.HelmChartReadyCondition, err) {
			r.warnf(cd, "HelmChartDownloadFailed", err.Error())
		}
		return err
	}

	l.Info("Initializing Helm client")
	actionConfig, err := r.InitializeConfiguration(cd, l.WithName("helm-actor").V(1).Info)
	if err != nil {
		return err
	}

	l.Info("Validating Helm chart with provided values")
	if err := r.EnsureReleaseWithValues(ctx, actionConfig, hcChart, cd); err != nil {
		err = fmt.Errorf("failed to validate template with provided configuration: %w", err)
		if r.setCondition(cd, kcmv1.HelmChartReadyCondition, err) {
			r.warnf(cd, "ValidationError", "Invalid configuration provided: %s", err)
		}
		return err
	}

	r.setCondition(cd, kcmv1.HelmChartReadyCondition, nil)
	return nil
}

func (*ClusterDeploymentReconciler) initClusterConditions(cd *kcmv1.ClusterDeployment) (changed bool) {
	// NOTE: do not put here the PredeclaredSecretsExistCondition since it won't be set if no secrets have been set
	for _, typ := range [5]string{
		kcmv1.CredentialReadyCondition,
		kcmv1.HelmReleaseReadyCondition,
		kcmv1.HelmChartReadyCondition,
		kcmv1.TemplateReadyCondition,
		kcmv1.ReadyCondition,
	} {
		// Skip initialization if the condition already exists.
		// This ensures we don't overwrite an existing condition and can accurately detect actual
		// conditions changes later.
		if apimeta.FindStatusCondition(cd.Status.Conditions, typ) != nil {
			continue
		}
		// Skip setting HelmReleaseReady if in DryRun mode
		if typ == kcmv1.HelmReleaseReadyCondition && cd.Spec.DryRun {
			continue
		}
		if apimeta.SetStatusCondition(&cd.Status.Conditions, metav1.Condition{
			Type:               typ,
			Status:             metav1.ConditionUnknown,
			Reason:             kcmv1.ProgressingReason,
			ObservedGeneration: cd.Generation,
		}) {
			changed = true
		}
	}
	return changed
}

func (r *ClusterDeploymentReconciler) aggregateCapiConditions(ctx context.Context, scope *clusterScope) (requeue bool, _ error) {
	cd := scope.cd
	clusters := &clusterapiv1.ClusterList{}
	if err := scope.rgnClient.List(ctx, clusters, client.MatchingLabels{kcmv1.FluxHelmChartNameKey: cd.Name}, client.Limit(1)); err != nil {
		return false, fmt.Errorf("failed to list clusters for ClusterDeployment %s: %w", client.ObjectKeyFromObject(cd), err)
	}
	if len(clusters.Items) == 0 {
		return false, nil
	}
	cluster := &clusters.Items[0]

	capiCondition, err := conditionsutil.GetCAPIClusterSummaryCondition(cd, cluster)
	if err != nil {
		return true, fmt.Errorf("failed to get condition summary from Cluster %s: %w", client.ObjectKeyFromObject(cluster), err)
	}

	if apimeta.SetStatusCondition(cd.GetConditions(), *capiCondition) {
		if capiCondition.Status == metav1.ConditionTrue {
			r.eventf(cd, "CAPIClusterIsReady", "Cluster has been provisioned")
			return false, nil
		}
		if cd.DeletionTimestamp.IsZero() {
			r.eventf(cd, "CAPIClusterIsProvisioning", "Cluster is provisioning")
		} else {
			r.eventf(cd, "CAPIClusterIsDeleting", "Cluster is deleting")
		}
	}
	return capiCondition.Status != metav1.ConditionTrue, nil
}

func (*ClusterDeploymentReconciler) setCondition(cd *kcmv1.ClusterDeployment, typ string, err error) (changed bool) {
	reason, cstatus, msg := kcmv1.SucceededReason, metav1.ConditionTrue, ""
	if err != nil {
		reason, cstatus, msg = kcmv1.FailedReason, metav1.ConditionFalse, err.Error()
	}

	return apimeta.SetStatusCondition(&cd.Status.Conditions, metav1.Condition{
		Type:               typ,
		Status:             cstatus,
		Reason:             reason,
		Message:            msg,
		ObservedGeneration: cd.Generation,
	})
}

// updateServices reconciles services provided in ClusterDeployment.Spec.ServiceSpec.
func (r *ClusterDeploymentReconciler) updateServices(ctx context.Context, cd *kcmv1.ClusterDeployment) error {
	l := ctrl.LoggerFrom(ctx)
	l.Info("Reconciling Services")

	if r.IsDisabledValidationWH {
		l.Info("Validating service dependencies")
		err := validationutil.ValidateServiceDependencyOverall(cd.Spec.ServiceSpec.Services)
		r.setCondition(cd, kcmv1.ServicesDependencyValidationCondition, err)
		if err != nil {
			l.Error(err, "failed to validate service dependencies, will not retrigger this error")
			return nil
		}
	}

	err := r.createOrUpdateServiceSet(ctx, cd)
	if err != nil {
		return fmt.Errorf("failed to create or update ServiceSet for ClusterDeployment %s: %w", client.ObjectKeyFromObject(cd), err)
	}

	var (
		serviceStatuses []kcmv1.ServiceState
		upgradePaths    []kcmv1.ServiceUpgradePaths
		errs            error
	)

	// we'll update services' statuses and join errors
	serviceStatuses, err = r.collectServicesStatuses(ctx, cd)
	cd.Status.Services = serviceStatuses
	errs = errors.Join(errs, err)

	// we'll update services' upgrade paths and join errors
	upgradePaths, err = serviceset.ServicesUpgradePaths(ctx, r.MgmtClient, cd.Spec.ServiceSpec.Services, cd.Namespace)
	cd.Status.ServicesUpgradePaths = upgradePaths
	errs = errors.Join(errs, err)
	return errs
}

// setServicesCondition updates ClusterDeployment's condition which shows number of successfully
// deployed services out of total number of desired services.
func (r *ClusterDeploymentReconciler) setServicesCondition(ctx context.Context, cd *kcmv1.ClusterDeployment) error {
	serviceSetList := new(kcmv1.ServiceSetList)
	if err := r.MgmtClient.List(ctx, serviceSetList, client.MatchingFields{kcmv1.ServiceSetClusterIndexKey: cd.Name}); err != nil {
		return fmt.Errorf("failed to list ServiceSets for ClusterDeployment %s: %w", client.ObjectKeyFromObject(cd), err)
	}

	var totalServices, readyServices int

	c := metav1.Condition{
		Type:   kcmv1.ServicesInReadyStateCondition,
		Status: metav1.ConditionTrue,
		Reason: kcmv1.SucceededReason,
	}

	serviceStateMap := make(
		map[client.ObjectKey][]*kcmv1.ServiceState,
		len(cd.Spec.ServiceSpec.Services),
	)

	for _, serviceSet := range serviceSetList.Items {
		// we'll skip serviceSets being deleted
		if !serviceSet.DeletionTimestamp.IsZero() {
			continue
		}
		for _, svc := range serviceSet.Status.Services {
			// We won't count services being deleted neither in total services count
			// nor in ready services count, because semantically such services obviously
			// can't be counted as deployed - deletion process is already started -, and
			// in the same time we can't count such services in total service count since
			// these services are not in the list of desired services.
			// Thus if service in "Deleting" state it will be skipped.
			// We might consider changing condition message from "X/Y" to "X/Y/Z" where
			// X - ready services, Y - services being deleted and Z - total number of
			// services being processed: desired services and services being deleted.
			if svc.State == kcmv1.ServiceStateDeleting {
				continue
			}

			key := serviceset.ServiceKey(svc.Namespace, svc.Name)
			serviceStateMap[key] = append(serviceStateMap[key], &svc)
		}
	}

	totalServices = len(serviceStateMap)
	for _, svcStates := range serviceStateMap {
		// For each of the services identified by ServiceKey(namespace, name), we loop
		// through its ServiceStates collected from all the ServiceSets that deploy that
		// service. If any one of these ServiceStates have deployed state then we
		// consider that service to have been successfully deployed.
		//
		// Consider if we have a ClusterDeployment and MultiClusterService both deploying
		// the same service on a cluster with the MultiClusterService having higher priority.
		// In such a case, it is possible to have 2 ServiceState entries for the service where
		// one reports success and the other reports failure. Therefore, we take into account
		// both the ServiceStates so we don't erroneously report failure when the service has
		// successfully been deployed on the cluster by the MultiClusterService.
		for _, svcState := range svcStates {
			if svcState.State == kcmv1.ServiceStateDeployed {
				readyServices++
			}
		}
	}

	if readyServices < totalServices {
		c.Status = metav1.ConditionFalse
		c.Reason = kcmv1.FailedReason
	}

	c.Message = fmt.Sprintf("%d/%d", readyServices, totalServices)
	apimeta.SetStatusCondition(&cd.Status.Conditions, c)
	return nil
}

// updateStatus updates the status for the ClusterDeployment object.
func (r *ClusterDeploymentReconciler) updateStatus(ctx context.Context, cd *kcmv1.ClusterDeployment, template *kcmv1.ClusterTemplate) error {
	if err := r.setServicesCondition(ctx, cd); err != nil {
		return fmt.Errorf("failed to set services condition: %w", err)
	}

	cd.Status.ObservedGeneration = cd.Generation
	cd.Status.Conditions = updateStatusConditions(cd.Status.Conditions)

	if err := r.setAvailableUpgrades(ctx, cd, template); err != nil {
		return errors.New("failed to set available upgrades")
	}

	if err := r.MgmtClient.Status().Update(ctx, cd); err != nil {
		return fmt.Errorf("failed to update status for clusterDeployment %s/%s: %w", cd.Namespace, cd.Name, err)
	}

	return nil
}

func (r *ClusterDeploymentReconciler) getSourceArtifact(ctx context.Context, ref *helmcontrollerv2.CrossNamespaceSourceReference) (*fluxmeta.Artifact, error) {
	if ref == nil {
		return nil, errors.New("helm chart source is not provided")
	}

	key := client.ObjectKey{Namespace: ref.Namespace, Name: ref.Name}
	hc := new(sourcev1.HelmChart)
	if err := r.MgmtClient.Get(ctx, key, hc); err != nil {
		return nil, fmt.Errorf("failed to get HelmChart %s: %w", key, err)
	}

	return hc.GetArtifact(), nil
}

func (r *ClusterDeploymentReconciler) reconcileDelete(ctx context.Context, mgmt *kcmv1.Management, scope *clusterScope) (result ctrl.Result, err error) {
	l := ctrl.LoggerFrom(ctx)

	cd := scope.cd

	defer func() {
		if err == nil {
			metrics.TrackMetricTemplateUsage(ctx, kcmv1.ClusterTemplateKind, cd.Spec.Template, kcmv1.ClusterDeploymentKind, cd.ObjectMeta, false)

			for _, svc := range cd.Spec.ServiceSpec.Services {
				metrics.TrackMetricTemplateUsage(ctx, kcmv1.ServiceTemplateKind, svc.Template, kcmv1.ClusterDeploymentKind, cd.ObjectMeta, false)
			}
		}
		err = errors.Join(err, r.updateStatus(ctx, cd, nil))
	}()

	if r.IsDisabledValidationWH {
		if err := validationutil.ClusterDeploymentDeletionAllowed(ctx, r.MgmtClient, cd); err != nil {
			r.warnf(cd, "ClusterDeploymentDeletionNotAllowed", err.Error())
			r.setCondition(cd, kcmv1.DeletingCondition, err)
			return ctrl.Result{}, err
		}
	}

	if _, err := r.aggregateCapiConditions(ctx, scope); err != nil {
		return ctrl.Result{}, fmt.Errorf("failed to aggregate conditions from CAPI Cluster for ClusterDeployment %s: %w", client.ObjectKeyFromObject(cd), err)
	}

	if cd.Spec.CleanupOnDeletion {
		if apimeta.IsStatusConditionTrue(cd.Status.Conditions, kcmv1.CloudResourcesDeletedCondition) {
			l.V(1).Info("cleanup of potentially orphaned cloud resources has been successfully concluded, skipping")
		} else {
			l.V(1).Info("cleanup on deletion is set, removing resources")
			requeue, err := r.deleteChildResources(ctx, scope)
			if err != nil {
				l.Error(err, "deleting potentially orphaned cloud resources")
				r.setCondition(cd, kcmv1.CloudResourcesDeletedCondition, err)
				return ctrl.Result{}, err
			}

			if requeue {
				l.V(1).Info("timeout during removing potentially orphaned cloud resources, requeuing", "requeue_after", r.defaultRequeueTime)
				return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
			}

			r.setCondition(cd, kcmv1.CloudResourcesDeletedCondition, nil)
			l.V(1).Info("successfully removed potentially orphaned cloud resources")
		}
	}

	if err := r.releaseProviderCluster(ctx, mgmt, scope); err != nil {
		if r.IsDisabledValidationWH && errors.Is(err, errClusterTemplateNotFound) {
			r.setCondition(cd, kcmv1.DeletingCondition, err)
			l.Error(err, "failed to release provider cluster object due to absent ClusterTemplate, will not retrigger")
			// there is not much to do, we cannot release the clusterdeployment without the clustertemplate
			return ctrl.Result{}, nil
		}

		return ctrl.Result{}, err
	}

	ssExists, err := r.deleteServiceSets(ctx, cd)
	if err != nil {
		r.setCondition(cd, kcmv1.DeletingCondition,
			fmt.Errorf("failed to delete ServiceSets for cluster %s: %w", client.ObjectKeyFromObject(cd), err))
		return ctrl.Result{}, err
	}

	err = r.MgmtClient.Get(ctx, client.ObjectKeyFromObject(cd), &helmcontrollerv2.HelmRelease{})
	if err == nil { // if NO error
		if err := helm.DeleteHelmRelease(ctx, r.MgmtClient, cd.Name, cd.Namespace); err != nil {
			r.setCondition(cd, kcmv1.DeletingCondition, err)
			return ctrl.Result{}, err
		}

		l.Info("HelmRelease still exists, retrying")
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	} else if !apierrors.IsNotFound(err) {
		r.setCondition(cd, kcmv1.DeletingCondition, err)
		return ctrl.Result{}, err
	}
	r.eventf(cd, "HelmReleaseDeleted", "HelmRelease %s has been deleted", client.ObjectKeyFromObject(cd))

	if ssExists {
		r.setCondition(cd, kcmv1.DeletingCondition, errors.New("waiting for ServiceSets to be deleted"))
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	}
	r.eventf(cd, "ServiceSetsDeleted", "ServiceSets for cluster %s have been deleted", client.ObjectKeyFromObject(cd))

	if cdsRequeue, err := r.deleteClusterDataSource(ctx, scope); err != nil {
		l.Error(err, "failed to delete ClusterDataSource")
		r.setCondition(cd, kcmv1.DeletingCondition, err)
		return ctrl.Result{}, err
	} else if cdsRequeue {
		l.Info("Waiting for the ClusterDataSource to be deleted, requeuing")
		r.setCondition(cd, kcmv1.DeletingCondition, errors.New("waiting for ClusterDataSource to be deleted"))
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	}
	r.eventf(cd, "ClusterDataSourceDeleted", "ClusterDataSource %s has been deleted", client.ObjectKeyFromObject(cd))

	cluster, err := r.getPartialCapiCluster(ctx, scope.rgnClient, cd)
	if err == nil && cluster != nil { // if NO error
		l.Info("Cluster still exists, retrying", "cluster name", client.ObjectKeyFromObject(cluster))
		return ctrl.Result{RequeueAfter: r.defaultRequeueTime}, nil
	}
	if err != nil {
		r.setCondition(cd, kcmv1.DeletingCondition, err)
		l.Error(err, "failed to get Cluster")
		return ctrl.Result{}, err
	}

	r.setCondition(cd, kcmv1.DeletingCondition, nil)
	if controllerutil.RemoveFinalizer(cd, kcmv1.ClusterDeploymentFinalizer) {
		l.Info("Removing Finalizer", "finalizer", kcmv1.ClusterDeploymentFinalizer)
		if err := r.MgmtClient.Update(ctx, cd); err != nil {
			return ctrl.Result{}, fmt.Errorf("failed to update clusterDeployment %s: %w", client.ObjectKeyFromObject(cd), err)
		}
		r.eventf(cd, "SuccessfulDelete", "ClusterDeployment has been deleted")
	}

	l.Info("ClusterDeployment deleted")

	return ctrl.Result{}, nil
}

func (r *ClusterDeploymentReconciler) deleteClusterDataSource(ctx context.Context, scope *clusterScope) (bool, error) {
	if scope.kine == nil || scope.kine.dataSource == nil || scope.cd.Spec.DataSource == "" {
		return false, nil
	}

	cds := new(kcmv1.ClusterDataSource)
	cdsKey := client.ObjectKey{Name: scope.cd.Name, Namespace: scope.cd.Namespace}
	err := r.MgmtClient.Get(ctx, cdsKey, cds)
	if err == nil { // if NO error
		if err := r.MgmtClient.Delete(ctx, cds); client.IgnoreNotFound(err) != nil {
			return false, fmt.Errorf("failed to delete ClusterDataSource %s: %w", cdsKey, err)
		}
		return true, nil // cld-related CDS exists, wait for it deletion
	}

	if client.IgnoreNotFound(err) != nil {
		return false, fmt.Errorf("failed to get ClusterDataSource %s: %w", cdsKey, err)
	}

	// not found, we can proceed further
	return false, nil
}

func (r *ClusterDeploymentReconciler) deleteServiceSets(ctx context.Context, cd *kcmv1.ClusterDeployment) (requeue bool, err error) {
	serviceSets := &kcmv1.ServiceSetList{}
	if err := r.MgmtClient.List(ctx, serviceSets, client.InNamespace(cd.Namespace), client.MatchingFields{kcmv1.OwnerRefIndexKey: cd.Name}); err != nil {
		return false, err
	}
	if len(serviceSets.Items) == 0 {
		return false, nil
	}
	for _, ss := range serviceSets.Items {
		if err := r.MgmtClient.Delete(ctx, &ss); err != nil {
			return false, err
		}
	}
	return true, nil
}

func (*ClusterDeploymentReconciler) deleteChildResources(ctx context.Context, scope *clusterScope) (requeue bool, _ error) {
	l := ctrl.LoggerFrom(ctx).WithName("child-cleanup")

	factory, _ := kubeutil.DefaultClientFactoryWithRestConfig()

	const secretKey = "value" // key in the secret, which holds the kubeconfig bytes
	kubeconfigSecretRef := kubeutil.GetKubeconfigSecretKey(client.ObjectKeyFromObject(scope.cd))
	cl, err := kubeutil.GetChildClient(ctx, scope.rgnClient, kubeconfigSecretRef, secretKey, scope.rgnClient.Scheme(), factory)
	if client.IgnoreNotFound(err) != nil {
		return false, fmt.Errorf("failed to get child cluster of ClusterDeployment %s: %w", client.ObjectKeyFromObject(scope.cd), err)
	}

	// secret has been deleted, nothing to do
	if cl == nil {
		l.V(1).Info("Secret with the kubeconfig has not been found, skipping procedure", "secret", kubeconfigSecretRef.String(), "key", secretKey)
		return false, nil
	}

	const deletionTimeout = 10 * time.Second // magic number
	eg, gctx := errgroup.WithContext(ctx)
	gctx = ctrl.LoggerInto(gctx, l)
	now := time.Now()
	eg.Go(func() error {
		if err := kubeutil.DeleteAllExceptAndWait(
			gctx,
			cl,
			&corev1.Service{},
			&corev1.ServiceList{},
			deletionTimeout,
			func(s *corev1.Service) bool { return s.Spec.Type != corev1.ServiceTypeLoadBalancer }, // preserve non-load balancer services
		); err != nil {
			return fmt.Errorf("failed to deletecollection of Services: %w", err)
		}
		return nil
	})
	eg.Go(func() error {
		if err := kubeutil.DeletePVCsAndOwnersAndWait(gctx, cl, deletionTimeout, nil); err != nil {
			return fmt.Errorf("failed to delete PVCs: %w", err)
		}
		return nil
	})

	if err := eg.Wait(); err != nil {
		l.Error(err, "failed to delete objects and wait", "duration", time.Since(now))
		if errors.Is(err, context.DeadlineExceeded) {
			return true, nil // requeue
		}

		return false, err // already wrapped
	}

	l.V(1).Info("Successfully cleaned up")

	return false, nil
}

func (*ClusterDeploymentReconciler) getProviderGVKs(providerInterface *kcmv1.ProviderInterface) []schema.GroupVersionKind {
	if providerInterface == nil {
		return nil
	}
	gvks := make([]schema.GroupVersionKind, 0, len(providerInterface.Spec.ClusterGVKs))

	for _, el := range providerInterface.Spec.ClusterGVKs {
		gvks = append(gvks, schema.GroupVersionKind{
			Group:   el.Group,
			Version: el.Version,
			Kind:    el.Kind,
		})
	}

	return gvks
}

func (r *ClusterDeploymentReconciler) releaseProviderCluster(ctx context.Context, mgmt *kcmv1.Management, scope *clusterScope) error {
	cd := scope.cd
	infraProviders, err := r.getInfraProvidersNames(ctx, cd.Namespace, cd.Spec.Template)
	if err != nil {
		return err
	}

	var parent validationutil.ClusterParent = mgmt
	if scope.region != nil {
		parent = scope.region
	}

	providerInterfaces := make([]*kcmv1.ProviderInterface, 0, len(infraProviders))
	for _, infraProvider := range infraProviders {
		if pi := providerinterface.FindProviderInterfaceForInfra(ctx, scope.rgnClient, parent, infraProvider); pi != nil {
			providerInterfaces = append(providerInterfaces, pi)
		}
	}

	// Associate the provider with it's GVK
	for _, pi := range providerInterfaces {
		gvks := r.getProviderGVKs(pi)
		if len(gvks) == 0 {
			continue
		}

		cluster, err := r.getProviderCluster(ctx, scope.rgnClient, cd.Namespace, cd.Name, gvks...)
		if err != nil {
			if !errors.Is(err, errClusterNotFound) {
				return err
			}
			return nil
		}

		found, err := r.clusterCAPIMachinesExist(ctx, scope.rgnClient, cd.Namespace, cluster.Name)
		if err != nil {
			continue
		}

		if !found {
			finalizersUpdated, err := r.removeClusterFinalizer(ctx, scope.rgnClient, cluster)
			if finalizersUpdated {
				r.eventf(cd, "ClusterDeleted", "Cluster %s has been deleted", client.ObjectKeyFromObject(cd))
			}
			if err != nil {
				return fmt.Errorf("failed to remove finalizer from %s %s: %w", cluster.Kind, client.ObjectKeyFromObject(cluster), err)
			}
		}
	}

	return nil
}

// getInfraProvidersNames returns the list of exposed infrastructure providers with the `infrastructure-` prefix for provided template
func (r *ClusterDeploymentReconciler) getInfraProvidersNames(ctx context.Context, templateNamespace, templateName string) ([]string, error) {
	template := &kcmv1.ClusterTemplate{}
	templateRef := client.ObjectKey{Name: templateName, Namespace: templateNamespace}
	if err := r.MgmtClient.Get(ctx, templateRef, template); err != nil {
		ctrl.LoggerFrom(ctx).Error(err, "Failed to get ClusterTemplate", "template namespace", templateNamespace, "template name", templateName)
		if apierrors.IsNotFound(err) {
			return nil, fmt.Errorf("failed to get ClusterTemplate %s: %w", templateRef, errClusterTemplateNotFound)
		}
		return nil, err
	}

	ips := make([]string, 0, len(template.Status.Providers))
	for _, v := range template.Status.Providers {
		if strings.HasPrefix(v, kcmv1.InfrastructureProviderPrefix) {
			ips = append(ips, v)
		}
	}

	return ips, nil
}

// getProviderCluster fetches a first provider Cluster from the given list of GVKs.
func (*ClusterDeploymentReconciler) getProviderCluster(ctx context.Context, rgnClient client.Client, namespace, name string, gvks ...schema.GroupVersionKind) (*metav1.PartialObjectMetadata, error) {
	for _, gvk := range gvks {
		itemsList := &metav1.PartialObjectMetadataList{}
		itemsList.SetGroupVersionKind(gvk)
		if err := rgnClient.List(ctx, itemsList, client.InNamespace(namespace), client.MatchingLabels{kcmv1.FluxHelmChartNameKey: name}); err != nil {
			return nil, fmt.Errorf("failed to list %s in namespace %s: %w", gvk.Kind, namespace, err)
		}

		if len(itemsList.Items) > 0 {
			return &itemsList.Items[0], nil
		}
	}

	return nil, errClusterNotFound
}

func (*ClusterDeploymentReconciler) removeClusterFinalizer(ctx context.Context, rgnClient client.Client, cluster *metav1.PartialObjectMetadata) (finalizersUpdated bool, err error) {
	originalCluster := *cluster
	if finalizersUpdated = controllerutil.RemoveFinalizer(cluster, kcmv1.BlockingFinalizer); finalizersUpdated {
		ctrl.LoggerFrom(ctx).Info("Allow to stop cluster", "finalizer", kcmv1.BlockingFinalizer)
		if err := rgnClient.Patch(ctx, cluster, client.MergeFrom(&originalCluster)); err != nil {
			return false, fmt.Errorf("failed to patch cluster %s/%s: %w", cluster.Namespace, cluster.Name, err)
		}
	}

	return finalizersUpdated, nil
}

func (*ClusterDeploymentReconciler) clusterCAPIMachinesExist(ctx context.Context, rgnClient client.Client, namespace, clusterName string) (bool, error) {
	itemsList := &metav1.PartialObjectMetadataList{}
	itemsList.SetGroupVersionKind(clusterapiv1.GroupVersion.WithKind("Machine"))
	if err := rgnClient.List(ctx, itemsList, client.InNamespace(namespace), client.Limit(1), client.MatchingLabels{clusterapiv1.ClusterNameLabel: clusterName}); err != nil {
		return false, err
	}
	return len(itemsList.Items) != 0, nil
}

func (r *ClusterDeploymentReconciler) setAvailableUpgrades(ctx context.Context, clusterDeployment *kcmv1.ClusterDeployment, clusterTpl *kcmv1.ClusterTemplate) error {
	if clusterTpl == nil {
		return nil
	}

	chains := new(kcmv1.ClusterTemplateChainList)
	if err := r.MgmtClient.List(ctx, chains,
		client.InNamespace(clusterTpl.Namespace),
		client.MatchingFields{kcmv1.TemplateChainSupportedTemplatesIndexKey: clusterTpl.Name},
	); err != nil {
		return fmt.Errorf("failed to list ClusterTemplateChains: %w", err)
	}

	availableUpgradesMap := make(map[string]kcmv1.AvailableUpgrade)
	for _, chain := range chains.Items {
		for _, supportedTemplate := range chain.Spec.SupportedTemplates {
			if supportedTemplate.Name == clusterTpl.Name {
				for _, availableUpgrade := range supportedTemplate.AvailableUpgrades {
					availableUpgradesMap[availableUpgrade.Name] = availableUpgrade
				}
			}
		}
	}

	availableUpgrades := make([]string, 0, len(availableUpgradesMap))
	for _, availableUpgrade := range availableUpgradesMap {
		availableUpgrades = append(availableUpgrades, availableUpgrade.Name)
	}

	clusterDeployment.Status.AvailableUpgrades = availableUpgrades
	return nil
}

// templatesValidUpdateSource is a source of update and create events which enqueues ClusterDeployment objects if the referenced ServiceTemplate or ClusterTemplate object gets the valid status.
func (*ClusterDeploymentReconciler) templatesValidUpdateSource(cl client.Client, cache crcache.Cache, obj client.Object) source.TypedSource[ctrl.Request] {
	var isServiceTemplateKind bool // quick kludge to avoid complicated switches
	var indexKey string

	switch obj.(type) {
	case *kcmv1.ServiceTemplate:
		isServiceTemplateKind = true
		indexKey = kcmv1.ClusterDeploymentServiceTemplatesIndexKey
	case *kcmv1.ClusterTemplate:
		indexKey = kcmv1.ClusterDeploymentTemplateIndexKey
	default:
		panic(fmt.Sprintf("unexpected type %T, expected one of [%T, %T]", obj, new(kcmv1.ServiceTemplate), new(kcmv1.ClusterTemplate)))
	}

	return source.TypedKind(cache, obj, handler.TypedEnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
		clds := new(kcmv1.ClusterDeploymentList)
		if err := cl.List(ctx, clds, client.InNamespace(o.GetNamespace()), client.MatchingFields{indexKey: o.GetName()}); err != nil {
			return nil
		}

		resp := make([]ctrl.Request, 0, len(clds.Items))
		for _, v := range clds.Items {
			resp = append(resp, ctrl.Request{NamespacedName: client.ObjectKeyFromObject(&v)})
		}

		return resp
	}), predicate.TypedFuncs[client.Object]{
		GenericFunc: func(event.TypedGenericEvent[client.Object]) bool { return false },
		DeleteFunc:  func(event.TypedDeleteEvent[client.Object]) bool { return false },
		UpdateFunc: func(tue event.TypedUpdateEvent[client.Object]) bool {
			// NOTE: might be optimized, probably with go's core types gone (>=go1.25)
			if isServiceTemplateKind {
				sto, ok := tue.ObjectOld.(*kcmv1.ServiceTemplate)
				if !ok {
					return false
				}
				stn, ok := tue.ObjectNew.(*kcmv1.ServiceTemplate)
				if !ok {
					return false
				}
				return stn.Status.Valid && !sto.Status.Valid
			}

			cto, ok := tue.ObjectOld.(*kcmv1.ClusterTemplate)
			if !ok {
				return false
			}
			ctn, ok := tue.ObjectNew.(*kcmv1.ClusterTemplate)
			if !ok {
				return false
			}
			return ctn.Status.Valid && !cto.Status.Valid
		},
	})
}

func (r *ClusterDeploymentReconciler) processClusterIPAM(ctx context.Context, cd *kcmv1.ClusterDeployment) error {
	// a compliment check of input values
	if cd.Spec.IPAMClaim.ClusterIPAMClaimRef == "" && cd.Spec.IPAMClaim.ClusterIPAMClaimSpec == nil {
		return nil
	}

	clusterIpamClaim := kcmv1.ClusterIPAMClaim{}
	// if the ClusterIPAMClaimSpec is not nil we need to create a new ClusterIPAMClaim object
	// or ensure the configuration of the existing ClusterIPAMClaim object. Then we need to
	// update the ClusterIPAMClaimRef in case it does not match the name of the ClusterIPAMClaim object.
	if cd.Spec.IPAMClaim.ClusterIPAMClaimSpec != nil {
		claimName := cd.Name + "-ipam"
		clusterIpamClaim.Name = claimName
		clusterIpamClaim.Namespace = cd.Namespace
		kubeutil.AddOwnerReference(&clusterIpamClaim, cd)
		_, err := ctrl.CreateOrUpdate(ctx, r.MgmtClient, &clusterIpamClaim, func() error {
			clusterIpamClaim.Spec = *cd.Spec.IPAMClaim.ClusterIPAMClaimSpec
			clusterIpamClaim.Spec.ClusterIPAMRef = claimName
			return nil
		})
		if err != nil {
			return fmt.Errorf("failed to create or update ClusterIPAMClaim: %w", err)
		}

		if cd.Spec.IPAMClaim.ClusterIPAMClaimRef != clusterIpamClaim.Name {
			cd.Spec.IPAMClaim.ClusterIPAMClaimRef = claimName
			if err := r.MgmtClient.Update(ctx, cd); err != nil {
				return fmt.Errorf("failed to update ClusterDeployment: %w", err)
			}
			return errClusterDeploymentSpecUpdated
		}
	} else {
		clusterIpamClaimRef := client.ObjectKey{Name: cd.Spec.IPAMClaim.ClusterIPAMClaimRef, Namespace: cd.Namespace}
		err := r.MgmtClient.Get(ctx, clusterIpamClaimRef, &clusterIpamClaim)
		if err != nil {
			return fmt.Errorf("failed to fetch ClusterIPAMClaim: %w", err)
		}
		if clusterIpamClaim.Spec.Cluster != cd.Name {
			return errors.Join(errInvalidIPAMClaimRef, fmt.Errorf(
				"ClusterIPAMClaim.Spec.Cluster %s does not match ClusterDeployment.Name %s", clusterIpamClaim.Spec.Cluster, cd.Name))
		}
	}

	if !clusterIpamClaim.Status.Bound {
		return errIPAMNotReady
	}

	clusterIpamRef := client.ObjectKey{Name: clusterIpamClaim.Spec.ClusterIPAMRef, Namespace: cd.Namespace}
	clusterIpam := kcmv1.ClusterIPAM{}
	if err := r.MgmtClient.Get(ctx, clusterIpamRef, &clusterIpam); err != nil {
		return fmt.Errorf("failed to fetch ClusterIPAM: %w", err)
	}

	needsUpdate, err := configNeedsUpdate(cd.Spec.Config, clusterIpam.Status.ProviderData)
	if err != nil {
		return fmt.Errorf("failed to determine whether config needs update: %w", err)
	}
	if needsUpdate {
		if err := cd.AddHelmValues(func(values map[string]any) error {
			values["ipamEnabled"] = true
			for _, v := range clusterIpam.Status.ProviderData {
				values[v.Name] = v
			}
			return nil
		}); err != nil {
			return fmt.Errorf("failed to add IPAM Helm values: %w", err)
		}
		if err := r.MgmtClient.Update(ctx, cd); err != nil {
			return fmt.Errorf("failed to update ClusterDeployment: %w", err)
		}
		return errClusterDeploymentSpecUpdated
	}

	return nil
}

func (r *ClusterDeploymentReconciler) handleCertificateSecrets(ctx context.Context, rgnClient client.Client, cd *kcmv1.ClusterDeployment) error {
	secretsToHandle := []string{r.K0sURLCertSecretName, r.RegistryCertSecretName}

	l := ctrl.LoggerFrom(ctx).WithName("handle-secrets")

	if cd.Namespace == r.SystemNamespace { // nothing to copy
		return nil
	}

	l.V(1).Info("Copying certificate secrets from the system namespace to the ClusterDeployment namespace")
	for _, secretName := range secretsToHandle {
		if err := kubeutil.CopySecret(
			ctx,
			r.MgmtClient,
			rgnClient,
			client.ObjectKey{Namespace: r.SystemNamespace, Name: secretName},
			cd.Namespace,
			"",
			nil,
			nil,
		); err != nil {
			l.Error(err, "failed to copy Secret for the ClusterDeployment")
			return err
		}
	}
	return nil
}

func (r *ClusterDeploymentReconciler) collectServicesStatuses(ctx context.Context, cd *kcmv1.ClusterDeployment) ([]kcmv1.ServiceState, error) {
	serviceSets := new(kcmv1.ServiceSetList)
	if err := r.MgmtClient.List(ctx, serviceSets, client.InNamespace(cd.Namespace), client.MatchingFields{kcmv1.ServiceSetClusterIndexKey: cd.Name}); err != nil {
		return nil, fmt.Errorf("failed to list ServiceSets: %w", err)
	}
	aggregatedServiceStatuses := make([]kcmv1.ServiceState, 0, len(serviceSets.Items))

	for _, serviceSet := range serviceSets.Items {
		aggregatedServiceStatuses = append(aggregatedServiceStatuses, serviceSet.Status.Services...)
	}

	return aggregatedServiceStatuses, nil
}

func configNeedsUpdate(config *apiextv1.JSON, providerData []kcmv1.ClusterIPAMProviderData) (bool, error) {
	// Check if values are already present in the config
	valuesNeedUpdate := false

	// Convert cd.Spec.Config to a map for checking
	var currentValues map[string]any
	if config != nil {
		if err := json.Unmarshal(config.Raw, &currentValues); err != nil {
			return false, fmt.Errorf("failed to unmarshal current config values: %w", err)
		}
	} else {
		currentValues = make(map[string]any)
	}

	// Check if ipamEnabled is already set correctly
	ipamEnabled, ipamEnabledExists := currentValues["ipamEnabled"].(bool)
	if !ipamEnabledExists || !ipamEnabled {
		valuesNeedUpdate = true
	}

	// Check if all provider data values are present
	if !valuesNeedUpdate {
		for _, v := range providerData {
			if _, exists := currentValues[v.Name]; !exists {
				valuesNeedUpdate = true
				break
			}
		}
	}
	return valuesNeedUpdate, nil
}

// createOrUpdateServiceSet creates or updates the ServiceSet for the given ClusterDeployment.
func (r *ClusterDeploymentReconciler) createOrUpdateServiceSet(
	ctx context.Context,
	cd *kcmv1.ClusterDeployment,
) error {
	l := ctrl.LoggerFrom(ctx).WithName("handle-service-set")

	var err error
	providerSpec, err := serviceset.StateManagementProviderConfigFromServiceSpec(cd.Spec.ServiceSpec)
	if err != nil {
		return fmt.Errorf("failed to convert ServiceSpec to provider config: %w", err)
	}

	key := client.ObjectKey{
		Name: providerSpec.Name,
	}
	provider := new(kcmv1.StateManagementProvider)
	if err := r.MgmtClient.Get(ctx, key, provider); err != nil {
		return fmt.Errorf("failed to get StateManagementProvider %s: %w", key.String(), err)
	}

	serviceSetObjectKey := client.ObjectKeyFromObject(cd)
	opRequisites := serviceset.OperationRequisites{
		ObjectKey:            client.ObjectKeyFromObject(cd),
		Services:             cd.Spec.ServiceSpec.Services,
		ProviderSpec:         providerSpec,
		PropagateCredentials: cd.Spec.PropagateCredentials,
	}

	serviceSet, op, err := serviceset.GetServiceSetWithOperation(ctx, r.MgmtClient, opRequisites)
	if err != nil {
		return fmt.Errorf("failed to get ServiceSet %s: %w", serviceSetObjectKey.String(), err)
	}

	if op == kcmv1.ServiceSetOperationNone {
		return nil
	}

	upgradePaths, err := serviceset.ServicesUpgradePaths(
		ctx, r.MgmtClient, serviceset.ServicesWithDesiredChains(cd.Spec.ServiceSpec.Services, serviceSet.Spec.Services), cd.Namespace,
	)
	if err != nil {
		return fmt.Errorf("failed to determine upgrade paths for services: %w", err)
	}
	l.V(1).Info("Determined upgrade paths for services", "upgradePaths", upgradePaths)

	filteredServices, err := serviceset.FilterServiceDependencies(ctx, r.MgmtClient, r.SystemNamespace, nil, cd, cd.Spec.ServiceSpec.Services)
	if err != nil {
		return fmt.Errorf("failed to filter for services that are not dependent on any other service: %w", err)
	}
	l.V(1).Info("Services to deploy after filtering services that are not dependent on any other service", "services", filteredServices)

	err = serviceset.ResolveServiceVersions(ctx, r.MgmtClient, cd.Namespace, filteredServices)
	if err != nil {
		return fmt.Errorf("failed to retrieve version information for filtered services: %w", err)
	}

	serviceSetServices := serviceSet.Spec.Services
	err = serviceset.ResolveServiceVersions(ctx, r.MgmtClient, cd.Namespace, serviceSetServices)
	if err != nil {
		return fmt.Errorf("failed to retrieve version information for service set services: %w", err)
	}

	resultingServices := serviceset.ServicesToDeploy(upgradePaths, filteredServices, serviceSet)
	l.V(1).Info("Services to deploy", "services", resultingServices)

	serviceSet, err = serviceset.NewBuilder(cd, serviceSet, provider.Spec.Selector).
		WithServicesToDeploy(resultingServices).Build()
	if err != nil {
		return fmt.Errorf("failed to build ServiceSet: %w", err)
	}

	serviceSetProcessor := serviceset.NewProcessor(r.MgmtClient)
	err = serviceSetProcessor.CreateOrUpdateServiceSet(ctx, op, serviceSet)
	if err != nil {
		return fmt.Errorf("failed to create or update ServiceSet %s: %w", serviceSetObjectKey.String(), err)
	}
	return nil
}

func (*ClusterDeploymentReconciler) eventf(cd *kcmv1.ClusterDeployment, reason, message string, args ...any) {
	record.Eventf(cd, cd.Generation, reason, message, args...)
}

func (*ClusterDeploymentReconciler) warnf(cd *kcmv1.ClusterDeployment, reason, message string, args ...any) {
	record.Warnf(cd, cd.Generation, reason, message, args...)
}

// SetupWithManager sets up the controller with the Manager.
func (r *ClusterDeploymentReconciler) SetupWithManager(mgr ctrl.Manager) error {
	r.MgmtClient = mgr.GetClient()
	r.helmActor = helm.NewActor(mgr.GetConfig(), r.MgmtClient.RESTMapper())

	r.defaultRequeueTime = 10 * time.Second

	managedController := ctrl.NewControllerManagedBy(mgr).
		WithOptions(controller.TypedOptions[ctrl.Request]{
			RateLimiter: ratelimitutil.DefaultFastSlow(),
		}).
		For(&kcmv1.ClusterDeployment{}).
		Owns(&kcmv1.ClusterDataSource{}, builder.WithPredicates(predicate.Funcs{
			GenericFunc: func(event.TypedGenericEvent[client.Object]) bool { return false },
			CreateFunc:  func(event.TypedCreateEvent[client.Object]) bool { return false },
			DeleteFunc:  func(event.TypedDeleteEvent[client.Object]) bool { return false },
			UpdateFunc: func(tue event.TypedUpdateEvent[client.Object]) bool {
				oldCDS, ok := tue.ObjectOld.(*kcmv1.ClusterDataSource)
				if !ok {
					return false
				}
				newCDS, ok := tue.ObjectNew.(*kcmv1.ClusterDataSource)
				if !ok {
					return false
				}
				return oldCDS.Status != newCDS.Status
			},
		})).
		Watches(&helmcontrollerv2.HelmRelease{},
			handler.EnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
				clusterDeploymentRef := client.ObjectKeyFromObject(o)
				if err := r.MgmtClient.Get(ctx, clusterDeploymentRef, &kcmv1.ClusterDeployment{}); err != nil {
					return []ctrl.Request{}
				}

				return []ctrl.Request{{NamespacedName: clusterDeploymentRef}}
			}),
		).
		Watches(&kcmv1.ClusterTemplateChain{},
			handler.EnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
				chain, ok := o.(*kcmv1.ClusterTemplateChain)
				if !ok {
					return nil
				}

				var req []ctrl.Request
				for _, template := range getTemplateNamesManagedByChain(chain) {
					clusterDeployments := &kcmv1.ClusterDeploymentList{}
					err := r.MgmtClient.List(ctx, clusterDeployments,
						client.InNamespace(chain.Namespace),
						client.MatchingFields{kcmv1.ClusterDeploymentTemplateIndexKey: template})
					if err != nil {
						return []ctrl.Request{}
					}
					for _, cluster := range clusterDeployments.Items {
						req = append(req, ctrl.Request{
							NamespacedName: client.ObjectKey{
								Namespace: cluster.Namespace,
								Name:      cluster.Name,
							},
						})
					}
				}
				return req
			}),
			builder.WithPredicates(predicate.Funcs{
				UpdateFunc:  func(event.UpdateEvent) bool { return false },
				GenericFunc: func(event.GenericEvent) bool { return false },
			}),
		).
		Watches(&kcmv1.Credential{},
			handler.EnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
				clusterDeployments := &kcmv1.ClusterDeploymentList{}
				err := r.MgmtClient.List(ctx, clusterDeployments,
					client.InNamespace(o.GetNamespace()),
					client.MatchingFields{kcmv1.ClusterDeploymentCredentialIndexKey: o.GetName()})
				if err != nil {
					return []ctrl.Request{}
				}

				req := []ctrl.Request{}
				for _, cluster := range clusterDeployments.Items {
					req = append(req, ctrl.Request{
						NamespacedName: client.ObjectKey{
							Namespace: cluster.Namespace,
							Name:      cluster.Name,
						},
					})
				}

				return req
			}),
		).
		Watches(&kcmv1.ClusterAuthentication{},
			handler.EnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
				clusterDeployments := new(kcmv1.ClusterDeploymentList)
				if err := r.MgmtClient.List(ctx, clusterDeployments,
					client.InNamespace(o.GetNamespace()),
					client.MatchingFields{kcmv1.ClusterDeploymentAuthenticationIndexKey: o.GetName()}); err != nil {
					return nil
				}

				req := make([]ctrl.Request, len(clusterDeployments.Items))
				for i, cluster := range clusterDeployments.Items {
					req[i] = ctrl.Request{
						NamespacedName: client.ObjectKey{
							Namespace: cluster.Namespace,
							Name:      cluster.Name,
						},
					}
				}

				return req
			}),
			// NOTE: on deletion of a ClusterAuthentication we should not delete auth-related helm values
			builder.WithPredicates(predicate.Funcs{
				GenericFunc: func(event.TypedGenericEvent[client.Object]) bool { return false },
				DeleteFunc:  func(event.TypedDeleteEvent[client.Object]) bool { return false },
				UpdateFunc:  func(event.TypedUpdateEvent[client.Object]) bool { return true },
				CreateFunc:  func(event.TypedCreateEvent[client.Object]) bool { return true },
			}),
		).
		Watches(&kcmv1.Region{}, handler.EnqueueRequestsFromMapFunc(func(ctx context.Context, o client.Object) []ctrl.Request {
			creds := new(kcmv1.CredentialList)
			if err := r.MgmtClient.List(ctx, creds, client.MatchingFields{kcmv1.CredentialRegionIndexKey: o.GetName()}); err != nil {
				return nil
			}

			reqs := []ctrl.Request{}
			for _, cred := range creds.Items {
				clds := new(kcmv1.ClusterDeploymentList)
				if err := r.MgmtClient.List(ctx, clds,
					client.MatchingFields{kcmv1.ClusterDeploymentCredentialIndexKey: cred.Name},
					client.InNamespace(cred.Namespace)); err != nil {
					continue
				}
				for _, cld := range clds.Items {
					reqs = append(reqs, ctrl.Request{NamespacedName: client.ObjectKeyFromObject(&cld)})
				}
			}

			return reqs
		}),
			builder.WithPredicates(predicate.Funcs{
				GenericFunc: func(event.TypedGenericEvent[client.Object]) bool { return false },
				CreateFunc:  func(event.TypedCreateEvent[client.Object]) bool { return false },
				DeleteFunc:  func(event.TypedDeleteEvent[client.Object]) bool { return false },
				UpdateFunc: func(tue event.TypedUpdateEvent[client.Object]) bool {
					if tue.ObjectNew == nil || tue.ObjectOld == nil {
						return false
					}
					_, okn := tue.ObjectNew.GetAnnotations()[kcmv1.RegionPauseAnnotation]
					_, oko := tue.ObjectOld.GetAnnotations()[kcmv1.RegionPauseAnnotation]
					return !okn && oko // new without; old with
				},
			}),
		).
		Owns(&kcmv1.ServiceSet{}, builder.WithPredicates(predicate.Funcs{
			CreateFunc:  func(event.CreateEvent) bool { return false },
			GenericFunc: func(event.GenericEvent) bool { return false },
		}))

	if r.IsDisabledValidationWH {
		setupLog := mgr.GetLogger().WithName("clusterdeployment_ctrl_setup")
		managedController.WatchesRawSource(r.templatesValidUpdateSource(mgr.GetClient(), mgr.GetCache(), &kcmv1.ServiceTemplate{}))
		setupLog.Info("Validations are disabled, watcher for ServiceTemplate objects is set")
		managedController.WatchesRawSource(r.templatesValidUpdateSource(mgr.GetClient(), mgr.GetCache(), &kcmv1.ClusterTemplate{}))
		setupLog.Info("Validations are disabled, watcher for ClusterTemplate objects is set")
	}

	return managedController.Complete(r)
}
