## DB Module

This contains the core functions for dealing with CouchDB. That includes
document CRUD operations, querying views and creating/deleting databases.


### Events

The db module is an EventEmitter. See the
[events package](http://kan.so/packages/details/events) for more information.

#### unauthorized

Emitted by the db module when a request results in a 401 Unauthorized
response. This is listened to used by the session module to help detect
session timeouts etc.

```javascript
var db = require("db");

db.on('unauthorized', function (req) {
    // req is the ajax request object which returned 401
});
```


### API


#### guessCurrent([loc])

Attempts to guess the database name and design doc id from the current URL,
or the loc paramter. Returns an object with 'db', 'design\_doc' and 'root'
properties, or null for a URL not matching the expected format (perhaps
behing a vhost).

You wouldn't normally use this function directly, but use `db.current()` to
return a DB object bound to the current database instead.

* __loc__ - _String_ - An alternative URL to use instead of window.location (optional)

__returns:__ _Object|null_


#### escapeUrlParams([obj])

Converts an object to a string of properly escaped URL parameters.

* __obj__ - _Object_ - An object containing url parameters, with parameter names stored as property names (or keys).

__returns:__ _String_


#### encode(str)

Encodes a document id or view, list or show name. This also will make sure
the forward-slash is not escaped for documents with id's beginning with
"\\_design/".

* __str__ - _String_ - the name or id to escape

__returns:__ _String_


#### stringifyQuery(query)

Properly encodes query parameters to CouchDB views etc. Handle complex
keys and other non-string parameters by passing through JSON.stringify.
Returns a shallow-copied clone of the original query after complex values
have been stringified.

* __query__ - _Object_

__returns:__ _Object_


#### request(options, callback)

Make a request, with some default settings, proper callback
handling, and optional caching. Used behind-the-scenes by
most other DB module functions.

* __options__ - _Object_
* __callback(err,response)__ - _Function_


#### createDatabase(name, callback)

Creates a CouchDB database.

If you're running behind a virtual host you'll need to set up
appropriate rewrites for a DELETE request to '/' either turning off safe
rewrites or setting up a new vhost entry.

* __name__ - _String_
* __callback(err,response)__ - _Function_


#### deleteDatabase(name, callback)

Deletes a CouchDB database.

If you're running behind a virtual host you'll need to set up
appropriate rewrites for a DELETE request to '/' either turning off safe
rewrites or setting up a new vhost entry.

* __name__ - _String_
* __callback(err,response)__ - _Function_


#### allDbs(callback)

Lists all databses

If you're running behind a virtual host you'll need to set up
appropriate rewrites for a DELETE request to '/' either turning off safe
rewrites or setting up a new vhost entry.

* __callback(err,response)__ - _Function_


#### newUUID(cacheNum, callback)

Returns a new UUID generated by CouchDB. Its possible to cache
multiple UUIDs for later use, to avoid making too many requests.

* __cacheNum__ - _Number_ - (optional, default: 1)
* __callback(err,response)__ - _Function_


#### use(url)

Creates a new DB object with methods operating on the database at 'url'

The DB object also exposes the same module-level methods (eg, createDatabase)
so it can be used in-place of the db exports object, for example:

```javascript
var db = require('db').use('mydb');

db.createDatabase('example', function (err, resp) {
    // do something
});
```

* __url__ - _String_ - The url to bind the new DB object to

__returns:__ _DB_


#### current()

Attempts to guess the current DB name and return a DB object using that.
Should work reliably unless running behind a virtual host.

Throws an error if the current database url cannot be detected.

The DB object also exposes the same module-level methods (eg, createDatabase)
so it can be used in-place of the db exports object, for example:

```javascript
var db = require('db').current();

db.createDatabase('example', function (err, resp) {
    // do something
});
```

__returns:__ _DB_


#### DB.getRewrite(name, path, [q], callback)

Fetches a rewrite from the database the app is running on. Results
are passed to the callback, with the first argument of the callback
reserved for any exceptions that occurred (node.js style).

* __name__ - _String_ - the name of the design doc
* __path__ - _String_
* __q__ - _Object_ - (optional)
* __callback(err,response)__ - _Function_


#### DB.allDesignDocs([q], callback)

Queries all design documents in the database.

* __q__ - _Object_ - query parameters to pass to /_all_docs (optional)
* __callback(err,response)__ - _Function_


#### DB.allDocs([q], callback)

Queries all documents in the database (include design docs).

* __q__ - _Object_ - query parameters to pass to /_all_docs (optional)
* __callback(err,response)__ - _Function_


#### DB.getDoc(id, [q], callback)

Fetches a document from the database the app is running on. Results are
passed to the callback, with the first argument of the callback reserved
for any exceptions that occurred (node.js style).

* __id__ - _String_
* __q__ - _Object_ - (optional)
* __callback(err,response)__ - _Function_


#### DB.saveDoc(doc, callback)

Saves a document to the database the app is running on. Results are
passed to the callback, with the first argument of the callback reserved
for any exceptions that occurred (node.js style).

* __doc__ - _Object_
* __callback(err,response)__ - _Function_


#### DB.removeDoc(doc, callback)

Deletes a document from the database the app is running on. Results are
passed to the callback, with the first argument of the callback reserved
for any exceptions that occurred (node.js style).

* __doc__ - _Object_
* __callback(err,response)__ - _Function_


#### DB.getView(name, view, [q], callback)

Fetches a view from the database the app is running on. Results are
passed to the callback, with the first argument of the callback reserved
for any exceptions that occurred (node.js style).

* __name__ - _String_ - name of the design doc to use
* __view__ - _String_ - name of the view
* __q__ - _Object_ - (optional)
* __callback(err,response)__ - _Function_


#### DB.getList(name, list, view, [q], callback)

Transforms and fetches a view through a list from the database the app
is running on. Results are passed to the callback, with the first
argument of the callback reserved for any exceptions that occurred
(node.js style).

* __name__ - _String_ - name of the design doc to use
* __list__ - _String_ - name of the list function
* __view__ - _String_ - name of the view to apply the list function to
* __q__ - _Object_ - (optional)
* __callback(err,response)__ - _Function_


#### DB.getShow(name, show, docid, [q], callback)

Transforms and fetches a document through a show from the database the app
is running on. Results are passed to the callback, with the first
argument of the callback reserved for any exceptions that occurred
(node.js style).

* __name__ - _String_ - name of the design doc to use
* __show__ - _String_ - name of the show function
* __docid__ - _String_ - id of the document to apply the show function to
* __q__ - _Object_ - (optional)
* __callback(err,response)__ - _Function_


#### DB.getDesignDoc(name, callback)

Fetch a design document from CouchDB.

* __name__ - The name of (i.e. path to) the design document without the preceeding "\_design/".
* __callback__ - The callback to invoke when the request completes.


#### DB.info(callback)

Gets information about the database.

* __callback(err,response)__ - _Function_


#### DB.changes([q], callback)

Listen to the changes feed for a database.

\__Options:__
* \_filter_ - the filter function to use
* \_since_ - the update_seq to start listening from
* \_heartbeat_ - the heartbeat time (defaults to 10 seconds)
* \_include_docs_ - whether to include docs in the results

Returning false from the callback will cancel the changes listener

* __q__ - _Object_ - (optional) query parameters (see options above)
* __callback(err,response)__ - _Function_


#### DB.bulkSave(docs, [options], callback)

Saves a list of documents, without using separate requests.
This function uses CouchDB's HTTP bulk document API (\_bulk_docs).
The return value is an array of objects, each containing an 'id'
and a 'rev' field. The return value is passed to the callback you
provide via its second argument; the first argument of the callback
is reserved for any exceptions that occurred (node.js style).

**Options:**
* *all\_or\_nothing* - Require that all documents be saved
  successfully (or saved with a conflict); otherwise roll
  back the operation.

* __docs__ - _Array_ - An array of documents; each document is an object
* __options__ - _Object_ - (optional) Options for the bulk-save operation.
* __callback(err,response)__ - _Function_ - A function to accept results and/or errors. Document update conflicts are reported in the results array.


#### DB.bulkGet(keys, [q], callback)

Requests a list of documents, using only a single HTTP request.
This function uses CouchDB's HTTP bulk document API (\_all_docs).
The return value is an array of objects, each of which is a document.
The return value is passed to the callback you provide via its second
argument; the first argument of the callback is reserved for any
exceptions that occurred (node.js style).

* __keys__ - _Array_ - An array of documents identifiers (i.e. strings).
* __q__ - _Object_ - (optional) Query parameters for the bulk-read operation.
* __callback(err,response)__ - _Function_ - A function to accept results and/or errors. Document update conflicts are reported in the results array.
