var couchdb = require('./couchdb'),
    settings = require('./settings'),
    versions = require('./versions'),
    cache = require('./cache'),
    utils = require('./utils'),
    logger = require('./logger'),
    semver = require('semver'),
    mime = require('mime'),
    async = require('async'),
    http = require('http'),
    https = require('https'),
    path = require('path'),
    url = require('url'),
    urlParse = url.parse,
    urlFormat = url.format,
    fs = require('fs'),
    _ = require('underscore')._;


/**
 * Temporary directory used to download new packages to before unpacking and
 * adding to cache.
 */

exports.TMP_DIR = process.env.HOME + '/.kanso/tmp';


/**
 * Attaches a .tar.gz file to a package document prior to publishing a new
 * version.
 *
 * @param {Object} doc - the document to attach the file to
 * @param {Object} cfg - the values from kanso.json
 * @param {String} tfile - the path to the .tar.gz file to attach
 * @param {Function} callback
 */

exports.attachTar = function (doc, cfg, tfile, callback) {
    fs.readFile(tfile, function (err, content) {
        if (err) {
            return callback(err);
        }
        doc._attachments = doc._attachments || {};
        doc._attachments[cfg.name + '-' + cfg.version + '.tar.gz'] = {
            content_type: 'application/x-compressed-tar',
            data: content.toString('base64')
        };
        callback(null, doc);
    });
};


/**
 * Attaches README file to a package document prior to publishing a new version.
 *
 * @param {Object} doc - the document to attach the file to
 * @param {Object} cfg - the values from kanso.json
 * @param {String} readme - the path to the README file to attach
 * @param {Function} callback
 */

exports.attachREADME = function (doc, cfg, readme, callback) {
    if (!readme) {
        return callback(null, doc);
    }
    fs.readFile(readme, function (err, content) {
        if (err) {
            return callback(err);
        }
        var basename = path.basename(readme);
        doc._attachments = doc._attachments || {};
        doc._attachments['docs/' + cfg.version + '/' + basename] = {
            content_type: mime.lookup(readme),
            data: content.toString('base64')
        };
        callback(null, doc);
    });
};


/**
 * Update a package document for a new published version. Adds .tar.gz archive
 * and README file as attachments.
 *
 * @param {Object} doc - the doc object to update
 * @param {Date} time - created time of the package
 * @param {Object} cfg - the config options from kanso.json
 * @param {String} tfile - the path to the .tar.gz archive for the new version
 * @param {String} readme - path to the README file for the new version
 * @param {Function} callback
 */

exports.updateDoc = function (doc, time, cfg, tfile, readme, callback) {
    doc.time.modified = utils.ISODateString(time);
    doc.time[cfg.version] = utils.ISODateString(time);

    doc.versions[cfg.version] = cfg;

    var vers = Object.keys(doc.versions);
    var highest = versions.max(vers);

    if (highest === cfg.version) {
        doc.tags = doc.tags || {};
        doc.tags.latest = cfg.version;

        doc.name = cfg.name;
        doc.author = cfg.author;
        doc.website = cfg.website;
        doc.maintainers = cfg.maintainers;
        doc.description = cfg.description;
        doc.categories = cfg.categories;
    }

    async.parallel([
        async.apply(exports.attachTar, doc, cfg, tfile),
        async.apply(exports.attachREADME, doc, cfg, readme)
    ],
    function (err) {
        callback(err, doc);
    });
};


/**
 * Creates a new package document for publishing a new package.
 *
 * @param {String} user - owner of the package
 * @param {Date} time - created time of the package
 * @param {Object} cfg - the config options from kanso.json
 * @param {String} tfile - the path to the .tar.gz archive for this package
 * @param {String} readme - path to the README file
 * @param {Function} callback
 */

exports.createDoc = function (user, time, cfg, tfile, readme, callback) {
    var doc = {
        _id: cfg.name,
        name: cfg.name,
        type: 'package',
        submitted_by: user,
        versions: {},
        time: {
            created: utils.ISODateString(time)
        }
    };
    exports.updateDoc(doc, time, cfg, tfile, readme, callback);
};


/**
 * Cache object used to store package documents from repositories when using the
 * get function.
 *
 * Stored as exports.CACHE[repository][package_name] = doc
 */

exports.CACHE = {};


/**
 * Get package document from repository. Returns null if package is missing,
 * will return cached copies if use_cache is set to true (default is false).
 *
 * @param {String} repository - the URL to the repository
 * @param {String} name - the name of the package to lookup
 * @param {Boolean} use_cache - whether to return cached results
 * @param {Function} callback
 */

exports.get = function (repository, name, /*optional*/use_cache, callback) {
    if (!callback) {
        callback = use_cache;
        use_cache = false;
    }
    if (!repository) {
        return callback(null, null);
    }
    if (!exports.CACHE[repository]) {
        exports.CACHE[repository] = {};
    }
    if (use_cache) {
        // use cached copy of document if available
        if (exports.CACHE[repository][name]) {
            return callback(null, exports.CACHE[repository][name]);
        }
    }
    var db = couchdb(repository);
    var id = couchdb.encode(name || '');
    db.client('GET', id, null, function (err, data, res) {
        res = res || {};
        if (res.statusCode !== 404 && err) {
            return callback(err);
        }
        exports.CACHE[repository][name] = data;
        callback(null, (res.statusCode === 200) ? data: null);
    });
};


/**
 * Finds a README file (with any or no extension) and returns its filename
 * if found or null if not.
 *
 * @param {String} dir - the package directory to search
 * @param {Function} callback
 */

exports.findREADME = function (dir, callback) {
    fs.readdir(dir, function (err, files) {
        if (err) {
            return callback(err);
        }
        // make sure the files are sorted so the same filename will match
        // first reliably in the same set
        files = files.sort();

        var readme = _.detect(files, function (f) {
            return /^README$|^README\..*$/.test(f);
        });
        var filename = path.join(dir, readme);
        if (readme) {
            fs.stat(filename, function (err, stat) {
                if (err) {
                    return callback(err);
                }
                return callback(null, stat.isFile() ? filename: null);
            });
        }
        else {
            return callback(null, null);
        }
    });
};


exports.publish = function (path, repository, /*optional*/options, callback) {
    if (!callback) {
        callback = options;
        options = {};
    }
    var time = options.server_time || new Date();
    var user = options.user;
    if (!user) {
        var parsed = url.parse(repository);
        if (parsed.auth) {
            user = parsed.auth.split(':')[0];
        }
    }
    settings.load(path, function (err, cfg) {
        if (err) {
            return callback(err);
        }
        async.series({
            get: async.apply(exports.get, repository, cfg.name),
            cache: async.apply(cache.update, cfg.name, cfg.version, path),
            readme: async.apply(exports.findREADME, path)
        },
        function (err, results) {
            if (err) {
                return callback(err);
            }
            var curr = results.get;
            var tfile = results.cache[0];
            var dir = results.cache[1];
            var readme = results.readme;

            var db = couchdb(repository);

            if (!curr) {
                return exports.createDoc(
                    user, time, cfg, tfile, readme, function (err, doc) {
                        db.save(cfg.name, doc, callback);
                    }
                );
            }
            else if (curr.versions && curr.versions[cfg.version]) {
                if (!options.force) {
                    return callback(
                        'Entry already exists for ' + cfg.name + ' ' +
                        cfg.version
                    );
                }
            }
            return exports.updateDoc(
                curr, time, cfg, tfile, readme, function (err, doc) {
                    db.save(cfg.name, doc, callback);
                }
            );
        });
    });
};

exports.unpublish = function (repository, name, version, options, callback) {
    if (!callback) {
        callback = options;
        options = {};
    }
    var db = couchdb(repository);

    function removeVersionFromCache(v) {
        return function (err) {
            if (err) {
                return callback(err);
            }
            cache.clear(name, v, callback);
        };
    }

    function removeVersion(v, data) {
        delete data.versions[v];
        delete data._attachments[name + '-' + v + '.tar.gz'];
        delete data.time[v];
        data.time.modified = utils.ISODateString(new Date());

        if (data.tags['latest'] === v) {
            var versions = Object.keys(data.versions).sort(semver.compare);
            data.tags['latest'] = versions[versions.length - 1];
        }

        if (Object.keys(data.versions).length) {
            db.save(name, data, removeVersionFromCache(v));
        }
        else {
            // no more version remaining
            db.delete(name, data._rev, removeVersionFromCache());
        }
    }
    exports.get(repository, name, function (err, data) {
        if (err) {
            return callback(err);
        }
        if (!data) {
            return callback('No entry exists for ' + name);
        }
        if (!version) {
            return db.delete(name, data._rev, removeVersionFromCache());
        }
        if (data.versions && data.versions[version]) {
            return removeVersion(version, data);
        }
        if (data.tags && data.tags[version]) {
            var v = data.tags[version];
            if (version !== 'latest') {
                delete data.tags[version];
            }
            return removeVersion(v, data);
        }
        return callback('No entry exists for ' + name + ' ' + version);
    });
};


/**
 * Find the maximum version of package by name which satisfies the provided
 * range. Returns the version, the package document and the repository URL
 * where the package is located. Repositories are checked in priority order
 * (earlier in the array is higher priority).
 *
 * @param {String} name - the name of the package to lookup
 * @param {String|Array} range - acceptable version range (or array of ranges)
 * @param {Array} repos - the repository URLs to check
 * @param {Function} callback
 */

exports.resolve = function (name, ranges, repos, callback) {
    if (!Array.isArray(ranges)) {
        ranges = [ranges];
    }
    if (!repos || !repos.length) {
        return callback(new Error('No repositories specified'));
    }
    exports.maxSatisfying(name, ranges, repos, function (err, m, vers) {
        if (err) {
            return callback(err);
        }
        if (!m) {
            if (vers && vers.length) {
                var e = new Error(
                    'No package for ' + name + ' @ ' + range + '\n' +
                    'Available versions: ' + vers.join(', ')
                );
                e.missing = true;
                e.versions = vers;
                return callback(e);
            }
            else {
                var e = new Error('No package for ' + name);
                e.missing = true;
                return callback(e);
            }
        }
        return callback(null, m.version, m.config, m.repository);
    });
};


// TODO
// the concurrency of fetch requests
//var concurrency = 5;
//var queue = async.queue(worker, concurrency);

//exports.buildTree = function (deps, repositories, callback) {
    // for each dep find available versions and get its deps
    // after all deps and versions, check all ranges to find max for each
//};


/**
 * Checks a list of repositories for all available versions of a package.
 * Returns an object keyed by version containing a 'repository' property
 * containing a repository url for each key, which is the highest-priority
 * repository with that version (in order of the original array, earlier is
 * higher priority), and a 'doc' property containing the package document from
 * that repository.
 *
 * @param {String} name - the name of the package to lookup
 * @param {Array} repositories - an array of repository URLs
 * @param {Function} callback
 */

exports.availableVersions = function (name, repositories, callback) {
    var versions = {};
    async.forEach(repositories, function (repo, cb) {
        exports.get(repo, name, true, function (err, doc) {
            if (err) {
                return cb(err);
            }
            if (doc && doc.versions) {
                for (var k in doc.versions) {
                    if (!versions[k]) {
                        versions[k] = {
                            repository: repo,
                            config: doc.versions[k],
                            source: 'repository'
                        };
                    }
                }
            }
            cb();
        });
    },
    function (err) {
        callback(err, versions);
    });
};


/**
 * Checks all repositories in the list for available versions, then tests
 * all versions against all ranges and returns the highest version which
 * satisfies all ranges (null if none satisfy all of them). The object
 * returned has two properties 'repository' which is the repository the version
 * was found on, and 'version' which is the actual max satisfying version.
 *
 * @param {String} name - the name of the package to look up
 * @param {Array} ranges - an array of version range requirements
 * @param {Array} repositories - an array of repository URLs
 * @param {Function} callback
 */

exports.maxSatisfying = function (name, ranges, repositories, callback) {
    // TODO: what to do about tags?
    exports.availableVersions(name, repositories, function (err, vers) {
        if (err) {
            return callback(err);
        }
        var max = versions.maxSatisfying(Object.keys(vers), ranges);
        if (!max) {
            return callback(null, null);
        }
        return callback(null, {
            source: vers[max].source,
            repository: vers[max].repository,
            config: vers[max].config,
            version: max
        }, vers);
    });
};


// data argument is an optional cached package document from a previous call to
// repository.resolve to avoid multiple db.get calls if possible

exports.fetch = function (name, version, repos, /*opt*/repository, callback) {

    logger.debug('fetching', name + ' (' + version + ')');

    if (!callback) {
        callback = repository;
        repository = null;
    }
    if (!callback) {
        callback = data;
        data = null;
    }

    cache.get(name, version, function (err, tarfile, cachedir) {
        if (cachedir && tarfile) {
            settings.load(cachedir, function (err, cfg) {
                if (err) {
                    return callback(err);
                }
                callback(null, tarfile, cachedir, version, cfg, true);
            });
            return;
        }

        exports.resolve(name, version, repos, function (err, v, cfg, repo) {
            if (err) {
                return callback(err);
            }
            cache.get(name, v, function (err, c_tfile, c_cdir) {
                if (err) {
                    return callback(err);
                }
                if (c_tfile && c_cdir) {
                    settings.load(cachedir, function (err, cfg) {
                        if (err) {
                            return callback(err);
                        }
                        callback(null, c_tfile, c_cdir, v, cfg, true);
                    });
                    return;
                }
                var filename = name + '-' + v + '.tar.gz';
                var url = repo + '/' + name + '/' + filename;
                logger.info('downloading', utils.noAuthURL(url));
                exports.download(url, function (err, tarfile) {
                    if (err) {
                        return callback(err);
                    }
                    cache.moveTar(name, v, tarfile,
                        function (err, tfile, cdir) {
                            callback(err, tfile, cdir, v, cfg);
                        }
                    );
                });
            });
        });

    });
};


exports.download = function (file, callback) {
    var target = exports.TMP_DIR + '/' + path.basename(file);
    var urlinfo = url.parse(file);
    var proto = (urlinfo.protocol === 'https:') ? https: http;

    var _cb = callback;
    callback = function (err) {
        var that = this;
        var args = arguments;
        if (err) {
            utils.rm('-rf', target, function (err) {
                if (err) {
                    // let the original error through, but still output this one
                    logger.error(err);
                }
                _cb.apply(that, args);
            });
            return;
        }
        _cb.apply(that, args);
    };

    utils.ensureDir(exports.TMP_DIR, function (err) {
        if (err) {
            return callback(err);
        }
        var headers = {};
        if (urlinfo.auth) {
            var enc = new Buffer(urlinfo.auth).toString('base64');
            headers.Authorization = "Basic " + enc;
        }
        var request = proto.request({
            host: urlinfo.hostname,
            port: urlinfo.port,
            method: 'GET',
            path: urlinfo.pathname,
            headers: headers
        });
        request.on('response', function (response) {
            if (response.statusCode >= 300) {
                return callback(couchdb.statusCodeError(response.statusCode));
            }
            var outfile = fs.createWriteStream(target);
            response.on('data', function (chunk) {
                outfile.write(chunk);
            });
            response.on('end', function () {
                outfile.end();
                callback(null, target);
            });
        });
        request.end();
    });
};
