use dwn_core::message::descriptor::RecordsWriteBuilder;
use tracing_test::traced_test;

use crate::utils::init_dwn;

#[tokio::test]
#[traced_test]
async fn test_valid_attestation() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    assert!(dwn.process_message(&actor.did, msg).await.is_ok());
}

#[tokio::test]
#[traced_test]
async fn test_invalid_payload() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    msg.attestation.as_mut().unwrap().payload = "abcdefghijklmnop".to_string();

    assert!(dwn.process_message(&actor.did, msg).await.is_err());
}

#[tokio::test]
#[traced_test]
async fn test_empty_signatures() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    msg.attestation.as_mut().unwrap().signatures.clear();

    assert!(dwn.process_message(&actor.did, msg).await.is_err());
}

#[tokio::test]
#[traced_test]
async fn test_invalid_signature() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    msg.attestation.as_mut().unwrap().signatures[0].signature = "abcdefghijklmnop".to_string();

    assert!(dwn.process_message(&actor.did, msg).await.is_err());
}

#[tokio::test]
#[traced_test]
async fn test_multiple_signatures() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    let sig = msg.attestation.as_mut().unwrap().signatures[0].clone();
    msg.attestation.as_mut().unwrap().signatures.push(sig);

    assert!(dwn.process_message(&actor.did, msg).await.is_ok());
}

#[tokio::test]
#[traced_test]
async fn test_multiple_signatures_invalid() {
    let (actor, _, dwn) = init_dwn();

    let mut msg = RecordsWriteBuilder::default().build().unwrap();
    actor.authorize(&mut msg).unwrap();
    actor.sign(&mut msg).unwrap();

    let mut sig = msg.attestation.as_mut().unwrap().signatures[0].clone();
    sig.signature = "abcdefghijklmnop".to_string();
    msg.attestation.as_mut().unwrap().signatures.push(sig);

    assert!(dwn.process_message(&actor.did, msg).await.is_err());
}
