# gunshi Plugin System: Renderer Decorator Pattern

## Overview

The gunshi plugin system uses the renderer decorator pattern to enable customization of CLI output (headers, usage, validation errors). This design allows plugins to wrap existing rendering functionality to modify or extend output.

## Design

### 1. Type Definitions

```typescript
// packages/gunshi/src/types.ts

/**
 * Renderer decorator type.
 * A function that wraps a base renderer to add or modify its behavior.
 * @param baseRenderer The base renderer function to decorate
 * @param ctx The command context
 * @returns The decorated result
 */
export type RendererDecorator<T> = (
  baseRenderer: (ctx: CommandContext) => Promise<T>,
  ctx: CommandContext
) => Promise<T>

/**
 * Validation errors renderer decorator type.
 * A function that wraps a validation errors renderer to add or modify its behavior.
 * @param baseRenderer The base validation errors renderer function to decorate
 * @param ctx The command context
 * @param error The aggregate error containing validation errors
 * @returns The decorated result
 */
export type ValidationErrorsDecorator = (
  baseRenderer: (ctx: CommandContext, error: AggregateError) => Promise<string>,
  ctx: CommandContext,
  error: AggregateError
) => Promise<string>
```

### 2. PluginContext API

```typescript
// packages/gunshi/src/plugin.ts

export class PluginContext {
  #globalOptions: Map<string, ArgSchema> = new Map()
  #decorators: RendererDecorators

  constructor(decorators: RendererDecorators) {
    this.#decorators = decorators
  }

  /**
   * Decorate the header renderer.
   * @param decorator - A decorator function that wraps the base header renderer.
   */
  decorateHeaderRenderer(decorator: RendererDecorator<string>): void {
    this.#decorators.addHeaderDecorator(decorator)
  }

  /**
   * Decorate the usage renderer.
   * @param decorator - A decorator function that wraps the base usage renderer.
   */
  decorateUsageRenderer(decorator: RendererDecorator<string>): void {
    this.#decorators.addUsageDecorator(decorator)
  }

  /**
   * Decorate the validation errors renderer.
   * @param decorator - A decorator function that wraps the base validation errors renderer.
   */
  decorateValidationErrorsRenderer(decorator: ValidationErrorsDecorator): void {
    this.#decorators.addValidationErrorsDecorator(decorator)
  }
}
```

### 3. RendererDecorators Class

```typescript
// packages/gunshi/src/decorators.ts

export class RendererDecorators {
  #headerDecorators: RendererDecorator<string>[] = []
  #usageDecorators: RendererDecorator<string>[] = []
  #validationDecorators: ValidationErrorsDecorator[] = []

  addHeaderDecorator(decorator: RendererDecorator<string>): void {
    this.#headerDecorators.push(decorator)
  }

  addUsageDecorator(decorator: RendererDecorator<string>): void {
    this.#usageDecorators.push(decorator)
  }

  addValidationErrorsDecorator(decorator: ValidationErrorsDecorator): void {
    this.#validationDecorators.push(decorator)
  }

  getHeaderRenderer(): (ctx: CommandContext) => Promise<string> {
    return this.#buildRenderer(this.#headerDecorators, defaultHeaderRenderer)
  }

  getUsageRenderer(): (ctx: CommandContext) => Promise<string> {
    return this.#buildRenderer(this.#usageDecorators, defaultUsageRenderer)
  }

  getValidationErrorsRenderer(): (ctx: CommandContext, error: AggregateError) => Promise<string> {
    return this.#buildValidationRenderer(this.#validationDecorators, defaultValidationRenderer)
  }

  // Build decorator chain
  #buildRenderer<T>(
    decorators: RendererDecorator<T>[],
    defaultRenderer: (ctx: CommandContext) => Promise<T>
  ): (ctx: CommandContext) => Promise<T> {
    return decorators.reduce(
      (baseRenderer, decorator) => (ctx: CommandContext) => decorator(baseRenderer, ctx),
      defaultRenderer
    )
  }
}
```

## Implementation Details

### 1. Default Renderers

```typescript
// Default base renderers (return empty strings)
const defaultHeaderRenderer = async (ctx: CommandContext): Promise<string> => ''
const defaultUsageRenderer = async (ctx: CommandContext): Promise<string> => ''
const defaultValidationRenderer = async (
  ctx: CommandContext,
  error: AggregateError
): Promise<string> => ''
```

### 2. Usage in CLI Execution Flow

```typescript
// packages/gunshi/src/cli.ts

async function showHeader<A extends Args>(
  ctx: CommandContext<A>,
  decorators: RendererDecorators
): Promise<string | undefined> {
  // Get decorator chain
  const renderer = ctx.env.renderHeader || decorators.getHeaderRenderer()
  const header = await renderer(ctx)

  if (header) {
    ctx.log(header)
    ctx.log() // Empty line
    return header
  }
}

async function showUsage<A extends Args>(
  ctx: CommandContext<A>,
  decorators: RendererDecorators
): Promise<string | undefined> {
  const renderer = ctx.env.renderUsage || decorators.getUsageRenderer()
  const usage = await renderer(ctx)

  if (usage) {
    ctx.log(usage)
    return usage
  }
}

async function showValidationErrors<A extends Args>(
  ctx: CommandContext<A>,
  error: AggregateError,
  decorators: RendererDecorators
): Promise<void> {
  const renderer = ctx.env.renderValidationErrors || decorators.getValidationErrorsRenderer()
  ctx.log(await renderer(ctx, error))
}
```

## Usage Examples

### 1. Default Renderer Plugin

```typescript
// packages/gunshi/src/plugins/renderer.ts

export default function defaultRenderer(ctx: PluginContext) {
  // Header renderer
  ctx.decorateHeaderRenderer(async (_baseRenderer, cmdCtx) => {
    const { name, description, version } = cmdCtx.env
    const parts: string[] = []

    if (name) {
      parts.push(name)
      if (version) parts.push(`v${version}`)
    }

    if (description) {
      if (parts.length > 0) parts.push('-')
      parts.push(description)
    }

    return parts.join(' ')
  })

  // Usage renderer
  ctx.decorateUsageRenderer(async (_baseRenderer, cmdCtx) => {
    const sections: string[] = []

    // Usage section
    sections.push(renderUsageSection(cmdCtx))

    // Description section
    if (cmdCtx.description) {
      sections.push(renderDescriptionSection(cmdCtx))
    }

    // Options section
    if (Object.keys(cmdCtx.args).length > 0) {
      sections.push(renderOptionsSection(cmdCtx))
    }

    // Examples section
    if (cmdCtx.examples) {
      sections.push(renderExamplesSection(cmdCtx))
    }

    // Commands section
    if (cmdCtx.env.subCommands?.size > 0) {
      sections.push(renderCommandsSection(cmdCtx))
    }

    return sections.filter(Boolean).join('\n\n')
  })

  // Validation errors renderer
  ctx.decorateValidationErrorsRenderer(async (_baseRenderer, cmdCtx, error) => {
    const lines: string[] = ['Validation errors:']

    for (const err of error.errors) {
      if (err instanceof Error) {
        lines.push(`  - ${err.message}`)
      }
    }

    lines.push('')
    lines.push('Run with --help for usage information.')

    return lines.join('\n')
  })
}
```

### 2. Color Output Decorator

```typescript
const colorPlugin: Plugin = ctx => {
  // Add color to header
  ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
    const result = await baseRenderer(cmdCtx)
    return chalk.bold.blue(result)
  })

  // Display errors in red
  ctx.decorateValidationErrorsRenderer(async (baseRenderer, cmdCtx, error) => {
    const result = await baseRenderer(cmdCtx, error)
    return chalk.red(result)
  })

  // Emphasize section headers in usage
  ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
    const result = await baseRenderer(cmdCtx)
    return result.replace(/^(Usage|Description|Options|Examples|Commands):/gm, match =>
      chalk.bold.underline(match)
    )
  })
}
```

### 3. Internationalization Decorator

```typescript
const i18nPlugin: Plugin = ctx => {
  ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
    // Get base rendering result
    const defaultOutput = await baseRenderer(cmdCtx)

    // Translate based on locale
    const locale = cmdCtx.locale.toString()
    if (locale === 'ja-JP') {
      return defaultOutput
        .replace(/Usage:/g, '使い方:')
        .replace(/Description:/g, '説明:')
        .replace(/Options:/g, 'オプション:')
        .replace(/Examples:/g, '例:')
        .replace(/Commands:/g, 'コマンド:')
    }

    return defaultOutput
  })
}
```

### 4. Conditional Rendering

```typescript
const conditionalPlugin: Plugin = ctx => {
  // Show examples only in detailed mode
  ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
    const result = await baseRenderer(cmdCtx)

    // Remove examples if --verbose option is not present
    if (!cmdCtx.values.verbose) {
      const lines = result.split('\n')
      const examplesIndex = lines.findIndex(line => line.startsWith('Examples:'))
      if (examplesIndex !== -1) {
        const nextSectionIndex = lines.findIndex(
          (line, i) => i > examplesIndex && line.match(/^[A-Z]\w+:/)
        )
        lines.splice(examplesIndex, nextSectionIndex - examplesIndex)
      }
      return lines.join('\n')
    }

    return result
  })
}
```

### 5. Composing Multiple Decorators

```typescript
const compositePlugin: Plugin = ctx => {
  // Base renderer
  ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
    return `${cmdCtx.env.name} v${cmdCtx.env.version}`
  })

  // Add prefix
  ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
    const result = await baseRenderer(cmdCtx)
    return `🚀 ${result}`
  })

  // Add border
  ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
    const result = await baseRenderer(cmdCtx)
    const border = '='.repeat(result.length + 4)
    return `${border}\n| ${result} |\n${border}`
  })
}

// Execution result:
// ================================
// | 🚀 myapp v1.0.0 |
// ================================
```

## Key Features

### 1. Decorator Chain Construction

- Decorators are applied in registration order
- Each decorator wraps the result of the previous one
- First decorator wraps the base renderer

### 2. Flexible Output Control

- Add, modify, or remove output
- Dynamic output based on conditions
- Combination of multiple decorators

### 3. Separation of Concerns

- Each decorator has a single responsibility
- Separation of basic rendering logic and customization
- Independence between plugins

### 4. Asynchronous Support

- All renderers are async functions
- Access to external resources possible (APIs, files, etc.)

## Implementation Considerations

### 1. Decorator Execution Order

```typescript
// Decorators are applied in registration order
ctx.decorateHeaderRenderer(decoratorA) // Inner
ctx.decorateHeaderRenderer(decoratorB) // Outer

// Execution order: decoratorB(decoratorA(baseRenderer))
```

### 2. Calling Base Renderer

```typescript
// Always call base renderer
ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
  const result = await baseRenderer(cmdCtx) // Important!
  return modify(result)
})

// If base renderer is not called, previous decorator results are lost
ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
  // ❌ Not calling baseRenderer
  return 'Fixed output'
})
```

### 3. Error Handling

```typescript
ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
  try {
    const result = await baseRenderer(cmdCtx)
    return await enhanceWithExternalData(result)
  } catch (error) {
    // Fallback on error
    console.warn('Enhancement failed:', error)
    return baseRenderer(cmdCtx)
  }
})
```

### 4. Performance Considerations

```typescript
// Cache heavy processing
const cache = new Map<string, string>()

ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
  const cacheKey = `${cmdCtx.name}-${cmdCtx.locale}`

  if (cache.has(cacheKey)) {
    return cache.get(cacheKey)!
  }

  const result = await baseRenderer(cmdCtx)
  const enhanced = await expensiveOperation(result)
  cache.set(cacheKey, enhanced)

  return enhanced
})
```

## Relationship with Default Renderer

### 1. Default Renderer Plugin

- Basic rendering implementation provided by gunshi
- Registered as the first decorator
- Other plugins extend this base

### 2. Customization Hierarchy

```
Empty base renderer
  ↓
Default renderer plugin (basic implementation)
  ↓
User plugin 1 (add colors, etc.)
  ↓
User plugin 2 (format changes, etc.)
  ↓
Final output
```

## Comparison with Command Decorators

| Aspect           | Renderer Decorator           | Command Decorator                |
| ---------------- | ---------------------------- | -------------------------------- |
| Target           | Output formatting            | Command execution                |
| Return Value     | String (display content)     | void or string                   |
| Execution Timing | When showing help, on errors | During command execution         |
| Main Use         | UI/UX customization          | Execution control, auth, logging |
| Base Function    | Renderer function            | Command runner                   |

## Summary

The renderer decorator pattern provides:

1. **Flexible output customization** - Free modification of headers, usage, and error messages
2. **Plugin cooperation** - Multiple decorators work together to generate final output
3. **Separation of concerns** - Clear separation and reuse of rendering logic
4. **Extensibility** - Easy to add new output formats and styles
5. **Consistent API** - Same decorator pattern for all renderers

This design makes gunshi's CLI output highly customizable and supports various use cases (internationalization, accessibility, corporate branding, etc.).
