# gunshi Plugin System: Command Execution Hooks with Decorator Pattern

## Overview

This document introduces command execution hooks using the decorator pattern in the gunshi plugin system. This design allows plugins to wrap command execution and enables pre/post processing and conditional behavior modification.

## Design

### 1. Type Definition Extensions

```typescript
// packages/gunshi/src/types.ts

/**
 * Command runner.
 * @returns void or string (for CLI output)
 */
export type CommandRunner<A extends Args = Args> = (
  ctx: Readonly<CommandContext<A>>
) => Awaitable<void | string>

/**
 * Command decorator.
 * A function that wraps a command runner to add or modify its behavior.
 */
export type CommandDecorator<A extends Args = Args> = (
  baseRunner: CommandRunner<A>
) => CommandRunner<A>
```

### 2. CommandContext Extension

```typescript
// packages/gunshi/src/types.ts

export interface CommandContext<A extends Args = Args, V = ArgValues<A>> {
  // ... existing properties ...

  /**
   * User-defined data storage for plugins and custom extensions.
   * Plugins can store arbitrary data here to share between decorators.
   */
  data: Record<string, unknown>
}
```

### 3. PluginContext Extension

```typescript
// packages/gunshi/src/plugin.ts

export class PluginContext {
  #globalOptions: Map<string, ArgSchema> = new Map()
  #decorators: RendererDecorators
  #commandDecorators: CommandDecorator[] = []

  constructor(decorators: RendererDecorators) {
    this.#decorators = decorators
  }

  // ... existing methods ...

  /**
   * Get the command decorators
   * @returns The command decorators
   */
  get commandDecorators(): readonly CommandDecorator[] {
    return [...this.#commandDecorators]
  }

  /**
   * Decorate the command execution.
   * Decorators are applied in reverse order (last registered is executed first).
   * @param decorator - A decorator function that wraps the command runner
   */
  decorateCommand(decorator: CommandDecorator): void {
    this.#commandDecorators.push(decorator)
  }
}
```

## Implementation Details

### 1. Context Creation Update

> [!IMPORTANT]
> This feature is not still implemented

```typescript
// packages/gunshi/src/context.ts

export async function createCommandContext<A extends Args = Args>(
  options: CommandContextOptions<A>
): Promise<CommandContext<A>> {
  // ... existing code ...

  return {
    // ... existing properties ...
    data: {} // Initialize new data storage
  }
}
```

### 2. CLI Execution Flow Update

```typescript
// packages/gunshi/src/cli.ts

async function executeCommand<A extends Args = Args>(
  cmd: Command<A> | LazyCommand<A>,
  ctx: CommandContext<A>,
  name: string,
  pluginContext: PluginContext
): Promise<string | undefined> {
  const resolved = isLazyCommand<A>(cmd) ? await resolveLazyCommand<A>(cmd, name, true) : cmd
  const baseRunner = resolved.run || (() => {})

  // Apply plugin decorators
  const decorators = pluginContext.commandDecorators
  const decoratedRunner = decorators.reduceRight(
    (runner, decorator) => decorator(runner),
    baseRunner
  )

  // Execute and return result
  const result = await decoratedRunner(ctx)

  // Return string if one was returned
  return typeof result === 'string' ? result : undefined
}

// Update cli function as well
export async function cli<A extends Args = Args>(
  argv: string[],
  entry: Command<A> | CommandRunner<A> | LazyCommand<A>,
  options: CliOptions<A> = {}
): Promise<string | undefined> {
  // ... existing processing ...

  // Remove special option processing (moved to decorators)
  // if (values.version) { ... }  // Remove
  // if (values.help) { ... }     // Remove

  // Return executeCommand result
  return await executeCommand(command, commandContext, name || '', pluginContext)
}
```

## Usage Examples

### 1. Help Plugin

```typescript
const helpPlugin: Plugin = ctx => {
  // Add global option
  ctx.addGlobalOption('help', {
    type: 'boolean',
    description: 'Show help'
  })

  // Apply help decorator
  ctx.decorateCommand(baseRunner => async context => {
    // Check for --help option or help subcommand
    if (context.values.help || context.positionals[0] === 'help') {
      const buffer: string[] = []

      // Header
      const header = await context.env.renderHeader?.(context)
      if (header) buffer.push(header, '')

      // Usage
      const usage = await context.env.renderUsage?.(context)
      if (usage) buffer.push(usage)

      const output = buffer.join('\n')

      if (!context.env.usageSilent) {
        context.log(output)
        return
      }

      return output
    }

    // Normal command execution
    return baseRunner(context)
  })
}
```

### 2. Timing Plugin

```typescript
const timingPlugin: Plugin = ctx => {
  ctx.decorateCommand(baseRunner => async context => {
    const start = Date.now()
    context.data.startTime = start

    try {
      const result = await baseRunner(context)
      const duration = Date.now() - start
      console.log(`⏱️  Command completed in ${duration}ms`)
      return result
    } catch (error) {
      const duration = Date.now() - start
      console.log(`❌ Command failed after ${duration}ms`)
      throw error
    }
  })
}
```

### 3. Authentication Plugin

```typescript
const authPlugin: Plugin = ctx => {
  ctx.decorateCommand(baseRunner => async context => {
    // Skip specific commands
    if (context.name === 'login' || context.name === 'help') {
      return baseRunner(context)
    }

    const token = process.env.AUTH_TOKEN
    if (!token) {
      throw new Error('Authentication required. Please run "login" first.')
    }

    // Add authentication info to context
    context.data.auth = { token, userId: await validateToken(token) }

    return baseRunner(context)
  })
}
```

### 4. Dry-run Plugin

```typescript
const dryRunPlugin: Plugin = ctx => {
  ctx.addGlobalOption('dryRun', {
    type: 'boolean',
    description: 'Run command in dry-run mode'
  })

  ctx.decorateCommand(baseRunner => async context => {
    if (context.values.dryRun) {
      console.log('🔍 Running in dry-run mode')
      context.data.isDryRun = true

      // Don't execute actual command in dry-run mode
      const preview = `Would execute: ${context.name || 'command'}`
      console.log(preview)

      return context.env.usageSilent ? preview : undefined
    }

    return baseRunner(context)
  })
}
```

### 5. Version Plugin

```typescript
const versionPlugin: Plugin = ctx => {
  ctx.addGlobalOption('version', {
    type: 'boolean',
    description: 'Show version'
  })

  ctx.decorateCommand(baseRunner => async context => {
    if (context.values.version) {
      const version = context.env.version || 'unknown'

      if (!context.env.usageSilent) {
        context.log(version)
        return
      }

      return version
    }

    return baseRunner(context)
  })
}
```

### 6. Logging Plugin

```typescript
const loggingPlugin: Plugin = ctx => {
  ctx.decorateCommand(baseRunner => async context => {
    console.log(`[LOG] Command: ${context.name}`)
    console.log(`[LOG] Arguments:`, context.values)

    try {
      const result = await baseRunner(context)
      console.log(`[LOG] Command completed successfully`)
      return result
    } catch (error) {
      console.error(`[LOG] Command failed:`, error)
      throw error
    }
  })
}
```

## Key Features

### 1. Clear Design Through Function Composition

- Each decorator has a single responsibility
- Complex behaviors achieved through decorator combination
- Implementable as pure functions

### 2. Consistency with Existing Patterns

- Same pattern as renderer decorators already used in gunshi
- Low learning cost
- Unified API design

### 3. Flexible Execution Control

- Execute arbitrary processing before/after command execution
- Skip command execution based on conditions
- Control CLI function return values

### 4. Data Sharing Mechanism

> [!IMPORTANT]
> This feature is not still implemented

- Data sharing between decorators via `context.data`
- Conflict avoidance using namespaces
- Type safety ensured by users

## Implementation Considerations

### 1. Decorator Execution Order

```typescript
// Later registered decorators execute first
ctx.decorateCommand(authDecorator) // Executes second
ctx.decorateCommand(loggingDecorator) // Executes first

// Execution order: loggingDecorator → authDecorator → baseRunner
```

### 2. Error Handling

```typescript
ctx.decorateCommand(baseRunner => async context => {
  try {
    return await baseRunner(context)
  } catch (error) {
    // Handle or re-throw error
    if (error instanceof ValidationError) {
      console.error('Validation failed:', error.message)
      // Return custom error message
      return 'Validation failed. Please check your input.'
    }
    throw error
  }
})
```

### 3. Short-circuit Evaluation

```typescript
// When help is displayed, subsequent decorators are not executed
ctx.decorateCommand(baseRunner => async context => {
  if (context.values.help) {
    return showHelp(context) // Exit here
  }
  // baseRunner is not called
})
```

### 4. Data Namespacing

> [!IMPORTANT]
> This feature is not still implemented

```typescript
// Avoid data conflicts between plugins
ctx.decorateCommand(baseRunner => async context => {
  // Use plugin-specific namespace
  context.data['myPlugin'] = {
    startTime: Date.now(),
    config: loadConfig()
  }

  return baseRunner(context)
})
```

## Comparison with Before/After Pattern

| Aspect                    | Decorator | Before/After | Evaluation                           |
| ------------------------- | --------- | ------------ | ------------------------------------ |
| Implementation Complexity | ★★★       | ★★           | Before/After is simpler              |
| Flexibility               | ★★★★★     | ★★★          | Decorator is more flexible           |
| Testability               | ★★★★★     | ★★★          | Decorator is easier to test          |
| Consistency with Existing | ★★★★★     | ★★★          | gunshi already uses decorators       |
| Return Value Control      | ★★★★★     | ★            | Decorator is superior                |
| Execution Flow Control    | ★★★★★     | ★★           | Decorator is more powerful           |
| Learning Cost             | ★★★       | ★★★★         | Before/After is easier to understand |

## Implementation Example: Issue #69 Support

```typescript
// Support both "mycli help" and "mycli --help"
const comprehensiveHelpPlugin: Plugin = ctx => {
  ctx.addGlobalOption('help', {
    type: 'boolean',
    description: 'Show help'
  })

  ctx.decorateCommand(baseRunner => async context => {
    const shouldShowHelp =
      context.values.help || // --help option
      context.positionals[0] === 'help' // help subcommand

    if (shouldShowHelp) {
      // Handle "help subcommand" case
      if (context.positionals[0] === 'help' && context.positionals[1]) {
        const subCommandName = context.positionals[1]
        const subCommand = context.env.subCommands?.get(subCommandName)

        if (subCommand) {
          // Display subcommand help
          return showSubCommandHelp(context, subCommand)
        }
      }

      // Display main help
      return showMainHelp(context)
    }

    return baseRunner(context)
  })
}
```

## Summary

Command execution hooks with the decorator pattern provide:

1. **Consistency with gunshi's existing design** - Same pattern as renderer decorators
2. **Flexible execution control** - Pre/post processing, conditional branching, short-circuit evaluation
3. **Special option pluginization** - `--help`, `--version`, etc.
4. **Clear separation of responsibilities through function composition** - Each decorator has a single function
5. **Easy testing** - Each decorator can be tested independently

This design makes gunshi's plugin system more powerful and extensible. The decorator pattern is the optimal choice for use cases requiring command execution interception and modification.
