# CommandContext Extension via Composition API

## Overview

This design enables type-safe extension of CommandContext through plugins while minimizing changes to the existing gunshi API.

## Core Design

### 1. Plugin Type Extension

```typescript
// packages/gunshi/src/plugin.ts

// Extend existing Plugin type (function + optional properties)
export type Plugin = ((ctx: PluginContext) => Awaitable<void>) & {
  name?: string
  extension?: ContextExtension<any>
}

// Context extension type definition
export interface ContextExtension<T = any> {
  readonly key: symbol
  readonly factory: (core: CommandContextCore) => T
}

// Plugin definition options
export interface PluginOptions<T = any> {
  name: string
  setup: (ctx: PluginContext) => Awaitable<void>
  extension?: (core: CommandContextCore) => T
}

// Plugin creation function
export function plugin<T = any>(options: PluginOptions<T>): Plugin {
  const { name, setup, extension } = options

  const pluginFn = setup as Plugin
  pluginFn.name = name

  if (extension) {
    pluginFn.extension = {
      key: Symbol(name),
      factory: extension
    }
  }

  return pluginFn
}
```

### 2. Command Definition Extension

```typescript
// packages/gunshi/src/definition.ts

// Extended Command type
export interface ExtendedCommand<
  A extends Args = Args,
  E extends Record<string, ContextExtension> = {}
> extends Command<A> {
  _extensions?: E
  run?: (
    ctx: CommandContext<A> & {
      ext: { [K in keyof E]: ReturnType<E[K]['factory']> }
    }
  ) => Awaitable<void | string>
}

// define overloads
export function define<A extends Args = Args>(definition: Command<A>): Command<A>

export function define<A extends Args = Args, E extends Record<string, ContextExtension> = {}>(
  definition: ExtendedCommand<A, E> & {
    extensions?: E
  }
): ExtendedCommand<A, E>

export function define<A extends Args = Args>(definition: any): any {
  if (definition.extensions) {
    const { extensions, ...rest } = definition
    return { ...rest, _extensions: extensions }
  }
  return definition
}

// lazy overloads
export function lazy<A extends Args = Args>(
  loader: CommandLoader<A>,
  definition?: Command<A>
): LazyCommand<A>

export function lazy<A extends Args = Args, E extends Record<string, ContextExtension> = {}>(
  loader: CommandLoader<A>,
  definition?: ExtendedCommand<A, E> & { extensions?: E }
): LazyCommand<A> & { _extensions?: E }

export function lazy<A extends Args = Args>(loader: CommandLoader<A>, definition?: any): any {
  const lazyCommand = loader as LazyCommand<A>

  if (definition != null) {
    // Copy existing properties
    lazyCommand.commandName = definition.name
    lazyCommand.description = definition.description
    lazyCommand.args = definition.args
    lazyCommand.examples = definition.examples
    lazyCommand.resource = definition.resource
    lazyCommand.toKebab = definition.toKebab

    // Handle extensions
    if (definition.extensions) {
      ;(lazyCommand as any)._extensions = definition.extensions
    } else if (definition._extensions) {
      ;(lazyCommand as any)._extensions = definition._extensions
    }
  }

  return lazyCommand
}
```

### 3. CommandContext Creation Update

```typescript
// packages/gunshi/src/context.ts

// CommandContextCore type (base type without extensions)
export type CommandContextCore<A extends Args = Args, V = ArgValues<A>> = Readonly<
  CommandContext<A, V>
>

export async function createCommandContext<
  A extends Args = Args,
  V extends ArgValues<A> = ArgValues<A>
>(params: CommandContextParams<A, V>): Promise<CommandContext<A, V>> {
  // Existing context creation logic
  const core = await createCoreContext(params)

  // If command requires extensions
  const command = params.command
  if ('_extensions' in command && command._extensions) {
    const ext = {} as any

    // Apply each extension
    for (const [key, extension] of Object.entries(command._extensions)) {
      ext[key] = extension.factory(core)
    }

    // Return context with extensions
    return Object.freeze({
      ...core,
      ext: Object.freeze(ext)
    }) as any
  }

  // Without extensions (backward compatibility)
  return core as any
}
```

## Usage Examples

### Plugin Definition

```typescript
// Authentication plugin
const authPlugin = plugin({
  name: 'auth',

  setup(ctx) {
    ctx.addGlobalOption('token', {
      type: 'string',
      description: 'Authentication token'
    })

    ctx.decorateCommand(baseRunner => async context => {
      if (!context.values.token) {
        throw new Error('Authentication required')
      }
      return baseRunner(context)
    })
  },

  extension(core) {
    return {
      get user() {
        return decodeToken(core.values.token)
      },
      get isAuthenticated() {
        return !!core.values.token
      },
      async logout() {
        await revokeToken(core.values.token)
      }
    }
  }
})

// Database plugin
const dbPlugin = plugin({
  name: 'db',

  setup(ctx) {
    ctx.addGlobalOption('db-url', {
      type: 'string',
      default: 'sqlite://./data.db'
    })
  },

  extension(core) {
    const connection = createConnection(core.values['db-url'])

    return {
      query: (sql: string) => connection.query(sql),
      transaction: (fn: () => Promise<void>) => connection.transaction(fn),
      close: () => connection.close()
    }
  }
})

// Simple logging plugin (no extension)
const loggingPlugin = plugin({
  name: 'logging',

  setup(ctx) {
    ctx.decorateCommand(baseRunner => async context => {
      console.log(`[${new Date().toISOString()}] Command: ${context.name}`)
      try {
        const result = await baseRunner(context)
        console.log(`[${new Date().toISOString()}] Success`)
        return result
      } catch (error) {
        console.error(`[${new Date().toISOString()}] Error:`, error)
        throw error
      }
    })
  }
})
```

### Command Definition

```typescript
// Command using extensions
const deployCommand = define({
  name: 'deploy',
  description: 'Deploy the application',

  args: {
    environment: {
      type: 'string',
      required: true,
      description: 'Target environment'
    },
    version: {
      type: 'string',
      description: 'Version to deploy'
    }
  },

  // Declare required extensions
  extensions: {
    auth: authPlugin.extension!,
    db: dbPlugin.extension!
  },

  async run(ctx) {
    // ctx.ext.auth and ctx.ext.db are type-safe
    ctx.log(`Deploying as ${ctx.ext.auth.user.name}`)

    // Record in database
    await ctx.ext.db.query(
      'INSERT INTO deployments (user, environment, version, timestamp) VALUES (?, ?, ?, ?)',
      [ctx.ext.auth.user.id, ctx.values.environment, ctx.values.version || 'latest', Date.now()]
    )

    // Perform deployment
    await performDeployment(ctx.values.environment, ctx.values.version)

    return `Successfully deployed to ${ctx.values.environment}`
  }
})

// Using with lazy loading
const lazyDeployCommand = lazy(() => import('./commands/deploy.js'), {
  name: 'deploy',
  description: 'Deploy the application',
  extensions: {
    auth: authPlugin.extension!,
    db: dbPlugin.extension!
  }
})

// Traditional command (backward compatibility)
const simpleCommand = define({
  name: 'hello',
  args: {
    name: { type: 'string' }
  },
  run(ctx) {
    return `Hello, ${ctx.values.name || 'World'}!`
  }
})
```

### CLI Integration

```typescript
// packages/gunshi/src/cli.ts

export async function cli<A extends Args = Args>(
  argv: string[],
  entry: Command<A> | CommandRunner<A> | LazyCommand<A>,
  options: CliOptions<A> = {}
): Promise<string | undefined> {
  // ... existing processing ...

  const decorators = new Decorators()
  const pluginContext = new PluginContext(decorators)

  // Apply plugins
  for (const plugin of plugins) {
    await plugin(pluginContext)

    // If plugin has extension, register it automatically
    if (plugin.extension && plugin.name) {
      pluginContext.registerExtension(plugin.name, plugin.extension)
    }
  }

  // ... command resolution etc ...

  // Create context (extensions are applied automatically)
  const commandContext = await createCommandContext({
    args,
    values,
    positionals,
    rest,
    argv,
    tokens,
    omitted,
    callMode,
    command,
    cliOptions: cliOptions
  })

  // Execute command
  return await executeCommand(command, commandContext, name || '', decorators.commandDecorators)
}
```

## Design Features

### 1. **Minimal Changes**

- Does not break existing API
- New features are completely optional
- Allows gradual migration

### 2. **Type Safety**

- Extension existence is guaranteed at compile time
- IDE completion works perfectly
- Prevents runtime errors

### 3. **Simplicity**

- Easy to understand concepts
- Natural JavaScript patterns
- Minimal new APIs

### 4. **Extensibility**

- Type composition between plugins is possible
- Easy to add future features
- Flexible usage patterns

## Implementation Impact

### Required File Changes

#### 1. **plugin.ts**

- Add optional properties to `Plugin` type
- Add `plugin()` function
- Add `ContextExtension` interface
- Add `PluginOptions` interface

#### 2. **definition.ts**

- Add `ExtendedCommand` interface
- Add overloads for `define()` and `lazy()`
- Convert `extensions` to `_extensions` internally

#### 3. **context.ts**

- Add extension application logic
- Add `CommandContextCore` type
- Process extensions in `createCommandContext`

#### 4. **types.ts**

- Adjust existing types as needed
- Maintain backward compatibility

### Type Relationship Diagram

```
Plugin (function type)
  ├── name?: string
  └── extension?: ContextExtension<T>
          ├── key: symbol
          └── factory: (core) => T

Command
  ├── existing properties
  └── _extensions?: Record<string, ContextExtension>

CommandContext
  ├── existing properties
  └── ext?: { [key]: T }  // when extensions are applied
```

## Migration Guide

### Impact on Existing Code

1. **Existing plugins**: No changes needed, work as-is
2. **Existing commands**: No changes needed, work as-is
3. **Adopting new features**: Add `extensions` as needed

### Gradual Migration Example

```typescript
// Step 1: Existing command (no changes)
const oldCommand = define({
  name: 'old',
  run(ctx) {
    /* existing code */
  }
})

// Step 2: Use new extensions
const newCommand = define({
  name: 'new',
  extensions: { auth: authPlugin.extension! },
  run(ctx) {
    // ctx.ext.auth is available
  }
})

// Step 3: Gradually add extensions
const enhancedCommand = define({
  name: 'enhanced',
  extensions: {
    auth: authPlugin.extension!,
    db: dbPlugin.extension!,
    cache: cachePlugin.extension!
  },
  run(ctx) {
    // Utilize multiple extensions
  }
})
```

## Summary

This final design enables:

1. **Type-safe extensions like Hono and H3**
2. **Complete backward compatibility** with existing code
3. **Intuitive API** for plugin authors
4. **Maximum functionality** with minimal implementation cost

gunshi will be able to provide type-safe extension capabilities through plugins as a modern CLI framework. This design balances simplicity and power while providing an excellent developer experience.
