# i18n Plugin Dependency Management System Design

## Option 1: Explicit Plugin Dependency Management - Detailed Design

### 1. Plugin System Extension

#### 1.1 Extended Plugin Definition

```typescript
// Add to packages/gunshi/src/types.ts
export interface PluginDependency {
  name: string
  optional?: boolean // Optional dependency
  version?: string // For future version management
}

// Extend PluginOptions in packages/gunshi/src/plugin.ts
export interface PluginOptions<T, G> {
  name: string
  dependencies?: PluginDependency[] | string[] // List of dependent plugins
  setup?: PluginFunction<G>
  extension?: PluginExtension<T, G>
}
```

#### 1.2 Extended PluginContext

```typescript
// Add to PluginContext class in packages/gunshi/src/plugin.ts
export class PluginContext<G extends GunshiParams = DefaultGunshiParams> {
  #extensions: Map<string, unknown> = new Map() // Store registered extensions

  /**
   * Get extension from another plugin
   * @param pluginName Plugin name
   * @returns Plugin extension, undefined if not found
   */
  getExtension<T>(pluginName: string): T | undefined {
    return this.#extensions.get(pluginName) as T | undefined
  }

  /**
   * Check if dependent plugin exists
   * @param pluginName Plugin name
   * @returns true if exists
   */
  hasExtension(pluginName: string): boolean {
    return this.#extensions.has(pluginName)
  }

  /**
   * Register plugin extension (internal use)
   * @internal
   */
  registerExtension(pluginName: string, extension: unknown): void {
    this.#extensions.set(pluginName, extension)
  }
}
```

### 2. i18n Plugin Implementation

```typescript
// packages/gunshi/src/plugins/i18n.ts
export interface I18nExtension {
  locale: Intl.Locale
  translate: <T extends string = string>(key: T, values?: Record<string, unknown>) => string
  // Built-in resource management
  getBuiltinResource(key: string): string | undefined
  setResource(locale: string, resource: Record<string, string>): void
}

export default function i18n(options: I18nPluginOptions = {}) {
  return plugin({
    name: 'i18n',
    extension: async (ctx: CommandContextCore, cmd: Command): Promise<I18nExtension> => {
      const locale = resolveLocale(options.locale)
      const adapter = createAdapter(options)

      // Load built-in resources
      const builtinResources = await loadBuiltinResources(locale)

      // Load command resources
      const commandResources = await loadCommandResources(ctx, cmd)

      // Implement translate function
      function translate(key: string, values?: Record<string, unknown>): string {
        // Handle built-in keys
        if (key.startsWith('_')) {
          return builtinResources.get(key) || key
        }
        // Handle regular keys
        return adapter.translate(locale.toString(), key, values) || ''
      }

      return {
        locale,
        translate,
        getBuiltinResource: key => builtinResources.get(key),
        setResource: (locale, resource) => adapter.setResource(locale, resource)
      }
    }
  })
}
```

### 3. Renderer Plugin Implementation

```typescript
// packages/gunshi/src/plugins/renderer.ts
export default function renderer() {
  return plugin({
    name: 'default-renderer',
    dependencies: ['i18n'], // Depends on i18n plugin

    setup: ctx => {
      // Header renderer
      ctx.decorateHeaderRenderer(async (_baseRenderer, cmdCtx) => {
        const i18n = ctx.getExtension<I18nExtension>('i18n')
        return await renderHeader(cmdCtx, i18n)
      })

      // Usage renderer
      ctx.decorateUsageRenderer(async (_baseRenderer, cmdCtx) => {
        const i18n = ctx.getExtension<I18nExtension>('i18n')
        return await renderUsage(cmdCtx, i18n)
      })

      // Validation errors renderer
      ctx.decorateValidationErrorsRenderer(async (_baseRenderer, cmdCtx, error) => {
        const i18n = ctx.getExtension<I18nExtension>('i18n')
        return await renderValidationErrors(cmdCtx, error, i18n)
      })
    }
  })
}

// Updated rendering functions
async function renderUsage<G extends GunshiParams>(
  ctx: Readonly<CommandContext<G>>,
  i18n?: I18nExtension
): Promise<string> {
  // Get translate function with fallback
  const translate = i18n?.translate || ((key: string) => key)

  // Maintain existing logic while replacing ctx.translate with translate
  const messages: string[] = []
  messages.push(`${translate('_USAGE')}:`)
  // ...
}
```

### 4. Plugin Loader Implementation

```typescript
// Usage example in packages/gunshi/src/cli.ts
export async function cli(options: CliOptions) {
  const plugins = options.plugins || []

  // 1. Resolve dependencies
  const sortedPlugins = resolveDependencies(plugins)

  // 2. Load plugins and register extensions
  const extensions: Record<string, CommandContextExtension> = {}
  const pluginContext = new PluginContext(decorators)

  for (const plugin of sortedPlugins) {
    // Execute plugin setup
    await plugin(pluginContext)

    // Register extension if available
    if (plugin.extension) {
      extensions[plugin.name] = plugin.extension
      // Also register in PluginContext for other plugins to reference
      const ext = await plugin.extension.factory(coreContext, command)
      pluginContext.registerExtension(plugin.name, ext)
    }
  }

  // 3. Pass extensions when creating CommandContext
  const ctx = await createCommandContext({
    // ...
    extensions
    // ...
  })
}

// Dependency resolution utility
function resolveDependencies(plugins: Plugin[]): Plugin[] {
  // Resolve dependencies with topological sort
  const sorted: Plugin[] = []
  const visited = new Set<string>()
  const visiting = new Set<string>()

  function visit(plugin: Plugin) {
    if (visited.has(plugin.name)) return
    if (visiting.has(plugin.name)) {
      throw new Error(`Circular dependency detected: ${plugin.name}`)
    }

    visiting.add(plugin.name)

    // Process dependent plugins first
    const deps = plugin.dependencies || []
    for (const dep of deps) {
      const depPlugin = plugins.find(p => p.name === dep.name || p.name === dep)
      if (!depPlugin && !dep.optional) {
        throw new Error(`Missing required dependency: ${dep}`)
      }
      if (depPlugin) visit(depPlugin)
    }

    visiting.delete(plugin.name)
    visited.add(plugin.name)
    sorted.push(plugin)
  }

  for (const plugin of plugins) {
    visit(plugin)
  }

  return sorted
}
```

### 5. Usage Examples

```typescript
// User code
import { cli } from 'gunshi'
import i18n from 'gunshi/plugins/i18n'
import renderer from 'gunshi/plugins/renderer'

await cli({
  name: 'my-cli',
  plugins: [
    i18n({ locale: 'ja' }), // Register i18n first
    renderer() // renderer depends on i18n
  ]
})

// Using in custom plugin
const myPlugin = plugin({
  name: 'my-plugin',
  dependencies: ['i18n'], // Depends on i18n
  setup: ctx => {
    ctx.decorateCommand(baseRunner => async cmdCtx => {
      const i18n = ctx.getExtension<I18nExtension>('i18n')
      if (i18n) {
        console.log(i18n.translate('my-message'))
      }
      return baseRunner(cmdCtx)
    })
  }
})
```

### 6. Advantages

1. **Explicit dependencies**: Clear dependencies between plugins
2. **Type safety**: TypeScript type checking works
3. **Flexibility**: Supports optional dependencies
4. **Error handling**: Early detection of dependency issues
5. **Extensibility**: Ready for future version management

This design enables complete pluginization of i18n functionality while making it available for use by other plugins.
