# CliOptions Command Hooks Design

## Overview

This document describes the design for adding `onBeforeCommand`, `onAfterCommand`, and `onErrorCommand` hooks to the `CliOptions` interface in the gunshi CLI framework.

## Motivation

While the plugin system provides `decorateCommand` for extending command execution, there's a need for application-level hooks that:

1. Run at the outermost layer of command execution
2. Provide a simpler API for common use cases
3. Apply globally to all commands (including subcommands)
4. Maintain clear separation between application logic and plugin extensions

## API Design

### Interface Definition

```typescript
export interface CliOptions<G extends GunshiParamsConstraint = DefaultGunshiParams> {
  // ... existing properties

  /**
   * Hook that runs before any command execution
   * @param ctx - The command context
   */
  onBeforeCommand?: (ctx: Readonly<CommandContext<G>>) => Awaitable<void>

  /**
   * Hook that runs after successful command execution
   * @param ctx - The command context
   * @param result - The command execution result
   */
  onAfterCommand?: (ctx: Readonly<CommandContext<G>>, result: string | void) => Awaitable<void>

  /**
   * Hook that runs when a command throws an error
   * @param ctx - The command context
   * @param error - The error thrown during execution
   */
  onErrorCommand?: (ctx: Readonly<CommandContext<G>>, error: Error) => Awaitable<void>
}
```

### Usage Example

```typescript
await cli(args, command, {
  name: 'my-cli',
  version: '1.0.0',

  onBeforeCommand: async ctx => {
    // Initialize resources
    await initializeDatabase()
    ctx.startTime = Date.now()
    console.log(`Starting command: ${ctx.command.name}`)
  },

  onAfterCommand: async (ctx, result) => {
    // Cleanup and metrics
    const duration = Date.now() - ctx.startTime
    console.log(`Command completed in ${duration}ms`)
    await closeDatabase()
  },

  onErrorCommand: async (ctx, error) => {
    // Error handling
    await logError(error, ctx)
    await cleanupOnError()
  },

  plugins: [performancePlugin(), loggingPlugin()]
})
```

## Execution Order

The hooks execute at the outermost layer, wrapping both plugin decorators and command execution:

```
1. CliOptions.onBeforeCommand      ← Outermost layer
2. Plugin.decorateCommand (start)
3. Command.run
4. Plugin.decorateCommand (end)
5. CliOptions.onAfterCommand       ← Outermost layer

On error:
- CliOptions.onErrorCommand        ← Catches all errors
```

### Visual Representation

```
┌─────────────────────────────────────────┐
│  CliOptions.onBeforeCommand             │
│  ┌───────────────────────────────────┐  │
│  │  Plugin.decorateCommand           │  │
│  │  ┌─────────────────────────────┐  │  │
│  │  │  Command.run                │  │  │
│  │  └─────────────────────────────┘  │  │
│  └───────────────────────────────────┘  │
│  CliOptions.onAfterCommand              │
└─────────────────────────────────────────┘
         ↓ (on error)
   CliOptions.onErrorCommand
```

## Use Cases

### 1. Global Resource Management

```typescript
{
  onBeforeCommand: async (ctx) => {
    await connectToDatabase()
    await initializeLogger()
  },

  onAfterCommand: async (ctx) => {
    await disconnectFromDatabase()
    await flushLogs()
  }
}
```

### 2. Request Tracking

```typescript
{
  onBeforeCommand: async (ctx) => {
    ctx.requestId = generateRequestId()
    ctx.startTime = performance.now()
  },

  onAfterCommand: async (ctx, result) => {
    const duration = performance.now() - ctx.startTime
    await sendMetrics({
      command: ctx.command.name,
      duration,
      requestId: ctx.requestId,
      success: true
    })
  }
}
```

### 3. Authentication

```typescript
{
  onBeforeCommand: async ctx => {
    const token = await getAuthToken()
    if (!token) {
      throw new Error('Authentication required')
    }
    ctx.auth = { token }
  }
}
```

### 4. Error Reporting

```typescript
{
  onErrorCommand: async (ctx, error) => {
    await sendErrorReport({
      error: error.message,
      stack: error.stack,
      command: ctx.command.name,
      args: ctx.args
    })
  }
}
```

## Comparison with Plugin decorateCommand

| Aspect          | CliOptions Hooks         | Plugin decorateCommand |
| --------------- | ------------------------ | ---------------------- |
| Scope           | Application-wide         | Plugin-specific        |
| Execution Order | Outermost layer          | Inner layer            |
| Use Case        | App lifecycle management | Feature extensions     |
| Reusability     | Single application       | Multiple applications  |
| Complexity      | Simple API               | More flexible          |

## Implementation Considerations

### 1. Error Handling

```typescript
async function executeCommand(command, ctx, options) {
  try {
    // Execute onBeforeCommand
    if (options.onBeforeCommand) {
      await options.onBeforeCommand(ctx)
    }

    // Execute command with plugins
    const result = await executeWithPlugins(command, ctx)

    // Execute onAfterCommand only on success
    if (options.onAfterCommand) {
      await options.onAfterCommand(ctx, result)
    }

    return result
  } catch (error) {
    // Execute onErrorCommand
    if (options.onErrorCommand) {
      try {
        await options.onErrorCommand(ctx, error)
      } catch (hookError) {
        // Log but don't swallow the original error
        console.error('Error in onErrorCommand hook:', hookError)
      }
    }
    throw error
  }
}
```

### 2. Subcommand Support

Hooks should apply to all commands, including subcommands:

```typescript
// Hooks run for both main and subcommands
await cli(args, mainCommand, {
  subCommands: new Map([
    ['deploy', deployCommand],
    ['test', testCommand]
  ]),

  onBeforeCommand: async ctx => {
    // Runs for main, deploy, and test commands
  }
})
```

### 3. Context Extension

Hooks can extend the context for use by plugins and commands:

```typescript
{
  onBeforeCommand: async ctx => {
    // Add application-specific context
    ctx.app = {
      config: await loadConfig(),
      services: await initializeServices()
    }
  }
}
```

## Benefits

1. **Clear Separation of Concerns**: Application logic vs. plugin extensions
2. **Simplified API**: Easier than creating custom plugins for common tasks
3. **Predictable Execution Order**: Outermost layer ensures proper setup/teardown
4. **Global Application**: Automatically applies to all commands
5. **Better Error Handling**: Centralized error handling for all commands

## Migration Path

Existing applications using plugins for these purposes can gradually migrate:

```typescript
// Before: Using plugin
const appPlugin = plugin({
  id: 'app',
  setup: ctx => {
    ctx.decorateCommand(baseRunner => async cmdCtx => {
      await setup()
      try {
        return await baseRunner(cmdCtx)
      } finally {
        await cleanup()
      }
    })
  }
})

// After: Using CliOptions hooks
await cli(args, command, {
  onBeforeCommand: async () => await setup(),
  onAfterCommand: async () => await cleanup()
})
```

## Future Considerations

1. **Additional Hooks**: Consider adding more lifecycle hooks if needed (e.g., `onBeforeParse`, `onAfterResolve`)
2. **Hook Composition**: Allow multiple hooks to be composed
3. **Async Hook Management**: Ensure proper handling of long-running async operations
4. **Performance Impact**: Monitor overhead of additional hook layers

## Conclusion

Adding `onBeforeCommand`, `onAfterCommand`, and `onErrorCommand` hooks to `CliOptions` provides a clean, intuitive API for application-level command lifecycle management while maintaining the flexibility of the existing plugin system.
