# Custom Rendering Design Proposal

## Background and Purpose

Currently in gunshi, `plugin-renderer` provides unified rendering of headers, usage, and validation errors. However, the current decorator pattern-based extension has the following limitations.

### Current Issues

1. **Uncertainty in Rendering Control**
   - Even if a plugin tries to disable rendering, it can be overridden by other plugins
   - Example: `plugin-completion` returns an empty string in `decorateHeaderRenderer` to hide the header, but other plugins registered later can override this behavior

2. **Decorator Execution Order**
   - Decorators registered later are executed first (LIFO: Last In, First Out)
   - Depends on plugin registration order, which can cause unintended behavior

3. **Lack of Complete Control**
   - No way for a plugin to reliably enforce "completely disable rendering for this command"
   - Difficult coordination between plugins

### Concrete Example

```javascript
// completion plugin disables header
ctx.decorateHeaderRenderer(async (_baseRenderer, _cmdCtx) => '')

// However, another plugin registered later can override it
ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
  return 'Custom Header\n' // completion plugin's intent is ignored
})
```

This issue can be reproduced with `reproduction.js`.

### Purpose of This Proposal

This proposal presents design options to achieve the following for specific Commands:

- **Reliably** disable default rendering
- **Prioritize** custom rendering logic
- Avoid conflicts between plugins and enable clear control

## Current Architecture

### Rendering Flow

1. `plugin-renderer` registers decorators

   ```typescript
   ctx.decorateHeaderRenderer(...)
   ctx.decorateUsageRenderer(...)
   ctx.decorateValidationErrorsRenderer(...)
   ```

2. Extension through decorator pattern
   - Multiple decorators are applied in a chain
   - The last registered decorator is executed first

3. Role of each renderer
   - `renderHeader`: Display command header information
   - `renderUsage`: Display command usage
   - `renderValidationErrors`: Display validation errors

## Design Options

### Option 1: Add Rendering Control Flags to Command Definition

Add a `rendering` property to the Command definition to control rendering behavior.

```typescript
interface Command<G extends GunshiParamsConstraint = DefaultGunshiParams> {
  // Existing properties...

  /**
   * Rendering control options
   * @since v0.27.0
   */
  rendering?: {
    /**
     * Header rendering configuration
     * - null: Disable rendering
     * - function: Use custom renderer
     * - undefined (when omitted): Use default renderer
     */
    header?: ((ctx: CommandContext<G>) => Promise<string>) | null

    /**
     * Usage rendering configuration
     * - null: Disable rendering
     * - function: Use custom renderer
     * - undefined (when omitted): Use default renderer
     */
    usage?: ((ctx: CommandContext<G>) => Promise<string>) | null

    /**
     * Validation errors rendering configuration
     * - null: Disable rendering
     * - function: Use custom renderer
     * - undefined (when omitted): Use default renderer
     */
    validationErrors?: ((ctx: CommandContext<G>, error: AggregateError) => Promise<string>) | null
  }
}
```

**Advantages:**

- Simple and intuitive
- Can be explicitly specified at Command definition time
- Allows partial disabling and customization

**Disadvantages:**

- Command definition becomes more complex
- Affects all existing Command definitions

**Implementation Example:**

```typescript
const myCommand: Command = {
  name: 'custom',
  description: 'Custom rendering command',
  rendering: {
    header: null, // Disable header
    usage: async ctx => {
      return `My custom usage for ${ctx.name}` // Access command name via ctx.name
    }
    // validationErrors omitted = use default
  },
  run: async ctx => {
    // ...
  }
}
```

**Architectural Challenge:**
Since the current CommandContext type doesn't have a direct reference to the Command object, accessing the `rendering` property of the Command definition from within rendering functions requires one of the following approaches:

1. Add a command property to CommandContext (breaking change)
2. Build a mechanism to pass Command information separately during rendering
3. Add a mechanism to store Command information globally or in context

### Option 2: Dedicated Rendering Disable Plugin

Create a dedicated plugin to control rendering for specific commands.

```typescript
interface DisableRenderingOptions {
  /**
   * List of command names to disable rendering for
   */
  commands: string[]

  /**
   * Targets to disable
   */
  targets?: {
    header?: boolean
    usage?: boolean
    validationErrors?: boolean
  }
}

export default function disableRendering(options: DisableRenderingOptions) {
  return plugin({
    id: namespacedId('disable-rendering'),
    name: 'disable rendering',

    setup(ctx) {
      const { commands, targets = {} } = options
      const { header = true, usage = true, validationErrors = true } = targets

      if (header) {
        ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
          if (commands.includes(cmdCtx.name || '')) {
            return ''
          }
          return baseRenderer(cmdCtx)
        })
      }

      if (usage) {
        ctx.decorateUsageRenderer(async (baseRenderer, cmdCtx) => {
          if (commands.includes(cmdCtx.name || '')) {
            return ''
          }
          return baseRenderer(cmdCtx)
        })
      }

      if (validationErrors) {
        ctx.decorateValidationErrorsRenderer(async (baseRenderer, cmdCtx, error) => {
          if (commands.includes(cmdCtx.name || '')) {
            return ''
          }
          return baseRenderer(cmdCtx, error)
        })
      }
    }
  })
}
```

**Advantages:**

- Leverages existing mechanisms
- Flexible control through plugins
- No need to change Command definitions

**Disadvantages:**

- Requires separate implementation for complete custom rendering
- Specification by command name makes it difficult to handle dynamic commands

**Usage Example:**

```typescript
await cli(args, entry, {
  plugins: [
    renderer(),
    disableRendering({
      commands: ['custom', 'special'],
      targets: {
        header: true,
        usage: false // Don't disable usage
      }
    })
  ]
})
```

### Option 3: Add Rendering Skip Flags to CommandContext

Add dynamic rendering control flags to CommandContext.

```typescript
interface CommandContext<G extends GunshiParamsConstraint = DefaultGunshiParams> {
  // Existing properties...

  /**
   * Rendering control flags
   * Can be set within CommandRunner
   * @since v0.27.0
   */
  skipDefaultRendering?: {
    header?: boolean
    usage?: boolean
    validationErrors?: boolean
  }
}
```

**Advantages:**

- Dynamically controllable at runtime
- Can switch rendering based on conditions

**Disadvantages:**

- Must be controlled within CommandRunner
- Cannot be controlled before execution

**Implementation Example:**

```typescript
const myCommand: Command = {
  name: 'dynamic',
  run: async ctx => {
    // Control rendering based on conditions
    if (ctx.flags.quiet) {
      ctx.skipDefaultRendering = {
        header: true,
        usage: true
      }
    }
    // ...
  }
}
```

### Option 4: Extend Rendering Pipeline

Introduce a pipeline system for more advanced rendering control.

```typescript
interface RenderingPipeline {
  /**
   * Renderer map per Command
   */
  commandRenderers: Map<
    string,
    {
      header?: (ctx: CommandContext) => Promise<string>
      usage?: (ctx: CommandContext) => Promise<string>
      validationErrors?: (ctx: CommandContext, error: AggregateError) => Promise<string>
    }
  >

  /**
   * Renderer selection logic
   */
  selectRenderer(
    commandName: string,
    type: 'header' | 'usage' | 'validationErrors'
  ): ((ctx: CommandContext, ...args: any[]) => Promise<string>) | null

  /**
   * Register custom renderer
   */
  registerCommandRenderer(
    commandName: string,
    renderers: {
      header?: (ctx: CommandContext) => Promise<string>
      usage?: (ctx: CommandContext) => Promise<string>
      validationErrors?: (ctx: CommandContext, error: AggregateError) => Promise<string>
    }
  ): void
}
```

**Advantages:**

- Advanced control possible
- Can manage multiple renderers
- Enables coordination between plugins

**Disadvantages:**

- Complex implementation
- Requires introduction of new concepts

### Option 5: Command-specific Renderer Plugin Extension

Leverage the plugin extension feature to provide Command-specific renderers.

```typescript
interface CommandWithCustomRenderer extends Command {
  /**
   * Renderer plugin ID to use
   */
  rendererId?: string
}

// Custom renderer plugin
export default function customRenderer() {
  return plugin({
    id: namespacedId('custom-renderer'),
    name: 'custom renderer',

    extension: (ctx, cmd) => ({
      renderHeader: async () => {
        return `=== ${cmd.name} ===\n${cmd.description}`
      },
      renderUsage: async () => {
        return 'Custom usage information'
      },
      renderValidationErrors: async error => {
        return `Error: ${error.message}`
      }
    }),

    setup(ctx) {
      ctx.decorateHeaderRenderer(async (baseRenderer, cmdCtx) => {
        // Note: Cannot directly access Command object in current architecture
        // This implementation requires architectural changes
        const ext = cmdCtx.extensions[namespacedId('custom-renderer')]
        return ext?.renderHeader?.() || baseRenderer(cmdCtx)
      })
    }
  })
}
```

**Advantages:**

- Integrated with plugin system
- Highly extensible
- Multiple custom renderers can coexist

**Disadvantages:**

- Somewhat complex implementation
- Creates dependencies between plugins

## Recommended Implementation Approach

As the most practical approach, we recommend combining **Option 1 (Command definition rendering control flags)** and **Option 2 (dedicated plugin)**.

### Implementation Steps

1. **Phase 1: Extend Command Definition**
   - Add `rendering` property to `Command` interface
   - Implement basic disable flags (`disableHeader`, etc.)

2. **Phase 2: plugin-renderer Support**
   - Modify decorators to consider the `rendering` property of Command definitions
   - Prioritize custom renderers when specified

3. **Phase 3: Provide Dedicated Plugin**
   - Create dedicated plugin for cases requiring more advanced control
   - Support dynamic control and command name-based control

### Implementation Example

```typescript
// Command definition
const myCommand: Command = {
  name: 'special',
  description: 'Special command with custom rendering',
  rendering: {
    header: null, // Disable header
    usage: async ctx => {
      return `
Special Command Usage:
  ${ctx.env.name} special [options]

This is a custom usage message.
      `.trim()
    }
  },
  run: async ctx => {
    console.log('Running special command')
  }
}

// CLI execution
await cli(args, entry, {
  subCommands: new Map([['special', myCommand]]),
  plugins: [
    renderer(), // Default renderer
    disableRendering({
      // Additional control (Option 2 plugin)
      commands: ['another-command'],
      targets: { header: true }
    })
  ]
})
```

## Summary

The proposed design enables gunshi users to:

1. Disable default rendering for specific commands
2. Implement custom rendering logic
3. Integrate with existing plugin system
4. Adopt gradually

This enables building more flexible CLI applications.
