# Custom Rendering Implementation Plan - Option 1: Add Rendering Control Flags to Command Definition

## Overview

This document provides a concrete implementation plan for Option 1 "Add rendering control flags to Command definition" proposed in `design/6.custom-rendering-proposal.md`.

## Understanding the Rendering Architecture

### Current Rendering Priority

From the implementation of the `normalizeCliOptions` function, rendering functions are determined in the following priority order:

1. **Renderers explicitly specified in CliOptions** (highest priority)

   ```typescript
   if (resolvedOptions.renderHeader === undefined) {
     resolvedOptions.renderHeader = decorators.getHeaderRenderer()
   }
   ```

2. **Renderers registered through decorator pattern** (plugin extensions)
   - Plugins register via `ctx.decorateHeaderRenderer()` etc.
   - Executed in LIFO (Last In, First Out) order

3. **Default renderer** (returns empty string)

### How to Apply Command Definition's rendering Property

Process the `rendering` property of the Command object directly within the `createCommandContext` function:

1. **Processing within createCommandContext**
   - Check the `rendering` option from the `command` parameter
   - If the `rendering` option exists, apply it preferentially when creating CommandEnvironment

2. **Implementation flow**

   ```typescript
   // Within createCommandContext
   const env = Object.assign(create<CommandEnvironment<G>>(), CLI_OPTIONS_DEFAULT, cliOptions)

   // Preferentially apply Command definition's rendering option
   if (command.rendering) {
     if ('header' in command.rendering) {
       env.renderHeader =
         command.rendering.header === null ? null : command.rendering.header || env.renderHeader
     }
     // Process usage and validationErrors similarly
   }
   ```

This approach provides:

- No need to search for Command during rendering
- Command definition's `rendering` takes highest priority
- Simple and efficient implementation

## Implementation Scope

### 1. Type Definition Extensions

#### 1.1 Extending the Command Interface

**File**: `packages/gunshi/src/types.ts`

```typescript
/**
 * Rendering control options
 * @since v0.27.0
 */
export interface RenderingOptions<G extends GunshiParamsConstraint = DefaultGunshiParams> {
  /**
   * Header rendering configuration
   * - `null`: Disable rendering
   * - `function`: Use custom renderer
   * - `undefined` (when omitted): Use default renderer
   */
  header?: ((ctx: CommandContext<G>) => Awaitable<string>) | null

  /**
   * Usage rendering configuration
   * - `null`: Disable rendering
   * - `function`: Use custom renderer
   * - `undefined` (when omitted): Use default renderer
   */
  usage?: ((ctx: CommandContext<G>) => Awaitable<string>) | null

  /**
   * Validation errors rendering configuration
   * - `null`: Disable rendering
   * - `function`: Use custom renderer
   * - `undefined` (when omitted): Use default renderer
   */
  validationErrors?: ((ctx: CommandContext<G>, error: AggregateError) => Awaitable<string>) | null
}

export interface Command<G extends GunshiParamsConstraint = DefaultGunshiParams> {
  // ... existing properties ...

  /**
   * Rendering control options
   * @since v0.27.0
   */
  rendering?: RenderingOptions<G>
}
```

### 2. Core Library Support

#### 2.1 Modifying createCommandContext

**File**: `packages/gunshi/src/context.ts`

Process the `rendering` option of the Command definition within the `createCommandContext` function:

```typescript
export async function createCommandContext<...>({
  // ...existing parameters
  command,
  cliOptions,
  // ...
}): Promise<...> {
  // ...existing processing

  const env = Object.assign(create<CommandEnvironment<G>>(), CLI_OPTIONS_DEFAULT, cliOptions)

  // Preferentially apply Command definition's rendering option
  if (command.rendering) {
    // Process header
    if ('header' in command.rendering) {
      if (command.rendering.header === null) {
        env.renderHeader = null
      } else if (typeof command.rendering.header === 'function') {
        env.renderHeader = command.rendering.header
      }
    }

    // Process usage
    if ('usage' in command.rendering) {
      if (command.rendering.usage === null) {
        env.renderUsage = null
      } else if (typeof command.rendering.usage === 'function') {
        env.renderUsage = command.rendering.usage
      }
    }

    // Process validationErrors
    if ('validationErrors' in command.rendering) {
      if (command.rendering.validationErrors === null) {
        env.renderValidationErrors = null
      } else if (typeof command.rendering.validationErrors === 'function') {
        env.renderValidationErrors = command.rendering.validationErrors
      }
    }
  }

  // ...remaining processing
}
```

This implementation ensures that the Command definition's `rendering` option is applied with highest priority, regardless of whether plugin-renderer is installed.

### 3. plugin-renderer Support

#### 3.1 Understanding Rendering Priority

**File**: `packages/plugin-renderer/src/index.ts`

Since the Command definition's `rendering` option is already applied within createCommandContext, plugin-renderer requires no special processing.

Rendering priority:

1. Command definition's `rendering` option (applied within createCommandContext)
2. Renderers explicitly specified in CliOptions
3. Renderers via plugin-renderer's decorator
4. Default renderer

```typescript
setup: ctx => {
  // Keep existing implementation
  ctx.decorateHeaderRenderer(
    async (_baseRenderer, cmdCtx) => await renderHeader<RendererCommandContext>(cmdCtx)
  )

  ctx.decorateUsageRenderer(
    async (_baseRenderer, cmdCtx) => await renderUsage<RendererCommandContext>(cmdCtx)
  )

  ctx.decorateValidationErrorsRenderer(
    async (_baseRenderer, cmdCtx, error) =>
      await renderValidationErrors<RendererCommandContext>(cmdCtx, error)
  )
}
```

Since the Command definition's `rendering` option is already reflected in CommandEnvironment, plugin-renderer works correctly without additional processing.

### 4. Implementation Order and Milestones

#### Phase 1: Basic Implementation (1 week)

1. **Add type definitions**
   - Define `RenderingOptions` interface
   - Add `rendering` property to `Command` interface
   - Add type tests

2. **Core library implementation**
   - Modify `createCommandContext` function
   - Add processing for Command definition's `rendering` option
   - Verify operation without plugin-renderer

3. **Verify plugin-renderer**
   - Confirm correct operation with existing code
   - No additional modifications needed

4. **Create unit tests**
   - Test operation without plugin-renderer
   - Test operation with plugin-renderer
   - Test custom renderers
   - Test compatibility with existing features

#### Phase 2: Integration Tests (1 day)

1. **Integration tests**
   - Test combinations with other plugins
   - Verify edge cases

#### Phase 3: Optimization and Extensions (optional)

1. **Performance optimization**
   - Cache rendering processing
   - Skip unnecessary processing

2. **Additional features**
   - Conditional rendering
   - Rendering presets

## Implementation Details

### 1. File Change List

```
packages/gunshi/src/
├── types.ts                    # RenderingOptions, Command extension
├── context.ts                  # createCommandContext modification
├── index.ts                    # Add exports as needed
├── type.test-d.ts              # Type tests
└── rendering.test.ts           # Rendering option tests

packages/plugin-renderer/src/
├── index.ts                    # No changes needed
└── test/
    └── rendering-option.test.ts # Integration tests with Command definition's rendering option
```

### 2. Backward Compatibility

- Existing Commands don't have `rendering` property, so behavior is unchanged
- `rendering` property is optional, allowing gradual migration
- Can coexist with existing plugin decorators

### 3. Edge Case Handling

#### 3.1 Explicit Value Handling

Three values are clearly distinguished: `null`, `undefined`, and function:

- `null`: Explicit disabling
- `undefined` or omitted: Use default renderer
- function: Use custom renderer

#### 3.2 Plugin Priority

- Command definition's `rendering` has highest priority
- Plugin decorators are applied next
- If plugin-renderer is not present, core library's default implementation is used

#### 3.3 Integration with plugin-renderer

When plugin-renderer is installed:

- plugin-renderer respects Command definition's `rendering` option
- Advanced rendering from plugin-renderer is applied only when no custom renderer exists

When plugin-renderer is not installed:

- Core library's `renderWithOptions` function operates
- Only basic rendering functionality is provided

#### 3.4 Error Handling

- If custom renderer throws an error, fall back to default renderer
- Log errors in development mode

## Test Strategy

### 1. Unit Tests

```typescript
// packages/gunshi/src/rendering.test.ts
describe('header rendering with custom options', () => {
  it('should disable header when header is null', async () => {
    const cmd: Command = {
      name: 'test',
      rendering: { header: null }
    }
    // ... test implementation
  })

  it('should use custom header renderer', async () => {
    const cmd: Command = {
      name: 'test',
      rendering: {
        header: async ctx => `Custom: ${ctx.name}`
      }
    }
    // ... test implementation
  })

  it('should use default renderer when header is undefined', async () => {
    const cmd: Command = {
      name: 'test',
      rendering: {} // header property doesn't exist
    }
    // ... test implementation
  })
})
```

### 2. Integration Tests

```typescript
// packages/gunshi/src/rendering.test.ts
describe('custom rendering integration', () => {
  it('should work without plugin-renderer', async () => {
    const cmd: Command = {
      name: 'test',
      rendering: {
        header: null,
        usage: async ctx => 'Custom usage'
      }
    }

    await cli(['test'], cmd, {
      plugins: [] // without plugin-renderer
    })
    // Verify rendering option works correctly
  })

  it('should work with plugin-renderer', async () => {
    const cmd: Command = {
      name: 'test',
      rendering: {
        header: null
      }
    }

    await cli(['test'], cmd, {
      plugins: [renderer()] // with plugin-renderer
    })
    // Verify coordination with plugin-renderer
  })

  it('should work with multiple plugins', async () => {
    await cli(['test'], testCommand, {
      plugins: [renderer(), i18n(), completion()]
    })
    // ... assertions
  })
})
```

## Risks and Mitigations

### 1. Performance Impact

**Risk**: Overhead from checking rendering options
**Mitigation**:

- Pre-check existence of rendering options
- Execute additional processing only when necessary

### 2. Type Complexity

**Risk**: Degraded developer experience due to complex generic types
**Mitigation**:

- Improve type inference
- Clear type error messages
- Abundant code examples

### 3. Maintainability

**Risk**: Increased maintenance cost from new features
**Mitigation**:

- Clear separation of concerns
- Comprehensive test coverage

## Success Metrics

1. **Functionality**
   - Complete control over rendering for specific commands
   - No impact on existing functionality
   - Resolution of conflicts between plugins

2. **Developer Experience**
   - Intuitive API
   - Clear error messages

3. **Performance**
   - No performance degradation for existing commands
   - Optimized rendering processing

## Future Extensibility

1. **Rendering Presets**

   ```typescript
   rendering: {
     preset: 'minimal' // or 'verbose', 'custom'
   }
   ```

2. **Conditional Rendering**

   ```typescript
   rendering: {
     header: ctx => (ctx.flags.quiet ? null : defaultHeader(ctx))
   }
   ```

3. **Rendering Middleware**
   ```typescript
   rendering: {
     middleware: [trimWhitespace, addColors, formatMarkdown]
   }
   ```

## API Usage Examples

### Basic Usage Examples

```typescript
// Disable rendering
const silentCommand: Command = {
  name: 'silent',
  description: 'Silent command',
  rendering: {
    header: null, // Disable header
    usage: null, // Disable usage
    validationErrors: null // Disable error display
  },
  run: async ctx => {
    // Execute without displaying anything
  }
}

// Custom rendering
const customCommand: Command = {
  name: 'custom',
  description: 'Custom rendering command',
  rendering: {
    header: async ctx => {
      return `=== ${ctx.name?.toUpperCase()} ===`
    },
    usage: async ctx => {
      return `Usage: ${ctx.env.name} ${ctx.name} [options]`
    }
    // validationErrors omitted = use default renderer
  },
  run: async ctx => {
    // Execute with custom header and usage
  }
}

// Partial customization
const partialCommand: Command = {
  name: 'partial',
  description: 'Partial customization',
  rendering: {
    header: null // Disable only header
    // usage and validationErrors use defaults
  },
  run: async ctx => {
    // Execute without header
  }
}
```

### plugin-completion Modification Example

```typescript
// Current implementation (can be overridden by other plugins)
ctx.decorateHeaderRenderer(async (_baseRenderer, _cmdCtx) => '')

// New implementation (reliably disabled in Command definition)
const completeCommand: Command = {
  name: 'complete',
  description: 'Generate shell completion script',
  rendering: {
    header: null // Reliably disable header
  },
  run: async ctx => {
    // completion processing
  }
}
```

## Summary

This implementation plan enables gunshi users to have complete control over rendering for specific commands. Key features:

1. **Simple implementation**: Directly process Command definition's `rendering` option within `createCommandContext`
2. **Highest priority**: Command definition's `rendering` option takes precedence over all other rendering settings
3. **plugin-renderer independent**: Core library supports rendering option, works without plugin-renderer
4. **Intuitive API**: Disable with `null`, customize with function, default behavior with `undefined`
5. **Backward compatibility**: No impact on existing code, gradual migration possible
6. **Reliable control**: Solves issues for plugins with special requirements like `plugin-completion`

This design enables building more flexible and robust CLI applications without major architectural changes.
