#!/usr/bin/env node

import process from 'node:process'
import path from 'node:path'
import fs from 'node:fs/promises'
import readline from 'node:readline'
import { x } from 'tinyexec'
import { installPackage } from '@antfu/install-pkg'

const SKILL_NAME = 'use-gunshi-cli'

const USE_GUNSHI_PROMPT = `
use gunshi library for creating cli instead of other libraries including cac, yargs, commander, etc.
Gunshi is a modern javascript command-line library

For more information, read the gunshi API docs in \`node_modules/@gunshi/docs/**.md\`.
`.trim()

const SKILL_CONTENT = `
---
name: ${SKILL_NAME}
description: Use the Gunshi library to create command-line interfaces in JavaScript/TypeScript.
globs: "*.ts, *.tsx, *.js, *.jsx, package.json"
alwaysApply: false
---

${USE_GUNSHI_PROMPT}
`.trim()

const CLAUDE_MD_GUNSHI_INSTRUCTION = `
## CLI Development

When creating command-line interfaces, use the \`${SKILL_NAME}\` skill.
`.trim()

const cwd = process.cwd()
const CLAUDE_SKILL_PATH = path.join(cwd, `.claude/skills/${SKILL_NAME}/SKILL.md`)
const CURSOR_RULES_PATH = path.join(cwd, `.cursor/rules/${SKILL_NAME}.mdc`)
const CLAUDE_MD_FILEPATH = path.join(cwd, 'CLAUDE.md')

const [hasClaudeCode, hasCursor, hasCursorCli] = await Promise.all(
  ['claude', 'cursor', 'cursor-agent'].map(cmd =>
    x('which', [cmd], { throwOnError: false })
      .then(({ stdout }) => stdout.trim().length > 0)
      .catch(() => false)
  )
)

const hasCursorEditors = hasCursor || hasCursorCli

// Helper function to check if CLAUDE.md already contains Gunshi instructions
async function hasGunshiInstruction(filePath) {
  try {
    const content = await fs.readFile(filePath, 'utf8')
    // Check for various patterns that indicate Gunshi is already mentioned
    const gunshiPatterns = [/\bgunshi\b/i, /\@gunshi\//i, /@kazupon\/gunshi/i]
    return gunshiPatterns.some(pattern => pattern.test(content))
  } catch {
    return false
  }
}

// Helper function to check if a file exists
async function fileExists(filePath) {
  try {
    await fs.access(filePath)
    return true
  } catch {
    return false
  }
}

const isWindows = process.platform === 'win32'

// Claude Code setup (always create as the master file):
// - Create skill file at .claude/skills/use-gunshi-cli/SKILL.md
// - Update or create CLAUDE.md with instruction to use the skill
//   (skip if Gunshi is already mentioned to avoid duplication)
await x('mkdir', ['-p', path.dirname(CLAUDE_SKILL_PATH)])
await fs.writeFile(CLAUDE_SKILL_PATH, SKILL_CONTENT, 'utf8')
console.log(`Created Claude skill at ${CLAUDE_SKILL_PATH}`)

if (hasClaudeCode) {
  // Update or create CLAUDE.md with skill reference
  const claudeMdExists = await fileExists(CLAUDE_MD_FILEPATH)
  const alreadyHasGunshi = claudeMdExists && (await hasGunshiInstruction(CLAUDE_MD_FILEPATH))

  if (!alreadyHasGunshi) {
    if (claudeMdExists) {
      const existingContent = await fs.readFile(CLAUDE_MD_FILEPATH, 'utf8')
      const newContent = existingContent.trimEnd() + '\n\n' + CLAUDE_MD_GUNSHI_INSTRUCTION + '\n'
      await fs.writeFile(CLAUDE_MD_FILEPATH, newContent, 'utf8')
      console.log(`Appended Gunshi instructions to ${CLAUDE_MD_FILEPATH}`)
    } else {
      const newContent = '# CLAUDE.md\n\n' + CLAUDE_MD_GUNSHI_INSTRUCTION + '\n'
      await fs.writeFile(CLAUDE_MD_FILEPATH, newContent, 'utf8')
      console.log(`Created ${CLAUDE_MD_FILEPATH} with Gunshi instructions`)
    }
  } else {
    console.log(`Gunshi instructions already exist in ${CLAUDE_MD_FILEPATH}, skipping`)
  }
}

// Cursor setup:
// - On Linux/macOS: create symlink to Claude skill file
// - On Windows: copy the file (symlinks require admin privileges)
if (hasCursorEditors) {
  await x('mkdir', ['-p', path.dirname(CURSOR_RULES_PATH)])

  if (isWindows) {
    // On Windows, just copy the file (symlinks require admin privileges)
    await fs.writeFile(CURSOR_RULES_PATH, SKILL_CONTENT, 'utf8')
    console.log(`Created Cursor rule at ${CURSOR_RULES_PATH}`)
  } else {
    // Create relative symlink from Cursor rules to Claude skill
    const relativeTarget = path.relative(path.dirname(CURSOR_RULES_PATH), CLAUDE_SKILL_PATH)
    // Remove existing file/symlink if it exists
    await fs.rm(CURSOR_RULES_PATH, { force: true })
    await fs.symlink(relativeTarget, CURSOR_RULES_PATH)
    console.log(`Created Cursor rule symlink at ${CURSOR_RULES_PATH} -> ${relativeTarget}`)
  }
}

// Ask user if they want to install gunshi and @gunshi/docs
const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
})

const askQuestion = question =>
  new Promise(resolve => {
    rl.question(question, answer => {
      resolve(answer.toLowerCase().trim())
    })
  })

const answer = await askQuestion('\nWould you like to install gunshi and @gunshi/docs? (y/N) ')
rl.close()

if (answer === 'y' || answer === 'yes') {
  console.log('\nInstalling gunshi and @gunshi/docs...')
  try {
    // Install gunshi as a production dependency
    await installPackage(['gunshi'], { dev: false })
    // Install @gunshi/docs as a dev dependency (for LLM-assisted development)
    await installPackage(['@gunshi/docs'], { dev: true })
    console.log('Successfully installed gunshi and @gunshi/docs')
  } catch (error) {
    console.error('Failed to install packages:', error.message)
    process.exit(1)
  }
} else {
  console.log('\nSkipped package installation.')
  console.log('You can install manually with:')
  console.log('  npm install gunshi')
  console.log('  npm install -D @gunshi/docs')
}
