/**
 * @author kazuya kawaguchi (a.k.a. kazupon)
 * @license MIT
 */

import type {
  Args,
  Command,
  DefaultGunshiParams,
  ExtendContext,
  ExtractArgs,
  ExtractExtensions,
  GunshiParamsConstraint,
  Prettify
} from '@gunshi/plugin'
import type { CommandResourceFetcher, I18nCommand } from './types.ts'

/**
 * The result type of the {@linkcode defineI18n} function
 *
 * @internal
 */
type I18nCommandDefinitionResult<
  G extends GunshiParamsConstraint = DefaultGunshiParams,
  C = {}
> = Prettify<
  Omit<C, 'resource'> &
    ('resource' extends keyof C
      ? { resource: CommandResourceFetcher<G> }
      : { resource?: CommandResourceFetcher<G> | undefined }) & {
      [K in Exclude<keyof I18nCommand<G>, keyof C | 'resource'>]?: I18nCommand<G>[K]
    }
>

/**
 * The result type of the {@linkcode withI18nResource} function
 *
 * @internal
 */
type WithI18nResourceResult<
  G extends GunshiParamsConstraint = DefaultGunshiParams,
  C extends Command<G> = Command<G>
> = Prettify<
  C & { resource: CommandResourceFetcher<G> } & {
    [K in Exclude<keyof I18nCommand<G>, keyof C | 'resource'>]?: I18nCommand<G>[K]
  }
>

/**
 * Define an i18n-aware {@link I18nCommand | command}.
 *
 * The difference from the {@linkcode define} function is that you can define a `resource` option that can load a locale.
 *
 * @example
 * ```ts
 * import { defineI18n } from '@gunshi/plugin-i18n'
 *
 * const greetCommand = defineI18n({
 *   name: 'greet',
 *   args: {
 *     name: { type: 'string', description: 'Name to greet' }
 *   },
 *   resource: locale => {
 *     switch (locale.toString()) {
 *       case 'ja-JP': {
 *         return {
 *           'description': '誰かにあいさつ',
 *           'arg:name': 'あいさつするための名前'
 *         }
 *       }
 *       // other locales ...
 *     }
 *   },
 *   run: ctx => {
 *     console.log(`Hello, ${ctx.values.name}!`)
 *   }
 * })
 * ```
 *
 * @typeParam G - A {@linkcode GunshiParamsConstraint} type
 * @typeParam A - An {@linkcode Args} type extracted from {@linkcode GunshiParamsConstraint}
 * @typeParam C - The inferred command type
 *
 * @param definition - A {@link I18nCommand | command} definition with i18n support
 * @returns A defined {@link I18nCommand | command} with compatible {@linkcode Command} type
 */
export function defineI18n<
  G extends GunshiParamsConstraint = DefaultGunshiParams,
  A extends Args = ExtractArgs<G>,
  C = {}
>(
  definition: C & { args?: A } & Omit<
      I18nCommand<{ args: A; extensions: ExtractExtensions<G> }>,
      'resource' | 'args'
    > & { resource?: CommandResourceFetcher<{ args: A; extensions: ExtractExtensions<G> }> }
): I18nCommandDefinitionResult<{ args: A; extensions: ExtractExtensions<G> }, C>

/**
 * Define an i18n-aware {@link I18nCommand | command}.
 *
 * @param definition - A {@link I18nCommand | command} definition with i18n support
 * @returns A defined {@link I18nCommand | command} with compatible {@linkcode Command} type
 */
export function defineI18n(
  definition: any // eslint-disable-line @typescript-eslint/no-explicit-any -- NOTE(kazupon): for implementation
  // eslint-disable-next-line @typescript-eslint/no-explicit-any -- NOTE(kazupon): for implementation
): any {
  // eslint-disable-next-line @typescript-eslint/no-unsafe-return -- NOTE(kazupon): for implementation
  return definition
}

/**
 * Return type for {@link defineI18nWithTypes}
 *
 * @typeParam DefaultExtensions - The {@linkcode ExtendContext} type extracted from G
 * @typeParam DefaultArgs - The {@linkcode Args} type extracted from G
 *
 * @internal
 */
type DefineI18nWithTypesReturn<
  DefaultExtensions extends ExtendContext,
  DefaultArgs extends Args
> = <A extends DefaultArgs = DefaultArgs, C = {}>(
  definition: C & { args?: A } & Omit<
      I18nCommand<{ args: A; extensions: DefaultExtensions }>,
      'resource' | 'args'
    > & { resource?: CommandResourceFetcher<{ args: A; extensions: DefaultExtensions }> }
) => I18nCommandDefinitionResult<{ args: A; extensions: DefaultExtensions }, C>

/**
 * Define an i18n-aware {@link I18nCommand | command} with types
 *
 * This helper function allows specifying the type parameter of {@linkcode GunshiParams}
 * while inferring the {@linkcode Args} type, {@linkcode ExtendContext} type from the definition.
 *
 * @example
 * ```ts
 * import { defineI18nWithTypes } from '@gunshi/plugin-i18n'
 *
 * // Define a command with specific extensions type
 * type MyExtensions = { logger: { log: (message: string) => void } }
 *
 * const greetCommand = defineI18nWithTypes<{ extensions: MyExtensions }>()({
 *   name: 'greet',
 *   args: {
 *     name: { type: 'string', description: 'Name to greet' }
 *   },
 *   resource: locale => {
 *     switch (locale.toString()) {
 *       case 'ja-JP': {
 *         return {
 *           'description': '誰かにあいさつ',
 *           'arg:name': 'あいさつするための名前'
 *         }
 *       }
 *       // other locales ...
 *     }
 *   },
 *   run: ctx => {
 *     // ctx.values is inferred as { name?: string }
 *     // ctx.extensions is MyExtensions
 *   }
 * })
 * ```
 *
 * @typeParam G - A {@linkcode GunshiParams} type
 *
 * @returns A function that takes a command definition via {@linkcode defineI18n}
 */
export function defineI18nWithTypes<G extends GunshiParamsConstraint>(): DefineI18nWithTypesReturn<
  ExtractExtensions<G>,
  ExtractArgs<G>
> {
  // Extract extensions from G, with proper defaults
  type DefaultExtensions = ExtractExtensions<G>
  type DefaultArgs = ExtractArgs<G>

  return (<A extends DefaultArgs = DefaultArgs, C = {}>(
    definition: C & { args?: A } & Omit<
        I18nCommand<{ args: A; extensions: DefaultExtensions }>,
        'resource' | 'args'
      > & { resource?: CommandResourceFetcher<{ args: A; extensions: DefaultExtensions }> }
  ): I18nCommandDefinitionResult<{ args: A; extensions: DefaultExtensions }, C> => {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/no-unsafe-return -- NOTE(kazupon): defineI18n returns compatible type but TypeScript cannot infer it
    return defineI18n(definition) as any
  }) as DefineI18nWithTypesReturn<DefaultExtensions, DefaultArgs>
}

/**
 * Add i18n resource to an existing command
 *
 * @example
 * ```ts
 * import { define } from 'gunshi'
 * import { withI18nResource } from '@gunshi/plugin-i18n'
 *
 * const myCommand = define({
 *   name: 'myCommand',
 *   args: {
 *     input: { type: 'string', description: 'Input value' }
 *   },
 *   run: ctx => {
 *     console.log(`Input: ${ctx.values.input}`)
 *   }
 * })
 *
 * const i18nCommand = withI18nResource(basicCommand, async locale => {
 *   const resource = await import(
 *     `./path/to/resources/test/${locale.toString()}.json`,
 *     { with: { type: 'json' } }
 *   ).then(l => l.default || l)
 *   return resource
 * })
 * ```
 *
 * @param command - A defined {@link Command | command} with {@linkcode define} function
 * @param resource - A {@link CommandResourceFetcher | resource fetcher} for the command
 * @returns A {@link I18nCommand | command} with i18n resource support
 */
export function withI18nResource<G extends GunshiParamsConstraint, C extends Command<G>>(
  command: C,
  resource: CommandResourceFetcher<G>
): WithI18nResourceResult<G, C> {
  return {
    ...command,
    resource
  } as WithI18nResourceResult<G, C>
}
