import type { CompleteCommitsParser, ParsedCommit, RawCommit, RawReference, RefLabel, Reference, Contributor, CommitMessage, CommitRange } from '@/types'
import { GpgSigLabel } from '@/enums'
import { getRawCommits, warn } from '@/utils'
import { createHash } from 'node:crypto'
import { defaultConfig, DEFAULT_RELEASE_TAG_PATTERN } from '@/defaults'

const parsedCommitsCache = new Map<string, ParsedCommit>()
let recentReleaseTag: ParsedCommit['releaseTag']

export const parseCommits = (commits: CommitRange | (RawCommit | string)[], commitsParser?: CompleteCommitsParser, prevReleaseTagPattern?: RegExp, commitsScope?: string): ParsedCommit[] => {
	prevReleaseTagPattern ??= DEFAULT_RELEASE_TAG_PATTERN
	const rawCommits = Array.isArray(commits) ? commits : getRawCommits(commits, prevReleaseTagPattern, commitsScope)
	const parser = commitsParser ?? defaultConfig.commitsParser

	const parsedCommits = rawCommits.map(commit => parseCommit(commit, parser, prevReleaseTagPattern))
		.filter(commit => commit !== null)

	recentReleaseTag = undefined

	return parsedCommits
}

export const parseCommit = (commit: RawCommit | string, commitsParser?: CompleteCommitsParser, prevReleaseTagPattern?: RegExp): ParsedCommit | null => {
	commitsParser ??= defaultConfig.commitsParser
	prevReleaseTagPattern ??= DEFAULT_RELEASE_TAG_PATTERN

	if (typeof commit === 'string') commit = { message: commit }

	const { tagRefs, hash = getFakeCommitHash(commit.message) } = commit

	if (parsedCommitsCache.has(hash)) return parsedCommitsCache.get(hash) ?? null

	const message = commit.message.trim()
	if (!message) throw new Error(`Message is missing for commit: ${JSON.stringify(commit)}`)

	let parsedMessage
	try {
		parsedMessage = parseCommitMessage(message, commitsParser)
	} catch (error) {
		warn(`Error parsing commit '${hash}':`, (error as Error).message)
		return null
	}
	const { type, scope, subject, body, breakingChanges, footer } = parsedMessage
	const tags = tagRefs ? [...tagRefs.matchAll(commitsParser.tagPattern)].map(m => m.groups?.tag ?? '') : []

	const signers = footer
		? [...footer.matchAll(commitsParser.signerPattern)].map(m => m.groups as unknown as Contributor)
		: []

	const authors: Contributor[] = []
	const addAuthor = (contributor: Contributor): void => {
		if (!authors.some(a => a.email === contributor.email)) authors.push(contributor)
	}

	const author = commit.authorName && commit.authorEmail
		? getContributorDetails({ name: commit.authorName, email: commit.authorEmail }, signers)
		: undefined
	if (author) addAuthor(author)

	const committer = commit.committerName && commit.committerEmail
		? getContributorDetails({ name: commit.committerName, email: commit.committerEmail }, signers)
		: undefined
	if (committer) addAuthor(committer)

	const coAuthors = footer
		? [...footer.matchAll(commitsParser.coAuthorPattern)].map(m => m.groups as unknown as Contributor)
			.map(coAuthor => getContributorDetails(coAuthor, signers))
		: []
	coAuthors.forEach(coAuthor => addAuthor(coAuthor))

	const refs = parseRefs((footer ?? ''), commitsParser)

	const gpgSig = commit.gpgSigCode
		? {
			code: commit.gpgSigCode,
			label: GpgSigLabel[commit.gpgSigCode],
			keyId: commit.gpgSigKeyId,
		}
		: undefined

	let date = commit[commitsParser.dateSource === 'committerDate' ? 'committerTs' : 'authorTs']
	if (typeof date === 'string') date = formatDate(new Date(+date * 1000), commitsParser.dateFormat)

	const releaseTag = tags.find(tag => prevReleaseTagPattern.exec(tag))

	const associatedReleaseTag = releaseTag ?? recentReleaseTag
	if (associatedReleaseTag) recentReleaseTag = associatedReleaseTag

	const parsedCommit = { hash, type, scope, subject, body, breakingChanges, footer, committer, gpgSig, date, releaseTag, associatedReleaseTag,
		tags: tags.length ? tags : undefined,
		authors: authors.length ? authors : undefined,
		refs: refs.length ? refs : undefined,
	}

	if (hash && !parsedCommitsCache.has(hash)) parsedCommitsCache.set(hash, parsedCommit)

	return parsedCommit
}

const getFakeCommitHash = (message: string): string =>
	'fake_' + createHash('sha256').update(message, 'utf8').digest('hex').slice(0, 7)

const parseCommitMessage = (message: string, parser: CompleteCommitsParser): CommitMessage => {
	const [header, ...details] = message.split('\n\n')

	const headerMatch = parser.headerPattern.exec(header)
	if (!headerMatch?.groups) throw new Error(`Commit header '${header}' doesn't match expected format`)
	const { type, scope, bang, subject } = headerMatch.groups

	let breakingChanges
	const breakingChangesPart = details.find(detail => parser.breakingChangesPattern.test(detail))
	if (breakingChangesPart) {
		breakingChanges = parseBreakingChanges(breakingChangesPart, parser)
		details.splice(details.indexOf(breakingChangesPart), 1)
	} else if (bang) {
		breakingChanges = subject
	}

	const footerStart = details.findIndex(detail =>
		detail.match(parser.refActionPattern)
		?? detail.match(parser.coAuthorPattern)
		?? detail.match(parser.signerPattern))
	const [body, footer] = footerStart === -1
		? [details.join('\n\n'), '']
		: [details.slice(0, footerStart).join('\n\n'), details.slice(footerStart).join('\n\n')]

	return {
		type,
		scope: scope || undefined,
		subject,
		body: body || undefined,
		breakingChanges,
		footer: footer || undefined,
	}
}

const parseBreakingChanges = (value: string, parser: CompleteCommitsParser): ParsedCommit['breakingChanges'] => {
	const breakingChanges = parser.breakingChangesPattern.exec(value)?.groups?.content
	if (!breakingChanges) throw new Error(`Failed to extract breaking changes content from '${value}' using pattern "${parser.breakingChangesPattern}"`)

	const breakingChangeList = [...breakingChanges.matchAll(parser.breakingChangeListPattern)]

	return breakingChangeList.length
		? breakingChangeList.map(m => m[1])
		: breakingChanges
}

const getContributorDetails = (contributor: Contributor, signers: Contributor[]): Contributor => {
	const hasSignedOff = signers.some(signer => signer.email === contributor.email && signer.name === contributor.name)
	return {
		...contributor,
		hasSignedOff,
		ghLogin: contributor.name,
		ghUrl: `https://github.com/${contributor.name}`,
	}
}

const parseRefs = (value: string, parser: CompleteCommitsParser): Reference[] =>
	[...value.matchAll(parser.refPattern)].map(m => m.groups as unknown as RawReference)
		.filter(rawRef => parser.refActionPattern.test(rawRef.action))
		.flatMap(rawRef =>
			[...rawRef.labels.matchAll(parser.refLabelPattern)].map(m => ({ label: m.groups as unknown as RefLabel, raw: m.input }))
				.filter(({ label }) => !!label.number)
				.map(({ label, raw }) => ({
					raw,
					action: rawRef.action,
					owner: label.owner,
					repo: label.repo,
					number: label.number,
				})),
		)

const formatDate = (date: Date, format: string): string => {
	const pad = (num: number) => num.toString().padStart(2, '0')

	if (format === 'US') {
		return date.toLocaleString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })
	} else if (format === 'ISO') {
		return date.toISOString().split('T')[0]
	}

	const dateParts: Record<string, string> = {
		YYYY: date.getUTCFullYear().toString(),
		MM: pad(date.getUTCMonth() + 1),
		DD: pad(date.getUTCDate()),
		HH: pad(date.getUTCHours()),
		mm: pad(date.getUTCMinutes()),
		ss: pad(date.getUTCSeconds()),
	}

	return format.replace(/YYYY|MM|DD|HH|mm|ss/g, k => dateParts[k])
}