using System;
using System.Collections.Generic;
using PChecker.IO.Debugging;
using PChecker.IO.Logging;
using PChecker.SystematicTesting;
using PChecker.Testing;

namespace PChecker;

/// <summary>
/// Checker class that implements the run method which acts as the entry point into the P checker.
/// </summary>
public static class Checker
{
    /// <summary>
    /// Run the P checker for the given configuration
    /// </summary>
    /// <param name="configuration"></param>
    /// <returns>exit code for the p checker</returns>
    public static void Run(CheckerConfiguration configuration)
    {
        var logger = new ConsoleLogger();
        // if the replay option is passed then we ignore all the other options and replay the schedule
        if (configuration.SchedulingStrategy == "replay")
        {
            logger.WriteLine(
                $"Replay option is used, checker is ignoring all other parameters and using the {configuration.ScheduleFile} to replay the schedule");
            logger.WriteLine($"... Replaying {configuration.ScheduleFile}");

            switch (configuration.Mode)
            {
                case CheckerMode.BugFinding:
                {
                    var engine = TestingEngine.Create(configuration);
                    engine.Run();
                    Error.Write(logger, ConsoleColor.Yellow, engine.GetReport());
                }
                    break;
                case CheckerMode.PEx:
                    ExhaustiveEngine.Create(configuration).Run();
                    break;
                default:
                    Error.Report($"[PTool] Checker with {configuration.Mode} mode is currently unsupported.");
                    break;
            }
        }
        else
        {
            logger.WriteLine(".. Checking " + configuration.AssemblyToBeAnalyzed);

            // Creates and runs the testing process scheduler.
            switch (configuration.Mode)
            {
                case CheckerMode.BugFinding:
                    List<string> testCases = TestingProcess.FetchTestCases(configuration);
                    if (configuration.ListTestCases)
                    {
                        break;
                    }
                    foreach (string tc in testCases)
                    {
                        configuration.TestCaseName = tc;
                        TestingProcess.Create(configuration).Run();
                    }
                    break;
                case CheckerMode.PEx:
                    ExhaustiveEngine.Create(configuration).Run();
                    break;
                default:
                    Error.Report($"[PTool] Checker with {configuration.Mode} mode is currently unsupported.");
                    break;
            }

            logger.WriteLine(". Done");
        }
    }
}
