package main

import (
	"fmt"
	"os"

	"github.com/tableauio/tableau/proto/tableaupb"
	"google.golang.org/protobuf/compiler/protogen"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/descriptorpb"
)

const checkExt = "check" // protoconf file extension
const pbExt = "pb"       // protobuf file extension

func generateFileHeader(gen *protogen.Plugin, file *protogen.File, g *protogen.GeneratedFile, doNotEdit bool) {
	generateCommonHeader(gen, g, doNotEdit)
	if file.Proto.GetOptions().GetDeprecated() {
		g.P("// ", file.Desc.Path(), " is a deprecated file.")
	} else {
		g.P("// source: ", file.Desc.Path())
	}
}

func generateCommonHeader(gen *protogen.Plugin, g *protogen.GeneratedFile, doNotEdit bool) {
	heading := "// Code generated by protoc-gen-go-tableau-checker."
	if doNotEdit {
		heading += " DO NOT EDIT."
	}
	g.P(heading)
	g.P("// versions:")
	g.P("// - protoc-gen-go-tableau-checker v", version)
	g.P("// - protoc                        ", protocVersion(gen))
}

func protocVersion(gen *protogen.Plugin) string {
	v := gen.Request.GetCompilerVersion()
	if v == nil {
		return "(unknown)"
	}
	var suffix string
	if s := v.GetSuffix(); s != "" {
		suffix = "-" + s
	}
	return fmt.Sprintf("v%d.%d.%d%s", v.GetMajor(), v.GetMinor(), v.GetPatch(), suffix)
}

// Exists returns whether the given file or directory exists
func Exists(path string) (bool, error) {
	_, err := os.Stat(path)
	if err == nil {
		return true, nil
	}
	if os.IsNotExist(err) {
		return false, nil
	}
	return false, err
}

func NeedGenFile(f *protogen.File) bool {
	if !f.Generate {
		return false
	}

	opts := f.Desc.Options().(*descriptorpb.FileOptions)
	workbook := proto.GetExtension(opts, tableaupb.E_Workbook).(*tableaupb.WorkbookOptions)
	if workbook == nil {
		return false
	}

	for _, message := range f.Messages {
		opts := message.Desc.Options().(*descriptorpb.MessageOptions)
		worksheet := proto.GetExtension(opts, tableaupb.E_Worksheet).(*tableaupb.WorksheetOptions)
		if worksheet != nil {
			return true
		}
	}
	return false
}
