# Panser: A command line application for (de)serializing data #

Copyright (C) 2017 Christopher R. Field.

This file is written in ASCII-encoded plain text using the [Github Flavored Markdown (GFM)](https://help.github.com/articles/github-flavored-markdown/) lightweight markup language.

## What is Panser? ##

The Panser project is a Command-Line Interface (CLI) application for (de)serializing data formats in a UNIX, pipe-friendly manner. The project is primarily written in the [Rust](http://www.rust-lang.org) programming language. The idea is to have a single application for reading data in one format on stdin and writing the same data but in a different format to stdout. It is possible to read data from a file and write to a file, but the application is focused on creating streams of data that can be piped into a socket, such as a TCP stream. The primary motivator for the application is to read [JSON](http://www.json.org/) data and output to the [MessagePack](http://msgpack.org/index.html) format which could be used with a TCP stream to build a low-level Application Programming Interface (API) for a network-enabled application. The reverse is also a desired goal, reading in MessagePack data (binary, machine-readable) and transcoding it to JSON (text, human-readable).

After accomplishing the primary goal of transcoding between JSON and MessagePack (Msgpack) formats, additional formats were gradually added using the [serde](https://github.com/serde-rs/serde) project and related libraries. Almost all of the formats listed in the [Data Formats](https://serde.rs/#data-formats) section of the [Overview](https://serde.rs/) for the serde project are implemented. The intention is to add more formats as more crates are developed using the serde framework.

## Examples ##

The [xxd](http://linuxcommand.org/man_pages/xxd1.html) utility is used to display binary formats as a series of bytes in hex notation. 

Convert [JSON](http://www.json.org) from stdin to [MessagePack](http://msgpack.org) (Msgpack) and write to stdout. Panser converts JSON to Msgpack by default. See the `-h,--help` text for more information and options. Specifically, see the `-f,--from` and `-t,--to` help text for lists of supported formats. 

```bash
$ echo '{"bool":true}' | panser | xxd -i
  0x81, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3
```

Similarly, convert JSON from a file to Msgpack and write to stdout. If no file is specified, then input data is read continuously from stdin. The file extension is used to determine the input data format unless the `-f,--from` option is explicitly used.

```bash
$ panser file.json | xxd -i
  0x81, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3
```

Redirection can also be used.

```bash
$ panser < file.json | xxd -i
  0x81, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3
```

Convert JSON to pretty, more human readable JSON. The [Hjson](https://hjson.org) format is a more human readable format.

```bash
$ echo '{"bool":true,"number":1.234}' | panser -t Hjson
{
    "bool": true,
    "number": 1.234
}
```

Write data to a file instead of stdout. The output file will contain the binary MessagePack data.

```bash
$ echo '{"bool":true,"number":1.234}' | panser -o file.msgpack
```

Write data to a file instead of stdout, but use the file extension to determine the format. The output file will contain the binary data in the [CBOR](http://cbor.io/) format.

```bash
$ echo '{"bool":true,"number":1.234}' | panser -o file.cbor
```

Add size-based framing to the output. [Framing](https://en.wikipedia.org/wiki/Frame_(networking)) is a process to create specific frames of data from a continuous stream. It aids in buffering, reducing memory usage, and simplifying network handling code in applications. Size-based framing is prepending the total serialized data length as an unsigned 32-bit integer in Big Endian (Network Order) to the frame, or message.

```bash
$ echo '{"bool":true,"number":1.234}' | panser --sized-output | xxd -i
  0x00, 0x00, 0x00, 0x17, 0x82, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3, 0xa6,
  0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0xcb, 0x3f, 0xf3, 0xbe, 0x76, 0xc8,
  0xb4, 0x39, 0x58
```

The same can be done for input to remove the size-based framing. Note the use of the `-f` option to indicate the input format is MessagePack and _not_ JSON. The first four bytes are removed. Framing can be added or removed from any supported format, not just MessagePack.

```bash
$ echo '{"bool":true,"number":1.234}' | panser --sized-output | panser -f msgpack --sized-input | xxd -i
  0x82, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3, 0xa6, 0x6e, 0x75, 0x6d, 0x62,
  0x65, 0x72, 0xcb, 0x3f, 0xf3, 0xbe, 0x76, 0xc8, 0xb4, 0x39, 0x58
```

An alternative form of framing uses a delimiter byte between frames, or messages. Panser can handle delimiter-based framing in a similar manner to size-based framing. The `--delimited-input` and `--delimited-output` options take a value that is string representation of a single byte. A radix suffix can be added to indicate the byte notation: (b) binary, (d) decimal, (h) hexadecimal, or (o) octal. If no radix suffix is used, then hexadecimal notation is assumed. Here, the ASCII newline character (`\n`, 1010b, 10d, 0Ah, and 012o) is used to delimit the binary data.

```bash
$ echo '{"bool":true,"number":1.234}' | panser --delimited-output 0Ah | panser -f msgpack --delimited-input 0Ah | xxd -i
  0x82, 0xa4, 0x62, 0x6f, 0x6f, 0x6c, 0xc3, 0xa6, 0x6e, 0x75, 0x6d, 0x62,
  0x65, 0x72, 0xcb, 0x3f, 0xf3, 0xbe, 0x76, 0xc8, 0xb4, 0x39, 0x58
```

The delimiter-based framing can be used to create an interactive console for panser.

```
$ panser -d 0Ah -t Hjson
{"bool":true}
{
    "bool": true
}
{"bool":true,"number":1.234}
{
    "bool": true,
    "number": 1.234,
}
```

Data can be sent to a network device using the [nc](https://linux.die.net/man/1/nc) command. The JSON will be transcoded to size-based framed MessagePack and streamed to the server at the IP address and TCP port used with the `nc` command. This was actually the primary motivation for creating the `panser` application.

```bash
$ echo '{"bool":true,"numeber":1.234}' | panser --sized-output | nc 127.0.0.1 1234
```

Interestingly, panser can be used in conjunction with the [wsta](https://github.com/esphen/wsta) application to send and receive data from a web socket server.

```bash
$ ehco '{"bool":true}' | panser | wsta 127.0.0.1:1234 | panser -f msgpack -t json
{"bool":true}
```

## Install ##

### Dependencies ###

- [Cargo](https://crates.io/), v0.17 or higher
- [Rust](https://www.rust-lang.org/), v1.16 or higher

Download and install the latest version of [Rust](https://www.rust-lang.org) before proceeding. [Cargo](https://crates.io) will be installed automatically with Rust.

### Source Distribution ###

Obtain the appropriate source distribution as an archive file and run the following commands from a terminal:

    $ tar xf panser-#.#.#.tar.gz
    $ cd panser-#.#.#
    $ cargo install

where `#.#.#` is replaced with the version number of the source distribution, respectively. It might be desirable to change the install location by using the `--root` option with the `cargo install` command. See the `cargo install --help` for more information about installing a Rust binary crate using Cargo.

## Build ##

### Dependencies ###

- [Cargo](https://crates.io/), v0.17 or higher
- [Pandoc](http://pandoc.org), v1.18 or higher
- [Rust](https://www.rust-lang.org), v1.16 or higher

Download and install the latest version of [Rust](https://www.rust-lang.org) before proceeding. [Cargo](https://crates.io) will be installed automatically with Rust. [Pandoc](http://pandoc.org) is used to convert the markdown files in the `man` folder to a manpage.

### Application ###

Obtain the appropriate source distribution as an archive file and run the following commands from a terminal:

    $ tar xf panser-#.#.#.tar.gz
    $ cd panser-#.#.#
    $ cargo build

where `#.#.#` is replaced with the version number of the source distribution, respectively. The `--release` flag can be added to the cargo command to build a release application instead of a debug application. 

### Documentation ###

Obtain the appropriate source distribution as an archive file and run the following commands from a terminal:

    $ tar xf panser-#.#.#.tar.gz
    $ cd panser-#.#.#
    $ pandoc -s -t man -o man/panser.1 man/panser.1.md

where `#.#.#` is replaced with the version number of the source distribution, respectively.

## License ##

See the LICENSE file for a information about licensing and copyright.

## Contributors ##

See the AUTHORS file for information about contributors. Contributors are listed alphabetically by family name.

