/*
 * Copyright 2020, Yahoo Inc.
 * Licensed under the Apache License, Version 2.0
 * See LICENSE file in project root for terms.
 */
package com.yahoo.elide.datastores.aggregation.timegrains.serde;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

import com.yahoo.elide.core.utils.coerce.converters.Serde;
import com.yahoo.elide.datastores.aggregation.timegrains.Year;
import org.junit.jupiter.api.Test;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

public class YearSerdeTest {
    private DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy");

    @Test
    public void testDateSerialize() {
        LocalDateTime localDate = LocalDateTime.of(2020, java.time.Month.of(01), 01, 00, 00, 00);

        Year expectedDate = new Year(localDate);
        Serde serde = new Year.YearSerde();
        Object actual = serde.serialize(expectedDate);
        assertEquals("2020", actual);
    }

    @Test
    public void testDateDeserialize() {
        LocalDateTime localDate = LocalDateTime.of(2020, java.time.Month.of(01), 01, 00, 00, 00);

        Year expectedDate = new Year(localDate);
        Serde serde = new Year.YearSerde();
        Object actualDate = serde.deserialize(Year.class, "2020");
        assertEquals(expectedDate, actualDate);
    }

    @Test
    public void testDeserializeTimestamp() {
        LocalDateTime localDate = LocalDateTime.of(2020, java.time.Month.of(01), 01, 00, 00, 00);

        Year expectedDate = new Year(localDate);
        Timestamp timestamp = new Timestamp(expectedDate.getTime());
        Serde serde = new Year.YearSerde();
        Object actualDate = serde.deserialize(Year.class, timestamp);
        assertEquals(expectedDate, actualDate);
    }

    @Test
    public void testDeserializeOffsetDateTime() {
        LocalDateTime localDate = LocalDateTime.of(2020, java.time.Month.of(01), 01, 00, 00, 00);
        Year expectedDate = new Year(localDate);

        OffsetDateTime dateTime = OffsetDateTime.of(2020, 1, 1, 0, 0, 0, 0, ZoneOffset.UTC);
        Serde serde = new Year.YearSerde();
        Object actualDate = serde.deserialize(Year.class, dateTime);
        assertEquals(expectedDate, actualDate);
    }

    @Test
    public void testDeserializeDateInvalidFormat() {

        String dateInString = "January";
        Serde serde = new Year.YearSerde();
        assertThrows(DateTimeParseException.class, () ->
            serde.deserialize(Year.class, dateInString)
        );
    }
}
