
# Test cases for jump table resolver

import os
import logging

import angr


test_location = os.path.join(os.path.dirname(os.path.realpath(__file__)), '..', '..', 'binaries', 'tests')
l = logging.getLogger("angr.tests.test_jumptables")


class J:
    def __init__(self, baddr, taddr, ents):
        self.block_addr = baddr
        self.table_addr = taddr
        self.entries = ents

    def __hash__(self):
        return hash((self.block_addr, self.table_addr, len(self.entries)))


def compare(jump_tables, groundtruth):
    for j in groundtruth:
        assert j.block_addr in jump_tables, "Jump table @ block %#x is not found in CFG." % j.block_addr
        jumptable_addr = jump_tables[j.block_addr].jumptable_addr
        assert j.table_addr == jumptable_addr, \
                                "Mismatch jump table addresses (expecting %s, got %s)." % (
                                    ("%#x" % j.table_addr) if j.table_addr is not None else "None",
                                    ("%#x" % jumptable_addr) if jumptable_addr is not None else "None",
                                )
        assert j.entries == jump_tables[j.block_addr].jumptable_entries


def test_amd64_dir_gcc_O0():
    p = angr.Project(os.path.join(test_location, "x86_64", "dir_gcc_-O0"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x40404c, 0x41b0a0, [0x404741, 0x404737, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4043ef, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x404280, 0x40429d, 0x4042b6, 0x4042c5, 0x4047ad, 0x4042d1, 0x4042e0, 0x4042ec, 0x40430a, 0x4047ad, 0x4047ad, 0x40431e, 0x4047ad, 0x40432d, 0x4047ad, 0x4047ad, 0x404341, 0x404355, 0x404361, 0x404377, 0x4043c3, 0x4047ad, 0x4047ad, 0x4043d9, 0x4047ad, 0x40472b, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x404058, 0x404067, 0x40407b, 0x40408a, 0x4047ad, 0x404096, 0x4040fd, 0x404113, 0x404147, 0x4047ad, 0x404153, 0x40415f, 0x40416e, 0x40417d, 0x404193, 0x4041a9, 0x4041b8, 0x4041c4, 0x4041d0, 0x4041dc, 0x4041f2, 0x404201, 0x404217, 0x404271, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x4047ad, 0x40440c, 0x40468c, 0x404543, 0x4042fb, 0x4040ee, 0x4044e6, 0x404526, 0x40449a, 0x404418, 0x4045ea, 0x40462a, 0x404680, 0x4046f7, 0x404453, 0x4044a6, 0x404670]),
        J(0x404d53, 0x41b938, [0x404d60, 0x404de7, 0x404ea3, 0x404f00, 0x404fa5]),
        J(0x404df6, 0x41b960, [0x404e61, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e02, 0x404e02, 0x404e02, 0x404e02, 0x404e02, 0x404e02, 0x404e02, 0x404e02, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e55, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e18, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e5b, 0x404e6a, 0x404e25, 0x404e2b, 0x404e6a, 0x404e6a, 0x404e31, 0x404e37, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e3d, 0x404e6a, 0x404e6a, 0x404e6a, 0x404e43, 0x404e6a, 0x404e49, 0x404e6a, 0x404e4f, 0x404e6a, 0x404e18]),
        J(0x404f12, 0x41bd28, [0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f1e, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f62, 0x404f62, 0x404f62, 0x404f62, 0x404f62, 0x404f62, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f84, 0x404f40, 0x404f40, 0x404f40, 0x404f40, 0x404f40, 0x404f40]),
        J(0x4051b0, 0x41bf58, [0x405412, 0x4051bd, 0x40528b, 0x4052c0, 0x4053b5, 0x405412, 0x40541d]),
        J(0x405b34, 0x41c088, [0x405b9f, 0x405b64, 0x405b4c, 0x405b9f, 0x405b58, 0x405b9f, 0x405b40, 0x405b9f, 0x405b7c, 0x405b9f, 0x405b70, 0x405b9f, 0x405b88, 0x405b9f, 0x405b94]),
        J(0x408899, 0x41c380, [0x408942, 0x4088a5, 0x4088f2, 0x408917, 0x408936]),
        J(0x410b5b, 0x41e410, [0x410cc9, 0x410c72, 0x410c85, 0x410c6e, 0x410c76, 0x410b79, 0x410b6b, 0x410bba, 0x410bc7, 0x410bc7, 0x410bc7]),
        J(0x410da4, 0x41e468, [0x410db0, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x41103c, 0x411042, 0x41105a, 0x41104e, 0x411060, 0x411048, 0x411054, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411195, 0x411102, 0x411102, 0x411102, 0x4110f7, 0x411102, 0x41155c, 0x411102, 0x41111e, 0x411102, 0x411102, 0x411102, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x411102, 0x411102, 0x411102, 0x411102, 0x410ef9, 0x411195, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x411102, 0x411066, 0x41155c, 0x411102, 0x41155c, 0x411102, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x41155c, 0x4110c5, 0x411102, 0x4110c5, 0x4110f7]),
        J(0x410f84, 0x41e860, [0x410f90, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x410f90, 0x410f90, 0x410f90, 0x41102d, 0x41102d, 0x41102d, 0x410f90, 0x41102d, 0x410f90, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x41102d, 0x410f90, 0x410f90, 0x410f90]),
        J(0x411300, 0x41e950, [0x4117b2, 0x4117b2, 0x41130c, 0x4117b2, 0x41130c, 0x4117b2, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x41130c, 0x4117b2]),
        J(0x412aae, 0x41ea88, [0x414b93, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x412abe, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414905, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x412c2f, 0x412c85, 0x413105, 0x4131d1, 0x414be3, 0x413bb4, 0x414295, 0x413bd1, 0x413bfd, 0x414be3, 0x414be3, 0x414be3, 0x413cbe, 0x413d2f, 0x414be3, 0x413ed8, 0x414be3, 0x413f09, 0x413f19, 0x4140b6, 0x41423e, 0x414295, 0x4145b5, 0x4140a4, 0x414660, 0x414749, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x414be3, 0x412c02, 0x412c5c, 0x412cb2, 0x4131ee, 0x413217, 0x414be3, 0x414295, 0x412c5c, 0x414be3, 0x413c79, 0x413c25, 0x413c51, 0x413cea, 0x413daa, 0x414be3, 0x413ee9, 0x414be3, 0x412e7d, 0x413f44, 0x4140c6, 0x4141f4, 0x414be3, 0x414634, 0x4131bf, 0x4146b6, 0x41495e]),
        J(0x415060, 0x41f0d8, [0x415073, 0x415078, 0x4150a8, 0x4150e3, 0x41512c, 0x415183, 0x4151ef, 0x415269, 0x4152f9, 0x415397]),
        J(0x415faa, 0x41f268, [0x415fcf, 0x4160ee, 0x4160ee, 0x415ff4, 0x4160ee, 0x416012, 0x4160ee, 0x4160ee, 0x4160ee, 0x416030, 0x4160ee, 0x41604e, 0x4160ee, 0x4160ee, 0x41606c, 0x4160ee, 0x4160ee, 0x4160ee, 0x416087, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160b8, 0x4160d3, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x415fb6, 0x415fe8, 0x4160ee, 0x4160ee, 0x4160ee, 0x416012, 0x4160ee, 0x4160ee, 0x4160ee, 0x416030, 0x4160ee, 0x41604e, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x4160ee, 0x416087, 0x4160ee, 0x4160ee, 0x4160a2]),
        J(0x416586, 0x41f4c8, [0x4165ab, 0x4166ca, 0x4166ca, 0x4165d0, 0x4166ca, 0x4165ee, 0x4166ca, 0x4166ca, 0x4166ca, 0x41660c, 0x4166ca, 0x41662a, 0x4166ca, 0x4166ca, 0x416648, 0x4166ca, 0x4166ca, 0x4166ca, 0x416663, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x416694, 0x4166af, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x416592, 0x4165c4, 0x4166ca, 0x4166ca, 0x4166ca, 0x4165ee, 0x4166ca, 0x4166ca, 0x4166ca, 0x41660c, 0x4166ca, 0x41662a, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x4166ca, 0x416663, 0x4166ca, 0x4166ca, 0x41667e]),
        J(0x4177ec, 0x41f6b0, [0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177ff, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8, 0x4177f8]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_amd64_chmod_gcc_O1():
    p = angr.Project(os.path.join(test_location, "x86_64", "chmod_gcc_-O1"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x402322, 0x4095a0, [0x40246d, 0x4024bf, 0x4023d9, 0x4024bf, 0x40288b, 0x4023a5, 0x4024bf, 0x4024bf, 0x40232c, 0x4024bf, 0x4024bf, 0x402422]),
        J(0x402fc1, 0x409778, [0x402ff2, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402ffd, 0x402fcb, 0x402fe0, 0x402fe9, 0x402ffd, 0x402ffd, 0x402fd4, 0x402fda]),
        J(0x40360a, 0x409920, [0x403614, 0x403f96, 0x403723, 0x40363c, 0x403f66, 0x403769, 0x40367d, 0x40367d, 0x40367d]),
        J(0x4038d0, 0x409968, [0x4038ef, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403a91, 0x4038e5, 0x403a56, 0x403a76, 0x403a5d, 0x403a4f, 0x4038db, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403b48, 0x403ad2, 0x403ad2, 0x403ad2, 0x403ac9, 0x403ad2, 0x403df3, 0x403ad2, 0x403af0, 0x403ad2, 0x403ad2, 0x403ad2, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403ad2, 0x403ad2, 0x403ad2, 0x403ad2, 0x40397c, 0x403b48, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403ad2, 0x403a64, 0x403df3, 0x403ad2, 0x403df3, 0x403ad2, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403aa4, 0x403ad2, 0x403aa4, 0x403ac9]),
        J(0x4039dc, 0x409D60, [0x4039e6, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x4039e6, 0x4039e6, 0x4039e6, 0x403df3, 0x403df3, 0x403df3, 0x4039e6, 0x403df3, 0x4039e6, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x403df3, 0x4039e6, 0x4039e6, 0x4039e6]),
        J(0x403c83, 0x409E50, [0x403f0d, 0x403f0d, 0x403c8d, 0x403f0d, 0x403c8d, 0x403f0d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403c8d, 0x403f0d]),
        J(0x4048ca, 0x40A250, [0x4048d2, 0x4048d7, 0x40490b, 0x404946, 0x404988, 0x4049d8, 0x404a2a, 0x404a8c, 0x404af7, 0x404b73]),
        J(0x406325, 0x40A310, [0x405fd8, 0x405f8c, 0x405feb, 0x405fd8, 0x405f79, 0x405fd8, 0x406300, 0x405fd8, 0x405fb2, 0x405fd8, 0x405f9f, 0x405fd8, 0x405fc5]),
        J(0x406399, 0x40A378, [0x405fdf, 0x405f93, 0x406374, 0x405fdf, 0x405f80, 0x405fdf, 0x406331, 0x405fdf, 0x405fb9, 0x405fdf, 0x405fa6, 0x405fdf, 0x405fcc]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_amd64_cfgswitches_gcc():
    p = angr.Project(os.path.join(test_location, "x86_64", "cfg_switches"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x4006e1, 0x400a40, [0x40075b, 0x4006EE, 0x4006FA, 0x4006EE, 0x40070B, 0x4006EE, 0x40071C, 0x4006EE, 0x40072D, 0x4006EE, 0x40073E, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40075b, 0x40074F]),
        J(0x4005bc, 0x4008c0, [0x4006b0, 0x4006b0, 0x4005C9, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4005D8, 0x4006b0, 0x4005E7, 0x4006b0, 0x4005F6, 0x400605, 0x400614, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x400623, 0x4006b0, 0x400632, 0x4006b0, 0x4006b0, 0x4006b0, 0x40063E, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x4006b0, 0x40064A]),
        J(0x40053a, 0x4007f8, [0x40058d, 0x400547, 0x40058d, 0x400552, 0x40058d, 0x40055D, 0x40058d, 0x400568, 0x40058d, 0x400573, 0x40058d, 0x40058d, 0x400580]),
        J(0x40065a, 0x4009f8, [0x4006a3, 0x400667, 0x400673, 0x40067f, 0x40068b, 0x400697]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_i386_cfgswitches_gcc_O0():
    p = angr.Project(os.path.join(test_location, "i386", "cfg_switches_O0"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x8048432, 0x8048770, [0x8048487, 0x8048441, 0x8048487, 0x804844c, 0x8048487, 0x8048457, 0x8048487, 0x8048462, 0x8048487, 0x804846d, 0x8048487, 0x8048487, 0x804847a]),
        J(0x80484b4, 0x8048804, [0x8048612, 0x8048612, 0x80484c3, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x80484d8, 0x8048612, 0x80484ed, 0x8048612, 0x8048502, 0x8048517, 0x804852c, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048541, 0x8048612, 0x8048556, 0x8048612, 0x8048612, 0x8048612, 0x804856b, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048612, 0x8048580]),
        J(0x8048596, 0x80488a0, [0x80485ff, 0x80485a5, 0x80485b7, 0x80485c9, 0x80485db, 0x80485ed]),
        J(0x8048654, 0x80488cc, [0x80486d5, 0x8048662, 0x8048671, 0x8048662, 0x8048682, 0x8048662, 0x8048693, 0x8048662, 0x80486a4, 0x8048662, 0x80486b5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486d5, 0x80486c6]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_i386_cfgswitches_gcc_O1():
    p = angr.Project(os.path.join(test_location, "i386", "cfg_switches_O1"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x804842f, 0x8048700, [0x804845a, 0x8048436, 0x804845a, 0x804843c, 0x804845a, 0x8048442, 0x804845a, 0x8048448, 0x804845a, 0x804844e, 0x804845a, 0x804845a, 0x8048454]),
        J(0x804847e, 0x8048734, [0x80485d5, 0x80485d5, 0x8048485, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x804849f, 0x80485d5, 0x80484b1, 0x80485d5, 0x80484c3, 0x80484d5, 0x80484e7, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80484f9, 0x80485d5, 0x804850b, 0x80485d5, 0x80485d5, 0x80485d5, 0x8048520, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x80485d5, 0x8048535]),
        J(0x804854c, 0x80487d0, [0x80485c0, 0x8048557, 0x804856c, 0x8048581, 0x8048596, 0x80485ab]),
        J(0x8048610, 0x80487e8, [0x8048626, 0x8048626, 0x8048617, 0x8048626, 0x8048635, 0x8048626, 0x8048646, 0x8048626, 0x8048657, 0x8048626, 0x8048668]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_i386_cfgswitches_gcc_O2():
    p = angr.Project(os.path.join(test_location, "i386", "cfg_switches_O2"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x8048336, 0x8048888, [0x804834b, 0x804834b, 0x804833d, 0x804834b, 0x8048377, 0x804834b, 0x8048387, 0x804834b, 0x8048357, 0x804834b, 0x8048367]),
        J(0x80484c9, 0x80487a0, [0x8048530, 0x80484e0, 0x8048530, 0x80484f0, 0x8048530, 0x8048500, 0x8048530, 0x8048510, 0x8048530, 0x8048520, 0x8048530, 0x8048530, 0x80484d0]),
        J(0x804855e, 0x80487d4, [0x8048670, 0x8048670, 0x8048590, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x80485b0, 0x8048670, 0x80485c8, 0x8048670, 0x80485e0, 0x80485f8, 0x8048610, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048628, 0x8048670, 0x8048640, 0x8048670, 0x8048670, 0x8048670, 0x8048658, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048670, 0x8048568]),
        J(0x8048583, 0x8048870, [0x8048700, 0x80486e8, 0x80486d0, 0x80486b8, 0x80486a0, 0x8048688]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_kprca_00009():
    p = angr.Project(os.path.join(test_location, "cgc", "KPRCA_00009"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x804919a, 0x804b890, [0x80491a4, 0x80494f9, 0x80494fe, 0x8049503, 0x8049508, 0x804950d, 0x8049512, 0x804951a]),
        J(0x8049295, 0x804b870, [0x804929f, 0x8049561, 0x8049566, 0x804956b, 0x8049570, 0x8049575, 0x804957a, 0x8049582]),
        J(0x8048282, 0x804b638, [0x8048289, 0x8048cc9, 0x80483c3, 0x80485f6, 0x80483d5, 0x8048cc9, 0x80483f0, 0x8048cc9, 0x8048436, 0x80485ad, 0x8048448, 0x804845a, 0x8048487, 0x80484a2, 0x80484ca, 0x8048cc9, 0x8048520, 0x8048cc9, 0x804852f, 0x8048cc9, 0x8048541, 0x8048cc9, 0x804855c, 0x8048cc9, 0x80485a0, 0x80485cb, 0x80485ed, 0x8048cc9, 0x8048611, 0x8048cc9, 0x804864d, 0x8048cc9, 0x80486a3, 0x8048cc9, 0x80486c1, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x80486d4, 0x8048cc9, 0x80486ed, 0x8048714, 0x8048726, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x8048741, 0x8048cc9, 0x804876d, 0x8048cc9, 0x804878d, 0x804879d, 0x8048cc9, 0x8048cc9, 0x80487a4, 0x8048cc9, 0x80487bc, 0x80487e3, 0x80487fa, 0x8048817, 0x8048cc9, 0x8048cc9, 0x804882b]),
        J(0x8049211, 0x804b850, [0x804921b, 0x8049533, 0x8049538, 0x804953d, 0x8049542, 0x8049547, 0x804954c, 0x8049554]),
        J(0x80492bd, 0x804b830, [0x80492c7, 0x80495a1, 0x80495a6, 0x80495ab, 0x80495b0, 0x80495b5, 0x80495ba, 0x80495c2]),
        J(0x8048cd8, 0x804b77c, [0x8048cdf, 0x8048cfd, 0x8048dea, 0x8048e11, 0x8048e38]),
        J(0x8048d48, 0x804b800, [0x8048d52, 0x8048f12, 0x8048f17, 0x8048f1c]),
        J(0x8049ebc, 0x804ba58, [0x8049f0a, 0x8049f68, 0x8049ec3, 0x8049f68, 0x8049f68, 0x8049f18, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f1e, 0x8049f24, 0x8049f2a]),
        J(0x8048d20, 0x804b820, [0x8048d2a, 0x8048eee, 0x8048ef3, 0x8048ef8]),
        J(0x8049eec, 0x804b9fc, [0x8049ef3, 0x8049f68, 0x8049f68, 0x8049f30, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f0a, 0x8049f36, 0x8049f3c, 0x8049f42, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f68, 0x8049f48, 0x8049f68, 0x8049f58, 0x8049f68, 0x8049f5e, 0x8049f68, 0x8049f64]),
        J(0x8048d70, 0x804b810, [0x8048d7a, 0x8048fc9, 0x8048fd1, 0x8048fd9]),
        J(0x8048d9b, 0x804b7f0, [0x8048da5, 0x8048fe1, 0x8048fe9, 0x8048ff1]),
        J(0x8048dd8, 0x804b950, [0x8048de2, 0x804913c, 0x8049141, 0x8049146, 0x804914b, 0x8049150, 0x8049155, 0x804915d]),
        J(0x8049169, 0x804b970, [0x8049173, 0x80491ac, 0x80491b1, 0x80491b6, 0x80491bb, 0x80491c0, 0x80491c5, 0x80491cd]),
        J(0x80489da, 0x804b734, [0x80489e1, 0x8048b46, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x8048cc9, 0x8048b5e, 0x8048b6d]),
        J(0x8048e85, 0x804b910, [0x8048e8f, 0x8049223, 0x8049228, 0x804922d, 0x8049232, 0x8049237, 0x804923c, 0x8049244]),
        J(0x8048edc, 0x804b8f0, [0x8048ee6, 0x8049387, 0x804938c, 0x8049391, 0x8049396, 0x804939b, 0x80493a0, 0x80493a8]),
        J(0x8048f81, 0x804b8d0, [0x8048f8b, 0x804942b, 0x8049430, 0x8049435, 0x804943a, 0x804943f, 0x8049444, 0x804944c]),
        J(0x8049327, 0x804b930, [0x8049331, 0x8049600, 0x8049605, 0x804960a]),
        J(0x804902b, 0x804b8b0, [0x8049035, 0x8049482, 0x8049487, 0x804948c, 0x8049491, 0x8049496, 0x804949b, 0x80494a3]),
        J(0x8049375, 0x804b940, [0x804937f, 0x8049627, 0x804962c, 0x8049631]),
        J(0x8048e49, 0x804b790, [0x8048e53, 0x80490fc, 0x8049101, 0x8049106, 0x804910b, 0x8049110, 0x8049115, 0x804911d]),
        J(0x8048e26, 0x804b7b0, [0x8048e30, 0x804908f, 0x8049094, 0x8049099, 0x804909e, 0x80490a3, 0x80490a8, 0x80490b0]),
        J(0x8048dff, 0x804b7d0, [0x8048e09, 0x804903d, 0x8049042, 0x8049047, 0x804904c, 0x8049051, 0x8049056, 0x804905e]),
        J(0x8048473, 0x804b75c, [0x804847f, 0x8048a11, 0x8048a16, 0x8048a1b, 0x8048a20, 0x8048a25, 0x8048a2a, 0x8048a32]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_armel_cfgswitches_gcc():
    p = angr.Project(os.path.join(test_location, "armel", "cfg_switches"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x10734, 0x10748, [0x107FC, 0x10808, 0x107FC, 0x10818, 0x107FC, 0x10828, 0x107FC, 0x10838, 0x107FC, 0x10848, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10864, 0x10858]),
        J(0x10434, 0x10458, [0x10488, 0x104e8, 0x10498, 0x104e8, 0x104A8, 0x104e8, 0x104B8, 0x104e8, 0x104C8, 0x104e8, 0x104e8, 0x104D8]),
        J(0x10524, 0x10538, [0x105CC, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x105D8, 0x106b4, 0x105E4, 0x106b4, 0x105F0, 0x105FC, 0x10608, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x10614, 0x106b4, 0x10620, 0x106b4, 0x106b4, 0x106b4, 0x1062C, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x106b4, 0x10638]),
        J(0x10640, 0x10654, [0x10668, 0x10674, 0x10680, 0x1068c, 0x10698]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_armel_lwip_tcpecho_bm():
    p = angr.Project(os.path.join(test_location, "armel", "lwip_tcpecho_bm.elf"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x14fb, 0x14fe, [0x1519, 0x1519, 0x152f, 0x1541, 0x1553, 0x159d, 0x1565, 0x156d, 0x1575, 0x159d, 0x159d, 0x157d, 0x1585, 0x158d, 0x1595, 0x159d, 0x159d, 0x159d, 0x15a1, 0x1513]),
        J(0x2d45, 0x2d48, [0x2d8d, 0x2d4f, 0x2d91, 0x2d7f, 0x2d7f]),
        J(0x2e8d, 0x2e90, [0x2eb3, 0x2e97, 0x2eb7, 0x2ea7, 0x2ea7]),
        J(0x2e9f, 0x2ea2, [0x2f93, 0x2fc9, 0x2fc9, 0x2ebb]),
        J(0x4b63, 0x4b66, [0x4b91, 0x4c81, 0x4d53, 0x4d5b, 0x4ded, 0x4d53, 0x4e21, 0x4e5f]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_s390x_cfgswitches():
    p = angr.Project(os.path.join(test_location, "s390x", "cfg_switches"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    all_jumptables = {
        J(0x400638, 0x400c00, [0x40064e, 0x40065c, 0x400692, 0x40065c, 0x4006a4, 0x40065c, 0x40066e, 0x40065c, 0x400680]),
        J(0x4007d4, 0x400a50, [0x4007ea, 0x4007ca, 0x4007f4, 0x4007ca, 0x4007fe, 0x4007ca, 0x400808, 0x4007ca, 0x400812, 0x4007ca, 0x4007ca, 0x40081c]),
        J(0x400872, 0x400ab0, [0x4008ae, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x4008be, 0x400854, 0x4008ce, 0x400854, 0x4008de, 0x4008ee, 0x4008fe, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x40090e, 0x400854, 0x40091e, 0x400854, 0x400854, 0x400854, 0x40092e, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400854, 0x400888]),
        J(0x40093e, 0x400bd8, [0x400984, 0x400974, 0x400964, 0x400954, 0x400994]),
    }

    compare(cfg.jump_tables, all_jumptables)


def test_arm_libsoap():

    # This is the ADDLS type of jump table (IndirectJumpType.JumpTable_AddressComputed) where no actual table is used
    # libsoap.so seems to be compiled from gSOAP, which is an open-source product

    p = angr.Project(os.path.join(test_location, "armel", "libsoap.so"), auto_load_libs=False)
    cfg = p.analyses.CFGFast(data_references=True)

    all_jumptabes = {
        J(0x411c8c, None, [0x411c9c, 0x411ca0, 0x411ca4, 0x411ca8, 0x411cac, 0x411cb0, 0x411cb4, 0x411cb8, 0x411cbc, 0x411cc0, 0x411cc4, 0x411cc8, 0x411ccc, 0x411cd0, 0x411cd4, 0x411cd8, 0x411cdc, 0x411ce0, 0x411ce4, 0x411ce8, 0x411cec, 0x411cf0, 0x411cf4, 0x411cf8, 0x411cfc, 0x411d00, 0x411d04, 0x411d08, 0x411d0c, 0x411d10, 0x411d14, 0x411d18, 0x411d1c, 0x411d20, 0x411d24, 0x411d28, 0x411d2c, 0x411d30, 0x411d34, 0x411d38, 0x411d3c, 0x411d40, 0x411d44, 0x411d48, 0x411d4c, 0x411d50, 0x411d54, 0x411d58, 0x411d5c, 0x411d60, 0x411d64, 0x411d68, 0x411d6c, 0x411d70]),
        J(0x411f54, None, [0x411f64, 0x411f68, 0x411f6c, 0x411f70, 0x411f74, 0x411f78, 0x411f7c, 0x411f80, 0x411f84, 0x411f88, 0x411f8c, 0x411f90, 0x411f94, 0x411f98, 0x411f9c, 0x411fa0, 0x411fa4, 0x411fa8, 0x411fac, 0x411fb0, 0x411fb4, 0x411fb8, 0x411fbc, 0x411fc0, 0x411fc4, 0x411fc8, 0x411fcc, 0x411fd0, 0x411fd4, 0x411fd8, 0x411fdc, 0x411fe0, 0x411fe4, 0x411fe8, 0x411fec, 0x411ff0, 0x411ff4, 0x411ff8, 0x411ffc, 0x412000, 0x412004, 0x412008, 0x41200c, 0x412010, 0x412014, 0x412018, 0x41201c, 0x412020, 0x412024, 0x412028, 0x41202c, 0x412030, 0x412034, 0x412038]),
        J(0x41b0b4, None, [0x41b0c4, 0x41b0c8, 0x41b0cc, 0x41b0d0, 0x41b0d4]),
        # 0x41d0e8 and 0x41d0fc are the same jump table - they appear twice because the CFG is not normalized (the two
        # blocks 0x41d0e8 and 0x41d0fc overlap and end at the same instruction)
        J(0x41d0e8, None, [0x41d10c, 0x41d110, 0x41d114, 0x41d118, 0x41d11c, 0x41d120, 0x41d124, 0x41d128, 0x41d12c, 0x41d130, 0x41d134, 0x41d138, 0x41d13c, 0x41d140, 0x41d144, 0x41d148, 0x41d14c, 0x41d150, 0x41d154, 0x41d158, 0x41d15c, 0x41d160, 0x41d164, 0x41d168, 0x41d16c, 0x41d170, 0x41d174, 0x41d178, 0x41d17c]),
        J(0x41d0fc, None, [0x41d10c, 0x41d110, 0x41d114, 0x41d118, 0x41d11c, 0x41d120, 0x41d124, 0x41d128, 0x41d12c, 0x41d130, 0x41d134, 0x41d138, 0x41d13c, 0x41d140, 0x41d144, 0x41d148, 0x41d14c, 0x41d150, 0x41d154, 0x41d158, 0x41d15c, 0x41d160, 0x41d164, 0x41d168, 0x41d16c, 0x41d170, 0x41d174, 0x41d178, 0x41d17c]),
        J(0x41d9d0, None, [0x41d9e0, 0x41d9e4, 0x41d9e8, 0x41d9ec, 0x41d9f0, 0x41d9f4, 0x41d9f8, 0x41d9fc, 0x41da00, 0x41da04, 0x41da08, 0x41da0c, 0x41da10, 0x41da14, 0x41da18, 0x41da1c, 0x41da20, 0x41da24, 0x41da28, 0x41da2c, 0x41da30, 0x41da34, 0x41da38, 0x41da3c, 0x41da40, 0x41da44, 0x41da48, 0x41da4c, 0x41da50, 0x41da54, 0x41da58, 0x41da5c, 0x41da60, 0x41da64, 0x41da68, 0x41da6c, 0x41da70, 0x41da74, 0x41da78, 0x41da7c, 0x41da80, 0x41da84, 0x41da88, 0x41da8c, 0x41da90, 0x41da94, 0x41da98, 0x41da9c, 0x41daa0, 0x41daa4, 0x41daa8, 0x41daac, 0x41dab0, 0x41dab4]),
        J(0x41e070, None, [0x41e080, 0x41e084, 0x41e088, 0x41e08c, 0x41e090, 0x41e094, 0x41e098, 0x41e09c, 0x41e0a0, 0x41e0a4, 0x41e0a8, 0x41e0ac, 0x41e0b0, 0x41e0b4, 0x41e0b8, 0x41e0bc, 0x41e0c0, 0x41e0c4, 0x41e0c8, 0x41e0cc, 0x41e0d0, 0x41e0d4, 0x41e0d8, 0x41e0dc, 0x41e0e0, 0x41e0e4, 0x41e0e8, 0x41e0ec, 0x41e0f0, 0x41e0f4, 0x41e0f8, 0x41e0fc, 0x41e100, 0x41e104, 0x41e108, 0x41e10c, 0x41e110, 0x41e114, 0x41e118, 0x41e11c, 0x41e120, 0x41e124, 0x41e128, 0x41e12c, 0x41e130, 0x41e134, 0x41e138, 0x41e13c, 0x41e140, 0x41e144, 0x41e148, 0x41e14c, 0x41e150, 0x41e154, 0x41e158, 0x41e15c, 0x41e160, 0x41e164, 0x41e168, 0x41e16c, 0x41e170, 0x41e174, 0x41e178, 0x41e17c, 0x41e180, 0x41e184, 0x41e188, 0x41e18c, 0x41e190]),
    }

    compare(cfg.jump_tables, all_jumptabes)

    assert 0x41d0e8 in cfg.model.jump_tables
    # normalizing the CFG should remove 0x41d0e8
    cfg.normalize()
    assert 0x41d0e8 not in cfg.model.jump_tables


#
# The jump table should be occupied and marked as data
#

def test_jumptable_occupied_as_data():

    # GitHub issue #1671

    p = angr.Project(os.path.join(test_location, "i386", "windows", "printenv.exe"), auto_load_libs=False)
    cfg = p.analyses.CFGFast()

    # it has a jump table at 0x402e4d with 10 entries
    assert 0x402e4d in cfg.indirect_jumps
    assert cfg.indirect_jumps[0x402e4d].jumptable is True
    assert cfg.indirect_jumps[0x402e4d].jumptable_addr == 0x402e54
    assert cfg.indirect_jumps[0x402e4d].jumptable_size == 4 * 10
    assert cfg.indirect_jumps[0x402e4d].jumptable_entry_size == 4

    # 40 bytes starting at 0x402e4d should be marked as "data"
    for addr in range(0x402e54, 0x402e54 + 40, 4):
        assert cfg._seg_list.occupied_by_sort(addr) == "data"

    # node 0x402e4d should have 10 successors
    assert len(cfg.model.get_any_node(0x402e4d).successors) == 10


if __name__ == "__main__":
    test_amd64_chmod_gcc_O1()
    test_amd64_dir_gcc_O0()
    test_amd64_cfgswitches_gcc()
    test_i386_cfgswitches_gcc_O0()
    test_i386_cfgswitches_gcc_O1()
    test_i386_cfgswitches_gcc_O2()
    test_kprca_00009()
    test_armel_cfgswitches_gcc()
    test_armel_lwip_tcpecho_bm()
    test_s390x_cfgswitches()
    test_arm_libsoap()
    test_jumptable_occupied_as_data()
