/*
 * Decompiled with CFR 0.152.
 */
package androidx.test.tools.crawler.postprocessing.logcat;

import androidx.test.tools.crawler.postprocessing.logcat.LogLine;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

public final class LogcatParser {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormat.forPattern("yyyyMM-dd HH:mm:ss.SSS");
    private static final Pattern LOG_PREFIX_PATTERN = Pattern.compile("^((\\d{2}-\\d{2} \\d{2}:\\d{2}:\\d{2}.\\d{3}):?\\s(\\w)\\/(.+?)\\((\\d+)\\):\\s).*");
    private static final Pattern STACK_FRAME_PATTERN = Pattern.compile("\\sat ([^ :]+)\\((Unknown Source|Native Method|[^ :]+:\\d+)\\).*");
    private static final Pattern CAUSED_BY_PATTERN = Pattern.compile("Caused by:\\s+(.+).*");
    private static final Pattern NATIVE_CRASH_PATTERN = Pattern.compile("pid: (\\d+), tid: \\d+, name: .+\\s+>>> ([^ :]+) <<<.*");
    private static final Pattern TIME_PATTERN = Pattern.compile("((?<min>\\d+)m)?((?<sec>\\d+)s)?(?<ms>\\d+)ms");
    private static final Pattern START_TIME_DEFAULT_PATTERN = Pattern.compile("Displayed (\\S+)/(\\S+): \\+(?<defaultTime>[\\dms]*\\d+ms)( \\(total \\+(?<totalTime>[\\dms]*\\d+ms)\\))?");
    private static final Pattern START_TIME_FULLY_DRAWN_PATTERN = Pattern.compile("Fully drawn (\\S+)/(\\S+): \\+(?<defaultTime>[\\dms]*\\d+ms)( \\(total \\+(?<totalTime>[\\dms]*\\d+ms)\\))?");
    private static final Pattern ROBO_TEST_COMPLETE_PATTERN = Pattern.compile(".*### Robo Test Complete ###.*");
    private static final Pattern SUPPRESSED_CRASH_PATTERN = Pattern.compile("Suppressing debuggerd output because prctl\\(PR_GET_DUMPABLE\\)==0");
    private static final Pattern PROCESS_DIED_PATTERN = Pattern.compile("Process (?<package>[^ :]+) \\(pid (?<pid>\\d+)\\) has died");
    private static final Pattern LOW_MEMORY_KILLER_PATTERN = Pattern.compile("Killing '(?<package>[^ :]+)' \\((?<pid>\\d+)\\), uid \\d+, adj 0.*");
    private static final Pattern NON_SDK_API_USED_VIOLATION_PATTERN = Pattern.compile("StrictMode policy violation: android.os.strictmode.NonSdkApiUsedViolation: (?<apisig>[^\\s]+)");
    private static final Pattern START_PROC_PATTERN = Pattern.compile("Start proc (?<pid>\\d+):(?<package>[^/]+)/.*");
    private static final Pattern CRASH_RUNNING_INSTRUMENTATION_PATTERN = Pattern.compile("Crash of app (?<package>[^ :]+) running instrumentation .*");
    private static final Pattern PROCESS_PATTERN = Pattern.compile("Process: (?<package>[^ :]+), PID: (?<pid>\\d+)");
    private static final Pattern TEST_CASE_STARTED = Pattern.compile("^started: (\\S+)\\((\\S+)\\)$");
    private static final Pattern TEST_CASE_FINISHED = Pattern.compile("^finished: (\\S+)\\((\\S+)\\)$");
    private static final String JUNIT_CLASS = "junit.framework.TestSuite";
    private static final Pattern SEPARATOR_PATTERN = Pattern.compile("^[-]+ beginning of [A-Za-z/]+");
    private final BufferedReader logcatReader;
    private List<String> unparseableLines = new ArrayList<String>();
    private static final ImmutableMap<String, LogLine.LogSeverity> allowedSeverityNames = ImmutableMap.builder().put("V", LogLine.LogSeverity.VERBOSE).put("D", LogLine.LogSeverity.DEBUG).put("I", LogLine.LogSeverity.INFO).put("W", LogLine.LogSeverity.WARNING).put("E", LogLine.LogSeverity.ERROR).put("F", LogLine.LogSeverity.FATAL).put("S", LogLine.LogSeverity.FATAL).build();

    public LogcatParser(BufferedReader logcatReader) {
        this.logcatReader = logcatReader;
    }

    public List<LogLine> parse() throws IOException {
        String line;
        ArrayList<LogLine> entries = new ArrayList<LogLine>();
        this.unparseableLines.clear();
        int lineNumber = 0;
        while ((line = this.logcatReader.readLine()) != null) {
            ++lineNumber;
            if (line.trim().isEmpty()) continue;
            LogLine parsed = LogcatParser.parseLine(line, lineNumber);
            if (parsed != null) {
                entries.add(parsed);
                continue;
            }
            if (SEPARATOR_PATTERN.matcher(line).matches()) continue;
            this.unparseableLines.add(line);
        }
        return entries;
    }

    private static LogLine parseLine(String line, int lineNumber) {
        Matcher matcher = LOG_PREFIX_PATTERN.matcher(line);
        if (!matcher.matches()) {
            return null;
        }
        LogLine.Builder result = LogLine.builder();
        result.fullLogLine(line);
        DateTime dt = DATE_TIME_FORMATTER.parseDateTime(Calendar.getInstance().get(1) + matcher.group(2));
        String tag = matcher.group(4);
        result.timeStamp(dt.toInstant());
        result.severity(LogcatParser.nameToSeverity(matcher.group(3)));
        result.tag(tag);
        result.processId(Integer.parseInt(matcher.group(5)));
        result.lineNumber(lineNumber);
        String message = line.substring(matcher.group(1).length());
        result.message(message);
        matcher = START_TIME_DEFAULT_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.STARTUP_TIME_DEFAULT);
            LogcatParser.addTotalTime(matcher, result);
            return result.build();
        }
        matcher = START_TIME_FULLY_DRAWN_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.STARTUP_TIME_FULLY_DRAWN);
            LogcatParser.addTotalTime(matcher, result);
            return result.build();
        }
        matcher = STACK_FRAME_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.STACK_FRAME);
            result.sourceMethod(matcher.group(1));
            result.sourceLine(matcher.group(2));
            return result.build();
        }
        matcher = CAUSED_BY_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.CAUSED_BY);
            result.causedBy(matcher.group(1));
            return result.build();
        }
        matcher = NATIVE_CRASH_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.NATIVE_CRASH);
            result.targetProcessId(Integer.parseInt(matcher.group(1)));
            result.packageName(matcher.group(2));
            return result.build();
        }
        matcher = ROBO_TEST_COMPLETE_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.ROBO_TEST_COMPLETE);
            return result.build();
        }
        matcher = SUPPRESSED_CRASH_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.SUPPRESSED_CRASH);
            return result.build();
        }
        matcher = PROCESS_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.PROCESS);
            result.targetProcessId(Integer.parseInt(matcher.group("pid")));
            result.packageName(matcher.group("package"));
            return result.build();
        }
        matcher = PROCESS_DIED_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.PROCESS_DIED);
            result.targetProcessId(Integer.parseInt(matcher.group("pid")));
            result.packageName(matcher.group("package"));
            return result.build();
        }
        matcher = CRASH_RUNNING_INSTRUMENTATION_PATTERN.matcher(message);
        if (matcher.matches()) {
            result.type(LogLine.LogMessageType.CRASH_DURING_INSTRUMENTATION);
            result.packageName(matcher.group("package"));
            return result.build();
        }
        matcher = LOW_MEMORY_KILLER_PATTERN.matcher(message);
        if (matcher.matches() && tag.equals("lowmemorykiller")) {
            result.type(LogLine.LogMessageType.LOW_MEMORY_KILLER);
            result.targetProcessId(Integer.parseInt(matcher.group("pid")));
            result.packageName(matcher.group("package"));
            return result.build();
        }
        matcher = NON_SDK_API_USED_VIOLATION_PATTERN.matcher(message);
        if (matcher.matches() && tag.equals("StrictMode")) {
            result.type(LogLine.LogMessageType.NON_SDK_API_USED_VIOLATION);
            result.nonSdkApiSignature(matcher.group("apisig"));
            return result.build();
        }
        matcher = START_PROC_PATTERN.matcher(message);
        if (matcher.matches() && tag.equals("ActivityManager")) {
            result.type(LogLine.LogMessageType.START_PROC);
            result.targetProcessId(Integer.parseInt(matcher.group("pid")));
            String packageName = matcher.group("package");
            int index = packageName.indexOf(58);
            if (index >= 0) {
                packageName = packageName.substring(0, index);
            }
            result.packageName(packageName);
            return result.build();
        }
        if (tag.equals("TestRunner")) {
            matcher = TEST_CASE_STARTED.matcher(message);
            if (matcher.matches() && !matcher.group(2).startsWith(JUNIT_CLASS)) {
                result.type(LogLine.LogMessageType.INSTRUMENTATION_TEST_STARTED);
                result.sourceMethod(matcher.group(1));
                result.sourceClass(matcher.group(2));
                return result.build();
            }
            matcher = TEST_CASE_FINISHED.matcher(message);
            if (matcher.matches() && !matcher.group(2).startsWith(JUNIT_CLASS)) {
                result.type(LogLine.LogMessageType.INSTRUMENTATION_TEST_FINISHED);
                result.sourceMethod(matcher.group(1));
                result.sourceClass(matcher.group(2));
                return result.build();
            }
        }
        result.type(LogLine.LogMessageType.OTHER);
        return result.build();
    }

    private static void addTotalTime(Matcher matcher, LogLine.Builder result) {
        if (!Strings.isNullOrEmpty(matcher.group("totalTime"))) {
            result.startupTime(LogcatParser.parseStartupTime(matcher.group("totalTime")));
        } else {
            result.startupTime(LogcatParser.parseStartupTime(matcher.group("defaultTime")));
        }
    }

    private static Duration parseStartupTime(String timeString) {
        Matcher timeMatcher = TIME_PATTERN.matcher(timeString);
        Preconditions.checkArgument(timeMatcher.matches(), "parseStartupTime should only be called on a properly formatted string");
        Duration startupDuration = Duration.millis(Integer.parseInt(timeMatcher.group("ms")));
        if (timeMatcher.group("sec") != null) {
            startupDuration = startupDuration.plus(Duration.standardSeconds(Integer.parseInt(timeMatcher.group("sec"))));
        }
        if (timeMatcher.group("min") != null) {
            startupDuration = startupDuration.plus(Duration.standardMinutes(Integer.parseInt(timeMatcher.group("min"))));
        }
        return startupDuration;
    }

    private static LogLine.LogSeverity nameToSeverity(String severityName) {
        String upperName = severityName.toUpperCase();
        LogLine.LogSeverity result = allowedSeverityNames.get(upperName);
        return result != null ? result : LogLine.LogSeverity.UNKNOWN;
    }

    public List<String> getUnparseableLines() {
        return this.unparseableLines;
    }
}

