/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.IntMath;
import com.google.common.math.MathPreconditions;
import java.math.RoundingMode;

@Beta
public final class LongMath {
    @VisibleForTesting
    static final long MAX_POWER_OF_SQRT2_UNSIGNED = -5402926248376769404L;
    @VisibleForTesting
    static final long[] POWERS_OF_10 = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L};
    @VisibleForTesting
    static final long[] HALF_POWERS_OF_10 = new long[]{3L, 31L, 316L, 3162L, 31622L, 316227L, 3162277L, 31622776L, 316227766L, 3162277660L, 31622776601L, 316227766016L, 3162277660168L, 31622776601683L, 316227766016837L, 3162277660168379L, 31622776601683793L, 316227766016837933L, 3162277660168379331L};
    @VisibleForTesting
    static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    static final long[] FACTORIALS = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final int[] BIGGEST_BINOMIALS = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 3810779, 121977, 16175, 4337, 1733, 887, 534, 361, 265, 206, 169, 143, 125, 111, 101, 94, 88, 83, 79, 76, 74, 72, 70, 69, 68, 67, 67, 66, 66, 66, 66};
    @VisibleForTesting
    static final int[] BIGGEST_SIMPLE_BINOMIALS = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 2642246, 86251, 11724, 3218, 1313, 684, 419, 287, 214, 169, 139, 119, 105, 95, 87, 81, 76, 73, 70, 68, 66, 64, 63, 62, 62, 61, 61, 61};

    public static boolean isPowerOfTwo(long l) {
        return l > 0L & (l & l - 1L) == 0L;
    }

    public static int log2(long l, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", l);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(LongMath.isPowerOfTwo(l));
            }
            case DOWN: 
            case FLOOR: {
                return 63 - Long.numberOfLeadingZeros(l);
            }
            case UP: 
            case CEILING: {
                return 64 - Long.numberOfLeadingZeros(l - 1L);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int n = Long.numberOfLeadingZeros(l);
                long l2 = -5402926248376769404L >>> n;
                int n2 = 63 - n;
                return l <= l2 ? n2 : n2 + 1;
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    public static int log10(long l, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", l);
        if (LongMath.fitsInInt(l)) {
            return IntMath.log10((int)l, roundingMode);
        }
        int n = LongMath.log10Floor(l);
        long l2 = POWERS_OF_10[n];
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l == l2);
            }
            case DOWN: 
            case FLOOR: {
                return n;
            }
            case UP: 
            case CEILING: {
                return l == l2 ? n : n + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return l <= HALF_POWERS_OF_10[n] ? n : n + 1;
            }
        }
        throw new AssertionError();
    }

    static int log10Floor(long l) {
        for (int i = 1; i < POWERS_OF_10.length; ++i) {
            if (l >= POWERS_OF_10[i]) continue;
            return i - 1;
        }
        return POWERS_OF_10.length - 1;
    }

    public static long pow(long l, int n) {
        MathPreconditions.checkNonNegative("exponent", n);
        if (-2L <= l && l <= 2L) {
            switch ((int)l) {
                case 0: {
                    return n == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (n & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    return n < 64 ? 1L << n : 0L;
                }
                case -2: {
                    if (n < 64) {
                        return (n & 1) == 0 ? 1L << n : -(1L << n);
                    }
                    return 0L;
                }
            }
        }
        long l2 = 1L;
        while (true) {
            switch (n) {
                case 0: {
                    return l2;
                }
                case 1: {
                    return l2 * l;
                }
            }
            l2 *= (n & 1) == 0 ? 1L : l;
            l *= l;
            n >>= 1;
        }
    }

    public static long sqrt(long l, RoundingMode roundingMode) {
        MathPreconditions.checkNonNegative("x", l);
        if (LongMath.fitsInInt(l)) {
            return IntMath.sqrt((int)l, roundingMode);
        }
        long l2 = LongMath.sqrtFloor(l);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l2 * l2 == l);
            }
            case DOWN: 
            case FLOOR: {
                return l2;
            }
            case UP: 
            case CEILING: {
                return l2 * l2 == l ? l2 : l2 + 1L;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long l3 = l2 * l2 + l2;
                return l3 >= l | l3 < 0L ? l2 : l2 + 1L;
            }
        }
        throw new AssertionError();
    }

    private static long sqrtFloor(long l) {
        long l2 = (long)Math.sqrt(l);
        long l3 = l2 + l / l2 >> 1;
        if (l3 == l2) {
            return l2;
        }
        while ((l3 = (l2 = l3) + l / l2 >> 1) < l2) {
        }
        return l2;
    }

    public static long divide(long l, long l2, RoundingMode roundingMode) {
        boolean bl;
        Preconditions.checkNotNull(roundingMode);
        long l3 = l / l2;
        long l4 = l - l2 * l3;
        if (l4 == 0L) {
            return l3;
        }
        int n = 1 | (int)((l ^ l2) >> 63);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l4 == 0L);
            }
            case DOWN: {
                bl = false;
                break;
            }
            case UP: {
                bl = true;
                break;
            }
            case CEILING: {
                bl = n > 0;
                break;
            }
            case FLOOR: {
                bl = n < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long l5 = Math.abs(l4);
                long l6 = l5 - (Math.abs(l2) - l5);
                if (l6 == 0L) {
                    bl = roundingMode == RoundingMode.HALF_UP | roundingMode == RoundingMode.HALF_EVEN & (l3 & 1L) != 0L;
                    break;
                }
                bl = l6 > 0L;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return bl ? l3 + (long)n : l3;
    }

    public static int mod(long l, int n) {
        return (int)LongMath.mod(l, (long)n);
    }

    public static long mod(long l, long l2) {
        if (l2 <= 0L) {
            throw new ArithmeticException("Modulus " + l2 + " must be > 0");
        }
        long l3 = l % l2;
        return l3 >= 0L ? l3 : l3 + l2;
    }

    public static long gcd(long l, long l2) {
        MathPreconditions.checkNonNegative("a", l);
        MathPreconditions.checkNonNegative("b", l2);
        if (l == 0L | l2 == 0L) {
            return l | l2;
        }
        int n = Long.numberOfTrailingZeros(l);
        l >>= n;
        int n2 = Long.numberOfTrailingZeros(l2);
        l2 >>= n2;
        while (l != l2) {
            if (l < l2) {
                long l3 = l2;
                l2 = l;
                l = l3;
            }
            l -= l2;
            l >>= Long.numberOfTrailingZeros(l);
        }
        return l << Math.min(n, n2);
    }

    public static long checkedAdd(long l, long l2) {
        long l3 = l + l2;
        MathPreconditions.checkNoOverflow((l ^ l2) < 0L | (l ^ l3) >= 0L);
        return l3;
    }

    public static long checkedSubtract(long l, long l2) {
        long l3 = l - l2;
        MathPreconditions.checkNoOverflow((l ^ l2) >= 0L | (l ^ l3) >= 0L);
        return l3;
    }

    public static long checkedMultiply(long l, long l2) {
        int n = Long.numberOfLeadingZeros(l) + Long.numberOfLeadingZeros(l ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(l2) + Long.numberOfLeadingZeros(l2 ^ 0xFFFFFFFFFFFFFFFFL);
        if (n > 65) {
            return l * l2;
        }
        MathPreconditions.checkNoOverflow(n >= 64);
        MathPreconditions.checkNoOverflow(l >= 0L | l2 != Long.MIN_VALUE);
        long l3 = l * l2;
        MathPreconditions.checkNoOverflow(l == 0L || l3 / l == l2);
        return l3;
    }

    public static long checkedPow(long l, int n) {
        MathPreconditions.checkNonNegative("exponent", n);
        if (l >= -2L & l <= 2L) {
            switch ((int)l) {
                case 0: {
                    return n == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (n & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    MathPreconditions.checkNoOverflow(n < 63);
                    return 1L << n;
                }
                case -2: {
                    MathPreconditions.checkNoOverflow(n < 64);
                    return (n & 1) == 0 ? 1L << n : -1L << n;
                }
            }
        }
        long l2 = 1L;
        while (true) {
            switch (n) {
                case 0: {
                    return l2;
                }
                case 1: {
                    return LongMath.checkedMultiply(l2, l);
                }
            }
            if ((n & 1) != 0) {
                l2 = LongMath.checkedMultiply(l2, l);
            }
            if ((n >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(l <= 3037000499L);
            l *= l;
        }
    }

    public static long factorial(int n) {
        MathPreconditions.checkNonNegative("n", n);
        return n < FACTORIALS.length ? FACTORIALS[n] : Long.MAX_VALUE;
    }

    public static long binomial(int n, int n2) {
        MathPreconditions.checkNonNegative("n", n);
        MathPreconditions.checkNonNegative("k", n2);
        Preconditions.checkArgument(n2 <= n, "k (%s) > n (%s)", n2, n);
        if (n2 > n >> 1) {
            n2 = n - n2;
        }
        if (n2 >= BIGGEST_BINOMIALS.length || n > BIGGEST_BINOMIALS[n2]) {
            return Long.MAX_VALUE;
        }
        long l = 1L;
        if (n2 < BIGGEST_SIMPLE_BINOMIALS.length && n <= BIGGEST_SIMPLE_BINOMIALS[n2]) {
            for (int i = 0; i < n2; ++i) {
                l *= (long)(n - i);
                l /= (long)(i + 1);
            }
        } else {
            int n3 = 1;
            while (n3 <= n2) {
                int n4 = IntMath.gcd(n, n3);
                l /= (long)(n3 / n4);
                l *= (long)(n / n4);
                ++n3;
                --n;
            }
        }
        return l;
    }

    static boolean fitsInInt(long l) {
        return (long)((int)l) == l;
    }

    private LongMath() {
    }
}

