/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import java.math.BigInteger;
import java.math.RoundingMode;

@Beta
public final class DoubleMath {
    private static final double MIN_INT_AS_DOUBLE = -2.147483648E9;
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final double LN_2 = Math.log(2.0);
    @VisibleForTesting
    static final int MAX_FACTORIAL = 170;
    @VisibleForTesting
    static final double[] EVERY_SIXTEENTH_FACTORIAL = new double[]{1.0, 2.0922789888E13, 2.631308369336935E35, 1.2413915592536073E61, 1.2688693218588417E89, 7.156945704626381E118, 9.916779348709496E149, 1.974506857221074E182, 3.856204823625804E215, 5.5502938327393044E249, 4.7147236359920616E284};

    static double roundIntermediate(double d, RoundingMode roundingMode) {
        if (!DoubleUtils.isFinite(d)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isMathematicalInteger(d));
                return d;
            }
            case FLOOR: {
                return d >= 0.0 ? d : Math.floor(d);
            }
            case CEILING: {
                return d >= 0.0 ? Math.ceil(d) : d;
            }
            case DOWN: {
                return d;
            }
            case UP: {
                return d >= 0.0 ? Math.ceil(d) : Math.floor(d);
            }
            case HALF_EVEN: {
                return Math.rint(d);
            }
            case HALF_UP: {
                if (DoubleMath.isMathematicalInteger(d)) {
                    return d;
                }
                return d >= 0.0 ? d + 0.5 : d - 0.5;
            }
            case HALF_DOWN: {
                if (DoubleMath.isMathematicalInteger(d)) {
                    return d;
                }
                if (d >= 0.0) {
                    double d2 = d + 0.5;
                    return d2 == d ? d : DoubleUtils.next(d2, false);
                }
                double d3 = d - 0.5;
                return d3 == d ? d : DoubleUtils.next(d3, true);
            }
        }
        throw new AssertionError();
    }

    public static int roundToInt(double d, RoundingMode roundingMode) {
        double d2 = DoubleMath.roundIntermediate(d, roundingMode);
        MathPreconditions.checkInRange(d2 > -2.147483649E9 & d2 < 2.147483648E9);
        return (int)d2;
    }

    public static long roundToLong(double d, RoundingMode roundingMode) {
        double d2 = DoubleMath.roundIntermediate(d, roundingMode);
        MathPreconditions.checkInRange(-9.223372036854776E18 - d2 < 1.0 & d2 < 9.223372036854776E18);
        return (long)d2;
    }

    public static BigInteger roundToBigInteger(double d, RoundingMode roundingMode) {
        if (-9.223372036854776E18 - (d = DoubleMath.roundIntermediate(d, roundingMode)) < 1.0 & d < 9.223372036854776E18) {
            return BigInteger.valueOf((long)d);
        }
        int n = DoubleUtils.getExponent(d);
        if (n < 0) {
            return BigInteger.ZERO;
        }
        long l = DoubleUtils.getSignificand(d);
        BigInteger bigInteger = BigInteger.valueOf(l).shiftLeft(n - 52);
        return d < 0.0 ? bigInteger.negate() : bigInteger;
    }

    public static boolean isPowerOfTwo(double d) {
        return d > 0.0 && DoubleUtils.isFinite(d) && LongMath.isPowerOfTwo(DoubleUtils.getSignificand(d));
    }

    public static double log2(double d) {
        return Math.log(d) / LN_2;
    }

    public static int log2(double d, RoundingMode roundingMode) {
        boolean bl;
        Preconditions.checkArgument(d > 0.0 && DoubleUtils.isFinite(d), "x must be positive and finite");
        int n = DoubleUtils.getExponent(d);
        if (!DoubleUtils.isNormal(d)) {
            return DoubleMath.log2(d * 4.503599627370496E15, roundingMode) - 52;
        }
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isPowerOfTwo(d));
            }
            case FLOOR: {
                bl = false;
                break;
            }
            case CEILING: {
                bl = !DoubleMath.isPowerOfTwo(d);
                break;
            }
            case DOWN: {
                bl = n < 0 & !DoubleMath.isPowerOfTwo(d);
                break;
            }
            case UP: {
                bl = n >= 0 & !DoubleMath.isPowerOfTwo(d);
                break;
            }
            case HALF_EVEN: 
            case HALF_UP: 
            case HALF_DOWN: {
                double d2 = DoubleUtils.scaleNormalize(d);
                bl = d2 * d2 > 2.0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return bl ? n + 1 : n;
    }

    public static boolean isMathematicalInteger(double d) {
        return DoubleUtils.isFinite(d) && (d == 0.0 || 52 - Long.numberOfTrailingZeros(DoubleUtils.getSignificand(d)) <= DoubleUtils.getExponent(d));
    }

    public static double factorial(int n) {
        MathPreconditions.checkNonNegative("n", n);
        if (n > 170) {
            return Double.POSITIVE_INFINITY;
        }
        double d = 1.0;
        for (int i = 1 + (n & 0xFFFFFFF0); i <= n; ++i) {
            d *= (double)i;
        }
        return d * EVERY_SIXTEENTH_FACTORIAL[n >> 4];
    }
}

