from typing import Optional, Union

import numpy as np
import pandas as pd
from plotly import graph_objs as go
from plotly.offline import plot

from shapash.style.style_utils import define_style, get_palette
from shapash.utils.sampling import subset_sampling
from shapash.utils.utils import adjust_title_height, truncate_str, tuning_colorscale


def plot_scatter_prediction(
    x_data,
    y_pred,
    y_proba_values,
    y_target,
    prediction_error,
    case,
    style_dict,
    round_digit,
    label_dict=None,
    selection=None,
    label_num=None,
    label_value=None,
    max_points=2000,
    width=900,
    height=600,
    file_name=None,
    auto_open=False,
):
    """
    Generate a scatter or violin plot of predictions versus target values to visually compare the model's predictions

    This plot allows the user to understand the distribution of predictions in comparison to the target variable.
    With the web app, it is possible to select the wrong or correct predictions or a subset of predictions.

    Parameters
    ----------
    x_data : pandas.DataFrame or numpy.ndarray
        The input dataset used for predictions. Contains feature values for each observation.
    y_pred : numpy.ndarray
        Array of predicted values generated by the model.
    y_proba_values : numpy.ndarray
        Array of predicted probabilities for classification tasks.
    y_target : numpy.ndarray
        Array of true target values for each observation.
    prediction_error : numpy.ndarray
        Array of prediction errors (difference between true and predicted values) for regression tasks.
    case : str
        Defines the type of task. Accepts either "classification" or "regression".
    style_dict : dict
        Dictionary containing style settings (e.g., colors, title fonts, etc.) for customizing the plot.
    round_digit : int
        Number of decimal places to round numeric values in the plot.
    label_dict : dict, optional
        Dictionary containing labels for the target classes (classification) or other custom labels for the plot, by default None.
    selection : list, optional
        List of indices representing a subset of data to plot. If None, the entire dataset or a sample (based on `max_points`) is used, by default None.
    label_num : int or None, optional
        Specific numeric label for filtering or categorizing the dataset, by default None.
    label_value : str or None, optional
        Specific string label for filtering or categorizing the dataset, by default None.
    max_points : int, optional
        Maximum number of data points to plot. If the dataset exceeds this number, a random sample is selected, by default 2000.
    width : int, optional
        Width of the plot, by default 900.
    height : int, optional
        Height of the plot, by default 600.
    file_name : str, optional
        Path to save the plot as an HTML file. If None, the plot will not be saved, by default None.
    auto_open : bool, optional
        If True, the plot will automatically open in a web browser after being generated, by default False.

    Returns
    -------
    plotly.graph_objects.Figure
        A Plotly figure object containing the scatter or violin plot of true versus predicted values.

    Raises
    ------
    ValueError
        If `case` is not one of "classification" or "regression".

    Examples
    --------
    >>> plot_scatter_prediction(
            x_data=x_data_df,
            y_pred=predictions,
            y_proba_values=proba_values,
            y_target=true_values,
            prediction_error=errors,
            case="classification",
            style_dict=style_dict,
            round_digit=2
        )
    """
    if y_target is not None:
        # Sampling
        list_ind, addnote = subset_sampling(x_data, selection, max_points)

        # Classification Case
        if case == "classification":
            fig, subtitle = _prediction_classification_plot(
                y_target, y_pred, y_proba_values, list_ind, style_dict, label_dict, label_num, label_value
            )

        # Regression Case
        elif case == "regression":
            fig, subtitle = _prediction_regression_plot(
                y_target, y_pred, prediction_error, list_ind, style_dict, round_digit
            )

        # Add traces, title and template
        title = "True Values Vs Predicted Values"
        if subtitle and addnote:
            title += "<br><sup>" + subtitle + " - " + addnote + "</sup>"
        elif subtitle:
            title += "<br><sup>" + subtitle + "</sup>"
        else:
            title += "<br><sup>" + addnote + "</sup>"
        dict_t = style_dict["dict_title"] | {"text": title, "y": adjust_title_height(height)}
        dict_xaxis = style_dict["dict_xaxis"] | {"text": truncate_str("True Values", 110)}
        dict_yaxis = style_dict["dict_yaxis"] | {"text": "Predicted Values"}

        fig.update_traces(marker={"size": 10, "opacity": 0.8, "line": {"width": 0.8, "color": "white"}})

        fig.update_layout(
            template="none",
            title=dict_t,
            width=width,
            height=height,
            xaxis_title=dict_xaxis,
            yaxis_title=dict_yaxis,
            hovermode="closest",
        )

        fig.update_yaxes(automargin=True)
        fig.update_xaxes(automargin=True)
        if file_name:
            plot(fig, filename=file_name, auto_open=auto_open)

    else:
        fig = go.Figure()
        fig.update_layout(
            xaxis={"visible": False},
            yaxis={"visible": False},
            annotations=[
                {
                    "text": "Provide the y_target argument in the compile() method to display this plot.",
                    "xref": "paper",
                    "yref": "paper",
                    "showarrow": False,
                    "font": {"size": 14},
                }
            ],
        )

    return fig


def _prediction_classification_plot(
    y_target, y_pred, y_proba_values, list_ind, style_dict, label_dict, label_num, label_value
):
    """
    _prediction_classification_plot displays a Plotly violin plot of predictions in comparison to the target variable.
    This plot represents Trues Values versus Predicted Values.

    This plot allows the user to understand the distribution of predictions in comparison to the target variable.
    With the web app, it is possible to select the wrong or correct predictions or a subset of predictions.

    Parameters
    ----------
    y_target : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the true target values for each observation.
    y_pred : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the predicted class labels for each observation.
    y_proba_values : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the predicted probabilities for each observation, with one column per class.
    list_ind : list
        List of indices representing the subset of data points to be plotted.
    style_dict : dict
        Dictionary containing styling options (e.g., colors, line styles) for the plot.
    label_dict : dict
        Dictionary mapping the target class labels to human-readable names, used for the x-axis.
    label_num : int
        Index of the class in `y_proba_values` to plot the probability values for (e.g., 0 for class 0, 1 for class 1, etc.).
    label_value : str
        The human-readable name of the class corresponding to `label_num`, used in the plot's subtitle.

    Returns
    -------
    fig : plotly.graph_objects.Figure
        The Plotly figure containing the violin and scatter plot of predictions.
    subtitle : str
        A subtitle for the plot, providing additional information about the plotted class.

    Raises
    ------
    Exception
        Raised if `label_num` or `label_value` is not provided or invalid.

    Notes
    -----
    - Correct predictions are marked in one color, and incorrect predictions are marked in another.
    - Hovering over a data point reveals detailed information about that observation, including predicted value, true value, and prediction class.
    - The x-axis can be customized to display class labels if `label_dict` is provided.
    """
    fig = go.Figure()

    if label_num is None or label_value is None:
        raise Exception({"message": "Label not found"})

    # predict
    if y_proba_values is not None:
        # Assign proba values of the target
        y_proba_values["proba_target"] = y_proba_values.iloc[:, label_num]
        y_proba_values = y_proba_values[["proba_target"]]
        # Proba subset:
        y_proba_values = y_proba_values.loc[list_ind, :]
        target = y_target.loc[list_ind, :]
        y_pred = y_pred.loc[list_ind, :]
        df_pred = pd.concat(
            [y_proba_values.reset_index(), y_pred.reset_index(drop=True), target.reset_index(drop=True)], axis=1
        )
        df_pred.set_index(df_pred.columns[0], inplace=True)
        df_pred.columns = ["proba_values", "predict_class", "target"]
        df_pred["wrong_predict"] = 1
        df_pred.loc[(df_pred["predict_class"] == df_pred["target"]), "wrong_predict"] = 0
        subtitle = f"Response: <b>{label_value}</b>"

    # Plot distribution
    fig.add_trace(
        go.Violin(
            x=df_pred["target"].values.flatten(),
            y=df_pred["proba_values"].values.flatten(),
            points=False,
            legendgroup="M",
            scalegroup="M",
            name="Correct Prediction",
            line_color=style_dict["violin_area_classif"][1],
            pointpos=-0.1,
            showlegend=False,
            jitter=0.075,
            meanline_visible=True,
            spanmode="hard",
            customdata=df_pred["proba_values"].index.values,
            scalemode="count",
        )
    )

    # Plot points depending if wrong or correct prediction
    df_correct_predict = df_pred[(df_pred["wrong_predict"] == 0)]
    df_wrong_predict = df_pred[(df_pred["wrong_predict"] == 1)]
    hv_text_correct_predict = [
        f"Id: {x}<br />Predicted Values: {y:.3f}<br />Predicted class: {w}<br />True Values: {z}<br />"
        for x, y, w, z in zip(
            df_correct_predict.index,
            df_correct_predict.proba_values.values.round(3).flatten(),
            df_correct_predict.predict_class.values.flatten(),
            df_correct_predict.target.values.flatten(),
        )
    ]
    hv_text_wrong_predict = [
        f"Id: {x}<br />Predicted Values: {y:.3f}<br />Predicted class: {w}<br />True Values: {z}<br />"
        for x, y, w, z in zip(
            df_wrong_predict.index,
            df_wrong_predict.proba_values.values.round(3).flatten(),
            df_wrong_predict.predict_class.values.flatten(),
            df_wrong_predict.target.values.flatten(),
        )
    ]
    rng = np.random.default_rng(seed=79)
    fig.add_trace(
        go.Scatter(
            x=df_correct_predict["target"].values.flatten() + rng.normal(0, 0.02, len(df_correct_predict)),
            y=df_correct_predict["proba_values"].values.flatten(),
            mode="markers",
            marker_color=style_dict["prediction_plot"][1],
            showlegend=True,
            name="Correct Prediction",
            hovertext=hv_text_correct_predict,
            hovertemplate="<b>%{hovertext}</b><br />",
            customdata=df_correct_predict["proba_values"].index.values,
        )
    )

    fig.add_trace(
        go.Scatter(
            x=df_wrong_predict["target"].values.flatten() + rng.normal(0, 0.02, len(df_wrong_predict)),
            y=df_wrong_predict["proba_values"].values.flatten(),
            mode="markers",
            marker_color=style_dict["prediction_plot"][0],
            showlegend=True,
            name="Wrong Prediction",
            hovertext=hv_text_wrong_predict,
            hovertemplate="<b>%{hovertext}</b><br />",
            customdata=df_wrong_predict["proba_values"].index.values,
        )
    )

    fig.update_layout(violingap=0, violinmode="overlay")
    if label_dict is not None:
        fig.update_xaxes(
            tickmode="array",
            tickvals=list(df_pred["target"].unique()),
            ticktext=list(df_pred["target"].apply(lambda x: label_dict[x]).unique()),
        )
    if label_dict is None:
        fig.update_xaxes(tickvals=sorted(list(df_pred["target"].unique())))

    return fig, subtitle


def _prediction_regression_plot(y_target, y_pred, prediction_error, list_ind, style_dict, round_digit):
    """
    _prediction_regression_plot displays a Plotly scatter plot of predictions in comparison to the target variable.
    This plot represents Trues Values versus Predicted Values.

    This plot allows the user to understand the distribution of predictions in comparison to the target variable.
    With the web app, it is possible to select the wrong or correct predictions or a subset of predictions.

    Parameters
    ----------
    y_target : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the true target values for each observation.
    y_pred : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the predicted values for each observation.
    prediction_error : pandas.DataFrame or numpy.ndarray
        Array or DataFrame containing the prediction error values (e.g., absolute or relative error) for each observation.
    list_ind : list
        List of indices representing the subset of data points to be plotted.
    style_dict : dict
        Dictionary containing styling options (e.g., colors, line styles) for the plot.
    round_digit : int
        Number of decimal places to round the predicted values for display purposes.

    Returns
    -------
    fig : plotly.graph_objects.Figure
        The Plotly figure containing the scatter plot of predicted versus true values, with an optional density plot.
    subtitle : str or None
        A subtitle for the plot describing the type of prediction error (e.g., absolute or relative).

    Raises
    ------
    Exception
        If certain required values for the plot (e.g., prediction errors) are missing or invalid.

    Notes
    -----
    - The plot is created by first calculating the prediction error and categorizing it into bins. The color of each point reflects the size of the error.
    - If the target distribution has a large number of points, the function applies outlier trimming (removes extreme points) to focus on the bulk of the data.
    - A density plot is added as a background to the scatter plot to show the distribution of the true values.
    - The scatter plot includes hover text that displays detailed information for each point, such as the ID, true value, predicted value, and prediction error.
    - The x- and y-axes are automatically scaled to match the range of the true and predicted values.
    - A diagonal reference line (y = x) is included to indicate the ideal case where predictions exactly match the true values.
    """
    fig = go.Figure()

    subtitle = None
    if prediction_error is not None:
        if (y_target == 0).any().iloc[0]:
            subtitle = "Prediction Error = abs(True Values - Predicted Values)"
        else:
            subtitle = "Prediction Error = abs(True Values - Predicted Values) / True Values"
        df_equal_bins = prediction_error.describe(percentiles=np.arange(0.1, 1, 0.1).tolist())
        equal_bins = df_equal_bins.loc[~df_equal_bins.index.isin(["count", "mean", "std"])].values
        equal_bins = np.unique(equal_bins)
        bins_list = [i for i in equal_bins]
        values = pd.DataFrame(pd.cut([val[0] for val in prediction_error.values], bins=bins_list, labels=False))
        col_scale, _, _ = tuning_colorscale(
            init_colorscale=style_dict["init_contrib_colorscale"], values=values, keep_90_pct=False
        )

        y_target = y_target.loc[list_ind]
        if len(y_target) > 500:
            lower_quantile = y_target.iloc[:, 0].quantile(0.005)
            upper_quantile = y_target.iloc[:, 0].quantile(0.995)
            y_target_tmp = y_target.iloc[:, 0][
                (y_target.iloc[:, 0] > lower_quantile) & (y_target.iloc[:, 0] < upper_quantile)
            ]
            if len(y_target_tmp) > 0.95 * len(y_target):
                y_target = y_target_tmp
            else:
                y_target_tmp = y_target.iloc[:, 0][(y_target.iloc[:, 0] < upper_quantile)]
                if len(y_target_tmp) > 0.95 * len(y_target):
                    y_target = y_target_tmp
                else:
                    y_target_tmp = y_target.iloc[:, 0][(y_target.iloc[:, 0] > lower_quantile)]
                    if len(y_target_tmp) > 0.95 * len(y_target):
                        y_target = y_target_tmp

        y_target_values = y_target.values.flatten()

        y_pred = y_pred.loc[y_target.index]
        prediction_error = np.array(prediction_error.loc[y_target.index])

        feature_values_array = y_target_values
        if len(feature_values_array) > 2:
            y_pred_flatten = y_pred.values.flatten()
            y_pred_flatten_min = min(y_pred_flatten)

            h = max(y_pred_flatten) - y_pred_flatten_min

            feature_values_min, feature_values_max = min(feature_values_array), max(feature_values_array)
            val_inter = feature_values_max - feature_values_min
            from sklearn.neighbors import KernelDensity

            feature_np = np.array(feature_values_array)
            feature_np = feature_np[~np.isnan(feature_np)][:, None]
            kde = KernelDensity(bandwidth=val_inter / 300, kernel="epanechnikov").fit(feature_np)
            xs = np.linspace(feature_values_min, feature_values_max, 1000)
            log_dens = kde.score_samples(xs[:, None])
            y_upper = np.exp(log_dens) * h / (np.max(np.exp(log_dens)) * 3) + y_pred_flatten_min
            y_lower = np.full_like(y_upper, y_pred_flatten_min)

            # Create the density plot
            density_plot = go.Scatter(
                x=np.concatenate([xs, xs[::-1]]),
                y=np.concatenate([y_upper, y_lower[::-1]]),
                fill="toself",
                hoverinfo="none",
                showlegend=False,
                line={"color": style_dict["contrib_distribution"]},
            )
            # Add density plot
            fig.add_trace(density_plot)

        # round predict
        y_pred = y_pred.map(lambda x: round(x, round_digit))
        y_pred_flatten = y_pred.values.flatten()

        hv_text = [
            f"Id: {x}<br />True Values: {y:,.2f}<br />Predicted Values: {z:,.2f}<br />Prediction Error: {w:,.2f}"
            for x, y, z, w in zip(y_target.index, y_target_values, y_pred.values.flatten(), prediction_error.flatten())
        ]

        fig.add_scatter(
            x=y_target_values,
            y=y_pred_flatten,
            mode="markers",
            hovertext=hv_text,
            hovertemplate="<b>%{hovertext}</b><br />",
            customdata=y_pred.index.values,
            showlegend=False,
        )

        colorpoints = pd.cut([val[0] for val in prediction_error], bins=bins_list, labels=False) / 10
        colorbar_title = "Prediction Error"
        fig.data[-1].marker.color = colorpoints.flatten()
        fig.data[-1].marker.coloraxis = "coloraxis"
        fig.layout.coloraxis.colorscale = col_scale
        fig.layout.coloraxis.colorbar = {
            "title": {"text": colorbar_title},
            "tickvals": [col_scale[0][0], col_scale[-1][0] - 0.15],
            "ticktext": [float(f"{equal_bins[0]:0.3f}"), float(f"{equal_bins[-2]:0.3f}")],
            "tickformat": ".2s",
            "yanchor": "top",
            "y": 1.1,
        }
        range_axis = [
            min(y_target_values.min(), y_pred_flatten.min()),
            max(y_target_values.max(), y_pred_flatten.max()),
        ]
        fig.update_xaxes(range=range_axis)
        fig.update_yaxes(range=range_axis)
        fig.update_layout(
            shapes=[
                {
                    "type": "line",
                    "yref": "y domain",
                    "xref": "x domain",
                    "y0": 0,
                    "y1": 1,
                    "x0": 0,
                    "x1": 1,
                    "line": dict(color="grey", width=1, dash="dot"),
                }
            ]
        )

    return fig, subtitle


def plot_confusion_matrix(
    y_true: Union[np.ndarray, list],
    y_pred: Union[np.ndarray, list],
    colors_dict: Optional[dict] = None,
    width: int = 700,
    height: int = 500,
    palette_name: str = "default",
    file_name=None,
    auto_open=False,
) -> go.Figure:
    """
    Creates an interactive confusion matrix using Plotly.

    Parameters
    ----------
    y_true : array-like
        Ground truth (correct) target values.
    y_pred : array-like
        Estimated targets as returned by a classifier.
    colors_dict : dict, optional
        Custom colors for the confusion matrix.
    width : int, optional
        The width of the figure in pixels.
    height : int, optional
        The height of the figure in pixels.
    palette_name : str, optional
        The color palette to use for the heatmap.
    file_name: string, optional
        Specify the save path of html files. If None, no file will be saved.
    auto_open: bool, optional
        Automatically open the plot.

    Returns
    -------
    go.Figure
        The generated confusion matrix as a Plotly figure.
    """
    # Create a confusion matrix as a DataFrame
    labels = sorted(set(y_true).union(set(y_pred)))
    se_y_true = pd.Series(y_true, name="Actual")
    se_y_pred = pd.Series(y_pred, name="Predicted")
    df_cm = pd.crosstab(se_y_true, se_y_pred).reindex(index=labels, columns=labels, fill_value=0)

    if colors_dict:
        style_dict = {}
        keys = ["dict_title", "init_confusion_matrix_colorscale", "dict_xaxis", "dict_yaxis"]
        if any(key not in colors_dict for key in keys):
            style_dict = define_style(get_palette(palette_name))
        style_dict.update(colors_dict)
    else:
        style_dict = define_style(get_palette(palette_name))

    init_colorscale = style_dict["init_confusion_matrix_colorscale"]
    linspace = np.linspace(0, 1, len(init_colorscale))
    col_scale = [(value, color) for value, color in zip(linspace, init_colorscale)]

    # Convert the DataFrame to a NumPy array
    x_labels = list(df_cm.columns)
    y_labels = list(df_cm.index)
    z = df_cm.loc[x_labels, y_labels].values

    title = "Confusion Matrix"
    dict_t = style_dict["dict_title"] | {"text": title, "y": adjust_title_height(height)}
    dict_xaxis = style_dict["dict_xaxis"] | {"text": se_y_pred.name}
    dict_yaxis = style_dict["dict_yaxis"] | {"text": se_y_true.name}

    # Determine if labels are numeric
    x_numeric = all(str(label).isdigit() for label in x_labels)
    y_numeric = all(str(label).isdigit() for label in y_labels)

    hv_text = [
        [f"Actual: {y}<br>Predicted: {x}<br>Count: {value}" for x, value in zip(x_labels, row)]
        for y, row in zip(y_labels, z)
    ]

    if not x_numeric:
        if len(x_labels) < 6:
            k = 10
        else:
            k = 6

        # Shorten labels that exceed the threshold
        x_labels = [x.replace(x[k + k // 2 : -k + k // 2], "...") if len(x) > 2 * k + 3 else x for x in x_labels]

    if not y_numeric:
        if len(y_labels) < 6:
            k = 10
        else:
            k = 6

        # Shorten labels that exceed the threshold
        y_labels = [x.replace(x[k + k // 2 : -k + k // 2], "...") if len(x) > 2 * k + 3 else x for x in y_labels]

    # Create the heatmap using go.Heatmap
    heatmap = go.Heatmap(
        z=z,
        x=x_labels,
        y=y_labels,
        colorscale=col_scale,
        hovertext=hv_text,
        hovertemplate="%{hovertext}<extra></extra>",
        showscale=True,
    )

    fig = go.Figure(data=[heatmap])

    # Add annotations for each cell
    annotations = []
    for i, y_label in enumerate(y_labels):
        for j, x_label in enumerate(x_labels):
            annotations.append(
                dict(
                    x=x_label,
                    y=y_label,
                    text=str(z[i][j]),
                    showarrow=False,
                    font=dict(color="black" if z[i][j] < z.max() / 2 else "white"),
                )
            )

    # Update layout
    fig.update_layout(
        annotations=annotations,
        title=dict_t,
        xaxis=dict(
            title=dict_xaxis,
            tickangle=45,
            tickmode="array" if x_numeric else "linear",
            tickvals=[int(label) for label in x_labels] if x_numeric else None,
            ticktext=x_labels if x_numeric else None,
        ),
        yaxis=dict(
            title=dict_yaxis,
            autorange="reversed",  # Reverse y-axis to match conventional confusion matrix
            tickmode="array" if y_numeric else "linear",
            tickvals=[int(label) for label in y_labels] if y_numeric else None,
            ticktext=y_labels if y_numeric else None,
        ),
        width=width,
        height=height,
        margin=dict(l=150, r=20, t=100, b=70),
    )

    if file_name:
        plot(fig, filename=file_name, auto_open=auto_open)

    return fig
