'''
Copyright (c) 2013 Qin Xuye <qin@qinxuye.me>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Created on 2013-5-25

@author: Chine
'''
import unittest
import tempfile
import os
import random
import shutil

from cola.core.mq.store import Store, StoreNoSpaceForPut


class Test(unittest.TestCase):


    def setUp(self):
        self.dir_ = tempfile.mkdtemp()
        self.node = Store(self.dir_)

    def tearDown(self):
        self.node.shutdown()
        shutil.rmtree(self.dir_)

    def testLockExists(self):
        self.assertFalse(os.path.exists(os.path.join(self.dir_, 'lock')))
           
    def testPutGet(self):
        get_num = lambda: random.randint(10000, 20000)
             
        num1 = get_num()
        self.assertEqual(self.node.put(str(num1)), str(num1))
        num2 = get_num()
        self.assertEqual(self.node.put(str(num2)), str(num2))
             
        self.assertEqual(self.node.get(), str(num1))
        self.assertEqual(self.node.get(), str(num2))
          
    def testBatchPutGet(self):
        self.node.shutdown()
           
        size = 60
        # '1'*20 will spend 30 bytes in `mq store`
        batch1 = ['1' * 20, '2' * 20]
        batch2 = ['3' * 20, '4' * 20]
           
        self.node = Store(self.dir_, size)
           
        self.assertEqual(self.node.put(batch1), batch1)
        self.assertEqual(self.node.put(batch2), batch2)
           
        self.assertEqual(len(self.node.legal_files), 2)
           
        gets = sorted([self.node.get() for _ in range(4)])
        res = list(batch1)
        res.extend(batch2)
        self.assertEqual(gets, res)
           
        self.node.put('5' * 20)
        self.assertEqual(self.node.get(), '5' * 20)
           
        self.node.put('6' * 20)
          
        self.assertEqual(len(self.node.legal_files), 1)
           
        self.assertEqual(self.node.get(), '6' * 20)
        self.assertEqual(self.node.get(), None)
           
        self.assertRaises(StoreNoSpaceForPut, lambda: self.node.put('7' * 100))
        
    def testPutCloseGet(self):
        self.node.put('1'*10)
        self.node.shutdown()
        self.node = Store(self.dir_)
        self.assertEqual(self.node.get(), '1'*10)

if __name__ == "__main__":
    #import sys;sys.argv = ['', 'Test.testName']
    unittest.main()