//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES
//
//===----------------------------------------------------------------------===//

// <algorithm>

// template<class T, class Proj = identity,
//          indirect_strict_weak_order<projected<const T*, Proj>> Comp = ranges::less>
//   constexpr const T& ranges::min(const T& a, const T& b, Comp comp = {}, Proj proj = {});
// template<copyable T, class Proj = identity,
//          indirect_strict_weak_order<projected<const T*, Proj>> Comp = ranges::less>
//   constexpr T ranges::min(initializer_list<T> r, Comp comp = {}, Proj proj = {});
// template<input_range R, class Proj = identity,
//          indirect_strict_weak_order<projected<iterator_t<R>, Proj>> Comp = ranges::less>
//   requires indirectly_copyable_storable<iterator_t<R>, range_value_t<R>*>
//   constexpr range_value_t<R>
//     ranges::min(R&& r, Comp comp = {}, Proj proj = {});

#include <cuda/std/__algorithm_>
#include <cuda/std/array>
#include <cuda/std/cassert>
#include <cuda/std/functional>
#include <cuda/std/ranges>

#include "almost_satisfies_types.h"
#include "test_iterators.h"
#include "test_macros.h"

template <class T>
_CCCL_CONCEPT HasMinR = _CCCL_REQUIRES_EXPR((T), T t)((unused(cuda::std::ranges::min(t))));

struct NoLessThanOp
{};
struct NotTotallyOrdered
{
  int i;
  __host__ __device__ bool operator<(const NotTotallyOrdered& o) const
  {
    return i < o.i;
  }
};

struct Movable
{
  Movable& operator=(Movable&&) = default;
  Movable(Movable&&)            = default;
  Movable(const Movable&)       = delete;
};

static_assert(!HasMinR<int>);

static_assert(HasMinR<int (&)[10]>);
static_assert(HasMinR<int (&&)[10]>);
static_assert(!HasMinR<NoLessThanOp (&)[10]>);
static_assert(!HasMinR<NotTotallyOrdered (&)[10]>);
static_assert(!HasMinR<Movable (&)[10]>);

static_assert(HasMinR<cuda::std::initializer_list<int>>);
static_assert(!HasMinR<cuda::std::initializer_list<NoLessThanOp>>);
static_assert(!HasMinR<cuda::std::initializer_list<NotTotallyOrdered>>);
static_assert(!HasMinR<cuda::std::initializer_list<Movable>>);
static_assert(!HasMinR<InputRangeNotDerivedFrom>);
static_assert(!HasMinR<InputRangeNotIndirectlyReadable>);
static_assert(!HasMinR<InputRangeNotInputOrOutputIterator>);
static_assert(!HasMinR<InputRangeNotSentinelSemiregular>);
static_assert(!HasMinR<InputRangeNotSentinelEqualityComparableWith>);

#if TEST_STD_VER > 2017
template <class T, class U = T>
concept HasMin2 = requires { cuda::std::ranges::min(cuda::std::declval<T>(), cuda::std::declval<U>()); };
#else
template <class T, class U = T, class = void>
constexpr bool HasMin2 = false;

template <class T, class U>
constexpr bool
  HasMin2<T, U, cuda::std::void_t<decltype(cuda::std::ranges::min(cuda::std::declval<T>(), cuda::std::declval<U>()))>> =
    true;
#endif

static_assert(HasMin2<int>);
static_assert(!HasMin2<int, long>);

static_assert(cuda::std::is_same_v<decltype(cuda::std::ranges::min(1, 2)), const int&>);

__host__ __device__ constexpr void test_2_arguments()
{
  assert(cuda::std::ranges::min(1, 2) == 1);
  assert(cuda::std::ranges::min(2, 1) == 1);
  // test comparator
  assert(cuda::std::ranges::min(1, 2, cuda::std::ranges::greater{}) == 2);
  // test projection
  assert(cuda::std::ranges::min(
           1,
           2,
           cuda::std::ranges::less{},
           [](int i) {
             return i == 1 ? 10 : i;
           })
         == 2);

  { // check that cuda::std::invoke is used
    struct S
    {
      int i;
    };
    S a[3]             = {S{2}, S{1}, S{3}};
    decltype(auto) ret = cuda::std::ranges::min(a[0], a[1], {}, &S::i);
    static_assert(cuda::std::is_same_v<decltype(ret), const S&>);
    assert(&ret == &a[1]);
    assert(ret.i == 1);
  }

  { // check that pointers are compared and not a range
    int i{};
    int* a[] = {&i, &i + 1};
    auto ret = cuda::std::ranges::min(a[0], a[1]);
    assert(ret == &i);
  }

  { // test predicate and projection count
    int compares    = 0;
    int projections = 0;
    auto comparator = [&](int x, int y) {
      ++compares;
      return x < y;
    };
    auto projection = [&](int x) {
      ++projections;
      return x;
    };
    auto ret = cuda::std::ranges::min(1, 2, comparator, projection);
    assert(ret == 1);
    assert(compares == 1);
    assert(projections == 2);
  }

  { // check that the first argument is returned
    struct S
    {
      int check;
      int other;
    };
    auto ret = cuda::std::ranges::min(S{0, 1}, S{0, 2}, {}, &S::check);
    assert(ret.other == 1);
  }
}

__host__ __device__ constexpr void test_initializer_list()
{
  {
    // test projection
    auto proj = [](int i) {
      return i == 5 ? -100 : i;
    };
    int ret = cuda::std::ranges::min({7, 6, 9, 3, 5, 1, 2, 4}, cuda::std::ranges::less{}, proj);
    assert(ret == 5);
  }

  {
    // test comparator
    int ret = cuda::std::ranges::min({7, 6, 9, 3, 5, 1, 2, 4}, cuda::std::ranges::greater{});
    assert(ret == 9);
  }

  {
    int compares    = 0;
    int projections = 0;
    auto comparator = [&](int a, int b) {
      ++compares;
      return a < b;
    };
    auto projection = [&](int a) {
      ++projections;
      return a;
    };
    decltype(auto) ret = cuda::std::ranges::min({1, 2, 3}, comparator, projection);
    static_assert(cuda::std::same_as<decltype(ret), int>);
    assert(ret == 1);
    assert(compares == 2);
    assert(projections == 4);
  }

  {
    struct S
    {
      int i;
    };
    decltype(auto) ret = cuda::std::ranges::min({S{2}, S{1}, S{3}}, {}, &S::i);
    static_assert(cuda::std::is_same_v<decltype(ret), S>);
    assert(ret.i == 1);
  }

  {
    int a[]    = {7, 6, 9, 3, 5, 1, 2, 4};
    using It   = cpp20_input_iterator<int*>;
    using Sent = sentinel_wrapper<It>;
    auto range = cuda::std::ranges::subrange<It, Sent>(It(a), Sent(It(a + 8)));
    auto ret   = cuda::std::ranges::min(range);
    assert(ret == 1);
  }
}

template <class It, class Sent = It>
__host__ __device__ constexpr void test_range_types()
{
  int a[]    = {7, 6, 9, 3, 5, 1, 2, 4};
  auto range = cuda::std::ranges::subrange<It, Sent>(It(a), Sent(It(a + 8)));
  int ret    = cuda::std::ranges::min(range);
  assert(ret == 1);
}

__host__ __device__ constexpr void test_range()
{
  { // check that all range types work
    test_range_types<cpp20_input_iterator<int*>, sentinel_wrapper<cpp20_input_iterator<int*>>>();
    test_range_types<forward_iterator<int*>>();
    test_range_types<bidirectional_iterator<int*>>();
    test_range_types<random_access_iterator<int*>>();
    test_range_types<contiguous_iterator<int*>>();
  }

  int a[] = {7, 6, 9, 3, 5, 1, 2, 4};
  {
    // test projection
    auto proj = [](int& i) {
      return i == 5 ? -100 : i;
    };
    int ret = cuda::std::ranges::min(a, cuda::std::ranges::less{}, proj);
    assert(ret == 5);
  }

  {
    // test comparator
    int ret = cuda::std::ranges::min(a, cuda::std::ranges::greater{});
    assert(ret == 9);
  }

  { // check that predicate and projection call counts are correct
    int compares    = 0;
    int projections = 0;
    auto comparator = [&](int x, int y) {
      ++compares;
      return x < y;
    };
    auto projection = [&](int x) {
      ++projections;
      return x;
    };
    decltype(auto) ret = cuda::std::ranges::min(cuda::std::array<int, 3>{1, 2, 3}, comparator, projection);
    static_assert(cuda::std::same_as<decltype(ret), int>);
    assert(ret == 1);
    assert(compares == 2);
    assert(projections == 4);
  }

  { // check that cuda::std::invoke is used
    struct S
    {
      int i;
    };
    S b[3]             = {S{2}, S{1}, S{3}};
    decltype(auto) ret = cuda::std::ranges::min(b, {}, &S::i);
    static_assert(cuda::std::same_as<decltype(ret), S>);
    assert(ret.i == 1);
  }

  { // check that the first smallest element is returned
    { // where the first element is the smallest
      struct S
      {
        int check;
        int other;
      };
      S b[]    = {S{0, 1}, S{1, 2}, S{0, 3}};
      auto ret = cuda::std::ranges::min(b, {}, &S::check);
      assert(ret.check == 0);
      assert(ret.other == 1);
    }
    { // where the first element isn't the smallest
      struct S
      {
        int check;
        int other;
      };
      S b[]    = {S{2, 1}, S{0, 2}, S{0, 3}};
      auto ret = cuda::std::ranges::min(b, {}, &S::check);
      assert(ret.check == 0);
      assert(ret.other == 2);
    }
  }
}

__host__ __device__ constexpr bool test()
{
  test_2_arguments();
  test_initializer_list();
  test_range();

  return true;
}

int main(int, char**)
{
  test();
#if !TEST_COMPILER(GCC, <, 10)
  static_assert(test());
#endif // !TEST_COMPILER(GCC, <, 10)

  return 0;
}
