# Copyright (C) Lutra Consulting Limited
#
# SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-MerginMaps-Commercial

import functools
from blinker import signal
from flask import current_app, render_template
from flask_login import current_user
from itsdangerous import URLSafeTimedSerializer
from sqlalchemy import func

from .commands import add_commands
from .config import Configuration
from .models import User

# signal for other versions to listen to
user_account_closed = signal("user_account_closed")
user_created = signal("user_created")

CANNOT_EDIT_PROFILE_MSG = "You cannot edit profile of this user"


def register(app):
    """Register mergin auth module in Flask app
    Adds Flask blueprint with autogenerated Flask/Connexion routes from openAPI definition.
    Register db events/hooks.
    """
    app.config.from_object(Configuration)
    app.connexion_app.add_api(
        "auth/api.yaml",
        base_path="/",
        options={"swagger_ui": False, "serve_spec": False},
        validate_responses=True,
    )
    # this is a hack to modify dict and name of flask blueprint registered with "base_path" name
    # due to lack of connexion flexibility for flask blueprint naming
    app.blueprints["/"].name = "auth"
    app.blueprints["auth"] = app.blueprints.pop("/")
    add_commands(app)


_permissions = {}


def register_permission(name, fn):
    _permissions[name] = fn


register_permission("admin", lambda user: user.is_admin)


def auth_required(f=None, permissions=None):
    if f is None:
        if permissions:
            permissions_fn = []
            for name in permissions:
                if name not in _permissions:
                    raise KeyError("Unknown permission: {}".format(name))
                permissions_fn.append(_permissions[name])
        return functools.partial(auth_required, permissions=permissions_fn)

    @functools.wraps(f)
    def wrapped_func(*args, **kwargs):
        if not current_user or not current_user.is_authenticated:
            return "Authentication information is missing or invalid.", 401
        if permissions:
            for check_permission in permissions:
                if not check_permission(current_user):
                    return "Permission denied.", 403
        return f(*args, **kwargs)

    return wrapped_func


def edit_profile_enabled(f):
    """Decorator to check if user can edit their profile (it is not allowed for SSO users)"""

    @functools.wraps(f)
    def wrapped_func(*args, **kwargs):
        if not current_user or not current_user.is_authenticated:
            return "Authentication information is missing or invalid.", 401
        if not current_user.can_edit_profile:
            return CANNOT_EDIT_PROFILE_MSG, 403
        return f(*args, **kwargs)

    return wrapped_func


def authenticate(login, password):
    if "@" in login:
        query = func.lower(User.email) == func.lower(login)
    else:
        query = func.lower(User.username) == func.lower(login)
    user = User.query.filter(query).one_or_none()
    if user and user.check_password(password):
        return user


def generate_confirmation_token(app, email, salt):
    serializer = URLSafeTimedSerializer(app.config["SECRET_KEY"])
    return serializer.dumps(email, salt=salt)


def confirm_token(token, salt, expiration=3600):
    serializer = URLSafeTimedSerializer(current_app.config["SECRET_KEY"])
    try:
        email = serializer.loads(token, salt=salt, max_age=expiration)
    except:
        return
    return email


def send_confirmation_email(app, user, url, template, header, **kwargs):
    """
    Send confirmation email from selected template with customizable email subject and confirmation URL.
    Optional kwargs are passed to render_template method if needed for particular template.
    """
    from ..celery import send_email_async

    salt = (
        app.config["SECURITY_EMAIL_SALT"]
        if url == "confirm-email"
        else app.config["SECURITY_PASSWORD_SALT"]
    )
    token = generate_confirmation_token(app, user.email, salt)
    confirm_url = f"{url}/{token}"
    html = render_template(
        template, subject=header, confirm_url=confirm_url, user=user, **kwargs
    )
    email_data = {
        "subject": header,
        "html": html,
        "recipients": [user.email],
        "sender": app.config["MAIL_DEFAULT_SENDER"],
    }
    send_email_async.delay(**email_data)
