/*
 * Copyright (C) 2007 - 2018 52°North Initiative for Geospatial Open Source
 * Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * If the program is linked with libraries which are licensed under one of
 * the following licenses, the combination of the program with the linked
 * library is not considered a "derivative work" of the program:
 *
 *       • Apache License, version 2.0
 *       • Apache Software License, version 1.0
 *       • GNU Lesser General Public License, version 3
 *       • Mozilla Public License, versions 1.0, 1.1 and 2.0
 *       • Common Development and Distribution License (CDDL), version 1.0
 *
 * Therefore the distribution of the program linked with libraries licensed
 * under the aforementioned licenses, is permitted by the copyright holders
 * if the distribution is compliant with both the GNU General Public
 * License version 2 and the aforementioned licenses.
 *
 * As an exception to the terms of the GPL, you may copy, modify,
 * propagate, and distribute a work formed by combining 52°North WPS
 * GeoTools Modules with the Eclipse Libraries, or a work derivative of
 * such a combination, even if such copying, modification, propagation, or
 * distribution would otherwise violate the terms of the GPL. Nothing in
 * this exception exempts you from complying with the GPL in all respects
 * for all of the code used other than the Eclipse Libraries. You may
 * include this exception and its grant of permissions when you distribute
 * 52°North WPS GeoTools Modules. Inclusion of this notice with such a
 * distribution constitutes a grant of such permissions. If you do not wish
 * to grant these permissions, remove this paragraph from your
 * distribution. "52°North WPS GeoTools Modules" means the 52°North WPS
 * modules using GeoTools functionality - software licensed under version 2
 * or any later version of the GPL, or a work based on such software and
 * licensed under the GPL. "Eclipse Libraries" means Eclipse Modeling
 * Framework Project and XML Schema Definition software distributed by the
 * Eclipse Foundation and licensed under the Eclipse Public License Version
 * 1.0 ("EPL"), or a work based on such software and licensed under the EPL.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 */

package org.n52.wps.io.datahandler.generator;

import java.awt.geom.Rectangle2D;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.httpclient.HttpException;
import org.geotools.coverage.grid.GridCoverage2D;
import org.n52.wps.io.data.GenericFileDataWithGT;
import org.n52.wps.io.data.IData;
import org.n52.wps.io.data.binding.complex.GTRasterDataBinding;
import org.n52.wps.io.data.binding.complex.GTVectorDataBinding;
import org.n52.wps.io.data.binding.complex.GenericFileDataWithGTBinding;
import org.n52.wps.io.data.binding.complex.GeotiffBinding;
import org.n52.wps.io.data.binding.complex.ShapefileBinding;
import org.opengis.referencing.ReferenceIdentifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class GeoserverWMSGenerator extends AbstractGeoserverWXSGenerator {

    private static Logger LOGGER = LoggerFactory.getLogger(GeoserverWMSGenerator.class);

    public GeoserverWMSGenerator() {
        super();
        this.supportedIDataTypes.add(GTRasterDataBinding.class);
        this.supportedIDataTypes.add(ShapefileBinding.class);
        this.supportedIDataTypes.add(GeotiffBinding.class);
        this.supportedIDataTypes.add(GTVectorDataBinding.class);
        this.supportedIDataTypes.add(GenericFileDataWithGTBinding.class);
    }

    @Override
    public InputStream generateStream(IData data, String mimeType, String schema) throws IOException {

        InputStream stream = null;
        try {
            String getMapURL = storeLayer(data);
            stream = new ByteArrayInputStream(getMapURL.getBytes("UTF-8"));
        } catch (IOException e) {
            LOGGER.error("Error generating WMS output. Reason: ", e);
            throw new RuntimeException("Error generating WMS output. Reason: " + e);
        } catch (ParserConfigurationException e) {
            LOGGER.error("Error generating WMS output. Reason: ", e);
            throw new RuntimeException("Error generating WMS output. Reason: " + e);
        }
        return stream;
    }

    private String storeLayer(IData coll) throws HttpException, IOException, ParserConfigurationException {
        File file = null;
        String layerName = "";
        if (coll instanceof GTVectorDataBinding) {
            GTVectorDataBinding gtData = (GTVectorDataBinding) coll;

            try {
                GenericFileDataWithGT fileData = new GenericFileDataWithGT(gtData.getPayload());
                file = fileData.getBaseFile(true);
            } catch (IOException e1) {
                e1.printStackTrace();
                throw new RuntimeException("Error generating shp file for storage in WFS. Reason: " + e1);
            }

            // zip shp file
            String path = file.getAbsolutePath();
            String baseName = path.substring(0, path.length() - ".shp".length());
            File shx = new File(baseName + ".shx");
            File dbf = new File(baseName + ".dbf");
            File prj = new File(baseName + ".prj");
            File zipped = org.n52.wps.io.IOUtils.zip(file, shx, dbf, prj);

            file = zipped;

        }
        if (coll instanceof GTRasterDataBinding) {
            GTRasterDataBinding gtData = (GTRasterDataBinding) coll;
            GenericFileDataWithGT fileData = new GenericFileDataWithGT(gtData.getPayload(), null);
            file = fileData.getBaseFile(true);

        }
        if (coll instanceof ShapefileBinding) {
            ShapefileBinding data = (ShapefileBinding) coll;
            file = data.getZippedPayload();

        }
        if (coll instanceof GeotiffBinding) {
            GeotiffBinding data = (GeotiffBinding) coll;
            file = (File) data.getPayload();
        }
        if (coll instanceof GenericFileDataWithGTBinding) {
            file = ((GenericFileDataWithGTBinding) coll).getPayload().getBaseFile(true);
        }

        layerName = makeUniqueFileName(file.getName());
        GeoServerUploader geoserverUploader = new GeoServerUploader(username, password, host, port);

        int width = 0;
        int height = 0;

        String bboxString = "";
        String srsString = "";

        String result = geoserverUploader.createWorkspace();
        LOGGER.debug(result);
        if (coll instanceof GTVectorDataBinding) {
            result = geoserverUploader.uploadShp(file, layerName);
        }
        if (coll instanceof GTRasterDataBinding) {
            result = geoserverUploader.uploadGeotiff(file, layerName);

            GridCoverage2D gridCoverage2D = ((GTRasterDataBinding) coll).getPayload();

            Rectangle2D bounds = gridCoverage2D.getEnvelope2D().getBounds();

            double minX = bounds.getMinX();
            double maxX = bounds.getMaxX();
            double minY = bounds.getMinY();
            double maxY = bounds.getMaxY();

            bboxString = minX + "," + minY + "," + maxX + "," + maxY;

            ReferenceIdentifier srsIdentifier = null;

            try {

                srsIdentifier = gridCoverage2D.getEnvelope2D()
                                              .getCoordinateReferenceSystem()
                                              .getCoordinateSystem()
                                              .getIdentifiers()
                                              .iterator()
                                              .next();

            } catch (Exception e) {
                LOGGER.info("Could not get CRS from grid.");
            }

            srsString = srsIdentifier.getCodeSpace() + ":" + srsIdentifier.getCode();

            width = gridCoverage2D.getRenderedImage().getWidth();
            height = gridCoverage2D.getRenderedImage().getHeight();
        }
        if (coll instanceof GenericFileDataWithGTBinding) {// TODO, could also be a shapefile
            result = geoserverUploader.uploadGeotiff(file, layerName);
        }

        LOGGER.debug(result);

        String getMapLink = "http://" + host + ":" + port
                + "/geoserver/wms?Service=WMS&Request=GetMap&Version=1.1.1&layers=" + "N52:" + layerName + "&width="
                + width + "&height=" + height + "&format=image/png" + "&bbox=" + bboxString + "&srs=" + srsString;
        // String directLink = geoserverBaseURL +
        // "?Service=WMS&Request=GetMap&Version=1.1.0&Layers=N52:"+wmsLayerName+"&WIDTH=300&HEIGHT=300";;

        return getMapLink;

    }

    private Document createXML(String layerName, String getCapabilitiesLink) throws ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        Document doc = factory.newDocumentBuilder().newDocument();

        Element root = doc.createElement("OWSResponse");
        root.setAttribute("type", "WMS");

        Element resourceIDElement = doc.createElement("ResourceID");
        resourceIDElement.appendChild(doc.createTextNode(layerName));
        root.appendChild(resourceIDElement);

        Element getCapabilitiesLinkElement = doc.createElement("GetCapabilitiesLink");
        getCapabilitiesLinkElement.appendChild(doc.createTextNode(getCapabilitiesLink));
        root.appendChild(getCapabilitiesLinkElement);
        /*
         * Element directResourceLinkElement = doc.createElement("DirectResourceLink");
         * directResourceLinkElement.appendChild(doc.createTextNode(getMapRequest));
         * root.appendChild(directResourceLinkElement);
         */
        doc.appendChild(root);

        return doc;
    }

}
