// Package main
/*
 * Copyright (c) 2018-2025, NVIDIA CORPORATION. All rights reserved.
 */
package main

import (
	"bufio"
	"fmt"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/NVIDIA/aistore/cmn/nlog"
)

type DiskStat struct {
	// based on https://www.kernel.org/doc/Documentation/iostats.txt
	ReadComplete  int64 // 1 - # of reads completed
	ReadMerged    int64 // 2 - # of reads merged
	ReadSectors   int64 // 3 - # of sectors read
	ReadMs        int64 // 4 - # ms spent reading
	WriteComplete int64 // 5 - # writes completed
	WriteMerged   int64 // 6 - # writes merged
	WriteSectors  int64 // 7 - # of sectors written
	WriteMs       int64 // 8 - # of milliseconds spent writing
	IOPending     int64 // 9 - # of I/Os currently in progress
	IOMs          int64 // 10 - # of milliseconds spent doing I/Os
	IOMsWeighted  int64 // 11 - weighted # of milliseconds spent doing I/Os
}

func timeTrack(start time.Time, name string) {
	elapsed := time.Since(start)
	fmt.Println(strings.Join([]string{time.Now().Format(time.RFC3339Nano), name,
		strconv.FormatInt(elapsed.Nanoseconds(), 10)}, ","))
}

func GetDiskstats() (output map[string]DiskStat) {
	defer timeTrack(time.Now(), "Get DiskStats Duration")

	output = make(map[string]DiskStat)

	file, err := os.Open("/proc/diskstats")
	if err != nil {
		nlog.Errorln(err)
		return
	}

	scanner := bufio.NewScanner(file)

	for scanner.Scan() {
		line := scanner.Text()
		if line == "" {
			continue
		}

		fields := strings.Fields(line)
		if len(fields) < 14 {
			continue
		}

		deviceName := fields[2]
		output[deviceName] = DiskStat{
			_exI64(fields[3]),
			_exI64(fields[4]),
			_exI64(fields[5]),
			_exI64(fields[6]),
			_exI64(fields[7]),
			_exI64(fields[8]),
			_exI64(fields[9]),
			_exI64(fields[10]),
			_exI64(fields[11]),
			_exI64(fields[12]),
			_exI64(fields[13]),
		}
	}

	return output
}

func _exI64(field string) int64 {
	val, err := strconv.ParseInt(field, 10, 64)
	if err != nil {
		nlog.Errorf("Failed to convert field value %q to int: %v \n",
			field, err)
		return 0
	}
	return val
}
