// Package cmn provides common constants, types, and utilities for AIS clients
// and AIStore.
/*
 * Copyright (c) 2018-2024, NVIDIA CORPORATION. All rights reserved.
 */
package cos

import (
	"errors"
	"io"
	"os"
	"syscall"

	"github.com/NVIDIA/aistore/cmn/debug"
)

var ioErrs = []error{
	io.ErrShortWrite,

	syscall.EIO,     // I/O error
	syscall.ENOTDIR, // mountpath is missing
	syscall.EBUSY,   // device or resource is busy
	syscall.ENXIO,   // No such device
	syscall.EBADF,   // Bad file number
	syscall.ENODEV,  // No such device
	syscall.EROFS,   // readonly filesystem
	syscall.EDQUOT,  // quota exceeded
	syscall.ESTALE,  // stale file handle
	syscall.ENOSPC,  // no space left
}

// Checks if the error is generated by any IO operation and if the error
// is severe enough to run the FSHC for mountpath testing
func IsIOError(err error) bool {
	debug.Assert(err != nil)
	for _, ioErr := range ioErrs {
		if errors.Is(err, ioErr) {
			return true
		}
	}
	return false
}

func IsErrXattrNotFound(err error) bool {
	// NOTE: syscall.ENOATTR confirmed to be returned on Darwin, instead of syscall.ENODATA.
	return os.IsNotExist(err) || err == syscall.ENOATTR
}
