// Package readers provides implementation for common reader types
/*
 * Copyright (c) 2024, NVIDIA CORPORATION. All rights reserved.
 */
package readers

import (
	"encoding/binary"
	"math/rand/v2"
	"sync/atomic"

	"github.com/NVIDIA/aistore/cmn/cos"
)

var (
	truffle = []byte{
		0x4f, 0x1c, 0x14, 0x8d, 0x4d, 0xaa, 0x52, 0x8d, 0x45, 0x01, 0x12, 0x8b, 0x01, 0x4d, 0x41, 0xd4,
		0xfa, 0x83, 0x41, 0xff, 0x94, 0x0f, 0x45, 0xc2, 0xb6, 0x0f, 0x49, 0xd2, 0xca, 0xff, 0x21, 0x4d,
		0x4d, 0xd4, 0xe4, 0x85, 0x84, 0x0f, 0x02, 0x11, 0x1a, 0x53, 0x8b, 0x45, 0x24, 0x14, 0x85, 0x45,
		0x0f, 0xd2, 0x04, 0x8e, 0x00, 0x02, 0x4c, 0x00, 0xa4, 0x89, 0xb0, 0x24, 0xde, 0xbf, 0x44, 0x00,
		0x6c, 0x8b, 0x20, 0x24, 0x85, 0x45, 0x0f, 0xed, 0x18, 0x8c, 0x00, 0x01, 0x45, 0x00, 0xea, 0x39,
		0x8e, 0x0f, 0x01, 0x0f, 0x5b, 0x64, 0x8b, 0x45, 0x24, 0x54, 0x0f, 0x04, 0x44, 0x1f, 0xd1, 0xe4,
		0x85, 0x45, 0x75, 0xd2, 0x45, 0x05, 0xd2, 0x31, 0x2c, 0xeb, 0x8d, 0x45, 0x07, 0x7a, 0xc1, 0x41,
		0x03, 0xff, 0x0f, 0x45, 0xef, 0xaf, 0x63, 0x4d, 0x4b, 0xed, 0x1c, 0x8d, 0x48, 0x2c, 0x5b, 0x8d,
		0xeb, 0x08, 0x48, 0x13, 0x94, 0x8b, 0x88, 0x24, 0x1a, 0x53, 0x4c, 0x00, 0x9c, 0x8b, 0x80, 0x24,
		0x77, 0xf2, 0x41, 0x00, 0xc2, 0x89, 0xc9, 0x84, 0x0e, 0x74, 0x01, 0xb8, 0xde, 0xbf, 0x4c, 0x00,
		0x1d, 0x8d, 0x19, 0x9a, 0x02, 0x1c, 0x70, 0xeb, 0x7a, 0x80, 0x02, 0x2b, 0x05, 0x77, 0x31, 0x45,
		0xeb, 0xdb, 0x4c, 0x2d, 0x62, 0x8d, 0x4d, 0x2b, 0x9b, 0x8b, 0x01, 0x40, 0x77, 0xf2, 0x52, 0x8b,
		0x49, 0x1c, 0x14, 0x8d, 0x48, 0x94, 0x52, 0x8d, 0x8b, 0x09, 0x49, 0x12, 0xd3, 0x01, 0xfa, 0x83,
		0x0f, 0xff, 0xc2, 0x94, 0xb6, 0x0f, 0x48, 0xd2, 0xca, 0xff, 0x21, 0x49, 0x0f, 0xd3, 0x00, 0x1f,
		0x85, 0x4d, 0x74, 0xdb, 0x49, 0x2e, 0x13, 0x8b, 0x48, 0x90, 0xd2, 0x85, 0x72, 0x7c, 0x89, 0x48,
		0x24, 0x54, 0xb8, 0x58, 0x00, 0x10, 0x1a, 0x53, 0xf7, 0x48, 0x70, 0xe2, 0x49, 0x59, 0x53, 0x8d,
		0x49, 0x08, 0xd3, 0x89, 0xf7, 0x48, 0x48, 0xda, 0xd0, 0x39, 0x4a, 0x77, 0x8b, 0x48, 0x24, 0x44,
		0xeb, 0x58, 0x31, 0x05, 0x45, 0xc0, 0xdb, 0x31, 0x89, 0x44, 0x48, 0xd1, 0xdf, 0x89, 0x89, 0x4c,
		0x49, 0xde, 0xc0, 0x89, 0x89, 0x4d, 0x8b, 0xc1, 0x24, 0x44, 0x48, 0x24, 0x9c, 0x8b, 0xc0, 0x24,
		0x5b, 0x64, 0x48, 0x00, 0xc4, 0x81, 0x00, 0xc8, 0x8e, 0xc7, 0xc3, 0x5d, 0xc0, 0x31, 0xdb, 0x31,
		0xc1, 0x89, 0x89, 0x48, 0x31, 0xdf, 0x45, 0xf6, 0xc0, 0x31, 0x89, 0x4d, 0x48, 0xc1, 0xc4, 0x81,
		0x00, 0xc8, 0xde, 0xbf, 0xc3, 0x5d, 0xa5, 0xe8, 0x01, 0x33, 0x0f, 0x00, 0x44, 0x1f, 0x1a, 0x53,
		0x9b, 0xe8, 0x01, 0x33, 0x49, 0x00, 0xcd, 0x63, 0x89, 0x48, 0x24, 0x4c, 0x8b, 0x78, 0x04, 0x5a,
		0x89, 0x4c, 0xe8, 0xd8, 0x2c, 0xc8, 0xd1, 0xe4, 0x89, 0x48, 0x24, 0x84, 0x00, 0x90, 0x8e, 0xc7,
		0x89, 0x48, 0x24, 0x5c, 0x48, 0x28, 0x8c, 0x8b, 0xb0, 0x24, 0x5b, 0x64, 0x48, 0x00, 0x09, 0x63,
		0x89, 0x48, 0x24, 0x4c, 0xe8, 0x70, 0x48, 0xc6, 0xff, 0xfe, 0x8d, 0x48, 0x0b, 0x05, 0x34, 0x78,
		0xbb, 0x01, 0x00, 0x13, 0x1a, 0x53, 0x35, 0xe8, 0xfe, 0x51, 0x48, 0xff, 0x44, 0x8b, 0x78, 0x24,
		0x48, 0x00, 0x84, 0x89, 0x98, 0x24, 0xde, 0xbf, 0x48, 0x00, 0x5c, 0x89, 0x30, 0x24, 0x8b, 0x48,
		0x4b, 0xe8, 0xfe, 0x4f, 0x48, 0xff, 0x05, 0x8d, 0xf8, 0x37, 0x01, 0x32, 0x05, 0xbb, 0x8e, 0xc7,
		0x24, 0x8c, 0x00, 0xd8, 0x77, 0xf2, 0x8b, 0x48, 0x40, 0x49, 0x89, 0x48, 0x24, 0x4c, 0xe8, 0x60,
		0xe8, 0x00, 0x51, 0x1a, 0xff, 0xfe, 0x8b, 0x48, 0x24, 0x44, 0xe8, 0x70, 0x4f, 0x30, 0xff, 0xfe,
		0x47, 0xec, 0xff, 0xfe, 0x8d, 0x48, 0xbd, 0x05, 0x34, 0x07, 0xbb, 0x01, 0x00, 0x0f, 0x1a, 0x53,
		0x8d, 0x48, 0x8d, 0x05, 0x35, 0x4a, 0xbb, 0x01, 0x00, 0x1c, 0xd1, 0xe4, 0x1f, 0x0f, 0x00, 0x40,
		0x5b, 0xe8, 0xfe, 0x50, 0x48, 0xff, 0x84, 0x8b, 0x98, 0x24, 0xde, 0xbf, 0x48, 0x00, 0x5c, 0x8b,
		0xfb, 0xe8, 0xfe, 0x50, 0x48, 0xff, 0x84, 0x8b, 0x90, 0x24, 0x5b, 0x64, 0x48, 0x00, 0x5c, 0x8b,
		0x30, 0x24, 0x49, 0xe8, 0xfe, 0x50, 0x48, 0xff, 0x05, 0x8d, 0xf0, 0x19, 0x01, 0x33, 0x0e, 0xbb,
		0x28, 0x24, 0xe9, 0xe8, 0xfe, 0x50, 0x48, 0xff, 0x05, 0x8d, 0xa8, 0xe7, 0x01, 0x33, 0x0b, 0xbb,
		0x1a, 0x53, 0xe8, 0x00, 0x50, 0x38, 0xff, 0xfe, 0x8b, 0x48, 0x24, 0x44, 0xe8, 0x60, 0x4e, 0xae,
		0x77, 0xf2, 0xe8, 0x00, 0x50, 0xd8, 0xff, 0xfe, 0x8b, 0x48, 0x24, 0x44, 0xe8, 0x48, 0x4f, 0x4e,
		0xff, 0xfe, 0x8d, 0x48, 0xcf, 0x05, 0x5a, 0xc0, 0xbb, 0x01, 0x00, 0x01, 0x8e, 0xc7, 0x90, 0x66,
		0xff, 0xfe, 0x8d, 0x48, 0x1b, 0x05, 0x32, 0xe2, 0xbb, 0x01, 0x00, 0x02, 0xde, 0xbf, 0x90, 0x66,
		0x1b, 0xe8, 0xfe, 0x50, 0x48, 0xff, 0x44, 0x8b, 0x68, 0x24, 0x91, 0xe8, 0xfe, 0x4e, 0xe8, 0xff,
		0xbb, 0xe8, 0xfe, 0x50, 0xe8, 0xff, 0x48, 0x96, 0xff, 0xfe, 0x8d, 0x48, 0x08, 0x05, 0x34, 0x24,
		0x49, 0xcc, 0xff, 0xfe, 0xe7, 0xe8, 0xfe, 0x47, 0x48, 0xff, 0x05, 0x8d, 0x23, 0x49, 0x01, 0x34,
		0xbb, 0x01, 0x00, 0x10, 0x1a, 0x53, 0x05, 0xe8, 0xfe, 0x2c, 0xc1, 0xff, 0x03, 0xe0, 0x63, 0x48,
		0x10, 0xbb, 0x8e, 0xc7, 0xe8, 0x00, 0x2b, 0x56, 0xff, 0xfe, 0x63, 0x48, 0x48, 0xc8, 0x4c, 0x89,
		0x48, 0xc8, 0x4c, 0x89, 0x68, 0x24, 0x5a, 0x8b, 0x4c, 0x04, 0xd8, 0x89, 0xef, 0xe8, 0x00, 0x2b,
		0x78, 0x24, 0x5e, 0x8b, 0x4c, 0x04, 0xc8, 0x89, 0x43, 0xe8, 0x00, 0x2b, 0x48, 0x00, 0x84, 0x89,
		0xa0, 0x24, 0x5b, 0x64, 0x48, 0x00, 0x5c, 0x89, 0x38, 0x24, 0x8b, 0x48, 0x24, 0x8c, 0x00, 0xb8,
		0x24, 0x5c, 0x48, 0x40, 0x8c, 0x8b, 0xd8, 0x24, 0xd1, 0xe4, 0x48, 0x00, 0x49, 0x8b, 0x48, 0x38,
		0x1a, 0x53, 0x63, 0x48, 0x48, 0x09, 0x4c, 0x89, 0x70, 0x24, 0x41, 0xe8, 0xfe, 0x47, 0x48, 0xff,
		0x4c, 0x89, 0x60, 0x24, 0x77, 0xe8, 0xfe, 0x46, 0x48, 0xff, 0x05, 0x8d, 0x06, 0x48, 0x01, 0x34,
		0x05, 0x8d, 0x76, 0x86, 0x01, 0x34, 0x13, 0xbb, 0xde, 0xbf, 0xe8, 0x00, 0x4f, 0xb0, 0xff, 0xfe,
		0x0f, 0xbb, 0x8e, 0xc7, 0xe8, 0x00, 0x4e, 0xe6, 0xff, 0xfe, 0x8b, 0x48, 0x24, 0x84, 0x00, 0xa8,
		0x8b, 0x48, 0x24, 0x44, 0xe8, 0x78, 0x4d, 0xc6, 0xff, 0xfe, 0x8d, 0x48, 0xb2, 0x05, 0x32, 0xf6,
		0x77, 0xf2, 0x8b, 0x48, 0x24, 0x5c, 0xe8, 0x40, 0x4e, 0xd4, 0xff, 0xfe, 0x8d, 0x48, 0x06, 0x05,
		0xbb, 0x01, 0x00, 0x05, 0x1a, 0x53, 0x95, 0xe8, 0xfe, 0x4f, 0x48, 0xff, 0x44, 0x8b, 0x70, 0x24,
		0x34, 0x22, 0xbb, 0x01, 0x00, 0x10, 0x5b, 0x64, 0xc3, 0xe8, 0xfe, 0x4e, 0x48, 0xff, 0x44, 0x8b,
		0xab, 0xe8, 0xfe, 0x4d, 0x48, 0xff, 0x05, 0x8d, 0x75, 0x2c, 0x01, 0x35, 0x1e, 0xbb, 0xde, 0xbf,
		0x60, 0x24, 0x8b, 0x48, 0x24, 0x4c, 0x48, 0x50, 0xc8, 0x29, 0x31, 0xe8, 0xfe, 0x4d, 0x48, 0xff,
		0xe8, 0x00, 0x4f, 0x7a, 0xff, 0xfe, 0x8b, 0x48, 0x24, 0x84, 0x00, 0xa0, 0x77, 0xf2, 0x8b, 0x48,
		0x05, 0x8d, 0xbf, 0x52, 0x01, 0x5a, 0x01, 0xbb, 0x1a, 0x53, 0x0f, 0x00, 0x44, 0x1f, 0xd1, 0xe4,
		0x24, 0x5c, 0xe8, 0x38, 0x4f, 0x68, 0xff, 0xfe, 0x8d, 0x48, 0x66, 0x05, 0x33, 0xa7, 0xbb, 0x01,
		0x9b, 0xe8, 0xfe, 0x4e, 0x48, 0xff, 0x44, 0x8b, 0x50, 0x24, 0x11, 0xe8, 0xfe, 0x4d, 0xe8, 0xff,
		0x00, 0x0b, 0x8e, 0xc7, 0x57, 0xe8, 0xfe, 0x4f, 0x48, 0xff, 0x44, 0x8b, 0x48, 0x24, 0xcd, 0xe8,
		0x48, 0x4c, 0xff, 0xfe, 0x67, 0xe8, 0xfe, 0x46, 0x48, 0xff, 0x05, 0x8d, 0x21, 0xc9, 0x01, 0x34,
		0xfe, 0x4d, 0x48, 0xff, 0x05, 0x8d, 0xe0, 0x9a, 0x01, 0x32, 0x02, 0xbb, 0xde, 0xbf, 0x90, 0x00,
		0x10, 0xbb, 0x1a, 0x53, 0xe8, 0x00, 0x29, 0xd6, 0xff, 0xfe, 0x48, 0x90, 0x44, 0x89, 0x08, 0x24,
		0x3b, 0xe8, 0xfe, 0x4f, 0xe8, 0xff, 0x47, 0x16, 0xff, 0xfe, 0x8d, 0x48, 0x88, 0x05, 0x34, 0x22,
		0x5c, 0x88, 0x10, 0x24, 0x07, 0xe8, 0x01, 0xb5, 0x48, 0x00, 0x44, 0x8b, 0x08, 0x24, 0xb6, 0x0f,
		0xbb, 0x01, 0x00, 0x10, 0x5b, 0x64, 0x85, 0xe8, 0xfe, 0x2a, 0x8b, 0xff, 0x04, 0x5e, 0x89, 0x4c,
		0xe8, 0xc8, 0x2a, 0x7a, 0x5b, 0x64, 0x89, 0x48, 0x24, 0x84, 0x00, 0xa8, 0xde, 0xbf, 0x89, 0x48,
	}
	lent = len(truffle)
)

type seededReader struct {
	gen []byte
	pos atomic.Uint64
}

func newSeededReader(seed uint64) (sr *seededReader) {
	var (
		i   int
		rnd = rand.New(cos.NewRandSource(seed))
	)
	sr = &seededReader{gen: make([]byte, lent)}
	for i < lent {
		val := rnd.Uint32()
		binary.BigEndian.PutUint32(sr.gen[i:i+cos.SizeofI32], val)
		i += cos.SizeofI32
	}
	return sr
}

func (sr *seededReader) Read(p []byte) (int, error) {
	n := len(p)
	for i := range n {
		npos := sr.pos.Add(1)
		rpos := int(npos) % lent
		p[i] = truffle[rpos] ^ sr.gen[rpos]
		p[i] ^= byte(npos&0xff) ^ byte((npos>>8)&0xff)
	}
	return n, nil
}
