#[allow(dead_code)]
use clap::Parser;
use formula::CNFFormat;

use itertools::Itertools;
use rayon::prelude::*;
use std::path::Path;
use std::time::Instant;
use std::{fmt::Debug, str::FromStr};

use crate::fault_trees::FaultTree;
use crate::solver::Solver;

mod fault_trees;
mod formula;
mod nodes;
mod solver;

#[derive(Parser, Debug)]
struct Arguments {
    #[clap(subcommand)]
    command: Command,
}

#[derive(Parser, Clone, Debug)]
enum Command {
    #[clap(
        about = "Outputs information about the FT: the amount of basic events, of gates and the number of clauses generated by the method."
    )]
    Info(InfoCommand),
    #[clap(
        about = "Translates the FT implicit formula to a CNF equisatisfiable formula. Outputs a DIMACS file with a wcnf."
    )]
    Translate(TranslateCommand),
    #[clap(
        about = "Executes a Solver to obtain the TEP of the FT at a given timepoint or timebound"
    )]
    Solve(SolveCommand),
}

#[derive(Parser, Debug, Clone)]
struct InfoCommand {
    /// Input file containing the fault tree in GALILEO format.
    #[arg(short, long, required = true)]
    input: String,
    /// Simplify the FT by removing one children gates.
    #[arg(long, default_value_t = false)]
    simplify: bool,
}
#[derive(Parser, Debug, Clone)]
struct SolveCommand {
    /// Input file containing the fault tree in GALILEO format.
    #[arg(short, long, required = true)]
    input: String,
    /// Solver path and arguments.
    /// Example: /solvers/gpmc:mode=1
    #[arg(short, long, value_delimiter = ':')]
    solver_path: Vec<String>,
    /// Timebounds, creates a range of values according to the command arguments: [start, end, step].
    #[arg(
        long,
        value_delimiter = ' ',
        num_args = 3,
        conflicts_with = "timepoint"
    )]
    timebounds: Option<Vec<f64>>,
    /// Compute TEP of the FT a given timepoint
    #[arg(long, default_value_t = 1.0, conflicts_with = "timebounds")]
    timepoint: f64,
    /// Output format for the CNF formual. The format gives the extension to the file. Currently supports MC21 and MCC.
    #[arg(long, default_value = "MC21")]
    format: String,
    /// Number of threads to use when timebounds are used.
    #[arg(long, default_value_t = 4)]
    num_threads: usize,
    /// Verbosity, if true, prints details at finish.
    #[arg(long, default_value_t = false)]
    verb: bool,
    /// Simplify the FT by removing one children gates.
    #[arg(long, default_value_t = true)]
    simplify: bool,
}

#[derive(Parser, Debug, Clone)]
struct TranslateCommand {
    /// Input file containing the fault tree in GALILEO format.
    #[arg(short, long, required = true)]
    input: String,
    /// Output file, writes a .wcnf file.
    #[arg(short, long)]
    output: String,
    /// Output format for the CNF formual. The format gives the extension to the file. Currently supports MC21 and MCC.
    #[arg(long, default_value = "MC21")]
    format: String,
    /// Verbosity, if true, prints more information
    #[arg(long, default_value_t = false)]
    verb: bool,
    /// Timebounds, creates a range of values according to the command arguments: [start, end, step]. Conflicts with timepoint.
    #[arg(
        long,
        value_delimiter = ' ',
        num_args = 3,
        conflicts_with = "timepoint"
    )]
    timebounds: Option<Vec<f64>>,
    /// If provided, the weights will be written to a separate file.
    #[arg(long)]
    w_file: Option<String>,
    /// Compute TEP of the FT a given timepoint. Conflicts with timebounds.
    #[arg(long, default_value_t = 1.0, conflicts_with = "timebounds")]
    timepoint: f64,
    /// Simplify the FT by removing one children gates.
    #[arg(long, default_value_t = true)]
    simplify: bool,
}

/// Outputs relevant information about the FT.
fn ft_info(command: InfoCommand) {
    let dft_filename = command.input;
    let mut ft: FaultTree<String> = FaultTree::new();
    let simplify = command.simplify;
    ft.read_from_file(&dft_filename, simplify);
    let (num_be, num_gates, num_clauses) = ft.get_info();
    println!("Number of Basic Events: {}", num_be);
    println!("Number of Gates: {}", num_gates);
    println!("Number of Clauses generated: {}", num_clauses);
}

/// Translates the FT explicit formula to a CNF file.
fn translate(command: TranslateCommand) {
    let dft_filename = command.input;
    let cnf_filename = command.output;
    let w_file = command.w_file;
    let timepoint = command.timepoint;
    let verbose = command.verb;
    let simplify = command.simplify;
    let format =
        CNFFormat::from_str(&command.format).expect("Unsupported format. Try MCC or MC21.");

    let mut ft: FaultTree<String> = FaultTree::new();
    let time_start = Instant::now();
    if verbose {
        println!("Processing file {}", dft_filename);
    }
    ft.read_from_file(&dft_filename, simplify);

    match command.timebounds {
        None => {
            let cnf_path = format!("{}_t={}.wcnf", cnf_filename, timepoint);
            ft.dump_cnf_to_file(cnf_path, format, timepoint, w_file);
            let duration = time_start.elapsed();
            if verbose {
                println!("Time elapsed: {:?}.", duration)
            }
        }
        Some(ts) => {
            let (start, end, step) = (ts[0], ts[1], ts[2]);
            let n_steps = if step != 0.0 {
                (end / step) as i64 + 1
            } else {
                2
            };
            let timebounds = (start as i64..n_steps)
                .into_iter()
                .map(|v| start + ((100 * v) as f64 * step).round() / 100.0)
                .collect_vec();
            for t in timebounds {
                let cnf_path = format!("{}_t={}.wcnf", cnf_filename, t);
                ft.dump_cnf_to_file(cnf_path, format, t.to_owned(), w_file.clone());
                let duration = time_start.elapsed();
                if verbose {
                    println!("Time elapsed: {:?}.", duration)
                }
            }
        }
    }
}

/// Compute TEP of FT, given a solver and the configuration needed.
/// Can perform multiple timepoints if range was given and use multi-threading
/// to handle each run.
fn compute_tep(command: SolveCommand) {
    let dft_filename = command.input;
    let solver_cmd = command.solver_path;
    let format =
        CNFFormat::from_str(&command.format).expect("Unsupported format. Try MCC or MC21.");
    let num_workers = command.num_threads;
    let timepoint = command.timepoint;
    let verbose = command.verb;
    let simplify = command.simplify;

    rayon::ThreadPoolBuilder::new()
        .num_threads(num_workers)
        .build_global()
        .unwrap();
    let mut ft: FaultTree<String> = FaultTree::new();
    let time_start = Instant::now();
    ft.read_from_file(&dft_filename, simplify);
    if verbose {
        println!("Processing file {}", dft_filename);
    }
    match command.timebounds {
        None => {
            let solver = Solver::from_vec(solver_cmd.to_owned());
            if verbose {
                println!(
                    "Running Solver {} for timepoint {}",
                    solver._name(),
                    timepoint
                );
            }
            let tep = solver.compute_probabilty(&ft, format, timepoint);
            let duration = time_start.elapsed();
            if !verbose {
                println!("System failure at timebound {:?} is {:?}", timepoint, tep)
            } else {
                let path = Path::new(dft_filename.as_str());
                let model_name = path.file_name().unwrap();
                println!(
                    "Model: {:?}.\nTimebound: {:?}.\nProbability of failure: {:?}.\nTime elapsed: {:?}",
                    model_name, timepoint, tep, duration
                );
            };
        }
        Some(ts) => {
            let (start, end, step) = (ts[0], ts[1], ts[2]);
            let n_steps = if step != 0.0 {
                (end / step) as i64 + 1
            } else {
                2
            };
            let timebounds = (start as i64..n_steps)
                .into_iter()
                .map(|v| start + ((100 * v) as f64 * step).round() / 100.0)
                .collect_vec();
            let _probs: Vec<(f64, f64)> = timebounds
            .into_par_iter()
            .filter_map(move |t| {
                let ft = &ft;
                if t > end {
                    None
                } else {
                    let solver = Solver::from_vec(solver_cmd.to_owned());
                    if verbose {
                        println!("Running Solver {} for timepoint {}", solver._name(), t);
                    }
                    let wmc = solver.compute_probabilty(ft, format, t);
                    let duration = time_start.elapsed();
                    if !verbose {
                        println!("System failure at timebound {:?} is {:?}", t, wmc)
                    } else {
                        let path = Path::new(dft_filename.as_str());
                        let model_name = path.file_name().unwrap();
                        println!(
                            "Model: {:?}.\nTimebound: {:?}.\nProbability of failure: {:?}.\nTime elapsed: {:?}",
                            model_name, t, wmc, duration
                        );
                    };
                    Some((t, wmc))
                }
            })
            .collect();
        }
    }
}


fn main() {
    let arguments = Arguments::parse();
    match arguments.command {
        Command::Info(command) => ft_info(command),
        Command::Solve(command) => compute_tep(command),
        Command::Translate(command) => translate(command),
    }
}
