// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using System;
using System.ComponentModel;
using System.Dynamic;

namespace Microsoft.CSharp.RuntimeBinder
{
    /// <summary>
    /// Represents a dynamic conversion in C#, providing the binding semantics and the details about the operation. 
    /// Instances of this class are generated by the C# compiler.
    /// </summary>
    internal sealed class CSharpConvertBinder : ConvertBinder
    {
        internal CSharpConversionKind ConversionKind { get { return _conversionKind; } }
        private CSharpConversionKind _conversionKind;

        internal bool IsChecked { get { return _isChecked; } }
        private bool _isChecked;

        internal Type CallingContext { get { return _callingContext; } }
        private Type _callingContext;

        private RuntimeBinder _binder;

        /// <summary>
        /// Initializes a new intsance of the <see cref="CSharpConvertBinder" />.
        /// </summary>
        /// <param name="type">The type to convert to.</param>
        /// <param name="conversionKind">The kind of conversion for this operation.</param>
        /// <param name="isChecked">True if the operation is defined in a checked context; otherwise false.</param>        
        public CSharpConvertBinder(
            Type type,
            CSharpConversionKind conversionKind,
            bool isChecked,
            Type callingContext) :
            base(type, conversionKind == CSharpConversionKind.ExplicitConversion)
        {
            _conversionKind = conversionKind;
            _isChecked = isChecked;
            _callingContext = callingContext;
            _binder = RuntimeBinder.GetInstance();
        }

        /// <summary>
        /// Performs the binding of the dynamic convert operation if the target dynamic object cannot bind.
        /// </summary>
        /// <param name="target">The target of the dynamic convert operation.</param>
        /// <param name="errorSuggestion">The binding result to use if binding fails, or null.</param>
        /// <returns>The <see cref="DynamicMetaObject"/> representing the result of the binding.</returns>
        public override DynamicMetaObject FallbackConvert(DynamicMetaObject target, DynamicMetaObject errorSuggestion)
        {
#if ENABLECOMBINDER 
            DynamicMetaObject com;
            if (!BinderHelper.IsWindowsRuntimeObject(target) && ComBinder.TryConvert(this, target, out com))
            {
                return com;
            }
#endif 
            return BinderHelper.Bind(this, _binder, new[] { target }, null, errorSuggestion);
        }
    }
}
