﻿// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using Xunit;
using System;
using System.Text;
using System.Globalization;

namespace System.Globalization.Tests
{
    public class NormalizationAll
    {
        [Fact]
        public void NormalizeTest()
        {
            bool failed = false;

            for (int i = 0; i < s_normalizationData.Length; i++)
            {
                NormalizationForm form = NormalizationForm.FormC;
                try
                {
                    form = NormalizationForm.FormC;
                    // Form C
                    failed |= VerifyConformanceInvariant(NormalizationForm.FormC, s_normalizationData[i][0], s_normalizationData[i][1], s_normalizationData[i][2], s_normalizationData[i][3], s_normalizationData[i][4]);

                    form = NormalizationForm.FormD;
                    // Form D
                    failed |= VerifyConformanceInvariant(NormalizationForm.FormD, s_normalizationData[i][0], s_normalizationData[i][1], s_normalizationData[i][2], s_normalizationData[i][3], s_normalizationData[i][4]);

                    form = NormalizationForm.FormKC;

                    // Form KC
                    failed |= VerifyConformanceInvariant(NormalizationForm.FormKC, s_normalizationData[i][0], s_normalizationData[i][1], s_normalizationData[i][2], s_normalizationData[i][3], s_normalizationData[i][4]);

                    form = NormalizationForm.FormKD;
                    // Form KD
                    failed |= VerifyConformanceInvariant(NormalizationForm.FormKD, s_normalizationData[i][0], s_normalizationData[i][1], s_normalizationData[i][2], s_normalizationData[i][3], s_normalizationData[i][4]);
                }
                catch
                {
                    Console.WriteLine("Wrong index: {0} length = {1} form = {2}", i, s_normalizationData[i].Length, form);
                    for (int j = 0; j < s_normalizationData[i].Length; j++ )
                    {
                        Console.WriteLine("'{0}'", DumpStringAsCodepoints(s_normalizationData[i][j]));
                    }
                    throw;
                }
            }

            Assert.False(failed);
        }

        // ------------------------------------------------------------------------
        // VerifyConformanceInvariant() 
        //
        // Verifies the first normalization conformance invariant for the
        // specified normalization form, where the rule as defined for all
        // Unicode normalization forms is as follows:
        //
        // 1. The following invariants must be true for all conformant 
        //    implementations:
        //
        //    NFC
        //      c2 ==  NFC(c1) ==  NFC(c2) ==  NFC(c3)
        //      c4 ==  NFC(c4) ==  NFC(c5)
        //
        //    NFD
        //      c3 ==  NFD(c1) ==  NFD(c2) ==  NFD(c3)
        //      c5 ==  NFD(c4) ==  NFD(c5)
        //
        //    NFKC
        //      c4 == NFKC(c1) == NFKC(c2) == NFKC(c3) == NFKC(c4) == NFKC(c5)
        //
        //    NFKD
        //      c5 == NFKD(c1) == NFKD(c2) == NFKD(c3) == NFKD(c4) == NFKD(c5)
        //
        // NOTE: The <platform> argument defines whether the results should
        // be taken from NLS (via NLS_PLATFORM) or NLS+ (via NLS_PLUS_PLATFORM)
        // ------------------------------------------------------------------------

        private static bool VerifyConformanceInvariant(
            NormalizationForm normForm,
            string c1,
            string c2,
            string c3,
            string c4,
            string c5
            )
        {
            bool failed = false;
            // Initialize variables according to <platform>
            string strNFc1 = c1.Normalize(normForm);
            bool bNFc1 = c1.IsNormalized(normForm);
            string strNFc2 = c2.Normalize(normForm);
            bool bNFc2 = c2.IsNormalized(normForm);
            string strNFc3 = c3.Normalize(normForm);
            bool bNFc3 = c3.IsNormalized(normForm);
            string strNFc4 = c4.Normalize(normForm);
            bool bNFc4 = c4.IsNormalized(normForm);
            string strNFc5 = c5.Normalize(normForm);
            bool bNFc5 = c5.IsNormalized(normForm);

            // Verify conformance invariants particular to <normForm>
            switch (normForm)
            {
                case NormalizationForm.FormC:
                    // c2 ==  NFC(c1) ==  NFC(c2) ==  NFC(c3)
                    failed |= CheckFailue(normForm, c2, strNFc1);
                    failed |= CheckFailue(normForm, c2, strNFc2);
                    failed |= CheckFailue(normForm, c2, strNFc3);

                    // c4 ==  NFC(c4) ==  NFC(c5)
                    failed |= CheckFailue(normForm, c4, strNFc4);
                    failed |= CheckFailue(normForm, c4, strNFc5);

                    // c2 is normalized to Form C
                    if (!bNFc2)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c2, normForm);
                        failed = true;
                    }

                    // c4 is normalized to Form C
                    if (!bNFc4)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c4, normForm);
                        failed = true;
                    }
                    break;

                case NormalizationForm.FormD:
                    // c3 ==  NFD(c1) ==  NFD(c2) ==  NFD(c3)
                    failed |= CheckFailue(normForm, c3, strNFc1);
                    failed |= CheckFailue(normForm, c3, strNFc2);
                    failed |= CheckFailue(normForm, c3, strNFc3);

                    // c5 ==  NFD(c4) ==  NFD(c5)
                    failed |= CheckFailue(normForm, c5, strNFc4);
                    failed |= CheckFailue(normForm, c5, strNFc5);

                    // c3 is normalized to Form D
                    if (!bNFc3)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c3, normForm);
                        failed = true;
                    }
                    // c5 is normalized to Form D
                    if (!bNFc5)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c5, normForm);
                        failed = true;
                    }
                    break;

                case NormalizationForm.FormKC:
                    // c4 == NFKC(c1) == NFKC(c2) == NFKC(c3) == NFKC(c4) 
                    //    == NFKC(c5)
                    failed |= CheckFailue(normForm, c4, strNFc1);
                    failed |= CheckFailue(normForm, c4, strNFc2);
                    failed |= CheckFailue(normForm, c4, strNFc3);
                    failed |= CheckFailue(normForm, c4, strNFc4);
                    failed |= CheckFailue(normForm, c4, strNFc5);

                    // c4 is normalized to Form KC
                    if (!bNFc4)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c4, normForm);
                        failed = true;
                    }
                    break;

                case NormalizationForm.FormKD:
                    // c5 == NFKD(c1) == NFKD(c2) == NFKD(c3) == NFKD(c4) 
                    //    == NFKD(c5)
                    failed |= CheckFailue(normForm, c5, strNFc1);
                    failed |= CheckFailue(normForm, c5, strNFc2);
                    failed |= CheckFailue(normForm, c5, strNFc3);
                    failed |= CheckFailue(normForm, c5, strNFc4);
                    failed |= CheckFailue(normForm, c5, strNFc5);

                    // c5 is normalized to Form KD
                    if (!bNFc5)
                    {
                        Console.WriteLine("'{0}' is marked as not normalized with form {1}", c5, normForm);
                        failed = true;
                    }
                    break;
            }
            return failed;
        }

        private static bool CheckFailue(NormalizationForm normForm, string c, string cForm)
        {
            if (!c.Equals(cForm))
            {
                Console.WriteLine("'{0}' is not matched with the normalized form '{1} with {2} normalization", DumpStringAsCodepoints(c), DumpStringAsCodepoints(cForm), cForm);
                return true;
            }

            return false;
        }

        private static string DumpStringAsCodepoints(string s)
        {
            StringBuilder sb = new StringBuilder();
            for (int i=0; i<s.Length; i++)
            {
                sb.Append("\\x");
                sb.Append(((int)s[i]).ToString("X4"));
            }
            return sb.ToString();
        }

/*

The Unicode data here came from:
http://www.unicode.org/Public/UCD/latest/ucd/NormalizationTest.txt 

Copyright: http://www.unicode.org/copyright.html

COPYRIGHT AND PERMISSION NOTICE

Copyright © 1991-2015 Unicode, Inc. All rights reserved.
Distributed under the Terms of Use in 
http://www.unicode.org/copyright.html.

Permission is hereby granted, free of charge, to any person obtaining
a copy of the Unicode data files and any associated documentation
(the "Data Files") or Unicode software and any associated documentation
(the "Software") to deal in the Data Files or Software
without restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, and/or sell copies of
the Data Files or Software, and to permit persons to whom the Data Files
or Software are furnished to do so, provided that
(a) this copyright and permission notice appear with all copies 
of the Data Files or Software,
(b) this copyright and permission notice appear in associated 
documentation, and
(c) there is clear notice in each modified Data File or in the Software
as well as in the documentation associated with the Data File(s) or
Software that the data or software has been modified.

THE DATA FILES AND SOFTWARE ARE PROVIDED "AS IS", WITHOUT WARRANTY OF
ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT OF THIRD PARTY RIGHTS.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS
NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL
DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THE DATA FILES OR SOFTWARE.

Except as contained in this notice, the name of a copyright holder
shall not be used in advertising or otherwise to promote the sale,
use or other dealings in these Data Files or Software without prior
written authorization of the copyright holder.

*/

        private static string [][] s_normalizationData = new string [][] {
            new string [] { "\u1E0A", "\u1E0A", "\u0044\u0307", "\u1E0A", "\u0044\u0307" },
            new string [] { "\u1E0C", "\u1E0C", "\u0044\u0323", "\u1E0C", "\u0044\u0323" },
            new string [] { "\u1E0A\u0323", "\u1E0C\u0307", "\u0044\u0323\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307" },
            new string [] { "\u1E0C\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307" },
            new string [] { "\u0044\u0307\u0323", "\u1E0C\u0307", "\u0044\u0323\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307" },
            new string [] { "\u0044\u0323\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307", "\u1E0C\u0307", "\u0044\u0323\u0307" },
            new string [] { "\u1E0A\u031B", "\u1E0A\u031B", "\u0044\u031B\u0307", "\u1E0A\u031B", "\u0044\u031B\u0307" },
            new string [] { "\u1E0C\u031B", "\u1E0C\u031B", "\u0044\u031B\u0323", "\u1E0C\u031B", "\u0044\u031B\u0323" },
            new string [] { "\u1E0A\u031B\u0323", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307" },
            new string [] { "\u1E0C\u031B\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307" },
            new string [] { "\u0044\u031B\u0307\u0323", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307" },
            new string [] { "\u0044\u031B\u0323\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307", "\u1E0C\u031B\u0307", "\u0044\u031B\u0323\u0307" },
            new string [] { "\u00C8", "\u00C8", "\u0045\u0300", "\u00C8", "\u0045\u0300" },
            new string [] { "\u0112", "\u0112", "\u0045\u0304", "\u0112", "\u0045\u0304" },
            new string [] { "\u0045\u0300", "\u00C8", "\u0045\u0300", "\u00C8", "\u0045\u0300" },
            new string [] { "\u0045\u0304", "\u0112", "\u0045\u0304", "\u0112", "\u0045\u0304" },
            new string [] { "\u1E14", "\u1E14", "\u0045\u0304\u0300", "\u1E14", "\u0045\u0304\u0300" },
            new string [] { "\u0112\u0300", "\u1E14", "\u0045\u0304\u0300", "\u1E14", "\u0045\u0304\u0300" },
            new string [] { "\u1E14\u0304", "\u1E14\u0304", "\u0045\u0304\u0300\u0304", "\u1E14\u0304", "\u0045\u0304\u0300\u0304" },
            new string [] { "\u0045\u0304\u0300", "\u1E14", "\u0045\u0304\u0300", "\u1E14", "\u0045\u0304\u0300" },
            new string [] { "\u0045\u0300\u0304", "\u00C8\u0304", "\u0045\u0300\u0304", "\u00C8\u0304", "\u0045\u0300\u0304" },
            new string [] { "\u05B8\u05B9\u05B1\u0591\u05C3\u05B0\u05AC\u059F", "\u05B1\u05B8\u05B9\u0591\u05C3\u05B0\u05AC\u059F", "\u05B1\u05B8\u05B9\u0591\u05C3\u05B0\u05AC\u059F", "\u05B1\u05B8\u05B9\u0591\u05C3\u05B0\u05AC\u059F", "\u05B1\u05B8\u05B9\u0591\u05C3\u05B0\u05AC\u059F" },
            new string [] { "\u0592\u05B7\u05BC\u05A5\u05B0\u05C0\u05C4\u05AD", "\u05B0\u05B7\u05BC\u05A5\u0592\u05C0\u05AD\u05C4", "\u05B0\u05B7\u05BC\u05A5\u0592\u05C0\u05AD\u05C4", "\u05B0\u05B7\u05BC\u05A5\u0592\u05C0\u05AD\u05C4", "\u05B0\u05B7\u05BC\u05A5\u0592\u05C0\u05AD\u05C4" },
            new string [] { "\u00A0", "\u00A0", "\u00A0", "\u0020", "\u0020" },
            new string [] { "\u00A8", "\u00A8", "\u00A8", "\u0020\u0308", "\u0020\u0308" },
            new string [] { "\u00AA", "\u00AA", "\u00AA", "\u0061", "\u0061" },
            new string [] { "\u00AF", "\u00AF", "\u00AF", "\u0020\u0304", "\u0020\u0304" },
            new string [] { "\u00B2", "\u00B2", "\u00B2", "\u0032", "\u0032" },
            new string [] { "\u00B3", "\u00B3", "\u00B3", "\u0033", "\u0033" },
            new string [] { "\u00B4", "\u00B4", "\u00B4", "\u0020\u0301", "\u0020\u0301" },
            new string [] { "\u00B5", "\u00B5", "\u00B5", "\u03BC", "\u03BC" },
            new string [] { "\u00B8", "\u00B8", "\u00B8", "\u0020\u0327", "\u0020\u0327" },
            new string [] { "\u00B9", "\u00B9", "\u00B9", "\u0031", "\u0031" },
            new string [] { "\u00BA", "\u00BA", "\u00BA", "\u006F", "\u006F" },
            new string [] { "\u00BC", "\u00BC", "\u00BC", "\u0031\u2044\u0034", "\u0031\u2044\u0034" },
            new string [] { "\u00BD", "\u00BD", "\u00BD", "\u0031\u2044\u0032", "\u0031\u2044\u0032" },
            new string [] { "\u00BE", "\u00BE", "\u00BE", "\u0033\u2044\u0034", "\u0033\u2044\u0034" },
            new string [] { "\u00C0", "\u00C0", "\u0041\u0300", "\u00C0", "\u0041\u0300" },
            new string [] { "\u00C1", "\u00C1", "\u0041\u0301", "\u00C1", "\u0041\u0301" },
            new string [] { "\u00C2", "\u00C2", "\u0041\u0302", "\u00C2", "\u0041\u0302" },
            new string [] { "\u00C3", "\u00C3", "\u0041\u0303", "\u00C3", "\u0041\u0303" },
            new string [] { "\u00C4", "\u00C4", "\u0041\u0308", "\u00C4", "\u0041\u0308" },
            new string [] { "\u00C5", "\u00C5", "\u0041\u030A", "\u00C5", "\u0041\u030A" },
            new string [] { "\u00C7", "\u00C7", "\u0043\u0327", "\u00C7", "\u0043\u0327" },
            new string [] { "\u00C8", "\u00C8", "\u0045\u0300", "\u00C8", "\u0045\u0300" },
            new string [] { "\u00C9", "\u00C9", "\u0045\u0301", "\u00C9", "\u0045\u0301" },
            new string [] { "\u00CA", "\u00CA", "\u0045\u0302", "\u00CA", "\u0045\u0302" },
            new string [] { "\u00CB", "\u00CB", "\u0045\u0308", "\u00CB", "\u0045\u0308" },
            new string [] { "\u00CC", "\u00CC", "\u0049\u0300", "\u00CC", "\u0049\u0300" },
            new string [] { "\u00CD", "\u00CD", "\u0049\u0301", "\u00CD", "\u0049\u0301" },
            new string [] { "\u00CE", "\u00CE", "\u0049\u0302", "\u00CE", "\u0049\u0302" },
            new string [] { "\u00CF", "\u00CF", "\u0049\u0308", "\u00CF", "\u0049\u0308" },
            new string [] { "\u00D1", "\u00D1", "\u004E\u0303", "\u00D1", "\u004E\u0303" },
            new string [] { "\u00D2", "\u00D2", "\u004F\u0300", "\u00D2", "\u004F\u0300" },
            new string [] { "\u00D3", "\u00D3", "\u004F\u0301", "\u00D3", "\u004F\u0301" },
            new string [] { "\u00D4", "\u00D4", "\u004F\u0302", "\u00D4", "\u004F\u0302" },
            new string [] { "\u00D5", "\u00D5", "\u004F\u0303", "\u00D5", "\u004F\u0303" },
            new string [] { "\u00D6", "\u00D6", "\u004F\u0308", "\u00D6", "\u004F\u0308" },
            new string [] { "\u00D9", "\u00D9", "\u0055\u0300", "\u00D9", "\u0055\u0300" },
            new string [] { "\u00DA", "\u00DA", "\u0055\u0301", "\u00DA", "\u0055\u0301" },
            new string [] { "\u00DB", "\u00DB", "\u0055\u0302", "\u00DB", "\u0055\u0302" },
            new string [] { "\u00DC", "\u00DC", "\u0055\u0308", "\u00DC", "\u0055\u0308" },
            new string [] { "\u00DD", "\u00DD", "\u0059\u0301", "\u00DD", "\u0059\u0301" },
            new string [] { "\u00E0", "\u00E0", "\u0061\u0300", "\u00E0", "\u0061\u0300" },
            new string [] { "\u00E1", "\u00E1", "\u0061\u0301", "\u00E1", "\u0061\u0301" },
            new string [] { "\u00E2", "\u00E2", "\u0061\u0302", "\u00E2", "\u0061\u0302" },
            new string [] { "\u00E3", "\u00E3", "\u0061\u0303", "\u00E3", "\u0061\u0303" },
            new string [] { "\u00E4", "\u00E4", "\u0061\u0308", "\u00E4", "\u0061\u0308" },
            new string [] { "\u00E5", "\u00E5", "\u0061\u030A", "\u00E5", "\u0061\u030A" },
            new string [] { "\u00E7", "\u00E7", "\u0063\u0327", "\u00E7", "\u0063\u0327" },
            new string [] { "\u00E8", "\u00E8", "\u0065\u0300", "\u00E8", "\u0065\u0300" },
            new string [] { "\u00E9", "\u00E9", "\u0065\u0301", "\u00E9", "\u0065\u0301" },
            new string [] { "\u00EA", "\u00EA", "\u0065\u0302", "\u00EA", "\u0065\u0302" },
            new string [] { "\u00EB", "\u00EB", "\u0065\u0308", "\u00EB", "\u0065\u0308" },
            new string [] { "\u00EC", "\u00EC", "\u0069\u0300", "\u00EC", "\u0069\u0300" },
            new string [] { "\u00ED", "\u00ED", "\u0069\u0301", "\u00ED", "\u0069\u0301" },
            new string [] { "\u00EE", "\u00EE", "\u0069\u0302", "\u00EE", "\u0069\u0302" },
            new string [] { "\u00EF", "\u00EF", "\u0069\u0308", "\u00EF", "\u0069\u0308" },
            new string [] { "\u00F1", "\u00F1", "\u006E\u0303", "\u00F1", "\u006E\u0303" },
            new string [] { "\u00F2", "\u00F2", "\u006F\u0300", "\u00F2", "\u006F\u0300" },
            new string [] { "\u00F3", "\u00F3", "\u006F\u0301", "\u00F3", "\u006F\u0301" },
            new string [] { "\u00F4", "\u00F4", "\u006F\u0302", "\u00F4", "\u006F\u0302" },
            new string [] { "\u00F5", "\u00F5", "\u006F\u0303", "\u00F5", "\u006F\u0303" },
            new string [] { "\u00F6", "\u00F6", "\u006F\u0308", "\u00F6", "\u006F\u0308" },
            new string [] { "\u00F9", "\u00F9", "\u0075\u0300", "\u00F9", "\u0075\u0300" },
            new string [] { "\u00FA", "\u00FA", "\u0075\u0301", "\u00FA", "\u0075\u0301" },
            new string [] { "\u00FB", "\u00FB", "\u0075\u0302", "\u00FB", "\u0075\u0302" },
            new string [] { "\u00FC", "\u00FC", "\u0075\u0308", "\u00FC", "\u0075\u0308" },
            new string [] { "\u00FD", "\u00FD", "\u0079\u0301", "\u00FD", "\u0079\u0301" },
            new string [] { "\u00FF", "\u00FF", "\u0079\u0308", "\u00FF", "\u0079\u0308" },
            new string [] { "\u0100", "\u0100", "\u0041\u0304", "\u0100", "\u0041\u0304" },
            new string [] { "\u0101", "\u0101", "\u0061\u0304", "\u0101", "\u0061\u0304" },
            new string [] { "\u0102", "\u0102", "\u0041\u0306", "\u0102", "\u0041\u0306" },
            new string [] { "\u0103", "\u0103", "\u0061\u0306", "\u0103", "\u0061\u0306" },
            new string [] { "\u0104", "\u0104", "\u0041\u0328", "\u0104", "\u0041\u0328" },
            new string [] { "\u0105", "\u0105", "\u0061\u0328", "\u0105", "\u0061\u0328" },
            new string [] { "\u0106", "\u0106", "\u0043\u0301", "\u0106", "\u0043\u0301" },
            new string [] { "\u0107", "\u0107", "\u0063\u0301", "\u0107", "\u0063\u0301" },
            new string [] { "\u0108", "\u0108", "\u0043\u0302", "\u0108", "\u0043\u0302" },
            new string [] { "\u0109", "\u0109", "\u0063\u0302", "\u0109", "\u0063\u0302" },
            new string [] { "\u010A", "\u010A", "\u0043\u0307", "\u010A", "\u0043\u0307" },
            new string [] { "\u010B", "\u010B", "\u0063\u0307", "\u010B", "\u0063\u0307" },
            new string [] { "\u010C", "\u010C", "\u0043\u030C", "\u010C", "\u0043\u030C" },
            new string [] { "\u010D", "\u010D", "\u0063\u030C", "\u010D", "\u0063\u030C" },
            new string [] { "\u010E", "\u010E", "\u0044\u030C", "\u010E", "\u0044\u030C" },
            new string [] { "\u010F", "\u010F", "\u0064\u030C", "\u010F", "\u0064\u030C" },
            new string [] { "\u0112", "\u0112", "\u0045\u0304", "\u0112", "\u0045\u0304" },
            new string [] { "\u0113", "\u0113", "\u0065\u0304", "\u0113", "\u0065\u0304" },
            new string [] { "\u0114", "\u0114", "\u0045\u0306", "\u0114", "\u0045\u0306" },
            new string [] { "\u0115", "\u0115", "\u0065\u0306", "\u0115", "\u0065\u0306" },
            new string [] { "\u0116", "\u0116", "\u0045\u0307", "\u0116", "\u0045\u0307" },
            new string [] { "\u0117", "\u0117", "\u0065\u0307", "\u0117", "\u0065\u0307" },
            new string [] { "\u0118", "\u0118", "\u0045\u0328", "\u0118", "\u0045\u0328" },
            new string [] { "\u0119", "\u0119", "\u0065\u0328", "\u0119", "\u0065\u0328" },
            new string [] { "\u011A", "\u011A", "\u0045\u030C", "\u011A", "\u0045\u030C" },
            new string [] { "\u011B", "\u011B", "\u0065\u030C", "\u011B", "\u0065\u030C" },
            new string [] { "\u011C", "\u011C", "\u0047\u0302", "\u011C", "\u0047\u0302" },
            new string [] { "\u011D", "\u011D", "\u0067\u0302", "\u011D", "\u0067\u0302" },
            new string [] { "\u011E", "\u011E", "\u0047\u0306", "\u011E", "\u0047\u0306" },
            new string [] { "\u011F", "\u011F", "\u0067\u0306", "\u011F", "\u0067\u0306" },
            new string [] { "\u0120", "\u0120", "\u0047\u0307", "\u0120", "\u0047\u0307" },
            new string [] { "\u0121", "\u0121", "\u0067\u0307", "\u0121", "\u0067\u0307" },
            new string [] { "\u0122", "\u0122", "\u0047\u0327", "\u0122", "\u0047\u0327" },
            new string [] { "\u0123", "\u0123", "\u0067\u0327", "\u0123", "\u0067\u0327" },
            new string [] { "\u0124", "\u0124", "\u0048\u0302", "\u0124", "\u0048\u0302" },
            new string [] { "\u0125", "\u0125", "\u0068\u0302", "\u0125", "\u0068\u0302" },
            new string [] { "\u0128", "\u0128", "\u0049\u0303", "\u0128", "\u0049\u0303" },
            new string [] { "\u0129", "\u0129", "\u0069\u0303", "\u0129", "\u0069\u0303" },
            new string [] { "\u012A", "\u012A", "\u0049\u0304", "\u012A", "\u0049\u0304" },
            new string [] { "\u012B", "\u012B", "\u0069\u0304", "\u012B", "\u0069\u0304" },
            new string [] { "\u012C", "\u012C", "\u0049\u0306", "\u012C", "\u0049\u0306" },
            new string [] { "\u012D", "\u012D", "\u0069\u0306", "\u012D", "\u0069\u0306" },
            new string [] { "\u012E", "\u012E", "\u0049\u0328", "\u012E", "\u0049\u0328" },
            new string [] { "\u012F", "\u012F", "\u0069\u0328", "\u012F", "\u0069\u0328" },
            new string [] { "\u0130", "\u0130", "\u0049\u0307", "\u0130", "\u0049\u0307" },
            new string [] { "\u0132", "\u0132", "\u0132", "\u0049\u004A", "\u0049\u004A" },
            new string [] { "\u0133", "\u0133", "\u0133", "\u0069\u006A", "\u0069\u006A" },
            new string [] { "\u0134", "\u0134", "\u004A\u0302", "\u0134", "\u004A\u0302" },
            new string [] { "\u0135", "\u0135", "\u006A\u0302", "\u0135", "\u006A\u0302" },
            new string [] { "\u0136", "\u0136", "\u004B\u0327", "\u0136", "\u004B\u0327" },
            new string [] { "\u0137", "\u0137", "\u006B\u0327", "\u0137", "\u006B\u0327" },
            new string [] { "\u0139", "\u0139", "\u004C\u0301", "\u0139", "\u004C\u0301" },
            new string [] { "\u013A", "\u013A", "\u006C\u0301", "\u013A", "\u006C\u0301" },
            new string [] { "\u013B", "\u013B", "\u004C\u0327", "\u013B", "\u004C\u0327" },
            new string [] { "\u013C", "\u013C", "\u006C\u0327", "\u013C", "\u006C\u0327" },
            new string [] { "\u013D", "\u013D", "\u004C\u030C", "\u013D", "\u004C\u030C" },
            new string [] { "\u013E", "\u013E", "\u006C\u030C", "\u013E", "\u006C\u030C" },
            new string [] { "\u013F", "\u013F", "\u013F", "\u004C\u00B7", "\u004C\u00B7" },
            new string [] { "\u0140", "\u0140", "\u0140", "\u006C\u00B7", "\u006C\u00B7" },
            new string [] { "\u0143", "\u0143", "\u004E\u0301", "\u0143", "\u004E\u0301" },
            new string [] { "\u0144", "\u0144", "\u006E\u0301", "\u0144", "\u006E\u0301" },
            new string [] { "\u0145", "\u0145", "\u004E\u0327", "\u0145", "\u004E\u0327" },
            new string [] { "\u0146", "\u0146", "\u006E\u0327", "\u0146", "\u006E\u0327" },
            new string [] { "\u0147", "\u0147", "\u004E\u030C", "\u0147", "\u004E\u030C" },
            new string [] { "\u0148", "\u0148", "\u006E\u030C", "\u0148", "\u006E\u030C" },
            new string [] { "\u0149", "\u0149", "\u0149", "\u02BC\u006E", "\u02BC\u006E" },
            new string [] { "\u014C", "\u014C", "\u004F\u0304", "\u014C", "\u004F\u0304" },
            new string [] { "\u014D", "\u014D", "\u006F\u0304", "\u014D", "\u006F\u0304" },
            new string [] { "\u014E", "\u014E", "\u004F\u0306", "\u014E", "\u004F\u0306" },
            new string [] { "\u014F", "\u014F", "\u006F\u0306", "\u014F", "\u006F\u0306" },
            new string [] { "\u0150", "\u0150", "\u004F\u030B", "\u0150", "\u004F\u030B" },
            new string [] { "\u0151", "\u0151", "\u006F\u030B", "\u0151", "\u006F\u030B" },
            new string [] { "\u0154", "\u0154", "\u0052\u0301", "\u0154", "\u0052\u0301" },
            new string [] { "\u0155", "\u0155", "\u0072\u0301", "\u0155", "\u0072\u0301" },
            new string [] { "\u0156", "\u0156", "\u0052\u0327", "\u0156", "\u0052\u0327" },
            new string [] { "\u0157", "\u0157", "\u0072\u0327", "\u0157", "\u0072\u0327" },
            new string [] { "\u0158", "\u0158", "\u0052\u030C", "\u0158", "\u0052\u030C" },
            new string [] { "\u0159", "\u0159", "\u0072\u030C", "\u0159", "\u0072\u030C" },
            new string [] { "\u015A", "\u015A", "\u0053\u0301", "\u015A", "\u0053\u0301" },
            new string [] { "\u015B", "\u015B", "\u0073\u0301", "\u015B", "\u0073\u0301" },
            new string [] { "\u015C", "\u015C", "\u0053\u0302", "\u015C", "\u0053\u0302" },
            new string [] { "\u015D", "\u015D", "\u0073\u0302", "\u015D", "\u0073\u0302" },
            new string [] { "\u015E", "\u015E", "\u0053\u0327", "\u015E", "\u0053\u0327" },
            new string [] { "\u015F", "\u015F", "\u0073\u0327", "\u015F", "\u0073\u0327" },
            new string [] { "\u0160", "\u0160", "\u0053\u030C", "\u0160", "\u0053\u030C" },
            new string [] { "\u0161", "\u0161", "\u0073\u030C", "\u0161", "\u0073\u030C" },
            new string [] { "\u0162", "\u0162", "\u0054\u0327", "\u0162", "\u0054\u0327" },
            new string [] { "\u0163", "\u0163", "\u0074\u0327", "\u0163", "\u0074\u0327" },
            new string [] { "\u0164", "\u0164", "\u0054\u030C", "\u0164", "\u0054\u030C" },
            new string [] { "\u0165", "\u0165", "\u0074\u030C", "\u0165", "\u0074\u030C" },
            new string [] { "\u0168", "\u0168", "\u0055\u0303", "\u0168", "\u0055\u0303" },
            new string [] { "\u0169", "\u0169", "\u0075\u0303", "\u0169", "\u0075\u0303" },
            new string [] { "\u016A", "\u016A", "\u0055\u0304", "\u016A", "\u0055\u0304" },
            new string [] { "\u016B", "\u016B", "\u0075\u0304", "\u016B", "\u0075\u0304" },
            new string [] { "\u016C", "\u016C", "\u0055\u0306", "\u016C", "\u0055\u0306" },
            new string [] { "\u016D", "\u016D", "\u0075\u0306", "\u016D", "\u0075\u0306" },
            new string [] { "\u016E", "\u016E", "\u0055\u030A", "\u016E", "\u0055\u030A" },
            new string [] { "\u016F", "\u016F", "\u0075\u030A", "\u016F", "\u0075\u030A" },
            new string [] { "\u0170", "\u0170", "\u0055\u030B", "\u0170", "\u0055\u030B" },
            new string [] { "\u0171", "\u0171", "\u0075\u030B", "\u0171", "\u0075\u030B" },
            new string [] { "\u0172", "\u0172", "\u0055\u0328", "\u0172", "\u0055\u0328" },
            new string [] { "\u0173", "\u0173", "\u0075\u0328", "\u0173", "\u0075\u0328" },
            new string [] { "\u0174", "\u0174", "\u0057\u0302", "\u0174", "\u0057\u0302" },
            new string [] { "\u0175", "\u0175", "\u0077\u0302", "\u0175", "\u0077\u0302" },
            new string [] { "\u0176", "\u0176", "\u0059\u0302", "\u0176", "\u0059\u0302" },
            new string [] { "\u0177", "\u0177", "\u0079\u0302", "\u0177", "\u0079\u0302" },
            new string [] { "\u0178", "\u0178", "\u0059\u0308", "\u0178", "\u0059\u0308" },
            new string [] { "\u0179", "\u0179", "\u005A\u0301", "\u0179", "\u005A\u0301" },
            new string [] { "\u017A", "\u017A", "\u007A\u0301", "\u017A", "\u007A\u0301" },
            new string [] { "\u017B", "\u017B", "\u005A\u0307", "\u017B", "\u005A\u0307" },
            new string [] { "\u017C", "\u017C", "\u007A\u0307", "\u017C", "\u007A\u0307" },
            new string [] { "\u017D", "\u017D", "\u005A\u030C", "\u017D", "\u005A\u030C" },
            new string [] { "\u017E", "\u017E", "\u007A\u030C", "\u017E", "\u007A\u030C" },
            new string [] { "\u017F", "\u017F", "\u017F", "\u0073", "\u0073" },
            new string [] { "\u01A0", "\u01A0", "\u004F\u031B", "\u01A0", "\u004F\u031B" },
            new string [] { "\u01A1", "\u01A1", "\u006F\u031B", "\u01A1", "\u006F\u031B" },
            new string [] { "\u01AF", "\u01AF", "\u0055\u031B", "\u01AF", "\u0055\u031B" },
            new string [] { "\u01B0", "\u01B0", "\u0075\u031B", "\u01B0", "\u0075\u031B" },
            new string [] { "\u01C4", "\u01C4", "\u01C4", "\u0044\u017D", "\u0044\u005A\u030C" },
            new string [] { "\u01C5", "\u01C5", "\u01C5", "\u0044\u017E", "\u0044\u007A\u030C" },
            new string [] { "\u01C6", "\u01C6", "\u01C6", "\u0064\u017E", "\u0064\u007A\u030C" },
            new string [] { "\u01C7", "\u01C7", "\u01C7", "\u004C\u004A", "\u004C\u004A" },
            new string [] { "\u01C8", "\u01C8", "\u01C8", "\u004C\u006A", "\u004C\u006A" },
            new string [] { "\u01C9", "\u01C9", "\u01C9", "\u006C\u006A", "\u006C\u006A" },
            new string [] { "\u01CA", "\u01CA", "\u01CA", "\u004E\u004A", "\u004E\u004A" },
            new string [] { "\u01CB", "\u01CB", "\u01CB", "\u004E\u006A", "\u004E\u006A" },
            new string [] { "\u01CC", "\u01CC", "\u01CC", "\u006E\u006A", "\u006E\u006A" },
            new string [] { "\u01CD", "\u01CD", "\u0041\u030C", "\u01CD", "\u0041\u030C" },
            new string [] { "\u01CE", "\u01CE", "\u0061\u030C", "\u01CE", "\u0061\u030C" },
            new string [] { "\u01CF", "\u01CF", "\u0049\u030C", "\u01CF", "\u0049\u030C" },
            new string [] { "\u01D0", "\u01D0", "\u0069\u030C", "\u01D0", "\u0069\u030C" },
            new string [] { "\u01D1", "\u01D1", "\u004F\u030C", "\u01D1", "\u004F\u030C" },
            new string [] { "\u01D2", "\u01D2", "\u006F\u030C", "\u01D2", "\u006F\u030C" },
            new string [] { "\u01D3", "\u01D3", "\u0055\u030C", "\u01D3", "\u0055\u030C" },
            new string [] { "\u01D4", "\u01D4", "\u0075\u030C", "\u01D4", "\u0075\u030C" },
            new string [] { "\u01D5", "\u01D5", "\u0055\u0308\u0304", "\u01D5", "\u0055\u0308\u0304" },
            new string [] { "\u01D6", "\u01D6", "\u0075\u0308\u0304", "\u01D6", "\u0075\u0308\u0304" },
            new string [] { "\u01D7", "\u01D7", "\u0055\u0308\u0301", "\u01D7", "\u0055\u0308\u0301" },
            new string [] { "\u01D8", "\u01D8", "\u0075\u0308\u0301", "\u01D8", "\u0075\u0308\u0301" },
            new string [] { "\u01D9", "\u01D9", "\u0055\u0308\u030C", "\u01D9", "\u0055\u0308\u030C" },
            new string [] { "\u01DA", "\u01DA", "\u0075\u0308\u030C", "\u01DA", "\u0075\u0308\u030C" },
            new string [] { "\u01DB", "\u01DB", "\u0055\u0308\u0300", "\u01DB", "\u0055\u0308\u0300" },
            new string [] { "\u01DC", "\u01DC", "\u0075\u0308\u0300", "\u01DC", "\u0075\u0308\u0300" },
            new string [] { "\u01DE", "\u01DE", "\u0041\u0308\u0304", "\u01DE", "\u0041\u0308\u0304" },
            new string [] { "\u01DF", "\u01DF", "\u0061\u0308\u0304", "\u01DF", "\u0061\u0308\u0304" },
            new string [] { "\u01E0", "\u01E0", "\u0041\u0307\u0304", "\u01E0", "\u0041\u0307\u0304" },
            new string [] { "\u01E1", "\u01E1", "\u0061\u0307\u0304", "\u01E1", "\u0061\u0307\u0304" },
            new string [] { "\u01E2", "\u01E2", "\u00C6\u0304", "\u01E2", "\u00C6\u0304" },
            new string [] { "\u01E3", "\u01E3", "\u00E6\u0304", "\u01E3", "\u00E6\u0304" },
            new string [] { "\u01E6", "\u01E6", "\u0047\u030C", "\u01E6", "\u0047\u030C" },
            new string [] { "\u01E7", "\u01E7", "\u0067\u030C", "\u01E7", "\u0067\u030C" },
            new string [] { "\u01E8", "\u01E8", "\u004B\u030C", "\u01E8", "\u004B\u030C" },
            new string [] { "\u01E9", "\u01E9", "\u006B\u030C", "\u01E9", "\u006B\u030C" },
            new string [] { "\u01EA", "\u01EA", "\u004F\u0328", "\u01EA", "\u004F\u0328" },
            new string [] { "\u01EB", "\u01EB", "\u006F\u0328", "\u01EB", "\u006F\u0328" },
            new string [] { "\u01EC", "\u01EC", "\u004F\u0328\u0304", "\u01EC", "\u004F\u0328\u0304" },
            new string [] { "\u01ED", "\u01ED", "\u006F\u0328\u0304", "\u01ED", "\u006F\u0328\u0304" },
            new string [] { "\u01EE", "\u01EE", "\u01B7\u030C", "\u01EE", "\u01B7\u030C" },
            new string [] { "\u01EF", "\u01EF", "\u0292\u030C", "\u01EF", "\u0292\u030C" },
            new string [] { "\u01F0", "\u01F0", "\u006A\u030C", "\u01F0", "\u006A\u030C" },
            new string [] { "\u01F1", "\u01F1", "\u01F1", "\u0044\u005A", "\u0044\u005A" },
            new string [] { "\u01F2", "\u01F2", "\u01F2", "\u0044\u007A", "\u0044\u007A" },
            new string [] { "\u01F3", "\u01F3", "\u01F3", "\u0064\u007A", "\u0064\u007A" },
            new string [] { "\u01F4", "\u01F4", "\u0047\u0301", "\u01F4", "\u0047\u0301" },
            new string [] { "\u01F5", "\u01F5", "\u0067\u0301", "\u01F5", "\u0067\u0301" },
            new string [] { "\u01F8", "\u01F8", "\u004E\u0300", "\u01F8", "\u004E\u0300" },
            new string [] { "\u01F9", "\u01F9", "\u006E\u0300", "\u01F9", "\u006E\u0300" },
            new string [] { "\u01FA", "\u01FA", "\u0041\u030A\u0301", "\u01FA", "\u0041\u030A\u0301" },
            new string [] { "\u01FB", "\u01FB", "\u0061\u030A\u0301", "\u01FB", "\u0061\u030A\u0301" },
            new string [] { "\u01FC", "\u01FC", "\u00C6\u0301", "\u01FC", "\u00C6\u0301" },
            new string [] { "\u01FD", "\u01FD", "\u00E6\u0301", "\u01FD", "\u00E6\u0301" },
            new string [] { "\u01FE", "\u01FE", "\u00D8\u0301", "\u01FE", "\u00D8\u0301" },
            new string [] { "\u01FF", "\u01FF", "\u00F8\u0301", "\u01FF", "\u00F8\u0301" },
            new string [] { "\u0200", "\u0200", "\u0041\u030F", "\u0200", "\u0041\u030F" },
            new string [] { "\u0201", "\u0201", "\u0061\u030F", "\u0201", "\u0061\u030F" },
            new string [] { "\u0202", "\u0202", "\u0041\u0311", "\u0202", "\u0041\u0311" },
            new string [] { "\u0203", "\u0203", "\u0061\u0311", "\u0203", "\u0061\u0311" },
            new string [] { "\u0204", "\u0204", "\u0045\u030F", "\u0204", "\u0045\u030F" },
            new string [] { "\u0205", "\u0205", "\u0065\u030F", "\u0205", "\u0065\u030F" },
            new string [] { "\u0206", "\u0206", "\u0045\u0311", "\u0206", "\u0045\u0311" },
            new string [] { "\u0207", "\u0207", "\u0065\u0311", "\u0207", "\u0065\u0311" },
            new string [] { "\u0208", "\u0208", "\u0049\u030F", "\u0208", "\u0049\u030F" },
            new string [] { "\u0209", "\u0209", "\u0069\u030F", "\u0209", "\u0069\u030F" },
            new string [] { "\u020A", "\u020A", "\u0049\u0311", "\u020A", "\u0049\u0311" },
            new string [] { "\u020B", "\u020B", "\u0069\u0311", "\u020B", "\u0069\u0311" },
            new string [] { "\u020C", "\u020C", "\u004F\u030F", "\u020C", "\u004F\u030F" },
            new string [] { "\u020D", "\u020D", "\u006F\u030F", "\u020D", "\u006F\u030F" },
            new string [] { "\u020E", "\u020E", "\u004F\u0311", "\u020E", "\u004F\u0311" },
            new string [] { "\u020F", "\u020F", "\u006F\u0311", "\u020F", "\u006F\u0311" },
            new string [] { "\u0210", "\u0210", "\u0052\u030F", "\u0210", "\u0052\u030F" },
            new string [] { "\u0211", "\u0211", "\u0072\u030F", "\u0211", "\u0072\u030F" },
            new string [] { "\u0212", "\u0212", "\u0052\u0311", "\u0212", "\u0052\u0311" },
            new string [] { "\u0213", "\u0213", "\u0072\u0311", "\u0213", "\u0072\u0311" },
            new string [] { "\u0214", "\u0214", "\u0055\u030F", "\u0214", "\u0055\u030F" },
            new string [] { "\u0215", "\u0215", "\u0075\u030F", "\u0215", "\u0075\u030F" },
            new string [] { "\u0216", "\u0216", "\u0055\u0311", "\u0216", "\u0055\u0311" },
            new string [] { "\u0217", "\u0217", "\u0075\u0311", "\u0217", "\u0075\u0311" },
            new string [] { "\u0218", "\u0218", "\u0053\u0326", "\u0218", "\u0053\u0326" },
            new string [] { "\u0219", "\u0219", "\u0073\u0326", "\u0219", "\u0073\u0326" },
            new string [] { "\u021A", "\u021A", "\u0054\u0326", "\u021A", "\u0054\u0326" },
            new string [] { "\u021B", "\u021B", "\u0074\u0326", "\u021B", "\u0074\u0326" },
            new string [] { "\u021E", "\u021E", "\u0048\u030C", "\u021E", "\u0048\u030C" },
            new string [] { "\u021F", "\u021F", "\u0068\u030C", "\u021F", "\u0068\u030C" },
            new string [] { "\u0226", "\u0226", "\u0041\u0307", "\u0226", "\u0041\u0307" },
            new string [] { "\u0227", "\u0227", "\u0061\u0307", "\u0227", "\u0061\u0307" },
            new string [] { "\u0228", "\u0228", "\u0045\u0327", "\u0228", "\u0045\u0327" },
            new string [] { "\u0229", "\u0229", "\u0065\u0327", "\u0229", "\u0065\u0327" },
            new string [] { "\u022A", "\u022A", "\u004F\u0308\u0304", "\u022A", "\u004F\u0308\u0304" },
            new string [] { "\u022B", "\u022B", "\u006F\u0308\u0304", "\u022B", "\u006F\u0308\u0304" },
            new string [] { "\u022C", "\u022C", "\u004F\u0303\u0304", "\u022C", "\u004F\u0303\u0304" },
            new string [] { "\u022D", "\u022D", "\u006F\u0303\u0304", "\u022D", "\u006F\u0303\u0304" },
            new string [] { "\u022E", "\u022E", "\u004F\u0307", "\u022E", "\u004F\u0307" },
            new string [] { "\u022F", "\u022F", "\u006F\u0307", "\u022F", "\u006F\u0307" },
            new string [] { "\u0230", "\u0230", "\u004F\u0307\u0304", "\u0230", "\u004F\u0307\u0304" },
            new string [] { "\u0231", "\u0231", "\u006F\u0307\u0304", "\u0231", "\u006F\u0307\u0304" },
            new string [] { "\u0232", "\u0232", "\u0059\u0304", "\u0232", "\u0059\u0304" },
            new string [] { "\u0233", "\u0233", "\u0079\u0304", "\u0233", "\u0079\u0304" },
            new string [] { "\u02B0", "\u02B0", "\u02B0", "\u0068", "\u0068" },
            new string [] { "\u02B1", "\u02B1", "\u02B1", "\u0266", "\u0266" },
            new string [] { "\u02B2", "\u02B2", "\u02B2", "\u006A", "\u006A" },
            new string [] { "\u02B3", "\u02B3", "\u02B3", "\u0072", "\u0072" },
            new string [] { "\u02B4", "\u02B4", "\u02B4", "\u0279", "\u0279" },
            new string [] { "\u02B5", "\u02B5", "\u02B5", "\u027B", "\u027B" },
            new string [] { "\u02B6", "\u02B6", "\u02B6", "\u0281", "\u0281" },
            new string [] { "\u02B7", "\u02B7", "\u02B7", "\u0077", "\u0077" },
            new string [] { "\u02B8", "\u02B8", "\u02B8", "\u0079", "\u0079" },
            new string [] { "\u02D8", "\u02D8", "\u02D8", "\u0020\u0306", "\u0020\u0306" },
            new string [] { "\u02D9", "\u02D9", "\u02D9", "\u0020\u0307", "\u0020\u0307" },
            new string [] { "\u02DA", "\u02DA", "\u02DA", "\u0020\u030A", "\u0020\u030A" },
            new string [] { "\u02DB", "\u02DB", "\u02DB", "\u0020\u0328", "\u0020\u0328" },
            new string [] { "\u02DC", "\u02DC", "\u02DC", "\u0020\u0303", "\u0020\u0303" },
            new string [] { "\u02DD", "\u02DD", "\u02DD", "\u0020\u030B", "\u0020\u030B" },
            new string [] { "\u02E0", "\u02E0", "\u02E0", "\u0263", "\u0263" },
            new string [] { "\u02E1", "\u02E1", "\u02E1", "\u006C", "\u006C" },
            new string [] { "\u02E2", "\u02E2", "\u02E2", "\u0073", "\u0073" },
            new string [] { "\u02E3", "\u02E3", "\u02E3", "\u0078", "\u0078" },
            new string [] { "\u02E4", "\u02E4", "\u02E4", "\u0295", "\u0295" },
            new string [] { "\u0340", "\u0300", "\u0300", "\u0300", "\u0300" },
            new string [] { "\u0341", "\u0301", "\u0301", "\u0301", "\u0301" },
            new string [] { "\u0343", "\u0313", "\u0313", "\u0313", "\u0313" },
            new string [] { "\u0344", "\u0308\u0301", "\u0308\u0301", "\u0308\u0301", "\u0308\u0301" },
            new string [] { "\u0374", "\u02B9", "\u02B9", "\u02B9", "\u02B9" },
            new string [] { "\u037A", "\u037A", "\u037A", "\u0020\u0345", "\u0020\u0345" },
            new string [] { "\u037E", "\u003B", "\u003B", "\u003B", "\u003B" },
            new string [] { "\u0384", "\u0384", "\u0384", "\u0020\u0301", "\u0020\u0301" },
            new string [] { "\u0385", "\u0385", "\u00A8\u0301", "\u0020\u0308\u0301", "\u0020\u0308\u0301" },
            new string [] { "\u0386", "\u0386", "\u0391\u0301", "\u0386", "\u0391\u0301" },
            new string [] { "\u0387", "\u00B7", "\u00B7", "\u00B7", "\u00B7" },
            new string [] { "\u0388", "\u0388", "\u0395\u0301", "\u0388", "\u0395\u0301" },
            new string [] { "\u0389", "\u0389", "\u0397\u0301", "\u0389", "\u0397\u0301" },
            new string [] { "\u038A", "\u038A", "\u0399\u0301", "\u038A", "\u0399\u0301" },
            new string [] { "\u038C", "\u038C", "\u039F\u0301", "\u038C", "\u039F\u0301" },
            new string [] { "\u038E", "\u038E", "\u03A5\u0301", "\u038E", "\u03A5\u0301" },
            new string [] { "\u038F", "\u038F", "\u03A9\u0301", "\u038F", "\u03A9\u0301" },
            new string [] { "\u0390", "\u0390", "\u03B9\u0308\u0301", "\u0390", "\u03B9\u0308\u0301" },
            new string [] { "\u03AA", "\u03AA", "\u0399\u0308", "\u03AA", "\u0399\u0308" },
            new string [] { "\u03AB", "\u03AB", "\u03A5\u0308", "\u03AB", "\u03A5\u0308" },
            new string [] { "\u03AC", "\u03AC", "\u03B1\u0301", "\u03AC", "\u03B1\u0301" },
            new string [] { "\u03AD", "\u03AD", "\u03B5\u0301", "\u03AD", "\u03B5\u0301" },
            new string [] { "\u03AE", "\u03AE", "\u03B7\u0301", "\u03AE", "\u03B7\u0301" },
            new string [] { "\u03AF", "\u03AF", "\u03B9\u0301", "\u03AF", "\u03B9\u0301" },
            new string [] { "\u03B0", "\u03B0", "\u03C5\u0308\u0301", "\u03B0", "\u03C5\u0308\u0301" },
            new string [] { "\u03CA", "\u03CA", "\u03B9\u0308", "\u03CA", "\u03B9\u0308" },
            new string [] { "\u03CB", "\u03CB", "\u03C5\u0308", "\u03CB", "\u03C5\u0308" },
            new string [] { "\u03CC", "\u03CC", "\u03BF\u0301", "\u03CC", "\u03BF\u0301" },
            new string [] { "\u03CD", "\u03CD", "\u03C5\u0301", "\u03CD", "\u03C5\u0301" },
            new string [] { "\u03CE", "\u03CE", "\u03C9\u0301", "\u03CE", "\u03C9\u0301" },
            new string [] { "\u03D0", "\u03D0", "\u03D0", "\u03B2", "\u03B2" },
            new string [] { "\u03D1", "\u03D1", "\u03D1", "\u03B8", "\u03B8" },
            new string [] { "\u03D2", "\u03D2", "\u03D2", "\u03A5", "\u03A5" },
            new string [] { "\u03D3", "\u03D3", "\u03D2\u0301", "\u038E", "\u03A5\u0301" },
            new string [] { "\u03D4", "\u03D4", "\u03D2\u0308", "\u03AB", "\u03A5\u0308" },
            new string [] { "\u03D5", "\u03D5", "\u03D5", "\u03C6", "\u03C6" },
            new string [] { "\u03D6", "\u03D6", "\u03D6", "\u03C0", "\u03C0" },
            new string [] { "\u03F0", "\u03F0", "\u03F0", "\u03BA", "\u03BA" },
            new string [] { "\u03F1", "\u03F1", "\u03F1", "\u03C1", "\u03C1" },
            new string [] { "\u03F2", "\u03F2", "\u03F2", "\u03C2", "\u03C2" },
            new string [] { "\u03F4", "\u03F4", "\u03F4", "\u0398", "\u0398" },
            new string [] { "\u03F5", "\u03F5", "\u03F5", "\u03B5", "\u03B5" },
            new string [] { "\u03F9", "\u03F9", "\u03F9", "\u03A3", "\u03A3" },
            new string [] { "\u0400", "\u0400", "\u0415\u0300", "\u0400", "\u0415\u0300" },
            new string [] { "\u0401", "\u0401", "\u0415\u0308", "\u0401", "\u0415\u0308" },
            new string [] { "\u0403", "\u0403", "\u0413\u0301", "\u0403", "\u0413\u0301" },
            new string [] { "\u0407", "\u0407", "\u0406\u0308", "\u0407", "\u0406\u0308" },
            new string [] { "\u040C", "\u040C", "\u041A\u0301", "\u040C", "\u041A\u0301" },
            new string [] { "\u040D", "\u040D", "\u0418\u0300", "\u040D", "\u0418\u0300" },
            new string [] { "\u040E", "\u040E", "\u0423\u0306", "\u040E", "\u0423\u0306" },
            new string [] { "\u0419", "\u0419", "\u0418\u0306", "\u0419", "\u0418\u0306" },
            new string [] { "\u0439", "\u0439", "\u0438\u0306", "\u0439", "\u0438\u0306" },
            new string [] { "\u0450", "\u0450", "\u0435\u0300", "\u0450", "\u0435\u0300" },
            new string [] { "\u0451", "\u0451", "\u0435\u0308", "\u0451", "\u0435\u0308" },
            new string [] { "\u0453", "\u0453", "\u0433\u0301", "\u0453", "\u0433\u0301" },
            new string [] { "\u0457", "\u0457", "\u0456\u0308", "\u0457", "\u0456\u0308" },
            new string [] { "\u045C", "\u045C", "\u043A\u0301", "\u045C", "\u043A\u0301" },
            new string [] { "\u045D", "\u045D", "\u0438\u0300", "\u045D", "\u0438\u0300" },
            new string [] { "\u045E", "\u045E", "\u0443\u0306", "\u045E", "\u0443\u0306" },
            new string [] { "\u0476", "\u0476", "\u0474\u030F", "\u0476", "\u0474\u030F" },
            new string [] { "\u0477", "\u0477", "\u0475\u030F", "\u0477", "\u0475\u030F" },
            new string [] { "\u04C1", "\u04C1", "\u0416\u0306", "\u04C1", "\u0416\u0306" },
            new string [] { "\u04C2", "\u04C2", "\u0436\u0306", "\u04C2", "\u0436\u0306" },
            new string [] { "\u04D0", "\u04D0", "\u0410\u0306", "\u04D0", "\u0410\u0306" },
            new string [] { "\u04D1", "\u04D1", "\u0430\u0306", "\u04D1", "\u0430\u0306" },
            new string [] { "\u04D2", "\u04D2", "\u0410\u0308", "\u04D2", "\u0410\u0308" },
            new string [] { "\u04D3", "\u04D3", "\u0430\u0308", "\u04D3", "\u0430\u0308" },
            new string [] { "\u04D6", "\u04D6", "\u0415\u0306", "\u04D6", "\u0415\u0306" },
            new string [] { "\u04D7", "\u04D7", "\u0435\u0306", "\u04D7", "\u0435\u0306" },
            new string [] { "\u04DA", "\u04DA", "\u04D8\u0308", "\u04DA", "\u04D8\u0308" },
            new string [] { "\u04DB", "\u04DB", "\u04D9\u0308", "\u04DB", "\u04D9\u0308" },
            new string [] { "\u04DC", "\u04DC", "\u0416\u0308", "\u04DC", "\u0416\u0308" },
            new string [] { "\u04DD", "\u04DD", "\u0436\u0308", "\u04DD", "\u0436\u0308" },
            new string [] { "\u04DE", "\u04DE", "\u0417\u0308", "\u04DE", "\u0417\u0308" },
            new string [] { "\u04DF", "\u04DF", "\u0437\u0308", "\u04DF", "\u0437\u0308" },
            new string [] { "\u04E2", "\u04E2", "\u0418\u0304", "\u04E2", "\u0418\u0304" },
            new string [] { "\u04E3", "\u04E3", "\u0438\u0304", "\u04E3", "\u0438\u0304" },
            new string [] { "\u04E4", "\u04E4", "\u0418\u0308", "\u04E4", "\u0418\u0308" },
            new string [] { "\u04E5", "\u04E5", "\u0438\u0308", "\u04E5", "\u0438\u0308" },
            new string [] { "\u04E6", "\u04E6", "\u041E\u0308", "\u04E6", "\u041E\u0308" },
            new string [] { "\u04E7", "\u04E7", "\u043E\u0308", "\u04E7", "\u043E\u0308" },
            new string [] { "\u04EA", "\u04EA", "\u04E8\u0308", "\u04EA", "\u04E8\u0308" },
            new string [] { "\u04EB", "\u04EB", "\u04E9\u0308", "\u04EB", "\u04E9\u0308" },
            new string [] { "\u04EC", "\u04EC", "\u042D\u0308", "\u04EC", "\u042D\u0308" },
            new string [] { "\u04ED", "\u04ED", "\u044D\u0308", "\u04ED", "\u044D\u0308" },
            new string [] { "\u04EE", "\u04EE", "\u0423\u0304", "\u04EE", "\u0423\u0304" },
            new string [] { "\u04EF", "\u04EF", "\u0443\u0304", "\u04EF", "\u0443\u0304" },
            new string [] { "\u04F0", "\u04F0", "\u0423\u0308", "\u04F0", "\u0423\u0308" },
            new string [] { "\u04F1", "\u04F1", "\u0443\u0308", "\u04F1", "\u0443\u0308" },
            new string [] { "\u04F2", "\u04F2", "\u0423\u030B", "\u04F2", "\u0423\u030B" },
            new string [] { "\u04F3", "\u04F3", "\u0443\u030B", "\u04F3", "\u0443\u030B" },
            new string [] { "\u04F4", "\u04F4", "\u0427\u0308", "\u04F4", "\u0427\u0308" },
            new string [] { "\u04F5", "\u04F5", "\u0447\u0308", "\u04F5", "\u0447\u0308" },
            new string [] { "\u04F8", "\u04F8", "\u042B\u0308", "\u04F8", "\u042B\u0308" },
            new string [] { "\u04F9", "\u04F9", "\u044B\u0308", "\u04F9", "\u044B\u0308" },
            new string [] { "\u0587", "\u0587", "\u0587", "\u0565\u0582", "\u0565\u0582" },
            new string [] { "\u0622", "\u0622", "\u0627\u0653", "\u0622", "\u0627\u0653" },
            new string [] { "\u0623", "\u0623", "\u0627\u0654", "\u0623", "\u0627\u0654" },
            new string [] { "\u0624", "\u0624", "\u0648\u0654", "\u0624", "\u0648\u0654" },
            new string [] { "\u0625", "\u0625", "\u0627\u0655", "\u0625", "\u0627\u0655" },
            new string [] { "\u0626", "\u0626", "\u064A\u0654", "\u0626", "\u064A\u0654" },
            new string [] { "\u0675", "\u0675", "\u0675", "\u0627\u0674", "\u0627\u0674" },
            new string [] { "\u0676", "\u0676", "\u0676", "\u0648\u0674", "\u0648\u0674" },
            new string [] { "\u0677", "\u0677", "\u0677", "\u06C7\u0674", "\u06C7\u0674" },
            new string [] { "\u0678", "\u0678", "\u0678", "\u064A\u0674", "\u064A\u0674" },
            new string [] { "\u06C0", "\u06C0", "\u06D5\u0654", "\u06C0", "\u06D5\u0654" },
            new string [] { "\u06C2", "\u06C2", "\u06C1\u0654", "\u06C2", "\u06C1\u0654" },
            new string [] { "\u06D3", "\u06D3", "\u06D2\u0654", "\u06D3", "\u06D2\u0654" },
            new string [] { "\u0929", "\u0929", "\u0928\u093C", "\u0929", "\u0928\u093C" },
            new string [] { "\u0931", "\u0931", "\u0930\u093C", "\u0931", "\u0930\u093C" },
            new string [] { "\u0934", "\u0934", "\u0933\u093C", "\u0934", "\u0933\u093C" },
            new string [] { "\u0958", "\u0915\u093C", "\u0915\u093C", "\u0915\u093C", "\u0915\u093C" },
            new string [] { "\u0959", "\u0916\u093C", "\u0916\u093C", "\u0916\u093C", "\u0916\u093C" },
            new string [] { "\u095A", "\u0917\u093C", "\u0917\u093C", "\u0917\u093C", "\u0917\u093C" },
            new string [] { "\u095B", "\u091C\u093C", "\u091C\u093C", "\u091C\u093C", "\u091C\u093C" },
            new string [] { "\u095C", "\u0921\u093C", "\u0921\u093C", "\u0921\u093C", "\u0921\u093C" },
            new string [] { "\u095D", "\u0922\u093C", "\u0922\u093C", "\u0922\u093C", "\u0922\u093C" },
            new string [] { "\u095E", "\u092B\u093C", "\u092B\u093C", "\u092B\u093C", "\u092B\u093C" },
            new string [] { "\u095F", "\u092F\u093C", "\u092F\u093C", "\u092F\u093C", "\u092F\u093C" },
            new string [] { "\u09CB", "\u09CB", "\u09C7\u09BE", "\u09CB", "\u09C7\u09BE" },
            new string [] { "\u09CC", "\u09CC", "\u09C7\u09D7", "\u09CC", "\u09C7\u09D7" },
            new string [] { "\u09DC", "\u09A1\u09BC", "\u09A1\u09BC", "\u09A1\u09BC", "\u09A1\u09BC" },
            new string [] { "\u09DD", "\u09A2\u09BC", "\u09A2\u09BC", "\u09A2\u09BC", "\u09A2\u09BC" },
            new string [] { "\u09DF", "\u09AF\u09BC", "\u09AF\u09BC", "\u09AF\u09BC", "\u09AF\u09BC" },
            new string [] { "\u0A33", "\u0A32\u0A3C", "\u0A32\u0A3C", "\u0A32\u0A3C", "\u0A32\u0A3C" },
            new string [] { "\u0A36", "\u0A38\u0A3C", "\u0A38\u0A3C", "\u0A38\u0A3C", "\u0A38\u0A3C" },
            new string [] { "\u0A59", "\u0A16\u0A3C", "\u0A16\u0A3C", "\u0A16\u0A3C", "\u0A16\u0A3C" },
            new string [] { "\u0A5A", "\u0A17\u0A3C", "\u0A17\u0A3C", "\u0A17\u0A3C", "\u0A17\u0A3C" },
            new string [] { "\u0A5B", "\u0A1C\u0A3C", "\u0A1C\u0A3C", "\u0A1C\u0A3C", "\u0A1C\u0A3C" },
            new string [] { "\u0A5E", "\u0A2B\u0A3C", "\u0A2B\u0A3C", "\u0A2B\u0A3C", "\u0A2B\u0A3C" },
            new string [] { "\u0B48", "\u0B48", "\u0B47\u0B56", "\u0B48", "\u0B47\u0B56" },
            new string [] { "\u0B4B", "\u0B4B", "\u0B47\u0B3E", "\u0B4B", "\u0B47\u0B3E" },
            new string [] { "\u0B4C", "\u0B4C", "\u0B47\u0B57", "\u0B4C", "\u0B47\u0B57" },
            new string [] { "\u0B5C", "\u0B21\u0B3C", "\u0B21\u0B3C", "\u0B21\u0B3C", "\u0B21\u0B3C" },
            new string [] { "\u0B5D", "\u0B22\u0B3C", "\u0B22\u0B3C", "\u0B22\u0B3C", "\u0B22\u0B3C" },
            new string [] { "\u0B94", "\u0B94", "\u0B92\u0BD7", "\u0B94", "\u0B92\u0BD7" },
            new string [] { "\u0BCA", "\u0BCA", "\u0BC6\u0BBE", "\u0BCA", "\u0BC6\u0BBE" },
            new string [] { "\u0BCB", "\u0BCB", "\u0BC7\u0BBE", "\u0BCB", "\u0BC7\u0BBE" },
            new string [] { "\u0BCC", "\u0BCC", "\u0BC6\u0BD7", "\u0BCC", "\u0BC6\u0BD7" },
            new string [] { "\u0C48", "\u0C48", "\u0C46\u0C56", "\u0C48", "\u0C46\u0C56" },
            new string [] { "\u0CC0", "\u0CC0", "\u0CBF\u0CD5", "\u0CC0", "\u0CBF\u0CD5" },
            new string [] { "\u0CC7", "\u0CC7", "\u0CC6\u0CD5", "\u0CC7", "\u0CC6\u0CD5" },
            new string [] { "\u0CC8", "\u0CC8", "\u0CC6\u0CD6", "\u0CC8", "\u0CC6\u0CD6" },
            new string [] { "\u0CCA", "\u0CCA", "\u0CC6\u0CC2", "\u0CCA", "\u0CC6\u0CC2" },
            new string [] { "\u0CCB", "\u0CCB", "\u0CC6\u0CC2\u0CD5", "\u0CCB", "\u0CC6\u0CC2\u0CD5" },
            new string [] { "\u0D4A", "\u0D4A", "\u0D46\u0D3E", "\u0D4A", "\u0D46\u0D3E" },
            new string [] { "\u0D4B", "\u0D4B", "\u0D47\u0D3E", "\u0D4B", "\u0D47\u0D3E" },
            new string [] { "\u0D4C", "\u0D4C", "\u0D46\u0D57", "\u0D4C", "\u0D46\u0D57" },
            new string [] { "\u0DDA", "\u0DDA", "\u0DD9\u0DCA", "\u0DDA", "\u0DD9\u0DCA" },
            new string [] { "\u0DDC", "\u0DDC", "\u0DD9\u0DCF", "\u0DDC", "\u0DD9\u0DCF" },
            new string [] { "\u0DDD", "\u0DDD", "\u0DD9\u0DCF\u0DCA", "\u0DDD", "\u0DD9\u0DCF\u0DCA" },
            new string [] { "\u0DDE", "\u0DDE", "\u0DD9\u0DDF", "\u0DDE", "\u0DD9\u0DDF" },
            new string [] { "\u0E33", "\u0E33", "\u0E33", "\u0E4D\u0E32", "\u0E4D\u0E32" },
            new string [] { "\u0EB3", "\u0EB3", "\u0EB3", "\u0ECD\u0EB2", "\u0ECD\u0EB2" },
            new string [] { "\u0EDC", "\u0EDC", "\u0EDC", "\u0EAB\u0E99", "\u0EAB\u0E99" },
            new string [] { "\u0EDD", "\u0EDD", "\u0EDD", "\u0EAB\u0EA1", "\u0EAB\u0EA1" },
            new string [] { "\u0F0C", "\u0F0C", "\u0F0C", "\u0F0B", "\u0F0B" },
            new string [] { "\u0F43", "\u0F42\u0FB7", "\u0F42\u0FB7", "\u0F42\u0FB7", "\u0F42\u0FB7" },
            new string [] { "\u0F4D", "\u0F4C\u0FB7", "\u0F4C\u0FB7", "\u0F4C\u0FB7", "\u0F4C\u0FB7" },
            new string [] { "\u0F52", "\u0F51\u0FB7", "\u0F51\u0FB7", "\u0F51\u0FB7", "\u0F51\u0FB7" },
            new string [] { "\u0F57", "\u0F56\u0FB7", "\u0F56\u0FB7", "\u0F56\u0FB7", "\u0F56\u0FB7" },
            new string [] { "\u0F5C", "\u0F5B\u0FB7", "\u0F5B\u0FB7", "\u0F5B\u0FB7", "\u0F5B\u0FB7" },
            new string [] { "\u0F69", "\u0F40\u0FB5", "\u0F40\u0FB5", "\u0F40\u0FB5", "\u0F40\u0FB5" },
            new string [] { "\u0F73", "\u0F71\u0F72", "\u0F71\u0F72", "\u0F71\u0F72", "\u0F71\u0F72" },
            new string [] { "\u0F75", "\u0F71\u0F74", "\u0F71\u0F74", "\u0F71\u0F74", "\u0F71\u0F74" },
            new string [] { "\u0F76", "\u0FB2\u0F80", "\u0FB2\u0F80", "\u0FB2\u0F80", "\u0FB2\u0F80" },
            new string [] { "\u0F77", "\u0F77", "\u0F77", "\u0FB2\u0F71\u0F80", "\u0FB2\u0F71\u0F80" },
            new string [] { "\u0F78", "\u0FB3\u0F80", "\u0FB3\u0F80", "\u0FB3\u0F80", "\u0FB3\u0F80" },
            new string [] { "\u0F79", "\u0F79", "\u0F79", "\u0FB3\u0F71\u0F80", "\u0FB3\u0F71\u0F80" },
            new string [] { "\u0F81", "\u0F71\u0F80", "\u0F71\u0F80", "\u0F71\u0F80", "\u0F71\u0F80" },
            new string [] { "\u0F93", "\u0F92\u0FB7", "\u0F92\u0FB7", "\u0F92\u0FB7", "\u0F92\u0FB7" },
            new string [] { "\u0F9D", "\u0F9C\u0FB7", "\u0F9C\u0FB7", "\u0F9C\u0FB7", "\u0F9C\u0FB7" },
            new string [] { "\u0FA2", "\u0FA1\u0FB7", "\u0FA1\u0FB7", "\u0FA1\u0FB7", "\u0FA1\u0FB7" },
            new string [] { "\u0FA7", "\u0FA6\u0FB7", "\u0FA6\u0FB7", "\u0FA6\u0FB7", "\u0FA6\u0FB7" },
            new string [] { "\u0FAC", "\u0FAB\u0FB7", "\u0FAB\u0FB7", "\u0FAB\u0FB7", "\u0FAB\u0FB7" },
            new string [] { "\u0FB9", "\u0F90\u0FB5", "\u0F90\u0FB5", "\u0F90\u0FB5", "\u0F90\u0FB5" },
            new string [] { "\u1026", "\u1026", "\u1025\u102E", "\u1026", "\u1025\u102E" },
            new string [] { "\u10FC", "\u10FC", "\u10FC", "\u10DC", "\u10DC" },
            new string [] { "\u1B06", "\u1B06", "\u1B05\u1B35", "\u1B06", "\u1B05\u1B35" },
            new string [] { "\u1B08", "\u1B08", "\u1B07\u1B35", "\u1B08", "\u1B07\u1B35" },
            new string [] { "\u1B0A", "\u1B0A", "\u1B09\u1B35", "\u1B0A", "\u1B09\u1B35" },
            new string [] { "\u1B0C", "\u1B0C", "\u1B0B\u1B35", "\u1B0C", "\u1B0B\u1B35" },
            new string [] { "\u1B0E", "\u1B0E", "\u1B0D\u1B35", "\u1B0E", "\u1B0D\u1B35" },
            new string [] { "\u1B12", "\u1B12", "\u1B11\u1B35", "\u1B12", "\u1B11\u1B35" },
            new string [] { "\u1B3B", "\u1B3B", "\u1B3A\u1B35", "\u1B3B", "\u1B3A\u1B35" },
            new string [] { "\u1B3D", "\u1B3D", "\u1B3C\u1B35", "\u1B3D", "\u1B3C\u1B35" },
            new string [] { "\u1B40", "\u1B40", "\u1B3E\u1B35", "\u1B40", "\u1B3E\u1B35" },
            new string [] { "\u1B41", "\u1B41", "\u1B3F\u1B35", "\u1B41", "\u1B3F\u1B35" },
            new string [] { "\u1B43", "\u1B43", "\u1B42\u1B35", "\u1B43", "\u1B42\u1B35" },
            new string [] { "\u1D2C", "\u1D2C", "\u1D2C", "\u0041", "\u0041" },
            new string [] { "\u1D2D", "\u1D2D", "\u1D2D", "\u00C6", "\u00C6" },
            new string [] { "\u1D2E", "\u1D2E", "\u1D2E", "\u0042", "\u0042" },
            new string [] { "\u1D30", "\u1D30", "\u1D30", "\u0044", "\u0044" },
            new string [] { "\u1D31", "\u1D31", "\u1D31", "\u0045", "\u0045" },
            new string [] { "\u1D32", "\u1D32", "\u1D32", "\u018E", "\u018E" },
            new string [] { "\u1D33", "\u1D33", "\u1D33", "\u0047", "\u0047" },
            new string [] { "\u1D34", "\u1D34", "\u1D34", "\u0048", "\u0048" },
            new string [] { "\u1D35", "\u1D35", "\u1D35", "\u0049", "\u0049" },
            new string [] { "\u1D36", "\u1D36", "\u1D36", "\u004A", "\u004A" },
            new string [] { "\u1D37", "\u1D37", "\u1D37", "\u004B", "\u004B" },
            new string [] { "\u1D38", "\u1D38", "\u1D38", "\u004C", "\u004C" },
            new string [] { "\u1D39", "\u1D39", "\u1D39", "\u004D", "\u004D" },
            new string [] { "\u1D3A", "\u1D3A", "\u1D3A", "\u004E", "\u004E" },
            new string [] { "\u1D3C", "\u1D3C", "\u1D3C", "\u004F", "\u004F" },
            new string [] { "\u1D3D", "\u1D3D", "\u1D3D", "\u0222", "\u0222" },
            new string [] { "\u1D3E", "\u1D3E", "\u1D3E", "\u0050", "\u0050" },
            new string [] { "\u1D3F", "\u1D3F", "\u1D3F", "\u0052", "\u0052" },
            new string [] { "\u1D40", "\u1D40", "\u1D40", "\u0054", "\u0054" },
            new string [] { "\u1D41", "\u1D41", "\u1D41", "\u0055", "\u0055" },
            new string [] { "\u1D42", "\u1D42", "\u1D42", "\u0057", "\u0057" },
            new string [] { "\u1D43", "\u1D43", "\u1D43", "\u0061", "\u0061" },
            new string [] { "\u1D44", "\u1D44", "\u1D44", "\u0250", "\u0250" },
            new string [] { "\u1D45", "\u1D45", "\u1D45", "\u0251", "\u0251" },
            new string [] { "\u1D46", "\u1D46", "\u1D46", "\u1D02", "\u1D02" },
            new string [] { "\u1D47", "\u1D47", "\u1D47", "\u0062", "\u0062" },
            new string [] { "\u1D48", "\u1D48", "\u1D48", "\u0064", "\u0064" },
            new string [] { "\u1D49", "\u1D49", "\u1D49", "\u0065", "\u0065" },
            new string [] { "\u1D4A", "\u1D4A", "\u1D4A", "\u0259", "\u0259" },
            new string [] { "\u1D4B", "\u1D4B", "\u1D4B", "\u025B", "\u025B" },
            new string [] { "\u1D4C", "\u1D4C", "\u1D4C", "\u025C", "\u025C" },
            new string [] { "\u1D4D", "\u1D4D", "\u1D4D", "\u0067", "\u0067" },
            new string [] { "\u1D4F", "\u1D4F", "\u1D4F", "\u006B", "\u006B" },
            new string [] { "\u1D50", "\u1D50", "\u1D50", "\u006D", "\u006D" },
            new string [] { "\u1D51", "\u1D51", "\u1D51", "\u014B", "\u014B" },
            new string [] { "\u1D52", "\u1D52", "\u1D52", "\u006F", "\u006F" },
            new string [] { "\u1D53", "\u1D53", "\u1D53", "\u0254", "\u0254" },
            new string [] { "\u1D54", "\u1D54", "\u1D54", "\u1D16", "\u1D16" },
            new string [] { "\u1D55", "\u1D55", "\u1D55", "\u1D17", "\u1D17" },
            new string [] { "\u1D56", "\u1D56", "\u1D56", "\u0070", "\u0070" },
            new string [] { "\u1D57", "\u1D57", "\u1D57", "\u0074", "\u0074" },
            new string [] { "\u1D58", "\u1D58", "\u1D58", "\u0075", "\u0075" },
            new string [] { "\u1D59", "\u1D59", "\u1D59", "\u1D1D", "\u1D1D" },
            new string [] { "\u1D5A", "\u1D5A", "\u1D5A", "\u026F", "\u026F" },
            new string [] { "\u1D5B", "\u1D5B", "\u1D5B", "\u0076", "\u0076" },
            new string [] { "\u1D5C", "\u1D5C", "\u1D5C", "\u1D25", "\u1D25" },
            new string [] { "\u1D5D", "\u1D5D", "\u1D5D", "\u03B2", "\u03B2" },
            new string [] { "\u1D5E", "\u1D5E", "\u1D5E", "\u03B3", "\u03B3" },
            new string [] { "\u1D5F", "\u1D5F", "\u1D5F", "\u03B4", "\u03B4" },
            new string [] { "\u1D60", "\u1D60", "\u1D60", "\u03C6", "\u03C6" },
            new string [] { "\u1D61", "\u1D61", "\u1D61", "\u03C7", "\u03C7" },
            new string [] { "\u1D62", "\u1D62", "\u1D62", "\u0069", "\u0069" },
            new string [] { "\u1D63", "\u1D63", "\u1D63", "\u0072", "\u0072" },
            new string [] { "\u1D64", "\u1D64", "\u1D64", "\u0075", "\u0075" },
            new string [] { "\u1D65", "\u1D65", "\u1D65", "\u0076", "\u0076" },
            new string [] { "\u1D66", "\u1D66", "\u1D66", "\u03B2", "\u03B2" },
            new string [] { "\u1D67", "\u1D67", "\u1D67", "\u03B3", "\u03B3" },
            new string [] { "\u1D68", "\u1D68", "\u1D68", "\u03C1", "\u03C1" },
            new string [] { "\u1D69", "\u1D69", "\u1D69", "\u03C6", "\u03C6" },
            new string [] { "\u1D6A", "\u1D6A", "\u1D6A", "\u03C7", "\u03C7" },
            new string [] { "\u1D78", "\u1D78", "\u1D78", "\u043D", "\u043D" },
            new string [] { "\u1D9B", "\u1D9B", "\u1D9B", "\u0252", "\u0252" },
            new string [] { "\u1D9C", "\u1D9C", "\u1D9C", "\u0063", "\u0063" },
            new string [] { "\u1D9D", "\u1D9D", "\u1D9D", "\u0255", "\u0255" },
            new string [] { "\u1D9E", "\u1D9E", "\u1D9E", "\u00F0", "\u00F0" },
            new string [] { "\u1D9F", "\u1D9F", "\u1D9F", "\u025C", "\u025C" },
            new string [] { "\u1DA0", "\u1DA0", "\u1DA0", "\u0066", "\u0066" },
            new string [] { "\u1DA1", "\u1DA1", "\u1DA1", "\u025F", "\u025F" },
            new string [] { "\u1DA2", "\u1DA2", "\u1DA2", "\u0261", "\u0261" },
            new string [] { "\u1DA3", "\u1DA3", "\u1DA3", "\u0265", "\u0265" },
            new string [] { "\u1DA4", "\u1DA4", "\u1DA4", "\u0268", "\u0268" },
            new string [] { "\u1DA5", "\u1DA5", "\u1DA5", "\u0269", "\u0269" },
            new string [] { "\u1DA6", "\u1DA6", "\u1DA6", "\u026A", "\u026A" },
            new string [] { "\u1DA7", "\u1DA7", "\u1DA7", "\u1D7B", "\u1D7B" },
            new string [] { "\u1DA8", "\u1DA8", "\u1DA8", "\u029D", "\u029D" },
            new string [] { "\u1DA9", "\u1DA9", "\u1DA9", "\u026D", "\u026D" },
            new string [] { "\u1DAA", "\u1DAA", "\u1DAA", "\u1D85", "\u1D85" },
            new string [] { "\u1DAB", "\u1DAB", "\u1DAB", "\u029F", "\u029F" },
            new string [] { "\u1DAC", "\u1DAC", "\u1DAC", "\u0271", "\u0271" },
            new string [] { "\u1DAD", "\u1DAD", "\u1DAD", "\u0270", "\u0270" },
            new string [] { "\u1DAE", "\u1DAE", "\u1DAE", "\u0272", "\u0272" },
            new string [] { "\u1DAF", "\u1DAF", "\u1DAF", "\u0273", "\u0273" },
            new string [] { "\u1DB0", "\u1DB0", "\u1DB0", "\u0274", "\u0274" },
            new string [] { "\u1DB1", "\u1DB1", "\u1DB1", "\u0275", "\u0275" },
            new string [] { "\u1DB2", "\u1DB2", "\u1DB2", "\u0278", "\u0278" },
            new string [] { "\u1DB3", "\u1DB3", "\u1DB3", "\u0282", "\u0282" },
            new string [] { "\u1DB4", "\u1DB4", "\u1DB4", "\u0283", "\u0283" },
            new string [] { "\u1DB5", "\u1DB5", "\u1DB5", "\u01AB", "\u01AB" },
            new string [] { "\u1DB6", "\u1DB6", "\u1DB6", "\u0289", "\u0289" },
            new string [] { "\u1DB7", "\u1DB7", "\u1DB7", "\u028A", "\u028A" },
            new string [] { "\u1DB8", "\u1DB8", "\u1DB8", "\u1D1C", "\u1D1C" },
            new string [] { "\u1DB9", "\u1DB9", "\u1DB9", "\u028B", "\u028B" },
            new string [] { "\u1DBA", "\u1DBA", "\u1DBA", "\u028C", "\u028C" },
            new string [] { "\u1DBB", "\u1DBB", "\u1DBB", "\u007A", "\u007A" },
            new string [] { "\u1DBC", "\u1DBC", "\u1DBC", "\u0290", "\u0290" },
            new string [] { "\u1DBD", "\u1DBD", "\u1DBD", "\u0291", "\u0291" },
            new string [] { "\u1DBE", "\u1DBE", "\u1DBE", "\u0292", "\u0292" },
            new string [] { "\u1DBF", "\u1DBF", "\u1DBF", "\u03B8", "\u03B8" },
            new string [] { "\u1E00", "\u1E00", "\u0041\u0325", "\u1E00", "\u0041\u0325" },
            new string [] { "\u1E01", "\u1E01", "\u0061\u0325", "\u1E01", "\u0061\u0325" },
            new string [] { "\u1E02", "\u1E02", "\u0042\u0307", "\u1E02", "\u0042\u0307" },
            new string [] { "\u1E03", "\u1E03", "\u0062\u0307", "\u1E03", "\u0062\u0307" },
            new string [] { "\u1E04", "\u1E04", "\u0042\u0323", "\u1E04", "\u0042\u0323" },
            new string [] { "\u1E05", "\u1E05", "\u0062\u0323", "\u1E05", "\u0062\u0323" },
            new string [] { "\u1E06", "\u1E06", "\u0042\u0331", "\u1E06", "\u0042\u0331" },
            new string [] { "\u1E07", "\u1E07", "\u0062\u0331", "\u1E07", "\u0062\u0331" },
            new string [] { "\u1E08", "\u1E08", "\u0043\u0327\u0301", "\u1E08", "\u0043\u0327\u0301" },
            new string [] { "\u1E09", "\u1E09", "\u0063\u0327\u0301", "\u1E09", "\u0063\u0327\u0301" },
            new string [] { "\u1E0A", "\u1E0A", "\u0044\u0307", "\u1E0A", "\u0044\u0307" },
            new string [] { "\u1E0B", "\u1E0B", "\u0064\u0307", "\u1E0B", "\u0064\u0307" },
            new string [] { "\u1E0C", "\u1E0C", "\u0044\u0323", "\u1E0C", "\u0044\u0323" },
            new string [] { "\u1E0D", "\u1E0D", "\u0064\u0323", "\u1E0D", "\u0064\u0323" },
            new string [] { "\u1E0E", "\u1E0E", "\u0044\u0331", "\u1E0E", "\u0044\u0331" },
            new string [] { "\u1E0F", "\u1E0F", "\u0064\u0331", "\u1E0F", "\u0064\u0331" },
            new string [] { "\u1E10", "\u1E10", "\u0044\u0327", "\u1E10", "\u0044\u0327" },
            new string [] { "\u1E11", "\u1E11", "\u0064\u0327", "\u1E11", "\u0064\u0327" },
            new string [] { "\u1E12", "\u1E12", "\u0044\u032D", "\u1E12", "\u0044\u032D" },
            new string [] { "\u1E13", "\u1E13", "\u0064\u032D", "\u1E13", "\u0064\u032D" },
            new string [] { "\u1E14", "\u1E14", "\u0045\u0304\u0300", "\u1E14", "\u0045\u0304\u0300" },
            new string [] { "\u1E15", "\u1E15", "\u0065\u0304\u0300", "\u1E15", "\u0065\u0304\u0300" },
            new string [] { "\u1E16", "\u1E16", "\u0045\u0304\u0301", "\u1E16", "\u0045\u0304\u0301" },
            new string [] { "\u1E17", "\u1E17", "\u0065\u0304\u0301", "\u1E17", "\u0065\u0304\u0301" },
            new string [] { "\u1E18", "\u1E18", "\u0045\u032D", "\u1E18", "\u0045\u032D" },
            new string [] { "\u1E19", "\u1E19", "\u0065\u032D", "\u1E19", "\u0065\u032D" },
            new string [] { "\u1E1A", "\u1E1A", "\u0045\u0330", "\u1E1A", "\u0045\u0330" },
            new string [] { "\u1E1B", "\u1E1B", "\u0065\u0330", "\u1E1B", "\u0065\u0330" },
            new string [] { "\u1E1C", "\u1E1C", "\u0045\u0327\u0306", "\u1E1C", "\u0045\u0327\u0306" },
            new string [] { "\u1E1D", "\u1E1D", "\u0065\u0327\u0306", "\u1E1D", "\u0065\u0327\u0306" },
            new string [] { "\u1E1E", "\u1E1E", "\u0046\u0307", "\u1E1E", "\u0046\u0307" },
            new string [] { "\u1E1F", "\u1E1F", "\u0066\u0307", "\u1E1F", "\u0066\u0307" },
            new string [] { "\u1E20", "\u1E20", "\u0047\u0304", "\u1E20", "\u0047\u0304" },
            new string [] { "\u1E21", "\u1E21", "\u0067\u0304", "\u1E21", "\u0067\u0304" },
            new string [] { "\u1E22", "\u1E22", "\u0048\u0307", "\u1E22", "\u0048\u0307" },
            new string [] { "\u1E23", "\u1E23", "\u0068\u0307", "\u1E23", "\u0068\u0307" },
            new string [] { "\u1E24", "\u1E24", "\u0048\u0323", "\u1E24", "\u0048\u0323" },
            new string [] { "\u1E25", "\u1E25", "\u0068\u0323", "\u1E25", "\u0068\u0323" },
            new string [] { "\u1E26", "\u1E26", "\u0048\u0308", "\u1E26", "\u0048\u0308" },
            new string [] { "\u1E27", "\u1E27", "\u0068\u0308", "\u1E27", "\u0068\u0308" },
            new string [] { "\u1E28", "\u1E28", "\u0048\u0327", "\u1E28", "\u0048\u0327" },
            new string [] { "\u1E29", "\u1E29", "\u0068\u0327", "\u1E29", "\u0068\u0327" },
            new string [] { "\u1E2A", "\u1E2A", "\u0048\u032E", "\u1E2A", "\u0048\u032E" },
            new string [] { "\u1E2B", "\u1E2B", "\u0068\u032E", "\u1E2B", "\u0068\u032E" },
            new string [] { "\u1E2C", "\u1E2C", "\u0049\u0330", "\u1E2C", "\u0049\u0330" },
            new string [] { "\u1E2D", "\u1E2D", "\u0069\u0330", "\u1E2D", "\u0069\u0330" },
            new string [] { "\u1E2E", "\u1E2E", "\u0049\u0308\u0301", "\u1E2E", "\u0049\u0308\u0301" },
            new string [] { "\u1E2F", "\u1E2F", "\u0069\u0308\u0301", "\u1E2F", "\u0069\u0308\u0301" },
            new string [] { "\u1E30", "\u1E30", "\u004B\u0301", "\u1E30", "\u004B\u0301" },
            new string [] { "\u1E31", "\u1E31", "\u006B\u0301", "\u1E31", "\u006B\u0301" },
            new string [] { "\u1E32", "\u1E32", "\u004B\u0323", "\u1E32", "\u004B\u0323" },
            new string [] { "\u1E33", "\u1E33", "\u006B\u0323", "\u1E33", "\u006B\u0323" },
            new string [] { "\u1E34", "\u1E34", "\u004B\u0331", "\u1E34", "\u004B\u0331" },
            new string [] { "\u1E35", "\u1E35", "\u006B\u0331", "\u1E35", "\u006B\u0331" },
            new string [] { "\u1E36", "\u1E36", "\u004C\u0323", "\u1E36", "\u004C\u0323" },
            new string [] { "\u1E37", "\u1E37", "\u006C\u0323", "\u1E37", "\u006C\u0323" },
            new string [] { "\u1E38", "\u1E38", "\u004C\u0323\u0304", "\u1E38", "\u004C\u0323\u0304" },
            new string [] { "\u1E39", "\u1E39", "\u006C\u0323\u0304", "\u1E39", "\u006C\u0323\u0304" },
            new string [] { "\u1E3A", "\u1E3A", "\u004C\u0331", "\u1E3A", "\u004C\u0331" },
            new string [] { "\u1E3B", "\u1E3B", "\u006C\u0331", "\u1E3B", "\u006C\u0331" },
            new string [] { "\u1E3C", "\u1E3C", "\u004C\u032D", "\u1E3C", "\u004C\u032D" },
            new string [] { "\u1E3D", "\u1E3D", "\u006C\u032D", "\u1E3D", "\u006C\u032D" },
            new string [] { "\u1E3E", "\u1E3E", "\u004D\u0301", "\u1E3E", "\u004D\u0301" },
            new string [] { "\u1E3F", "\u1E3F", "\u006D\u0301", "\u1E3F", "\u006D\u0301" },
            new string [] { "\u1E40", "\u1E40", "\u004D\u0307", "\u1E40", "\u004D\u0307" },
            new string [] { "\u1E41", "\u1E41", "\u006D\u0307", "\u1E41", "\u006D\u0307" },
            new string [] { "\u1E42", "\u1E42", "\u004D\u0323", "\u1E42", "\u004D\u0323" },
            new string [] { "\u1E43", "\u1E43", "\u006D\u0323", "\u1E43", "\u006D\u0323" },
            new string [] { "\u1E44", "\u1E44", "\u004E\u0307", "\u1E44", "\u004E\u0307" },
            new string [] { "\u1E45", "\u1E45", "\u006E\u0307", "\u1E45", "\u006E\u0307" },
            new string [] { "\u1E46", "\u1E46", "\u004E\u0323", "\u1E46", "\u004E\u0323" },
            new string [] { "\u1E47", "\u1E47", "\u006E\u0323", "\u1E47", "\u006E\u0323" },
            new string [] { "\u1E48", "\u1E48", "\u004E\u0331", "\u1E48", "\u004E\u0331" },
            new string [] { "\u1E49", "\u1E49", "\u006E\u0331", "\u1E49", "\u006E\u0331" },
            new string [] { "\u1E4A", "\u1E4A", "\u004E\u032D", "\u1E4A", "\u004E\u032D" },
            new string [] { "\u1E4B", "\u1E4B", "\u006E\u032D", "\u1E4B", "\u006E\u032D" },
            new string [] { "\u1E4C", "\u1E4C", "\u004F\u0303\u0301", "\u1E4C", "\u004F\u0303\u0301" },
            new string [] { "\u1E4D", "\u1E4D", "\u006F\u0303\u0301", "\u1E4D", "\u006F\u0303\u0301" },
            new string [] { "\u1E4E", "\u1E4E", "\u004F\u0303\u0308", "\u1E4E", "\u004F\u0303\u0308" },
            new string [] { "\u1E4F", "\u1E4F", "\u006F\u0303\u0308", "\u1E4F", "\u006F\u0303\u0308" },
            new string [] { "\u1E50", "\u1E50", "\u004F\u0304\u0300", "\u1E50", "\u004F\u0304\u0300" },
            new string [] { "\u1E51", "\u1E51", "\u006F\u0304\u0300", "\u1E51", "\u006F\u0304\u0300" },
            new string [] { "\u1E52", "\u1E52", "\u004F\u0304\u0301", "\u1E52", "\u004F\u0304\u0301" },
            new string [] { "\u1E53", "\u1E53", "\u006F\u0304\u0301", "\u1E53", "\u006F\u0304\u0301" },
            new string [] { "\u1E54", "\u1E54", "\u0050\u0301", "\u1E54", "\u0050\u0301" },
            new string [] { "\u1E55", "\u1E55", "\u0070\u0301", "\u1E55", "\u0070\u0301" },
            new string [] { "\u1E56", "\u1E56", "\u0050\u0307", "\u1E56", "\u0050\u0307" },
            new string [] { "\u1E57", "\u1E57", "\u0070\u0307", "\u1E57", "\u0070\u0307" },
            new string [] { "\u1E58", "\u1E58", "\u0052\u0307", "\u1E58", "\u0052\u0307" },
            new string [] { "\u1E59", "\u1E59", "\u0072\u0307", "\u1E59", "\u0072\u0307" },
            new string [] { "\u1E5A", "\u1E5A", "\u0052\u0323", "\u1E5A", "\u0052\u0323" },
            new string [] { "\u1E5B", "\u1E5B", "\u0072\u0323", "\u1E5B", "\u0072\u0323" },
            new string [] { "\u1E5C", "\u1E5C", "\u0052\u0323\u0304", "\u1E5C", "\u0052\u0323\u0304" },
            new string [] { "\u1E5D", "\u1E5D", "\u0072\u0323\u0304", "\u1E5D", "\u0072\u0323\u0304" },
            new string [] { "\u1E5E", "\u1E5E", "\u0052\u0331", "\u1E5E", "\u0052\u0331" },
            new string [] { "\u1E5F", "\u1E5F", "\u0072\u0331", "\u1E5F", "\u0072\u0331" },
            new string [] { "\u1E60", "\u1E60", "\u0053\u0307", "\u1E60", "\u0053\u0307" },
            new string [] { "\u1E61", "\u1E61", "\u0073\u0307", "\u1E61", "\u0073\u0307" },
            new string [] { "\u1E62", "\u1E62", "\u0053\u0323", "\u1E62", "\u0053\u0323" },
            new string [] { "\u1E63", "\u1E63", "\u0073\u0323", "\u1E63", "\u0073\u0323" },
            new string [] { "\u1E64", "\u1E64", "\u0053\u0301\u0307", "\u1E64", "\u0053\u0301\u0307" },
            new string [] { "\u1E65", "\u1E65", "\u0073\u0301\u0307", "\u1E65", "\u0073\u0301\u0307" },
            new string [] { "\u1E66", "\u1E66", "\u0053\u030C\u0307", "\u1E66", "\u0053\u030C\u0307" },
            new string [] { "\u1E67", "\u1E67", "\u0073\u030C\u0307", "\u1E67", "\u0073\u030C\u0307" },
            new string [] { "\u1E68", "\u1E68", "\u0053\u0323\u0307", "\u1E68", "\u0053\u0323\u0307" },
            new string [] { "\u1E69", "\u1E69", "\u0073\u0323\u0307", "\u1E69", "\u0073\u0323\u0307" },
            new string [] { "\u1E6A", "\u1E6A", "\u0054\u0307", "\u1E6A", "\u0054\u0307" },
            new string [] { "\u1E6B", "\u1E6B", "\u0074\u0307", "\u1E6B", "\u0074\u0307" },
            new string [] { "\u1E6C", "\u1E6C", "\u0054\u0323", "\u1E6C", "\u0054\u0323" },
            new string [] { "\u1E6D", "\u1E6D", "\u0074\u0323", "\u1E6D", "\u0074\u0323" },
            new string [] { "\u1E6E", "\u1E6E", "\u0054\u0331", "\u1E6E", "\u0054\u0331" },
            new string [] { "\u1E6F", "\u1E6F", "\u0074\u0331", "\u1E6F", "\u0074\u0331" },
            new string [] { "\u1E70", "\u1E70", "\u0054\u032D", "\u1E70", "\u0054\u032D" },
            new string [] { "\u1E71", "\u1E71", "\u0074\u032D", "\u1E71", "\u0074\u032D" },
            new string [] { "\u1E72", "\u1E72", "\u0055\u0324", "\u1E72", "\u0055\u0324" },
            new string [] { "\u1E73", "\u1E73", "\u0075\u0324", "\u1E73", "\u0075\u0324" },
            new string [] { "\u1E74", "\u1E74", "\u0055\u0330", "\u1E74", "\u0055\u0330" },
            new string [] { "\u1E75", "\u1E75", "\u0075\u0330", "\u1E75", "\u0075\u0330" },
            new string [] { "\u1E76", "\u1E76", "\u0055\u032D", "\u1E76", "\u0055\u032D" },
            new string [] { "\u1E77", "\u1E77", "\u0075\u032D", "\u1E77", "\u0075\u032D" },
            new string [] { "\u1E78", "\u1E78", "\u0055\u0303\u0301", "\u1E78", "\u0055\u0303\u0301" },
            new string [] { "\u1E79", "\u1E79", "\u0075\u0303\u0301", "\u1E79", "\u0075\u0303\u0301" },
            new string [] { "\u1E7A", "\u1E7A", "\u0055\u0304\u0308", "\u1E7A", "\u0055\u0304\u0308" },
            new string [] { "\u1E7B", "\u1E7B", "\u0075\u0304\u0308", "\u1E7B", "\u0075\u0304\u0308" },
            new string [] { "\u1E7C", "\u1E7C", "\u0056\u0303", "\u1E7C", "\u0056\u0303" },
            new string [] { "\u1E7D", "\u1E7D", "\u0076\u0303", "\u1E7D", "\u0076\u0303" },
            new string [] { "\u1E7E", "\u1E7E", "\u0056\u0323", "\u1E7E", "\u0056\u0323" },
            new string [] { "\u1E7F", "\u1E7F", "\u0076\u0323", "\u1E7F", "\u0076\u0323" },
            new string [] { "\u1E80", "\u1E80", "\u0057\u0300", "\u1E80", "\u0057\u0300" },
            new string [] { "\u1E81", "\u1E81", "\u0077\u0300", "\u1E81", "\u0077\u0300" },
            new string [] { "\u1E82", "\u1E82", "\u0057\u0301", "\u1E82", "\u0057\u0301" },
            new string [] { "\u1E83", "\u1E83", "\u0077\u0301", "\u1E83", "\u0077\u0301" },
            new string [] { "\u1E84", "\u1E84", "\u0057\u0308", "\u1E84", "\u0057\u0308" },
            new string [] { "\u1E85", "\u1E85", "\u0077\u0308", "\u1E85", "\u0077\u0308" },
            new string [] { "\u1E86", "\u1E86", "\u0057\u0307", "\u1E86", "\u0057\u0307" },
            new string [] { "\u1E87", "\u1E87", "\u0077\u0307", "\u1E87", "\u0077\u0307" },
            new string [] { "\u1E88", "\u1E88", "\u0057\u0323", "\u1E88", "\u0057\u0323" },
            new string [] { "\u1E89", "\u1E89", "\u0077\u0323", "\u1E89", "\u0077\u0323" },
            new string [] { "\u1E8A", "\u1E8A", "\u0058\u0307", "\u1E8A", "\u0058\u0307" },
            new string [] { "\u1E8B", "\u1E8B", "\u0078\u0307", "\u1E8B", "\u0078\u0307" },
            new string [] { "\u1E8C", "\u1E8C", "\u0058\u0308", "\u1E8C", "\u0058\u0308" },
            new string [] { "\u1E8D", "\u1E8D", "\u0078\u0308", "\u1E8D", "\u0078\u0308" },
            new string [] { "\u1E8E", "\u1E8E", "\u0059\u0307", "\u1E8E", "\u0059\u0307" },
            new string [] { "\u1E8F", "\u1E8F", "\u0079\u0307", "\u1E8F", "\u0079\u0307" },
            new string [] { "\u1E90", "\u1E90", "\u005A\u0302", "\u1E90", "\u005A\u0302" },
            new string [] { "\u1E91", "\u1E91", "\u007A\u0302", "\u1E91", "\u007A\u0302" },
            new string [] { "\u1E92", "\u1E92", "\u005A\u0323", "\u1E92", "\u005A\u0323" },
            new string [] { "\u1E93", "\u1E93", "\u007A\u0323", "\u1E93", "\u007A\u0323" },
            new string [] { "\u1E94", "\u1E94", "\u005A\u0331", "\u1E94", "\u005A\u0331" },
            new string [] { "\u1E95", "\u1E95", "\u007A\u0331", "\u1E95", "\u007A\u0331" },
            new string [] { "\u1E96", "\u1E96", "\u0068\u0331", "\u1E96", "\u0068\u0331" },
            new string [] { "\u1E97", "\u1E97", "\u0074\u0308", "\u1E97", "\u0074\u0308" },
            new string [] { "\u1E98", "\u1E98", "\u0077\u030A", "\u1E98", "\u0077\u030A" },
            new string [] { "\u1E99", "\u1E99", "\u0079\u030A", "\u1E99", "\u0079\u030A" },
            new string [] { "\u1E9A", "\u1E9A", "\u1E9A", "\u0061\u02BE", "\u0061\u02BE" },
            new string [] { "\u1E9B", "\u1E9B", "\u017F\u0307", "\u1E61", "\u0073\u0307" },
            new string [] { "\u1EA0", "\u1EA0", "\u0041\u0323", "\u1EA0", "\u0041\u0323" },
            new string [] { "\u1EA1", "\u1EA1", "\u0061\u0323", "\u1EA1", "\u0061\u0323" },
            new string [] { "\u1EA2", "\u1EA2", "\u0041\u0309", "\u1EA2", "\u0041\u0309" },
            new string [] { "\u1EA3", "\u1EA3", "\u0061\u0309", "\u1EA3", "\u0061\u0309" },
            new string [] { "\u1EA4", "\u1EA4", "\u0041\u0302\u0301", "\u1EA4", "\u0041\u0302\u0301" },
            new string [] { "\u1EA5", "\u1EA5", "\u0061\u0302\u0301", "\u1EA5", "\u0061\u0302\u0301" },
            new string [] { "\u1EA6", "\u1EA6", "\u0041\u0302\u0300", "\u1EA6", "\u0041\u0302\u0300" },
            new string [] { "\u1EA7", "\u1EA7", "\u0061\u0302\u0300", "\u1EA7", "\u0061\u0302\u0300" },
            new string [] { "\u1EA8", "\u1EA8", "\u0041\u0302\u0309", "\u1EA8", "\u0041\u0302\u0309" },
            new string [] { "\u1EA9", "\u1EA9", "\u0061\u0302\u0309", "\u1EA9", "\u0061\u0302\u0309" },
            new string [] { "\u1EAA", "\u1EAA", "\u0041\u0302\u0303", "\u1EAA", "\u0041\u0302\u0303" },
            new string [] { "\u1EAB", "\u1EAB", "\u0061\u0302\u0303", "\u1EAB", "\u0061\u0302\u0303" },
            new string [] { "\u1EAC", "\u1EAC", "\u0041\u0323\u0302", "\u1EAC", "\u0041\u0323\u0302" },
            new string [] { "\u1EAD", "\u1EAD", "\u0061\u0323\u0302", "\u1EAD", "\u0061\u0323\u0302" },
            new string [] { "\u1EAE", "\u1EAE", "\u0041\u0306\u0301", "\u1EAE", "\u0041\u0306\u0301" },
            new string [] { "\u1EAF", "\u1EAF", "\u0061\u0306\u0301", "\u1EAF", "\u0061\u0306\u0301" },
            new string [] { "\u1EB0", "\u1EB0", "\u0041\u0306\u0300", "\u1EB0", "\u0041\u0306\u0300" },
            new string [] { "\u1EB1", "\u1EB1", "\u0061\u0306\u0300", "\u1EB1", "\u0061\u0306\u0300" },
            new string [] { "\u1EB2", "\u1EB2", "\u0041\u0306\u0309", "\u1EB2", "\u0041\u0306\u0309" },
            new string [] { "\u1EB3", "\u1EB3", "\u0061\u0306\u0309", "\u1EB3", "\u0061\u0306\u0309" },
            new string [] { "\u1EB4", "\u1EB4", "\u0041\u0306\u0303", "\u1EB4", "\u0041\u0306\u0303" },
            new string [] { "\u1EB5", "\u1EB5", "\u0061\u0306\u0303", "\u1EB5", "\u0061\u0306\u0303" },
            new string [] { "\u1EB6", "\u1EB6", "\u0041\u0323\u0306", "\u1EB6", "\u0041\u0323\u0306" },
            new string [] { "\u1EB7", "\u1EB7", "\u0061\u0323\u0306", "\u1EB7", "\u0061\u0323\u0306" },
            new string [] { "\u1EB8", "\u1EB8", "\u0045\u0323", "\u1EB8", "\u0045\u0323" },
            new string [] { "\u1EB9", "\u1EB9", "\u0065\u0323", "\u1EB9", "\u0065\u0323" },
            new string [] { "\u1EBA", "\u1EBA", "\u0045\u0309", "\u1EBA", "\u0045\u0309" },
            new string [] { "\u1EBB", "\u1EBB", "\u0065\u0309", "\u1EBB", "\u0065\u0309" },
            new string [] { "\u1EBC", "\u1EBC", "\u0045\u0303", "\u1EBC", "\u0045\u0303" },
            new string [] { "\u1EBD", "\u1EBD", "\u0065\u0303", "\u1EBD", "\u0065\u0303" },
            new string [] { "\u1EBE", "\u1EBE", "\u0045\u0302\u0301", "\u1EBE", "\u0045\u0302\u0301" },
            new string [] { "\u1EBF", "\u1EBF", "\u0065\u0302\u0301", "\u1EBF", "\u0065\u0302\u0301" },
            new string [] { "\u1EC0", "\u1EC0", "\u0045\u0302\u0300", "\u1EC0", "\u0045\u0302\u0300" },
            new string [] { "\u1EC1", "\u1EC1", "\u0065\u0302\u0300", "\u1EC1", "\u0065\u0302\u0300" },
            new string [] { "\u1EC2", "\u1EC2", "\u0045\u0302\u0309", "\u1EC2", "\u0045\u0302\u0309" },
            new string [] { "\u1EC3", "\u1EC3", "\u0065\u0302\u0309", "\u1EC3", "\u0065\u0302\u0309" },
            new string [] { "\u1EC4", "\u1EC4", "\u0045\u0302\u0303", "\u1EC4", "\u0045\u0302\u0303" },
            new string [] { "\u1EC5", "\u1EC5", "\u0065\u0302\u0303", "\u1EC5", "\u0065\u0302\u0303" },
            new string [] { "\u1EC6", "\u1EC6", "\u0045\u0323\u0302", "\u1EC6", "\u0045\u0323\u0302" },
            new string [] { "\u1EC7", "\u1EC7", "\u0065\u0323\u0302", "\u1EC7", "\u0065\u0323\u0302" },
            new string [] { "\u1EC8", "\u1EC8", "\u0049\u0309", "\u1EC8", "\u0049\u0309" },
            new string [] { "\u1EC9", "\u1EC9", "\u0069\u0309", "\u1EC9", "\u0069\u0309" },
            new string [] { "\u1ECA", "\u1ECA", "\u0049\u0323", "\u1ECA", "\u0049\u0323" },
            new string [] { "\u1ECB", "\u1ECB", "\u0069\u0323", "\u1ECB", "\u0069\u0323" },
            new string [] { "\u1ECC", "\u1ECC", "\u004F\u0323", "\u1ECC", "\u004F\u0323" },
            new string [] { "\u1ECD", "\u1ECD", "\u006F\u0323", "\u1ECD", "\u006F\u0323" },
            new string [] { "\u1ECE", "\u1ECE", "\u004F\u0309", "\u1ECE", "\u004F\u0309" },
            new string [] { "\u1ECF", "\u1ECF", "\u006F\u0309", "\u1ECF", "\u006F\u0309" },
            new string [] { "\u1ED0", "\u1ED0", "\u004F\u0302\u0301", "\u1ED0", "\u004F\u0302\u0301" },
            new string [] { "\u1ED1", "\u1ED1", "\u006F\u0302\u0301", "\u1ED1", "\u006F\u0302\u0301" },
            new string [] { "\u1ED2", "\u1ED2", "\u004F\u0302\u0300", "\u1ED2", "\u004F\u0302\u0300" },
            new string [] { "\u1ED3", "\u1ED3", "\u006F\u0302\u0300", "\u1ED3", "\u006F\u0302\u0300" },
            new string [] { "\u1ED4", "\u1ED4", "\u004F\u0302\u0309", "\u1ED4", "\u004F\u0302\u0309" },
            new string [] { "\u1ED5", "\u1ED5", "\u006F\u0302\u0309", "\u1ED5", "\u006F\u0302\u0309" },
            new string [] { "\u1ED6", "\u1ED6", "\u004F\u0302\u0303", "\u1ED6", "\u004F\u0302\u0303" },
            new string [] { "\u1ED7", "\u1ED7", "\u006F\u0302\u0303", "\u1ED7", "\u006F\u0302\u0303" },
            new string [] { "\u1ED8", "\u1ED8", "\u004F\u0323\u0302", "\u1ED8", "\u004F\u0323\u0302" },
            new string [] { "\u1ED9", "\u1ED9", "\u006F\u0323\u0302", "\u1ED9", "\u006F\u0323\u0302" },
            new string [] { "\u1EDA", "\u1EDA", "\u004F\u031B\u0301", "\u1EDA", "\u004F\u031B\u0301" },
            new string [] { "\u1EDB", "\u1EDB", "\u006F\u031B\u0301", "\u1EDB", "\u006F\u031B\u0301" },
            new string [] { "\u1EDC", "\u1EDC", "\u004F\u031B\u0300", "\u1EDC", "\u004F\u031B\u0300" },
            new string [] { "\u1EDD", "\u1EDD", "\u006F\u031B\u0300", "\u1EDD", "\u006F\u031B\u0300" },
            new string [] { "\u1EDE", "\u1EDE", "\u004F\u031B\u0309", "\u1EDE", "\u004F\u031B\u0309" },
            new string [] { "\u1EDF", "\u1EDF", "\u006F\u031B\u0309", "\u1EDF", "\u006F\u031B\u0309" },
            new string [] { "\u1EE0", "\u1EE0", "\u004F\u031B\u0303", "\u1EE0", "\u004F\u031B\u0303" },
            new string [] { "\u1EE1", "\u1EE1", "\u006F\u031B\u0303", "\u1EE1", "\u006F\u031B\u0303" },
            new string [] { "\u1EE2", "\u1EE2", "\u004F\u031B\u0323", "\u1EE2", "\u004F\u031B\u0323" },
            new string [] { "\u1EE3", "\u1EE3", "\u006F\u031B\u0323", "\u1EE3", "\u006F\u031B\u0323" },
            new string [] { "\u1EE4", "\u1EE4", "\u0055\u0323", "\u1EE4", "\u0055\u0323" },
            new string [] { "\u1EE5", "\u1EE5", "\u0075\u0323", "\u1EE5", "\u0075\u0323" },
            new string [] { "\u1EE6", "\u1EE6", "\u0055\u0309", "\u1EE6", "\u0055\u0309" },
            new string [] { "\u1EE7", "\u1EE7", "\u0075\u0309", "\u1EE7", "\u0075\u0309" },
            new string [] { "\u1EE8", "\u1EE8", "\u0055\u031B\u0301", "\u1EE8", "\u0055\u031B\u0301" },
            new string [] { "\u1EE9", "\u1EE9", "\u0075\u031B\u0301", "\u1EE9", "\u0075\u031B\u0301" },
            new string [] { "\u1EEA", "\u1EEA", "\u0055\u031B\u0300", "\u1EEA", "\u0055\u031B\u0300" },
            new string [] { "\u1EEB", "\u1EEB", "\u0075\u031B\u0300", "\u1EEB", "\u0075\u031B\u0300" },
            new string [] { "\u1EEC", "\u1EEC", "\u0055\u031B\u0309", "\u1EEC", "\u0055\u031B\u0309" },
            new string [] { "\u1EED", "\u1EED", "\u0075\u031B\u0309", "\u1EED", "\u0075\u031B\u0309" },
            new string [] { "\u1EEE", "\u1EEE", "\u0055\u031B\u0303", "\u1EEE", "\u0055\u031B\u0303" },
            new string [] { "\u1EEF", "\u1EEF", "\u0075\u031B\u0303", "\u1EEF", "\u0075\u031B\u0303" },
            new string [] { "\u1EF0", "\u1EF0", "\u0055\u031B\u0323", "\u1EF0", "\u0055\u031B\u0323" },
            new string [] { "\u1EF1", "\u1EF1", "\u0075\u031B\u0323", "\u1EF1", "\u0075\u031B\u0323" },
            new string [] { "\u1EF2", "\u1EF2", "\u0059\u0300", "\u1EF2", "\u0059\u0300" },
            new string [] { "\u1EF3", "\u1EF3", "\u0079\u0300", "\u1EF3", "\u0079\u0300" },
            new string [] { "\u1EF4", "\u1EF4", "\u0059\u0323", "\u1EF4", "\u0059\u0323" },
            new string [] { "\u1EF5", "\u1EF5", "\u0079\u0323", "\u1EF5", "\u0079\u0323" },
            new string [] { "\u1EF6", "\u1EF6", "\u0059\u0309", "\u1EF6", "\u0059\u0309" },
            new string [] { "\u1EF7", "\u1EF7", "\u0079\u0309", "\u1EF7", "\u0079\u0309" },
            new string [] { "\u1EF8", "\u1EF8", "\u0059\u0303", "\u1EF8", "\u0059\u0303" },
            new string [] { "\u1EF9", "\u1EF9", "\u0079\u0303", "\u1EF9", "\u0079\u0303" },
            new string [] { "\u1F00", "\u1F00", "\u03B1\u0313", "\u1F00", "\u03B1\u0313" },
            new string [] { "\u1F01", "\u1F01", "\u03B1\u0314", "\u1F01", "\u03B1\u0314" },
            new string [] { "\u1F02", "\u1F02", "\u03B1\u0313\u0300", "\u1F02", "\u03B1\u0313\u0300" },
            new string [] { "\u1F03", "\u1F03", "\u03B1\u0314\u0300", "\u1F03", "\u03B1\u0314\u0300" },
            new string [] { "\u1F04", "\u1F04", "\u03B1\u0313\u0301", "\u1F04", "\u03B1\u0313\u0301" },
            new string [] { "\u1F05", "\u1F05", "\u03B1\u0314\u0301", "\u1F05", "\u03B1\u0314\u0301" },
            new string [] { "\u1F06", "\u1F06", "\u03B1\u0313\u0342", "\u1F06", "\u03B1\u0313\u0342" },
            new string [] { "\u1F07", "\u1F07", "\u03B1\u0314\u0342", "\u1F07", "\u03B1\u0314\u0342" },
            new string [] { "\u1F08", "\u1F08", "\u0391\u0313", "\u1F08", "\u0391\u0313" },
            new string [] { "\u1F09", "\u1F09", "\u0391\u0314", "\u1F09", "\u0391\u0314" },
            new string [] { "\u1F0A", "\u1F0A", "\u0391\u0313\u0300", "\u1F0A", "\u0391\u0313\u0300" },
            new string [] { "\u1F0B", "\u1F0B", "\u0391\u0314\u0300", "\u1F0B", "\u0391\u0314\u0300" },
            new string [] { "\u1F0C", "\u1F0C", "\u0391\u0313\u0301", "\u1F0C", "\u0391\u0313\u0301" },
            new string [] { "\u1F0D", "\u1F0D", "\u0391\u0314\u0301", "\u1F0D", "\u0391\u0314\u0301" },
            new string [] { "\u1F0E", "\u1F0E", "\u0391\u0313\u0342", "\u1F0E", "\u0391\u0313\u0342" },
            new string [] { "\u1F0F", "\u1F0F", "\u0391\u0314\u0342", "\u1F0F", "\u0391\u0314\u0342" },
            new string [] { "\u1F10", "\u1F10", "\u03B5\u0313", "\u1F10", "\u03B5\u0313" },
            new string [] { "\u1F11", "\u1F11", "\u03B5\u0314", "\u1F11", "\u03B5\u0314" },
            new string [] { "\u1F12", "\u1F12", "\u03B5\u0313\u0300", "\u1F12", "\u03B5\u0313\u0300" },
            new string [] { "\u1F13", "\u1F13", "\u03B5\u0314\u0300", "\u1F13", "\u03B5\u0314\u0300" },
            new string [] { "\u1F14", "\u1F14", "\u03B5\u0313\u0301", "\u1F14", "\u03B5\u0313\u0301" },
            new string [] { "\u1F15", "\u1F15", "\u03B5\u0314\u0301", "\u1F15", "\u03B5\u0314\u0301" },
            new string [] { "\u1F18", "\u1F18", "\u0395\u0313", "\u1F18", "\u0395\u0313" },
            new string [] { "\u1F19", "\u1F19", "\u0395\u0314", "\u1F19", "\u0395\u0314" },
            new string [] { "\u1F1A", "\u1F1A", "\u0395\u0313\u0300", "\u1F1A", "\u0395\u0313\u0300" },
            new string [] { "\u1F1B", "\u1F1B", "\u0395\u0314\u0300", "\u1F1B", "\u0395\u0314\u0300" },
            new string [] { "\u1F1C", "\u1F1C", "\u0395\u0313\u0301", "\u1F1C", "\u0395\u0313\u0301" },
            new string [] { "\u1F1D", "\u1F1D", "\u0395\u0314\u0301", "\u1F1D", "\u0395\u0314\u0301" },
            new string [] { "\u1F20", "\u1F20", "\u03B7\u0313", "\u1F20", "\u03B7\u0313" },
            new string [] { "\u1F21", "\u1F21", "\u03B7\u0314", "\u1F21", "\u03B7\u0314" },
            new string [] { "\u1F22", "\u1F22", "\u03B7\u0313\u0300", "\u1F22", "\u03B7\u0313\u0300" },
            new string [] { "\u1F23", "\u1F23", "\u03B7\u0314\u0300", "\u1F23", "\u03B7\u0314\u0300" },
            new string [] { "\u1F24", "\u1F24", "\u03B7\u0313\u0301", "\u1F24", "\u03B7\u0313\u0301" },
            new string [] { "\u1F25", "\u1F25", "\u03B7\u0314\u0301", "\u1F25", "\u03B7\u0314\u0301" },
            new string [] { "\u1F26", "\u1F26", "\u03B7\u0313\u0342", "\u1F26", "\u03B7\u0313\u0342" },
            new string [] { "\u1F27", "\u1F27", "\u03B7\u0314\u0342", "\u1F27", "\u03B7\u0314\u0342" },
            new string [] { "\u1F28", "\u1F28", "\u0397\u0313", "\u1F28", "\u0397\u0313" },
            new string [] { "\u1F29", "\u1F29", "\u0397\u0314", "\u1F29", "\u0397\u0314" },
            new string [] { "\u1F2A", "\u1F2A", "\u0397\u0313\u0300", "\u1F2A", "\u0397\u0313\u0300" },
            new string [] { "\u1F2B", "\u1F2B", "\u0397\u0314\u0300", "\u1F2B", "\u0397\u0314\u0300" },
            new string [] { "\u1F2C", "\u1F2C", "\u0397\u0313\u0301", "\u1F2C", "\u0397\u0313\u0301" },
            new string [] { "\u1F2D", "\u1F2D", "\u0397\u0314\u0301", "\u1F2D", "\u0397\u0314\u0301" },
            new string [] { "\u1F2E", "\u1F2E", "\u0397\u0313\u0342", "\u1F2E", "\u0397\u0313\u0342" },
            new string [] { "\u1F2F", "\u1F2F", "\u0397\u0314\u0342", "\u1F2F", "\u0397\u0314\u0342" },
            new string [] { "\u1F30", "\u1F30", "\u03B9\u0313", "\u1F30", "\u03B9\u0313" },
            new string [] { "\u1F31", "\u1F31", "\u03B9\u0314", "\u1F31", "\u03B9\u0314" },
            new string [] { "\u1F32", "\u1F32", "\u03B9\u0313\u0300", "\u1F32", "\u03B9\u0313\u0300" },
            new string [] { "\u1F33", "\u1F33", "\u03B9\u0314\u0300", "\u1F33", "\u03B9\u0314\u0300" },
            new string [] { "\u1F34", "\u1F34", "\u03B9\u0313\u0301", "\u1F34", "\u03B9\u0313\u0301" },
            new string [] { "\u1F35", "\u1F35", "\u03B9\u0314\u0301", "\u1F35", "\u03B9\u0314\u0301" },
            new string [] { "\u1F36", "\u1F36", "\u03B9\u0313\u0342", "\u1F36", "\u03B9\u0313\u0342" },
            new string [] { "\u1F37", "\u1F37", "\u03B9\u0314\u0342", "\u1F37", "\u03B9\u0314\u0342" },
            new string [] { "\u1F38", "\u1F38", "\u0399\u0313", "\u1F38", "\u0399\u0313" },
            new string [] { "\u1F39", "\u1F39", "\u0399\u0314", "\u1F39", "\u0399\u0314" },
            new string [] { "\u1F3A", "\u1F3A", "\u0399\u0313\u0300", "\u1F3A", "\u0399\u0313\u0300" },
            new string [] { "\u1F3B", "\u1F3B", "\u0399\u0314\u0300", "\u1F3B", "\u0399\u0314\u0300" },
            new string [] { "\u1F3C", "\u1F3C", "\u0399\u0313\u0301", "\u1F3C", "\u0399\u0313\u0301" },
            new string [] { "\u1F3D", "\u1F3D", "\u0399\u0314\u0301", "\u1F3D", "\u0399\u0314\u0301" },
            new string [] { "\u1F3E", "\u1F3E", "\u0399\u0313\u0342", "\u1F3E", "\u0399\u0313\u0342" },
            new string [] { "\u1F3F", "\u1F3F", "\u0399\u0314\u0342", "\u1F3F", "\u0399\u0314\u0342" },
            new string [] { "\u1F40", "\u1F40", "\u03BF\u0313", "\u1F40", "\u03BF\u0313" },
            new string [] { "\u1F41", "\u1F41", "\u03BF\u0314", "\u1F41", "\u03BF\u0314" },
            new string [] { "\u1F42", "\u1F42", "\u03BF\u0313\u0300", "\u1F42", "\u03BF\u0313\u0300" },
            new string [] { "\u1F43", "\u1F43", "\u03BF\u0314\u0300", "\u1F43", "\u03BF\u0314\u0300" },
            new string [] { "\u1F44", "\u1F44", "\u03BF\u0313\u0301", "\u1F44", "\u03BF\u0313\u0301" },
            new string [] { "\u1F45", "\u1F45", "\u03BF\u0314\u0301", "\u1F45", "\u03BF\u0314\u0301" },
            new string [] { "\u1F48", "\u1F48", "\u039F\u0313", "\u1F48", "\u039F\u0313" },
            new string [] { "\u1F49", "\u1F49", "\u039F\u0314", "\u1F49", "\u039F\u0314" },
            new string [] { "\u1F4A", "\u1F4A", "\u039F\u0313\u0300", "\u1F4A", "\u039F\u0313\u0300" },
            new string [] { "\u1F4B", "\u1F4B", "\u039F\u0314\u0300", "\u1F4B", "\u039F\u0314\u0300" },
            new string [] { "\u1F4C", "\u1F4C", "\u039F\u0313\u0301", "\u1F4C", "\u039F\u0313\u0301" },
            new string [] { "\u1F4D", "\u1F4D", "\u039F\u0314\u0301", "\u1F4D", "\u039F\u0314\u0301" },
            new string [] { "\u1F50", "\u1F50", "\u03C5\u0313", "\u1F50", "\u03C5\u0313" },
            new string [] { "\u1F51", "\u1F51", "\u03C5\u0314", "\u1F51", "\u03C5\u0314" },
            new string [] { "\u1F52", "\u1F52", "\u03C5\u0313\u0300", "\u1F52", "\u03C5\u0313\u0300" },
            new string [] { "\u1F53", "\u1F53", "\u03C5\u0314\u0300", "\u1F53", "\u03C5\u0314\u0300" },
            new string [] { "\u1F54", "\u1F54", "\u03C5\u0313\u0301", "\u1F54", "\u03C5\u0313\u0301" },
            new string [] { "\u1F55", "\u1F55", "\u03C5\u0314\u0301", "\u1F55", "\u03C5\u0314\u0301" },
            new string [] { "\u1F56", "\u1F56", "\u03C5\u0313\u0342", "\u1F56", "\u03C5\u0313\u0342" },
            new string [] { "\u1F57", "\u1F57", "\u03C5\u0314\u0342", "\u1F57", "\u03C5\u0314\u0342" },
            new string [] { "\u1F59", "\u1F59", "\u03A5\u0314", "\u1F59", "\u03A5\u0314" },
            new string [] { "\u1F5B", "\u1F5B", "\u03A5\u0314\u0300", "\u1F5B", "\u03A5\u0314\u0300" },
            new string [] { "\u1F5D", "\u1F5D", "\u03A5\u0314\u0301", "\u1F5D", "\u03A5\u0314\u0301" },
            new string [] { "\u1F5F", "\u1F5F", "\u03A5\u0314\u0342", "\u1F5F", "\u03A5\u0314\u0342" },
            new string [] { "\u1F60", "\u1F60", "\u03C9\u0313", "\u1F60", "\u03C9\u0313" },
            new string [] { "\u1F61", "\u1F61", "\u03C9\u0314", "\u1F61", "\u03C9\u0314" },
            new string [] { "\u1F62", "\u1F62", "\u03C9\u0313\u0300", "\u1F62", "\u03C9\u0313\u0300" },
            new string [] { "\u1F63", "\u1F63", "\u03C9\u0314\u0300", "\u1F63", "\u03C9\u0314\u0300" },
            new string [] { "\u1F64", "\u1F64", "\u03C9\u0313\u0301", "\u1F64", "\u03C9\u0313\u0301" },
            new string [] { "\u1F65", "\u1F65", "\u03C9\u0314\u0301", "\u1F65", "\u03C9\u0314\u0301" },
            new string [] { "\u1F66", "\u1F66", "\u03C9\u0313\u0342", "\u1F66", "\u03C9\u0313\u0342" },
            new string [] { "\u1F67", "\u1F67", "\u03C9\u0314\u0342", "\u1F67", "\u03C9\u0314\u0342" },
            new string [] { "\u1F68", "\u1F68", "\u03A9\u0313", "\u1F68", "\u03A9\u0313" },
            new string [] { "\u1F69", "\u1F69", "\u03A9\u0314", "\u1F69", "\u03A9\u0314" },
            new string [] { "\u1F6A", "\u1F6A", "\u03A9\u0313\u0300", "\u1F6A", "\u03A9\u0313\u0300" },
            new string [] { "\u1F6B", "\u1F6B", "\u03A9\u0314\u0300", "\u1F6B", "\u03A9\u0314\u0300" },
            new string [] { "\u1F6C", "\u1F6C", "\u03A9\u0313\u0301", "\u1F6C", "\u03A9\u0313\u0301" },
            new string [] { "\u1F6D", "\u1F6D", "\u03A9\u0314\u0301", "\u1F6D", "\u03A9\u0314\u0301" },
            new string [] { "\u1F6E", "\u1F6E", "\u03A9\u0313\u0342", "\u1F6E", "\u03A9\u0313\u0342" },
            new string [] { "\u1F6F", "\u1F6F", "\u03A9\u0314\u0342", "\u1F6F", "\u03A9\u0314\u0342" },
            new string [] { "\u1F70", "\u1F70", "\u03B1\u0300", "\u1F70", "\u03B1\u0300" },
            new string [] { "\u1F71", "\u03AC", "\u03B1\u0301", "\u03AC", "\u03B1\u0301" },
            new string [] { "\u1F72", "\u1F72", "\u03B5\u0300", "\u1F72", "\u03B5\u0300" },
            new string [] { "\u1F73", "\u03AD", "\u03B5\u0301", "\u03AD", "\u03B5\u0301" },
            new string [] { "\u1F74", "\u1F74", "\u03B7\u0300", "\u1F74", "\u03B7\u0300" },
            new string [] { "\u1F75", "\u03AE", "\u03B7\u0301", "\u03AE", "\u03B7\u0301" },
            new string [] { "\u1F76", "\u1F76", "\u03B9\u0300", "\u1F76", "\u03B9\u0300" },
            new string [] { "\u1F77", "\u03AF", "\u03B9\u0301", "\u03AF", "\u03B9\u0301" },
            new string [] { "\u1F78", "\u1F78", "\u03BF\u0300", "\u1F78", "\u03BF\u0300" },
            new string [] { "\u1F79", "\u03CC", "\u03BF\u0301", "\u03CC", "\u03BF\u0301" },
            new string [] { "\u1F7A", "\u1F7A", "\u03C5\u0300", "\u1F7A", "\u03C5\u0300" },
            new string [] { "\u1F7B", "\u03CD", "\u03C5\u0301", "\u03CD", "\u03C5\u0301" },
            new string [] { "\u1F7C", "\u1F7C", "\u03C9\u0300", "\u1F7C", "\u03C9\u0300" },
            new string [] { "\u1F7D", "\u03CE", "\u03C9\u0301", "\u03CE", "\u03C9\u0301" },
            new string [] { "\u1F80", "\u1F80", "\u03B1\u0313\u0345", "\u1F80", "\u03B1\u0313\u0345" },
            new string [] { "\u1F81", "\u1F81", "\u03B1\u0314\u0345", "\u1F81", "\u03B1\u0314\u0345" },
            new string [] { "\u1F82", "\u1F82", "\u03B1\u0313\u0300\u0345", "\u1F82", "\u03B1\u0313\u0300\u0345" },
            new string [] { "\u1F83", "\u1F83", "\u03B1\u0314\u0300\u0345", "\u1F83", "\u03B1\u0314\u0300\u0345" },
            new string [] { "\u1F84", "\u1F84", "\u03B1\u0313\u0301\u0345", "\u1F84", "\u03B1\u0313\u0301\u0345" },
            new string [] { "\u1F85", "\u1F85", "\u03B1\u0314\u0301\u0345", "\u1F85", "\u03B1\u0314\u0301\u0345" },
            new string [] { "\u1F86", "\u1F86", "\u03B1\u0313\u0342\u0345", "\u1F86", "\u03B1\u0313\u0342\u0345" },
            new string [] { "\u1F87", "\u1F87", "\u03B1\u0314\u0342\u0345", "\u1F87", "\u03B1\u0314\u0342\u0345" },
            new string [] { "\u1F88", "\u1F88", "\u0391\u0313\u0345", "\u1F88", "\u0391\u0313\u0345" },
            new string [] { "\u1F89", "\u1F89", "\u0391\u0314\u0345", "\u1F89", "\u0391\u0314\u0345" },
            new string [] { "\u1F8A", "\u1F8A", "\u0391\u0313\u0300\u0345", "\u1F8A", "\u0391\u0313\u0300\u0345" },
            new string [] { "\u1F8B", "\u1F8B", "\u0391\u0314\u0300\u0345", "\u1F8B", "\u0391\u0314\u0300\u0345" },
            new string [] { "\u1F8C", "\u1F8C", "\u0391\u0313\u0301\u0345", "\u1F8C", "\u0391\u0313\u0301\u0345" },
            new string [] { "\u1F8D", "\u1F8D", "\u0391\u0314\u0301\u0345", "\u1F8D", "\u0391\u0314\u0301\u0345" },
            new string [] { "\u1F8E", "\u1F8E", "\u0391\u0313\u0342\u0345", "\u1F8E", "\u0391\u0313\u0342\u0345" },
            new string [] { "\u1F8F", "\u1F8F", "\u0391\u0314\u0342\u0345", "\u1F8F", "\u0391\u0314\u0342\u0345" },
            new string [] { "\u1F90", "\u1F90", "\u03B7\u0313\u0345", "\u1F90", "\u03B7\u0313\u0345" },
            new string [] { "\u1F91", "\u1F91", "\u03B7\u0314\u0345", "\u1F91", "\u03B7\u0314\u0345" },
            new string [] { "\u1F92", "\u1F92", "\u03B7\u0313\u0300\u0345", "\u1F92", "\u03B7\u0313\u0300\u0345" },
            new string [] { "\u1F93", "\u1F93", "\u03B7\u0314\u0300\u0345", "\u1F93", "\u03B7\u0314\u0300\u0345" },
            new string [] { "\u1F94", "\u1F94", "\u03B7\u0313\u0301\u0345", "\u1F94", "\u03B7\u0313\u0301\u0345" },
            new string [] { "\u1F95", "\u1F95", "\u03B7\u0314\u0301\u0345", "\u1F95", "\u03B7\u0314\u0301\u0345" },
            new string [] { "\u1F96", "\u1F96", "\u03B7\u0313\u0342\u0345", "\u1F96", "\u03B7\u0313\u0342\u0345" },
            new string [] { "\u1F97", "\u1F97", "\u03B7\u0314\u0342\u0345", "\u1F97", "\u03B7\u0314\u0342\u0345" },
            new string [] { "\u1F98", "\u1F98", "\u0397\u0313\u0345", "\u1F98", "\u0397\u0313\u0345" },
            new string [] { "\u1F99", "\u1F99", "\u0397\u0314\u0345", "\u1F99", "\u0397\u0314\u0345" },
            new string [] { "\u1F9A", "\u1F9A", "\u0397\u0313\u0300\u0345", "\u1F9A", "\u0397\u0313\u0300\u0345" },
            new string [] { "\u1F9B", "\u1F9B", "\u0397\u0314\u0300\u0345", "\u1F9B", "\u0397\u0314\u0300\u0345" },
            new string [] { "\u1F9C", "\u1F9C", "\u0397\u0313\u0301\u0345", "\u1F9C", "\u0397\u0313\u0301\u0345" },
            new string [] { "\u1F9D", "\u1F9D", "\u0397\u0314\u0301\u0345", "\u1F9D", "\u0397\u0314\u0301\u0345" },
            new string [] { "\u1F9E", "\u1F9E", "\u0397\u0313\u0342\u0345", "\u1F9E", "\u0397\u0313\u0342\u0345" },
            new string [] { "\u1F9F", "\u1F9F", "\u0397\u0314\u0342\u0345", "\u1F9F", "\u0397\u0314\u0342\u0345" },
            new string [] { "\u1FA0", "\u1FA0", "\u03C9\u0313\u0345", "\u1FA0", "\u03C9\u0313\u0345" },
            new string [] { "\u1FA1", "\u1FA1", "\u03C9\u0314\u0345", "\u1FA1", "\u03C9\u0314\u0345" },
            new string [] { "\u1FA2", "\u1FA2", "\u03C9\u0313\u0300\u0345", "\u1FA2", "\u03C9\u0313\u0300\u0345" },
            new string [] { "\u1FA3", "\u1FA3", "\u03C9\u0314\u0300\u0345", "\u1FA3", "\u03C9\u0314\u0300\u0345" },
            new string [] { "\u1FA4", "\u1FA4", "\u03C9\u0313\u0301\u0345", "\u1FA4", "\u03C9\u0313\u0301\u0345" },
            new string [] { "\u1FA5", "\u1FA5", "\u03C9\u0314\u0301\u0345", "\u1FA5", "\u03C9\u0314\u0301\u0345" },
            new string [] { "\u1FA6", "\u1FA6", "\u03C9\u0313\u0342\u0345", "\u1FA6", "\u03C9\u0313\u0342\u0345" },
            new string [] { "\u1FA7", "\u1FA7", "\u03C9\u0314\u0342\u0345", "\u1FA7", "\u03C9\u0314\u0342\u0345" },
            new string [] { "\u1FA8", "\u1FA8", "\u03A9\u0313\u0345", "\u1FA8", "\u03A9\u0313\u0345" },
            new string [] { "\u1FA9", "\u1FA9", "\u03A9\u0314\u0345", "\u1FA9", "\u03A9\u0314\u0345" },
            new string [] { "\u1FAA", "\u1FAA", "\u03A9\u0313\u0300\u0345", "\u1FAA", "\u03A9\u0313\u0300\u0345" },
            new string [] { "\u1FAB", "\u1FAB", "\u03A9\u0314\u0300\u0345", "\u1FAB", "\u03A9\u0314\u0300\u0345" },
            new string [] { "\u1FAC", "\u1FAC", "\u03A9\u0313\u0301\u0345", "\u1FAC", "\u03A9\u0313\u0301\u0345" },
            new string [] { "\u1FAD", "\u1FAD", "\u03A9\u0314\u0301\u0345", "\u1FAD", "\u03A9\u0314\u0301\u0345" },
            new string [] { "\u1FAE", "\u1FAE", "\u03A9\u0313\u0342\u0345", "\u1FAE", "\u03A9\u0313\u0342\u0345" },
            new string [] { "\u1FAF", "\u1FAF", "\u03A9\u0314\u0342\u0345", "\u1FAF", "\u03A9\u0314\u0342\u0345" },
            new string [] { "\u1FB0", "\u1FB0", "\u03B1\u0306", "\u1FB0", "\u03B1\u0306" },
            new string [] { "\u1FB1", "\u1FB1", "\u03B1\u0304", "\u1FB1", "\u03B1\u0304" },
            new string [] { "\u1FB2", "\u1FB2", "\u03B1\u0300\u0345", "\u1FB2", "\u03B1\u0300\u0345" },
            new string [] { "\u1FB3", "\u1FB3", "\u03B1\u0345", "\u1FB3", "\u03B1\u0345" },
            new string [] { "\u1FB4", "\u1FB4", "\u03B1\u0301\u0345", "\u1FB4", "\u03B1\u0301\u0345" },
            new string [] { "\u1FB6", "\u1FB6", "\u03B1\u0342", "\u1FB6", "\u03B1\u0342" },
            new string [] { "\u1FB7", "\u1FB7", "\u03B1\u0342\u0345", "\u1FB7", "\u03B1\u0342\u0345" },
            new string [] { "\u1FB8", "\u1FB8", "\u0391\u0306", "\u1FB8", "\u0391\u0306" },
            new string [] { "\u1FB9", "\u1FB9", "\u0391\u0304", "\u1FB9", "\u0391\u0304" },
            new string [] { "\u1FBA", "\u1FBA", "\u0391\u0300", "\u1FBA", "\u0391\u0300" },
            new string [] { "\u1FBB", "\u0386", "\u0391\u0301", "\u0386", "\u0391\u0301" },
            new string [] { "\u1FBC", "\u1FBC", "\u0391\u0345", "\u1FBC", "\u0391\u0345" },
            new string [] { "\u1FBD", "\u1FBD", "\u1FBD", "\u0020\u0313", "\u0020\u0313" },
            new string [] { "\u1FBE", "\u03B9", "\u03B9", "\u03B9", "\u03B9" },
            new string [] { "\u1FBF", "\u1FBF", "\u1FBF", "\u0020\u0313", "\u0020\u0313" },
            new string [] { "\u1FC0", "\u1FC0", "\u1FC0", "\u0020\u0342", "\u0020\u0342" },
            new string [] { "\u1FC1", "\u1FC1", "\u00A8\u0342", "\u0020\u0308\u0342", "\u0020\u0308\u0342" },
            new string [] { "\u1FC2", "\u1FC2", "\u03B7\u0300\u0345", "\u1FC2", "\u03B7\u0300\u0345" },
            new string [] { "\u1FC3", "\u1FC3", "\u03B7\u0345", "\u1FC3", "\u03B7\u0345" },
            new string [] { "\u1FC4", "\u1FC4", "\u03B7\u0301\u0345", "\u1FC4", "\u03B7\u0301\u0345" },
            new string [] { "\u1FC6", "\u1FC6", "\u03B7\u0342", "\u1FC6", "\u03B7\u0342" },
            new string [] { "\u1FC7", "\u1FC7", "\u03B7\u0342\u0345", "\u1FC7", "\u03B7\u0342\u0345" },
            new string [] { "\u1FC8", "\u1FC8", "\u0395\u0300", "\u1FC8", "\u0395\u0300" },
            new string [] { "\u1FC9", "\u0388", "\u0395\u0301", "\u0388", "\u0395\u0301" },
            new string [] { "\u1FCA", "\u1FCA", "\u0397\u0300", "\u1FCA", "\u0397\u0300" },
            new string [] { "\u1FCB", "\u0389", "\u0397\u0301", "\u0389", "\u0397\u0301" },
            new string [] { "\u1FCC", "\u1FCC", "\u0397\u0345", "\u1FCC", "\u0397\u0345" },
            new string [] { "\u1FCD", "\u1FCD", "\u1FBF\u0300", "\u0020\u0313\u0300", "\u0020\u0313\u0300" },
            new string [] { "\u1FCE", "\u1FCE", "\u1FBF\u0301", "\u0020\u0313\u0301", "\u0020\u0313\u0301" },
            new string [] { "\u1FCF", "\u1FCF", "\u1FBF\u0342", "\u0020\u0313\u0342", "\u0020\u0313\u0342" },
            new string [] { "\u1FD0", "\u1FD0", "\u03B9\u0306", "\u1FD0", "\u03B9\u0306" },
            new string [] { "\u1FD1", "\u1FD1", "\u03B9\u0304", "\u1FD1", "\u03B9\u0304" },
            new string [] { "\u1FD2", "\u1FD2", "\u03B9\u0308\u0300", "\u1FD2", "\u03B9\u0308\u0300" },
            new string [] { "\u1FD3", "\u0390", "\u03B9\u0308\u0301", "\u0390", "\u03B9\u0308\u0301" },
            new string [] { "\u1FD6", "\u1FD6", "\u03B9\u0342", "\u1FD6", "\u03B9\u0342" },
            new string [] { "\u1FD7", "\u1FD7", "\u03B9\u0308\u0342", "\u1FD7", "\u03B9\u0308\u0342" },
            new string [] { "\u1FD8", "\u1FD8", "\u0399\u0306", "\u1FD8", "\u0399\u0306" },
            new string [] { "\u1FD9", "\u1FD9", "\u0399\u0304", "\u1FD9", "\u0399\u0304" },
            new string [] { "\u1FDA", "\u1FDA", "\u0399\u0300", "\u1FDA", "\u0399\u0300" },
            new string [] { "\u1FDB", "\u038A", "\u0399\u0301", "\u038A", "\u0399\u0301" },
            new string [] { "\u1FDD", "\u1FDD", "\u1FFE\u0300", "\u0020\u0314\u0300", "\u0020\u0314\u0300" },
            new string [] { "\u1FDE", "\u1FDE", "\u1FFE\u0301", "\u0020\u0314\u0301", "\u0020\u0314\u0301" },
            new string [] { "\u1FDF", "\u1FDF", "\u1FFE\u0342", "\u0020\u0314\u0342", "\u0020\u0314\u0342" },
            new string [] { "\u1FE0", "\u1FE0", "\u03C5\u0306", "\u1FE0", "\u03C5\u0306" },
            new string [] { "\u1FE1", "\u1FE1", "\u03C5\u0304", "\u1FE1", "\u03C5\u0304" },
            new string [] { "\u1FE2", "\u1FE2", "\u03C5\u0308\u0300", "\u1FE2", "\u03C5\u0308\u0300" },
            new string [] { "\u1FE3", "\u03B0", "\u03C5\u0308\u0301", "\u03B0", "\u03C5\u0308\u0301" },
            new string [] { "\u1FE4", "\u1FE4", "\u03C1\u0313", "\u1FE4", "\u03C1\u0313" },
            new string [] { "\u1FE5", "\u1FE5", "\u03C1\u0314", "\u1FE5", "\u03C1\u0314" },
            new string [] { "\u1FE6", "\u1FE6", "\u03C5\u0342", "\u1FE6", "\u03C5\u0342" },
            new string [] { "\u1FE7", "\u1FE7", "\u03C5\u0308\u0342", "\u1FE7", "\u03C5\u0308\u0342" },
            new string [] { "\u1FE8", "\u1FE8", "\u03A5\u0306", "\u1FE8", "\u03A5\u0306" },
            new string [] { "\u1FE9", "\u1FE9", "\u03A5\u0304", "\u1FE9", "\u03A5\u0304" },
            new string [] { "\u1FEA", "\u1FEA", "\u03A5\u0300", "\u1FEA", "\u03A5\u0300" },
            new string [] { "\u1FEB", "\u038E", "\u03A5\u0301", "\u038E", "\u03A5\u0301" },
            new string [] { "\u1FEC", "\u1FEC", "\u03A1\u0314", "\u1FEC", "\u03A1\u0314" },
            new string [] { "\u1FED", "\u1FED", "\u00A8\u0300", "\u0020\u0308\u0300", "\u0020\u0308\u0300" },
            new string [] { "\u1FEE", "\u0385", "\u00A8\u0301", "\u0020\u0308\u0301", "\u0020\u0308\u0301" },
            new string [] { "\u1FEF", "\u0060", "\u0060", "\u0060", "\u0060" },
            new string [] { "\u1FF2", "\u1FF2", "\u03C9\u0300\u0345", "\u1FF2", "\u03C9\u0300\u0345" },
            new string [] { "\u1FF3", "\u1FF3", "\u03C9\u0345", "\u1FF3", "\u03C9\u0345" },
            new string [] { "\u1FF4", "\u1FF4", "\u03C9\u0301\u0345", "\u1FF4", "\u03C9\u0301\u0345" },
            new string [] { "\u1FF6", "\u1FF6", "\u03C9\u0342", "\u1FF6", "\u03C9\u0342" },
            new string [] { "\u1FF7", "\u1FF7", "\u03C9\u0342\u0345", "\u1FF7", "\u03C9\u0342\u0345" },
            new string [] { "\u1FF8", "\u1FF8", "\u039F\u0300", "\u1FF8", "\u039F\u0300" },
            new string [] { "\u1FF9", "\u038C", "\u039F\u0301", "\u038C", "\u039F\u0301" },
            new string [] { "\u1FFA", "\u1FFA", "\u03A9\u0300", "\u1FFA", "\u03A9\u0300" },
            new string [] { "\u1FFB", "\u038F", "\u03A9\u0301", "\u038F", "\u03A9\u0301" },
            new string [] { "\u1FFC", "\u1FFC", "\u03A9\u0345", "\u1FFC", "\u03A9\u0345" },
            new string [] { "\u1FFD", "\u00B4", "\u00B4", "\u0020\u0301", "\u0020\u0301" },
            new string [] { "\u1FFE", "\u1FFE", "\u1FFE", "\u0020\u0314", "\u0020\u0314" },
            new string [] { "\u2000", "\u2002", "\u2002", "\u0020", "\u0020" },
            new string [] { "\u2001", "\u2003", "\u2003", "\u0020", "\u0020" },
            new string [] { "\u2002", "\u2002", "\u2002", "\u0020", "\u0020" },
            new string [] { "\u2003", "\u2003", "\u2003", "\u0020", "\u0020" },
            new string [] { "\u2004", "\u2004", "\u2004", "\u0020", "\u0020" },
            new string [] { "\u2005", "\u2005", "\u2005", "\u0020", "\u0020" },
            new string [] { "\u2006", "\u2006", "\u2006", "\u0020", "\u0020" },
            new string [] { "\u2007", "\u2007", "\u2007", "\u0020", "\u0020" },
            new string [] { "\u2008", "\u2008", "\u2008", "\u0020", "\u0020" },
            new string [] { "\u2009", "\u2009", "\u2009", "\u0020", "\u0020" },
            new string [] { "\u200A", "\u200A", "\u200A", "\u0020", "\u0020" },
            new string [] { "\u2011", "\u2011", "\u2011", "\u2010", "\u2010" },
            new string [] { "\u2017", "\u2017", "\u2017", "\u0020\u0333", "\u0020\u0333" },
            new string [] { "\u2024", "\u2024", "\u2024", "\u002E", "\u002E" },
            new string [] { "\u2025", "\u2025", "\u2025", "\u002E\u002E", "\u002E\u002E" },
            new string [] { "\u2026", "\u2026", "\u2026", "\u002E\u002E\u002E", "\u002E\u002E\u002E" },
            new string [] { "\u202F", "\u202F", "\u202F", "\u0020", "\u0020" },
            new string [] { "\u2033", "\u2033", "\u2033", "\u2032\u2032", "\u2032\u2032" },
            new string [] { "\u2034", "\u2034", "\u2034", "\u2032\u2032\u2032", "\u2032\u2032\u2032" },
            new string [] { "\u2036", "\u2036", "\u2036", "\u2035\u2035", "\u2035\u2035" },
            new string [] { "\u2037", "\u2037", "\u2037", "\u2035\u2035\u2035", "\u2035\u2035\u2035" },
            new string [] { "\u203C", "\u203C", "\u203C", "\u0021\u0021", "\u0021\u0021" },
            new string [] { "\u203E", "\u203E", "\u203E", "\u0020\u0305", "\u0020\u0305" },
            new string [] { "\u2047", "\u2047", "\u2047", "\u003F\u003F", "\u003F\u003F" },
            new string [] { "\u2048", "\u2048", "\u2048", "\u003F\u0021", "\u003F\u0021" },
            new string [] { "\u2049", "\u2049", "\u2049", "\u0021\u003F", "\u0021\u003F" },
            new string [] { "\u2057", "\u2057", "\u2057", "\u2032\u2032\u2032\u2032", "\u2032\u2032\u2032\u2032" },
            new string [] { "\u205F", "\u205F", "\u205F", "\u0020", "\u0020" },
            new string [] { "\u2070", "\u2070", "\u2070", "\u0030", "\u0030" },
            new string [] { "\u2071", "\u2071", "\u2071", "\u0069", "\u0069" },
            new string [] { "\u2074", "\u2074", "\u2074", "\u0034", "\u0034" },
            new string [] { "\u2075", "\u2075", "\u2075", "\u0035", "\u0035" },
            new string [] { "\u2076", "\u2076", "\u2076", "\u0036", "\u0036" },
            new string [] { "\u2077", "\u2077", "\u2077", "\u0037", "\u0037" },
            new string [] { "\u2078", "\u2078", "\u2078", "\u0038", "\u0038" },
            new string [] { "\u2079", "\u2079", "\u2079", "\u0039", "\u0039" },
            new string [] { "\u207A", "\u207A", "\u207A", "\u002B", "\u002B" },
            new string [] { "\u207B", "\u207B", "\u207B", "\u2212", "\u2212" },
            new string [] { "\u207C", "\u207C", "\u207C", "\u003D", "\u003D" },
            new string [] { "\u207D", "\u207D", "\u207D", "\u0028", "\u0028" },
            new string [] { "\u207E", "\u207E", "\u207E", "\u0029", "\u0029" },
            new string [] { "\u207F", "\u207F", "\u207F", "\u006E", "\u006E" },
            new string [] { "\u2080", "\u2080", "\u2080", "\u0030", "\u0030" },
            new string [] { "\u2081", "\u2081", "\u2081", "\u0031", "\u0031" },
            new string [] { "\u2082", "\u2082", "\u2082", "\u0032", "\u0032" },
            new string [] { "\u2083", "\u2083", "\u2083", "\u0033", "\u0033" },
            new string [] { "\u2084", "\u2084", "\u2084", "\u0034", "\u0034" },
            new string [] { "\u2085", "\u2085", "\u2085", "\u0035", "\u0035" },
            new string [] { "\u2086", "\u2086", "\u2086", "\u0036", "\u0036" },
            new string [] { "\u2087", "\u2087", "\u2087", "\u0037", "\u0037" },
            new string [] { "\u2088", "\u2088", "\u2088", "\u0038", "\u0038" },
            new string [] { "\u2089", "\u2089", "\u2089", "\u0039", "\u0039" },
            new string [] { "\u208A", "\u208A", "\u208A", "\u002B", "\u002B" },
            new string [] { "\u208B", "\u208B", "\u208B", "\u2212", "\u2212" },
            new string [] { "\u208C", "\u208C", "\u208C", "\u003D", "\u003D" },
            new string [] { "\u208D", "\u208D", "\u208D", "\u0028", "\u0028" },
            new string [] { "\u208E", "\u208E", "\u208E", "\u0029", "\u0029" },
            new string [] { "\u2090", "\u2090", "\u2090", "\u0061", "\u0061" },
            new string [] { "\u2091", "\u2091", "\u2091", "\u0065", "\u0065" },
            new string [] { "\u2092", "\u2092", "\u2092", "\u006F", "\u006F" },
            new string [] { "\u2093", "\u2093", "\u2093", "\u0078", "\u0078" },
            new string [] { "\u2094", "\u2094", "\u2094", "\u0259", "\u0259" },
            new string [] { "\u2095", "\u2095", "\u2095", "\u0068", "\u0068" },
            new string [] { "\u2096", "\u2096", "\u2096", "\u006B", "\u006B" },
            new string [] { "\u2097", "\u2097", "\u2097", "\u006C", "\u006C" },
            new string [] { "\u2098", "\u2098", "\u2098", "\u006D", "\u006D" },
            new string [] { "\u2099", "\u2099", "\u2099", "\u006E", "\u006E" },
            new string [] { "\u209A", "\u209A", "\u209A", "\u0070", "\u0070" },
            new string [] { "\u209B", "\u209B", "\u209B", "\u0073", "\u0073" },
            new string [] { "\u209C", "\u209C", "\u209C", "\u0074", "\u0074" },
            new string [] { "\u20A8", "\u20A8", "\u20A8", "\u0052\u0073", "\u0052\u0073" },
            new string [] { "\u2100", "\u2100", "\u2100", "\u0061\u002F\u0063", "\u0061\u002F\u0063" },
            new string [] { "\u2101", "\u2101", "\u2101", "\u0061\u002F\u0073", "\u0061\u002F\u0073" },
            new string [] { "\u2102", "\u2102", "\u2102", "\u0043", "\u0043" },
            new string [] { "\u2103", "\u2103", "\u2103", "\u00B0\u0043", "\u00B0\u0043" },
            new string [] { "\u2105", "\u2105", "\u2105", "\u0063\u002F\u006F", "\u0063\u002F\u006F" },
            new string [] { "\u2106", "\u2106", "\u2106", "\u0063\u002F\u0075", "\u0063\u002F\u0075" },
            new string [] { "\u2107", "\u2107", "\u2107", "\u0190", "\u0190" },
            new string [] { "\u2109", "\u2109", "\u2109", "\u00B0\u0046", "\u00B0\u0046" },
            new string [] { "\u210A", "\u210A", "\u210A", "\u0067", "\u0067" },
            new string [] { "\u210B", "\u210B", "\u210B", "\u0048", "\u0048" },
            new string [] { "\u210C", "\u210C", "\u210C", "\u0048", "\u0048" },
            new string [] { "\u210D", "\u210D", "\u210D", "\u0048", "\u0048" },
            new string [] { "\u210E", "\u210E", "\u210E", "\u0068", "\u0068" },
            new string [] { "\u210F", "\u210F", "\u210F", "\u0127", "\u0127" },
            new string [] { "\u2110", "\u2110", "\u2110", "\u0049", "\u0049" },
            new string [] { "\u2111", "\u2111", "\u2111", "\u0049", "\u0049" },
            new string [] { "\u2112", "\u2112", "\u2112", "\u004C", "\u004C" },
            new string [] { "\u2113", "\u2113", "\u2113", "\u006C", "\u006C" },
            new string [] { "\u2115", "\u2115", "\u2115", "\u004E", "\u004E" },
            new string [] { "\u2116", "\u2116", "\u2116", "\u004E\u006F", "\u004E\u006F" },
            new string [] { "\u2119", "\u2119", "\u2119", "\u0050", "\u0050" },
            new string [] { "\u211A", "\u211A", "\u211A", "\u0051", "\u0051" },
            new string [] { "\u211B", "\u211B", "\u211B", "\u0052", "\u0052" },
            new string [] { "\u211C", "\u211C", "\u211C", "\u0052", "\u0052" },
            new string [] { "\u211D", "\u211D", "\u211D", "\u0052", "\u0052" },
            new string [] { "\u2120", "\u2120", "\u2120", "\u0053\u004D", "\u0053\u004D" },
            new string [] { "\u2121", "\u2121", "\u2121", "\u0054\u0045\u004C", "\u0054\u0045\u004C" },
            new string [] { "\u2122", "\u2122", "\u2122", "\u0054\u004D", "\u0054\u004D" },
            new string [] { "\u2124", "\u2124", "\u2124", "\u005A", "\u005A" },
            new string [] { "\u2126", "\u03A9", "\u03A9", "\u03A9", "\u03A9" },
            new string [] { "\u2128", "\u2128", "\u2128", "\u005A", "\u005A" },
            new string [] { "\u212A", "\u004B", "\u004B", "\u004B", "\u004B" },
            new string [] { "\u212B", "\u00C5", "\u0041\u030A", "\u00C5", "\u0041\u030A" },
            new string [] { "\u212C", "\u212C", "\u212C", "\u0042", "\u0042" },
            new string [] { "\u212D", "\u212D", "\u212D", "\u0043", "\u0043" },
            new string [] { "\u212F", "\u212F", "\u212F", "\u0065", "\u0065" },
            new string [] { "\u2130", "\u2130", "\u2130", "\u0045", "\u0045" },
            new string [] { "\u2131", "\u2131", "\u2131", "\u0046", "\u0046" },
            new string [] { "\u2133", "\u2133", "\u2133", "\u004D", "\u004D" },
            new string [] { "\u2134", "\u2134", "\u2134", "\u006F", "\u006F" },
            new string [] { "\u2135", "\u2135", "\u2135", "\u05D0", "\u05D0" },
            new string [] { "\u2136", "\u2136", "\u2136", "\u05D1", "\u05D1" },
            new string [] { "\u2137", "\u2137", "\u2137", "\u05D2", "\u05D2" },
            new string [] { "\u2138", "\u2138", "\u2138", "\u05D3", "\u05D3" },
            new string [] { "\u2139", "\u2139", "\u2139", "\u0069", "\u0069" },
            new string [] { "\u213B", "\u213B", "\u213B", "\u0046\u0041\u0058", "\u0046\u0041\u0058" },
            new string [] { "\u213C", "\u213C", "\u213C", "\u03C0", "\u03C0" },
            new string [] { "\u213D", "\u213D", "\u213D", "\u03B3", "\u03B3" },
            new string [] { "\u213E", "\u213E", "\u213E", "\u0393", "\u0393" },
            new string [] { "\u213F", "\u213F", "\u213F", "\u03A0", "\u03A0" },
            new string [] { "\u2140", "\u2140", "\u2140", "\u2211", "\u2211" },
            new string [] { "\u2145", "\u2145", "\u2145", "\u0044", "\u0044" },
            new string [] { "\u2146", "\u2146", "\u2146", "\u0064", "\u0064" },
            new string [] { "\u2147", "\u2147", "\u2147", "\u0065", "\u0065" },
            new string [] { "\u2148", "\u2148", "\u2148", "\u0069", "\u0069" },
            new string [] { "\u2149", "\u2149", "\u2149", "\u006A", "\u006A" },
            new string [] { "\u2150", "\u2150", "\u2150", "\u0031\u2044\u0037", "\u0031\u2044\u0037" },
            new string [] { "\u2151", "\u2151", "\u2151", "\u0031\u2044\u0039", "\u0031\u2044\u0039" },
            new string [] { "\u2152", "\u2152", "\u2152", "\u0031\u2044\u0031\u0030", "\u0031\u2044\u0031\u0030" },
            new string [] { "\u2153", "\u2153", "\u2153", "\u0031\u2044\u0033", "\u0031\u2044\u0033" },
            new string [] { "\u2154", "\u2154", "\u2154", "\u0032\u2044\u0033", "\u0032\u2044\u0033" },
            new string [] { "\u2155", "\u2155", "\u2155", "\u0031\u2044\u0035", "\u0031\u2044\u0035" },
            new string [] { "\u2156", "\u2156", "\u2156", "\u0032\u2044\u0035", "\u0032\u2044\u0035" },
            new string [] { "\u2157", "\u2157", "\u2157", "\u0033\u2044\u0035", "\u0033\u2044\u0035" },
            new string [] { "\u2158", "\u2158", "\u2158", "\u0034\u2044\u0035", "\u0034\u2044\u0035" },
            new string [] { "\u2159", "\u2159", "\u2159", "\u0031\u2044\u0036", "\u0031\u2044\u0036" },
            new string [] { "\u215A", "\u215A", "\u215A", "\u0035\u2044\u0036", "\u0035\u2044\u0036" },
            new string [] { "\u215B", "\u215B", "\u215B", "\u0031\u2044\u0038", "\u0031\u2044\u0038" },
            new string [] { "\u215C", "\u215C", "\u215C", "\u0033\u2044\u0038", "\u0033\u2044\u0038" },
            new string [] { "\u215D", "\u215D", "\u215D", "\u0035\u2044\u0038", "\u0035\u2044\u0038" },
            new string [] { "\u215E", "\u215E", "\u215E", "\u0037\u2044\u0038", "\u0037\u2044\u0038" },
            new string [] { "\u215F", "\u215F", "\u215F", "\u0031\u2044", "\u0031\u2044" },
            new string [] { "\u2160", "\u2160", "\u2160", "\u0049", "\u0049" },
            new string [] { "\u2161", "\u2161", "\u2161", "\u0049\u0049", "\u0049\u0049" },
            new string [] { "\u2162", "\u2162", "\u2162", "\u0049\u0049\u0049", "\u0049\u0049\u0049" },
            new string [] { "\u2163", "\u2163", "\u2163", "\u0049\u0056", "\u0049\u0056" },
            new string [] { "\u2164", "\u2164", "\u2164", "\u0056", "\u0056" },
            new string [] { "\u2165", "\u2165", "\u2165", "\u0056\u0049", "\u0056\u0049" },
            new string [] { "\u2166", "\u2166", "\u2166", "\u0056\u0049\u0049", "\u0056\u0049\u0049" },
            new string [] { "\u2167", "\u2167", "\u2167", "\u0056\u0049\u0049\u0049", "\u0056\u0049\u0049\u0049" },
            new string [] { "\u2168", "\u2168", "\u2168", "\u0049\u0058", "\u0049\u0058" },
            new string [] { "\u2169", "\u2169", "\u2169", "\u0058", "\u0058" },
            new string [] { "\u216A", "\u216A", "\u216A", "\u0058\u0049", "\u0058\u0049" },
            new string [] { "\u216B", "\u216B", "\u216B", "\u0058\u0049\u0049", "\u0058\u0049\u0049" },
            new string [] { "\u216C", "\u216C", "\u216C", "\u004C", "\u004C" },
            new string [] { "\u216D", "\u216D", "\u216D", "\u0043", "\u0043" },
            new string [] { "\u216E", "\u216E", "\u216E", "\u0044", "\u0044" },
            new string [] { "\u216F", "\u216F", "\u216F", "\u004D", "\u004D" },
            new string [] { "\u2170", "\u2170", "\u2170", "\u0069", "\u0069" },
            new string [] { "\u2171", "\u2171", "\u2171", "\u0069\u0069", "\u0069\u0069" },
            new string [] { "\u2172", "\u2172", "\u2172", "\u0069\u0069\u0069", "\u0069\u0069\u0069" },
            new string [] { "\u2173", "\u2173", "\u2173", "\u0069\u0076", "\u0069\u0076" },
            new string [] { "\u2174", "\u2174", "\u2174", "\u0076", "\u0076" },
            new string [] { "\u2175", "\u2175", "\u2175", "\u0076\u0069", "\u0076\u0069" },
            new string [] { "\u2176", "\u2176", "\u2176", "\u0076\u0069\u0069", "\u0076\u0069\u0069" },
            new string [] { "\u2177", "\u2177", "\u2177", "\u0076\u0069\u0069\u0069", "\u0076\u0069\u0069\u0069" },
            new string [] { "\u2178", "\u2178", "\u2178", "\u0069\u0078", "\u0069\u0078" },
            new string [] { "\u2179", "\u2179", "\u2179", "\u0078", "\u0078" },
            new string [] { "\u217A", "\u217A", "\u217A", "\u0078\u0069", "\u0078\u0069" },
            new string [] { "\u217B", "\u217B", "\u217B", "\u0078\u0069\u0069", "\u0078\u0069\u0069" },
            new string [] { "\u217C", "\u217C", "\u217C", "\u006C", "\u006C" },
            new string [] { "\u217D", "\u217D", "\u217D", "\u0063", "\u0063" },
            new string [] { "\u217E", "\u217E", "\u217E", "\u0064", "\u0064" },
            new string [] { "\u217F", "\u217F", "\u217F", "\u006D", "\u006D" },
            new string [] { "\u2189", "\u2189", "\u2189", "\u0030\u2044\u0033", "\u0030\u2044\u0033" },
            new string [] { "\u219A", "\u219A", "\u2190\u0338", "\u219A", "\u2190\u0338" },
            new string [] { "\u219B", "\u219B", "\u2192\u0338", "\u219B", "\u2192\u0338" },
            new string [] { "\u21AE", "\u21AE", "\u2194\u0338", "\u21AE", "\u2194\u0338" },
            new string [] { "\u21CD", "\u21CD", "\u21D0\u0338", "\u21CD", "\u21D0\u0338" },
            new string [] { "\u21CE", "\u21CE", "\u21D4\u0338", "\u21CE", "\u21D4\u0338" },
            new string [] { "\u21CF", "\u21CF", "\u21D2\u0338", "\u21CF", "\u21D2\u0338" },
            new string [] { "\u2204", "\u2204", "\u2203\u0338", "\u2204", "\u2203\u0338" },
            new string [] { "\u2209", "\u2209", "\u2208\u0338", "\u2209", "\u2208\u0338" },
            new string [] { "\u220C", "\u220C", "\u220B\u0338", "\u220C", "\u220B\u0338" },
            new string [] { "\u2224", "\u2224", "\u2223\u0338", "\u2224", "\u2223\u0338" },
            new string [] { "\u2226", "\u2226", "\u2225\u0338", "\u2226", "\u2225\u0338" },
            new string [] { "\u222C", "\u222C", "\u222C", "\u222B\u222B", "\u222B\u222B" },
            new string [] { "\u222D", "\u222D", "\u222D", "\u222B\u222B\u222B", "\u222B\u222B\u222B" },
            new string [] { "\u222F", "\u222F", "\u222F", "\u222E\u222E", "\u222E\u222E" },
            new string [] { "\u2230", "\u2230", "\u2230", "\u222E\u222E\u222E", "\u222E\u222E\u222E" },
            new string [] { "\u2241", "\u2241", "\u223C\u0338", "\u2241", "\u223C\u0338" },
            new string [] { "\u2244", "\u2244", "\u2243\u0338", "\u2244", "\u2243\u0338" },
            new string [] { "\u2247", "\u2247", "\u2245\u0338", "\u2247", "\u2245\u0338" },
            new string [] { "\u2249", "\u2249", "\u2248\u0338", "\u2249", "\u2248\u0338" },
            new string [] { "\u2260", "\u2260", "\u003D\u0338", "\u2260", "\u003D\u0338" },
            new string [] { "\u2262", "\u2262", "\u2261\u0338", "\u2262", "\u2261\u0338" },
            new string [] { "\u226D", "\u226D", "\u224D\u0338", "\u226D", "\u224D\u0338" },
            new string [] { "\u226E", "\u226E", "\u003C\u0338", "\u226E", "\u003C\u0338" },
            new string [] { "\u226F", "\u226F", "\u003E\u0338", "\u226F", "\u003E\u0338" },
            new string [] { "\u2270", "\u2270", "\u2264\u0338", "\u2270", "\u2264\u0338" },
            new string [] { "\u2271", "\u2271", "\u2265\u0338", "\u2271", "\u2265\u0338" },
            new string [] { "\u2274", "\u2274", "\u2272\u0338", "\u2274", "\u2272\u0338" },
            new string [] { "\u2275", "\u2275", "\u2273\u0338", "\u2275", "\u2273\u0338" },
            new string [] { "\u2278", "\u2278", "\u2276\u0338", "\u2278", "\u2276\u0338" },
            new string [] { "\u2279", "\u2279", "\u2277\u0338", "\u2279", "\u2277\u0338" },
            new string [] { "\u2280", "\u2280", "\u227A\u0338", "\u2280", "\u227A\u0338" },
            new string [] { "\u2281", "\u2281", "\u227B\u0338", "\u2281", "\u227B\u0338" },
            new string [] { "\u2284", "\u2284", "\u2282\u0338", "\u2284", "\u2282\u0338" },
            new string [] { "\u2285", "\u2285", "\u2283\u0338", "\u2285", "\u2283\u0338" },
            new string [] { "\u2288", "\u2288", "\u2286\u0338", "\u2288", "\u2286\u0338" },
            new string [] { "\u2289", "\u2289", "\u2287\u0338", "\u2289", "\u2287\u0338" },
            new string [] { "\u22AC", "\u22AC", "\u22A2\u0338", "\u22AC", "\u22A2\u0338" },
            new string [] { "\u22AD", "\u22AD", "\u22A8\u0338", "\u22AD", "\u22A8\u0338" },
            new string [] { "\u22AE", "\u22AE", "\u22A9\u0338", "\u22AE", "\u22A9\u0338" },
            new string [] { "\u22AF", "\u22AF", "\u22AB\u0338", "\u22AF", "\u22AB\u0338" },
            new string [] { "\u22E0", "\u22E0", "\u227C\u0338", "\u22E0", "\u227C\u0338" },
            new string [] { "\u22E1", "\u22E1", "\u227D\u0338", "\u22E1", "\u227D\u0338" },
            new string [] { "\u22E2", "\u22E2", "\u2291\u0338", "\u22E2", "\u2291\u0338" },
            new string [] { "\u22E3", "\u22E3", "\u2292\u0338", "\u22E3", "\u2292\u0338" },
            new string [] { "\u22EA", "\u22EA", "\u22B2\u0338", "\u22EA", "\u22B2\u0338" },
            new string [] { "\u22EB", "\u22EB", "\u22B3\u0338", "\u22EB", "\u22B3\u0338" },
            new string [] { "\u22EC", "\u22EC", "\u22B4\u0338", "\u22EC", "\u22B4\u0338" },
            new string [] { "\u22ED", "\u22ED", "\u22B5\u0338", "\u22ED", "\u22B5\u0338" },
            new string [] { "\u2329", "\u3008", "\u3008", "\u3008", "\u3008" },
            new string [] { "\u232A", "\u3009", "\u3009", "\u3009", "\u3009" },
            new string [] { "\u2460", "\u2460", "\u2460", "\u0031", "\u0031" },
            new string [] { "\u2461", "\u2461", "\u2461", "\u0032", "\u0032" },
            new string [] { "\u2462", "\u2462", "\u2462", "\u0033", "\u0033" },
            new string [] { "\u2463", "\u2463", "\u2463", "\u0034", "\u0034" },
            new string [] { "\u2464", "\u2464", "\u2464", "\u0035", "\u0035" },
            new string [] { "\u2465", "\u2465", "\u2465", "\u0036", "\u0036" },
            new string [] { "\u2466", "\u2466", "\u2466", "\u0037", "\u0037" },
            new string [] { "\u2467", "\u2467", "\u2467", "\u0038", "\u0038" },
            new string [] { "\u2468", "\u2468", "\u2468", "\u0039", "\u0039" },
            new string [] { "\u2469", "\u2469", "\u2469", "\u0031\u0030", "\u0031\u0030" },
            new string [] { "\u246A", "\u246A", "\u246A", "\u0031\u0031", "\u0031\u0031" },
            new string [] { "\u246B", "\u246B", "\u246B", "\u0031\u0032", "\u0031\u0032" },
            new string [] { "\u246C", "\u246C", "\u246C", "\u0031\u0033", "\u0031\u0033" },
            new string [] { "\u246D", "\u246D", "\u246D", "\u0031\u0034", "\u0031\u0034" },
            new string [] { "\u246E", "\u246E", "\u246E", "\u0031\u0035", "\u0031\u0035" },
            new string [] { "\u246F", "\u246F", "\u246F", "\u0031\u0036", "\u0031\u0036" },
            new string [] { "\u2470", "\u2470", "\u2470", "\u0031\u0037", "\u0031\u0037" },
            new string [] { "\u2471", "\u2471", "\u2471", "\u0031\u0038", "\u0031\u0038" },
            new string [] { "\u2472", "\u2472", "\u2472", "\u0031\u0039", "\u0031\u0039" },
            new string [] { "\u2473", "\u2473", "\u2473", "\u0032\u0030", "\u0032\u0030" },
            new string [] { "\u2474", "\u2474", "\u2474", "\u0028\u0031\u0029", "\u0028\u0031\u0029" },
            new string [] { "\u2475", "\u2475", "\u2475", "\u0028\u0032\u0029", "\u0028\u0032\u0029" },
            new string [] { "\u2476", "\u2476", "\u2476", "\u0028\u0033\u0029", "\u0028\u0033\u0029" },
            new string [] { "\u2477", "\u2477", "\u2477", "\u0028\u0034\u0029", "\u0028\u0034\u0029" },
            new string [] { "\u2478", "\u2478", "\u2478", "\u0028\u0035\u0029", "\u0028\u0035\u0029" },
            new string [] { "\u2479", "\u2479", "\u2479", "\u0028\u0036\u0029", "\u0028\u0036\u0029" },
            new string [] { "\u247A", "\u247A", "\u247A", "\u0028\u0037\u0029", "\u0028\u0037\u0029" },
            new string [] { "\u247B", "\u247B", "\u247B", "\u0028\u0038\u0029", "\u0028\u0038\u0029" },
            new string [] { "\u247C", "\u247C", "\u247C", "\u0028\u0039\u0029", "\u0028\u0039\u0029" },
            new string [] { "\u247D", "\u247D", "\u247D", "\u0028\u0031\u0030\u0029", "\u0028\u0031\u0030\u0029" },
            new string [] { "\u247E", "\u247E", "\u247E", "\u0028\u0031\u0031\u0029", "\u0028\u0031\u0031\u0029" },
            new string [] { "\u247F", "\u247F", "\u247F", "\u0028\u0031\u0032\u0029", "\u0028\u0031\u0032\u0029" },
            new string [] { "\u2480", "\u2480", "\u2480", "\u0028\u0031\u0033\u0029", "\u0028\u0031\u0033\u0029" },
            new string [] { "\u2481", "\u2481", "\u2481", "\u0028\u0031\u0034\u0029", "\u0028\u0031\u0034\u0029" },
            new string [] { "\u2482", "\u2482", "\u2482", "\u0028\u0031\u0035\u0029", "\u0028\u0031\u0035\u0029" },
            new string [] { "\u2483", "\u2483", "\u2483", "\u0028\u0031\u0036\u0029", "\u0028\u0031\u0036\u0029" },
            new string [] { "\u2484", "\u2484", "\u2484", "\u0028\u0031\u0037\u0029", "\u0028\u0031\u0037\u0029" },
            new string [] { "\u2485", "\u2485", "\u2485", "\u0028\u0031\u0038\u0029", "\u0028\u0031\u0038\u0029" },
            new string [] { "\u2486", "\u2486", "\u2486", "\u0028\u0031\u0039\u0029", "\u0028\u0031\u0039\u0029" },
            new string [] { "\u2487", "\u2487", "\u2487", "\u0028\u0032\u0030\u0029", "\u0028\u0032\u0030\u0029" },
            new string [] { "\u2488", "\u2488", "\u2488", "\u0031\u002E", "\u0031\u002E" },
            new string [] { "\u2489", "\u2489", "\u2489", "\u0032\u002E", "\u0032\u002E" },
            new string [] { "\u248A", "\u248A", "\u248A", "\u0033\u002E", "\u0033\u002E" },
            new string [] { "\u248B", "\u248B", "\u248B", "\u0034\u002E", "\u0034\u002E" },
            new string [] { "\u248C", "\u248C", "\u248C", "\u0035\u002E", "\u0035\u002E" },
            new string [] { "\u248D", "\u248D", "\u248D", "\u0036\u002E", "\u0036\u002E" },
            new string [] { "\u248E", "\u248E", "\u248E", "\u0037\u002E", "\u0037\u002E" },
            new string [] { "\u248F", "\u248F", "\u248F", "\u0038\u002E", "\u0038\u002E" },
            new string [] { "\u2490", "\u2490", "\u2490", "\u0039\u002E", "\u0039\u002E" },
            new string [] { "\u2491", "\u2491", "\u2491", "\u0031\u0030\u002E", "\u0031\u0030\u002E" },
            new string [] { "\u2492", "\u2492", "\u2492", "\u0031\u0031\u002E", "\u0031\u0031\u002E" },
            new string [] { "\u2493", "\u2493", "\u2493", "\u0031\u0032\u002E", "\u0031\u0032\u002E" },
            new string [] { "\u2494", "\u2494", "\u2494", "\u0031\u0033\u002E", "\u0031\u0033\u002E" },
            new string [] { "\u2495", "\u2495", "\u2495", "\u0031\u0034\u002E", "\u0031\u0034\u002E" },
            new string [] { "\u2496", "\u2496", "\u2496", "\u0031\u0035\u002E", "\u0031\u0035\u002E" },
            new string [] { "\u2497", "\u2497", "\u2497", "\u0031\u0036\u002E", "\u0031\u0036\u002E" },
            new string [] { "\u2498", "\u2498", "\u2498", "\u0031\u0037\u002E", "\u0031\u0037\u002E" },
            new string [] { "\u2499", "\u2499", "\u2499", "\u0031\u0038\u002E", "\u0031\u0038\u002E" },
            new string [] { "\u249A", "\u249A", "\u249A", "\u0031\u0039\u002E", "\u0031\u0039\u002E" },
            new string [] { "\u249B", "\u249B", "\u249B", "\u0032\u0030\u002E", "\u0032\u0030\u002E" },
            new string [] { "\u249C", "\u249C", "\u249C", "\u0028\u0061\u0029", "\u0028\u0061\u0029" },
            new string [] { "\u249D", "\u249D", "\u249D", "\u0028\u0062\u0029", "\u0028\u0062\u0029" },
            new string [] { "\u249E", "\u249E", "\u249E", "\u0028\u0063\u0029", "\u0028\u0063\u0029" },
            new string [] { "\u249F", "\u249F", "\u249F", "\u0028\u0064\u0029", "\u0028\u0064\u0029" },
            new string [] { "\u24A0", "\u24A0", "\u24A0", "\u0028\u0065\u0029", "\u0028\u0065\u0029" },
            new string [] { "\u24A1", "\u24A1", "\u24A1", "\u0028\u0066\u0029", "\u0028\u0066\u0029" },
            new string [] { "\u24A2", "\u24A2", "\u24A2", "\u0028\u0067\u0029", "\u0028\u0067\u0029" },
            new string [] { "\u24A3", "\u24A3", "\u24A3", "\u0028\u0068\u0029", "\u0028\u0068\u0029" },
            new string [] { "\u24A4", "\u24A4", "\u24A4", "\u0028\u0069\u0029", "\u0028\u0069\u0029" },
            new string [] { "\u24A5", "\u24A5", "\u24A5", "\u0028\u006A\u0029", "\u0028\u006A\u0029" },
            new string [] { "\u24A6", "\u24A6", "\u24A6", "\u0028\u006B\u0029", "\u0028\u006B\u0029" },
            new string [] { "\u24A7", "\u24A7", "\u24A7", "\u0028\u006C\u0029", "\u0028\u006C\u0029" },
            new string [] { "\u24A8", "\u24A8", "\u24A8", "\u0028\u006D\u0029", "\u0028\u006D\u0029" },
            new string [] { "\u24A9", "\u24A9", "\u24A9", "\u0028\u006E\u0029", "\u0028\u006E\u0029" },
            new string [] { "\u24AA", "\u24AA", "\u24AA", "\u0028\u006F\u0029", "\u0028\u006F\u0029" },
            new string [] { "\u24AB", "\u24AB", "\u24AB", "\u0028\u0070\u0029", "\u0028\u0070\u0029" },
            new string [] { "\u24AC", "\u24AC", "\u24AC", "\u0028\u0071\u0029", "\u0028\u0071\u0029" },
            new string [] { "\u24AD", "\u24AD", "\u24AD", "\u0028\u0072\u0029", "\u0028\u0072\u0029" },
            new string [] { "\u24AE", "\u24AE", "\u24AE", "\u0028\u0073\u0029", "\u0028\u0073\u0029" },
            new string [] { "\u24AF", "\u24AF", "\u24AF", "\u0028\u0074\u0029", "\u0028\u0074\u0029" },
            new string [] { "\u24B0", "\u24B0", "\u24B0", "\u0028\u0075\u0029", "\u0028\u0075\u0029" },
            new string [] { "\u24B1", "\u24B1", "\u24B1", "\u0028\u0076\u0029", "\u0028\u0076\u0029" },
            new string [] { "\u24B2", "\u24B2", "\u24B2", "\u0028\u0077\u0029", "\u0028\u0077\u0029" },
            new string [] { "\u24B3", "\u24B3", "\u24B3", "\u0028\u0078\u0029", "\u0028\u0078\u0029" },
            new string [] { "\u24B4", "\u24B4", "\u24B4", "\u0028\u0079\u0029", "\u0028\u0079\u0029" },
            new string [] { "\u24B5", "\u24B5", "\u24B5", "\u0028\u007A\u0029", "\u0028\u007A\u0029" },
            new string [] { "\u24B6", "\u24B6", "\u24B6", "\u0041", "\u0041" },
            new string [] { "\u24B7", "\u24B7", "\u24B7", "\u0042", "\u0042" },
            new string [] { "\u24B8", "\u24B8", "\u24B8", "\u0043", "\u0043" },
            new string [] { "\u24B9", "\u24B9", "\u24B9", "\u0044", "\u0044" },
            new string [] { "\u24BA", "\u24BA", "\u24BA", "\u0045", "\u0045" },
            new string [] { "\u24BB", "\u24BB", "\u24BB", "\u0046", "\u0046" },
            new string [] { "\u24BC", "\u24BC", "\u24BC", "\u0047", "\u0047" },
            new string [] { "\u24BD", "\u24BD", "\u24BD", "\u0048", "\u0048" },
            new string [] { "\u24BE", "\u24BE", "\u24BE", "\u0049", "\u0049" },
            new string [] { "\u24BF", "\u24BF", "\u24BF", "\u004A", "\u004A" },
            new string [] { "\u24C0", "\u24C0", "\u24C0", "\u004B", "\u004B" },
            new string [] { "\u24C1", "\u24C1", "\u24C1", "\u004C", "\u004C" },
            new string [] { "\u24C2", "\u24C2", "\u24C2", "\u004D", "\u004D" },
            new string [] { "\u24C3", "\u24C3", "\u24C3", "\u004E", "\u004E" },
            new string [] { "\u24C4", "\u24C4", "\u24C4", "\u004F", "\u004F" },
            new string [] { "\u24C5", "\u24C5", "\u24C5", "\u0050", "\u0050" },
            new string [] { "\u24C6", "\u24C6", "\u24C6", "\u0051", "\u0051" },
            new string [] { "\u24C7", "\u24C7", "\u24C7", "\u0052", "\u0052" },
            new string [] { "\u24C8", "\u24C8", "\u24C8", "\u0053", "\u0053" },
            new string [] { "\u24C9", "\u24C9", "\u24C9", "\u0054", "\u0054" },
            new string [] { "\u24CA", "\u24CA", "\u24CA", "\u0055", "\u0055" },
            new string [] { "\u24CB", "\u24CB", "\u24CB", "\u0056", "\u0056" },
            new string [] { "\u24CC", "\u24CC", "\u24CC", "\u0057", "\u0057" },
            new string [] { "\u24CD", "\u24CD", "\u24CD", "\u0058", "\u0058" },
            new string [] { "\u24CE", "\u24CE", "\u24CE", "\u0059", "\u0059" },
            new string [] { "\u24CF", "\u24CF", "\u24CF", "\u005A", "\u005A" },
            new string [] { "\u24D0", "\u24D0", "\u24D0", "\u0061", "\u0061" },
            new string [] { "\u24D1", "\u24D1", "\u24D1", "\u0062", "\u0062" },
            new string [] { "\u24D2", "\u24D2", "\u24D2", "\u0063", "\u0063" },
            new string [] { "\u24D3", "\u24D3", "\u24D3", "\u0064", "\u0064" },
            new string [] { "\u24D4", "\u24D4", "\u24D4", "\u0065", "\u0065" },
            new string [] { "\u24D5", "\u24D5", "\u24D5", "\u0066", "\u0066" },
            new string [] { "\u24D6", "\u24D6", "\u24D6", "\u0067", "\u0067" },
            new string [] { "\u24D7", "\u24D7", "\u24D7", "\u0068", "\u0068" },
            new string [] { "\u24D8", "\u24D8", "\u24D8", "\u0069", "\u0069" },
            new string [] { "\u24D9", "\u24D9", "\u24D9", "\u006A", "\u006A" },
            new string [] { "\u24DA", "\u24DA", "\u24DA", "\u006B", "\u006B" },
            new string [] { "\u24DB", "\u24DB", "\u24DB", "\u006C", "\u006C" },
            new string [] { "\u24DC", "\u24DC", "\u24DC", "\u006D", "\u006D" },
            new string [] { "\u24DD", "\u24DD", "\u24DD", "\u006E", "\u006E" },
            new string [] { "\u24DE", "\u24DE", "\u24DE", "\u006F", "\u006F" },
            new string [] { "\u24DF", "\u24DF", "\u24DF", "\u0070", "\u0070" },
            new string [] { "\u24E0", "\u24E0", "\u24E0", "\u0071", "\u0071" },
            new string [] { "\u24E1", "\u24E1", "\u24E1", "\u0072", "\u0072" },
            new string [] { "\u24E2", "\u24E2", "\u24E2", "\u0073", "\u0073" },
            new string [] { "\u24E3", "\u24E3", "\u24E3", "\u0074", "\u0074" },
            new string [] { "\u24E4", "\u24E4", "\u24E4", "\u0075", "\u0075" },
            new string [] { "\u24E5", "\u24E5", "\u24E5", "\u0076", "\u0076" },
            new string [] { "\u24E6", "\u24E6", "\u24E6", "\u0077", "\u0077" },
            new string [] { "\u24E7", "\u24E7", "\u24E7", "\u0078", "\u0078" },
            new string [] { "\u24E8", "\u24E8", "\u24E8", "\u0079", "\u0079" },
            new string [] { "\u24E9", "\u24E9", "\u24E9", "\u007A", "\u007A" },
            new string [] { "\u24EA", "\u24EA", "\u24EA", "\u0030", "\u0030" },
            new string [] { "\u2A0C", "\u2A0C", "\u2A0C", "\u222B\u222B\u222B\u222B", "\u222B\u222B\u222B\u222B" },
            new string [] { "\u2A74", "\u2A74", "\u2A74", "\u003A\u003A\u003D", "\u003A\u003A\u003D" },
            new string [] { "\u2A75", "\u2A75", "\u2A75", "\u003D\u003D", "\u003D\u003D" },
            new string [] { "\u2A76", "\u2A76", "\u2A76", "\u003D\u003D\u003D", "\u003D\u003D\u003D" },
            new string [] { "\u2ADC", "\u2ADD\u0338", "\u2ADD\u0338", "\u2ADD\u0338", "\u2ADD\u0338" },
            new string [] { "\u2C7C", "\u2C7C", "\u2C7C", "\u006A", "\u006A" },
            new string [] { "\u2C7D", "\u2C7D", "\u2C7D", "\u0056", "\u0056" },
            new string [] { "\u2D6F", "\u2D6F", "\u2D6F", "\u2D61", "\u2D61" },
            new string [] { "\u2E9F", "\u2E9F", "\u2E9F", "\u6BCD", "\u6BCD" },
            new string [] { "\u2EF3", "\u2EF3", "\u2EF3", "\u9F9F", "\u9F9F" },
            new string [] { "\u2F00", "\u2F00", "\u2F00", "\u4E00", "\u4E00" },
            new string [] { "\u2F01", "\u2F01", "\u2F01", "\u4E28", "\u4E28" },
            new string [] { "\u2F02", "\u2F02", "\u2F02", "\u4E36", "\u4E36" },
            new string [] { "\u2F03", "\u2F03", "\u2F03", "\u4E3F", "\u4E3F" },
            new string [] { "\u2F04", "\u2F04", "\u2F04", "\u4E59", "\u4E59" },
            new string [] { "\u2F05", "\u2F05", "\u2F05", "\u4E85", "\u4E85" },
            new string [] { "\u2F06", "\u2F06", "\u2F06", "\u4E8C", "\u4E8C" },
            new string [] { "\u2F07", "\u2F07", "\u2F07", "\u4EA0", "\u4EA0" },
            new string [] { "\u2F08", "\u2F08", "\u2F08", "\u4EBA", "\u4EBA" },
            new string [] { "\u2F09", "\u2F09", "\u2F09", "\u513F", "\u513F" },
            new string [] { "\u2F0A", "\u2F0A", "\u2F0A", "\u5165", "\u5165" },
            new string [] { "\u2F0B", "\u2F0B", "\u2F0B", "\u516B", "\u516B" },
            new string [] { "\u2F0C", "\u2F0C", "\u2F0C", "\u5182", "\u5182" },
            new string [] { "\u2F0D", "\u2F0D", "\u2F0D", "\u5196", "\u5196" },
            new string [] { "\u2F0E", "\u2F0E", "\u2F0E", "\u51AB", "\u51AB" },
            new string [] { "\u2F0F", "\u2F0F", "\u2F0F", "\u51E0", "\u51E0" },
            new string [] { "\u2F10", "\u2F10", "\u2F10", "\u51F5", "\u51F5" },
            new string [] { "\u2F11", "\u2F11", "\u2F11", "\u5200", "\u5200" },
            new string [] { "\u2F12", "\u2F12", "\u2F12", "\u529B", "\u529B" },
            new string [] { "\u2F13", "\u2F13", "\u2F13", "\u52F9", "\u52F9" },
            new string [] { "\u2F14", "\u2F14", "\u2F14", "\u5315", "\u5315" },
            new string [] { "\u2F15", "\u2F15", "\u2F15", "\u531A", "\u531A" },
            new string [] { "\u2F16", "\u2F16", "\u2F16", "\u5338", "\u5338" },
            new string [] { "\u2F17", "\u2F17", "\u2F17", "\u5341", "\u5341" },
            new string [] { "\u2F18", "\u2F18", "\u2F18", "\u535C", "\u535C" },
            new string [] { "\u2F19", "\u2F19", "\u2F19", "\u5369", "\u5369" },
            new string [] { "\u2F1A", "\u2F1A", "\u2F1A", "\u5382", "\u5382" },
            new string [] { "\u2F1B", "\u2F1B", "\u2F1B", "\u53B6", "\u53B6" },
            new string [] { "\u2F1C", "\u2F1C", "\u2F1C", "\u53C8", "\u53C8" },
            new string [] { "\u2F1D", "\u2F1D", "\u2F1D", "\u53E3", "\u53E3" },
            new string [] { "\u2F1E", "\u2F1E", "\u2F1E", "\u56D7", "\u56D7" },
            new string [] { "\u2F1F", "\u2F1F", "\u2F1F", "\u571F", "\u571F" },
            new string [] { "\u2F20", "\u2F20", "\u2F20", "\u58EB", "\u58EB" },
            new string [] { "\u2F21", "\u2F21", "\u2F21", "\u5902", "\u5902" },
            new string [] { "\u2F22", "\u2F22", "\u2F22", "\u590A", "\u590A" },
            new string [] { "\u2F23", "\u2F23", "\u2F23", "\u5915", "\u5915" },
            new string [] { "\u2F24", "\u2F24", "\u2F24", "\u5927", "\u5927" },
            new string [] { "\u2F25", "\u2F25", "\u2F25", "\u5973", "\u5973" },
            new string [] { "\u2F26", "\u2F26", "\u2F26", "\u5B50", "\u5B50" },
            new string [] { "\u2F27", "\u2F27", "\u2F27", "\u5B80", "\u5B80" },
            new string [] { "\u2F28", "\u2F28", "\u2F28", "\u5BF8", "\u5BF8" },
            new string [] { "\u2F29", "\u2F29", "\u2F29", "\u5C0F", "\u5C0F" },
            new string [] { "\u2F2A", "\u2F2A", "\u2F2A", "\u5C22", "\u5C22" },
            new string [] { "\u2F2B", "\u2F2B", "\u2F2B", "\u5C38", "\u5C38" },
            new string [] { "\u2F2C", "\u2F2C", "\u2F2C", "\u5C6E", "\u5C6E" },
            new string [] { "\u2F2D", "\u2F2D", "\u2F2D", "\u5C71", "\u5C71" },
            new string [] { "\u2F2E", "\u2F2E", "\u2F2E", "\u5DDB", "\u5DDB" },
            new string [] { "\u2F2F", "\u2F2F", "\u2F2F", "\u5DE5", "\u5DE5" },
            new string [] { "\u2F30", "\u2F30", "\u2F30", "\u5DF1", "\u5DF1" },
            new string [] { "\u2F31", "\u2F31", "\u2F31", "\u5DFE", "\u5DFE" },
            new string [] { "\u2F32", "\u2F32", "\u2F32", "\u5E72", "\u5E72" },
            new string [] { "\u2F33", "\u2F33", "\u2F33", "\u5E7A", "\u5E7A" },
            new string [] { "\u2F34", "\u2F34", "\u2F34", "\u5E7F", "\u5E7F" },
            new string [] { "\u2F35", "\u2F35", "\u2F35", "\u5EF4", "\u5EF4" },
            new string [] { "\u2F36", "\u2F36", "\u2F36", "\u5EFE", "\u5EFE" },
            new string [] { "\u2F37", "\u2F37", "\u2F37", "\u5F0B", "\u5F0B" },
            new string [] { "\u2F38", "\u2F38", "\u2F38", "\u5F13", "\u5F13" },
            new string [] { "\u2F39", "\u2F39", "\u2F39", "\u5F50", "\u5F50" },
            new string [] { "\u2F3A", "\u2F3A", "\u2F3A", "\u5F61", "\u5F61" },
            new string [] { "\u2F3B", "\u2F3B", "\u2F3B", "\u5F73", "\u5F73" },
            new string [] { "\u2F3C", "\u2F3C", "\u2F3C", "\u5FC3", "\u5FC3" },
            new string [] { "\u2F3D", "\u2F3D", "\u2F3D", "\u6208", "\u6208" },
            new string [] { "\u2F3E", "\u2F3E", "\u2F3E", "\u6236", "\u6236" },
            new string [] { "\u2F3F", "\u2F3F", "\u2F3F", "\u624B", "\u624B" },
            new string [] { "\u2F40", "\u2F40", "\u2F40", "\u652F", "\u652F" },
            new string [] { "\u2F41", "\u2F41", "\u2F41", "\u6534", "\u6534" },
            new string [] { "\u2F42", "\u2F42", "\u2F42", "\u6587", "\u6587" },
            new string [] { "\u2F43", "\u2F43", "\u2F43", "\u6597", "\u6597" },
            new string [] { "\u2F44", "\u2F44", "\u2F44", "\u65A4", "\u65A4" },
            new string [] { "\u2F45", "\u2F45", "\u2F45", "\u65B9", "\u65B9" },
            new string [] { "\u2F46", "\u2F46", "\u2F46", "\u65E0", "\u65E0" },
            new string [] { "\u2F47", "\u2F47", "\u2F47", "\u65E5", "\u65E5" },
            new string [] { "\u2F48", "\u2F48", "\u2F48", "\u66F0", "\u66F0" },
            new string [] { "\u2F49", "\u2F49", "\u2F49", "\u6708", "\u6708" },
            new string [] { "\u2F4A", "\u2F4A", "\u2F4A", "\u6728", "\u6728" },
            new string [] { "\u2F4B", "\u2F4B", "\u2F4B", "\u6B20", "\u6B20" },
            new string [] { "\u2F4C", "\u2F4C", "\u2F4C", "\u6B62", "\u6B62" },
            new string [] { "\u2F4D", "\u2F4D", "\u2F4D", "\u6B79", "\u6B79" },
            new string [] { "\u2F4E", "\u2F4E", "\u2F4E", "\u6BB3", "\u6BB3" },
            new string [] { "\u2F4F", "\u2F4F", "\u2F4F", "\u6BCB", "\u6BCB" },
            new string [] { "\u2F50", "\u2F50", "\u2F50", "\u6BD4", "\u6BD4" },
            new string [] { "\u2F51", "\u2F51", "\u2F51", "\u6BDB", "\u6BDB" },
            new string [] { "\u2F52", "\u2F52", "\u2F52", "\u6C0F", "\u6C0F" },
            new string [] { "\u2F53", "\u2F53", "\u2F53", "\u6C14", "\u6C14" },
            new string [] { "\u2F54", "\u2F54", "\u2F54", "\u6C34", "\u6C34" },
            new string [] { "\u2F55", "\u2F55", "\u2F55", "\u706B", "\u706B" },
            new string [] { "\u2F56", "\u2F56", "\u2F56", "\u722A", "\u722A" },
            new string [] { "\u2F57", "\u2F57", "\u2F57", "\u7236", "\u7236" },
            new string [] { "\u2F58", "\u2F58", "\u2F58", "\u723B", "\u723B" },
            new string [] { "\u2F59", "\u2F59", "\u2F59", "\u723F", "\u723F" },
            new string [] { "\u2F5A", "\u2F5A", "\u2F5A", "\u7247", "\u7247" },
            new string [] { "\u2F5B", "\u2F5B", "\u2F5B", "\u7259", "\u7259" },
            new string [] { "\u2F5C", "\u2F5C", "\u2F5C", "\u725B", "\u725B" },
            new string [] { "\u2F5D", "\u2F5D", "\u2F5D", "\u72AC", "\u72AC" },
            new string [] { "\u2F5E", "\u2F5E", "\u2F5E", "\u7384", "\u7384" },
            new string [] { "\u2F5F", "\u2F5F", "\u2F5F", "\u7389", "\u7389" },
            new string [] { "\u2F60", "\u2F60", "\u2F60", "\u74DC", "\u74DC" },
            new string [] { "\u2F61", "\u2F61", "\u2F61", "\u74E6", "\u74E6" },
            new string [] { "\u2F62", "\u2F62", "\u2F62", "\u7518", "\u7518" },
            new string [] { "\u2F63", "\u2F63", "\u2F63", "\u751F", "\u751F" },
            new string [] { "\u2F64", "\u2F64", "\u2F64", "\u7528", "\u7528" },
            new string [] { "\u2F65", "\u2F65", "\u2F65", "\u7530", "\u7530" },
            new string [] { "\u2F66", "\u2F66", "\u2F66", "\u758B", "\u758B" },
            new string [] { "\u2F67", "\u2F67", "\u2F67", "\u7592", "\u7592" },
            new string [] { "\u2F68", "\u2F68", "\u2F68", "\u7676", "\u7676" },
            new string [] { "\u2F69", "\u2F69", "\u2F69", "\u767D", "\u767D" },
            new string [] { "\u2F6A", "\u2F6A", "\u2F6A", "\u76AE", "\u76AE" },
            new string [] { "\u2F6B", "\u2F6B", "\u2F6B", "\u76BF", "\u76BF" },
            new string [] { "\u2F6C", "\u2F6C", "\u2F6C", "\u76EE", "\u76EE" },
            new string [] { "\u2F6D", "\u2F6D", "\u2F6D", "\u77DB", "\u77DB" },
            new string [] { "\u2F6E", "\u2F6E", "\u2F6E", "\u77E2", "\u77E2" },
            new string [] { "\u2F6F", "\u2F6F", "\u2F6F", "\u77F3", "\u77F3" },
            new string [] { "\u2F70", "\u2F70", "\u2F70", "\u793A", "\u793A" },
            new string [] { "\u2F71", "\u2F71", "\u2F71", "\u79B8", "\u79B8" },
            new string [] { "\u2F72", "\u2F72", "\u2F72", "\u79BE", "\u79BE" },
            new string [] { "\u2F73", "\u2F73", "\u2F73", "\u7A74", "\u7A74" },
            new string [] { "\u2F74", "\u2F74", "\u2F74", "\u7ACB", "\u7ACB" },
            new string [] { "\u2F75", "\u2F75", "\u2F75", "\u7AF9", "\u7AF9" },
            new string [] { "\u2F76", "\u2F76", "\u2F76", "\u7C73", "\u7C73" },
            new string [] { "\u2F77", "\u2F77", "\u2F77", "\u7CF8", "\u7CF8" },
            new string [] { "\u2F78", "\u2F78", "\u2F78", "\u7F36", "\u7F36" },
            new string [] { "\u2F79", "\u2F79", "\u2F79", "\u7F51", "\u7F51" },
            new string [] { "\u2F7A", "\u2F7A", "\u2F7A", "\u7F8A", "\u7F8A" },
            new string [] { "\u2F7B", "\u2F7B", "\u2F7B", "\u7FBD", "\u7FBD" },
            new string [] { "\u2F7C", "\u2F7C", "\u2F7C", "\u8001", "\u8001" },
            new string [] { "\u2F7D", "\u2F7D", "\u2F7D", "\u800C", "\u800C" },
            new string [] { "\u2F7E", "\u2F7E", "\u2F7E", "\u8012", "\u8012" },
            new string [] { "\u2F7F", "\u2F7F", "\u2F7F", "\u8033", "\u8033" },
            new string [] { "\u2F80", "\u2F80", "\u2F80", "\u807F", "\u807F" },
            new string [] { "\u2F81", "\u2F81", "\u2F81", "\u8089", "\u8089" },
            new string [] { "\u2F82", "\u2F82", "\u2F82", "\u81E3", "\u81E3" },
            new string [] { "\u2F83", "\u2F83", "\u2F83", "\u81EA", "\u81EA" },
            new string [] { "\u2F84", "\u2F84", "\u2F84", "\u81F3", "\u81F3" },
            new string [] { "\u2F85", "\u2F85", "\u2F85", "\u81FC", "\u81FC" },
            new string [] { "\u2F86", "\u2F86", "\u2F86", "\u820C", "\u820C" },
            new string [] { "\u2F87", "\u2F87", "\u2F87", "\u821B", "\u821B" },
            new string [] { "\u2F88", "\u2F88", "\u2F88", "\u821F", "\u821F" },
            new string [] { "\u2F89", "\u2F89", "\u2F89", "\u826E", "\u826E" },
            new string [] { "\u2F8A", "\u2F8A", "\u2F8A", "\u8272", "\u8272" },
            new string [] { "\u2F8B", "\u2F8B", "\u2F8B", "\u8278", "\u8278" },
            new string [] { "\u2F8C", "\u2F8C", "\u2F8C", "\u864D", "\u864D" },
            new string [] { "\u2F8D", "\u2F8D", "\u2F8D", "\u866B", "\u866B" },
            new string [] { "\u2F8E", "\u2F8E", "\u2F8E", "\u8840", "\u8840" },
            new string [] { "\u2F8F", "\u2F8F", "\u2F8F", "\u884C", "\u884C" },
            new string [] { "\u2F90", "\u2F90", "\u2F90", "\u8863", "\u8863" },
            new string [] { "\u2F91", "\u2F91", "\u2F91", "\u897E", "\u897E" },
            new string [] { "\u2F92", "\u2F92", "\u2F92", "\u898B", "\u898B" },
            new string [] { "\u2F93", "\u2F93", "\u2F93", "\u89D2", "\u89D2" },
            new string [] { "\u2F94", "\u2F94", "\u2F94", "\u8A00", "\u8A00" },
            new string [] { "\u2F95", "\u2F95", "\u2F95", "\u8C37", "\u8C37" },
            new string [] { "\u2F96", "\u2F96", "\u2F96", "\u8C46", "\u8C46" },
            new string [] { "\u2F97", "\u2F97", "\u2F97", "\u8C55", "\u8C55" },
            new string [] { "\u2F98", "\u2F98", "\u2F98", "\u8C78", "\u8C78" },
            new string [] { "\u2F99", "\u2F99", "\u2F99", "\u8C9D", "\u8C9D" },
            new string [] { "\u2F9A", "\u2F9A", "\u2F9A", "\u8D64", "\u8D64" },
            new string [] { "\u2F9B", "\u2F9B", "\u2F9B", "\u8D70", "\u8D70" },
            new string [] { "\u2F9C", "\u2F9C", "\u2F9C", "\u8DB3", "\u8DB3" },
            new string [] { "\u2F9D", "\u2F9D", "\u2F9D", "\u8EAB", "\u8EAB" },
            new string [] { "\u2F9E", "\u2F9E", "\u2F9E", "\u8ECA", "\u8ECA" },
            new string [] { "\u2F9F", "\u2F9F", "\u2F9F", "\u8F9B", "\u8F9B" },
            new string [] { "\u2FA0", "\u2FA0", "\u2FA0", "\u8FB0", "\u8FB0" },
            new string [] { "\u2FA1", "\u2FA1", "\u2FA1", "\u8FB5", "\u8FB5" },
            new string [] { "\u2FA2", "\u2FA2", "\u2FA2", "\u9091", "\u9091" },
            new string [] { "\u2FA3", "\u2FA3", "\u2FA3", "\u9149", "\u9149" },
            new string [] { "\u2FA4", "\u2FA4", "\u2FA4", "\u91C6", "\u91C6" },
            new string [] { "\u2FA5", "\u2FA5", "\u2FA5", "\u91CC", "\u91CC" },
            new string [] { "\u2FA6", "\u2FA6", "\u2FA6", "\u91D1", "\u91D1" },
            new string [] { "\u2FA7", "\u2FA7", "\u2FA7", "\u9577", "\u9577" },
            new string [] { "\u2FA8", "\u2FA8", "\u2FA8", "\u9580", "\u9580" },
            new string [] { "\u2FA9", "\u2FA9", "\u2FA9", "\u961C", "\u961C" },
            new string [] { "\u2FAA", "\u2FAA", "\u2FAA", "\u96B6", "\u96B6" },
            new string [] { "\u2FAB", "\u2FAB", "\u2FAB", "\u96B9", "\u96B9" },
            new string [] { "\u2FAC", "\u2FAC", "\u2FAC", "\u96E8", "\u96E8" },
            new string [] { "\u2FAD", "\u2FAD", "\u2FAD", "\u9751", "\u9751" },
            new string [] { "\u2FAE", "\u2FAE", "\u2FAE", "\u975E", "\u975E" },
            new string [] { "\u2FAF", "\u2FAF", "\u2FAF", "\u9762", "\u9762" },
            new string [] { "\u2FB0", "\u2FB0", "\u2FB0", "\u9769", "\u9769" },
            new string [] { "\u2FB1", "\u2FB1", "\u2FB1", "\u97CB", "\u97CB" },
            new string [] { "\u2FB2", "\u2FB2", "\u2FB2", "\u97ED", "\u97ED" },
            new string [] { "\u2FB3", "\u2FB3", "\u2FB3", "\u97F3", "\u97F3" },
            new string [] { "\u2FB4", "\u2FB4", "\u2FB4", "\u9801", "\u9801" },
            new string [] { "\u2FB5", "\u2FB5", "\u2FB5", "\u98A8", "\u98A8" },
            new string [] { "\u2FB6", "\u2FB6", "\u2FB6", "\u98DB", "\u98DB" },
            new string [] { "\u2FB7", "\u2FB7", "\u2FB7", "\u98DF", "\u98DF" },
            new string [] { "\u2FB8", "\u2FB8", "\u2FB8", "\u9996", "\u9996" },
            new string [] { "\u2FB9", "\u2FB9", "\u2FB9", "\u9999", "\u9999" },
            new string [] { "\u2FBA", "\u2FBA", "\u2FBA", "\u99AC", "\u99AC" },
            new string [] { "\u2FBB", "\u2FBB", "\u2FBB", "\u9AA8", "\u9AA8" },
            new string [] { "\u2FBC", "\u2FBC", "\u2FBC", "\u9AD8", "\u9AD8" },
            new string [] { "\u2FBD", "\u2FBD", "\u2FBD", "\u9ADF", "\u9ADF" },
            new string [] { "\u2FBE", "\u2FBE", "\u2FBE", "\u9B25", "\u9B25" },
            new string [] { "\u2FBF", "\u2FBF", "\u2FBF", "\u9B2F", "\u9B2F" },
            new string [] { "\u2FC0", "\u2FC0", "\u2FC0", "\u9B32", "\u9B32" },
            new string [] { "\u2FC1", "\u2FC1", "\u2FC1", "\u9B3C", "\u9B3C" },
            new string [] { "\u2FC2", "\u2FC2", "\u2FC2", "\u9B5A", "\u9B5A" },
            new string [] { "\u2FC3", "\u2FC3", "\u2FC3", "\u9CE5", "\u9CE5" },
            new string [] { "\u2FC4", "\u2FC4", "\u2FC4", "\u9E75", "\u9E75" },
            new string [] { "\u2FC5", "\u2FC5", "\u2FC5", "\u9E7F", "\u9E7F" },
            new string [] { "\u2FC6", "\u2FC6", "\u2FC6", "\u9EA5", "\u9EA5" },
            new string [] { "\u2FC7", "\u2FC7", "\u2FC7", "\u9EBB", "\u9EBB" },
            new string [] { "\u2FC8", "\u2FC8", "\u2FC8", "\u9EC3", "\u9EC3" },
            new string [] { "\u2FC9", "\u2FC9", "\u2FC9", "\u9ECD", "\u9ECD" },
            new string [] { "\u2FCA", "\u2FCA", "\u2FCA", "\u9ED1", "\u9ED1" },
            new string [] { "\u2FCB", "\u2FCB", "\u2FCB", "\u9EF9", "\u9EF9" },
            new string [] { "\u2FCC", "\u2FCC", "\u2FCC", "\u9EFD", "\u9EFD" },
            new string [] { "\u2FCD", "\u2FCD", "\u2FCD", "\u9F0E", "\u9F0E" },
            new string [] { "\u2FCE", "\u2FCE", "\u2FCE", "\u9F13", "\u9F13" },
            new string [] { "\u2FCF", "\u2FCF", "\u2FCF", "\u9F20", "\u9F20" },
            new string [] { "\u2FD0", "\u2FD0", "\u2FD0", "\u9F3B", "\u9F3B" },
            new string [] { "\u2FD1", "\u2FD1", "\u2FD1", "\u9F4A", "\u9F4A" },
            new string [] { "\u2FD2", "\u2FD2", "\u2FD2", "\u9F52", "\u9F52" },
            new string [] { "\u2FD3", "\u2FD3", "\u2FD3", "\u9F8D", "\u9F8D" },
            new string [] { "\u2FD4", "\u2FD4", "\u2FD4", "\u9F9C", "\u9F9C" },
            new string [] { "\u2FD5", "\u2FD5", "\u2FD5", "\u9FA0", "\u9FA0" },
            new string [] { "\u3000", "\u3000", "\u3000", "\u0020", "\u0020" },
            new string [] { "\u3036", "\u3036", "\u3036", "\u3012", "\u3012" },
            new string [] { "\u3038", "\u3038", "\u3038", "\u5341", "\u5341" },
            new string [] { "\u3039", "\u3039", "\u3039", "\u5344", "\u5344" },
            new string [] { "\u303A", "\u303A", "\u303A", "\u5345", "\u5345" },
            new string [] { "\u304C", "\u304C", "\u304B\u3099", "\u304C", "\u304B\u3099" },
            new string [] { "\u304E", "\u304E", "\u304D\u3099", "\u304E", "\u304D\u3099" },
            new string [] { "\u3050", "\u3050", "\u304F\u3099", "\u3050", "\u304F\u3099" },
            new string [] { "\u3052", "\u3052", "\u3051\u3099", "\u3052", "\u3051\u3099" },
            new string [] { "\u3054", "\u3054", "\u3053\u3099", "\u3054", "\u3053\u3099" },
            new string [] { "\u3056", "\u3056", "\u3055\u3099", "\u3056", "\u3055\u3099" },
            new string [] { "\u3058", "\u3058", "\u3057\u3099", "\u3058", "\u3057\u3099" },
            new string [] { "\u305A", "\u305A", "\u3059\u3099", "\u305A", "\u3059\u3099" },
            new string [] { "\u305C", "\u305C", "\u305B\u3099", "\u305C", "\u305B\u3099" },
            new string [] { "\u305E", "\u305E", "\u305D\u3099", "\u305E", "\u305D\u3099" },
            new string [] { "\u3060", "\u3060", "\u305F\u3099", "\u3060", "\u305F\u3099" },
            new string [] { "\u3062", "\u3062", "\u3061\u3099", "\u3062", "\u3061\u3099" },
            new string [] { "\u3065", "\u3065", "\u3064\u3099", "\u3065", "\u3064\u3099" },
            new string [] { "\u3067", "\u3067", "\u3066\u3099", "\u3067", "\u3066\u3099" },
            new string [] { "\u3069", "\u3069", "\u3068\u3099", "\u3069", "\u3068\u3099" },
            new string [] { "\u3070", "\u3070", "\u306F\u3099", "\u3070", "\u306F\u3099" },
            new string [] { "\u3071", "\u3071", "\u306F\u309A", "\u3071", "\u306F\u309A" },
            new string [] { "\u3073", "\u3073", "\u3072\u3099", "\u3073", "\u3072\u3099" },
            new string [] { "\u3074", "\u3074", "\u3072\u309A", "\u3074", "\u3072\u309A" },
            new string [] { "\u3076", "\u3076", "\u3075\u3099", "\u3076", "\u3075\u3099" },
            new string [] { "\u3077", "\u3077", "\u3075\u309A", "\u3077", "\u3075\u309A" },
            new string [] { "\u3079", "\u3079", "\u3078\u3099", "\u3079", "\u3078\u3099" },
            new string [] { "\u307A", "\u307A", "\u3078\u309A", "\u307A", "\u3078\u309A" },
            new string [] { "\u307C", "\u307C", "\u307B\u3099", "\u307C", "\u307B\u3099" },
            new string [] { "\u307D", "\u307D", "\u307B\u309A", "\u307D", "\u307B\u309A" },
            new string [] { "\u3094", "\u3094", "\u3046\u3099", "\u3094", "\u3046\u3099" },
            new string [] { "\u309B", "\u309B", "\u309B", "\u0020\u3099", "\u0020\u3099" },
            new string [] { "\u309C", "\u309C", "\u309C", "\u0020\u309A", "\u0020\u309A" },
            new string [] { "\u309E", "\u309E", "\u309D\u3099", "\u309E", "\u309D\u3099" },
            new string [] { "\u309F", "\u309F", "\u309F", "\u3088\u308A", "\u3088\u308A" },
            new string [] { "\u30AC", "\u30AC", "\u30AB\u3099", "\u30AC", "\u30AB\u3099" },
            new string [] { "\u30AE", "\u30AE", "\u30AD\u3099", "\u30AE", "\u30AD\u3099" },
            new string [] { "\u30B0", "\u30B0", "\u30AF\u3099", "\u30B0", "\u30AF\u3099" },
            new string [] { "\u30B2", "\u30B2", "\u30B1\u3099", "\u30B2", "\u30B1\u3099" },
            new string [] { "\u30B4", "\u30B4", "\u30B3\u3099", "\u30B4", "\u30B3\u3099" },
            new string [] { "\u30B6", "\u30B6", "\u30B5\u3099", "\u30B6", "\u30B5\u3099" },
            new string [] { "\u30B8", "\u30B8", "\u30B7\u3099", "\u30B8", "\u30B7\u3099" },
            new string [] { "\u30BA", "\u30BA", "\u30B9\u3099", "\u30BA", "\u30B9\u3099" },
            new string [] { "\u30BC", "\u30BC", "\u30BB\u3099", "\u30BC", "\u30BB\u3099" },
            new string [] { "\u30BE", "\u30BE", "\u30BD\u3099", "\u30BE", "\u30BD\u3099" },
            new string [] { "\u30C0", "\u30C0", "\u30BF\u3099", "\u30C0", "\u30BF\u3099" },
            new string [] { "\u30C2", "\u30C2", "\u30C1\u3099", "\u30C2", "\u30C1\u3099" },
            new string [] { "\u30C5", "\u30C5", "\u30C4\u3099", "\u30C5", "\u30C4\u3099" },
            new string [] { "\u30C7", "\u30C7", "\u30C6\u3099", "\u30C7", "\u30C6\u3099" },
            new string [] { "\u30C9", "\u30C9", "\u30C8\u3099", "\u30C9", "\u30C8\u3099" },
            new string [] { "\u30D0", "\u30D0", "\u30CF\u3099", "\u30D0", "\u30CF\u3099" },
            new string [] { "\u30D1", "\u30D1", "\u30CF\u309A", "\u30D1", "\u30CF\u309A" },
            new string [] { "\u30D3", "\u30D3", "\u30D2\u3099", "\u30D3", "\u30D2\u3099" },
            new string [] { "\u30D4", "\u30D4", "\u30D2\u309A", "\u30D4", "\u30D2\u309A" },
            new string [] { "\u30D6", "\u30D6", "\u30D5\u3099", "\u30D6", "\u30D5\u3099" },
            new string [] { "\u30D7", "\u30D7", "\u30D5\u309A", "\u30D7", "\u30D5\u309A" },
            new string [] { "\u30D9", "\u30D9", "\u30D8\u3099", "\u30D9", "\u30D8\u3099" },
            new string [] { "\u30DA", "\u30DA", "\u30D8\u309A", "\u30DA", "\u30D8\u309A" },
            new string [] { "\u30DC", "\u30DC", "\u30DB\u3099", "\u30DC", "\u30DB\u3099" },
            new string [] { "\u30DD", "\u30DD", "\u30DB\u309A", "\u30DD", "\u30DB\u309A" },
            new string [] { "\u30F4", "\u30F4", "\u30A6\u3099", "\u30F4", "\u30A6\u3099" },
            new string [] { "\u30F7", "\u30F7", "\u30EF\u3099", "\u30F7", "\u30EF\u3099" },
            new string [] { "\u30F8", "\u30F8", "\u30F0\u3099", "\u30F8", "\u30F0\u3099" },
            new string [] { "\u30F9", "\u30F9", "\u30F1\u3099", "\u30F9", "\u30F1\u3099" },
            new string [] { "\u30FA", "\u30FA", "\u30F2\u3099", "\u30FA", "\u30F2\u3099" },
            new string [] { "\u30FE", "\u30FE", "\u30FD\u3099", "\u30FE", "\u30FD\u3099" },
            new string [] { "\u30FF", "\u30FF", "\u30FF", "\u30B3\u30C8", "\u30B3\u30C8" },
            new string [] { "\u3131", "\u3131", "\u3131", "\u1100", "\u1100" },
            new string [] { "\u3132", "\u3132", "\u3132", "\u1101", "\u1101" },
            new string [] { "\u3133", "\u3133", "\u3133", "\u11AA", "\u11AA" },
            new string [] { "\u3134", "\u3134", "\u3134", "\u1102", "\u1102" },
            new string [] { "\u3135", "\u3135", "\u3135", "\u11AC", "\u11AC" },
            new string [] { "\u3136", "\u3136", "\u3136", "\u11AD", "\u11AD" },
            new string [] { "\u3137", "\u3137", "\u3137", "\u1103", "\u1103" },
            new string [] { "\u3138", "\u3138", "\u3138", "\u1104", "\u1104" },
            new string [] { "\u3139", "\u3139", "\u3139", "\u1105", "\u1105" },
            new string [] { "\u313A", "\u313A", "\u313A", "\u11B0", "\u11B0" },
            new string [] { "\u313B", "\u313B", "\u313B", "\u11B1", "\u11B1" },
            new string [] { "\u313C", "\u313C", "\u313C", "\u11B2", "\u11B2" },
            new string [] { "\u313D", "\u313D", "\u313D", "\u11B3", "\u11B3" },
            new string [] { "\u313E", "\u313E", "\u313E", "\u11B4", "\u11B4" },
            new string [] { "\u313F", "\u313F", "\u313F", "\u11B5", "\u11B5" },
            new string [] { "\u3140", "\u3140", "\u3140", "\u111A", "\u111A" },
            new string [] { "\u3141", "\u3141", "\u3141", "\u1106", "\u1106" },
            new string [] { "\u3142", "\u3142", "\u3142", "\u1107", "\u1107" },
            new string [] { "\u3143", "\u3143", "\u3143", "\u1108", "\u1108" },
            new string [] { "\u3144", "\u3144", "\u3144", "\u1121", "\u1121" },
            new string [] { "\u3145", "\u3145", "\u3145", "\u1109", "\u1109" },
            new string [] { "\u3146", "\u3146", "\u3146", "\u110A", "\u110A" },
            new string [] { "\u3147", "\u3147", "\u3147", "\u110B", "\u110B" },
            new string [] { "\u3148", "\u3148", "\u3148", "\u110C", "\u110C" },
            new string [] { "\u3149", "\u3149", "\u3149", "\u110D", "\u110D" },
            new string [] { "\u314A", "\u314A", "\u314A", "\u110E", "\u110E" },
            new string [] { "\u314B", "\u314B", "\u314B", "\u110F", "\u110F" },
            new string [] { "\u314C", "\u314C", "\u314C", "\u1110", "\u1110" },
            new string [] { "\u314D", "\u314D", "\u314D", "\u1111", "\u1111" },
            new string [] { "\u314E", "\u314E", "\u314E", "\u1112", "\u1112" },
            new string [] { "\u314F", "\u314F", "\u314F", "\u1161", "\u1161" },
            new string [] { "\u3150", "\u3150", "\u3150", "\u1162", "\u1162" },
            new string [] { "\u3151", "\u3151", "\u3151", "\u1163", "\u1163" },
            new string [] { "\u3152", "\u3152", "\u3152", "\u1164", "\u1164" },
            new string [] { "\u3153", "\u3153", "\u3153", "\u1165", "\u1165" },
            new string [] { "\u3154", "\u3154", "\u3154", "\u1166", "\u1166" },
            new string [] { "\u3155", "\u3155", "\u3155", "\u1167", "\u1167" },
            new string [] { "\u3156", "\u3156", "\u3156", "\u1168", "\u1168" },
            new string [] { "\u3157", "\u3157", "\u3157", "\u1169", "\u1169" },
            new string [] { "\u3158", "\u3158", "\u3158", "\u116A", "\u116A" },
            new string [] { "\u3159", "\u3159", "\u3159", "\u116B", "\u116B" },
            new string [] { "\u315A", "\u315A", "\u315A", "\u116C", "\u116C" },
            new string [] { "\u315B", "\u315B", "\u315B", "\u116D", "\u116D" },
            new string [] { "\u315C", "\u315C", "\u315C", "\u116E", "\u116E" },
            new string [] { "\u315D", "\u315D", "\u315D", "\u116F", "\u116F" },
            new string [] { "\u315E", "\u315E", "\u315E", "\u1170", "\u1170" },
            new string [] { "\u315F", "\u315F", "\u315F", "\u1171", "\u1171" },
            new string [] { "\u3160", "\u3160", "\u3160", "\u1172", "\u1172" },
            new string [] { "\u3161", "\u3161", "\u3161", "\u1173", "\u1173" },
            new string [] { "\u3162", "\u3162", "\u3162", "\u1174", "\u1174" },
            new string [] { "\u3163", "\u3163", "\u3163", "\u1175", "\u1175" },
            new string [] { "\u3164", "\u3164", "\u3164", "\u1160", "\u1160" },
            new string [] { "\u3165", "\u3165", "\u3165", "\u1114", "\u1114" },
            new string [] { "\u3166", "\u3166", "\u3166", "\u1115", "\u1115" },
            new string [] { "\u3167", "\u3167", "\u3167", "\u11C7", "\u11C7" },
            new string [] { "\u3168", "\u3168", "\u3168", "\u11C8", "\u11C8" },
            new string [] { "\u3169", "\u3169", "\u3169", "\u11CC", "\u11CC" },
            new string [] { "\u316A", "\u316A", "\u316A", "\u11CE", "\u11CE" },
            new string [] { "\u316B", "\u316B", "\u316B", "\u11D3", "\u11D3" },
            new string [] { "\u316C", "\u316C", "\u316C", "\u11D7", "\u11D7" },
            new string [] { "\u316D", "\u316D", "\u316D", "\u11D9", "\u11D9" },
            new string [] { "\u316E", "\u316E", "\u316E", "\u111C", "\u111C" },
            new string [] { "\u316F", "\u316F", "\u316F", "\u11DD", "\u11DD" },
            new string [] { "\u3170", "\u3170", "\u3170", "\u11DF", "\u11DF" },
            new string [] { "\u3171", "\u3171", "\u3171", "\u111D", "\u111D" },
            new string [] { "\u3172", "\u3172", "\u3172", "\u111E", "\u111E" },
            new string [] { "\u3173", "\u3173", "\u3173", "\u1120", "\u1120" },
            new string [] { "\u3174", "\u3174", "\u3174", "\u1122", "\u1122" },
            new string [] { "\u3175", "\u3175", "\u3175", "\u1123", "\u1123" },
            new string [] { "\u3176", "\u3176", "\u3176", "\u1127", "\u1127" },
            new string [] { "\u3177", "\u3177", "\u3177", "\u1129", "\u1129" },
            new string [] { "\u3178", "\u3178", "\u3178", "\u112B", "\u112B" },
            new string [] { "\u3179", "\u3179", "\u3179", "\u112C", "\u112C" },
            new string [] { "\u317A", "\u317A", "\u317A", "\u112D", "\u112D" },
            new string [] { "\u317B", "\u317B", "\u317B", "\u112E", "\u112E" },
            new string [] { "\u317C", "\u317C", "\u317C", "\u112F", "\u112F" },
            new string [] { "\u317D", "\u317D", "\u317D", "\u1132", "\u1132" },
            new string [] { "\u317E", "\u317E", "\u317E", "\u1136", "\u1136" },
            new string [] { "\u317F", "\u317F", "\u317F", "\u1140", "\u1140" },
            new string [] { "\u3180", "\u3180", "\u3180", "\u1147", "\u1147" },
            new string [] { "\u3181", "\u3181", "\u3181", "\u114C", "\u114C" },
            new string [] { "\u3182", "\u3182", "\u3182", "\u11F1", "\u11F1" },
            new string [] { "\u3183", "\u3183", "\u3183", "\u11F2", "\u11F2" },
            new string [] { "\u3184", "\u3184", "\u3184", "\u1157", "\u1157" },
            new string [] { "\u3185", "\u3185", "\u3185", "\u1158", "\u1158" },
            new string [] { "\u3186", "\u3186", "\u3186", "\u1159", "\u1159" },
            new string [] { "\u3187", "\u3187", "\u3187", "\u1184", "\u1184" },
            new string [] { "\u3188", "\u3188", "\u3188", "\u1185", "\u1185" },
            new string [] { "\u3189", "\u3189", "\u3189", "\u1188", "\u1188" },
            new string [] { "\u318A", "\u318A", "\u318A", "\u1191", "\u1191" },
            new string [] { "\u318B", "\u318B", "\u318B", "\u1192", "\u1192" },
            new string [] { "\u318C", "\u318C", "\u318C", "\u1194", "\u1194" },
            new string [] { "\u318D", "\u318D", "\u318D", "\u119E", "\u119E" },
            new string [] { "\u318E", "\u318E", "\u318E", "\u11A1", "\u11A1" },
            new string [] { "\u3192", "\u3192", "\u3192", "\u4E00", "\u4E00" },
            new string [] { "\u3193", "\u3193", "\u3193", "\u4E8C", "\u4E8C" },
            new string [] { "\u3194", "\u3194", "\u3194", "\u4E09", "\u4E09" },
            new string [] { "\u3195", "\u3195", "\u3195", "\u56DB", "\u56DB" },
            new string [] { "\u3196", "\u3196", "\u3196", "\u4E0A", "\u4E0A" },
            new string [] { "\u3197", "\u3197", "\u3197", "\u4E2D", "\u4E2D" },
            new string [] { "\u3198", "\u3198", "\u3198", "\u4E0B", "\u4E0B" },
            new string [] { "\u3199", "\u3199", "\u3199", "\u7532", "\u7532" },
            new string [] { "\u319A", "\u319A", "\u319A", "\u4E59", "\u4E59" },
            new string [] { "\u319B", "\u319B", "\u319B", "\u4E19", "\u4E19" },
            new string [] { "\u319C", "\u319C", "\u319C", "\u4E01", "\u4E01" },
            new string [] { "\u319D", "\u319D", "\u319D", "\u5929", "\u5929" },
            new string [] { "\u319E", "\u319E", "\u319E", "\u5730", "\u5730" },
            new string [] { "\u319F", "\u319F", "\u319F", "\u4EBA", "\u4EBA" },
            new string [] { "\u3200", "\u3200", "\u3200", "\u0028\u1100\u0029", "\u0028\u1100\u0029" },
            new string [] { "\u3201", "\u3201", "\u3201", "\u0028\u1102\u0029", "\u0028\u1102\u0029" },
            new string [] { "\u3202", "\u3202", "\u3202", "\u0028\u1103\u0029", "\u0028\u1103\u0029" },
            new string [] { "\u3203", "\u3203", "\u3203", "\u0028\u1105\u0029", "\u0028\u1105\u0029" },
            new string [] { "\u3204", "\u3204", "\u3204", "\u0028\u1106\u0029", "\u0028\u1106\u0029" },
            new string [] { "\u3205", "\u3205", "\u3205", "\u0028\u1107\u0029", "\u0028\u1107\u0029" },
            new string [] { "\u3206", "\u3206", "\u3206", "\u0028\u1109\u0029", "\u0028\u1109\u0029" },
            new string [] { "\u3207", "\u3207", "\u3207", "\u0028\u110B\u0029", "\u0028\u110B\u0029" },
            new string [] { "\u3208", "\u3208", "\u3208", "\u0028\u110C\u0029", "\u0028\u110C\u0029" },
            new string [] { "\u3209", "\u3209", "\u3209", "\u0028\u110E\u0029", "\u0028\u110E\u0029" },
            new string [] { "\u320A", "\u320A", "\u320A", "\u0028\u110F\u0029", "\u0028\u110F\u0029" },
            new string [] { "\u320B", "\u320B", "\u320B", "\u0028\u1110\u0029", "\u0028\u1110\u0029" },
            new string [] { "\u320C", "\u320C", "\u320C", "\u0028\u1111\u0029", "\u0028\u1111\u0029" },
            new string [] { "\u320D", "\u320D", "\u320D", "\u0028\u1112\u0029", "\u0028\u1112\u0029" },
            new string [] { "\u320E", "\u320E", "\u320E", "\u0028\uAC00\u0029", "\u0028\u1100\u1161\u0029" },
            new string [] { "\u320F", "\u320F", "\u320F", "\u0028\uB098\u0029", "\u0028\u1102\u1161\u0029" },
            new string [] { "\u3210", "\u3210", "\u3210", "\u0028\uB2E4\u0029", "\u0028\u1103\u1161\u0029" },
            new string [] { "\u3211", "\u3211", "\u3211", "\u0028\uB77C\u0029", "\u0028\u1105\u1161\u0029" },
            new string [] { "\u3212", "\u3212", "\u3212", "\u0028\uB9C8\u0029", "\u0028\u1106\u1161\u0029" },
            new string [] { "\u3213", "\u3213", "\u3213", "\u0028\uBC14\u0029", "\u0028\u1107\u1161\u0029" },
            new string [] { "\u3214", "\u3214", "\u3214", "\u0028\uC0AC\u0029", "\u0028\u1109\u1161\u0029" },
            new string [] { "\u3215", "\u3215", "\u3215", "\u0028\uC544\u0029", "\u0028\u110B\u1161\u0029" },
            new string [] { "\u3216", "\u3216", "\u3216", "\u0028\uC790\u0029", "\u0028\u110C\u1161\u0029" },
            new string [] { "\u3217", "\u3217", "\u3217", "\u0028\uCC28\u0029", "\u0028\u110E\u1161\u0029" },
            new string [] { "\u3218", "\u3218", "\u3218", "\u0028\uCE74\u0029", "\u0028\u110F\u1161\u0029" },
            new string [] { "\u3219", "\u3219", "\u3219", "\u0028\uD0C0\u0029", "\u0028\u1110\u1161\u0029" },
            new string [] { "\u321A", "\u321A", "\u321A", "\u0028\uD30C\u0029", "\u0028\u1111\u1161\u0029" },
            new string [] { "\u321B", "\u321B", "\u321B", "\u0028\uD558\u0029", "\u0028\u1112\u1161\u0029" },
            new string [] { "\u321C", "\u321C", "\u321C", "\u0028\uC8FC\u0029", "\u0028\u110C\u116E\u0029" },
            new string [] { "\u321D", "\u321D", "\u321D", "\u0028\uC624\uC804\u0029", "\u0028\u110B\u1169\u110C\u1165\u11AB\u0029" },
            new string [] { "\u321E", "\u321E", "\u321E", "\u0028\uC624\uD6C4\u0029", "\u0028\u110B\u1169\u1112\u116E\u0029" },
            new string [] { "\u3220", "\u3220", "\u3220", "\u0028\u4E00\u0029", "\u0028\u4E00\u0029" },
            new string [] { "\u3221", "\u3221", "\u3221", "\u0028\u4E8C\u0029", "\u0028\u4E8C\u0029" },
            new string [] { "\u3222", "\u3222", "\u3222", "\u0028\u4E09\u0029", "\u0028\u4E09\u0029" },
            new string [] { "\u3223", "\u3223", "\u3223", "\u0028\u56DB\u0029", "\u0028\u56DB\u0029" },
            new string [] { "\u3224", "\u3224", "\u3224", "\u0028\u4E94\u0029", "\u0028\u4E94\u0029" },
            new string [] { "\u3225", "\u3225", "\u3225", "\u0028\u516D\u0029", "\u0028\u516D\u0029" },
            new string [] { "\u3226", "\u3226", "\u3226", "\u0028\u4E03\u0029", "\u0028\u4E03\u0029" },
            new string [] { "\u3227", "\u3227", "\u3227", "\u0028\u516B\u0029", "\u0028\u516B\u0029" },
            new string [] { "\u3228", "\u3228", "\u3228", "\u0028\u4E5D\u0029", "\u0028\u4E5D\u0029" },
            new string [] { "\u3229", "\u3229", "\u3229", "\u0028\u5341\u0029", "\u0028\u5341\u0029" },
            new string [] { "\u322A", "\u322A", "\u322A", "\u0028\u6708\u0029", "\u0028\u6708\u0029" },
            new string [] { "\u322B", "\u322B", "\u322B", "\u0028\u706B\u0029", "\u0028\u706B\u0029" },
            new string [] { "\u322C", "\u322C", "\u322C", "\u0028\u6C34\u0029", "\u0028\u6C34\u0029" },
            new string [] { "\u322D", "\u322D", "\u322D", "\u0028\u6728\u0029", "\u0028\u6728\u0029" },
            new string [] { "\u322E", "\u322E", "\u322E", "\u0028\u91D1\u0029", "\u0028\u91D1\u0029" },
            new string [] { "\u322F", "\u322F", "\u322F", "\u0028\u571F\u0029", "\u0028\u571F\u0029" },
            new string [] { "\u3230", "\u3230", "\u3230", "\u0028\u65E5\u0029", "\u0028\u65E5\u0029" },
            new string [] { "\u3231", "\u3231", "\u3231", "\u0028\u682A\u0029", "\u0028\u682A\u0029" },
            new string [] { "\u3232", "\u3232", "\u3232", "\u0028\u6709\u0029", "\u0028\u6709\u0029" },
            new string [] { "\u3233", "\u3233", "\u3233", "\u0028\u793E\u0029", "\u0028\u793E\u0029" },
            new string [] { "\u3234", "\u3234", "\u3234", "\u0028\u540D\u0029", "\u0028\u540D\u0029" },
            new string [] { "\u3235", "\u3235", "\u3235", "\u0028\u7279\u0029", "\u0028\u7279\u0029" },
            new string [] { "\u3236", "\u3236", "\u3236", "\u0028\u8CA1\u0029", "\u0028\u8CA1\u0029" },
            new string [] { "\u3237", "\u3237", "\u3237", "\u0028\u795D\u0029", "\u0028\u795D\u0029" },
            new string [] { "\u3238", "\u3238", "\u3238", "\u0028\u52B4\u0029", "\u0028\u52B4\u0029" },
            new string [] { "\u3239", "\u3239", "\u3239", "\u0028\u4EE3\u0029", "\u0028\u4EE3\u0029" },
            new string [] { "\u323A", "\u323A", "\u323A", "\u0028\u547C\u0029", "\u0028\u547C\u0029" },
            new string [] { "\u323B", "\u323B", "\u323B", "\u0028\u5B66\u0029", "\u0028\u5B66\u0029" },
            new string [] { "\u323C", "\u323C", "\u323C", "\u0028\u76E3\u0029", "\u0028\u76E3\u0029" },
            new string [] { "\u323D", "\u323D", "\u323D", "\u0028\u4F01\u0029", "\u0028\u4F01\u0029" },
            new string [] { "\u323E", "\u323E", "\u323E", "\u0028\u8CC7\u0029", "\u0028\u8CC7\u0029" },
            new string [] { "\u323F", "\u323F", "\u323F", "\u0028\u5354\u0029", "\u0028\u5354\u0029" },
            new string [] { "\u3240", "\u3240", "\u3240", "\u0028\u796D\u0029", "\u0028\u796D\u0029" },
            new string [] { "\u3241", "\u3241", "\u3241", "\u0028\u4F11\u0029", "\u0028\u4F11\u0029" },
            new string [] { "\u3242", "\u3242", "\u3242", "\u0028\u81EA\u0029", "\u0028\u81EA\u0029" },
            new string [] { "\u3243", "\u3243", "\u3243", "\u0028\u81F3\u0029", "\u0028\u81F3\u0029" },
            new string [] { "\u3244", "\u3244", "\u3244", "\u554F", "\u554F" },
            new string [] { "\u3245", "\u3245", "\u3245", "\u5E7C", "\u5E7C" },
            new string [] { "\u3246", "\u3246", "\u3246", "\u6587", "\u6587" },
            new string [] { "\u3247", "\u3247", "\u3247", "\u7B8F", "\u7B8F" },
            new string [] { "\u3250", "\u3250", "\u3250", "\u0050\u0054\u0045", "\u0050\u0054\u0045" },
            new string [] { "\u3251", "\u3251", "\u3251", "\u0032\u0031", "\u0032\u0031" },
            new string [] { "\u3252", "\u3252", "\u3252", "\u0032\u0032", "\u0032\u0032" },
            new string [] { "\u3253", "\u3253", "\u3253", "\u0032\u0033", "\u0032\u0033" },
            new string [] { "\u3254", "\u3254", "\u3254", "\u0032\u0034", "\u0032\u0034" },
            new string [] { "\u3255", "\u3255", "\u3255", "\u0032\u0035", "\u0032\u0035" },
            new string [] { "\u3256", "\u3256", "\u3256", "\u0032\u0036", "\u0032\u0036" },
            new string [] { "\u3257", "\u3257", "\u3257", "\u0032\u0037", "\u0032\u0037" },
            new string [] { "\u3258", "\u3258", "\u3258", "\u0032\u0038", "\u0032\u0038" },
            new string [] { "\u3259", "\u3259", "\u3259", "\u0032\u0039", "\u0032\u0039" },
            new string [] { "\u325A", "\u325A", "\u325A", "\u0033\u0030", "\u0033\u0030" },
            new string [] { "\u325B", "\u325B", "\u325B", "\u0033\u0031", "\u0033\u0031" },
            new string [] { "\u325C", "\u325C", "\u325C", "\u0033\u0032", "\u0033\u0032" },
            new string [] { "\u325D", "\u325D", "\u325D", "\u0033\u0033", "\u0033\u0033" },
            new string [] { "\u325E", "\u325E", "\u325E", "\u0033\u0034", "\u0033\u0034" },
            new string [] { "\u325F", "\u325F", "\u325F", "\u0033\u0035", "\u0033\u0035" },
            new string [] { "\u3260", "\u3260", "\u3260", "\u1100", "\u1100" },
            new string [] { "\u3261", "\u3261", "\u3261", "\u1102", "\u1102" },
            new string [] { "\u3262", "\u3262", "\u3262", "\u1103", "\u1103" },
            new string [] { "\u3263", "\u3263", "\u3263", "\u1105", "\u1105" },
            new string [] { "\u3264", "\u3264", "\u3264", "\u1106", "\u1106" },
            new string [] { "\u3265", "\u3265", "\u3265", "\u1107", "\u1107" },
            new string [] { "\u3266", "\u3266", "\u3266", "\u1109", "\u1109" },
            new string [] { "\u3267", "\u3267", "\u3267", "\u110B", "\u110B" },
            new string [] { "\u3268", "\u3268", "\u3268", "\u110C", "\u110C" },
            new string [] { "\u3269", "\u3269", "\u3269", "\u110E", "\u110E" },
            new string [] { "\u326A", "\u326A", "\u326A", "\u110F", "\u110F" },
            new string [] { "\u326B", "\u326B", "\u326B", "\u1110", "\u1110" },
            new string [] { "\u326C", "\u326C", "\u326C", "\u1111", "\u1111" },
            new string [] { "\u326D", "\u326D", "\u326D", "\u1112", "\u1112" },
            new string [] { "\u326E", "\u326E", "\u326E", "\uAC00", "\u1100\u1161" },
            new string [] { "\u326F", "\u326F", "\u326F", "\uB098", "\u1102\u1161" },
            new string [] { "\u3270", "\u3270", "\u3270", "\uB2E4", "\u1103\u1161" },
            new string [] { "\u3271", "\u3271", "\u3271", "\uB77C", "\u1105\u1161" },
            new string [] { "\u3272", "\u3272", "\u3272", "\uB9C8", "\u1106\u1161" },
            new string [] { "\u3273", "\u3273", "\u3273", "\uBC14", "\u1107\u1161" },
            new string [] { "\u3274", "\u3274", "\u3274", "\uC0AC", "\u1109\u1161" },
            new string [] { "\u3275", "\u3275", "\u3275", "\uC544", "\u110B\u1161" },
            new string [] { "\u3276", "\u3276", "\u3276", "\uC790", "\u110C\u1161" },
            new string [] { "\u3277", "\u3277", "\u3277", "\uCC28", "\u110E\u1161" },
            new string [] { "\u3278", "\u3278", "\u3278", "\uCE74", "\u110F\u1161" },
            new string [] { "\u3279", "\u3279", "\u3279", "\uD0C0", "\u1110\u1161" },
            new string [] { "\u327A", "\u327A", "\u327A", "\uD30C", "\u1111\u1161" },
            new string [] { "\u327B", "\u327B", "\u327B", "\uD558", "\u1112\u1161" },
            new string [] { "\u327C", "\u327C", "\u327C", "\uCC38\uACE0", "\u110E\u1161\u11B7\u1100\u1169" },
            new string [] { "\u327D", "\u327D", "\u327D", "\uC8FC\uC758", "\u110C\u116E\u110B\u1174" },
            new string [] { "\u327E", "\u327E", "\u327E", "\uC6B0", "\u110B\u116E" },
            new string [] { "\u3280", "\u3280", "\u3280", "\u4E00", "\u4E00" },
            new string [] { "\u3281", "\u3281", "\u3281", "\u4E8C", "\u4E8C" },
            new string [] { "\u3282", "\u3282", "\u3282", "\u4E09", "\u4E09" },
            new string [] { "\u3283", "\u3283", "\u3283", "\u56DB", "\u56DB" },
            new string [] { "\u3284", "\u3284", "\u3284", "\u4E94", "\u4E94" },
            new string [] { "\u3285", "\u3285", "\u3285", "\u516D", "\u516D" },
            new string [] { "\u3286", "\u3286", "\u3286", "\u4E03", "\u4E03" },
            new string [] { "\u3287", "\u3287", "\u3287", "\u516B", "\u516B" },
            new string [] { "\u3288", "\u3288", "\u3288", "\u4E5D", "\u4E5D" },
            new string [] { "\u3289", "\u3289", "\u3289", "\u5341", "\u5341" },
            new string [] { "\u328A", "\u328A", "\u328A", "\u6708", "\u6708" },
            new string [] { "\u328B", "\u328B", "\u328B", "\u706B", "\u706B" },
            new string [] { "\u328C", "\u328C", "\u328C", "\u6C34", "\u6C34" },
            new string [] { "\u328D", "\u328D", "\u328D", "\u6728", "\u6728" },
            new string [] { "\u328E", "\u328E", "\u328E", "\u91D1", "\u91D1" },
            new string [] { "\u328F", "\u328F", "\u328F", "\u571F", "\u571F" },
            new string [] { "\u3290", "\u3290", "\u3290", "\u65E5", "\u65E5" },
            new string [] { "\u3291", "\u3291", "\u3291", "\u682A", "\u682A" },
            new string [] { "\u3292", "\u3292", "\u3292", "\u6709", "\u6709" },
            new string [] { "\u3293", "\u3293", "\u3293", "\u793E", "\u793E" },
            new string [] { "\u3294", "\u3294", "\u3294", "\u540D", "\u540D" },
            new string [] { "\u3295", "\u3295", "\u3295", "\u7279", "\u7279" },
            new string [] { "\u3296", "\u3296", "\u3296", "\u8CA1", "\u8CA1" },
            new string [] { "\u3297", "\u3297", "\u3297", "\u795D", "\u795D" },
            new string [] { "\u3298", "\u3298", "\u3298", "\u52B4", "\u52B4" },
            new string [] { "\u3299", "\u3299", "\u3299", "\u79D8", "\u79D8" },
            new string [] { "\u329A", "\u329A", "\u329A", "\u7537", "\u7537" },
            new string [] { "\u329B", "\u329B", "\u329B", "\u5973", "\u5973" },
            new string [] { "\u329C", "\u329C", "\u329C", "\u9069", "\u9069" },
            new string [] { "\u329D", "\u329D", "\u329D", "\u512A", "\u512A" },
            new string [] { "\u329E", "\u329E", "\u329E", "\u5370", "\u5370" },
            new string [] { "\u329F", "\u329F", "\u329F", "\u6CE8", "\u6CE8" },
            new string [] { "\u32A0", "\u32A0", "\u32A0", "\u9805", "\u9805" },
            new string [] { "\u32A1", "\u32A1", "\u32A1", "\u4F11", "\u4F11" },
            new string [] { "\u32A2", "\u32A2", "\u32A2", "\u5199", "\u5199" },
            new string [] { "\u32A3", "\u32A3", "\u32A3", "\u6B63", "\u6B63" },
            new string [] { "\u32A4", "\u32A4", "\u32A4", "\u4E0A", "\u4E0A" },
            new string [] { "\u32A5", "\u32A5", "\u32A5", "\u4E2D", "\u4E2D" },
            new string [] { "\u32A6", "\u32A6", "\u32A6", "\u4E0B", "\u4E0B" },
            new string [] { "\u32A7", "\u32A7", "\u32A7", "\u5DE6", "\u5DE6" },
            new string [] { "\u32A8", "\u32A8", "\u32A8", "\u53F3", "\u53F3" },
            new string [] { "\u32A9", "\u32A9", "\u32A9", "\u533B", "\u533B" },
            new string [] { "\u32AA", "\u32AA", "\u32AA", "\u5B97", "\u5B97" },
            new string [] { "\u32AB", "\u32AB", "\u32AB", "\u5B66", "\u5B66" },
            new string [] { "\u32AC", "\u32AC", "\u32AC", "\u76E3", "\u76E3" },
            new string [] { "\u32AD", "\u32AD", "\u32AD", "\u4F01", "\u4F01" },
            new string [] { "\u32AE", "\u32AE", "\u32AE", "\u8CC7", "\u8CC7" },
            new string [] { "\u32AF", "\u32AF", "\u32AF", "\u5354", "\u5354" },
            new string [] { "\u32B0", "\u32B0", "\u32B0", "\u591C", "\u591C" },
            new string [] { "\u32B1", "\u32B1", "\u32B1", "\u0033\u0036", "\u0033\u0036" },
            new string [] { "\u32B2", "\u32B2", "\u32B2", "\u0033\u0037", "\u0033\u0037" },
            new string [] { "\u32B3", "\u32B3", "\u32B3", "\u0033\u0038", "\u0033\u0038" },
            new string [] { "\u32B4", "\u32B4", "\u32B4", "\u0033\u0039", "\u0033\u0039" },
            new string [] { "\u32B5", "\u32B5", "\u32B5", "\u0034\u0030", "\u0034\u0030" },
            new string [] { "\u32B6", "\u32B6", "\u32B6", "\u0034\u0031", "\u0034\u0031" },
            new string [] { "\u32B7", "\u32B7", "\u32B7", "\u0034\u0032", "\u0034\u0032" },
            new string [] { "\u32B8", "\u32B8", "\u32B8", "\u0034\u0033", "\u0034\u0033" },
            new string [] { "\u32B9", "\u32B9", "\u32B9", "\u0034\u0034", "\u0034\u0034" },
            new string [] { "\u32BA", "\u32BA", "\u32BA", "\u0034\u0035", "\u0034\u0035" },
            new string [] { "\u32BB", "\u32BB", "\u32BB", "\u0034\u0036", "\u0034\u0036" },
            new string [] { "\u32BC", "\u32BC", "\u32BC", "\u0034\u0037", "\u0034\u0037" },
            new string [] { "\u32BD", "\u32BD", "\u32BD", "\u0034\u0038", "\u0034\u0038" },
            new string [] { "\u32BE", "\u32BE", "\u32BE", "\u0034\u0039", "\u0034\u0039" },
            new string [] { "\u32BF", "\u32BF", "\u32BF", "\u0035\u0030", "\u0035\u0030" },
            new string [] { "\u32C0", "\u32C0", "\u32C0", "\u0031\u6708", "\u0031\u6708" },
            new string [] { "\u32C1", "\u32C1", "\u32C1", "\u0032\u6708", "\u0032\u6708" },
            new string [] { "\u32C2", "\u32C2", "\u32C2", "\u0033\u6708", "\u0033\u6708" },
            new string [] { "\u32C3", "\u32C3", "\u32C3", "\u0034\u6708", "\u0034\u6708" },
            new string [] { "\u32C4", "\u32C4", "\u32C4", "\u0035\u6708", "\u0035\u6708" },
            new string [] { "\u32C5", "\u32C5", "\u32C5", "\u0036\u6708", "\u0036\u6708" },
            new string [] { "\u32C6", "\u32C6", "\u32C6", "\u0037\u6708", "\u0037\u6708" },
            new string [] { "\u32C7", "\u32C7", "\u32C7", "\u0038\u6708", "\u0038\u6708" },
            new string [] { "\u32C8", "\u32C8", "\u32C8", "\u0039\u6708", "\u0039\u6708" },
            new string [] { "\u32C9", "\u32C9", "\u32C9", "\u0031\u0030\u6708", "\u0031\u0030\u6708" },
            new string [] { "\u32CA", "\u32CA", "\u32CA", "\u0031\u0031\u6708", "\u0031\u0031\u6708" },
            new string [] { "\u32CB", "\u32CB", "\u32CB", "\u0031\u0032\u6708", "\u0031\u0032\u6708" },
            new string [] { "\u32CC", "\u32CC", "\u32CC", "\u0048\u0067", "\u0048\u0067" },
            new string [] { "\u32CD", "\u32CD", "\u32CD", "\u0065\u0072\u0067", "\u0065\u0072\u0067" },
            new string [] { "\u32CE", "\u32CE", "\u32CE", "\u0065\u0056", "\u0065\u0056" },
            new string [] { "\u32CF", "\u32CF", "\u32CF", "\u004C\u0054\u0044", "\u004C\u0054\u0044" },
            new string [] { "\u32D0", "\u32D0", "\u32D0", "\u30A2", "\u30A2" },
            new string [] { "\u32D1", "\u32D1", "\u32D1", "\u30A4", "\u30A4" },
            new string [] { "\u32D2", "\u32D2", "\u32D2", "\u30A6", "\u30A6" },
            new string [] { "\u32D3", "\u32D3", "\u32D3", "\u30A8", "\u30A8" },
            new string [] { "\u32D4", "\u32D4", "\u32D4", "\u30AA", "\u30AA" },
            new string [] { "\u32D5", "\u32D5", "\u32D5", "\u30AB", "\u30AB" },
            new string [] { "\u32D6", "\u32D6", "\u32D6", "\u30AD", "\u30AD" },
            new string [] { "\u32D7", "\u32D7", "\u32D7", "\u30AF", "\u30AF" },
            new string [] { "\u32D8", "\u32D8", "\u32D8", "\u30B1", "\u30B1" },
            new string [] { "\u32D9", "\u32D9", "\u32D9", "\u30B3", "\u30B3" },
            new string [] { "\u32DA", "\u32DA", "\u32DA", "\u30B5", "\u30B5" },
            new string [] { "\u32DB", "\u32DB", "\u32DB", "\u30B7", "\u30B7" },
            new string [] { "\u32DC", "\u32DC", "\u32DC", "\u30B9", "\u30B9" },
            new string [] { "\u32DD", "\u32DD", "\u32DD", "\u30BB", "\u30BB" },
            new string [] { "\u32DE", "\u32DE", "\u32DE", "\u30BD", "\u30BD" },
            new string [] { "\u32DF", "\u32DF", "\u32DF", "\u30BF", "\u30BF" },
            new string [] { "\u32E0", "\u32E0", "\u32E0", "\u30C1", "\u30C1" },
            new string [] { "\u32E1", "\u32E1", "\u32E1", "\u30C4", "\u30C4" },
            new string [] { "\u32E2", "\u32E2", "\u32E2", "\u30C6", "\u30C6" },
            new string [] { "\u32E3", "\u32E3", "\u32E3", "\u30C8", "\u30C8" },
            new string [] { "\u32E4", "\u32E4", "\u32E4", "\u30CA", "\u30CA" },
            new string [] { "\u32E5", "\u32E5", "\u32E5", "\u30CB", "\u30CB" },
            new string [] { "\u32E6", "\u32E6", "\u32E6", "\u30CC", "\u30CC" },
            new string [] { "\u32E7", "\u32E7", "\u32E7", "\u30CD", "\u30CD" },
            new string [] { "\u32E8", "\u32E8", "\u32E8", "\u30CE", "\u30CE" },
            new string [] { "\u32E9", "\u32E9", "\u32E9", "\u30CF", "\u30CF" },
            new string [] { "\u32EA", "\u32EA", "\u32EA", "\u30D2", "\u30D2" },
            new string [] { "\u32EB", "\u32EB", "\u32EB", "\u30D5", "\u30D5" },
            new string [] { "\u32EC", "\u32EC", "\u32EC", "\u30D8", "\u30D8" },
            new string [] { "\u32ED", "\u32ED", "\u32ED", "\u30DB", "\u30DB" },
            new string [] { "\u32EE", "\u32EE", "\u32EE", "\u30DE", "\u30DE" },
            new string [] { "\u32EF", "\u32EF", "\u32EF", "\u30DF", "\u30DF" },
            new string [] { "\u32F0", "\u32F0", "\u32F0", "\u30E0", "\u30E0" },
            new string [] { "\u32F1", "\u32F1", "\u32F1", "\u30E1", "\u30E1" },
            new string [] { "\u32F2", "\u32F2", "\u32F2", "\u30E2", "\u30E2" },
            new string [] { "\u32F3", "\u32F3", "\u32F3", "\u30E4", "\u30E4" },
            new string [] { "\u32F4", "\u32F4", "\u32F4", "\u30E6", "\u30E6" },
            new string [] { "\u32F5", "\u32F5", "\u32F5", "\u30E8", "\u30E8" },
            new string [] { "\u32F6", "\u32F6", "\u32F6", "\u30E9", "\u30E9" },
            new string [] { "\u32F7", "\u32F7", "\u32F7", "\u30EA", "\u30EA" },
            new string [] { "\u32F8", "\u32F8", "\u32F8", "\u30EB", "\u30EB" },
            new string [] { "\u32F9", "\u32F9", "\u32F9", "\u30EC", "\u30EC" },
            new string [] { "\u32FA", "\u32FA", "\u32FA", "\u30ED", "\u30ED" },
            new string [] { "\u32FB", "\u32FB", "\u32FB", "\u30EF", "\u30EF" },
            new string [] { "\u32FC", "\u32FC", "\u32FC", "\u30F0", "\u30F0" },
            new string [] { "\u32FD", "\u32FD", "\u32FD", "\u30F1", "\u30F1" },
            new string [] { "\u32FE", "\u32FE", "\u32FE", "\u30F2", "\u30F2" },
            new string [] { "\u3300", "\u3300", "\u3300", "\u30A2\u30D1\u30FC\u30C8", "\u30A2\u30CF\u309A\u30FC\u30C8" },
            new string [] { "\u3301", "\u3301", "\u3301", "\u30A2\u30EB\u30D5\u30A1", "\u30A2\u30EB\u30D5\u30A1" },
            new string [] { "\u3302", "\u3302", "\u3302", "\u30A2\u30F3\u30DA\u30A2", "\u30A2\u30F3\u30D8\u309A\u30A2" },
            new string [] { "\u3303", "\u3303", "\u3303", "\u30A2\u30FC\u30EB", "\u30A2\u30FC\u30EB" },
            new string [] { "\u3304", "\u3304", "\u3304", "\u30A4\u30CB\u30F3\u30B0", "\u30A4\u30CB\u30F3\u30AF\u3099" },
            new string [] { "\u3305", "\u3305", "\u3305", "\u30A4\u30F3\u30C1", "\u30A4\u30F3\u30C1" },
            new string [] { "\u3306", "\u3306", "\u3306", "\u30A6\u30A9\u30F3", "\u30A6\u30A9\u30F3" },
            new string [] { "\u3307", "\u3307", "\u3307", "\u30A8\u30B9\u30AF\u30FC\u30C9", "\u30A8\u30B9\u30AF\u30FC\u30C8\u3099" },
            new string [] { "\u3308", "\u3308", "\u3308", "\u30A8\u30FC\u30AB\u30FC", "\u30A8\u30FC\u30AB\u30FC" },
            new string [] { "\u3309", "\u3309", "\u3309", "\u30AA\u30F3\u30B9", "\u30AA\u30F3\u30B9" },
            new string [] { "\u330A", "\u330A", "\u330A", "\u30AA\u30FC\u30E0", "\u30AA\u30FC\u30E0" },
            new string [] { "\u330B", "\u330B", "\u330B", "\u30AB\u30A4\u30EA", "\u30AB\u30A4\u30EA" },
            new string [] { "\u330C", "\u330C", "\u330C", "\u30AB\u30E9\u30C3\u30C8", "\u30AB\u30E9\u30C3\u30C8" },
            new string [] { "\u330D", "\u330D", "\u330D", "\u30AB\u30ED\u30EA\u30FC", "\u30AB\u30ED\u30EA\u30FC" },
            new string [] { "\u330E", "\u330E", "\u330E", "\u30AC\u30ED\u30F3", "\u30AB\u3099\u30ED\u30F3" },
            new string [] { "\u330F", "\u330F", "\u330F", "\u30AC\u30F3\u30DE", "\u30AB\u3099\u30F3\u30DE" },
            new string [] { "\u3310", "\u3310", "\u3310", "\u30AE\u30AC", "\u30AD\u3099\u30AB\u3099" },
            new string [] { "\u3311", "\u3311", "\u3311", "\u30AE\u30CB\u30FC", "\u30AD\u3099\u30CB\u30FC" },
            new string [] { "\u3312", "\u3312", "\u3312", "\u30AD\u30E5\u30EA\u30FC", "\u30AD\u30E5\u30EA\u30FC" },
            new string [] { "\u3313", "\u3313", "\u3313", "\u30AE\u30EB\u30C0\u30FC", "\u30AD\u3099\u30EB\u30BF\u3099\u30FC" },
            new string [] { "\u3314", "\u3314", "\u3314", "\u30AD\u30ED", "\u30AD\u30ED" },
            new string [] { "\u3315", "\u3315", "\u3315", "\u30AD\u30ED\u30B0\u30E9\u30E0", "\u30AD\u30ED\u30AF\u3099\u30E9\u30E0" },
            new string [] { "\u3316", "\u3316", "\u3316", "\u30AD\u30ED\u30E1\u30FC\u30C8\u30EB", "\u30AD\u30ED\u30E1\u30FC\u30C8\u30EB" },
            new string [] { "\u3317", "\u3317", "\u3317", "\u30AD\u30ED\u30EF\u30C3\u30C8", "\u30AD\u30ED\u30EF\u30C3\u30C8" },
            new string [] { "\u3318", "\u3318", "\u3318", "\u30B0\u30E9\u30E0", "\u30AF\u3099\u30E9\u30E0" },
            new string [] { "\u3319", "\u3319", "\u3319", "\u30B0\u30E9\u30E0\u30C8\u30F3", "\u30AF\u3099\u30E9\u30E0\u30C8\u30F3" },
            new string [] { "\u331A", "\u331A", "\u331A", "\u30AF\u30EB\u30BC\u30A4\u30ED", "\u30AF\u30EB\u30BB\u3099\u30A4\u30ED" },
            new string [] { "\u331B", "\u331B", "\u331B", "\u30AF\u30ED\u30FC\u30CD", "\u30AF\u30ED\u30FC\u30CD" },
            new string [] { "\u331C", "\u331C", "\u331C", "\u30B1\u30FC\u30B9", "\u30B1\u30FC\u30B9" },
            new string [] { "\u331D", "\u331D", "\u331D", "\u30B3\u30EB\u30CA", "\u30B3\u30EB\u30CA" },
            new string [] { "\u331E", "\u331E", "\u331E", "\u30B3\u30FC\u30DD", "\u30B3\u30FC\u30DB\u309A" },
            new string [] { "\u331F", "\u331F", "\u331F", "\u30B5\u30A4\u30AF\u30EB", "\u30B5\u30A4\u30AF\u30EB" },
            new string [] { "\u3320", "\u3320", "\u3320", "\u30B5\u30F3\u30C1\u30FC\u30E0", "\u30B5\u30F3\u30C1\u30FC\u30E0" },
            new string [] { "\u3321", "\u3321", "\u3321", "\u30B7\u30EA\u30F3\u30B0", "\u30B7\u30EA\u30F3\u30AF\u3099" },
            new string [] { "\u3322", "\u3322", "\u3322", "\u30BB\u30F3\u30C1", "\u30BB\u30F3\u30C1" },
            new string [] { "\u3323", "\u3323", "\u3323", "\u30BB\u30F3\u30C8", "\u30BB\u30F3\u30C8" },
            new string [] { "\u3324", "\u3324", "\u3324", "\u30C0\u30FC\u30B9", "\u30BF\u3099\u30FC\u30B9" },
            new string [] { "\u3325", "\u3325", "\u3325", "\u30C7\u30B7", "\u30C6\u3099\u30B7" },
            new string [] { "\u3326", "\u3326", "\u3326", "\u30C9\u30EB", "\u30C8\u3099\u30EB" },
            new string [] { "\u3327", "\u3327", "\u3327", "\u30C8\u30F3", "\u30C8\u30F3" },
            new string [] { "\u3328", "\u3328", "\u3328", "\u30CA\u30CE", "\u30CA\u30CE" },
            new string [] { "\u3329", "\u3329", "\u3329", "\u30CE\u30C3\u30C8", "\u30CE\u30C3\u30C8" },
            new string [] { "\u332A", "\u332A", "\u332A", "\u30CF\u30A4\u30C4", "\u30CF\u30A4\u30C4" },
            new string [] { "\u332B", "\u332B", "\u332B", "\u30D1\u30FC\u30BB\u30F3\u30C8", "\u30CF\u309A\u30FC\u30BB\u30F3\u30C8" },
            new string [] { "\u332C", "\u332C", "\u332C", "\u30D1\u30FC\u30C4", "\u30CF\u309A\u30FC\u30C4" },
            new string [] { "\u332D", "\u332D", "\u332D", "\u30D0\u30FC\u30EC\u30EB", "\u30CF\u3099\u30FC\u30EC\u30EB" },
            new string [] { "\u332E", "\u332E", "\u332E", "\u30D4\u30A2\u30B9\u30C8\u30EB", "\u30D2\u309A\u30A2\u30B9\u30C8\u30EB" },
            new string [] { "\u332F", "\u332F", "\u332F", "\u30D4\u30AF\u30EB", "\u30D2\u309A\u30AF\u30EB" },
            new string [] { "\u3330", "\u3330", "\u3330", "\u30D4\u30B3", "\u30D2\u309A\u30B3" },
            new string [] { "\u3331", "\u3331", "\u3331", "\u30D3\u30EB", "\u30D2\u3099\u30EB" },
            new string [] { "\u3332", "\u3332", "\u3332", "\u30D5\u30A1\u30E9\u30C3\u30C9", "\u30D5\u30A1\u30E9\u30C3\u30C8\u3099" },
            new string [] { "\u3333", "\u3333", "\u3333", "\u30D5\u30A3\u30FC\u30C8", "\u30D5\u30A3\u30FC\u30C8" },
            new string [] { "\u3334", "\u3334", "\u3334", "\u30D6\u30C3\u30B7\u30A7\u30EB", "\u30D5\u3099\u30C3\u30B7\u30A7\u30EB" },
            new string [] { "\u3335", "\u3335", "\u3335", "\u30D5\u30E9\u30F3", "\u30D5\u30E9\u30F3" },
            new string [] { "\u3336", "\u3336", "\u3336", "\u30D8\u30AF\u30BF\u30FC\u30EB", "\u30D8\u30AF\u30BF\u30FC\u30EB" },
            new string [] { "\u3337", "\u3337", "\u3337", "\u30DA\u30BD", "\u30D8\u309A\u30BD" },
            new string [] { "\u3338", "\u3338", "\u3338", "\u30DA\u30CB\u30D2", "\u30D8\u309A\u30CB\u30D2" },
            new string [] { "\u3339", "\u3339", "\u3339", "\u30D8\u30EB\u30C4", "\u30D8\u30EB\u30C4" },
            new string [] { "\u333A", "\u333A", "\u333A", "\u30DA\u30F3\u30B9", "\u30D8\u309A\u30F3\u30B9" },
            new string [] { "\u333B", "\u333B", "\u333B", "\u30DA\u30FC\u30B8", "\u30D8\u309A\u30FC\u30B7\u3099" },
            new string [] { "\u333C", "\u333C", "\u333C", "\u30D9\u30FC\u30BF", "\u30D8\u3099\u30FC\u30BF" },
            new string [] { "\u333D", "\u333D", "\u333D", "\u30DD\u30A4\u30F3\u30C8", "\u30DB\u309A\u30A4\u30F3\u30C8" },
            new string [] { "\u333E", "\u333E", "\u333E", "\u30DC\u30EB\u30C8", "\u30DB\u3099\u30EB\u30C8" },
            new string [] { "\u333F", "\u333F", "\u333F", "\u30DB\u30F3", "\u30DB\u30F3" },
            new string [] { "\u3340", "\u3340", "\u3340", "\u30DD\u30F3\u30C9", "\u30DB\u309A\u30F3\u30C8\u3099" },
            new string [] { "\u3341", "\u3341", "\u3341", "\u30DB\u30FC\u30EB", "\u30DB\u30FC\u30EB" },
            new string [] { "\u3342", "\u3342", "\u3342", "\u30DB\u30FC\u30F3", "\u30DB\u30FC\u30F3" },
            new string [] { "\u3343", "\u3343", "\u3343", "\u30DE\u30A4\u30AF\u30ED", "\u30DE\u30A4\u30AF\u30ED" },
            new string [] { "\u3344", "\u3344", "\u3344", "\u30DE\u30A4\u30EB", "\u30DE\u30A4\u30EB" },
            new string [] { "\u3345", "\u3345", "\u3345", "\u30DE\u30C3\u30CF", "\u30DE\u30C3\u30CF" },
            new string [] { "\u3346", "\u3346", "\u3346", "\u30DE\u30EB\u30AF", "\u30DE\u30EB\u30AF" },
            new string [] { "\u3347", "\u3347", "\u3347", "\u30DE\u30F3\u30B7\u30E7\u30F3", "\u30DE\u30F3\u30B7\u30E7\u30F3" },
            new string [] { "\u3348", "\u3348", "\u3348", "\u30DF\u30AF\u30ED\u30F3", "\u30DF\u30AF\u30ED\u30F3" },
            new string [] { "\u3349", "\u3349", "\u3349", "\u30DF\u30EA", "\u30DF\u30EA" },
            new string [] { "\u334A", "\u334A", "\u334A", "\u30DF\u30EA\u30D0\u30FC\u30EB", "\u30DF\u30EA\u30CF\u3099\u30FC\u30EB" },
            new string [] { "\u334B", "\u334B", "\u334B", "\u30E1\u30AC", "\u30E1\u30AB\u3099" },
            new string [] { "\u334C", "\u334C", "\u334C", "\u30E1\u30AC\u30C8\u30F3", "\u30E1\u30AB\u3099\u30C8\u30F3" },
            new string [] { "\u334D", "\u334D", "\u334D", "\u30E1\u30FC\u30C8\u30EB", "\u30E1\u30FC\u30C8\u30EB" },
            new string [] { "\u334E", "\u334E", "\u334E", "\u30E4\u30FC\u30C9", "\u30E4\u30FC\u30C8\u3099" },
            new string [] { "\u334F", "\u334F", "\u334F", "\u30E4\u30FC\u30EB", "\u30E4\u30FC\u30EB" },
            new string [] { "\u3350", "\u3350", "\u3350", "\u30E6\u30A2\u30F3", "\u30E6\u30A2\u30F3" },
            new string [] { "\u3351", "\u3351", "\u3351", "\u30EA\u30C3\u30C8\u30EB", "\u30EA\u30C3\u30C8\u30EB" },
            new string [] { "\u3352", "\u3352", "\u3352", "\u30EA\u30E9", "\u30EA\u30E9" },
            new string [] { "\u3353", "\u3353", "\u3353", "\u30EB\u30D4\u30FC", "\u30EB\u30D2\u309A\u30FC" },
            new string [] { "\u3354", "\u3354", "\u3354", "\u30EB\u30FC\u30D6\u30EB", "\u30EB\u30FC\u30D5\u3099\u30EB" },
            new string [] { "\u3355", "\u3355", "\u3355", "\u30EC\u30E0", "\u30EC\u30E0" },
            new string [] { "\u3356", "\u3356", "\u3356", "\u30EC\u30F3\u30C8\u30B2\u30F3", "\u30EC\u30F3\u30C8\u30B1\u3099\u30F3" },
            new string [] { "\u3357", "\u3357", "\u3357", "\u30EF\u30C3\u30C8", "\u30EF\u30C3\u30C8" },
            new string [] { "\u3358", "\u3358", "\u3358", "\u0030\u70B9", "\u0030\u70B9" },
            new string [] { "\u3359", "\u3359", "\u3359", "\u0031\u70B9", "\u0031\u70B9" },
            new string [] { "\u335A", "\u335A", "\u335A", "\u0032\u70B9", "\u0032\u70B9" },
            new string [] { "\u335B", "\u335B", "\u335B", "\u0033\u70B9", "\u0033\u70B9" },
            new string [] { "\u335C", "\u335C", "\u335C", "\u0034\u70B9", "\u0034\u70B9" },
            new string [] { "\u335D", "\u335D", "\u335D", "\u0035\u70B9", "\u0035\u70B9" },
            new string [] { "\u335E", "\u335E", "\u335E", "\u0036\u70B9", "\u0036\u70B9" },
            new string [] { "\u335F", "\u335F", "\u335F", "\u0037\u70B9", "\u0037\u70B9" },
            new string [] { "\u3360", "\u3360", "\u3360", "\u0038\u70B9", "\u0038\u70B9" },
            new string [] { "\u3361", "\u3361", "\u3361", "\u0039\u70B9", "\u0039\u70B9" },
            new string [] { "\u3362", "\u3362", "\u3362", "\u0031\u0030\u70B9", "\u0031\u0030\u70B9" },
            new string [] { "\u3363", "\u3363", "\u3363", "\u0031\u0031\u70B9", "\u0031\u0031\u70B9" },
            new string [] { "\u3364", "\u3364", "\u3364", "\u0031\u0032\u70B9", "\u0031\u0032\u70B9" },
            new string [] { "\u3365", "\u3365", "\u3365", "\u0031\u0033\u70B9", "\u0031\u0033\u70B9" },
            new string [] { "\u3366", "\u3366", "\u3366", "\u0031\u0034\u70B9", "\u0031\u0034\u70B9" },
            new string [] { "\u3367", "\u3367", "\u3367", "\u0031\u0035\u70B9", "\u0031\u0035\u70B9" },
            new string [] { "\u3368", "\u3368", "\u3368", "\u0031\u0036\u70B9", "\u0031\u0036\u70B9" },
            new string [] { "\u3369", "\u3369", "\u3369", "\u0031\u0037\u70B9", "\u0031\u0037\u70B9" },
            new string [] { "\u336A", "\u336A", "\u336A", "\u0031\u0038\u70B9", "\u0031\u0038\u70B9" },
            new string [] { "\u336B", "\u336B", "\u336B", "\u0031\u0039\u70B9", "\u0031\u0039\u70B9" },
            new string [] { "\u336C", "\u336C", "\u336C", "\u0032\u0030\u70B9", "\u0032\u0030\u70B9" },
            new string [] { "\u336D", "\u336D", "\u336D", "\u0032\u0031\u70B9", "\u0032\u0031\u70B9" },
            new string [] { "\u336E", "\u336E", "\u336E", "\u0032\u0032\u70B9", "\u0032\u0032\u70B9" },
            new string [] { "\u336F", "\u336F", "\u336F", "\u0032\u0033\u70B9", "\u0032\u0033\u70B9" },
            new string [] { "\u3370", "\u3370", "\u3370", "\u0032\u0034\u70B9", "\u0032\u0034\u70B9" },
            new string [] { "\u3371", "\u3371", "\u3371", "\u0068\u0050\u0061", "\u0068\u0050\u0061" },
            new string [] { "\u3372", "\u3372", "\u3372", "\u0064\u0061", "\u0064\u0061" },
            new string [] { "\u3373", "\u3373", "\u3373", "\u0041\u0055", "\u0041\u0055" },
            new string [] { "\u3374", "\u3374", "\u3374", "\u0062\u0061\u0072", "\u0062\u0061\u0072" },
            new string [] { "\u3375", "\u3375", "\u3375", "\u006F\u0056", "\u006F\u0056" },
            new string [] { "\u3376", "\u3376", "\u3376", "\u0070\u0063", "\u0070\u0063" },
            new string [] { "\u3377", "\u3377", "\u3377", "\u0064\u006D", "\u0064\u006D" },
            new string [] { "\u3378", "\u3378", "\u3378", "\u0064\u006D\u0032", "\u0064\u006D\u0032" },
            new string [] { "\u3379", "\u3379", "\u3379", "\u0064\u006D\u0033", "\u0064\u006D\u0033" },
            new string [] { "\u337A", "\u337A", "\u337A", "\u0049\u0055", "\u0049\u0055" },
            new string [] { "\u337B", "\u337B", "\u337B", "\u5E73\u6210", "\u5E73\u6210" },
            new string [] { "\u337C", "\u337C", "\u337C", "\u662D\u548C", "\u662D\u548C" },
            new string [] { "\u337D", "\u337D", "\u337D", "\u5927\u6B63", "\u5927\u6B63" },
            new string [] { "\u337E", "\u337E", "\u337E", "\u660E\u6CBB", "\u660E\u6CBB" },
            new string [] { "\u337F", "\u337F", "\u337F", "\u682A\u5F0F\u4F1A\u793E", "\u682A\u5F0F\u4F1A\u793E" },
            new string [] { "\u3380", "\u3380", "\u3380", "\u0070\u0041", "\u0070\u0041" },
            new string [] { "\u3381", "\u3381", "\u3381", "\u006E\u0041", "\u006E\u0041" },
            new string [] { "\u3382", "\u3382", "\u3382", "\u03BC\u0041", "\u03BC\u0041" },
            new string [] { "\u3383", "\u3383", "\u3383", "\u006D\u0041", "\u006D\u0041" },
            new string [] { "\u3384", "\u3384", "\u3384", "\u006B\u0041", "\u006B\u0041" },
            new string [] { "\u3385", "\u3385", "\u3385", "\u004B\u0042", "\u004B\u0042" },
            new string [] { "\u3386", "\u3386", "\u3386", "\u004D\u0042", "\u004D\u0042" },
            new string [] { "\u3387", "\u3387", "\u3387", "\u0047\u0042", "\u0047\u0042" },
            new string [] { "\u3388", "\u3388", "\u3388", "\u0063\u0061\u006C", "\u0063\u0061\u006C" },
            new string [] { "\u3389", "\u3389", "\u3389", "\u006B\u0063\u0061\u006C", "\u006B\u0063\u0061\u006C" },
            new string [] { "\u338A", "\u338A", "\u338A", "\u0070\u0046", "\u0070\u0046" },
            new string [] { "\u338B", "\u338B", "\u338B", "\u006E\u0046", "\u006E\u0046" },
            new string [] { "\u338C", "\u338C", "\u338C", "\u03BC\u0046", "\u03BC\u0046" },
            new string [] { "\u338D", "\u338D", "\u338D", "\u03BC\u0067", "\u03BC\u0067" },
            new string [] { "\u338E", "\u338E", "\u338E", "\u006D\u0067", "\u006D\u0067" },
            new string [] { "\u338F", "\u338F", "\u338F", "\u006B\u0067", "\u006B\u0067" },
            new string [] { "\u3390", "\u3390", "\u3390", "\u0048\u007A", "\u0048\u007A" },
            new string [] { "\u3391", "\u3391", "\u3391", "\u006B\u0048\u007A", "\u006B\u0048\u007A" },
            new string [] { "\u3392", "\u3392", "\u3392", "\u004D\u0048\u007A", "\u004D\u0048\u007A" },
            new string [] { "\u3393", "\u3393", "\u3393", "\u0047\u0048\u007A", "\u0047\u0048\u007A" },
            new string [] { "\u3394", "\u3394", "\u3394", "\u0054\u0048\u007A", "\u0054\u0048\u007A" },
            new string [] { "\u3395", "\u3395", "\u3395", "\u03BC\u006C", "\u03BC\u006C" },
            new string [] { "\u3396", "\u3396", "\u3396", "\u006D\u006C", "\u006D\u006C" },
            new string [] { "\u3397", "\u3397", "\u3397", "\u0064\u006C", "\u0064\u006C" },
            new string [] { "\u3398", "\u3398", "\u3398", "\u006B\u006C", "\u006B\u006C" },
            new string [] { "\u3399", "\u3399", "\u3399", "\u0066\u006D", "\u0066\u006D" },
            new string [] { "\u339A", "\u339A", "\u339A", "\u006E\u006D", "\u006E\u006D" },
            new string [] { "\u339B", "\u339B", "\u339B", "\u03BC\u006D", "\u03BC\u006D" },
            new string [] { "\u339C", "\u339C", "\u339C", "\u006D\u006D", "\u006D\u006D" },
            new string [] { "\u339D", "\u339D", "\u339D", "\u0063\u006D", "\u0063\u006D" },
            new string [] { "\u339E", "\u339E", "\u339E", "\u006B\u006D", "\u006B\u006D" },
            new string [] { "\u339F", "\u339F", "\u339F", "\u006D\u006D\u0032", "\u006D\u006D\u0032" },
            new string [] { "\u33A0", "\u33A0", "\u33A0", "\u0063\u006D\u0032", "\u0063\u006D\u0032" },
            new string [] { "\u33A1", "\u33A1", "\u33A1", "\u006D\u0032", "\u006D\u0032" },
            new string [] { "\u33A2", "\u33A2", "\u33A2", "\u006B\u006D\u0032", "\u006B\u006D\u0032" },
            new string [] { "\u33A3", "\u33A3", "\u33A3", "\u006D\u006D\u0033", "\u006D\u006D\u0033" },
            new string [] { "\u33A4", "\u33A4", "\u33A4", "\u0063\u006D\u0033", "\u0063\u006D\u0033" },
            new string [] { "\u33A5", "\u33A5", "\u33A5", "\u006D\u0033", "\u006D\u0033" },
            new string [] { "\u33A6", "\u33A6", "\u33A6", "\u006B\u006D\u0033", "\u006B\u006D\u0033" },
            new string [] { "\u33A7", "\u33A7", "\u33A7", "\u006D\u2215\u0073", "\u006D\u2215\u0073" },
            new string [] { "\u33A8", "\u33A8", "\u33A8", "\u006D\u2215\u0073\u0032", "\u006D\u2215\u0073\u0032" },
            new string [] { "\u33A9", "\u33A9", "\u33A9", "\u0050\u0061", "\u0050\u0061" },
            new string [] { "\u33AA", "\u33AA", "\u33AA", "\u006B\u0050\u0061", "\u006B\u0050\u0061" },
            new string [] { "\u33AB", "\u33AB", "\u33AB", "\u004D\u0050\u0061", "\u004D\u0050\u0061" },
            new string [] { "\u33AC", "\u33AC", "\u33AC", "\u0047\u0050\u0061", "\u0047\u0050\u0061" },
            new string [] { "\u33AD", "\u33AD", "\u33AD", "\u0072\u0061\u0064", "\u0072\u0061\u0064" },
            new string [] { "\u33AE", "\u33AE", "\u33AE", "\u0072\u0061\u0064\u2215\u0073", "\u0072\u0061\u0064\u2215\u0073" },
            new string [] { "\u33AF", "\u33AF", "\u33AF", "\u0072\u0061\u0064\u2215\u0073\u0032", "\u0072\u0061\u0064\u2215\u0073\u0032" },
            new string [] { "\u33B0", "\u33B0", "\u33B0", "\u0070\u0073", "\u0070\u0073" },
            new string [] { "\u33B1", "\u33B1", "\u33B1", "\u006E\u0073", "\u006E\u0073" },
            new string [] { "\u33B2", "\u33B2", "\u33B2", "\u03BC\u0073", "\u03BC\u0073" },
            new string [] { "\u33B3", "\u33B3", "\u33B3", "\u006D\u0073", "\u006D\u0073" },
            new string [] { "\u33B4", "\u33B4", "\u33B4", "\u0070\u0056", "\u0070\u0056" },
            new string [] { "\u33B5", "\u33B5", "\u33B5", "\u006E\u0056", "\u006E\u0056" },
            new string [] { "\u33B6", "\u33B6", "\u33B6", "\u03BC\u0056", "\u03BC\u0056" },
            new string [] { "\u33B7", "\u33B7", "\u33B7", "\u006D\u0056", "\u006D\u0056" },
            new string [] { "\u33B8", "\u33B8", "\u33B8", "\u006B\u0056", "\u006B\u0056" },
            new string [] { "\u33B9", "\u33B9", "\u33B9", "\u004D\u0056", "\u004D\u0056" },
            new string [] { "\u33BA", "\u33BA", "\u33BA", "\u0070\u0057", "\u0070\u0057" },
            new string [] { "\u33BB", "\u33BB", "\u33BB", "\u006E\u0057", "\u006E\u0057" },
            new string [] { "\u33BC", "\u33BC", "\u33BC", "\u03BC\u0057", "\u03BC\u0057" },
            new string [] { "\u33BD", "\u33BD", "\u33BD", "\u006D\u0057", "\u006D\u0057" },
            new string [] { "\u33BE", "\u33BE", "\u33BE", "\u006B\u0057", "\u006B\u0057" },
            new string [] { "\u33BF", "\u33BF", "\u33BF", "\u004D\u0057", "\u004D\u0057" },
            new string [] { "\u33C0", "\u33C0", "\u33C0", "\u006B\u03A9", "\u006B\u03A9" },
            new string [] { "\u33C1", "\u33C1", "\u33C1", "\u004D\u03A9", "\u004D\u03A9" },
            new string [] { "\u33C2", "\u33C2", "\u33C2", "\u0061\u002E\u006D\u002E", "\u0061\u002E\u006D\u002E" },
            new string [] { "\u33C3", "\u33C3", "\u33C3", "\u0042\u0071", "\u0042\u0071" },
            new string [] { "\u33C4", "\u33C4", "\u33C4", "\u0063\u0063", "\u0063\u0063" },
            new string [] { "\u33C5", "\u33C5", "\u33C5", "\u0063\u0064", "\u0063\u0064" },
            new string [] { "\u33C6", "\u33C6", "\u33C6", "\u0043\u2215\u006B\u0067", "\u0043\u2215\u006B\u0067" },
            new string [] { "\u33C7", "\u33C7", "\u33C7", "\u0043\u006F\u002E", "\u0043\u006F\u002E" },
            new string [] { "\u33C8", "\u33C8", "\u33C8", "\u0064\u0042", "\u0064\u0042" },
            new string [] { "\u33C9", "\u33C9", "\u33C9", "\u0047\u0079", "\u0047\u0079" },
            new string [] { "\u33CA", "\u33CA", "\u33CA", "\u0068\u0061", "\u0068\u0061" },
            new string [] { "\u33CB", "\u33CB", "\u33CB", "\u0048\u0050", "\u0048\u0050" },
            new string [] { "\u33CC", "\u33CC", "\u33CC", "\u0069\u006E", "\u0069\u006E" },
            new string [] { "\u33CD", "\u33CD", "\u33CD", "\u004B\u004B", "\u004B\u004B" },
            new string [] { "\u33CE", "\u33CE", "\u33CE", "\u004B\u004D", "\u004B\u004D" },
            new string [] { "\u33CF", "\u33CF", "\u33CF", "\u006B\u0074", "\u006B\u0074" },
            new string [] { "\u33D0", "\u33D0", "\u33D0", "\u006C\u006D", "\u006C\u006D" },
            new string [] { "\u33D1", "\u33D1", "\u33D1", "\u006C\u006E", "\u006C\u006E" },
            new string [] { "\u33D2", "\u33D2", "\u33D2", "\u006C\u006F\u0067", "\u006C\u006F\u0067" },
            new string [] { "\u33D3", "\u33D3", "\u33D3", "\u006C\u0078", "\u006C\u0078" },
            new string [] { "\u33D4", "\u33D4", "\u33D4", "\u006D\u0062", "\u006D\u0062" },
            new string [] { "\u33D5", "\u33D5", "\u33D5", "\u006D\u0069\u006C", "\u006D\u0069\u006C" },
            new string [] { "\u33D6", "\u33D6", "\u33D6", "\u006D\u006F\u006C", "\u006D\u006F\u006C" },
            new string [] { "\u33D7", "\u33D7", "\u33D7", "\u0050\u0048", "\u0050\u0048" },
            new string [] { "\u33D8", "\u33D8", "\u33D8", "\u0070\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E" },
            new string [] { "\u33D9", "\u33D9", "\u33D9", "\u0050\u0050\u004D", "\u0050\u0050\u004D" },
            new string [] { "\u33DA", "\u33DA", "\u33DA", "\u0050\u0052", "\u0050\u0052" },
            new string [] { "\u33DB", "\u33DB", "\u33DB", "\u0073\u0072", "\u0073\u0072" },
            new string [] { "\u33DC", "\u33DC", "\u33DC", "\u0053\u0076", "\u0053\u0076" },
            new string [] { "\u33DD", "\u33DD", "\u33DD", "\u0057\u0062", "\u0057\u0062" },
            new string [] { "\u33DE", "\u33DE", "\u33DE", "\u0056\u2215\u006D", "\u0056\u2215\u006D" },
            new string [] { "\u33DF", "\u33DF", "\u33DF", "\u0041\u2215\u006D", "\u0041\u2215\u006D" },
            new string [] { "\u33E0", "\u33E0", "\u33E0", "\u0031\u65E5", "\u0031\u65E5" },
            new string [] { "\u33E1", "\u33E1", "\u33E1", "\u0032\u65E5", "\u0032\u65E5" },
            new string [] { "\u33E2", "\u33E2", "\u33E2", "\u0033\u65E5", "\u0033\u65E5" },
            new string [] { "\u33E3", "\u33E3", "\u33E3", "\u0034\u65E5", "\u0034\u65E5" },
            new string [] { "\u33E4", "\u33E4", "\u33E4", "\u0035\u65E5", "\u0035\u65E5" },
            new string [] { "\u33E5", "\u33E5", "\u33E5", "\u0036\u65E5", "\u0036\u65E5" },
            new string [] { "\u33E6", "\u33E6", "\u33E6", "\u0037\u65E5", "\u0037\u65E5" },
            new string [] { "\u33E7", "\u33E7", "\u33E7", "\u0038\u65E5", "\u0038\u65E5" },
            new string [] { "\u33E8", "\u33E8", "\u33E8", "\u0039\u65E5", "\u0039\u65E5" },
            new string [] { "\u33E9", "\u33E9", "\u33E9", "\u0031\u0030\u65E5", "\u0031\u0030\u65E5" },
            new string [] { "\u33EA", "\u33EA", "\u33EA", "\u0031\u0031\u65E5", "\u0031\u0031\u65E5" },
            new string [] { "\u33EB", "\u33EB", "\u33EB", "\u0031\u0032\u65E5", "\u0031\u0032\u65E5" },
            new string [] { "\u33EC", "\u33EC", "\u33EC", "\u0031\u0033\u65E5", "\u0031\u0033\u65E5" },
            new string [] { "\u33ED", "\u33ED", "\u33ED", "\u0031\u0034\u65E5", "\u0031\u0034\u65E5" },
            new string [] { "\u33EE", "\u33EE", "\u33EE", "\u0031\u0035\u65E5", "\u0031\u0035\u65E5" },
            new string [] { "\u33EF", "\u33EF", "\u33EF", "\u0031\u0036\u65E5", "\u0031\u0036\u65E5" },
            new string [] { "\u33F0", "\u33F0", "\u33F0", "\u0031\u0037\u65E5", "\u0031\u0037\u65E5" },
            new string [] { "\u33F1", "\u33F1", "\u33F1", "\u0031\u0038\u65E5", "\u0031\u0038\u65E5" },
            new string [] { "\u33F2", "\u33F2", "\u33F2", "\u0031\u0039\u65E5", "\u0031\u0039\u65E5" },
            new string [] { "\u33F3", "\u33F3", "\u33F3", "\u0032\u0030\u65E5", "\u0032\u0030\u65E5" },
            new string [] { "\u33F4", "\u33F4", "\u33F4", "\u0032\u0031\u65E5", "\u0032\u0031\u65E5" },
            new string [] { "\u33F5", "\u33F5", "\u33F5", "\u0032\u0032\u65E5", "\u0032\u0032\u65E5" },
            new string [] { "\u33F6", "\u33F6", "\u33F6", "\u0032\u0033\u65E5", "\u0032\u0033\u65E5" },
            new string [] { "\u33F7", "\u33F7", "\u33F7", "\u0032\u0034\u65E5", "\u0032\u0034\u65E5" },
            new string [] { "\u33F8", "\u33F8", "\u33F8", "\u0032\u0035\u65E5", "\u0032\u0035\u65E5" },
            new string [] { "\u33F9", "\u33F9", "\u33F9", "\u0032\u0036\u65E5", "\u0032\u0036\u65E5" },
            new string [] { "\u33FA", "\u33FA", "\u33FA", "\u0032\u0037\u65E5", "\u0032\u0037\u65E5" },
            new string [] { "\u33FB", "\u33FB", "\u33FB", "\u0032\u0038\u65E5", "\u0032\u0038\u65E5" },
            new string [] { "\u33FC", "\u33FC", "\u33FC", "\u0032\u0039\u65E5", "\u0032\u0039\u65E5" },
            new string [] { "\u33FD", "\u33FD", "\u33FD", "\u0033\u0030\u65E5", "\u0033\u0030\u65E5" },
            new string [] { "\u33FE", "\u33FE", "\u33FE", "\u0033\u0031\u65E5", "\u0033\u0031\u65E5" },
            new string [] { "\u33FF", "\u33FF", "\u33FF", "\u0067\u0061\u006C", "\u0067\u0061\u006C" },
            new string [] { "\uA770", "\uA770", "\uA770", "\uA76F", "\uA76F" },
            new string [] { "\uAC00", "\uAC00", "\u1100\u1161", "\uAC00", "\u1100\u1161" },
            new string [] { "\uAC01", "\uAC01", "\u1100\u1161\u11A8", "\uAC01", "\u1100\u1161\u11A8" },
            new string [] { "\uAC02", "\uAC02", "\u1100\u1161\u11A9", "\uAC02", "\u1100\u1161\u11A9" },
            new string [] { "\uAC03", "\uAC03", "\u1100\u1161\u11AA", "\uAC03", "\u1100\u1161\u11AA" },
            new string [] { "\uAC04", "\uAC04", "\u1100\u1161\u11AB", "\uAC04", "\u1100\u1161\u11AB" },
            new string [] { "\uAC05", "\uAC05", "\u1100\u1161\u11AC", "\uAC05", "\u1100\u1161\u11AC" },
            new string [] { "\uAC06", "\uAC06", "\u1100\u1161\u11AD", "\uAC06", "\u1100\u1161\u11AD" },
            new string [] { "\uAC07", "\uAC07", "\u1100\u1161\u11AE", "\uAC07", "\u1100\u1161\u11AE" },
            new string [] { "\uAC08", "\uAC08", "\u1100\u1161\u11AF", "\uAC08", "\u1100\u1161\u11AF" },
            new string [] { "\uAC09", "\uAC09", "\u1100\u1161\u11B0", "\uAC09", "\u1100\u1161\u11B0" },
            new string [] { "\uAC0A", "\uAC0A", "\u1100\u1161\u11B1", "\uAC0A", "\u1100\u1161\u11B1" },
            new string [] { "\uAC0B", "\uAC0B", "\u1100\u1161\u11B2", "\uAC0B", "\u1100\u1161\u11B2" },
            new string [] { "\uAC0C", "\uAC0C", "\u1100\u1161\u11B3", "\uAC0C", "\u1100\u1161\u11B3" },
            new string [] { "\uAC0D", "\uAC0D", "\u1100\u1161\u11B4", "\uAC0D", "\u1100\u1161\u11B4" },
            new string [] { "\uAC0E", "\uAC0E", "\u1100\u1161\u11B5", "\uAC0E", "\u1100\u1161\u11B5" },
            new string [] { "\uAC0F", "\uAC0F", "\u1100\u1161\u11B6", "\uAC0F", "\u1100\u1161\u11B6" },
            new string [] { "\uAC10", "\uAC10", "\u1100\u1161\u11B7", "\uAC10", "\u1100\u1161\u11B7" },
            new string [] { "\uAC11", "\uAC11", "\u1100\u1161\u11B8", "\uAC11", "\u1100\u1161\u11B8" },
            new string [] { "\uAC12", "\uAC12", "\u1100\u1161\u11B9", "\uAC12", "\u1100\u1161\u11B9" },
            new string [] { "\uAC13", "\uAC13", "\u1100\u1161\u11BA", "\uAC13", "\u1100\u1161\u11BA" },
            new string [] { "\uAC14", "\uAC14", "\u1100\u1161\u11BB", "\uAC14", "\u1100\u1161\u11BB" },
            new string [] { "\uAC15", "\uAC15", "\u1100\u1161\u11BC", "\uAC15", "\u1100\u1161\u11BC" },
            new string [] { "\uAC16", "\uAC16", "\u1100\u1161\u11BD", "\uAC16", "\u1100\u1161\u11BD" },
            new string [] { "\uAC17", "\uAC17", "\u1100\u1161\u11BE", "\uAC17", "\u1100\u1161\u11BE" },
            new string [] { "\uAC18", "\uAC18", "\u1100\u1161\u11BF", "\uAC18", "\u1100\u1161\u11BF" },
            new string [] { "\uAC19", "\uAC19", "\u1100\u1161\u11C0", "\uAC19", "\u1100\u1161\u11C0" },
            new string [] { "\uAC1A", "\uAC1A", "\u1100\u1161\u11C1", "\uAC1A", "\u1100\u1161\u11C1" },
            new string [] { "\uAC1B", "\uAC1B", "\u1100\u1161\u11C2", "\uAC1B", "\u1100\u1161\u11C2" },
            new string [] { "\uAC1C", "\uAC1C", "\u1100\u1162", "\uAC1C", "\u1100\u1162" },
            new string [] { "\uAC1D", "\uAC1D", "\u1100\u1162\u11A8", "\uAC1D", "\u1100\u1162\u11A8" },
            new string [] { "\uAC1E", "\uAC1E", "\u1100\u1162\u11A9", "\uAC1E", "\u1100\u1162\u11A9" },
            new string [] { "\uAC1F", "\uAC1F", "\u1100\u1162\u11AA", "\uAC1F", "\u1100\u1162\u11AA" },
            new string [] { "\uAC20", "\uAC20", "\u1100\u1162\u11AB", "\uAC20", "\u1100\u1162\u11AB" },
            new string [] { "\uAC21", "\uAC21", "\u1100\u1162\u11AC", "\uAC21", "\u1100\u1162\u11AC" },
            new string [] { "\uAC22", "\uAC22", "\u1100\u1162\u11AD", "\uAC22", "\u1100\u1162\u11AD" },
            new string [] { "\uAC23", "\uAC23", "\u1100\u1162\u11AE", "\uAC23", "\u1100\u1162\u11AE" },
            new string [] { "\uAC24", "\uAC24", "\u1100\u1162\u11AF", "\uAC24", "\u1100\u1162\u11AF" },
            new string [] { "\uAC25", "\uAC25", "\u1100\u1162\u11B0", "\uAC25", "\u1100\u1162\u11B0" },
            new string [] { "\uAC26", "\uAC26", "\u1100\u1162\u11B1", "\uAC26", "\u1100\u1162\u11B1" },
            new string [] { "\uAC27", "\uAC27", "\u1100\u1162\u11B2", "\uAC27", "\u1100\u1162\u11B2" },
            new string [] { "\uAC28", "\uAC28", "\u1100\u1162\u11B3", "\uAC28", "\u1100\u1162\u11B3" },
            new string [] { "\uAC29", "\uAC29", "\u1100\u1162\u11B4", "\uAC29", "\u1100\u1162\u11B4" },
            new string [] { "\uAC2A", "\uAC2A", "\u1100\u1162\u11B5", "\uAC2A", "\u1100\u1162\u11B5" },
            new string [] { "\uAC2B", "\uAC2B", "\u1100\u1162\u11B6", "\uAC2B", "\u1100\u1162\u11B6" },
            new string [] { "\uAC2C", "\uAC2C", "\u1100\u1162\u11B7", "\uAC2C", "\u1100\u1162\u11B7" },
            new string [] { "\uAC2D", "\uAC2D", "\u1100\u1162\u11B8", "\uAC2D", "\u1100\u1162\u11B8" },
            new string [] { "\uAC2E", "\uAC2E", "\u1100\u1162\u11B9", "\uAC2E", "\u1100\u1162\u11B9" },
            new string [] { "\uAC2F", "\uAC2F", "\u1100\u1162\u11BA", "\uAC2F", "\u1100\u1162\u11BA" },
            new string [] { "\uAC30", "\uAC30", "\u1100\u1162\u11BB", "\uAC30", "\u1100\u1162\u11BB" },
            new string [] { "\uAC31", "\uAC31", "\u1100\u1162\u11BC", "\uAC31", "\u1100\u1162\u11BC" },
            new string [] { "\uAC32", "\uAC32", "\u1100\u1162\u11BD", "\uAC32", "\u1100\u1162\u11BD" },
            new string [] { "\uAC33", "\uAC33", "\u1100\u1162\u11BE", "\uAC33", "\u1100\u1162\u11BE" },
            new string [] { "\uAC34", "\uAC34", "\u1100\u1162\u11BF", "\uAC34", "\u1100\u1162\u11BF" },
            new string [] { "\uAC35", "\uAC35", "\u1100\u1162\u11C0", "\uAC35", "\u1100\u1162\u11C0" },
            new string [] { "\uAC36", "\uAC36", "\u1100\u1162\u11C1", "\uAC36", "\u1100\u1162\u11C1" },
            new string [] { "\uAC37", "\uAC37", "\u1100\u1162\u11C2", "\uAC37", "\u1100\u1162\u11C2" },
            new string [] { "\uAC38", "\uAC38", "\u1100\u1163", "\uAC38", "\u1100\u1163" },
            new string [] { "\uAC39", "\uAC39", "\u1100\u1163\u11A8", "\uAC39", "\u1100\u1163\u11A8" },
            new string [] { "\uAC3A", "\uAC3A", "\u1100\u1163\u11A9", "\uAC3A", "\u1100\u1163\u11A9" },
            new string [] { "\uAC3B", "\uAC3B", "\u1100\u1163\u11AA", "\uAC3B", "\u1100\u1163\u11AA" },
            new string [] { "\uAC3C", "\uAC3C", "\u1100\u1163\u11AB", "\uAC3C", "\u1100\u1163\u11AB" },
            new string [] { "\uAC3D", "\uAC3D", "\u1100\u1163\u11AC", "\uAC3D", "\u1100\u1163\u11AC" },
            new string [] { "\uAC3E", "\uAC3E", "\u1100\u1163\u11AD", "\uAC3E", "\u1100\u1163\u11AD" },
            new string [] { "\uAC3F", "\uAC3F", "\u1100\u1163\u11AE", "\uAC3F", "\u1100\u1163\u11AE" },
            new string [] { "\uAC40", "\uAC40", "\u1100\u1163\u11AF", "\uAC40", "\u1100\u1163\u11AF" },
            new string [] { "\uAC41", "\uAC41", "\u1100\u1163\u11B0", "\uAC41", "\u1100\u1163\u11B0" },
            new string [] { "\uAC42", "\uAC42", "\u1100\u1163\u11B1", "\uAC42", "\u1100\u1163\u11B1" },
            new string [] { "\uAC43", "\uAC43", "\u1100\u1163\u11B2", "\uAC43", "\u1100\u1163\u11B2" },
            new string [] { "\uAC44", "\uAC44", "\u1100\u1163\u11B3", "\uAC44", "\u1100\u1163\u11B3" },
            new string [] { "\uAC45", "\uAC45", "\u1100\u1163\u11B4", "\uAC45", "\u1100\u1163\u11B4" },
            new string [] { "\uAC46", "\uAC46", "\u1100\u1163\u11B5", "\uAC46", "\u1100\u1163\u11B5" },
            new string [] { "\uAC47", "\uAC47", "\u1100\u1163\u11B6", "\uAC47", "\u1100\u1163\u11B6" },
            new string [] { "\uAC48", "\uAC48", "\u1100\u1163\u11B7", "\uAC48", "\u1100\u1163\u11B7" },
            new string [] { "\uAC49", "\uAC49", "\u1100\u1163\u11B8", "\uAC49", "\u1100\u1163\u11B8" },
            new string [] { "\uAC4A", "\uAC4A", "\u1100\u1163\u11B9", "\uAC4A", "\u1100\u1163\u11B9" },
            new string [] { "\uAC4B", "\uAC4B", "\u1100\u1163\u11BA", "\uAC4B", "\u1100\u1163\u11BA" },
            new string [] { "\uAC4C", "\uAC4C", "\u1100\u1163\u11BB", "\uAC4C", "\u1100\u1163\u11BB" },
            new string [] { "\uAC4D", "\uAC4D", "\u1100\u1163\u11BC", "\uAC4D", "\u1100\u1163\u11BC" },
            new string [] { "\uAC4E", "\uAC4E", "\u1100\u1163\u11BD", "\uAC4E", "\u1100\u1163\u11BD" },
            new string [] { "\uAC4F", "\uAC4F", "\u1100\u1163\u11BE", "\uAC4F", "\u1100\u1163\u11BE" },
            new string [] { "\uAC50", "\uAC50", "\u1100\u1163\u11BF", "\uAC50", "\u1100\u1163\u11BF" },
            new string [] { "\uAC51", "\uAC51", "\u1100\u1163\u11C0", "\uAC51", "\u1100\u1163\u11C0" },
            new string [] { "\uAC52", "\uAC52", "\u1100\u1163\u11C1", "\uAC52", "\u1100\u1163\u11C1" },
            new string [] { "\uAC53", "\uAC53", "\u1100\u1163\u11C2", "\uAC53", "\u1100\u1163\u11C2" },
            new string [] { "\uAC54", "\uAC54", "\u1100\u1164", "\uAC54", "\u1100\u1164" },
            new string [] { "\uAC55", "\uAC55", "\u1100\u1164\u11A8", "\uAC55", "\u1100\u1164\u11A8" },
            new string [] { "\uAC56", "\uAC56", "\u1100\u1164\u11A9", "\uAC56", "\u1100\u1164\u11A9" },
            new string [] { "\uAC57", "\uAC57", "\u1100\u1164\u11AA", "\uAC57", "\u1100\u1164\u11AA" },
            new string [] { "\uAC58", "\uAC58", "\u1100\u1164\u11AB", "\uAC58", "\u1100\u1164\u11AB" },
            new string [] { "\uAC59", "\uAC59", "\u1100\u1164\u11AC", "\uAC59", "\u1100\u1164\u11AC" },
            new string [] { "\uAC5A", "\uAC5A", "\u1100\u1164\u11AD", "\uAC5A", "\u1100\u1164\u11AD" },
            new string [] { "\uAC5B", "\uAC5B", "\u1100\u1164\u11AE", "\uAC5B", "\u1100\u1164\u11AE" },
            new string [] { "\uAC5C", "\uAC5C", "\u1100\u1164\u11AF", "\uAC5C", "\u1100\u1164\u11AF" },
            new string [] { "\uAC5D", "\uAC5D", "\u1100\u1164\u11B0", "\uAC5D", "\u1100\u1164\u11B0" },
            new string [] { "\uAC5E", "\uAC5E", "\u1100\u1164\u11B1", "\uAC5E", "\u1100\u1164\u11B1" },
            new string [] { "\uAC5F", "\uAC5F", "\u1100\u1164\u11B2", "\uAC5F", "\u1100\u1164\u11B2" },
            new string [] { "\uAC60", "\uAC60", "\u1100\u1164\u11B3", "\uAC60", "\u1100\u1164\u11B3" },
            new string [] { "\uAC61", "\uAC61", "\u1100\u1164\u11B4", "\uAC61", "\u1100\u1164\u11B4" },
            new string [] { "\uAC62", "\uAC62", "\u1100\u1164\u11B5", "\uAC62", "\u1100\u1164\u11B5" },
            new string [] { "\uAC63", "\uAC63", "\u1100\u1164\u11B6", "\uAC63", "\u1100\u1164\u11B6" },
            new string [] { "\uAC64", "\uAC64", "\u1100\u1164\u11B7", "\uAC64", "\u1100\u1164\u11B7" },
            new string [] { "\uAC65", "\uAC65", "\u1100\u1164\u11B8", "\uAC65", "\u1100\u1164\u11B8" },
            new string [] { "\uAC66", "\uAC66", "\u1100\u1164\u11B9", "\uAC66", "\u1100\u1164\u11B9" },
            new string [] { "\uAC67", "\uAC67", "\u1100\u1164\u11BA", "\uAC67", "\u1100\u1164\u11BA" },
            new string [] { "\uAC68", "\uAC68", "\u1100\u1164\u11BB", "\uAC68", "\u1100\u1164\u11BB" },
            new string [] { "\uAC69", "\uAC69", "\u1100\u1164\u11BC", "\uAC69", "\u1100\u1164\u11BC" },
            new string [] { "\uAC6A", "\uAC6A", "\u1100\u1164\u11BD", "\uAC6A", "\u1100\u1164\u11BD" },
            new string [] { "\uAC6B", "\uAC6B", "\u1100\u1164\u11BE", "\uAC6B", "\u1100\u1164\u11BE" },
            new string [] { "\uAC6C", "\uAC6C", "\u1100\u1164\u11BF", "\uAC6C", "\u1100\u1164\u11BF" },
            new string [] { "\uAC6D", "\uAC6D", "\u1100\u1164\u11C0", "\uAC6D", "\u1100\u1164\u11C0" },
            new string [] { "\uAC6E", "\uAC6E", "\u1100\u1164\u11C1", "\uAC6E", "\u1100\u1164\u11C1" },
            new string [] { "\uAC6F", "\uAC6F", "\u1100\u1164\u11C2", "\uAC6F", "\u1100\u1164\u11C2" },
            new string [] { "\uAC70", "\uAC70", "\u1100\u1165", "\uAC70", "\u1100\u1165" },
            new string [] { "\uAC71", "\uAC71", "\u1100\u1165\u11A8", "\uAC71", "\u1100\u1165\u11A8" },
            new string [] { "\uAC72", "\uAC72", "\u1100\u1165\u11A9", "\uAC72", "\u1100\u1165\u11A9" },
            new string [] { "\uAC73", "\uAC73", "\u1100\u1165\u11AA", "\uAC73", "\u1100\u1165\u11AA" },
            new string [] { "\uAC74", "\uAC74", "\u1100\u1165\u11AB", "\uAC74", "\u1100\u1165\u11AB" },
            new string [] { "\uAC75", "\uAC75", "\u1100\u1165\u11AC", "\uAC75", "\u1100\u1165\u11AC" },
            new string [] { "\uAC76", "\uAC76", "\u1100\u1165\u11AD", "\uAC76", "\u1100\u1165\u11AD" },
            new string [] { "\uAC77", "\uAC77", "\u1100\u1165\u11AE", "\uAC77", "\u1100\u1165\u11AE" },
            new string [] { "\uAC78", "\uAC78", "\u1100\u1165\u11AF", "\uAC78", "\u1100\u1165\u11AF" },
            new string [] { "\uAC79", "\uAC79", "\u1100\u1165\u11B0", "\uAC79", "\u1100\u1165\u11B0" },
            new string [] { "\uAC7A", "\uAC7A", "\u1100\u1165\u11B1", "\uAC7A", "\u1100\u1165\u11B1" },
            new string [] { "\uAC7B", "\uAC7B", "\u1100\u1165\u11B2", "\uAC7B", "\u1100\u1165\u11B2" },
            new string [] { "\uAC7C", "\uAC7C", "\u1100\u1165\u11B3", "\uAC7C", "\u1100\u1165\u11B3" },
            new string [] { "\uAC7D", "\uAC7D", "\u1100\u1165\u11B4", "\uAC7D", "\u1100\u1165\u11B4" },
            new string [] { "\uAC7E", "\uAC7E", "\u1100\u1165\u11B5", "\uAC7E", "\u1100\u1165\u11B5" },
            new string [] { "\uAC7F", "\uAC7F", "\u1100\u1165\u11B6", "\uAC7F", "\u1100\u1165\u11B6" },
            new string [] { "\uAC80", "\uAC80", "\u1100\u1165\u11B7", "\uAC80", "\u1100\u1165\u11B7" },
            new string [] { "\uAC81", "\uAC81", "\u1100\u1165\u11B8", "\uAC81", "\u1100\u1165\u11B8" },
            new string [] { "\uAC82", "\uAC82", "\u1100\u1165\u11B9", "\uAC82", "\u1100\u1165\u11B9" },
            new string [] { "\uAC83", "\uAC83", "\u1100\u1165\u11BA", "\uAC83", "\u1100\u1165\u11BA" },
            new string [] { "\uAC84", "\uAC84", "\u1100\u1165\u11BB", "\uAC84", "\u1100\u1165\u11BB" },
            new string [] { "\uAC85", "\uAC85", "\u1100\u1165\u11BC", "\uAC85", "\u1100\u1165\u11BC" },
            new string [] { "\uAC86", "\uAC86", "\u1100\u1165\u11BD", "\uAC86", "\u1100\u1165\u11BD" },
            new string [] { "\uAC87", "\uAC87", "\u1100\u1165\u11BE", "\uAC87", "\u1100\u1165\u11BE" },
            new string [] { "\uAC88", "\uAC88", "\u1100\u1165\u11BF", "\uAC88", "\u1100\u1165\u11BF" },
            new string [] { "\uAC89", "\uAC89", "\u1100\u1165\u11C0", "\uAC89", "\u1100\u1165\u11C0" },
            new string [] { "\uAC8A", "\uAC8A", "\u1100\u1165\u11C1", "\uAC8A", "\u1100\u1165\u11C1" },
            new string [] { "\uAC8B", "\uAC8B", "\u1100\u1165\u11C2", "\uAC8B", "\u1100\u1165\u11C2" },
            new string [] { "\uAC8C", "\uAC8C", "\u1100\u1166", "\uAC8C", "\u1100\u1166" },
            new string [] { "\uAC8D", "\uAC8D", "\u1100\u1166\u11A8", "\uAC8D", "\u1100\u1166\u11A8" },
            new string [] { "\uAC8E", "\uAC8E", "\u1100\u1166\u11A9", "\uAC8E", "\u1100\u1166\u11A9" },
            new string [] { "\uAC8F", "\uAC8F", "\u1100\u1166\u11AA", "\uAC8F", "\u1100\u1166\u11AA" },
            new string [] { "\uAC90", "\uAC90", "\u1100\u1166\u11AB", "\uAC90", "\u1100\u1166\u11AB" },
            new string [] { "\uAC91", "\uAC91", "\u1100\u1166\u11AC", "\uAC91", "\u1100\u1166\u11AC" },
            new string [] { "\uAC92", "\uAC92", "\u1100\u1166\u11AD", "\uAC92", "\u1100\u1166\u11AD" },
            new string [] { "\uAC93", "\uAC93", "\u1100\u1166\u11AE", "\uAC93", "\u1100\u1166\u11AE" },
            new string [] { "\uAC94", "\uAC94", "\u1100\u1166\u11AF", "\uAC94", "\u1100\u1166\u11AF" },
            new string [] { "\uAC95", "\uAC95", "\u1100\u1166\u11B0", "\uAC95", "\u1100\u1166\u11B0" },
            new string [] { "\uAC96", "\uAC96", "\u1100\u1166\u11B1", "\uAC96", "\u1100\u1166\u11B1" },
            new string [] { "\uAC97", "\uAC97", "\u1100\u1166\u11B2", "\uAC97", "\u1100\u1166\u11B2" },
            new string [] { "\uAC98", "\uAC98", "\u1100\u1166\u11B3", "\uAC98", "\u1100\u1166\u11B3" },
            new string [] { "\uAC99", "\uAC99", "\u1100\u1166\u11B4", "\uAC99", "\u1100\u1166\u11B4" },
            new string [] { "\uAC9A", "\uAC9A", "\u1100\u1166\u11B5", "\uAC9A", "\u1100\u1166\u11B5" },
            new string [] { "\uAC9B", "\uAC9B", "\u1100\u1166\u11B6", "\uAC9B", "\u1100\u1166\u11B6" },
            new string [] { "\uAC9C", "\uAC9C", "\u1100\u1166\u11B7", "\uAC9C", "\u1100\u1166\u11B7" },
            new string [] { "\uAC9D", "\uAC9D", "\u1100\u1166\u11B8", "\uAC9D", "\u1100\u1166\u11B8" },
            new string [] { "\uAC9E", "\uAC9E", "\u1100\u1166\u11B9", "\uAC9E", "\u1100\u1166\u11B9" },
            new string [] { "\uAC9F", "\uAC9F", "\u1100\u1166\u11BA", "\uAC9F", "\u1100\u1166\u11BA" },
            new string [] { "\uACA0", "\uACA0", "\u1100\u1166\u11BB", "\uACA0", "\u1100\u1166\u11BB" },
            new string [] { "\uACA1", "\uACA1", "\u1100\u1166\u11BC", "\uACA1", "\u1100\u1166\u11BC" },
            new string [] { "\uACA2", "\uACA2", "\u1100\u1166\u11BD", "\uACA2", "\u1100\u1166\u11BD" },
            new string [] { "\uACA3", "\uACA3", "\u1100\u1166\u11BE", "\uACA3", "\u1100\u1166\u11BE" },
            new string [] { "\uACA4", "\uACA4", "\u1100\u1166\u11BF", "\uACA4", "\u1100\u1166\u11BF" },
            new string [] { "\uACA5", "\uACA5", "\u1100\u1166\u11C0", "\uACA5", "\u1100\u1166\u11C0" },
            new string [] { "\uACA6", "\uACA6", "\u1100\u1166\u11C1", "\uACA6", "\u1100\u1166\u11C1" },
            new string [] { "\uACA7", "\uACA7", "\u1100\u1166\u11C2", "\uACA7", "\u1100\u1166\u11C2" },
            new string [] { "\uACA8", "\uACA8", "\u1100\u1167", "\uACA8", "\u1100\u1167" },
            new string [] { "\uACA9", "\uACA9", "\u1100\u1167\u11A8", "\uACA9", "\u1100\u1167\u11A8" },
            new string [] { "\uACAA", "\uACAA", "\u1100\u1167\u11A9", "\uACAA", "\u1100\u1167\u11A9" },
            new string [] { "\uACAB", "\uACAB", "\u1100\u1167\u11AA", "\uACAB", "\u1100\u1167\u11AA" },
            new string [] { "\uACAC", "\uACAC", "\u1100\u1167\u11AB", "\uACAC", "\u1100\u1167\u11AB" },
            new string [] { "\uACAD", "\uACAD", "\u1100\u1167\u11AC", "\uACAD", "\u1100\u1167\u11AC" },
            new string [] { "\uACAE", "\uACAE", "\u1100\u1167\u11AD", "\uACAE", "\u1100\u1167\u11AD" },
            new string [] { "\uACAF", "\uACAF", "\u1100\u1167\u11AE", "\uACAF", "\u1100\u1167\u11AE" },
            new string [] { "\uACB0", "\uACB0", "\u1100\u1167\u11AF", "\uACB0", "\u1100\u1167\u11AF" },
            new string [] { "\uACB1", "\uACB1", "\u1100\u1167\u11B0", "\uACB1", "\u1100\u1167\u11B0" },
            new string [] { "\uACB2", "\uACB2", "\u1100\u1167\u11B1", "\uACB2", "\u1100\u1167\u11B1" },
            new string [] { "\uACB3", "\uACB3", "\u1100\u1167\u11B2", "\uACB3", "\u1100\u1167\u11B2" },
            new string [] { "\uACB4", "\uACB4", "\u1100\u1167\u11B3", "\uACB4", "\u1100\u1167\u11B3" },
            new string [] { "\uACB5", "\uACB5", "\u1100\u1167\u11B4", "\uACB5", "\u1100\u1167\u11B4" },
            new string [] { "\uACB6", "\uACB6", "\u1100\u1167\u11B5", "\uACB6", "\u1100\u1167\u11B5" },
            new string [] { "\uACB7", "\uACB7", "\u1100\u1167\u11B6", "\uACB7", "\u1100\u1167\u11B6" },
            new string [] { "\uACB8", "\uACB8", "\u1100\u1167\u11B7", "\uACB8", "\u1100\u1167\u11B7" },
            new string [] { "\uACB9", "\uACB9", "\u1100\u1167\u11B8", "\uACB9", "\u1100\u1167\u11B8" },
            new string [] { "\uACBA", "\uACBA", "\u1100\u1167\u11B9", "\uACBA", "\u1100\u1167\u11B9" },
            new string [] { "\uACBB", "\uACBB", "\u1100\u1167\u11BA", "\uACBB", "\u1100\u1167\u11BA" },
            new string [] { "\uACBC", "\uACBC", "\u1100\u1167\u11BB", "\uACBC", "\u1100\u1167\u11BB" },
            new string [] { "\uACBD", "\uACBD", "\u1100\u1167\u11BC", "\uACBD", "\u1100\u1167\u11BC" },
            new string [] { "\uACBE", "\uACBE", "\u1100\u1167\u11BD", "\uACBE", "\u1100\u1167\u11BD" },
            new string [] { "\uACBF", "\uACBF", "\u1100\u1167\u11BE", "\uACBF", "\u1100\u1167\u11BE" },
            new string [] { "\uACC0", "\uACC0", "\u1100\u1167\u11BF", "\uACC0", "\u1100\u1167\u11BF" },
            new string [] { "\uACC1", "\uACC1", "\u1100\u1167\u11C0", "\uACC1", "\u1100\u1167\u11C0" },
            new string [] { "\uACC2", "\uACC2", "\u1100\u1167\u11C1", "\uACC2", "\u1100\u1167\u11C1" },
            new string [] { "\uACC3", "\uACC3", "\u1100\u1167\u11C2", "\uACC3", "\u1100\u1167\u11C2" },
            new string [] { "\uACC4", "\uACC4", "\u1100\u1168", "\uACC4", "\u1100\u1168" },
            new string [] { "\uACC5", "\uACC5", "\u1100\u1168\u11A8", "\uACC5", "\u1100\u1168\u11A8" },
            new string [] { "\uACC6", "\uACC6", "\u1100\u1168\u11A9", "\uACC6", "\u1100\u1168\u11A9" },
            new string [] { "\uACC7", "\uACC7", "\u1100\u1168\u11AA", "\uACC7", "\u1100\u1168\u11AA" },
            new string [] { "\uACC8", "\uACC8", "\u1100\u1168\u11AB", "\uACC8", "\u1100\u1168\u11AB" },
            new string [] { "\uACC9", "\uACC9", "\u1100\u1168\u11AC", "\uACC9", "\u1100\u1168\u11AC" },
            new string [] { "\uACCA", "\uACCA", "\u1100\u1168\u11AD", "\uACCA", "\u1100\u1168\u11AD" },
            new string [] { "\uACCB", "\uACCB", "\u1100\u1168\u11AE", "\uACCB", "\u1100\u1168\u11AE" },
            new string [] { "\uACCC", "\uACCC", "\u1100\u1168\u11AF", "\uACCC", "\u1100\u1168\u11AF" },
            new string [] { "\uACCD", "\uACCD", "\u1100\u1168\u11B0", "\uACCD", "\u1100\u1168\u11B0" },
            new string [] { "\uACCE", "\uACCE", "\u1100\u1168\u11B1", "\uACCE", "\u1100\u1168\u11B1" },
            new string [] { "\uACCF", "\uACCF", "\u1100\u1168\u11B2", "\uACCF", "\u1100\u1168\u11B2" },
            new string [] { "\uACD0", "\uACD0", "\u1100\u1168\u11B3", "\uACD0", "\u1100\u1168\u11B3" },
            new string [] { "\uACD1", "\uACD1", "\u1100\u1168\u11B4", "\uACD1", "\u1100\u1168\u11B4" },
            new string [] { "\uACD2", "\uACD2", "\u1100\u1168\u11B5", "\uACD2", "\u1100\u1168\u11B5" },
            new string [] { "\uACD3", "\uACD3", "\u1100\u1168\u11B6", "\uACD3", "\u1100\u1168\u11B6" },
            new string [] { "\uACD4", "\uACD4", "\u1100\u1168\u11B7", "\uACD4", "\u1100\u1168\u11B7" },
            new string [] { "\uACD5", "\uACD5", "\u1100\u1168\u11B8", "\uACD5", "\u1100\u1168\u11B8" },
            new string [] { "\uACD6", "\uACD6", "\u1100\u1168\u11B9", "\uACD6", "\u1100\u1168\u11B9" },
            new string [] { "\uACD7", "\uACD7", "\u1100\u1168\u11BA", "\uACD7", "\u1100\u1168\u11BA" },
            new string [] { "\uACD8", "\uACD8", "\u1100\u1168\u11BB", "\uACD8", "\u1100\u1168\u11BB" },
            new string [] { "\uACD9", "\uACD9", "\u1100\u1168\u11BC", "\uACD9", "\u1100\u1168\u11BC" },
            new string [] { "\uACDA", "\uACDA", "\u1100\u1168\u11BD", "\uACDA", "\u1100\u1168\u11BD" },
            new string [] { "\uACDB", "\uACDB", "\u1100\u1168\u11BE", "\uACDB", "\u1100\u1168\u11BE" },
            new string [] { "\uACDC", "\uACDC", "\u1100\u1168\u11BF", "\uACDC", "\u1100\u1168\u11BF" },
            new string [] { "\uACDD", "\uACDD", "\u1100\u1168\u11C0", "\uACDD", "\u1100\u1168\u11C0" },
            new string [] { "\uACDE", "\uACDE", "\u1100\u1168\u11C1", "\uACDE", "\u1100\u1168\u11C1" },
            new string [] { "\uACDF", "\uACDF", "\u1100\u1168\u11C2", "\uACDF", "\u1100\u1168\u11C2" },
            new string [] { "\uACE0", "\uACE0", "\u1100\u1169", "\uACE0", "\u1100\u1169" },
            new string [] { "\uACE1", "\uACE1", "\u1100\u1169\u11A8", "\uACE1", "\u1100\u1169\u11A8" },
            new string [] { "\uACE2", "\uACE2", "\u1100\u1169\u11A9", "\uACE2", "\u1100\u1169\u11A9" },
            new string [] { "\uACE3", "\uACE3", "\u1100\u1169\u11AA", "\uACE3", "\u1100\u1169\u11AA" },
            new string [] { "\uACE4", "\uACE4", "\u1100\u1169\u11AB", "\uACE4", "\u1100\u1169\u11AB" },
            new string [] { "\uACE5", "\uACE5", "\u1100\u1169\u11AC", "\uACE5", "\u1100\u1169\u11AC" },
            new string [] { "\uACE6", "\uACE6", "\u1100\u1169\u11AD", "\uACE6", "\u1100\u1169\u11AD" },
            new string [] { "\uACE7", "\uACE7", "\u1100\u1169\u11AE", "\uACE7", "\u1100\u1169\u11AE" },
            new string [] { "\uACE8", "\uACE8", "\u1100\u1169\u11AF", "\uACE8", "\u1100\u1169\u11AF" },
            new string [] { "\uACE9", "\uACE9", "\u1100\u1169\u11B0", "\uACE9", "\u1100\u1169\u11B0" },
            new string [] { "\uACEA", "\uACEA", "\u1100\u1169\u11B1", "\uACEA", "\u1100\u1169\u11B1" },
            new string [] { "\uACEB", "\uACEB", "\u1100\u1169\u11B2", "\uACEB", "\u1100\u1169\u11B2" },
            new string [] { "\uACEC", "\uACEC", "\u1100\u1169\u11B3", "\uACEC", "\u1100\u1169\u11B3" },
            new string [] { "\uACED", "\uACED", "\u1100\u1169\u11B4", "\uACED", "\u1100\u1169\u11B4" },
            new string [] { "\uACEE", "\uACEE", "\u1100\u1169\u11B5", "\uACEE", "\u1100\u1169\u11B5" },
            new string [] { "\uACEF", "\uACEF", "\u1100\u1169\u11B6", "\uACEF", "\u1100\u1169\u11B6" },
            new string [] { "\uACF0", "\uACF0", "\u1100\u1169\u11B7", "\uACF0", "\u1100\u1169\u11B7" },
            new string [] { "\uACF1", "\uACF1", "\u1100\u1169\u11B8", "\uACF1", "\u1100\u1169\u11B8" },
            new string [] { "\uACF2", "\uACF2", "\u1100\u1169\u11B9", "\uACF2", "\u1100\u1169\u11B9" },
            new string [] { "\uACF3", "\uACF3", "\u1100\u1169\u11BA", "\uACF3", "\u1100\u1169\u11BA" },
            new string [] { "\uACF4", "\uACF4", "\u1100\u1169\u11BB", "\uACF4", "\u1100\u1169\u11BB" },
            new string [] { "\uACF5", "\uACF5", "\u1100\u1169\u11BC", "\uACF5", "\u1100\u1169\u11BC" },
            new string [] { "\uACF6", "\uACF6", "\u1100\u1169\u11BD", "\uACF6", "\u1100\u1169\u11BD" },
            new string [] { "\uACF7", "\uACF7", "\u1100\u1169\u11BE", "\uACF7", "\u1100\u1169\u11BE" },
            new string [] { "\uACF8", "\uACF8", "\u1100\u1169\u11BF", "\uACF8", "\u1100\u1169\u11BF" },
            new string [] { "\uACF9", "\uACF9", "\u1100\u1169\u11C0", "\uACF9", "\u1100\u1169\u11C0" },
            new string [] { "\uACFA", "\uACFA", "\u1100\u1169\u11C1", "\uACFA", "\u1100\u1169\u11C1" },
            new string [] { "\uACFB", "\uACFB", "\u1100\u1169\u11C2", "\uACFB", "\u1100\u1169\u11C2" },
            new string [] { "\uACFC", "\uACFC", "\u1100\u116A", "\uACFC", "\u1100\u116A" },
            new string [] { "\uACFD", "\uACFD", "\u1100\u116A\u11A8", "\uACFD", "\u1100\u116A\u11A8" },
            new string [] { "\uACFE", "\uACFE", "\u1100\u116A\u11A9", "\uACFE", "\u1100\u116A\u11A9" },
            new string [] { "\uACFF", "\uACFF", "\u1100\u116A\u11AA", "\uACFF", "\u1100\u116A\u11AA" },
            new string [] { "\uAD00", "\uAD00", "\u1100\u116A\u11AB", "\uAD00", "\u1100\u116A\u11AB" },
            new string [] { "\uAD01", "\uAD01", "\u1100\u116A\u11AC", "\uAD01", "\u1100\u116A\u11AC" },
            new string [] { "\uAD02", "\uAD02", "\u1100\u116A\u11AD", "\uAD02", "\u1100\u116A\u11AD" },
            new string [] { "\uAD03", "\uAD03", "\u1100\u116A\u11AE", "\uAD03", "\u1100\u116A\u11AE" },
            new string [] { "\uAD04", "\uAD04", "\u1100\u116A\u11AF", "\uAD04", "\u1100\u116A\u11AF" },
            new string [] { "\uAD05", "\uAD05", "\u1100\u116A\u11B0", "\uAD05", "\u1100\u116A\u11B0" },
            new string [] { "\uAD06", "\uAD06", "\u1100\u116A\u11B1", "\uAD06", "\u1100\u116A\u11B1" },
            new string [] { "\uAD07", "\uAD07", "\u1100\u116A\u11B2", "\uAD07", "\u1100\u116A\u11B2" },
            new string [] { "\uAD08", "\uAD08", "\u1100\u116A\u11B3", "\uAD08", "\u1100\u116A\u11B3" },
            new string [] { "\uAD09", "\uAD09", "\u1100\u116A\u11B4", "\uAD09", "\u1100\u116A\u11B4" },
            new string [] { "\uAD0A", "\uAD0A", "\u1100\u116A\u11B5", "\uAD0A", "\u1100\u116A\u11B5" },
            new string [] { "\uAD0B", "\uAD0B", "\u1100\u116A\u11B6", "\uAD0B", "\u1100\u116A\u11B6" },
            new string [] { "\uAD0C", "\uAD0C", "\u1100\u116A\u11B7", "\uAD0C", "\u1100\u116A\u11B7" },
            new string [] { "\uAD0D", "\uAD0D", "\u1100\u116A\u11B8", "\uAD0D", "\u1100\u116A\u11B8" },
            new string [] { "\uAD0E", "\uAD0E", "\u1100\u116A\u11B9", "\uAD0E", "\u1100\u116A\u11B9" },
            new string [] { "\uAD0F", "\uAD0F", "\u1100\u116A\u11BA", "\uAD0F", "\u1100\u116A\u11BA" },
            new string [] { "\uAD10", "\uAD10", "\u1100\u116A\u11BB", "\uAD10", "\u1100\u116A\u11BB" },
            new string [] { "\uAD11", "\uAD11", "\u1100\u116A\u11BC", "\uAD11", "\u1100\u116A\u11BC" },
            new string [] { "\uAD12", "\uAD12", "\u1100\u116A\u11BD", "\uAD12", "\u1100\u116A\u11BD" },
            new string [] { "\uAD13", "\uAD13", "\u1100\u116A\u11BE", "\uAD13", "\u1100\u116A\u11BE" },
            new string [] { "\uAD14", "\uAD14", "\u1100\u116A\u11BF", "\uAD14", "\u1100\u116A\u11BF" },
            new string [] { "\uAD15", "\uAD15", "\u1100\u116A\u11C0", "\uAD15", "\u1100\u116A\u11C0" },
            new string [] { "\uAD16", "\uAD16", "\u1100\u116A\u11C1", "\uAD16", "\u1100\u116A\u11C1" },
            new string [] { "\uAD17", "\uAD17", "\u1100\u116A\u11C2", "\uAD17", "\u1100\u116A\u11C2" },
            new string [] { "\uAD18", "\uAD18", "\u1100\u116B", "\uAD18", "\u1100\u116B" },
            new string [] { "\uAD19", "\uAD19", "\u1100\u116B\u11A8", "\uAD19", "\u1100\u116B\u11A8" },
            new string [] { "\uAD1A", "\uAD1A", "\u1100\u116B\u11A9", "\uAD1A", "\u1100\u116B\u11A9" },
            new string [] { "\uAD1B", "\uAD1B", "\u1100\u116B\u11AA", "\uAD1B", "\u1100\u116B\u11AA" },
            new string [] { "\uAD1C", "\uAD1C", "\u1100\u116B\u11AB", "\uAD1C", "\u1100\u116B\u11AB" },
            new string [] { "\uAD1D", "\uAD1D", "\u1100\u116B\u11AC", "\uAD1D", "\u1100\u116B\u11AC" },
            new string [] { "\uAD1E", "\uAD1E", "\u1100\u116B\u11AD", "\uAD1E", "\u1100\u116B\u11AD" },
            new string [] { "\uAD1F", "\uAD1F", "\u1100\u116B\u11AE", "\uAD1F", "\u1100\u116B\u11AE" },
            new string [] { "\uAD20", "\uAD20", "\u1100\u116B\u11AF", "\uAD20", "\u1100\u116B\u11AF" },
            new string [] { "\uAD21", "\uAD21", "\u1100\u116B\u11B0", "\uAD21", "\u1100\u116B\u11B0" },
            new string [] { "\uAD22", "\uAD22", "\u1100\u116B\u11B1", "\uAD22", "\u1100\u116B\u11B1" },
            new string [] { "\uAD23", "\uAD23", "\u1100\u116B\u11B2", "\uAD23", "\u1100\u116B\u11B2" },
            new string [] { "\uAD24", "\uAD24", "\u1100\u116B\u11B3", "\uAD24", "\u1100\u116B\u11B3" },
            new string [] { "\uAD25", "\uAD25", "\u1100\u116B\u11B4", "\uAD25", "\u1100\u116B\u11B4" },
            new string [] { "\uAD26", "\uAD26", "\u1100\u116B\u11B5", "\uAD26", "\u1100\u116B\u11B5" },
            new string [] { "\uAD27", "\uAD27", "\u1100\u116B\u11B6", "\uAD27", "\u1100\u116B\u11B6" },
            new string [] { "\uAD28", "\uAD28", "\u1100\u116B\u11B7", "\uAD28", "\u1100\u116B\u11B7" },
            new string [] { "\uAD29", "\uAD29", "\u1100\u116B\u11B8", "\uAD29", "\u1100\u116B\u11B8" },
            new string [] { "\uAD2A", "\uAD2A", "\u1100\u116B\u11B9", "\uAD2A", "\u1100\u116B\u11B9" },
            new string [] { "\uAD2B", "\uAD2B", "\u1100\u116B\u11BA", "\uAD2B", "\u1100\u116B\u11BA" },
            new string [] { "\uAD2C", "\uAD2C", "\u1100\u116B\u11BB", "\uAD2C", "\u1100\u116B\u11BB" },
            new string [] { "\uAD2D", "\uAD2D", "\u1100\u116B\u11BC", "\uAD2D", "\u1100\u116B\u11BC" },
            new string [] { "\uAD2E", "\uAD2E", "\u1100\u116B\u11BD", "\uAD2E", "\u1100\u116B\u11BD" },
            new string [] { "\uAD2F", "\uAD2F", "\u1100\u116B\u11BE", "\uAD2F", "\u1100\u116B\u11BE" },
            new string [] { "\uAD30", "\uAD30", "\u1100\u116B\u11BF", "\uAD30", "\u1100\u116B\u11BF" },
            new string [] { "\uAD31", "\uAD31", "\u1100\u116B\u11C0", "\uAD31", "\u1100\u116B\u11C0" },
            new string [] { "\uAD32", "\uAD32", "\u1100\u116B\u11C1", "\uAD32", "\u1100\u116B\u11C1" },
            new string [] { "\uAD33", "\uAD33", "\u1100\u116B\u11C2", "\uAD33", "\u1100\u116B\u11C2" },
            new string [] { "\uAD34", "\uAD34", "\u1100\u116C", "\uAD34", "\u1100\u116C" },
            new string [] { "\uAD35", "\uAD35", "\u1100\u116C\u11A8", "\uAD35", "\u1100\u116C\u11A8" },
            new string [] { "\uAD36", "\uAD36", "\u1100\u116C\u11A9", "\uAD36", "\u1100\u116C\u11A9" },
            new string [] { "\uAD37", "\uAD37", "\u1100\u116C\u11AA", "\uAD37", "\u1100\u116C\u11AA" },
            new string [] { "\uAD38", "\uAD38", "\u1100\u116C\u11AB", "\uAD38", "\u1100\u116C\u11AB" },
            new string [] { "\uAD39", "\uAD39", "\u1100\u116C\u11AC", "\uAD39", "\u1100\u116C\u11AC" },
            new string [] { "\uAD3A", "\uAD3A", "\u1100\u116C\u11AD", "\uAD3A", "\u1100\u116C\u11AD" },
            new string [] { "\uAD3B", "\uAD3B", "\u1100\u116C\u11AE", "\uAD3B", "\u1100\u116C\u11AE" },
            new string [] { "\uAD3C", "\uAD3C", "\u1100\u116C\u11AF", "\uAD3C", "\u1100\u116C\u11AF" },
            new string [] { "\uAD3D", "\uAD3D", "\u1100\u116C\u11B0", "\uAD3D", "\u1100\u116C\u11B0" },
            new string [] { "\uAD3E", "\uAD3E", "\u1100\u116C\u11B1", "\uAD3E", "\u1100\u116C\u11B1" },
            new string [] { "\uAD3F", "\uAD3F", "\u1100\u116C\u11B2", "\uAD3F", "\u1100\u116C\u11B2" },
            new string [] { "\uAD40", "\uAD40", "\u1100\u116C\u11B3", "\uAD40", "\u1100\u116C\u11B3" },
            new string [] { "\uAD41", "\uAD41", "\u1100\u116C\u11B4", "\uAD41", "\u1100\u116C\u11B4" },
            new string [] { "\uAD42", "\uAD42", "\u1100\u116C\u11B5", "\uAD42", "\u1100\u116C\u11B5" },
            new string [] { "\uAD43", "\uAD43", "\u1100\u116C\u11B6", "\uAD43", "\u1100\u116C\u11B6" },
            new string [] { "\uAD44", "\uAD44", "\u1100\u116C\u11B7", "\uAD44", "\u1100\u116C\u11B7" },
            new string [] { "\uAD45", "\uAD45", "\u1100\u116C\u11B8", "\uAD45", "\u1100\u116C\u11B8" },
            new string [] { "\uAD46", "\uAD46", "\u1100\u116C\u11B9", "\uAD46", "\u1100\u116C\u11B9" },
            new string [] { "\uAD47", "\uAD47", "\u1100\u116C\u11BA", "\uAD47", "\u1100\u116C\u11BA" },
            new string [] { "\uAD48", "\uAD48", "\u1100\u116C\u11BB", "\uAD48", "\u1100\u116C\u11BB" },
            new string [] { "\uAD49", "\uAD49", "\u1100\u116C\u11BC", "\uAD49", "\u1100\u116C\u11BC" },
            new string [] { "\uAD4A", "\uAD4A", "\u1100\u116C\u11BD", "\uAD4A", "\u1100\u116C\u11BD" },
            new string [] { "\uAD4B", "\uAD4B", "\u1100\u116C\u11BE", "\uAD4B", "\u1100\u116C\u11BE" },
            new string [] { "\uAD4C", "\uAD4C", "\u1100\u116C\u11BF", "\uAD4C", "\u1100\u116C\u11BF" },
            new string [] { "\uAD4D", "\uAD4D", "\u1100\u116C\u11C0", "\uAD4D", "\u1100\u116C\u11C0" },
            new string [] { "\uAD4E", "\uAD4E", "\u1100\u116C\u11C1", "\uAD4E", "\u1100\u116C\u11C1" },
            new string [] { "\uAD4F", "\uAD4F", "\u1100\u116C\u11C2", "\uAD4F", "\u1100\u116C\u11C2" },
            new string [] { "\uAD50", "\uAD50", "\u1100\u116D", "\uAD50", "\u1100\u116D" },
            new string [] { "\uAD51", "\uAD51", "\u1100\u116D\u11A8", "\uAD51", "\u1100\u116D\u11A8" },
            new string [] { "\uAD52", "\uAD52", "\u1100\u116D\u11A9", "\uAD52", "\u1100\u116D\u11A9" },
            new string [] { "\uAD53", "\uAD53", "\u1100\u116D\u11AA", "\uAD53", "\u1100\u116D\u11AA" },
            new string [] { "\uAD54", "\uAD54", "\u1100\u116D\u11AB", "\uAD54", "\u1100\u116D\u11AB" },
            new string [] { "\uAD55", "\uAD55", "\u1100\u116D\u11AC", "\uAD55", "\u1100\u116D\u11AC" },
            new string [] { "\uAD56", "\uAD56", "\u1100\u116D\u11AD", "\uAD56", "\u1100\u116D\u11AD" },
            new string [] { "\uAD57", "\uAD57", "\u1100\u116D\u11AE", "\uAD57", "\u1100\u116D\u11AE" },
            new string [] { "\uAD58", "\uAD58", "\u1100\u116D\u11AF", "\uAD58", "\u1100\u116D\u11AF" },
            new string [] { "\uAD59", "\uAD59", "\u1100\u116D\u11B0", "\uAD59", "\u1100\u116D\u11B0" },
            new string [] { "\uAD5A", "\uAD5A", "\u1100\u116D\u11B1", "\uAD5A", "\u1100\u116D\u11B1" },
            new string [] { "\uAD5B", "\uAD5B", "\u1100\u116D\u11B2", "\uAD5B", "\u1100\u116D\u11B2" },
            new string [] { "\uAD5C", "\uAD5C", "\u1100\u116D\u11B3", "\uAD5C", "\u1100\u116D\u11B3" },
            new string [] { "\uAD5D", "\uAD5D", "\u1100\u116D\u11B4", "\uAD5D", "\u1100\u116D\u11B4" },
            new string [] { "\uAD5E", "\uAD5E", "\u1100\u116D\u11B5", "\uAD5E", "\u1100\u116D\u11B5" },
            new string [] { "\uAD5F", "\uAD5F", "\u1100\u116D\u11B6", "\uAD5F", "\u1100\u116D\u11B6" },
            new string [] { "\uAD60", "\uAD60", "\u1100\u116D\u11B7", "\uAD60", "\u1100\u116D\u11B7" },
            new string [] { "\uAD61", "\uAD61", "\u1100\u116D\u11B8", "\uAD61", "\u1100\u116D\u11B8" },
            new string [] { "\uAD62", "\uAD62", "\u1100\u116D\u11B9", "\uAD62", "\u1100\u116D\u11B9" },
            new string [] { "\uAD63", "\uAD63", "\u1100\u116D\u11BA", "\uAD63", "\u1100\u116D\u11BA" },
            new string [] { "\uAD64", "\uAD64", "\u1100\u116D\u11BB", "\uAD64", "\u1100\u116D\u11BB" },
            new string [] { "\uAD65", "\uAD65", "\u1100\u116D\u11BC", "\uAD65", "\u1100\u116D\u11BC" },
            new string [] { "\uAD66", "\uAD66", "\u1100\u116D\u11BD", "\uAD66", "\u1100\u116D\u11BD" },
            new string [] { "\uAD67", "\uAD67", "\u1100\u116D\u11BE", "\uAD67", "\u1100\u116D\u11BE" },
            new string [] { "\uAD68", "\uAD68", "\u1100\u116D\u11BF", "\uAD68", "\u1100\u116D\u11BF" },
            new string [] { "\uAD69", "\uAD69", "\u1100\u116D\u11C0", "\uAD69", "\u1100\u116D\u11C0" },
            new string [] { "\uAD6A", "\uAD6A", "\u1100\u116D\u11C1", "\uAD6A", "\u1100\u116D\u11C1" },
            new string [] { "\uAD6B", "\uAD6B", "\u1100\u116D\u11C2", "\uAD6B", "\u1100\u116D\u11C2" },
            new string [] { "\uAD6C", "\uAD6C", "\u1100\u116E", "\uAD6C", "\u1100\u116E" },
            new string [] { "\uAD6D", "\uAD6D", "\u1100\u116E\u11A8", "\uAD6D", "\u1100\u116E\u11A8" },
            new string [] { "\uAD6E", "\uAD6E", "\u1100\u116E\u11A9", "\uAD6E", "\u1100\u116E\u11A9" },
            new string [] { "\uAD6F", "\uAD6F", "\u1100\u116E\u11AA", "\uAD6F", "\u1100\u116E\u11AA" },
            new string [] { "\uAD70", "\uAD70", "\u1100\u116E\u11AB", "\uAD70", "\u1100\u116E\u11AB" },
            new string [] { "\uAD71", "\uAD71", "\u1100\u116E\u11AC", "\uAD71", "\u1100\u116E\u11AC" },
            new string [] { "\uAD72", "\uAD72", "\u1100\u116E\u11AD", "\uAD72", "\u1100\u116E\u11AD" },
            new string [] { "\uAD73", "\uAD73", "\u1100\u116E\u11AE", "\uAD73", "\u1100\u116E\u11AE" },
            new string [] { "\uAD74", "\uAD74", "\u1100\u116E\u11AF", "\uAD74", "\u1100\u116E\u11AF" },
            new string [] { "\uAD75", "\uAD75", "\u1100\u116E\u11B0", "\uAD75", "\u1100\u116E\u11B0" },
            new string [] { "\uAD76", "\uAD76", "\u1100\u116E\u11B1", "\uAD76", "\u1100\u116E\u11B1" },
            new string [] { "\uAD77", "\uAD77", "\u1100\u116E\u11B2", "\uAD77", "\u1100\u116E\u11B2" },
            new string [] { "\uAD78", "\uAD78", "\u1100\u116E\u11B3", "\uAD78", "\u1100\u116E\u11B3" },
            new string [] { "\uAD79", "\uAD79", "\u1100\u116E\u11B4", "\uAD79", "\u1100\u116E\u11B4" },
            new string [] { "\uAD7A", "\uAD7A", "\u1100\u116E\u11B5", "\uAD7A", "\u1100\u116E\u11B5" },
            new string [] { "\uAD7B", "\uAD7B", "\u1100\u116E\u11B6", "\uAD7B", "\u1100\u116E\u11B6" },
            new string [] { "\uAD7C", "\uAD7C", "\u1100\u116E\u11B7", "\uAD7C", "\u1100\u116E\u11B7" },
            new string [] { "\uAD7D", "\uAD7D", "\u1100\u116E\u11B8", "\uAD7D", "\u1100\u116E\u11B8" },
            new string [] { "\uAD7E", "\uAD7E", "\u1100\u116E\u11B9", "\uAD7E", "\u1100\u116E\u11B9" },
            new string [] { "\uAD7F", "\uAD7F", "\u1100\u116E\u11BA", "\uAD7F", "\u1100\u116E\u11BA" },
            new string [] { "\uAD80", "\uAD80", "\u1100\u116E\u11BB", "\uAD80", "\u1100\u116E\u11BB" },
            new string [] { "\uAD81", "\uAD81", "\u1100\u116E\u11BC", "\uAD81", "\u1100\u116E\u11BC" },
            new string [] { "\uAD82", "\uAD82", "\u1100\u116E\u11BD", "\uAD82", "\u1100\u116E\u11BD" },
            new string [] { "\uAD83", "\uAD83", "\u1100\u116E\u11BE", "\uAD83", "\u1100\u116E\u11BE" },
            new string [] { "\uAD84", "\uAD84", "\u1100\u116E\u11BF", "\uAD84", "\u1100\u116E\u11BF" },
            new string [] { "\uAD85", "\uAD85", "\u1100\u116E\u11C0", "\uAD85", "\u1100\u116E\u11C0" },
            new string [] { "\uAD86", "\uAD86", "\u1100\u116E\u11C1", "\uAD86", "\u1100\u116E\u11C1" },
            new string [] { "\uAD87", "\uAD87", "\u1100\u116E\u11C2", "\uAD87", "\u1100\u116E\u11C2" },
            new string [] { "\uAD88", "\uAD88", "\u1100\u116F", "\uAD88", "\u1100\u116F" },
            new string [] { "\uAD89", "\uAD89", "\u1100\u116F\u11A8", "\uAD89", "\u1100\u116F\u11A8" },
            new string [] { "\uAD8A", "\uAD8A", "\u1100\u116F\u11A9", "\uAD8A", "\u1100\u116F\u11A9" },
            new string [] { "\uAD8B", "\uAD8B", "\u1100\u116F\u11AA", "\uAD8B", "\u1100\u116F\u11AA" },
            new string [] { "\uAD8C", "\uAD8C", "\u1100\u116F\u11AB", "\uAD8C", "\u1100\u116F\u11AB" },
            new string [] { "\uAD8D", "\uAD8D", "\u1100\u116F\u11AC", "\uAD8D", "\u1100\u116F\u11AC" },
            new string [] { "\uAD8E", "\uAD8E", "\u1100\u116F\u11AD", "\uAD8E", "\u1100\u116F\u11AD" },
            new string [] { "\uAD8F", "\uAD8F", "\u1100\u116F\u11AE", "\uAD8F", "\u1100\u116F\u11AE" },
            new string [] { "\uAD90", "\uAD90", "\u1100\u116F\u11AF", "\uAD90", "\u1100\u116F\u11AF" },
            new string [] { "\uAD91", "\uAD91", "\u1100\u116F\u11B0", "\uAD91", "\u1100\u116F\u11B0" },
            new string [] { "\uAD92", "\uAD92", "\u1100\u116F\u11B1", "\uAD92", "\u1100\u116F\u11B1" },
            new string [] { "\uAD93", "\uAD93", "\u1100\u116F\u11B2", "\uAD93", "\u1100\u116F\u11B2" },
            new string [] { "\uAD94", "\uAD94", "\u1100\u116F\u11B3", "\uAD94", "\u1100\u116F\u11B3" },
            new string [] { "\uAD95", "\uAD95", "\u1100\u116F\u11B4", "\uAD95", "\u1100\u116F\u11B4" },
            new string [] { "\uAD96", "\uAD96", "\u1100\u116F\u11B5", "\uAD96", "\u1100\u116F\u11B5" },
            new string [] { "\uAD97", "\uAD97", "\u1100\u116F\u11B6", "\uAD97", "\u1100\u116F\u11B6" },
            new string [] { "\uAD98", "\uAD98", "\u1100\u116F\u11B7", "\uAD98", "\u1100\u116F\u11B7" },
            new string [] { "\uAD99", "\uAD99", "\u1100\u116F\u11B8", "\uAD99", "\u1100\u116F\u11B8" },
            new string [] { "\uAD9A", "\uAD9A", "\u1100\u116F\u11B9", "\uAD9A", "\u1100\u116F\u11B9" },
            new string [] { "\uAD9B", "\uAD9B", "\u1100\u116F\u11BA", "\uAD9B", "\u1100\u116F\u11BA" },
            new string [] { "\uAD9C", "\uAD9C", "\u1100\u116F\u11BB", "\uAD9C", "\u1100\u116F\u11BB" },
            new string [] { "\uAD9D", "\uAD9D", "\u1100\u116F\u11BC", "\uAD9D", "\u1100\u116F\u11BC" },
            new string [] { "\uAD9E", "\uAD9E", "\u1100\u116F\u11BD", "\uAD9E", "\u1100\u116F\u11BD" },
            new string [] { "\uAD9F", "\uAD9F", "\u1100\u116F\u11BE", "\uAD9F", "\u1100\u116F\u11BE" },
            new string [] { "\uADA0", "\uADA0", "\u1100\u116F\u11BF", "\uADA0", "\u1100\u116F\u11BF" },
            new string [] { "\uADA1", "\uADA1", "\u1100\u116F\u11C0", "\uADA1", "\u1100\u116F\u11C0" },
            new string [] { "\uADA2", "\uADA2", "\u1100\u116F\u11C1", "\uADA2", "\u1100\u116F\u11C1" },
            new string [] { "\uADA3", "\uADA3", "\u1100\u116F\u11C2", "\uADA3", "\u1100\u116F\u11C2" },
            new string [] { "\uADA4", "\uADA4", "\u1100\u1170", "\uADA4", "\u1100\u1170" },
            new string [] { "\uADA5", "\uADA5", "\u1100\u1170\u11A8", "\uADA5", "\u1100\u1170\u11A8" },
            new string [] { "\uADA6", "\uADA6", "\u1100\u1170\u11A9", "\uADA6", "\u1100\u1170\u11A9" },
            new string [] { "\uADA7", "\uADA7", "\u1100\u1170\u11AA", "\uADA7", "\u1100\u1170\u11AA" },
            new string [] { "\uADA8", "\uADA8", "\u1100\u1170\u11AB", "\uADA8", "\u1100\u1170\u11AB" },
            new string [] { "\uADA9", "\uADA9", "\u1100\u1170\u11AC", "\uADA9", "\u1100\u1170\u11AC" },
            new string [] { "\uADAA", "\uADAA", "\u1100\u1170\u11AD", "\uADAA", "\u1100\u1170\u11AD" },
            new string [] { "\uADAB", "\uADAB", "\u1100\u1170\u11AE", "\uADAB", "\u1100\u1170\u11AE" },
            new string [] { "\uADAC", "\uADAC", "\u1100\u1170\u11AF", "\uADAC", "\u1100\u1170\u11AF" },
            new string [] { "\uADAD", "\uADAD", "\u1100\u1170\u11B0", "\uADAD", "\u1100\u1170\u11B0" },
            new string [] { "\uADAE", "\uADAE", "\u1100\u1170\u11B1", "\uADAE", "\u1100\u1170\u11B1" },
            new string [] { "\uADAF", "\uADAF", "\u1100\u1170\u11B2", "\uADAF", "\u1100\u1170\u11B2" },
            new string [] { "\uADB0", "\uADB0", "\u1100\u1170\u11B3", "\uADB0", "\u1100\u1170\u11B3" },
            new string [] { "\uADB1", "\uADB1", "\u1100\u1170\u11B4", "\uADB1", "\u1100\u1170\u11B4" },
            new string [] { "\uADB2", "\uADB2", "\u1100\u1170\u11B5", "\uADB2", "\u1100\u1170\u11B5" },
            new string [] { "\uADB3", "\uADB3", "\u1100\u1170\u11B6", "\uADB3", "\u1100\u1170\u11B6" },
            new string [] { "\uADB4", "\uADB4", "\u1100\u1170\u11B7", "\uADB4", "\u1100\u1170\u11B7" },
            new string [] { "\uADB5", "\uADB5", "\u1100\u1170\u11B8", "\uADB5", "\u1100\u1170\u11B8" },
            new string [] { "\uADB6", "\uADB6", "\u1100\u1170\u11B9", "\uADB6", "\u1100\u1170\u11B9" },
            new string [] { "\uADB7", "\uADB7", "\u1100\u1170\u11BA", "\uADB7", "\u1100\u1170\u11BA" },
            new string [] { "\uADB8", "\uADB8", "\u1100\u1170\u11BB", "\uADB8", "\u1100\u1170\u11BB" },
            new string [] { "\uADB9", "\uADB9", "\u1100\u1170\u11BC", "\uADB9", "\u1100\u1170\u11BC" },
            new string [] { "\uADBA", "\uADBA", "\u1100\u1170\u11BD", "\uADBA", "\u1100\u1170\u11BD" },
            new string [] { "\uADBB", "\uADBB", "\u1100\u1170\u11BE", "\uADBB", "\u1100\u1170\u11BE" },
            new string [] { "\uADBC", "\uADBC", "\u1100\u1170\u11BF", "\uADBC", "\u1100\u1170\u11BF" },
            new string [] { "\uADBD", "\uADBD", "\u1100\u1170\u11C0", "\uADBD", "\u1100\u1170\u11C0" },
            new string [] { "\uADBE", "\uADBE", "\u1100\u1170\u11C1", "\uADBE", "\u1100\u1170\u11C1" },
            new string [] { "\uADBF", "\uADBF", "\u1100\u1170\u11C2", "\uADBF", "\u1100\u1170\u11C2" },
            new string [] { "\uADC0", "\uADC0", "\u1100\u1171", "\uADC0", "\u1100\u1171" },
            new string [] { "\uADC1", "\uADC1", "\u1100\u1171\u11A8", "\uADC1", "\u1100\u1171\u11A8" },
            new string [] { "\uADC2", "\uADC2", "\u1100\u1171\u11A9", "\uADC2", "\u1100\u1171\u11A9" },
            new string [] { "\uADC3", "\uADC3", "\u1100\u1171\u11AA", "\uADC3", "\u1100\u1171\u11AA" },
            new string [] { "\uADC4", "\uADC4", "\u1100\u1171\u11AB", "\uADC4", "\u1100\u1171\u11AB" },
            new string [] { "\uADC5", "\uADC5", "\u1100\u1171\u11AC", "\uADC5", "\u1100\u1171\u11AC" },
            new string [] { "\uADC6", "\uADC6", "\u1100\u1171\u11AD", "\uADC6", "\u1100\u1171\u11AD" },
            new string [] { "\uADC7", "\uADC7", "\u1100\u1171\u11AE", "\uADC7", "\u1100\u1171\u11AE" },
            new string [] { "\uADC8", "\uADC8", "\u1100\u1171\u11AF", "\uADC8", "\u1100\u1171\u11AF" },
            new string [] { "\uADC9", "\uADC9", "\u1100\u1171\u11B0", "\uADC9", "\u1100\u1171\u11B0" },
            new string [] { "\uADCA", "\uADCA", "\u1100\u1171\u11B1", "\uADCA", "\u1100\u1171\u11B1" },
            new string [] { "\uADCB", "\uADCB", "\u1100\u1171\u11B2", "\uADCB", "\u1100\u1171\u11B2" },
            new string [] { "\uADCC", "\uADCC", "\u1100\u1171\u11B3", "\uADCC", "\u1100\u1171\u11B3" },
            new string [] { "\uADCD", "\uADCD", "\u1100\u1171\u11B4", "\uADCD", "\u1100\u1171\u11B4" },
            new string [] { "\uADCE", "\uADCE", "\u1100\u1171\u11B5", "\uADCE", "\u1100\u1171\u11B5" },
            new string [] { "\uADCF", "\uADCF", "\u1100\u1171\u11B6", "\uADCF", "\u1100\u1171\u11B6" },
            new string [] { "\uADD0", "\uADD0", "\u1100\u1171\u11B7", "\uADD0", "\u1100\u1171\u11B7" },
            new string [] { "\uADD1", "\uADD1", "\u1100\u1171\u11B8", "\uADD1", "\u1100\u1171\u11B8" },
            new string [] { "\uADD2", "\uADD2", "\u1100\u1171\u11B9", "\uADD2", "\u1100\u1171\u11B9" },
            new string [] { "\uADD3", "\uADD3", "\u1100\u1171\u11BA", "\uADD3", "\u1100\u1171\u11BA" },
            new string [] { "\uADD4", "\uADD4", "\u1100\u1171\u11BB", "\uADD4", "\u1100\u1171\u11BB" },
            new string [] { "\uADD5", "\uADD5", "\u1100\u1171\u11BC", "\uADD5", "\u1100\u1171\u11BC" },
            new string [] { "\uADD6", "\uADD6", "\u1100\u1171\u11BD", "\uADD6", "\u1100\u1171\u11BD" },
            new string [] { "\uADD7", "\uADD7", "\u1100\u1171\u11BE", "\uADD7", "\u1100\u1171\u11BE" },
            new string [] { "\uADD8", "\uADD8", "\u1100\u1171\u11BF", "\uADD8", "\u1100\u1171\u11BF" },
            new string [] { "\uADD9", "\uADD9", "\u1100\u1171\u11C0", "\uADD9", "\u1100\u1171\u11C0" },
            new string [] { "\uADDA", "\uADDA", "\u1100\u1171\u11C1", "\uADDA", "\u1100\u1171\u11C1" },
            new string [] { "\uADDB", "\uADDB", "\u1100\u1171\u11C2", "\uADDB", "\u1100\u1171\u11C2" },
            new string [] { "\uADDC", "\uADDC", "\u1100\u1172", "\uADDC", "\u1100\u1172" },
            new string [] { "\uADDD", "\uADDD", "\u1100\u1172\u11A8", "\uADDD", "\u1100\u1172\u11A8" },
            new string [] { "\uADDE", "\uADDE", "\u1100\u1172\u11A9", "\uADDE", "\u1100\u1172\u11A9" },
            new string [] { "\uADDF", "\uADDF", "\u1100\u1172\u11AA", "\uADDF", "\u1100\u1172\u11AA" },
            new string [] { "\uADE0", "\uADE0", "\u1100\u1172\u11AB", "\uADE0", "\u1100\u1172\u11AB" },
            new string [] { "\uADE1", "\uADE1", "\u1100\u1172\u11AC", "\uADE1", "\u1100\u1172\u11AC" },
            new string [] { "\uADE2", "\uADE2", "\u1100\u1172\u11AD", "\uADE2", "\u1100\u1172\u11AD" },
            new string [] { "\uADE3", "\uADE3", "\u1100\u1172\u11AE", "\uADE3", "\u1100\u1172\u11AE" },
            new string [] { "\uADE4", "\uADE4", "\u1100\u1172\u11AF", "\uADE4", "\u1100\u1172\u11AF" },
            new string [] { "\uADE5", "\uADE5", "\u1100\u1172\u11B0", "\uADE5", "\u1100\u1172\u11B0" },
            new string [] { "\uADE6", "\uADE6", "\u1100\u1172\u11B1", "\uADE6", "\u1100\u1172\u11B1" },
            new string [] { "\uADE7", "\uADE7", "\u1100\u1172\u11B2", "\uADE7", "\u1100\u1172\u11B2" },
            new string [] { "\uADE8", "\uADE8", "\u1100\u1172\u11B3", "\uADE8", "\u1100\u1172\u11B3" },
            new string [] { "\uADE9", "\uADE9", "\u1100\u1172\u11B4", "\uADE9", "\u1100\u1172\u11B4" },
            new string [] { "\uADEA", "\uADEA", "\u1100\u1172\u11B5", "\uADEA", "\u1100\u1172\u11B5" },
            new string [] { "\uADEB", "\uADEB", "\u1100\u1172\u11B6", "\uADEB", "\u1100\u1172\u11B6" },
            new string [] { "\uADEC", "\uADEC", "\u1100\u1172\u11B7", "\uADEC", "\u1100\u1172\u11B7" },
            new string [] { "\uADED", "\uADED", "\u1100\u1172\u11B8", "\uADED", "\u1100\u1172\u11B8" },
            new string [] { "\uADEE", "\uADEE", "\u1100\u1172\u11B9", "\uADEE", "\u1100\u1172\u11B9" },
            new string [] { "\uADEF", "\uADEF", "\u1100\u1172\u11BA", "\uADEF", "\u1100\u1172\u11BA" },
            new string [] { "\uADF0", "\uADF0", "\u1100\u1172\u11BB", "\uADF0", "\u1100\u1172\u11BB" },
            new string [] { "\uADF1", "\uADF1", "\u1100\u1172\u11BC", "\uADF1", "\u1100\u1172\u11BC" },
            new string [] { "\uADF2", "\uADF2", "\u1100\u1172\u11BD", "\uADF2", "\u1100\u1172\u11BD" },
            new string [] { "\uADF3", "\uADF3", "\u1100\u1172\u11BE", "\uADF3", "\u1100\u1172\u11BE" },
            new string [] { "\uADF4", "\uADF4", "\u1100\u1172\u11BF", "\uADF4", "\u1100\u1172\u11BF" },
            new string [] { "\uADF5", "\uADF5", "\u1100\u1172\u11C0", "\uADF5", "\u1100\u1172\u11C0" },
            new string [] { "\uADF6", "\uADF6", "\u1100\u1172\u11C1", "\uADF6", "\u1100\u1172\u11C1" },
            new string [] { "\uADF7", "\uADF7", "\u1100\u1172\u11C2", "\uADF7", "\u1100\u1172\u11C2" },
            new string [] { "\uADF8", "\uADF8", "\u1100\u1173", "\uADF8", "\u1100\u1173" },
            new string [] { "\uADF9", "\uADF9", "\u1100\u1173\u11A8", "\uADF9", "\u1100\u1173\u11A8" },
            new string [] { "\uADFA", "\uADFA", "\u1100\u1173\u11A9", "\uADFA", "\u1100\u1173\u11A9" },
            new string [] { "\uADFB", "\uADFB", "\u1100\u1173\u11AA", "\uADFB", "\u1100\u1173\u11AA" },
            new string [] { "\uADFC", "\uADFC", "\u1100\u1173\u11AB", "\uADFC", "\u1100\u1173\u11AB" },
            new string [] { "\uADFD", "\uADFD", "\u1100\u1173\u11AC", "\uADFD", "\u1100\u1173\u11AC" },
            new string [] { "\uADFE", "\uADFE", "\u1100\u1173\u11AD", "\uADFE", "\u1100\u1173\u11AD" },
            new string [] { "\uADFF", "\uADFF", "\u1100\u1173\u11AE", "\uADFF", "\u1100\u1173\u11AE" },
            new string [] { "\uAE00", "\uAE00", "\u1100\u1173\u11AF", "\uAE00", "\u1100\u1173\u11AF" },
            new string [] { "\uAE01", "\uAE01", "\u1100\u1173\u11B0", "\uAE01", "\u1100\u1173\u11B0" },
            new string [] { "\uAE02", "\uAE02", "\u1100\u1173\u11B1", "\uAE02", "\u1100\u1173\u11B1" },
            new string [] { "\uAE03", "\uAE03", "\u1100\u1173\u11B2", "\uAE03", "\u1100\u1173\u11B2" },
            new string [] { "\uAE04", "\uAE04", "\u1100\u1173\u11B3", "\uAE04", "\u1100\u1173\u11B3" },
            new string [] { "\uAE05", "\uAE05", "\u1100\u1173\u11B4", "\uAE05", "\u1100\u1173\u11B4" },
            new string [] { "\uAE06", "\uAE06", "\u1100\u1173\u11B5", "\uAE06", "\u1100\u1173\u11B5" },
            new string [] { "\uAE07", "\uAE07", "\u1100\u1173\u11B6", "\uAE07", "\u1100\u1173\u11B6" },
            new string [] { "\uAE08", "\uAE08", "\u1100\u1173\u11B7", "\uAE08", "\u1100\u1173\u11B7" },
            new string [] { "\uAE09", "\uAE09", "\u1100\u1173\u11B8", "\uAE09", "\u1100\u1173\u11B8" },
            new string [] { "\uAE0A", "\uAE0A", "\u1100\u1173\u11B9", "\uAE0A", "\u1100\u1173\u11B9" },
            new string [] { "\uAE0B", "\uAE0B", "\u1100\u1173\u11BA", "\uAE0B", "\u1100\u1173\u11BA" },
            new string [] { "\uAE0C", "\uAE0C", "\u1100\u1173\u11BB", "\uAE0C", "\u1100\u1173\u11BB" },
            new string [] { "\uAE0D", "\uAE0D", "\u1100\u1173\u11BC", "\uAE0D", "\u1100\u1173\u11BC" },
            new string [] { "\uAE0E", "\uAE0E", "\u1100\u1173\u11BD", "\uAE0E", "\u1100\u1173\u11BD" },
            new string [] { "\uAE0F", "\uAE0F", "\u1100\u1173\u11BE", "\uAE0F", "\u1100\u1173\u11BE" },
            new string [] { "\uAE10", "\uAE10", "\u1100\u1173\u11BF", "\uAE10", "\u1100\u1173\u11BF" },
            new string [] { "\uAE11", "\uAE11", "\u1100\u1173\u11C0", "\uAE11", "\u1100\u1173\u11C0" },
            new string [] { "\uAE12", "\uAE12", "\u1100\u1173\u11C1", "\uAE12", "\u1100\u1173\u11C1" },
            new string [] { "\uAE13", "\uAE13", "\u1100\u1173\u11C2", "\uAE13", "\u1100\u1173\u11C2" },
            new string [] { "\uAE14", "\uAE14", "\u1100\u1174", "\uAE14", "\u1100\u1174" },
            new string [] { "\uAE15", "\uAE15", "\u1100\u1174\u11A8", "\uAE15", "\u1100\u1174\u11A8" },
            new string [] { "\uAE16", "\uAE16", "\u1100\u1174\u11A9", "\uAE16", "\u1100\u1174\u11A9" },
            new string [] { "\uAE17", "\uAE17", "\u1100\u1174\u11AA", "\uAE17", "\u1100\u1174\u11AA" },
            new string [] { "\uAE18", "\uAE18", "\u1100\u1174\u11AB", "\uAE18", "\u1100\u1174\u11AB" },
            new string [] { "\uAE19", "\uAE19", "\u1100\u1174\u11AC", "\uAE19", "\u1100\u1174\u11AC" },
            new string [] { "\uAE1A", "\uAE1A", "\u1100\u1174\u11AD", "\uAE1A", "\u1100\u1174\u11AD" },
            new string [] { "\uAE1B", "\uAE1B", "\u1100\u1174\u11AE", "\uAE1B", "\u1100\u1174\u11AE" },
            new string [] { "\uAE1C", "\uAE1C", "\u1100\u1174\u11AF", "\uAE1C", "\u1100\u1174\u11AF" },
            new string [] { "\uAE1D", "\uAE1D", "\u1100\u1174\u11B0", "\uAE1D", "\u1100\u1174\u11B0" },
            new string [] { "\uAE1E", "\uAE1E", "\u1100\u1174\u11B1", "\uAE1E", "\u1100\u1174\u11B1" },
            new string [] { "\uAE1F", "\uAE1F", "\u1100\u1174\u11B2", "\uAE1F", "\u1100\u1174\u11B2" },
            new string [] { "\uAE20", "\uAE20", "\u1100\u1174\u11B3", "\uAE20", "\u1100\u1174\u11B3" },
            new string [] { "\uAE21", "\uAE21", "\u1100\u1174\u11B4", "\uAE21", "\u1100\u1174\u11B4" },
            new string [] { "\uAE22", "\uAE22", "\u1100\u1174\u11B5", "\uAE22", "\u1100\u1174\u11B5" },
            new string [] { "\uAE23", "\uAE23", "\u1100\u1174\u11B6", "\uAE23", "\u1100\u1174\u11B6" },
            new string [] { "\uAE24", "\uAE24", "\u1100\u1174\u11B7", "\uAE24", "\u1100\u1174\u11B7" },
            new string [] { "\uAE25", "\uAE25", "\u1100\u1174\u11B8", "\uAE25", "\u1100\u1174\u11B8" },
            new string [] { "\uAE26", "\uAE26", "\u1100\u1174\u11B9", "\uAE26", "\u1100\u1174\u11B9" },
            new string [] { "\uAE27", "\uAE27", "\u1100\u1174\u11BA", "\uAE27", "\u1100\u1174\u11BA" },
            new string [] { "\uAE28", "\uAE28", "\u1100\u1174\u11BB", "\uAE28", "\u1100\u1174\u11BB" },
            new string [] { "\uAE29", "\uAE29", "\u1100\u1174\u11BC", "\uAE29", "\u1100\u1174\u11BC" },
            new string [] { "\uAE2A", "\uAE2A", "\u1100\u1174\u11BD", "\uAE2A", "\u1100\u1174\u11BD" },
            new string [] { "\uAE2B", "\uAE2B", "\u1100\u1174\u11BE", "\uAE2B", "\u1100\u1174\u11BE" },
            new string [] { "\uAE2C", "\uAE2C", "\u1100\u1174\u11BF", "\uAE2C", "\u1100\u1174\u11BF" },
            new string [] { "\uAE2D", "\uAE2D", "\u1100\u1174\u11C0", "\uAE2D", "\u1100\u1174\u11C0" },
            new string [] { "\uAE2E", "\uAE2E", "\u1100\u1174\u11C1", "\uAE2E", "\u1100\u1174\u11C1" },
            new string [] { "\uAE2F", "\uAE2F", "\u1100\u1174\u11C2", "\uAE2F", "\u1100\u1174\u11C2" },
            new string [] { "\uAE30", "\uAE30", "\u1100\u1175", "\uAE30", "\u1100\u1175" },
            new string [] { "\uAE31", "\uAE31", "\u1100\u1175\u11A8", "\uAE31", "\u1100\u1175\u11A8" },
            new string [] { "\uAE32", "\uAE32", "\u1100\u1175\u11A9", "\uAE32", "\u1100\u1175\u11A9" },
            new string [] { "\uAE33", "\uAE33", "\u1100\u1175\u11AA", "\uAE33", "\u1100\u1175\u11AA" },
            new string [] { "\uAE34", "\uAE34", "\u1100\u1175\u11AB", "\uAE34", "\u1100\u1175\u11AB" },
            new string [] { "\uAE35", "\uAE35", "\u1100\u1175\u11AC", "\uAE35", "\u1100\u1175\u11AC" },
            new string [] { "\uAE36", "\uAE36", "\u1100\u1175\u11AD", "\uAE36", "\u1100\u1175\u11AD" },
            new string [] { "\uAE37", "\uAE37", "\u1100\u1175\u11AE", "\uAE37", "\u1100\u1175\u11AE" },
            new string [] { "\uAE38", "\uAE38", "\u1100\u1175\u11AF", "\uAE38", "\u1100\u1175\u11AF" },
            new string [] { "\uAE39", "\uAE39", "\u1100\u1175\u11B0", "\uAE39", "\u1100\u1175\u11B0" },
            new string [] { "\uAE3A", "\uAE3A", "\u1100\u1175\u11B1", "\uAE3A", "\u1100\u1175\u11B1" },
            new string [] { "\uAE3B", "\uAE3B", "\u1100\u1175\u11B2", "\uAE3B", "\u1100\u1175\u11B2" },
            new string [] { "\uAE3C", "\uAE3C", "\u1100\u1175\u11B3", "\uAE3C", "\u1100\u1175\u11B3" },
            new string [] { "\uAE3D", "\uAE3D", "\u1100\u1175\u11B4", "\uAE3D", "\u1100\u1175\u11B4" },
            new string [] { "\uAE3E", "\uAE3E", "\u1100\u1175\u11B5", "\uAE3E", "\u1100\u1175\u11B5" },
            new string [] { "\uAE3F", "\uAE3F", "\u1100\u1175\u11B6", "\uAE3F", "\u1100\u1175\u11B6" },
            new string [] { "\uAE40", "\uAE40", "\u1100\u1175\u11B7", "\uAE40", "\u1100\u1175\u11B7" },
            new string [] { "\uAE41", "\uAE41", "\u1100\u1175\u11B8", "\uAE41", "\u1100\u1175\u11B8" },
            new string [] { "\uAE42", "\uAE42", "\u1100\u1175\u11B9", "\uAE42", "\u1100\u1175\u11B9" },
            new string [] { "\uAE43", "\uAE43", "\u1100\u1175\u11BA", "\uAE43", "\u1100\u1175\u11BA" },
            new string [] { "\uAE44", "\uAE44", "\u1100\u1175\u11BB", "\uAE44", "\u1100\u1175\u11BB" },
            new string [] { "\uAE45", "\uAE45", "\u1100\u1175\u11BC", "\uAE45", "\u1100\u1175\u11BC" },
            new string [] { "\uAE46", "\uAE46", "\u1100\u1175\u11BD", "\uAE46", "\u1100\u1175\u11BD" },
            new string [] { "\uAE47", "\uAE47", "\u1100\u1175\u11BE", "\uAE47", "\u1100\u1175\u11BE" },
            new string [] { "\uAE48", "\uAE48", "\u1100\u1175\u11BF", "\uAE48", "\u1100\u1175\u11BF" },
            new string [] { "\uAE49", "\uAE49", "\u1100\u1175\u11C0", "\uAE49", "\u1100\u1175\u11C0" },
            new string [] { "\uAE4A", "\uAE4A", "\u1100\u1175\u11C1", "\uAE4A", "\u1100\u1175\u11C1" },
            new string [] { "\uAE4B", "\uAE4B", "\u1100\u1175\u11C2", "\uAE4B", "\u1100\u1175\u11C2" },
            new string [] { "\uAE4C", "\uAE4C", "\u1101\u1161", "\uAE4C", "\u1101\u1161" },
            new string [] { "\uAE4D", "\uAE4D", "\u1101\u1161\u11A8", "\uAE4D", "\u1101\u1161\u11A8" },
            new string [] { "\uAE4E", "\uAE4E", "\u1101\u1161\u11A9", "\uAE4E", "\u1101\u1161\u11A9" },
            new string [] { "\uAE4F", "\uAE4F", "\u1101\u1161\u11AA", "\uAE4F", "\u1101\u1161\u11AA" },
            new string [] { "\uAE50", "\uAE50", "\u1101\u1161\u11AB", "\uAE50", "\u1101\u1161\u11AB" },
            new string [] { "\uAE51", "\uAE51", "\u1101\u1161\u11AC", "\uAE51", "\u1101\u1161\u11AC" },
            new string [] { "\uAE52", "\uAE52", "\u1101\u1161\u11AD", "\uAE52", "\u1101\u1161\u11AD" },
            new string [] { "\uAE53", "\uAE53", "\u1101\u1161\u11AE", "\uAE53", "\u1101\u1161\u11AE" },
            new string [] { "\uAE54", "\uAE54", "\u1101\u1161\u11AF", "\uAE54", "\u1101\u1161\u11AF" },
            new string [] { "\uAE55", "\uAE55", "\u1101\u1161\u11B0", "\uAE55", "\u1101\u1161\u11B0" },
            new string [] { "\uAE56", "\uAE56", "\u1101\u1161\u11B1", "\uAE56", "\u1101\u1161\u11B1" },
            new string [] { "\uAE57", "\uAE57", "\u1101\u1161\u11B2", "\uAE57", "\u1101\u1161\u11B2" },
            new string [] { "\uAE58", "\uAE58", "\u1101\u1161\u11B3", "\uAE58", "\u1101\u1161\u11B3" },
            new string [] { "\uAE59", "\uAE59", "\u1101\u1161\u11B4", "\uAE59", "\u1101\u1161\u11B4" },
            new string [] { "\uAE5A", "\uAE5A", "\u1101\u1161\u11B5", "\uAE5A", "\u1101\u1161\u11B5" },
            new string [] { "\uAE5B", "\uAE5B", "\u1101\u1161\u11B6", "\uAE5B", "\u1101\u1161\u11B6" },
            new string [] { "\uAE5C", "\uAE5C", "\u1101\u1161\u11B7", "\uAE5C", "\u1101\u1161\u11B7" },
            new string [] { "\uAE5D", "\uAE5D", "\u1101\u1161\u11B8", "\uAE5D", "\u1101\u1161\u11B8" },
            new string [] { "\uAE5E", "\uAE5E", "\u1101\u1161\u11B9", "\uAE5E", "\u1101\u1161\u11B9" },
            new string [] { "\uAE5F", "\uAE5F", "\u1101\u1161\u11BA", "\uAE5F", "\u1101\u1161\u11BA" },
            new string [] { "\uAE60", "\uAE60", "\u1101\u1161\u11BB", "\uAE60", "\u1101\u1161\u11BB" },
            new string [] { "\uAE61", "\uAE61", "\u1101\u1161\u11BC", "\uAE61", "\u1101\u1161\u11BC" },
            new string [] { "\uAE62", "\uAE62", "\u1101\u1161\u11BD", "\uAE62", "\u1101\u1161\u11BD" },
            new string [] { "\uAE63", "\uAE63", "\u1101\u1161\u11BE", "\uAE63", "\u1101\u1161\u11BE" },
            new string [] { "\uAE64", "\uAE64", "\u1101\u1161\u11BF", "\uAE64", "\u1101\u1161\u11BF" },
            new string [] { "\uAE65", "\uAE65", "\u1101\u1161\u11C0", "\uAE65", "\u1101\u1161\u11C0" },
            new string [] { "\uAE66", "\uAE66", "\u1101\u1161\u11C1", "\uAE66", "\u1101\u1161\u11C1" },
            new string [] { "\uAE67", "\uAE67", "\u1101\u1161\u11C2", "\uAE67", "\u1101\u1161\u11C2" },
            new string [] { "\uAE68", "\uAE68", "\u1101\u1162", "\uAE68", "\u1101\u1162" },
            new string [] { "\uAE69", "\uAE69", "\u1101\u1162\u11A8", "\uAE69", "\u1101\u1162\u11A8" },
            new string [] { "\uAE6A", "\uAE6A", "\u1101\u1162\u11A9", "\uAE6A", "\u1101\u1162\u11A9" },
            new string [] { "\uAE6B", "\uAE6B", "\u1101\u1162\u11AA", "\uAE6B", "\u1101\u1162\u11AA" },
            new string [] { "\uAE6C", "\uAE6C", "\u1101\u1162\u11AB", "\uAE6C", "\u1101\u1162\u11AB" },
            new string [] { "\uAE6D", "\uAE6D", "\u1101\u1162\u11AC", "\uAE6D", "\u1101\u1162\u11AC" },
            new string [] { "\uAE6E", "\uAE6E", "\u1101\u1162\u11AD", "\uAE6E", "\u1101\u1162\u11AD" },
            new string [] { "\uAE6F", "\uAE6F", "\u1101\u1162\u11AE", "\uAE6F", "\u1101\u1162\u11AE" },
            new string [] { "\uAE70", "\uAE70", "\u1101\u1162\u11AF", "\uAE70", "\u1101\u1162\u11AF" },
            new string [] { "\uAE71", "\uAE71", "\u1101\u1162\u11B0", "\uAE71", "\u1101\u1162\u11B0" },
            new string [] { "\uAE72", "\uAE72", "\u1101\u1162\u11B1", "\uAE72", "\u1101\u1162\u11B1" },
            new string [] { "\uAE73", "\uAE73", "\u1101\u1162\u11B2", "\uAE73", "\u1101\u1162\u11B2" },
            new string [] { "\uAE74", "\uAE74", "\u1101\u1162\u11B3", "\uAE74", "\u1101\u1162\u11B3" },
            new string [] { "\uAE75", "\uAE75", "\u1101\u1162\u11B4", "\uAE75", "\u1101\u1162\u11B4" },
            new string [] { "\uAE76", "\uAE76", "\u1101\u1162\u11B5", "\uAE76", "\u1101\u1162\u11B5" },
            new string [] { "\uAE77", "\uAE77", "\u1101\u1162\u11B6", "\uAE77", "\u1101\u1162\u11B6" },
            new string [] { "\uAE78", "\uAE78", "\u1101\u1162\u11B7", "\uAE78", "\u1101\u1162\u11B7" },
            new string [] { "\uAE79", "\uAE79", "\u1101\u1162\u11B8", "\uAE79", "\u1101\u1162\u11B8" },
            new string [] { "\uAE7A", "\uAE7A", "\u1101\u1162\u11B9", "\uAE7A", "\u1101\u1162\u11B9" },
            new string [] { "\uAE7B", "\uAE7B", "\u1101\u1162\u11BA", "\uAE7B", "\u1101\u1162\u11BA" },
            new string [] { "\uAE7C", "\uAE7C", "\u1101\u1162\u11BB", "\uAE7C", "\u1101\u1162\u11BB" },
            new string [] { "\uAE7D", "\uAE7D", "\u1101\u1162\u11BC", "\uAE7D", "\u1101\u1162\u11BC" },
            new string [] { "\uAE7E", "\uAE7E", "\u1101\u1162\u11BD", "\uAE7E", "\u1101\u1162\u11BD" },
            new string [] { "\uAE7F", "\uAE7F", "\u1101\u1162\u11BE", "\uAE7F", "\u1101\u1162\u11BE" },
            new string [] { "\uAE80", "\uAE80", "\u1101\u1162\u11BF", "\uAE80", "\u1101\u1162\u11BF" },
            new string [] { "\uAE81", "\uAE81", "\u1101\u1162\u11C0", "\uAE81", "\u1101\u1162\u11C0" },
            new string [] { "\uAE82", "\uAE82", "\u1101\u1162\u11C1", "\uAE82", "\u1101\u1162\u11C1" },
            new string [] { "\uAE83", "\uAE83", "\u1101\u1162\u11C2", "\uAE83", "\u1101\u1162\u11C2" },
            new string [] { "\uAE84", "\uAE84", "\u1101\u1163", "\uAE84", "\u1101\u1163" },
            new string [] { "\uAE85", "\uAE85", "\u1101\u1163\u11A8", "\uAE85", "\u1101\u1163\u11A8" },
            new string [] { "\uAE86", "\uAE86", "\u1101\u1163\u11A9", "\uAE86", "\u1101\u1163\u11A9" },
            new string [] { "\uAE87", "\uAE87", "\u1101\u1163\u11AA", "\uAE87", "\u1101\u1163\u11AA" },
            new string [] { "\uAE88", "\uAE88", "\u1101\u1163\u11AB", "\uAE88", "\u1101\u1163\u11AB" },
            new string [] { "\uAE89", "\uAE89", "\u1101\u1163\u11AC", "\uAE89", "\u1101\u1163\u11AC" },
            new string [] { "\uAE8A", "\uAE8A", "\u1101\u1163\u11AD", "\uAE8A", "\u1101\u1163\u11AD" },
            new string [] { "\uAE8B", "\uAE8B", "\u1101\u1163\u11AE", "\uAE8B", "\u1101\u1163\u11AE" },
            new string [] { "\uAE8C", "\uAE8C", "\u1101\u1163\u11AF", "\uAE8C", "\u1101\u1163\u11AF" },
            new string [] { "\uAE8D", "\uAE8D", "\u1101\u1163\u11B0", "\uAE8D", "\u1101\u1163\u11B0" },
            new string [] { "\uAE8E", "\uAE8E", "\u1101\u1163\u11B1", "\uAE8E", "\u1101\u1163\u11B1" },
            new string [] { "\uAE8F", "\uAE8F", "\u1101\u1163\u11B2", "\uAE8F", "\u1101\u1163\u11B2" },
            new string [] { "\uAE90", "\uAE90", "\u1101\u1163\u11B3", "\uAE90", "\u1101\u1163\u11B3" },
            new string [] { "\uAE91", "\uAE91", "\u1101\u1163\u11B4", "\uAE91", "\u1101\u1163\u11B4" },
            new string [] { "\uAE92", "\uAE92", "\u1101\u1163\u11B5", "\uAE92", "\u1101\u1163\u11B5" },
            new string [] { "\uAE93", "\uAE93", "\u1101\u1163\u11B6", "\uAE93", "\u1101\u1163\u11B6" },
            new string [] { "\uAE94", "\uAE94", "\u1101\u1163\u11B7", "\uAE94", "\u1101\u1163\u11B7" },
            new string [] { "\uAE95", "\uAE95", "\u1101\u1163\u11B8", "\uAE95", "\u1101\u1163\u11B8" },
            new string [] { "\uAE96", "\uAE96", "\u1101\u1163\u11B9", "\uAE96", "\u1101\u1163\u11B9" },
            new string [] { "\uAE97", "\uAE97", "\u1101\u1163\u11BA", "\uAE97", "\u1101\u1163\u11BA" },
            new string [] { "\uAE98", "\uAE98", "\u1101\u1163\u11BB", "\uAE98", "\u1101\u1163\u11BB" },
            new string [] { "\uAE99", "\uAE99", "\u1101\u1163\u11BC", "\uAE99", "\u1101\u1163\u11BC" },
            new string [] { "\uAE9A", "\uAE9A", "\u1101\u1163\u11BD", "\uAE9A", "\u1101\u1163\u11BD" },
            new string [] { "\uAE9B", "\uAE9B", "\u1101\u1163\u11BE", "\uAE9B", "\u1101\u1163\u11BE" },
            new string [] { "\uAE9C", "\uAE9C", "\u1101\u1163\u11BF", "\uAE9C", "\u1101\u1163\u11BF" },
            new string [] { "\uAE9D", "\uAE9D", "\u1101\u1163\u11C0", "\uAE9D", "\u1101\u1163\u11C0" },
            new string [] { "\uAE9E", "\uAE9E", "\u1101\u1163\u11C1", "\uAE9E", "\u1101\u1163\u11C1" },
            new string [] { "\uAE9F", "\uAE9F", "\u1101\u1163\u11C2", "\uAE9F", "\u1101\u1163\u11C2" },
            new string [] { "\uAEA0", "\uAEA0", "\u1101\u1164", "\uAEA0", "\u1101\u1164" },
            new string [] { "\uAEA1", "\uAEA1", "\u1101\u1164\u11A8", "\uAEA1", "\u1101\u1164\u11A8" },
            new string [] { "\uAEA2", "\uAEA2", "\u1101\u1164\u11A9", "\uAEA2", "\u1101\u1164\u11A9" },
            new string [] { "\uAEA3", "\uAEA3", "\u1101\u1164\u11AA", "\uAEA3", "\u1101\u1164\u11AA" },
            new string [] { "\uAEA4", "\uAEA4", "\u1101\u1164\u11AB", "\uAEA4", "\u1101\u1164\u11AB" },
            new string [] { "\uAEA5", "\uAEA5", "\u1101\u1164\u11AC", "\uAEA5", "\u1101\u1164\u11AC" },
            new string [] { "\uAEA6", "\uAEA6", "\u1101\u1164\u11AD", "\uAEA6", "\u1101\u1164\u11AD" },
            new string [] { "\uAEA7", "\uAEA7", "\u1101\u1164\u11AE", "\uAEA7", "\u1101\u1164\u11AE" },
            new string [] { "\uAEA8", "\uAEA8", "\u1101\u1164\u11AF", "\uAEA8", "\u1101\u1164\u11AF" },
            new string [] { "\uAEA9", "\uAEA9", "\u1101\u1164\u11B0", "\uAEA9", "\u1101\u1164\u11B0" },
            new string [] { "\uAEAA", "\uAEAA", "\u1101\u1164\u11B1", "\uAEAA", "\u1101\u1164\u11B1" },
            new string [] { "\uAEAB", "\uAEAB", "\u1101\u1164\u11B2", "\uAEAB", "\u1101\u1164\u11B2" },
            new string [] { "\uAEAC", "\uAEAC", "\u1101\u1164\u11B3", "\uAEAC", "\u1101\u1164\u11B3" },
            new string [] { "\uAEAD", "\uAEAD", "\u1101\u1164\u11B4", "\uAEAD", "\u1101\u1164\u11B4" },
            new string [] { "\uAEAE", "\uAEAE", "\u1101\u1164\u11B5", "\uAEAE", "\u1101\u1164\u11B5" },
            new string [] { "\uAEAF", "\uAEAF", "\u1101\u1164\u11B6", "\uAEAF", "\u1101\u1164\u11B6" },
            new string [] { "\uAEB0", "\uAEB0", "\u1101\u1164\u11B7", "\uAEB0", "\u1101\u1164\u11B7" },
            new string [] { "\uAEB1", "\uAEB1", "\u1101\u1164\u11B8", "\uAEB1", "\u1101\u1164\u11B8" },
            new string [] { "\uAEB2", "\uAEB2", "\u1101\u1164\u11B9", "\uAEB2", "\u1101\u1164\u11B9" },
            new string [] { "\uAEB3", "\uAEB3", "\u1101\u1164\u11BA", "\uAEB3", "\u1101\u1164\u11BA" },
            new string [] { "\uAEB4", "\uAEB4", "\u1101\u1164\u11BB", "\uAEB4", "\u1101\u1164\u11BB" },
            new string [] { "\uAEB5", "\uAEB5", "\u1101\u1164\u11BC", "\uAEB5", "\u1101\u1164\u11BC" },
            new string [] { "\uAEB6", "\uAEB6", "\u1101\u1164\u11BD", "\uAEB6", "\u1101\u1164\u11BD" },
            new string [] { "\uAEB7", "\uAEB7", "\u1101\u1164\u11BE", "\uAEB7", "\u1101\u1164\u11BE" },
            new string [] { "\uAEB8", "\uAEB8", "\u1101\u1164\u11BF", "\uAEB8", "\u1101\u1164\u11BF" },
            new string [] { "\uAEB9", "\uAEB9", "\u1101\u1164\u11C0", "\uAEB9", "\u1101\u1164\u11C0" },
            new string [] { "\uAEBA", "\uAEBA", "\u1101\u1164\u11C1", "\uAEBA", "\u1101\u1164\u11C1" },
            new string [] { "\uAEBB", "\uAEBB", "\u1101\u1164\u11C2", "\uAEBB", "\u1101\u1164\u11C2" },
            new string [] { "\uAEBC", "\uAEBC", "\u1101\u1165", "\uAEBC", "\u1101\u1165" },
            new string [] { "\uAEBD", "\uAEBD", "\u1101\u1165\u11A8", "\uAEBD", "\u1101\u1165\u11A8" },
            new string [] { "\uAEBE", "\uAEBE", "\u1101\u1165\u11A9", "\uAEBE", "\u1101\u1165\u11A9" },
            new string [] { "\uAEBF", "\uAEBF", "\u1101\u1165\u11AA", "\uAEBF", "\u1101\u1165\u11AA" },
            new string [] { "\uAEC0", "\uAEC0", "\u1101\u1165\u11AB", "\uAEC0", "\u1101\u1165\u11AB" },
            new string [] { "\uAEC1", "\uAEC1", "\u1101\u1165\u11AC", "\uAEC1", "\u1101\u1165\u11AC" },
            new string [] { "\uAEC2", "\uAEC2", "\u1101\u1165\u11AD", "\uAEC2", "\u1101\u1165\u11AD" },
            new string [] { "\uAEC3", "\uAEC3", "\u1101\u1165\u11AE", "\uAEC3", "\u1101\u1165\u11AE" },
            new string [] { "\uAEC4", "\uAEC4", "\u1101\u1165\u11AF", "\uAEC4", "\u1101\u1165\u11AF" },
            new string [] { "\uAEC5", "\uAEC5", "\u1101\u1165\u11B0", "\uAEC5", "\u1101\u1165\u11B0" },
            new string [] { "\uAEC6", "\uAEC6", "\u1101\u1165\u11B1", "\uAEC6", "\u1101\u1165\u11B1" },
            new string [] { "\uAEC7", "\uAEC7", "\u1101\u1165\u11B2", "\uAEC7", "\u1101\u1165\u11B2" },
            new string [] { "\uAEC8", "\uAEC8", "\u1101\u1165\u11B3", "\uAEC8", "\u1101\u1165\u11B3" },
            new string [] { "\uAEC9", "\uAEC9", "\u1101\u1165\u11B4", "\uAEC9", "\u1101\u1165\u11B4" },
            new string [] { "\uAECA", "\uAECA", "\u1101\u1165\u11B5", "\uAECA", "\u1101\u1165\u11B5" },
            new string [] { "\uAECB", "\uAECB", "\u1101\u1165\u11B6", "\uAECB", "\u1101\u1165\u11B6" },
            new string [] { "\uAECC", "\uAECC", "\u1101\u1165\u11B7", "\uAECC", "\u1101\u1165\u11B7" },
            new string [] { "\uAECD", "\uAECD", "\u1101\u1165\u11B8", "\uAECD", "\u1101\u1165\u11B8" },
            new string [] { "\uAECE", "\uAECE", "\u1101\u1165\u11B9", "\uAECE", "\u1101\u1165\u11B9" },
            new string [] { "\uAECF", "\uAECF", "\u1101\u1165\u11BA", "\uAECF", "\u1101\u1165\u11BA" },
            new string [] { "\uAED0", "\uAED0", "\u1101\u1165\u11BB", "\uAED0", "\u1101\u1165\u11BB" },
            new string [] { "\uAED1", "\uAED1", "\u1101\u1165\u11BC", "\uAED1", "\u1101\u1165\u11BC" },
            new string [] { "\uAED2", "\uAED2", "\u1101\u1165\u11BD", "\uAED2", "\u1101\u1165\u11BD" },
            new string [] { "\uAED3", "\uAED3", "\u1101\u1165\u11BE", "\uAED3", "\u1101\u1165\u11BE" },
            new string [] { "\uAED4", "\uAED4", "\u1101\u1165\u11BF", "\uAED4", "\u1101\u1165\u11BF" },
            new string [] { "\uAED5", "\uAED5", "\u1101\u1165\u11C0", "\uAED5", "\u1101\u1165\u11C0" },
            new string [] { "\uAED6", "\uAED6", "\u1101\u1165\u11C1", "\uAED6", "\u1101\u1165\u11C1" },
            new string [] { "\uAED7", "\uAED7", "\u1101\u1165\u11C2", "\uAED7", "\u1101\u1165\u11C2" },
            new string [] { "\uAED8", "\uAED8", "\u1101\u1166", "\uAED8", "\u1101\u1166" },
            new string [] { "\uAED9", "\uAED9", "\u1101\u1166\u11A8", "\uAED9", "\u1101\u1166\u11A8" },
            new string [] { "\uAEDA", "\uAEDA", "\u1101\u1166\u11A9", "\uAEDA", "\u1101\u1166\u11A9" },
            new string [] { "\uAEDB", "\uAEDB", "\u1101\u1166\u11AA", "\uAEDB", "\u1101\u1166\u11AA" },
            new string [] { "\uAEDC", "\uAEDC", "\u1101\u1166\u11AB", "\uAEDC", "\u1101\u1166\u11AB" },
            new string [] { "\uAEDD", "\uAEDD", "\u1101\u1166\u11AC", "\uAEDD", "\u1101\u1166\u11AC" },
            new string [] { "\uAEDE", "\uAEDE", "\u1101\u1166\u11AD", "\uAEDE", "\u1101\u1166\u11AD" },
            new string [] { "\uAEDF", "\uAEDF", "\u1101\u1166\u11AE", "\uAEDF", "\u1101\u1166\u11AE" },
            new string [] { "\uAEE0", "\uAEE0", "\u1101\u1166\u11AF", "\uAEE0", "\u1101\u1166\u11AF" },
            new string [] { "\uAEE1", "\uAEE1", "\u1101\u1166\u11B0", "\uAEE1", "\u1101\u1166\u11B0" },
            new string [] { "\uAEE2", "\uAEE2", "\u1101\u1166\u11B1", "\uAEE2", "\u1101\u1166\u11B1" },
            new string [] { "\uAEE3", "\uAEE3", "\u1101\u1166\u11B2", "\uAEE3", "\u1101\u1166\u11B2" },
            new string [] { "\uAEE4", "\uAEE4", "\u1101\u1166\u11B3", "\uAEE4", "\u1101\u1166\u11B3" },
            new string [] { "\uAEE5", "\uAEE5", "\u1101\u1166\u11B4", "\uAEE5", "\u1101\u1166\u11B4" },
            new string [] { "\uAEE6", "\uAEE6", "\u1101\u1166\u11B5", "\uAEE6", "\u1101\u1166\u11B5" },
            new string [] { "\uAEE7", "\uAEE7", "\u1101\u1166\u11B6", "\uAEE7", "\u1101\u1166\u11B6" },
            new string [] { "\uAEE8", "\uAEE8", "\u1101\u1166\u11B7", "\uAEE8", "\u1101\u1166\u11B7" },
            new string [] { "\uAEE9", "\uAEE9", "\u1101\u1166\u11B8", "\uAEE9", "\u1101\u1166\u11B8" },
            new string [] { "\uAEEA", "\uAEEA", "\u1101\u1166\u11B9", "\uAEEA", "\u1101\u1166\u11B9" },
            new string [] { "\uAEEB", "\uAEEB", "\u1101\u1166\u11BA", "\uAEEB", "\u1101\u1166\u11BA" },
            new string [] { "\uAEEC", "\uAEEC", "\u1101\u1166\u11BB", "\uAEEC", "\u1101\u1166\u11BB" },
            new string [] { "\uAEED", "\uAEED", "\u1101\u1166\u11BC", "\uAEED", "\u1101\u1166\u11BC" },
            new string [] { "\uAEEE", "\uAEEE", "\u1101\u1166\u11BD", "\uAEEE", "\u1101\u1166\u11BD" },
            new string [] { "\uAEEF", "\uAEEF", "\u1101\u1166\u11BE", "\uAEEF", "\u1101\u1166\u11BE" },
            new string [] { "\uAEF0", "\uAEF0", "\u1101\u1166\u11BF", "\uAEF0", "\u1101\u1166\u11BF" },
            new string [] { "\uAEF1", "\uAEF1", "\u1101\u1166\u11C0", "\uAEF1", "\u1101\u1166\u11C0" },
            new string [] { "\uAEF2", "\uAEF2", "\u1101\u1166\u11C1", "\uAEF2", "\u1101\u1166\u11C1" },
            new string [] { "\uAEF3", "\uAEF3", "\u1101\u1166\u11C2", "\uAEF3", "\u1101\u1166\u11C2" },
            new string [] { "\uAEF4", "\uAEF4", "\u1101\u1167", "\uAEF4", "\u1101\u1167" },
            new string [] { "\uAEF5", "\uAEF5", "\u1101\u1167\u11A8", "\uAEF5", "\u1101\u1167\u11A8" },
            new string [] { "\uAEF6", "\uAEF6", "\u1101\u1167\u11A9", "\uAEF6", "\u1101\u1167\u11A9" },
            new string [] { "\uAEF7", "\uAEF7", "\u1101\u1167\u11AA", "\uAEF7", "\u1101\u1167\u11AA" },
            new string [] { "\uAEF8", "\uAEF8", "\u1101\u1167\u11AB", "\uAEF8", "\u1101\u1167\u11AB" },
            new string [] { "\uAEF9", "\uAEF9", "\u1101\u1167\u11AC", "\uAEF9", "\u1101\u1167\u11AC" },
            new string [] { "\uAEFA", "\uAEFA", "\u1101\u1167\u11AD", "\uAEFA", "\u1101\u1167\u11AD" },
            new string [] { "\uAEFB", "\uAEFB", "\u1101\u1167\u11AE", "\uAEFB", "\u1101\u1167\u11AE" },
            new string [] { "\uAEFC", "\uAEFC", "\u1101\u1167\u11AF", "\uAEFC", "\u1101\u1167\u11AF" },
            new string [] { "\uAEFD", "\uAEFD", "\u1101\u1167\u11B0", "\uAEFD", "\u1101\u1167\u11B0" },
            new string [] { "\uAEFE", "\uAEFE", "\u1101\u1167\u11B1", "\uAEFE", "\u1101\u1167\u11B1" },
            new string [] { "\uAEFF", "\uAEFF", "\u1101\u1167\u11B2", "\uAEFF", "\u1101\u1167\u11B2" },
            new string [] { "\uAF00", "\uAF00", "\u1101\u1167\u11B3", "\uAF00", "\u1101\u1167\u11B3" },
            new string [] { "\uAF01", "\uAF01", "\u1101\u1167\u11B4", "\uAF01", "\u1101\u1167\u11B4" },
            new string [] { "\uAF02", "\uAF02", "\u1101\u1167\u11B5", "\uAF02", "\u1101\u1167\u11B5" },
            new string [] { "\uAF03", "\uAF03", "\u1101\u1167\u11B6", "\uAF03", "\u1101\u1167\u11B6" },
            new string [] { "\uAF04", "\uAF04", "\u1101\u1167\u11B7", "\uAF04", "\u1101\u1167\u11B7" },
            new string [] { "\uAF05", "\uAF05", "\u1101\u1167\u11B8", "\uAF05", "\u1101\u1167\u11B8" },
            new string [] { "\uAF06", "\uAF06", "\u1101\u1167\u11B9", "\uAF06", "\u1101\u1167\u11B9" },
            new string [] { "\uAF07", "\uAF07", "\u1101\u1167\u11BA", "\uAF07", "\u1101\u1167\u11BA" },
            new string [] { "\uAF08", "\uAF08", "\u1101\u1167\u11BB", "\uAF08", "\u1101\u1167\u11BB" },
            new string [] { "\uAF09", "\uAF09", "\u1101\u1167\u11BC", "\uAF09", "\u1101\u1167\u11BC" },
            new string [] { "\uAF0A", "\uAF0A", "\u1101\u1167\u11BD", "\uAF0A", "\u1101\u1167\u11BD" },
            new string [] { "\uAF0B", "\uAF0B", "\u1101\u1167\u11BE", "\uAF0B", "\u1101\u1167\u11BE" },
            new string [] { "\uAF0C", "\uAF0C", "\u1101\u1167\u11BF", "\uAF0C", "\u1101\u1167\u11BF" },
            new string [] { "\uAF0D", "\uAF0D", "\u1101\u1167\u11C0", "\uAF0D", "\u1101\u1167\u11C0" },
            new string [] { "\uAF0E", "\uAF0E", "\u1101\u1167\u11C1", "\uAF0E", "\u1101\u1167\u11C1" },
            new string [] { "\uAF0F", "\uAF0F", "\u1101\u1167\u11C2", "\uAF0F", "\u1101\u1167\u11C2" },
            new string [] { "\uAF10", "\uAF10", "\u1101\u1168", "\uAF10", "\u1101\u1168" },
            new string [] { "\uAF11", "\uAF11", "\u1101\u1168\u11A8", "\uAF11", "\u1101\u1168\u11A8" },
            new string [] { "\uAF12", "\uAF12", "\u1101\u1168\u11A9", "\uAF12", "\u1101\u1168\u11A9" },
            new string [] { "\uAF13", "\uAF13", "\u1101\u1168\u11AA", "\uAF13", "\u1101\u1168\u11AA" },
            new string [] { "\uAF14", "\uAF14", "\u1101\u1168\u11AB", "\uAF14", "\u1101\u1168\u11AB" },
            new string [] { "\uAF15", "\uAF15", "\u1101\u1168\u11AC", "\uAF15", "\u1101\u1168\u11AC" },
            new string [] { "\uAF16", "\uAF16", "\u1101\u1168\u11AD", "\uAF16", "\u1101\u1168\u11AD" },
            new string [] { "\uAF17", "\uAF17", "\u1101\u1168\u11AE", "\uAF17", "\u1101\u1168\u11AE" },
            new string [] { "\uAF18", "\uAF18", "\u1101\u1168\u11AF", "\uAF18", "\u1101\u1168\u11AF" },
            new string [] { "\uAF19", "\uAF19", "\u1101\u1168\u11B0", "\uAF19", "\u1101\u1168\u11B0" },
            new string [] { "\uAF1A", "\uAF1A", "\u1101\u1168\u11B1", "\uAF1A", "\u1101\u1168\u11B1" },
            new string [] { "\uAF1B", "\uAF1B", "\u1101\u1168\u11B2", "\uAF1B", "\u1101\u1168\u11B2" },
            new string [] { "\uAF1C", "\uAF1C", "\u1101\u1168\u11B3", "\uAF1C", "\u1101\u1168\u11B3" },
            new string [] { "\uAF1D", "\uAF1D", "\u1101\u1168\u11B4", "\uAF1D", "\u1101\u1168\u11B4" },
            new string [] { "\uAF1E", "\uAF1E", "\u1101\u1168\u11B5", "\uAF1E", "\u1101\u1168\u11B5" },
            new string [] { "\uAF1F", "\uAF1F", "\u1101\u1168\u11B6", "\uAF1F", "\u1101\u1168\u11B6" },
            new string [] { "\uAF20", "\uAF20", "\u1101\u1168\u11B7", "\uAF20", "\u1101\u1168\u11B7" },
            new string [] { "\uAF21", "\uAF21", "\u1101\u1168\u11B8", "\uAF21", "\u1101\u1168\u11B8" },
            new string [] { "\uAF22", "\uAF22", "\u1101\u1168\u11B9", "\uAF22", "\u1101\u1168\u11B9" },
            new string [] { "\uAF23", "\uAF23", "\u1101\u1168\u11BA", "\uAF23", "\u1101\u1168\u11BA" },
            new string [] { "\uAF24", "\uAF24", "\u1101\u1168\u11BB", "\uAF24", "\u1101\u1168\u11BB" },
            new string [] { "\uAF25", "\uAF25", "\u1101\u1168\u11BC", "\uAF25", "\u1101\u1168\u11BC" },
            new string [] { "\uAF26", "\uAF26", "\u1101\u1168\u11BD", "\uAF26", "\u1101\u1168\u11BD" },
            new string [] { "\uAF27", "\uAF27", "\u1101\u1168\u11BE", "\uAF27", "\u1101\u1168\u11BE" },
            new string [] { "\uAF28", "\uAF28", "\u1101\u1168\u11BF", "\uAF28", "\u1101\u1168\u11BF" },
            new string [] { "\uAF29", "\uAF29", "\u1101\u1168\u11C0", "\uAF29", "\u1101\u1168\u11C0" },
            new string [] { "\uAF2A", "\uAF2A", "\u1101\u1168\u11C1", "\uAF2A", "\u1101\u1168\u11C1" },
            new string [] { "\uAF2B", "\uAF2B", "\u1101\u1168\u11C2", "\uAF2B", "\u1101\u1168\u11C2" },
            new string [] { "\uAF2C", "\uAF2C", "\u1101\u1169", "\uAF2C", "\u1101\u1169" },
            new string [] { "\uAF2D", "\uAF2D", "\u1101\u1169\u11A8", "\uAF2D", "\u1101\u1169\u11A8" },
            new string [] { "\uAF2E", "\uAF2E", "\u1101\u1169\u11A9", "\uAF2E", "\u1101\u1169\u11A9" },
            new string [] { "\uAF2F", "\uAF2F", "\u1101\u1169\u11AA", "\uAF2F", "\u1101\u1169\u11AA" },
            new string [] { "\uAF30", "\uAF30", "\u1101\u1169\u11AB", "\uAF30", "\u1101\u1169\u11AB" },
            new string [] { "\uAF31", "\uAF31", "\u1101\u1169\u11AC", "\uAF31", "\u1101\u1169\u11AC" },
            new string [] { "\uAF32", "\uAF32", "\u1101\u1169\u11AD", "\uAF32", "\u1101\u1169\u11AD" },
            new string [] { "\uAF33", "\uAF33", "\u1101\u1169\u11AE", "\uAF33", "\u1101\u1169\u11AE" },
            new string [] { "\uAF34", "\uAF34", "\u1101\u1169\u11AF", "\uAF34", "\u1101\u1169\u11AF" },
            new string [] { "\uAF35", "\uAF35", "\u1101\u1169\u11B0", "\uAF35", "\u1101\u1169\u11B0" },
            new string [] { "\uAF36", "\uAF36", "\u1101\u1169\u11B1", "\uAF36", "\u1101\u1169\u11B1" },
            new string [] { "\uAF37", "\uAF37", "\u1101\u1169\u11B2", "\uAF37", "\u1101\u1169\u11B2" },
            new string [] { "\uAF38", "\uAF38", "\u1101\u1169\u11B3", "\uAF38", "\u1101\u1169\u11B3" },
            new string [] { "\uAF39", "\uAF39", "\u1101\u1169\u11B4", "\uAF39", "\u1101\u1169\u11B4" },
            new string [] { "\uAF3A", "\uAF3A", "\u1101\u1169\u11B5", "\uAF3A", "\u1101\u1169\u11B5" },
            new string [] { "\uAF3B", "\uAF3B", "\u1101\u1169\u11B6", "\uAF3B", "\u1101\u1169\u11B6" },
            new string [] { "\uAF3C", "\uAF3C", "\u1101\u1169\u11B7", "\uAF3C", "\u1101\u1169\u11B7" },
            new string [] { "\uAF3D", "\uAF3D", "\u1101\u1169\u11B8", "\uAF3D", "\u1101\u1169\u11B8" },
            new string [] { "\uAF3E", "\uAF3E", "\u1101\u1169\u11B9", "\uAF3E", "\u1101\u1169\u11B9" },
            new string [] { "\uAF3F", "\uAF3F", "\u1101\u1169\u11BA", "\uAF3F", "\u1101\u1169\u11BA" },
            new string [] { "\uAF40", "\uAF40", "\u1101\u1169\u11BB", "\uAF40", "\u1101\u1169\u11BB" },
            new string [] { "\uAF41", "\uAF41", "\u1101\u1169\u11BC", "\uAF41", "\u1101\u1169\u11BC" },
            new string [] { "\uAF42", "\uAF42", "\u1101\u1169\u11BD", "\uAF42", "\u1101\u1169\u11BD" },
            new string [] { "\uAF43", "\uAF43", "\u1101\u1169\u11BE", "\uAF43", "\u1101\u1169\u11BE" },
            new string [] { "\uAF44", "\uAF44", "\u1101\u1169\u11BF", "\uAF44", "\u1101\u1169\u11BF" },
            new string [] { "\uAF45", "\uAF45", "\u1101\u1169\u11C0", "\uAF45", "\u1101\u1169\u11C0" },
            new string [] { "\uAF46", "\uAF46", "\u1101\u1169\u11C1", "\uAF46", "\u1101\u1169\u11C1" },
            new string [] { "\uAF47", "\uAF47", "\u1101\u1169\u11C2", "\uAF47", "\u1101\u1169\u11C2" },
            new string [] { "\uAF48", "\uAF48", "\u1101\u116A", "\uAF48", "\u1101\u116A" },
            new string [] { "\uAF49", "\uAF49", "\u1101\u116A\u11A8", "\uAF49", "\u1101\u116A\u11A8" },
            new string [] { "\uAF4A", "\uAF4A", "\u1101\u116A\u11A9", "\uAF4A", "\u1101\u116A\u11A9" },
            new string [] { "\uAF4B", "\uAF4B", "\u1101\u116A\u11AA", "\uAF4B", "\u1101\u116A\u11AA" },
            new string [] { "\uAF4C", "\uAF4C", "\u1101\u116A\u11AB", "\uAF4C", "\u1101\u116A\u11AB" },
            new string [] { "\uAF4D", "\uAF4D", "\u1101\u116A\u11AC", "\uAF4D", "\u1101\u116A\u11AC" },
            new string [] { "\uAF4E", "\uAF4E", "\u1101\u116A\u11AD", "\uAF4E", "\u1101\u116A\u11AD" },
            new string [] { "\uAF4F", "\uAF4F", "\u1101\u116A\u11AE", "\uAF4F", "\u1101\u116A\u11AE" },
            new string [] { "\uAF50", "\uAF50", "\u1101\u116A\u11AF", "\uAF50", "\u1101\u116A\u11AF" },
            new string [] { "\uAF51", "\uAF51", "\u1101\u116A\u11B0", "\uAF51", "\u1101\u116A\u11B0" },
            new string [] { "\uAF52", "\uAF52", "\u1101\u116A\u11B1", "\uAF52", "\u1101\u116A\u11B1" },
            new string [] { "\uAF53", "\uAF53", "\u1101\u116A\u11B2", "\uAF53", "\u1101\u116A\u11B2" },
            new string [] { "\uAF54", "\uAF54", "\u1101\u116A\u11B3", "\uAF54", "\u1101\u116A\u11B3" },
            new string [] { "\uAF55", "\uAF55", "\u1101\u116A\u11B4", "\uAF55", "\u1101\u116A\u11B4" },
            new string [] { "\uAF56", "\uAF56", "\u1101\u116A\u11B5", "\uAF56", "\u1101\u116A\u11B5" },
            new string [] { "\uAF57", "\uAF57", "\u1101\u116A\u11B6", "\uAF57", "\u1101\u116A\u11B6" },
            new string [] { "\uAF58", "\uAF58", "\u1101\u116A\u11B7", "\uAF58", "\u1101\u116A\u11B7" },
            new string [] { "\uAF59", "\uAF59", "\u1101\u116A\u11B8", "\uAF59", "\u1101\u116A\u11B8" },
            new string [] { "\uAF5A", "\uAF5A", "\u1101\u116A\u11B9", "\uAF5A", "\u1101\u116A\u11B9" },
            new string [] { "\uAF5B", "\uAF5B", "\u1101\u116A\u11BA", "\uAF5B", "\u1101\u116A\u11BA" },
            new string [] { "\uAF5C", "\uAF5C", "\u1101\u116A\u11BB", "\uAF5C", "\u1101\u116A\u11BB" },
            new string [] { "\uAF5D", "\uAF5D", "\u1101\u116A\u11BC", "\uAF5D", "\u1101\u116A\u11BC" },
            new string [] { "\uAF5E", "\uAF5E", "\u1101\u116A\u11BD", "\uAF5E", "\u1101\u116A\u11BD" },
            new string [] { "\uAF5F", "\uAF5F", "\u1101\u116A\u11BE", "\uAF5F", "\u1101\u116A\u11BE" },
            new string [] { "\uAF60", "\uAF60", "\u1101\u116A\u11BF", "\uAF60", "\u1101\u116A\u11BF" },
            new string [] { "\uAF61", "\uAF61", "\u1101\u116A\u11C0", "\uAF61", "\u1101\u116A\u11C0" },
            new string [] { "\uAF62", "\uAF62", "\u1101\u116A\u11C1", "\uAF62", "\u1101\u116A\u11C1" },
            new string [] { "\uAF63", "\uAF63", "\u1101\u116A\u11C2", "\uAF63", "\u1101\u116A\u11C2" },
            new string [] { "\uAF64", "\uAF64", "\u1101\u116B", "\uAF64", "\u1101\u116B" },
            new string [] { "\uAF65", "\uAF65", "\u1101\u116B\u11A8", "\uAF65", "\u1101\u116B\u11A8" },
            new string [] { "\uAF66", "\uAF66", "\u1101\u116B\u11A9", "\uAF66", "\u1101\u116B\u11A9" },
            new string [] { "\uAF67", "\uAF67", "\u1101\u116B\u11AA", "\uAF67", "\u1101\u116B\u11AA" },
            new string [] { "\uAF68", "\uAF68", "\u1101\u116B\u11AB", "\uAF68", "\u1101\u116B\u11AB" },
            new string [] { "\uAF69", "\uAF69", "\u1101\u116B\u11AC", "\uAF69", "\u1101\u116B\u11AC" },
            new string [] { "\uAF6A", "\uAF6A", "\u1101\u116B\u11AD", "\uAF6A", "\u1101\u116B\u11AD" },
            new string [] { "\uAF6B", "\uAF6B", "\u1101\u116B\u11AE", "\uAF6B", "\u1101\u116B\u11AE" },
            new string [] { "\uAF6C", "\uAF6C", "\u1101\u116B\u11AF", "\uAF6C", "\u1101\u116B\u11AF" },
            new string [] { "\uAF6D", "\uAF6D", "\u1101\u116B\u11B0", "\uAF6D", "\u1101\u116B\u11B0" },
            new string [] { "\uAF6E", "\uAF6E", "\u1101\u116B\u11B1", "\uAF6E", "\u1101\u116B\u11B1" },
            new string [] { "\uAF6F", "\uAF6F", "\u1101\u116B\u11B2", "\uAF6F", "\u1101\u116B\u11B2" },
            new string [] { "\uAF70", "\uAF70", "\u1101\u116B\u11B3", "\uAF70", "\u1101\u116B\u11B3" },
            new string [] { "\uAF71", "\uAF71", "\u1101\u116B\u11B4", "\uAF71", "\u1101\u116B\u11B4" },
            new string [] { "\uAF72", "\uAF72", "\u1101\u116B\u11B5", "\uAF72", "\u1101\u116B\u11B5" },
            new string [] { "\uAF73", "\uAF73", "\u1101\u116B\u11B6", "\uAF73", "\u1101\u116B\u11B6" },
            new string [] { "\uAF74", "\uAF74", "\u1101\u116B\u11B7", "\uAF74", "\u1101\u116B\u11B7" },
            new string [] { "\uAF75", "\uAF75", "\u1101\u116B\u11B8", "\uAF75", "\u1101\u116B\u11B8" },
            new string [] { "\uAF76", "\uAF76", "\u1101\u116B\u11B9", "\uAF76", "\u1101\u116B\u11B9" },
            new string [] { "\uAF77", "\uAF77", "\u1101\u116B\u11BA", "\uAF77", "\u1101\u116B\u11BA" },
            new string [] { "\uAF78", "\uAF78", "\u1101\u116B\u11BB", "\uAF78", "\u1101\u116B\u11BB" },
            new string [] { "\uAF79", "\uAF79", "\u1101\u116B\u11BC", "\uAF79", "\u1101\u116B\u11BC" },
            new string [] { "\uAF7A", "\uAF7A", "\u1101\u116B\u11BD", "\uAF7A", "\u1101\u116B\u11BD" },
            new string [] { "\uAF7B", "\uAF7B", "\u1101\u116B\u11BE", "\uAF7B", "\u1101\u116B\u11BE" },
            new string [] { "\uAF7C", "\uAF7C", "\u1101\u116B\u11BF", "\uAF7C", "\u1101\u116B\u11BF" },
            new string [] { "\uAF7D", "\uAF7D", "\u1101\u116B\u11C0", "\uAF7D", "\u1101\u116B\u11C0" },
            new string [] { "\uAF7E", "\uAF7E", "\u1101\u116B\u11C1", "\uAF7E", "\u1101\u116B\u11C1" },
            new string [] { "\uAF7F", "\uAF7F", "\u1101\u116B\u11C2", "\uAF7F", "\u1101\u116B\u11C2" },
            new string [] { "\uAF80", "\uAF80", "\u1101\u116C", "\uAF80", "\u1101\u116C" },
            new string [] { "\uAF81", "\uAF81", "\u1101\u116C\u11A8", "\uAF81", "\u1101\u116C\u11A8" },
            new string [] { "\uAF82", "\uAF82", "\u1101\u116C\u11A9", "\uAF82", "\u1101\u116C\u11A9" },
            new string [] { "\uAF83", "\uAF83", "\u1101\u116C\u11AA", "\uAF83", "\u1101\u116C\u11AA" },
            new string [] { "\uAF84", "\uAF84", "\u1101\u116C\u11AB", "\uAF84", "\u1101\u116C\u11AB" },
            new string [] { "\uAF85", "\uAF85", "\u1101\u116C\u11AC", "\uAF85", "\u1101\u116C\u11AC" },
            new string [] { "\uAF86", "\uAF86", "\u1101\u116C\u11AD", "\uAF86", "\u1101\u116C\u11AD" },
            new string [] { "\uAF87", "\uAF87", "\u1101\u116C\u11AE", "\uAF87", "\u1101\u116C\u11AE" },
            new string [] { "\uAF88", "\uAF88", "\u1101\u116C\u11AF", "\uAF88", "\u1101\u116C\u11AF" },
            new string [] { "\uAF89", "\uAF89", "\u1101\u116C\u11B0", "\uAF89", "\u1101\u116C\u11B0" },
            new string [] { "\uAF8A", "\uAF8A", "\u1101\u116C\u11B1", "\uAF8A", "\u1101\u116C\u11B1" },
            new string [] { "\uAF8B", "\uAF8B", "\u1101\u116C\u11B2", "\uAF8B", "\u1101\u116C\u11B2" },
            new string [] { "\uAF8C", "\uAF8C", "\u1101\u116C\u11B3", "\uAF8C", "\u1101\u116C\u11B3" },
            new string [] { "\uAF8D", "\uAF8D", "\u1101\u116C\u11B4", "\uAF8D", "\u1101\u116C\u11B4" },
            new string [] { "\uAF8E", "\uAF8E", "\u1101\u116C\u11B5", "\uAF8E", "\u1101\u116C\u11B5" },
            new string [] { "\uAF8F", "\uAF8F", "\u1101\u116C\u11B6", "\uAF8F", "\u1101\u116C\u11B6" },
            new string [] { "\uAF90", "\uAF90", "\u1101\u116C\u11B7", "\uAF90", "\u1101\u116C\u11B7" },
            new string [] { "\uAF91", "\uAF91", "\u1101\u116C\u11B8", "\uAF91", "\u1101\u116C\u11B8" },
            new string [] { "\uAF92", "\uAF92", "\u1101\u116C\u11B9", "\uAF92", "\u1101\u116C\u11B9" },
            new string [] { "\uAF93", "\uAF93", "\u1101\u116C\u11BA", "\uAF93", "\u1101\u116C\u11BA" },
            new string [] { "\uAF94", "\uAF94", "\u1101\u116C\u11BB", "\uAF94", "\u1101\u116C\u11BB" },
            new string [] { "\uAF95", "\uAF95", "\u1101\u116C\u11BC", "\uAF95", "\u1101\u116C\u11BC" },
            new string [] { "\uAF96", "\uAF96", "\u1101\u116C\u11BD", "\uAF96", "\u1101\u116C\u11BD" },
            new string [] { "\uAF97", "\uAF97", "\u1101\u116C\u11BE", "\uAF97", "\u1101\u116C\u11BE" },
            new string [] { "\uAF98", "\uAF98", "\u1101\u116C\u11BF", "\uAF98", "\u1101\u116C\u11BF" },
            new string [] { "\uAF99", "\uAF99", "\u1101\u116C\u11C0", "\uAF99", "\u1101\u116C\u11C0" },
            new string [] { "\uAF9A", "\uAF9A", "\u1101\u116C\u11C1", "\uAF9A", "\u1101\u116C\u11C1" },
            new string [] { "\uAF9B", "\uAF9B", "\u1101\u116C\u11C2", "\uAF9B", "\u1101\u116C\u11C2" },
            new string [] { "\uAF9C", "\uAF9C", "\u1101\u116D", "\uAF9C", "\u1101\u116D" },
            new string [] { "\uAF9D", "\uAF9D", "\u1101\u116D\u11A8", "\uAF9D", "\u1101\u116D\u11A8" },
            new string [] { "\uAF9E", "\uAF9E", "\u1101\u116D\u11A9", "\uAF9E", "\u1101\u116D\u11A9" },
            new string [] { "\uAF9F", "\uAF9F", "\u1101\u116D\u11AA", "\uAF9F", "\u1101\u116D\u11AA" },
            new string [] { "\uAFA0", "\uAFA0", "\u1101\u116D\u11AB", "\uAFA0", "\u1101\u116D\u11AB" },
            new string [] { "\uAFA1", "\uAFA1", "\u1101\u116D\u11AC", "\uAFA1", "\u1101\u116D\u11AC" },
            new string [] { "\uAFA2", "\uAFA2", "\u1101\u116D\u11AD", "\uAFA2", "\u1101\u116D\u11AD" },
            new string [] { "\uAFA3", "\uAFA3", "\u1101\u116D\u11AE", "\uAFA3", "\u1101\u116D\u11AE" },
            new string [] { "\uAFA4", "\uAFA4", "\u1101\u116D\u11AF", "\uAFA4", "\u1101\u116D\u11AF" },
            new string [] { "\uAFA5", "\uAFA5", "\u1101\u116D\u11B0", "\uAFA5", "\u1101\u116D\u11B0" },
            new string [] { "\uAFA6", "\uAFA6", "\u1101\u116D\u11B1", "\uAFA6", "\u1101\u116D\u11B1" },
            new string [] { "\uAFA7", "\uAFA7", "\u1101\u116D\u11B2", "\uAFA7", "\u1101\u116D\u11B2" },
            new string [] { "\uAFA8", "\uAFA8", "\u1101\u116D\u11B3", "\uAFA8", "\u1101\u116D\u11B3" },
            new string [] { "\uAFA9", "\uAFA9", "\u1101\u116D\u11B4", "\uAFA9", "\u1101\u116D\u11B4" },
            new string [] { "\uAFAA", "\uAFAA", "\u1101\u116D\u11B5", "\uAFAA", "\u1101\u116D\u11B5" },
            new string [] { "\uAFAB", "\uAFAB", "\u1101\u116D\u11B6", "\uAFAB", "\u1101\u116D\u11B6" },
            new string [] { "\uAFAC", "\uAFAC", "\u1101\u116D\u11B7", "\uAFAC", "\u1101\u116D\u11B7" },
            new string [] { "\uAFAD", "\uAFAD", "\u1101\u116D\u11B8", "\uAFAD", "\u1101\u116D\u11B8" },
            new string [] { "\uAFAE", "\uAFAE", "\u1101\u116D\u11B9", "\uAFAE", "\u1101\u116D\u11B9" },
            new string [] { "\uAFAF", "\uAFAF", "\u1101\u116D\u11BA", "\uAFAF", "\u1101\u116D\u11BA" },
            new string [] { "\uAFB0", "\uAFB0", "\u1101\u116D\u11BB", "\uAFB0", "\u1101\u116D\u11BB" },
            new string [] { "\uAFB1", "\uAFB1", "\u1101\u116D\u11BC", "\uAFB1", "\u1101\u116D\u11BC" },
            new string [] { "\uAFB2", "\uAFB2", "\u1101\u116D\u11BD", "\uAFB2", "\u1101\u116D\u11BD" },
            new string [] { "\uAFB3", "\uAFB3", "\u1101\u116D\u11BE", "\uAFB3", "\u1101\u116D\u11BE" },
            new string [] { "\uAFB4", "\uAFB4", "\u1101\u116D\u11BF", "\uAFB4", "\u1101\u116D\u11BF" },
            new string [] { "\uAFB5", "\uAFB5", "\u1101\u116D\u11C0", "\uAFB5", "\u1101\u116D\u11C0" },
            new string [] { "\uAFB6", "\uAFB6", "\u1101\u116D\u11C1", "\uAFB6", "\u1101\u116D\u11C1" },
            new string [] { "\uAFB7", "\uAFB7", "\u1101\u116D\u11C2", "\uAFB7", "\u1101\u116D\u11C2" },
            new string [] { "\uAFB8", "\uAFB8", "\u1101\u116E", "\uAFB8", "\u1101\u116E" },
            new string [] { "\uAFB9", "\uAFB9", "\u1101\u116E\u11A8", "\uAFB9", "\u1101\u116E\u11A8" },
            new string [] { "\uAFBA", "\uAFBA", "\u1101\u116E\u11A9", "\uAFBA", "\u1101\u116E\u11A9" },
            new string [] { "\uAFBB", "\uAFBB", "\u1101\u116E\u11AA", "\uAFBB", "\u1101\u116E\u11AA" },
            new string [] { "\uAFBC", "\uAFBC", "\u1101\u116E\u11AB", "\uAFBC", "\u1101\u116E\u11AB" },
            new string [] { "\uAFBD", "\uAFBD", "\u1101\u116E\u11AC", "\uAFBD", "\u1101\u116E\u11AC" },
            new string [] { "\uAFBE", "\uAFBE", "\u1101\u116E\u11AD", "\uAFBE", "\u1101\u116E\u11AD" },
            new string [] { "\uAFBF", "\uAFBF", "\u1101\u116E\u11AE", "\uAFBF", "\u1101\u116E\u11AE" },
            new string [] { "\uAFC0", "\uAFC0", "\u1101\u116E\u11AF", "\uAFC0", "\u1101\u116E\u11AF" },
            new string [] { "\uAFC1", "\uAFC1", "\u1101\u116E\u11B0", "\uAFC1", "\u1101\u116E\u11B0" },
            new string [] { "\uAFC2", "\uAFC2", "\u1101\u116E\u11B1", "\uAFC2", "\u1101\u116E\u11B1" },
            new string [] { "\uAFC3", "\uAFC3", "\u1101\u116E\u11B2", "\uAFC3", "\u1101\u116E\u11B2" },
            new string [] { "\uAFC4", "\uAFC4", "\u1101\u116E\u11B3", "\uAFC4", "\u1101\u116E\u11B3" },
            new string [] { "\uAFC5", "\uAFC5", "\u1101\u116E\u11B4", "\uAFC5", "\u1101\u116E\u11B4" },
            new string [] { "\uAFC6", "\uAFC6", "\u1101\u116E\u11B5", "\uAFC6", "\u1101\u116E\u11B5" },
            new string [] { "\uAFC7", "\uAFC7", "\u1101\u116E\u11B6", "\uAFC7", "\u1101\u116E\u11B6" },
            new string [] { "\uAFC8", "\uAFC8", "\u1101\u116E\u11B7", "\uAFC8", "\u1101\u116E\u11B7" },
            new string [] { "\uAFC9", "\uAFC9", "\u1101\u116E\u11B8", "\uAFC9", "\u1101\u116E\u11B8" },
            new string [] { "\uAFCA", "\uAFCA", "\u1101\u116E\u11B9", "\uAFCA", "\u1101\u116E\u11B9" },
            new string [] { "\uAFCB", "\uAFCB", "\u1101\u116E\u11BA", "\uAFCB", "\u1101\u116E\u11BA" },
            new string [] { "\uAFCC", "\uAFCC", "\u1101\u116E\u11BB", "\uAFCC", "\u1101\u116E\u11BB" },
            new string [] { "\uAFCD", "\uAFCD", "\u1101\u116E\u11BC", "\uAFCD", "\u1101\u116E\u11BC" },
            new string [] { "\uAFCE", "\uAFCE", "\u1101\u116E\u11BD", "\uAFCE", "\u1101\u116E\u11BD" },
            new string [] { "\uAFCF", "\uAFCF", "\u1101\u116E\u11BE", "\uAFCF", "\u1101\u116E\u11BE" },
            new string [] { "\uAFD0", "\uAFD0", "\u1101\u116E\u11BF", "\uAFD0", "\u1101\u116E\u11BF" },
            new string [] { "\uAFD1", "\uAFD1", "\u1101\u116E\u11C0", "\uAFD1", "\u1101\u116E\u11C0" },
            new string [] { "\uAFD2", "\uAFD2", "\u1101\u116E\u11C1", "\uAFD2", "\u1101\u116E\u11C1" },
            new string [] { "\uAFD3", "\uAFD3", "\u1101\u116E\u11C2", "\uAFD3", "\u1101\u116E\u11C2" },
            new string [] { "\uAFD4", "\uAFD4", "\u1101\u116F", "\uAFD4", "\u1101\u116F" },
            new string [] { "\uAFD5", "\uAFD5", "\u1101\u116F\u11A8", "\uAFD5", "\u1101\u116F\u11A8" },
            new string [] { "\uAFD6", "\uAFD6", "\u1101\u116F\u11A9", "\uAFD6", "\u1101\u116F\u11A9" },
            new string [] { "\uAFD7", "\uAFD7", "\u1101\u116F\u11AA", "\uAFD7", "\u1101\u116F\u11AA" },
            new string [] { "\uAFD8", "\uAFD8", "\u1101\u116F\u11AB", "\uAFD8", "\u1101\u116F\u11AB" },
            new string [] { "\uAFD9", "\uAFD9", "\u1101\u116F\u11AC", "\uAFD9", "\u1101\u116F\u11AC" },
            new string [] { "\uAFDA", "\uAFDA", "\u1101\u116F\u11AD", "\uAFDA", "\u1101\u116F\u11AD" },
            new string [] { "\uAFDB", "\uAFDB", "\u1101\u116F\u11AE", "\uAFDB", "\u1101\u116F\u11AE" },
            new string [] { "\uAFDC", "\uAFDC", "\u1101\u116F\u11AF", "\uAFDC", "\u1101\u116F\u11AF" },
            new string [] { "\uAFDD", "\uAFDD", "\u1101\u116F\u11B0", "\uAFDD", "\u1101\u116F\u11B0" },
            new string [] { "\uAFDE", "\uAFDE", "\u1101\u116F\u11B1", "\uAFDE", "\u1101\u116F\u11B1" },
            new string [] { "\uAFDF", "\uAFDF", "\u1101\u116F\u11B2", "\uAFDF", "\u1101\u116F\u11B2" },
            new string [] { "\uAFE0", "\uAFE0", "\u1101\u116F\u11B3", "\uAFE0", "\u1101\u116F\u11B3" },
            new string [] { "\uAFE1", "\uAFE1", "\u1101\u116F\u11B4", "\uAFE1", "\u1101\u116F\u11B4" },
            new string [] { "\uAFE2", "\uAFE2", "\u1101\u116F\u11B5", "\uAFE2", "\u1101\u116F\u11B5" },
            new string [] { "\uAFE3", "\uAFE3", "\u1101\u116F\u11B6", "\uAFE3", "\u1101\u116F\u11B6" },
            new string [] { "\uAFE4", "\uAFE4", "\u1101\u116F\u11B7", "\uAFE4", "\u1101\u116F\u11B7" },
            new string [] { "\uAFE5", "\uAFE5", "\u1101\u116F\u11B8", "\uAFE5", "\u1101\u116F\u11B8" },
            new string [] { "\uAFE6", "\uAFE6", "\u1101\u116F\u11B9", "\uAFE6", "\u1101\u116F\u11B9" },
            new string [] { "\uAFE7", "\uAFE7", "\u1101\u116F\u11BA", "\uAFE7", "\u1101\u116F\u11BA" },
            new string [] { "\uAFE8", "\uAFE8", "\u1101\u116F\u11BB", "\uAFE8", "\u1101\u116F\u11BB" },
            new string [] { "\uAFE9", "\uAFE9", "\u1101\u116F\u11BC", "\uAFE9", "\u1101\u116F\u11BC" },
            new string [] { "\uAFEA", "\uAFEA", "\u1101\u116F\u11BD", "\uAFEA", "\u1101\u116F\u11BD" },
            new string [] { "\uAFEB", "\uAFEB", "\u1101\u116F\u11BE", "\uAFEB", "\u1101\u116F\u11BE" },
            new string [] { "\uAFEC", "\uAFEC", "\u1101\u116F\u11BF", "\uAFEC", "\u1101\u116F\u11BF" },
            new string [] { "\uAFED", "\uAFED", "\u1101\u116F\u11C0", "\uAFED", "\u1101\u116F\u11C0" },
            new string [] { "\uAFEE", "\uAFEE", "\u1101\u116F\u11C1", "\uAFEE", "\u1101\u116F\u11C1" },
            new string [] { "\uAFEF", "\uAFEF", "\u1101\u116F\u11C2", "\uAFEF", "\u1101\u116F\u11C2" },
            new string [] { "\uAFF0", "\uAFF0", "\u1101\u1170", "\uAFF0", "\u1101\u1170" },
            new string [] { "\uAFF1", "\uAFF1", "\u1101\u1170\u11A8", "\uAFF1", "\u1101\u1170\u11A8" },
            new string [] { "\uAFF2", "\uAFF2", "\u1101\u1170\u11A9", "\uAFF2", "\u1101\u1170\u11A9" },
            new string [] { "\uAFF3", "\uAFF3", "\u1101\u1170\u11AA", "\uAFF3", "\u1101\u1170\u11AA" },
            new string [] { "\uAFF4", "\uAFF4", "\u1101\u1170\u11AB", "\uAFF4", "\u1101\u1170\u11AB" },
            new string [] { "\uAFF5", "\uAFF5", "\u1101\u1170\u11AC", "\uAFF5", "\u1101\u1170\u11AC" },
            new string [] { "\uAFF6", "\uAFF6", "\u1101\u1170\u11AD", "\uAFF6", "\u1101\u1170\u11AD" },
            new string [] { "\uAFF7", "\uAFF7", "\u1101\u1170\u11AE", "\uAFF7", "\u1101\u1170\u11AE" },
            new string [] { "\uAFF8", "\uAFF8", "\u1101\u1170\u11AF", "\uAFF8", "\u1101\u1170\u11AF" },
            new string [] { "\uAFF9", "\uAFF9", "\u1101\u1170\u11B0", "\uAFF9", "\u1101\u1170\u11B0" },
            new string [] { "\uAFFA", "\uAFFA", "\u1101\u1170\u11B1", "\uAFFA", "\u1101\u1170\u11B1" },
            new string [] { "\uAFFB", "\uAFFB", "\u1101\u1170\u11B2", "\uAFFB", "\u1101\u1170\u11B2" },
            new string [] { "\uAFFC", "\uAFFC", "\u1101\u1170\u11B3", "\uAFFC", "\u1101\u1170\u11B3" },
            new string [] { "\uAFFD", "\uAFFD", "\u1101\u1170\u11B4", "\uAFFD", "\u1101\u1170\u11B4" },
            new string [] { "\uAFFE", "\uAFFE", "\u1101\u1170\u11B5", "\uAFFE", "\u1101\u1170\u11B5" },
            new string [] { "\uAFFF", "\uAFFF", "\u1101\u1170\u11B6", "\uAFFF", "\u1101\u1170\u11B6" },
            new string [] { "\uB000", "\uB000", "\u1101\u1170\u11B7", "\uB000", "\u1101\u1170\u11B7" },
            new string [] { "\uB001", "\uB001", "\u1101\u1170\u11B8", "\uB001", "\u1101\u1170\u11B8" },
            new string [] { "\uB002", "\uB002", "\u1101\u1170\u11B9", "\uB002", "\u1101\u1170\u11B9" },
            new string [] { "\uB003", "\uB003", "\u1101\u1170\u11BA", "\uB003", "\u1101\u1170\u11BA" },
            new string [] { "\uB004", "\uB004", "\u1101\u1170\u11BB", "\uB004", "\u1101\u1170\u11BB" },
            new string [] { "\uB005", "\uB005", "\u1101\u1170\u11BC", "\uB005", "\u1101\u1170\u11BC" },
            new string [] { "\uB006", "\uB006", "\u1101\u1170\u11BD", "\uB006", "\u1101\u1170\u11BD" },
            new string [] { "\uB007", "\uB007", "\u1101\u1170\u11BE", "\uB007", "\u1101\u1170\u11BE" },
            new string [] { "\uB008", "\uB008", "\u1101\u1170\u11BF", "\uB008", "\u1101\u1170\u11BF" },
            new string [] { "\uB009", "\uB009", "\u1101\u1170\u11C0", "\uB009", "\u1101\u1170\u11C0" },
            new string [] { "\uB00A", "\uB00A", "\u1101\u1170\u11C1", "\uB00A", "\u1101\u1170\u11C1" },
            new string [] { "\uB00B", "\uB00B", "\u1101\u1170\u11C2", "\uB00B", "\u1101\u1170\u11C2" },
            new string [] { "\uB00C", "\uB00C", "\u1101\u1171", "\uB00C", "\u1101\u1171" },
            new string [] { "\uB00D", "\uB00D", "\u1101\u1171\u11A8", "\uB00D", "\u1101\u1171\u11A8" },
            new string [] { "\uB00E", "\uB00E", "\u1101\u1171\u11A9", "\uB00E", "\u1101\u1171\u11A9" },
            new string [] { "\uB00F", "\uB00F", "\u1101\u1171\u11AA", "\uB00F", "\u1101\u1171\u11AA" },
            new string [] { "\uB010", "\uB010", "\u1101\u1171\u11AB", "\uB010", "\u1101\u1171\u11AB" },
            new string [] { "\uB011", "\uB011", "\u1101\u1171\u11AC", "\uB011", "\u1101\u1171\u11AC" },
            new string [] { "\uB012", "\uB012", "\u1101\u1171\u11AD", "\uB012", "\u1101\u1171\u11AD" },
            new string [] { "\uB013", "\uB013", "\u1101\u1171\u11AE", "\uB013", "\u1101\u1171\u11AE" },
            new string [] { "\uB014", "\uB014", "\u1101\u1171\u11AF", "\uB014", "\u1101\u1171\u11AF" },
            new string [] { "\uB015", "\uB015", "\u1101\u1171\u11B0", "\uB015", "\u1101\u1171\u11B0" },
            new string [] { "\uB016", "\uB016", "\u1101\u1171\u11B1", "\uB016", "\u1101\u1171\u11B1" },
            new string [] { "\uB017", "\uB017", "\u1101\u1171\u11B2", "\uB017", "\u1101\u1171\u11B2" },
            new string [] { "\uB018", "\uB018", "\u1101\u1171\u11B3", "\uB018", "\u1101\u1171\u11B3" },
            new string [] { "\uB019", "\uB019", "\u1101\u1171\u11B4", "\uB019", "\u1101\u1171\u11B4" },
            new string [] { "\uB01A", "\uB01A", "\u1101\u1171\u11B5", "\uB01A", "\u1101\u1171\u11B5" },
            new string [] { "\uB01B", "\uB01B", "\u1101\u1171\u11B6", "\uB01B", "\u1101\u1171\u11B6" },
            new string [] { "\uB01C", "\uB01C", "\u1101\u1171\u11B7", "\uB01C", "\u1101\u1171\u11B7" },
            new string [] { "\uB01D", "\uB01D", "\u1101\u1171\u11B8", "\uB01D", "\u1101\u1171\u11B8" },
            new string [] { "\uB01E", "\uB01E", "\u1101\u1171\u11B9", "\uB01E", "\u1101\u1171\u11B9" },
            new string [] { "\uB01F", "\uB01F", "\u1101\u1171\u11BA", "\uB01F", "\u1101\u1171\u11BA" },
            new string [] { "\uB020", "\uB020", "\u1101\u1171\u11BB", "\uB020", "\u1101\u1171\u11BB" },
            new string [] { "\uB021", "\uB021", "\u1101\u1171\u11BC", "\uB021", "\u1101\u1171\u11BC" },
            new string [] { "\uB022", "\uB022", "\u1101\u1171\u11BD", "\uB022", "\u1101\u1171\u11BD" },
            new string [] { "\uB023", "\uB023", "\u1101\u1171\u11BE", "\uB023", "\u1101\u1171\u11BE" },
            new string [] { "\uB024", "\uB024", "\u1101\u1171\u11BF", "\uB024", "\u1101\u1171\u11BF" },
            new string [] { "\uB025", "\uB025", "\u1101\u1171\u11C0", "\uB025", "\u1101\u1171\u11C0" },
            new string [] { "\uB026", "\uB026", "\u1101\u1171\u11C1", "\uB026", "\u1101\u1171\u11C1" },
            new string [] { "\uB027", "\uB027", "\u1101\u1171\u11C2", "\uB027", "\u1101\u1171\u11C2" },
            new string [] { "\uB028", "\uB028", "\u1101\u1172", "\uB028", "\u1101\u1172" },
            new string [] { "\uB029", "\uB029", "\u1101\u1172\u11A8", "\uB029", "\u1101\u1172\u11A8" },
            new string [] { "\uB02A", "\uB02A", "\u1101\u1172\u11A9", "\uB02A", "\u1101\u1172\u11A9" },
            new string [] { "\uB02B", "\uB02B", "\u1101\u1172\u11AA", "\uB02B", "\u1101\u1172\u11AA" },
            new string [] { "\uB02C", "\uB02C", "\u1101\u1172\u11AB", "\uB02C", "\u1101\u1172\u11AB" },
            new string [] { "\uB02D", "\uB02D", "\u1101\u1172\u11AC", "\uB02D", "\u1101\u1172\u11AC" },
            new string [] { "\uB02E", "\uB02E", "\u1101\u1172\u11AD", "\uB02E", "\u1101\u1172\u11AD" },
            new string [] { "\uB02F", "\uB02F", "\u1101\u1172\u11AE", "\uB02F", "\u1101\u1172\u11AE" },
            new string [] { "\uB030", "\uB030", "\u1101\u1172\u11AF", "\uB030", "\u1101\u1172\u11AF" },
            new string [] { "\uB031", "\uB031", "\u1101\u1172\u11B0", "\uB031", "\u1101\u1172\u11B0" },
            new string [] { "\uB032", "\uB032", "\u1101\u1172\u11B1", "\uB032", "\u1101\u1172\u11B1" },
            new string [] { "\uB033", "\uB033", "\u1101\u1172\u11B2", "\uB033", "\u1101\u1172\u11B2" },
            new string [] { "\uB034", "\uB034", "\u1101\u1172\u11B3", "\uB034", "\u1101\u1172\u11B3" },
            new string [] { "\uB035", "\uB035", "\u1101\u1172\u11B4", "\uB035", "\u1101\u1172\u11B4" },
            new string [] { "\uB036", "\uB036", "\u1101\u1172\u11B5", "\uB036", "\u1101\u1172\u11B5" },
            new string [] { "\uB037", "\uB037", "\u1101\u1172\u11B6", "\uB037", "\u1101\u1172\u11B6" },
            new string [] { "\uB038", "\uB038", "\u1101\u1172\u11B7", "\uB038", "\u1101\u1172\u11B7" },
            new string [] { "\uB039", "\uB039", "\u1101\u1172\u11B8", "\uB039", "\u1101\u1172\u11B8" },
            new string [] { "\uB03A", "\uB03A", "\u1101\u1172\u11B9", "\uB03A", "\u1101\u1172\u11B9" },
            new string [] { "\uB03B", "\uB03B", "\u1101\u1172\u11BA", "\uB03B", "\u1101\u1172\u11BA" },
            new string [] { "\uB03C", "\uB03C", "\u1101\u1172\u11BB", "\uB03C", "\u1101\u1172\u11BB" },
            new string [] { "\uB03D", "\uB03D", "\u1101\u1172\u11BC", "\uB03D", "\u1101\u1172\u11BC" },
            new string [] { "\uB03E", "\uB03E", "\u1101\u1172\u11BD", "\uB03E", "\u1101\u1172\u11BD" },
            new string [] { "\uB03F", "\uB03F", "\u1101\u1172\u11BE", "\uB03F", "\u1101\u1172\u11BE" },
            new string [] { "\uB040", "\uB040", "\u1101\u1172\u11BF", "\uB040", "\u1101\u1172\u11BF" },
            new string [] { "\uB041", "\uB041", "\u1101\u1172\u11C0", "\uB041", "\u1101\u1172\u11C0" },
            new string [] { "\uB042", "\uB042", "\u1101\u1172\u11C1", "\uB042", "\u1101\u1172\u11C1" },
            new string [] { "\uB043", "\uB043", "\u1101\u1172\u11C2", "\uB043", "\u1101\u1172\u11C2" },
            new string [] { "\uB044", "\uB044", "\u1101\u1173", "\uB044", "\u1101\u1173" },
            new string [] { "\uB045", "\uB045", "\u1101\u1173\u11A8", "\uB045", "\u1101\u1173\u11A8" },
            new string [] { "\uB046", "\uB046", "\u1101\u1173\u11A9", "\uB046", "\u1101\u1173\u11A9" },
            new string [] { "\uB047", "\uB047", "\u1101\u1173\u11AA", "\uB047", "\u1101\u1173\u11AA" },
            new string [] { "\uB048", "\uB048", "\u1101\u1173\u11AB", "\uB048", "\u1101\u1173\u11AB" },
            new string [] { "\uB049", "\uB049", "\u1101\u1173\u11AC", "\uB049", "\u1101\u1173\u11AC" },
            new string [] { "\uB04A", "\uB04A", "\u1101\u1173\u11AD", "\uB04A", "\u1101\u1173\u11AD" },
            new string [] { "\uB04B", "\uB04B", "\u1101\u1173\u11AE", "\uB04B", "\u1101\u1173\u11AE" },
            new string [] { "\uB04C", "\uB04C", "\u1101\u1173\u11AF", "\uB04C", "\u1101\u1173\u11AF" },
            new string [] { "\uB04D", "\uB04D", "\u1101\u1173\u11B0", "\uB04D", "\u1101\u1173\u11B0" },
            new string [] { "\uB04E", "\uB04E", "\u1101\u1173\u11B1", "\uB04E", "\u1101\u1173\u11B1" },
            new string [] { "\uB04F", "\uB04F", "\u1101\u1173\u11B2", "\uB04F", "\u1101\u1173\u11B2" },
            new string [] { "\uB050", "\uB050", "\u1101\u1173\u11B3", "\uB050", "\u1101\u1173\u11B3" },
            new string [] { "\uB051", "\uB051", "\u1101\u1173\u11B4", "\uB051", "\u1101\u1173\u11B4" },
            new string [] { "\uB052", "\uB052", "\u1101\u1173\u11B5", "\uB052", "\u1101\u1173\u11B5" },
            new string [] { "\uB053", "\uB053", "\u1101\u1173\u11B6", "\uB053", "\u1101\u1173\u11B6" },
            new string [] { "\uB054", "\uB054", "\u1101\u1173\u11B7", "\uB054", "\u1101\u1173\u11B7" },
            new string [] { "\uB055", "\uB055", "\u1101\u1173\u11B8", "\uB055", "\u1101\u1173\u11B8" },
            new string [] { "\uB056", "\uB056", "\u1101\u1173\u11B9", "\uB056", "\u1101\u1173\u11B9" },
            new string [] { "\uB057", "\uB057", "\u1101\u1173\u11BA", "\uB057", "\u1101\u1173\u11BA" },
            new string [] { "\uB058", "\uB058", "\u1101\u1173\u11BB", "\uB058", "\u1101\u1173\u11BB" },
            new string [] { "\uB059", "\uB059", "\u1101\u1173\u11BC", "\uB059", "\u1101\u1173\u11BC" },
            new string [] { "\uB05A", "\uB05A", "\u1101\u1173\u11BD", "\uB05A", "\u1101\u1173\u11BD" },
            new string [] { "\uB05B", "\uB05B", "\u1101\u1173\u11BE", "\uB05B", "\u1101\u1173\u11BE" },
            new string [] { "\uB05C", "\uB05C", "\u1101\u1173\u11BF", "\uB05C", "\u1101\u1173\u11BF" },
            new string [] { "\uB05D", "\uB05D", "\u1101\u1173\u11C0", "\uB05D", "\u1101\u1173\u11C0" },
            new string [] { "\uB05E", "\uB05E", "\u1101\u1173\u11C1", "\uB05E", "\u1101\u1173\u11C1" },
            new string [] { "\uB05F", "\uB05F", "\u1101\u1173\u11C2", "\uB05F", "\u1101\u1173\u11C2" },
            new string [] { "\uB060", "\uB060", "\u1101\u1174", "\uB060", "\u1101\u1174" },
            new string [] { "\uB061", "\uB061", "\u1101\u1174\u11A8", "\uB061", "\u1101\u1174\u11A8" },
            new string [] { "\uB062", "\uB062", "\u1101\u1174\u11A9", "\uB062", "\u1101\u1174\u11A9" },
            new string [] { "\uB063", "\uB063", "\u1101\u1174\u11AA", "\uB063", "\u1101\u1174\u11AA" },
            new string [] { "\uB064", "\uB064", "\u1101\u1174\u11AB", "\uB064", "\u1101\u1174\u11AB" },
            new string [] { "\uB065", "\uB065", "\u1101\u1174\u11AC", "\uB065", "\u1101\u1174\u11AC" },
            new string [] { "\uB066", "\uB066", "\u1101\u1174\u11AD", "\uB066", "\u1101\u1174\u11AD" },
            new string [] { "\uB067", "\uB067", "\u1101\u1174\u11AE", "\uB067", "\u1101\u1174\u11AE" },
            new string [] { "\uB068", "\uB068", "\u1101\u1174\u11AF", "\uB068", "\u1101\u1174\u11AF" },
            new string [] { "\uB069", "\uB069", "\u1101\u1174\u11B0", "\uB069", "\u1101\u1174\u11B0" },
            new string [] { "\uB06A", "\uB06A", "\u1101\u1174\u11B1", "\uB06A", "\u1101\u1174\u11B1" },
            new string [] { "\uB06B", "\uB06B", "\u1101\u1174\u11B2", "\uB06B", "\u1101\u1174\u11B2" },
            new string [] { "\uB06C", "\uB06C", "\u1101\u1174\u11B3", "\uB06C", "\u1101\u1174\u11B3" },
            new string [] { "\uB06D", "\uB06D", "\u1101\u1174\u11B4", "\uB06D", "\u1101\u1174\u11B4" },
            new string [] { "\uB06E", "\uB06E", "\u1101\u1174\u11B5", "\uB06E", "\u1101\u1174\u11B5" },
            new string [] { "\uB06F", "\uB06F", "\u1101\u1174\u11B6", "\uB06F", "\u1101\u1174\u11B6" },
            new string [] { "\uB070", "\uB070", "\u1101\u1174\u11B7", "\uB070", "\u1101\u1174\u11B7" },
            new string [] { "\uB071", "\uB071", "\u1101\u1174\u11B8", "\uB071", "\u1101\u1174\u11B8" },
            new string [] { "\uB072", "\uB072", "\u1101\u1174\u11B9", "\uB072", "\u1101\u1174\u11B9" },
            new string [] { "\uB073", "\uB073", "\u1101\u1174\u11BA", "\uB073", "\u1101\u1174\u11BA" },
            new string [] { "\uB074", "\uB074", "\u1101\u1174\u11BB", "\uB074", "\u1101\u1174\u11BB" },
            new string [] { "\uB075", "\uB075", "\u1101\u1174\u11BC", "\uB075", "\u1101\u1174\u11BC" },
            new string [] { "\uB076", "\uB076", "\u1101\u1174\u11BD", "\uB076", "\u1101\u1174\u11BD" },
            new string [] { "\uB077", "\uB077", "\u1101\u1174\u11BE", "\uB077", "\u1101\u1174\u11BE" },
            new string [] { "\uB078", "\uB078", "\u1101\u1174\u11BF", "\uB078", "\u1101\u1174\u11BF" },
            new string [] { "\uB079", "\uB079", "\u1101\u1174\u11C0", "\uB079", "\u1101\u1174\u11C0" },
            new string [] { "\uB07A", "\uB07A", "\u1101\u1174\u11C1", "\uB07A", "\u1101\u1174\u11C1" },
            new string [] { "\uB07B", "\uB07B", "\u1101\u1174\u11C2", "\uB07B", "\u1101\u1174\u11C2" },
            new string [] { "\uB07C", "\uB07C", "\u1101\u1175", "\uB07C", "\u1101\u1175" },
            new string [] { "\uB07D", "\uB07D", "\u1101\u1175\u11A8", "\uB07D", "\u1101\u1175\u11A8" },
            new string [] { "\uB07E", "\uB07E", "\u1101\u1175\u11A9", "\uB07E", "\u1101\u1175\u11A9" },
            new string [] { "\uB07F", "\uB07F", "\u1101\u1175\u11AA", "\uB07F", "\u1101\u1175\u11AA" },
            new string [] { "\uB080", "\uB080", "\u1101\u1175\u11AB", "\uB080", "\u1101\u1175\u11AB" },
            new string [] { "\uB081", "\uB081", "\u1101\u1175\u11AC", "\uB081", "\u1101\u1175\u11AC" },
            new string [] { "\uB082", "\uB082", "\u1101\u1175\u11AD", "\uB082", "\u1101\u1175\u11AD" },
            new string [] { "\uB083", "\uB083", "\u1101\u1175\u11AE", "\uB083", "\u1101\u1175\u11AE" },
            new string [] { "\uB084", "\uB084", "\u1101\u1175\u11AF", "\uB084", "\u1101\u1175\u11AF" },
            new string [] { "\uB085", "\uB085", "\u1101\u1175\u11B0", "\uB085", "\u1101\u1175\u11B0" },
            new string [] { "\uB086", "\uB086", "\u1101\u1175\u11B1", "\uB086", "\u1101\u1175\u11B1" },
            new string [] { "\uB087", "\uB087", "\u1101\u1175\u11B2", "\uB087", "\u1101\u1175\u11B2" },
            new string [] { "\uB088", "\uB088", "\u1101\u1175\u11B3", "\uB088", "\u1101\u1175\u11B3" },
            new string [] { "\uB089", "\uB089", "\u1101\u1175\u11B4", "\uB089", "\u1101\u1175\u11B4" },
            new string [] { "\uB08A", "\uB08A", "\u1101\u1175\u11B5", "\uB08A", "\u1101\u1175\u11B5" },
            new string [] { "\uB08B", "\uB08B", "\u1101\u1175\u11B6", "\uB08B", "\u1101\u1175\u11B6" },
            new string [] { "\uB08C", "\uB08C", "\u1101\u1175\u11B7", "\uB08C", "\u1101\u1175\u11B7" },
            new string [] { "\uB08D", "\uB08D", "\u1101\u1175\u11B8", "\uB08D", "\u1101\u1175\u11B8" },
            new string [] { "\uB08E", "\uB08E", "\u1101\u1175\u11B9", "\uB08E", "\u1101\u1175\u11B9" },
            new string [] { "\uB08F", "\uB08F", "\u1101\u1175\u11BA", "\uB08F", "\u1101\u1175\u11BA" },
            new string [] { "\uB090", "\uB090", "\u1101\u1175\u11BB", "\uB090", "\u1101\u1175\u11BB" },
            new string [] { "\uB091", "\uB091", "\u1101\u1175\u11BC", "\uB091", "\u1101\u1175\u11BC" },
            new string [] { "\uB092", "\uB092", "\u1101\u1175\u11BD", "\uB092", "\u1101\u1175\u11BD" },
            new string [] { "\uB093", "\uB093", "\u1101\u1175\u11BE", "\uB093", "\u1101\u1175\u11BE" },
            new string [] { "\uB094", "\uB094", "\u1101\u1175\u11BF", "\uB094", "\u1101\u1175\u11BF" },
            new string [] { "\uB095", "\uB095", "\u1101\u1175\u11C0", "\uB095", "\u1101\u1175\u11C0" },
            new string [] { "\uB096", "\uB096", "\u1101\u1175\u11C1", "\uB096", "\u1101\u1175\u11C1" },
            new string [] { "\uB097", "\uB097", "\u1101\u1175\u11C2", "\uB097", "\u1101\u1175\u11C2" },
            new string [] { "\uB098", "\uB098", "\u1102\u1161", "\uB098", "\u1102\u1161" },
            new string [] { "\uB099", "\uB099", "\u1102\u1161\u11A8", "\uB099", "\u1102\u1161\u11A8" },
            new string [] { "\uB09A", "\uB09A", "\u1102\u1161\u11A9", "\uB09A", "\u1102\u1161\u11A9" },
            new string [] { "\uB09B", "\uB09B", "\u1102\u1161\u11AA", "\uB09B", "\u1102\u1161\u11AA" },
            new string [] { "\uB09C", "\uB09C", "\u1102\u1161\u11AB", "\uB09C", "\u1102\u1161\u11AB" },
            new string [] { "\uB09D", "\uB09D", "\u1102\u1161\u11AC", "\uB09D", "\u1102\u1161\u11AC" },
            new string [] { "\uB09E", "\uB09E", "\u1102\u1161\u11AD", "\uB09E", "\u1102\u1161\u11AD" },
            new string [] { "\uB09F", "\uB09F", "\u1102\u1161\u11AE", "\uB09F", "\u1102\u1161\u11AE" },
            new string [] { "\uB0A0", "\uB0A0", "\u1102\u1161\u11AF", "\uB0A0", "\u1102\u1161\u11AF" },
            new string [] { "\uB0A1", "\uB0A1", "\u1102\u1161\u11B0", "\uB0A1", "\u1102\u1161\u11B0" },
            new string [] { "\uB0A2", "\uB0A2", "\u1102\u1161\u11B1", "\uB0A2", "\u1102\u1161\u11B1" },
            new string [] { "\uB0A3", "\uB0A3", "\u1102\u1161\u11B2", "\uB0A3", "\u1102\u1161\u11B2" },
            new string [] { "\uB0A4", "\uB0A4", "\u1102\u1161\u11B3", "\uB0A4", "\u1102\u1161\u11B3" },
            new string [] { "\uB0A5", "\uB0A5", "\u1102\u1161\u11B4", "\uB0A5", "\u1102\u1161\u11B4" },
            new string [] { "\uB0A6", "\uB0A6", "\u1102\u1161\u11B5", "\uB0A6", "\u1102\u1161\u11B5" },
            new string [] { "\uB0A7", "\uB0A7", "\u1102\u1161\u11B6", "\uB0A7", "\u1102\u1161\u11B6" },
            new string [] { "\uB0A8", "\uB0A8", "\u1102\u1161\u11B7", "\uB0A8", "\u1102\u1161\u11B7" },
            new string [] { "\uB0A9", "\uB0A9", "\u1102\u1161\u11B8", "\uB0A9", "\u1102\u1161\u11B8" },
            new string [] { "\uB0AA", "\uB0AA", "\u1102\u1161\u11B9", "\uB0AA", "\u1102\u1161\u11B9" },
            new string [] { "\uB0AB", "\uB0AB", "\u1102\u1161\u11BA", "\uB0AB", "\u1102\u1161\u11BA" },
            new string [] { "\uB0AC", "\uB0AC", "\u1102\u1161\u11BB", "\uB0AC", "\u1102\u1161\u11BB" },
            new string [] { "\uB0AD", "\uB0AD", "\u1102\u1161\u11BC", "\uB0AD", "\u1102\u1161\u11BC" },
            new string [] { "\uB0AE", "\uB0AE", "\u1102\u1161\u11BD", "\uB0AE", "\u1102\u1161\u11BD" },
            new string [] { "\uB0AF", "\uB0AF", "\u1102\u1161\u11BE", "\uB0AF", "\u1102\u1161\u11BE" },
            new string [] { "\uB0B0", "\uB0B0", "\u1102\u1161\u11BF", "\uB0B0", "\u1102\u1161\u11BF" },
            new string [] { "\uB0B1", "\uB0B1", "\u1102\u1161\u11C0", "\uB0B1", "\u1102\u1161\u11C0" },
            new string [] { "\uB0B2", "\uB0B2", "\u1102\u1161\u11C1", "\uB0B2", "\u1102\u1161\u11C1" },
            new string [] { "\uB0B3", "\uB0B3", "\u1102\u1161\u11C2", "\uB0B3", "\u1102\u1161\u11C2" },
            new string [] { "\uB0B4", "\uB0B4", "\u1102\u1162", "\uB0B4", "\u1102\u1162" },
            new string [] { "\uB0B5", "\uB0B5", "\u1102\u1162\u11A8", "\uB0B5", "\u1102\u1162\u11A8" },
            new string [] { "\uB0B6", "\uB0B6", "\u1102\u1162\u11A9", "\uB0B6", "\u1102\u1162\u11A9" },
            new string [] { "\uB0B7", "\uB0B7", "\u1102\u1162\u11AA", "\uB0B7", "\u1102\u1162\u11AA" },
            new string [] { "\uB0B8", "\uB0B8", "\u1102\u1162\u11AB", "\uB0B8", "\u1102\u1162\u11AB" },
            new string [] { "\uB0B9", "\uB0B9", "\u1102\u1162\u11AC", "\uB0B9", "\u1102\u1162\u11AC" },
            new string [] { "\uB0BA", "\uB0BA", "\u1102\u1162\u11AD", "\uB0BA", "\u1102\u1162\u11AD" },
            new string [] { "\uB0BB", "\uB0BB", "\u1102\u1162\u11AE", "\uB0BB", "\u1102\u1162\u11AE" },
            new string [] { "\uB0BC", "\uB0BC", "\u1102\u1162\u11AF", "\uB0BC", "\u1102\u1162\u11AF" },
            new string [] { "\uB0BD", "\uB0BD", "\u1102\u1162\u11B0", "\uB0BD", "\u1102\u1162\u11B0" },
            new string [] { "\uB0BE", "\uB0BE", "\u1102\u1162\u11B1", "\uB0BE", "\u1102\u1162\u11B1" },
            new string [] { "\uB0BF", "\uB0BF", "\u1102\u1162\u11B2", "\uB0BF", "\u1102\u1162\u11B2" },
            new string [] { "\uB0C0", "\uB0C0", "\u1102\u1162\u11B3", "\uB0C0", "\u1102\u1162\u11B3" },
            new string [] { "\uB0C1", "\uB0C1", "\u1102\u1162\u11B4", "\uB0C1", "\u1102\u1162\u11B4" },
            new string [] { "\uB0C2", "\uB0C2", "\u1102\u1162\u11B5", "\uB0C2", "\u1102\u1162\u11B5" },
            new string [] { "\uB0C3", "\uB0C3", "\u1102\u1162\u11B6", "\uB0C3", "\u1102\u1162\u11B6" },
            new string [] { "\uB0C4", "\uB0C4", "\u1102\u1162\u11B7", "\uB0C4", "\u1102\u1162\u11B7" },
            new string [] { "\uB0C5", "\uB0C5", "\u1102\u1162\u11B8", "\uB0C5", "\u1102\u1162\u11B8" },
            new string [] { "\uB0C6", "\uB0C6", "\u1102\u1162\u11B9", "\uB0C6", "\u1102\u1162\u11B9" },
            new string [] { "\uB0C7", "\uB0C7", "\u1102\u1162\u11BA", "\uB0C7", "\u1102\u1162\u11BA" },
            new string [] { "\uB0C8", "\uB0C8", "\u1102\u1162\u11BB", "\uB0C8", "\u1102\u1162\u11BB" },
            new string [] { "\uB0C9", "\uB0C9", "\u1102\u1162\u11BC", "\uB0C9", "\u1102\u1162\u11BC" },
            new string [] { "\uB0CA", "\uB0CA", "\u1102\u1162\u11BD", "\uB0CA", "\u1102\u1162\u11BD" },
            new string [] { "\uB0CB", "\uB0CB", "\u1102\u1162\u11BE", "\uB0CB", "\u1102\u1162\u11BE" },
            new string [] { "\uB0CC", "\uB0CC", "\u1102\u1162\u11BF", "\uB0CC", "\u1102\u1162\u11BF" },
            new string [] { "\uB0CD", "\uB0CD", "\u1102\u1162\u11C0", "\uB0CD", "\u1102\u1162\u11C0" },
            new string [] { "\uB0CE", "\uB0CE", "\u1102\u1162\u11C1", "\uB0CE", "\u1102\u1162\u11C1" },
            new string [] { "\uB0CF", "\uB0CF", "\u1102\u1162\u11C2", "\uB0CF", "\u1102\u1162\u11C2" },
            new string [] { "\uB0D0", "\uB0D0", "\u1102\u1163", "\uB0D0", "\u1102\u1163" },
            new string [] { "\uB0D1", "\uB0D1", "\u1102\u1163\u11A8", "\uB0D1", "\u1102\u1163\u11A8" },
            new string [] { "\uB0D2", "\uB0D2", "\u1102\u1163\u11A9", "\uB0D2", "\u1102\u1163\u11A9" },
            new string [] { "\uB0D3", "\uB0D3", "\u1102\u1163\u11AA", "\uB0D3", "\u1102\u1163\u11AA" },
            new string [] { "\uB0D4", "\uB0D4", "\u1102\u1163\u11AB", "\uB0D4", "\u1102\u1163\u11AB" },
            new string [] { "\uB0D5", "\uB0D5", "\u1102\u1163\u11AC", "\uB0D5", "\u1102\u1163\u11AC" },
            new string [] { "\uB0D6", "\uB0D6", "\u1102\u1163\u11AD", "\uB0D6", "\u1102\u1163\u11AD" },
            new string [] { "\uB0D7", "\uB0D7", "\u1102\u1163\u11AE", "\uB0D7", "\u1102\u1163\u11AE" },
            new string [] { "\uB0D8", "\uB0D8", "\u1102\u1163\u11AF", "\uB0D8", "\u1102\u1163\u11AF" },
            new string [] { "\uB0D9", "\uB0D9", "\u1102\u1163\u11B0", "\uB0D9", "\u1102\u1163\u11B0" },
            new string [] { "\uB0DA", "\uB0DA", "\u1102\u1163\u11B1", "\uB0DA", "\u1102\u1163\u11B1" },
            new string [] { "\uB0DB", "\uB0DB", "\u1102\u1163\u11B2", "\uB0DB", "\u1102\u1163\u11B2" },
            new string [] { "\uB0DC", "\uB0DC", "\u1102\u1163\u11B3", "\uB0DC", "\u1102\u1163\u11B3" },
            new string [] { "\uB0DD", "\uB0DD", "\u1102\u1163\u11B4", "\uB0DD", "\u1102\u1163\u11B4" },
            new string [] { "\uB0DE", "\uB0DE", "\u1102\u1163\u11B5", "\uB0DE", "\u1102\u1163\u11B5" },
            new string [] { "\uB0DF", "\uB0DF", "\u1102\u1163\u11B6", "\uB0DF", "\u1102\u1163\u11B6" },
            new string [] { "\uB0E0", "\uB0E0", "\u1102\u1163\u11B7", "\uB0E0", "\u1102\u1163\u11B7" },
            new string [] { "\uB0E1", "\uB0E1", "\u1102\u1163\u11B8", "\uB0E1", "\u1102\u1163\u11B8" },
            new string [] { "\uB0E2", "\uB0E2", "\u1102\u1163\u11B9", "\uB0E2", "\u1102\u1163\u11B9" },
            new string [] { "\uB0E3", "\uB0E3", "\u1102\u1163\u11BA", "\uB0E3", "\u1102\u1163\u11BA" },
            new string [] { "\uB0E4", "\uB0E4", "\u1102\u1163\u11BB", "\uB0E4", "\u1102\u1163\u11BB" },
            new string [] { "\uB0E5", "\uB0E5", "\u1102\u1163\u11BC", "\uB0E5", "\u1102\u1163\u11BC" },
            new string [] { "\uB0E6", "\uB0E6", "\u1102\u1163\u11BD", "\uB0E6", "\u1102\u1163\u11BD" },
            new string [] { "\uB0E7", "\uB0E7", "\u1102\u1163\u11BE", "\uB0E7", "\u1102\u1163\u11BE" },
            new string [] { "\uB0E8", "\uB0E8", "\u1102\u1163\u11BF", "\uB0E8", "\u1102\u1163\u11BF" },
            new string [] { "\uB0E9", "\uB0E9", "\u1102\u1163\u11C0", "\uB0E9", "\u1102\u1163\u11C0" },
            new string [] { "\uB0EA", "\uB0EA", "\u1102\u1163\u11C1", "\uB0EA", "\u1102\u1163\u11C1" },
            new string [] { "\uB0EB", "\uB0EB", "\u1102\u1163\u11C2", "\uB0EB", "\u1102\u1163\u11C2" },
            new string [] { "\uB0EC", "\uB0EC", "\u1102\u1164", "\uB0EC", "\u1102\u1164" },
            new string [] { "\uB0ED", "\uB0ED", "\u1102\u1164\u11A8", "\uB0ED", "\u1102\u1164\u11A8" },
            new string [] { "\uB0EE", "\uB0EE", "\u1102\u1164\u11A9", "\uB0EE", "\u1102\u1164\u11A9" },
            new string [] { "\uB0EF", "\uB0EF", "\u1102\u1164\u11AA", "\uB0EF", "\u1102\u1164\u11AA" },
            new string [] { "\uB0F0", "\uB0F0", "\u1102\u1164\u11AB", "\uB0F0", "\u1102\u1164\u11AB" },
            new string [] { "\uB0F1", "\uB0F1", "\u1102\u1164\u11AC", "\uB0F1", "\u1102\u1164\u11AC" },
            new string [] { "\uB0F2", "\uB0F2", "\u1102\u1164\u11AD", "\uB0F2", "\u1102\u1164\u11AD" },
            new string [] { "\uB0F3", "\uB0F3", "\u1102\u1164\u11AE", "\uB0F3", "\u1102\u1164\u11AE" },
            new string [] { "\uB0F4", "\uB0F4", "\u1102\u1164\u11AF", "\uB0F4", "\u1102\u1164\u11AF" },
            new string [] { "\uB0F5", "\uB0F5", "\u1102\u1164\u11B0", "\uB0F5", "\u1102\u1164\u11B0" },
            new string [] { "\uB0F6", "\uB0F6", "\u1102\u1164\u11B1", "\uB0F6", "\u1102\u1164\u11B1" },
            new string [] { "\uB0F7", "\uB0F7", "\u1102\u1164\u11B2", "\uB0F7", "\u1102\u1164\u11B2" },
            new string [] { "\uB0F8", "\uB0F8", "\u1102\u1164\u11B3", "\uB0F8", "\u1102\u1164\u11B3" },
            new string [] { "\uB0F9", "\uB0F9", "\u1102\u1164\u11B4", "\uB0F9", "\u1102\u1164\u11B4" },
            new string [] { "\uB0FA", "\uB0FA", "\u1102\u1164\u11B5", "\uB0FA", "\u1102\u1164\u11B5" },
            new string [] { "\uB0FB", "\uB0FB", "\u1102\u1164\u11B6", "\uB0FB", "\u1102\u1164\u11B6" },
            new string [] { "\uB0FC", "\uB0FC", "\u1102\u1164\u11B7", "\uB0FC", "\u1102\u1164\u11B7" },
            new string [] { "\uB0FD", "\uB0FD", "\u1102\u1164\u11B8", "\uB0FD", "\u1102\u1164\u11B8" },
            new string [] { "\uB0FE", "\uB0FE", "\u1102\u1164\u11B9", "\uB0FE", "\u1102\u1164\u11B9" },
            new string [] { "\uB0FF", "\uB0FF", "\u1102\u1164\u11BA", "\uB0FF", "\u1102\u1164\u11BA" },
            new string [] { "\uB100", "\uB100", "\u1102\u1164\u11BB", "\uB100", "\u1102\u1164\u11BB" },
            new string [] { "\uB101", "\uB101", "\u1102\u1164\u11BC", "\uB101", "\u1102\u1164\u11BC" },
            new string [] { "\uB102", "\uB102", "\u1102\u1164\u11BD", "\uB102", "\u1102\u1164\u11BD" },
            new string [] { "\uB103", "\uB103", "\u1102\u1164\u11BE", "\uB103", "\u1102\u1164\u11BE" },
            new string [] { "\uB104", "\uB104", "\u1102\u1164\u11BF", "\uB104", "\u1102\u1164\u11BF" },
            new string [] { "\uB105", "\uB105", "\u1102\u1164\u11C0", "\uB105", "\u1102\u1164\u11C0" },
            new string [] { "\uB106", "\uB106", "\u1102\u1164\u11C1", "\uB106", "\u1102\u1164\u11C1" },
            new string [] { "\uB107", "\uB107", "\u1102\u1164\u11C2", "\uB107", "\u1102\u1164\u11C2" },
            new string [] { "\uB108", "\uB108", "\u1102\u1165", "\uB108", "\u1102\u1165" },
            new string [] { "\uB109", "\uB109", "\u1102\u1165\u11A8", "\uB109", "\u1102\u1165\u11A8" },
            new string [] { "\uB10A", "\uB10A", "\u1102\u1165\u11A9", "\uB10A", "\u1102\u1165\u11A9" },
            new string [] { "\uB10B", "\uB10B", "\u1102\u1165\u11AA", "\uB10B", "\u1102\u1165\u11AA" },
            new string [] { "\uB10C", "\uB10C", "\u1102\u1165\u11AB", "\uB10C", "\u1102\u1165\u11AB" },
            new string [] { "\uB10D", "\uB10D", "\u1102\u1165\u11AC", "\uB10D", "\u1102\u1165\u11AC" },
            new string [] { "\uB10E", "\uB10E", "\u1102\u1165\u11AD", "\uB10E", "\u1102\u1165\u11AD" },
            new string [] { "\uB10F", "\uB10F", "\u1102\u1165\u11AE", "\uB10F", "\u1102\u1165\u11AE" },
            new string [] { "\uB110", "\uB110", "\u1102\u1165\u11AF", "\uB110", "\u1102\u1165\u11AF" },
            new string [] { "\uB111", "\uB111", "\u1102\u1165\u11B0", "\uB111", "\u1102\u1165\u11B0" },
            new string [] { "\uB112", "\uB112", "\u1102\u1165\u11B1", "\uB112", "\u1102\u1165\u11B1" },
            new string [] { "\uB113", "\uB113", "\u1102\u1165\u11B2", "\uB113", "\u1102\u1165\u11B2" },
            new string [] { "\uB114", "\uB114", "\u1102\u1165\u11B3", "\uB114", "\u1102\u1165\u11B3" },
            new string [] { "\uB115", "\uB115", "\u1102\u1165\u11B4", "\uB115", "\u1102\u1165\u11B4" },
            new string [] { "\uB116", "\uB116", "\u1102\u1165\u11B5", "\uB116", "\u1102\u1165\u11B5" },
            new string [] { "\uB117", "\uB117", "\u1102\u1165\u11B6", "\uB117", "\u1102\u1165\u11B6" },
            new string [] { "\uB118", "\uB118", "\u1102\u1165\u11B7", "\uB118", "\u1102\u1165\u11B7" },
            new string [] { "\uB119", "\uB119", "\u1102\u1165\u11B8", "\uB119", "\u1102\u1165\u11B8" },
            new string [] { "\uB11A", "\uB11A", "\u1102\u1165\u11B9", "\uB11A", "\u1102\u1165\u11B9" },
            new string [] { "\uB11B", "\uB11B", "\u1102\u1165\u11BA", "\uB11B", "\u1102\u1165\u11BA" },
            new string [] { "\uB11C", "\uB11C", "\u1102\u1165\u11BB", "\uB11C", "\u1102\u1165\u11BB" },
            new string [] { "\uB11D", "\uB11D", "\u1102\u1165\u11BC", "\uB11D", "\u1102\u1165\u11BC" },
            new string [] { "\uB11E", "\uB11E", "\u1102\u1165\u11BD", "\uB11E", "\u1102\u1165\u11BD" },
            new string [] { "\uB11F", "\uB11F", "\u1102\u1165\u11BE", "\uB11F", "\u1102\u1165\u11BE" },
            new string [] { "\uB120", "\uB120", "\u1102\u1165\u11BF", "\uB120", "\u1102\u1165\u11BF" },
            new string [] { "\uB121", "\uB121", "\u1102\u1165\u11C0", "\uB121", "\u1102\u1165\u11C0" },
            new string [] { "\uB122", "\uB122", "\u1102\u1165\u11C1", "\uB122", "\u1102\u1165\u11C1" },
            new string [] { "\uB123", "\uB123", "\u1102\u1165\u11C2", "\uB123", "\u1102\u1165\u11C2" },
            new string [] { "\uB124", "\uB124", "\u1102\u1166", "\uB124", "\u1102\u1166" },
            new string [] { "\uB125", "\uB125", "\u1102\u1166\u11A8", "\uB125", "\u1102\u1166\u11A8" },
            new string [] { "\uB126", "\uB126", "\u1102\u1166\u11A9", "\uB126", "\u1102\u1166\u11A9" },
            new string [] { "\uB127", "\uB127", "\u1102\u1166\u11AA", "\uB127", "\u1102\u1166\u11AA" },
            new string [] { "\uB128", "\uB128", "\u1102\u1166\u11AB", "\uB128", "\u1102\u1166\u11AB" },
            new string [] { "\uB129", "\uB129", "\u1102\u1166\u11AC", "\uB129", "\u1102\u1166\u11AC" },
            new string [] { "\uB12A", "\uB12A", "\u1102\u1166\u11AD", "\uB12A", "\u1102\u1166\u11AD" },
            new string [] { "\uB12B", "\uB12B", "\u1102\u1166\u11AE", "\uB12B", "\u1102\u1166\u11AE" },
            new string [] { "\uB12C", "\uB12C", "\u1102\u1166\u11AF", "\uB12C", "\u1102\u1166\u11AF" },
            new string [] { "\uB12D", "\uB12D", "\u1102\u1166\u11B0", "\uB12D", "\u1102\u1166\u11B0" },
            new string [] { "\uB12E", "\uB12E", "\u1102\u1166\u11B1", "\uB12E", "\u1102\u1166\u11B1" },
            new string [] { "\uB12F", "\uB12F", "\u1102\u1166\u11B2", "\uB12F", "\u1102\u1166\u11B2" },
            new string [] { "\uB130", "\uB130", "\u1102\u1166\u11B3", "\uB130", "\u1102\u1166\u11B3" },
            new string [] { "\uB131", "\uB131", "\u1102\u1166\u11B4", "\uB131", "\u1102\u1166\u11B4" },
            new string [] { "\uB132", "\uB132", "\u1102\u1166\u11B5", "\uB132", "\u1102\u1166\u11B5" },
            new string [] { "\uB133", "\uB133", "\u1102\u1166\u11B6", "\uB133", "\u1102\u1166\u11B6" },
            new string [] { "\uB134", "\uB134", "\u1102\u1166\u11B7", "\uB134", "\u1102\u1166\u11B7" },
            new string [] { "\uB135", "\uB135", "\u1102\u1166\u11B8", "\uB135", "\u1102\u1166\u11B8" },
            new string [] { "\uB136", "\uB136", "\u1102\u1166\u11B9", "\uB136", "\u1102\u1166\u11B9" },
            new string [] { "\uB137", "\uB137", "\u1102\u1166\u11BA", "\uB137", "\u1102\u1166\u11BA" },
            new string [] { "\uB138", "\uB138", "\u1102\u1166\u11BB", "\uB138", "\u1102\u1166\u11BB" },
            new string [] { "\uB139", "\uB139", "\u1102\u1166\u11BC", "\uB139", "\u1102\u1166\u11BC" },
            new string [] { "\uB13A", "\uB13A", "\u1102\u1166\u11BD", "\uB13A", "\u1102\u1166\u11BD" },
            new string [] { "\uB13B", "\uB13B", "\u1102\u1166\u11BE", "\uB13B", "\u1102\u1166\u11BE" },
            new string [] { "\uB13C", "\uB13C", "\u1102\u1166\u11BF", "\uB13C", "\u1102\u1166\u11BF" },
            new string [] { "\uB13D", "\uB13D", "\u1102\u1166\u11C0", "\uB13D", "\u1102\u1166\u11C0" },
            new string [] { "\uB13E", "\uB13E", "\u1102\u1166\u11C1", "\uB13E", "\u1102\u1166\u11C1" },
            new string [] { "\uB13F", "\uB13F", "\u1102\u1166\u11C2", "\uB13F", "\u1102\u1166\u11C2" },
            new string [] { "\uB140", "\uB140", "\u1102\u1167", "\uB140", "\u1102\u1167" },
            new string [] { "\uB141", "\uB141", "\u1102\u1167\u11A8", "\uB141", "\u1102\u1167\u11A8" },
            new string [] { "\uB142", "\uB142", "\u1102\u1167\u11A9", "\uB142", "\u1102\u1167\u11A9" },
            new string [] { "\uB143", "\uB143", "\u1102\u1167\u11AA", "\uB143", "\u1102\u1167\u11AA" },
            new string [] { "\uB144", "\uB144", "\u1102\u1167\u11AB", "\uB144", "\u1102\u1167\u11AB" },
            new string [] { "\uB145", "\uB145", "\u1102\u1167\u11AC", "\uB145", "\u1102\u1167\u11AC" },
            new string [] { "\uB146", "\uB146", "\u1102\u1167\u11AD", "\uB146", "\u1102\u1167\u11AD" },
            new string [] { "\uB147", "\uB147", "\u1102\u1167\u11AE", "\uB147", "\u1102\u1167\u11AE" },
            new string [] { "\uB148", "\uB148", "\u1102\u1167\u11AF", "\uB148", "\u1102\u1167\u11AF" },
            new string [] { "\uB149", "\uB149", "\u1102\u1167\u11B0", "\uB149", "\u1102\u1167\u11B0" },
            new string [] { "\uB14A", "\uB14A", "\u1102\u1167\u11B1", "\uB14A", "\u1102\u1167\u11B1" },
            new string [] { "\uB14B", "\uB14B", "\u1102\u1167\u11B2", "\uB14B", "\u1102\u1167\u11B2" },
            new string [] { "\uB14C", "\uB14C", "\u1102\u1167\u11B3", "\uB14C", "\u1102\u1167\u11B3" },
            new string [] { "\uB14D", "\uB14D", "\u1102\u1167\u11B4", "\uB14D", "\u1102\u1167\u11B4" },
            new string [] { "\uB14E", "\uB14E", "\u1102\u1167\u11B5", "\uB14E", "\u1102\u1167\u11B5" },
            new string [] { "\uB14F", "\uB14F", "\u1102\u1167\u11B6", "\uB14F", "\u1102\u1167\u11B6" },
            new string [] { "\uB150", "\uB150", "\u1102\u1167\u11B7", "\uB150", "\u1102\u1167\u11B7" },
            new string [] { "\uB151", "\uB151", "\u1102\u1167\u11B8", "\uB151", "\u1102\u1167\u11B8" },
            new string [] { "\uB152", "\uB152", "\u1102\u1167\u11B9", "\uB152", "\u1102\u1167\u11B9" },
            new string [] { "\uB153", "\uB153", "\u1102\u1167\u11BA", "\uB153", "\u1102\u1167\u11BA" },
            new string [] { "\uB154", "\uB154", "\u1102\u1167\u11BB", "\uB154", "\u1102\u1167\u11BB" },
            new string [] { "\uB155", "\uB155", "\u1102\u1167\u11BC", "\uB155", "\u1102\u1167\u11BC" },
            new string [] { "\uB156", "\uB156", "\u1102\u1167\u11BD", "\uB156", "\u1102\u1167\u11BD" },
            new string [] { "\uB157", "\uB157", "\u1102\u1167\u11BE", "\uB157", "\u1102\u1167\u11BE" },
            new string [] { "\uB158", "\uB158", "\u1102\u1167\u11BF", "\uB158", "\u1102\u1167\u11BF" },
            new string [] { "\uB159", "\uB159", "\u1102\u1167\u11C0", "\uB159", "\u1102\u1167\u11C0" },
            new string [] { "\uB15A", "\uB15A", "\u1102\u1167\u11C1", "\uB15A", "\u1102\u1167\u11C1" },
            new string [] { "\uB15B", "\uB15B", "\u1102\u1167\u11C2", "\uB15B", "\u1102\u1167\u11C2" },
            new string [] { "\uB15C", "\uB15C", "\u1102\u1168", "\uB15C", "\u1102\u1168" },
            new string [] { "\uB15D", "\uB15D", "\u1102\u1168\u11A8", "\uB15D", "\u1102\u1168\u11A8" },
            new string [] { "\uB15E", "\uB15E", "\u1102\u1168\u11A9", "\uB15E", "\u1102\u1168\u11A9" },
            new string [] { "\uB15F", "\uB15F", "\u1102\u1168\u11AA", "\uB15F", "\u1102\u1168\u11AA" },
            new string [] { "\uB160", "\uB160", "\u1102\u1168\u11AB", "\uB160", "\u1102\u1168\u11AB" },
            new string [] { "\uB161", "\uB161", "\u1102\u1168\u11AC", "\uB161", "\u1102\u1168\u11AC" },
            new string [] { "\uB162", "\uB162", "\u1102\u1168\u11AD", "\uB162", "\u1102\u1168\u11AD" },
            new string [] { "\uB163", "\uB163", "\u1102\u1168\u11AE", "\uB163", "\u1102\u1168\u11AE" },
            new string [] { "\uB164", "\uB164", "\u1102\u1168\u11AF", "\uB164", "\u1102\u1168\u11AF" },
            new string [] { "\uB165", "\uB165", "\u1102\u1168\u11B0", "\uB165", "\u1102\u1168\u11B0" },
            new string [] { "\uB166", "\uB166", "\u1102\u1168\u11B1", "\uB166", "\u1102\u1168\u11B1" },
            new string [] { "\uB167", "\uB167", "\u1102\u1168\u11B2", "\uB167", "\u1102\u1168\u11B2" },
            new string [] { "\uB168", "\uB168", "\u1102\u1168\u11B3", "\uB168", "\u1102\u1168\u11B3" },
            new string [] { "\uB169", "\uB169", "\u1102\u1168\u11B4", "\uB169", "\u1102\u1168\u11B4" },
            new string [] { "\uB16A", "\uB16A", "\u1102\u1168\u11B5", "\uB16A", "\u1102\u1168\u11B5" },
            new string [] { "\uB16B", "\uB16B", "\u1102\u1168\u11B6", "\uB16B", "\u1102\u1168\u11B6" },
            new string [] { "\uB16C", "\uB16C", "\u1102\u1168\u11B7", "\uB16C", "\u1102\u1168\u11B7" },
            new string [] { "\uB16D", "\uB16D", "\u1102\u1168\u11B8", "\uB16D", "\u1102\u1168\u11B8" },
            new string [] { "\uB16E", "\uB16E", "\u1102\u1168\u11B9", "\uB16E", "\u1102\u1168\u11B9" },
            new string [] { "\uB16F", "\uB16F", "\u1102\u1168\u11BA", "\uB16F", "\u1102\u1168\u11BA" },
            new string [] { "\uB170", "\uB170", "\u1102\u1168\u11BB", "\uB170", "\u1102\u1168\u11BB" },
            new string [] { "\uB171", "\uB171", "\u1102\u1168\u11BC", "\uB171", "\u1102\u1168\u11BC" },
            new string [] { "\uB172", "\uB172", "\u1102\u1168\u11BD", "\uB172", "\u1102\u1168\u11BD" },
            new string [] { "\uB173", "\uB173", "\u1102\u1168\u11BE", "\uB173", "\u1102\u1168\u11BE" },
            new string [] { "\uB174", "\uB174", "\u1102\u1168\u11BF", "\uB174", "\u1102\u1168\u11BF" },
            new string [] { "\uB175", "\uB175", "\u1102\u1168\u11C0", "\uB175", "\u1102\u1168\u11C0" },
            new string [] { "\uB176", "\uB176", "\u1102\u1168\u11C1", "\uB176", "\u1102\u1168\u11C1" },
            new string [] { "\uB177", "\uB177", "\u1102\u1168\u11C2", "\uB177", "\u1102\u1168\u11C2" },
            new string [] { "\uB178", "\uB178", "\u1102\u1169", "\uB178", "\u1102\u1169" },
            new string [] { "\uB179", "\uB179", "\u1102\u1169\u11A8", "\uB179", "\u1102\u1169\u11A8" },
            new string [] { "\uB17A", "\uB17A", "\u1102\u1169\u11A9", "\uB17A", "\u1102\u1169\u11A9" },
            new string [] { "\uB17B", "\uB17B", "\u1102\u1169\u11AA", "\uB17B", "\u1102\u1169\u11AA" },
            new string [] { "\uB17C", "\uB17C", "\u1102\u1169\u11AB", "\uB17C", "\u1102\u1169\u11AB" },
            new string [] { "\uB17D", "\uB17D", "\u1102\u1169\u11AC", "\uB17D", "\u1102\u1169\u11AC" },
            new string [] { "\uB17E", "\uB17E", "\u1102\u1169\u11AD", "\uB17E", "\u1102\u1169\u11AD" },
            new string [] { "\uB17F", "\uB17F", "\u1102\u1169\u11AE", "\uB17F", "\u1102\u1169\u11AE" },
            new string [] { "\uB180", "\uB180", "\u1102\u1169\u11AF", "\uB180", "\u1102\u1169\u11AF" },
            new string [] { "\uB181", "\uB181", "\u1102\u1169\u11B0", "\uB181", "\u1102\u1169\u11B0" },
            new string [] { "\uB182", "\uB182", "\u1102\u1169\u11B1", "\uB182", "\u1102\u1169\u11B1" },
            new string [] { "\uB183", "\uB183", "\u1102\u1169\u11B2", "\uB183", "\u1102\u1169\u11B2" },
            new string [] { "\uB184", "\uB184", "\u1102\u1169\u11B3", "\uB184", "\u1102\u1169\u11B3" },
            new string [] { "\uB185", "\uB185", "\u1102\u1169\u11B4", "\uB185", "\u1102\u1169\u11B4" },
            new string [] { "\uB186", "\uB186", "\u1102\u1169\u11B5", "\uB186", "\u1102\u1169\u11B5" },
            new string [] { "\uB187", "\uB187", "\u1102\u1169\u11B6", "\uB187", "\u1102\u1169\u11B6" },
            new string [] { "\uB188", "\uB188", "\u1102\u1169\u11B7", "\uB188", "\u1102\u1169\u11B7" },
            new string [] { "\uB189", "\uB189", "\u1102\u1169\u11B8", "\uB189", "\u1102\u1169\u11B8" },
            new string [] { "\uB18A", "\uB18A", "\u1102\u1169\u11B9", "\uB18A", "\u1102\u1169\u11B9" },
            new string [] { "\uB18B", "\uB18B", "\u1102\u1169\u11BA", "\uB18B", "\u1102\u1169\u11BA" },
            new string [] { "\uB18C", "\uB18C", "\u1102\u1169\u11BB", "\uB18C", "\u1102\u1169\u11BB" },
            new string [] { "\uB18D", "\uB18D", "\u1102\u1169\u11BC", "\uB18D", "\u1102\u1169\u11BC" },
            new string [] { "\uB18E", "\uB18E", "\u1102\u1169\u11BD", "\uB18E", "\u1102\u1169\u11BD" },
            new string [] { "\uB18F", "\uB18F", "\u1102\u1169\u11BE", "\uB18F", "\u1102\u1169\u11BE" },
            new string [] { "\uB190", "\uB190", "\u1102\u1169\u11BF", "\uB190", "\u1102\u1169\u11BF" },
            new string [] { "\uB191", "\uB191", "\u1102\u1169\u11C0", "\uB191", "\u1102\u1169\u11C0" },
            new string [] { "\uB192", "\uB192", "\u1102\u1169\u11C1", "\uB192", "\u1102\u1169\u11C1" },
            new string [] { "\uB193", "\uB193", "\u1102\u1169\u11C2", "\uB193", "\u1102\u1169\u11C2" },
            new string [] { "\uB194", "\uB194", "\u1102\u116A", "\uB194", "\u1102\u116A" },
            new string [] { "\uB195", "\uB195", "\u1102\u116A\u11A8", "\uB195", "\u1102\u116A\u11A8" },
            new string [] { "\uB196", "\uB196", "\u1102\u116A\u11A9", "\uB196", "\u1102\u116A\u11A9" },
            new string [] { "\uB197", "\uB197", "\u1102\u116A\u11AA", "\uB197", "\u1102\u116A\u11AA" },
            new string [] { "\uB198", "\uB198", "\u1102\u116A\u11AB", "\uB198", "\u1102\u116A\u11AB" },
            new string [] { "\uB199", "\uB199", "\u1102\u116A\u11AC", "\uB199", "\u1102\u116A\u11AC" },
            new string [] { "\uB19A", "\uB19A", "\u1102\u116A\u11AD", "\uB19A", "\u1102\u116A\u11AD" },
            new string [] { "\uB19B", "\uB19B", "\u1102\u116A\u11AE", "\uB19B", "\u1102\u116A\u11AE" },
            new string [] { "\uB19C", "\uB19C", "\u1102\u116A\u11AF", "\uB19C", "\u1102\u116A\u11AF" },
            new string [] { "\uB19D", "\uB19D", "\u1102\u116A\u11B0", "\uB19D", "\u1102\u116A\u11B0" },
            new string [] { "\uB19E", "\uB19E", "\u1102\u116A\u11B1", "\uB19E", "\u1102\u116A\u11B1" },
            new string [] { "\uB19F", "\uB19F", "\u1102\u116A\u11B2", "\uB19F", "\u1102\u116A\u11B2" },
            new string [] { "\uB1A0", "\uB1A0", "\u1102\u116A\u11B3", "\uB1A0", "\u1102\u116A\u11B3" },
            new string [] { "\uB1A1", "\uB1A1", "\u1102\u116A\u11B4", "\uB1A1", "\u1102\u116A\u11B4" },
            new string [] { "\uB1A2", "\uB1A2", "\u1102\u116A\u11B5", "\uB1A2", "\u1102\u116A\u11B5" },
            new string [] { "\uB1A3", "\uB1A3", "\u1102\u116A\u11B6", "\uB1A3", "\u1102\u116A\u11B6" },
            new string [] { "\uB1A4", "\uB1A4", "\u1102\u116A\u11B7", "\uB1A4", "\u1102\u116A\u11B7" },
            new string [] { "\uB1A5", "\uB1A5", "\u1102\u116A\u11B8", "\uB1A5", "\u1102\u116A\u11B8" },
            new string [] { "\uB1A6", "\uB1A6", "\u1102\u116A\u11B9", "\uB1A6", "\u1102\u116A\u11B9" },
            new string [] { "\uB1A7", "\uB1A7", "\u1102\u116A\u11BA", "\uB1A7", "\u1102\u116A\u11BA" },
            new string [] { "\uB1A8", "\uB1A8", "\u1102\u116A\u11BB", "\uB1A8", "\u1102\u116A\u11BB" },
            new string [] { "\uB1A9", "\uB1A9", "\u1102\u116A\u11BC", "\uB1A9", "\u1102\u116A\u11BC" },
            new string [] { "\uB1AA", "\uB1AA", "\u1102\u116A\u11BD", "\uB1AA", "\u1102\u116A\u11BD" },
            new string [] { "\uB1AB", "\uB1AB", "\u1102\u116A\u11BE", "\uB1AB", "\u1102\u116A\u11BE" },
            new string [] { "\uB1AC", "\uB1AC", "\u1102\u116A\u11BF", "\uB1AC", "\u1102\u116A\u11BF" },
            new string [] { "\uB1AD", "\uB1AD", "\u1102\u116A\u11C0", "\uB1AD", "\u1102\u116A\u11C0" },
            new string [] { "\uB1AE", "\uB1AE", "\u1102\u116A\u11C1", "\uB1AE", "\u1102\u116A\u11C1" },
            new string [] { "\uB1AF", "\uB1AF", "\u1102\u116A\u11C2", "\uB1AF", "\u1102\u116A\u11C2" },
            new string [] { "\uB1B0", "\uB1B0", "\u1102\u116B", "\uB1B0", "\u1102\u116B" },
            new string [] { "\uB1B1", "\uB1B1", "\u1102\u116B\u11A8", "\uB1B1", "\u1102\u116B\u11A8" },
            new string [] { "\uB1B2", "\uB1B2", "\u1102\u116B\u11A9", "\uB1B2", "\u1102\u116B\u11A9" },
            new string [] { "\uB1B3", "\uB1B3", "\u1102\u116B\u11AA", "\uB1B3", "\u1102\u116B\u11AA" },
            new string [] { "\uB1B4", "\uB1B4", "\u1102\u116B\u11AB", "\uB1B4", "\u1102\u116B\u11AB" },
            new string [] { "\uB1B5", "\uB1B5", "\u1102\u116B\u11AC", "\uB1B5", "\u1102\u116B\u11AC" },
            new string [] { "\uB1B6", "\uB1B6", "\u1102\u116B\u11AD", "\uB1B6", "\u1102\u116B\u11AD" },
            new string [] { "\uB1B7", "\uB1B7", "\u1102\u116B\u11AE", "\uB1B7", "\u1102\u116B\u11AE" },
            new string [] { "\uB1B8", "\uB1B8", "\u1102\u116B\u11AF", "\uB1B8", "\u1102\u116B\u11AF" },
            new string [] { "\uB1B9", "\uB1B9", "\u1102\u116B\u11B0", "\uB1B9", "\u1102\u116B\u11B0" },
            new string [] { "\uB1BA", "\uB1BA", "\u1102\u116B\u11B1", "\uB1BA", "\u1102\u116B\u11B1" },
            new string [] { "\uB1BB", "\uB1BB", "\u1102\u116B\u11B2", "\uB1BB", "\u1102\u116B\u11B2" },
            new string [] { "\uB1BC", "\uB1BC", "\u1102\u116B\u11B3", "\uB1BC", "\u1102\u116B\u11B3" },
            new string [] { "\uB1BD", "\uB1BD", "\u1102\u116B\u11B4", "\uB1BD", "\u1102\u116B\u11B4" },
            new string [] { "\uB1BE", "\uB1BE", "\u1102\u116B\u11B5", "\uB1BE", "\u1102\u116B\u11B5" },
            new string [] { "\uB1BF", "\uB1BF", "\u1102\u116B\u11B6", "\uB1BF", "\u1102\u116B\u11B6" },
            new string [] { "\uB1C0", "\uB1C0", "\u1102\u116B\u11B7", "\uB1C0", "\u1102\u116B\u11B7" },
            new string [] { "\uB1C1", "\uB1C1", "\u1102\u116B\u11B8", "\uB1C1", "\u1102\u116B\u11B8" },
            new string [] { "\uB1C2", "\uB1C2", "\u1102\u116B\u11B9", "\uB1C2", "\u1102\u116B\u11B9" },
            new string [] { "\uB1C3", "\uB1C3", "\u1102\u116B\u11BA", "\uB1C3", "\u1102\u116B\u11BA" },
            new string [] { "\uB1C4", "\uB1C4", "\u1102\u116B\u11BB", "\uB1C4", "\u1102\u116B\u11BB" },
            new string [] { "\uB1C5", "\uB1C5", "\u1102\u116B\u11BC", "\uB1C5", "\u1102\u116B\u11BC" },
            new string [] { "\uB1C6", "\uB1C6", "\u1102\u116B\u11BD", "\uB1C6", "\u1102\u116B\u11BD" },
            new string [] { "\uB1C7", "\uB1C7", "\u1102\u116B\u11BE", "\uB1C7", "\u1102\u116B\u11BE" },
            new string [] { "\uB1C8", "\uB1C8", "\u1102\u116B\u11BF", "\uB1C8", "\u1102\u116B\u11BF" },
            new string [] { "\uB1C9", "\uB1C9", "\u1102\u116B\u11C0", "\uB1C9", "\u1102\u116B\u11C0" },
            new string [] { "\uB1CA", "\uB1CA", "\u1102\u116B\u11C1", "\uB1CA", "\u1102\u116B\u11C1" },
            new string [] { "\uB1CB", "\uB1CB", "\u1102\u116B\u11C2", "\uB1CB", "\u1102\u116B\u11C2" },
            new string [] { "\uB1CC", "\uB1CC", "\u1102\u116C", "\uB1CC", "\u1102\u116C" },
            new string [] { "\uB1CD", "\uB1CD", "\u1102\u116C\u11A8", "\uB1CD", "\u1102\u116C\u11A8" },
            new string [] { "\uB1CE", "\uB1CE", "\u1102\u116C\u11A9", "\uB1CE", "\u1102\u116C\u11A9" },
            new string [] { "\uB1CF", "\uB1CF", "\u1102\u116C\u11AA", "\uB1CF", "\u1102\u116C\u11AA" },
            new string [] { "\uB1D0", "\uB1D0", "\u1102\u116C\u11AB", "\uB1D0", "\u1102\u116C\u11AB" },
            new string [] { "\uB1D1", "\uB1D1", "\u1102\u116C\u11AC", "\uB1D1", "\u1102\u116C\u11AC" },
            new string [] { "\uB1D2", "\uB1D2", "\u1102\u116C\u11AD", "\uB1D2", "\u1102\u116C\u11AD" },
            new string [] { "\uB1D3", "\uB1D3", "\u1102\u116C\u11AE", "\uB1D3", "\u1102\u116C\u11AE" },
            new string [] { "\uB1D4", "\uB1D4", "\u1102\u116C\u11AF", "\uB1D4", "\u1102\u116C\u11AF" },
            new string [] { "\uB1D5", "\uB1D5", "\u1102\u116C\u11B0", "\uB1D5", "\u1102\u116C\u11B0" },
            new string [] { "\uB1D6", "\uB1D6", "\u1102\u116C\u11B1", "\uB1D6", "\u1102\u116C\u11B1" },
            new string [] { "\uB1D7", "\uB1D7", "\u1102\u116C\u11B2", "\uB1D7", "\u1102\u116C\u11B2" },
            new string [] { "\uB1D8", "\uB1D8", "\u1102\u116C\u11B3", "\uB1D8", "\u1102\u116C\u11B3" },
            new string [] { "\uB1D9", "\uB1D9", "\u1102\u116C\u11B4", "\uB1D9", "\u1102\u116C\u11B4" },
            new string [] { "\uB1DA", "\uB1DA", "\u1102\u116C\u11B5", "\uB1DA", "\u1102\u116C\u11B5" },
            new string [] { "\uB1DB", "\uB1DB", "\u1102\u116C\u11B6", "\uB1DB", "\u1102\u116C\u11B6" },
            new string [] { "\uB1DC", "\uB1DC", "\u1102\u116C\u11B7", "\uB1DC", "\u1102\u116C\u11B7" },
            new string [] { "\uB1DD", "\uB1DD", "\u1102\u116C\u11B8", "\uB1DD", "\u1102\u116C\u11B8" },
            new string [] { "\uB1DE", "\uB1DE", "\u1102\u116C\u11B9", "\uB1DE", "\u1102\u116C\u11B9" },
            new string [] { "\uB1DF", "\uB1DF", "\u1102\u116C\u11BA", "\uB1DF", "\u1102\u116C\u11BA" },
            new string [] { "\uB1E0", "\uB1E0", "\u1102\u116C\u11BB", "\uB1E0", "\u1102\u116C\u11BB" },
            new string [] { "\uB1E1", "\uB1E1", "\u1102\u116C\u11BC", "\uB1E1", "\u1102\u116C\u11BC" },
            new string [] { "\uB1E2", "\uB1E2", "\u1102\u116C\u11BD", "\uB1E2", "\u1102\u116C\u11BD" },
            new string [] { "\uB1E3", "\uB1E3", "\u1102\u116C\u11BE", "\uB1E3", "\u1102\u116C\u11BE" },
            new string [] { "\uB1E4", "\uB1E4", "\u1102\u116C\u11BF", "\uB1E4", "\u1102\u116C\u11BF" },
            new string [] { "\uB1E5", "\uB1E5", "\u1102\u116C\u11C0", "\uB1E5", "\u1102\u116C\u11C0" },
            new string [] { "\uB1E6", "\uB1E6", "\u1102\u116C\u11C1", "\uB1E6", "\u1102\u116C\u11C1" },
            new string [] { "\uB1E7", "\uB1E7", "\u1102\u116C\u11C2", "\uB1E7", "\u1102\u116C\u11C2" },
            new string [] { "\uB1E8", "\uB1E8", "\u1102\u116D", "\uB1E8", "\u1102\u116D" },
            new string [] { "\uB1E9", "\uB1E9", "\u1102\u116D\u11A8", "\uB1E9", "\u1102\u116D\u11A8" },
            new string [] { "\uB1EA", "\uB1EA", "\u1102\u116D\u11A9", "\uB1EA", "\u1102\u116D\u11A9" },
            new string [] { "\uB1EB", "\uB1EB", "\u1102\u116D\u11AA", "\uB1EB", "\u1102\u116D\u11AA" },
            new string [] { "\uB1EC", "\uB1EC", "\u1102\u116D\u11AB", "\uB1EC", "\u1102\u116D\u11AB" },
            new string [] { "\uB1ED", "\uB1ED", "\u1102\u116D\u11AC", "\uB1ED", "\u1102\u116D\u11AC" },
            new string [] { "\uB1EE", "\uB1EE", "\u1102\u116D\u11AD", "\uB1EE", "\u1102\u116D\u11AD" },
            new string [] { "\uB1EF", "\uB1EF", "\u1102\u116D\u11AE", "\uB1EF", "\u1102\u116D\u11AE" },
            new string [] { "\uB1F0", "\uB1F0", "\u1102\u116D\u11AF", "\uB1F0", "\u1102\u116D\u11AF" },
            new string [] { "\uB1F1", "\uB1F1", "\u1102\u116D\u11B0", "\uB1F1", "\u1102\u116D\u11B0" },
            new string [] { "\uB1F2", "\uB1F2", "\u1102\u116D\u11B1", "\uB1F2", "\u1102\u116D\u11B1" },
            new string [] { "\uB1F3", "\uB1F3", "\u1102\u116D\u11B2", "\uB1F3", "\u1102\u116D\u11B2" },
            new string [] { "\uB1F4", "\uB1F4", "\u1102\u116D\u11B3", "\uB1F4", "\u1102\u116D\u11B3" },
            new string [] { "\uB1F5", "\uB1F5", "\u1102\u116D\u11B4", "\uB1F5", "\u1102\u116D\u11B4" },
            new string [] { "\uB1F6", "\uB1F6", "\u1102\u116D\u11B5", "\uB1F6", "\u1102\u116D\u11B5" },
            new string [] { "\uB1F7", "\uB1F7", "\u1102\u116D\u11B6", "\uB1F7", "\u1102\u116D\u11B6" },
            new string [] { "\uB1F8", "\uB1F8", "\u1102\u116D\u11B7", "\uB1F8", "\u1102\u116D\u11B7" },
            new string [] { "\uB1F9", "\uB1F9", "\u1102\u116D\u11B8", "\uB1F9", "\u1102\u116D\u11B8" },
            new string [] { "\uB1FA", "\uB1FA", "\u1102\u116D\u11B9", "\uB1FA", "\u1102\u116D\u11B9" },
            new string [] { "\uB1FB", "\uB1FB", "\u1102\u116D\u11BA", "\uB1FB", "\u1102\u116D\u11BA" },
            new string [] { "\uB1FC", "\uB1FC", "\u1102\u116D\u11BB", "\uB1FC", "\u1102\u116D\u11BB" },
            new string [] { "\uB1FD", "\uB1FD", "\u1102\u116D\u11BC", "\uB1FD", "\u1102\u116D\u11BC" },
            new string [] { "\uB1FE", "\uB1FE", "\u1102\u116D\u11BD", "\uB1FE", "\u1102\u116D\u11BD" },
            new string [] { "\uB1FF", "\uB1FF", "\u1102\u116D\u11BE", "\uB1FF", "\u1102\u116D\u11BE" },
            new string [] { "\uB200", "\uB200", "\u1102\u116D\u11BF", "\uB200", "\u1102\u116D\u11BF" },
            new string [] { "\uB201", "\uB201", "\u1102\u116D\u11C0", "\uB201", "\u1102\u116D\u11C0" },
            new string [] { "\uB202", "\uB202", "\u1102\u116D\u11C1", "\uB202", "\u1102\u116D\u11C1" },
            new string [] { "\uB203", "\uB203", "\u1102\u116D\u11C2", "\uB203", "\u1102\u116D\u11C2" },
            new string [] { "\uB204", "\uB204", "\u1102\u116E", "\uB204", "\u1102\u116E" },
            new string [] { "\uB205", "\uB205", "\u1102\u116E\u11A8", "\uB205", "\u1102\u116E\u11A8" },
            new string [] { "\uB206", "\uB206", "\u1102\u116E\u11A9", "\uB206", "\u1102\u116E\u11A9" },
            new string [] { "\uB207", "\uB207", "\u1102\u116E\u11AA", "\uB207", "\u1102\u116E\u11AA" },
            new string [] { "\uB208", "\uB208", "\u1102\u116E\u11AB", "\uB208", "\u1102\u116E\u11AB" },
            new string [] { "\uB209", "\uB209", "\u1102\u116E\u11AC", "\uB209", "\u1102\u116E\u11AC" },
            new string [] { "\uB20A", "\uB20A", "\u1102\u116E\u11AD", "\uB20A", "\u1102\u116E\u11AD" },
            new string [] { "\uB20B", "\uB20B", "\u1102\u116E\u11AE", "\uB20B", "\u1102\u116E\u11AE" },
            new string [] { "\uB20C", "\uB20C", "\u1102\u116E\u11AF", "\uB20C", "\u1102\u116E\u11AF" },
            new string [] { "\uB20D", "\uB20D", "\u1102\u116E\u11B0", "\uB20D", "\u1102\u116E\u11B0" },
            new string [] { "\uB20E", "\uB20E", "\u1102\u116E\u11B1", "\uB20E", "\u1102\u116E\u11B1" },
            new string [] { "\uB20F", "\uB20F", "\u1102\u116E\u11B2", "\uB20F", "\u1102\u116E\u11B2" },
            new string [] { "\uB210", "\uB210", "\u1102\u116E\u11B3", "\uB210", "\u1102\u116E\u11B3" },
            new string [] { "\uB211", "\uB211", "\u1102\u116E\u11B4", "\uB211", "\u1102\u116E\u11B4" },
            new string [] { "\uB212", "\uB212", "\u1102\u116E\u11B5", "\uB212", "\u1102\u116E\u11B5" },
            new string [] { "\uB213", "\uB213", "\u1102\u116E\u11B6", "\uB213", "\u1102\u116E\u11B6" },
            new string [] { "\uB214", "\uB214", "\u1102\u116E\u11B7", "\uB214", "\u1102\u116E\u11B7" },
            new string [] { "\uB215", "\uB215", "\u1102\u116E\u11B8", "\uB215", "\u1102\u116E\u11B8" },
            new string [] { "\uB216", "\uB216", "\u1102\u116E\u11B9", "\uB216", "\u1102\u116E\u11B9" },
            new string [] { "\uB217", "\uB217", "\u1102\u116E\u11BA", "\uB217", "\u1102\u116E\u11BA" },
            new string [] { "\uB218", "\uB218", "\u1102\u116E\u11BB", "\uB218", "\u1102\u116E\u11BB" },
            new string [] { "\uB219", "\uB219", "\u1102\u116E\u11BC", "\uB219", "\u1102\u116E\u11BC" },
            new string [] { "\uB21A", "\uB21A", "\u1102\u116E\u11BD", "\uB21A", "\u1102\u116E\u11BD" },
            new string [] { "\uB21B", "\uB21B", "\u1102\u116E\u11BE", "\uB21B", "\u1102\u116E\u11BE" },
            new string [] { "\uB21C", "\uB21C", "\u1102\u116E\u11BF", "\uB21C", "\u1102\u116E\u11BF" },
            new string [] { "\uB21D", "\uB21D", "\u1102\u116E\u11C0", "\uB21D", "\u1102\u116E\u11C0" },
            new string [] { "\uB21E", "\uB21E", "\u1102\u116E\u11C1", "\uB21E", "\u1102\u116E\u11C1" },
            new string [] { "\uB21F", "\uB21F", "\u1102\u116E\u11C2", "\uB21F", "\u1102\u116E\u11C2" },
            new string [] { "\uB220", "\uB220", "\u1102\u116F", "\uB220", "\u1102\u116F" },
            new string [] { "\uB221", "\uB221", "\u1102\u116F\u11A8", "\uB221", "\u1102\u116F\u11A8" },
            new string [] { "\uB222", "\uB222", "\u1102\u116F\u11A9", "\uB222", "\u1102\u116F\u11A9" },
            new string [] { "\uB223", "\uB223", "\u1102\u116F\u11AA", "\uB223", "\u1102\u116F\u11AA" },
            new string [] { "\uB224", "\uB224", "\u1102\u116F\u11AB", "\uB224", "\u1102\u116F\u11AB" },
            new string [] { "\uB225", "\uB225", "\u1102\u116F\u11AC", "\uB225", "\u1102\u116F\u11AC" },
            new string [] { "\uB226", "\uB226", "\u1102\u116F\u11AD", "\uB226", "\u1102\u116F\u11AD" },
            new string [] { "\uB227", "\uB227", "\u1102\u116F\u11AE", "\uB227", "\u1102\u116F\u11AE" },
            new string [] { "\uB228", "\uB228", "\u1102\u116F\u11AF", "\uB228", "\u1102\u116F\u11AF" },
            new string [] { "\uB229", "\uB229", "\u1102\u116F\u11B0", "\uB229", "\u1102\u116F\u11B0" },
            new string [] { "\uB22A", "\uB22A", "\u1102\u116F\u11B1", "\uB22A", "\u1102\u116F\u11B1" },
            new string [] { "\uB22B", "\uB22B", "\u1102\u116F\u11B2", "\uB22B", "\u1102\u116F\u11B2" },
            new string [] { "\uB22C", "\uB22C", "\u1102\u116F\u11B3", "\uB22C", "\u1102\u116F\u11B3" },
            new string [] { "\uB22D", "\uB22D", "\u1102\u116F\u11B4", "\uB22D", "\u1102\u116F\u11B4" },
            new string [] { "\uB22E", "\uB22E", "\u1102\u116F\u11B5", "\uB22E", "\u1102\u116F\u11B5" },
            new string [] { "\uB22F", "\uB22F", "\u1102\u116F\u11B6", "\uB22F", "\u1102\u116F\u11B6" },
            new string [] { "\uB230", "\uB230", "\u1102\u116F\u11B7", "\uB230", "\u1102\u116F\u11B7" },
            new string [] { "\uB231", "\uB231", "\u1102\u116F\u11B8", "\uB231", "\u1102\u116F\u11B8" },
            new string [] { "\uB232", "\uB232", "\u1102\u116F\u11B9", "\uB232", "\u1102\u116F\u11B9" },
            new string [] { "\uB233", "\uB233", "\u1102\u116F\u11BA", "\uB233", "\u1102\u116F\u11BA" },
            new string [] { "\uB234", "\uB234", "\u1102\u116F\u11BB", "\uB234", "\u1102\u116F\u11BB" },
            new string [] { "\uB235", "\uB235", "\u1102\u116F\u11BC", "\uB235", "\u1102\u116F\u11BC" },
            new string [] { "\uB236", "\uB236", "\u1102\u116F\u11BD", "\uB236", "\u1102\u116F\u11BD" },
            new string [] { "\uB237", "\uB237", "\u1102\u116F\u11BE", "\uB237", "\u1102\u116F\u11BE" },
            new string [] { "\uB238", "\uB238", "\u1102\u116F\u11BF", "\uB238", "\u1102\u116F\u11BF" },
            new string [] { "\uB239", "\uB239", "\u1102\u116F\u11C0", "\uB239", "\u1102\u116F\u11C0" },
            new string [] { "\uB23A", "\uB23A", "\u1102\u116F\u11C1", "\uB23A", "\u1102\u116F\u11C1" },
            new string [] { "\uB23B", "\uB23B", "\u1102\u116F\u11C2", "\uB23B", "\u1102\u116F\u11C2" },
            new string [] { "\uB23C", "\uB23C", "\u1102\u1170", "\uB23C", "\u1102\u1170" },
            new string [] { "\uB23D", "\uB23D", "\u1102\u1170\u11A8", "\uB23D", "\u1102\u1170\u11A8" },
            new string [] { "\uB23E", "\uB23E", "\u1102\u1170\u11A9", "\uB23E", "\u1102\u1170\u11A9" },
            new string [] { "\uB23F", "\uB23F", "\u1102\u1170\u11AA", "\uB23F", "\u1102\u1170\u11AA" },
            new string [] { "\uB240", "\uB240", "\u1102\u1170\u11AB", "\uB240", "\u1102\u1170\u11AB" },
            new string [] { "\uB241", "\uB241", "\u1102\u1170\u11AC", "\uB241", "\u1102\u1170\u11AC" },
            new string [] { "\uB242", "\uB242", "\u1102\u1170\u11AD", "\uB242", "\u1102\u1170\u11AD" },
            new string [] { "\uB243", "\uB243", "\u1102\u1170\u11AE", "\uB243", "\u1102\u1170\u11AE" },
            new string [] { "\uB244", "\uB244", "\u1102\u1170\u11AF", "\uB244", "\u1102\u1170\u11AF" },
            new string [] { "\uB245", "\uB245", "\u1102\u1170\u11B0", "\uB245", "\u1102\u1170\u11B0" },
            new string [] { "\uB246", "\uB246", "\u1102\u1170\u11B1", "\uB246", "\u1102\u1170\u11B1" },
            new string [] { "\uB247", "\uB247", "\u1102\u1170\u11B2", "\uB247", "\u1102\u1170\u11B2" },
            new string [] { "\uB248", "\uB248", "\u1102\u1170\u11B3", "\uB248", "\u1102\u1170\u11B3" },
            new string [] { "\uB249", "\uB249", "\u1102\u1170\u11B4", "\uB249", "\u1102\u1170\u11B4" },
            new string [] { "\uB24A", "\uB24A", "\u1102\u1170\u11B5", "\uB24A", "\u1102\u1170\u11B5" },
            new string [] { "\uB24B", "\uB24B", "\u1102\u1170\u11B6", "\uB24B", "\u1102\u1170\u11B6" },
            new string [] { "\uB24C", "\uB24C", "\u1102\u1170\u11B7", "\uB24C", "\u1102\u1170\u11B7" },
            new string [] { "\uB24D", "\uB24D", "\u1102\u1170\u11B8", "\uB24D", "\u1102\u1170\u11B8" },
            new string [] { "\uB24E", "\uB24E", "\u1102\u1170\u11B9", "\uB24E", "\u1102\u1170\u11B9" },
            new string [] { "\uB24F", "\uB24F", "\u1102\u1170\u11BA", "\uB24F", "\u1102\u1170\u11BA" },
            new string [] { "\uB250", "\uB250", "\u1102\u1170\u11BB", "\uB250", "\u1102\u1170\u11BB" },
            new string [] { "\uB251", "\uB251", "\u1102\u1170\u11BC", "\uB251", "\u1102\u1170\u11BC" },
            new string [] { "\uB252", "\uB252", "\u1102\u1170\u11BD", "\uB252", "\u1102\u1170\u11BD" },
            new string [] { "\uB253", "\uB253", "\u1102\u1170\u11BE", "\uB253", "\u1102\u1170\u11BE" },
            new string [] { "\uB254", "\uB254", "\u1102\u1170\u11BF", "\uB254", "\u1102\u1170\u11BF" },
            new string [] { "\uB255", "\uB255", "\u1102\u1170\u11C0", "\uB255", "\u1102\u1170\u11C0" },
            new string [] { "\uB256", "\uB256", "\u1102\u1170\u11C1", "\uB256", "\u1102\u1170\u11C1" },
            new string [] { "\uB257", "\uB257", "\u1102\u1170\u11C2", "\uB257", "\u1102\u1170\u11C2" },
            new string [] { "\uB258", "\uB258", "\u1102\u1171", "\uB258", "\u1102\u1171" },
            new string [] { "\uB259", "\uB259", "\u1102\u1171\u11A8", "\uB259", "\u1102\u1171\u11A8" },
            new string [] { "\uB25A", "\uB25A", "\u1102\u1171\u11A9", "\uB25A", "\u1102\u1171\u11A9" },
            new string [] { "\uB25B", "\uB25B", "\u1102\u1171\u11AA", "\uB25B", "\u1102\u1171\u11AA" },
            new string [] { "\uB25C", "\uB25C", "\u1102\u1171\u11AB", "\uB25C", "\u1102\u1171\u11AB" },
            new string [] { "\uB25D", "\uB25D", "\u1102\u1171\u11AC", "\uB25D", "\u1102\u1171\u11AC" },
            new string [] { "\uB25E", "\uB25E", "\u1102\u1171\u11AD", "\uB25E", "\u1102\u1171\u11AD" },
            new string [] { "\uB25F", "\uB25F", "\u1102\u1171\u11AE", "\uB25F", "\u1102\u1171\u11AE" },
            new string [] { "\uB260", "\uB260", "\u1102\u1171\u11AF", "\uB260", "\u1102\u1171\u11AF" },
            new string [] { "\uB261", "\uB261", "\u1102\u1171\u11B0", "\uB261", "\u1102\u1171\u11B0" },
            new string [] { "\uB262", "\uB262", "\u1102\u1171\u11B1", "\uB262", "\u1102\u1171\u11B1" },
            new string [] { "\uB263", "\uB263", "\u1102\u1171\u11B2", "\uB263", "\u1102\u1171\u11B2" },
            new string [] { "\uB264", "\uB264", "\u1102\u1171\u11B3", "\uB264", "\u1102\u1171\u11B3" },
            new string [] { "\uB265", "\uB265", "\u1102\u1171\u11B4", "\uB265", "\u1102\u1171\u11B4" },
            new string [] { "\uB266", "\uB266", "\u1102\u1171\u11B5", "\uB266", "\u1102\u1171\u11B5" },
            new string [] { "\uB267", "\uB267", "\u1102\u1171\u11B6", "\uB267", "\u1102\u1171\u11B6" },
            new string [] { "\uB268", "\uB268", "\u1102\u1171\u11B7", "\uB268", "\u1102\u1171\u11B7" },
            new string [] { "\uB269", "\uB269", "\u1102\u1171\u11B8", "\uB269", "\u1102\u1171\u11B8" },
            new string [] { "\uB26A", "\uB26A", "\u1102\u1171\u11B9", "\uB26A", "\u1102\u1171\u11B9" },
            new string [] { "\uB26B", "\uB26B", "\u1102\u1171\u11BA", "\uB26B", "\u1102\u1171\u11BA" },
            new string [] { "\uB26C", "\uB26C", "\u1102\u1171\u11BB", "\uB26C", "\u1102\u1171\u11BB" },
            new string [] { "\uB26D", "\uB26D", "\u1102\u1171\u11BC", "\uB26D", "\u1102\u1171\u11BC" },
            new string [] { "\uB26E", "\uB26E", "\u1102\u1171\u11BD", "\uB26E", "\u1102\u1171\u11BD" },
            new string [] { "\uB26F", "\uB26F", "\u1102\u1171\u11BE", "\uB26F", "\u1102\u1171\u11BE" },
            new string [] { "\uB270", "\uB270", "\u1102\u1171\u11BF", "\uB270", "\u1102\u1171\u11BF" },
            new string [] { "\uB271", "\uB271", "\u1102\u1171\u11C0", "\uB271", "\u1102\u1171\u11C0" },
            new string [] { "\uB272", "\uB272", "\u1102\u1171\u11C1", "\uB272", "\u1102\u1171\u11C1" },
            new string [] { "\uB273", "\uB273", "\u1102\u1171\u11C2", "\uB273", "\u1102\u1171\u11C2" },
            new string [] { "\uB274", "\uB274", "\u1102\u1172", "\uB274", "\u1102\u1172" },
            new string [] { "\uB275", "\uB275", "\u1102\u1172\u11A8", "\uB275", "\u1102\u1172\u11A8" },
            new string [] { "\uB276", "\uB276", "\u1102\u1172\u11A9", "\uB276", "\u1102\u1172\u11A9" },
            new string [] { "\uB277", "\uB277", "\u1102\u1172\u11AA", "\uB277", "\u1102\u1172\u11AA" },
            new string [] { "\uB278", "\uB278", "\u1102\u1172\u11AB", "\uB278", "\u1102\u1172\u11AB" },
            new string [] { "\uB279", "\uB279", "\u1102\u1172\u11AC", "\uB279", "\u1102\u1172\u11AC" },
            new string [] { "\uB27A", "\uB27A", "\u1102\u1172\u11AD", "\uB27A", "\u1102\u1172\u11AD" },
            new string [] { "\uB27B", "\uB27B", "\u1102\u1172\u11AE", "\uB27B", "\u1102\u1172\u11AE" },
            new string [] { "\uB27C", "\uB27C", "\u1102\u1172\u11AF", "\uB27C", "\u1102\u1172\u11AF" },
            new string [] { "\uB27D", "\uB27D", "\u1102\u1172\u11B0", "\uB27D", "\u1102\u1172\u11B0" },
            new string [] { "\uB27E", "\uB27E", "\u1102\u1172\u11B1", "\uB27E", "\u1102\u1172\u11B1" },
            new string [] { "\uB27F", "\uB27F", "\u1102\u1172\u11B2", "\uB27F", "\u1102\u1172\u11B2" },
            new string [] { "\uB280", "\uB280", "\u1102\u1172\u11B3", "\uB280", "\u1102\u1172\u11B3" },
            new string [] { "\uB281", "\uB281", "\u1102\u1172\u11B4", "\uB281", "\u1102\u1172\u11B4" },
            new string [] { "\uB282", "\uB282", "\u1102\u1172\u11B5", "\uB282", "\u1102\u1172\u11B5" },
            new string [] { "\uB283", "\uB283", "\u1102\u1172\u11B6", "\uB283", "\u1102\u1172\u11B6" },
            new string [] { "\uB284", "\uB284", "\u1102\u1172\u11B7", "\uB284", "\u1102\u1172\u11B7" },
            new string [] { "\uB285", "\uB285", "\u1102\u1172\u11B8", "\uB285", "\u1102\u1172\u11B8" },
            new string [] { "\uB286", "\uB286", "\u1102\u1172\u11B9", "\uB286", "\u1102\u1172\u11B9" },
            new string [] { "\uB287", "\uB287", "\u1102\u1172\u11BA", "\uB287", "\u1102\u1172\u11BA" },
            new string [] { "\uB288", "\uB288", "\u1102\u1172\u11BB", "\uB288", "\u1102\u1172\u11BB" },
            new string [] { "\uB289", "\uB289", "\u1102\u1172\u11BC", "\uB289", "\u1102\u1172\u11BC" },
            new string [] { "\uB28A", "\uB28A", "\u1102\u1172\u11BD", "\uB28A", "\u1102\u1172\u11BD" },
            new string [] { "\uB28B", "\uB28B", "\u1102\u1172\u11BE", "\uB28B", "\u1102\u1172\u11BE" },
            new string [] { "\uB28C", "\uB28C", "\u1102\u1172\u11BF", "\uB28C", "\u1102\u1172\u11BF" },
            new string [] { "\uB28D", "\uB28D", "\u1102\u1172\u11C0", "\uB28D", "\u1102\u1172\u11C0" },
            new string [] { "\uB28E", "\uB28E", "\u1102\u1172\u11C1", "\uB28E", "\u1102\u1172\u11C1" },
            new string [] { "\uB28F", "\uB28F", "\u1102\u1172\u11C2", "\uB28F", "\u1102\u1172\u11C2" },
            new string [] { "\uB290", "\uB290", "\u1102\u1173", "\uB290", "\u1102\u1173" },
            new string [] { "\uB291", "\uB291", "\u1102\u1173\u11A8", "\uB291", "\u1102\u1173\u11A8" },
            new string [] { "\uB292", "\uB292", "\u1102\u1173\u11A9", "\uB292", "\u1102\u1173\u11A9" },
            new string [] { "\uB293", "\uB293", "\u1102\u1173\u11AA", "\uB293", "\u1102\u1173\u11AA" },
            new string [] { "\uB294", "\uB294", "\u1102\u1173\u11AB", "\uB294", "\u1102\u1173\u11AB" },
            new string [] { "\uB295", "\uB295", "\u1102\u1173\u11AC", "\uB295", "\u1102\u1173\u11AC" },
            new string [] { "\uB296", "\uB296", "\u1102\u1173\u11AD", "\uB296", "\u1102\u1173\u11AD" },
            new string [] { "\uB297", "\uB297", "\u1102\u1173\u11AE", "\uB297", "\u1102\u1173\u11AE" },
            new string [] { "\uB298", "\uB298", "\u1102\u1173\u11AF", "\uB298", "\u1102\u1173\u11AF" },
            new string [] { "\uB299", "\uB299", "\u1102\u1173\u11B0", "\uB299", "\u1102\u1173\u11B0" },
            new string [] { "\uB29A", "\uB29A", "\u1102\u1173\u11B1", "\uB29A", "\u1102\u1173\u11B1" },
            new string [] { "\uB29B", "\uB29B", "\u1102\u1173\u11B2", "\uB29B", "\u1102\u1173\u11B2" },
            new string [] { "\uB29C", "\uB29C", "\u1102\u1173\u11B3", "\uB29C", "\u1102\u1173\u11B3" },
            new string [] { "\uB29D", "\uB29D", "\u1102\u1173\u11B4", "\uB29D", "\u1102\u1173\u11B4" },
            new string [] { "\uB29E", "\uB29E", "\u1102\u1173\u11B5", "\uB29E", "\u1102\u1173\u11B5" },
            new string [] { "\uB29F", "\uB29F", "\u1102\u1173\u11B6", "\uB29F", "\u1102\u1173\u11B6" },
            new string [] { "\uB2A0", "\uB2A0", "\u1102\u1173\u11B7", "\uB2A0", "\u1102\u1173\u11B7" },
            new string [] { "\uB2A1", "\uB2A1", "\u1102\u1173\u11B8", "\uB2A1", "\u1102\u1173\u11B8" },
            new string [] { "\uB2A2", "\uB2A2", "\u1102\u1173\u11B9", "\uB2A2", "\u1102\u1173\u11B9" },
            new string [] { "\uB2A3", "\uB2A3", "\u1102\u1173\u11BA", "\uB2A3", "\u1102\u1173\u11BA" },
            new string [] { "\uB2A4", "\uB2A4", "\u1102\u1173\u11BB", "\uB2A4", "\u1102\u1173\u11BB" },
            new string [] { "\uB2A5", "\uB2A5", "\u1102\u1173\u11BC", "\uB2A5", "\u1102\u1173\u11BC" },
            new string [] { "\uB2A6", "\uB2A6", "\u1102\u1173\u11BD", "\uB2A6", "\u1102\u1173\u11BD" },
            new string [] { "\uB2A7", "\uB2A7", "\u1102\u1173\u11BE", "\uB2A7", "\u1102\u1173\u11BE" },
            new string [] { "\uB2A8", "\uB2A8", "\u1102\u1173\u11BF", "\uB2A8", "\u1102\u1173\u11BF" },
            new string [] { "\uB2A9", "\uB2A9", "\u1102\u1173\u11C0", "\uB2A9", "\u1102\u1173\u11C0" },
            new string [] { "\uB2AA", "\uB2AA", "\u1102\u1173\u11C1", "\uB2AA", "\u1102\u1173\u11C1" },
            new string [] { "\uB2AB", "\uB2AB", "\u1102\u1173\u11C2", "\uB2AB", "\u1102\u1173\u11C2" },
            new string [] { "\uB2AC", "\uB2AC", "\u1102\u1174", "\uB2AC", "\u1102\u1174" },
            new string [] { "\uB2AD", "\uB2AD", "\u1102\u1174\u11A8", "\uB2AD", "\u1102\u1174\u11A8" },
            new string [] { "\uB2AE", "\uB2AE", "\u1102\u1174\u11A9", "\uB2AE", "\u1102\u1174\u11A9" },
            new string [] { "\uB2AF", "\uB2AF", "\u1102\u1174\u11AA", "\uB2AF", "\u1102\u1174\u11AA" },
            new string [] { "\uB2B0", "\uB2B0", "\u1102\u1174\u11AB", "\uB2B0", "\u1102\u1174\u11AB" },
            new string [] { "\uB2B1", "\uB2B1", "\u1102\u1174\u11AC", "\uB2B1", "\u1102\u1174\u11AC" },
            new string [] { "\uB2B2", "\uB2B2", "\u1102\u1174\u11AD", "\uB2B2", "\u1102\u1174\u11AD" },
            new string [] { "\uB2B3", "\uB2B3", "\u1102\u1174\u11AE", "\uB2B3", "\u1102\u1174\u11AE" },
            new string [] { "\uB2B4", "\uB2B4", "\u1102\u1174\u11AF", "\uB2B4", "\u1102\u1174\u11AF" },
            new string [] { "\uB2B5", "\uB2B5", "\u1102\u1174\u11B0", "\uB2B5", "\u1102\u1174\u11B0" },
            new string [] { "\uB2B6", "\uB2B6", "\u1102\u1174\u11B1", "\uB2B6", "\u1102\u1174\u11B1" },
            new string [] { "\uB2B7", "\uB2B7", "\u1102\u1174\u11B2", "\uB2B7", "\u1102\u1174\u11B2" },
            new string [] { "\uB2B8", "\uB2B8", "\u1102\u1174\u11B3", "\uB2B8", "\u1102\u1174\u11B3" },
            new string [] { "\uB2B9", "\uB2B9", "\u1102\u1174\u11B4", "\uB2B9", "\u1102\u1174\u11B4" },
            new string [] { "\uB2BA", "\uB2BA", "\u1102\u1174\u11B5", "\uB2BA", "\u1102\u1174\u11B5" },
            new string [] { "\uB2BB", "\uB2BB", "\u1102\u1174\u11B6", "\uB2BB", "\u1102\u1174\u11B6" },
            new string [] { "\uB2BC", "\uB2BC", "\u1102\u1174\u11B7", "\uB2BC", "\u1102\u1174\u11B7" },
            new string [] { "\uB2BD", "\uB2BD", "\u1102\u1174\u11B8", "\uB2BD", "\u1102\u1174\u11B8" },
            new string [] { "\uB2BE", "\uB2BE", "\u1102\u1174\u11B9", "\uB2BE", "\u1102\u1174\u11B9" },
            new string [] { "\uB2BF", "\uB2BF", "\u1102\u1174\u11BA", "\uB2BF", "\u1102\u1174\u11BA" },
            new string [] { "\uB2C0", "\uB2C0", "\u1102\u1174\u11BB", "\uB2C0", "\u1102\u1174\u11BB" },
            new string [] { "\uB2C1", "\uB2C1", "\u1102\u1174\u11BC", "\uB2C1", "\u1102\u1174\u11BC" },
            new string [] { "\uB2C2", "\uB2C2", "\u1102\u1174\u11BD", "\uB2C2", "\u1102\u1174\u11BD" },
            new string [] { "\uB2C3", "\uB2C3", "\u1102\u1174\u11BE", "\uB2C3", "\u1102\u1174\u11BE" },
            new string [] { "\uB2C4", "\uB2C4", "\u1102\u1174\u11BF", "\uB2C4", "\u1102\u1174\u11BF" },
            new string [] { "\uB2C5", "\uB2C5", "\u1102\u1174\u11C0", "\uB2C5", "\u1102\u1174\u11C0" },
            new string [] { "\uB2C6", "\uB2C6", "\u1102\u1174\u11C1", "\uB2C6", "\u1102\u1174\u11C1" },
            new string [] { "\uB2C7", "\uB2C7", "\u1102\u1174\u11C2", "\uB2C7", "\u1102\u1174\u11C2" },
            new string [] { "\uB2C8", "\uB2C8", "\u1102\u1175", "\uB2C8", "\u1102\u1175" },
            new string [] { "\uB2C9", "\uB2C9", "\u1102\u1175\u11A8", "\uB2C9", "\u1102\u1175\u11A8" },
            new string [] { "\uB2CA", "\uB2CA", "\u1102\u1175\u11A9", "\uB2CA", "\u1102\u1175\u11A9" },
            new string [] { "\uB2CB", "\uB2CB", "\u1102\u1175\u11AA", "\uB2CB", "\u1102\u1175\u11AA" },
            new string [] { "\uB2CC", "\uB2CC", "\u1102\u1175\u11AB", "\uB2CC", "\u1102\u1175\u11AB" },
            new string [] { "\uB2CD", "\uB2CD", "\u1102\u1175\u11AC", "\uB2CD", "\u1102\u1175\u11AC" },
            new string [] { "\uB2CE", "\uB2CE", "\u1102\u1175\u11AD", "\uB2CE", "\u1102\u1175\u11AD" },
            new string [] { "\uB2CF", "\uB2CF", "\u1102\u1175\u11AE", "\uB2CF", "\u1102\u1175\u11AE" },
            new string [] { "\uB2D0", "\uB2D0", "\u1102\u1175\u11AF", "\uB2D0", "\u1102\u1175\u11AF" },
            new string [] { "\uB2D1", "\uB2D1", "\u1102\u1175\u11B0", "\uB2D1", "\u1102\u1175\u11B0" },
            new string [] { "\uB2D2", "\uB2D2", "\u1102\u1175\u11B1", "\uB2D2", "\u1102\u1175\u11B1" },
            new string [] { "\uB2D3", "\uB2D3", "\u1102\u1175\u11B2", "\uB2D3", "\u1102\u1175\u11B2" },
            new string [] { "\uB2D4", "\uB2D4", "\u1102\u1175\u11B3", "\uB2D4", "\u1102\u1175\u11B3" },
            new string [] { "\uB2D5", "\uB2D5", "\u1102\u1175\u11B4", "\uB2D5", "\u1102\u1175\u11B4" },
            new string [] { "\uB2D6", "\uB2D6", "\u1102\u1175\u11B5", "\uB2D6", "\u1102\u1175\u11B5" },
            new string [] { "\uB2D7", "\uB2D7", "\u1102\u1175\u11B6", "\uB2D7", "\u1102\u1175\u11B6" },
            new string [] { "\uB2D8", "\uB2D8", "\u1102\u1175\u11B7", "\uB2D8", "\u1102\u1175\u11B7" },
            new string [] { "\uB2D9", "\uB2D9", "\u1102\u1175\u11B8", "\uB2D9", "\u1102\u1175\u11B8" },
            new string [] { "\uB2DA", "\uB2DA", "\u1102\u1175\u11B9", "\uB2DA", "\u1102\u1175\u11B9" },
            new string [] { "\uB2DB", "\uB2DB", "\u1102\u1175\u11BA", "\uB2DB", "\u1102\u1175\u11BA" },
            new string [] { "\uB2DC", "\uB2DC", "\u1102\u1175\u11BB", "\uB2DC", "\u1102\u1175\u11BB" },
            new string [] { "\uB2DD", "\uB2DD", "\u1102\u1175\u11BC", "\uB2DD", "\u1102\u1175\u11BC" },
            new string [] { "\uB2DE", "\uB2DE", "\u1102\u1175\u11BD", "\uB2DE", "\u1102\u1175\u11BD" },
            new string [] { "\uB2DF", "\uB2DF", "\u1102\u1175\u11BE", "\uB2DF", "\u1102\u1175\u11BE" },
            new string [] { "\uB2E0", "\uB2E0", "\u1102\u1175\u11BF", "\uB2E0", "\u1102\u1175\u11BF" },
            new string [] { "\uB2E1", "\uB2E1", "\u1102\u1175\u11C0", "\uB2E1", "\u1102\u1175\u11C0" },
            new string [] { "\uB2E2", "\uB2E2", "\u1102\u1175\u11C1", "\uB2E2", "\u1102\u1175\u11C1" },
            new string [] { "\uB2E3", "\uB2E3", "\u1102\u1175\u11C2", "\uB2E3", "\u1102\u1175\u11C2" },
            new string [] { "\uB2E4", "\uB2E4", "\u1103\u1161", "\uB2E4", "\u1103\u1161" },
            new string [] { "\uB2E5", "\uB2E5", "\u1103\u1161\u11A8", "\uB2E5", "\u1103\u1161\u11A8" },
            new string [] { "\uB2E6", "\uB2E6", "\u1103\u1161\u11A9", "\uB2E6", "\u1103\u1161\u11A9" },
            new string [] { "\uB2E7", "\uB2E7", "\u1103\u1161\u11AA", "\uB2E7", "\u1103\u1161\u11AA" },
            new string [] { "\uB2E8", "\uB2E8", "\u1103\u1161\u11AB", "\uB2E8", "\u1103\u1161\u11AB" },
            new string [] { "\uB2E9", "\uB2E9", "\u1103\u1161\u11AC", "\uB2E9", "\u1103\u1161\u11AC" },
            new string [] { "\uB2EA", "\uB2EA", "\u1103\u1161\u11AD", "\uB2EA", "\u1103\u1161\u11AD" },
            new string [] { "\uB2EB", "\uB2EB", "\u1103\u1161\u11AE", "\uB2EB", "\u1103\u1161\u11AE" },
            new string [] { "\uB2EC", "\uB2EC", "\u1103\u1161\u11AF", "\uB2EC", "\u1103\u1161\u11AF" },
            new string [] { "\uB2ED", "\uB2ED", "\u1103\u1161\u11B0", "\uB2ED", "\u1103\u1161\u11B0" },
            new string [] { "\uB2EE", "\uB2EE", "\u1103\u1161\u11B1", "\uB2EE", "\u1103\u1161\u11B1" },
            new string [] { "\uB2EF", "\uB2EF", "\u1103\u1161\u11B2", "\uB2EF", "\u1103\u1161\u11B2" },
            new string [] { "\uB2F0", "\uB2F0", "\u1103\u1161\u11B3", "\uB2F0", "\u1103\u1161\u11B3" },
            new string [] { "\uB2F1", "\uB2F1", "\u1103\u1161\u11B4", "\uB2F1", "\u1103\u1161\u11B4" },
            new string [] { "\uB2F2", "\uB2F2", "\u1103\u1161\u11B5", "\uB2F2", "\u1103\u1161\u11B5" },
            new string [] { "\uB2F3", "\uB2F3", "\u1103\u1161\u11B6", "\uB2F3", "\u1103\u1161\u11B6" },
            new string [] { "\uB2F4", "\uB2F4", "\u1103\u1161\u11B7", "\uB2F4", "\u1103\u1161\u11B7" },
            new string [] { "\uB2F5", "\uB2F5", "\u1103\u1161\u11B8", "\uB2F5", "\u1103\u1161\u11B8" },
            new string [] { "\uB2F6", "\uB2F6", "\u1103\u1161\u11B9", "\uB2F6", "\u1103\u1161\u11B9" },
            new string [] { "\uB2F7", "\uB2F7", "\u1103\u1161\u11BA", "\uB2F7", "\u1103\u1161\u11BA" },
            new string [] { "\uB2F8", "\uB2F8", "\u1103\u1161\u11BB", "\uB2F8", "\u1103\u1161\u11BB" },
            new string [] { "\uB2F9", "\uB2F9", "\u1103\u1161\u11BC", "\uB2F9", "\u1103\u1161\u11BC" },
            new string [] { "\uB2FA", "\uB2FA", "\u1103\u1161\u11BD", "\uB2FA", "\u1103\u1161\u11BD" },
            new string [] { "\uB2FB", "\uB2FB", "\u1103\u1161\u11BE", "\uB2FB", "\u1103\u1161\u11BE" },
            new string [] { "\uB2FC", "\uB2FC", "\u1103\u1161\u11BF", "\uB2FC", "\u1103\u1161\u11BF" },
            new string [] { "\uB2FD", "\uB2FD", "\u1103\u1161\u11C0", "\uB2FD", "\u1103\u1161\u11C0" },
            new string [] { "\uB2FE", "\uB2FE", "\u1103\u1161\u11C1", "\uB2FE", "\u1103\u1161\u11C1" },
            new string [] { "\uB2FF", "\uB2FF", "\u1103\u1161\u11C2", "\uB2FF", "\u1103\u1161\u11C2" },
            new string [] { "\uB300", "\uB300", "\u1103\u1162", "\uB300", "\u1103\u1162" },
            new string [] { "\uB301", "\uB301", "\u1103\u1162\u11A8", "\uB301", "\u1103\u1162\u11A8" },
            new string [] { "\uB302", "\uB302", "\u1103\u1162\u11A9", "\uB302", "\u1103\u1162\u11A9" },
            new string [] { "\uB303", "\uB303", "\u1103\u1162\u11AA", "\uB303", "\u1103\u1162\u11AA" },
            new string [] { "\uB304", "\uB304", "\u1103\u1162\u11AB", "\uB304", "\u1103\u1162\u11AB" },
            new string [] { "\uB305", "\uB305", "\u1103\u1162\u11AC", "\uB305", "\u1103\u1162\u11AC" },
            new string [] { "\uB306", "\uB306", "\u1103\u1162\u11AD", "\uB306", "\u1103\u1162\u11AD" },
            new string [] { "\uB307", "\uB307", "\u1103\u1162\u11AE", "\uB307", "\u1103\u1162\u11AE" },
            new string [] { "\uB308", "\uB308", "\u1103\u1162\u11AF", "\uB308", "\u1103\u1162\u11AF" },
            new string [] { "\uB309", "\uB309", "\u1103\u1162\u11B0", "\uB309", "\u1103\u1162\u11B0" },
            new string [] { "\uB30A", "\uB30A", "\u1103\u1162\u11B1", "\uB30A", "\u1103\u1162\u11B1" },
            new string [] { "\uB30B", "\uB30B", "\u1103\u1162\u11B2", "\uB30B", "\u1103\u1162\u11B2" },
            new string [] { "\uB30C", "\uB30C", "\u1103\u1162\u11B3", "\uB30C", "\u1103\u1162\u11B3" },
            new string [] { "\uB30D", "\uB30D", "\u1103\u1162\u11B4", "\uB30D", "\u1103\u1162\u11B4" },
            new string [] { "\uB30E", "\uB30E", "\u1103\u1162\u11B5", "\uB30E", "\u1103\u1162\u11B5" },
            new string [] { "\uB30F", "\uB30F", "\u1103\u1162\u11B6", "\uB30F", "\u1103\u1162\u11B6" },
            new string [] { "\uB310", "\uB310", "\u1103\u1162\u11B7", "\uB310", "\u1103\u1162\u11B7" },
            new string [] { "\uB311", "\uB311", "\u1103\u1162\u11B8", "\uB311", "\u1103\u1162\u11B8" },
            new string [] { "\uB312", "\uB312", "\u1103\u1162\u11B9", "\uB312", "\u1103\u1162\u11B9" },
            new string [] { "\uB313", "\uB313", "\u1103\u1162\u11BA", "\uB313", "\u1103\u1162\u11BA" },
            new string [] { "\uB314", "\uB314", "\u1103\u1162\u11BB", "\uB314", "\u1103\u1162\u11BB" },
            new string [] { "\uB315", "\uB315", "\u1103\u1162\u11BC", "\uB315", "\u1103\u1162\u11BC" },
            new string [] { "\uB316", "\uB316", "\u1103\u1162\u11BD", "\uB316", "\u1103\u1162\u11BD" },
            new string [] { "\uB317", "\uB317", "\u1103\u1162\u11BE", "\uB317", "\u1103\u1162\u11BE" },
            new string [] { "\uB318", "\uB318", "\u1103\u1162\u11BF", "\uB318", "\u1103\u1162\u11BF" },
            new string [] { "\uB319", "\uB319", "\u1103\u1162\u11C0", "\uB319", "\u1103\u1162\u11C0" },
            new string [] { "\uB31A", "\uB31A", "\u1103\u1162\u11C1", "\uB31A", "\u1103\u1162\u11C1" },
            new string [] { "\uB31B", "\uB31B", "\u1103\u1162\u11C2", "\uB31B", "\u1103\u1162\u11C2" },
            new string [] { "\uB31C", "\uB31C", "\u1103\u1163", "\uB31C", "\u1103\u1163" },
            new string [] { "\uB31D", "\uB31D", "\u1103\u1163\u11A8", "\uB31D", "\u1103\u1163\u11A8" },
            new string [] { "\uB31E", "\uB31E", "\u1103\u1163\u11A9", "\uB31E", "\u1103\u1163\u11A9" },
            new string [] { "\uB31F", "\uB31F", "\u1103\u1163\u11AA", "\uB31F", "\u1103\u1163\u11AA" },
            new string [] { "\uB320", "\uB320", "\u1103\u1163\u11AB", "\uB320", "\u1103\u1163\u11AB" },
            new string [] { "\uB321", "\uB321", "\u1103\u1163\u11AC", "\uB321", "\u1103\u1163\u11AC" },
            new string [] { "\uB322", "\uB322", "\u1103\u1163\u11AD", "\uB322", "\u1103\u1163\u11AD" },
            new string [] { "\uB323", "\uB323", "\u1103\u1163\u11AE", "\uB323", "\u1103\u1163\u11AE" },
            new string [] { "\uB324", "\uB324", "\u1103\u1163\u11AF", "\uB324", "\u1103\u1163\u11AF" },
            new string [] { "\uB325", "\uB325", "\u1103\u1163\u11B0", "\uB325", "\u1103\u1163\u11B0" },
            new string [] { "\uB326", "\uB326", "\u1103\u1163\u11B1", "\uB326", "\u1103\u1163\u11B1" },
            new string [] { "\uB327", "\uB327", "\u1103\u1163\u11B2", "\uB327", "\u1103\u1163\u11B2" },
            new string [] { "\uB328", "\uB328", "\u1103\u1163\u11B3", "\uB328", "\u1103\u1163\u11B3" },
            new string [] { "\uB329", "\uB329", "\u1103\u1163\u11B4", "\uB329", "\u1103\u1163\u11B4" },
            new string [] { "\uB32A", "\uB32A", "\u1103\u1163\u11B5", "\uB32A", "\u1103\u1163\u11B5" },
            new string [] { "\uB32B", "\uB32B", "\u1103\u1163\u11B6", "\uB32B", "\u1103\u1163\u11B6" },
            new string [] { "\uB32C", "\uB32C", "\u1103\u1163\u11B7", "\uB32C", "\u1103\u1163\u11B7" },
            new string [] { "\uB32D", "\uB32D", "\u1103\u1163\u11B8", "\uB32D", "\u1103\u1163\u11B8" },
            new string [] { "\uB32E", "\uB32E", "\u1103\u1163\u11B9", "\uB32E", "\u1103\u1163\u11B9" },
            new string [] { "\uB32F", "\uB32F", "\u1103\u1163\u11BA", "\uB32F", "\u1103\u1163\u11BA" },
            new string [] { "\uB330", "\uB330", "\u1103\u1163\u11BB", "\uB330", "\u1103\u1163\u11BB" },
            new string [] { "\uB331", "\uB331", "\u1103\u1163\u11BC", "\uB331", "\u1103\u1163\u11BC" },
            new string [] { "\uB332", "\uB332", "\u1103\u1163\u11BD", "\uB332", "\u1103\u1163\u11BD" },
            new string [] { "\uB333", "\uB333", "\u1103\u1163\u11BE", "\uB333", "\u1103\u1163\u11BE" },
            new string [] { "\uB334", "\uB334", "\u1103\u1163\u11BF", "\uB334", "\u1103\u1163\u11BF" },
            new string [] { "\uB335", "\uB335", "\u1103\u1163\u11C0", "\uB335", "\u1103\u1163\u11C0" },
            new string [] { "\uB336", "\uB336", "\u1103\u1163\u11C1", "\uB336", "\u1103\u1163\u11C1" },
            new string [] { "\uB337", "\uB337", "\u1103\u1163\u11C2", "\uB337", "\u1103\u1163\u11C2" },
            new string [] { "\uB338", "\uB338", "\u1103\u1164", "\uB338", "\u1103\u1164" },
            new string [] { "\uB339", "\uB339", "\u1103\u1164\u11A8", "\uB339", "\u1103\u1164\u11A8" },
            new string [] { "\uB33A", "\uB33A", "\u1103\u1164\u11A9", "\uB33A", "\u1103\u1164\u11A9" },
            new string [] { "\uB33B", "\uB33B", "\u1103\u1164\u11AA", "\uB33B", "\u1103\u1164\u11AA" },
            new string [] { "\uB33C", "\uB33C", "\u1103\u1164\u11AB", "\uB33C", "\u1103\u1164\u11AB" },
            new string [] { "\uB33D", "\uB33D", "\u1103\u1164\u11AC", "\uB33D", "\u1103\u1164\u11AC" },
            new string [] { "\uB33E", "\uB33E", "\u1103\u1164\u11AD", "\uB33E", "\u1103\u1164\u11AD" },
            new string [] { "\uB33F", "\uB33F", "\u1103\u1164\u11AE", "\uB33F", "\u1103\u1164\u11AE" },
            new string [] { "\uB340", "\uB340", "\u1103\u1164\u11AF", "\uB340", "\u1103\u1164\u11AF" },
            new string [] { "\uB341", "\uB341", "\u1103\u1164\u11B0", "\uB341", "\u1103\u1164\u11B0" },
            new string [] { "\uB342", "\uB342", "\u1103\u1164\u11B1", "\uB342", "\u1103\u1164\u11B1" },
            new string [] { "\uB343", "\uB343", "\u1103\u1164\u11B2", "\uB343", "\u1103\u1164\u11B2" },
            new string [] { "\uB344", "\uB344", "\u1103\u1164\u11B3", "\uB344", "\u1103\u1164\u11B3" },
            new string [] { "\uB345", "\uB345", "\u1103\u1164\u11B4", "\uB345", "\u1103\u1164\u11B4" },
            new string [] { "\uB346", "\uB346", "\u1103\u1164\u11B5", "\uB346", "\u1103\u1164\u11B5" },
            new string [] { "\uB347", "\uB347", "\u1103\u1164\u11B6", "\uB347", "\u1103\u1164\u11B6" },
            new string [] { "\uB348", "\uB348", "\u1103\u1164\u11B7", "\uB348", "\u1103\u1164\u11B7" },
            new string [] { "\uB349", "\uB349", "\u1103\u1164\u11B8", "\uB349", "\u1103\u1164\u11B8" },
            new string [] { "\uB34A", "\uB34A", "\u1103\u1164\u11B9", "\uB34A", "\u1103\u1164\u11B9" },
            new string [] { "\uB34B", "\uB34B", "\u1103\u1164\u11BA", "\uB34B", "\u1103\u1164\u11BA" },
            new string [] { "\uB34C", "\uB34C", "\u1103\u1164\u11BB", "\uB34C", "\u1103\u1164\u11BB" },
            new string [] { "\uB34D", "\uB34D", "\u1103\u1164\u11BC", "\uB34D", "\u1103\u1164\u11BC" },
            new string [] { "\uB34E", "\uB34E", "\u1103\u1164\u11BD", "\uB34E", "\u1103\u1164\u11BD" },
            new string [] { "\uB34F", "\uB34F", "\u1103\u1164\u11BE", "\uB34F", "\u1103\u1164\u11BE" },
            new string [] { "\uB350", "\uB350", "\u1103\u1164\u11BF", "\uB350", "\u1103\u1164\u11BF" },
            new string [] { "\uB351", "\uB351", "\u1103\u1164\u11C0", "\uB351", "\u1103\u1164\u11C0" },
            new string [] { "\uB352", "\uB352", "\u1103\u1164\u11C1", "\uB352", "\u1103\u1164\u11C1" },
            new string [] { "\uB353", "\uB353", "\u1103\u1164\u11C2", "\uB353", "\u1103\u1164\u11C2" },
            new string [] { "\uB354", "\uB354", "\u1103\u1165", "\uB354", "\u1103\u1165" },
            new string [] { "\uB355", "\uB355", "\u1103\u1165\u11A8", "\uB355", "\u1103\u1165\u11A8" },
            new string [] { "\uB356", "\uB356", "\u1103\u1165\u11A9", "\uB356", "\u1103\u1165\u11A9" },
            new string [] { "\uB357", "\uB357", "\u1103\u1165\u11AA", "\uB357", "\u1103\u1165\u11AA" },
            new string [] { "\uB358", "\uB358", "\u1103\u1165\u11AB", "\uB358", "\u1103\u1165\u11AB" },
            new string [] { "\uB359", "\uB359", "\u1103\u1165\u11AC", "\uB359", "\u1103\u1165\u11AC" },
            new string [] { "\uB35A", "\uB35A", "\u1103\u1165\u11AD", "\uB35A", "\u1103\u1165\u11AD" },
            new string [] { "\uB35B", "\uB35B", "\u1103\u1165\u11AE", "\uB35B", "\u1103\u1165\u11AE" },
            new string [] { "\uB35C", "\uB35C", "\u1103\u1165\u11AF", "\uB35C", "\u1103\u1165\u11AF" },
            new string [] { "\uB35D", "\uB35D", "\u1103\u1165\u11B0", "\uB35D", "\u1103\u1165\u11B0" },
            new string [] { "\uB35E", "\uB35E", "\u1103\u1165\u11B1", "\uB35E", "\u1103\u1165\u11B1" },
            new string [] { "\uB35F", "\uB35F", "\u1103\u1165\u11B2", "\uB35F", "\u1103\u1165\u11B2" },
            new string [] { "\uB360", "\uB360", "\u1103\u1165\u11B3", "\uB360", "\u1103\u1165\u11B3" },
            new string [] { "\uB361", "\uB361", "\u1103\u1165\u11B4", "\uB361", "\u1103\u1165\u11B4" },
            new string [] { "\uB362", "\uB362", "\u1103\u1165\u11B5", "\uB362", "\u1103\u1165\u11B5" },
            new string [] { "\uB363", "\uB363", "\u1103\u1165\u11B6", "\uB363", "\u1103\u1165\u11B6" },
            new string [] { "\uB364", "\uB364", "\u1103\u1165\u11B7", "\uB364", "\u1103\u1165\u11B7" },
            new string [] { "\uB365", "\uB365", "\u1103\u1165\u11B8", "\uB365", "\u1103\u1165\u11B8" },
            new string [] { "\uB366", "\uB366", "\u1103\u1165\u11B9", "\uB366", "\u1103\u1165\u11B9" },
            new string [] { "\uB367", "\uB367", "\u1103\u1165\u11BA", "\uB367", "\u1103\u1165\u11BA" },
            new string [] { "\uB368", "\uB368", "\u1103\u1165\u11BB", "\uB368", "\u1103\u1165\u11BB" },
            new string [] { "\uB369", "\uB369", "\u1103\u1165\u11BC", "\uB369", "\u1103\u1165\u11BC" },
            new string [] { "\uB36A", "\uB36A", "\u1103\u1165\u11BD", "\uB36A", "\u1103\u1165\u11BD" },
            new string [] { "\uB36B", "\uB36B", "\u1103\u1165\u11BE", "\uB36B", "\u1103\u1165\u11BE" },
            new string [] { "\uB36C", "\uB36C", "\u1103\u1165\u11BF", "\uB36C", "\u1103\u1165\u11BF" },
            new string [] { "\uB36D", "\uB36D", "\u1103\u1165\u11C0", "\uB36D", "\u1103\u1165\u11C0" },
            new string [] { "\uB36E", "\uB36E", "\u1103\u1165\u11C1", "\uB36E", "\u1103\u1165\u11C1" },
            new string [] { "\uB36F", "\uB36F", "\u1103\u1165\u11C2", "\uB36F", "\u1103\u1165\u11C2" },
            new string [] { "\uB370", "\uB370", "\u1103\u1166", "\uB370", "\u1103\u1166" },
            new string [] { "\uB371", "\uB371", "\u1103\u1166\u11A8", "\uB371", "\u1103\u1166\u11A8" },
            new string [] { "\uB372", "\uB372", "\u1103\u1166\u11A9", "\uB372", "\u1103\u1166\u11A9" },
            new string [] { "\uB373", "\uB373", "\u1103\u1166\u11AA", "\uB373", "\u1103\u1166\u11AA" },
            new string [] { "\uB374", "\uB374", "\u1103\u1166\u11AB", "\uB374", "\u1103\u1166\u11AB" },
            new string [] { "\uB375", "\uB375", "\u1103\u1166\u11AC", "\uB375", "\u1103\u1166\u11AC" },
            new string [] { "\uB376", "\uB376", "\u1103\u1166\u11AD", "\uB376", "\u1103\u1166\u11AD" },
            new string [] { "\uB377", "\uB377", "\u1103\u1166\u11AE", "\uB377", "\u1103\u1166\u11AE" },
            new string [] { "\uB378", "\uB378", "\u1103\u1166\u11AF", "\uB378", "\u1103\u1166\u11AF" },
            new string [] { "\uB379", "\uB379", "\u1103\u1166\u11B0", "\uB379", "\u1103\u1166\u11B0" },
            new string [] { "\uB37A", "\uB37A", "\u1103\u1166\u11B1", "\uB37A", "\u1103\u1166\u11B1" },
            new string [] { "\uB37B", "\uB37B", "\u1103\u1166\u11B2", "\uB37B", "\u1103\u1166\u11B2" },
            new string [] { "\uB37C", "\uB37C", "\u1103\u1166\u11B3", "\uB37C", "\u1103\u1166\u11B3" },
            new string [] { "\uB37D", "\uB37D", "\u1103\u1166\u11B4", "\uB37D", "\u1103\u1166\u11B4" },
            new string [] { "\uB37E", "\uB37E", "\u1103\u1166\u11B5", "\uB37E", "\u1103\u1166\u11B5" },
            new string [] { "\uB37F", "\uB37F", "\u1103\u1166\u11B6", "\uB37F", "\u1103\u1166\u11B6" },
            new string [] { "\uB380", "\uB380", "\u1103\u1166\u11B7", "\uB380", "\u1103\u1166\u11B7" },
            new string [] { "\uB381", "\uB381", "\u1103\u1166\u11B8", "\uB381", "\u1103\u1166\u11B8" },
            new string [] { "\uB382", "\uB382", "\u1103\u1166\u11B9", "\uB382", "\u1103\u1166\u11B9" },
            new string [] { "\uB383", "\uB383", "\u1103\u1166\u11BA", "\uB383", "\u1103\u1166\u11BA" },
            new string [] { "\uB384", "\uB384", "\u1103\u1166\u11BB", "\uB384", "\u1103\u1166\u11BB" },
            new string [] { "\uB385", "\uB385", "\u1103\u1166\u11BC", "\uB385", "\u1103\u1166\u11BC" },
            new string [] { "\uB386", "\uB386", "\u1103\u1166\u11BD", "\uB386", "\u1103\u1166\u11BD" },
            new string [] { "\uB387", "\uB387", "\u1103\u1166\u11BE", "\uB387", "\u1103\u1166\u11BE" },
            new string [] { "\uB388", "\uB388", "\u1103\u1166\u11BF", "\uB388", "\u1103\u1166\u11BF" },
            new string [] { "\uB389", "\uB389", "\u1103\u1166\u11C0", "\uB389", "\u1103\u1166\u11C0" },
            new string [] { "\uB38A", "\uB38A", "\u1103\u1166\u11C1", "\uB38A", "\u1103\u1166\u11C1" },
            new string [] { "\uB38B", "\uB38B", "\u1103\u1166\u11C2", "\uB38B", "\u1103\u1166\u11C2" },
            new string [] { "\uB38C", "\uB38C", "\u1103\u1167", "\uB38C", "\u1103\u1167" },
            new string [] { "\uB38D", "\uB38D", "\u1103\u1167\u11A8", "\uB38D", "\u1103\u1167\u11A8" },
            new string [] { "\uB38E", "\uB38E", "\u1103\u1167\u11A9", "\uB38E", "\u1103\u1167\u11A9" },
            new string [] { "\uB38F", "\uB38F", "\u1103\u1167\u11AA", "\uB38F", "\u1103\u1167\u11AA" },
            new string [] { "\uB390", "\uB390", "\u1103\u1167\u11AB", "\uB390", "\u1103\u1167\u11AB" },
            new string [] { "\uB391", "\uB391", "\u1103\u1167\u11AC", "\uB391", "\u1103\u1167\u11AC" },
            new string [] { "\uB392", "\uB392", "\u1103\u1167\u11AD", "\uB392", "\u1103\u1167\u11AD" },
            new string [] { "\uB393", "\uB393", "\u1103\u1167\u11AE", "\uB393", "\u1103\u1167\u11AE" },
            new string [] { "\uB394", "\uB394", "\u1103\u1167\u11AF", "\uB394", "\u1103\u1167\u11AF" },
            new string [] { "\uB395", "\uB395", "\u1103\u1167\u11B0", "\uB395", "\u1103\u1167\u11B0" },
            new string [] { "\uB396", "\uB396", "\u1103\u1167\u11B1", "\uB396", "\u1103\u1167\u11B1" },
            new string [] { "\uB397", "\uB397", "\u1103\u1167\u11B2", "\uB397", "\u1103\u1167\u11B2" },
            new string [] { "\uB398", "\uB398", "\u1103\u1167\u11B3", "\uB398", "\u1103\u1167\u11B3" },
            new string [] { "\uB399", "\uB399", "\u1103\u1167\u11B4", "\uB399", "\u1103\u1167\u11B4" },
            new string [] { "\uB39A", "\uB39A", "\u1103\u1167\u11B5", "\uB39A", "\u1103\u1167\u11B5" },
            new string [] { "\uB39B", "\uB39B", "\u1103\u1167\u11B6", "\uB39B", "\u1103\u1167\u11B6" },
            new string [] { "\uB39C", "\uB39C", "\u1103\u1167\u11B7", "\uB39C", "\u1103\u1167\u11B7" },
            new string [] { "\uB39D", "\uB39D", "\u1103\u1167\u11B8", "\uB39D", "\u1103\u1167\u11B8" },
            new string [] { "\uB39E", "\uB39E", "\u1103\u1167\u11B9", "\uB39E", "\u1103\u1167\u11B9" },
            new string [] { "\uB39F", "\uB39F", "\u1103\u1167\u11BA", "\uB39F", "\u1103\u1167\u11BA" },
            new string [] { "\uB3A0", "\uB3A0", "\u1103\u1167\u11BB", "\uB3A0", "\u1103\u1167\u11BB" },
            new string [] { "\uB3A1", "\uB3A1", "\u1103\u1167\u11BC", "\uB3A1", "\u1103\u1167\u11BC" },
            new string [] { "\uB3A2", "\uB3A2", "\u1103\u1167\u11BD", "\uB3A2", "\u1103\u1167\u11BD" },
            new string [] { "\uB3A3", "\uB3A3", "\u1103\u1167\u11BE", "\uB3A3", "\u1103\u1167\u11BE" },
            new string [] { "\uB3A4", "\uB3A4", "\u1103\u1167\u11BF", "\uB3A4", "\u1103\u1167\u11BF" },
            new string [] { "\uB3A5", "\uB3A5", "\u1103\u1167\u11C0", "\uB3A5", "\u1103\u1167\u11C0" },
            new string [] { "\uB3A6", "\uB3A6", "\u1103\u1167\u11C1", "\uB3A6", "\u1103\u1167\u11C1" },
            new string [] { "\uB3A7", "\uB3A7", "\u1103\u1167\u11C2", "\uB3A7", "\u1103\u1167\u11C2" },
            new string [] { "\uB3A8", "\uB3A8", "\u1103\u1168", "\uB3A8", "\u1103\u1168" },
            new string [] { "\uB3A9", "\uB3A9", "\u1103\u1168\u11A8", "\uB3A9", "\u1103\u1168\u11A8" },
            new string [] { "\uB3AA", "\uB3AA", "\u1103\u1168\u11A9", "\uB3AA", "\u1103\u1168\u11A9" },
            new string [] { "\uB3AB", "\uB3AB", "\u1103\u1168\u11AA", "\uB3AB", "\u1103\u1168\u11AA" },
            new string [] { "\uB3AC", "\uB3AC", "\u1103\u1168\u11AB", "\uB3AC", "\u1103\u1168\u11AB" },
            new string [] { "\uB3AD", "\uB3AD", "\u1103\u1168\u11AC", "\uB3AD", "\u1103\u1168\u11AC" },
            new string [] { "\uB3AE", "\uB3AE", "\u1103\u1168\u11AD", "\uB3AE", "\u1103\u1168\u11AD" },
            new string [] { "\uB3AF", "\uB3AF", "\u1103\u1168\u11AE", "\uB3AF", "\u1103\u1168\u11AE" },
            new string [] { "\uB3B0", "\uB3B0", "\u1103\u1168\u11AF", "\uB3B0", "\u1103\u1168\u11AF" },
            new string [] { "\uB3B1", "\uB3B1", "\u1103\u1168\u11B0", "\uB3B1", "\u1103\u1168\u11B0" },
            new string [] { "\uB3B2", "\uB3B2", "\u1103\u1168\u11B1", "\uB3B2", "\u1103\u1168\u11B1" },
            new string [] { "\uB3B3", "\uB3B3", "\u1103\u1168\u11B2", "\uB3B3", "\u1103\u1168\u11B2" },
            new string [] { "\uB3B4", "\uB3B4", "\u1103\u1168\u11B3", "\uB3B4", "\u1103\u1168\u11B3" },
            new string [] { "\uB3B5", "\uB3B5", "\u1103\u1168\u11B4", "\uB3B5", "\u1103\u1168\u11B4" },
            new string [] { "\uB3B6", "\uB3B6", "\u1103\u1168\u11B5", "\uB3B6", "\u1103\u1168\u11B5" },
            new string [] { "\uB3B7", "\uB3B7", "\u1103\u1168\u11B6", "\uB3B7", "\u1103\u1168\u11B6" },
            new string [] { "\uB3B8", "\uB3B8", "\u1103\u1168\u11B7", "\uB3B8", "\u1103\u1168\u11B7" },
            new string [] { "\uB3B9", "\uB3B9", "\u1103\u1168\u11B8", "\uB3B9", "\u1103\u1168\u11B8" },
            new string [] { "\uB3BA", "\uB3BA", "\u1103\u1168\u11B9", "\uB3BA", "\u1103\u1168\u11B9" },
            new string [] { "\uB3BB", "\uB3BB", "\u1103\u1168\u11BA", "\uB3BB", "\u1103\u1168\u11BA" },
            new string [] { "\uB3BC", "\uB3BC", "\u1103\u1168\u11BB", "\uB3BC", "\u1103\u1168\u11BB" },
            new string [] { "\uB3BD", "\uB3BD", "\u1103\u1168\u11BC", "\uB3BD", "\u1103\u1168\u11BC" },
            new string [] { "\uB3BE", "\uB3BE", "\u1103\u1168\u11BD", "\uB3BE", "\u1103\u1168\u11BD" },
            new string [] { "\uB3BF", "\uB3BF", "\u1103\u1168\u11BE", "\uB3BF", "\u1103\u1168\u11BE" },
            new string [] { "\uB3C0", "\uB3C0", "\u1103\u1168\u11BF", "\uB3C0", "\u1103\u1168\u11BF" },
            new string [] { "\uB3C1", "\uB3C1", "\u1103\u1168\u11C0", "\uB3C1", "\u1103\u1168\u11C0" },
            new string [] { "\uB3C2", "\uB3C2", "\u1103\u1168\u11C1", "\uB3C2", "\u1103\u1168\u11C1" },
            new string [] { "\uB3C3", "\uB3C3", "\u1103\u1168\u11C2", "\uB3C3", "\u1103\u1168\u11C2" },
            new string [] { "\uB3C4", "\uB3C4", "\u1103\u1169", "\uB3C4", "\u1103\u1169" },
            new string [] { "\uB3C5", "\uB3C5", "\u1103\u1169\u11A8", "\uB3C5", "\u1103\u1169\u11A8" },
            new string [] { "\uB3C6", "\uB3C6", "\u1103\u1169\u11A9", "\uB3C6", "\u1103\u1169\u11A9" },
            new string [] { "\uB3C7", "\uB3C7", "\u1103\u1169\u11AA", "\uB3C7", "\u1103\u1169\u11AA" },
            new string [] { "\uB3C8", "\uB3C8", "\u1103\u1169\u11AB", "\uB3C8", "\u1103\u1169\u11AB" },
            new string [] { "\uB3C9", "\uB3C9", "\u1103\u1169\u11AC", "\uB3C9", "\u1103\u1169\u11AC" },
            new string [] { "\uB3CA", "\uB3CA", "\u1103\u1169\u11AD", "\uB3CA", "\u1103\u1169\u11AD" },
            new string [] { "\uB3CB", "\uB3CB", "\u1103\u1169\u11AE", "\uB3CB", "\u1103\u1169\u11AE" },
            new string [] { "\uB3CC", "\uB3CC", "\u1103\u1169\u11AF", "\uB3CC", "\u1103\u1169\u11AF" },
            new string [] { "\uB3CD", "\uB3CD", "\u1103\u1169\u11B0", "\uB3CD", "\u1103\u1169\u11B0" },
            new string [] { "\uB3CE", "\uB3CE", "\u1103\u1169\u11B1", "\uB3CE", "\u1103\u1169\u11B1" },
            new string [] { "\uB3CF", "\uB3CF", "\u1103\u1169\u11B2", "\uB3CF", "\u1103\u1169\u11B2" },
            new string [] { "\uB3D0", "\uB3D0", "\u1103\u1169\u11B3", "\uB3D0", "\u1103\u1169\u11B3" },
            new string [] { "\uB3D1", "\uB3D1", "\u1103\u1169\u11B4", "\uB3D1", "\u1103\u1169\u11B4" },
            new string [] { "\uB3D2", "\uB3D2", "\u1103\u1169\u11B5", "\uB3D2", "\u1103\u1169\u11B5" },
            new string [] { "\uB3D3", "\uB3D3", "\u1103\u1169\u11B6", "\uB3D3", "\u1103\u1169\u11B6" },
            new string [] { "\uB3D4", "\uB3D4", "\u1103\u1169\u11B7", "\uB3D4", "\u1103\u1169\u11B7" },
            new string [] { "\uB3D5", "\uB3D5", "\u1103\u1169\u11B8", "\uB3D5", "\u1103\u1169\u11B8" },
            new string [] { "\uB3D6", "\uB3D6", "\u1103\u1169\u11B9", "\uB3D6", "\u1103\u1169\u11B9" },
            new string [] { "\uB3D7", "\uB3D7", "\u1103\u1169\u11BA", "\uB3D7", "\u1103\u1169\u11BA" },
            new string [] { "\uB3D8", "\uB3D8", "\u1103\u1169\u11BB", "\uB3D8", "\u1103\u1169\u11BB" },
            new string [] { "\uB3D9", "\uB3D9", "\u1103\u1169\u11BC", "\uB3D9", "\u1103\u1169\u11BC" },
            new string [] { "\uB3DA", "\uB3DA", "\u1103\u1169\u11BD", "\uB3DA", "\u1103\u1169\u11BD" },
            new string [] { "\uB3DB", "\uB3DB", "\u1103\u1169\u11BE", "\uB3DB", "\u1103\u1169\u11BE" },
            new string [] { "\uB3DC", "\uB3DC", "\u1103\u1169\u11BF", "\uB3DC", "\u1103\u1169\u11BF" },
            new string [] { "\uB3DD", "\uB3DD", "\u1103\u1169\u11C0", "\uB3DD", "\u1103\u1169\u11C0" },
            new string [] { "\uB3DE", "\uB3DE", "\u1103\u1169\u11C1", "\uB3DE", "\u1103\u1169\u11C1" },
            new string [] { "\uB3DF", "\uB3DF", "\u1103\u1169\u11C2", "\uB3DF", "\u1103\u1169\u11C2" },
            new string [] { "\uB3E0", "\uB3E0", "\u1103\u116A", "\uB3E0", "\u1103\u116A" },
            new string [] { "\uB3E1", "\uB3E1", "\u1103\u116A\u11A8", "\uB3E1", "\u1103\u116A\u11A8" },
            new string [] { "\uB3E2", "\uB3E2", "\u1103\u116A\u11A9", "\uB3E2", "\u1103\u116A\u11A9" },
            new string [] { "\uB3E3", "\uB3E3", "\u1103\u116A\u11AA", "\uB3E3", "\u1103\u116A\u11AA" },
            new string [] { "\uB3E4", "\uB3E4", "\u1103\u116A\u11AB", "\uB3E4", "\u1103\u116A\u11AB" },
            new string [] { "\uB3E5", "\uB3E5", "\u1103\u116A\u11AC", "\uB3E5", "\u1103\u116A\u11AC" },
            new string [] { "\uB3E6", "\uB3E6", "\u1103\u116A\u11AD", "\uB3E6", "\u1103\u116A\u11AD" },
            new string [] { "\uB3E7", "\uB3E7", "\u1103\u116A\u11AE", "\uB3E7", "\u1103\u116A\u11AE" },
            new string [] { "\uB3E8", "\uB3E8", "\u1103\u116A\u11AF", "\uB3E8", "\u1103\u116A\u11AF" },
            new string [] { "\uB3E9", "\uB3E9", "\u1103\u116A\u11B0", "\uB3E9", "\u1103\u116A\u11B0" },
            new string [] { "\uB3EA", "\uB3EA", "\u1103\u116A\u11B1", "\uB3EA", "\u1103\u116A\u11B1" },
            new string [] { "\uB3EB", "\uB3EB", "\u1103\u116A\u11B2", "\uB3EB", "\u1103\u116A\u11B2" },
            new string [] { "\uB3EC", "\uB3EC", "\u1103\u116A\u11B3", "\uB3EC", "\u1103\u116A\u11B3" },
            new string [] { "\uB3ED", "\uB3ED", "\u1103\u116A\u11B4", "\uB3ED", "\u1103\u116A\u11B4" },
            new string [] { "\uB3EE", "\uB3EE", "\u1103\u116A\u11B5", "\uB3EE", "\u1103\u116A\u11B5" },
            new string [] { "\uB3EF", "\uB3EF", "\u1103\u116A\u11B6", "\uB3EF", "\u1103\u116A\u11B6" },
            new string [] { "\uB3F0", "\uB3F0", "\u1103\u116A\u11B7", "\uB3F0", "\u1103\u116A\u11B7" },
            new string [] { "\uB3F1", "\uB3F1", "\u1103\u116A\u11B8", "\uB3F1", "\u1103\u116A\u11B8" },
            new string [] { "\uB3F2", "\uB3F2", "\u1103\u116A\u11B9", "\uB3F2", "\u1103\u116A\u11B9" },
            new string [] { "\uB3F3", "\uB3F3", "\u1103\u116A\u11BA", "\uB3F3", "\u1103\u116A\u11BA" },
            new string [] { "\uB3F4", "\uB3F4", "\u1103\u116A\u11BB", "\uB3F4", "\u1103\u116A\u11BB" },
            new string [] { "\uB3F5", "\uB3F5", "\u1103\u116A\u11BC", "\uB3F5", "\u1103\u116A\u11BC" },
            new string [] { "\uB3F6", "\uB3F6", "\u1103\u116A\u11BD", "\uB3F6", "\u1103\u116A\u11BD" },
            new string [] { "\uB3F7", "\uB3F7", "\u1103\u116A\u11BE", "\uB3F7", "\u1103\u116A\u11BE" },
            new string [] { "\uB3F8", "\uB3F8", "\u1103\u116A\u11BF", "\uB3F8", "\u1103\u116A\u11BF" },
            new string [] { "\uB3F9", "\uB3F9", "\u1103\u116A\u11C0", "\uB3F9", "\u1103\u116A\u11C0" },
            new string [] { "\uB3FA", "\uB3FA", "\u1103\u116A\u11C1", "\uB3FA", "\u1103\u116A\u11C1" },
            new string [] { "\uB3FB", "\uB3FB", "\u1103\u116A\u11C2", "\uB3FB", "\u1103\u116A\u11C2" },
            new string [] { "\uB3FC", "\uB3FC", "\u1103\u116B", "\uB3FC", "\u1103\u116B" },
            new string [] { "\uB3FD", "\uB3FD", "\u1103\u116B\u11A8", "\uB3FD", "\u1103\u116B\u11A8" },
            new string [] { "\uB3FE", "\uB3FE", "\u1103\u116B\u11A9", "\uB3FE", "\u1103\u116B\u11A9" },
            new string [] { "\uB3FF", "\uB3FF", "\u1103\u116B\u11AA", "\uB3FF", "\u1103\u116B\u11AA" },
            new string [] { "\uB400", "\uB400", "\u1103\u116B\u11AB", "\uB400", "\u1103\u116B\u11AB" },
            new string [] { "\uB401", "\uB401", "\u1103\u116B\u11AC", "\uB401", "\u1103\u116B\u11AC" },
            new string [] { "\uB402", "\uB402", "\u1103\u116B\u11AD", "\uB402", "\u1103\u116B\u11AD" },
            new string [] { "\uB403", "\uB403", "\u1103\u116B\u11AE", "\uB403", "\u1103\u116B\u11AE" },
            new string [] { "\uB404", "\uB404", "\u1103\u116B\u11AF", "\uB404", "\u1103\u116B\u11AF" },
            new string [] { "\uB405", "\uB405", "\u1103\u116B\u11B0", "\uB405", "\u1103\u116B\u11B0" },
            new string [] { "\uB406", "\uB406", "\u1103\u116B\u11B1", "\uB406", "\u1103\u116B\u11B1" },
            new string [] { "\uB407", "\uB407", "\u1103\u116B\u11B2", "\uB407", "\u1103\u116B\u11B2" },
            new string [] { "\uB408", "\uB408", "\u1103\u116B\u11B3", "\uB408", "\u1103\u116B\u11B3" },
            new string [] { "\uB409", "\uB409", "\u1103\u116B\u11B4", "\uB409", "\u1103\u116B\u11B4" },
            new string [] { "\uB40A", "\uB40A", "\u1103\u116B\u11B5", "\uB40A", "\u1103\u116B\u11B5" },
            new string [] { "\uB40B", "\uB40B", "\u1103\u116B\u11B6", "\uB40B", "\u1103\u116B\u11B6" },
            new string [] { "\uB40C", "\uB40C", "\u1103\u116B\u11B7", "\uB40C", "\u1103\u116B\u11B7" },
            new string [] { "\uB40D", "\uB40D", "\u1103\u116B\u11B8", "\uB40D", "\u1103\u116B\u11B8" },
            new string [] { "\uB40E", "\uB40E", "\u1103\u116B\u11B9", "\uB40E", "\u1103\u116B\u11B9" },
            new string [] { "\uB40F", "\uB40F", "\u1103\u116B\u11BA", "\uB40F", "\u1103\u116B\u11BA" },
            new string [] { "\uB410", "\uB410", "\u1103\u116B\u11BB", "\uB410", "\u1103\u116B\u11BB" },
            new string [] { "\uB411", "\uB411", "\u1103\u116B\u11BC", "\uB411", "\u1103\u116B\u11BC" },
            new string [] { "\uB412", "\uB412", "\u1103\u116B\u11BD", "\uB412", "\u1103\u116B\u11BD" },
            new string [] { "\uB413", "\uB413", "\u1103\u116B\u11BE", "\uB413", "\u1103\u116B\u11BE" },
            new string [] { "\uB414", "\uB414", "\u1103\u116B\u11BF", "\uB414", "\u1103\u116B\u11BF" },
            new string [] { "\uB415", "\uB415", "\u1103\u116B\u11C0", "\uB415", "\u1103\u116B\u11C0" },
            new string [] { "\uB416", "\uB416", "\u1103\u116B\u11C1", "\uB416", "\u1103\u116B\u11C1" },
            new string [] { "\uB417", "\uB417", "\u1103\u116B\u11C2", "\uB417", "\u1103\u116B\u11C2" },
            new string [] { "\uB418", "\uB418", "\u1103\u116C", "\uB418", "\u1103\u116C" },
            new string [] { "\uB419", "\uB419", "\u1103\u116C\u11A8", "\uB419", "\u1103\u116C\u11A8" },
            new string [] { "\uB41A", "\uB41A", "\u1103\u116C\u11A9", "\uB41A", "\u1103\u116C\u11A9" },
            new string [] { "\uB41B", "\uB41B", "\u1103\u116C\u11AA", "\uB41B", "\u1103\u116C\u11AA" },
            new string [] { "\uB41C", "\uB41C", "\u1103\u116C\u11AB", "\uB41C", "\u1103\u116C\u11AB" },
            new string [] { "\uB41D", "\uB41D", "\u1103\u116C\u11AC", "\uB41D", "\u1103\u116C\u11AC" },
            new string [] { "\uB41E", "\uB41E", "\u1103\u116C\u11AD", "\uB41E", "\u1103\u116C\u11AD" },
            new string [] { "\uB41F", "\uB41F", "\u1103\u116C\u11AE", "\uB41F", "\u1103\u116C\u11AE" },
            new string [] { "\uB420", "\uB420", "\u1103\u116C\u11AF", "\uB420", "\u1103\u116C\u11AF" },
            new string [] { "\uB421", "\uB421", "\u1103\u116C\u11B0", "\uB421", "\u1103\u116C\u11B0" },
            new string [] { "\uB422", "\uB422", "\u1103\u116C\u11B1", "\uB422", "\u1103\u116C\u11B1" },
            new string [] { "\uB423", "\uB423", "\u1103\u116C\u11B2", "\uB423", "\u1103\u116C\u11B2" },
            new string [] { "\uB424", "\uB424", "\u1103\u116C\u11B3", "\uB424", "\u1103\u116C\u11B3" },
            new string [] { "\uB425", "\uB425", "\u1103\u116C\u11B4", "\uB425", "\u1103\u116C\u11B4" },
            new string [] { "\uB426", "\uB426", "\u1103\u116C\u11B5", "\uB426", "\u1103\u116C\u11B5" },
            new string [] { "\uB427", "\uB427", "\u1103\u116C\u11B6", "\uB427", "\u1103\u116C\u11B6" },
            new string [] { "\uB428", "\uB428", "\u1103\u116C\u11B7", "\uB428", "\u1103\u116C\u11B7" },
            new string [] { "\uB429", "\uB429", "\u1103\u116C\u11B8", "\uB429", "\u1103\u116C\u11B8" },
            new string [] { "\uB42A", "\uB42A", "\u1103\u116C\u11B9", "\uB42A", "\u1103\u116C\u11B9" },
            new string [] { "\uB42B", "\uB42B", "\u1103\u116C\u11BA", "\uB42B", "\u1103\u116C\u11BA" },
            new string [] { "\uB42C", "\uB42C", "\u1103\u116C\u11BB", "\uB42C", "\u1103\u116C\u11BB" },
            new string [] { "\uB42D", "\uB42D", "\u1103\u116C\u11BC", "\uB42D", "\u1103\u116C\u11BC" },
            new string [] { "\uB42E", "\uB42E", "\u1103\u116C\u11BD", "\uB42E", "\u1103\u116C\u11BD" },
            new string [] { "\uB42F", "\uB42F", "\u1103\u116C\u11BE", "\uB42F", "\u1103\u116C\u11BE" },
            new string [] { "\uB430", "\uB430", "\u1103\u116C\u11BF", "\uB430", "\u1103\u116C\u11BF" },
            new string [] { "\uB431", "\uB431", "\u1103\u116C\u11C0", "\uB431", "\u1103\u116C\u11C0" },
            new string [] { "\uB432", "\uB432", "\u1103\u116C\u11C1", "\uB432", "\u1103\u116C\u11C1" },
            new string [] { "\uB433", "\uB433", "\u1103\u116C\u11C2", "\uB433", "\u1103\u116C\u11C2" },
            new string [] { "\uB434", "\uB434", "\u1103\u116D", "\uB434", "\u1103\u116D" },
            new string [] { "\uB435", "\uB435", "\u1103\u116D\u11A8", "\uB435", "\u1103\u116D\u11A8" },
            new string [] { "\uB436", "\uB436", "\u1103\u116D\u11A9", "\uB436", "\u1103\u116D\u11A9" },
            new string [] { "\uB437", "\uB437", "\u1103\u116D\u11AA", "\uB437", "\u1103\u116D\u11AA" },
            new string [] { "\uB438", "\uB438", "\u1103\u116D\u11AB", "\uB438", "\u1103\u116D\u11AB" },
            new string [] { "\uB439", "\uB439", "\u1103\u116D\u11AC", "\uB439", "\u1103\u116D\u11AC" },
            new string [] { "\uB43A", "\uB43A", "\u1103\u116D\u11AD", "\uB43A", "\u1103\u116D\u11AD" },
            new string [] { "\uB43B", "\uB43B", "\u1103\u116D\u11AE", "\uB43B", "\u1103\u116D\u11AE" },
            new string [] { "\uB43C", "\uB43C", "\u1103\u116D\u11AF", "\uB43C", "\u1103\u116D\u11AF" },
            new string [] { "\uB43D", "\uB43D", "\u1103\u116D\u11B0", "\uB43D", "\u1103\u116D\u11B0" },
            new string [] { "\uB43E", "\uB43E", "\u1103\u116D\u11B1", "\uB43E", "\u1103\u116D\u11B1" },
            new string [] { "\uB43F", "\uB43F", "\u1103\u116D\u11B2", "\uB43F", "\u1103\u116D\u11B2" },
            new string [] { "\uB440", "\uB440", "\u1103\u116D\u11B3", "\uB440", "\u1103\u116D\u11B3" },
            new string [] { "\uB441", "\uB441", "\u1103\u116D\u11B4", "\uB441", "\u1103\u116D\u11B4" },
            new string [] { "\uB442", "\uB442", "\u1103\u116D\u11B5", "\uB442", "\u1103\u116D\u11B5" },
            new string [] { "\uB443", "\uB443", "\u1103\u116D\u11B6", "\uB443", "\u1103\u116D\u11B6" },
            new string [] { "\uB444", "\uB444", "\u1103\u116D\u11B7", "\uB444", "\u1103\u116D\u11B7" },
            new string [] { "\uB445", "\uB445", "\u1103\u116D\u11B8", "\uB445", "\u1103\u116D\u11B8" },
            new string [] { "\uB446", "\uB446", "\u1103\u116D\u11B9", "\uB446", "\u1103\u116D\u11B9" },
            new string [] { "\uB447", "\uB447", "\u1103\u116D\u11BA", "\uB447", "\u1103\u116D\u11BA" },
            new string [] { "\uB448", "\uB448", "\u1103\u116D\u11BB", "\uB448", "\u1103\u116D\u11BB" },
            new string [] { "\uB449", "\uB449", "\u1103\u116D\u11BC", "\uB449", "\u1103\u116D\u11BC" },
            new string [] { "\uB44A", "\uB44A", "\u1103\u116D\u11BD", "\uB44A", "\u1103\u116D\u11BD" },
            new string [] { "\uB44B", "\uB44B", "\u1103\u116D\u11BE", "\uB44B", "\u1103\u116D\u11BE" },
            new string [] { "\uB44C", "\uB44C", "\u1103\u116D\u11BF", "\uB44C", "\u1103\u116D\u11BF" },
            new string [] { "\uB44D", "\uB44D", "\u1103\u116D\u11C0", "\uB44D", "\u1103\u116D\u11C0" },
            new string [] { "\uB44E", "\uB44E", "\u1103\u116D\u11C1", "\uB44E", "\u1103\u116D\u11C1" },
            new string [] { "\uB44F", "\uB44F", "\u1103\u116D\u11C2", "\uB44F", "\u1103\u116D\u11C2" },
            new string [] { "\uB450", "\uB450", "\u1103\u116E", "\uB450", "\u1103\u116E" },
            new string [] { "\uB451", "\uB451", "\u1103\u116E\u11A8", "\uB451", "\u1103\u116E\u11A8" },
            new string [] { "\uB452", "\uB452", "\u1103\u116E\u11A9", "\uB452", "\u1103\u116E\u11A9" },
            new string [] { "\uB453", "\uB453", "\u1103\u116E\u11AA", "\uB453", "\u1103\u116E\u11AA" },
            new string [] { "\uB454", "\uB454", "\u1103\u116E\u11AB", "\uB454", "\u1103\u116E\u11AB" },
            new string [] { "\uB455", "\uB455", "\u1103\u116E\u11AC", "\uB455", "\u1103\u116E\u11AC" },
            new string [] { "\uB456", "\uB456", "\u1103\u116E\u11AD", "\uB456", "\u1103\u116E\u11AD" },
            new string [] { "\uB457", "\uB457", "\u1103\u116E\u11AE", "\uB457", "\u1103\u116E\u11AE" },
            new string [] { "\uB458", "\uB458", "\u1103\u116E\u11AF", "\uB458", "\u1103\u116E\u11AF" },
            new string [] { "\uB459", "\uB459", "\u1103\u116E\u11B0", "\uB459", "\u1103\u116E\u11B0" },
            new string [] { "\uB45A", "\uB45A", "\u1103\u116E\u11B1", "\uB45A", "\u1103\u116E\u11B1" },
            new string [] { "\uB45B", "\uB45B", "\u1103\u116E\u11B2", "\uB45B", "\u1103\u116E\u11B2" },
            new string [] { "\uB45C", "\uB45C", "\u1103\u116E\u11B3", "\uB45C", "\u1103\u116E\u11B3" },
            new string [] { "\uB45D", "\uB45D", "\u1103\u116E\u11B4", "\uB45D", "\u1103\u116E\u11B4" },
            new string [] { "\uB45E", "\uB45E", "\u1103\u116E\u11B5", "\uB45E", "\u1103\u116E\u11B5" },
            new string [] { "\uB45F", "\uB45F", "\u1103\u116E\u11B6", "\uB45F", "\u1103\u116E\u11B6" },
            new string [] { "\uB460", "\uB460", "\u1103\u116E\u11B7", "\uB460", "\u1103\u116E\u11B7" },
            new string [] { "\uB461", "\uB461", "\u1103\u116E\u11B8", "\uB461", "\u1103\u116E\u11B8" },
            new string [] { "\uB462", "\uB462", "\u1103\u116E\u11B9", "\uB462", "\u1103\u116E\u11B9" },
            new string [] { "\uB463", "\uB463", "\u1103\u116E\u11BA", "\uB463", "\u1103\u116E\u11BA" },
            new string [] { "\uB464", "\uB464", "\u1103\u116E\u11BB", "\uB464", "\u1103\u116E\u11BB" },
            new string [] { "\uB465", "\uB465", "\u1103\u116E\u11BC", "\uB465", "\u1103\u116E\u11BC" },
            new string [] { "\uB466", "\uB466", "\u1103\u116E\u11BD", "\uB466", "\u1103\u116E\u11BD" },
            new string [] { "\uB467", "\uB467", "\u1103\u116E\u11BE", "\uB467", "\u1103\u116E\u11BE" },
            new string [] { "\uB468", "\uB468", "\u1103\u116E\u11BF", "\uB468", "\u1103\u116E\u11BF" },
            new string [] { "\uB469", "\uB469", "\u1103\u116E\u11C0", "\uB469", "\u1103\u116E\u11C0" },
            new string [] { "\uB46A", "\uB46A", "\u1103\u116E\u11C1", "\uB46A", "\u1103\u116E\u11C1" },
            new string [] { "\uB46B", "\uB46B", "\u1103\u116E\u11C2", "\uB46B", "\u1103\u116E\u11C2" },
            new string [] { "\uB46C", "\uB46C", "\u1103\u116F", "\uB46C", "\u1103\u116F" },
            new string [] { "\uB46D", "\uB46D", "\u1103\u116F\u11A8", "\uB46D", "\u1103\u116F\u11A8" },
            new string [] { "\uB46E", "\uB46E", "\u1103\u116F\u11A9", "\uB46E", "\u1103\u116F\u11A9" },
            new string [] { "\uB46F", "\uB46F", "\u1103\u116F\u11AA", "\uB46F", "\u1103\u116F\u11AA" },
            new string [] { "\uB470", "\uB470", "\u1103\u116F\u11AB", "\uB470", "\u1103\u116F\u11AB" },
            new string [] { "\uB471", "\uB471", "\u1103\u116F\u11AC", "\uB471", "\u1103\u116F\u11AC" },
            new string [] { "\uB472", "\uB472", "\u1103\u116F\u11AD", "\uB472", "\u1103\u116F\u11AD" },
            new string [] { "\uB473", "\uB473", "\u1103\u116F\u11AE", "\uB473", "\u1103\u116F\u11AE" },
            new string [] { "\uB474", "\uB474", "\u1103\u116F\u11AF", "\uB474", "\u1103\u116F\u11AF" },
            new string [] { "\uB475", "\uB475", "\u1103\u116F\u11B0", "\uB475", "\u1103\u116F\u11B0" },
            new string [] { "\uB476", "\uB476", "\u1103\u116F\u11B1", "\uB476", "\u1103\u116F\u11B1" },
            new string [] { "\uB477", "\uB477", "\u1103\u116F\u11B2", "\uB477", "\u1103\u116F\u11B2" },
            new string [] { "\uB478", "\uB478", "\u1103\u116F\u11B3", "\uB478", "\u1103\u116F\u11B3" },
            new string [] { "\uB479", "\uB479", "\u1103\u116F\u11B4", "\uB479", "\u1103\u116F\u11B4" },
            new string [] { "\uB47A", "\uB47A", "\u1103\u116F\u11B5", "\uB47A", "\u1103\u116F\u11B5" },
            new string [] { "\uB47B", "\uB47B", "\u1103\u116F\u11B6", "\uB47B", "\u1103\u116F\u11B6" },
            new string [] { "\uB47C", "\uB47C", "\u1103\u116F\u11B7", "\uB47C", "\u1103\u116F\u11B7" },
            new string [] { "\uB47D", "\uB47D", "\u1103\u116F\u11B8", "\uB47D", "\u1103\u116F\u11B8" },
            new string [] { "\uB47E", "\uB47E", "\u1103\u116F\u11B9", "\uB47E", "\u1103\u116F\u11B9" },
            new string [] { "\uB47F", "\uB47F", "\u1103\u116F\u11BA", "\uB47F", "\u1103\u116F\u11BA" },
            new string [] { "\uB480", "\uB480", "\u1103\u116F\u11BB", "\uB480", "\u1103\u116F\u11BB" },
            new string [] { "\uB481", "\uB481", "\u1103\u116F\u11BC", "\uB481", "\u1103\u116F\u11BC" },
            new string [] { "\uB482", "\uB482", "\u1103\u116F\u11BD", "\uB482", "\u1103\u116F\u11BD" },
            new string [] { "\uB483", "\uB483", "\u1103\u116F\u11BE", "\uB483", "\u1103\u116F\u11BE" },
            new string [] { "\uB484", "\uB484", "\u1103\u116F\u11BF", "\uB484", "\u1103\u116F\u11BF" },
            new string [] { "\uB485", "\uB485", "\u1103\u116F\u11C0", "\uB485", "\u1103\u116F\u11C0" },
            new string [] { "\uB486", "\uB486", "\u1103\u116F\u11C1", "\uB486", "\u1103\u116F\u11C1" },
            new string [] { "\uB487", "\uB487", "\u1103\u116F\u11C2", "\uB487", "\u1103\u116F\u11C2" },
            new string [] { "\uB488", "\uB488", "\u1103\u1170", "\uB488", "\u1103\u1170" },
            new string [] { "\uB489", "\uB489", "\u1103\u1170\u11A8", "\uB489", "\u1103\u1170\u11A8" },
            new string [] { "\uB48A", "\uB48A", "\u1103\u1170\u11A9", "\uB48A", "\u1103\u1170\u11A9" },
            new string [] { "\uB48B", "\uB48B", "\u1103\u1170\u11AA", "\uB48B", "\u1103\u1170\u11AA" },
            new string [] { "\uB48C", "\uB48C", "\u1103\u1170\u11AB", "\uB48C", "\u1103\u1170\u11AB" },
            new string [] { "\uB48D", "\uB48D", "\u1103\u1170\u11AC", "\uB48D", "\u1103\u1170\u11AC" },
            new string [] { "\uB48E", "\uB48E", "\u1103\u1170\u11AD", "\uB48E", "\u1103\u1170\u11AD" },
            new string [] { "\uB48F", "\uB48F", "\u1103\u1170\u11AE", "\uB48F", "\u1103\u1170\u11AE" },
            new string [] { "\uB490", "\uB490", "\u1103\u1170\u11AF", "\uB490", "\u1103\u1170\u11AF" },
            new string [] { "\uB491", "\uB491", "\u1103\u1170\u11B0", "\uB491", "\u1103\u1170\u11B0" },
            new string [] { "\uB492", "\uB492", "\u1103\u1170\u11B1", "\uB492", "\u1103\u1170\u11B1" },
            new string [] { "\uB493", "\uB493", "\u1103\u1170\u11B2", "\uB493", "\u1103\u1170\u11B2" },
            new string [] { "\uB494", "\uB494", "\u1103\u1170\u11B3", "\uB494", "\u1103\u1170\u11B3" },
            new string [] { "\uB495", "\uB495", "\u1103\u1170\u11B4", "\uB495", "\u1103\u1170\u11B4" },
            new string [] { "\uB496", "\uB496", "\u1103\u1170\u11B5", "\uB496", "\u1103\u1170\u11B5" },
            new string [] { "\uB497", "\uB497", "\u1103\u1170\u11B6", "\uB497", "\u1103\u1170\u11B6" },
            new string [] { "\uB498", "\uB498", "\u1103\u1170\u11B7", "\uB498", "\u1103\u1170\u11B7" },
            new string [] { "\uB499", "\uB499", "\u1103\u1170\u11B8", "\uB499", "\u1103\u1170\u11B8" },
            new string [] { "\uB49A", "\uB49A", "\u1103\u1170\u11B9", "\uB49A", "\u1103\u1170\u11B9" },
            new string [] { "\uB49B", "\uB49B", "\u1103\u1170\u11BA", "\uB49B", "\u1103\u1170\u11BA" },
            new string [] { "\uB49C", "\uB49C", "\u1103\u1170\u11BB", "\uB49C", "\u1103\u1170\u11BB" },
            new string [] { "\uB49D", "\uB49D", "\u1103\u1170\u11BC", "\uB49D", "\u1103\u1170\u11BC" },
            new string [] { "\uB49E", "\uB49E", "\u1103\u1170\u11BD", "\uB49E", "\u1103\u1170\u11BD" },
            new string [] { "\uB49F", "\uB49F", "\u1103\u1170\u11BE", "\uB49F", "\u1103\u1170\u11BE" },
            new string [] { "\uB4A0", "\uB4A0", "\u1103\u1170\u11BF", "\uB4A0", "\u1103\u1170\u11BF" },
            new string [] { "\uB4A1", "\uB4A1", "\u1103\u1170\u11C0", "\uB4A1", "\u1103\u1170\u11C0" },
            new string [] { "\uB4A2", "\uB4A2", "\u1103\u1170\u11C1", "\uB4A2", "\u1103\u1170\u11C1" },
            new string [] { "\uB4A3", "\uB4A3", "\u1103\u1170\u11C2", "\uB4A3", "\u1103\u1170\u11C2" },
            new string [] { "\uB4A4", "\uB4A4", "\u1103\u1171", "\uB4A4", "\u1103\u1171" },
            new string [] { "\uB4A5", "\uB4A5", "\u1103\u1171\u11A8", "\uB4A5", "\u1103\u1171\u11A8" },
            new string [] { "\uB4A6", "\uB4A6", "\u1103\u1171\u11A9", "\uB4A6", "\u1103\u1171\u11A9" },
            new string [] { "\uB4A7", "\uB4A7", "\u1103\u1171\u11AA", "\uB4A7", "\u1103\u1171\u11AA" },
            new string [] { "\uB4A8", "\uB4A8", "\u1103\u1171\u11AB", "\uB4A8", "\u1103\u1171\u11AB" },
            new string [] { "\uB4A9", "\uB4A9", "\u1103\u1171\u11AC", "\uB4A9", "\u1103\u1171\u11AC" },
            new string [] { "\uB4AA", "\uB4AA", "\u1103\u1171\u11AD", "\uB4AA", "\u1103\u1171\u11AD" },
            new string [] { "\uB4AB", "\uB4AB", "\u1103\u1171\u11AE", "\uB4AB", "\u1103\u1171\u11AE" },
            new string [] { "\uB4AC", "\uB4AC", "\u1103\u1171\u11AF", "\uB4AC", "\u1103\u1171\u11AF" },
            new string [] { "\uB4AD", "\uB4AD", "\u1103\u1171\u11B0", "\uB4AD", "\u1103\u1171\u11B0" },
            new string [] { "\uB4AE", "\uB4AE", "\u1103\u1171\u11B1", "\uB4AE", "\u1103\u1171\u11B1" },
            new string [] { "\uB4AF", "\uB4AF", "\u1103\u1171\u11B2", "\uB4AF", "\u1103\u1171\u11B2" },
            new string [] { "\uB4B0", "\uB4B0", "\u1103\u1171\u11B3", "\uB4B0", "\u1103\u1171\u11B3" },
            new string [] { "\uB4B1", "\uB4B1", "\u1103\u1171\u11B4", "\uB4B1", "\u1103\u1171\u11B4" },
            new string [] { "\uB4B2", "\uB4B2", "\u1103\u1171\u11B5", "\uB4B2", "\u1103\u1171\u11B5" },
            new string [] { "\uB4B3", "\uB4B3", "\u1103\u1171\u11B6", "\uB4B3", "\u1103\u1171\u11B6" },
            new string [] { "\uB4B4", "\uB4B4", "\u1103\u1171\u11B7", "\uB4B4", "\u1103\u1171\u11B7" },
            new string [] { "\uB4B5", "\uB4B5", "\u1103\u1171\u11B8", "\uB4B5", "\u1103\u1171\u11B8" },
            new string [] { "\uB4B6", "\uB4B6", "\u1103\u1171\u11B9", "\uB4B6", "\u1103\u1171\u11B9" },
            new string [] { "\uB4B7", "\uB4B7", "\u1103\u1171\u11BA", "\uB4B7", "\u1103\u1171\u11BA" },
            new string [] { "\uB4B8", "\uB4B8", "\u1103\u1171\u11BB", "\uB4B8", "\u1103\u1171\u11BB" },
            new string [] { "\uB4B9", "\uB4B9", "\u1103\u1171\u11BC", "\uB4B9", "\u1103\u1171\u11BC" },
            new string [] { "\uB4BA", "\uB4BA", "\u1103\u1171\u11BD", "\uB4BA", "\u1103\u1171\u11BD" },
            new string [] { "\uB4BB", "\uB4BB", "\u1103\u1171\u11BE", "\uB4BB", "\u1103\u1171\u11BE" },
            new string [] { "\uB4BC", "\uB4BC", "\u1103\u1171\u11BF", "\uB4BC", "\u1103\u1171\u11BF" },
            new string [] { "\uB4BD", "\uB4BD", "\u1103\u1171\u11C0", "\uB4BD", "\u1103\u1171\u11C0" },
            new string [] { "\uB4BE", "\uB4BE", "\u1103\u1171\u11C1", "\uB4BE", "\u1103\u1171\u11C1" },
            new string [] { "\uB4BF", "\uB4BF", "\u1103\u1171\u11C2", "\uB4BF", "\u1103\u1171\u11C2" },
            new string [] { "\uB4C0", "\uB4C0", "\u1103\u1172", "\uB4C0", "\u1103\u1172" },
            new string [] { "\uB4C1", "\uB4C1", "\u1103\u1172\u11A8", "\uB4C1", "\u1103\u1172\u11A8" },
            new string [] { "\uB4C2", "\uB4C2", "\u1103\u1172\u11A9", "\uB4C2", "\u1103\u1172\u11A9" },
            new string [] { "\uB4C3", "\uB4C3", "\u1103\u1172\u11AA", "\uB4C3", "\u1103\u1172\u11AA" },
            new string [] { "\uB4C4", "\uB4C4", "\u1103\u1172\u11AB", "\uB4C4", "\u1103\u1172\u11AB" },
            new string [] { "\uB4C5", "\uB4C5", "\u1103\u1172\u11AC", "\uB4C5", "\u1103\u1172\u11AC" },
            new string [] { "\uB4C6", "\uB4C6", "\u1103\u1172\u11AD", "\uB4C6", "\u1103\u1172\u11AD" },
            new string [] { "\uB4C7", "\uB4C7", "\u1103\u1172\u11AE", "\uB4C7", "\u1103\u1172\u11AE" },
            new string [] { "\uB4C8", "\uB4C8", "\u1103\u1172\u11AF", "\uB4C8", "\u1103\u1172\u11AF" },
            new string [] { "\uB4C9", "\uB4C9", "\u1103\u1172\u11B0", "\uB4C9", "\u1103\u1172\u11B0" },
            new string [] { "\uB4CA", "\uB4CA", "\u1103\u1172\u11B1", "\uB4CA", "\u1103\u1172\u11B1" },
            new string [] { "\uB4CB", "\uB4CB", "\u1103\u1172\u11B2", "\uB4CB", "\u1103\u1172\u11B2" },
            new string [] { "\uB4CC", "\uB4CC", "\u1103\u1172\u11B3", "\uB4CC", "\u1103\u1172\u11B3" },
            new string [] { "\uB4CD", "\uB4CD", "\u1103\u1172\u11B4", "\uB4CD", "\u1103\u1172\u11B4" },
            new string [] { "\uB4CE", "\uB4CE", "\u1103\u1172\u11B5", "\uB4CE", "\u1103\u1172\u11B5" },
            new string [] { "\uB4CF", "\uB4CF", "\u1103\u1172\u11B6", "\uB4CF", "\u1103\u1172\u11B6" },
            new string [] { "\uB4D0", "\uB4D0", "\u1103\u1172\u11B7", "\uB4D0", "\u1103\u1172\u11B7" },
            new string [] { "\uB4D1", "\uB4D1", "\u1103\u1172\u11B8", "\uB4D1", "\u1103\u1172\u11B8" },
            new string [] { "\uB4D2", "\uB4D2", "\u1103\u1172\u11B9", "\uB4D2", "\u1103\u1172\u11B9" },
            new string [] { "\uB4D3", "\uB4D3", "\u1103\u1172\u11BA", "\uB4D3", "\u1103\u1172\u11BA" },
            new string [] { "\uB4D4", "\uB4D4", "\u1103\u1172\u11BB", "\uB4D4", "\u1103\u1172\u11BB" },
            new string [] { "\uB4D5", "\uB4D5", "\u1103\u1172\u11BC", "\uB4D5", "\u1103\u1172\u11BC" },
            new string [] { "\uB4D6", "\uB4D6", "\u1103\u1172\u11BD", "\uB4D6", "\u1103\u1172\u11BD" },
            new string [] { "\uB4D7", "\uB4D7", "\u1103\u1172\u11BE", "\uB4D7", "\u1103\u1172\u11BE" },
            new string [] { "\uB4D8", "\uB4D8", "\u1103\u1172\u11BF", "\uB4D8", "\u1103\u1172\u11BF" },
            new string [] { "\uB4D9", "\uB4D9", "\u1103\u1172\u11C0", "\uB4D9", "\u1103\u1172\u11C0" },
            new string [] { "\uB4DA", "\uB4DA", "\u1103\u1172\u11C1", "\uB4DA", "\u1103\u1172\u11C1" },
            new string [] { "\uB4DB", "\uB4DB", "\u1103\u1172\u11C2", "\uB4DB", "\u1103\u1172\u11C2" },
            new string [] { "\uB4DC", "\uB4DC", "\u1103\u1173", "\uB4DC", "\u1103\u1173" },
            new string [] { "\uB4DD", "\uB4DD", "\u1103\u1173\u11A8", "\uB4DD", "\u1103\u1173\u11A8" },
            new string [] { "\uB4DE", "\uB4DE", "\u1103\u1173\u11A9", "\uB4DE", "\u1103\u1173\u11A9" },
            new string [] { "\uB4DF", "\uB4DF", "\u1103\u1173\u11AA", "\uB4DF", "\u1103\u1173\u11AA" },
            new string [] { "\uB4E0", "\uB4E0", "\u1103\u1173\u11AB", "\uB4E0", "\u1103\u1173\u11AB" },
            new string [] { "\uB4E1", "\uB4E1", "\u1103\u1173\u11AC", "\uB4E1", "\u1103\u1173\u11AC" },
            new string [] { "\uB4E2", "\uB4E2", "\u1103\u1173\u11AD", "\uB4E2", "\u1103\u1173\u11AD" },
            new string [] { "\uB4E3", "\uB4E3", "\u1103\u1173\u11AE", "\uB4E3", "\u1103\u1173\u11AE" },
            new string [] { "\uB4E4", "\uB4E4", "\u1103\u1173\u11AF", "\uB4E4", "\u1103\u1173\u11AF" },
            new string [] { "\uB4E5", "\uB4E5", "\u1103\u1173\u11B0", "\uB4E5", "\u1103\u1173\u11B0" },
            new string [] { "\uB4E6", "\uB4E6", "\u1103\u1173\u11B1", "\uB4E6", "\u1103\u1173\u11B1" },
            new string [] { "\uB4E7", "\uB4E7", "\u1103\u1173\u11B2", "\uB4E7", "\u1103\u1173\u11B2" },
            new string [] { "\uB4E8", "\uB4E8", "\u1103\u1173\u11B3", "\uB4E8", "\u1103\u1173\u11B3" },
            new string [] { "\uB4E9", "\uB4E9", "\u1103\u1173\u11B4", "\uB4E9", "\u1103\u1173\u11B4" },
            new string [] { "\uB4EA", "\uB4EA", "\u1103\u1173\u11B5", "\uB4EA", "\u1103\u1173\u11B5" },
            new string [] { "\uB4EB", "\uB4EB", "\u1103\u1173\u11B6", "\uB4EB", "\u1103\u1173\u11B6" },
            new string [] { "\uB4EC", "\uB4EC", "\u1103\u1173\u11B7", "\uB4EC", "\u1103\u1173\u11B7" },
            new string [] { "\uB4ED", "\uB4ED", "\u1103\u1173\u11B8", "\uB4ED", "\u1103\u1173\u11B8" },
            new string [] { "\uB4EE", "\uB4EE", "\u1103\u1173\u11B9", "\uB4EE", "\u1103\u1173\u11B9" },
            new string [] { "\uB4EF", "\uB4EF", "\u1103\u1173\u11BA", "\uB4EF", "\u1103\u1173\u11BA" },
            new string [] { "\uB4F0", "\uB4F0", "\u1103\u1173\u11BB", "\uB4F0", "\u1103\u1173\u11BB" },
            new string [] { "\uB4F1", "\uB4F1", "\u1103\u1173\u11BC", "\uB4F1", "\u1103\u1173\u11BC" },
            new string [] { "\uB4F2", "\uB4F2", "\u1103\u1173\u11BD", "\uB4F2", "\u1103\u1173\u11BD" },
            new string [] { "\uB4F3", "\uB4F3", "\u1103\u1173\u11BE", "\uB4F3", "\u1103\u1173\u11BE" },
            new string [] { "\uB4F4", "\uB4F4", "\u1103\u1173\u11BF", "\uB4F4", "\u1103\u1173\u11BF" },
            new string [] { "\uB4F5", "\uB4F5", "\u1103\u1173\u11C0", "\uB4F5", "\u1103\u1173\u11C0" },
            new string [] { "\uB4F6", "\uB4F6", "\u1103\u1173\u11C1", "\uB4F6", "\u1103\u1173\u11C1" },
            new string [] { "\uB4F7", "\uB4F7", "\u1103\u1173\u11C2", "\uB4F7", "\u1103\u1173\u11C2" },
            new string [] { "\uB4F8", "\uB4F8", "\u1103\u1174", "\uB4F8", "\u1103\u1174" },
            new string [] { "\uB4F9", "\uB4F9", "\u1103\u1174\u11A8", "\uB4F9", "\u1103\u1174\u11A8" },
            new string [] { "\uB4FA", "\uB4FA", "\u1103\u1174\u11A9", "\uB4FA", "\u1103\u1174\u11A9" },
            new string [] { "\uB4FB", "\uB4FB", "\u1103\u1174\u11AA", "\uB4FB", "\u1103\u1174\u11AA" },
            new string [] { "\uB4FC", "\uB4FC", "\u1103\u1174\u11AB", "\uB4FC", "\u1103\u1174\u11AB" },
            new string [] { "\uB4FD", "\uB4FD", "\u1103\u1174\u11AC", "\uB4FD", "\u1103\u1174\u11AC" },
            new string [] { "\uB4FE", "\uB4FE", "\u1103\u1174\u11AD", "\uB4FE", "\u1103\u1174\u11AD" },
            new string [] { "\uB4FF", "\uB4FF", "\u1103\u1174\u11AE", "\uB4FF", "\u1103\u1174\u11AE" },
            new string [] { "\uB500", "\uB500", "\u1103\u1174\u11AF", "\uB500", "\u1103\u1174\u11AF" },
            new string [] { "\uB501", "\uB501", "\u1103\u1174\u11B0", "\uB501", "\u1103\u1174\u11B0" },
            new string [] { "\uB502", "\uB502", "\u1103\u1174\u11B1", "\uB502", "\u1103\u1174\u11B1" },
            new string [] { "\uB503", "\uB503", "\u1103\u1174\u11B2", "\uB503", "\u1103\u1174\u11B2" },
            new string [] { "\uB504", "\uB504", "\u1103\u1174\u11B3", "\uB504", "\u1103\u1174\u11B3" },
            new string [] { "\uB505", "\uB505", "\u1103\u1174\u11B4", "\uB505", "\u1103\u1174\u11B4" },
            new string [] { "\uB506", "\uB506", "\u1103\u1174\u11B5", "\uB506", "\u1103\u1174\u11B5" },
            new string [] { "\uB507", "\uB507", "\u1103\u1174\u11B6", "\uB507", "\u1103\u1174\u11B6" },
            new string [] { "\uB508", "\uB508", "\u1103\u1174\u11B7", "\uB508", "\u1103\u1174\u11B7" },
            new string [] { "\uB509", "\uB509", "\u1103\u1174\u11B8", "\uB509", "\u1103\u1174\u11B8" },
            new string [] { "\uB50A", "\uB50A", "\u1103\u1174\u11B9", "\uB50A", "\u1103\u1174\u11B9" },
            new string [] { "\uB50B", "\uB50B", "\u1103\u1174\u11BA", "\uB50B", "\u1103\u1174\u11BA" },
            new string [] { "\uB50C", "\uB50C", "\u1103\u1174\u11BB", "\uB50C", "\u1103\u1174\u11BB" },
            new string [] { "\uB50D", "\uB50D", "\u1103\u1174\u11BC", "\uB50D", "\u1103\u1174\u11BC" },
            new string [] { "\uB50E", "\uB50E", "\u1103\u1174\u11BD", "\uB50E", "\u1103\u1174\u11BD" },
            new string [] { "\uB50F", "\uB50F", "\u1103\u1174\u11BE", "\uB50F", "\u1103\u1174\u11BE" },
            new string [] { "\uB510", "\uB510", "\u1103\u1174\u11BF", "\uB510", "\u1103\u1174\u11BF" },
            new string [] { "\uB511", "\uB511", "\u1103\u1174\u11C0", "\uB511", "\u1103\u1174\u11C0" },
            new string [] { "\uB512", "\uB512", "\u1103\u1174\u11C1", "\uB512", "\u1103\u1174\u11C1" },
            new string [] { "\uB513", "\uB513", "\u1103\u1174\u11C2", "\uB513", "\u1103\u1174\u11C2" },
            new string [] { "\uB514", "\uB514", "\u1103\u1175", "\uB514", "\u1103\u1175" },
            new string [] { "\uB515", "\uB515", "\u1103\u1175\u11A8", "\uB515", "\u1103\u1175\u11A8" },
            new string [] { "\uB516", "\uB516", "\u1103\u1175\u11A9", "\uB516", "\u1103\u1175\u11A9" },
            new string [] { "\uB517", "\uB517", "\u1103\u1175\u11AA", "\uB517", "\u1103\u1175\u11AA" },
            new string [] { "\uB518", "\uB518", "\u1103\u1175\u11AB", "\uB518", "\u1103\u1175\u11AB" },
            new string [] { "\uB519", "\uB519", "\u1103\u1175\u11AC", "\uB519", "\u1103\u1175\u11AC" },
            new string [] { "\uB51A", "\uB51A", "\u1103\u1175\u11AD", "\uB51A", "\u1103\u1175\u11AD" },
            new string [] { "\uB51B", "\uB51B", "\u1103\u1175\u11AE", "\uB51B", "\u1103\u1175\u11AE" },
            new string [] { "\uB51C", "\uB51C", "\u1103\u1175\u11AF", "\uB51C", "\u1103\u1175\u11AF" },
            new string [] { "\uB51D", "\uB51D", "\u1103\u1175\u11B0", "\uB51D", "\u1103\u1175\u11B0" },
            new string [] { "\uB51E", "\uB51E", "\u1103\u1175\u11B1", "\uB51E", "\u1103\u1175\u11B1" },
            new string [] { "\uB51F", "\uB51F", "\u1103\u1175\u11B2", "\uB51F", "\u1103\u1175\u11B2" },
            new string [] { "\uB520", "\uB520", "\u1103\u1175\u11B3", "\uB520", "\u1103\u1175\u11B3" },
            new string [] { "\uB521", "\uB521", "\u1103\u1175\u11B4", "\uB521", "\u1103\u1175\u11B4" },
            new string [] { "\uB522", "\uB522", "\u1103\u1175\u11B5", "\uB522", "\u1103\u1175\u11B5" },
            new string [] { "\uB523", "\uB523", "\u1103\u1175\u11B6", "\uB523", "\u1103\u1175\u11B6" },
            new string [] { "\uB524", "\uB524", "\u1103\u1175\u11B7", "\uB524", "\u1103\u1175\u11B7" },
            new string [] { "\uB525", "\uB525", "\u1103\u1175\u11B8", "\uB525", "\u1103\u1175\u11B8" },
            new string [] { "\uB526", "\uB526", "\u1103\u1175\u11B9", "\uB526", "\u1103\u1175\u11B9" },
            new string [] { "\uB527", "\uB527", "\u1103\u1175\u11BA", "\uB527", "\u1103\u1175\u11BA" },
            new string [] { "\uB528", "\uB528", "\u1103\u1175\u11BB", "\uB528", "\u1103\u1175\u11BB" },
            new string [] { "\uB529", "\uB529", "\u1103\u1175\u11BC", "\uB529", "\u1103\u1175\u11BC" },
            new string [] { "\uB52A", "\uB52A", "\u1103\u1175\u11BD", "\uB52A", "\u1103\u1175\u11BD" },
            new string [] { "\uB52B", "\uB52B", "\u1103\u1175\u11BE", "\uB52B", "\u1103\u1175\u11BE" },
            new string [] { "\uB52C", "\uB52C", "\u1103\u1175\u11BF", "\uB52C", "\u1103\u1175\u11BF" },
            new string [] { "\uB52D", "\uB52D", "\u1103\u1175\u11C0", "\uB52D", "\u1103\u1175\u11C0" },
            new string [] { "\uB52E", "\uB52E", "\u1103\u1175\u11C1", "\uB52E", "\u1103\u1175\u11C1" },
            new string [] { "\uB52F", "\uB52F", "\u1103\u1175\u11C2", "\uB52F", "\u1103\u1175\u11C2" },
            new string [] { "\uB530", "\uB530", "\u1104\u1161", "\uB530", "\u1104\u1161" },
            new string [] { "\uB531", "\uB531", "\u1104\u1161\u11A8", "\uB531", "\u1104\u1161\u11A8" },
            new string [] { "\uB532", "\uB532", "\u1104\u1161\u11A9", "\uB532", "\u1104\u1161\u11A9" },
            new string [] { "\uB533", "\uB533", "\u1104\u1161\u11AA", "\uB533", "\u1104\u1161\u11AA" },
            new string [] { "\uB534", "\uB534", "\u1104\u1161\u11AB", "\uB534", "\u1104\u1161\u11AB" },
            new string [] { "\uB535", "\uB535", "\u1104\u1161\u11AC", "\uB535", "\u1104\u1161\u11AC" },
            new string [] { "\uB536", "\uB536", "\u1104\u1161\u11AD", "\uB536", "\u1104\u1161\u11AD" },
            new string [] { "\uB537", "\uB537", "\u1104\u1161\u11AE", "\uB537", "\u1104\u1161\u11AE" },
            new string [] { "\uB538", "\uB538", "\u1104\u1161\u11AF", "\uB538", "\u1104\u1161\u11AF" },
            new string [] { "\uB539", "\uB539", "\u1104\u1161\u11B0", "\uB539", "\u1104\u1161\u11B0" },
            new string [] { "\uB53A", "\uB53A", "\u1104\u1161\u11B1", "\uB53A", "\u1104\u1161\u11B1" },
            new string [] { "\uB53B", "\uB53B", "\u1104\u1161\u11B2", "\uB53B", "\u1104\u1161\u11B2" },
            new string [] { "\uB53C", "\uB53C", "\u1104\u1161\u11B3", "\uB53C", "\u1104\u1161\u11B3" },
            new string [] { "\uB53D", "\uB53D", "\u1104\u1161\u11B4", "\uB53D", "\u1104\u1161\u11B4" },
            new string [] { "\uB53E", "\uB53E", "\u1104\u1161\u11B5", "\uB53E", "\u1104\u1161\u11B5" },
            new string [] { "\uB53F", "\uB53F", "\u1104\u1161\u11B6", "\uB53F", "\u1104\u1161\u11B6" },
            new string [] { "\uB540", "\uB540", "\u1104\u1161\u11B7", "\uB540", "\u1104\u1161\u11B7" },
            new string [] { "\uB541", "\uB541", "\u1104\u1161\u11B8", "\uB541", "\u1104\u1161\u11B8" },
            new string [] { "\uB542", "\uB542", "\u1104\u1161\u11B9", "\uB542", "\u1104\u1161\u11B9" },
            new string [] { "\uB543", "\uB543", "\u1104\u1161\u11BA", "\uB543", "\u1104\u1161\u11BA" },
            new string [] { "\uB544", "\uB544", "\u1104\u1161\u11BB", "\uB544", "\u1104\u1161\u11BB" },
            new string [] { "\uB545", "\uB545", "\u1104\u1161\u11BC", "\uB545", "\u1104\u1161\u11BC" },
            new string [] { "\uB546", "\uB546", "\u1104\u1161\u11BD", "\uB546", "\u1104\u1161\u11BD" },
            new string [] { "\uB547", "\uB547", "\u1104\u1161\u11BE", "\uB547", "\u1104\u1161\u11BE" },
            new string [] { "\uB548", "\uB548", "\u1104\u1161\u11BF", "\uB548", "\u1104\u1161\u11BF" },
            new string [] { "\uB549", "\uB549", "\u1104\u1161\u11C0", "\uB549", "\u1104\u1161\u11C0" },
            new string [] { "\uB54A", "\uB54A", "\u1104\u1161\u11C1", "\uB54A", "\u1104\u1161\u11C1" },
            new string [] { "\uB54B", "\uB54B", "\u1104\u1161\u11C2", "\uB54B", "\u1104\u1161\u11C2" },
            new string [] { "\uB54C", "\uB54C", "\u1104\u1162", "\uB54C", "\u1104\u1162" },
            new string [] { "\uB54D", "\uB54D", "\u1104\u1162\u11A8", "\uB54D", "\u1104\u1162\u11A8" },
            new string [] { "\uB54E", "\uB54E", "\u1104\u1162\u11A9", "\uB54E", "\u1104\u1162\u11A9" },
            new string [] { "\uB54F", "\uB54F", "\u1104\u1162\u11AA", "\uB54F", "\u1104\u1162\u11AA" },
            new string [] { "\uB550", "\uB550", "\u1104\u1162\u11AB", "\uB550", "\u1104\u1162\u11AB" },
            new string [] { "\uB551", "\uB551", "\u1104\u1162\u11AC", "\uB551", "\u1104\u1162\u11AC" },
            new string [] { "\uB552", "\uB552", "\u1104\u1162\u11AD", "\uB552", "\u1104\u1162\u11AD" },
            new string [] { "\uB553", "\uB553", "\u1104\u1162\u11AE", "\uB553", "\u1104\u1162\u11AE" },
            new string [] { "\uB554", "\uB554", "\u1104\u1162\u11AF", "\uB554", "\u1104\u1162\u11AF" },
            new string [] { "\uB555", "\uB555", "\u1104\u1162\u11B0", "\uB555", "\u1104\u1162\u11B0" },
            new string [] { "\uB556", "\uB556", "\u1104\u1162\u11B1", "\uB556", "\u1104\u1162\u11B1" },
            new string [] { "\uB557", "\uB557", "\u1104\u1162\u11B2", "\uB557", "\u1104\u1162\u11B2" },
            new string [] { "\uB558", "\uB558", "\u1104\u1162\u11B3", "\uB558", "\u1104\u1162\u11B3" },
            new string [] { "\uB559", "\uB559", "\u1104\u1162\u11B4", "\uB559", "\u1104\u1162\u11B4" },
            new string [] { "\uB55A", "\uB55A", "\u1104\u1162\u11B5", "\uB55A", "\u1104\u1162\u11B5" },
            new string [] { "\uB55B", "\uB55B", "\u1104\u1162\u11B6", "\uB55B", "\u1104\u1162\u11B6" },
            new string [] { "\uB55C", "\uB55C", "\u1104\u1162\u11B7", "\uB55C", "\u1104\u1162\u11B7" },
            new string [] { "\uB55D", "\uB55D", "\u1104\u1162\u11B8", "\uB55D", "\u1104\u1162\u11B8" },
            new string [] { "\uB55E", "\uB55E", "\u1104\u1162\u11B9", "\uB55E", "\u1104\u1162\u11B9" },
            new string [] { "\uB55F", "\uB55F", "\u1104\u1162\u11BA", "\uB55F", "\u1104\u1162\u11BA" },
            new string [] { "\uB560", "\uB560", "\u1104\u1162\u11BB", "\uB560", "\u1104\u1162\u11BB" },
            new string [] { "\uB561", "\uB561", "\u1104\u1162\u11BC", "\uB561", "\u1104\u1162\u11BC" },
            new string [] { "\uB562", "\uB562", "\u1104\u1162\u11BD", "\uB562", "\u1104\u1162\u11BD" },
            new string [] { "\uB563", "\uB563", "\u1104\u1162\u11BE", "\uB563", "\u1104\u1162\u11BE" },
            new string [] { "\uB564", "\uB564", "\u1104\u1162\u11BF", "\uB564", "\u1104\u1162\u11BF" },
            new string [] { "\uB565", "\uB565", "\u1104\u1162\u11C0", "\uB565", "\u1104\u1162\u11C0" },
            new string [] { "\uB566", "\uB566", "\u1104\u1162\u11C1", "\uB566", "\u1104\u1162\u11C1" },
            new string [] { "\uB567", "\uB567", "\u1104\u1162\u11C2", "\uB567", "\u1104\u1162\u11C2" },
            new string [] { "\uB568", "\uB568", "\u1104\u1163", "\uB568", "\u1104\u1163" },
            new string [] { "\uB569", "\uB569", "\u1104\u1163\u11A8", "\uB569", "\u1104\u1163\u11A8" },
            new string [] { "\uB56A", "\uB56A", "\u1104\u1163\u11A9", "\uB56A", "\u1104\u1163\u11A9" },
            new string [] { "\uB56B", "\uB56B", "\u1104\u1163\u11AA", "\uB56B", "\u1104\u1163\u11AA" },
            new string [] { "\uB56C", "\uB56C", "\u1104\u1163\u11AB", "\uB56C", "\u1104\u1163\u11AB" },
            new string [] { "\uB56D", "\uB56D", "\u1104\u1163\u11AC", "\uB56D", "\u1104\u1163\u11AC" },
            new string [] { "\uB56E", "\uB56E", "\u1104\u1163\u11AD", "\uB56E", "\u1104\u1163\u11AD" },
            new string [] { "\uB56F", "\uB56F", "\u1104\u1163\u11AE", "\uB56F", "\u1104\u1163\u11AE" },
            new string [] { "\uB570", "\uB570", "\u1104\u1163\u11AF", "\uB570", "\u1104\u1163\u11AF" },
            new string [] { "\uB571", "\uB571", "\u1104\u1163\u11B0", "\uB571", "\u1104\u1163\u11B0" },
            new string [] { "\uB572", "\uB572", "\u1104\u1163\u11B1", "\uB572", "\u1104\u1163\u11B1" },
            new string [] { "\uB573", "\uB573", "\u1104\u1163\u11B2", "\uB573", "\u1104\u1163\u11B2" },
            new string [] { "\uB574", "\uB574", "\u1104\u1163\u11B3", "\uB574", "\u1104\u1163\u11B3" },
            new string [] { "\uB575", "\uB575", "\u1104\u1163\u11B4", "\uB575", "\u1104\u1163\u11B4" },
            new string [] { "\uB576", "\uB576", "\u1104\u1163\u11B5", "\uB576", "\u1104\u1163\u11B5" },
            new string [] { "\uB577", "\uB577", "\u1104\u1163\u11B6", "\uB577", "\u1104\u1163\u11B6" },
            new string [] { "\uB578", "\uB578", "\u1104\u1163\u11B7", "\uB578", "\u1104\u1163\u11B7" },
            new string [] { "\uB579", "\uB579", "\u1104\u1163\u11B8", "\uB579", "\u1104\u1163\u11B8" },
            new string [] { "\uB57A", "\uB57A", "\u1104\u1163\u11B9", "\uB57A", "\u1104\u1163\u11B9" },
            new string [] { "\uB57B", "\uB57B", "\u1104\u1163\u11BA", "\uB57B", "\u1104\u1163\u11BA" },
            new string [] { "\uB57C", "\uB57C", "\u1104\u1163\u11BB", "\uB57C", "\u1104\u1163\u11BB" },
            new string [] { "\uB57D", "\uB57D", "\u1104\u1163\u11BC", "\uB57D", "\u1104\u1163\u11BC" },
            new string [] { "\uB57E", "\uB57E", "\u1104\u1163\u11BD", "\uB57E", "\u1104\u1163\u11BD" },
            new string [] { "\uB57F", "\uB57F", "\u1104\u1163\u11BE", "\uB57F", "\u1104\u1163\u11BE" },
            new string [] { "\uB580", "\uB580", "\u1104\u1163\u11BF", "\uB580", "\u1104\u1163\u11BF" },
            new string [] { "\uB581", "\uB581", "\u1104\u1163\u11C0", "\uB581", "\u1104\u1163\u11C0" },
            new string [] { "\uB582", "\uB582", "\u1104\u1163\u11C1", "\uB582", "\u1104\u1163\u11C1" },
            new string [] { "\uB583", "\uB583", "\u1104\u1163\u11C2", "\uB583", "\u1104\u1163\u11C2" },
            new string [] { "\uB584", "\uB584", "\u1104\u1164", "\uB584", "\u1104\u1164" },
            new string [] { "\uB585", "\uB585", "\u1104\u1164\u11A8", "\uB585", "\u1104\u1164\u11A8" },
            new string [] { "\uB586", "\uB586", "\u1104\u1164\u11A9", "\uB586", "\u1104\u1164\u11A9" },
            new string [] { "\uB587", "\uB587", "\u1104\u1164\u11AA", "\uB587", "\u1104\u1164\u11AA" },
            new string [] { "\uB588", "\uB588", "\u1104\u1164\u11AB", "\uB588", "\u1104\u1164\u11AB" },
            new string [] { "\uB589", "\uB589", "\u1104\u1164\u11AC", "\uB589", "\u1104\u1164\u11AC" },
            new string [] { "\uB58A", "\uB58A", "\u1104\u1164\u11AD", "\uB58A", "\u1104\u1164\u11AD" },
            new string [] { "\uB58B", "\uB58B", "\u1104\u1164\u11AE", "\uB58B", "\u1104\u1164\u11AE" },
            new string [] { "\uB58C", "\uB58C", "\u1104\u1164\u11AF", "\uB58C", "\u1104\u1164\u11AF" },
            new string [] { "\uB58D", "\uB58D", "\u1104\u1164\u11B0", "\uB58D", "\u1104\u1164\u11B0" },
            new string [] { "\uB58E", "\uB58E", "\u1104\u1164\u11B1", "\uB58E", "\u1104\u1164\u11B1" },
            new string [] { "\uB58F", "\uB58F", "\u1104\u1164\u11B2", "\uB58F", "\u1104\u1164\u11B2" },
            new string [] { "\uB590", "\uB590", "\u1104\u1164\u11B3", "\uB590", "\u1104\u1164\u11B3" },
            new string [] { "\uB591", "\uB591", "\u1104\u1164\u11B4", "\uB591", "\u1104\u1164\u11B4" },
            new string [] { "\uB592", "\uB592", "\u1104\u1164\u11B5", "\uB592", "\u1104\u1164\u11B5" },
            new string [] { "\uB593", "\uB593", "\u1104\u1164\u11B6", "\uB593", "\u1104\u1164\u11B6" },
            new string [] { "\uB594", "\uB594", "\u1104\u1164\u11B7", "\uB594", "\u1104\u1164\u11B7" },
            new string [] { "\uB595", "\uB595", "\u1104\u1164\u11B8", "\uB595", "\u1104\u1164\u11B8" },
            new string [] { "\uB596", "\uB596", "\u1104\u1164\u11B9", "\uB596", "\u1104\u1164\u11B9" },
            new string [] { "\uB597", "\uB597", "\u1104\u1164\u11BA", "\uB597", "\u1104\u1164\u11BA" },
            new string [] { "\uB598", "\uB598", "\u1104\u1164\u11BB", "\uB598", "\u1104\u1164\u11BB" },
            new string [] { "\uB599", "\uB599", "\u1104\u1164\u11BC", "\uB599", "\u1104\u1164\u11BC" },
            new string [] { "\uB59A", "\uB59A", "\u1104\u1164\u11BD", "\uB59A", "\u1104\u1164\u11BD" },
            new string [] { "\uB59B", "\uB59B", "\u1104\u1164\u11BE", "\uB59B", "\u1104\u1164\u11BE" },
            new string [] { "\uB59C", "\uB59C", "\u1104\u1164\u11BF", "\uB59C", "\u1104\u1164\u11BF" },
            new string [] { "\uB59D", "\uB59D", "\u1104\u1164\u11C0", "\uB59D", "\u1104\u1164\u11C0" },
            new string [] { "\uB59E", "\uB59E", "\u1104\u1164\u11C1", "\uB59E", "\u1104\u1164\u11C1" },
            new string [] { "\uB59F", "\uB59F", "\u1104\u1164\u11C2", "\uB59F", "\u1104\u1164\u11C2" },
            new string [] { "\uB5A0", "\uB5A0", "\u1104\u1165", "\uB5A0", "\u1104\u1165" },
            new string [] { "\uB5A1", "\uB5A1", "\u1104\u1165\u11A8", "\uB5A1", "\u1104\u1165\u11A8" },
            new string [] { "\uB5A2", "\uB5A2", "\u1104\u1165\u11A9", "\uB5A2", "\u1104\u1165\u11A9" },
            new string [] { "\uB5A3", "\uB5A3", "\u1104\u1165\u11AA", "\uB5A3", "\u1104\u1165\u11AA" },
            new string [] { "\uB5A4", "\uB5A4", "\u1104\u1165\u11AB", "\uB5A4", "\u1104\u1165\u11AB" },
            new string [] { "\uB5A5", "\uB5A5", "\u1104\u1165\u11AC", "\uB5A5", "\u1104\u1165\u11AC" },
            new string [] { "\uB5A6", "\uB5A6", "\u1104\u1165\u11AD", "\uB5A6", "\u1104\u1165\u11AD" },
            new string [] { "\uB5A7", "\uB5A7", "\u1104\u1165\u11AE", "\uB5A7", "\u1104\u1165\u11AE" },
            new string [] { "\uB5A8", "\uB5A8", "\u1104\u1165\u11AF", "\uB5A8", "\u1104\u1165\u11AF" },
            new string [] { "\uB5A9", "\uB5A9", "\u1104\u1165\u11B0", "\uB5A9", "\u1104\u1165\u11B0" },
            new string [] { "\uB5AA", "\uB5AA", "\u1104\u1165\u11B1", "\uB5AA", "\u1104\u1165\u11B1" },
            new string [] { "\uB5AB", "\uB5AB", "\u1104\u1165\u11B2", "\uB5AB", "\u1104\u1165\u11B2" },
            new string [] { "\uB5AC", "\uB5AC", "\u1104\u1165\u11B3", "\uB5AC", "\u1104\u1165\u11B3" },
            new string [] { "\uB5AD", "\uB5AD", "\u1104\u1165\u11B4", "\uB5AD", "\u1104\u1165\u11B4" },
            new string [] { "\uB5AE", "\uB5AE", "\u1104\u1165\u11B5", "\uB5AE", "\u1104\u1165\u11B5" },
            new string [] { "\uB5AF", "\uB5AF", "\u1104\u1165\u11B6", "\uB5AF", "\u1104\u1165\u11B6" },
            new string [] { "\uB5B0", "\uB5B0", "\u1104\u1165\u11B7", "\uB5B0", "\u1104\u1165\u11B7" },
            new string [] { "\uB5B1", "\uB5B1", "\u1104\u1165\u11B8", "\uB5B1", "\u1104\u1165\u11B8" },
            new string [] { "\uB5B2", "\uB5B2", "\u1104\u1165\u11B9", "\uB5B2", "\u1104\u1165\u11B9" },
            new string [] { "\uB5B3", "\uB5B3", "\u1104\u1165\u11BA", "\uB5B3", "\u1104\u1165\u11BA" },
            new string [] { "\uB5B4", "\uB5B4", "\u1104\u1165\u11BB", "\uB5B4", "\u1104\u1165\u11BB" },
            new string [] { "\uB5B5", "\uB5B5", "\u1104\u1165\u11BC", "\uB5B5", "\u1104\u1165\u11BC" },
            new string [] { "\uB5B6", "\uB5B6", "\u1104\u1165\u11BD", "\uB5B6", "\u1104\u1165\u11BD" },
            new string [] { "\uB5B7", "\uB5B7", "\u1104\u1165\u11BE", "\uB5B7", "\u1104\u1165\u11BE" },
            new string [] { "\uB5B8", "\uB5B8", "\u1104\u1165\u11BF", "\uB5B8", "\u1104\u1165\u11BF" },
            new string [] { "\uB5B9", "\uB5B9", "\u1104\u1165\u11C0", "\uB5B9", "\u1104\u1165\u11C0" },
            new string [] { "\uB5BA", "\uB5BA", "\u1104\u1165\u11C1", "\uB5BA", "\u1104\u1165\u11C1" },
            new string [] { "\uB5BB", "\uB5BB", "\u1104\u1165\u11C2", "\uB5BB", "\u1104\u1165\u11C2" },
            new string [] { "\uB5BC", "\uB5BC", "\u1104\u1166", "\uB5BC", "\u1104\u1166" },
            new string [] { "\uB5BD", "\uB5BD", "\u1104\u1166\u11A8", "\uB5BD", "\u1104\u1166\u11A8" },
            new string [] { "\uB5BE", "\uB5BE", "\u1104\u1166\u11A9", "\uB5BE", "\u1104\u1166\u11A9" },
            new string [] { "\uB5BF", "\uB5BF", "\u1104\u1166\u11AA", "\uB5BF", "\u1104\u1166\u11AA" },
            new string [] { "\uB5C0", "\uB5C0", "\u1104\u1166\u11AB", "\uB5C0", "\u1104\u1166\u11AB" },
            new string [] { "\uB5C1", "\uB5C1", "\u1104\u1166\u11AC", "\uB5C1", "\u1104\u1166\u11AC" },
            new string [] { "\uB5C2", "\uB5C2", "\u1104\u1166\u11AD", "\uB5C2", "\u1104\u1166\u11AD" },
            new string [] { "\uB5C3", "\uB5C3", "\u1104\u1166\u11AE", "\uB5C3", "\u1104\u1166\u11AE" },
            new string [] { "\uB5C4", "\uB5C4", "\u1104\u1166\u11AF", "\uB5C4", "\u1104\u1166\u11AF" },
            new string [] { "\uB5C5", "\uB5C5", "\u1104\u1166\u11B0", "\uB5C5", "\u1104\u1166\u11B0" },
            new string [] { "\uB5C6", "\uB5C6", "\u1104\u1166\u11B1", "\uB5C6", "\u1104\u1166\u11B1" },
            new string [] { "\uB5C7", "\uB5C7", "\u1104\u1166\u11B2", "\uB5C7", "\u1104\u1166\u11B2" },
            new string [] { "\uB5C8", "\uB5C8", "\u1104\u1166\u11B3", "\uB5C8", "\u1104\u1166\u11B3" },
            new string [] { "\uB5C9", "\uB5C9", "\u1104\u1166\u11B4", "\uB5C9", "\u1104\u1166\u11B4" },
            new string [] { "\uB5CA", "\uB5CA", "\u1104\u1166\u11B5", "\uB5CA", "\u1104\u1166\u11B5" },
            new string [] { "\uB5CB", "\uB5CB", "\u1104\u1166\u11B6", "\uB5CB", "\u1104\u1166\u11B6" },
            new string [] { "\uB5CC", "\uB5CC", "\u1104\u1166\u11B7", "\uB5CC", "\u1104\u1166\u11B7" },
            new string [] { "\uB5CD", "\uB5CD", "\u1104\u1166\u11B8", "\uB5CD", "\u1104\u1166\u11B8" },
            new string [] { "\uB5CE", "\uB5CE", "\u1104\u1166\u11B9", "\uB5CE", "\u1104\u1166\u11B9" },
            new string [] { "\uB5CF", "\uB5CF", "\u1104\u1166\u11BA", "\uB5CF", "\u1104\u1166\u11BA" },
            new string [] { "\uB5D0", "\uB5D0", "\u1104\u1166\u11BB", "\uB5D0", "\u1104\u1166\u11BB" },
            new string [] { "\uB5D1", "\uB5D1", "\u1104\u1166\u11BC", "\uB5D1", "\u1104\u1166\u11BC" },
            new string [] { "\uB5D2", "\uB5D2", "\u1104\u1166\u11BD", "\uB5D2", "\u1104\u1166\u11BD" },
            new string [] { "\uB5D3", "\uB5D3", "\u1104\u1166\u11BE", "\uB5D3", "\u1104\u1166\u11BE" },
            new string [] { "\uB5D4", "\uB5D4", "\u1104\u1166\u11BF", "\uB5D4", "\u1104\u1166\u11BF" },
            new string [] { "\uB5D5", "\uB5D5", "\u1104\u1166\u11C0", "\uB5D5", "\u1104\u1166\u11C0" },
            new string [] { "\uB5D6", "\uB5D6", "\u1104\u1166\u11C1", "\uB5D6", "\u1104\u1166\u11C1" },
            new string [] { "\uB5D7", "\uB5D7", "\u1104\u1166\u11C2", "\uB5D7", "\u1104\u1166\u11C2" },
            new string [] { "\uB5D8", "\uB5D8", "\u1104\u1167", "\uB5D8", "\u1104\u1167" },
            new string [] { "\uB5D9", "\uB5D9", "\u1104\u1167\u11A8", "\uB5D9", "\u1104\u1167\u11A8" },
            new string [] { "\uB5DA", "\uB5DA", "\u1104\u1167\u11A9", "\uB5DA", "\u1104\u1167\u11A9" },
            new string [] { "\uB5DB", "\uB5DB", "\u1104\u1167\u11AA", "\uB5DB", "\u1104\u1167\u11AA" },
            new string [] { "\uB5DC", "\uB5DC", "\u1104\u1167\u11AB", "\uB5DC", "\u1104\u1167\u11AB" },
            new string [] { "\uB5DD", "\uB5DD", "\u1104\u1167\u11AC", "\uB5DD", "\u1104\u1167\u11AC" },
            new string [] { "\uB5DE", "\uB5DE", "\u1104\u1167\u11AD", "\uB5DE", "\u1104\u1167\u11AD" },
            new string [] { "\uB5DF", "\uB5DF", "\u1104\u1167\u11AE", "\uB5DF", "\u1104\u1167\u11AE" },
            new string [] { "\uB5E0", "\uB5E0", "\u1104\u1167\u11AF", "\uB5E0", "\u1104\u1167\u11AF" },
            new string [] { "\uB5E1", "\uB5E1", "\u1104\u1167\u11B0", "\uB5E1", "\u1104\u1167\u11B0" },
            new string [] { "\uB5E2", "\uB5E2", "\u1104\u1167\u11B1", "\uB5E2", "\u1104\u1167\u11B1" },
            new string [] { "\uB5E3", "\uB5E3", "\u1104\u1167\u11B2", "\uB5E3", "\u1104\u1167\u11B2" },
            new string [] { "\uB5E4", "\uB5E4", "\u1104\u1167\u11B3", "\uB5E4", "\u1104\u1167\u11B3" },
            new string [] { "\uB5E5", "\uB5E5", "\u1104\u1167\u11B4", "\uB5E5", "\u1104\u1167\u11B4" },
            new string [] { "\uB5E6", "\uB5E6", "\u1104\u1167\u11B5", "\uB5E6", "\u1104\u1167\u11B5" },
            new string [] { "\uB5E7", "\uB5E7", "\u1104\u1167\u11B6", "\uB5E7", "\u1104\u1167\u11B6" },
            new string [] { "\uB5E8", "\uB5E8", "\u1104\u1167\u11B7", "\uB5E8", "\u1104\u1167\u11B7" },
            new string [] { "\uB5E9", "\uB5E9", "\u1104\u1167\u11B8", "\uB5E9", "\u1104\u1167\u11B8" },
            new string [] { "\uB5EA", "\uB5EA", "\u1104\u1167\u11B9", "\uB5EA", "\u1104\u1167\u11B9" },
            new string [] { "\uB5EB", "\uB5EB", "\u1104\u1167\u11BA", "\uB5EB", "\u1104\u1167\u11BA" },
            new string [] { "\uB5EC", "\uB5EC", "\u1104\u1167\u11BB", "\uB5EC", "\u1104\u1167\u11BB" },
            new string [] { "\uB5ED", "\uB5ED", "\u1104\u1167\u11BC", "\uB5ED", "\u1104\u1167\u11BC" },
            new string [] { "\uB5EE", "\uB5EE", "\u1104\u1167\u11BD", "\uB5EE", "\u1104\u1167\u11BD" },
            new string [] { "\uB5EF", "\uB5EF", "\u1104\u1167\u11BE", "\uB5EF", "\u1104\u1167\u11BE" },
            new string [] { "\uB5F0", "\uB5F0", "\u1104\u1167\u11BF", "\uB5F0", "\u1104\u1167\u11BF" },
            new string [] { "\uB5F1", "\uB5F1", "\u1104\u1167\u11C0", "\uB5F1", "\u1104\u1167\u11C0" },
            new string [] { "\uB5F2", "\uB5F2", "\u1104\u1167\u11C1", "\uB5F2", "\u1104\u1167\u11C1" },
            new string [] { "\uB5F3", "\uB5F3", "\u1104\u1167\u11C2", "\uB5F3", "\u1104\u1167\u11C2" },
            new string [] { "\uB5F4", "\uB5F4", "\u1104\u1168", "\uB5F4", "\u1104\u1168" },
            new string [] { "\uB5F5", "\uB5F5", "\u1104\u1168\u11A8", "\uB5F5", "\u1104\u1168\u11A8" },
            new string [] { "\uB5F6", "\uB5F6", "\u1104\u1168\u11A9", "\uB5F6", "\u1104\u1168\u11A9" },
            new string [] { "\uB5F7", "\uB5F7", "\u1104\u1168\u11AA", "\uB5F7", "\u1104\u1168\u11AA" },
            new string [] { "\uB5F8", "\uB5F8", "\u1104\u1168\u11AB", "\uB5F8", "\u1104\u1168\u11AB" },
            new string [] { "\uB5F9", "\uB5F9", "\u1104\u1168\u11AC", "\uB5F9", "\u1104\u1168\u11AC" },
            new string [] { "\uB5FA", "\uB5FA", "\u1104\u1168\u11AD", "\uB5FA", "\u1104\u1168\u11AD" },
            new string [] { "\uB5FB", "\uB5FB", "\u1104\u1168\u11AE", "\uB5FB", "\u1104\u1168\u11AE" },
            new string [] { "\uB5FC", "\uB5FC", "\u1104\u1168\u11AF", "\uB5FC", "\u1104\u1168\u11AF" },
            new string [] { "\uB5FD", "\uB5FD", "\u1104\u1168\u11B0", "\uB5FD", "\u1104\u1168\u11B0" },
            new string [] { "\uB5FE", "\uB5FE", "\u1104\u1168\u11B1", "\uB5FE", "\u1104\u1168\u11B1" },
            new string [] { "\uB5FF", "\uB5FF", "\u1104\u1168\u11B2", "\uB5FF", "\u1104\u1168\u11B2" },
            new string [] { "\uB600", "\uB600", "\u1104\u1168\u11B3", "\uB600", "\u1104\u1168\u11B3" },
            new string [] { "\uB601", "\uB601", "\u1104\u1168\u11B4", "\uB601", "\u1104\u1168\u11B4" },
            new string [] { "\uB602", "\uB602", "\u1104\u1168\u11B5", "\uB602", "\u1104\u1168\u11B5" },
            new string [] { "\uB603", "\uB603", "\u1104\u1168\u11B6", "\uB603", "\u1104\u1168\u11B6" },
            new string [] { "\uB604", "\uB604", "\u1104\u1168\u11B7", "\uB604", "\u1104\u1168\u11B7" },
            new string [] { "\uB605", "\uB605", "\u1104\u1168\u11B8", "\uB605", "\u1104\u1168\u11B8" },
            new string [] { "\uB606", "\uB606", "\u1104\u1168\u11B9", "\uB606", "\u1104\u1168\u11B9" },
            new string [] { "\uB607", "\uB607", "\u1104\u1168\u11BA", "\uB607", "\u1104\u1168\u11BA" },
            new string [] { "\uB608", "\uB608", "\u1104\u1168\u11BB", "\uB608", "\u1104\u1168\u11BB" },
            new string [] { "\uB609", "\uB609", "\u1104\u1168\u11BC", "\uB609", "\u1104\u1168\u11BC" },
            new string [] { "\uB60A", "\uB60A", "\u1104\u1168\u11BD", "\uB60A", "\u1104\u1168\u11BD" },
            new string [] { "\uB60B", "\uB60B", "\u1104\u1168\u11BE", "\uB60B", "\u1104\u1168\u11BE" },
            new string [] { "\uB60C", "\uB60C", "\u1104\u1168\u11BF", "\uB60C", "\u1104\u1168\u11BF" },
            new string [] { "\uB60D", "\uB60D", "\u1104\u1168\u11C0", "\uB60D", "\u1104\u1168\u11C0" },
            new string [] { "\uB60E", "\uB60E", "\u1104\u1168\u11C1", "\uB60E", "\u1104\u1168\u11C1" },
            new string [] { "\uB60F", "\uB60F", "\u1104\u1168\u11C2", "\uB60F", "\u1104\u1168\u11C2" },
            new string [] { "\uB610", "\uB610", "\u1104\u1169", "\uB610", "\u1104\u1169" },
            new string [] { "\uB611", "\uB611", "\u1104\u1169\u11A8", "\uB611", "\u1104\u1169\u11A8" },
            new string [] { "\uB612", "\uB612", "\u1104\u1169\u11A9", "\uB612", "\u1104\u1169\u11A9" },
            new string [] { "\uB613", "\uB613", "\u1104\u1169\u11AA", "\uB613", "\u1104\u1169\u11AA" },
            new string [] { "\uB614", "\uB614", "\u1104\u1169\u11AB", "\uB614", "\u1104\u1169\u11AB" },
            new string [] { "\uB615", "\uB615", "\u1104\u1169\u11AC", "\uB615", "\u1104\u1169\u11AC" },
            new string [] { "\uB616", "\uB616", "\u1104\u1169\u11AD", "\uB616", "\u1104\u1169\u11AD" },
            new string [] { "\uB617", "\uB617", "\u1104\u1169\u11AE", "\uB617", "\u1104\u1169\u11AE" },
            new string [] { "\uB618", "\uB618", "\u1104\u1169\u11AF", "\uB618", "\u1104\u1169\u11AF" },
            new string [] { "\uB619", "\uB619", "\u1104\u1169\u11B0", "\uB619", "\u1104\u1169\u11B0" },
            new string [] { "\uB61A", "\uB61A", "\u1104\u1169\u11B1", "\uB61A", "\u1104\u1169\u11B1" },
            new string [] { "\uB61B", "\uB61B", "\u1104\u1169\u11B2", "\uB61B", "\u1104\u1169\u11B2" },
            new string [] { "\uB61C", "\uB61C", "\u1104\u1169\u11B3", "\uB61C", "\u1104\u1169\u11B3" },
            new string [] { "\uB61D", "\uB61D", "\u1104\u1169\u11B4", "\uB61D", "\u1104\u1169\u11B4" },
            new string [] { "\uB61E", "\uB61E", "\u1104\u1169\u11B5", "\uB61E", "\u1104\u1169\u11B5" },
            new string [] { "\uB61F", "\uB61F", "\u1104\u1169\u11B6", "\uB61F", "\u1104\u1169\u11B6" },
            new string [] { "\uB620", "\uB620", "\u1104\u1169\u11B7", "\uB620", "\u1104\u1169\u11B7" },
            new string [] { "\uB621", "\uB621", "\u1104\u1169\u11B8", "\uB621", "\u1104\u1169\u11B8" },
            new string [] { "\uB622", "\uB622", "\u1104\u1169\u11B9", "\uB622", "\u1104\u1169\u11B9" },
            new string [] { "\uB623", "\uB623", "\u1104\u1169\u11BA", "\uB623", "\u1104\u1169\u11BA" },
            new string [] { "\uB624", "\uB624", "\u1104\u1169\u11BB", "\uB624", "\u1104\u1169\u11BB" },
            new string [] { "\uB625", "\uB625", "\u1104\u1169\u11BC", "\uB625", "\u1104\u1169\u11BC" },
            new string [] { "\uB626", "\uB626", "\u1104\u1169\u11BD", "\uB626", "\u1104\u1169\u11BD" },
            new string [] { "\uB627", "\uB627", "\u1104\u1169\u11BE", "\uB627", "\u1104\u1169\u11BE" },
            new string [] { "\uB628", "\uB628", "\u1104\u1169\u11BF", "\uB628", "\u1104\u1169\u11BF" },
            new string [] { "\uB629", "\uB629", "\u1104\u1169\u11C0", "\uB629", "\u1104\u1169\u11C0" },
            new string [] { "\uB62A", "\uB62A", "\u1104\u1169\u11C1", "\uB62A", "\u1104\u1169\u11C1" },
            new string [] { "\uB62B", "\uB62B", "\u1104\u1169\u11C2", "\uB62B", "\u1104\u1169\u11C2" },
            new string [] { "\uB62C", "\uB62C", "\u1104\u116A", "\uB62C", "\u1104\u116A" },
            new string [] { "\uB62D", "\uB62D", "\u1104\u116A\u11A8", "\uB62D", "\u1104\u116A\u11A8" },
            new string [] { "\uB62E", "\uB62E", "\u1104\u116A\u11A9", "\uB62E", "\u1104\u116A\u11A9" },
            new string [] { "\uB62F", "\uB62F", "\u1104\u116A\u11AA", "\uB62F", "\u1104\u116A\u11AA" },
            new string [] { "\uB630", "\uB630", "\u1104\u116A\u11AB", "\uB630", "\u1104\u116A\u11AB" },
            new string [] { "\uB631", "\uB631", "\u1104\u116A\u11AC", "\uB631", "\u1104\u116A\u11AC" },
            new string [] { "\uB632", "\uB632", "\u1104\u116A\u11AD", "\uB632", "\u1104\u116A\u11AD" },
            new string [] { "\uB633", "\uB633", "\u1104\u116A\u11AE", "\uB633", "\u1104\u116A\u11AE" },
            new string [] { "\uB634", "\uB634", "\u1104\u116A\u11AF", "\uB634", "\u1104\u116A\u11AF" },
            new string [] { "\uB635", "\uB635", "\u1104\u116A\u11B0", "\uB635", "\u1104\u116A\u11B0" },
            new string [] { "\uB636", "\uB636", "\u1104\u116A\u11B1", "\uB636", "\u1104\u116A\u11B1" },
            new string [] { "\uB637", "\uB637", "\u1104\u116A\u11B2", "\uB637", "\u1104\u116A\u11B2" },
            new string [] { "\uB638", "\uB638", "\u1104\u116A\u11B3", "\uB638", "\u1104\u116A\u11B3" },
            new string [] { "\uB639", "\uB639", "\u1104\u116A\u11B4", "\uB639", "\u1104\u116A\u11B4" },
            new string [] { "\uB63A", "\uB63A", "\u1104\u116A\u11B5", "\uB63A", "\u1104\u116A\u11B5" },
            new string [] { "\uB63B", "\uB63B", "\u1104\u116A\u11B6", "\uB63B", "\u1104\u116A\u11B6" },
            new string [] { "\uB63C", "\uB63C", "\u1104\u116A\u11B7", "\uB63C", "\u1104\u116A\u11B7" },
            new string [] { "\uB63D", "\uB63D", "\u1104\u116A\u11B8", "\uB63D", "\u1104\u116A\u11B8" },
            new string [] { "\uB63E", "\uB63E", "\u1104\u116A\u11B9", "\uB63E", "\u1104\u116A\u11B9" },
            new string [] { "\uB63F", "\uB63F", "\u1104\u116A\u11BA", "\uB63F", "\u1104\u116A\u11BA" },
            new string [] { "\uB640", "\uB640", "\u1104\u116A\u11BB", "\uB640", "\u1104\u116A\u11BB" },
            new string [] { "\uB641", "\uB641", "\u1104\u116A\u11BC", "\uB641", "\u1104\u116A\u11BC" },
            new string [] { "\uB642", "\uB642", "\u1104\u116A\u11BD", "\uB642", "\u1104\u116A\u11BD" },
            new string [] { "\uB643", "\uB643", "\u1104\u116A\u11BE", "\uB643", "\u1104\u116A\u11BE" },
            new string [] { "\uB644", "\uB644", "\u1104\u116A\u11BF", "\uB644", "\u1104\u116A\u11BF" },
            new string [] { "\uB645", "\uB645", "\u1104\u116A\u11C0", "\uB645", "\u1104\u116A\u11C0" },
            new string [] { "\uB646", "\uB646", "\u1104\u116A\u11C1", "\uB646", "\u1104\u116A\u11C1" },
            new string [] { "\uB647", "\uB647", "\u1104\u116A\u11C2", "\uB647", "\u1104\u116A\u11C2" },
            new string [] { "\uB648", "\uB648", "\u1104\u116B", "\uB648", "\u1104\u116B" },
            new string [] { "\uB649", "\uB649", "\u1104\u116B\u11A8", "\uB649", "\u1104\u116B\u11A8" },
            new string [] { "\uB64A", "\uB64A", "\u1104\u116B\u11A9", "\uB64A", "\u1104\u116B\u11A9" },
            new string [] { "\uB64B", "\uB64B", "\u1104\u116B\u11AA", "\uB64B", "\u1104\u116B\u11AA" },
            new string [] { "\uB64C", "\uB64C", "\u1104\u116B\u11AB", "\uB64C", "\u1104\u116B\u11AB" },
            new string [] { "\uB64D", "\uB64D", "\u1104\u116B\u11AC", "\uB64D", "\u1104\u116B\u11AC" },
            new string [] { "\uB64E", "\uB64E", "\u1104\u116B\u11AD", "\uB64E", "\u1104\u116B\u11AD" },
            new string [] { "\uB64F", "\uB64F", "\u1104\u116B\u11AE", "\uB64F", "\u1104\u116B\u11AE" },
            new string [] { "\uB650", "\uB650", "\u1104\u116B\u11AF", "\uB650", "\u1104\u116B\u11AF" },
            new string [] { "\uB651", "\uB651", "\u1104\u116B\u11B0", "\uB651", "\u1104\u116B\u11B0" },
            new string [] { "\uB652", "\uB652", "\u1104\u116B\u11B1", "\uB652", "\u1104\u116B\u11B1" },
            new string [] { "\uB653", "\uB653", "\u1104\u116B\u11B2", "\uB653", "\u1104\u116B\u11B2" },
            new string [] { "\uB654", "\uB654", "\u1104\u116B\u11B3", "\uB654", "\u1104\u116B\u11B3" },
            new string [] { "\uB655", "\uB655", "\u1104\u116B\u11B4", "\uB655", "\u1104\u116B\u11B4" },
            new string [] { "\uB656", "\uB656", "\u1104\u116B\u11B5", "\uB656", "\u1104\u116B\u11B5" },
            new string [] { "\uB657", "\uB657", "\u1104\u116B\u11B6", "\uB657", "\u1104\u116B\u11B6" },
            new string [] { "\uB658", "\uB658", "\u1104\u116B\u11B7", "\uB658", "\u1104\u116B\u11B7" },
            new string [] { "\uB659", "\uB659", "\u1104\u116B\u11B8", "\uB659", "\u1104\u116B\u11B8" },
            new string [] { "\uB65A", "\uB65A", "\u1104\u116B\u11B9", "\uB65A", "\u1104\u116B\u11B9" },
            new string [] { "\uB65B", "\uB65B", "\u1104\u116B\u11BA", "\uB65B", "\u1104\u116B\u11BA" },
            new string [] { "\uB65C", "\uB65C", "\u1104\u116B\u11BB", "\uB65C", "\u1104\u116B\u11BB" },
            new string [] { "\uB65D", "\uB65D", "\u1104\u116B\u11BC", "\uB65D", "\u1104\u116B\u11BC" },
            new string [] { "\uB65E", "\uB65E", "\u1104\u116B\u11BD", "\uB65E", "\u1104\u116B\u11BD" },
            new string [] { "\uB65F", "\uB65F", "\u1104\u116B\u11BE", "\uB65F", "\u1104\u116B\u11BE" },
            new string [] { "\uB660", "\uB660", "\u1104\u116B\u11BF", "\uB660", "\u1104\u116B\u11BF" },
            new string [] { "\uB661", "\uB661", "\u1104\u116B\u11C0", "\uB661", "\u1104\u116B\u11C0" },
            new string [] { "\uB662", "\uB662", "\u1104\u116B\u11C1", "\uB662", "\u1104\u116B\u11C1" },
            new string [] { "\uB663", "\uB663", "\u1104\u116B\u11C2", "\uB663", "\u1104\u116B\u11C2" },
            new string [] { "\uB664", "\uB664", "\u1104\u116C", "\uB664", "\u1104\u116C" },
            new string [] { "\uB665", "\uB665", "\u1104\u116C\u11A8", "\uB665", "\u1104\u116C\u11A8" },
            new string [] { "\uB666", "\uB666", "\u1104\u116C\u11A9", "\uB666", "\u1104\u116C\u11A9" },
            new string [] { "\uB667", "\uB667", "\u1104\u116C\u11AA", "\uB667", "\u1104\u116C\u11AA" },
            new string [] { "\uB668", "\uB668", "\u1104\u116C\u11AB", "\uB668", "\u1104\u116C\u11AB" },
            new string [] { "\uB669", "\uB669", "\u1104\u116C\u11AC", "\uB669", "\u1104\u116C\u11AC" },
            new string [] { "\uB66A", "\uB66A", "\u1104\u116C\u11AD", "\uB66A", "\u1104\u116C\u11AD" },
            new string [] { "\uB66B", "\uB66B", "\u1104\u116C\u11AE", "\uB66B", "\u1104\u116C\u11AE" },
            new string [] { "\uB66C", "\uB66C", "\u1104\u116C\u11AF", "\uB66C", "\u1104\u116C\u11AF" },
            new string [] { "\uB66D", "\uB66D", "\u1104\u116C\u11B0", "\uB66D", "\u1104\u116C\u11B0" },
            new string [] { "\uB66E", "\uB66E", "\u1104\u116C\u11B1", "\uB66E", "\u1104\u116C\u11B1" },
            new string [] { "\uB66F", "\uB66F", "\u1104\u116C\u11B2", "\uB66F", "\u1104\u116C\u11B2" },
            new string [] { "\uB670", "\uB670", "\u1104\u116C\u11B3", "\uB670", "\u1104\u116C\u11B3" },
            new string [] { "\uB671", "\uB671", "\u1104\u116C\u11B4", "\uB671", "\u1104\u116C\u11B4" },
            new string [] { "\uB672", "\uB672", "\u1104\u116C\u11B5", "\uB672", "\u1104\u116C\u11B5" },
            new string [] { "\uB673", "\uB673", "\u1104\u116C\u11B6", "\uB673", "\u1104\u116C\u11B6" },
            new string [] { "\uB674", "\uB674", "\u1104\u116C\u11B7", "\uB674", "\u1104\u116C\u11B7" },
            new string [] { "\uB675", "\uB675", "\u1104\u116C\u11B8", "\uB675", "\u1104\u116C\u11B8" },
            new string [] { "\uB676", "\uB676", "\u1104\u116C\u11B9", "\uB676", "\u1104\u116C\u11B9" },
            new string [] { "\uB677", "\uB677", "\u1104\u116C\u11BA", "\uB677", "\u1104\u116C\u11BA" },
            new string [] { "\uB678", "\uB678", "\u1104\u116C\u11BB", "\uB678", "\u1104\u116C\u11BB" },
            new string [] { "\uB679", "\uB679", "\u1104\u116C\u11BC", "\uB679", "\u1104\u116C\u11BC" },
            new string [] { "\uB67A", "\uB67A", "\u1104\u116C\u11BD", "\uB67A", "\u1104\u116C\u11BD" },
            new string [] { "\uB67B", "\uB67B", "\u1104\u116C\u11BE", "\uB67B", "\u1104\u116C\u11BE" },
            new string [] { "\uB67C", "\uB67C", "\u1104\u116C\u11BF", "\uB67C", "\u1104\u116C\u11BF" },
            new string [] { "\uB67D", "\uB67D", "\u1104\u116C\u11C0", "\uB67D", "\u1104\u116C\u11C0" },
            new string [] { "\uB67E", "\uB67E", "\u1104\u116C\u11C1", "\uB67E", "\u1104\u116C\u11C1" },
            new string [] { "\uB67F", "\uB67F", "\u1104\u116C\u11C2", "\uB67F", "\u1104\u116C\u11C2" },
            new string [] { "\uB680", "\uB680", "\u1104\u116D", "\uB680", "\u1104\u116D" },
            new string [] { "\uB681", "\uB681", "\u1104\u116D\u11A8", "\uB681", "\u1104\u116D\u11A8" },
            new string [] { "\uB682", "\uB682", "\u1104\u116D\u11A9", "\uB682", "\u1104\u116D\u11A9" },
            new string [] { "\uB683", "\uB683", "\u1104\u116D\u11AA", "\uB683", "\u1104\u116D\u11AA" },
            new string [] { "\uB684", "\uB684", "\u1104\u116D\u11AB", "\uB684", "\u1104\u116D\u11AB" },
            new string [] { "\uB685", "\uB685", "\u1104\u116D\u11AC", "\uB685", "\u1104\u116D\u11AC" },
            new string [] { "\uB686", "\uB686", "\u1104\u116D\u11AD", "\uB686", "\u1104\u116D\u11AD" },
            new string [] { "\uB687", "\uB687", "\u1104\u116D\u11AE", "\uB687", "\u1104\u116D\u11AE" },
            new string [] { "\uB688", "\uB688", "\u1104\u116D\u11AF", "\uB688", "\u1104\u116D\u11AF" },
            new string [] { "\uB689", "\uB689", "\u1104\u116D\u11B0", "\uB689", "\u1104\u116D\u11B0" },
            new string [] { "\uB68A", "\uB68A", "\u1104\u116D\u11B1", "\uB68A", "\u1104\u116D\u11B1" },
            new string [] { "\uB68B", "\uB68B", "\u1104\u116D\u11B2", "\uB68B", "\u1104\u116D\u11B2" },
            new string [] { "\uB68C", "\uB68C", "\u1104\u116D\u11B3", "\uB68C", "\u1104\u116D\u11B3" },
            new string [] { "\uB68D", "\uB68D", "\u1104\u116D\u11B4", "\uB68D", "\u1104\u116D\u11B4" },
            new string [] { "\uB68E", "\uB68E", "\u1104\u116D\u11B5", "\uB68E", "\u1104\u116D\u11B5" },
            new string [] { "\uB68F", "\uB68F", "\u1104\u116D\u11B6", "\uB68F", "\u1104\u116D\u11B6" },
            new string [] { "\uB690", "\uB690", "\u1104\u116D\u11B7", "\uB690", "\u1104\u116D\u11B7" },
            new string [] { "\uB691", "\uB691", "\u1104\u116D\u11B8", "\uB691", "\u1104\u116D\u11B8" },
            new string [] { "\uB692", "\uB692", "\u1104\u116D\u11B9", "\uB692", "\u1104\u116D\u11B9" },
            new string [] { "\uB693", "\uB693", "\u1104\u116D\u11BA", "\uB693", "\u1104\u116D\u11BA" },
            new string [] { "\uB694", "\uB694", "\u1104\u116D\u11BB", "\uB694", "\u1104\u116D\u11BB" },
            new string [] { "\uB695", "\uB695", "\u1104\u116D\u11BC", "\uB695", "\u1104\u116D\u11BC" },
            new string [] { "\uB696", "\uB696", "\u1104\u116D\u11BD", "\uB696", "\u1104\u116D\u11BD" },
            new string [] { "\uB697", "\uB697", "\u1104\u116D\u11BE", "\uB697", "\u1104\u116D\u11BE" },
            new string [] { "\uB698", "\uB698", "\u1104\u116D\u11BF", "\uB698", "\u1104\u116D\u11BF" },
            new string [] { "\uB699", "\uB699", "\u1104\u116D\u11C0", "\uB699", "\u1104\u116D\u11C0" },
            new string [] { "\uB69A", "\uB69A", "\u1104\u116D\u11C1", "\uB69A", "\u1104\u116D\u11C1" },
            new string [] { "\uB69B", "\uB69B", "\u1104\u116D\u11C2", "\uB69B", "\u1104\u116D\u11C2" },
            new string [] { "\uB69C", "\uB69C", "\u1104\u116E", "\uB69C", "\u1104\u116E" },
            new string [] { "\uB69D", "\uB69D", "\u1104\u116E\u11A8", "\uB69D", "\u1104\u116E\u11A8" },
            new string [] { "\uB69E", "\uB69E", "\u1104\u116E\u11A9", "\uB69E", "\u1104\u116E\u11A9" },
            new string [] { "\uB69F", "\uB69F", "\u1104\u116E\u11AA", "\uB69F", "\u1104\u116E\u11AA" },
            new string [] { "\uB6A0", "\uB6A0", "\u1104\u116E\u11AB", "\uB6A0", "\u1104\u116E\u11AB" },
            new string [] { "\uB6A1", "\uB6A1", "\u1104\u116E\u11AC", "\uB6A1", "\u1104\u116E\u11AC" },
            new string [] { "\uB6A2", "\uB6A2", "\u1104\u116E\u11AD", "\uB6A2", "\u1104\u116E\u11AD" },
            new string [] { "\uB6A3", "\uB6A3", "\u1104\u116E\u11AE", "\uB6A3", "\u1104\u116E\u11AE" },
            new string [] { "\uB6A4", "\uB6A4", "\u1104\u116E\u11AF", "\uB6A4", "\u1104\u116E\u11AF" },
            new string [] { "\uB6A5", "\uB6A5", "\u1104\u116E\u11B0", "\uB6A5", "\u1104\u116E\u11B0" },
            new string [] { "\uB6A6", "\uB6A6", "\u1104\u116E\u11B1", "\uB6A6", "\u1104\u116E\u11B1" },
            new string [] { "\uB6A7", "\uB6A7", "\u1104\u116E\u11B2", "\uB6A7", "\u1104\u116E\u11B2" },
            new string [] { "\uB6A8", "\uB6A8", "\u1104\u116E\u11B3", "\uB6A8", "\u1104\u116E\u11B3" },
            new string [] { "\uB6A9", "\uB6A9", "\u1104\u116E\u11B4", "\uB6A9", "\u1104\u116E\u11B4" },
            new string [] { "\uB6AA", "\uB6AA", "\u1104\u116E\u11B5", "\uB6AA", "\u1104\u116E\u11B5" },
            new string [] { "\uB6AB", "\uB6AB", "\u1104\u116E\u11B6", "\uB6AB", "\u1104\u116E\u11B6" },
            new string [] { "\uB6AC", "\uB6AC", "\u1104\u116E\u11B7", "\uB6AC", "\u1104\u116E\u11B7" },
            new string [] { "\uB6AD", "\uB6AD", "\u1104\u116E\u11B8", "\uB6AD", "\u1104\u116E\u11B8" },
            new string [] { "\uB6AE", "\uB6AE", "\u1104\u116E\u11B9", "\uB6AE", "\u1104\u116E\u11B9" },
            new string [] { "\uB6AF", "\uB6AF", "\u1104\u116E\u11BA", "\uB6AF", "\u1104\u116E\u11BA" },
            new string [] { "\uB6B0", "\uB6B0", "\u1104\u116E\u11BB", "\uB6B0", "\u1104\u116E\u11BB" },
            new string [] { "\uB6B1", "\uB6B1", "\u1104\u116E\u11BC", "\uB6B1", "\u1104\u116E\u11BC" },
            new string [] { "\uB6B2", "\uB6B2", "\u1104\u116E\u11BD", "\uB6B2", "\u1104\u116E\u11BD" },
            new string [] { "\uB6B3", "\uB6B3", "\u1104\u116E\u11BE", "\uB6B3", "\u1104\u116E\u11BE" },
            new string [] { "\uB6B4", "\uB6B4", "\u1104\u116E\u11BF", "\uB6B4", "\u1104\u116E\u11BF" },
            new string [] { "\uB6B5", "\uB6B5", "\u1104\u116E\u11C0", "\uB6B5", "\u1104\u116E\u11C0" },
            new string [] { "\uB6B6", "\uB6B6", "\u1104\u116E\u11C1", "\uB6B6", "\u1104\u116E\u11C1" },
            new string [] { "\uB6B7", "\uB6B7", "\u1104\u116E\u11C2", "\uB6B7", "\u1104\u116E\u11C2" },
            new string [] { "\uB6B8", "\uB6B8", "\u1104\u116F", "\uB6B8", "\u1104\u116F" },
            new string [] { "\uB6B9", "\uB6B9", "\u1104\u116F\u11A8", "\uB6B9", "\u1104\u116F\u11A8" },
            new string [] { "\uB6BA", "\uB6BA", "\u1104\u116F\u11A9", "\uB6BA", "\u1104\u116F\u11A9" },
            new string [] { "\uB6BB", "\uB6BB", "\u1104\u116F\u11AA", "\uB6BB", "\u1104\u116F\u11AA" },
            new string [] { "\uB6BC", "\uB6BC", "\u1104\u116F\u11AB", "\uB6BC", "\u1104\u116F\u11AB" },
            new string [] { "\uB6BD", "\uB6BD", "\u1104\u116F\u11AC", "\uB6BD", "\u1104\u116F\u11AC" },
            new string [] { "\uB6BE", "\uB6BE", "\u1104\u116F\u11AD", "\uB6BE", "\u1104\u116F\u11AD" },
            new string [] { "\uB6BF", "\uB6BF", "\u1104\u116F\u11AE", "\uB6BF", "\u1104\u116F\u11AE" },
            new string [] { "\uB6C0", "\uB6C0", "\u1104\u116F\u11AF", "\uB6C0", "\u1104\u116F\u11AF" },
            new string [] { "\uB6C1", "\uB6C1", "\u1104\u116F\u11B0", "\uB6C1", "\u1104\u116F\u11B0" },
            new string [] { "\uB6C2", "\uB6C2", "\u1104\u116F\u11B1", "\uB6C2", "\u1104\u116F\u11B1" },
            new string [] { "\uB6C3", "\uB6C3", "\u1104\u116F\u11B2", "\uB6C3", "\u1104\u116F\u11B2" },
            new string [] { "\uB6C4", "\uB6C4", "\u1104\u116F\u11B3", "\uB6C4", "\u1104\u116F\u11B3" },
            new string [] { "\uB6C5", "\uB6C5", "\u1104\u116F\u11B4", "\uB6C5", "\u1104\u116F\u11B4" },
            new string [] { "\uB6C6", "\uB6C6", "\u1104\u116F\u11B5", "\uB6C6", "\u1104\u116F\u11B5" },
            new string [] { "\uB6C7", "\uB6C7", "\u1104\u116F\u11B6", "\uB6C7", "\u1104\u116F\u11B6" },
            new string [] { "\uB6C8", "\uB6C8", "\u1104\u116F\u11B7", "\uB6C8", "\u1104\u116F\u11B7" },
            new string [] { "\uB6C9", "\uB6C9", "\u1104\u116F\u11B8", "\uB6C9", "\u1104\u116F\u11B8" },
            new string [] { "\uB6CA", "\uB6CA", "\u1104\u116F\u11B9", "\uB6CA", "\u1104\u116F\u11B9" },
            new string [] { "\uB6CB", "\uB6CB", "\u1104\u116F\u11BA", "\uB6CB", "\u1104\u116F\u11BA" },
            new string [] { "\uB6CC", "\uB6CC", "\u1104\u116F\u11BB", "\uB6CC", "\u1104\u116F\u11BB" },
            new string [] { "\uB6CD", "\uB6CD", "\u1104\u116F\u11BC", "\uB6CD", "\u1104\u116F\u11BC" },
            new string [] { "\uB6CE", "\uB6CE", "\u1104\u116F\u11BD", "\uB6CE", "\u1104\u116F\u11BD" },
            new string [] { "\uB6CF", "\uB6CF", "\u1104\u116F\u11BE", "\uB6CF", "\u1104\u116F\u11BE" },
            new string [] { "\uB6D0", "\uB6D0", "\u1104\u116F\u11BF", "\uB6D0", "\u1104\u116F\u11BF" },
            new string [] { "\uB6D1", "\uB6D1", "\u1104\u116F\u11C0", "\uB6D1", "\u1104\u116F\u11C0" },
            new string [] { "\uB6D2", "\uB6D2", "\u1104\u116F\u11C1", "\uB6D2", "\u1104\u116F\u11C1" },
            new string [] { "\uB6D3", "\uB6D3", "\u1104\u116F\u11C2", "\uB6D3", "\u1104\u116F\u11C2" },
            new string [] { "\uB6D4", "\uB6D4", "\u1104\u1170", "\uB6D4", "\u1104\u1170" },
            new string [] { "\uB6D5", "\uB6D5", "\u1104\u1170\u11A8", "\uB6D5", "\u1104\u1170\u11A8" },
            new string [] { "\uB6D6", "\uB6D6", "\u1104\u1170\u11A9", "\uB6D6", "\u1104\u1170\u11A9" },
            new string [] { "\uB6D7", "\uB6D7", "\u1104\u1170\u11AA", "\uB6D7", "\u1104\u1170\u11AA" },
            new string [] { "\uB6D8", "\uB6D8", "\u1104\u1170\u11AB", "\uB6D8", "\u1104\u1170\u11AB" },
            new string [] { "\uB6D9", "\uB6D9", "\u1104\u1170\u11AC", "\uB6D9", "\u1104\u1170\u11AC" },
            new string [] { "\uB6DA", "\uB6DA", "\u1104\u1170\u11AD", "\uB6DA", "\u1104\u1170\u11AD" },
            new string [] { "\uB6DB", "\uB6DB", "\u1104\u1170\u11AE", "\uB6DB", "\u1104\u1170\u11AE" },
            new string [] { "\uB6DC", "\uB6DC", "\u1104\u1170\u11AF", "\uB6DC", "\u1104\u1170\u11AF" },
            new string [] { "\uB6DD", "\uB6DD", "\u1104\u1170\u11B0", "\uB6DD", "\u1104\u1170\u11B0" },
            new string [] { "\uB6DE", "\uB6DE", "\u1104\u1170\u11B1", "\uB6DE", "\u1104\u1170\u11B1" },
            new string [] { "\uB6DF", "\uB6DF", "\u1104\u1170\u11B2", "\uB6DF", "\u1104\u1170\u11B2" },
            new string [] { "\uB6E0", "\uB6E0", "\u1104\u1170\u11B3", "\uB6E0", "\u1104\u1170\u11B3" },
            new string [] { "\uB6E1", "\uB6E1", "\u1104\u1170\u11B4", "\uB6E1", "\u1104\u1170\u11B4" },
            new string [] { "\uB6E2", "\uB6E2", "\u1104\u1170\u11B5", "\uB6E2", "\u1104\u1170\u11B5" },
            new string [] { "\uB6E3", "\uB6E3", "\u1104\u1170\u11B6", "\uB6E3", "\u1104\u1170\u11B6" },
            new string [] { "\uB6E4", "\uB6E4", "\u1104\u1170\u11B7", "\uB6E4", "\u1104\u1170\u11B7" },
            new string [] { "\uB6E5", "\uB6E5", "\u1104\u1170\u11B8", "\uB6E5", "\u1104\u1170\u11B8" },
            new string [] { "\uB6E6", "\uB6E6", "\u1104\u1170\u11B9", "\uB6E6", "\u1104\u1170\u11B9" },
            new string [] { "\uB6E7", "\uB6E7", "\u1104\u1170\u11BA", "\uB6E7", "\u1104\u1170\u11BA" },
            new string [] { "\uB6E8", "\uB6E8", "\u1104\u1170\u11BB", "\uB6E8", "\u1104\u1170\u11BB" },
            new string [] { "\uB6E9", "\uB6E9", "\u1104\u1170\u11BC", "\uB6E9", "\u1104\u1170\u11BC" },
            new string [] { "\uB6EA", "\uB6EA", "\u1104\u1170\u11BD", "\uB6EA", "\u1104\u1170\u11BD" },
            new string [] { "\uB6EB", "\uB6EB", "\u1104\u1170\u11BE", "\uB6EB", "\u1104\u1170\u11BE" },
            new string [] { "\uB6EC", "\uB6EC", "\u1104\u1170\u11BF", "\uB6EC", "\u1104\u1170\u11BF" },
            new string [] { "\uB6ED", "\uB6ED", "\u1104\u1170\u11C0", "\uB6ED", "\u1104\u1170\u11C0" },
            new string [] { "\uB6EE", "\uB6EE", "\u1104\u1170\u11C1", "\uB6EE", "\u1104\u1170\u11C1" },
            new string [] { "\uB6EF", "\uB6EF", "\u1104\u1170\u11C2", "\uB6EF", "\u1104\u1170\u11C2" },
            new string [] { "\uB6F0", "\uB6F0", "\u1104\u1171", "\uB6F0", "\u1104\u1171" },
            new string [] { "\uB6F1", "\uB6F1", "\u1104\u1171\u11A8", "\uB6F1", "\u1104\u1171\u11A8" },
            new string [] { "\uB6F2", "\uB6F2", "\u1104\u1171\u11A9", "\uB6F2", "\u1104\u1171\u11A9" },
            new string [] { "\uB6F3", "\uB6F3", "\u1104\u1171\u11AA", "\uB6F3", "\u1104\u1171\u11AA" },
            new string [] { "\uB6F4", "\uB6F4", "\u1104\u1171\u11AB", "\uB6F4", "\u1104\u1171\u11AB" },
            new string [] { "\uB6F5", "\uB6F5", "\u1104\u1171\u11AC", "\uB6F5", "\u1104\u1171\u11AC" },
            new string [] { "\uB6F6", "\uB6F6", "\u1104\u1171\u11AD", "\uB6F6", "\u1104\u1171\u11AD" },
            new string [] { "\uB6F7", "\uB6F7", "\u1104\u1171\u11AE", "\uB6F7", "\u1104\u1171\u11AE" },
            new string [] { "\uB6F8", "\uB6F8", "\u1104\u1171\u11AF", "\uB6F8", "\u1104\u1171\u11AF" },
            new string [] { "\uB6F9", "\uB6F9", "\u1104\u1171\u11B0", "\uB6F9", "\u1104\u1171\u11B0" },
            new string [] { "\uB6FA", "\uB6FA", "\u1104\u1171\u11B1", "\uB6FA", "\u1104\u1171\u11B1" },
            new string [] { "\uB6FB", "\uB6FB", "\u1104\u1171\u11B2", "\uB6FB", "\u1104\u1171\u11B2" },
            new string [] { "\uB6FC", "\uB6FC", "\u1104\u1171\u11B3", "\uB6FC", "\u1104\u1171\u11B3" },
            new string [] { "\uB6FD", "\uB6FD", "\u1104\u1171\u11B4", "\uB6FD", "\u1104\u1171\u11B4" },
            new string [] { "\uB6FE", "\uB6FE", "\u1104\u1171\u11B5", "\uB6FE", "\u1104\u1171\u11B5" },
            new string [] { "\uB6FF", "\uB6FF", "\u1104\u1171\u11B6", "\uB6FF", "\u1104\u1171\u11B6" },
            new string [] { "\uB700", "\uB700", "\u1104\u1171\u11B7", "\uB700", "\u1104\u1171\u11B7" },
            new string [] { "\uB701", "\uB701", "\u1104\u1171\u11B8", "\uB701", "\u1104\u1171\u11B8" },
            new string [] { "\uB702", "\uB702", "\u1104\u1171\u11B9", "\uB702", "\u1104\u1171\u11B9" },
            new string [] { "\uB703", "\uB703", "\u1104\u1171\u11BA", "\uB703", "\u1104\u1171\u11BA" },
            new string [] { "\uB704", "\uB704", "\u1104\u1171\u11BB", "\uB704", "\u1104\u1171\u11BB" },
            new string [] { "\uB705", "\uB705", "\u1104\u1171\u11BC", "\uB705", "\u1104\u1171\u11BC" },
            new string [] { "\uB706", "\uB706", "\u1104\u1171\u11BD", "\uB706", "\u1104\u1171\u11BD" },
            new string [] { "\uB707", "\uB707", "\u1104\u1171\u11BE", "\uB707", "\u1104\u1171\u11BE" },
            new string [] { "\uB708", "\uB708", "\u1104\u1171\u11BF", "\uB708", "\u1104\u1171\u11BF" },
            new string [] { "\uB709", "\uB709", "\u1104\u1171\u11C0", "\uB709", "\u1104\u1171\u11C0" },
            new string [] { "\uB70A", "\uB70A", "\u1104\u1171\u11C1", "\uB70A", "\u1104\u1171\u11C1" },
            new string [] { "\uB70B", "\uB70B", "\u1104\u1171\u11C2", "\uB70B", "\u1104\u1171\u11C2" },
            new string [] { "\uB70C", "\uB70C", "\u1104\u1172", "\uB70C", "\u1104\u1172" },
            new string [] { "\uB70D", "\uB70D", "\u1104\u1172\u11A8", "\uB70D", "\u1104\u1172\u11A8" },
            new string [] { "\uB70E", "\uB70E", "\u1104\u1172\u11A9", "\uB70E", "\u1104\u1172\u11A9" },
            new string [] { "\uB70F", "\uB70F", "\u1104\u1172\u11AA", "\uB70F", "\u1104\u1172\u11AA" },
            new string [] { "\uB710", "\uB710", "\u1104\u1172\u11AB", "\uB710", "\u1104\u1172\u11AB" },
            new string [] { "\uB711", "\uB711", "\u1104\u1172\u11AC", "\uB711", "\u1104\u1172\u11AC" },
            new string [] { "\uB712", "\uB712", "\u1104\u1172\u11AD", "\uB712", "\u1104\u1172\u11AD" },
            new string [] { "\uB713", "\uB713", "\u1104\u1172\u11AE", "\uB713", "\u1104\u1172\u11AE" },
            new string [] { "\uB714", "\uB714", "\u1104\u1172\u11AF", "\uB714", "\u1104\u1172\u11AF" },
            new string [] { "\uB715", "\uB715", "\u1104\u1172\u11B0", "\uB715", "\u1104\u1172\u11B0" },
            new string [] { "\uB716", "\uB716", "\u1104\u1172\u11B1", "\uB716", "\u1104\u1172\u11B1" },
            new string [] { "\uB717", "\uB717", "\u1104\u1172\u11B2", "\uB717", "\u1104\u1172\u11B2" },
            new string [] { "\uB718", "\uB718", "\u1104\u1172\u11B3", "\uB718", "\u1104\u1172\u11B3" },
            new string [] { "\uB719", "\uB719", "\u1104\u1172\u11B4", "\uB719", "\u1104\u1172\u11B4" },
            new string [] { "\uB71A", "\uB71A", "\u1104\u1172\u11B5", "\uB71A", "\u1104\u1172\u11B5" },
            new string [] { "\uB71B", "\uB71B", "\u1104\u1172\u11B6", "\uB71B", "\u1104\u1172\u11B6" },
            new string [] { "\uB71C", "\uB71C", "\u1104\u1172\u11B7", "\uB71C", "\u1104\u1172\u11B7" },
            new string [] { "\uB71D", "\uB71D", "\u1104\u1172\u11B8", "\uB71D", "\u1104\u1172\u11B8" },
            new string [] { "\uB71E", "\uB71E", "\u1104\u1172\u11B9", "\uB71E", "\u1104\u1172\u11B9" },
            new string [] { "\uB71F", "\uB71F", "\u1104\u1172\u11BA", "\uB71F", "\u1104\u1172\u11BA" },
            new string [] { "\uB720", "\uB720", "\u1104\u1172\u11BB", "\uB720", "\u1104\u1172\u11BB" },
            new string [] { "\uB721", "\uB721", "\u1104\u1172\u11BC", "\uB721", "\u1104\u1172\u11BC" },
            new string [] { "\uB722", "\uB722", "\u1104\u1172\u11BD", "\uB722", "\u1104\u1172\u11BD" },
            new string [] { "\uB723", "\uB723", "\u1104\u1172\u11BE", "\uB723", "\u1104\u1172\u11BE" },
            new string [] { "\uB724", "\uB724", "\u1104\u1172\u11BF", "\uB724", "\u1104\u1172\u11BF" },
            new string [] { "\uB725", "\uB725", "\u1104\u1172\u11C0", "\uB725", "\u1104\u1172\u11C0" },
            new string [] { "\uB726", "\uB726", "\u1104\u1172\u11C1", "\uB726", "\u1104\u1172\u11C1" },
            new string [] { "\uB727", "\uB727", "\u1104\u1172\u11C2", "\uB727", "\u1104\u1172\u11C2" },
            new string [] { "\uB728", "\uB728", "\u1104\u1173", "\uB728", "\u1104\u1173" },
            new string [] { "\uB729", "\uB729", "\u1104\u1173\u11A8", "\uB729", "\u1104\u1173\u11A8" },
            new string [] { "\uB72A", "\uB72A", "\u1104\u1173\u11A9", "\uB72A", "\u1104\u1173\u11A9" },
            new string [] { "\uB72B", "\uB72B", "\u1104\u1173\u11AA", "\uB72B", "\u1104\u1173\u11AA" },
            new string [] { "\uB72C", "\uB72C", "\u1104\u1173\u11AB", "\uB72C", "\u1104\u1173\u11AB" },
            new string [] { "\uB72D", "\uB72D", "\u1104\u1173\u11AC", "\uB72D", "\u1104\u1173\u11AC" },
            new string [] { "\uB72E", "\uB72E", "\u1104\u1173\u11AD", "\uB72E", "\u1104\u1173\u11AD" },
            new string [] { "\uB72F", "\uB72F", "\u1104\u1173\u11AE", "\uB72F", "\u1104\u1173\u11AE" },
            new string [] { "\uB730", "\uB730", "\u1104\u1173\u11AF", "\uB730", "\u1104\u1173\u11AF" },
            new string [] { "\uB731", "\uB731", "\u1104\u1173\u11B0", "\uB731", "\u1104\u1173\u11B0" },
            new string [] { "\uB732", "\uB732", "\u1104\u1173\u11B1", "\uB732", "\u1104\u1173\u11B1" },
            new string [] { "\uB733", "\uB733", "\u1104\u1173\u11B2", "\uB733", "\u1104\u1173\u11B2" },
            new string [] { "\uB734", "\uB734", "\u1104\u1173\u11B3", "\uB734", "\u1104\u1173\u11B3" },
            new string [] { "\uB735", "\uB735", "\u1104\u1173\u11B4", "\uB735", "\u1104\u1173\u11B4" },
            new string [] { "\uB736", "\uB736", "\u1104\u1173\u11B5", "\uB736", "\u1104\u1173\u11B5" },
            new string [] { "\uB737", "\uB737", "\u1104\u1173\u11B6", "\uB737", "\u1104\u1173\u11B6" },
            new string [] { "\uB738", "\uB738", "\u1104\u1173\u11B7", "\uB738", "\u1104\u1173\u11B7" },
            new string [] { "\uB739", "\uB739", "\u1104\u1173\u11B8", "\uB739", "\u1104\u1173\u11B8" },
            new string [] { "\uB73A", "\uB73A", "\u1104\u1173\u11B9", "\uB73A", "\u1104\u1173\u11B9" },
            new string [] { "\uB73B", "\uB73B", "\u1104\u1173\u11BA", "\uB73B", "\u1104\u1173\u11BA" },
            new string [] { "\uB73C", "\uB73C", "\u1104\u1173\u11BB", "\uB73C", "\u1104\u1173\u11BB" },
            new string [] { "\uB73D", "\uB73D", "\u1104\u1173\u11BC", "\uB73D", "\u1104\u1173\u11BC" },
            new string [] { "\uB73E", "\uB73E", "\u1104\u1173\u11BD", "\uB73E", "\u1104\u1173\u11BD" },
            new string [] { "\uB73F", "\uB73F", "\u1104\u1173\u11BE", "\uB73F", "\u1104\u1173\u11BE" },
            new string [] { "\uB740", "\uB740", "\u1104\u1173\u11BF", "\uB740", "\u1104\u1173\u11BF" },
            new string [] { "\uB741", "\uB741", "\u1104\u1173\u11C0", "\uB741", "\u1104\u1173\u11C0" },
            new string [] { "\uB742", "\uB742", "\u1104\u1173\u11C1", "\uB742", "\u1104\u1173\u11C1" },
            new string [] { "\uB743", "\uB743", "\u1104\u1173\u11C2", "\uB743", "\u1104\u1173\u11C2" },
            new string [] { "\uB744", "\uB744", "\u1104\u1174", "\uB744", "\u1104\u1174" },
            new string [] { "\uB745", "\uB745", "\u1104\u1174\u11A8", "\uB745", "\u1104\u1174\u11A8" },
            new string [] { "\uB746", "\uB746", "\u1104\u1174\u11A9", "\uB746", "\u1104\u1174\u11A9" },
            new string [] { "\uB747", "\uB747", "\u1104\u1174\u11AA", "\uB747", "\u1104\u1174\u11AA" },
            new string [] { "\uB748", "\uB748", "\u1104\u1174\u11AB", "\uB748", "\u1104\u1174\u11AB" },
            new string [] { "\uB749", "\uB749", "\u1104\u1174\u11AC", "\uB749", "\u1104\u1174\u11AC" },
            new string [] { "\uB74A", "\uB74A", "\u1104\u1174\u11AD", "\uB74A", "\u1104\u1174\u11AD" },
            new string [] { "\uB74B", "\uB74B", "\u1104\u1174\u11AE", "\uB74B", "\u1104\u1174\u11AE" },
            new string [] { "\uB74C", "\uB74C", "\u1104\u1174\u11AF", "\uB74C", "\u1104\u1174\u11AF" },
            new string [] { "\uB74D", "\uB74D", "\u1104\u1174\u11B0", "\uB74D", "\u1104\u1174\u11B0" },
            new string [] { "\uB74E", "\uB74E", "\u1104\u1174\u11B1", "\uB74E", "\u1104\u1174\u11B1" },
            new string [] { "\uB74F", "\uB74F", "\u1104\u1174\u11B2", "\uB74F", "\u1104\u1174\u11B2" },
            new string [] { "\uB750", "\uB750", "\u1104\u1174\u11B3", "\uB750", "\u1104\u1174\u11B3" },
            new string [] { "\uB751", "\uB751", "\u1104\u1174\u11B4", "\uB751", "\u1104\u1174\u11B4" },
            new string [] { "\uB752", "\uB752", "\u1104\u1174\u11B5", "\uB752", "\u1104\u1174\u11B5" },
            new string [] { "\uB753", "\uB753", "\u1104\u1174\u11B6", "\uB753", "\u1104\u1174\u11B6" },
            new string [] { "\uB754", "\uB754", "\u1104\u1174\u11B7", "\uB754", "\u1104\u1174\u11B7" },
            new string [] { "\uB755", "\uB755", "\u1104\u1174\u11B8", "\uB755", "\u1104\u1174\u11B8" },
            new string [] { "\uB756", "\uB756", "\u1104\u1174\u11B9", "\uB756", "\u1104\u1174\u11B9" },
            new string [] { "\uB757", "\uB757", "\u1104\u1174\u11BA", "\uB757", "\u1104\u1174\u11BA" },
            new string [] { "\uB758", "\uB758", "\u1104\u1174\u11BB", "\uB758", "\u1104\u1174\u11BB" },
            new string [] { "\uB759", "\uB759", "\u1104\u1174\u11BC", "\uB759", "\u1104\u1174\u11BC" },
            new string [] { "\uB75A", "\uB75A", "\u1104\u1174\u11BD", "\uB75A", "\u1104\u1174\u11BD" },
            new string [] { "\uB75B", "\uB75B", "\u1104\u1174\u11BE", "\uB75B", "\u1104\u1174\u11BE" },
            new string [] { "\uB75C", "\uB75C", "\u1104\u1174\u11BF", "\uB75C", "\u1104\u1174\u11BF" },
            new string [] { "\uB75D", "\uB75D", "\u1104\u1174\u11C0", "\uB75D", "\u1104\u1174\u11C0" },
            new string [] { "\uB75E", "\uB75E", "\u1104\u1174\u11C1", "\uB75E", "\u1104\u1174\u11C1" },
            new string [] { "\uB75F", "\uB75F", "\u1104\u1174\u11C2", "\uB75F", "\u1104\u1174\u11C2" },
            new string [] { "\uB760", "\uB760", "\u1104\u1175", "\uB760", "\u1104\u1175" },
            new string [] { "\uB761", "\uB761", "\u1104\u1175\u11A8", "\uB761", "\u1104\u1175\u11A8" },
            new string [] { "\uB762", "\uB762", "\u1104\u1175\u11A9", "\uB762", "\u1104\u1175\u11A9" },
            new string [] { "\uB763", "\uB763", "\u1104\u1175\u11AA", "\uB763", "\u1104\u1175\u11AA" },
            new string [] { "\uB764", "\uB764", "\u1104\u1175\u11AB", "\uB764", "\u1104\u1175\u11AB" },
            new string [] { "\uB765", "\uB765", "\u1104\u1175\u11AC", "\uB765", "\u1104\u1175\u11AC" },
            new string [] { "\uB766", "\uB766", "\u1104\u1175\u11AD", "\uB766", "\u1104\u1175\u11AD" },
            new string [] { "\uB767", "\uB767", "\u1104\u1175\u11AE", "\uB767", "\u1104\u1175\u11AE" },
            new string [] { "\uB768", "\uB768", "\u1104\u1175\u11AF", "\uB768", "\u1104\u1175\u11AF" },
            new string [] { "\uB769", "\uB769", "\u1104\u1175\u11B0", "\uB769", "\u1104\u1175\u11B0" },
            new string [] { "\uB76A", "\uB76A", "\u1104\u1175\u11B1", "\uB76A", "\u1104\u1175\u11B1" },
            new string [] { "\uB76B", "\uB76B", "\u1104\u1175\u11B2", "\uB76B", "\u1104\u1175\u11B2" },
            new string [] { "\uB76C", "\uB76C", "\u1104\u1175\u11B3", "\uB76C", "\u1104\u1175\u11B3" },
            new string [] { "\uB76D", "\uB76D", "\u1104\u1175\u11B4", "\uB76D", "\u1104\u1175\u11B4" },
            new string [] { "\uB76E", "\uB76E", "\u1104\u1175\u11B5", "\uB76E", "\u1104\u1175\u11B5" },
            new string [] { "\uB76F", "\uB76F", "\u1104\u1175\u11B6", "\uB76F", "\u1104\u1175\u11B6" },
            new string [] { "\uB770", "\uB770", "\u1104\u1175\u11B7", "\uB770", "\u1104\u1175\u11B7" },
            new string [] { "\uB771", "\uB771", "\u1104\u1175\u11B8", "\uB771", "\u1104\u1175\u11B8" },
            new string [] { "\uB772", "\uB772", "\u1104\u1175\u11B9", "\uB772", "\u1104\u1175\u11B9" },
            new string [] { "\uB773", "\uB773", "\u1104\u1175\u11BA", "\uB773", "\u1104\u1175\u11BA" },
            new string [] { "\uB774", "\uB774", "\u1104\u1175\u11BB", "\uB774", "\u1104\u1175\u11BB" },
            new string [] { "\uB775", "\uB775", "\u1104\u1175\u11BC", "\uB775", "\u1104\u1175\u11BC" },
            new string [] { "\uB776", "\uB776", "\u1104\u1175\u11BD", "\uB776", "\u1104\u1175\u11BD" },
            new string [] { "\uB777", "\uB777", "\u1104\u1175\u11BE", "\uB777", "\u1104\u1175\u11BE" },
            new string [] { "\uB778", "\uB778", "\u1104\u1175\u11BF", "\uB778", "\u1104\u1175\u11BF" },
            new string [] { "\uB779", "\uB779", "\u1104\u1175\u11C0", "\uB779", "\u1104\u1175\u11C0" },
            new string [] { "\uB77A", "\uB77A", "\u1104\u1175\u11C1", "\uB77A", "\u1104\u1175\u11C1" },
            new string [] { "\uB77B", "\uB77B", "\u1104\u1175\u11C2", "\uB77B", "\u1104\u1175\u11C2" },
            new string [] { "\uB77C", "\uB77C", "\u1105\u1161", "\uB77C", "\u1105\u1161" },
            new string [] { "\uB77D", "\uB77D", "\u1105\u1161\u11A8", "\uB77D", "\u1105\u1161\u11A8" },
            new string [] { "\uB77E", "\uB77E", "\u1105\u1161\u11A9", "\uB77E", "\u1105\u1161\u11A9" },
            new string [] { "\uB77F", "\uB77F", "\u1105\u1161\u11AA", "\uB77F", "\u1105\u1161\u11AA" },
            new string [] { "\uB780", "\uB780", "\u1105\u1161\u11AB", "\uB780", "\u1105\u1161\u11AB" },
            new string [] { "\uB781", "\uB781", "\u1105\u1161\u11AC", "\uB781", "\u1105\u1161\u11AC" },
            new string [] { "\uB782", "\uB782", "\u1105\u1161\u11AD", "\uB782", "\u1105\u1161\u11AD" },
            new string [] { "\uB783", "\uB783", "\u1105\u1161\u11AE", "\uB783", "\u1105\u1161\u11AE" },
            new string [] { "\uB784", "\uB784", "\u1105\u1161\u11AF", "\uB784", "\u1105\u1161\u11AF" },
            new string [] { "\uB785", "\uB785", "\u1105\u1161\u11B0", "\uB785", "\u1105\u1161\u11B0" },
            new string [] { "\uB786", "\uB786", "\u1105\u1161\u11B1", "\uB786", "\u1105\u1161\u11B1" },
            new string [] { "\uB787", "\uB787", "\u1105\u1161\u11B2", "\uB787", "\u1105\u1161\u11B2" },
            new string [] { "\uB788", "\uB788", "\u1105\u1161\u11B3", "\uB788", "\u1105\u1161\u11B3" },
            new string [] { "\uB789", "\uB789", "\u1105\u1161\u11B4", "\uB789", "\u1105\u1161\u11B4" },
            new string [] { "\uB78A", "\uB78A", "\u1105\u1161\u11B5", "\uB78A", "\u1105\u1161\u11B5" },
            new string [] { "\uB78B", "\uB78B", "\u1105\u1161\u11B6", "\uB78B", "\u1105\u1161\u11B6" },
            new string [] { "\uB78C", "\uB78C", "\u1105\u1161\u11B7", "\uB78C", "\u1105\u1161\u11B7" },
            new string [] { "\uB78D", "\uB78D", "\u1105\u1161\u11B8", "\uB78D", "\u1105\u1161\u11B8" },
            new string [] { "\uB78E", "\uB78E", "\u1105\u1161\u11B9", "\uB78E", "\u1105\u1161\u11B9" },
            new string [] { "\uB78F", "\uB78F", "\u1105\u1161\u11BA", "\uB78F", "\u1105\u1161\u11BA" },
            new string [] { "\uB790", "\uB790", "\u1105\u1161\u11BB", "\uB790", "\u1105\u1161\u11BB" },
            new string [] { "\uB791", "\uB791", "\u1105\u1161\u11BC", "\uB791", "\u1105\u1161\u11BC" },
            new string [] { "\uB792", "\uB792", "\u1105\u1161\u11BD", "\uB792", "\u1105\u1161\u11BD" },
            new string [] { "\uB793", "\uB793", "\u1105\u1161\u11BE", "\uB793", "\u1105\u1161\u11BE" },
            new string [] { "\uB794", "\uB794", "\u1105\u1161\u11BF", "\uB794", "\u1105\u1161\u11BF" },
            new string [] { "\uB795", "\uB795", "\u1105\u1161\u11C0", "\uB795", "\u1105\u1161\u11C0" },
            new string [] { "\uB796", "\uB796", "\u1105\u1161\u11C1", "\uB796", "\u1105\u1161\u11C1" },
            new string [] { "\uB797", "\uB797", "\u1105\u1161\u11C2", "\uB797", "\u1105\u1161\u11C2" },
            new string [] { "\uB798", "\uB798", "\u1105\u1162", "\uB798", "\u1105\u1162" },
            new string [] { "\uB799", "\uB799", "\u1105\u1162\u11A8", "\uB799", "\u1105\u1162\u11A8" },
            new string [] { "\uB79A", "\uB79A", "\u1105\u1162\u11A9", "\uB79A", "\u1105\u1162\u11A9" },
            new string [] { "\uB79B", "\uB79B", "\u1105\u1162\u11AA", "\uB79B", "\u1105\u1162\u11AA" },
            new string [] { "\uB79C", "\uB79C", "\u1105\u1162\u11AB", "\uB79C", "\u1105\u1162\u11AB" },
            new string [] { "\uB79D", "\uB79D", "\u1105\u1162\u11AC", "\uB79D", "\u1105\u1162\u11AC" },
            new string [] { "\uB79E", "\uB79E", "\u1105\u1162\u11AD", "\uB79E", "\u1105\u1162\u11AD" },
            new string [] { "\uB79F", "\uB79F", "\u1105\u1162\u11AE", "\uB79F", "\u1105\u1162\u11AE" },
            new string [] { "\uB7A0", "\uB7A0", "\u1105\u1162\u11AF", "\uB7A0", "\u1105\u1162\u11AF" },
            new string [] { "\uB7A1", "\uB7A1", "\u1105\u1162\u11B0", "\uB7A1", "\u1105\u1162\u11B0" },
            new string [] { "\uB7A2", "\uB7A2", "\u1105\u1162\u11B1", "\uB7A2", "\u1105\u1162\u11B1" },
            new string [] { "\uB7A3", "\uB7A3", "\u1105\u1162\u11B2", "\uB7A3", "\u1105\u1162\u11B2" },
            new string [] { "\uB7A4", "\uB7A4", "\u1105\u1162\u11B3", "\uB7A4", "\u1105\u1162\u11B3" },
            new string [] { "\uB7A5", "\uB7A5", "\u1105\u1162\u11B4", "\uB7A5", "\u1105\u1162\u11B4" },
            new string [] { "\uB7A6", "\uB7A6", "\u1105\u1162\u11B5", "\uB7A6", "\u1105\u1162\u11B5" },
            new string [] { "\uB7A7", "\uB7A7", "\u1105\u1162\u11B6", "\uB7A7", "\u1105\u1162\u11B6" },
            new string [] { "\uB7A8", "\uB7A8", "\u1105\u1162\u11B7", "\uB7A8", "\u1105\u1162\u11B7" },
            new string [] { "\uB7A9", "\uB7A9", "\u1105\u1162\u11B8", "\uB7A9", "\u1105\u1162\u11B8" },
            new string [] { "\uB7AA", "\uB7AA", "\u1105\u1162\u11B9", "\uB7AA", "\u1105\u1162\u11B9" },
            new string [] { "\uB7AB", "\uB7AB", "\u1105\u1162\u11BA", "\uB7AB", "\u1105\u1162\u11BA" },
            new string [] { "\uB7AC", "\uB7AC", "\u1105\u1162\u11BB", "\uB7AC", "\u1105\u1162\u11BB" },
            new string [] { "\uB7AD", "\uB7AD", "\u1105\u1162\u11BC", "\uB7AD", "\u1105\u1162\u11BC" },
            new string [] { "\uB7AE", "\uB7AE", "\u1105\u1162\u11BD", "\uB7AE", "\u1105\u1162\u11BD" },
            new string [] { "\uB7AF", "\uB7AF", "\u1105\u1162\u11BE", "\uB7AF", "\u1105\u1162\u11BE" },
            new string [] { "\uB7B0", "\uB7B0", "\u1105\u1162\u11BF", "\uB7B0", "\u1105\u1162\u11BF" },
            new string [] { "\uB7B1", "\uB7B1", "\u1105\u1162\u11C0", "\uB7B1", "\u1105\u1162\u11C0" },
            new string [] { "\uB7B2", "\uB7B2", "\u1105\u1162\u11C1", "\uB7B2", "\u1105\u1162\u11C1" },
            new string [] { "\uB7B3", "\uB7B3", "\u1105\u1162\u11C2", "\uB7B3", "\u1105\u1162\u11C2" },
            new string [] { "\uB7B4", "\uB7B4", "\u1105\u1163", "\uB7B4", "\u1105\u1163" },
            new string [] { "\uB7B5", "\uB7B5", "\u1105\u1163\u11A8", "\uB7B5", "\u1105\u1163\u11A8" },
            new string [] { "\uB7B6", "\uB7B6", "\u1105\u1163\u11A9", "\uB7B6", "\u1105\u1163\u11A9" },
            new string [] { "\uB7B7", "\uB7B7", "\u1105\u1163\u11AA", "\uB7B7", "\u1105\u1163\u11AA" },
            new string [] { "\uB7B8", "\uB7B8", "\u1105\u1163\u11AB", "\uB7B8", "\u1105\u1163\u11AB" },
            new string [] { "\uB7B9", "\uB7B9", "\u1105\u1163\u11AC", "\uB7B9", "\u1105\u1163\u11AC" },
            new string [] { "\uB7BA", "\uB7BA", "\u1105\u1163\u11AD", "\uB7BA", "\u1105\u1163\u11AD" },
            new string [] { "\uB7BB", "\uB7BB", "\u1105\u1163\u11AE", "\uB7BB", "\u1105\u1163\u11AE" },
            new string [] { "\uB7BC", "\uB7BC", "\u1105\u1163\u11AF", "\uB7BC", "\u1105\u1163\u11AF" },
            new string [] { "\uB7BD", "\uB7BD", "\u1105\u1163\u11B0", "\uB7BD", "\u1105\u1163\u11B0" },
            new string [] { "\uB7BE", "\uB7BE", "\u1105\u1163\u11B1", "\uB7BE", "\u1105\u1163\u11B1" },
            new string [] { "\uB7BF", "\uB7BF", "\u1105\u1163\u11B2", "\uB7BF", "\u1105\u1163\u11B2" },
            new string [] { "\uB7C0", "\uB7C0", "\u1105\u1163\u11B3", "\uB7C0", "\u1105\u1163\u11B3" },
            new string [] { "\uB7C1", "\uB7C1", "\u1105\u1163\u11B4", "\uB7C1", "\u1105\u1163\u11B4" },
            new string [] { "\uB7C2", "\uB7C2", "\u1105\u1163\u11B5", "\uB7C2", "\u1105\u1163\u11B5" },
            new string [] { "\uB7C3", "\uB7C3", "\u1105\u1163\u11B6", "\uB7C3", "\u1105\u1163\u11B6" },
            new string [] { "\uB7C4", "\uB7C4", "\u1105\u1163\u11B7", "\uB7C4", "\u1105\u1163\u11B7" },
            new string [] { "\uB7C5", "\uB7C5", "\u1105\u1163\u11B8", "\uB7C5", "\u1105\u1163\u11B8" },
            new string [] { "\uB7C6", "\uB7C6", "\u1105\u1163\u11B9", "\uB7C6", "\u1105\u1163\u11B9" },
            new string [] { "\uB7C7", "\uB7C7", "\u1105\u1163\u11BA", "\uB7C7", "\u1105\u1163\u11BA" },
            new string [] { "\uB7C8", "\uB7C8", "\u1105\u1163\u11BB", "\uB7C8", "\u1105\u1163\u11BB" },
            new string [] { "\uB7C9", "\uB7C9", "\u1105\u1163\u11BC", "\uB7C9", "\u1105\u1163\u11BC" },
            new string [] { "\uB7CA", "\uB7CA", "\u1105\u1163\u11BD", "\uB7CA", "\u1105\u1163\u11BD" },
            new string [] { "\uB7CB", "\uB7CB", "\u1105\u1163\u11BE", "\uB7CB", "\u1105\u1163\u11BE" },
            new string [] { "\uB7CC", "\uB7CC", "\u1105\u1163\u11BF", "\uB7CC", "\u1105\u1163\u11BF" },
            new string [] { "\uB7CD", "\uB7CD", "\u1105\u1163\u11C0", "\uB7CD", "\u1105\u1163\u11C0" },
            new string [] { "\uB7CE", "\uB7CE", "\u1105\u1163\u11C1", "\uB7CE", "\u1105\u1163\u11C1" },
            new string [] { "\uB7CF", "\uB7CF", "\u1105\u1163\u11C2", "\uB7CF", "\u1105\u1163\u11C2" },
            new string [] { "\uB7D0", "\uB7D0", "\u1105\u1164", "\uB7D0", "\u1105\u1164" },
            new string [] { "\uB7D1", "\uB7D1", "\u1105\u1164\u11A8", "\uB7D1", "\u1105\u1164\u11A8" },
            new string [] { "\uB7D2", "\uB7D2", "\u1105\u1164\u11A9", "\uB7D2", "\u1105\u1164\u11A9" },
            new string [] { "\uB7D3", "\uB7D3", "\u1105\u1164\u11AA", "\uB7D3", "\u1105\u1164\u11AA" },
            new string [] { "\uB7D4", "\uB7D4", "\u1105\u1164\u11AB", "\uB7D4", "\u1105\u1164\u11AB" },
            new string [] { "\uB7D5", "\uB7D5", "\u1105\u1164\u11AC", "\uB7D5", "\u1105\u1164\u11AC" },
            new string [] { "\uB7D6", "\uB7D6", "\u1105\u1164\u11AD", "\uB7D6", "\u1105\u1164\u11AD" },
            new string [] { "\uB7D7", "\uB7D7", "\u1105\u1164\u11AE", "\uB7D7", "\u1105\u1164\u11AE" },
            new string [] { "\uB7D8", "\uB7D8", "\u1105\u1164\u11AF", "\uB7D8", "\u1105\u1164\u11AF" },
            new string [] { "\uB7D9", "\uB7D9", "\u1105\u1164\u11B0", "\uB7D9", "\u1105\u1164\u11B0" },
            new string [] { "\uB7DA", "\uB7DA", "\u1105\u1164\u11B1", "\uB7DA", "\u1105\u1164\u11B1" },
            new string [] { "\uB7DB", "\uB7DB", "\u1105\u1164\u11B2", "\uB7DB", "\u1105\u1164\u11B2" },
            new string [] { "\uB7DC", "\uB7DC", "\u1105\u1164\u11B3", "\uB7DC", "\u1105\u1164\u11B3" },
            new string [] { "\uB7DD", "\uB7DD", "\u1105\u1164\u11B4", "\uB7DD", "\u1105\u1164\u11B4" },
            new string [] { "\uB7DE", "\uB7DE", "\u1105\u1164\u11B5", "\uB7DE", "\u1105\u1164\u11B5" },
            new string [] { "\uB7DF", "\uB7DF", "\u1105\u1164\u11B6", "\uB7DF", "\u1105\u1164\u11B6" },
            new string [] { "\uB7E0", "\uB7E0", "\u1105\u1164\u11B7", "\uB7E0", "\u1105\u1164\u11B7" },
            new string [] { "\uB7E1", "\uB7E1", "\u1105\u1164\u11B8", "\uB7E1", "\u1105\u1164\u11B8" },
            new string [] { "\uB7E2", "\uB7E2", "\u1105\u1164\u11B9", "\uB7E2", "\u1105\u1164\u11B9" },
            new string [] { "\uB7E3", "\uB7E3", "\u1105\u1164\u11BA", "\uB7E3", "\u1105\u1164\u11BA" },
            new string [] { "\uB7E4", "\uB7E4", "\u1105\u1164\u11BB", "\uB7E4", "\u1105\u1164\u11BB" },
            new string [] { "\uB7E5", "\uB7E5", "\u1105\u1164\u11BC", "\uB7E5", "\u1105\u1164\u11BC" },
            new string [] { "\uB7E6", "\uB7E6", "\u1105\u1164\u11BD", "\uB7E6", "\u1105\u1164\u11BD" },
            new string [] { "\uB7E7", "\uB7E7", "\u1105\u1164\u11BE", "\uB7E7", "\u1105\u1164\u11BE" },
            new string [] { "\uB7E8", "\uB7E8", "\u1105\u1164\u11BF", "\uB7E8", "\u1105\u1164\u11BF" },
            new string [] { "\uB7E9", "\uB7E9", "\u1105\u1164\u11C0", "\uB7E9", "\u1105\u1164\u11C0" },
            new string [] { "\uB7EA", "\uB7EA", "\u1105\u1164\u11C1", "\uB7EA", "\u1105\u1164\u11C1" },
            new string [] { "\uB7EB", "\uB7EB", "\u1105\u1164\u11C2", "\uB7EB", "\u1105\u1164\u11C2" },
            new string [] { "\uB7EC", "\uB7EC", "\u1105\u1165", "\uB7EC", "\u1105\u1165" },
            new string [] { "\uB7ED", "\uB7ED", "\u1105\u1165\u11A8", "\uB7ED", "\u1105\u1165\u11A8" },
            new string [] { "\uB7EE", "\uB7EE", "\u1105\u1165\u11A9", "\uB7EE", "\u1105\u1165\u11A9" },
            new string [] { "\uB7EF", "\uB7EF", "\u1105\u1165\u11AA", "\uB7EF", "\u1105\u1165\u11AA" },
            new string [] { "\uB7F0", "\uB7F0", "\u1105\u1165\u11AB", "\uB7F0", "\u1105\u1165\u11AB" },
            new string [] { "\uB7F1", "\uB7F1", "\u1105\u1165\u11AC", "\uB7F1", "\u1105\u1165\u11AC" },
            new string [] { "\uB7F2", "\uB7F2", "\u1105\u1165\u11AD", "\uB7F2", "\u1105\u1165\u11AD" },
            new string [] { "\uB7F3", "\uB7F3", "\u1105\u1165\u11AE", "\uB7F3", "\u1105\u1165\u11AE" },
            new string [] { "\uB7F4", "\uB7F4", "\u1105\u1165\u11AF", "\uB7F4", "\u1105\u1165\u11AF" },
            new string [] { "\uB7F5", "\uB7F5", "\u1105\u1165\u11B0", "\uB7F5", "\u1105\u1165\u11B0" },
            new string [] { "\uB7F6", "\uB7F6", "\u1105\u1165\u11B1", "\uB7F6", "\u1105\u1165\u11B1" },
            new string [] { "\uB7F7", "\uB7F7", "\u1105\u1165\u11B2", "\uB7F7", "\u1105\u1165\u11B2" },
            new string [] { "\uB7F8", "\uB7F8", "\u1105\u1165\u11B3", "\uB7F8", "\u1105\u1165\u11B3" },
            new string [] { "\uB7F9", "\uB7F9", "\u1105\u1165\u11B4", "\uB7F9", "\u1105\u1165\u11B4" },
            new string [] { "\uB7FA", "\uB7FA", "\u1105\u1165\u11B5", "\uB7FA", "\u1105\u1165\u11B5" },
            new string [] { "\uB7FB", "\uB7FB", "\u1105\u1165\u11B6", "\uB7FB", "\u1105\u1165\u11B6" },
            new string [] { "\uB7FC", "\uB7FC", "\u1105\u1165\u11B7", "\uB7FC", "\u1105\u1165\u11B7" },
            new string [] { "\uB7FD", "\uB7FD", "\u1105\u1165\u11B8", "\uB7FD", "\u1105\u1165\u11B8" },
            new string [] { "\uB7FE", "\uB7FE", "\u1105\u1165\u11B9", "\uB7FE", "\u1105\u1165\u11B9" },
            new string [] { "\uB7FF", "\uB7FF", "\u1105\u1165\u11BA", "\uB7FF", "\u1105\u1165\u11BA" },
            new string [] { "\uB800", "\uB800", "\u1105\u1165\u11BB", "\uB800", "\u1105\u1165\u11BB" },
            new string [] { "\uB801", "\uB801", "\u1105\u1165\u11BC", "\uB801", "\u1105\u1165\u11BC" },
            new string [] { "\uB802", "\uB802", "\u1105\u1165\u11BD", "\uB802", "\u1105\u1165\u11BD" },
            new string [] { "\uB803", "\uB803", "\u1105\u1165\u11BE", "\uB803", "\u1105\u1165\u11BE" },
            new string [] { "\uB804", "\uB804", "\u1105\u1165\u11BF", "\uB804", "\u1105\u1165\u11BF" },
            new string [] { "\uB805", "\uB805", "\u1105\u1165\u11C0", "\uB805", "\u1105\u1165\u11C0" },
            new string [] { "\uB806", "\uB806", "\u1105\u1165\u11C1", "\uB806", "\u1105\u1165\u11C1" },
            new string [] { "\uB807", "\uB807", "\u1105\u1165\u11C2", "\uB807", "\u1105\u1165\u11C2" },
            new string [] { "\uB808", "\uB808", "\u1105\u1166", "\uB808", "\u1105\u1166" },
            new string [] { "\uB809", "\uB809", "\u1105\u1166\u11A8", "\uB809", "\u1105\u1166\u11A8" },
            new string [] { "\uB80A", "\uB80A", "\u1105\u1166\u11A9", "\uB80A", "\u1105\u1166\u11A9" },
            new string [] { "\uB80B", "\uB80B", "\u1105\u1166\u11AA", "\uB80B", "\u1105\u1166\u11AA" },
            new string [] { "\uB80C", "\uB80C", "\u1105\u1166\u11AB", "\uB80C", "\u1105\u1166\u11AB" },
            new string [] { "\uB80D", "\uB80D", "\u1105\u1166\u11AC", "\uB80D", "\u1105\u1166\u11AC" },
            new string [] { "\uB80E", "\uB80E", "\u1105\u1166\u11AD", "\uB80E", "\u1105\u1166\u11AD" },
            new string [] { "\uB80F", "\uB80F", "\u1105\u1166\u11AE", "\uB80F", "\u1105\u1166\u11AE" },
            new string [] { "\uB810", "\uB810", "\u1105\u1166\u11AF", "\uB810", "\u1105\u1166\u11AF" },
            new string [] { "\uB811", "\uB811", "\u1105\u1166\u11B0", "\uB811", "\u1105\u1166\u11B0" },
            new string [] { "\uB812", "\uB812", "\u1105\u1166\u11B1", "\uB812", "\u1105\u1166\u11B1" },
            new string [] { "\uB813", "\uB813", "\u1105\u1166\u11B2", "\uB813", "\u1105\u1166\u11B2" },
            new string [] { "\uB814", "\uB814", "\u1105\u1166\u11B3", "\uB814", "\u1105\u1166\u11B3" },
            new string [] { "\uB815", "\uB815", "\u1105\u1166\u11B4", "\uB815", "\u1105\u1166\u11B4" },
            new string [] { "\uB816", "\uB816", "\u1105\u1166\u11B5", "\uB816", "\u1105\u1166\u11B5" },
            new string [] { "\uB817", "\uB817", "\u1105\u1166\u11B6", "\uB817", "\u1105\u1166\u11B6" },
            new string [] { "\uB818", "\uB818", "\u1105\u1166\u11B7", "\uB818", "\u1105\u1166\u11B7" },
            new string [] { "\uB819", "\uB819", "\u1105\u1166\u11B8", "\uB819", "\u1105\u1166\u11B8" },
            new string [] { "\uB81A", "\uB81A", "\u1105\u1166\u11B9", "\uB81A", "\u1105\u1166\u11B9" },
            new string [] { "\uB81B", "\uB81B", "\u1105\u1166\u11BA", "\uB81B", "\u1105\u1166\u11BA" },
            new string [] { "\uB81C", "\uB81C", "\u1105\u1166\u11BB", "\uB81C", "\u1105\u1166\u11BB" },
            new string [] { "\uB81D", "\uB81D", "\u1105\u1166\u11BC", "\uB81D", "\u1105\u1166\u11BC" },
            new string [] { "\uB81E", "\uB81E", "\u1105\u1166\u11BD", "\uB81E", "\u1105\u1166\u11BD" },
            new string [] { "\uB81F", "\uB81F", "\u1105\u1166\u11BE", "\uB81F", "\u1105\u1166\u11BE" },
            new string [] { "\uB820", "\uB820", "\u1105\u1166\u11BF", "\uB820", "\u1105\u1166\u11BF" },
            new string [] { "\uB821", "\uB821", "\u1105\u1166\u11C0", "\uB821", "\u1105\u1166\u11C0" },
            new string [] { "\uB822", "\uB822", "\u1105\u1166\u11C1", "\uB822", "\u1105\u1166\u11C1" },
            new string [] { "\uB823", "\uB823", "\u1105\u1166\u11C2", "\uB823", "\u1105\u1166\u11C2" },
            new string [] { "\uB824", "\uB824", "\u1105\u1167", "\uB824", "\u1105\u1167" },
            new string [] { "\uB825", "\uB825", "\u1105\u1167\u11A8", "\uB825", "\u1105\u1167\u11A8" },
            new string [] { "\uB826", "\uB826", "\u1105\u1167\u11A9", "\uB826", "\u1105\u1167\u11A9" },
            new string [] { "\uB827", "\uB827", "\u1105\u1167\u11AA", "\uB827", "\u1105\u1167\u11AA" },
            new string [] { "\uB828", "\uB828", "\u1105\u1167\u11AB", "\uB828", "\u1105\u1167\u11AB" },
            new string [] { "\uB829", "\uB829", "\u1105\u1167\u11AC", "\uB829", "\u1105\u1167\u11AC" },
            new string [] { "\uB82A", "\uB82A", "\u1105\u1167\u11AD", "\uB82A", "\u1105\u1167\u11AD" },
            new string [] { "\uB82B", "\uB82B", "\u1105\u1167\u11AE", "\uB82B", "\u1105\u1167\u11AE" },
            new string [] { "\uB82C", "\uB82C", "\u1105\u1167\u11AF", "\uB82C", "\u1105\u1167\u11AF" },
            new string [] { "\uB82D", "\uB82D", "\u1105\u1167\u11B0", "\uB82D", "\u1105\u1167\u11B0" },
            new string [] { "\uB82E", "\uB82E", "\u1105\u1167\u11B1", "\uB82E", "\u1105\u1167\u11B1" },
            new string [] { "\uB82F", "\uB82F", "\u1105\u1167\u11B2", "\uB82F", "\u1105\u1167\u11B2" },
            new string [] { "\uB830", "\uB830", "\u1105\u1167\u11B3", "\uB830", "\u1105\u1167\u11B3" },
            new string [] { "\uB831", "\uB831", "\u1105\u1167\u11B4", "\uB831", "\u1105\u1167\u11B4" },
            new string [] { "\uB832", "\uB832", "\u1105\u1167\u11B5", "\uB832", "\u1105\u1167\u11B5" },
            new string [] { "\uB833", "\uB833", "\u1105\u1167\u11B6", "\uB833", "\u1105\u1167\u11B6" },
            new string [] { "\uB834", "\uB834", "\u1105\u1167\u11B7", "\uB834", "\u1105\u1167\u11B7" },
            new string [] { "\uB835", "\uB835", "\u1105\u1167\u11B8", "\uB835", "\u1105\u1167\u11B8" },
            new string [] { "\uB836", "\uB836", "\u1105\u1167\u11B9", "\uB836", "\u1105\u1167\u11B9" },
            new string [] { "\uB837", "\uB837", "\u1105\u1167\u11BA", "\uB837", "\u1105\u1167\u11BA" },
            new string [] { "\uB838", "\uB838", "\u1105\u1167\u11BB", "\uB838", "\u1105\u1167\u11BB" },
            new string [] { "\uB839", "\uB839", "\u1105\u1167\u11BC", "\uB839", "\u1105\u1167\u11BC" },
            new string [] { "\uB83A", "\uB83A", "\u1105\u1167\u11BD", "\uB83A", "\u1105\u1167\u11BD" },
            new string [] { "\uB83B", "\uB83B", "\u1105\u1167\u11BE", "\uB83B", "\u1105\u1167\u11BE" },
            new string [] { "\uB83C", "\uB83C", "\u1105\u1167\u11BF", "\uB83C", "\u1105\u1167\u11BF" },
            new string [] { "\uB83D", "\uB83D", "\u1105\u1167\u11C0", "\uB83D", "\u1105\u1167\u11C0" },
            new string [] { "\uB83E", "\uB83E", "\u1105\u1167\u11C1", "\uB83E", "\u1105\u1167\u11C1" },
            new string [] { "\uB83F", "\uB83F", "\u1105\u1167\u11C2", "\uB83F", "\u1105\u1167\u11C2" },
            new string [] { "\uB840", "\uB840", "\u1105\u1168", "\uB840", "\u1105\u1168" },
            new string [] { "\uB841", "\uB841", "\u1105\u1168\u11A8", "\uB841", "\u1105\u1168\u11A8" },
            new string [] { "\uB842", "\uB842", "\u1105\u1168\u11A9", "\uB842", "\u1105\u1168\u11A9" },
            new string [] { "\uB843", "\uB843", "\u1105\u1168\u11AA", "\uB843", "\u1105\u1168\u11AA" },
            new string [] { "\uB844", "\uB844", "\u1105\u1168\u11AB", "\uB844", "\u1105\u1168\u11AB" },
            new string [] { "\uB845", "\uB845", "\u1105\u1168\u11AC", "\uB845", "\u1105\u1168\u11AC" },
            new string [] { "\uB846", "\uB846", "\u1105\u1168\u11AD", "\uB846", "\u1105\u1168\u11AD" },
            new string [] { "\uB847", "\uB847", "\u1105\u1168\u11AE", "\uB847", "\u1105\u1168\u11AE" },
            new string [] { "\uB848", "\uB848", "\u1105\u1168\u11AF", "\uB848", "\u1105\u1168\u11AF" },
            new string [] { "\uB849", "\uB849", "\u1105\u1168\u11B0", "\uB849", "\u1105\u1168\u11B0" },
            new string [] { "\uB84A", "\uB84A", "\u1105\u1168\u11B1", "\uB84A", "\u1105\u1168\u11B1" },
            new string [] { "\uB84B", "\uB84B", "\u1105\u1168\u11B2", "\uB84B", "\u1105\u1168\u11B2" },
            new string [] { "\uB84C", "\uB84C", "\u1105\u1168\u11B3", "\uB84C", "\u1105\u1168\u11B3" },
            new string [] { "\uB84D", "\uB84D", "\u1105\u1168\u11B4", "\uB84D", "\u1105\u1168\u11B4" },
            new string [] { "\uB84E", "\uB84E", "\u1105\u1168\u11B5", "\uB84E", "\u1105\u1168\u11B5" },
            new string [] { "\uB84F", "\uB84F", "\u1105\u1168\u11B6", "\uB84F", "\u1105\u1168\u11B6" },
            new string [] { "\uB850", "\uB850", "\u1105\u1168\u11B7", "\uB850", "\u1105\u1168\u11B7" },
            new string [] { "\uB851", "\uB851", "\u1105\u1168\u11B8", "\uB851", "\u1105\u1168\u11B8" },
            new string [] { "\uB852", "\uB852", "\u1105\u1168\u11B9", "\uB852", "\u1105\u1168\u11B9" },
            new string [] { "\uB853", "\uB853", "\u1105\u1168\u11BA", "\uB853", "\u1105\u1168\u11BA" },
            new string [] { "\uB854", "\uB854", "\u1105\u1168\u11BB", "\uB854", "\u1105\u1168\u11BB" },
            new string [] { "\uB855", "\uB855", "\u1105\u1168\u11BC", "\uB855", "\u1105\u1168\u11BC" },
            new string [] { "\uB856", "\uB856", "\u1105\u1168\u11BD", "\uB856", "\u1105\u1168\u11BD" },
            new string [] { "\uB857", "\uB857", "\u1105\u1168\u11BE", "\uB857", "\u1105\u1168\u11BE" },
            new string [] { "\uB858", "\uB858", "\u1105\u1168\u11BF", "\uB858", "\u1105\u1168\u11BF" },
            new string [] { "\uB859", "\uB859", "\u1105\u1168\u11C0", "\uB859", "\u1105\u1168\u11C0" },
            new string [] { "\uB85A", "\uB85A", "\u1105\u1168\u11C1", "\uB85A", "\u1105\u1168\u11C1" },
            new string [] { "\uB85B", "\uB85B", "\u1105\u1168\u11C2", "\uB85B", "\u1105\u1168\u11C2" },
            new string [] { "\uB85C", "\uB85C", "\u1105\u1169", "\uB85C", "\u1105\u1169" },
            new string [] { "\uB85D", "\uB85D", "\u1105\u1169\u11A8", "\uB85D", "\u1105\u1169\u11A8" },
            new string [] { "\uB85E", "\uB85E", "\u1105\u1169\u11A9", "\uB85E", "\u1105\u1169\u11A9" },
            new string [] { "\uB85F", "\uB85F", "\u1105\u1169\u11AA", "\uB85F", "\u1105\u1169\u11AA" },
            new string [] { "\uB860", "\uB860", "\u1105\u1169\u11AB", "\uB860", "\u1105\u1169\u11AB" },
            new string [] { "\uB861", "\uB861", "\u1105\u1169\u11AC", "\uB861", "\u1105\u1169\u11AC" },
            new string [] { "\uB862", "\uB862", "\u1105\u1169\u11AD", "\uB862", "\u1105\u1169\u11AD" },
            new string [] { "\uB863", "\uB863", "\u1105\u1169\u11AE", "\uB863", "\u1105\u1169\u11AE" },
            new string [] { "\uB864", "\uB864", "\u1105\u1169\u11AF", "\uB864", "\u1105\u1169\u11AF" },
            new string [] { "\uB865", "\uB865", "\u1105\u1169\u11B0", "\uB865", "\u1105\u1169\u11B0" },
            new string [] { "\uB866", "\uB866", "\u1105\u1169\u11B1", "\uB866", "\u1105\u1169\u11B1" },
            new string [] { "\uB867", "\uB867", "\u1105\u1169\u11B2", "\uB867", "\u1105\u1169\u11B2" },
            new string [] { "\uB868", "\uB868", "\u1105\u1169\u11B3", "\uB868", "\u1105\u1169\u11B3" },
            new string [] { "\uB869", "\uB869", "\u1105\u1169\u11B4", "\uB869", "\u1105\u1169\u11B4" },
            new string [] { "\uB86A", "\uB86A", "\u1105\u1169\u11B5", "\uB86A", "\u1105\u1169\u11B5" },
            new string [] { "\uB86B", "\uB86B", "\u1105\u1169\u11B6", "\uB86B", "\u1105\u1169\u11B6" },
            new string [] { "\uB86C", "\uB86C", "\u1105\u1169\u11B7", "\uB86C", "\u1105\u1169\u11B7" },
            new string [] { "\uB86D", "\uB86D", "\u1105\u1169\u11B8", "\uB86D", "\u1105\u1169\u11B8" },
            new string [] { "\uB86E", "\uB86E", "\u1105\u1169\u11B9", "\uB86E", "\u1105\u1169\u11B9" },
            new string [] { "\uB86F", "\uB86F", "\u1105\u1169\u11BA", "\uB86F", "\u1105\u1169\u11BA" },
            new string [] { "\uB870", "\uB870", "\u1105\u1169\u11BB", "\uB870", "\u1105\u1169\u11BB" },
            new string [] { "\uB871", "\uB871", "\u1105\u1169\u11BC", "\uB871", "\u1105\u1169\u11BC" },
            new string [] { "\uB872", "\uB872", "\u1105\u1169\u11BD", "\uB872", "\u1105\u1169\u11BD" },
            new string [] { "\uB873", "\uB873", "\u1105\u1169\u11BE", "\uB873", "\u1105\u1169\u11BE" },
            new string [] { "\uB874", "\uB874", "\u1105\u1169\u11BF", "\uB874", "\u1105\u1169\u11BF" },
            new string [] { "\uB875", "\uB875", "\u1105\u1169\u11C0", "\uB875", "\u1105\u1169\u11C0" },
            new string [] { "\uB876", "\uB876", "\u1105\u1169\u11C1", "\uB876", "\u1105\u1169\u11C1" },
            new string [] { "\uB877", "\uB877", "\u1105\u1169\u11C2", "\uB877", "\u1105\u1169\u11C2" },
            new string [] { "\uB878", "\uB878", "\u1105\u116A", "\uB878", "\u1105\u116A" },
            new string [] { "\uB879", "\uB879", "\u1105\u116A\u11A8", "\uB879", "\u1105\u116A\u11A8" },
            new string [] { "\uB87A", "\uB87A", "\u1105\u116A\u11A9", "\uB87A", "\u1105\u116A\u11A9" },
            new string [] { "\uB87B", "\uB87B", "\u1105\u116A\u11AA", "\uB87B", "\u1105\u116A\u11AA" },
            new string [] { "\uB87C", "\uB87C", "\u1105\u116A\u11AB", "\uB87C", "\u1105\u116A\u11AB" },
            new string [] { "\uB87D", "\uB87D", "\u1105\u116A\u11AC", "\uB87D", "\u1105\u116A\u11AC" },
            new string [] { "\uB87E", "\uB87E", "\u1105\u116A\u11AD", "\uB87E", "\u1105\u116A\u11AD" },
            new string [] { "\uB87F", "\uB87F", "\u1105\u116A\u11AE", "\uB87F", "\u1105\u116A\u11AE" },
            new string [] { "\uB880", "\uB880", "\u1105\u116A\u11AF", "\uB880", "\u1105\u116A\u11AF" },
            new string [] { "\uB881", "\uB881", "\u1105\u116A\u11B0", "\uB881", "\u1105\u116A\u11B0" },
            new string [] { "\uB882", "\uB882", "\u1105\u116A\u11B1", "\uB882", "\u1105\u116A\u11B1" },
            new string [] { "\uB883", "\uB883", "\u1105\u116A\u11B2", "\uB883", "\u1105\u116A\u11B2" },
            new string [] { "\uB884", "\uB884", "\u1105\u116A\u11B3", "\uB884", "\u1105\u116A\u11B3" },
            new string [] { "\uB885", "\uB885", "\u1105\u116A\u11B4", "\uB885", "\u1105\u116A\u11B4" },
            new string [] { "\uB886", "\uB886", "\u1105\u116A\u11B5", "\uB886", "\u1105\u116A\u11B5" },
            new string [] { "\uB887", "\uB887", "\u1105\u116A\u11B6", "\uB887", "\u1105\u116A\u11B6" },
            new string [] { "\uB888", "\uB888", "\u1105\u116A\u11B7", "\uB888", "\u1105\u116A\u11B7" },
            new string [] { "\uB889", "\uB889", "\u1105\u116A\u11B8", "\uB889", "\u1105\u116A\u11B8" },
            new string [] { "\uB88A", "\uB88A", "\u1105\u116A\u11B9", "\uB88A", "\u1105\u116A\u11B9" },
            new string [] { "\uB88B", "\uB88B", "\u1105\u116A\u11BA", "\uB88B", "\u1105\u116A\u11BA" },
            new string [] { "\uB88C", "\uB88C", "\u1105\u116A\u11BB", "\uB88C", "\u1105\u116A\u11BB" },
            new string [] { "\uB88D", "\uB88D", "\u1105\u116A\u11BC", "\uB88D", "\u1105\u116A\u11BC" },
            new string [] { "\uB88E", "\uB88E", "\u1105\u116A\u11BD", "\uB88E", "\u1105\u116A\u11BD" },
            new string [] { "\uB88F", "\uB88F", "\u1105\u116A\u11BE", "\uB88F", "\u1105\u116A\u11BE" },
            new string [] { "\uB890", "\uB890", "\u1105\u116A\u11BF", "\uB890", "\u1105\u116A\u11BF" },
            new string [] { "\uB891", "\uB891", "\u1105\u116A\u11C0", "\uB891", "\u1105\u116A\u11C0" },
            new string [] { "\uB892", "\uB892", "\u1105\u116A\u11C1", "\uB892", "\u1105\u116A\u11C1" },
            new string [] { "\uB893", "\uB893", "\u1105\u116A\u11C2", "\uB893", "\u1105\u116A\u11C2" },
            new string [] { "\uB894", "\uB894", "\u1105\u116B", "\uB894", "\u1105\u116B" },
            new string [] { "\uB895", "\uB895", "\u1105\u116B\u11A8", "\uB895", "\u1105\u116B\u11A8" },
            new string [] { "\uB896", "\uB896", "\u1105\u116B\u11A9", "\uB896", "\u1105\u116B\u11A9" },
            new string [] { "\uB897", "\uB897", "\u1105\u116B\u11AA", "\uB897", "\u1105\u116B\u11AA" },
            new string [] { "\uB898", "\uB898", "\u1105\u116B\u11AB", "\uB898", "\u1105\u116B\u11AB" },
            new string [] { "\uB899", "\uB899", "\u1105\u116B\u11AC", "\uB899", "\u1105\u116B\u11AC" },
            new string [] { "\uB89A", "\uB89A", "\u1105\u116B\u11AD", "\uB89A", "\u1105\u116B\u11AD" },
            new string [] { "\uB89B", "\uB89B", "\u1105\u116B\u11AE", "\uB89B", "\u1105\u116B\u11AE" },
            new string [] { "\uB89C", "\uB89C", "\u1105\u116B\u11AF", "\uB89C", "\u1105\u116B\u11AF" },
            new string [] { "\uB89D", "\uB89D", "\u1105\u116B\u11B0", "\uB89D", "\u1105\u116B\u11B0" },
            new string [] { "\uB89E", "\uB89E", "\u1105\u116B\u11B1", "\uB89E", "\u1105\u116B\u11B1" },
            new string [] { "\uB89F", "\uB89F", "\u1105\u116B\u11B2", "\uB89F", "\u1105\u116B\u11B2" },
            new string [] { "\uB8A0", "\uB8A0", "\u1105\u116B\u11B3", "\uB8A0", "\u1105\u116B\u11B3" },
            new string [] { "\uB8A1", "\uB8A1", "\u1105\u116B\u11B4", "\uB8A1", "\u1105\u116B\u11B4" },
            new string [] { "\uB8A2", "\uB8A2", "\u1105\u116B\u11B5", "\uB8A2", "\u1105\u116B\u11B5" },
            new string [] { "\uB8A3", "\uB8A3", "\u1105\u116B\u11B6", "\uB8A3", "\u1105\u116B\u11B6" },
            new string [] { "\uB8A4", "\uB8A4", "\u1105\u116B\u11B7", "\uB8A4", "\u1105\u116B\u11B7" },
            new string [] { "\uB8A5", "\uB8A5", "\u1105\u116B\u11B8", "\uB8A5", "\u1105\u116B\u11B8" },
            new string [] { "\uB8A6", "\uB8A6", "\u1105\u116B\u11B9", "\uB8A6", "\u1105\u116B\u11B9" },
            new string [] { "\uB8A7", "\uB8A7", "\u1105\u116B\u11BA", "\uB8A7", "\u1105\u116B\u11BA" },
            new string [] { "\uB8A8", "\uB8A8", "\u1105\u116B\u11BB", "\uB8A8", "\u1105\u116B\u11BB" },
            new string [] { "\uB8A9", "\uB8A9", "\u1105\u116B\u11BC", "\uB8A9", "\u1105\u116B\u11BC" },
            new string [] { "\uB8AA", "\uB8AA", "\u1105\u116B\u11BD", "\uB8AA", "\u1105\u116B\u11BD" },
            new string [] { "\uB8AB", "\uB8AB", "\u1105\u116B\u11BE", "\uB8AB", "\u1105\u116B\u11BE" },
            new string [] { "\uB8AC", "\uB8AC", "\u1105\u116B\u11BF", "\uB8AC", "\u1105\u116B\u11BF" },
            new string [] { "\uB8AD", "\uB8AD", "\u1105\u116B\u11C0", "\uB8AD", "\u1105\u116B\u11C0" },
            new string [] { "\uB8AE", "\uB8AE", "\u1105\u116B\u11C1", "\uB8AE", "\u1105\u116B\u11C1" },
            new string [] { "\uB8AF", "\uB8AF", "\u1105\u116B\u11C2", "\uB8AF", "\u1105\u116B\u11C2" },
            new string [] { "\uB8B0", "\uB8B0", "\u1105\u116C", "\uB8B0", "\u1105\u116C" },
            new string [] { "\uB8B1", "\uB8B1", "\u1105\u116C\u11A8", "\uB8B1", "\u1105\u116C\u11A8" },
            new string [] { "\uB8B2", "\uB8B2", "\u1105\u116C\u11A9", "\uB8B2", "\u1105\u116C\u11A9" },
            new string [] { "\uB8B3", "\uB8B3", "\u1105\u116C\u11AA", "\uB8B3", "\u1105\u116C\u11AA" },
            new string [] { "\uB8B4", "\uB8B4", "\u1105\u116C\u11AB", "\uB8B4", "\u1105\u116C\u11AB" },
            new string [] { "\uB8B5", "\uB8B5", "\u1105\u116C\u11AC", "\uB8B5", "\u1105\u116C\u11AC" },
            new string [] { "\uB8B6", "\uB8B6", "\u1105\u116C\u11AD", "\uB8B6", "\u1105\u116C\u11AD" },
            new string [] { "\uB8B7", "\uB8B7", "\u1105\u116C\u11AE", "\uB8B7", "\u1105\u116C\u11AE" },
            new string [] { "\uB8B8", "\uB8B8", "\u1105\u116C\u11AF", "\uB8B8", "\u1105\u116C\u11AF" },
            new string [] { "\uB8B9", "\uB8B9", "\u1105\u116C\u11B0", "\uB8B9", "\u1105\u116C\u11B0" },
            new string [] { "\uB8BA", "\uB8BA", "\u1105\u116C\u11B1", "\uB8BA", "\u1105\u116C\u11B1" },
            new string [] { "\uB8BB", "\uB8BB", "\u1105\u116C\u11B2", "\uB8BB", "\u1105\u116C\u11B2" },
            new string [] { "\uB8BC", "\uB8BC", "\u1105\u116C\u11B3", "\uB8BC", "\u1105\u116C\u11B3" },
            new string [] { "\uB8BD", "\uB8BD", "\u1105\u116C\u11B4", "\uB8BD", "\u1105\u116C\u11B4" },
            new string [] { "\uB8BE", "\uB8BE", "\u1105\u116C\u11B5", "\uB8BE", "\u1105\u116C\u11B5" },
            new string [] { "\uB8BF", "\uB8BF", "\u1105\u116C\u11B6", "\uB8BF", "\u1105\u116C\u11B6" },
            new string [] { "\uB8C0", "\uB8C0", "\u1105\u116C\u11B7", "\uB8C0", "\u1105\u116C\u11B7" },
            new string [] { "\uB8C1", "\uB8C1", "\u1105\u116C\u11B8", "\uB8C1", "\u1105\u116C\u11B8" },
            new string [] { "\uB8C2", "\uB8C2", "\u1105\u116C\u11B9", "\uB8C2", "\u1105\u116C\u11B9" },
            new string [] { "\uB8C3", "\uB8C3", "\u1105\u116C\u11BA", "\uB8C3", "\u1105\u116C\u11BA" },
            new string [] { "\uB8C4", "\uB8C4", "\u1105\u116C\u11BB", "\uB8C4", "\u1105\u116C\u11BB" },
            new string [] { "\uB8C5", "\uB8C5", "\u1105\u116C\u11BC", "\uB8C5", "\u1105\u116C\u11BC" },
            new string [] { "\uB8C6", "\uB8C6", "\u1105\u116C\u11BD", "\uB8C6", "\u1105\u116C\u11BD" },
            new string [] { "\uB8C7", "\uB8C7", "\u1105\u116C\u11BE", "\uB8C7", "\u1105\u116C\u11BE" },
            new string [] { "\uB8C8", "\uB8C8", "\u1105\u116C\u11BF", "\uB8C8", "\u1105\u116C\u11BF" },
            new string [] { "\uB8C9", "\uB8C9", "\u1105\u116C\u11C0", "\uB8C9", "\u1105\u116C\u11C0" },
            new string [] { "\uB8CA", "\uB8CA", "\u1105\u116C\u11C1", "\uB8CA", "\u1105\u116C\u11C1" },
            new string [] { "\uB8CB", "\uB8CB", "\u1105\u116C\u11C2", "\uB8CB", "\u1105\u116C\u11C2" },
            new string [] { "\uB8CC", "\uB8CC", "\u1105\u116D", "\uB8CC", "\u1105\u116D" },
            new string [] { "\uB8CD", "\uB8CD", "\u1105\u116D\u11A8", "\uB8CD", "\u1105\u116D\u11A8" },
            new string [] { "\uB8CE", "\uB8CE", "\u1105\u116D\u11A9", "\uB8CE", "\u1105\u116D\u11A9" },
            new string [] { "\uB8CF", "\uB8CF", "\u1105\u116D\u11AA", "\uB8CF", "\u1105\u116D\u11AA" },
            new string [] { "\uB8D0", "\uB8D0", "\u1105\u116D\u11AB", "\uB8D0", "\u1105\u116D\u11AB" },
            new string [] { "\uB8D1", "\uB8D1", "\u1105\u116D\u11AC", "\uB8D1", "\u1105\u116D\u11AC" },
            new string [] { "\uB8D2", "\uB8D2", "\u1105\u116D\u11AD", "\uB8D2", "\u1105\u116D\u11AD" },
            new string [] { "\uB8D3", "\uB8D3", "\u1105\u116D\u11AE", "\uB8D3", "\u1105\u116D\u11AE" },
            new string [] { "\uB8D4", "\uB8D4", "\u1105\u116D\u11AF", "\uB8D4", "\u1105\u116D\u11AF" },
            new string [] { "\uB8D5", "\uB8D5", "\u1105\u116D\u11B0", "\uB8D5", "\u1105\u116D\u11B0" },
            new string [] { "\uB8D6", "\uB8D6", "\u1105\u116D\u11B1", "\uB8D6", "\u1105\u116D\u11B1" },
            new string [] { "\uB8D7", "\uB8D7", "\u1105\u116D\u11B2", "\uB8D7", "\u1105\u116D\u11B2" },
            new string [] { "\uB8D8", "\uB8D8", "\u1105\u116D\u11B3", "\uB8D8", "\u1105\u116D\u11B3" },
            new string [] { "\uB8D9", "\uB8D9", "\u1105\u116D\u11B4", "\uB8D9", "\u1105\u116D\u11B4" },
            new string [] { "\uB8DA", "\uB8DA", "\u1105\u116D\u11B5", "\uB8DA", "\u1105\u116D\u11B5" },
            new string [] { "\uB8DB", "\uB8DB", "\u1105\u116D\u11B6", "\uB8DB", "\u1105\u116D\u11B6" },
            new string [] { "\uB8DC", "\uB8DC", "\u1105\u116D\u11B7", "\uB8DC", "\u1105\u116D\u11B7" },
            new string [] { "\uB8DD", "\uB8DD", "\u1105\u116D\u11B8", "\uB8DD", "\u1105\u116D\u11B8" },
            new string [] { "\uB8DE", "\uB8DE", "\u1105\u116D\u11B9", "\uB8DE", "\u1105\u116D\u11B9" },
            new string [] { "\uB8DF", "\uB8DF", "\u1105\u116D\u11BA", "\uB8DF", "\u1105\u116D\u11BA" },
            new string [] { "\uB8E0", "\uB8E0", "\u1105\u116D\u11BB", "\uB8E0", "\u1105\u116D\u11BB" },
            new string [] { "\uB8E1", "\uB8E1", "\u1105\u116D\u11BC", "\uB8E1", "\u1105\u116D\u11BC" },
            new string [] { "\uB8E2", "\uB8E2", "\u1105\u116D\u11BD", "\uB8E2", "\u1105\u116D\u11BD" },
            new string [] { "\uB8E3", "\uB8E3", "\u1105\u116D\u11BE", "\uB8E3", "\u1105\u116D\u11BE" },
            new string [] { "\uB8E4", "\uB8E4", "\u1105\u116D\u11BF", "\uB8E4", "\u1105\u116D\u11BF" },
            new string [] { "\uB8E5", "\uB8E5", "\u1105\u116D\u11C0", "\uB8E5", "\u1105\u116D\u11C0" },
            new string [] { "\uB8E6", "\uB8E6", "\u1105\u116D\u11C1", "\uB8E6", "\u1105\u116D\u11C1" },
            new string [] { "\uB8E7", "\uB8E7", "\u1105\u116D\u11C2", "\uB8E7", "\u1105\u116D\u11C2" },
            new string [] { "\uB8E8", "\uB8E8", "\u1105\u116E", "\uB8E8", "\u1105\u116E" },
            new string [] { "\uB8E9", "\uB8E9", "\u1105\u116E\u11A8", "\uB8E9", "\u1105\u116E\u11A8" },
            new string [] { "\uB8EA", "\uB8EA", "\u1105\u116E\u11A9", "\uB8EA", "\u1105\u116E\u11A9" },
            new string [] { "\uB8EB", "\uB8EB", "\u1105\u116E\u11AA", "\uB8EB", "\u1105\u116E\u11AA" },
            new string [] { "\uB8EC", "\uB8EC", "\u1105\u116E\u11AB", "\uB8EC", "\u1105\u116E\u11AB" },
            new string [] { "\uB8ED", "\uB8ED", "\u1105\u116E\u11AC", "\uB8ED", "\u1105\u116E\u11AC" },
            new string [] { "\uB8EE", "\uB8EE", "\u1105\u116E\u11AD", "\uB8EE", "\u1105\u116E\u11AD" },
            new string [] { "\uB8EF", "\uB8EF", "\u1105\u116E\u11AE", "\uB8EF", "\u1105\u116E\u11AE" },
            new string [] { "\uB8F0", "\uB8F0", "\u1105\u116E\u11AF", "\uB8F0", "\u1105\u116E\u11AF" },
            new string [] { "\uB8F1", "\uB8F1", "\u1105\u116E\u11B0", "\uB8F1", "\u1105\u116E\u11B0" },
            new string [] { "\uB8F2", "\uB8F2", "\u1105\u116E\u11B1", "\uB8F2", "\u1105\u116E\u11B1" },
            new string [] { "\uB8F3", "\uB8F3", "\u1105\u116E\u11B2", "\uB8F3", "\u1105\u116E\u11B2" },
            new string [] { "\uB8F4", "\uB8F4", "\u1105\u116E\u11B3", "\uB8F4", "\u1105\u116E\u11B3" },
            new string [] { "\uB8F5", "\uB8F5", "\u1105\u116E\u11B4", "\uB8F5", "\u1105\u116E\u11B4" },
            new string [] { "\uB8F6", "\uB8F6", "\u1105\u116E\u11B5", "\uB8F6", "\u1105\u116E\u11B5" },
            new string [] { "\uB8F7", "\uB8F7", "\u1105\u116E\u11B6", "\uB8F7", "\u1105\u116E\u11B6" },
            new string [] { "\uB8F8", "\uB8F8", "\u1105\u116E\u11B7", "\uB8F8", "\u1105\u116E\u11B7" },
            new string [] { "\uB8F9", "\uB8F9", "\u1105\u116E\u11B8", "\uB8F9", "\u1105\u116E\u11B8" },
            new string [] { "\uB8FA", "\uB8FA", "\u1105\u116E\u11B9", "\uB8FA", "\u1105\u116E\u11B9" },
            new string [] { "\uB8FB", "\uB8FB", "\u1105\u116E\u11BA", "\uB8FB", "\u1105\u116E\u11BA" },
            new string [] { "\uB8FC", "\uB8FC", "\u1105\u116E\u11BB", "\uB8FC", "\u1105\u116E\u11BB" },
            new string [] { "\uB8FD", "\uB8FD", "\u1105\u116E\u11BC", "\uB8FD", "\u1105\u116E\u11BC" },
            new string [] { "\uB8FE", "\uB8FE", "\u1105\u116E\u11BD", "\uB8FE", "\u1105\u116E\u11BD" },
            new string [] { "\uB8FF", "\uB8FF", "\u1105\u116E\u11BE", "\uB8FF", "\u1105\u116E\u11BE" },
            new string [] { "\uB900", "\uB900", "\u1105\u116E\u11BF", "\uB900", "\u1105\u116E\u11BF" },
            new string [] { "\uB901", "\uB901", "\u1105\u116E\u11C0", "\uB901", "\u1105\u116E\u11C0" },
            new string [] { "\uB902", "\uB902", "\u1105\u116E\u11C1", "\uB902", "\u1105\u116E\u11C1" },
            new string [] { "\uB903", "\uB903", "\u1105\u116E\u11C2", "\uB903", "\u1105\u116E\u11C2" },
            new string [] { "\uB904", "\uB904", "\u1105\u116F", "\uB904", "\u1105\u116F" },
            new string [] { "\uB905", "\uB905", "\u1105\u116F\u11A8", "\uB905", "\u1105\u116F\u11A8" },
            new string [] { "\uB906", "\uB906", "\u1105\u116F\u11A9", "\uB906", "\u1105\u116F\u11A9" },
            new string [] { "\uB907", "\uB907", "\u1105\u116F\u11AA", "\uB907", "\u1105\u116F\u11AA" },
            new string [] { "\uB908", "\uB908", "\u1105\u116F\u11AB", "\uB908", "\u1105\u116F\u11AB" },
            new string [] { "\uB909", "\uB909", "\u1105\u116F\u11AC", "\uB909", "\u1105\u116F\u11AC" },
            new string [] { "\uB90A", "\uB90A", "\u1105\u116F\u11AD", "\uB90A", "\u1105\u116F\u11AD" },
            new string [] { "\uB90B", "\uB90B", "\u1105\u116F\u11AE", "\uB90B", "\u1105\u116F\u11AE" },
            new string [] { "\uB90C", "\uB90C", "\u1105\u116F\u11AF", "\uB90C", "\u1105\u116F\u11AF" },
            new string [] { "\uB90D", "\uB90D", "\u1105\u116F\u11B0", "\uB90D", "\u1105\u116F\u11B0" },
            new string [] { "\uB90E", "\uB90E", "\u1105\u116F\u11B1", "\uB90E", "\u1105\u116F\u11B1" },
            new string [] { "\uB90F", "\uB90F", "\u1105\u116F\u11B2", "\uB90F", "\u1105\u116F\u11B2" },
            new string [] { "\uB910", "\uB910", "\u1105\u116F\u11B3", "\uB910", "\u1105\u116F\u11B3" },
            new string [] { "\uB911", "\uB911", "\u1105\u116F\u11B4", "\uB911", "\u1105\u116F\u11B4" },
            new string [] { "\uB912", "\uB912", "\u1105\u116F\u11B5", "\uB912", "\u1105\u116F\u11B5" },
            new string [] { "\uB913", "\uB913", "\u1105\u116F\u11B6", "\uB913", "\u1105\u116F\u11B6" },
            new string [] { "\uB914", "\uB914", "\u1105\u116F\u11B7", "\uB914", "\u1105\u116F\u11B7" },
            new string [] { "\uB915", "\uB915", "\u1105\u116F\u11B8", "\uB915", "\u1105\u116F\u11B8" },
            new string [] { "\uB916", "\uB916", "\u1105\u116F\u11B9", "\uB916", "\u1105\u116F\u11B9" },
            new string [] { "\uB917", "\uB917", "\u1105\u116F\u11BA", "\uB917", "\u1105\u116F\u11BA" },
            new string [] { "\uB918", "\uB918", "\u1105\u116F\u11BB", "\uB918", "\u1105\u116F\u11BB" },
            new string [] { "\uB919", "\uB919", "\u1105\u116F\u11BC", "\uB919", "\u1105\u116F\u11BC" },
            new string [] { "\uB91A", "\uB91A", "\u1105\u116F\u11BD", "\uB91A", "\u1105\u116F\u11BD" },
            new string [] { "\uB91B", "\uB91B", "\u1105\u116F\u11BE", "\uB91B", "\u1105\u116F\u11BE" },
            new string [] { "\uB91C", "\uB91C", "\u1105\u116F\u11BF", "\uB91C", "\u1105\u116F\u11BF" },
            new string [] { "\uB91D", "\uB91D", "\u1105\u116F\u11C0", "\uB91D", "\u1105\u116F\u11C0" },
            new string [] { "\uB91E", "\uB91E", "\u1105\u116F\u11C1", "\uB91E", "\u1105\u116F\u11C1" },
            new string [] { "\uB91F", "\uB91F", "\u1105\u116F\u11C2", "\uB91F", "\u1105\u116F\u11C2" },
            new string [] { "\uB920", "\uB920", "\u1105\u1170", "\uB920", "\u1105\u1170" },
            new string [] { "\uB921", "\uB921", "\u1105\u1170\u11A8", "\uB921", "\u1105\u1170\u11A8" },
            new string [] { "\uB922", "\uB922", "\u1105\u1170\u11A9", "\uB922", "\u1105\u1170\u11A9" },
            new string [] { "\uB923", "\uB923", "\u1105\u1170\u11AA", "\uB923", "\u1105\u1170\u11AA" },
            new string [] { "\uB924", "\uB924", "\u1105\u1170\u11AB", "\uB924", "\u1105\u1170\u11AB" },
            new string [] { "\uB925", "\uB925", "\u1105\u1170\u11AC", "\uB925", "\u1105\u1170\u11AC" },
            new string [] { "\uB926", "\uB926", "\u1105\u1170\u11AD", "\uB926", "\u1105\u1170\u11AD" },
            new string [] { "\uB927", "\uB927", "\u1105\u1170\u11AE", "\uB927", "\u1105\u1170\u11AE" },
            new string [] { "\uB928", "\uB928", "\u1105\u1170\u11AF", "\uB928", "\u1105\u1170\u11AF" },
            new string [] { "\uB929", "\uB929", "\u1105\u1170\u11B0", "\uB929", "\u1105\u1170\u11B0" },
            new string [] { "\uB92A", "\uB92A", "\u1105\u1170\u11B1", "\uB92A", "\u1105\u1170\u11B1" },
            new string [] { "\uB92B", "\uB92B", "\u1105\u1170\u11B2", "\uB92B", "\u1105\u1170\u11B2" },
            new string [] { "\uB92C", "\uB92C", "\u1105\u1170\u11B3", "\uB92C", "\u1105\u1170\u11B3" },
            new string [] { "\uB92D", "\uB92D", "\u1105\u1170\u11B4", "\uB92D", "\u1105\u1170\u11B4" },
            new string [] { "\uB92E", "\uB92E", "\u1105\u1170\u11B5", "\uB92E", "\u1105\u1170\u11B5" },
            new string [] { "\uB92F", "\uB92F", "\u1105\u1170\u11B6", "\uB92F", "\u1105\u1170\u11B6" },
            new string [] { "\uB930", "\uB930", "\u1105\u1170\u11B7", "\uB930", "\u1105\u1170\u11B7" },
            new string [] { "\uB931", "\uB931", "\u1105\u1170\u11B8", "\uB931", "\u1105\u1170\u11B8" },
            new string [] { "\uB932", "\uB932", "\u1105\u1170\u11B9", "\uB932", "\u1105\u1170\u11B9" },
            new string [] { "\uB933", "\uB933", "\u1105\u1170\u11BA", "\uB933", "\u1105\u1170\u11BA" },
            new string [] { "\uB934", "\uB934", "\u1105\u1170\u11BB", "\uB934", "\u1105\u1170\u11BB" },
            new string [] { "\uB935", "\uB935", "\u1105\u1170\u11BC", "\uB935", "\u1105\u1170\u11BC" },
            new string [] { "\uB936", "\uB936", "\u1105\u1170\u11BD", "\uB936", "\u1105\u1170\u11BD" },
            new string [] { "\uB937", "\uB937", "\u1105\u1170\u11BE", "\uB937", "\u1105\u1170\u11BE" },
            new string [] { "\uB938", "\uB938", "\u1105\u1170\u11BF", "\uB938", "\u1105\u1170\u11BF" },
            new string [] { "\uB939", "\uB939", "\u1105\u1170\u11C0", "\uB939", "\u1105\u1170\u11C0" },
            new string [] { "\uB93A", "\uB93A", "\u1105\u1170\u11C1", "\uB93A", "\u1105\u1170\u11C1" },
            new string [] { "\uB93B", "\uB93B", "\u1105\u1170\u11C2", "\uB93B", "\u1105\u1170\u11C2" },
            new string [] { "\uB93C", "\uB93C", "\u1105\u1171", "\uB93C", "\u1105\u1171" },
            new string [] { "\uB93D", "\uB93D", "\u1105\u1171\u11A8", "\uB93D", "\u1105\u1171\u11A8" },
            new string [] { "\uB93E", "\uB93E", "\u1105\u1171\u11A9", "\uB93E", "\u1105\u1171\u11A9" },
            new string [] { "\uB93F", "\uB93F", "\u1105\u1171\u11AA", "\uB93F", "\u1105\u1171\u11AA" },
            new string [] { "\uB940", "\uB940", "\u1105\u1171\u11AB", "\uB940", "\u1105\u1171\u11AB" },
            new string [] { "\uB941", "\uB941", "\u1105\u1171\u11AC", "\uB941", "\u1105\u1171\u11AC" },
            new string [] { "\uB942", "\uB942", "\u1105\u1171\u11AD", "\uB942", "\u1105\u1171\u11AD" },
            new string [] { "\uB943", "\uB943", "\u1105\u1171\u11AE", "\uB943", "\u1105\u1171\u11AE" },
            new string [] { "\uB944", "\uB944", "\u1105\u1171\u11AF", "\uB944", "\u1105\u1171\u11AF" },
            new string [] { "\uB945", "\uB945", "\u1105\u1171\u11B0", "\uB945", "\u1105\u1171\u11B0" },
            new string [] { "\uB946", "\uB946", "\u1105\u1171\u11B1", "\uB946", "\u1105\u1171\u11B1" },
            new string [] { "\uB947", "\uB947", "\u1105\u1171\u11B2", "\uB947", "\u1105\u1171\u11B2" },
            new string [] { "\uB948", "\uB948", "\u1105\u1171\u11B3", "\uB948", "\u1105\u1171\u11B3" },
            new string [] { "\uB949", "\uB949", "\u1105\u1171\u11B4", "\uB949", "\u1105\u1171\u11B4" },
            new string [] { "\uB94A", "\uB94A", "\u1105\u1171\u11B5", "\uB94A", "\u1105\u1171\u11B5" },
            new string [] { "\uB94B", "\uB94B", "\u1105\u1171\u11B6", "\uB94B", "\u1105\u1171\u11B6" },
            new string [] { "\uB94C", "\uB94C", "\u1105\u1171\u11B7", "\uB94C", "\u1105\u1171\u11B7" },
            new string [] { "\uB94D", "\uB94D", "\u1105\u1171\u11B8", "\uB94D", "\u1105\u1171\u11B8" },
            new string [] { "\uB94E", "\uB94E", "\u1105\u1171\u11B9", "\uB94E", "\u1105\u1171\u11B9" },
            new string [] { "\uB94F", "\uB94F", "\u1105\u1171\u11BA", "\uB94F", "\u1105\u1171\u11BA" },
            new string [] { "\uB950", "\uB950", "\u1105\u1171\u11BB", "\uB950", "\u1105\u1171\u11BB" },
            new string [] { "\uB951", "\uB951", "\u1105\u1171\u11BC", "\uB951", "\u1105\u1171\u11BC" },
            new string [] { "\uB952", "\uB952", "\u1105\u1171\u11BD", "\uB952", "\u1105\u1171\u11BD" },
            new string [] { "\uB953", "\uB953", "\u1105\u1171\u11BE", "\uB953", "\u1105\u1171\u11BE" },
            new string [] { "\uB954", "\uB954", "\u1105\u1171\u11BF", "\uB954", "\u1105\u1171\u11BF" },
            new string [] { "\uB955", "\uB955", "\u1105\u1171\u11C0", "\uB955", "\u1105\u1171\u11C0" },
            new string [] { "\uB956", "\uB956", "\u1105\u1171\u11C1", "\uB956", "\u1105\u1171\u11C1" },
            new string [] { "\uB957", "\uB957", "\u1105\u1171\u11C2", "\uB957", "\u1105\u1171\u11C2" },
            new string [] { "\uB958", "\uB958", "\u1105\u1172", "\uB958", "\u1105\u1172" },
            new string [] { "\uB959", "\uB959", "\u1105\u1172\u11A8", "\uB959", "\u1105\u1172\u11A8" },
            new string [] { "\uB95A", "\uB95A", "\u1105\u1172\u11A9", "\uB95A", "\u1105\u1172\u11A9" },
            new string [] { "\uB95B", "\uB95B", "\u1105\u1172\u11AA", "\uB95B", "\u1105\u1172\u11AA" },
            new string [] { "\uB95C", "\uB95C", "\u1105\u1172\u11AB", "\uB95C", "\u1105\u1172\u11AB" },
            new string [] { "\uB95D", "\uB95D", "\u1105\u1172\u11AC", "\uB95D", "\u1105\u1172\u11AC" },
            new string [] { "\uB95E", "\uB95E", "\u1105\u1172\u11AD", "\uB95E", "\u1105\u1172\u11AD" },
            new string [] { "\uB95F", "\uB95F", "\u1105\u1172\u11AE", "\uB95F", "\u1105\u1172\u11AE" },
            new string [] { "\uB960", "\uB960", "\u1105\u1172\u11AF", "\uB960", "\u1105\u1172\u11AF" },
            new string [] { "\uB961", "\uB961", "\u1105\u1172\u11B0", "\uB961", "\u1105\u1172\u11B0" },
            new string [] { "\uB962", "\uB962", "\u1105\u1172\u11B1", "\uB962", "\u1105\u1172\u11B1" },
            new string [] { "\uB963", "\uB963", "\u1105\u1172\u11B2", "\uB963", "\u1105\u1172\u11B2" },
            new string [] { "\uB964", "\uB964", "\u1105\u1172\u11B3", "\uB964", "\u1105\u1172\u11B3" },
            new string [] { "\uB965", "\uB965", "\u1105\u1172\u11B4", "\uB965", "\u1105\u1172\u11B4" },
            new string [] { "\uB966", "\uB966", "\u1105\u1172\u11B5", "\uB966", "\u1105\u1172\u11B5" },
            new string [] { "\uB967", "\uB967", "\u1105\u1172\u11B6", "\uB967", "\u1105\u1172\u11B6" },
            new string [] { "\uB968", "\uB968", "\u1105\u1172\u11B7", "\uB968", "\u1105\u1172\u11B7" },
            new string [] { "\uB969", "\uB969", "\u1105\u1172\u11B8", "\uB969", "\u1105\u1172\u11B8" },
            new string [] { "\uB96A", "\uB96A", "\u1105\u1172\u11B9", "\uB96A", "\u1105\u1172\u11B9" },
            new string [] { "\uB96B", "\uB96B", "\u1105\u1172\u11BA", "\uB96B", "\u1105\u1172\u11BA" },
            new string [] { "\uB96C", "\uB96C", "\u1105\u1172\u11BB", "\uB96C", "\u1105\u1172\u11BB" },
            new string [] { "\uB96D", "\uB96D", "\u1105\u1172\u11BC", "\uB96D", "\u1105\u1172\u11BC" },
            new string [] { "\uB96E", "\uB96E", "\u1105\u1172\u11BD", "\uB96E", "\u1105\u1172\u11BD" },
            new string [] { "\uB96F", "\uB96F", "\u1105\u1172\u11BE", "\uB96F", "\u1105\u1172\u11BE" },
            new string [] { "\uB970", "\uB970", "\u1105\u1172\u11BF", "\uB970", "\u1105\u1172\u11BF" },
            new string [] { "\uB971", "\uB971", "\u1105\u1172\u11C0", "\uB971", "\u1105\u1172\u11C0" },
            new string [] { "\uB972", "\uB972", "\u1105\u1172\u11C1", "\uB972", "\u1105\u1172\u11C1" },
            new string [] { "\uB973", "\uB973", "\u1105\u1172\u11C2", "\uB973", "\u1105\u1172\u11C2" },
            new string [] { "\uB974", "\uB974", "\u1105\u1173", "\uB974", "\u1105\u1173" },
            new string [] { "\uB975", "\uB975", "\u1105\u1173\u11A8", "\uB975", "\u1105\u1173\u11A8" },
            new string [] { "\uB976", "\uB976", "\u1105\u1173\u11A9", "\uB976", "\u1105\u1173\u11A9" },
            new string [] { "\uB977", "\uB977", "\u1105\u1173\u11AA", "\uB977", "\u1105\u1173\u11AA" },
            new string [] { "\uB978", "\uB978", "\u1105\u1173\u11AB", "\uB978", "\u1105\u1173\u11AB" },
            new string [] { "\uB979", "\uB979", "\u1105\u1173\u11AC", "\uB979", "\u1105\u1173\u11AC" },
            new string [] { "\uB97A", "\uB97A", "\u1105\u1173\u11AD", "\uB97A", "\u1105\u1173\u11AD" },
            new string [] { "\uB97B", "\uB97B", "\u1105\u1173\u11AE", "\uB97B", "\u1105\u1173\u11AE" },
            new string [] { "\uB97C", "\uB97C", "\u1105\u1173\u11AF", "\uB97C", "\u1105\u1173\u11AF" },
            new string [] { "\uB97D", "\uB97D", "\u1105\u1173\u11B0", "\uB97D", "\u1105\u1173\u11B0" },
            new string [] { "\uB97E", "\uB97E", "\u1105\u1173\u11B1", "\uB97E", "\u1105\u1173\u11B1" },
            new string [] { "\uB97F", "\uB97F", "\u1105\u1173\u11B2", "\uB97F", "\u1105\u1173\u11B2" },
            new string [] { "\uB980", "\uB980", "\u1105\u1173\u11B3", "\uB980", "\u1105\u1173\u11B3" },
            new string [] { "\uB981", "\uB981", "\u1105\u1173\u11B4", "\uB981", "\u1105\u1173\u11B4" },
            new string [] { "\uB982", "\uB982", "\u1105\u1173\u11B5", "\uB982", "\u1105\u1173\u11B5" },
            new string [] { "\uB983", "\uB983", "\u1105\u1173\u11B6", "\uB983", "\u1105\u1173\u11B6" },
            new string [] { "\uB984", "\uB984", "\u1105\u1173\u11B7", "\uB984", "\u1105\u1173\u11B7" },
            new string [] { "\uB985", "\uB985", "\u1105\u1173\u11B8", "\uB985", "\u1105\u1173\u11B8" },
            new string [] { "\uB986", "\uB986", "\u1105\u1173\u11B9", "\uB986", "\u1105\u1173\u11B9" },
            new string [] { "\uB987", "\uB987", "\u1105\u1173\u11BA", "\uB987", "\u1105\u1173\u11BA" },
            new string [] { "\uB988", "\uB988", "\u1105\u1173\u11BB", "\uB988", "\u1105\u1173\u11BB" },
            new string [] { "\uB989", "\uB989", "\u1105\u1173\u11BC", "\uB989", "\u1105\u1173\u11BC" },
            new string [] { "\uB98A", "\uB98A", "\u1105\u1173\u11BD", "\uB98A", "\u1105\u1173\u11BD" },
            new string [] { "\uB98B", "\uB98B", "\u1105\u1173\u11BE", "\uB98B", "\u1105\u1173\u11BE" },
            new string [] { "\uB98C", "\uB98C", "\u1105\u1173\u11BF", "\uB98C", "\u1105\u1173\u11BF" },
            new string [] { "\uB98D", "\uB98D", "\u1105\u1173\u11C0", "\uB98D", "\u1105\u1173\u11C0" },
            new string [] { "\uB98E", "\uB98E", "\u1105\u1173\u11C1", "\uB98E", "\u1105\u1173\u11C1" },
            new string [] { "\uB98F", "\uB98F", "\u1105\u1173\u11C2", "\uB98F", "\u1105\u1173\u11C2" },
            new string [] { "\uB990", "\uB990", "\u1105\u1174", "\uB990", "\u1105\u1174" },
            new string [] { "\uB991", "\uB991", "\u1105\u1174\u11A8", "\uB991", "\u1105\u1174\u11A8" },
            new string [] { "\uB992", "\uB992", "\u1105\u1174\u11A9", "\uB992", "\u1105\u1174\u11A9" },
            new string [] { "\uB993", "\uB993", "\u1105\u1174\u11AA", "\uB993", "\u1105\u1174\u11AA" },
            new string [] { "\uB994", "\uB994", "\u1105\u1174\u11AB", "\uB994", "\u1105\u1174\u11AB" },
            new string [] { "\uB995", "\uB995", "\u1105\u1174\u11AC", "\uB995", "\u1105\u1174\u11AC" },
            new string [] { "\uB996", "\uB996", "\u1105\u1174\u11AD", "\uB996", "\u1105\u1174\u11AD" },
            new string [] { "\uB997", "\uB997", "\u1105\u1174\u11AE", "\uB997", "\u1105\u1174\u11AE" },
            new string [] { "\uB998", "\uB998", "\u1105\u1174\u11AF", "\uB998", "\u1105\u1174\u11AF" },
            new string [] { "\uB999", "\uB999", "\u1105\u1174\u11B0", "\uB999", "\u1105\u1174\u11B0" },
            new string [] { "\uB99A", "\uB99A", "\u1105\u1174\u11B1", "\uB99A", "\u1105\u1174\u11B1" },
            new string [] { "\uB99B", "\uB99B", "\u1105\u1174\u11B2", "\uB99B", "\u1105\u1174\u11B2" },
            new string [] { "\uB99C", "\uB99C", "\u1105\u1174\u11B3", "\uB99C", "\u1105\u1174\u11B3" },
            new string [] { "\uB99D", "\uB99D", "\u1105\u1174\u11B4", "\uB99D", "\u1105\u1174\u11B4" },
            new string [] { "\uB99E", "\uB99E", "\u1105\u1174\u11B5", "\uB99E", "\u1105\u1174\u11B5" },
            new string [] { "\uB99F", "\uB99F", "\u1105\u1174\u11B6", "\uB99F", "\u1105\u1174\u11B6" },
            new string [] { "\uB9A0", "\uB9A0", "\u1105\u1174\u11B7", "\uB9A0", "\u1105\u1174\u11B7" },
            new string [] { "\uB9A1", "\uB9A1", "\u1105\u1174\u11B8", "\uB9A1", "\u1105\u1174\u11B8" },
            new string [] { "\uB9A2", "\uB9A2", "\u1105\u1174\u11B9", "\uB9A2", "\u1105\u1174\u11B9" },
            new string [] { "\uB9A3", "\uB9A3", "\u1105\u1174\u11BA", "\uB9A3", "\u1105\u1174\u11BA" },
            new string [] { "\uB9A4", "\uB9A4", "\u1105\u1174\u11BB", "\uB9A4", "\u1105\u1174\u11BB" },
            new string [] { "\uB9A5", "\uB9A5", "\u1105\u1174\u11BC", "\uB9A5", "\u1105\u1174\u11BC" },
            new string [] { "\uB9A6", "\uB9A6", "\u1105\u1174\u11BD", "\uB9A6", "\u1105\u1174\u11BD" },
            new string [] { "\uB9A7", "\uB9A7", "\u1105\u1174\u11BE", "\uB9A7", "\u1105\u1174\u11BE" },
            new string [] { "\uB9A8", "\uB9A8", "\u1105\u1174\u11BF", "\uB9A8", "\u1105\u1174\u11BF" },
            new string [] { "\uB9A9", "\uB9A9", "\u1105\u1174\u11C0", "\uB9A9", "\u1105\u1174\u11C0" },
            new string [] { "\uB9AA", "\uB9AA", "\u1105\u1174\u11C1", "\uB9AA", "\u1105\u1174\u11C1" },
            new string [] { "\uB9AB", "\uB9AB", "\u1105\u1174\u11C2", "\uB9AB", "\u1105\u1174\u11C2" },
            new string [] { "\uB9AC", "\uB9AC", "\u1105\u1175", "\uB9AC", "\u1105\u1175" },
            new string [] { "\uB9AD", "\uB9AD", "\u1105\u1175\u11A8", "\uB9AD", "\u1105\u1175\u11A8" },
            new string [] { "\uB9AE", "\uB9AE", "\u1105\u1175\u11A9", "\uB9AE", "\u1105\u1175\u11A9" },
            new string [] { "\uB9AF", "\uB9AF", "\u1105\u1175\u11AA", "\uB9AF", "\u1105\u1175\u11AA" },
            new string [] { "\uB9B0", "\uB9B0", "\u1105\u1175\u11AB", "\uB9B0", "\u1105\u1175\u11AB" },
            new string [] { "\uB9B1", "\uB9B1", "\u1105\u1175\u11AC", "\uB9B1", "\u1105\u1175\u11AC" },
            new string [] { "\uB9B2", "\uB9B2", "\u1105\u1175\u11AD", "\uB9B2", "\u1105\u1175\u11AD" },
            new string [] { "\uB9B3", "\uB9B3", "\u1105\u1175\u11AE", "\uB9B3", "\u1105\u1175\u11AE" },
            new string [] { "\uB9B4", "\uB9B4", "\u1105\u1175\u11AF", "\uB9B4", "\u1105\u1175\u11AF" },
            new string [] { "\uB9B5", "\uB9B5", "\u1105\u1175\u11B0", "\uB9B5", "\u1105\u1175\u11B0" },
            new string [] { "\uB9B6", "\uB9B6", "\u1105\u1175\u11B1", "\uB9B6", "\u1105\u1175\u11B1" },
            new string [] { "\uB9B7", "\uB9B7", "\u1105\u1175\u11B2", "\uB9B7", "\u1105\u1175\u11B2" },
            new string [] { "\uB9B8", "\uB9B8", "\u1105\u1175\u11B3", "\uB9B8", "\u1105\u1175\u11B3" },
            new string [] { "\uB9B9", "\uB9B9", "\u1105\u1175\u11B4", "\uB9B9", "\u1105\u1175\u11B4" },
            new string [] { "\uB9BA", "\uB9BA", "\u1105\u1175\u11B5", "\uB9BA", "\u1105\u1175\u11B5" },
            new string [] { "\uB9BB", "\uB9BB", "\u1105\u1175\u11B6", "\uB9BB", "\u1105\u1175\u11B6" },
            new string [] { "\uB9BC", "\uB9BC", "\u1105\u1175\u11B7", "\uB9BC", "\u1105\u1175\u11B7" },
            new string [] { "\uB9BD", "\uB9BD", "\u1105\u1175\u11B8", "\uB9BD", "\u1105\u1175\u11B8" },
            new string [] { "\uB9BE", "\uB9BE", "\u1105\u1175\u11B9", "\uB9BE", "\u1105\u1175\u11B9" },
            new string [] { "\uB9BF", "\uB9BF", "\u1105\u1175\u11BA", "\uB9BF", "\u1105\u1175\u11BA" },
            new string [] { "\uB9C0", "\uB9C0", "\u1105\u1175\u11BB", "\uB9C0", "\u1105\u1175\u11BB" },
            new string [] { "\uB9C1", "\uB9C1", "\u1105\u1175\u11BC", "\uB9C1", "\u1105\u1175\u11BC" },
            new string [] { "\uB9C2", "\uB9C2", "\u1105\u1175\u11BD", "\uB9C2", "\u1105\u1175\u11BD" },
            new string [] { "\uB9C3", "\uB9C3", "\u1105\u1175\u11BE", "\uB9C3", "\u1105\u1175\u11BE" },
            new string [] { "\uB9C4", "\uB9C4", "\u1105\u1175\u11BF", "\uB9C4", "\u1105\u1175\u11BF" },
            new string [] { "\uB9C5", "\uB9C5", "\u1105\u1175\u11C0", "\uB9C5", "\u1105\u1175\u11C0" },
            new string [] { "\uB9C6", "\uB9C6", "\u1105\u1175\u11C1", "\uB9C6", "\u1105\u1175\u11C1" },
            new string [] { "\uB9C7", "\uB9C7", "\u1105\u1175\u11C2", "\uB9C7", "\u1105\u1175\u11C2" },
            new string [] { "\uB9C8", "\uB9C8", "\u1106\u1161", "\uB9C8", "\u1106\u1161" },
            new string [] { "\uB9C9", "\uB9C9", "\u1106\u1161\u11A8", "\uB9C9", "\u1106\u1161\u11A8" },
            new string [] { "\uB9CA", "\uB9CA", "\u1106\u1161\u11A9", "\uB9CA", "\u1106\u1161\u11A9" },
            new string [] { "\uB9CB", "\uB9CB", "\u1106\u1161\u11AA", "\uB9CB", "\u1106\u1161\u11AA" },
            new string [] { "\uB9CC", "\uB9CC", "\u1106\u1161\u11AB", "\uB9CC", "\u1106\u1161\u11AB" },
            new string [] { "\uB9CD", "\uB9CD", "\u1106\u1161\u11AC", "\uB9CD", "\u1106\u1161\u11AC" },
            new string [] { "\uB9CE", "\uB9CE", "\u1106\u1161\u11AD", "\uB9CE", "\u1106\u1161\u11AD" },
            new string [] { "\uB9CF", "\uB9CF", "\u1106\u1161\u11AE", "\uB9CF", "\u1106\u1161\u11AE" },
            new string [] { "\uB9D0", "\uB9D0", "\u1106\u1161\u11AF", "\uB9D0", "\u1106\u1161\u11AF" },
            new string [] { "\uB9D1", "\uB9D1", "\u1106\u1161\u11B0", "\uB9D1", "\u1106\u1161\u11B0" },
            new string [] { "\uB9D2", "\uB9D2", "\u1106\u1161\u11B1", "\uB9D2", "\u1106\u1161\u11B1" },
            new string [] { "\uB9D3", "\uB9D3", "\u1106\u1161\u11B2", "\uB9D3", "\u1106\u1161\u11B2" },
            new string [] { "\uB9D4", "\uB9D4", "\u1106\u1161\u11B3", "\uB9D4", "\u1106\u1161\u11B3" },
            new string [] { "\uB9D5", "\uB9D5", "\u1106\u1161\u11B4", "\uB9D5", "\u1106\u1161\u11B4" },
            new string [] { "\uB9D6", "\uB9D6", "\u1106\u1161\u11B5", "\uB9D6", "\u1106\u1161\u11B5" },
            new string [] { "\uB9D7", "\uB9D7", "\u1106\u1161\u11B6", "\uB9D7", "\u1106\u1161\u11B6" },
            new string [] { "\uB9D8", "\uB9D8", "\u1106\u1161\u11B7", "\uB9D8", "\u1106\u1161\u11B7" },
            new string [] { "\uB9D9", "\uB9D9", "\u1106\u1161\u11B8", "\uB9D9", "\u1106\u1161\u11B8" },
            new string [] { "\uB9DA", "\uB9DA", "\u1106\u1161\u11B9", "\uB9DA", "\u1106\u1161\u11B9" },
            new string [] { "\uB9DB", "\uB9DB", "\u1106\u1161\u11BA", "\uB9DB", "\u1106\u1161\u11BA" },
            new string [] { "\uB9DC", "\uB9DC", "\u1106\u1161\u11BB", "\uB9DC", "\u1106\u1161\u11BB" },
            new string [] { "\uB9DD", "\uB9DD", "\u1106\u1161\u11BC", "\uB9DD", "\u1106\u1161\u11BC" },
            new string [] { "\uB9DE", "\uB9DE", "\u1106\u1161\u11BD", "\uB9DE", "\u1106\u1161\u11BD" },
            new string [] { "\uB9DF", "\uB9DF", "\u1106\u1161\u11BE", "\uB9DF", "\u1106\u1161\u11BE" },
            new string [] { "\uB9E0", "\uB9E0", "\u1106\u1161\u11BF", "\uB9E0", "\u1106\u1161\u11BF" },
            new string [] { "\uB9E1", "\uB9E1", "\u1106\u1161\u11C0", "\uB9E1", "\u1106\u1161\u11C0" },
            new string [] { "\uB9E2", "\uB9E2", "\u1106\u1161\u11C1", "\uB9E2", "\u1106\u1161\u11C1" },
            new string [] { "\uB9E3", "\uB9E3", "\u1106\u1161\u11C2", "\uB9E3", "\u1106\u1161\u11C2" },
            new string [] { "\uB9E4", "\uB9E4", "\u1106\u1162", "\uB9E4", "\u1106\u1162" },
            new string [] { "\uB9E5", "\uB9E5", "\u1106\u1162\u11A8", "\uB9E5", "\u1106\u1162\u11A8" },
            new string [] { "\uB9E6", "\uB9E6", "\u1106\u1162\u11A9", "\uB9E6", "\u1106\u1162\u11A9" },
            new string [] { "\uB9E7", "\uB9E7", "\u1106\u1162\u11AA", "\uB9E7", "\u1106\u1162\u11AA" },
            new string [] { "\uB9E8", "\uB9E8", "\u1106\u1162\u11AB", "\uB9E8", "\u1106\u1162\u11AB" },
            new string [] { "\uB9E9", "\uB9E9", "\u1106\u1162\u11AC", "\uB9E9", "\u1106\u1162\u11AC" },
            new string [] { "\uB9EA", "\uB9EA", "\u1106\u1162\u11AD", "\uB9EA", "\u1106\u1162\u11AD" },
            new string [] { "\uB9EB", "\uB9EB", "\u1106\u1162\u11AE", "\uB9EB", "\u1106\u1162\u11AE" },
            new string [] { "\uB9EC", "\uB9EC", "\u1106\u1162\u11AF", "\uB9EC", "\u1106\u1162\u11AF" },
            new string [] { "\uB9ED", "\uB9ED", "\u1106\u1162\u11B0", "\uB9ED", "\u1106\u1162\u11B0" },
            new string [] { "\uB9EE", "\uB9EE", "\u1106\u1162\u11B1", "\uB9EE", "\u1106\u1162\u11B1" },
            new string [] { "\uB9EF", "\uB9EF", "\u1106\u1162\u11B2", "\uB9EF", "\u1106\u1162\u11B2" },
            new string [] { "\uB9F0", "\uB9F0", "\u1106\u1162\u11B3", "\uB9F0", "\u1106\u1162\u11B3" },
            new string [] { "\uB9F1", "\uB9F1", "\u1106\u1162\u11B4", "\uB9F1", "\u1106\u1162\u11B4" },
            new string [] { "\uB9F2", "\uB9F2", "\u1106\u1162\u11B5", "\uB9F2", "\u1106\u1162\u11B5" },
            new string [] { "\uB9F3", "\uB9F3", "\u1106\u1162\u11B6", "\uB9F3", "\u1106\u1162\u11B6" },
            new string [] { "\uB9F4", "\uB9F4", "\u1106\u1162\u11B7", "\uB9F4", "\u1106\u1162\u11B7" },
            new string [] { "\uB9F5", "\uB9F5", "\u1106\u1162\u11B8", "\uB9F5", "\u1106\u1162\u11B8" },
            new string [] { "\uB9F6", "\uB9F6", "\u1106\u1162\u11B9", "\uB9F6", "\u1106\u1162\u11B9" },
            new string [] { "\uB9F7", "\uB9F7", "\u1106\u1162\u11BA", "\uB9F7", "\u1106\u1162\u11BA" },
            new string [] { "\uB9F8", "\uB9F8", "\u1106\u1162\u11BB", "\uB9F8", "\u1106\u1162\u11BB" },
            new string [] { "\uB9F9", "\uB9F9", "\u1106\u1162\u11BC", "\uB9F9", "\u1106\u1162\u11BC" },
            new string [] { "\uB9FA", "\uB9FA", "\u1106\u1162\u11BD", "\uB9FA", "\u1106\u1162\u11BD" },
            new string [] { "\uB9FB", "\uB9FB", "\u1106\u1162\u11BE", "\uB9FB", "\u1106\u1162\u11BE" },
            new string [] { "\uB9FC", "\uB9FC", "\u1106\u1162\u11BF", "\uB9FC", "\u1106\u1162\u11BF" },
            new string [] { "\uB9FD", "\uB9FD", "\u1106\u1162\u11C0", "\uB9FD", "\u1106\u1162\u11C0" },
            new string [] { "\uB9FE", "\uB9FE", "\u1106\u1162\u11C1", "\uB9FE", "\u1106\u1162\u11C1" },
            new string [] { "\uB9FF", "\uB9FF", "\u1106\u1162\u11C2", "\uB9FF", "\u1106\u1162\u11C2" },
            new string [] { "\uBA00", "\uBA00", "\u1106\u1163", "\uBA00", "\u1106\u1163" },
            new string [] { "\uBA01", "\uBA01", "\u1106\u1163\u11A8", "\uBA01", "\u1106\u1163\u11A8" },
            new string [] { "\uBA02", "\uBA02", "\u1106\u1163\u11A9", "\uBA02", "\u1106\u1163\u11A9" },
            new string [] { "\uBA03", "\uBA03", "\u1106\u1163\u11AA", "\uBA03", "\u1106\u1163\u11AA" },
            new string [] { "\uBA04", "\uBA04", "\u1106\u1163\u11AB", "\uBA04", "\u1106\u1163\u11AB" },
            new string [] { "\uBA05", "\uBA05", "\u1106\u1163\u11AC", "\uBA05", "\u1106\u1163\u11AC" },
            new string [] { "\uBA06", "\uBA06", "\u1106\u1163\u11AD", "\uBA06", "\u1106\u1163\u11AD" },
            new string [] { "\uBA07", "\uBA07", "\u1106\u1163\u11AE", "\uBA07", "\u1106\u1163\u11AE" },
            new string [] { "\uBA08", "\uBA08", "\u1106\u1163\u11AF", "\uBA08", "\u1106\u1163\u11AF" },
            new string [] { "\uBA09", "\uBA09", "\u1106\u1163\u11B0", "\uBA09", "\u1106\u1163\u11B0" },
            new string [] { "\uBA0A", "\uBA0A", "\u1106\u1163\u11B1", "\uBA0A", "\u1106\u1163\u11B1" },
            new string [] { "\uBA0B", "\uBA0B", "\u1106\u1163\u11B2", "\uBA0B", "\u1106\u1163\u11B2" },
            new string [] { "\uBA0C", "\uBA0C", "\u1106\u1163\u11B3", "\uBA0C", "\u1106\u1163\u11B3" },
            new string [] { "\uBA0D", "\uBA0D", "\u1106\u1163\u11B4", "\uBA0D", "\u1106\u1163\u11B4" },
            new string [] { "\uBA0E", "\uBA0E", "\u1106\u1163\u11B5", "\uBA0E", "\u1106\u1163\u11B5" },
            new string [] { "\uBA0F", "\uBA0F", "\u1106\u1163\u11B6", "\uBA0F", "\u1106\u1163\u11B6" },
            new string [] { "\uBA10", "\uBA10", "\u1106\u1163\u11B7", "\uBA10", "\u1106\u1163\u11B7" },
            new string [] { "\uBA11", "\uBA11", "\u1106\u1163\u11B8", "\uBA11", "\u1106\u1163\u11B8" },
            new string [] { "\uBA12", "\uBA12", "\u1106\u1163\u11B9", "\uBA12", "\u1106\u1163\u11B9" },
            new string [] { "\uBA13", "\uBA13", "\u1106\u1163\u11BA", "\uBA13", "\u1106\u1163\u11BA" },
            new string [] { "\uBA14", "\uBA14", "\u1106\u1163\u11BB", "\uBA14", "\u1106\u1163\u11BB" },
            new string [] { "\uBA15", "\uBA15", "\u1106\u1163\u11BC", "\uBA15", "\u1106\u1163\u11BC" },
            new string [] { "\uBA16", "\uBA16", "\u1106\u1163\u11BD", "\uBA16", "\u1106\u1163\u11BD" },
            new string [] { "\uBA17", "\uBA17", "\u1106\u1163\u11BE", "\uBA17", "\u1106\u1163\u11BE" },
            new string [] { "\uBA18", "\uBA18", "\u1106\u1163\u11BF", "\uBA18", "\u1106\u1163\u11BF" },
            new string [] { "\uBA19", "\uBA19", "\u1106\u1163\u11C0", "\uBA19", "\u1106\u1163\u11C0" },
            new string [] { "\uBA1A", "\uBA1A", "\u1106\u1163\u11C1", "\uBA1A", "\u1106\u1163\u11C1" },
            new string [] { "\uBA1B", "\uBA1B", "\u1106\u1163\u11C2", "\uBA1B", "\u1106\u1163\u11C2" },
            new string [] { "\uBA1C", "\uBA1C", "\u1106\u1164", "\uBA1C", "\u1106\u1164" },
            new string [] { "\uBA1D", "\uBA1D", "\u1106\u1164\u11A8", "\uBA1D", "\u1106\u1164\u11A8" },
            new string [] { "\uBA1E", "\uBA1E", "\u1106\u1164\u11A9", "\uBA1E", "\u1106\u1164\u11A9" },
            new string [] { "\uBA1F", "\uBA1F", "\u1106\u1164\u11AA", "\uBA1F", "\u1106\u1164\u11AA" },
            new string [] { "\uBA20", "\uBA20", "\u1106\u1164\u11AB", "\uBA20", "\u1106\u1164\u11AB" },
            new string [] { "\uBA21", "\uBA21", "\u1106\u1164\u11AC", "\uBA21", "\u1106\u1164\u11AC" },
            new string [] { "\uBA22", "\uBA22", "\u1106\u1164\u11AD", "\uBA22", "\u1106\u1164\u11AD" },
            new string [] { "\uBA23", "\uBA23", "\u1106\u1164\u11AE", "\uBA23", "\u1106\u1164\u11AE" },
            new string [] { "\uBA24", "\uBA24", "\u1106\u1164\u11AF", "\uBA24", "\u1106\u1164\u11AF" },
            new string [] { "\uBA25", "\uBA25", "\u1106\u1164\u11B0", "\uBA25", "\u1106\u1164\u11B0" },
            new string [] { "\uBA26", "\uBA26", "\u1106\u1164\u11B1", "\uBA26", "\u1106\u1164\u11B1" },
            new string [] { "\uBA27", "\uBA27", "\u1106\u1164\u11B2", "\uBA27", "\u1106\u1164\u11B2" },
            new string [] { "\uBA28", "\uBA28", "\u1106\u1164\u11B3", "\uBA28", "\u1106\u1164\u11B3" },
            new string [] { "\uBA29", "\uBA29", "\u1106\u1164\u11B4", "\uBA29", "\u1106\u1164\u11B4" },
            new string [] { "\uBA2A", "\uBA2A", "\u1106\u1164\u11B5", "\uBA2A", "\u1106\u1164\u11B5" },
            new string [] { "\uBA2B", "\uBA2B", "\u1106\u1164\u11B6", "\uBA2B", "\u1106\u1164\u11B6" },
            new string [] { "\uBA2C", "\uBA2C", "\u1106\u1164\u11B7", "\uBA2C", "\u1106\u1164\u11B7" },
            new string [] { "\uBA2D", "\uBA2D", "\u1106\u1164\u11B8", "\uBA2D", "\u1106\u1164\u11B8" },
            new string [] { "\uBA2E", "\uBA2E", "\u1106\u1164\u11B9", "\uBA2E", "\u1106\u1164\u11B9" },
            new string [] { "\uBA2F", "\uBA2F", "\u1106\u1164\u11BA", "\uBA2F", "\u1106\u1164\u11BA" },
            new string [] { "\uBA30", "\uBA30", "\u1106\u1164\u11BB", "\uBA30", "\u1106\u1164\u11BB" },
            new string [] { "\uBA31", "\uBA31", "\u1106\u1164\u11BC", "\uBA31", "\u1106\u1164\u11BC" },
            new string [] { "\uBA32", "\uBA32", "\u1106\u1164\u11BD", "\uBA32", "\u1106\u1164\u11BD" },
            new string [] { "\uBA33", "\uBA33", "\u1106\u1164\u11BE", "\uBA33", "\u1106\u1164\u11BE" },
            new string [] { "\uBA34", "\uBA34", "\u1106\u1164\u11BF", "\uBA34", "\u1106\u1164\u11BF" },
            new string [] { "\uBA35", "\uBA35", "\u1106\u1164\u11C0", "\uBA35", "\u1106\u1164\u11C0" },
            new string [] { "\uBA36", "\uBA36", "\u1106\u1164\u11C1", "\uBA36", "\u1106\u1164\u11C1" },
            new string [] { "\uBA37", "\uBA37", "\u1106\u1164\u11C2", "\uBA37", "\u1106\u1164\u11C2" },
            new string [] { "\uBA38", "\uBA38", "\u1106\u1165", "\uBA38", "\u1106\u1165" },
            new string [] { "\uBA39", "\uBA39", "\u1106\u1165\u11A8", "\uBA39", "\u1106\u1165\u11A8" },
            new string [] { "\uBA3A", "\uBA3A", "\u1106\u1165\u11A9", "\uBA3A", "\u1106\u1165\u11A9" },
            new string [] { "\uBA3B", "\uBA3B", "\u1106\u1165\u11AA", "\uBA3B", "\u1106\u1165\u11AA" },
            new string [] { "\uBA3C", "\uBA3C", "\u1106\u1165\u11AB", "\uBA3C", "\u1106\u1165\u11AB" },
            new string [] { "\uBA3D", "\uBA3D", "\u1106\u1165\u11AC", "\uBA3D", "\u1106\u1165\u11AC" },
            new string [] { "\uBA3E", "\uBA3E", "\u1106\u1165\u11AD", "\uBA3E", "\u1106\u1165\u11AD" },
            new string [] { "\uBA3F", "\uBA3F", "\u1106\u1165\u11AE", "\uBA3F", "\u1106\u1165\u11AE" },
            new string [] { "\uBA40", "\uBA40", "\u1106\u1165\u11AF", "\uBA40", "\u1106\u1165\u11AF" },
            new string [] { "\uBA41", "\uBA41", "\u1106\u1165\u11B0", "\uBA41", "\u1106\u1165\u11B0" },
            new string [] { "\uBA42", "\uBA42", "\u1106\u1165\u11B1", "\uBA42", "\u1106\u1165\u11B1" },
            new string [] { "\uBA43", "\uBA43", "\u1106\u1165\u11B2", "\uBA43", "\u1106\u1165\u11B2" },
            new string [] { "\uBA44", "\uBA44", "\u1106\u1165\u11B3", "\uBA44", "\u1106\u1165\u11B3" },
            new string [] { "\uBA45", "\uBA45", "\u1106\u1165\u11B4", "\uBA45", "\u1106\u1165\u11B4" },
            new string [] { "\uBA46", "\uBA46", "\u1106\u1165\u11B5", "\uBA46", "\u1106\u1165\u11B5" },
            new string [] { "\uBA47", "\uBA47", "\u1106\u1165\u11B6", "\uBA47", "\u1106\u1165\u11B6" },
            new string [] { "\uBA48", "\uBA48", "\u1106\u1165\u11B7", "\uBA48", "\u1106\u1165\u11B7" },
            new string [] { "\uBA49", "\uBA49", "\u1106\u1165\u11B8", "\uBA49", "\u1106\u1165\u11B8" },
            new string [] { "\uBA4A", "\uBA4A", "\u1106\u1165\u11B9", "\uBA4A", "\u1106\u1165\u11B9" },
            new string [] { "\uBA4B", "\uBA4B", "\u1106\u1165\u11BA", "\uBA4B", "\u1106\u1165\u11BA" },
            new string [] { "\uBA4C", "\uBA4C", "\u1106\u1165\u11BB", "\uBA4C", "\u1106\u1165\u11BB" },
            new string [] { "\uBA4D", "\uBA4D", "\u1106\u1165\u11BC", "\uBA4D", "\u1106\u1165\u11BC" },
            new string [] { "\uBA4E", "\uBA4E", "\u1106\u1165\u11BD", "\uBA4E", "\u1106\u1165\u11BD" },
            new string [] { "\uBA4F", "\uBA4F", "\u1106\u1165\u11BE", "\uBA4F", "\u1106\u1165\u11BE" },
            new string [] { "\uBA50", "\uBA50", "\u1106\u1165\u11BF", "\uBA50", "\u1106\u1165\u11BF" },
            new string [] { "\uBA51", "\uBA51", "\u1106\u1165\u11C0", "\uBA51", "\u1106\u1165\u11C0" },
            new string [] { "\uBA52", "\uBA52", "\u1106\u1165\u11C1", "\uBA52", "\u1106\u1165\u11C1" },
            new string [] { "\uBA53", "\uBA53", "\u1106\u1165\u11C2", "\uBA53", "\u1106\u1165\u11C2" },
            new string [] { "\uBA54", "\uBA54", "\u1106\u1166", "\uBA54", "\u1106\u1166" },
            new string [] { "\uBA55", "\uBA55", "\u1106\u1166\u11A8", "\uBA55", "\u1106\u1166\u11A8" },
            new string [] { "\uBA56", "\uBA56", "\u1106\u1166\u11A9", "\uBA56", "\u1106\u1166\u11A9" },
            new string [] { "\uBA57", "\uBA57", "\u1106\u1166\u11AA", "\uBA57", "\u1106\u1166\u11AA" },
            new string [] { "\uBA58", "\uBA58", "\u1106\u1166\u11AB", "\uBA58", "\u1106\u1166\u11AB" },
            new string [] { "\uBA59", "\uBA59", "\u1106\u1166\u11AC", "\uBA59", "\u1106\u1166\u11AC" },
            new string [] { "\uBA5A", "\uBA5A", "\u1106\u1166\u11AD", "\uBA5A", "\u1106\u1166\u11AD" },
            new string [] { "\uBA5B", "\uBA5B", "\u1106\u1166\u11AE", "\uBA5B", "\u1106\u1166\u11AE" },
            new string [] { "\uBA5C", "\uBA5C", "\u1106\u1166\u11AF", "\uBA5C", "\u1106\u1166\u11AF" },
            new string [] { "\uBA5D", "\uBA5D", "\u1106\u1166\u11B0", "\uBA5D", "\u1106\u1166\u11B0" },
            new string [] { "\uBA5E", "\uBA5E", "\u1106\u1166\u11B1", "\uBA5E", "\u1106\u1166\u11B1" },
            new string [] { "\uBA5F", "\uBA5F", "\u1106\u1166\u11B2", "\uBA5F", "\u1106\u1166\u11B2" },
            new string [] { "\uBA60", "\uBA60", "\u1106\u1166\u11B3", "\uBA60", "\u1106\u1166\u11B3" },
            new string [] { "\uBA61", "\uBA61", "\u1106\u1166\u11B4", "\uBA61", "\u1106\u1166\u11B4" },
            new string [] { "\uBA62", "\uBA62", "\u1106\u1166\u11B5", "\uBA62", "\u1106\u1166\u11B5" },
            new string [] { "\uBA63", "\uBA63", "\u1106\u1166\u11B6", "\uBA63", "\u1106\u1166\u11B6" },
            new string [] { "\uBA64", "\uBA64", "\u1106\u1166\u11B7", "\uBA64", "\u1106\u1166\u11B7" },
            new string [] { "\uBA65", "\uBA65", "\u1106\u1166\u11B8", "\uBA65", "\u1106\u1166\u11B8" },
            new string [] { "\uBA66", "\uBA66", "\u1106\u1166\u11B9", "\uBA66", "\u1106\u1166\u11B9" },
            new string [] { "\uBA67", "\uBA67", "\u1106\u1166\u11BA", "\uBA67", "\u1106\u1166\u11BA" },
            new string [] { "\uBA68", "\uBA68", "\u1106\u1166\u11BB", "\uBA68", "\u1106\u1166\u11BB" },
            new string [] { "\uBA69", "\uBA69", "\u1106\u1166\u11BC", "\uBA69", "\u1106\u1166\u11BC" },
            new string [] { "\uBA6A", "\uBA6A", "\u1106\u1166\u11BD", "\uBA6A", "\u1106\u1166\u11BD" },
            new string [] { "\uBA6B", "\uBA6B", "\u1106\u1166\u11BE", "\uBA6B", "\u1106\u1166\u11BE" },
            new string [] { "\uBA6C", "\uBA6C", "\u1106\u1166\u11BF", "\uBA6C", "\u1106\u1166\u11BF" },
            new string [] { "\uBA6D", "\uBA6D", "\u1106\u1166\u11C0", "\uBA6D", "\u1106\u1166\u11C0" },
            new string [] { "\uBA6E", "\uBA6E", "\u1106\u1166\u11C1", "\uBA6E", "\u1106\u1166\u11C1" },
            new string [] { "\uBA6F", "\uBA6F", "\u1106\u1166\u11C2", "\uBA6F", "\u1106\u1166\u11C2" },
            new string [] { "\uBA70", "\uBA70", "\u1106\u1167", "\uBA70", "\u1106\u1167" },
            new string [] { "\uBA71", "\uBA71", "\u1106\u1167\u11A8", "\uBA71", "\u1106\u1167\u11A8" },
            new string [] { "\uBA72", "\uBA72", "\u1106\u1167\u11A9", "\uBA72", "\u1106\u1167\u11A9" },
            new string [] { "\uBA73", "\uBA73", "\u1106\u1167\u11AA", "\uBA73", "\u1106\u1167\u11AA" },
            new string [] { "\uBA74", "\uBA74", "\u1106\u1167\u11AB", "\uBA74", "\u1106\u1167\u11AB" },
            new string [] { "\uBA75", "\uBA75", "\u1106\u1167\u11AC", "\uBA75", "\u1106\u1167\u11AC" },
            new string [] { "\uBA76", "\uBA76", "\u1106\u1167\u11AD", "\uBA76", "\u1106\u1167\u11AD" },
            new string [] { "\uBA77", "\uBA77", "\u1106\u1167\u11AE", "\uBA77", "\u1106\u1167\u11AE" },
            new string [] { "\uBA78", "\uBA78", "\u1106\u1167\u11AF", "\uBA78", "\u1106\u1167\u11AF" },
            new string [] { "\uBA79", "\uBA79", "\u1106\u1167\u11B0", "\uBA79", "\u1106\u1167\u11B0" },
            new string [] { "\uBA7A", "\uBA7A", "\u1106\u1167\u11B1", "\uBA7A", "\u1106\u1167\u11B1" },
            new string [] { "\uBA7B", "\uBA7B", "\u1106\u1167\u11B2", "\uBA7B", "\u1106\u1167\u11B2" },
            new string [] { "\uBA7C", "\uBA7C", "\u1106\u1167\u11B3", "\uBA7C", "\u1106\u1167\u11B3" },
            new string [] { "\uBA7D", "\uBA7D", "\u1106\u1167\u11B4", "\uBA7D", "\u1106\u1167\u11B4" },
            new string [] { "\uBA7E", "\uBA7E", "\u1106\u1167\u11B5", "\uBA7E", "\u1106\u1167\u11B5" },
            new string [] { "\uBA7F", "\uBA7F", "\u1106\u1167\u11B6", "\uBA7F", "\u1106\u1167\u11B6" },
            new string [] { "\uBA80", "\uBA80", "\u1106\u1167\u11B7", "\uBA80", "\u1106\u1167\u11B7" },
            new string [] { "\uBA81", "\uBA81", "\u1106\u1167\u11B8", "\uBA81", "\u1106\u1167\u11B8" },
            new string [] { "\uBA82", "\uBA82", "\u1106\u1167\u11B9", "\uBA82", "\u1106\u1167\u11B9" },
            new string [] { "\uBA83", "\uBA83", "\u1106\u1167\u11BA", "\uBA83", "\u1106\u1167\u11BA" },
            new string [] { "\uBA84", "\uBA84", "\u1106\u1167\u11BB", "\uBA84", "\u1106\u1167\u11BB" },
            new string [] { "\uBA85", "\uBA85", "\u1106\u1167\u11BC", "\uBA85", "\u1106\u1167\u11BC" },
            new string [] { "\uBA86", "\uBA86", "\u1106\u1167\u11BD", "\uBA86", "\u1106\u1167\u11BD" },
            new string [] { "\uBA87", "\uBA87", "\u1106\u1167\u11BE", "\uBA87", "\u1106\u1167\u11BE" },
            new string [] { "\uBA88", "\uBA88", "\u1106\u1167\u11BF", "\uBA88", "\u1106\u1167\u11BF" },
            new string [] { "\uBA89", "\uBA89", "\u1106\u1167\u11C0", "\uBA89", "\u1106\u1167\u11C0" },
            new string [] { "\uBA8A", "\uBA8A", "\u1106\u1167\u11C1", "\uBA8A", "\u1106\u1167\u11C1" },
            new string [] { "\uBA8B", "\uBA8B", "\u1106\u1167\u11C2", "\uBA8B", "\u1106\u1167\u11C2" },
            new string [] { "\uBA8C", "\uBA8C", "\u1106\u1168", "\uBA8C", "\u1106\u1168" },
            new string [] { "\uBA8D", "\uBA8D", "\u1106\u1168\u11A8", "\uBA8D", "\u1106\u1168\u11A8" },
            new string [] { "\uBA8E", "\uBA8E", "\u1106\u1168\u11A9", "\uBA8E", "\u1106\u1168\u11A9" },
            new string [] { "\uBA8F", "\uBA8F", "\u1106\u1168\u11AA", "\uBA8F", "\u1106\u1168\u11AA" },
            new string [] { "\uBA90", "\uBA90", "\u1106\u1168\u11AB", "\uBA90", "\u1106\u1168\u11AB" },
            new string [] { "\uBA91", "\uBA91", "\u1106\u1168\u11AC", "\uBA91", "\u1106\u1168\u11AC" },
            new string [] { "\uBA92", "\uBA92", "\u1106\u1168\u11AD", "\uBA92", "\u1106\u1168\u11AD" },
            new string [] { "\uBA93", "\uBA93", "\u1106\u1168\u11AE", "\uBA93", "\u1106\u1168\u11AE" },
            new string [] { "\uBA94", "\uBA94", "\u1106\u1168\u11AF", "\uBA94", "\u1106\u1168\u11AF" },
            new string [] { "\uBA95", "\uBA95", "\u1106\u1168\u11B0", "\uBA95", "\u1106\u1168\u11B0" },
            new string [] { "\uBA96", "\uBA96", "\u1106\u1168\u11B1", "\uBA96", "\u1106\u1168\u11B1" },
            new string [] { "\uBA97", "\uBA97", "\u1106\u1168\u11B2", "\uBA97", "\u1106\u1168\u11B2" },
            new string [] { "\uBA98", "\uBA98", "\u1106\u1168\u11B3", "\uBA98", "\u1106\u1168\u11B3" },
            new string [] { "\uBA99", "\uBA99", "\u1106\u1168\u11B4", "\uBA99", "\u1106\u1168\u11B4" },
            new string [] { "\uBA9A", "\uBA9A", "\u1106\u1168\u11B5", "\uBA9A", "\u1106\u1168\u11B5" },
            new string [] { "\uBA9B", "\uBA9B", "\u1106\u1168\u11B6", "\uBA9B", "\u1106\u1168\u11B6" },
            new string [] { "\uBA9C", "\uBA9C", "\u1106\u1168\u11B7", "\uBA9C", "\u1106\u1168\u11B7" },
            new string [] { "\uBA9D", "\uBA9D", "\u1106\u1168\u11B8", "\uBA9D", "\u1106\u1168\u11B8" },
            new string [] { "\uBA9E", "\uBA9E", "\u1106\u1168\u11B9", "\uBA9E", "\u1106\u1168\u11B9" },
            new string [] { "\uBA9F", "\uBA9F", "\u1106\u1168\u11BA", "\uBA9F", "\u1106\u1168\u11BA" },
            new string [] { "\uBAA0", "\uBAA0", "\u1106\u1168\u11BB", "\uBAA0", "\u1106\u1168\u11BB" },
            new string [] { "\uBAA1", "\uBAA1", "\u1106\u1168\u11BC", "\uBAA1", "\u1106\u1168\u11BC" },
            new string [] { "\uBAA2", "\uBAA2", "\u1106\u1168\u11BD", "\uBAA2", "\u1106\u1168\u11BD" },
            new string [] { "\uBAA3", "\uBAA3", "\u1106\u1168\u11BE", "\uBAA3", "\u1106\u1168\u11BE" },
            new string [] { "\uBAA4", "\uBAA4", "\u1106\u1168\u11BF", "\uBAA4", "\u1106\u1168\u11BF" },
            new string [] { "\uBAA5", "\uBAA5", "\u1106\u1168\u11C0", "\uBAA5", "\u1106\u1168\u11C0" },
            new string [] { "\uBAA6", "\uBAA6", "\u1106\u1168\u11C1", "\uBAA6", "\u1106\u1168\u11C1" },
            new string [] { "\uBAA7", "\uBAA7", "\u1106\u1168\u11C2", "\uBAA7", "\u1106\u1168\u11C2" },
            new string [] { "\uBAA8", "\uBAA8", "\u1106\u1169", "\uBAA8", "\u1106\u1169" },
            new string [] { "\uBAA9", "\uBAA9", "\u1106\u1169\u11A8", "\uBAA9", "\u1106\u1169\u11A8" },
            new string [] { "\uBAAA", "\uBAAA", "\u1106\u1169\u11A9", "\uBAAA", "\u1106\u1169\u11A9" },
            new string [] { "\uBAAB", "\uBAAB", "\u1106\u1169\u11AA", "\uBAAB", "\u1106\u1169\u11AA" },
            new string [] { "\uBAAC", "\uBAAC", "\u1106\u1169\u11AB", "\uBAAC", "\u1106\u1169\u11AB" },
            new string [] { "\uBAAD", "\uBAAD", "\u1106\u1169\u11AC", "\uBAAD", "\u1106\u1169\u11AC" },
            new string [] { "\uBAAE", "\uBAAE", "\u1106\u1169\u11AD", "\uBAAE", "\u1106\u1169\u11AD" },
            new string [] { "\uBAAF", "\uBAAF", "\u1106\u1169\u11AE", "\uBAAF", "\u1106\u1169\u11AE" },
            new string [] { "\uBAB0", "\uBAB0", "\u1106\u1169\u11AF", "\uBAB0", "\u1106\u1169\u11AF" },
            new string [] { "\uBAB1", "\uBAB1", "\u1106\u1169\u11B0", "\uBAB1", "\u1106\u1169\u11B0" },
            new string [] { "\uBAB2", "\uBAB2", "\u1106\u1169\u11B1", "\uBAB2", "\u1106\u1169\u11B1" },
            new string [] { "\uBAB3", "\uBAB3", "\u1106\u1169\u11B2", "\uBAB3", "\u1106\u1169\u11B2" },
            new string [] { "\uBAB4", "\uBAB4", "\u1106\u1169\u11B3", "\uBAB4", "\u1106\u1169\u11B3" },
            new string [] { "\uBAB5", "\uBAB5", "\u1106\u1169\u11B4", "\uBAB5", "\u1106\u1169\u11B4" },
            new string [] { "\uBAB6", "\uBAB6", "\u1106\u1169\u11B5", "\uBAB6", "\u1106\u1169\u11B5" },
            new string [] { "\uBAB7", "\uBAB7", "\u1106\u1169\u11B6", "\uBAB7", "\u1106\u1169\u11B6" },
            new string [] { "\uBAB8", "\uBAB8", "\u1106\u1169\u11B7", "\uBAB8", "\u1106\u1169\u11B7" },
            new string [] { "\uBAB9", "\uBAB9", "\u1106\u1169\u11B8", "\uBAB9", "\u1106\u1169\u11B8" },
            new string [] { "\uBABA", "\uBABA", "\u1106\u1169\u11B9", "\uBABA", "\u1106\u1169\u11B9" },
            new string [] { "\uBABB", "\uBABB", "\u1106\u1169\u11BA", "\uBABB", "\u1106\u1169\u11BA" },
            new string [] { "\uBABC", "\uBABC", "\u1106\u1169\u11BB", "\uBABC", "\u1106\u1169\u11BB" },
            new string [] { "\uBABD", "\uBABD", "\u1106\u1169\u11BC", "\uBABD", "\u1106\u1169\u11BC" },
            new string [] { "\uBABE", "\uBABE", "\u1106\u1169\u11BD", "\uBABE", "\u1106\u1169\u11BD" },
            new string [] { "\uBABF", "\uBABF", "\u1106\u1169\u11BE", "\uBABF", "\u1106\u1169\u11BE" },
            new string [] { "\uBAC0", "\uBAC0", "\u1106\u1169\u11BF", "\uBAC0", "\u1106\u1169\u11BF" },
            new string [] { "\uBAC1", "\uBAC1", "\u1106\u1169\u11C0", "\uBAC1", "\u1106\u1169\u11C0" },
            new string [] { "\uBAC2", "\uBAC2", "\u1106\u1169\u11C1", "\uBAC2", "\u1106\u1169\u11C1" },
            new string [] { "\uBAC3", "\uBAC3", "\u1106\u1169\u11C2", "\uBAC3", "\u1106\u1169\u11C2" },
            new string [] { "\uBAC4", "\uBAC4", "\u1106\u116A", "\uBAC4", "\u1106\u116A" },
            new string [] { "\uBAC5", "\uBAC5", "\u1106\u116A\u11A8", "\uBAC5", "\u1106\u116A\u11A8" },
            new string [] { "\uBAC6", "\uBAC6", "\u1106\u116A\u11A9", "\uBAC6", "\u1106\u116A\u11A9" },
            new string [] { "\uBAC7", "\uBAC7", "\u1106\u116A\u11AA", "\uBAC7", "\u1106\u116A\u11AA" },
            new string [] { "\uBAC8", "\uBAC8", "\u1106\u116A\u11AB", "\uBAC8", "\u1106\u116A\u11AB" },
            new string [] { "\uBAC9", "\uBAC9", "\u1106\u116A\u11AC", "\uBAC9", "\u1106\u116A\u11AC" },
            new string [] { "\uBACA", "\uBACA", "\u1106\u116A\u11AD", "\uBACA", "\u1106\u116A\u11AD" },
            new string [] { "\uBACB", "\uBACB", "\u1106\u116A\u11AE", "\uBACB", "\u1106\u116A\u11AE" },
            new string [] { "\uBACC", "\uBACC", "\u1106\u116A\u11AF", "\uBACC", "\u1106\u116A\u11AF" },
            new string [] { "\uBACD", "\uBACD", "\u1106\u116A\u11B0", "\uBACD", "\u1106\u116A\u11B0" },
            new string [] { "\uBACE", "\uBACE", "\u1106\u116A\u11B1", "\uBACE", "\u1106\u116A\u11B1" },
            new string [] { "\uBACF", "\uBACF", "\u1106\u116A\u11B2", "\uBACF", "\u1106\u116A\u11B2" },
            new string [] { "\uBAD0", "\uBAD0", "\u1106\u116A\u11B3", "\uBAD0", "\u1106\u116A\u11B3" },
            new string [] { "\uBAD1", "\uBAD1", "\u1106\u116A\u11B4", "\uBAD1", "\u1106\u116A\u11B4" },
            new string [] { "\uBAD2", "\uBAD2", "\u1106\u116A\u11B5", "\uBAD2", "\u1106\u116A\u11B5" },
            new string [] { "\uBAD3", "\uBAD3", "\u1106\u116A\u11B6", "\uBAD3", "\u1106\u116A\u11B6" },
            new string [] { "\uBAD4", "\uBAD4", "\u1106\u116A\u11B7", "\uBAD4", "\u1106\u116A\u11B7" },
            new string [] { "\uBAD5", "\uBAD5", "\u1106\u116A\u11B8", "\uBAD5", "\u1106\u116A\u11B8" },
            new string [] { "\uBAD6", "\uBAD6", "\u1106\u116A\u11B9", "\uBAD6", "\u1106\u116A\u11B9" },
            new string [] { "\uBAD7", "\uBAD7", "\u1106\u116A\u11BA", "\uBAD7", "\u1106\u116A\u11BA" },
            new string [] { "\uBAD8", "\uBAD8", "\u1106\u116A\u11BB", "\uBAD8", "\u1106\u116A\u11BB" },
            new string [] { "\uBAD9", "\uBAD9", "\u1106\u116A\u11BC", "\uBAD9", "\u1106\u116A\u11BC" },
            new string [] { "\uBADA", "\uBADA", "\u1106\u116A\u11BD", "\uBADA", "\u1106\u116A\u11BD" },
            new string [] { "\uBADB", "\uBADB", "\u1106\u116A\u11BE", "\uBADB", "\u1106\u116A\u11BE" },
            new string [] { "\uBADC", "\uBADC", "\u1106\u116A\u11BF", "\uBADC", "\u1106\u116A\u11BF" },
            new string [] { "\uBADD", "\uBADD", "\u1106\u116A\u11C0", "\uBADD", "\u1106\u116A\u11C0" },
            new string [] { "\uBADE", "\uBADE", "\u1106\u116A\u11C1", "\uBADE", "\u1106\u116A\u11C1" },
            new string [] { "\uBADF", "\uBADF", "\u1106\u116A\u11C2", "\uBADF", "\u1106\u116A\u11C2" },
            new string [] { "\uBAE0", "\uBAE0", "\u1106\u116B", "\uBAE0", "\u1106\u116B" },
            new string [] { "\uBAE1", "\uBAE1", "\u1106\u116B\u11A8", "\uBAE1", "\u1106\u116B\u11A8" },
            new string [] { "\uBAE2", "\uBAE2", "\u1106\u116B\u11A9", "\uBAE2", "\u1106\u116B\u11A9" },
            new string [] { "\uBAE3", "\uBAE3", "\u1106\u116B\u11AA", "\uBAE3", "\u1106\u116B\u11AA" },
            new string [] { "\uBAE4", "\uBAE4", "\u1106\u116B\u11AB", "\uBAE4", "\u1106\u116B\u11AB" },
            new string [] { "\uBAE5", "\uBAE5", "\u1106\u116B\u11AC", "\uBAE5", "\u1106\u116B\u11AC" },
            new string [] { "\uBAE6", "\uBAE6", "\u1106\u116B\u11AD", "\uBAE6", "\u1106\u116B\u11AD" },
            new string [] { "\uBAE7", "\uBAE7", "\u1106\u116B\u11AE", "\uBAE7", "\u1106\u116B\u11AE" },
            new string [] { "\uBAE8", "\uBAE8", "\u1106\u116B\u11AF", "\uBAE8", "\u1106\u116B\u11AF" },
            new string [] { "\uBAE9", "\uBAE9", "\u1106\u116B\u11B0", "\uBAE9", "\u1106\u116B\u11B0" },
            new string [] { "\uBAEA", "\uBAEA", "\u1106\u116B\u11B1", "\uBAEA", "\u1106\u116B\u11B1" },
            new string [] { "\uBAEB", "\uBAEB", "\u1106\u116B\u11B2", "\uBAEB", "\u1106\u116B\u11B2" },
            new string [] { "\uBAEC", "\uBAEC", "\u1106\u116B\u11B3", "\uBAEC", "\u1106\u116B\u11B3" },
            new string [] { "\uBAED", "\uBAED", "\u1106\u116B\u11B4", "\uBAED", "\u1106\u116B\u11B4" },
            new string [] { "\uBAEE", "\uBAEE", "\u1106\u116B\u11B5", "\uBAEE", "\u1106\u116B\u11B5" },
            new string [] { "\uBAEF", "\uBAEF", "\u1106\u116B\u11B6", "\uBAEF", "\u1106\u116B\u11B6" },
            new string [] { "\uBAF0", "\uBAF0", "\u1106\u116B\u11B7", "\uBAF0", "\u1106\u116B\u11B7" },
            new string [] { "\uBAF1", "\uBAF1", "\u1106\u116B\u11B8", "\uBAF1", "\u1106\u116B\u11B8" },
            new string [] { "\uBAF2", "\uBAF2", "\u1106\u116B\u11B9", "\uBAF2", "\u1106\u116B\u11B9" },
            new string [] { "\uBAF3", "\uBAF3", "\u1106\u116B\u11BA", "\uBAF3", "\u1106\u116B\u11BA" },
            new string [] { "\uBAF4", "\uBAF4", "\u1106\u116B\u11BB", "\uBAF4", "\u1106\u116B\u11BB" },
            new string [] { "\uBAF5", "\uBAF5", "\u1106\u116B\u11BC", "\uBAF5", "\u1106\u116B\u11BC" },
            new string [] { "\uBAF6", "\uBAF6", "\u1106\u116B\u11BD", "\uBAF6", "\u1106\u116B\u11BD" },
            new string [] { "\uBAF7", "\uBAF7", "\u1106\u116B\u11BE", "\uBAF7", "\u1106\u116B\u11BE" },
            new string [] { "\uBAF8", "\uBAF8", "\u1106\u116B\u11BF", "\uBAF8", "\u1106\u116B\u11BF" },
            new string [] { "\uBAF9", "\uBAF9", "\u1106\u116B\u11C0", "\uBAF9", "\u1106\u116B\u11C0" },
            new string [] { "\uBAFA", "\uBAFA", "\u1106\u116B\u11C1", "\uBAFA", "\u1106\u116B\u11C1" },
            new string [] { "\uBAFB", "\uBAFB", "\u1106\u116B\u11C2", "\uBAFB", "\u1106\u116B\u11C2" },
            new string [] { "\uBAFC", "\uBAFC", "\u1106\u116C", "\uBAFC", "\u1106\u116C" },
            new string [] { "\uBAFD", "\uBAFD", "\u1106\u116C\u11A8", "\uBAFD", "\u1106\u116C\u11A8" },
            new string [] { "\uBAFE", "\uBAFE", "\u1106\u116C\u11A9", "\uBAFE", "\u1106\u116C\u11A9" },
            new string [] { "\uBAFF", "\uBAFF", "\u1106\u116C\u11AA", "\uBAFF", "\u1106\u116C\u11AA" },
            new string [] { "\uBB00", "\uBB00", "\u1106\u116C\u11AB", "\uBB00", "\u1106\u116C\u11AB" },
            new string [] { "\uBB01", "\uBB01", "\u1106\u116C\u11AC", "\uBB01", "\u1106\u116C\u11AC" },
            new string [] { "\uBB02", "\uBB02", "\u1106\u116C\u11AD", "\uBB02", "\u1106\u116C\u11AD" },
            new string [] { "\uBB03", "\uBB03", "\u1106\u116C\u11AE", "\uBB03", "\u1106\u116C\u11AE" },
            new string [] { "\uBB04", "\uBB04", "\u1106\u116C\u11AF", "\uBB04", "\u1106\u116C\u11AF" },
            new string [] { "\uBB05", "\uBB05", "\u1106\u116C\u11B0", "\uBB05", "\u1106\u116C\u11B0" },
            new string [] { "\uBB06", "\uBB06", "\u1106\u116C\u11B1", "\uBB06", "\u1106\u116C\u11B1" },
            new string [] { "\uBB07", "\uBB07", "\u1106\u116C\u11B2", "\uBB07", "\u1106\u116C\u11B2" },
            new string [] { "\uBB08", "\uBB08", "\u1106\u116C\u11B3", "\uBB08", "\u1106\u116C\u11B3" },
            new string [] { "\uBB09", "\uBB09", "\u1106\u116C\u11B4", "\uBB09", "\u1106\u116C\u11B4" },
            new string [] { "\uBB0A", "\uBB0A", "\u1106\u116C\u11B5", "\uBB0A", "\u1106\u116C\u11B5" },
            new string [] { "\uBB0B", "\uBB0B", "\u1106\u116C\u11B6", "\uBB0B", "\u1106\u116C\u11B6" },
            new string [] { "\uBB0C", "\uBB0C", "\u1106\u116C\u11B7", "\uBB0C", "\u1106\u116C\u11B7" },
            new string [] { "\uBB0D", "\uBB0D", "\u1106\u116C\u11B8", "\uBB0D", "\u1106\u116C\u11B8" },
            new string [] { "\uBB0E", "\uBB0E", "\u1106\u116C\u11B9", "\uBB0E", "\u1106\u116C\u11B9" },
            new string [] { "\uBB0F", "\uBB0F", "\u1106\u116C\u11BA", "\uBB0F", "\u1106\u116C\u11BA" },
            new string [] { "\uBB10", "\uBB10", "\u1106\u116C\u11BB", "\uBB10", "\u1106\u116C\u11BB" },
            new string [] { "\uBB11", "\uBB11", "\u1106\u116C\u11BC", "\uBB11", "\u1106\u116C\u11BC" },
            new string [] { "\uBB12", "\uBB12", "\u1106\u116C\u11BD", "\uBB12", "\u1106\u116C\u11BD" },
            new string [] { "\uBB13", "\uBB13", "\u1106\u116C\u11BE", "\uBB13", "\u1106\u116C\u11BE" },
            new string [] { "\uBB14", "\uBB14", "\u1106\u116C\u11BF", "\uBB14", "\u1106\u116C\u11BF" },
            new string [] { "\uBB15", "\uBB15", "\u1106\u116C\u11C0", "\uBB15", "\u1106\u116C\u11C0" },
            new string [] { "\uBB16", "\uBB16", "\u1106\u116C\u11C1", "\uBB16", "\u1106\u116C\u11C1" },
            new string [] { "\uBB17", "\uBB17", "\u1106\u116C\u11C2", "\uBB17", "\u1106\u116C\u11C2" },
            new string [] { "\uBB18", "\uBB18", "\u1106\u116D", "\uBB18", "\u1106\u116D" },
            new string [] { "\uBB19", "\uBB19", "\u1106\u116D\u11A8", "\uBB19", "\u1106\u116D\u11A8" },
            new string [] { "\uBB1A", "\uBB1A", "\u1106\u116D\u11A9", "\uBB1A", "\u1106\u116D\u11A9" },
            new string [] { "\uBB1B", "\uBB1B", "\u1106\u116D\u11AA", "\uBB1B", "\u1106\u116D\u11AA" },
            new string [] { "\uBB1C", "\uBB1C", "\u1106\u116D\u11AB", "\uBB1C", "\u1106\u116D\u11AB" },
            new string [] { "\uBB1D", "\uBB1D", "\u1106\u116D\u11AC", "\uBB1D", "\u1106\u116D\u11AC" },
            new string [] { "\uBB1E", "\uBB1E", "\u1106\u116D\u11AD", "\uBB1E", "\u1106\u116D\u11AD" },
            new string [] { "\uBB1F", "\uBB1F", "\u1106\u116D\u11AE", "\uBB1F", "\u1106\u116D\u11AE" },
            new string [] { "\uBB20", "\uBB20", "\u1106\u116D\u11AF", "\uBB20", "\u1106\u116D\u11AF" },
            new string [] { "\uBB21", "\uBB21", "\u1106\u116D\u11B0", "\uBB21", "\u1106\u116D\u11B0" },
            new string [] { "\uBB22", "\uBB22", "\u1106\u116D\u11B1", "\uBB22", "\u1106\u116D\u11B1" },
            new string [] { "\uBB23", "\uBB23", "\u1106\u116D\u11B2", "\uBB23", "\u1106\u116D\u11B2" },
            new string [] { "\uBB24", "\uBB24", "\u1106\u116D\u11B3", "\uBB24", "\u1106\u116D\u11B3" },
            new string [] { "\uBB25", "\uBB25", "\u1106\u116D\u11B4", "\uBB25", "\u1106\u116D\u11B4" },
            new string [] { "\uBB26", "\uBB26", "\u1106\u116D\u11B5", "\uBB26", "\u1106\u116D\u11B5" },
            new string [] { "\uBB27", "\uBB27", "\u1106\u116D\u11B6", "\uBB27", "\u1106\u116D\u11B6" },
            new string [] { "\uBB28", "\uBB28", "\u1106\u116D\u11B7", "\uBB28", "\u1106\u116D\u11B7" },
            new string [] { "\uBB29", "\uBB29", "\u1106\u116D\u11B8", "\uBB29", "\u1106\u116D\u11B8" },
            new string [] { "\uBB2A", "\uBB2A", "\u1106\u116D\u11B9", "\uBB2A", "\u1106\u116D\u11B9" },
            new string [] { "\uBB2B", "\uBB2B", "\u1106\u116D\u11BA", "\uBB2B", "\u1106\u116D\u11BA" },
            new string [] { "\uBB2C", "\uBB2C", "\u1106\u116D\u11BB", "\uBB2C", "\u1106\u116D\u11BB" },
            new string [] { "\uBB2D", "\uBB2D", "\u1106\u116D\u11BC", "\uBB2D", "\u1106\u116D\u11BC" },
            new string [] { "\uBB2E", "\uBB2E", "\u1106\u116D\u11BD", "\uBB2E", "\u1106\u116D\u11BD" },
            new string [] { "\uBB2F", "\uBB2F", "\u1106\u116D\u11BE", "\uBB2F", "\u1106\u116D\u11BE" },
            new string [] { "\uBB30", "\uBB30", "\u1106\u116D\u11BF", "\uBB30", "\u1106\u116D\u11BF" },
            new string [] { "\uBB31", "\uBB31", "\u1106\u116D\u11C0", "\uBB31", "\u1106\u116D\u11C0" },
            new string [] { "\uBB32", "\uBB32", "\u1106\u116D\u11C1", "\uBB32", "\u1106\u116D\u11C1" },
            new string [] { "\uBB33", "\uBB33", "\u1106\u116D\u11C2", "\uBB33", "\u1106\u116D\u11C2" },
            new string [] { "\uBB34", "\uBB34", "\u1106\u116E", "\uBB34", "\u1106\u116E" },
            new string [] { "\uBB35", "\uBB35", "\u1106\u116E\u11A8", "\uBB35", "\u1106\u116E\u11A8" },
            new string [] { "\uBB36", "\uBB36", "\u1106\u116E\u11A9", "\uBB36", "\u1106\u116E\u11A9" },
            new string [] { "\uBB37", "\uBB37", "\u1106\u116E\u11AA", "\uBB37", "\u1106\u116E\u11AA" },
            new string [] { "\uBB38", "\uBB38", "\u1106\u116E\u11AB", "\uBB38", "\u1106\u116E\u11AB" },
            new string [] { "\uBB39", "\uBB39", "\u1106\u116E\u11AC", "\uBB39", "\u1106\u116E\u11AC" },
            new string [] { "\uBB3A", "\uBB3A", "\u1106\u116E\u11AD", "\uBB3A", "\u1106\u116E\u11AD" },
            new string [] { "\uBB3B", "\uBB3B", "\u1106\u116E\u11AE", "\uBB3B", "\u1106\u116E\u11AE" },
            new string [] { "\uBB3C", "\uBB3C", "\u1106\u116E\u11AF", "\uBB3C", "\u1106\u116E\u11AF" },
            new string [] { "\uBB3D", "\uBB3D", "\u1106\u116E\u11B0", "\uBB3D", "\u1106\u116E\u11B0" },
            new string [] { "\uBB3E", "\uBB3E", "\u1106\u116E\u11B1", "\uBB3E", "\u1106\u116E\u11B1" },
            new string [] { "\uBB3F", "\uBB3F", "\u1106\u116E\u11B2", "\uBB3F", "\u1106\u116E\u11B2" },
            new string [] { "\uBB40", "\uBB40", "\u1106\u116E\u11B3", "\uBB40", "\u1106\u116E\u11B3" },
            new string [] { "\uBB41", "\uBB41", "\u1106\u116E\u11B4", "\uBB41", "\u1106\u116E\u11B4" },
            new string [] { "\uBB42", "\uBB42", "\u1106\u116E\u11B5", "\uBB42", "\u1106\u116E\u11B5" },
            new string [] { "\uBB43", "\uBB43", "\u1106\u116E\u11B6", "\uBB43", "\u1106\u116E\u11B6" },
            new string [] { "\uBB44", "\uBB44", "\u1106\u116E\u11B7", "\uBB44", "\u1106\u116E\u11B7" },
            new string [] { "\uBB45", "\uBB45", "\u1106\u116E\u11B8", "\uBB45", "\u1106\u116E\u11B8" },
            new string [] { "\uBB46", "\uBB46", "\u1106\u116E\u11B9", "\uBB46", "\u1106\u116E\u11B9" },
            new string [] { "\uBB47", "\uBB47", "\u1106\u116E\u11BA", "\uBB47", "\u1106\u116E\u11BA" },
            new string [] { "\uBB48", "\uBB48", "\u1106\u116E\u11BB", "\uBB48", "\u1106\u116E\u11BB" },
            new string [] { "\uBB49", "\uBB49", "\u1106\u116E\u11BC", "\uBB49", "\u1106\u116E\u11BC" },
            new string [] { "\uBB4A", "\uBB4A", "\u1106\u116E\u11BD", "\uBB4A", "\u1106\u116E\u11BD" },
            new string [] { "\uBB4B", "\uBB4B", "\u1106\u116E\u11BE", "\uBB4B", "\u1106\u116E\u11BE" },
            new string [] { "\uBB4C", "\uBB4C", "\u1106\u116E\u11BF", "\uBB4C", "\u1106\u116E\u11BF" },
            new string [] { "\uBB4D", "\uBB4D", "\u1106\u116E\u11C0", "\uBB4D", "\u1106\u116E\u11C0" },
            new string [] { "\uBB4E", "\uBB4E", "\u1106\u116E\u11C1", "\uBB4E", "\u1106\u116E\u11C1" },
            new string [] { "\uBB4F", "\uBB4F", "\u1106\u116E\u11C2", "\uBB4F", "\u1106\u116E\u11C2" },
            new string [] { "\uBB50", "\uBB50", "\u1106\u116F", "\uBB50", "\u1106\u116F" },
            new string [] { "\uBB51", "\uBB51", "\u1106\u116F\u11A8", "\uBB51", "\u1106\u116F\u11A8" },
            new string [] { "\uBB52", "\uBB52", "\u1106\u116F\u11A9", "\uBB52", "\u1106\u116F\u11A9" },
            new string [] { "\uBB53", "\uBB53", "\u1106\u116F\u11AA", "\uBB53", "\u1106\u116F\u11AA" },
            new string [] { "\uBB54", "\uBB54", "\u1106\u116F\u11AB", "\uBB54", "\u1106\u116F\u11AB" },
            new string [] { "\uBB55", "\uBB55", "\u1106\u116F\u11AC", "\uBB55", "\u1106\u116F\u11AC" },
            new string [] { "\uBB56", "\uBB56", "\u1106\u116F\u11AD", "\uBB56", "\u1106\u116F\u11AD" },
            new string [] { "\uBB57", "\uBB57", "\u1106\u116F\u11AE", "\uBB57", "\u1106\u116F\u11AE" },
            new string [] { "\uBB58", "\uBB58", "\u1106\u116F\u11AF", "\uBB58", "\u1106\u116F\u11AF" },
            new string [] { "\uBB59", "\uBB59", "\u1106\u116F\u11B0", "\uBB59", "\u1106\u116F\u11B0" },
            new string [] { "\uBB5A", "\uBB5A", "\u1106\u116F\u11B1", "\uBB5A", "\u1106\u116F\u11B1" },
            new string [] { "\uBB5B", "\uBB5B", "\u1106\u116F\u11B2", "\uBB5B", "\u1106\u116F\u11B2" },
            new string [] { "\uBB5C", "\uBB5C", "\u1106\u116F\u11B3", "\uBB5C", "\u1106\u116F\u11B3" },
            new string [] { "\uBB5D", "\uBB5D", "\u1106\u116F\u11B4", "\uBB5D", "\u1106\u116F\u11B4" },
            new string [] { "\uBB5E", "\uBB5E", "\u1106\u116F\u11B5", "\uBB5E", "\u1106\u116F\u11B5" },
            new string [] { "\uBB5F", "\uBB5F", "\u1106\u116F\u11B6", "\uBB5F", "\u1106\u116F\u11B6" },
            new string [] { "\uBB60", "\uBB60", "\u1106\u116F\u11B7", "\uBB60", "\u1106\u116F\u11B7" },
            new string [] { "\uBB61", "\uBB61", "\u1106\u116F\u11B8", "\uBB61", "\u1106\u116F\u11B8" },
            new string [] { "\uBB62", "\uBB62", "\u1106\u116F\u11B9", "\uBB62", "\u1106\u116F\u11B9" },
            new string [] { "\uBB63", "\uBB63", "\u1106\u116F\u11BA", "\uBB63", "\u1106\u116F\u11BA" },
            new string [] { "\uBB64", "\uBB64", "\u1106\u116F\u11BB", "\uBB64", "\u1106\u116F\u11BB" },
            new string [] { "\uBB65", "\uBB65", "\u1106\u116F\u11BC", "\uBB65", "\u1106\u116F\u11BC" },
            new string [] { "\uBB66", "\uBB66", "\u1106\u116F\u11BD", "\uBB66", "\u1106\u116F\u11BD" },
            new string [] { "\uBB67", "\uBB67", "\u1106\u116F\u11BE", "\uBB67", "\u1106\u116F\u11BE" },
            new string [] { "\uBB68", "\uBB68", "\u1106\u116F\u11BF", "\uBB68", "\u1106\u116F\u11BF" },
            new string [] { "\uBB69", "\uBB69", "\u1106\u116F\u11C0", "\uBB69", "\u1106\u116F\u11C0" },
            new string [] { "\uBB6A", "\uBB6A", "\u1106\u116F\u11C1", "\uBB6A", "\u1106\u116F\u11C1" },
            new string [] { "\uBB6B", "\uBB6B", "\u1106\u116F\u11C2", "\uBB6B", "\u1106\u116F\u11C2" },
            new string [] { "\uBB6C", "\uBB6C", "\u1106\u1170", "\uBB6C", "\u1106\u1170" },
            new string [] { "\uBB6D", "\uBB6D", "\u1106\u1170\u11A8", "\uBB6D", "\u1106\u1170\u11A8" },
            new string [] { "\uBB6E", "\uBB6E", "\u1106\u1170\u11A9", "\uBB6E", "\u1106\u1170\u11A9" },
            new string [] { "\uBB6F", "\uBB6F", "\u1106\u1170\u11AA", "\uBB6F", "\u1106\u1170\u11AA" },
            new string [] { "\uBB70", "\uBB70", "\u1106\u1170\u11AB", "\uBB70", "\u1106\u1170\u11AB" },
            new string [] { "\uBB71", "\uBB71", "\u1106\u1170\u11AC", "\uBB71", "\u1106\u1170\u11AC" },
            new string [] { "\uBB72", "\uBB72", "\u1106\u1170\u11AD", "\uBB72", "\u1106\u1170\u11AD" },
            new string [] { "\uBB73", "\uBB73", "\u1106\u1170\u11AE", "\uBB73", "\u1106\u1170\u11AE" },
            new string [] { "\uBB74", "\uBB74", "\u1106\u1170\u11AF", "\uBB74", "\u1106\u1170\u11AF" },
            new string [] { "\uBB75", "\uBB75", "\u1106\u1170\u11B0", "\uBB75", "\u1106\u1170\u11B0" },
            new string [] { "\uBB76", "\uBB76", "\u1106\u1170\u11B1", "\uBB76", "\u1106\u1170\u11B1" },
            new string [] { "\uBB77", "\uBB77", "\u1106\u1170\u11B2", "\uBB77", "\u1106\u1170\u11B2" },
            new string [] { "\uBB78", "\uBB78", "\u1106\u1170\u11B3", "\uBB78", "\u1106\u1170\u11B3" },
            new string [] { "\uBB79", "\uBB79", "\u1106\u1170\u11B4", "\uBB79", "\u1106\u1170\u11B4" },
            new string [] { "\uBB7A", "\uBB7A", "\u1106\u1170\u11B5", "\uBB7A", "\u1106\u1170\u11B5" },
            new string [] { "\uBB7B", "\uBB7B", "\u1106\u1170\u11B6", "\uBB7B", "\u1106\u1170\u11B6" },
            new string [] { "\uBB7C", "\uBB7C", "\u1106\u1170\u11B7", "\uBB7C", "\u1106\u1170\u11B7" },
            new string [] { "\uBB7D", "\uBB7D", "\u1106\u1170\u11B8", "\uBB7D", "\u1106\u1170\u11B8" },
            new string [] { "\uBB7E", "\uBB7E", "\u1106\u1170\u11B9", "\uBB7E", "\u1106\u1170\u11B9" },
            new string [] { "\uBB7F", "\uBB7F", "\u1106\u1170\u11BA", "\uBB7F", "\u1106\u1170\u11BA" },
            new string [] { "\uBB80", "\uBB80", "\u1106\u1170\u11BB", "\uBB80", "\u1106\u1170\u11BB" },
            new string [] { "\uBB81", "\uBB81", "\u1106\u1170\u11BC", "\uBB81", "\u1106\u1170\u11BC" },
            new string [] { "\uBB82", "\uBB82", "\u1106\u1170\u11BD", "\uBB82", "\u1106\u1170\u11BD" },
            new string [] { "\uBB83", "\uBB83", "\u1106\u1170\u11BE", "\uBB83", "\u1106\u1170\u11BE" },
            new string [] { "\uBB84", "\uBB84", "\u1106\u1170\u11BF", "\uBB84", "\u1106\u1170\u11BF" },
            new string [] { "\uBB85", "\uBB85", "\u1106\u1170\u11C0", "\uBB85", "\u1106\u1170\u11C0" },
            new string [] { "\uBB86", "\uBB86", "\u1106\u1170\u11C1", "\uBB86", "\u1106\u1170\u11C1" },
            new string [] { "\uBB87", "\uBB87", "\u1106\u1170\u11C2", "\uBB87", "\u1106\u1170\u11C2" },
            new string [] { "\uBB88", "\uBB88", "\u1106\u1171", "\uBB88", "\u1106\u1171" },
            new string [] { "\uBB89", "\uBB89", "\u1106\u1171\u11A8", "\uBB89", "\u1106\u1171\u11A8" },
            new string [] { "\uBB8A", "\uBB8A", "\u1106\u1171\u11A9", "\uBB8A", "\u1106\u1171\u11A9" },
            new string [] { "\uBB8B", "\uBB8B", "\u1106\u1171\u11AA", "\uBB8B", "\u1106\u1171\u11AA" },
            new string [] { "\uBB8C", "\uBB8C", "\u1106\u1171\u11AB", "\uBB8C", "\u1106\u1171\u11AB" },
            new string [] { "\uBB8D", "\uBB8D", "\u1106\u1171\u11AC", "\uBB8D", "\u1106\u1171\u11AC" },
            new string [] { "\uBB8E", "\uBB8E", "\u1106\u1171\u11AD", "\uBB8E", "\u1106\u1171\u11AD" },
            new string [] { "\uBB8F", "\uBB8F", "\u1106\u1171\u11AE", "\uBB8F", "\u1106\u1171\u11AE" },
            new string [] { "\uBB90", "\uBB90", "\u1106\u1171\u11AF", "\uBB90", "\u1106\u1171\u11AF" },
            new string [] { "\uBB91", "\uBB91", "\u1106\u1171\u11B0", "\uBB91", "\u1106\u1171\u11B0" },
            new string [] { "\uBB92", "\uBB92", "\u1106\u1171\u11B1", "\uBB92", "\u1106\u1171\u11B1" },
            new string [] { "\uBB93", "\uBB93", "\u1106\u1171\u11B2", "\uBB93", "\u1106\u1171\u11B2" },
            new string [] { "\uBB94", "\uBB94", "\u1106\u1171\u11B3", "\uBB94", "\u1106\u1171\u11B3" },
            new string [] { "\uBB95", "\uBB95", "\u1106\u1171\u11B4", "\uBB95", "\u1106\u1171\u11B4" },
            new string [] { "\uBB96", "\uBB96", "\u1106\u1171\u11B5", "\uBB96", "\u1106\u1171\u11B5" },
            new string [] { "\uBB97", "\uBB97", "\u1106\u1171\u11B6", "\uBB97", "\u1106\u1171\u11B6" },
            new string [] { "\uBB98", "\uBB98", "\u1106\u1171\u11B7", "\uBB98", "\u1106\u1171\u11B7" },
            new string [] { "\uBB99", "\uBB99", "\u1106\u1171\u11B8", "\uBB99", "\u1106\u1171\u11B8" },
            new string [] { "\uBB9A", "\uBB9A", "\u1106\u1171\u11B9", "\uBB9A", "\u1106\u1171\u11B9" },
            new string [] { "\uBB9B", "\uBB9B", "\u1106\u1171\u11BA", "\uBB9B", "\u1106\u1171\u11BA" },
            new string [] { "\uBB9C", "\uBB9C", "\u1106\u1171\u11BB", "\uBB9C", "\u1106\u1171\u11BB" },
            new string [] { "\uBB9D", "\uBB9D", "\u1106\u1171\u11BC", "\uBB9D", "\u1106\u1171\u11BC" },
            new string [] { "\uBB9E", "\uBB9E", "\u1106\u1171\u11BD", "\uBB9E", "\u1106\u1171\u11BD" },
            new string [] { "\uBB9F", "\uBB9F", "\u1106\u1171\u11BE", "\uBB9F", "\u1106\u1171\u11BE" },
            new string [] { "\uBBA0", "\uBBA0", "\u1106\u1171\u11BF", "\uBBA0", "\u1106\u1171\u11BF" },
            new string [] { "\uBBA1", "\uBBA1", "\u1106\u1171\u11C0", "\uBBA1", "\u1106\u1171\u11C0" },
            new string [] { "\uBBA2", "\uBBA2", "\u1106\u1171\u11C1", "\uBBA2", "\u1106\u1171\u11C1" },
            new string [] { "\uBBA3", "\uBBA3", "\u1106\u1171\u11C2", "\uBBA3", "\u1106\u1171\u11C2" },
            new string [] { "\uBBA4", "\uBBA4", "\u1106\u1172", "\uBBA4", "\u1106\u1172" },
            new string [] { "\uBBA5", "\uBBA5", "\u1106\u1172\u11A8", "\uBBA5", "\u1106\u1172\u11A8" },
            new string [] { "\uBBA6", "\uBBA6", "\u1106\u1172\u11A9", "\uBBA6", "\u1106\u1172\u11A9" },
            new string [] { "\uBBA7", "\uBBA7", "\u1106\u1172\u11AA", "\uBBA7", "\u1106\u1172\u11AA" },
            new string [] { "\uBBA8", "\uBBA8", "\u1106\u1172\u11AB", "\uBBA8", "\u1106\u1172\u11AB" },
            new string [] { "\uBBA9", "\uBBA9", "\u1106\u1172\u11AC", "\uBBA9", "\u1106\u1172\u11AC" },
            new string [] { "\uBBAA", "\uBBAA", "\u1106\u1172\u11AD", "\uBBAA", "\u1106\u1172\u11AD" },
            new string [] { "\uBBAB", "\uBBAB", "\u1106\u1172\u11AE", "\uBBAB", "\u1106\u1172\u11AE" },
            new string [] { "\uBBAC", "\uBBAC", "\u1106\u1172\u11AF", "\uBBAC", "\u1106\u1172\u11AF" },
            new string [] { "\uBBAD", "\uBBAD", "\u1106\u1172\u11B0", "\uBBAD", "\u1106\u1172\u11B0" },
            new string [] { "\uBBAE", "\uBBAE", "\u1106\u1172\u11B1", "\uBBAE", "\u1106\u1172\u11B1" },
            new string [] { "\uBBAF", "\uBBAF", "\u1106\u1172\u11B2", "\uBBAF", "\u1106\u1172\u11B2" },
            new string [] { "\uBBB0", "\uBBB0", "\u1106\u1172\u11B3", "\uBBB0", "\u1106\u1172\u11B3" },
            new string [] { "\uBBB1", "\uBBB1", "\u1106\u1172\u11B4", "\uBBB1", "\u1106\u1172\u11B4" },
            new string [] { "\uBBB2", "\uBBB2", "\u1106\u1172\u11B5", "\uBBB2", "\u1106\u1172\u11B5" },
            new string [] { "\uBBB3", "\uBBB3", "\u1106\u1172\u11B6", "\uBBB3", "\u1106\u1172\u11B6" },
            new string [] { "\uBBB4", "\uBBB4", "\u1106\u1172\u11B7", "\uBBB4", "\u1106\u1172\u11B7" },
            new string [] { "\uBBB5", "\uBBB5", "\u1106\u1172\u11B8", "\uBBB5", "\u1106\u1172\u11B8" },
            new string [] { "\uBBB6", "\uBBB6", "\u1106\u1172\u11B9", "\uBBB6", "\u1106\u1172\u11B9" },
            new string [] { "\uBBB7", "\uBBB7", "\u1106\u1172\u11BA", "\uBBB7", "\u1106\u1172\u11BA" },
            new string [] { "\uBBB8", "\uBBB8", "\u1106\u1172\u11BB", "\uBBB8", "\u1106\u1172\u11BB" },
            new string [] { "\uBBB9", "\uBBB9", "\u1106\u1172\u11BC", "\uBBB9", "\u1106\u1172\u11BC" },
            new string [] { "\uBBBA", "\uBBBA", "\u1106\u1172\u11BD", "\uBBBA", "\u1106\u1172\u11BD" },
            new string [] { "\uBBBB", "\uBBBB", "\u1106\u1172\u11BE", "\uBBBB", "\u1106\u1172\u11BE" },
            new string [] { "\uBBBC", "\uBBBC", "\u1106\u1172\u11BF", "\uBBBC", "\u1106\u1172\u11BF" },
            new string [] { "\uBBBD", "\uBBBD", "\u1106\u1172\u11C0", "\uBBBD", "\u1106\u1172\u11C0" },
            new string [] { "\uBBBE", "\uBBBE", "\u1106\u1172\u11C1", "\uBBBE", "\u1106\u1172\u11C1" },
            new string [] { "\uBBBF", "\uBBBF", "\u1106\u1172\u11C2", "\uBBBF", "\u1106\u1172\u11C2" },
            new string [] { "\uBBC0", "\uBBC0", "\u1106\u1173", "\uBBC0", "\u1106\u1173" },
            new string [] { "\uBBC1", "\uBBC1", "\u1106\u1173\u11A8", "\uBBC1", "\u1106\u1173\u11A8" },
            new string [] { "\uBBC2", "\uBBC2", "\u1106\u1173\u11A9", "\uBBC2", "\u1106\u1173\u11A9" },
            new string [] { "\uBBC3", "\uBBC3", "\u1106\u1173\u11AA", "\uBBC3", "\u1106\u1173\u11AA" },
            new string [] { "\uBBC4", "\uBBC4", "\u1106\u1173\u11AB", "\uBBC4", "\u1106\u1173\u11AB" },
            new string [] { "\uBBC5", "\uBBC5", "\u1106\u1173\u11AC", "\uBBC5", "\u1106\u1173\u11AC" },
            new string [] { "\uBBC6", "\uBBC6", "\u1106\u1173\u11AD", "\uBBC6", "\u1106\u1173\u11AD" },
            new string [] { "\uBBC7", "\uBBC7", "\u1106\u1173\u11AE", "\uBBC7", "\u1106\u1173\u11AE" },
            new string [] { "\uBBC8", "\uBBC8", "\u1106\u1173\u11AF", "\uBBC8", "\u1106\u1173\u11AF" },
            new string [] { "\uBBC9", "\uBBC9", "\u1106\u1173\u11B0", "\uBBC9", "\u1106\u1173\u11B0" },
            new string [] { "\uBBCA", "\uBBCA", "\u1106\u1173\u11B1", "\uBBCA", "\u1106\u1173\u11B1" },
            new string [] { "\uBBCB", "\uBBCB", "\u1106\u1173\u11B2", "\uBBCB", "\u1106\u1173\u11B2" },
            new string [] { "\uBBCC", "\uBBCC", "\u1106\u1173\u11B3", "\uBBCC", "\u1106\u1173\u11B3" },
            new string [] { "\uBBCD", "\uBBCD", "\u1106\u1173\u11B4", "\uBBCD", "\u1106\u1173\u11B4" },
            new string [] { "\uBBCE", "\uBBCE", "\u1106\u1173\u11B5", "\uBBCE", "\u1106\u1173\u11B5" },
            new string [] { "\uBBCF", "\uBBCF", "\u1106\u1173\u11B6", "\uBBCF", "\u1106\u1173\u11B6" },
            new string [] { "\uBBD0", "\uBBD0", "\u1106\u1173\u11B7", "\uBBD0", "\u1106\u1173\u11B7" },
            new string [] { "\uBBD1", "\uBBD1", "\u1106\u1173\u11B8", "\uBBD1", "\u1106\u1173\u11B8" },
            new string [] { "\uBBD2", "\uBBD2", "\u1106\u1173\u11B9", "\uBBD2", "\u1106\u1173\u11B9" },
            new string [] { "\uBBD3", "\uBBD3", "\u1106\u1173\u11BA", "\uBBD3", "\u1106\u1173\u11BA" },
            new string [] { "\uBBD4", "\uBBD4", "\u1106\u1173\u11BB", "\uBBD4", "\u1106\u1173\u11BB" },
            new string [] { "\uBBD5", "\uBBD5", "\u1106\u1173\u11BC", "\uBBD5", "\u1106\u1173\u11BC" },
            new string [] { "\uBBD6", "\uBBD6", "\u1106\u1173\u11BD", "\uBBD6", "\u1106\u1173\u11BD" },
            new string [] { "\uBBD7", "\uBBD7", "\u1106\u1173\u11BE", "\uBBD7", "\u1106\u1173\u11BE" },
            new string [] { "\uBBD8", "\uBBD8", "\u1106\u1173\u11BF", "\uBBD8", "\u1106\u1173\u11BF" },
            new string [] { "\uBBD9", "\uBBD9", "\u1106\u1173\u11C0", "\uBBD9", "\u1106\u1173\u11C0" },
            new string [] { "\uBBDA", "\uBBDA", "\u1106\u1173\u11C1", "\uBBDA", "\u1106\u1173\u11C1" },
            new string [] { "\uBBDB", "\uBBDB", "\u1106\u1173\u11C2", "\uBBDB", "\u1106\u1173\u11C2" },
            new string [] { "\uBBDC", "\uBBDC", "\u1106\u1174", "\uBBDC", "\u1106\u1174" },
            new string [] { "\uBBDD", "\uBBDD", "\u1106\u1174\u11A8", "\uBBDD", "\u1106\u1174\u11A8" },
            new string [] { "\uBBDE", "\uBBDE", "\u1106\u1174\u11A9", "\uBBDE", "\u1106\u1174\u11A9" },
            new string [] { "\uBBDF", "\uBBDF", "\u1106\u1174\u11AA", "\uBBDF", "\u1106\u1174\u11AA" },
            new string [] { "\uBBE0", "\uBBE0", "\u1106\u1174\u11AB", "\uBBE0", "\u1106\u1174\u11AB" },
            new string [] { "\uBBE1", "\uBBE1", "\u1106\u1174\u11AC", "\uBBE1", "\u1106\u1174\u11AC" },
            new string [] { "\uBBE2", "\uBBE2", "\u1106\u1174\u11AD", "\uBBE2", "\u1106\u1174\u11AD" },
            new string [] { "\uBBE3", "\uBBE3", "\u1106\u1174\u11AE", "\uBBE3", "\u1106\u1174\u11AE" },
            new string [] { "\uBBE4", "\uBBE4", "\u1106\u1174\u11AF", "\uBBE4", "\u1106\u1174\u11AF" },
            new string [] { "\uBBE5", "\uBBE5", "\u1106\u1174\u11B0", "\uBBE5", "\u1106\u1174\u11B0" },
            new string [] { "\uBBE6", "\uBBE6", "\u1106\u1174\u11B1", "\uBBE6", "\u1106\u1174\u11B1" },
            new string [] { "\uBBE7", "\uBBE7", "\u1106\u1174\u11B2", "\uBBE7", "\u1106\u1174\u11B2" },
            new string [] { "\uBBE8", "\uBBE8", "\u1106\u1174\u11B3", "\uBBE8", "\u1106\u1174\u11B3" },
            new string [] { "\uBBE9", "\uBBE9", "\u1106\u1174\u11B4", "\uBBE9", "\u1106\u1174\u11B4" },
            new string [] { "\uBBEA", "\uBBEA", "\u1106\u1174\u11B5", "\uBBEA", "\u1106\u1174\u11B5" },
            new string [] { "\uBBEB", "\uBBEB", "\u1106\u1174\u11B6", "\uBBEB", "\u1106\u1174\u11B6" },
            new string [] { "\uBBEC", "\uBBEC", "\u1106\u1174\u11B7", "\uBBEC", "\u1106\u1174\u11B7" },
            new string [] { "\uBBED", "\uBBED", "\u1106\u1174\u11B8", "\uBBED", "\u1106\u1174\u11B8" },
            new string [] { "\uBBEE", "\uBBEE", "\u1106\u1174\u11B9", "\uBBEE", "\u1106\u1174\u11B9" },
            new string [] { "\uBBEF", "\uBBEF", "\u1106\u1174\u11BA", "\uBBEF", "\u1106\u1174\u11BA" },
            new string [] { "\uBBF0", "\uBBF0", "\u1106\u1174\u11BB", "\uBBF0", "\u1106\u1174\u11BB" },
            new string [] { "\uBBF1", "\uBBF1", "\u1106\u1174\u11BC", "\uBBF1", "\u1106\u1174\u11BC" },
            new string [] { "\uBBF2", "\uBBF2", "\u1106\u1174\u11BD", "\uBBF2", "\u1106\u1174\u11BD" },
            new string [] { "\uBBF3", "\uBBF3", "\u1106\u1174\u11BE", "\uBBF3", "\u1106\u1174\u11BE" },
            new string [] { "\uBBF4", "\uBBF4", "\u1106\u1174\u11BF", "\uBBF4", "\u1106\u1174\u11BF" },
            new string [] { "\uBBF5", "\uBBF5", "\u1106\u1174\u11C0", "\uBBF5", "\u1106\u1174\u11C0" },
            new string [] { "\uBBF6", "\uBBF6", "\u1106\u1174\u11C1", "\uBBF6", "\u1106\u1174\u11C1" },
            new string [] { "\uBBF7", "\uBBF7", "\u1106\u1174\u11C2", "\uBBF7", "\u1106\u1174\u11C2" },
            new string [] { "\uBBF8", "\uBBF8", "\u1106\u1175", "\uBBF8", "\u1106\u1175" },
            new string [] { "\uBBF9", "\uBBF9", "\u1106\u1175\u11A8", "\uBBF9", "\u1106\u1175\u11A8" },
            new string [] { "\uBBFA", "\uBBFA", "\u1106\u1175\u11A9", "\uBBFA", "\u1106\u1175\u11A9" },
            new string [] { "\uBBFB", "\uBBFB", "\u1106\u1175\u11AA", "\uBBFB", "\u1106\u1175\u11AA" },
            new string [] { "\uBBFC", "\uBBFC", "\u1106\u1175\u11AB", "\uBBFC", "\u1106\u1175\u11AB" },
            new string [] { "\uBBFD", "\uBBFD", "\u1106\u1175\u11AC", "\uBBFD", "\u1106\u1175\u11AC" },
            new string [] { "\uBBFE", "\uBBFE", "\u1106\u1175\u11AD", "\uBBFE", "\u1106\u1175\u11AD" },
            new string [] { "\uBBFF", "\uBBFF", "\u1106\u1175\u11AE", "\uBBFF", "\u1106\u1175\u11AE" },
            new string [] { "\uBC00", "\uBC00", "\u1106\u1175\u11AF", "\uBC00", "\u1106\u1175\u11AF" },
            new string [] { "\uBC01", "\uBC01", "\u1106\u1175\u11B0", "\uBC01", "\u1106\u1175\u11B0" },
            new string [] { "\uBC02", "\uBC02", "\u1106\u1175\u11B1", "\uBC02", "\u1106\u1175\u11B1" },
            new string [] { "\uBC03", "\uBC03", "\u1106\u1175\u11B2", "\uBC03", "\u1106\u1175\u11B2" },
            new string [] { "\uBC04", "\uBC04", "\u1106\u1175\u11B3", "\uBC04", "\u1106\u1175\u11B3" },
            new string [] { "\uBC05", "\uBC05", "\u1106\u1175\u11B4", "\uBC05", "\u1106\u1175\u11B4" },
            new string [] { "\uBC06", "\uBC06", "\u1106\u1175\u11B5", "\uBC06", "\u1106\u1175\u11B5" },
            new string [] { "\uBC07", "\uBC07", "\u1106\u1175\u11B6", "\uBC07", "\u1106\u1175\u11B6" },
            new string [] { "\uBC08", "\uBC08", "\u1106\u1175\u11B7", "\uBC08", "\u1106\u1175\u11B7" },
            new string [] { "\uBC09", "\uBC09", "\u1106\u1175\u11B8", "\uBC09", "\u1106\u1175\u11B8" },
            new string [] { "\uBC0A", "\uBC0A", "\u1106\u1175\u11B9", "\uBC0A", "\u1106\u1175\u11B9" },
            new string [] { "\uBC0B", "\uBC0B", "\u1106\u1175\u11BA", "\uBC0B", "\u1106\u1175\u11BA" },
            new string [] { "\uBC0C", "\uBC0C", "\u1106\u1175\u11BB", "\uBC0C", "\u1106\u1175\u11BB" },
            new string [] { "\uBC0D", "\uBC0D", "\u1106\u1175\u11BC", "\uBC0D", "\u1106\u1175\u11BC" },
            new string [] { "\uBC0E", "\uBC0E", "\u1106\u1175\u11BD", "\uBC0E", "\u1106\u1175\u11BD" },
            new string [] { "\uBC0F", "\uBC0F", "\u1106\u1175\u11BE", "\uBC0F", "\u1106\u1175\u11BE" },
            new string [] { "\uBC10", "\uBC10", "\u1106\u1175\u11BF", "\uBC10", "\u1106\u1175\u11BF" },
            new string [] { "\uBC11", "\uBC11", "\u1106\u1175\u11C0", "\uBC11", "\u1106\u1175\u11C0" },
            new string [] { "\uBC12", "\uBC12", "\u1106\u1175\u11C1", "\uBC12", "\u1106\u1175\u11C1" },
            new string [] { "\uBC13", "\uBC13", "\u1106\u1175\u11C2", "\uBC13", "\u1106\u1175\u11C2" },
            new string [] { "\uBC14", "\uBC14", "\u1107\u1161", "\uBC14", "\u1107\u1161" },
            new string [] { "\uBC15", "\uBC15", "\u1107\u1161\u11A8", "\uBC15", "\u1107\u1161\u11A8" },
            new string [] { "\uBC16", "\uBC16", "\u1107\u1161\u11A9", "\uBC16", "\u1107\u1161\u11A9" },
            new string [] { "\uBC17", "\uBC17", "\u1107\u1161\u11AA", "\uBC17", "\u1107\u1161\u11AA" },
            new string [] { "\uBC18", "\uBC18", "\u1107\u1161\u11AB", "\uBC18", "\u1107\u1161\u11AB" },
            new string [] { "\uBC19", "\uBC19", "\u1107\u1161\u11AC", "\uBC19", "\u1107\u1161\u11AC" },
            new string [] { "\uBC1A", "\uBC1A", "\u1107\u1161\u11AD", "\uBC1A", "\u1107\u1161\u11AD" },
            new string [] { "\uBC1B", "\uBC1B", "\u1107\u1161\u11AE", "\uBC1B", "\u1107\u1161\u11AE" },
            new string [] { "\uBC1C", "\uBC1C", "\u1107\u1161\u11AF", "\uBC1C", "\u1107\u1161\u11AF" },
            new string [] { "\uBC1D", "\uBC1D", "\u1107\u1161\u11B0", "\uBC1D", "\u1107\u1161\u11B0" },
            new string [] { "\uBC1E", "\uBC1E", "\u1107\u1161\u11B1", "\uBC1E", "\u1107\u1161\u11B1" },
            new string [] { "\uBC1F", "\uBC1F", "\u1107\u1161\u11B2", "\uBC1F", "\u1107\u1161\u11B2" },
            new string [] { "\uBC20", "\uBC20", "\u1107\u1161\u11B3", "\uBC20", "\u1107\u1161\u11B3" },
            new string [] { "\uBC21", "\uBC21", "\u1107\u1161\u11B4", "\uBC21", "\u1107\u1161\u11B4" },
            new string [] { "\uBC22", "\uBC22", "\u1107\u1161\u11B5", "\uBC22", "\u1107\u1161\u11B5" },
            new string [] { "\uBC23", "\uBC23", "\u1107\u1161\u11B6", "\uBC23", "\u1107\u1161\u11B6" },
            new string [] { "\uBC24", "\uBC24", "\u1107\u1161\u11B7", "\uBC24", "\u1107\u1161\u11B7" },
            new string [] { "\uBC25", "\uBC25", "\u1107\u1161\u11B8", "\uBC25", "\u1107\u1161\u11B8" },
            new string [] { "\uBC26", "\uBC26", "\u1107\u1161\u11B9", "\uBC26", "\u1107\u1161\u11B9" },
            new string [] { "\uBC27", "\uBC27", "\u1107\u1161\u11BA", "\uBC27", "\u1107\u1161\u11BA" },
            new string [] { "\uBC28", "\uBC28", "\u1107\u1161\u11BB", "\uBC28", "\u1107\u1161\u11BB" },
            new string [] { "\uBC29", "\uBC29", "\u1107\u1161\u11BC", "\uBC29", "\u1107\u1161\u11BC" },
            new string [] { "\uBC2A", "\uBC2A", "\u1107\u1161\u11BD", "\uBC2A", "\u1107\u1161\u11BD" },
            new string [] { "\uBC2B", "\uBC2B", "\u1107\u1161\u11BE", "\uBC2B", "\u1107\u1161\u11BE" },
            new string [] { "\uBC2C", "\uBC2C", "\u1107\u1161\u11BF", "\uBC2C", "\u1107\u1161\u11BF" },
            new string [] { "\uBC2D", "\uBC2D", "\u1107\u1161\u11C0", "\uBC2D", "\u1107\u1161\u11C0" },
            new string [] { "\uBC2E", "\uBC2E", "\u1107\u1161\u11C1", "\uBC2E", "\u1107\u1161\u11C1" },
            new string [] { "\uBC2F", "\uBC2F", "\u1107\u1161\u11C2", "\uBC2F", "\u1107\u1161\u11C2" },
            new string [] { "\uBC30", "\uBC30", "\u1107\u1162", "\uBC30", "\u1107\u1162" },
            new string [] { "\uBC31", "\uBC31", "\u1107\u1162\u11A8", "\uBC31", "\u1107\u1162\u11A8" },
            new string [] { "\uBC32", "\uBC32", "\u1107\u1162\u11A9", "\uBC32", "\u1107\u1162\u11A9" },
            new string [] { "\uBC33", "\uBC33", "\u1107\u1162\u11AA", "\uBC33", "\u1107\u1162\u11AA" },
            new string [] { "\uBC34", "\uBC34", "\u1107\u1162\u11AB", "\uBC34", "\u1107\u1162\u11AB" },
            new string [] { "\uBC35", "\uBC35", "\u1107\u1162\u11AC", "\uBC35", "\u1107\u1162\u11AC" },
            new string [] { "\uBC36", "\uBC36", "\u1107\u1162\u11AD", "\uBC36", "\u1107\u1162\u11AD" },
            new string [] { "\uBC37", "\uBC37", "\u1107\u1162\u11AE", "\uBC37", "\u1107\u1162\u11AE" },
            new string [] { "\uBC38", "\uBC38", "\u1107\u1162\u11AF", "\uBC38", "\u1107\u1162\u11AF" },
            new string [] { "\uBC39", "\uBC39", "\u1107\u1162\u11B0", "\uBC39", "\u1107\u1162\u11B0" },
            new string [] { "\uBC3A", "\uBC3A", "\u1107\u1162\u11B1", "\uBC3A", "\u1107\u1162\u11B1" },
            new string [] { "\uBC3B", "\uBC3B", "\u1107\u1162\u11B2", "\uBC3B", "\u1107\u1162\u11B2" },
            new string [] { "\uBC3C", "\uBC3C", "\u1107\u1162\u11B3", "\uBC3C", "\u1107\u1162\u11B3" },
            new string [] { "\uBC3D", "\uBC3D", "\u1107\u1162\u11B4", "\uBC3D", "\u1107\u1162\u11B4" },
            new string [] { "\uBC3E", "\uBC3E", "\u1107\u1162\u11B5", "\uBC3E", "\u1107\u1162\u11B5" },
            new string [] { "\uBC3F", "\uBC3F", "\u1107\u1162\u11B6", "\uBC3F", "\u1107\u1162\u11B6" },
            new string [] { "\uBC40", "\uBC40", "\u1107\u1162\u11B7", "\uBC40", "\u1107\u1162\u11B7" },
            new string [] { "\uBC41", "\uBC41", "\u1107\u1162\u11B8", "\uBC41", "\u1107\u1162\u11B8" },
            new string [] { "\uBC42", "\uBC42", "\u1107\u1162\u11B9", "\uBC42", "\u1107\u1162\u11B9" },
            new string [] { "\uBC43", "\uBC43", "\u1107\u1162\u11BA", "\uBC43", "\u1107\u1162\u11BA" },
            new string [] { "\uBC44", "\uBC44", "\u1107\u1162\u11BB", "\uBC44", "\u1107\u1162\u11BB" },
            new string [] { "\uBC45", "\uBC45", "\u1107\u1162\u11BC", "\uBC45", "\u1107\u1162\u11BC" },
            new string [] { "\uBC46", "\uBC46", "\u1107\u1162\u11BD", "\uBC46", "\u1107\u1162\u11BD" },
            new string [] { "\uBC47", "\uBC47", "\u1107\u1162\u11BE", "\uBC47", "\u1107\u1162\u11BE" },
            new string [] { "\uBC48", "\uBC48", "\u1107\u1162\u11BF", "\uBC48", "\u1107\u1162\u11BF" },
            new string [] { "\uBC49", "\uBC49", "\u1107\u1162\u11C0", "\uBC49", "\u1107\u1162\u11C0" },
            new string [] { "\uBC4A", "\uBC4A", "\u1107\u1162\u11C1", "\uBC4A", "\u1107\u1162\u11C1" },
            new string [] { "\uBC4B", "\uBC4B", "\u1107\u1162\u11C2", "\uBC4B", "\u1107\u1162\u11C2" },
            new string [] { "\uBC4C", "\uBC4C", "\u1107\u1163", "\uBC4C", "\u1107\u1163" },
            new string [] { "\uBC4D", "\uBC4D", "\u1107\u1163\u11A8", "\uBC4D", "\u1107\u1163\u11A8" },
            new string [] { "\uBC4E", "\uBC4E", "\u1107\u1163\u11A9", "\uBC4E", "\u1107\u1163\u11A9" },
            new string [] { "\uBC4F", "\uBC4F", "\u1107\u1163\u11AA", "\uBC4F", "\u1107\u1163\u11AA" },
            new string [] { "\uBC50", "\uBC50", "\u1107\u1163\u11AB", "\uBC50", "\u1107\u1163\u11AB" },
            new string [] { "\uBC51", "\uBC51", "\u1107\u1163\u11AC", "\uBC51", "\u1107\u1163\u11AC" },
            new string [] { "\uBC52", "\uBC52", "\u1107\u1163\u11AD", "\uBC52", "\u1107\u1163\u11AD" },
            new string [] { "\uBC53", "\uBC53", "\u1107\u1163\u11AE", "\uBC53", "\u1107\u1163\u11AE" },
            new string [] { "\uBC54", "\uBC54", "\u1107\u1163\u11AF", "\uBC54", "\u1107\u1163\u11AF" },
            new string [] { "\uBC55", "\uBC55", "\u1107\u1163\u11B0", "\uBC55", "\u1107\u1163\u11B0" },
            new string [] { "\uBC56", "\uBC56", "\u1107\u1163\u11B1", "\uBC56", "\u1107\u1163\u11B1" },
            new string [] { "\uBC57", "\uBC57", "\u1107\u1163\u11B2", "\uBC57", "\u1107\u1163\u11B2" },
            new string [] { "\uBC58", "\uBC58", "\u1107\u1163\u11B3", "\uBC58", "\u1107\u1163\u11B3" },
            new string [] { "\uBC59", "\uBC59", "\u1107\u1163\u11B4", "\uBC59", "\u1107\u1163\u11B4" },
            new string [] { "\uBC5A", "\uBC5A", "\u1107\u1163\u11B5", "\uBC5A", "\u1107\u1163\u11B5" },
            new string [] { "\uBC5B", "\uBC5B", "\u1107\u1163\u11B6", "\uBC5B", "\u1107\u1163\u11B6" },
            new string [] { "\uBC5C", "\uBC5C", "\u1107\u1163\u11B7", "\uBC5C", "\u1107\u1163\u11B7" },
            new string [] { "\uBC5D", "\uBC5D", "\u1107\u1163\u11B8", "\uBC5D", "\u1107\u1163\u11B8" },
            new string [] { "\uBC5E", "\uBC5E", "\u1107\u1163\u11B9", "\uBC5E", "\u1107\u1163\u11B9" },
            new string [] { "\uBC5F", "\uBC5F", "\u1107\u1163\u11BA", "\uBC5F", "\u1107\u1163\u11BA" },
            new string [] { "\uBC60", "\uBC60", "\u1107\u1163\u11BB", "\uBC60", "\u1107\u1163\u11BB" },
            new string [] { "\uBC61", "\uBC61", "\u1107\u1163\u11BC", "\uBC61", "\u1107\u1163\u11BC" },
            new string [] { "\uBC62", "\uBC62", "\u1107\u1163\u11BD", "\uBC62", "\u1107\u1163\u11BD" },
            new string [] { "\uBC63", "\uBC63", "\u1107\u1163\u11BE", "\uBC63", "\u1107\u1163\u11BE" },
            new string [] { "\uBC64", "\uBC64", "\u1107\u1163\u11BF", "\uBC64", "\u1107\u1163\u11BF" },
            new string [] { "\uBC65", "\uBC65", "\u1107\u1163\u11C0", "\uBC65", "\u1107\u1163\u11C0" },
            new string [] { "\uBC66", "\uBC66", "\u1107\u1163\u11C1", "\uBC66", "\u1107\u1163\u11C1" },
            new string [] { "\uBC67", "\uBC67", "\u1107\u1163\u11C2", "\uBC67", "\u1107\u1163\u11C2" },
            new string [] { "\uBC68", "\uBC68", "\u1107\u1164", "\uBC68", "\u1107\u1164" },
            new string [] { "\uBC69", "\uBC69", "\u1107\u1164\u11A8", "\uBC69", "\u1107\u1164\u11A8" },
            new string [] { "\uBC6A", "\uBC6A", "\u1107\u1164\u11A9", "\uBC6A", "\u1107\u1164\u11A9" },
            new string [] { "\uBC6B", "\uBC6B", "\u1107\u1164\u11AA", "\uBC6B", "\u1107\u1164\u11AA" },
            new string [] { "\uBC6C", "\uBC6C", "\u1107\u1164\u11AB", "\uBC6C", "\u1107\u1164\u11AB" },
            new string [] { "\uBC6D", "\uBC6D", "\u1107\u1164\u11AC", "\uBC6D", "\u1107\u1164\u11AC" },
            new string [] { "\uBC6E", "\uBC6E", "\u1107\u1164\u11AD", "\uBC6E", "\u1107\u1164\u11AD" },
            new string [] { "\uBC6F", "\uBC6F", "\u1107\u1164\u11AE", "\uBC6F", "\u1107\u1164\u11AE" },
            new string [] { "\uBC70", "\uBC70", "\u1107\u1164\u11AF", "\uBC70", "\u1107\u1164\u11AF" },
            new string [] { "\uBC71", "\uBC71", "\u1107\u1164\u11B0", "\uBC71", "\u1107\u1164\u11B0" },
            new string [] { "\uBC72", "\uBC72", "\u1107\u1164\u11B1", "\uBC72", "\u1107\u1164\u11B1" },
            new string [] { "\uBC73", "\uBC73", "\u1107\u1164\u11B2", "\uBC73", "\u1107\u1164\u11B2" },
            new string [] { "\uBC74", "\uBC74", "\u1107\u1164\u11B3", "\uBC74", "\u1107\u1164\u11B3" },
            new string [] { "\uBC75", "\uBC75", "\u1107\u1164\u11B4", "\uBC75", "\u1107\u1164\u11B4" },
            new string [] { "\uBC76", "\uBC76", "\u1107\u1164\u11B5", "\uBC76", "\u1107\u1164\u11B5" },
            new string [] { "\uBC77", "\uBC77", "\u1107\u1164\u11B6", "\uBC77", "\u1107\u1164\u11B6" },
            new string [] { "\uBC78", "\uBC78", "\u1107\u1164\u11B7", "\uBC78", "\u1107\u1164\u11B7" },
            new string [] { "\uBC79", "\uBC79", "\u1107\u1164\u11B8", "\uBC79", "\u1107\u1164\u11B8" },
            new string [] { "\uBC7A", "\uBC7A", "\u1107\u1164\u11B9", "\uBC7A", "\u1107\u1164\u11B9" },
            new string [] { "\uBC7B", "\uBC7B", "\u1107\u1164\u11BA", "\uBC7B", "\u1107\u1164\u11BA" },
            new string [] { "\uBC7C", "\uBC7C", "\u1107\u1164\u11BB", "\uBC7C", "\u1107\u1164\u11BB" },
            new string [] { "\uBC7D", "\uBC7D", "\u1107\u1164\u11BC", "\uBC7D", "\u1107\u1164\u11BC" },
            new string [] { "\uBC7E", "\uBC7E", "\u1107\u1164\u11BD", "\uBC7E", "\u1107\u1164\u11BD" },
            new string [] { "\uBC7F", "\uBC7F", "\u1107\u1164\u11BE", "\uBC7F", "\u1107\u1164\u11BE" },
            new string [] { "\uBC80", "\uBC80", "\u1107\u1164\u11BF", "\uBC80", "\u1107\u1164\u11BF" },
            new string [] { "\uBC81", "\uBC81", "\u1107\u1164\u11C0", "\uBC81", "\u1107\u1164\u11C0" },
            new string [] { "\uBC82", "\uBC82", "\u1107\u1164\u11C1", "\uBC82", "\u1107\u1164\u11C1" },
            new string [] { "\uBC83", "\uBC83", "\u1107\u1164\u11C2", "\uBC83", "\u1107\u1164\u11C2" },
            new string [] { "\uBC84", "\uBC84", "\u1107\u1165", "\uBC84", "\u1107\u1165" },
            new string [] { "\uBC85", "\uBC85", "\u1107\u1165\u11A8", "\uBC85", "\u1107\u1165\u11A8" },
            new string [] { "\uBC86", "\uBC86", "\u1107\u1165\u11A9", "\uBC86", "\u1107\u1165\u11A9" },
            new string [] { "\uBC87", "\uBC87", "\u1107\u1165\u11AA", "\uBC87", "\u1107\u1165\u11AA" },
            new string [] { "\uBC88", "\uBC88", "\u1107\u1165\u11AB", "\uBC88", "\u1107\u1165\u11AB" },
            new string [] { "\uBC89", "\uBC89", "\u1107\u1165\u11AC", "\uBC89", "\u1107\u1165\u11AC" },
            new string [] { "\uBC8A", "\uBC8A", "\u1107\u1165\u11AD", "\uBC8A", "\u1107\u1165\u11AD" },
            new string [] { "\uBC8B", "\uBC8B", "\u1107\u1165\u11AE", "\uBC8B", "\u1107\u1165\u11AE" },
            new string [] { "\uBC8C", "\uBC8C", "\u1107\u1165\u11AF", "\uBC8C", "\u1107\u1165\u11AF" },
            new string [] { "\uBC8D", "\uBC8D", "\u1107\u1165\u11B0", "\uBC8D", "\u1107\u1165\u11B0" },
            new string [] { "\uBC8E", "\uBC8E", "\u1107\u1165\u11B1", "\uBC8E", "\u1107\u1165\u11B1" },
            new string [] { "\uBC8F", "\uBC8F", "\u1107\u1165\u11B2", "\uBC8F", "\u1107\u1165\u11B2" },
            new string [] { "\uBC90", "\uBC90", "\u1107\u1165\u11B3", "\uBC90", "\u1107\u1165\u11B3" },
            new string [] { "\uBC91", "\uBC91", "\u1107\u1165\u11B4", "\uBC91", "\u1107\u1165\u11B4" },
            new string [] { "\uBC92", "\uBC92", "\u1107\u1165\u11B5", "\uBC92", "\u1107\u1165\u11B5" },
            new string [] { "\uBC93", "\uBC93", "\u1107\u1165\u11B6", "\uBC93", "\u1107\u1165\u11B6" },
            new string [] { "\uBC94", "\uBC94", "\u1107\u1165\u11B7", "\uBC94", "\u1107\u1165\u11B7" },
            new string [] { "\uBC95", "\uBC95", "\u1107\u1165\u11B8", "\uBC95", "\u1107\u1165\u11B8" },
            new string [] { "\uBC96", "\uBC96", "\u1107\u1165\u11B9", "\uBC96", "\u1107\u1165\u11B9" },
            new string [] { "\uBC97", "\uBC97", "\u1107\u1165\u11BA", "\uBC97", "\u1107\u1165\u11BA" },
            new string [] { "\uBC98", "\uBC98", "\u1107\u1165\u11BB", "\uBC98", "\u1107\u1165\u11BB" },
            new string [] { "\uBC99", "\uBC99", "\u1107\u1165\u11BC", "\uBC99", "\u1107\u1165\u11BC" },
            new string [] { "\uBC9A", "\uBC9A", "\u1107\u1165\u11BD", "\uBC9A", "\u1107\u1165\u11BD" },
            new string [] { "\uBC9B", "\uBC9B", "\u1107\u1165\u11BE", "\uBC9B", "\u1107\u1165\u11BE" },
            new string [] { "\uBC9C", "\uBC9C", "\u1107\u1165\u11BF", "\uBC9C", "\u1107\u1165\u11BF" },
            new string [] { "\uBC9D", "\uBC9D", "\u1107\u1165\u11C0", "\uBC9D", "\u1107\u1165\u11C0" },
            new string [] { "\uBC9E", "\uBC9E", "\u1107\u1165\u11C1", "\uBC9E", "\u1107\u1165\u11C1" },
            new string [] { "\uBC9F", "\uBC9F", "\u1107\u1165\u11C2", "\uBC9F", "\u1107\u1165\u11C2" },
            new string [] { "\uBCA0", "\uBCA0", "\u1107\u1166", "\uBCA0", "\u1107\u1166" },
            new string [] { "\uBCA1", "\uBCA1", "\u1107\u1166\u11A8", "\uBCA1", "\u1107\u1166\u11A8" },
            new string [] { "\uBCA2", "\uBCA2", "\u1107\u1166\u11A9", "\uBCA2", "\u1107\u1166\u11A9" },
            new string [] { "\uBCA3", "\uBCA3", "\u1107\u1166\u11AA", "\uBCA3", "\u1107\u1166\u11AA" },
            new string [] { "\uBCA4", "\uBCA4", "\u1107\u1166\u11AB", "\uBCA4", "\u1107\u1166\u11AB" },
            new string [] { "\uBCA5", "\uBCA5", "\u1107\u1166\u11AC", "\uBCA5", "\u1107\u1166\u11AC" },
            new string [] { "\uBCA6", "\uBCA6", "\u1107\u1166\u11AD", "\uBCA6", "\u1107\u1166\u11AD" },
            new string [] { "\uBCA7", "\uBCA7", "\u1107\u1166\u11AE", "\uBCA7", "\u1107\u1166\u11AE" },
            new string [] { "\uBCA8", "\uBCA8", "\u1107\u1166\u11AF", "\uBCA8", "\u1107\u1166\u11AF" },
            new string [] { "\uBCA9", "\uBCA9", "\u1107\u1166\u11B0", "\uBCA9", "\u1107\u1166\u11B0" },
            new string [] { "\uBCAA", "\uBCAA", "\u1107\u1166\u11B1", "\uBCAA", "\u1107\u1166\u11B1" },
            new string [] { "\uBCAB", "\uBCAB", "\u1107\u1166\u11B2", "\uBCAB", "\u1107\u1166\u11B2" },
            new string [] { "\uBCAC", "\uBCAC", "\u1107\u1166\u11B3", "\uBCAC", "\u1107\u1166\u11B3" },
            new string [] { "\uBCAD", "\uBCAD", "\u1107\u1166\u11B4", "\uBCAD", "\u1107\u1166\u11B4" },
            new string [] { "\uBCAE", "\uBCAE", "\u1107\u1166\u11B5", "\uBCAE", "\u1107\u1166\u11B5" },
            new string [] { "\uBCAF", "\uBCAF", "\u1107\u1166\u11B6", "\uBCAF", "\u1107\u1166\u11B6" },
            new string [] { "\uBCB0", "\uBCB0", "\u1107\u1166\u11B7", "\uBCB0", "\u1107\u1166\u11B7" },
            new string [] { "\uBCB1", "\uBCB1", "\u1107\u1166\u11B8", "\uBCB1", "\u1107\u1166\u11B8" },
            new string [] { "\uBCB2", "\uBCB2", "\u1107\u1166\u11B9", "\uBCB2", "\u1107\u1166\u11B9" },
            new string [] { "\uBCB3", "\uBCB3", "\u1107\u1166\u11BA", "\uBCB3", "\u1107\u1166\u11BA" },
            new string [] { "\uBCB4", "\uBCB4", "\u1107\u1166\u11BB", "\uBCB4", "\u1107\u1166\u11BB" },
            new string [] { "\uBCB5", "\uBCB5", "\u1107\u1166\u11BC", "\uBCB5", "\u1107\u1166\u11BC" },
            new string [] { "\uBCB6", "\uBCB6", "\u1107\u1166\u11BD", "\uBCB6", "\u1107\u1166\u11BD" },
            new string [] { "\uBCB7", "\uBCB7", "\u1107\u1166\u11BE", "\uBCB7", "\u1107\u1166\u11BE" },
            new string [] { "\uBCB8", "\uBCB8", "\u1107\u1166\u11BF", "\uBCB8", "\u1107\u1166\u11BF" },
            new string [] { "\uBCB9", "\uBCB9", "\u1107\u1166\u11C0", "\uBCB9", "\u1107\u1166\u11C0" },
            new string [] { "\uBCBA", "\uBCBA", "\u1107\u1166\u11C1", "\uBCBA", "\u1107\u1166\u11C1" },
            new string [] { "\uBCBB", "\uBCBB", "\u1107\u1166\u11C2", "\uBCBB", "\u1107\u1166\u11C2" },
            new string [] { "\uBCBC", "\uBCBC", "\u1107\u1167", "\uBCBC", "\u1107\u1167" },
            new string [] { "\uBCBD", "\uBCBD", "\u1107\u1167\u11A8", "\uBCBD", "\u1107\u1167\u11A8" },
            new string [] { "\uBCBE", "\uBCBE", "\u1107\u1167\u11A9", "\uBCBE", "\u1107\u1167\u11A9" },
            new string [] { "\uBCBF", "\uBCBF", "\u1107\u1167\u11AA", "\uBCBF", "\u1107\u1167\u11AA" },
            new string [] { "\uBCC0", "\uBCC0", "\u1107\u1167\u11AB", "\uBCC0", "\u1107\u1167\u11AB" },
            new string [] { "\uBCC1", "\uBCC1", "\u1107\u1167\u11AC", "\uBCC1", "\u1107\u1167\u11AC" },
            new string [] { "\uBCC2", "\uBCC2", "\u1107\u1167\u11AD", "\uBCC2", "\u1107\u1167\u11AD" },
            new string [] { "\uBCC3", "\uBCC3", "\u1107\u1167\u11AE", "\uBCC3", "\u1107\u1167\u11AE" },
            new string [] { "\uBCC4", "\uBCC4", "\u1107\u1167\u11AF", "\uBCC4", "\u1107\u1167\u11AF" },
            new string [] { "\uBCC5", "\uBCC5", "\u1107\u1167\u11B0", "\uBCC5", "\u1107\u1167\u11B0" },
            new string [] { "\uBCC6", "\uBCC6", "\u1107\u1167\u11B1", "\uBCC6", "\u1107\u1167\u11B1" },
            new string [] { "\uBCC7", "\uBCC7", "\u1107\u1167\u11B2", "\uBCC7", "\u1107\u1167\u11B2" },
            new string [] { "\uBCC8", "\uBCC8", "\u1107\u1167\u11B3", "\uBCC8", "\u1107\u1167\u11B3" },
            new string [] { "\uBCC9", "\uBCC9", "\u1107\u1167\u11B4", "\uBCC9", "\u1107\u1167\u11B4" },
            new string [] { "\uBCCA", "\uBCCA", "\u1107\u1167\u11B5", "\uBCCA", "\u1107\u1167\u11B5" },
            new string [] { "\uBCCB", "\uBCCB", "\u1107\u1167\u11B6", "\uBCCB", "\u1107\u1167\u11B6" },
            new string [] { "\uBCCC", "\uBCCC", "\u1107\u1167\u11B7", "\uBCCC", "\u1107\u1167\u11B7" },
            new string [] { "\uBCCD", "\uBCCD", "\u1107\u1167\u11B8", "\uBCCD", "\u1107\u1167\u11B8" },
            new string [] { "\uBCCE", "\uBCCE", "\u1107\u1167\u11B9", "\uBCCE", "\u1107\u1167\u11B9" },
            new string [] { "\uBCCF", "\uBCCF", "\u1107\u1167\u11BA", "\uBCCF", "\u1107\u1167\u11BA" },
            new string [] { "\uBCD0", "\uBCD0", "\u1107\u1167\u11BB", "\uBCD0", "\u1107\u1167\u11BB" },
            new string [] { "\uBCD1", "\uBCD1", "\u1107\u1167\u11BC", "\uBCD1", "\u1107\u1167\u11BC" },
            new string [] { "\uBCD2", "\uBCD2", "\u1107\u1167\u11BD", "\uBCD2", "\u1107\u1167\u11BD" },
            new string [] { "\uBCD3", "\uBCD3", "\u1107\u1167\u11BE", "\uBCD3", "\u1107\u1167\u11BE" },
            new string [] { "\uBCD4", "\uBCD4", "\u1107\u1167\u11BF", "\uBCD4", "\u1107\u1167\u11BF" },
            new string [] { "\uBCD5", "\uBCD5", "\u1107\u1167\u11C0", "\uBCD5", "\u1107\u1167\u11C0" },
            new string [] { "\uBCD6", "\uBCD6", "\u1107\u1167\u11C1", "\uBCD6", "\u1107\u1167\u11C1" },
            new string [] { "\uBCD7", "\uBCD7", "\u1107\u1167\u11C2", "\uBCD7", "\u1107\u1167\u11C2" },
            new string [] { "\uBCD8", "\uBCD8", "\u1107\u1168", "\uBCD8", "\u1107\u1168" },
            new string [] { "\uBCD9", "\uBCD9", "\u1107\u1168\u11A8", "\uBCD9", "\u1107\u1168\u11A8" },
            new string [] { "\uBCDA", "\uBCDA", "\u1107\u1168\u11A9", "\uBCDA", "\u1107\u1168\u11A9" },
            new string [] { "\uBCDB", "\uBCDB", "\u1107\u1168\u11AA", "\uBCDB", "\u1107\u1168\u11AA" },
            new string [] { "\uBCDC", "\uBCDC", "\u1107\u1168\u11AB", "\uBCDC", "\u1107\u1168\u11AB" },
            new string [] { "\uBCDD", "\uBCDD", "\u1107\u1168\u11AC", "\uBCDD", "\u1107\u1168\u11AC" },
            new string [] { "\uBCDE", "\uBCDE", "\u1107\u1168\u11AD", "\uBCDE", "\u1107\u1168\u11AD" },
            new string [] { "\uBCDF", "\uBCDF", "\u1107\u1168\u11AE", "\uBCDF", "\u1107\u1168\u11AE" },
            new string [] { "\uBCE0", "\uBCE0", "\u1107\u1168\u11AF", "\uBCE0", "\u1107\u1168\u11AF" },
            new string [] { "\uBCE1", "\uBCE1", "\u1107\u1168\u11B0", "\uBCE1", "\u1107\u1168\u11B0" },
            new string [] { "\uBCE2", "\uBCE2", "\u1107\u1168\u11B1", "\uBCE2", "\u1107\u1168\u11B1" },
            new string [] { "\uBCE3", "\uBCE3", "\u1107\u1168\u11B2", "\uBCE3", "\u1107\u1168\u11B2" },
            new string [] { "\uBCE4", "\uBCE4", "\u1107\u1168\u11B3", "\uBCE4", "\u1107\u1168\u11B3" },
            new string [] { "\uBCE5", "\uBCE5", "\u1107\u1168\u11B4", "\uBCE5", "\u1107\u1168\u11B4" },
            new string [] { "\uBCE6", "\uBCE6", "\u1107\u1168\u11B5", "\uBCE6", "\u1107\u1168\u11B5" },
            new string [] { "\uBCE7", "\uBCE7", "\u1107\u1168\u11B6", "\uBCE7", "\u1107\u1168\u11B6" },
            new string [] { "\uBCE8", "\uBCE8", "\u1107\u1168\u11B7", "\uBCE8", "\u1107\u1168\u11B7" },
            new string [] { "\uBCE9", "\uBCE9", "\u1107\u1168\u11B8", "\uBCE9", "\u1107\u1168\u11B8" },
            new string [] { "\uBCEA", "\uBCEA", "\u1107\u1168\u11B9", "\uBCEA", "\u1107\u1168\u11B9" },
            new string [] { "\uBCEB", "\uBCEB", "\u1107\u1168\u11BA", "\uBCEB", "\u1107\u1168\u11BA" },
            new string [] { "\uBCEC", "\uBCEC", "\u1107\u1168\u11BB", "\uBCEC", "\u1107\u1168\u11BB" },
            new string [] { "\uBCED", "\uBCED", "\u1107\u1168\u11BC", "\uBCED", "\u1107\u1168\u11BC" },
            new string [] { "\uBCEE", "\uBCEE", "\u1107\u1168\u11BD", "\uBCEE", "\u1107\u1168\u11BD" },
            new string [] { "\uBCEF", "\uBCEF", "\u1107\u1168\u11BE", "\uBCEF", "\u1107\u1168\u11BE" },
            new string [] { "\uBCF0", "\uBCF0", "\u1107\u1168\u11BF", "\uBCF0", "\u1107\u1168\u11BF" },
            new string [] { "\uBCF1", "\uBCF1", "\u1107\u1168\u11C0", "\uBCF1", "\u1107\u1168\u11C0" },
            new string [] { "\uBCF2", "\uBCF2", "\u1107\u1168\u11C1", "\uBCF2", "\u1107\u1168\u11C1" },
            new string [] { "\uBCF3", "\uBCF3", "\u1107\u1168\u11C2", "\uBCF3", "\u1107\u1168\u11C2" },
            new string [] { "\uBCF4", "\uBCF4", "\u1107\u1169", "\uBCF4", "\u1107\u1169" },
            new string [] { "\uBCF5", "\uBCF5", "\u1107\u1169\u11A8", "\uBCF5", "\u1107\u1169\u11A8" },
            new string [] { "\uBCF6", "\uBCF6", "\u1107\u1169\u11A9", "\uBCF6", "\u1107\u1169\u11A9" },
            new string [] { "\uBCF7", "\uBCF7", "\u1107\u1169\u11AA", "\uBCF7", "\u1107\u1169\u11AA" },
            new string [] { "\uBCF8", "\uBCF8", "\u1107\u1169\u11AB", "\uBCF8", "\u1107\u1169\u11AB" },
            new string [] { "\uBCF9", "\uBCF9", "\u1107\u1169\u11AC", "\uBCF9", "\u1107\u1169\u11AC" },
            new string [] { "\uBCFA", "\uBCFA", "\u1107\u1169\u11AD", "\uBCFA", "\u1107\u1169\u11AD" },
            new string [] { "\uBCFB", "\uBCFB", "\u1107\u1169\u11AE", "\uBCFB", "\u1107\u1169\u11AE" },
            new string [] { "\uBCFC", "\uBCFC", "\u1107\u1169\u11AF", "\uBCFC", "\u1107\u1169\u11AF" },
            new string [] { "\uBCFD", "\uBCFD", "\u1107\u1169\u11B0", "\uBCFD", "\u1107\u1169\u11B0" },
            new string [] { "\uBCFE", "\uBCFE", "\u1107\u1169\u11B1", "\uBCFE", "\u1107\u1169\u11B1" },
            new string [] { "\uBCFF", "\uBCFF", "\u1107\u1169\u11B2", "\uBCFF", "\u1107\u1169\u11B2" },
            new string [] { "\uBD00", "\uBD00", "\u1107\u1169\u11B3", "\uBD00", "\u1107\u1169\u11B3" },
            new string [] { "\uBD01", "\uBD01", "\u1107\u1169\u11B4", "\uBD01", "\u1107\u1169\u11B4" },
            new string [] { "\uBD02", "\uBD02", "\u1107\u1169\u11B5", "\uBD02", "\u1107\u1169\u11B5" },
            new string [] { "\uBD03", "\uBD03", "\u1107\u1169\u11B6", "\uBD03", "\u1107\u1169\u11B6" },
            new string [] { "\uBD04", "\uBD04", "\u1107\u1169\u11B7", "\uBD04", "\u1107\u1169\u11B7" },
            new string [] { "\uBD05", "\uBD05", "\u1107\u1169\u11B8", "\uBD05", "\u1107\u1169\u11B8" },
            new string [] { "\uBD06", "\uBD06", "\u1107\u1169\u11B9", "\uBD06", "\u1107\u1169\u11B9" },
            new string [] { "\uBD07", "\uBD07", "\u1107\u1169\u11BA", "\uBD07", "\u1107\u1169\u11BA" },
            new string [] { "\uBD08", "\uBD08", "\u1107\u1169\u11BB", "\uBD08", "\u1107\u1169\u11BB" },
            new string [] { "\uBD09", "\uBD09", "\u1107\u1169\u11BC", "\uBD09", "\u1107\u1169\u11BC" },
            new string [] { "\uBD0A", "\uBD0A", "\u1107\u1169\u11BD", "\uBD0A", "\u1107\u1169\u11BD" },
            new string [] { "\uBD0B", "\uBD0B", "\u1107\u1169\u11BE", "\uBD0B", "\u1107\u1169\u11BE" },
            new string [] { "\uBD0C", "\uBD0C", "\u1107\u1169\u11BF", "\uBD0C", "\u1107\u1169\u11BF" },
            new string [] { "\uBD0D", "\uBD0D", "\u1107\u1169\u11C0", "\uBD0D", "\u1107\u1169\u11C0" },
            new string [] { "\uBD0E", "\uBD0E", "\u1107\u1169\u11C1", "\uBD0E", "\u1107\u1169\u11C1" },
            new string [] { "\uBD0F", "\uBD0F", "\u1107\u1169\u11C2", "\uBD0F", "\u1107\u1169\u11C2" },
            new string [] { "\uBD10", "\uBD10", "\u1107\u116A", "\uBD10", "\u1107\u116A" },
            new string [] { "\uBD11", "\uBD11", "\u1107\u116A\u11A8", "\uBD11", "\u1107\u116A\u11A8" },
            new string [] { "\uBD12", "\uBD12", "\u1107\u116A\u11A9", "\uBD12", "\u1107\u116A\u11A9" },
            new string [] { "\uBD13", "\uBD13", "\u1107\u116A\u11AA", "\uBD13", "\u1107\u116A\u11AA" },
            new string [] { "\uBD14", "\uBD14", "\u1107\u116A\u11AB", "\uBD14", "\u1107\u116A\u11AB" },
            new string [] { "\uBD15", "\uBD15", "\u1107\u116A\u11AC", "\uBD15", "\u1107\u116A\u11AC" },
            new string [] { "\uBD16", "\uBD16", "\u1107\u116A\u11AD", "\uBD16", "\u1107\u116A\u11AD" },
            new string [] { "\uBD17", "\uBD17", "\u1107\u116A\u11AE", "\uBD17", "\u1107\u116A\u11AE" },
            new string [] { "\uBD18", "\uBD18", "\u1107\u116A\u11AF", "\uBD18", "\u1107\u116A\u11AF" },
            new string [] { "\uBD19", "\uBD19", "\u1107\u116A\u11B0", "\uBD19", "\u1107\u116A\u11B0" },
            new string [] { "\uBD1A", "\uBD1A", "\u1107\u116A\u11B1", "\uBD1A", "\u1107\u116A\u11B1" },
            new string [] { "\uBD1B", "\uBD1B", "\u1107\u116A\u11B2", "\uBD1B", "\u1107\u116A\u11B2" },
            new string [] { "\uBD1C", "\uBD1C", "\u1107\u116A\u11B3", "\uBD1C", "\u1107\u116A\u11B3" },
            new string [] { "\uBD1D", "\uBD1D", "\u1107\u116A\u11B4", "\uBD1D", "\u1107\u116A\u11B4" },
            new string [] { "\uBD1E", "\uBD1E", "\u1107\u116A\u11B5", "\uBD1E", "\u1107\u116A\u11B5" },
            new string [] { "\uBD1F", "\uBD1F", "\u1107\u116A\u11B6", "\uBD1F", "\u1107\u116A\u11B6" },
            new string [] { "\uBD20", "\uBD20", "\u1107\u116A\u11B7", "\uBD20", "\u1107\u116A\u11B7" },
            new string [] { "\uBD21", "\uBD21", "\u1107\u116A\u11B8", "\uBD21", "\u1107\u116A\u11B8" },
            new string [] { "\uBD22", "\uBD22", "\u1107\u116A\u11B9", "\uBD22", "\u1107\u116A\u11B9" },
            new string [] { "\uBD23", "\uBD23", "\u1107\u116A\u11BA", "\uBD23", "\u1107\u116A\u11BA" },
            new string [] { "\uBD24", "\uBD24", "\u1107\u116A\u11BB", "\uBD24", "\u1107\u116A\u11BB" },
            new string [] { "\uBD25", "\uBD25", "\u1107\u116A\u11BC", "\uBD25", "\u1107\u116A\u11BC" },
            new string [] { "\uBD26", "\uBD26", "\u1107\u116A\u11BD", "\uBD26", "\u1107\u116A\u11BD" },
            new string [] { "\uBD27", "\uBD27", "\u1107\u116A\u11BE", "\uBD27", "\u1107\u116A\u11BE" },
            new string [] { "\uBD28", "\uBD28", "\u1107\u116A\u11BF", "\uBD28", "\u1107\u116A\u11BF" },
            new string [] { "\uBD29", "\uBD29", "\u1107\u116A\u11C0", "\uBD29", "\u1107\u116A\u11C0" },
            new string [] { "\uBD2A", "\uBD2A", "\u1107\u116A\u11C1", "\uBD2A", "\u1107\u116A\u11C1" },
            new string [] { "\uBD2B", "\uBD2B", "\u1107\u116A\u11C2", "\uBD2B", "\u1107\u116A\u11C2" },
            new string [] { "\uBD2C", "\uBD2C", "\u1107\u116B", "\uBD2C", "\u1107\u116B" },
            new string [] { "\uBD2D", "\uBD2D", "\u1107\u116B\u11A8", "\uBD2D", "\u1107\u116B\u11A8" },
            new string [] { "\uBD2E", "\uBD2E", "\u1107\u116B\u11A9", "\uBD2E", "\u1107\u116B\u11A9" },
            new string [] { "\uBD2F", "\uBD2F", "\u1107\u116B\u11AA", "\uBD2F", "\u1107\u116B\u11AA" },
            new string [] { "\uBD30", "\uBD30", "\u1107\u116B\u11AB", "\uBD30", "\u1107\u116B\u11AB" },
            new string [] { "\uBD31", "\uBD31", "\u1107\u116B\u11AC", "\uBD31", "\u1107\u116B\u11AC" },
            new string [] { "\uBD32", "\uBD32", "\u1107\u116B\u11AD", "\uBD32", "\u1107\u116B\u11AD" },
            new string [] { "\uBD33", "\uBD33", "\u1107\u116B\u11AE", "\uBD33", "\u1107\u116B\u11AE" },
            new string [] { "\uBD34", "\uBD34", "\u1107\u116B\u11AF", "\uBD34", "\u1107\u116B\u11AF" },
            new string [] { "\uBD35", "\uBD35", "\u1107\u116B\u11B0", "\uBD35", "\u1107\u116B\u11B0" },
            new string [] { "\uBD36", "\uBD36", "\u1107\u116B\u11B1", "\uBD36", "\u1107\u116B\u11B1" },
            new string [] { "\uBD37", "\uBD37", "\u1107\u116B\u11B2", "\uBD37", "\u1107\u116B\u11B2" },
            new string [] { "\uBD38", "\uBD38", "\u1107\u116B\u11B3", "\uBD38", "\u1107\u116B\u11B3" },
            new string [] { "\uBD39", "\uBD39", "\u1107\u116B\u11B4", "\uBD39", "\u1107\u116B\u11B4" },
            new string [] { "\uBD3A", "\uBD3A", "\u1107\u116B\u11B5", "\uBD3A", "\u1107\u116B\u11B5" },
            new string [] { "\uBD3B", "\uBD3B", "\u1107\u116B\u11B6", "\uBD3B", "\u1107\u116B\u11B6" },
            new string [] { "\uBD3C", "\uBD3C", "\u1107\u116B\u11B7", "\uBD3C", "\u1107\u116B\u11B7" },
            new string [] { "\uBD3D", "\uBD3D", "\u1107\u116B\u11B8", "\uBD3D", "\u1107\u116B\u11B8" },
            new string [] { "\uBD3E", "\uBD3E", "\u1107\u116B\u11B9", "\uBD3E", "\u1107\u116B\u11B9" },
            new string [] { "\uBD3F", "\uBD3F", "\u1107\u116B\u11BA", "\uBD3F", "\u1107\u116B\u11BA" },
            new string [] { "\uBD40", "\uBD40", "\u1107\u116B\u11BB", "\uBD40", "\u1107\u116B\u11BB" },
            new string [] { "\uBD41", "\uBD41", "\u1107\u116B\u11BC", "\uBD41", "\u1107\u116B\u11BC" },
            new string [] { "\uBD42", "\uBD42", "\u1107\u116B\u11BD", "\uBD42", "\u1107\u116B\u11BD" },
            new string [] { "\uBD43", "\uBD43", "\u1107\u116B\u11BE", "\uBD43", "\u1107\u116B\u11BE" },
            new string [] { "\uBD44", "\uBD44", "\u1107\u116B\u11BF", "\uBD44", "\u1107\u116B\u11BF" },
            new string [] { "\uBD45", "\uBD45", "\u1107\u116B\u11C0", "\uBD45", "\u1107\u116B\u11C0" },
            new string [] { "\uBD46", "\uBD46", "\u1107\u116B\u11C1", "\uBD46", "\u1107\u116B\u11C1" },
            new string [] { "\uBD47", "\uBD47", "\u1107\u116B\u11C2", "\uBD47", "\u1107\u116B\u11C2" },
            new string [] { "\uBD48", "\uBD48", "\u1107\u116C", "\uBD48", "\u1107\u116C" },
            new string [] { "\uBD49", "\uBD49", "\u1107\u116C\u11A8", "\uBD49", "\u1107\u116C\u11A8" },
            new string [] { "\uBD4A", "\uBD4A", "\u1107\u116C\u11A9", "\uBD4A", "\u1107\u116C\u11A9" },
            new string [] { "\uBD4B", "\uBD4B", "\u1107\u116C\u11AA", "\uBD4B", "\u1107\u116C\u11AA" },
            new string [] { "\uBD4C", "\uBD4C", "\u1107\u116C\u11AB", "\uBD4C", "\u1107\u116C\u11AB" },
            new string [] { "\uBD4D", "\uBD4D", "\u1107\u116C\u11AC", "\uBD4D", "\u1107\u116C\u11AC" },
            new string [] { "\uBD4E", "\uBD4E", "\u1107\u116C\u11AD", "\uBD4E", "\u1107\u116C\u11AD" },
            new string [] { "\uBD4F", "\uBD4F", "\u1107\u116C\u11AE", "\uBD4F", "\u1107\u116C\u11AE" },
            new string [] { "\uBD50", "\uBD50", "\u1107\u116C\u11AF", "\uBD50", "\u1107\u116C\u11AF" },
            new string [] { "\uBD51", "\uBD51", "\u1107\u116C\u11B0", "\uBD51", "\u1107\u116C\u11B0" },
            new string [] { "\uBD52", "\uBD52", "\u1107\u116C\u11B1", "\uBD52", "\u1107\u116C\u11B1" },
            new string [] { "\uBD53", "\uBD53", "\u1107\u116C\u11B2", "\uBD53", "\u1107\u116C\u11B2" },
            new string [] { "\uBD54", "\uBD54", "\u1107\u116C\u11B3", "\uBD54", "\u1107\u116C\u11B3" },
            new string [] { "\uBD55", "\uBD55", "\u1107\u116C\u11B4", "\uBD55", "\u1107\u116C\u11B4" },
            new string [] { "\uBD56", "\uBD56", "\u1107\u116C\u11B5", "\uBD56", "\u1107\u116C\u11B5" },
            new string [] { "\uBD57", "\uBD57", "\u1107\u116C\u11B6", "\uBD57", "\u1107\u116C\u11B6" },
            new string [] { "\uBD58", "\uBD58", "\u1107\u116C\u11B7", "\uBD58", "\u1107\u116C\u11B7" },
            new string [] { "\uBD59", "\uBD59", "\u1107\u116C\u11B8", "\uBD59", "\u1107\u116C\u11B8" },
            new string [] { "\uBD5A", "\uBD5A", "\u1107\u116C\u11B9", "\uBD5A", "\u1107\u116C\u11B9" },
            new string [] { "\uBD5B", "\uBD5B", "\u1107\u116C\u11BA", "\uBD5B", "\u1107\u116C\u11BA" },
            new string [] { "\uBD5C", "\uBD5C", "\u1107\u116C\u11BB", "\uBD5C", "\u1107\u116C\u11BB" },
            new string [] { "\uBD5D", "\uBD5D", "\u1107\u116C\u11BC", "\uBD5D", "\u1107\u116C\u11BC" },
            new string [] { "\uBD5E", "\uBD5E", "\u1107\u116C\u11BD", "\uBD5E", "\u1107\u116C\u11BD" },
            new string [] { "\uBD5F", "\uBD5F", "\u1107\u116C\u11BE", "\uBD5F", "\u1107\u116C\u11BE" },
            new string [] { "\uBD60", "\uBD60", "\u1107\u116C\u11BF", "\uBD60", "\u1107\u116C\u11BF" },
            new string [] { "\uBD61", "\uBD61", "\u1107\u116C\u11C0", "\uBD61", "\u1107\u116C\u11C0" },
            new string [] { "\uBD62", "\uBD62", "\u1107\u116C\u11C1", "\uBD62", "\u1107\u116C\u11C1" },
            new string [] { "\uBD63", "\uBD63", "\u1107\u116C\u11C2", "\uBD63", "\u1107\u116C\u11C2" },
            new string [] { "\uBD64", "\uBD64", "\u1107\u116D", "\uBD64", "\u1107\u116D" },
            new string [] { "\uBD65", "\uBD65", "\u1107\u116D\u11A8", "\uBD65", "\u1107\u116D\u11A8" },
            new string [] { "\uBD66", "\uBD66", "\u1107\u116D\u11A9", "\uBD66", "\u1107\u116D\u11A9" },
            new string [] { "\uBD67", "\uBD67", "\u1107\u116D\u11AA", "\uBD67", "\u1107\u116D\u11AA" },
            new string [] { "\uBD68", "\uBD68", "\u1107\u116D\u11AB", "\uBD68", "\u1107\u116D\u11AB" },
            new string [] { "\uBD69", "\uBD69", "\u1107\u116D\u11AC", "\uBD69", "\u1107\u116D\u11AC" },
            new string [] { "\uBD6A", "\uBD6A", "\u1107\u116D\u11AD", "\uBD6A", "\u1107\u116D\u11AD" },
            new string [] { "\uBD6B", "\uBD6B", "\u1107\u116D\u11AE", "\uBD6B", "\u1107\u116D\u11AE" },
            new string [] { "\uBD6C", "\uBD6C", "\u1107\u116D\u11AF", "\uBD6C", "\u1107\u116D\u11AF" },
            new string [] { "\uBD6D", "\uBD6D", "\u1107\u116D\u11B0", "\uBD6D", "\u1107\u116D\u11B0" },
            new string [] { "\uBD6E", "\uBD6E", "\u1107\u116D\u11B1", "\uBD6E", "\u1107\u116D\u11B1" },
            new string [] { "\uBD6F", "\uBD6F", "\u1107\u116D\u11B2", "\uBD6F", "\u1107\u116D\u11B2" },
            new string [] { "\uBD70", "\uBD70", "\u1107\u116D\u11B3", "\uBD70", "\u1107\u116D\u11B3" },
            new string [] { "\uBD71", "\uBD71", "\u1107\u116D\u11B4", "\uBD71", "\u1107\u116D\u11B4" },
            new string [] { "\uBD72", "\uBD72", "\u1107\u116D\u11B5", "\uBD72", "\u1107\u116D\u11B5" },
            new string [] { "\uBD73", "\uBD73", "\u1107\u116D\u11B6", "\uBD73", "\u1107\u116D\u11B6" },
            new string [] { "\uBD74", "\uBD74", "\u1107\u116D\u11B7", "\uBD74", "\u1107\u116D\u11B7" },
            new string [] { "\uBD75", "\uBD75", "\u1107\u116D\u11B8", "\uBD75", "\u1107\u116D\u11B8" },
            new string [] { "\uBD76", "\uBD76", "\u1107\u116D\u11B9", "\uBD76", "\u1107\u116D\u11B9" },
            new string [] { "\uBD77", "\uBD77", "\u1107\u116D\u11BA", "\uBD77", "\u1107\u116D\u11BA" },
            new string [] { "\uBD78", "\uBD78", "\u1107\u116D\u11BB", "\uBD78", "\u1107\u116D\u11BB" },
            new string [] { "\uBD79", "\uBD79", "\u1107\u116D\u11BC", "\uBD79", "\u1107\u116D\u11BC" },
            new string [] { "\uBD7A", "\uBD7A", "\u1107\u116D\u11BD", "\uBD7A", "\u1107\u116D\u11BD" },
            new string [] { "\uBD7B", "\uBD7B", "\u1107\u116D\u11BE", "\uBD7B", "\u1107\u116D\u11BE" },
            new string [] { "\uBD7C", "\uBD7C", "\u1107\u116D\u11BF", "\uBD7C", "\u1107\u116D\u11BF" },
            new string [] { "\uBD7D", "\uBD7D", "\u1107\u116D\u11C0", "\uBD7D", "\u1107\u116D\u11C0" },
            new string [] { "\uBD7E", "\uBD7E", "\u1107\u116D\u11C1", "\uBD7E", "\u1107\u116D\u11C1" },
            new string [] { "\uBD7F", "\uBD7F", "\u1107\u116D\u11C2", "\uBD7F", "\u1107\u116D\u11C2" },
            new string [] { "\uBD80", "\uBD80", "\u1107\u116E", "\uBD80", "\u1107\u116E" },
            new string [] { "\uBD81", "\uBD81", "\u1107\u116E\u11A8", "\uBD81", "\u1107\u116E\u11A8" },
            new string [] { "\uBD82", "\uBD82", "\u1107\u116E\u11A9", "\uBD82", "\u1107\u116E\u11A9" },
            new string [] { "\uBD83", "\uBD83", "\u1107\u116E\u11AA", "\uBD83", "\u1107\u116E\u11AA" },
            new string [] { "\uBD84", "\uBD84", "\u1107\u116E\u11AB", "\uBD84", "\u1107\u116E\u11AB" },
            new string [] { "\uBD85", "\uBD85", "\u1107\u116E\u11AC", "\uBD85", "\u1107\u116E\u11AC" },
            new string [] { "\uBD86", "\uBD86", "\u1107\u116E\u11AD", "\uBD86", "\u1107\u116E\u11AD" },
            new string [] { "\uBD87", "\uBD87", "\u1107\u116E\u11AE", "\uBD87", "\u1107\u116E\u11AE" },
            new string [] { "\uBD88", "\uBD88", "\u1107\u116E\u11AF", "\uBD88", "\u1107\u116E\u11AF" },
            new string [] { "\uBD89", "\uBD89", "\u1107\u116E\u11B0", "\uBD89", "\u1107\u116E\u11B0" },
            new string [] { "\uBD8A", "\uBD8A", "\u1107\u116E\u11B1", "\uBD8A", "\u1107\u116E\u11B1" },
            new string [] { "\uBD8B", "\uBD8B", "\u1107\u116E\u11B2", "\uBD8B", "\u1107\u116E\u11B2" },
            new string [] { "\uBD8C", "\uBD8C", "\u1107\u116E\u11B3", "\uBD8C", "\u1107\u116E\u11B3" },
            new string [] { "\uBD8D", "\uBD8D", "\u1107\u116E\u11B4", "\uBD8D", "\u1107\u116E\u11B4" },
            new string [] { "\uBD8E", "\uBD8E", "\u1107\u116E\u11B5", "\uBD8E", "\u1107\u116E\u11B5" },
            new string [] { "\uBD8F", "\uBD8F", "\u1107\u116E\u11B6", "\uBD8F", "\u1107\u116E\u11B6" },
            new string [] { "\uBD90", "\uBD90", "\u1107\u116E\u11B7", "\uBD90", "\u1107\u116E\u11B7" },
            new string [] { "\uBD91", "\uBD91", "\u1107\u116E\u11B8", "\uBD91", "\u1107\u116E\u11B8" },
            new string [] { "\uBD92", "\uBD92", "\u1107\u116E\u11B9", "\uBD92", "\u1107\u116E\u11B9" },
            new string [] { "\uBD93", "\uBD93", "\u1107\u116E\u11BA", "\uBD93", "\u1107\u116E\u11BA" },
            new string [] { "\uBD94", "\uBD94", "\u1107\u116E\u11BB", "\uBD94", "\u1107\u116E\u11BB" },
            new string [] { "\uBD95", "\uBD95", "\u1107\u116E\u11BC", "\uBD95", "\u1107\u116E\u11BC" },
            new string [] { "\uBD96", "\uBD96", "\u1107\u116E\u11BD", "\uBD96", "\u1107\u116E\u11BD" },
            new string [] { "\uBD97", "\uBD97", "\u1107\u116E\u11BE", "\uBD97", "\u1107\u116E\u11BE" },
            new string [] { "\uBD98", "\uBD98", "\u1107\u116E\u11BF", "\uBD98", "\u1107\u116E\u11BF" },
            new string [] { "\uBD99", "\uBD99", "\u1107\u116E\u11C0", "\uBD99", "\u1107\u116E\u11C0" },
            new string [] { "\uBD9A", "\uBD9A", "\u1107\u116E\u11C1", "\uBD9A", "\u1107\u116E\u11C1" },
            new string [] { "\uBD9B", "\uBD9B", "\u1107\u116E\u11C2", "\uBD9B", "\u1107\u116E\u11C2" },
            new string [] { "\uBD9C", "\uBD9C", "\u1107\u116F", "\uBD9C", "\u1107\u116F" },
            new string [] { "\uBD9D", "\uBD9D", "\u1107\u116F\u11A8", "\uBD9D", "\u1107\u116F\u11A8" },
            new string [] { "\uBD9E", "\uBD9E", "\u1107\u116F\u11A9", "\uBD9E", "\u1107\u116F\u11A9" },
            new string [] { "\uBD9F", "\uBD9F", "\u1107\u116F\u11AA", "\uBD9F", "\u1107\u116F\u11AA" },
            new string [] { "\uBDA0", "\uBDA0", "\u1107\u116F\u11AB", "\uBDA0", "\u1107\u116F\u11AB" },
            new string [] { "\uBDA1", "\uBDA1", "\u1107\u116F\u11AC", "\uBDA1", "\u1107\u116F\u11AC" },
            new string [] { "\uBDA2", "\uBDA2", "\u1107\u116F\u11AD", "\uBDA2", "\u1107\u116F\u11AD" },
            new string [] { "\uBDA3", "\uBDA3", "\u1107\u116F\u11AE", "\uBDA3", "\u1107\u116F\u11AE" },
            new string [] { "\uBDA4", "\uBDA4", "\u1107\u116F\u11AF", "\uBDA4", "\u1107\u116F\u11AF" },
            new string [] { "\uBDA5", "\uBDA5", "\u1107\u116F\u11B0", "\uBDA5", "\u1107\u116F\u11B0" },
            new string [] { "\uBDA6", "\uBDA6", "\u1107\u116F\u11B1", "\uBDA6", "\u1107\u116F\u11B1" },
            new string [] { "\uBDA7", "\uBDA7", "\u1107\u116F\u11B2", "\uBDA7", "\u1107\u116F\u11B2" },
            new string [] { "\uBDA8", "\uBDA8", "\u1107\u116F\u11B3", "\uBDA8", "\u1107\u116F\u11B3" },
            new string [] { "\uBDA9", "\uBDA9", "\u1107\u116F\u11B4", "\uBDA9", "\u1107\u116F\u11B4" },
            new string [] { "\uBDAA", "\uBDAA", "\u1107\u116F\u11B5", "\uBDAA", "\u1107\u116F\u11B5" },
            new string [] { "\uBDAB", "\uBDAB", "\u1107\u116F\u11B6", "\uBDAB", "\u1107\u116F\u11B6" },
            new string [] { "\uBDAC", "\uBDAC", "\u1107\u116F\u11B7", "\uBDAC", "\u1107\u116F\u11B7" },
            new string [] { "\uBDAD", "\uBDAD", "\u1107\u116F\u11B8", "\uBDAD", "\u1107\u116F\u11B8" },
            new string [] { "\uBDAE", "\uBDAE", "\u1107\u116F\u11B9", "\uBDAE", "\u1107\u116F\u11B9" },
            new string [] { "\uBDAF", "\uBDAF", "\u1107\u116F\u11BA", "\uBDAF", "\u1107\u116F\u11BA" },
            new string [] { "\uBDB0", "\uBDB0", "\u1107\u116F\u11BB", "\uBDB0", "\u1107\u116F\u11BB" },
            new string [] { "\uBDB1", "\uBDB1", "\u1107\u116F\u11BC", "\uBDB1", "\u1107\u116F\u11BC" },
            new string [] { "\uBDB2", "\uBDB2", "\u1107\u116F\u11BD", "\uBDB2", "\u1107\u116F\u11BD" },
            new string [] { "\uBDB3", "\uBDB3", "\u1107\u116F\u11BE", "\uBDB3", "\u1107\u116F\u11BE" },
            new string [] { "\uBDB4", "\uBDB4", "\u1107\u116F\u11BF", "\uBDB4", "\u1107\u116F\u11BF" },
            new string [] { "\uBDB5", "\uBDB5", "\u1107\u116F\u11C0", "\uBDB5", "\u1107\u116F\u11C0" },
            new string [] { "\uBDB6", "\uBDB6", "\u1107\u116F\u11C1", "\uBDB6", "\u1107\u116F\u11C1" },
            new string [] { "\uBDB7", "\uBDB7", "\u1107\u116F\u11C2", "\uBDB7", "\u1107\u116F\u11C2" },
            new string [] { "\uBDB8", "\uBDB8", "\u1107\u1170", "\uBDB8", "\u1107\u1170" },
            new string [] { "\uBDB9", "\uBDB9", "\u1107\u1170\u11A8", "\uBDB9", "\u1107\u1170\u11A8" },
            new string [] { "\uBDBA", "\uBDBA", "\u1107\u1170\u11A9", "\uBDBA", "\u1107\u1170\u11A9" },
            new string [] { "\uBDBB", "\uBDBB", "\u1107\u1170\u11AA", "\uBDBB", "\u1107\u1170\u11AA" },
            new string [] { "\uBDBC", "\uBDBC", "\u1107\u1170\u11AB", "\uBDBC", "\u1107\u1170\u11AB" },
            new string [] { "\uBDBD", "\uBDBD", "\u1107\u1170\u11AC", "\uBDBD", "\u1107\u1170\u11AC" },
            new string [] { "\uBDBE", "\uBDBE", "\u1107\u1170\u11AD", "\uBDBE", "\u1107\u1170\u11AD" },
            new string [] { "\uBDBF", "\uBDBF", "\u1107\u1170\u11AE", "\uBDBF", "\u1107\u1170\u11AE" },
            new string [] { "\uBDC0", "\uBDC0", "\u1107\u1170\u11AF", "\uBDC0", "\u1107\u1170\u11AF" },
            new string [] { "\uBDC1", "\uBDC1", "\u1107\u1170\u11B0", "\uBDC1", "\u1107\u1170\u11B0" },
            new string [] { "\uBDC2", "\uBDC2", "\u1107\u1170\u11B1", "\uBDC2", "\u1107\u1170\u11B1" },
            new string [] { "\uBDC3", "\uBDC3", "\u1107\u1170\u11B2", "\uBDC3", "\u1107\u1170\u11B2" },
            new string [] { "\uBDC4", "\uBDC4", "\u1107\u1170\u11B3", "\uBDC4", "\u1107\u1170\u11B3" },
            new string [] { "\uBDC5", "\uBDC5", "\u1107\u1170\u11B4", "\uBDC5", "\u1107\u1170\u11B4" },
            new string [] { "\uBDC6", "\uBDC6", "\u1107\u1170\u11B5", "\uBDC6", "\u1107\u1170\u11B5" },
            new string [] { "\uBDC7", "\uBDC7", "\u1107\u1170\u11B6", "\uBDC7", "\u1107\u1170\u11B6" },
            new string [] { "\uBDC8", "\uBDC8", "\u1107\u1170\u11B7", "\uBDC8", "\u1107\u1170\u11B7" },
            new string [] { "\uBDC9", "\uBDC9", "\u1107\u1170\u11B8", "\uBDC9", "\u1107\u1170\u11B8" },
            new string [] { "\uBDCA", "\uBDCA", "\u1107\u1170\u11B9", "\uBDCA", "\u1107\u1170\u11B9" },
            new string [] { "\uBDCB", "\uBDCB", "\u1107\u1170\u11BA", "\uBDCB", "\u1107\u1170\u11BA" },
            new string [] { "\uBDCC", "\uBDCC", "\u1107\u1170\u11BB", "\uBDCC", "\u1107\u1170\u11BB" },
            new string [] { "\uBDCD", "\uBDCD", "\u1107\u1170\u11BC", "\uBDCD", "\u1107\u1170\u11BC" },
            new string [] { "\uBDCE", "\uBDCE", "\u1107\u1170\u11BD", "\uBDCE", "\u1107\u1170\u11BD" },
            new string [] { "\uBDCF", "\uBDCF", "\u1107\u1170\u11BE", "\uBDCF", "\u1107\u1170\u11BE" },
            new string [] { "\uBDD0", "\uBDD0", "\u1107\u1170\u11BF", "\uBDD0", "\u1107\u1170\u11BF" },
            new string [] { "\uBDD1", "\uBDD1", "\u1107\u1170\u11C0", "\uBDD1", "\u1107\u1170\u11C0" },
            new string [] { "\uBDD2", "\uBDD2", "\u1107\u1170\u11C1", "\uBDD2", "\u1107\u1170\u11C1" },
            new string [] { "\uBDD3", "\uBDD3", "\u1107\u1170\u11C2", "\uBDD3", "\u1107\u1170\u11C2" },
            new string [] { "\uBDD4", "\uBDD4", "\u1107\u1171", "\uBDD4", "\u1107\u1171" },
            new string [] { "\uBDD5", "\uBDD5", "\u1107\u1171\u11A8", "\uBDD5", "\u1107\u1171\u11A8" },
            new string [] { "\uBDD6", "\uBDD6", "\u1107\u1171\u11A9", "\uBDD6", "\u1107\u1171\u11A9" },
            new string [] { "\uBDD7", "\uBDD7", "\u1107\u1171\u11AA", "\uBDD7", "\u1107\u1171\u11AA" },
            new string [] { "\uBDD8", "\uBDD8", "\u1107\u1171\u11AB", "\uBDD8", "\u1107\u1171\u11AB" },
            new string [] { "\uBDD9", "\uBDD9", "\u1107\u1171\u11AC", "\uBDD9", "\u1107\u1171\u11AC" },
            new string [] { "\uBDDA", "\uBDDA", "\u1107\u1171\u11AD", "\uBDDA", "\u1107\u1171\u11AD" },
            new string [] { "\uBDDB", "\uBDDB", "\u1107\u1171\u11AE", "\uBDDB", "\u1107\u1171\u11AE" },
            new string [] { "\uBDDC", "\uBDDC", "\u1107\u1171\u11AF", "\uBDDC", "\u1107\u1171\u11AF" },
            new string [] { "\uBDDD", "\uBDDD", "\u1107\u1171\u11B0", "\uBDDD", "\u1107\u1171\u11B0" },
            new string [] { "\uBDDE", "\uBDDE", "\u1107\u1171\u11B1", "\uBDDE", "\u1107\u1171\u11B1" },
            new string [] { "\uBDDF", "\uBDDF", "\u1107\u1171\u11B2", "\uBDDF", "\u1107\u1171\u11B2" },
            new string [] { "\uBDE0", "\uBDE0", "\u1107\u1171\u11B3", "\uBDE0", "\u1107\u1171\u11B3" },
            new string [] { "\uBDE1", "\uBDE1", "\u1107\u1171\u11B4", "\uBDE1", "\u1107\u1171\u11B4" },
            new string [] { "\uBDE2", "\uBDE2", "\u1107\u1171\u11B5", "\uBDE2", "\u1107\u1171\u11B5" },
            new string [] { "\uBDE3", "\uBDE3", "\u1107\u1171\u11B6", "\uBDE3", "\u1107\u1171\u11B6" },
            new string [] { "\uBDE4", "\uBDE4", "\u1107\u1171\u11B7", "\uBDE4", "\u1107\u1171\u11B7" },
            new string [] { "\uBDE5", "\uBDE5", "\u1107\u1171\u11B8", "\uBDE5", "\u1107\u1171\u11B8" },
            new string [] { "\uBDE6", "\uBDE6", "\u1107\u1171\u11B9", "\uBDE6", "\u1107\u1171\u11B9" },
            new string [] { "\uBDE7", "\uBDE7", "\u1107\u1171\u11BA", "\uBDE7", "\u1107\u1171\u11BA" },
            new string [] { "\uBDE8", "\uBDE8", "\u1107\u1171\u11BB", "\uBDE8", "\u1107\u1171\u11BB" },
            new string [] { "\uBDE9", "\uBDE9", "\u1107\u1171\u11BC", "\uBDE9", "\u1107\u1171\u11BC" },
            new string [] { "\uBDEA", "\uBDEA", "\u1107\u1171\u11BD", "\uBDEA", "\u1107\u1171\u11BD" },
            new string [] { "\uBDEB", "\uBDEB", "\u1107\u1171\u11BE", "\uBDEB", "\u1107\u1171\u11BE" },
            new string [] { "\uBDEC", "\uBDEC", "\u1107\u1171\u11BF", "\uBDEC", "\u1107\u1171\u11BF" },
            new string [] { "\uBDED", "\uBDED", "\u1107\u1171\u11C0", "\uBDED", "\u1107\u1171\u11C0" },
            new string [] { "\uBDEE", "\uBDEE", "\u1107\u1171\u11C1", "\uBDEE", "\u1107\u1171\u11C1" },
            new string [] { "\uBDEF", "\uBDEF", "\u1107\u1171\u11C2", "\uBDEF", "\u1107\u1171\u11C2" },
            new string [] { "\uBDF0", "\uBDF0", "\u1107\u1172", "\uBDF0", "\u1107\u1172" },
            new string [] { "\uBDF1", "\uBDF1", "\u1107\u1172\u11A8", "\uBDF1", "\u1107\u1172\u11A8" },
            new string [] { "\uBDF2", "\uBDF2", "\u1107\u1172\u11A9", "\uBDF2", "\u1107\u1172\u11A9" },
            new string [] { "\uBDF3", "\uBDF3", "\u1107\u1172\u11AA", "\uBDF3", "\u1107\u1172\u11AA" },
            new string [] { "\uBDF4", "\uBDF4", "\u1107\u1172\u11AB", "\uBDF4", "\u1107\u1172\u11AB" },
            new string [] { "\uBDF5", "\uBDF5", "\u1107\u1172\u11AC", "\uBDF5", "\u1107\u1172\u11AC" },
            new string [] { "\uBDF6", "\uBDF6", "\u1107\u1172\u11AD", "\uBDF6", "\u1107\u1172\u11AD" },
            new string [] { "\uBDF7", "\uBDF7", "\u1107\u1172\u11AE", "\uBDF7", "\u1107\u1172\u11AE" },
            new string [] { "\uBDF8", "\uBDF8", "\u1107\u1172\u11AF", "\uBDF8", "\u1107\u1172\u11AF" },
            new string [] { "\uBDF9", "\uBDF9", "\u1107\u1172\u11B0", "\uBDF9", "\u1107\u1172\u11B0" },
            new string [] { "\uBDFA", "\uBDFA", "\u1107\u1172\u11B1", "\uBDFA", "\u1107\u1172\u11B1" },
            new string [] { "\uBDFB", "\uBDFB", "\u1107\u1172\u11B2", "\uBDFB", "\u1107\u1172\u11B2" },
            new string [] { "\uBDFC", "\uBDFC", "\u1107\u1172\u11B3", "\uBDFC", "\u1107\u1172\u11B3" },
            new string [] { "\uBDFD", "\uBDFD", "\u1107\u1172\u11B4", "\uBDFD", "\u1107\u1172\u11B4" },
            new string [] { "\uBDFE", "\uBDFE", "\u1107\u1172\u11B5", "\uBDFE", "\u1107\u1172\u11B5" },
            new string [] { "\uBDFF", "\uBDFF", "\u1107\u1172\u11B6", "\uBDFF", "\u1107\u1172\u11B6" },
            new string [] { "\uBE00", "\uBE00", "\u1107\u1172\u11B7", "\uBE00", "\u1107\u1172\u11B7" },
            new string [] { "\uBE01", "\uBE01", "\u1107\u1172\u11B8", "\uBE01", "\u1107\u1172\u11B8" },
            new string [] { "\uBE02", "\uBE02", "\u1107\u1172\u11B9", "\uBE02", "\u1107\u1172\u11B9" },
            new string [] { "\uBE03", "\uBE03", "\u1107\u1172\u11BA", "\uBE03", "\u1107\u1172\u11BA" },
            new string [] { "\uBE04", "\uBE04", "\u1107\u1172\u11BB", "\uBE04", "\u1107\u1172\u11BB" },
            new string [] { "\uBE05", "\uBE05", "\u1107\u1172\u11BC", "\uBE05", "\u1107\u1172\u11BC" },
            new string [] { "\uBE06", "\uBE06", "\u1107\u1172\u11BD", "\uBE06", "\u1107\u1172\u11BD" },
            new string [] { "\uBE07", "\uBE07", "\u1107\u1172\u11BE", "\uBE07", "\u1107\u1172\u11BE" },
            new string [] { "\uBE08", "\uBE08", "\u1107\u1172\u11BF", "\uBE08", "\u1107\u1172\u11BF" },
            new string [] { "\uBE09", "\uBE09", "\u1107\u1172\u11C0", "\uBE09", "\u1107\u1172\u11C0" },
            new string [] { "\uBE0A", "\uBE0A", "\u1107\u1172\u11C1", "\uBE0A", "\u1107\u1172\u11C1" },
            new string [] { "\uBE0B", "\uBE0B", "\u1107\u1172\u11C2", "\uBE0B", "\u1107\u1172\u11C2" },
            new string [] { "\uBE0C", "\uBE0C", "\u1107\u1173", "\uBE0C", "\u1107\u1173" },
            new string [] { "\uBE0D", "\uBE0D", "\u1107\u1173\u11A8", "\uBE0D", "\u1107\u1173\u11A8" },
            new string [] { "\uBE0E", "\uBE0E", "\u1107\u1173\u11A9", "\uBE0E", "\u1107\u1173\u11A9" },
            new string [] { "\uBE0F", "\uBE0F", "\u1107\u1173\u11AA", "\uBE0F", "\u1107\u1173\u11AA" },
            new string [] { "\uBE10", "\uBE10", "\u1107\u1173\u11AB", "\uBE10", "\u1107\u1173\u11AB" },
            new string [] { "\uBE11", "\uBE11", "\u1107\u1173\u11AC", "\uBE11", "\u1107\u1173\u11AC" },
            new string [] { "\uBE12", "\uBE12", "\u1107\u1173\u11AD", "\uBE12", "\u1107\u1173\u11AD" },
            new string [] { "\uBE13", "\uBE13", "\u1107\u1173\u11AE", "\uBE13", "\u1107\u1173\u11AE" },
            new string [] { "\uBE14", "\uBE14", "\u1107\u1173\u11AF", "\uBE14", "\u1107\u1173\u11AF" },
            new string [] { "\uBE15", "\uBE15", "\u1107\u1173\u11B0", "\uBE15", "\u1107\u1173\u11B0" },
            new string [] { "\uBE16", "\uBE16", "\u1107\u1173\u11B1", "\uBE16", "\u1107\u1173\u11B1" },
            new string [] { "\uBE17", "\uBE17", "\u1107\u1173\u11B2", "\uBE17", "\u1107\u1173\u11B2" },
            new string [] { "\uBE18", "\uBE18", "\u1107\u1173\u11B3", "\uBE18", "\u1107\u1173\u11B3" },
            new string [] { "\uBE19", "\uBE19", "\u1107\u1173\u11B4", "\uBE19", "\u1107\u1173\u11B4" },
            new string [] { "\uBE1A", "\uBE1A", "\u1107\u1173\u11B5", "\uBE1A", "\u1107\u1173\u11B5" },
            new string [] { "\uBE1B", "\uBE1B", "\u1107\u1173\u11B6", "\uBE1B", "\u1107\u1173\u11B6" },
            new string [] { "\uBE1C", "\uBE1C", "\u1107\u1173\u11B7", "\uBE1C", "\u1107\u1173\u11B7" },
            new string [] { "\uBE1D", "\uBE1D", "\u1107\u1173\u11B8", "\uBE1D", "\u1107\u1173\u11B8" },
            new string [] { "\uBE1E", "\uBE1E", "\u1107\u1173\u11B9", "\uBE1E", "\u1107\u1173\u11B9" },
            new string [] { "\uBE1F", "\uBE1F", "\u1107\u1173\u11BA", "\uBE1F", "\u1107\u1173\u11BA" },
            new string [] { "\uBE20", "\uBE20", "\u1107\u1173\u11BB", "\uBE20", "\u1107\u1173\u11BB" },
            new string [] { "\uBE21", "\uBE21", "\u1107\u1173\u11BC", "\uBE21", "\u1107\u1173\u11BC" },
            new string [] { "\uBE22", "\uBE22", "\u1107\u1173\u11BD", "\uBE22", "\u1107\u1173\u11BD" },
            new string [] { "\uBE23", "\uBE23", "\u1107\u1173\u11BE", "\uBE23", "\u1107\u1173\u11BE" },
            new string [] { "\uBE24", "\uBE24", "\u1107\u1173\u11BF", "\uBE24", "\u1107\u1173\u11BF" },
            new string [] { "\uBE25", "\uBE25", "\u1107\u1173\u11C0", "\uBE25", "\u1107\u1173\u11C0" },
            new string [] { "\uBE26", "\uBE26", "\u1107\u1173\u11C1", "\uBE26", "\u1107\u1173\u11C1" },
            new string [] { "\uBE27", "\uBE27", "\u1107\u1173\u11C2", "\uBE27", "\u1107\u1173\u11C2" },
            new string [] { "\uBE28", "\uBE28", "\u1107\u1174", "\uBE28", "\u1107\u1174" },
            new string [] { "\uBE29", "\uBE29", "\u1107\u1174\u11A8", "\uBE29", "\u1107\u1174\u11A8" },
            new string [] { "\uBE2A", "\uBE2A", "\u1107\u1174\u11A9", "\uBE2A", "\u1107\u1174\u11A9" },
            new string [] { "\uBE2B", "\uBE2B", "\u1107\u1174\u11AA", "\uBE2B", "\u1107\u1174\u11AA" },
            new string [] { "\uBE2C", "\uBE2C", "\u1107\u1174\u11AB", "\uBE2C", "\u1107\u1174\u11AB" },
            new string [] { "\uBE2D", "\uBE2D", "\u1107\u1174\u11AC", "\uBE2D", "\u1107\u1174\u11AC" },
            new string [] { "\uBE2E", "\uBE2E", "\u1107\u1174\u11AD", "\uBE2E", "\u1107\u1174\u11AD" },
            new string [] { "\uBE2F", "\uBE2F", "\u1107\u1174\u11AE", "\uBE2F", "\u1107\u1174\u11AE" },
            new string [] { "\uBE30", "\uBE30", "\u1107\u1174\u11AF", "\uBE30", "\u1107\u1174\u11AF" },
            new string [] { "\uBE31", "\uBE31", "\u1107\u1174\u11B0", "\uBE31", "\u1107\u1174\u11B0" },
            new string [] { "\uBE32", "\uBE32", "\u1107\u1174\u11B1", "\uBE32", "\u1107\u1174\u11B1" },
            new string [] { "\uBE33", "\uBE33", "\u1107\u1174\u11B2", "\uBE33", "\u1107\u1174\u11B2" },
            new string [] { "\uBE34", "\uBE34", "\u1107\u1174\u11B3", "\uBE34", "\u1107\u1174\u11B3" },
            new string [] { "\uBE35", "\uBE35", "\u1107\u1174\u11B4", "\uBE35", "\u1107\u1174\u11B4" },
            new string [] { "\uBE36", "\uBE36", "\u1107\u1174\u11B5", "\uBE36", "\u1107\u1174\u11B5" },
            new string [] { "\uBE37", "\uBE37", "\u1107\u1174\u11B6", "\uBE37", "\u1107\u1174\u11B6" },
            new string [] { "\uBE38", "\uBE38", "\u1107\u1174\u11B7", "\uBE38", "\u1107\u1174\u11B7" },
            new string [] { "\uBE39", "\uBE39", "\u1107\u1174\u11B8", "\uBE39", "\u1107\u1174\u11B8" },
            new string [] { "\uBE3A", "\uBE3A", "\u1107\u1174\u11B9", "\uBE3A", "\u1107\u1174\u11B9" },
            new string [] { "\uBE3B", "\uBE3B", "\u1107\u1174\u11BA", "\uBE3B", "\u1107\u1174\u11BA" },
            new string [] { "\uBE3C", "\uBE3C", "\u1107\u1174\u11BB", "\uBE3C", "\u1107\u1174\u11BB" },
            new string [] { "\uBE3D", "\uBE3D", "\u1107\u1174\u11BC", "\uBE3D", "\u1107\u1174\u11BC" },
            new string [] { "\uBE3E", "\uBE3E", "\u1107\u1174\u11BD", "\uBE3E", "\u1107\u1174\u11BD" },
            new string [] { "\uBE3F", "\uBE3F", "\u1107\u1174\u11BE", "\uBE3F", "\u1107\u1174\u11BE" },
            new string [] { "\uBE40", "\uBE40", "\u1107\u1174\u11BF", "\uBE40", "\u1107\u1174\u11BF" },
            new string [] { "\uBE41", "\uBE41", "\u1107\u1174\u11C0", "\uBE41", "\u1107\u1174\u11C0" },
            new string [] { "\uBE42", "\uBE42", "\u1107\u1174\u11C1", "\uBE42", "\u1107\u1174\u11C1" },
            new string [] { "\uBE43", "\uBE43", "\u1107\u1174\u11C2", "\uBE43", "\u1107\u1174\u11C2" },
            new string [] { "\uBE44", "\uBE44", "\u1107\u1175", "\uBE44", "\u1107\u1175" },
            new string [] { "\uBE45", "\uBE45", "\u1107\u1175\u11A8", "\uBE45", "\u1107\u1175\u11A8" },
            new string [] { "\uBE46", "\uBE46", "\u1107\u1175\u11A9", "\uBE46", "\u1107\u1175\u11A9" },
            new string [] { "\uBE47", "\uBE47", "\u1107\u1175\u11AA", "\uBE47", "\u1107\u1175\u11AA" },
            new string [] { "\uBE48", "\uBE48", "\u1107\u1175\u11AB", "\uBE48", "\u1107\u1175\u11AB" },
            new string [] { "\uBE49", "\uBE49", "\u1107\u1175\u11AC", "\uBE49", "\u1107\u1175\u11AC" },
            new string [] { "\uBE4A", "\uBE4A", "\u1107\u1175\u11AD", "\uBE4A", "\u1107\u1175\u11AD" },
            new string [] { "\uBE4B", "\uBE4B", "\u1107\u1175\u11AE", "\uBE4B", "\u1107\u1175\u11AE" },
            new string [] { "\uBE4C", "\uBE4C", "\u1107\u1175\u11AF", "\uBE4C", "\u1107\u1175\u11AF" },
            new string [] { "\uBE4D", "\uBE4D", "\u1107\u1175\u11B0", "\uBE4D", "\u1107\u1175\u11B0" },
            new string [] { "\uBE4E", "\uBE4E", "\u1107\u1175\u11B1", "\uBE4E", "\u1107\u1175\u11B1" },
            new string [] { "\uBE4F", "\uBE4F", "\u1107\u1175\u11B2", "\uBE4F", "\u1107\u1175\u11B2" },
            new string [] { "\uBE50", "\uBE50", "\u1107\u1175\u11B3", "\uBE50", "\u1107\u1175\u11B3" },
            new string [] { "\uBE51", "\uBE51", "\u1107\u1175\u11B4", "\uBE51", "\u1107\u1175\u11B4" },
            new string [] { "\uBE52", "\uBE52", "\u1107\u1175\u11B5", "\uBE52", "\u1107\u1175\u11B5" },
            new string [] { "\uBE53", "\uBE53", "\u1107\u1175\u11B6", "\uBE53", "\u1107\u1175\u11B6" },
            new string [] { "\uBE54", "\uBE54", "\u1107\u1175\u11B7", "\uBE54", "\u1107\u1175\u11B7" },
            new string [] { "\uBE55", "\uBE55", "\u1107\u1175\u11B8", "\uBE55", "\u1107\u1175\u11B8" },
            new string [] { "\uBE56", "\uBE56", "\u1107\u1175\u11B9", "\uBE56", "\u1107\u1175\u11B9" },
            new string [] { "\uBE57", "\uBE57", "\u1107\u1175\u11BA", "\uBE57", "\u1107\u1175\u11BA" },
            new string [] { "\uBE58", "\uBE58", "\u1107\u1175\u11BB", "\uBE58", "\u1107\u1175\u11BB" },
            new string [] { "\uBE59", "\uBE59", "\u1107\u1175\u11BC", "\uBE59", "\u1107\u1175\u11BC" },
            new string [] { "\uBE5A", "\uBE5A", "\u1107\u1175\u11BD", "\uBE5A", "\u1107\u1175\u11BD" },
            new string [] { "\uBE5B", "\uBE5B", "\u1107\u1175\u11BE", "\uBE5B", "\u1107\u1175\u11BE" },
            new string [] { "\uBE5C", "\uBE5C", "\u1107\u1175\u11BF", "\uBE5C", "\u1107\u1175\u11BF" },
            new string [] { "\uBE5D", "\uBE5D", "\u1107\u1175\u11C0", "\uBE5D", "\u1107\u1175\u11C0" },
            new string [] { "\uBE5E", "\uBE5E", "\u1107\u1175\u11C1", "\uBE5E", "\u1107\u1175\u11C1" },
            new string [] { "\uBE5F", "\uBE5F", "\u1107\u1175\u11C2", "\uBE5F", "\u1107\u1175\u11C2" },
            new string [] { "\uBE60", "\uBE60", "\u1108\u1161", "\uBE60", "\u1108\u1161" },
            new string [] { "\uBE61", "\uBE61", "\u1108\u1161\u11A8", "\uBE61", "\u1108\u1161\u11A8" },
            new string [] { "\uBE62", "\uBE62", "\u1108\u1161\u11A9", "\uBE62", "\u1108\u1161\u11A9" },
            new string [] { "\uBE63", "\uBE63", "\u1108\u1161\u11AA", "\uBE63", "\u1108\u1161\u11AA" },
            new string [] { "\uBE64", "\uBE64", "\u1108\u1161\u11AB", "\uBE64", "\u1108\u1161\u11AB" },
            new string [] { "\uBE65", "\uBE65", "\u1108\u1161\u11AC", "\uBE65", "\u1108\u1161\u11AC" },
            new string [] { "\uBE66", "\uBE66", "\u1108\u1161\u11AD", "\uBE66", "\u1108\u1161\u11AD" },
            new string [] { "\uBE67", "\uBE67", "\u1108\u1161\u11AE", "\uBE67", "\u1108\u1161\u11AE" },
            new string [] { "\uBE68", "\uBE68", "\u1108\u1161\u11AF", "\uBE68", "\u1108\u1161\u11AF" },
            new string [] { "\uBE69", "\uBE69", "\u1108\u1161\u11B0", "\uBE69", "\u1108\u1161\u11B0" },
            new string [] { "\uBE6A", "\uBE6A", "\u1108\u1161\u11B1", "\uBE6A", "\u1108\u1161\u11B1" },
            new string [] { "\uBE6B", "\uBE6B", "\u1108\u1161\u11B2", "\uBE6B", "\u1108\u1161\u11B2" },
            new string [] { "\uBE6C", "\uBE6C", "\u1108\u1161\u11B3", "\uBE6C", "\u1108\u1161\u11B3" },
            new string [] { "\uBE6D", "\uBE6D", "\u1108\u1161\u11B4", "\uBE6D", "\u1108\u1161\u11B4" },
            new string [] { "\uBE6E", "\uBE6E", "\u1108\u1161\u11B5", "\uBE6E", "\u1108\u1161\u11B5" },
            new string [] { "\uBE6F", "\uBE6F", "\u1108\u1161\u11B6", "\uBE6F", "\u1108\u1161\u11B6" },
            new string [] { "\uBE70", "\uBE70", "\u1108\u1161\u11B7", "\uBE70", "\u1108\u1161\u11B7" },
            new string [] { "\uBE71", "\uBE71", "\u1108\u1161\u11B8", "\uBE71", "\u1108\u1161\u11B8" },
            new string [] { "\uBE72", "\uBE72", "\u1108\u1161\u11B9", "\uBE72", "\u1108\u1161\u11B9" },
            new string [] { "\uBE73", "\uBE73", "\u1108\u1161\u11BA", "\uBE73", "\u1108\u1161\u11BA" },
            new string [] { "\uBE74", "\uBE74", "\u1108\u1161\u11BB", "\uBE74", "\u1108\u1161\u11BB" },
            new string [] { "\uBE75", "\uBE75", "\u1108\u1161\u11BC", "\uBE75", "\u1108\u1161\u11BC" },
            new string [] { "\uBE76", "\uBE76", "\u1108\u1161\u11BD", "\uBE76", "\u1108\u1161\u11BD" },
            new string [] { "\uBE77", "\uBE77", "\u1108\u1161\u11BE", "\uBE77", "\u1108\u1161\u11BE" },
            new string [] { "\uBE78", "\uBE78", "\u1108\u1161\u11BF", "\uBE78", "\u1108\u1161\u11BF" },
            new string [] { "\uBE79", "\uBE79", "\u1108\u1161\u11C0", "\uBE79", "\u1108\u1161\u11C0" },
            new string [] { "\uBE7A", "\uBE7A", "\u1108\u1161\u11C1", "\uBE7A", "\u1108\u1161\u11C1" },
            new string [] { "\uBE7B", "\uBE7B", "\u1108\u1161\u11C2", "\uBE7B", "\u1108\u1161\u11C2" },
            new string [] { "\uBE7C", "\uBE7C", "\u1108\u1162", "\uBE7C", "\u1108\u1162" },
            new string [] { "\uBE7D", "\uBE7D", "\u1108\u1162\u11A8", "\uBE7D", "\u1108\u1162\u11A8" },
            new string [] { "\uBE7E", "\uBE7E", "\u1108\u1162\u11A9", "\uBE7E", "\u1108\u1162\u11A9" },
            new string [] { "\uBE7F", "\uBE7F", "\u1108\u1162\u11AA", "\uBE7F", "\u1108\u1162\u11AA" },
            new string [] { "\uBE80", "\uBE80", "\u1108\u1162\u11AB", "\uBE80", "\u1108\u1162\u11AB" },
            new string [] { "\uBE81", "\uBE81", "\u1108\u1162\u11AC", "\uBE81", "\u1108\u1162\u11AC" },
            new string [] { "\uBE82", "\uBE82", "\u1108\u1162\u11AD", "\uBE82", "\u1108\u1162\u11AD" },
            new string [] { "\uBE83", "\uBE83", "\u1108\u1162\u11AE", "\uBE83", "\u1108\u1162\u11AE" },
            new string [] { "\uBE84", "\uBE84", "\u1108\u1162\u11AF", "\uBE84", "\u1108\u1162\u11AF" },
            new string [] { "\uBE85", "\uBE85", "\u1108\u1162\u11B0", "\uBE85", "\u1108\u1162\u11B0" },
            new string [] { "\uBE86", "\uBE86", "\u1108\u1162\u11B1", "\uBE86", "\u1108\u1162\u11B1" },
            new string [] { "\uBE87", "\uBE87", "\u1108\u1162\u11B2", "\uBE87", "\u1108\u1162\u11B2" },
            new string [] { "\uBE88", "\uBE88", "\u1108\u1162\u11B3", "\uBE88", "\u1108\u1162\u11B3" },
            new string [] { "\uBE89", "\uBE89", "\u1108\u1162\u11B4", "\uBE89", "\u1108\u1162\u11B4" },
            new string [] { "\uBE8A", "\uBE8A", "\u1108\u1162\u11B5", "\uBE8A", "\u1108\u1162\u11B5" },
            new string [] { "\uBE8B", "\uBE8B", "\u1108\u1162\u11B6", "\uBE8B", "\u1108\u1162\u11B6" },
            new string [] { "\uBE8C", "\uBE8C", "\u1108\u1162\u11B7", "\uBE8C", "\u1108\u1162\u11B7" },
            new string [] { "\uBE8D", "\uBE8D", "\u1108\u1162\u11B8", "\uBE8D", "\u1108\u1162\u11B8" },
            new string [] { "\uBE8E", "\uBE8E", "\u1108\u1162\u11B9", "\uBE8E", "\u1108\u1162\u11B9" },
            new string [] { "\uBE8F", "\uBE8F", "\u1108\u1162\u11BA", "\uBE8F", "\u1108\u1162\u11BA" },
            new string [] { "\uBE90", "\uBE90", "\u1108\u1162\u11BB", "\uBE90", "\u1108\u1162\u11BB" },
            new string [] { "\uBE91", "\uBE91", "\u1108\u1162\u11BC", "\uBE91", "\u1108\u1162\u11BC" },
            new string [] { "\uBE92", "\uBE92", "\u1108\u1162\u11BD", "\uBE92", "\u1108\u1162\u11BD" },
            new string [] { "\uBE93", "\uBE93", "\u1108\u1162\u11BE", "\uBE93", "\u1108\u1162\u11BE" },
            new string [] { "\uBE94", "\uBE94", "\u1108\u1162\u11BF", "\uBE94", "\u1108\u1162\u11BF" },
            new string [] { "\uBE95", "\uBE95", "\u1108\u1162\u11C0", "\uBE95", "\u1108\u1162\u11C0" },
            new string [] { "\uBE96", "\uBE96", "\u1108\u1162\u11C1", "\uBE96", "\u1108\u1162\u11C1" },
            new string [] { "\uBE97", "\uBE97", "\u1108\u1162\u11C2", "\uBE97", "\u1108\u1162\u11C2" },
            new string [] { "\uBE98", "\uBE98", "\u1108\u1163", "\uBE98", "\u1108\u1163" },
            new string [] { "\uBE99", "\uBE99", "\u1108\u1163\u11A8", "\uBE99", "\u1108\u1163\u11A8" },
            new string [] { "\uBE9A", "\uBE9A", "\u1108\u1163\u11A9", "\uBE9A", "\u1108\u1163\u11A9" },
            new string [] { "\uBE9B", "\uBE9B", "\u1108\u1163\u11AA", "\uBE9B", "\u1108\u1163\u11AA" },
            new string [] { "\uBE9C", "\uBE9C", "\u1108\u1163\u11AB", "\uBE9C", "\u1108\u1163\u11AB" },
            new string [] { "\uBE9D", "\uBE9D", "\u1108\u1163\u11AC", "\uBE9D", "\u1108\u1163\u11AC" },
            new string [] { "\uBE9E", "\uBE9E", "\u1108\u1163\u11AD", "\uBE9E", "\u1108\u1163\u11AD" },
            new string [] { "\uBE9F", "\uBE9F", "\u1108\u1163\u11AE", "\uBE9F", "\u1108\u1163\u11AE" },
            new string [] { "\uBEA0", "\uBEA0", "\u1108\u1163\u11AF", "\uBEA0", "\u1108\u1163\u11AF" },
            new string [] { "\uBEA1", "\uBEA1", "\u1108\u1163\u11B0", "\uBEA1", "\u1108\u1163\u11B0" },
            new string [] { "\uBEA2", "\uBEA2", "\u1108\u1163\u11B1", "\uBEA2", "\u1108\u1163\u11B1" },
            new string [] { "\uBEA3", "\uBEA3", "\u1108\u1163\u11B2", "\uBEA3", "\u1108\u1163\u11B2" },
            new string [] { "\uBEA4", "\uBEA4", "\u1108\u1163\u11B3", "\uBEA4", "\u1108\u1163\u11B3" },
            new string [] { "\uBEA5", "\uBEA5", "\u1108\u1163\u11B4", "\uBEA5", "\u1108\u1163\u11B4" },
            new string [] { "\uBEA6", "\uBEA6", "\u1108\u1163\u11B5", "\uBEA6", "\u1108\u1163\u11B5" },
            new string [] { "\uBEA7", "\uBEA7", "\u1108\u1163\u11B6", "\uBEA7", "\u1108\u1163\u11B6" },
            new string [] { "\uBEA8", "\uBEA8", "\u1108\u1163\u11B7", "\uBEA8", "\u1108\u1163\u11B7" },
            new string [] { "\uBEA9", "\uBEA9", "\u1108\u1163\u11B8", "\uBEA9", "\u1108\u1163\u11B8" },
            new string [] { "\uBEAA", "\uBEAA", "\u1108\u1163\u11B9", "\uBEAA", "\u1108\u1163\u11B9" },
            new string [] { "\uBEAB", "\uBEAB", "\u1108\u1163\u11BA", "\uBEAB", "\u1108\u1163\u11BA" },
            new string [] { "\uBEAC", "\uBEAC", "\u1108\u1163\u11BB", "\uBEAC", "\u1108\u1163\u11BB" },
            new string [] { "\uBEAD", "\uBEAD", "\u1108\u1163\u11BC", "\uBEAD", "\u1108\u1163\u11BC" },
            new string [] { "\uBEAE", "\uBEAE", "\u1108\u1163\u11BD", "\uBEAE", "\u1108\u1163\u11BD" },
            new string [] { "\uBEAF", "\uBEAF", "\u1108\u1163\u11BE", "\uBEAF", "\u1108\u1163\u11BE" },
            new string [] { "\uBEB0", "\uBEB0", "\u1108\u1163\u11BF", "\uBEB0", "\u1108\u1163\u11BF" },
            new string [] { "\uBEB1", "\uBEB1", "\u1108\u1163\u11C0", "\uBEB1", "\u1108\u1163\u11C0" },
            new string [] { "\uBEB2", "\uBEB2", "\u1108\u1163\u11C1", "\uBEB2", "\u1108\u1163\u11C1" },
            new string [] { "\uBEB3", "\uBEB3", "\u1108\u1163\u11C2", "\uBEB3", "\u1108\u1163\u11C2" },
            new string [] { "\uBEB4", "\uBEB4", "\u1108\u1164", "\uBEB4", "\u1108\u1164" },
            new string [] { "\uBEB5", "\uBEB5", "\u1108\u1164\u11A8", "\uBEB5", "\u1108\u1164\u11A8" },
            new string [] { "\uBEB6", "\uBEB6", "\u1108\u1164\u11A9", "\uBEB6", "\u1108\u1164\u11A9" },
            new string [] { "\uBEB7", "\uBEB7", "\u1108\u1164\u11AA", "\uBEB7", "\u1108\u1164\u11AA" },
            new string [] { "\uBEB8", "\uBEB8", "\u1108\u1164\u11AB", "\uBEB8", "\u1108\u1164\u11AB" },
            new string [] { "\uBEB9", "\uBEB9", "\u1108\u1164\u11AC", "\uBEB9", "\u1108\u1164\u11AC" },
            new string [] { "\uBEBA", "\uBEBA", "\u1108\u1164\u11AD", "\uBEBA", "\u1108\u1164\u11AD" },
            new string [] { "\uBEBB", "\uBEBB", "\u1108\u1164\u11AE", "\uBEBB", "\u1108\u1164\u11AE" },
            new string [] { "\uBEBC", "\uBEBC", "\u1108\u1164\u11AF", "\uBEBC", "\u1108\u1164\u11AF" },
            new string [] { "\uBEBD", "\uBEBD", "\u1108\u1164\u11B0", "\uBEBD", "\u1108\u1164\u11B0" },
            new string [] { "\uBEBE", "\uBEBE", "\u1108\u1164\u11B1", "\uBEBE", "\u1108\u1164\u11B1" },
            new string [] { "\uBEBF", "\uBEBF", "\u1108\u1164\u11B2", "\uBEBF", "\u1108\u1164\u11B2" },
            new string [] { "\uBEC0", "\uBEC0", "\u1108\u1164\u11B3", "\uBEC0", "\u1108\u1164\u11B3" },
            new string [] { "\uBEC1", "\uBEC1", "\u1108\u1164\u11B4", "\uBEC1", "\u1108\u1164\u11B4" },
            new string [] { "\uBEC2", "\uBEC2", "\u1108\u1164\u11B5", "\uBEC2", "\u1108\u1164\u11B5" },
            new string [] { "\uBEC3", "\uBEC3", "\u1108\u1164\u11B6", "\uBEC3", "\u1108\u1164\u11B6" },
            new string [] { "\uBEC4", "\uBEC4", "\u1108\u1164\u11B7", "\uBEC4", "\u1108\u1164\u11B7" },
            new string [] { "\uBEC5", "\uBEC5", "\u1108\u1164\u11B8", "\uBEC5", "\u1108\u1164\u11B8" },
            new string [] { "\uBEC6", "\uBEC6", "\u1108\u1164\u11B9", "\uBEC6", "\u1108\u1164\u11B9" },
            new string [] { "\uBEC7", "\uBEC7", "\u1108\u1164\u11BA", "\uBEC7", "\u1108\u1164\u11BA" },
            new string [] { "\uBEC8", "\uBEC8", "\u1108\u1164\u11BB", "\uBEC8", "\u1108\u1164\u11BB" },
            new string [] { "\uBEC9", "\uBEC9", "\u1108\u1164\u11BC", "\uBEC9", "\u1108\u1164\u11BC" },
            new string [] { "\uBECA", "\uBECA", "\u1108\u1164\u11BD", "\uBECA", "\u1108\u1164\u11BD" },
            new string [] { "\uBECB", "\uBECB", "\u1108\u1164\u11BE", "\uBECB", "\u1108\u1164\u11BE" },
            new string [] { "\uBECC", "\uBECC", "\u1108\u1164\u11BF", "\uBECC", "\u1108\u1164\u11BF" },
            new string [] { "\uBECD", "\uBECD", "\u1108\u1164\u11C0", "\uBECD", "\u1108\u1164\u11C0" },
            new string [] { "\uBECE", "\uBECE", "\u1108\u1164\u11C1", "\uBECE", "\u1108\u1164\u11C1" },
            new string [] { "\uBECF", "\uBECF", "\u1108\u1164\u11C2", "\uBECF", "\u1108\u1164\u11C2" },
            new string [] { "\uBED0", "\uBED0", "\u1108\u1165", "\uBED0", "\u1108\u1165" },
            new string [] { "\uBED1", "\uBED1", "\u1108\u1165\u11A8", "\uBED1", "\u1108\u1165\u11A8" },
            new string [] { "\uBED2", "\uBED2", "\u1108\u1165\u11A9", "\uBED2", "\u1108\u1165\u11A9" },
            new string [] { "\uBED3", "\uBED3", "\u1108\u1165\u11AA", "\uBED3", "\u1108\u1165\u11AA" },
            new string [] { "\uBED4", "\uBED4", "\u1108\u1165\u11AB", "\uBED4", "\u1108\u1165\u11AB" },
            new string [] { "\uBED5", "\uBED5", "\u1108\u1165\u11AC", "\uBED5", "\u1108\u1165\u11AC" },
            new string [] { "\uBED6", "\uBED6", "\u1108\u1165\u11AD", "\uBED6", "\u1108\u1165\u11AD" },
            new string [] { "\uBED7", "\uBED7", "\u1108\u1165\u11AE", "\uBED7", "\u1108\u1165\u11AE" },
            new string [] { "\uBED8", "\uBED8", "\u1108\u1165\u11AF", "\uBED8", "\u1108\u1165\u11AF" },
            new string [] { "\uBED9", "\uBED9", "\u1108\u1165\u11B0", "\uBED9", "\u1108\u1165\u11B0" },
            new string [] { "\uBEDA", "\uBEDA", "\u1108\u1165\u11B1", "\uBEDA", "\u1108\u1165\u11B1" },
            new string [] { "\uBEDB", "\uBEDB", "\u1108\u1165\u11B2", "\uBEDB", "\u1108\u1165\u11B2" },
            new string [] { "\uBEDC", "\uBEDC", "\u1108\u1165\u11B3", "\uBEDC", "\u1108\u1165\u11B3" },
            new string [] { "\uBEDD", "\uBEDD", "\u1108\u1165\u11B4", "\uBEDD", "\u1108\u1165\u11B4" },
            new string [] { "\uBEDE", "\uBEDE", "\u1108\u1165\u11B5", "\uBEDE", "\u1108\u1165\u11B5" },
            new string [] { "\uBEDF", "\uBEDF", "\u1108\u1165\u11B6", "\uBEDF", "\u1108\u1165\u11B6" },
            new string [] { "\uBEE0", "\uBEE0", "\u1108\u1165\u11B7", "\uBEE0", "\u1108\u1165\u11B7" },
            new string [] { "\uBEE1", "\uBEE1", "\u1108\u1165\u11B8", "\uBEE1", "\u1108\u1165\u11B8" },
            new string [] { "\uBEE2", "\uBEE2", "\u1108\u1165\u11B9", "\uBEE2", "\u1108\u1165\u11B9" },
            new string [] { "\uBEE3", "\uBEE3", "\u1108\u1165\u11BA", "\uBEE3", "\u1108\u1165\u11BA" },
            new string [] { "\uBEE4", "\uBEE4", "\u1108\u1165\u11BB", "\uBEE4", "\u1108\u1165\u11BB" },
            new string [] { "\uBEE5", "\uBEE5", "\u1108\u1165\u11BC", "\uBEE5", "\u1108\u1165\u11BC" },
            new string [] { "\uBEE6", "\uBEE6", "\u1108\u1165\u11BD", "\uBEE6", "\u1108\u1165\u11BD" },
            new string [] { "\uBEE7", "\uBEE7", "\u1108\u1165\u11BE", "\uBEE7", "\u1108\u1165\u11BE" },
            new string [] { "\uBEE8", "\uBEE8", "\u1108\u1165\u11BF", "\uBEE8", "\u1108\u1165\u11BF" },
            new string [] { "\uBEE9", "\uBEE9", "\u1108\u1165\u11C0", "\uBEE9", "\u1108\u1165\u11C0" },
            new string [] { "\uBEEA", "\uBEEA", "\u1108\u1165\u11C1", "\uBEEA", "\u1108\u1165\u11C1" },
            new string [] { "\uBEEB", "\uBEEB", "\u1108\u1165\u11C2", "\uBEEB", "\u1108\u1165\u11C2" },
            new string [] { "\uBEEC", "\uBEEC", "\u1108\u1166", "\uBEEC", "\u1108\u1166" },
            new string [] { "\uBEED", "\uBEED", "\u1108\u1166\u11A8", "\uBEED", "\u1108\u1166\u11A8" },
            new string [] { "\uBEEE", "\uBEEE", "\u1108\u1166\u11A9", "\uBEEE", "\u1108\u1166\u11A9" },
            new string [] { "\uBEEF", "\uBEEF", "\u1108\u1166\u11AA", "\uBEEF", "\u1108\u1166\u11AA" },
            new string [] { "\uBEF0", "\uBEF0", "\u1108\u1166\u11AB", "\uBEF0", "\u1108\u1166\u11AB" },
            new string [] { "\uBEF1", "\uBEF1", "\u1108\u1166\u11AC", "\uBEF1", "\u1108\u1166\u11AC" },
            new string [] { "\uBEF2", "\uBEF2", "\u1108\u1166\u11AD", "\uBEF2", "\u1108\u1166\u11AD" },
            new string [] { "\uBEF3", "\uBEF3", "\u1108\u1166\u11AE", "\uBEF3", "\u1108\u1166\u11AE" },
            new string [] { "\uBEF4", "\uBEF4", "\u1108\u1166\u11AF", "\uBEF4", "\u1108\u1166\u11AF" },
            new string [] { "\uBEF5", "\uBEF5", "\u1108\u1166\u11B0", "\uBEF5", "\u1108\u1166\u11B0" },
            new string [] { "\uBEF6", "\uBEF6", "\u1108\u1166\u11B1", "\uBEF6", "\u1108\u1166\u11B1" },
            new string [] { "\uBEF7", "\uBEF7", "\u1108\u1166\u11B2", "\uBEF7", "\u1108\u1166\u11B2" },
            new string [] { "\uBEF8", "\uBEF8", "\u1108\u1166\u11B3", "\uBEF8", "\u1108\u1166\u11B3" },
            new string [] { "\uBEF9", "\uBEF9", "\u1108\u1166\u11B4", "\uBEF9", "\u1108\u1166\u11B4" },
            new string [] { "\uBEFA", "\uBEFA", "\u1108\u1166\u11B5", "\uBEFA", "\u1108\u1166\u11B5" },
            new string [] { "\uBEFB", "\uBEFB", "\u1108\u1166\u11B6", "\uBEFB", "\u1108\u1166\u11B6" },
            new string [] { "\uBEFC", "\uBEFC", "\u1108\u1166\u11B7", "\uBEFC", "\u1108\u1166\u11B7" },
            new string [] { "\uBEFD", "\uBEFD", "\u1108\u1166\u11B8", "\uBEFD", "\u1108\u1166\u11B8" },
            new string [] { "\uBEFE", "\uBEFE", "\u1108\u1166\u11B9", "\uBEFE", "\u1108\u1166\u11B9" },
            new string [] { "\uBEFF", "\uBEFF", "\u1108\u1166\u11BA", "\uBEFF", "\u1108\u1166\u11BA" },
            new string [] { "\uBF00", "\uBF00", "\u1108\u1166\u11BB", "\uBF00", "\u1108\u1166\u11BB" },
            new string [] { "\uBF01", "\uBF01", "\u1108\u1166\u11BC", "\uBF01", "\u1108\u1166\u11BC" },
            new string [] { "\uBF02", "\uBF02", "\u1108\u1166\u11BD", "\uBF02", "\u1108\u1166\u11BD" },
            new string [] { "\uBF03", "\uBF03", "\u1108\u1166\u11BE", "\uBF03", "\u1108\u1166\u11BE" },
            new string [] { "\uBF04", "\uBF04", "\u1108\u1166\u11BF", "\uBF04", "\u1108\u1166\u11BF" },
            new string [] { "\uBF05", "\uBF05", "\u1108\u1166\u11C0", "\uBF05", "\u1108\u1166\u11C0" },
            new string [] { "\uBF06", "\uBF06", "\u1108\u1166\u11C1", "\uBF06", "\u1108\u1166\u11C1" },
            new string [] { "\uBF07", "\uBF07", "\u1108\u1166\u11C2", "\uBF07", "\u1108\u1166\u11C2" },
            new string [] { "\uBF08", "\uBF08", "\u1108\u1167", "\uBF08", "\u1108\u1167" },
            new string [] { "\uBF09", "\uBF09", "\u1108\u1167\u11A8", "\uBF09", "\u1108\u1167\u11A8" },
            new string [] { "\uBF0A", "\uBF0A", "\u1108\u1167\u11A9", "\uBF0A", "\u1108\u1167\u11A9" },
            new string [] { "\uBF0B", "\uBF0B", "\u1108\u1167\u11AA", "\uBF0B", "\u1108\u1167\u11AA" },
            new string [] { "\uBF0C", "\uBF0C", "\u1108\u1167\u11AB", "\uBF0C", "\u1108\u1167\u11AB" },
            new string [] { "\uBF0D", "\uBF0D", "\u1108\u1167\u11AC", "\uBF0D", "\u1108\u1167\u11AC" },
            new string [] { "\uBF0E", "\uBF0E", "\u1108\u1167\u11AD", "\uBF0E", "\u1108\u1167\u11AD" },
            new string [] { "\uBF0F", "\uBF0F", "\u1108\u1167\u11AE", "\uBF0F", "\u1108\u1167\u11AE" },
            new string [] { "\uBF10", "\uBF10", "\u1108\u1167\u11AF", "\uBF10", "\u1108\u1167\u11AF" },
            new string [] { "\uBF11", "\uBF11", "\u1108\u1167\u11B0", "\uBF11", "\u1108\u1167\u11B0" },
            new string [] { "\uBF12", "\uBF12", "\u1108\u1167\u11B1", "\uBF12", "\u1108\u1167\u11B1" },
            new string [] { "\uBF13", "\uBF13", "\u1108\u1167\u11B2", "\uBF13", "\u1108\u1167\u11B2" },
            new string [] { "\uBF14", "\uBF14", "\u1108\u1167\u11B3", "\uBF14", "\u1108\u1167\u11B3" },
            new string [] { "\uBF15", "\uBF15", "\u1108\u1167\u11B4", "\uBF15", "\u1108\u1167\u11B4" },
            new string [] { "\uBF16", "\uBF16", "\u1108\u1167\u11B5", "\uBF16", "\u1108\u1167\u11B5" },
            new string [] { "\uBF17", "\uBF17", "\u1108\u1167\u11B6", "\uBF17", "\u1108\u1167\u11B6" },
            new string [] { "\uBF18", "\uBF18", "\u1108\u1167\u11B7", "\uBF18", "\u1108\u1167\u11B7" },
            new string [] { "\uBF19", "\uBF19", "\u1108\u1167\u11B8", "\uBF19", "\u1108\u1167\u11B8" },
            new string [] { "\uBF1A", "\uBF1A", "\u1108\u1167\u11B9", "\uBF1A", "\u1108\u1167\u11B9" },
            new string [] { "\uBF1B", "\uBF1B", "\u1108\u1167\u11BA", "\uBF1B", "\u1108\u1167\u11BA" },
            new string [] { "\uBF1C", "\uBF1C", "\u1108\u1167\u11BB", "\uBF1C", "\u1108\u1167\u11BB" },
            new string [] { "\uBF1D", "\uBF1D", "\u1108\u1167\u11BC", "\uBF1D", "\u1108\u1167\u11BC" },
            new string [] { "\uBF1E", "\uBF1E", "\u1108\u1167\u11BD", "\uBF1E", "\u1108\u1167\u11BD" },
            new string [] { "\uBF1F", "\uBF1F", "\u1108\u1167\u11BE", "\uBF1F", "\u1108\u1167\u11BE" },
            new string [] { "\uBF20", "\uBF20", "\u1108\u1167\u11BF", "\uBF20", "\u1108\u1167\u11BF" },
            new string [] { "\uBF21", "\uBF21", "\u1108\u1167\u11C0", "\uBF21", "\u1108\u1167\u11C0" },
            new string [] { "\uBF22", "\uBF22", "\u1108\u1167\u11C1", "\uBF22", "\u1108\u1167\u11C1" },
            new string [] { "\uBF23", "\uBF23", "\u1108\u1167\u11C2", "\uBF23", "\u1108\u1167\u11C2" },
            new string [] { "\uBF24", "\uBF24", "\u1108\u1168", "\uBF24", "\u1108\u1168" },
            new string [] { "\uBF25", "\uBF25", "\u1108\u1168\u11A8", "\uBF25", "\u1108\u1168\u11A8" },
            new string [] { "\uBF26", "\uBF26", "\u1108\u1168\u11A9", "\uBF26", "\u1108\u1168\u11A9" },
            new string [] { "\uBF27", "\uBF27", "\u1108\u1168\u11AA", "\uBF27", "\u1108\u1168\u11AA" },
            new string [] { "\uBF28", "\uBF28", "\u1108\u1168\u11AB", "\uBF28", "\u1108\u1168\u11AB" },
            new string [] { "\uBF29", "\uBF29", "\u1108\u1168\u11AC", "\uBF29", "\u1108\u1168\u11AC" },
            new string [] { "\uBF2A", "\uBF2A", "\u1108\u1168\u11AD", "\uBF2A", "\u1108\u1168\u11AD" },
            new string [] { "\uBF2B", "\uBF2B", "\u1108\u1168\u11AE", "\uBF2B", "\u1108\u1168\u11AE" },
            new string [] { "\uBF2C", "\uBF2C", "\u1108\u1168\u11AF", "\uBF2C", "\u1108\u1168\u11AF" },
            new string [] { "\uBF2D", "\uBF2D", "\u1108\u1168\u11B0", "\uBF2D", "\u1108\u1168\u11B0" },
            new string [] { "\uBF2E", "\uBF2E", "\u1108\u1168\u11B1", "\uBF2E", "\u1108\u1168\u11B1" },
            new string [] { "\uBF2F", "\uBF2F", "\u1108\u1168\u11B2", "\uBF2F", "\u1108\u1168\u11B2" },
            new string [] { "\uBF30", "\uBF30", "\u1108\u1168\u11B3", "\uBF30", "\u1108\u1168\u11B3" },
            new string [] { "\uBF31", "\uBF31", "\u1108\u1168\u11B4", "\uBF31", "\u1108\u1168\u11B4" },
            new string [] { "\uBF32", "\uBF32", "\u1108\u1168\u11B5", "\uBF32", "\u1108\u1168\u11B5" },
            new string [] { "\uBF33", "\uBF33", "\u1108\u1168\u11B6", "\uBF33", "\u1108\u1168\u11B6" },
            new string [] { "\uBF34", "\uBF34", "\u1108\u1168\u11B7", "\uBF34", "\u1108\u1168\u11B7" },
            new string [] { "\uBF35", "\uBF35", "\u1108\u1168\u11B8", "\uBF35", "\u1108\u1168\u11B8" },
            new string [] { "\uBF36", "\uBF36", "\u1108\u1168\u11B9", "\uBF36", "\u1108\u1168\u11B9" },
            new string [] { "\uBF37", "\uBF37", "\u1108\u1168\u11BA", "\uBF37", "\u1108\u1168\u11BA" },
            new string [] { "\uBF38", "\uBF38", "\u1108\u1168\u11BB", "\uBF38", "\u1108\u1168\u11BB" },
            new string [] { "\uBF39", "\uBF39", "\u1108\u1168\u11BC", "\uBF39", "\u1108\u1168\u11BC" },
            new string [] { "\uBF3A", "\uBF3A", "\u1108\u1168\u11BD", "\uBF3A", "\u1108\u1168\u11BD" },
            new string [] { "\uBF3B", "\uBF3B", "\u1108\u1168\u11BE", "\uBF3B", "\u1108\u1168\u11BE" },
            new string [] { "\uBF3C", "\uBF3C", "\u1108\u1168\u11BF", "\uBF3C", "\u1108\u1168\u11BF" },
            new string [] { "\uBF3D", "\uBF3D", "\u1108\u1168\u11C0", "\uBF3D", "\u1108\u1168\u11C0" },
            new string [] { "\uBF3E", "\uBF3E", "\u1108\u1168\u11C1", "\uBF3E", "\u1108\u1168\u11C1" },
            new string [] { "\uBF3F", "\uBF3F", "\u1108\u1168\u11C2", "\uBF3F", "\u1108\u1168\u11C2" },
            new string [] { "\uBF40", "\uBF40", "\u1108\u1169", "\uBF40", "\u1108\u1169" },
            new string [] { "\uBF41", "\uBF41", "\u1108\u1169\u11A8", "\uBF41", "\u1108\u1169\u11A8" },
            new string [] { "\uBF42", "\uBF42", "\u1108\u1169\u11A9", "\uBF42", "\u1108\u1169\u11A9" },
            new string [] { "\uBF43", "\uBF43", "\u1108\u1169\u11AA", "\uBF43", "\u1108\u1169\u11AA" },
            new string [] { "\uBF44", "\uBF44", "\u1108\u1169\u11AB", "\uBF44", "\u1108\u1169\u11AB" },
            new string [] { "\uBF45", "\uBF45", "\u1108\u1169\u11AC", "\uBF45", "\u1108\u1169\u11AC" },
            new string [] { "\uBF46", "\uBF46", "\u1108\u1169\u11AD", "\uBF46", "\u1108\u1169\u11AD" },
            new string [] { "\uBF47", "\uBF47", "\u1108\u1169\u11AE", "\uBF47", "\u1108\u1169\u11AE" },
            new string [] { "\uBF48", "\uBF48", "\u1108\u1169\u11AF", "\uBF48", "\u1108\u1169\u11AF" },
            new string [] { "\uBF49", "\uBF49", "\u1108\u1169\u11B0", "\uBF49", "\u1108\u1169\u11B0" },
            new string [] { "\uBF4A", "\uBF4A", "\u1108\u1169\u11B1", "\uBF4A", "\u1108\u1169\u11B1" },
            new string [] { "\uBF4B", "\uBF4B", "\u1108\u1169\u11B2", "\uBF4B", "\u1108\u1169\u11B2" },
            new string [] { "\uBF4C", "\uBF4C", "\u1108\u1169\u11B3", "\uBF4C", "\u1108\u1169\u11B3" },
            new string [] { "\uBF4D", "\uBF4D", "\u1108\u1169\u11B4", "\uBF4D", "\u1108\u1169\u11B4" },
            new string [] { "\uBF4E", "\uBF4E", "\u1108\u1169\u11B5", "\uBF4E", "\u1108\u1169\u11B5" },
            new string [] { "\uBF4F", "\uBF4F", "\u1108\u1169\u11B6", "\uBF4F", "\u1108\u1169\u11B6" },
            new string [] { "\uBF50", "\uBF50", "\u1108\u1169\u11B7", "\uBF50", "\u1108\u1169\u11B7" },
            new string [] { "\uBF51", "\uBF51", "\u1108\u1169\u11B8", "\uBF51", "\u1108\u1169\u11B8" },
            new string [] { "\uBF52", "\uBF52", "\u1108\u1169\u11B9", "\uBF52", "\u1108\u1169\u11B9" },
            new string [] { "\uBF53", "\uBF53", "\u1108\u1169\u11BA", "\uBF53", "\u1108\u1169\u11BA" },
            new string [] { "\uBF54", "\uBF54", "\u1108\u1169\u11BB", "\uBF54", "\u1108\u1169\u11BB" },
            new string [] { "\uBF55", "\uBF55", "\u1108\u1169\u11BC", "\uBF55", "\u1108\u1169\u11BC" },
            new string [] { "\uBF56", "\uBF56", "\u1108\u1169\u11BD", "\uBF56", "\u1108\u1169\u11BD" },
            new string [] { "\uBF57", "\uBF57", "\u1108\u1169\u11BE", "\uBF57", "\u1108\u1169\u11BE" },
            new string [] { "\uBF58", "\uBF58", "\u1108\u1169\u11BF", "\uBF58", "\u1108\u1169\u11BF" },
            new string [] { "\uBF59", "\uBF59", "\u1108\u1169\u11C0", "\uBF59", "\u1108\u1169\u11C0" },
            new string [] { "\uBF5A", "\uBF5A", "\u1108\u1169\u11C1", "\uBF5A", "\u1108\u1169\u11C1" },
            new string [] { "\uBF5B", "\uBF5B", "\u1108\u1169\u11C2", "\uBF5B", "\u1108\u1169\u11C2" },
            new string [] { "\uBF5C", "\uBF5C", "\u1108\u116A", "\uBF5C", "\u1108\u116A" },
            new string [] { "\uBF5D", "\uBF5D", "\u1108\u116A\u11A8", "\uBF5D", "\u1108\u116A\u11A8" },
            new string [] { "\uBF5E", "\uBF5E", "\u1108\u116A\u11A9", "\uBF5E", "\u1108\u116A\u11A9" },
            new string [] { "\uBF5F", "\uBF5F", "\u1108\u116A\u11AA", "\uBF5F", "\u1108\u116A\u11AA" },
            new string [] { "\uBF60", "\uBF60", "\u1108\u116A\u11AB", "\uBF60", "\u1108\u116A\u11AB" },
            new string [] { "\uBF61", "\uBF61", "\u1108\u116A\u11AC", "\uBF61", "\u1108\u116A\u11AC" },
            new string [] { "\uBF62", "\uBF62", "\u1108\u116A\u11AD", "\uBF62", "\u1108\u116A\u11AD" },
            new string [] { "\uBF63", "\uBF63", "\u1108\u116A\u11AE", "\uBF63", "\u1108\u116A\u11AE" },
            new string [] { "\uBF64", "\uBF64", "\u1108\u116A\u11AF", "\uBF64", "\u1108\u116A\u11AF" },
            new string [] { "\uBF65", "\uBF65", "\u1108\u116A\u11B0", "\uBF65", "\u1108\u116A\u11B0" },
            new string [] { "\uBF66", "\uBF66", "\u1108\u116A\u11B1", "\uBF66", "\u1108\u116A\u11B1" },
            new string [] { "\uBF67", "\uBF67", "\u1108\u116A\u11B2", "\uBF67", "\u1108\u116A\u11B2" },
            new string [] { "\uBF68", "\uBF68", "\u1108\u116A\u11B3", "\uBF68", "\u1108\u116A\u11B3" },
            new string [] { "\uBF69", "\uBF69", "\u1108\u116A\u11B4", "\uBF69", "\u1108\u116A\u11B4" },
            new string [] { "\uBF6A", "\uBF6A", "\u1108\u116A\u11B5", "\uBF6A", "\u1108\u116A\u11B5" },
            new string [] { "\uBF6B", "\uBF6B", "\u1108\u116A\u11B6", "\uBF6B", "\u1108\u116A\u11B6" },
            new string [] { "\uBF6C", "\uBF6C", "\u1108\u116A\u11B7", "\uBF6C", "\u1108\u116A\u11B7" },
            new string [] { "\uBF6D", "\uBF6D", "\u1108\u116A\u11B8", "\uBF6D", "\u1108\u116A\u11B8" },
            new string [] { "\uBF6E", "\uBF6E", "\u1108\u116A\u11B9", "\uBF6E", "\u1108\u116A\u11B9" },
            new string [] { "\uBF6F", "\uBF6F", "\u1108\u116A\u11BA", "\uBF6F", "\u1108\u116A\u11BA" },
            new string [] { "\uBF70", "\uBF70", "\u1108\u116A\u11BB", "\uBF70", "\u1108\u116A\u11BB" },
            new string [] { "\uBF71", "\uBF71", "\u1108\u116A\u11BC", "\uBF71", "\u1108\u116A\u11BC" },
            new string [] { "\uBF72", "\uBF72", "\u1108\u116A\u11BD", "\uBF72", "\u1108\u116A\u11BD" },
            new string [] { "\uBF73", "\uBF73", "\u1108\u116A\u11BE", "\uBF73", "\u1108\u116A\u11BE" },
            new string [] { "\uBF74", "\uBF74", "\u1108\u116A\u11BF", "\uBF74", "\u1108\u116A\u11BF" },
            new string [] { "\uBF75", "\uBF75", "\u1108\u116A\u11C0", "\uBF75", "\u1108\u116A\u11C0" },
            new string [] { "\uBF76", "\uBF76", "\u1108\u116A\u11C1", "\uBF76", "\u1108\u116A\u11C1" },
            new string [] { "\uBF77", "\uBF77", "\u1108\u116A\u11C2", "\uBF77", "\u1108\u116A\u11C2" },
            new string [] { "\uBF78", "\uBF78", "\u1108\u116B", "\uBF78", "\u1108\u116B" },
            new string [] { "\uBF79", "\uBF79", "\u1108\u116B\u11A8", "\uBF79", "\u1108\u116B\u11A8" },
            new string [] { "\uBF7A", "\uBF7A", "\u1108\u116B\u11A9", "\uBF7A", "\u1108\u116B\u11A9" },
            new string [] { "\uBF7B", "\uBF7B", "\u1108\u116B\u11AA", "\uBF7B", "\u1108\u116B\u11AA" },
            new string [] { "\uBF7C", "\uBF7C", "\u1108\u116B\u11AB", "\uBF7C", "\u1108\u116B\u11AB" },
            new string [] { "\uBF7D", "\uBF7D", "\u1108\u116B\u11AC", "\uBF7D", "\u1108\u116B\u11AC" },
            new string [] { "\uBF7E", "\uBF7E", "\u1108\u116B\u11AD", "\uBF7E", "\u1108\u116B\u11AD" },
            new string [] { "\uBF7F", "\uBF7F", "\u1108\u116B\u11AE", "\uBF7F", "\u1108\u116B\u11AE" },
            new string [] { "\uBF80", "\uBF80", "\u1108\u116B\u11AF", "\uBF80", "\u1108\u116B\u11AF" },
            new string [] { "\uBF81", "\uBF81", "\u1108\u116B\u11B0", "\uBF81", "\u1108\u116B\u11B0" },
            new string [] { "\uBF82", "\uBF82", "\u1108\u116B\u11B1", "\uBF82", "\u1108\u116B\u11B1" },
            new string [] { "\uBF83", "\uBF83", "\u1108\u116B\u11B2", "\uBF83", "\u1108\u116B\u11B2" },
            new string [] { "\uBF84", "\uBF84", "\u1108\u116B\u11B3", "\uBF84", "\u1108\u116B\u11B3" },
            new string [] { "\uBF85", "\uBF85", "\u1108\u116B\u11B4", "\uBF85", "\u1108\u116B\u11B4" },
            new string [] { "\uBF86", "\uBF86", "\u1108\u116B\u11B5", "\uBF86", "\u1108\u116B\u11B5" },
            new string [] { "\uBF87", "\uBF87", "\u1108\u116B\u11B6", "\uBF87", "\u1108\u116B\u11B6" },
            new string [] { "\uBF88", "\uBF88", "\u1108\u116B\u11B7", "\uBF88", "\u1108\u116B\u11B7" },
            new string [] { "\uBF89", "\uBF89", "\u1108\u116B\u11B8", "\uBF89", "\u1108\u116B\u11B8" },
            new string [] { "\uBF8A", "\uBF8A", "\u1108\u116B\u11B9", "\uBF8A", "\u1108\u116B\u11B9" },
            new string [] { "\uBF8B", "\uBF8B", "\u1108\u116B\u11BA", "\uBF8B", "\u1108\u116B\u11BA" },
            new string [] { "\uBF8C", "\uBF8C", "\u1108\u116B\u11BB", "\uBF8C", "\u1108\u116B\u11BB" },
            new string [] { "\uBF8D", "\uBF8D", "\u1108\u116B\u11BC", "\uBF8D", "\u1108\u116B\u11BC" },
            new string [] { "\uBF8E", "\uBF8E", "\u1108\u116B\u11BD", "\uBF8E", "\u1108\u116B\u11BD" },
            new string [] { "\uBF8F", "\uBF8F", "\u1108\u116B\u11BE", "\uBF8F", "\u1108\u116B\u11BE" },
            new string [] { "\uBF90", "\uBF90", "\u1108\u116B\u11BF", "\uBF90", "\u1108\u116B\u11BF" },
            new string [] { "\uBF91", "\uBF91", "\u1108\u116B\u11C0", "\uBF91", "\u1108\u116B\u11C0" },
            new string [] { "\uBF92", "\uBF92", "\u1108\u116B\u11C1", "\uBF92", "\u1108\u116B\u11C1" },
            new string [] { "\uBF93", "\uBF93", "\u1108\u116B\u11C2", "\uBF93", "\u1108\u116B\u11C2" },
            new string [] { "\uBF94", "\uBF94", "\u1108\u116C", "\uBF94", "\u1108\u116C" },
            new string [] { "\uBF95", "\uBF95", "\u1108\u116C\u11A8", "\uBF95", "\u1108\u116C\u11A8" },
            new string [] { "\uBF96", "\uBF96", "\u1108\u116C\u11A9", "\uBF96", "\u1108\u116C\u11A9" },
            new string [] { "\uBF97", "\uBF97", "\u1108\u116C\u11AA", "\uBF97", "\u1108\u116C\u11AA" },
            new string [] { "\uBF98", "\uBF98", "\u1108\u116C\u11AB", "\uBF98", "\u1108\u116C\u11AB" },
            new string [] { "\uBF99", "\uBF99", "\u1108\u116C\u11AC", "\uBF99", "\u1108\u116C\u11AC" },
            new string [] { "\uBF9A", "\uBF9A", "\u1108\u116C\u11AD", "\uBF9A", "\u1108\u116C\u11AD" },
            new string [] { "\uBF9B", "\uBF9B", "\u1108\u116C\u11AE", "\uBF9B", "\u1108\u116C\u11AE" },
            new string [] { "\uBF9C", "\uBF9C", "\u1108\u116C\u11AF", "\uBF9C", "\u1108\u116C\u11AF" },
            new string [] { "\uBF9D", "\uBF9D", "\u1108\u116C\u11B0", "\uBF9D", "\u1108\u116C\u11B0" },
            new string [] { "\uBF9E", "\uBF9E", "\u1108\u116C\u11B1", "\uBF9E", "\u1108\u116C\u11B1" },
            new string [] { "\uBF9F", "\uBF9F", "\u1108\u116C\u11B2", "\uBF9F", "\u1108\u116C\u11B2" },
            new string [] { "\uBFA0", "\uBFA0", "\u1108\u116C\u11B3", "\uBFA0", "\u1108\u116C\u11B3" },
            new string [] { "\uBFA1", "\uBFA1", "\u1108\u116C\u11B4", "\uBFA1", "\u1108\u116C\u11B4" },
            new string [] { "\uBFA2", "\uBFA2", "\u1108\u116C\u11B5", "\uBFA2", "\u1108\u116C\u11B5" },
            new string [] { "\uBFA3", "\uBFA3", "\u1108\u116C\u11B6", "\uBFA3", "\u1108\u116C\u11B6" },
            new string [] { "\uBFA4", "\uBFA4", "\u1108\u116C\u11B7", "\uBFA4", "\u1108\u116C\u11B7" },
            new string [] { "\uBFA5", "\uBFA5", "\u1108\u116C\u11B8", "\uBFA5", "\u1108\u116C\u11B8" },
            new string [] { "\uBFA6", "\uBFA6", "\u1108\u116C\u11B9", "\uBFA6", "\u1108\u116C\u11B9" },
            new string [] { "\uBFA7", "\uBFA7", "\u1108\u116C\u11BA", "\uBFA7", "\u1108\u116C\u11BA" },
            new string [] { "\uBFA8", "\uBFA8", "\u1108\u116C\u11BB", "\uBFA8", "\u1108\u116C\u11BB" },
            new string [] { "\uBFA9", "\uBFA9", "\u1108\u116C\u11BC", "\uBFA9", "\u1108\u116C\u11BC" },
            new string [] { "\uBFAA", "\uBFAA", "\u1108\u116C\u11BD", "\uBFAA", "\u1108\u116C\u11BD" },
            new string [] { "\uBFAB", "\uBFAB", "\u1108\u116C\u11BE", "\uBFAB", "\u1108\u116C\u11BE" },
            new string [] { "\uBFAC", "\uBFAC", "\u1108\u116C\u11BF", "\uBFAC", "\u1108\u116C\u11BF" },
            new string [] { "\uBFAD", "\uBFAD", "\u1108\u116C\u11C0", "\uBFAD", "\u1108\u116C\u11C0" },
            new string [] { "\uBFAE", "\uBFAE", "\u1108\u116C\u11C1", "\uBFAE", "\u1108\u116C\u11C1" },
            new string [] { "\uBFAF", "\uBFAF", "\u1108\u116C\u11C2", "\uBFAF", "\u1108\u116C\u11C2" },
            new string [] { "\uBFB0", "\uBFB0", "\u1108\u116D", "\uBFB0", "\u1108\u116D" },
            new string [] { "\uBFB1", "\uBFB1", "\u1108\u116D\u11A8", "\uBFB1", "\u1108\u116D\u11A8" },
            new string [] { "\uBFB2", "\uBFB2", "\u1108\u116D\u11A9", "\uBFB2", "\u1108\u116D\u11A9" },
            new string [] { "\uBFB3", "\uBFB3", "\u1108\u116D\u11AA", "\uBFB3", "\u1108\u116D\u11AA" },
            new string [] { "\uBFB4", "\uBFB4", "\u1108\u116D\u11AB", "\uBFB4", "\u1108\u116D\u11AB" },
            new string [] { "\uBFB5", "\uBFB5", "\u1108\u116D\u11AC", "\uBFB5", "\u1108\u116D\u11AC" },
            new string [] { "\uBFB6", "\uBFB6", "\u1108\u116D\u11AD", "\uBFB6", "\u1108\u116D\u11AD" },
            new string [] { "\uBFB7", "\uBFB7", "\u1108\u116D\u11AE", "\uBFB7", "\u1108\u116D\u11AE" },
            new string [] { "\uBFB8", "\uBFB8", "\u1108\u116D\u11AF", "\uBFB8", "\u1108\u116D\u11AF" },
            new string [] { "\uBFB9", "\uBFB9", "\u1108\u116D\u11B0", "\uBFB9", "\u1108\u116D\u11B0" },
            new string [] { "\uBFBA", "\uBFBA", "\u1108\u116D\u11B1", "\uBFBA", "\u1108\u116D\u11B1" },
            new string [] { "\uBFBB", "\uBFBB", "\u1108\u116D\u11B2", "\uBFBB", "\u1108\u116D\u11B2" },
            new string [] { "\uBFBC", "\uBFBC", "\u1108\u116D\u11B3", "\uBFBC", "\u1108\u116D\u11B3" },
            new string [] { "\uBFBD", "\uBFBD", "\u1108\u116D\u11B4", "\uBFBD", "\u1108\u116D\u11B4" },
            new string [] { "\uBFBE", "\uBFBE", "\u1108\u116D\u11B5", "\uBFBE", "\u1108\u116D\u11B5" },
            new string [] { "\uBFBF", "\uBFBF", "\u1108\u116D\u11B6", "\uBFBF", "\u1108\u116D\u11B6" },
            new string [] { "\uBFC0", "\uBFC0", "\u1108\u116D\u11B7", "\uBFC0", "\u1108\u116D\u11B7" },
            new string [] { "\uBFC1", "\uBFC1", "\u1108\u116D\u11B8", "\uBFC1", "\u1108\u116D\u11B8" },
            new string [] { "\uBFC2", "\uBFC2", "\u1108\u116D\u11B9", "\uBFC2", "\u1108\u116D\u11B9" },
            new string [] { "\uBFC3", "\uBFC3", "\u1108\u116D\u11BA", "\uBFC3", "\u1108\u116D\u11BA" },
            new string [] { "\uBFC4", "\uBFC4", "\u1108\u116D\u11BB", "\uBFC4", "\u1108\u116D\u11BB" },
            new string [] { "\uBFC5", "\uBFC5", "\u1108\u116D\u11BC", "\uBFC5", "\u1108\u116D\u11BC" },
            new string [] { "\uBFC6", "\uBFC6", "\u1108\u116D\u11BD", "\uBFC6", "\u1108\u116D\u11BD" },
            new string [] { "\uBFC7", "\uBFC7", "\u1108\u116D\u11BE", "\uBFC7", "\u1108\u116D\u11BE" },
            new string [] { "\uBFC8", "\uBFC8", "\u1108\u116D\u11BF", "\uBFC8", "\u1108\u116D\u11BF" },
            new string [] { "\uBFC9", "\uBFC9", "\u1108\u116D\u11C0", "\uBFC9", "\u1108\u116D\u11C0" },
            new string [] { "\uBFCA", "\uBFCA", "\u1108\u116D\u11C1", "\uBFCA", "\u1108\u116D\u11C1" },
            new string [] { "\uBFCB", "\uBFCB", "\u1108\u116D\u11C2", "\uBFCB", "\u1108\u116D\u11C2" },
            new string [] { "\uBFCC", "\uBFCC", "\u1108\u116E", "\uBFCC", "\u1108\u116E" },
            new string [] { "\uBFCD", "\uBFCD", "\u1108\u116E\u11A8", "\uBFCD", "\u1108\u116E\u11A8" },
            new string [] { "\uBFCE", "\uBFCE", "\u1108\u116E\u11A9", "\uBFCE", "\u1108\u116E\u11A9" },
            new string [] { "\uBFCF", "\uBFCF", "\u1108\u116E\u11AA", "\uBFCF", "\u1108\u116E\u11AA" },
            new string [] { "\uBFD0", "\uBFD0", "\u1108\u116E\u11AB", "\uBFD0", "\u1108\u116E\u11AB" },
            new string [] { "\uBFD1", "\uBFD1", "\u1108\u116E\u11AC", "\uBFD1", "\u1108\u116E\u11AC" },
            new string [] { "\uBFD2", "\uBFD2", "\u1108\u116E\u11AD", "\uBFD2", "\u1108\u116E\u11AD" },
            new string [] { "\uBFD3", "\uBFD3", "\u1108\u116E\u11AE", "\uBFD3", "\u1108\u116E\u11AE" },
            new string [] { "\uBFD4", "\uBFD4", "\u1108\u116E\u11AF", "\uBFD4", "\u1108\u116E\u11AF" },
            new string [] { "\uBFD5", "\uBFD5", "\u1108\u116E\u11B0", "\uBFD5", "\u1108\u116E\u11B0" },
            new string [] { "\uBFD6", "\uBFD6", "\u1108\u116E\u11B1", "\uBFD6", "\u1108\u116E\u11B1" },
            new string [] { "\uBFD7", "\uBFD7", "\u1108\u116E\u11B2", "\uBFD7", "\u1108\u116E\u11B2" },
            new string [] { "\uBFD8", "\uBFD8", "\u1108\u116E\u11B3", "\uBFD8", "\u1108\u116E\u11B3" },
            new string [] { "\uBFD9", "\uBFD9", "\u1108\u116E\u11B4", "\uBFD9", "\u1108\u116E\u11B4" },
            new string [] { "\uBFDA", "\uBFDA", "\u1108\u116E\u11B5", "\uBFDA", "\u1108\u116E\u11B5" },
            new string [] { "\uBFDB", "\uBFDB", "\u1108\u116E\u11B6", "\uBFDB", "\u1108\u116E\u11B6" },
            new string [] { "\uBFDC", "\uBFDC", "\u1108\u116E\u11B7", "\uBFDC", "\u1108\u116E\u11B7" },
            new string [] { "\uBFDD", "\uBFDD", "\u1108\u116E\u11B8", "\uBFDD", "\u1108\u116E\u11B8" },
            new string [] { "\uBFDE", "\uBFDE", "\u1108\u116E\u11B9", "\uBFDE", "\u1108\u116E\u11B9" },
            new string [] { "\uBFDF", "\uBFDF", "\u1108\u116E\u11BA", "\uBFDF", "\u1108\u116E\u11BA" },
            new string [] { "\uBFE0", "\uBFE0", "\u1108\u116E\u11BB", "\uBFE0", "\u1108\u116E\u11BB" },
            new string [] { "\uBFE1", "\uBFE1", "\u1108\u116E\u11BC", "\uBFE1", "\u1108\u116E\u11BC" },
            new string [] { "\uBFE2", "\uBFE2", "\u1108\u116E\u11BD", "\uBFE2", "\u1108\u116E\u11BD" },
            new string [] { "\uBFE3", "\uBFE3", "\u1108\u116E\u11BE", "\uBFE3", "\u1108\u116E\u11BE" },
            new string [] { "\uBFE4", "\uBFE4", "\u1108\u116E\u11BF", "\uBFE4", "\u1108\u116E\u11BF" },
            new string [] { "\uBFE5", "\uBFE5", "\u1108\u116E\u11C0", "\uBFE5", "\u1108\u116E\u11C0" },
            new string [] { "\uBFE6", "\uBFE6", "\u1108\u116E\u11C1", "\uBFE6", "\u1108\u116E\u11C1" },
            new string [] { "\uBFE7", "\uBFE7", "\u1108\u116E\u11C2", "\uBFE7", "\u1108\u116E\u11C2" },
            new string [] { "\uBFE8", "\uBFE8", "\u1108\u116F", "\uBFE8", "\u1108\u116F" },
            new string [] { "\uBFE9", "\uBFE9", "\u1108\u116F\u11A8", "\uBFE9", "\u1108\u116F\u11A8" },
            new string [] { "\uBFEA", "\uBFEA", "\u1108\u116F\u11A9", "\uBFEA", "\u1108\u116F\u11A9" },
            new string [] { "\uBFEB", "\uBFEB", "\u1108\u116F\u11AA", "\uBFEB", "\u1108\u116F\u11AA" },
            new string [] { "\uBFEC", "\uBFEC", "\u1108\u116F\u11AB", "\uBFEC", "\u1108\u116F\u11AB" },
            new string [] { "\uBFED", "\uBFED", "\u1108\u116F\u11AC", "\uBFED", "\u1108\u116F\u11AC" },
            new string [] { "\uBFEE", "\uBFEE", "\u1108\u116F\u11AD", "\uBFEE", "\u1108\u116F\u11AD" },
            new string [] { "\uBFEF", "\uBFEF", "\u1108\u116F\u11AE", "\uBFEF", "\u1108\u116F\u11AE" },
            new string [] { "\uBFF0", "\uBFF0", "\u1108\u116F\u11AF", "\uBFF0", "\u1108\u116F\u11AF" },
            new string [] { "\uBFF1", "\uBFF1", "\u1108\u116F\u11B0", "\uBFF1", "\u1108\u116F\u11B0" },
            new string [] { "\uBFF2", "\uBFF2", "\u1108\u116F\u11B1", "\uBFF2", "\u1108\u116F\u11B1" },
            new string [] { "\uBFF3", "\uBFF3", "\u1108\u116F\u11B2", "\uBFF3", "\u1108\u116F\u11B2" },
            new string [] { "\uBFF4", "\uBFF4", "\u1108\u116F\u11B3", "\uBFF4", "\u1108\u116F\u11B3" },
            new string [] { "\uBFF5", "\uBFF5", "\u1108\u116F\u11B4", "\uBFF5", "\u1108\u116F\u11B4" },
            new string [] { "\uBFF6", "\uBFF6", "\u1108\u116F\u11B5", "\uBFF6", "\u1108\u116F\u11B5" },
            new string [] { "\uBFF7", "\uBFF7", "\u1108\u116F\u11B6", "\uBFF7", "\u1108\u116F\u11B6" },
            new string [] { "\uBFF8", "\uBFF8", "\u1108\u116F\u11B7", "\uBFF8", "\u1108\u116F\u11B7" },
            new string [] { "\uBFF9", "\uBFF9", "\u1108\u116F\u11B8", "\uBFF9", "\u1108\u116F\u11B8" },
            new string [] { "\uBFFA", "\uBFFA", "\u1108\u116F\u11B9", "\uBFFA", "\u1108\u116F\u11B9" },
            new string [] { "\uBFFB", "\uBFFB", "\u1108\u116F\u11BA", "\uBFFB", "\u1108\u116F\u11BA" },
            new string [] { "\uBFFC", "\uBFFC", "\u1108\u116F\u11BB", "\uBFFC", "\u1108\u116F\u11BB" },
            new string [] { "\uBFFD", "\uBFFD", "\u1108\u116F\u11BC", "\uBFFD", "\u1108\u116F\u11BC" },
            new string [] { "\uBFFE", "\uBFFE", "\u1108\u116F\u11BD", "\uBFFE", "\u1108\u116F\u11BD" },
            new string [] { "\uBFFF", "\uBFFF", "\u1108\u116F\u11BE", "\uBFFF", "\u1108\u116F\u11BE" },
            new string [] { "\uC000", "\uC000", "\u1108\u116F\u11BF", "\uC000", "\u1108\u116F\u11BF" },
            new string [] { "\uC001", "\uC001", "\u1108\u116F\u11C0", "\uC001", "\u1108\u116F\u11C0" },
            new string [] { "\uC002", "\uC002", "\u1108\u116F\u11C1", "\uC002", "\u1108\u116F\u11C1" },
            new string [] { "\uC003", "\uC003", "\u1108\u116F\u11C2", "\uC003", "\u1108\u116F\u11C2" },
            new string [] { "\uC004", "\uC004", "\u1108\u1170", "\uC004", "\u1108\u1170" },
            new string [] { "\uC005", "\uC005", "\u1108\u1170\u11A8", "\uC005", "\u1108\u1170\u11A8" },
            new string [] { "\uC006", "\uC006", "\u1108\u1170\u11A9", "\uC006", "\u1108\u1170\u11A9" },
            new string [] { "\uC007", "\uC007", "\u1108\u1170\u11AA", "\uC007", "\u1108\u1170\u11AA" },
            new string [] { "\uC008", "\uC008", "\u1108\u1170\u11AB", "\uC008", "\u1108\u1170\u11AB" },
            new string [] { "\uC009", "\uC009", "\u1108\u1170\u11AC", "\uC009", "\u1108\u1170\u11AC" },
            new string [] { "\uC00A", "\uC00A", "\u1108\u1170\u11AD", "\uC00A", "\u1108\u1170\u11AD" },
            new string [] { "\uC00B", "\uC00B", "\u1108\u1170\u11AE", "\uC00B", "\u1108\u1170\u11AE" },
            new string [] { "\uC00C", "\uC00C", "\u1108\u1170\u11AF", "\uC00C", "\u1108\u1170\u11AF" },
            new string [] { "\uC00D", "\uC00D", "\u1108\u1170\u11B0", "\uC00D", "\u1108\u1170\u11B0" },
            new string [] { "\uC00E", "\uC00E", "\u1108\u1170\u11B1", "\uC00E", "\u1108\u1170\u11B1" },
            new string [] { "\uC00F", "\uC00F", "\u1108\u1170\u11B2", "\uC00F", "\u1108\u1170\u11B2" },
            new string [] { "\uC010", "\uC010", "\u1108\u1170\u11B3", "\uC010", "\u1108\u1170\u11B3" },
            new string [] { "\uC011", "\uC011", "\u1108\u1170\u11B4", "\uC011", "\u1108\u1170\u11B4" },
            new string [] { "\uC012", "\uC012", "\u1108\u1170\u11B5", "\uC012", "\u1108\u1170\u11B5" },
            new string [] { "\uC013", "\uC013", "\u1108\u1170\u11B6", "\uC013", "\u1108\u1170\u11B6" },
            new string [] { "\uC014", "\uC014", "\u1108\u1170\u11B7", "\uC014", "\u1108\u1170\u11B7" },
            new string [] { "\uC015", "\uC015", "\u1108\u1170\u11B8", "\uC015", "\u1108\u1170\u11B8" },
            new string [] { "\uC016", "\uC016", "\u1108\u1170\u11B9", "\uC016", "\u1108\u1170\u11B9" },
            new string [] { "\uC017", "\uC017", "\u1108\u1170\u11BA", "\uC017", "\u1108\u1170\u11BA" },
            new string [] { "\uC018", "\uC018", "\u1108\u1170\u11BB", "\uC018", "\u1108\u1170\u11BB" },
            new string [] { "\uC019", "\uC019", "\u1108\u1170\u11BC", "\uC019", "\u1108\u1170\u11BC" },
            new string [] { "\uC01A", "\uC01A", "\u1108\u1170\u11BD", "\uC01A", "\u1108\u1170\u11BD" },
            new string [] { "\uC01B", "\uC01B", "\u1108\u1170\u11BE", "\uC01B", "\u1108\u1170\u11BE" },
            new string [] { "\uC01C", "\uC01C", "\u1108\u1170\u11BF", "\uC01C", "\u1108\u1170\u11BF" },
            new string [] { "\uC01D", "\uC01D", "\u1108\u1170\u11C0", "\uC01D", "\u1108\u1170\u11C0" },
            new string [] { "\uC01E", "\uC01E", "\u1108\u1170\u11C1", "\uC01E", "\u1108\u1170\u11C1" },
            new string [] { "\uC01F", "\uC01F", "\u1108\u1170\u11C2", "\uC01F", "\u1108\u1170\u11C2" },
            new string [] { "\uC020", "\uC020", "\u1108\u1171", "\uC020", "\u1108\u1171" },
            new string [] { "\uC021", "\uC021", "\u1108\u1171\u11A8", "\uC021", "\u1108\u1171\u11A8" },
            new string [] { "\uC022", "\uC022", "\u1108\u1171\u11A9", "\uC022", "\u1108\u1171\u11A9" },
            new string [] { "\uC023", "\uC023", "\u1108\u1171\u11AA", "\uC023", "\u1108\u1171\u11AA" },
            new string [] { "\uC024", "\uC024", "\u1108\u1171\u11AB", "\uC024", "\u1108\u1171\u11AB" },
            new string [] { "\uC025", "\uC025", "\u1108\u1171\u11AC", "\uC025", "\u1108\u1171\u11AC" },
            new string [] { "\uC026", "\uC026", "\u1108\u1171\u11AD", "\uC026", "\u1108\u1171\u11AD" },
            new string [] { "\uC027", "\uC027", "\u1108\u1171\u11AE", "\uC027", "\u1108\u1171\u11AE" },
            new string [] { "\uC028", "\uC028", "\u1108\u1171\u11AF", "\uC028", "\u1108\u1171\u11AF" },
            new string [] { "\uC029", "\uC029", "\u1108\u1171\u11B0", "\uC029", "\u1108\u1171\u11B0" },
            new string [] { "\uC02A", "\uC02A", "\u1108\u1171\u11B1", "\uC02A", "\u1108\u1171\u11B1" },
            new string [] { "\uC02B", "\uC02B", "\u1108\u1171\u11B2", "\uC02B", "\u1108\u1171\u11B2" },
            new string [] { "\uC02C", "\uC02C", "\u1108\u1171\u11B3", "\uC02C", "\u1108\u1171\u11B3" },
            new string [] { "\uC02D", "\uC02D", "\u1108\u1171\u11B4", "\uC02D", "\u1108\u1171\u11B4" },
            new string [] { "\uC02E", "\uC02E", "\u1108\u1171\u11B5", "\uC02E", "\u1108\u1171\u11B5" },
            new string [] { "\uC02F", "\uC02F", "\u1108\u1171\u11B6", "\uC02F", "\u1108\u1171\u11B6" },
            new string [] { "\uC030", "\uC030", "\u1108\u1171\u11B7", "\uC030", "\u1108\u1171\u11B7" },
            new string [] { "\uC031", "\uC031", "\u1108\u1171\u11B8", "\uC031", "\u1108\u1171\u11B8" },
            new string [] { "\uC032", "\uC032", "\u1108\u1171\u11B9", "\uC032", "\u1108\u1171\u11B9" },
            new string [] { "\uC033", "\uC033", "\u1108\u1171\u11BA", "\uC033", "\u1108\u1171\u11BA" },
            new string [] { "\uC034", "\uC034", "\u1108\u1171\u11BB", "\uC034", "\u1108\u1171\u11BB" },
            new string [] { "\uC035", "\uC035", "\u1108\u1171\u11BC", "\uC035", "\u1108\u1171\u11BC" },
            new string [] { "\uC036", "\uC036", "\u1108\u1171\u11BD", "\uC036", "\u1108\u1171\u11BD" },
            new string [] { "\uC037", "\uC037", "\u1108\u1171\u11BE", "\uC037", "\u1108\u1171\u11BE" },
            new string [] { "\uC038", "\uC038", "\u1108\u1171\u11BF", "\uC038", "\u1108\u1171\u11BF" },
            new string [] { "\uC039", "\uC039", "\u1108\u1171\u11C0", "\uC039", "\u1108\u1171\u11C0" },
            new string [] { "\uC03A", "\uC03A", "\u1108\u1171\u11C1", "\uC03A", "\u1108\u1171\u11C1" },
            new string [] { "\uC03B", "\uC03B", "\u1108\u1171\u11C2", "\uC03B", "\u1108\u1171\u11C2" },
            new string [] { "\uC03C", "\uC03C", "\u1108\u1172", "\uC03C", "\u1108\u1172" },
            new string [] { "\uC03D", "\uC03D", "\u1108\u1172\u11A8", "\uC03D", "\u1108\u1172\u11A8" },
            new string [] { "\uC03E", "\uC03E", "\u1108\u1172\u11A9", "\uC03E", "\u1108\u1172\u11A9" },
            new string [] { "\uC03F", "\uC03F", "\u1108\u1172\u11AA", "\uC03F", "\u1108\u1172\u11AA" },
            new string [] { "\uC040", "\uC040", "\u1108\u1172\u11AB", "\uC040", "\u1108\u1172\u11AB" },
            new string [] { "\uC041", "\uC041", "\u1108\u1172\u11AC", "\uC041", "\u1108\u1172\u11AC" },
            new string [] { "\uC042", "\uC042", "\u1108\u1172\u11AD", "\uC042", "\u1108\u1172\u11AD" },
            new string [] { "\uC043", "\uC043", "\u1108\u1172\u11AE", "\uC043", "\u1108\u1172\u11AE" },
            new string [] { "\uC044", "\uC044", "\u1108\u1172\u11AF", "\uC044", "\u1108\u1172\u11AF" },
            new string [] { "\uC045", "\uC045", "\u1108\u1172\u11B0", "\uC045", "\u1108\u1172\u11B0" },
            new string [] { "\uC046", "\uC046", "\u1108\u1172\u11B1", "\uC046", "\u1108\u1172\u11B1" },
            new string [] { "\uC047", "\uC047", "\u1108\u1172\u11B2", "\uC047", "\u1108\u1172\u11B2" },
            new string [] { "\uC048", "\uC048", "\u1108\u1172\u11B3", "\uC048", "\u1108\u1172\u11B3" },
            new string [] { "\uC049", "\uC049", "\u1108\u1172\u11B4", "\uC049", "\u1108\u1172\u11B4" },
            new string [] { "\uC04A", "\uC04A", "\u1108\u1172\u11B5", "\uC04A", "\u1108\u1172\u11B5" },
            new string [] { "\uC04B", "\uC04B", "\u1108\u1172\u11B6", "\uC04B", "\u1108\u1172\u11B6" },
            new string [] { "\uC04C", "\uC04C", "\u1108\u1172\u11B7", "\uC04C", "\u1108\u1172\u11B7" },
            new string [] { "\uC04D", "\uC04D", "\u1108\u1172\u11B8", "\uC04D", "\u1108\u1172\u11B8" },
            new string [] { "\uC04E", "\uC04E", "\u1108\u1172\u11B9", "\uC04E", "\u1108\u1172\u11B9" },
            new string [] { "\uC04F", "\uC04F", "\u1108\u1172\u11BA", "\uC04F", "\u1108\u1172\u11BA" },
            new string [] { "\uC050", "\uC050", "\u1108\u1172\u11BB", "\uC050", "\u1108\u1172\u11BB" },
            new string [] { "\uC051", "\uC051", "\u1108\u1172\u11BC", "\uC051", "\u1108\u1172\u11BC" },
            new string [] { "\uC052", "\uC052", "\u1108\u1172\u11BD", "\uC052", "\u1108\u1172\u11BD" },
            new string [] { "\uC053", "\uC053", "\u1108\u1172\u11BE", "\uC053", "\u1108\u1172\u11BE" },
            new string [] { "\uC054", "\uC054", "\u1108\u1172\u11BF", "\uC054", "\u1108\u1172\u11BF" },
            new string [] { "\uC055", "\uC055", "\u1108\u1172\u11C0", "\uC055", "\u1108\u1172\u11C0" },
            new string [] { "\uC056", "\uC056", "\u1108\u1172\u11C1", "\uC056", "\u1108\u1172\u11C1" },
            new string [] { "\uC057", "\uC057", "\u1108\u1172\u11C2", "\uC057", "\u1108\u1172\u11C2" },
            new string [] { "\uC058", "\uC058", "\u1108\u1173", "\uC058", "\u1108\u1173" },
            new string [] { "\uC059", "\uC059", "\u1108\u1173\u11A8", "\uC059", "\u1108\u1173\u11A8" },
            new string [] { "\uC05A", "\uC05A", "\u1108\u1173\u11A9", "\uC05A", "\u1108\u1173\u11A9" },
            new string [] { "\uC05B", "\uC05B", "\u1108\u1173\u11AA", "\uC05B", "\u1108\u1173\u11AA" },
            new string [] { "\uC05C", "\uC05C", "\u1108\u1173\u11AB", "\uC05C", "\u1108\u1173\u11AB" },
            new string [] { "\uC05D", "\uC05D", "\u1108\u1173\u11AC", "\uC05D", "\u1108\u1173\u11AC" },
            new string [] { "\uC05E", "\uC05E", "\u1108\u1173\u11AD", "\uC05E", "\u1108\u1173\u11AD" },
            new string [] { "\uC05F", "\uC05F", "\u1108\u1173\u11AE", "\uC05F", "\u1108\u1173\u11AE" },
            new string [] { "\uC060", "\uC060", "\u1108\u1173\u11AF", "\uC060", "\u1108\u1173\u11AF" },
            new string [] { "\uC061", "\uC061", "\u1108\u1173\u11B0", "\uC061", "\u1108\u1173\u11B0" },
            new string [] { "\uC062", "\uC062", "\u1108\u1173\u11B1", "\uC062", "\u1108\u1173\u11B1" },
            new string [] { "\uC063", "\uC063", "\u1108\u1173\u11B2", "\uC063", "\u1108\u1173\u11B2" },
            new string [] { "\uC064", "\uC064", "\u1108\u1173\u11B3", "\uC064", "\u1108\u1173\u11B3" },
            new string [] { "\uC065", "\uC065", "\u1108\u1173\u11B4", "\uC065", "\u1108\u1173\u11B4" },
            new string [] { "\uC066", "\uC066", "\u1108\u1173\u11B5", "\uC066", "\u1108\u1173\u11B5" },
            new string [] { "\uC067", "\uC067", "\u1108\u1173\u11B6", "\uC067", "\u1108\u1173\u11B6" },
            new string [] { "\uC068", "\uC068", "\u1108\u1173\u11B7", "\uC068", "\u1108\u1173\u11B7" },
            new string [] { "\uC069", "\uC069", "\u1108\u1173\u11B8", "\uC069", "\u1108\u1173\u11B8" },
            new string [] { "\uC06A", "\uC06A", "\u1108\u1173\u11B9", "\uC06A", "\u1108\u1173\u11B9" },
            new string [] { "\uC06B", "\uC06B", "\u1108\u1173\u11BA", "\uC06B", "\u1108\u1173\u11BA" },
            new string [] { "\uC06C", "\uC06C", "\u1108\u1173\u11BB", "\uC06C", "\u1108\u1173\u11BB" },
            new string [] { "\uC06D", "\uC06D", "\u1108\u1173\u11BC", "\uC06D", "\u1108\u1173\u11BC" },
            new string [] { "\uC06E", "\uC06E", "\u1108\u1173\u11BD", "\uC06E", "\u1108\u1173\u11BD" },
            new string [] { "\uC06F", "\uC06F", "\u1108\u1173\u11BE", "\uC06F", "\u1108\u1173\u11BE" },
            new string [] { "\uC070", "\uC070", "\u1108\u1173\u11BF", "\uC070", "\u1108\u1173\u11BF" },
            new string [] { "\uC071", "\uC071", "\u1108\u1173\u11C0", "\uC071", "\u1108\u1173\u11C0" },
            new string [] { "\uC072", "\uC072", "\u1108\u1173\u11C1", "\uC072", "\u1108\u1173\u11C1" },
            new string [] { "\uC073", "\uC073", "\u1108\u1173\u11C2", "\uC073", "\u1108\u1173\u11C2" },
            new string [] { "\uC074", "\uC074", "\u1108\u1174", "\uC074", "\u1108\u1174" },
            new string [] { "\uC075", "\uC075", "\u1108\u1174\u11A8", "\uC075", "\u1108\u1174\u11A8" },
            new string [] { "\uC076", "\uC076", "\u1108\u1174\u11A9", "\uC076", "\u1108\u1174\u11A9" },
            new string [] { "\uC077", "\uC077", "\u1108\u1174\u11AA", "\uC077", "\u1108\u1174\u11AA" },
            new string [] { "\uC078", "\uC078", "\u1108\u1174\u11AB", "\uC078", "\u1108\u1174\u11AB" },
            new string [] { "\uC079", "\uC079", "\u1108\u1174\u11AC", "\uC079", "\u1108\u1174\u11AC" },
            new string [] { "\uC07A", "\uC07A", "\u1108\u1174\u11AD", "\uC07A", "\u1108\u1174\u11AD" },
            new string [] { "\uC07B", "\uC07B", "\u1108\u1174\u11AE", "\uC07B", "\u1108\u1174\u11AE" },
            new string [] { "\uC07C", "\uC07C", "\u1108\u1174\u11AF", "\uC07C", "\u1108\u1174\u11AF" },
            new string [] { "\uC07D", "\uC07D", "\u1108\u1174\u11B0", "\uC07D", "\u1108\u1174\u11B0" },
            new string [] { "\uC07E", "\uC07E", "\u1108\u1174\u11B1", "\uC07E", "\u1108\u1174\u11B1" },
            new string [] { "\uC07F", "\uC07F", "\u1108\u1174\u11B2", "\uC07F", "\u1108\u1174\u11B2" },
            new string [] { "\uC080", "\uC080", "\u1108\u1174\u11B3", "\uC080", "\u1108\u1174\u11B3" },
            new string [] { "\uC081", "\uC081", "\u1108\u1174\u11B4", "\uC081", "\u1108\u1174\u11B4" },
            new string [] { "\uC082", "\uC082", "\u1108\u1174\u11B5", "\uC082", "\u1108\u1174\u11B5" },
            new string [] { "\uC083", "\uC083", "\u1108\u1174\u11B6", "\uC083", "\u1108\u1174\u11B6" },
            new string [] { "\uC084", "\uC084", "\u1108\u1174\u11B7", "\uC084", "\u1108\u1174\u11B7" },
            new string [] { "\uC085", "\uC085", "\u1108\u1174\u11B8", "\uC085", "\u1108\u1174\u11B8" },
            new string [] { "\uC086", "\uC086", "\u1108\u1174\u11B9", "\uC086", "\u1108\u1174\u11B9" },
            new string [] { "\uC087", "\uC087", "\u1108\u1174\u11BA", "\uC087", "\u1108\u1174\u11BA" },
            new string [] { "\uC088", "\uC088", "\u1108\u1174\u11BB", "\uC088", "\u1108\u1174\u11BB" },
            new string [] { "\uC089", "\uC089", "\u1108\u1174\u11BC", "\uC089", "\u1108\u1174\u11BC" },
            new string [] { "\uC08A", "\uC08A", "\u1108\u1174\u11BD", "\uC08A", "\u1108\u1174\u11BD" },
            new string [] { "\uC08B", "\uC08B", "\u1108\u1174\u11BE", "\uC08B", "\u1108\u1174\u11BE" },
            new string [] { "\uC08C", "\uC08C", "\u1108\u1174\u11BF", "\uC08C", "\u1108\u1174\u11BF" },
            new string [] { "\uC08D", "\uC08D", "\u1108\u1174\u11C0", "\uC08D", "\u1108\u1174\u11C0" },
            new string [] { "\uC08E", "\uC08E", "\u1108\u1174\u11C1", "\uC08E", "\u1108\u1174\u11C1" },
            new string [] { "\uC08F", "\uC08F", "\u1108\u1174\u11C2", "\uC08F", "\u1108\u1174\u11C2" },
            new string [] { "\uC090", "\uC090", "\u1108\u1175", "\uC090", "\u1108\u1175" },
            new string [] { "\uC091", "\uC091", "\u1108\u1175\u11A8", "\uC091", "\u1108\u1175\u11A8" },
            new string [] { "\uC092", "\uC092", "\u1108\u1175\u11A9", "\uC092", "\u1108\u1175\u11A9" },
            new string [] { "\uC093", "\uC093", "\u1108\u1175\u11AA", "\uC093", "\u1108\u1175\u11AA" },
            new string [] { "\uC094", "\uC094", "\u1108\u1175\u11AB", "\uC094", "\u1108\u1175\u11AB" },
            new string [] { "\uC095", "\uC095", "\u1108\u1175\u11AC", "\uC095", "\u1108\u1175\u11AC" },
            new string [] { "\uC096", "\uC096", "\u1108\u1175\u11AD", "\uC096", "\u1108\u1175\u11AD" },
            new string [] { "\uC097", "\uC097", "\u1108\u1175\u11AE", "\uC097", "\u1108\u1175\u11AE" },
            new string [] { "\uC098", "\uC098", "\u1108\u1175\u11AF", "\uC098", "\u1108\u1175\u11AF" },
            new string [] { "\uC099", "\uC099", "\u1108\u1175\u11B0", "\uC099", "\u1108\u1175\u11B0" },
            new string [] { "\uC09A", "\uC09A", "\u1108\u1175\u11B1", "\uC09A", "\u1108\u1175\u11B1" },
            new string [] { "\uC09B", "\uC09B", "\u1108\u1175\u11B2", "\uC09B", "\u1108\u1175\u11B2" },
            new string [] { "\uC09C", "\uC09C", "\u1108\u1175\u11B3", "\uC09C", "\u1108\u1175\u11B3" },
            new string [] { "\uC09D", "\uC09D", "\u1108\u1175\u11B4", "\uC09D", "\u1108\u1175\u11B4" },
            new string [] { "\uC09E", "\uC09E", "\u1108\u1175\u11B5", "\uC09E", "\u1108\u1175\u11B5" },
            new string [] { "\uC09F", "\uC09F", "\u1108\u1175\u11B6", "\uC09F", "\u1108\u1175\u11B6" },
            new string [] { "\uC0A0", "\uC0A0", "\u1108\u1175\u11B7", "\uC0A0", "\u1108\u1175\u11B7" },
            new string [] { "\uC0A1", "\uC0A1", "\u1108\u1175\u11B8", "\uC0A1", "\u1108\u1175\u11B8" },
            new string [] { "\uC0A2", "\uC0A2", "\u1108\u1175\u11B9", "\uC0A2", "\u1108\u1175\u11B9" },
            new string [] { "\uC0A3", "\uC0A3", "\u1108\u1175\u11BA", "\uC0A3", "\u1108\u1175\u11BA" },
            new string [] { "\uC0A4", "\uC0A4", "\u1108\u1175\u11BB", "\uC0A4", "\u1108\u1175\u11BB" },
            new string [] { "\uC0A5", "\uC0A5", "\u1108\u1175\u11BC", "\uC0A5", "\u1108\u1175\u11BC" },
            new string [] { "\uC0A6", "\uC0A6", "\u1108\u1175\u11BD", "\uC0A6", "\u1108\u1175\u11BD" },
            new string [] { "\uC0A7", "\uC0A7", "\u1108\u1175\u11BE", "\uC0A7", "\u1108\u1175\u11BE" },
            new string [] { "\uC0A8", "\uC0A8", "\u1108\u1175\u11BF", "\uC0A8", "\u1108\u1175\u11BF" },
            new string [] { "\uC0A9", "\uC0A9", "\u1108\u1175\u11C0", "\uC0A9", "\u1108\u1175\u11C0" },
            new string [] { "\uC0AA", "\uC0AA", "\u1108\u1175\u11C1", "\uC0AA", "\u1108\u1175\u11C1" },
            new string [] { "\uC0AB", "\uC0AB", "\u1108\u1175\u11C2", "\uC0AB", "\u1108\u1175\u11C2" },
            new string [] { "\uC0AC", "\uC0AC", "\u1109\u1161", "\uC0AC", "\u1109\u1161" },
            new string [] { "\uC0AD", "\uC0AD", "\u1109\u1161\u11A8", "\uC0AD", "\u1109\u1161\u11A8" },
            new string [] { "\uC0AE", "\uC0AE", "\u1109\u1161\u11A9", "\uC0AE", "\u1109\u1161\u11A9" },
            new string [] { "\uC0AF", "\uC0AF", "\u1109\u1161\u11AA", "\uC0AF", "\u1109\u1161\u11AA" },
            new string [] { "\uC0B0", "\uC0B0", "\u1109\u1161\u11AB", "\uC0B0", "\u1109\u1161\u11AB" },
            new string [] { "\uC0B1", "\uC0B1", "\u1109\u1161\u11AC", "\uC0B1", "\u1109\u1161\u11AC" },
            new string [] { "\uC0B2", "\uC0B2", "\u1109\u1161\u11AD", "\uC0B2", "\u1109\u1161\u11AD" },
            new string [] { "\uC0B3", "\uC0B3", "\u1109\u1161\u11AE", "\uC0B3", "\u1109\u1161\u11AE" },
            new string [] { "\uC0B4", "\uC0B4", "\u1109\u1161\u11AF", "\uC0B4", "\u1109\u1161\u11AF" },
            new string [] { "\uC0B5", "\uC0B5", "\u1109\u1161\u11B0", "\uC0B5", "\u1109\u1161\u11B0" },
            new string [] { "\uC0B6", "\uC0B6", "\u1109\u1161\u11B1", "\uC0B6", "\u1109\u1161\u11B1" },
            new string [] { "\uC0B7", "\uC0B7", "\u1109\u1161\u11B2", "\uC0B7", "\u1109\u1161\u11B2" },
            new string [] { "\uC0B8", "\uC0B8", "\u1109\u1161\u11B3", "\uC0B8", "\u1109\u1161\u11B3" },
            new string [] { "\uC0B9", "\uC0B9", "\u1109\u1161\u11B4", "\uC0B9", "\u1109\u1161\u11B4" },
            new string [] { "\uC0BA", "\uC0BA", "\u1109\u1161\u11B5", "\uC0BA", "\u1109\u1161\u11B5" },
            new string [] { "\uC0BB", "\uC0BB", "\u1109\u1161\u11B6", "\uC0BB", "\u1109\u1161\u11B6" },
            new string [] { "\uC0BC", "\uC0BC", "\u1109\u1161\u11B7", "\uC0BC", "\u1109\u1161\u11B7" },
            new string [] { "\uC0BD", "\uC0BD", "\u1109\u1161\u11B8", "\uC0BD", "\u1109\u1161\u11B8" },
            new string [] { "\uC0BE", "\uC0BE", "\u1109\u1161\u11B9", "\uC0BE", "\u1109\u1161\u11B9" },
            new string [] { "\uC0BF", "\uC0BF", "\u1109\u1161\u11BA", "\uC0BF", "\u1109\u1161\u11BA" },
            new string [] { "\uC0C0", "\uC0C0", "\u1109\u1161\u11BB", "\uC0C0", "\u1109\u1161\u11BB" },
            new string [] { "\uC0C1", "\uC0C1", "\u1109\u1161\u11BC", "\uC0C1", "\u1109\u1161\u11BC" },
            new string [] { "\uC0C2", "\uC0C2", "\u1109\u1161\u11BD", "\uC0C2", "\u1109\u1161\u11BD" },
            new string [] { "\uC0C3", "\uC0C3", "\u1109\u1161\u11BE", "\uC0C3", "\u1109\u1161\u11BE" },
            new string [] { "\uC0C4", "\uC0C4", "\u1109\u1161\u11BF", "\uC0C4", "\u1109\u1161\u11BF" },
            new string [] { "\uC0C5", "\uC0C5", "\u1109\u1161\u11C0", "\uC0C5", "\u1109\u1161\u11C0" },
            new string [] { "\uC0C6", "\uC0C6", "\u1109\u1161\u11C1", "\uC0C6", "\u1109\u1161\u11C1" },
            new string [] { "\uC0C7", "\uC0C7", "\u1109\u1161\u11C2", "\uC0C7", "\u1109\u1161\u11C2" },
            new string [] { "\uC0C8", "\uC0C8", "\u1109\u1162", "\uC0C8", "\u1109\u1162" },
            new string [] { "\uC0C9", "\uC0C9", "\u1109\u1162\u11A8", "\uC0C9", "\u1109\u1162\u11A8" },
            new string [] { "\uC0CA", "\uC0CA", "\u1109\u1162\u11A9", "\uC0CA", "\u1109\u1162\u11A9" },
            new string [] { "\uC0CB", "\uC0CB", "\u1109\u1162\u11AA", "\uC0CB", "\u1109\u1162\u11AA" },
            new string [] { "\uC0CC", "\uC0CC", "\u1109\u1162\u11AB", "\uC0CC", "\u1109\u1162\u11AB" },
            new string [] { "\uC0CD", "\uC0CD", "\u1109\u1162\u11AC", "\uC0CD", "\u1109\u1162\u11AC" },
            new string [] { "\uC0CE", "\uC0CE", "\u1109\u1162\u11AD", "\uC0CE", "\u1109\u1162\u11AD" },
            new string [] { "\uC0CF", "\uC0CF", "\u1109\u1162\u11AE", "\uC0CF", "\u1109\u1162\u11AE" },
            new string [] { "\uC0D0", "\uC0D0", "\u1109\u1162\u11AF", "\uC0D0", "\u1109\u1162\u11AF" },
            new string [] { "\uC0D1", "\uC0D1", "\u1109\u1162\u11B0", "\uC0D1", "\u1109\u1162\u11B0" },
            new string [] { "\uC0D2", "\uC0D2", "\u1109\u1162\u11B1", "\uC0D2", "\u1109\u1162\u11B1" },
            new string [] { "\uC0D3", "\uC0D3", "\u1109\u1162\u11B2", "\uC0D3", "\u1109\u1162\u11B2" },
            new string [] { "\uC0D4", "\uC0D4", "\u1109\u1162\u11B3", "\uC0D4", "\u1109\u1162\u11B3" },
            new string [] { "\uC0D5", "\uC0D5", "\u1109\u1162\u11B4", "\uC0D5", "\u1109\u1162\u11B4" },
            new string [] { "\uC0D6", "\uC0D6", "\u1109\u1162\u11B5", "\uC0D6", "\u1109\u1162\u11B5" },
            new string [] { "\uC0D7", "\uC0D7", "\u1109\u1162\u11B6", "\uC0D7", "\u1109\u1162\u11B6" },
            new string [] { "\uC0D8", "\uC0D8", "\u1109\u1162\u11B7", "\uC0D8", "\u1109\u1162\u11B7" },
            new string [] { "\uC0D9", "\uC0D9", "\u1109\u1162\u11B8", "\uC0D9", "\u1109\u1162\u11B8" },
            new string [] { "\uC0DA", "\uC0DA", "\u1109\u1162\u11B9", "\uC0DA", "\u1109\u1162\u11B9" },
            new string [] { "\uC0DB", "\uC0DB", "\u1109\u1162\u11BA", "\uC0DB", "\u1109\u1162\u11BA" },
            new string [] { "\uC0DC", "\uC0DC", "\u1109\u1162\u11BB", "\uC0DC", "\u1109\u1162\u11BB" },
            new string [] { "\uC0DD", "\uC0DD", "\u1109\u1162\u11BC", "\uC0DD", "\u1109\u1162\u11BC" },
            new string [] { "\uC0DE", "\uC0DE", "\u1109\u1162\u11BD", "\uC0DE", "\u1109\u1162\u11BD" },
            new string [] { "\uC0DF", "\uC0DF", "\u1109\u1162\u11BE", "\uC0DF", "\u1109\u1162\u11BE" },
            new string [] { "\uC0E0", "\uC0E0", "\u1109\u1162\u11BF", "\uC0E0", "\u1109\u1162\u11BF" },
            new string [] { "\uC0E1", "\uC0E1", "\u1109\u1162\u11C0", "\uC0E1", "\u1109\u1162\u11C0" },
            new string [] { "\uC0E2", "\uC0E2", "\u1109\u1162\u11C1", "\uC0E2", "\u1109\u1162\u11C1" },
            new string [] { "\uC0E3", "\uC0E3", "\u1109\u1162\u11C2", "\uC0E3", "\u1109\u1162\u11C2" },
            new string [] { "\uC0E4", "\uC0E4", "\u1109\u1163", "\uC0E4", "\u1109\u1163" },
            new string [] { "\uC0E5", "\uC0E5", "\u1109\u1163\u11A8", "\uC0E5", "\u1109\u1163\u11A8" },
            new string [] { "\uC0E6", "\uC0E6", "\u1109\u1163\u11A9", "\uC0E6", "\u1109\u1163\u11A9" },
            new string [] { "\uC0E7", "\uC0E7", "\u1109\u1163\u11AA", "\uC0E7", "\u1109\u1163\u11AA" },
            new string [] { "\uC0E8", "\uC0E8", "\u1109\u1163\u11AB", "\uC0E8", "\u1109\u1163\u11AB" },
            new string [] { "\uC0E9", "\uC0E9", "\u1109\u1163\u11AC", "\uC0E9", "\u1109\u1163\u11AC" },
            new string [] { "\uC0EA", "\uC0EA", "\u1109\u1163\u11AD", "\uC0EA", "\u1109\u1163\u11AD" },
            new string [] { "\uC0EB", "\uC0EB", "\u1109\u1163\u11AE", "\uC0EB", "\u1109\u1163\u11AE" },
            new string [] { "\uC0EC", "\uC0EC", "\u1109\u1163\u11AF", "\uC0EC", "\u1109\u1163\u11AF" },
            new string [] { "\uC0ED", "\uC0ED", "\u1109\u1163\u11B0", "\uC0ED", "\u1109\u1163\u11B0" },
            new string [] { "\uC0EE", "\uC0EE", "\u1109\u1163\u11B1", "\uC0EE", "\u1109\u1163\u11B1" },
            new string [] { "\uC0EF", "\uC0EF", "\u1109\u1163\u11B2", "\uC0EF", "\u1109\u1163\u11B2" },
            new string [] { "\uC0F0", "\uC0F0", "\u1109\u1163\u11B3", "\uC0F0", "\u1109\u1163\u11B3" },
            new string [] { "\uC0F1", "\uC0F1", "\u1109\u1163\u11B4", "\uC0F1", "\u1109\u1163\u11B4" },
            new string [] { "\uC0F2", "\uC0F2", "\u1109\u1163\u11B5", "\uC0F2", "\u1109\u1163\u11B5" },
            new string [] { "\uC0F3", "\uC0F3", "\u1109\u1163\u11B6", "\uC0F3", "\u1109\u1163\u11B6" },
            new string [] { "\uC0F4", "\uC0F4", "\u1109\u1163\u11B7", "\uC0F4", "\u1109\u1163\u11B7" },
            new string [] { "\uC0F5", "\uC0F5", "\u1109\u1163\u11B8", "\uC0F5", "\u1109\u1163\u11B8" },
            new string [] { "\uC0F6", "\uC0F6", "\u1109\u1163\u11B9", "\uC0F6", "\u1109\u1163\u11B9" },
            new string [] { "\uC0F7", "\uC0F7", "\u1109\u1163\u11BA", "\uC0F7", "\u1109\u1163\u11BA" },
            new string [] { "\uC0F8", "\uC0F8", "\u1109\u1163\u11BB", "\uC0F8", "\u1109\u1163\u11BB" },
            new string [] { "\uC0F9", "\uC0F9", "\u1109\u1163\u11BC", "\uC0F9", "\u1109\u1163\u11BC" },
            new string [] { "\uC0FA", "\uC0FA", "\u1109\u1163\u11BD", "\uC0FA", "\u1109\u1163\u11BD" },
            new string [] { "\uC0FB", "\uC0FB", "\u1109\u1163\u11BE", "\uC0FB", "\u1109\u1163\u11BE" },
            new string [] { "\uC0FC", "\uC0FC", "\u1109\u1163\u11BF", "\uC0FC", "\u1109\u1163\u11BF" },
            new string [] { "\uC0FD", "\uC0FD", "\u1109\u1163\u11C0", "\uC0FD", "\u1109\u1163\u11C0" },
            new string [] { "\uC0FE", "\uC0FE", "\u1109\u1163\u11C1", "\uC0FE", "\u1109\u1163\u11C1" },
            new string [] { "\uC0FF", "\uC0FF", "\u1109\u1163\u11C2", "\uC0FF", "\u1109\u1163\u11C2" },
            new string [] { "\uC100", "\uC100", "\u1109\u1164", "\uC100", "\u1109\u1164" },
            new string [] { "\uC101", "\uC101", "\u1109\u1164\u11A8", "\uC101", "\u1109\u1164\u11A8" },
            new string [] { "\uC102", "\uC102", "\u1109\u1164\u11A9", "\uC102", "\u1109\u1164\u11A9" },
            new string [] { "\uC103", "\uC103", "\u1109\u1164\u11AA", "\uC103", "\u1109\u1164\u11AA" },
            new string [] { "\uC104", "\uC104", "\u1109\u1164\u11AB", "\uC104", "\u1109\u1164\u11AB" },
            new string [] { "\uC105", "\uC105", "\u1109\u1164\u11AC", "\uC105", "\u1109\u1164\u11AC" },
            new string [] { "\uC106", "\uC106", "\u1109\u1164\u11AD", "\uC106", "\u1109\u1164\u11AD" },
            new string [] { "\uC107", "\uC107", "\u1109\u1164\u11AE", "\uC107", "\u1109\u1164\u11AE" },
            new string [] { "\uC108", "\uC108", "\u1109\u1164\u11AF", "\uC108", "\u1109\u1164\u11AF" },
            new string [] { "\uC109", "\uC109", "\u1109\u1164\u11B0", "\uC109", "\u1109\u1164\u11B0" },
            new string [] { "\uC10A", "\uC10A", "\u1109\u1164\u11B1", "\uC10A", "\u1109\u1164\u11B1" },
            new string [] { "\uC10B", "\uC10B", "\u1109\u1164\u11B2", "\uC10B", "\u1109\u1164\u11B2" },
            new string [] { "\uC10C", "\uC10C", "\u1109\u1164\u11B3", "\uC10C", "\u1109\u1164\u11B3" },
            new string [] { "\uC10D", "\uC10D", "\u1109\u1164\u11B4", "\uC10D", "\u1109\u1164\u11B4" },
            new string [] { "\uC10E", "\uC10E", "\u1109\u1164\u11B5", "\uC10E", "\u1109\u1164\u11B5" },
            new string [] { "\uC10F", "\uC10F", "\u1109\u1164\u11B6", "\uC10F", "\u1109\u1164\u11B6" },
            new string [] { "\uC110", "\uC110", "\u1109\u1164\u11B7", "\uC110", "\u1109\u1164\u11B7" },
            new string [] { "\uC111", "\uC111", "\u1109\u1164\u11B8", "\uC111", "\u1109\u1164\u11B8" },
            new string [] { "\uC112", "\uC112", "\u1109\u1164\u11B9", "\uC112", "\u1109\u1164\u11B9" },
            new string [] { "\uC113", "\uC113", "\u1109\u1164\u11BA", "\uC113", "\u1109\u1164\u11BA" },
            new string [] { "\uC114", "\uC114", "\u1109\u1164\u11BB", "\uC114", "\u1109\u1164\u11BB" },
            new string [] { "\uC115", "\uC115", "\u1109\u1164\u11BC", "\uC115", "\u1109\u1164\u11BC" },
            new string [] { "\uC116", "\uC116", "\u1109\u1164\u11BD", "\uC116", "\u1109\u1164\u11BD" },
            new string [] { "\uC117", "\uC117", "\u1109\u1164\u11BE", "\uC117", "\u1109\u1164\u11BE" },
            new string [] { "\uC118", "\uC118", "\u1109\u1164\u11BF", "\uC118", "\u1109\u1164\u11BF" },
            new string [] { "\uC119", "\uC119", "\u1109\u1164\u11C0", "\uC119", "\u1109\u1164\u11C0" },
            new string [] { "\uC11A", "\uC11A", "\u1109\u1164\u11C1", "\uC11A", "\u1109\u1164\u11C1" },
            new string [] { "\uC11B", "\uC11B", "\u1109\u1164\u11C2", "\uC11B", "\u1109\u1164\u11C2" },
            new string [] { "\uC11C", "\uC11C", "\u1109\u1165", "\uC11C", "\u1109\u1165" },
            new string [] { "\uC11D", "\uC11D", "\u1109\u1165\u11A8", "\uC11D", "\u1109\u1165\u11A8" },
            new string [] { "\uC11E", "\uC11E", "\u1109\u1165\u11A9", "\uC11E", "\u1109\u1165\u11A9" },
            new string [] { "\uC11F", "\uC11F", "\u1109\u1165\u11AA", "\uC11F", "\u1109\u1165\u11AA" },
            new string [] { "\uC120", "\uC120", "\u1109\u1165\u11AB", "\uC120", "\u1109\u1165\u11AB" },
            new string [] { "\uC121", "\uC121", "\u1109\u1165\u11AC", "\uC121", "\u1109\u1165\u11AC" },
            new string [] { "\uC122", "\uC122", "\u1109\u1165\u11AD", "\uC122", "\u1109\u1165\u11AD" },
            new string [] { "\uC123", "\uC123", "\u1109\u1165\u11AE", "\uC123", "\u1109\u1165\u11AE" },
            new string [] { "\uC124", "\uC124", "\u1109\u1165\u11AF", "\uC124", "\u1109\u1165\u11AF" },
            new string [] { "\uC125", "\uC125", "\u1109\u1165\u11B0", "\uC125", "\u1109\u1165\u11B0" },
            new string [] { "\uC126", "\uC126", "\u1109\u1165\u11B1", "\uC126", "\u1109\u1165\u11B1" },
            new string [] { "\uC127", "\uC127", "\u1109\u1165\u11B2", "\uC127", "\u1109\u1165\u11B2" },
            new string [] { "\uC128", "\uC128", "\u1109\u1165\u11B3", "\uC128", "\u1109\u1165\u11B3" },
            new string [] { "\uC129", "\uC129", "\u1109\u1165\u11B4", "\uC129", "\u1109\u1165\u11B4" },
            new string [] { "\uC12A", "\uC12A", "\u1109\u1165\u11B5", "\uC12A", "\u1109\u1165\u11B5" },
            new string [] { "\uC12B", "\uC12B", "\u1109\u1165\u11B6", "\uC12B", "\u1109\u1165\u11B6" },
            new string [] { "\uC12C", "\uC12C", "\u1109\u1165\u11B7", "\uC12C", "\u1109\u1165\u11B7" },
            new string [] { "\uC12D", "\uC12D", "\u1109\u1165\u11B8", "\uC12D", "\u1109\u1165\u11B8" },
            new string [] { "\uC12E", "\uC12E", "\u1109\u1165\u11B9", "\uC12E", "\u1109\u1165\u11B9" },
            new string [] { "\uC12F", "\uC12F", "\u1109\u1165\u11BA", "\uC12F", "\u1109\u1165\u11BA" },
            new string [] { "\uC130", "\uC130", "\u1109\u1165\u11BB", "\uC130", "\u1109\u1165\u11BB" },
            new string [] { "\uC131", "\uC131", "\u1109\u1165\u11BC", "\uC131", "\u1109\u1165\u11BC" },
            new string [] { "\uC132", "\uC132", "\u1109\u1165\u11BD", "\uC132", "\u1109\u1165\u11BD" },
            new string [] { "\uC133", "\uC133", "\u1109\u1165\u11BE", "\uC133", "\u1109\u1165\u11BE" },
            new string [] { "\uC134", "\uC134", "\u1109\u1165\u11BF", "\uC134", "\u1109\u1165\u11BF" },
            new string [] { "\uC135", "\uC135", "\u1109\u1165\u11C0", "\uC135", "\u1109\u1165\u11C0" },
            new string [] { "\uC136", "\uC136", "\u1109\u1165\u11C1", "\uC136", "\u1109\u1165\u11C1" },
            new string [] { "\uC137", "\uC137", "\u1109\u1165\u11C2", "\uC137", "\u1109\u1165\u11C2" },
            new string [] { "\uC138", "\uC138", "\u1109\u1166", "\uC138", "\u1109\u1166" },
            new string [] { "\uC139", "\uC139", "\u1109\u1166\u11A8", "\uC139", "\u1109\u1166\u11A8" },
            new string [] { "\uC13A", "\uC13A", "\u1109\u1166\u11A9", "\uC13A", "\u1109\u1166\u11A9" },
            new string [] { "\uC13B", "\uC13B", "\u1109\u1166\u11AA", "\uC13B", "\u1109\u1166\u11AA" },
            new string [] { "\uC13C", "\uC13C", "\u1109\u1166\u11AB", "\uC13C", "\u1109\u1166\u11AB" },
            new string [] { "\uC13D", "\uC13D", "\u1109\u1166\u11AC", "\uC13D", "\u1109\u1166\u11AC" },
            new string [] { "\uC13E", "\uC13E", "\u1109\u1166\u11AD", "\uC13E", "\u1109\u1166\u11AD" },
            new string [] { "\uC13F", "\uC13F", "\u1109\u1166\u11AE", "\uC13F", "\u1109\u1166\u11AE" },
            new string [] { "\uC140", "\uC140", "\u1109\u1166\u11AF", "\uC140", "\u1109\u1166\u11AF" },
            new string [] { "\uC141", "\uC141", "\u1109\u1166\u11B0", "\uC141", "\u1109\u1166\u11B0" },
            new string [] { "\uC142", "\uC142", "\u1109\u1166\u11B1", "\uC142", "\u1109\u1166\u11B1" },
            new string [] { "\uC143", "\uC143", "\u1109\u1166\u11B2", "\uC143", "\u1109\u1166\u11B2" },
            new string [] { "\uC144", "\uC144", "\u1109\u1166\u11B3", "\uC144", "\u1109\u1166\u11B3" },
            new string [] { "\uC145", "\uC145", "\u1109\u1166\u11B4", "\uC145", "\u1109\u1166\u11B4" },
            new string [] { "\uC146", "\uC146", "\u1109\u1166\u11B5", "\uC146", "\u1109\u1166\u11B5" },
            new string [] { "\uC147", "\uC147", "\u1109\u1166\u11B6", "\uC147", "\u1109\u1166\u11B6" },
            new string [] { "\uC148", "\uC148", "\u1109\u1166\u11B7", "\uC148", "\u1109\u1166\u11B7" },
            new string [] { "\uC149", "\uC149", "\u1109\u1166\u11B8", "\uC149", "\u1109\u1166\u11B8" },
            new string [] { "\uC14A", "\uC14A", "\u1109\u1166\u11B9", "\uC14A", "\u1109\u1166\u11B9" },
            new string [] { "\uC14B", "\uC14B", "\u1109\u1166\u11BA", "\uC14B", "\u1109\u1166\u11BA" },
            new string [] { "\uC14C", "\uC14C", "\u1109\u1166\u11BB", "\uC14C", "\u1109\u1166\u11BB" },
            new string [] { "\uC14D", "\uC14D", "\u1109\u1166\u11BC", "\uC14D", "\u1109\u1166\u11BC" },
            new string [] { "\uC14E", "\uC14E", "\u1109\u1166\u11BD", "\uC14E", "\u1109\u1166\u11BD" },
            new string [] { "\uC14F", "\uC14F", "\u1109\u1166\u11BE", "\uC14F", "\u1109\u1166\u11BE" },
            new string [] { "\uC150", "\uC150", "\u1109\u1166\u11BF", "\uC150", "\u1109\u1166\u11BF" },
            new string [] { "\uC151", "\uC151", "\u1109\u1166\u11C0", "\uC151", "\u1109\u1166\u11C0" },
            new string [] { "\uC152", "\uC152", "\u1109\u1166\u11C1", "\uC152", "\u1109\u1166\u11C1" },
            new string [] { "\uC153", "\uC153", "\u1109\u1166\u11C2", "\uC153", "\u1109\u1166\u11C2" },
            new string [] { "\uC154", "\uC154", "\u1109\u1167", "\uC154", "\u1109\u1167" },
            new string [] { "\uC155", "\uC155", "\u1109\u1167\u11A8", "\uC155", "\u1109\u1167\u11A8" },
            new string [] { "\uC156", "\uC156", "\u1109\u1167\u11A9", "\uC156", "\u1109\u1167\u11A9" },
            new string [] { "\uC157", "\uC157", "\u1109\u1167\u11AA", "\uC157", "\u1109\u1167\u11AA" },
            new string [] { "\uC158", "\uC158", "\u1109\u1167\u11AB", "\uC158", "\u1109\u1167\u11AB" },
            new string [] { "\uC159", "\uC159", "\u1109\u1167\u11AC", "\uC159", "\u1109\u1167\u11AC" },
            new string [] { "\uC15A", "\uC15A", "\u1109\u1167\u11AD", "\uC15A", "\u1109\u1167\u11AD" },
            new string [] { "\uC15B", "\uC15B", "\u1109\u1167\u11AE", "\uC15B", "\u1109\u1167\u11AE" },
            new string [] { "\uC15C", "\uC15C", "\u1109\u1167\u11AF", "\uC15C", "\u1109\u1167\u11AF" },
            new string [] { "\uC15D", "\uC15D", "\u1109\u1167\u11B0", "\uC15D", "\u1109\u1167\u11B0" },
            new string [] { "\uC15E", "\uC15E", "\u1109\u1167\u11B1", "\uC15E", "\u1109\u1167\u11B1" },
            new string [] { "\uC15F", "\uC15F", "\u1109\u1167\u11B2", "\uC15F", "\u1109\u1167\u11B2" },
            new string [] { "\uC160", "\uC160", "\u1109\u1167\u11B3", "\uC160", "\u1109\u1167\u11B3" },
            new string [] { "\uC161", "\uC161", "\u1109\u1167\u11B4", "\uC161", "\u1109\u1167\u11B4" },
            new string [] { "\uC162", "\uC162", "\u1109\u1167\u11B5", "\uC162", "\u1109\u1167\u11B5" },
            new string [] { "\uC163", "\uC163", "\u1109\u1167\u11B6", "\uC163", "\u1109\u1167\u11B6" },
            new string [] { "\uC164", "\uC164", "\u1109\u1167\u11B7", "\uC164", "\u1109\u1167\u11B7" },
            new string [] { "\uC165", "\uC165", "\u1109\u1167\u11B8", "\uC165", "\u1109\u1167\u11B8" },
            new string [] { "\uC166", "\uC166", "\u1109\u1167\u11B9", "\uC166", "\u1109\u1167\u11B9" },
            new string [] { "\uC167", "\uC167", "\u1109\u1167\u11BA", "\uC167", "\u1109\u1167\u11BA" },
            new string [] { "\uC168", "\uC168", "\u1109\u1167\u11BB", "\uC168", "\u1109\u1167\u11BB" },
            new string [] { "\uC169", "\uC169", "\u1109\u1167\u11BC", "\uC169", "\u1109\u1167\u11BC" },
            new string [] { "\uC16A", "\uC16A", "\u1109\u1167\u11BD", "\uC16A", "\u1109\u1167\u11BD" },
            new string [] { "\uC16B", "\uC16B", "\u1109\u1167\u11BE", "\uC16B", "\u1109\u1167\u11BE" },
            new string [] { "\uC16C", "\uC16C", "\u1109\u1167\u11BF", "\uC16C", "\u1109\u1167\u11BF" },
            new string [] { "\uC16D", "\uC16D", "\u1109\u1167\u11C0", "\uC16D", "\u1109\u1167\u11C0" },
            new string [] { "\uC16E", "\uC16E", "\u1109\u1167\u11C1", "\uC16E", "\u1109\u1167\u11C1" },
            new string [] { "\uC16F", "\uC16F", "\u1109\u1167\u11C2", "\uC16F", "\u1109\u1167\u11C2" },
            new string [] { "\uC170", "\uC170", "\u1109\u1168", "\uC170", "\u1109\u1168" },
            new string [] { "\uC171", "\uC171", "\u1109\u1168\u11A8", "\uC171", "\u1109\u1168\u11A8" },
            new string [] { "\uC172", "\uC172", "\u1109\u1168\u11A9", "\uC172", "\u1109\u1168\u11A9" },
            new string [] { "\uC173", "\uC173", "\u1109\u1168\u11AA", "\uC173", "\u1109\u1168\u11AA" },
            new string [] { "\uC174", "\uC174", "\u1109\u1168\u11AB", "\uC174", "\u1109\u1168\u11AB" },
            new string [] { "\uC175", "\uC175", "\u1109\u1168\u11AC", "\uC175", "\u1109\u1168\u11AC" },
            new string [] { "\uC176", "\uC176", "\u1109\u1168\u11AD", "\uC176", "\u1109\u1168\u11AD" },
            new string [] { "\uC177", "\uC177", "\u1109\u1168\u11AE", "\uC177", "\u1109\u1168\u11AE" },
            new string [] { "\uC178", "\uC178", "\u1109\u1168\u11AF", "\uC178", "\u1109\u1168\u11AF" },
            new string [] { "\uC179", "\uC179", "\u1109\u1168\u11B0", "\uC179", "\u1109\u1168\u11B0" },
            new string [] { "\uC17A", "\uC17A", "\u1109\u1168\u11B1", "\uC17A", "\u1109\u1168\u11B1" },
            new string [] { "\uC17B", "\uC17B", "\u1109\u1168\u11B2", "\uC17B", "\u1109\u1168\u11B2" },
            new string [] { "\uC17C", "\uC17C", "\u1109\u1168\u11B3", "\uC17C", "\u1109\u1168\u11B3" },
            new string [] { "\uC17D", "\uC17D", "\u1109\u1168\u11B4", "\uC17D", "\u1109\u1168\u11B4" },
            new string [] { "\uC17E", "\uC17E", "\u1109\u1168\u11B5", "\uC17E", "\u1109\u1168\u11B5" },
            new string [] { "\uC17F", "\uC17F", "\u1109\u1168\u11B6", "\uC17F", "\u1109\u1168\u11B6" },
            new string [] { "\uC180", "\uC180", "\u1109\u1168\u11B7", "\uC180", "\u1109\u1168\u11B7" },
            new string [] { "\uC181", "\uC181", "\u1109\u1168\u11B8", "\uC181", "\u1109\u1168\u11B8" },
            new string [] { "\uC182", "\uC182", "\u1109\u1168\u11B9", "\uC182", "\u1109\u1168\u11B9" },
            new string [] { "\uC183", "\uC183", "\u1109\u1168\u11BA", "\uC183", "\u1109\u1168\u11BA" },
            new string [] { "\uC184", "\uC184", "\u1109\u1168\u11BB", "\uC184", "\u1109\u1168\u11BB" },
            new string [] { "\uC185", "\uC185", "\u1109\u1168\u11BC", "\uC185", "\u1109\u1168\u11BC" },
            new string [] { "\uC186", "\uC186", "\u1109\u1168\u11BD", "\uC186", "\u1109\u1168\u11BD" },
            new string [] { "\uC187", "\uC187", "\u1109\u1168\u11BE", "\uC187", "\u1109\u1168\u11BE" },
            new string [] { "\uC188", "\uC188", "\u1109\u1168\u11BF", "\uC188", "\u1109\u1168\u11BF" },
            new string [] { "\uC189", "\uC189", "\u1109\u1168\u11C0", "\uC189", "\u1109\u1168\u11C0" },
            new string [] { "\uC18A", "\uC18A", "\u1109\u1168\u11C1", "\uC18A", "\u1109\u1168\u11C1" },
            new string [] { "\uC18B", "\uC18B", "\u1109\u1168\u11C2", "\uC18B", "\u1109\u1168\u11C2" },
            new string [] { "\uC18C", "\uC18C", "\u1109\u1169", "\uC18C", "\u1109\u1169" },
            new string [] { "\uC18D", "\uC18D", "\u1109\u1169\u11A8", "\uC18D", "\u1109\u1169\u11A8" },
            new string [] { "\uC18E", "\uC18E", "\u1109\u1169\u11A9", "\uC18E", "\u1109\u1169\u11A9" },
            new string [] { "\uC18F", "\uC18F", "\u1109\u1169\u11AA", "\uC18F", "\u1109\u1169\u11AA" },
            new string [] { "\uC190", "\uC190", "\u1109\u1169\u11AB", "\uC190", "\u1109\u1169\u11AB" },
            new string [] { "\uC191", "\uC191", "\u1109\u1169\u11AC", "\uC191", "\u1109\u1169\u11AC" },
            new string [] { "\uC192", "\uC192", "\u1109\u1169\u11AD", "\uC192", "\u1109\u1169\u11AD" },
            new string [] { "\uC193", "\uC193", "\u1109\u1169\u11AE", "\uC193", "\u1109\u1169\u11AE" },
            new string [] { "\uC194", "\uC194", "\u1109\u1169\u11AF", "\uC194", "\u1109\u1169\u11AF" },
            new string [] { "\uC195", "\uC195", "\u1109\u1169\u11B0", "\uC195", "\u1109\u1169\u11B0" },
            new string [] { "\uC196", "\uC196", "\u1109\u1169\u11B1", "\uC196", "\u1109\u1169\u11B1" },
            new string [] { "\uC197", "\uC197", "\u1109\u1169\u11B2", "\uC197", "\u1109\u1169\u11B2" },
            new string [] { "\uC198", "\uC198", "\u1109\u1169\u11B3", "\uC198", "\u1109\u1169\u11B3" },
            new string [] { "\uC199", "\uC199", "\u1109\u1169\u11B4", "\uC199", "\u1109\u1169\u11B4" },
            new string [] { "\uC19A", "\uC19A", "\u1109\u1169\u11B5", "\uC19A", "\u1109\u1169\u11B5" },
            new string [] { "\uC19B", "\uC19B", "\u1109\u1169\u11B6", "\uC19B", "\u1109\u1169\u11B6" },
            new string [] { "\uC19C", "\uC19C", "\u1109\u1169\u11B7", "\uC19C", "\u1109\u1169\u11B7" },
            new string [] { "\uC19D", "\uC19D", "\u1109\u1169\u11B8", "\uC19D", "\u1109\u1169\u11B8" },
            new string [] { "\uC19E", "\uC19E", "\u1109\u1169\u11B9", "\uC19E", "\u1109\u1169\u11B9" },
            new string [] { "\uC19F", "\uC19F", "\u1109\u1169\u11BA", "\uC19F", "\u1109\u1169\u11BA" },
            new string [] { "\uC1A0", "\uC1A0", "\u1109\u1169\u11BB", "\uC1A0", "\u1109\u1169\u11BB" },
            new string [] { "\uC1A1", "\uC1A1", "\u1109\u1169\u11BC", "\uC1A1", "\u1109\u1169\u11BC" },
            new string [] { "\uC1A2", "\uC1A2", "\u1109\u1169\u11BD", "\uC1A2", "\u1109\u1169\u11BD" },
            new string [] { "\uC1A3", "\uC1A3", "\u1109\u1169\u11BE", "\uC1A3", "\u1109\u1169\u11BE" },
            new string [] { "\uC1A4", "\uC1A4", "\u1109\u1169\u11BF", "\uC1A4", "\u1109\u1169\u11BF" },
            new string [] { "\uC1A5", "\uC1A5", "\u1109\u1169\u11C0", "\uC1A5", "\u1109\u1169\u11C0" },
            new string [] { "\uC1A6", "\uC1A6", "\u1109\u1169\u11C1", "\uC1A6", "\u1109\u1169\u11C1" },
            new string [] { "\uC1A7", "\uC1A7", "\u1109\u1169\u11C2", "\uC1A7", "\u1109\u1169\u11C2" },
            new string [] { "\uC1A8", "\uC1A8", "\u1109\u116A", "\uC1A8", "\u1109\u116A" },
            new string [] { "\uC1A9", "\uC1A9", "\u1109\u116A\u11A8", "\uC1A9", "\u1109\u116A\u11A8" },
            new string [] { "\uC1AA", "\uC1AA", "\u1109\u116A\u11A9", "\uC1AA", "\u1109\u116A\u11A9" },
            new string [] { "\uC1AB", "\uC1AB", "\u1109\u116A\u11AA", "\uC1AB", "\u1109\u116A\u11AA" },
            new string [] { "\uC1AC", "\uC1AC", "\u1109\u116A\u11AB", "\uC1AC", "\u1109\u116A\u11AB" },
            new string [] { "\uC1AD", "\uC1AD", "\u1109\u116A\u11AC", "\uC1AD", "\u1109\u116A\u11AC" },
            new string [] { "\uC1AE", "\uC1AE", "\u1109\u116A\u11AD", "\uC1AE", "\u1109\u116A\u11AD" },
            new string [] { "\uC1AF", "\uC1AF", "\u1109\u116A\u11AE", "\uC1AF", "\u1109\u116A\u11AE" },
            new string [] { "\uC1B0", "\uC1B0", "\u1109\u116A\u11AF", "\uC1B0", "\u1109\u116A\u11AF" },
            new string [] { "\uC1B1", "\uC1B1", "\u1109\u116A\u11B0", "\uC1B1", "\u1109\u116A\u11B0" },
            new string [] { "\uC1B2", "\uC1B2", "\u1109\u116A\u11B1", "\uC1B2", "\u1109\u116A\u11B1" },
            new string [] { "\uC1B3", "\uC1B3", "\u1109\u116A\u11B2", "\uC1B3", "\u1109\u116A\u11B2" },
            new string [] { "\uC1B4", "\uC1B4", "\u1109\u116A\u11B3", "\uC1B4", "\u1109\u116A\u11B3" },
            new string [] { "\uC1B5", "\uC1B5", "\u1109\u116A\u11B4", "\uC1B5", "\u1109\u116A\u11B4" },
            new string [] { "\uC1B6", "\uC1B6", "\u1109\u116A\u11B5", "\uC1B6", "\u1109\u116A\u11B5" },
            new string [] { "\uC1B7", "\uC1B7", "\u1109\u116A\u11B6", "\uC1B7", "\u1109\u116A\u11B6" },
            new string [] { "\uC1B8", "\uC1B8", "\u1109\u116A\u11B7", "\uC1B8", "\u1109\u116A\u11B7" },
            new string [] { "\uC1B9", "\uC1B9", "\u1109\u116A\u11B8", "\uC1B9", "\u1109\u116A\u11B8" },
            new string [] { "\uC1BA", "\uC1BA", "\u1109\u116A\u11B9", "\uC1BA", "\u1109\u116A\u11B9" },
            new string [] { "\uC1BB", "\uC1BB", "\u1109\u116A\u11BA", "\uC1BB", "\u1109\u116A\u11BA" },
            new string [] { "\uC1BC", "\uC1BC", "\u1109\u116A\u11BB", "\uC1BC", "\u1109\u116A\u11BB" },
            new string [] { "\uC1BD", "\uC1BD", "\u1109\u116A\u11BC", "\uC1BD", "\u1109\u116A\u11BC" },
            new string [] { "\uC1BE", "\uC1BE", "\u1109\u116A\u11BD", "\uC1BE", "\u1109\u116A\u11BD" },
            new string [] { "\uC1BF", "\uC1BF", "\u1109\u116A\u11BE", "\uC1BF", "\u1109\u116A\u11BE" },
            new string [] { "\uC1C0", "\uC1C0", "\u1109\u116A\u11BF", "\uC1C0", "\u1109\u116A\u11BF" },
            new string [] { "\uC1C1", "\uC1C1", "\u1109\u116A\u11C0", "\uC1C1", "\u1109\u116A\u11C0" },
            new string [] { "\uC1C2", "\uC1C2", "\u1109\u116A\u11C1", "\uC1C2", "\u1109\u116A\u11C1" },
            new string [] { "\uC1C3", "\uC1C3", "\u1109\u116A\u11C2", "\uC1C3", "\u1109\u116A\u11C2" },
            new string [] { "\uC1C4", "\uC1C4", "\u1109\u116B", "\uC1C4", "\u1109\u116B" },
            new string [] { "\uC1C5", "\uC1C5", "\u1109\u116B\u11A8", "\uC1C5", "\u1109\u116B\u11A8" },
            new string [] { "\uC1C6", "\uC1C6", "\u1109\u116B\u11A9", "\uC1C6", "\u1109\u116B\u11A9" },
            new string [] { "\uC1C7", "\uC1C7", "\u1109\u116B\u11AA", "\uC1C7", "\u1109\u116B\u11AA" },
            new string [] { "\uC1C8", "\uC1C8", "\u1109\u116B\u11AB", "\uC1C8", "\u1109\u116B\u11AB" },
            new string [] { "\uC1C9", "\uC1C9", "\u1109\u116B\u11AC", "\uC1C9", "\u1109\u116B\u11AC" },
            new string [] { "\uC1CA", "\uC1CA", "\u1109\u116B\u11AD", "\uC1CA", "\u1109\u116B\u11AD" },
            new string [] { "\uC1CB", "\uC1CB", "\u1109\u116B\u11AE", "\uC1CB", "\u1109\u116B\u11AE" },
            new string [] { "\uC1CC", "\uC1CC", "\u1109\u116B\u11AF", "\uC1CC", "\u1109\u116B\u11AF" },
            new string [] { "\uC1CD", "\uC1CD", "\u1109\u116B\u11B0", "\uC1CD", "\u1109\u116B\u11B0" },
            new string [] { "\uC1CE", "\uC1CE", "\u1109\u116B\u11B1", "\uC1CE", "\u1109\u116B\u11B1" },
            new string [] { "\uC1CF", "\uC1CF", "\u1109\u116B\u11B2", "\uC1CF", "\u1109\u116B\u11B2" },
            new string [] { "\uC1D0", "\uC1D0", "\u1109\u116B\u11B3", "\uC1D0", "\u1109\u116B\u11B3" },
            new string [] { "\uC1D1", "\uC1D1", "\u1109\u116B\u11B4", "\uC1D1", "\u1109\u116B\u11B4" },
            new string [] { "\uC1D2", "\uC1D2", "\u1109\u116B\u11B5", "\uC1D2", "\u1109\u116B\u11B5" },
            new string [] { "\uC1D3", "\uC1D3", "\u1109\u116B\u11B6", "\uC1D3", "\u1109\u116B\u11B6" },
            new string [] { "\uC1D4", "\uC1D4", "\u1109\u116B\u11B7", "\uC1D4", "\u1109\u116B\u11B7" },
            new string [] { "\uC1D5", "\uC1D5", "\u1109\u116B\u11B8", "\uC1D5", "\u1109\u116B\u11B8" },
            new string [] { "\uC1D6", "\uC1D6", "\u1109\u116B\u11B9", "\uC1D6", "\u1109\u116B\u11B9" },
            new string [] { "\uC1D7", "\uC1D7", "\u1109\u116B\u11BA", "\uC1D7", "\u1109\u116B\u11BA" },
            new string [] { "\uC1D8", "\uC1D8", "\u1109\u116B\u11BB", "\uC1D8", "\u1109\u116B\u11BB" },
            new string [] { "\uC1D9", "\uC1D9", "\u1109\u116B\u11BC", "\uC1D9", "\u1109\u116B\u11BC" },
            new string [] { "\uC1DA", "\uC1DA", "\u1109\u116B\u11BD", "\uC1DA", "\u1109\u116B\u11BD" },
            new string [] { "\uC1DB", "\uC1DB", "\u1109\u116B\u11BE", "\uC1DB", "\u1109\u116B\u11BE" },
            new string [] { "\uC1DC", "\uC1DC", "\u1109\u116B\u11BF", "\uC1DC", "\u1109\u116B\u11BF" },
            new string [] { "\uC1DD", "\uC1DD", "\u1109\u116B\u11C0", "\uC1DD", "\u1109\u116B\u11C0" },
            new string [] { "\uC1DE", "\uC1DE", "\u1109\u116B\u11C1", "\uC1DE", "\u1109\u116B\u11C1" },
            new string [] { "\uC1DF", "\uC1DF", "\u1109\u116B\u11C2", "\uC1DF", "\u1109\u116B\u11C2" },
            new string [] { "\uC1E0", "\uC1E0", "\u1109\u116C", "\uC1E0", "\u1109\u116C" },
            new string [] { "\uC1E1", "\uC1E1", "\u1109\u116C\u11A8", "\uC1E1", "\u1109\u116C\u11A8" },
            new string [] { "\uC1E2", "\uC1E2", "\u1109\u116C\u11A9", "\uC1E2", "\u1109\u116C\u11A9" },
            new string [] { "\uC1E3", "\uC1E3", "\u1109\u116C\u11AA", "\uC1E3", "\u1109\u116C\u11AA" },
            new string [] { "\uC1E4", "\uC1E4", "\u1109\u116C\u11AB", "\uC1E4", "\u1109\u116C\u11AB" },
            new string [] { "\uC1E5", "\uC1E5", "\u1109\u116C\u11AC", "\uC1E5", "\u1109\u116C\u11AC" },
            new string [] { "\uC1E6", "\uC1E6", "\u1109\u116C\u11AD", "\uC1E6", "\u1109\u116C\u11AD" },
            new string [] { "\uC1E7", "\uC1E7", "\u1109\u116C\u11AE", "\uC1E7", "\u1109\u116C\u11AE" },
            new string [] { "\uC1E8", "\uC1E8", "\u1109\u116C\u11AF", "\uC1E8", "\u1109\u116C\u11AF" },
            new string [] { "\uC1E9", "\uC1E9", "\u1109\u116C\u11B0", "\uC1E9", "\u1109\u116C\u11B0" },
            new string [] { "\uC1EA", "\uC1EA", "\u1109\u116C\u11B1", "\uC1EA", "\u1109\u116C\u11B1" },
            new string [] { "\uC1EB", "\uC1EB", "\u1109\u116C\u11B2", "\uC1EB", "\u1109\u116C\u11B2" },
            new string [] { "\uC1EC", "\uC1EC", "\u1109\u116C\u11B3", "\uC1EC", "\u1109\u116C\u11B3" },
            new string [] { "\uC1ED", "\uC1ED", "\u1109\u116C\u11B4", "\uC1ED", "\u1109\u116C\u11B4" },
            new string [] { "\uC1EE", "\uC1EE", "\u1109\u116C\u11B5", "\uC1EE", "\u1109\u116C\u11B5" },
            new string [] { "\uC1EF", "\uC1EF", "\u1109\u116C\u11B6", "\uC1EF", "\u1109\u116C\u11B6" },
            new string [] { "\uC1F0", "\uC1F0", "\u1109\u116C\u11B7", "\uC1F0", "\u1109\u116C\u11B7" },
            new string [] { "\uC1F1", "\uC1F1", "\u1109\u116C\u11B8", "\uC1F1", "\u1109\u116C\u11B8" },
            new string [] { "\uC1F2", "\uC1F2", "\u1109\u116C\u11B9", "\uC1F2", "\u1109\u116C\u11B9" },
            new string [] { "\uC1F3", "\uC1F3", "\u1109\u116C\u11BA", "\uC1F3", "\u1109\u116C\u11BA" },
            new string [] { "\uC1F4", "\uC1F4", "\u1109\u116C\u11BB", "\uC1F4", "\u1109\u116C\u11BB" },
            new string [] { "\uC1F5", "\uC1F5", "\u1109\u116C\u11BC", "\uC1F5", "\u1109\u116C\u11BC" },
            new string [] { "\uC1F6", "\uC1F6", "\u1109\u116C\u11BD", "\uC1F6", "\u1109\u116C\u11BD" },
            new string [] { "\uC1F7", "\uC1F7", "\u1109\u116C\u11BE", "\uC1F7", "\u1109\u116C\u11BE" },
            new string [] { "\uC1F8", "\uC1F8", "\u1109\u116C\u11BF", "\uC1F8", "\u1109\u116C\u11BF" },
            new string [] { "\uC1F9", "\uC1F9", "\u1109\u116C\u11C0", "\uC1F9", "\u1109\u116C\u11C0" },
            new string [] { "\uC1FA", "\uC1FA", "\u1109\u116C\u11C1", "\uC1FA", "\u1109\u116C\u11C1" },
            new string [] { "\uC1FB", "\uC1FB", "\u1109\u116C\u11C2", "\uC1FB", "\u1109\u116C\u11C2" },
            new string [] { "\uC1FC", "\uC1FC", "\u1109\u116D", "\uC1FC", "\u1109\u116D" },
            new string [] { "\uC1FD", "\uC1FD", "\u1109\u116D\u11A8", "\uC1FD", "\u1109\u116D\u11A8" },
            new string [] { "\uC1FE", "\uC1FE", "\u1109\u116D\u11A9", "\uC1FE", "\u1109\u116D\u11A9" },
            new string [] { "\uC1FF", "\uC1FF", "\u1109\u116D\u11AA", "\uC1FF", "\u1109\u116D\u11AA" },
            new string [] { "\uC200", "\uC200", "\u1109\u116D\u11AB", "\uC200", "\u1109\u116D\u11AB" },
            new string [] { "\uC201", "\uC201", "\u1109\u116D\u11AC", "\uC201", "\u1109\u116D\u11AC" },
            new string [] { "\uC202", "\uC202", "\u1109\u116D\u11AD", "\uC202", "\u1109\u116D\u11AD" },
            new string [] { "\uC203", "\uC203", "\u1109\u116D\u11AE", "\uC203", "\u1109\u116D\u11AE" },
            new string [] { "\uC204", "\uC204", "\u1109\u116D\u11AF", "\uC204", "\u1109\u116D\u11AF" },
            new string [] { "\uC205", "\uC205", "\u1109\u116D\u11B0", "\uC205", "\u1109\u116D\u11B0" },
            new string [] { "\uC206", "\uC206", "\u1109\u116D\u11B1", "\uC206", "\u1109\u116D\u11B1" },
            new string [] { "\uC207", "\uC207", "\u1109\u116D\u11B2", "\uC207", "\u1109\u116D\u11B2" },
            new string [] { "\uC208", "\uC208", "\u1109\u116D\u11B3", "\uC208", "\u1109\u116D\u11B3" },
            new string [] { "\uC209", "\uC209", "\u1109\u116D\u11B4", "\uC209", "\u1109\u116D\u11B4" },
            new string [] { "\uC20A", "\uC20A", "\u1109\u116D\u11B5", "\uC20A", "\u1109\u116D\u11B5" },
            new string [] { "\uC20B", "\uC20B", "\u1109\u116D\u11B6", "\uC20B", "\u1109\u116D\u11B6" },
            new string [] { "\uC20C", "\uC20C", "\u1109\u116D\u11B7", "\uC20C", "\u1109\u116D\u11B7" },
            new string [] { "\uC20D", "\uC20D", "\u1109\u116D\u11B8", "\uC20D", "\u1109\u116D\u11B8" },
            new string [] { "\uC20E", "\uC20E", "\u1109\u116D\u11B9", "\uC20E", "\u1109\u116D\u11B9" },
            new string [] { "\uC20F", "\uC20F", "\u1109\u116D\u11BA", "\uC20F", "\u1109\u116D\u11BA" },
            new string [] { "\uC210", "\uC210", "\u1109\u116D\u11BB", "\uC210", "\u1109\u116D\u11BB" },
            new string [] { "\uC211", "\uC211", "\u1109\u116D\u11BC", "\uC211", "\u1109\u116D\u11BC" },
            new string [] { "\uC212", "\uC212", "\u1109\u116D\u11BD", "\uC212", "\u1109\u116D\u11BD" },
            new string [] { "\uC213", "\uC213", "\u1109\u116D\u11BE", "\uC213", "\u1109\u116D\u11BE" },
            new string [] { "\uC214", "\uC214", "\u1109\u116D\u11BF", "\uC214", "\u1109\u116D\u11BF" },
            new string [] { "\uC215", "\uC215", "\u1109\u116D\u11C0", "\uC215", "\u1109\u116D\u11C0" },
            new string [] { "\uC216", "\uC216", "\u1109\u116D\u11C1", "\uC216", "\u1109\u116D\u11C1" },
            new string [] { "\uC217", "\uC217", "\u1109\u116D\u11C2", "\uC217", "\u1109\u116D\u11C2" },
            new string [] { "\uC218", "\uC218", "\u1109\u116E", "\uC218", "\u1109\u116E" },
            new string [] { "\uC219", "\uC219", "\u1109\u116E\u11A8", "\uC219", "\u1109\u116E\u11A8" },
            new string [] { "\uC21A", "\uC21A", "\u1109\u116E\u11A9", "\uC21A", "\u1109\u116E\u11A9" },
            new string [] { "\uC21B", "\uC21B", "\u1109\u116E\u11AA", "\uC21B", "\u1109\u116E\u11AA" },
            new string [] { "\uC21C", "\uC21C", "\u1109\u116E\u11AB", "\uC21C", "\u1109\u116E\u11AB" },
            new string [] { "\uC21D", "\uC21D", "\u1109\u116E\u11AC", "\uC21D", "\u1109\u116E\u11AC" },
            new string [] { "\uC21E", "\uC21E", "\u1109\u116E\u11AD", "\uC21E", "\u1109\u116E\u11AD" },
            new string [] { "\uC21F", "\uC21F", "\u1109\u116E\u11AE", "\uC21F", "\u1109\u116E\u11AE" },
            new string [] { "\uC220", "\uC220", "\u1109\u116E\u11AF", "\uC220", "\u1109\u116E\u11AF" },
            new string [] { "\uC221", "\uC221", "\u1109\u116E\u11B0", "\uC221", "\u1109\u116E\u11B0" },
            new string [] { "\uC222", "\uC222", "\u1109\u116E\u11B1", "\uC222", "\u1109\u116E\u11B1" },
            new string [] { "\uC223", "\uC223", "\u1109\u116E\u11B2", "\uC223", "\u1109\u116E\u11B2" },
            new string [] { "\uC224", "\uC224", "\u1109\u116E\u11B3", "\uC224", "\u1109\u116E\u11B3" },
            new string [] { "\uC225", "\uC225", "\u1109\u116E\u11B4", "\uC225", "\u1109\u116E\u11B4" },
            new string [] { "\uC226", "\uC226", "\u1109\u116E\u11B5", "\uC226", "\u1109\u116E\u11B5" },
            new string [] { "\uC227", "\uC227", "\u1109\u116E\u11B6", "\uC227", "\u1109\u116E\u11B6" },
            new string [] { "\uC228", "\uC228", "\u1109\u116E\u11B7", "\uC228", "\u1109\u116E\u11B7" },
            new string [] { "\uC229", "\uC229", "\u1109\u116E\u11B8", "\uC229", "\u1109\u116E\u11B8" },
            new string [] { "\uC22A", "\uC22A", "\u1109\u116E\u11B9", "\uC22A", "\u1109\u116E\u11B9" },
            new string [] { "\uC22B", "\uC22B", "\u1109\u116E\u11BA", "\uC22B", "\u1109\u116E\u11BA" },
            new string [] { "\uC22C", "\uC22C", "\u1109\u116E\u11BB", "\uC22C", "\u1109\u116E\u11BB" },
            new string [] { "\uC22D", "\uC22D", "\u1109\u116E\u11BC", "\uC22D", "\u1109\u116E\u11BC" },
            new string [] { "\uC22E", "\uC22E", "\u1109\u116E\u11BD", "\uC22E", "\u1109\u116E\u11BD" },
            new string [] { "\uC22F", "\uC22F", "\u1109\u116E\u11BE", "\uC22F", "\u1109\u116E\u11BE" },
            new string [] { "\uC230", "\uC230", "\u1109\u116E\u11BF", "\uC230", "\u1109\u116E\u11BF" },
            new string [] { "\uC231", "\uC231", "\u1109\u116E\u11C0", "\uC231", "\u1109\u116E\u11C0" },
            new string [] { "\uC232", "\uC232", "\u1109\u116E\u11C1", "\uC232", "\u1109\u116E\u11C1" },
            new string [] { "\uC233", "\uC233", "\u1109\u116E\u11C2", "\uC233", "\u1109\u116E\u11C2" },
            new string [] { "\uC234", "\uC234", "\u1109\u116F", "\uC234", "\u1109\u116F" },
            new string [] { "\uC235", "\uC235", "\u1109\u116F\u11A8", "\uC235", "\u1109\u116F\u11A8" },
            new string [] { "\uC236", "\uC236", "\u1109\u116F\u11A9", "\uC236", "\u1109\u116F\u11A9" },
            new string [] { "\uC237", "\uC237", "\u1109\u116F\u11AA", "\uC237", "\u1109\u116F\u11AA" },
            new string [] { "\uC238", "\uC238", "\u1109\u116F\u11AB", "\uC238", "\u1109\u116F\u11AB" },
            new string [] { "\uC239", "\uC239", "\u1109\u116F\u11AC", "\uC239", "\u1109\u116F\u11AC" },
            new string [] { "\uC23A", "\uC23A", "\u1109\u116F\u11AD", "\uC23A", "\u1109\u116F\u11AD" },
            new string [] { "\uC23B", "\uC23B", "\u1109\u116F\u11AE", "\uC23B", "\u1109\u116F\u11AE" },
            new string [] { "\uC23C", "\uC23C", "\u1109\u116F\u11AF", "\uC23C", "\u1109\u116F\u11AF" },
            new string [] { "\uC23D", "\uC23D", "\u1109\u116F\u11B0", "\uC23D", "\u1109\u116F\u11B0" },
            new string [] { "\uC23E", "\uC23E", "\u1109\u116F\u11B1", "\uC23E", "\u1109\u116F\u11B1" },
            new string [] { "\uC23F", "\uC23F", "\u1109\u116F\u11B2", "\uC23F", "\u1109\u116F\u11B2" },
            new string [] { "\uC240", "\uC240", "\u1109\u116F\u11B3", "\uC240", "\u1109\u116F\u11B3" },
            new string [] { "\uC241", "\uC241", "\u1109\u116F\u11B4", "\uC241", "\u1109\u116F\u11B4" },
            new string [] { "\uC242", "\uC242", "\u1109\u116F\u11B5", "\uC242", "\u1109\u116F\u11B5" },
            new string [] { "\uC243", "\uC243", "\u1109\u116F\u11B6", "\uC243", "\u1109\u116F\u11B6" },
            new string [] { "\uC244", "\uC244", "\u1109\u116F\u11B7", "\uC244", "\u1109\u116F\u11B7" },
            new string [] { "\uC245", "\uC245", "\u1109\u116F\u11B8", "\uC245", "\u1109\u116F\u11B8" },
            new string [] { "\uC246", "\uC246", "\u1109\u116F\u11B9", "\uC246", "\u1109\u116F\u11B9" },
            new string [] { "\uC247", "\uC247", "\u1109\u116F\u11BA", "\uC247", "\u1109\u116F\u11BA" },
            new string [] { "\uC248", "\uC248", "\u1109\u116F\u11BB", "\uC248", "\u1109\u116F\u11BB" },
            new string [] { "\uC249", "\uC249", "\u1109\u116F\u11BC", "\uC249", "\u1109\u116F\u11BC" },
            new string [] { "\uC24A", "\uC24A", "\u1109\u116F\u11BD", "\uC24A", "\u1109\u116F\u11BD" },
            new string [] { "\uC24B", "\uC24B", "\u1109\u116F\u11BE", "\uC24B", "\u1109\u116F\u11BE" },
            new string [] { "\uC24C", "\uC24C", "\u1109\u116F\u11BF", "\uC24C", "\u1109\u116F\u11BF" },
            new string [] { "\uC24D", "\uC24D", "\u1109\u116F\u11C0", "\uC24D", "\u1109\u116F\u11C0" },
            new string [] { "\uC24E", "\uC24E", "\u1109\u116F\u11C1", "\uC24E", "\u1109\u116F\u11C1" },
            new string [] { "\uC24F", "\uC24F", "\u1109\u116F\u11C2", "\uC24F", "\u1109\u116F\u11C2" },
            new string [] { "\uC250", "\uC250", "\u1109\u1170", "\uC250", "\u1109\u1170" },
            new string [] { "\uC251", "\uC251", "\u1109\u1170\u11A8", "\uC251", "\u1109\u1170\u11A8" },
            new string [] { "\uC252", "\uC252", "\u1109\u1170\u11A9", "\uC252", "\u1109\u1170\u11A9" },
            new string [] { "\uC253", "\uC253", "\u1109\u1170\u11AA", "\uC253", "\u1109\u1170\u11AA" },
            new string [] { "\uC254", "\uC254", "\u1109\u1170\u11AB", "\uC254", "\u1109\u1170\u11AB" },
            new string [] { "\uC255", "\uC255", "\u1109\u1170\u11AC", "\uC255", "\u1109\u1170\u11AC" },
            new string [] { "\uC256", "\uC256", "\u1109\u1170\u11AD", "\uC256", "\u1109\u1170\u11AD" },
            new string [] { "\uC257", "\uC257", "\u1109\u1170\u11AE", "\uC257", "\u1109\u1170\u11AE" },
            new string [] { "\uC258", "\uC258", "\u1109\u1170\u11AF", "\uC258", "\u1109\u1170\u11AF" },
            new string [] { "\uC259", "\uC259", "\u1109\u1170\u11B0", "\uC259", "\u1109\u1170\u11B0" },
            new string [] { "\uC25A", "\uC25A", "\u1109\u1170\u11B1", "\uC25A", "\u1109\u1170\u11B1" },
            new string [] { "\uC25B", "\uC25B", "\u1109\u1170\u11B2", "\uC25B", "\u1109\u1170\u11B2" },
            new string [] { "\uC25C", "\uC25C", "\u1109\u1170\u11B3", "\uC25C", "\u1109\u1170\u11B3" },
            new string [] { "\uC25D", "\uC25D", "\u1109\u1170\u11B4", "\uC25D", "\u1109\u1170\u11B4" },
            new string [] { "\uC25E", "\uC25E", "\u1109\u1170\u11B5", "\uC25E", "\u1109\u1170\u11B5" },
            new string [] { "\uC25F", "\uC25F", "\u1109\u1170\u11B6", "\uC25F", "\u1109\u1170\u11B6" },
            new string [] { "\uC260", "\uC260", "\u1109\u1170\u11B7", "\uC260", "\u1109\u1170\u11B7" },
            new string [] { "\uC261", "\uC261", "\u1109\u1170\u11B8", "\uC261", "\u1109\u1170\u11B8" },
            new string [] { "\uC262", "\uC262", "\u1109\u1170\u11B9", "\uC262", "\u1109\u1170\u11B9" },
            new string [] { "\uC263", "\uC263", "\u1109\u1170\u11BA", "\uC263", "\u1109\u1170\u11BA" },
            new string [] { "\uC264", "\uC264", "\u1109\u1170\u11BB", "\uC264", "\u1109\u1170\u11BB" },
            new string [] { "\uC265", "\uC265", "\u1109\u1170\u11BC", "\uC265", "\u1109\u1170\u11BC" },
            new string [] { "\uC266", "\uC266", "\u1109\u1170\u11BD", "\uC266", "\u1109\u1170\u11BD" },
            new string [] { "\uC267", "\uC267", "\u1109\u1170\u11BE", "\uC267", "\u1109\u1170\u11BE" },
            new string [] { "\uC268", "\uC268", "\u1109\u1170\u11BF", "\uC268", "\u1109\u1170\u11BF" },
            new string [] { "\uC269", "\uC269", "\u1109\u1170\u11C0", "\uC269", "\u1109\u1170\u11C0" },
            new string [] { "\uC26A", "\uC26A", "\u1109\u1170\u11C1", "\uC26A", "\u1109\u1170\u11C1" },
            new string [] { "\uC26B", "\uC26B", "\u1109\u1170\u11C2", "\uC26B", "\u1109\u1170\u11C2" },
            new string [] { "\uC26C", "\uC26C", "\u1109\u1171", "\uC26C", "\u1109\u1171" },
            new string [] { "\uC26D", "\uC26D", "\u1109\u1171\u11A8", "\uC26D", "\u1109\u1171\u11A8" },
            new string [] { "\uC26E", "\uC26E", "\u1109\u1171\u11A9", "\uC26E", "\u1109\u1171\u11A9" },
            new string [] { "\uC26F", "\uC26F", "\u1109\u1171\u11AA", "\uC26F", "\u1109\u1171\u11AA" },
            new string [] { "\uC270", "\uC270", "\u1109\u1171\u11AB", "\uC270", "\u1109\u1171\u11AB" },
            new string [] { "\uC271", "\uC271", "\u1109\u1171\u11AC", "\uC271", "\u1109\u1171\u11AC" },
            new string [] { "\uC272", "\uC272", "\u1109\u1171\u11AD", "\uC272", "\u1109\u1171\u11AD" },
            new string [] { "\uC273", "\uC273", "\u1109\u1171\u11AE", "\uC273", "\u1109\u1171\u11AE" },
            new string [] { "\uC274", "\uC274", "\u1109\u1171\u11AF", "\uC274", "\u1109\u1171\u11AF" },
            new string [] { "\uC275", "\uC275", "\u1109\u1171\u11B0", "\uC275", "\u1109\u1171\u11B0" },
            new string [] { "\uC276", "\uC276", "\u1109\u1171\u11B1", "\uC276", "\u1109\u1171\u11B1" },
            new string [] { "\uC277", "\uC277", "\u1109\u1171\u11B2", "\uC277", "\u1109\u1171\u11B2" },
            new string [] { "\uC278", "\uC278", "\u1109\u1171\u11B3", "\uC278", "\u1109\u1171\u11B3" },
            new string [] { "\uC279", "\uC279", "\u1109\u1171\u11B4", "\uC279", "\u1109\u1171\u11B4" },
            new string [] { "\uC27A", "\uC27A", "\u1109\u1171\u11B5", "\uC27A", "\u1109\u1171\u11B5" },
            new string [] { "\uC27B", "\uC27B", "\u1109\u1171\u11B6", "\uC27B", "\u1109\u1171\u11B6" },
            new string [] { "\uC27C", "\uC27C", "\u1109\u1171\u11B7", "\uC27C", "\u1109\u1171\u11B7" },
            new string [] { "\uC27D", "\uC27D", "\u1109\u1171\u11B8", "\uC27D", "\u1109\u1171\u11B8" },
            new string [] { "\uC27E", "\uC27E", "\u1109\u1171\u11B9", "\uC27E", "\u1109\u1171\u11B9" },
            new string [] { "\uC27F", "\uC27F", "\u1109\u1171\u11BA", "\uC27F", "\u1109\u1171\u11BA" },
            new string [] { "\uC280", "\uC280", "\u1109\u1171\u11BB", "\uC280", "\u1109\u1171\u11BB" },
            new string [] { "\uC281", "\uC281", "\u1109\u1171\u11BC", "\uC281", "\u1109\u1171\u11BC" },
            new string [] { "\uC282", "\uC282", "\u1109\u1171\u11BD", "\uC282", "\u1109\u1171\u11BD" },
            new string [] { "\uC283", "\uC283", "\u1109\u1171\u11BE", "\uC283", "\u1109\u1171\u11BE" },
            new string [] { "\uC284", "\uC284", "\u1109\u1171\u11BF", "\uC284", "\u1109\u1171\u11BF" },
            new string [] { "\uC285", "\uC285", "\u1109\u1171\u11C0", "\uC285", "\u1109\u1171\u11C0" },
            new string [] { "\uC286", "\uC286", "\u1109\u1171\u11C1", "\uC286", "\u1109\u1171\u11C1" },
            new string [] { "\uC287", "\uC287", "\u1109\u1171\u11C2", "\uC287", "\u1109\u1171\u11C2" },
            new string [] { "\uC288", "\uC288", "\u1109\u1172", "\uC288", "\u1109\u1172" },
            new string [] { "\uC289", "\uC289", "\u1109\u1172\u11A8", "\uC289", "\u1109\u1172\u11A8" },
            new string [] { "\uC28A", "\uC28A", "\u1109\u1172\u11A9", "\uC28A", "\u1109\u1172\u11A9" },
            new string [] { "\uC28B", "\uC28B", "\u1109\u1172\u11AA", "\uC28B", "\u1109\u1172\u11AA" },
            new string [] { "\uC28C", "\uC28C", "\u1109\u1172\u11AB", "\uC28C", "\u1109\u1172\u11AB" },
            new string [] { "\uC28D", "\uC28D", "\u1109\u1172\u11AC", "\uC28D", "\u1109\u1172\u11AC" },
            new string [] { "\uC28E", "\uC28E", "\u1109\u1172\u11AD", "\uC28E", "\u1109\u1172\u11AD" },
            new string [] { "\uC28F", "\uC28F", "\u1109\u1172\u11AE", "\uC28F", "\u1109\u1172\u11AE" },
            new string [] { "\uC290", "\uC290", "\u1109\u1172\u11AF", "\uC290", "\u1109\u1172\u11AF" },
            new string [] { "\uC291", "\uC291", "\u1109\u1172\u11B0", "\uC291", "\u1109\u1172\u11B0" },
            new string [] { "\uC292", "\uC292", "\u1109\u1172\u11B1", "\uC292", "\u1109\u1172\u11B1" },
            new string [] { "\uC293", "\uC293", "\u1109\u1172\u11B2", "\uC293", "\u1109\u1172\u11B2" },
            new string [] { "\uC294", "\uC294", "\u1109\u1172\u11B3", "\uC294", "\u1109\u1172\u11B3" },
            new string [] { "\uC295", "\uC295", "\u1109\u1172\u11B4", "\uC295", "\u1109\u1172\u11B4" },
            new string [] { "\uC296", "\uC296", "\u1109\u1172\u11B5", "\uC296", "\u1109\u1172\u11B5" },
            new string [] { "\uC297", "\uC297", "\u1109\u1172\u11B6", "\uC297", "\u1109\u1172\u11B6" },
            new string [] { "\uC298", "\uC298", "\u1109\u1172\u11B7", "\uC298", "\u1109\u1172\u11B7" },
            new string [] { "\uC299", "\uC299", "\u1109\u1172\u11B8", "\uC299", "\u1109\u1172\u11B8" },
            new string [] { "\uC29A", "\uC29A", "\u1109\u1172\u11B9", "\uC29A", "\u1109\u1172\u11B9" },
            new string [] { "\uC29B", "\uC29B", "\u1109\u1172\u11BA", "\uC29B", "\u1109\u1172\u11BA" },
            new string [] { "\uC29C", "\uC29C", "\u1109\u1172\u11BB", "\uC29C", "\u1109\u1172\u11BB" },
            new string [] { "\uC29D", "\uC29D", "\u1109\u1172\u11BC", "\uC29D", "\u1109\u1172\u11BC" },
            new string [] { "\uC29E", "\uC29E", "\u1109\u1172\u11BD", "\uC29E", "\u1109\u1172\u11BD" },
            new string [] { "\uC29F", "\uC29F", "\u1109\u1172\u11BE", "\uC29F", "\u1109\u1172\u11BE" },
            new string [] { "\uC2A0", "\uC2A0", "\u1109\u1172\u11BF", "\uC2A0", "\u1109\u1172\u11BF" },
            new string [] { "\uC2A1", "\uC2A1", "\u1109\u1172\u11C0", "\uC2A1", "\u1109\u1172\u11C0" },
            new string [] { "\uC2A2", "\uC2A2", "\u1109\u1172\u11C1", "\uC2A2", "\u1109\u1172\u11C1" },
            new string [] { "\uC2A3", "\uC2A3", "\u1109\u1172\u11C2", "\uC2A3", "\u1109\u1172\u11C2" },
            new string [] { "\uC2A4", "\uC2A4", "\u1109\u1173", "\uC2A4", "\u1109\u1173" },
            new string [] { "\uC2A5", "\uC2A5", "\u1109\u1173\u11A8", "\uC2A5", "\u1109\u1173\u11A8" },
            new string [] { "\uC2A6", "\uC2A6", "\u1109\u1173\u11A9", "\uC2A6", "\u1109\u1173\u11A9" },
            new string [] { "\uC2A7", "\uC2A7", "\u1109\u1173\u11AA", "\uC2A7", "\u1109\u1173\u11AA" },
            new string [] { "\uC2A8", "\uC2A8", "\u1109\u1173\u11AB", "\uC2A8", "\u1109\u1173\u11AB" },
            new string [] { "\uC2A9", "\uC2A9", "\u1109\u1173\u11AC", "\uC2A9", "\u1109\u1173\u11AC" },
            new string [] { "\uC2AA", "\uC2AA", "\u1109\u1173\u11AD", "\uC2AA", "\u1109\u1173\u11AD" },
            new string [] { "\uC2AB", "\uC2AB", "\u1109\u1173\u11AE", "\uC2AB", "\u1109\u1173\u11AE" },
            new string [] { "\uC2AC", "\uC2AC", "\u1109\u1173\u11AF", "\uC2AC", "\u1109\u1173\u11AF" },
            new string [] { "\uC2AD", "\uC2AD", "\u1109\u1173\u11B0", "\uC2AD", "\u1109\u1173\u11B0" },
            new string [] { "\uC2AE", "\uC2AE", "\u1109\u1173\u11B1", "\uC2AE", "\u1109\u1173\u11B1" },
            new string [] { "\uC2AF", "\uC2AF", "\u1109\u1173\u11B2", "\uC2AF", "\u1109\u1173\u11B2" },
            new string [] { "\uC2B0", "\uC2B0", "\u1109\u1173\u11B3", "\uC2B0", "\u1109\u1173\u11B3" },
            new string [] { "\uC2B1", "\uC2B1", "\u1109\u1173\u11B4", "\uC2B1", "\u1109\u1173\u11B4" },
            new string [] { "\uC2B2", "\uC2B2", "\u1109\u1173\u11B5", "\uC2B2", "\u1109\u1173\u11B5" },
            new string [] { "\uC2B3", "\uC2B3", "\u1109\u1173\u11B6", "\uC2B3", "\u1109\u1173\u11B6" },
            new string [] { "\uC2B4", "\uC2B4", "\u1109\u1173\u11B7", "\uC2B4", "\u1109\u1173\u11B7" },
            new string [] { "\uC2B5", "\uC2B5", "\u1109\u1173\u11B8", "\uC2B5", "\u1109\u1173\u11B8" },
            new string [] { "\uC2B6", "\uC2B6", "\u1109\u1173\u11B9", "\uC2B6", "\u1109\u1173\u11B9" },
            new string [] { "\uC2B7", "\uC2B7", "\u1109\u1173\u11BA", "\uC2B7", "\u1109\u1173\u11BA" },
            new string [] { "\uC2B8", "\uC2B8", "\u1109\u1173\u11BB", "\uC2B8", "\u1109\u1173\u11BB" },
            new string [] { "\uC2B9", "\uC2B9", "\u1109\u1173\u11BC", "\uC2B9", "\u1109\u1173\u11BC" },
            new string [] { "\uC2BA", "\uC2BA", "\u1109\u1173\u11BD", "\uC2BA", "\u1109\u1173\u11BD" },
            new string [] { "\uC2BB", "\uC2BB", "\u1109\u1173\u11BE", "\uC2BB", "\u1109\u1173\u11BE" },
            new string [] { "\uC2BC", "\uC2BC", "\u1109\u1173\u11BF", "\uC2BC", "\u1109\u1173\u11BF" },
            new string [] { "\uC2BD", "\uC2BD", "\u1109\u1173\u11C0", "\uC2BD", "\u1109\u1173\u11C0" },
            new string [] { "\uC2BE", "\uC2BE", "\u1109\u1173\u11C1", "\uC2BE", "\u1109\u1173\u11C1" },
            new string [] { "\uC2BF", "\uC2BF", "\u1109\u1173\u11C2", "\uC2BF", "\u1109\u1173\u11C2" },
            new string [] { "\uC2C0", "\uC2C0", "\u1109\u1174", "\uC2C0", "\u1109\u1174" },
            new string [] { "\uC2C1", "\uC2C1", "\u1109\u1174\u11A8", "\uC2C1", "\u1109\u1174\u11A8" },
            new string [] { "\uC2C2", "\uC2C2", "\u1109\u1174\u11A9", "\uC2C2", "\u1109\u1174\u11A9" },
            new string [] { "\uC2C3", "\uC2C3", "\u1109\u1174\u11AA", "\uC2C3", "\u1109\u1174\u11AA" },
            new string [] { "\uC2C4", "\uC2C4", "\u1109\u1174\u11AB", "\uC2C4", "\u1109\u1174\u11AB" },
            new string [] { "\uC2C5", "\uC2C5", "\u1109\u1174\u11AC", "\uC2C5", "\u1109\u1174\u11AC" },
            new string [] { "\uC2C6", "\uC2C6", "\u1109\u1174\u11AD", "\uC2C6", "\u1109\u1174\u11AD" },
            new string [] { "\uC2C7", "\uC2C7", "\u1109\u1174\u11AE", "\uC2C7", "\u1109\u1174\u11AE" },
            new string [] { "\uC2C8", "\uC2C8", "\u1109\u1174\u11AF", "\uC2C8", "\u1109\u1174\u11AF" },
            new string [] { "\uC2C9", "\uC2C9", "\u1109\u1174\u11B0", "\uC2C9", "\u1109\u1174\u11B0" },
            new string [] { "\uC2CA", "\uC2CA", "\u1109\u1174\u11B1", "\uC2CA", "\u1109\u1174\u11B1" },
            new string [] { "\uC2CB", "\uC2CB", "\u1109\u1174\u11B2", "\uC2CB", "\u1109\u1174\u11B2" },
            new string [] { "\uC2CC", "\uC2CC", "\u1109\u1174\u11B3", "\uC2CC", "\u1109\u1174\u11B3" },
            new string [] { "\uC2CD", "\uC2CD", "\u1109\u1174\u11B4", "\uC2CD", "\u1109\u1174\u11B4" },
            new string [] { "\uC2CE", "\uC2CE", "\u1109\u1174\u11B5", "\uC2CE", "\u1109\u1174\u11B5" },
            new string [] { "\uC2CF", "\uC2CF", "\u1109\u1174\u11B6", "\uC2CF", "\u1109\u1174\u11B6" },
            new string [] { "\uC2D0", "\uC2D0", "\u1109\u1174\u11B7", "\uC2D0", "\u1109\u1174\u11B7" },
            new string [] { "\uC2D1", "\uC2D1", "\u1109\u1174\u11B8", "\uC2D1", "\u1109\u1174\u11B8" },
            new string [] { "\uC2D2", "\uC2D2", "\u1109\u1174\u11B9", "\uC2D2", "\u1109\u1174\u11B9" },
            new string [] { "\uC2D3", "\uC2D3", "\u1109\u1174\u11BA", "\uC2D3", "\u1109\u1174\u11BA" },
            new string [] { "\uC2D4", "\uC2D4", "\u1109\u1174\u11BB", "\uC2D4", "\u1109\u1174\u11BB" },
            new string [] { "\uC2D5", "\uC2D5", "\u1109\u1174\u11BC", "\uC2D5", "\u1109\u1174\u11BC" },
            new string [] { "\uC2D6", "\uC2D6", "\u1109\u1174\u11BD", "\uC2D6", "\u1109\u1174\u11BD" },
            new string [] { "\uC2D7", "\uC2D7", "\u1109\u1174\u11BE", "\uC2D7", "\u1109\u1174\u11BE" },
            new string [] { "\uC2D8", "\uC2D8", "\u1109\u1174\u11BF", "\uC2D8", "\u1109\u1174\u11BF" },
            new string [] { "\uC2D9", "\uC2D9", "\u1109\u1174\u11C0", "\uC2D9", "\u1109\u1174\u11C0" },
            new string [] { "\uC2DA", "\uC2DA", "\u1109\u1174\u11C1", "\uC2DA", "\u1109\u1174\u11C1" },
            new string [] { "\uC2DB", "\uC2DB", "\u1109\u1174\u11C2", "\uC2DB", "\u1109\u1174\u11C2" },
            new string [] { "\uC2DC", "\uC2DC", "\u1109\u1175", "\uC2DC", "\u1109\u1175" },
            new string [] { "\uC2DD", "\uC2DD", "\u1109\u1175\u11A8", "\uC2DD", "\u1109\u1175\u11A8" },
            new string [] { "\uC2DE", "\uC2DE", "\u1109\u1175\u11A9", "\uC2DE", "\u1109\u1175\u11A9" },
            new string [] { "\uC2DF", "\uC2DF", "\u1109\u1175\u11AA", "\uC2DF", "\u1109\u1175\u11AA" },
            new string [] { "\uC2E0", "\uC2E0", "\u1109\u1175\u11AB", "\uC2E0", "\u1109\u1175\u11AB" },
            new string [] { "\uC2E1", "\uC2E1", "\u1109\u1175\u11AC", "\uC2E1", "\u1109\u1175\u11AC" },
            new string [] { "\uC2E2", "\uC2E2", "\u1109\u1175\u11AD", "\uC2E2", "\u1109\u1175\u11AD" },
            new string [] { "\uC2E3", "\uC2E3", "\u1109\u1175\u11AE", "\uC2E3", "\u1109\u1175\u11AE" },
            new string [] { "\uC2E4", "\uC2E4", "\u1109\u1175\u11AF", "\uC2E4", "\u1109\u1175\u11AF" },
            new string [] { "\uC2E5", "\uC2E5", "\u1109\u1175\u11B0", "\uC2E5", "\u1109\u1175\u11B0" },
            new string [] { "\uC2E6", "\uC2E6", "\u1109\u1175\u11B1", "\uC2E6", "\u1109\u1175\u11B1" },
            new string [] { "\uC2E7", "\uC2E7", "\u1109\u1175\u11B2", "\uC2E7", "\u1109\u1175\u11B2" },
            new string [] { "\uC2E8", "\uC2E8", "\u1109\u1175\u11B3", "\uC2E8", "\u1109\u1175\u11B3" },
            new string [] { "\uC2E9", "\uC2E9", "\u1109\u1175\u11B4", "\uC2E9", "\u1109\u1175\u11B4" },
            new string [] { "\uC2EA", "\uC2EA", "\u1109\u1175\u11B5", "\uC2EA", "\u1109\u1175\u11B5" },
            new string [] { "\uC2EB", "\uC2EB", "\u1109\u1175\u11B6", "\uC2EB", "\u1109\u1175\u11B6" },
            new string [] { "\uC2EC", "\uC2EC", "\u1109\u1175\u11B7", "\uC2EC", "\u1109\u1175\u11B7" },
            new string [] { "\uC2ED", "\uC2ED", "\u1109\u1175\u11B8", "\uC2ED", "\u1109\u1175\u11B8" },
            new string [] { "\uC2EE", "\uC2EE", "\u1109\u1175\u11B9", "\uC2EE", "\u1109\u1175\u11B9" },
            new string [] { "\uC2EF", "\uC2EF", "\u1109\u1175\u11BA", "\uC2EF", "\u1109\u1175\u11BA" },
            new string [] { "\uC2F0", "\uC2F0", "\u1109\u1175\u11BB", "\uC2F0", "\u1109\u1175\u11BB" },
            new string [] { "\uC2F1", "\uC2F1", "\u1109\u1175\u11BC", "\uC2F1", "\u1109\u1175\u11BC" },
            new string [] { "\uC2F2", "\uC2F2", "\u1109\u1175\u11BD", "\uC2F2", "\u1109\u1175\u11BD" },
            new string [] { "\uC2F3", "\uC2F3", "\u1109\u1175\u11BE", "\uC2F3", "\u1109\u1175\u11BE" },
            new string [] { "\uC2F4", "\uC2F4", "\u1109\u1175\u11BF", "\uC2F4", "\u1109\u1175\u11BF" },
            new string [] { "\uC2F5", "\uC2F5", "\u1109\u1175\u11C0", "\uC2F5", "\u1109\u1175\u11C0" },
            new string [] { "\uC2F6", "\uC2F6", "\u1109\u1175\u11C1", "\uC2F6", "\u1109\u1175\u11C1" },
            new string [] { "\uC2F7", "\uC2F7", "\u1109\u1175\u11C2", "\uC2F7", "\u1109\u1175\u11C2" },
            new string [] { "\uC2F8", "\uC2F8", "\u110A\u1161", "\uC2F8", "\u110A\u1161" },
            new string [] { "\uC2F9", "\uC2F9", "\u110A\u1161\u11A8", "\uC2F9", "\u110A\u1161\u11A8" },
            new string [] { "\uC2FA", "\uC2FA", "\u110A\u1161\u11A9", "\uC2FA", "\u110A\u1161\u11A9" },
            new string [] { "\uC2FB", "\uC2FB", "\u110A\u1161\u11AA", "\uC2FB", "\u110A\u1161\u11AA" },
            new string [] { "\uC2FC", "\uC2FC", "\u110A\u1161\u11AB", "\uC2FC", "\u110A\u1161\u11AB" },
            new string [] { "\uC2FD", "\uC2FD", "\u110A\u1161\u11AC", "\uC2FD", "\u110A\u1161\u11AC" },
            new string [] { "\uC2FE", "\uC2FE", "\u110A\u1161\u11AD", "\uC2FE", "\u110A\u1161\u11AD" },
            new string [] { "\uC2FF", "\uC2FF", "\u110A\u1161\u11AE", "\uC2FF", "\u110A\u1161\u11AE" },
            new string [] { "\uC300", "\uC300", "\u110A\u1161\u11AF", "\uC300", "\u110A\u1161\u11AF" },
            new string [] { "\uC301", "\uC301", "\u110A\u1161\u11B0", "\uC301", "\u110A\u1161\u11B0" },
            new string [] { "\uC302", "\uC302", "\u110A\u1161\u11B1", "\uC302", "\u110A\u1161\u11B1" },
            new string [] { "\uC303", "\uC303", "\u110A\u1161\u11B2", "\uC303", "\u110A\u1161\u11B2" },
            new string [] { "\uC304", "\uC304", "\u110A\u1161\u11B3", "\uC304", "\u110A\u1161\u11B3" },
            new string [] { "\uC305", "\uC305", "\u110A\u1161\u11B4", "\uC305", "\u110A\u1161\u11B4" },
            new string [] { "\uC306", "\uC306", "\u110A\u1161\u11B5", "\uC306", "\u110A\u1161\u11B5" },
            new string [] { "\uC307", "\uC307", "\u110A\u1161\u11B6", "\uC307", "\u110A\u1161\u11B6" },
            new string [] { "\uC308", "\uC308", "\u110A\u1161\u11B7", "\uC308", "\u110A\u1161\u11B7" },
            new string [] { "\uC309", "\uC309", "\u110A\u1161\u11B8", "\uC309", "\u110A\u1161\u11B8" },
            new string [] { "\uC30A", "\uC30A", "\u110A\u1161\u11B9", "\uC30A", "\u110A\u1161\u11B9" },
            new string [] { "\uC30B", "\uC30B", "\u110A\u1161\u11BA", "\uC30B", "\u110A\u1161\u11BA" },
            new string [] { "\uC30C", "\uC30C", "\u110A\u1161\u11BB", "\uC30C", "\u110A\u1161\u11BB" },
            new string [] { "\uC30D", "\uC30D", "\u110A\u1161\u11BC", "\uC30D", "\u110A\u1161\u11BC" },
            new string [] { "\uC30E", "\uC30E", "\u110A\u1161\u11BD", "\uC30E", "\u110A\u1161\u11BD" },
            new string [] { "\uC30F", "\uC30F", "\u110A\u1161\u11BE", "\uC30F", "\u110A\u1161\u11BE" },
            new string [] { "\uC310", "\uC310", "\u110A\u1161\u11BF", "\uC310", "\u110A\u1161\u11BF" },
            new string [] { "\uC311", "\uC311", "\u110A\u1161\u11C0", "\uC311", "\u110A\u1161\u11C0" },
            new string [] { "\uC312", "\uC312", "\u110A\u1161\u11C1", "\uC312", "\u110A\u1161\u11C1" },
            new string [] { "\uC313", "\uC313", "\u110A\u1161\u11C2", "\uC313", "\u110A\u1161\u11C2" },
            new string [] { "\uC314", "\uC314", "\u110A\u1162", "\uC314", "\u110A\u1162" },
            new string [] { "\uC315", "\uC315", "\u110A\u1162\u11A8", "\uC315", "\u110A\u1162\u11A8" },
            new string [] { "\uC316", "\uC316", "\u110A\u1162\u11A9", "\uC316", "\u110A\u1162\u11A9" },
            new string [] { "\uC317", "\uC317", "\u110A\u1162\u11AA", "\uC317", "\u110A\u1162\u11AA" },
            new string [] { "\uC318", "\uC318", "\u110A\u1162\u11AB", "\uC318", "\u110A\u1162\u11AB" },
            new string [] { "\uC319", "\uC319", "\u110A\u1162\u11AC", "\uC319", "\u110A\u1162\u11AC" },
            new string [] { "\uC31A", "\uC31A", "\u110A\u1162\u11AD", "\uC31A", "\u110A\u1162\u11AD" },
            new string [] { "\uC31B", "\uC31B", "\u110A\u1162\u11AE", "\uC31B", "\u110A\u1162\u11AE" },
            new string [] { "\uC31C", "\uC31C", "\u110A\u1162\u11AF", "\uC31C", "\u110A\u1162\u11AF" },
            new string [] { "\uC31D", "\uC31D", "\u110A\u1162\u11B0", "\uC31D", "\u110A\u1162\u11B0" },
            new string [] { "\uC31E", "\uC31E", "\u110A\u1162\u11B1", "\uC31E", "\u110A\u1162\u11B1" },
            new string [] { "\uC31F", "\uC31F", "\u110A\u1162\u11B2", "\uC31F", "\u110A\u1162\u11B2" },
            new string [] { "\uC320", "\uC320", "\u110A\u1162\u11B3", "\uC320", "\u110A\u1162\u11B3" },
            new string [] { "\uC321", "\uC321", "\u110A\u1162\u11B4", "\uC321", "\u110A\u1162\u11B4" },
            new string [] { "\uC322", "\uC322", "\u110A\u1162\u11B5", "\uC322", "\u110A\u1162\u11B5" },
            new string [] { "\uC323", "\uC323", "\u110A\u1162\u11B6", "\uC323", "\u110A\u1162\u11B6" },
            new string [] { "\uC324", "\uC324", "\u110A\u1162\u11B7", "\uC324", "\u110A\u1162\u11B7" },
            new string [] { "\uC325", "\uC325", "\u110A\u1162\u11B8", "\uC325", "\u110A\u1162\u11B8" },
            new string [] { "\uC326", "\uC326", "\u110A\u1162\u11B9", "\uC326", "\u110A\u1162\u11B9" },
            new string [] { "\uC327", "\uC327", "\u110A\u1162\u11BA", "\uC327", "\u110A\u1162\u11BA" },
            new string [] { "\uC328", "\uC328", "\u110A\u1162\u11BB", "\uC328", "\u110A\u1162\u11BB" },
            new string [] { "\uC329", "\uC329", "\u110A\u1162\u11BC", "\uC329", "\u110A\u1162\u11BC" },
            new string [] { "\uC32A", "\uC32A", "\u110A\u1162\u11BD", "\uC32A", "\u110A\u1162\u11BD" },
            new string [] { "\uC32B", "\uC32B", "\u110A\u1162\u11BE", "\uC32B", "\u110A\u1162\u11BE" },
            new string [] { "\uC32C", "\uC32C", "\u110A\u1162\u11BF", "\uC32C", "\u110A\u1162\u11BF" },
            new string [] { "\uC32D", "\uC32D", "\u110A\u1162\u11C0", "\uC32D", "\u110A\u1162\u11C0" },
            new string [] { "\uC32E", "\uC32E", "\u110A\u1162\u11C1", "\uC32E", "\u110A\u1162\u11C1" },
            new string [] { "\uC32F", "\uC32F", "\u110A\u1162\u11C2", "\uC32F", "\u110A\u1162\u11C2" },
            new string [] { "\uC330", "\uC330", "\u110A\u1163", "\uC330", "\u110A\u1163" },
            new string [] { "\uC331", "\uC331", "\u110A\u1163\u11A8", "\uC331", "\u110A\u1163\u11A8" },
            new string [] { "\uC332", "\uC332", "\u110A\u1163\u11A9", "\uC332", "\u110A\u1163\u11A9" },
            new string [] { "\uC333", "\uC333", "\u110A\u1163\u11AA", "\uC333", "\u110A\u1163\u11AA" },
            new string [] { "\uC334", "\uC334", "\u110A\u1163\u11AB", "\uC334", "\u110A\u1163\u11AB" },
            new string [] { "\uC335", "\uC335", "\u110A\u1163\u11AC", "\uC335", "\u110A\u1163\u11AC" },
            new string [] { "\uC336", "\uC336", "\u110A\u1163\u11AD", "\uC336", "\u110A\u1163\u11AD" },
            new string [] { "\uC337", "\uC337", "\u110A\u1163\u11AE", "\uC337", "\u110A\u1163\u11AE" },
            new string [] { "\uC338", "\uC338", "\u110A\u1163\u11AF", "\uC338", "\u110A\u1163\u11AF" },
            new string [] { "\uC339", "\uC339", "\u110A\u1163\u11B0", "\uC339", "\u110A\u1163\u11B0" },
            new string [] { "\uC33A", "\uC33A", "\u110A\u1163\u11B1", "\uC33A", "\u110A\u1163\u11B1" },
            new string [] { "\uC33B", "\uC33B", "\u110A\u1163\u11B2", "\uC33B", "\u110A\u1163\u11B2" },
            new string [] { "\uC33C", "\uC33C", "\u110A\u1163\u11B3", "\uC33C", "\u110A\u1163\u11B3" },
            new string [] { "\uC33D", "\uC33D", "\u110A\u1163\u11B4", "\uC33D", "\u110A\u1163\u11B4" },
            new string [] { "\uC33E", "\uC33E", "\u110A\u1163\u11B5", "\uC33E", "\u110A\u1163\u11B5" },
            new string [] { "\uC33F", "\uC33F", "\u110A\u1163\u11B6", "\uC33F", "\u110A\u1163\u11B6" },
            new string [] { "\uC340", "\uC340", "\u110A\u1163\u11B7", "\uC340", "\u110A\u1163\u11B7" },
            new string [] { "\uC341", "\uC341", "\u110A\u1163\u11B8", "\uC341", "\u110A\u1163\u11B8" },
            new string [] { "\uC342", "\uC342", "\u110A\u1163\u11B9", "\uC342", "\u110A\u1163\u11B9" },
            new string [] { "\uC343", "\uC343", "\u110A\u1163\u11BA", "\uC343", "\u110A\u1163\u11BA" },
            new string [] { "\uC344", "\uC344", "\u110A\u1163\u11BB", "\uC344", "\u110A\u1163\u11BB" },
            new string [] { "\uC345", "\uC345", "\u110A\u1163\u11BC", "\uC345", "\u110A\u1163\u11BC" },
            new string [] { "\uC346", "\uC346", "\u110A\u1163\u11BD", "\uC346", "\u110A\u1163\u11BD" },
            new string [] { "\uC347", "\uC347", "\u110A\u1163\u11BE", "\uC347", "\u110A\u1163\u11BE" },
            new string [] { "\uC348", "\uC348", "\u110A\u1163\u11BF", "\uC348", "\u110A\u1163\u11BF" },
            new string [] { "\uC349", "\uC349", "\u110A\u1163\u11C0", "\uC349", "\u110A\u1163\u11C0" },
            new string [] { "\uC34A", "\uC34A", "\u110A\u1163\u11C1", "\uC34A", "\u110A\u1163\u11C1" },
            new string [] { "\uC34B", "\uC34B", "\u110A\u1163\u11C2", "\uC34B", "\u110A\u1163\u11C2" },
            new string [] { "\uC34C", "\uC34C", "\u110A\u1164", "\uC34C", "\u110A\u1164" },
            new string [] { "\uC34D", "\uC34D", "\u110A\u1164\u11A8", "\uC34D", "\u110A\u1164\u11A8" },
            new string [] { "\uC34E", "\uC34E", "\u110A\u1164\u11A9", "\uC34E", "\u110A\u1164\u11A9" },
            new string [] { "\uC34F", "\uC34F", "\u110A\u1164\u11AA", "\uC34F", "\u110A\u1164\u11AA" },
            new string [] { "\uC350", "\uC350", "\u110A\u1164\u11AB", "\uC350", "\u110A\u1164\u11AB" },
            new string [] { "\uC351", "\uC351", "\u110A\u1164\u11AC", "\uC351", "\u110A\u1164\u11AC" },
            new string [] { "\uC352", "\uC352", "\u110A\u1164\u11AD", "\uC352", "\u110A\u1164\u11AD" },
            new string [] { "\uC353", "\uC353", "\u110A\u1164\u11AE", "\uC353", "\u110A\u1164\u11AE" },
            new string [] { "\uC354", "\uC354", "\u110A\u1164\u11AF", "\uC354", "\u110A\u1164\u11AF" },
            new string [] { "\uC355", "\uC355", "\u110A\u1164\u11B0", "\uC355", "\u110A\u1164\u11B0" },
            new string [] { "\uC356", "\uC356", "\u110A\u1164\u11B1", "\uC356", "\u110A\u1164\u11B1" },
            new string [] { "\uC357", "\uC357", "\u110A\u1164\u11B2", "\uC357", "\u110A\u1164\u11B2" },
            new string [] { "\uC358", "\uC358", "\u110A\u1164\u11B3", "\uC358", "\u110A\u1164\u11B3" },
            new string [] { "\uC359", "\uC359", "\u110A\u1164\u11B4", "\uC359", "\u110A\u1164\u11B4" },
            new string [] { "\uC35A", "\uC35A", "\u110A\u1164\u11B5", "\uC35A", "\u110A\u1164\u11B5" },
            new string [] { "\uC35B", "\uC35B", "\u110A\u1164\u11B6", "\uC35B", "\u110A\u1164\u11B6" },
            new string [] { "\uC35C", "\uC35C", "\u110A\u1164\u11B7", "\uC35C", "\u110A\u1164\u11B7" },
            new string [] { "\uC35D", "\uC35D", "\u110A\u1164\u11B8", "\uC35D", "\u110A\u1164\u11B8" },
            new string [] { "\uC35E", "\uC35E", "\u110A\u1164\u11B9", "\uC35E", "\u110A\u1164\u11B9" },
            new string [] { "\uC35F", "\uC35F", "\u110A\u1164\u11BA", "\uC35F", "\u110A\u1164\u11BA" },
            new string [] { "\uC360", "\uC360", "\u110A\u1164\u11BB", "\uC360", "\u110A\u1164\u11BB" },
            new string [] { "\uC361", "\uC361", "\u110A\u1164\u11BC", "\uC361", "\u110A\u1164\u11BC" },
            new string [] { "\uC362", "\uC362", "\u110A\u1164\u11BD", "\uC362", "\u110A\u1164\u11BD" },
            new string [] { "\uC363", "\uC363", "\u110A\u1164\u11BE", "\uC363", "\u110A\u1164\u11BE" },
            new string [] { "\uC364", "\uC364", "\u110A\u1164\u11BF", "\uC364", "\u110A\u1164\u11BF" },
            new string [] { "\uC365", "\uC365", "\u110A\u1164\u11C0", "\uC365", "\u110A\u1164\u11C0" },
            new string [] { "\uC366", "\uC366", "\u110A\u1164\u11C1", "\uC366", "\u110A\u1164\u11C1" },
            new string [] { "\uC367", "\uC367", "\u110A\u1164\u11C2", "\uC367", "\u110A\u1164\u11C2" },
            new string [] { "\uC368", "\uC368", "\u110A\u1165", "\uC368", "\u110A\u1165" },
            new string [] { "\uC369", "\uC369", "\u110A\u1165\u11A8", "\uC369", "\u110A\u1165\u11A8" },
            new string [] { "\uC36A", "\uC36A", "\u110A\u1165\u11A9", "\uC36A", "\u110A\u1165\u11A9" },
            new string [] { "\uC36B", "\uC36B", "\u110A\u1165\u11AA", "\uC36B", "\u110A\u1165\u11AA" },
            new string [] { "\uC36C", "\uC36C", "\u110A\u1165\u11AB", "\uC36C", "\u110A\u1165\u11AB" },
            new string [] { "\uC36D", "\uC36D", "\u110A\u1165\u11AC", "\uC36D", "\u110A\u1165\u11AC" },
            new string [] { "\uC36E", "\uC36E", "\u110A\u1165\u11AD", "\uC36E", "\u110A\u1165\u11AD" },
            new string [] { "\uC36F", "\uC36F", "\u110A\u1165\u11AE", "\uC36F", "\u110A\u1165\u11AE" },
            new string [] { "\uC370", "\uC370", "\u110A\u1165\u11AF", "\uC370", "\u110A\u1165\u11AF" },
            new string [] { "\uC371", "\uC371", "\u110A\u1165\u11B0", "\uC371", "\u110A\u1165\u11B0" },
            new string [] { "\uC372", "\uC372", "\u110A\u1165\u11B1", "\uC372", "\u110A\u1165\u11B1" },
            new string [] { "\uC373", "\uC373", "\u110A\u1165\u11B2", "\uC373", "\u110A\u1165\u11B2" },
            new string [] { "\uC374", "\uC374", "\u110A\u1165\u11B3", "\uC374", "\u110A\u1165\u11B3" },
            new string [] { "\uC375", "\uC375", "\u110A\u1165\u11B4", "\uC375", "\u110A\u1165\u11B4" },
            new string [] { "\uC376", "\uC376", "\u110A\u1165\u11B5", "\uC376", "\u110A\u1165\u11B5" },
            new string [] { "\uC377", "\uC377", "\u110A\u1165\u11B6", "\uC377", "\u110A\u1165\u11B6" },
            new string [] { "\uC378", "\uC378", "\u110A\u1165\u11B7", "\uC378", "\u110A\u1165\u11B7" },
            new string [] { "\uC379", "\uC379", "\u110A\u1165\u11B8", "\uC379", "\u110A\u1165\u11B8" },
            new string [] { "\uC37A", "\uC37A", "\u110A\u1165\u11B9", "\uC37A", "\u110A\u1165\u11B9" },
            new string [] { "\uC37B", "\uC37B", "\u110A\u1165\u11BA", "\uC37B", "\u110A\u1165\u11BA" },
            new string [] { "\uC37C", "\uC37C", "\u110A\u1165\u11BB", "\uC37C", "\u110A\u1165\u11BB" },
            new string [] { "\uC37D", "\uC37D", "\u110A\u1165\u11BC", "\uC37D", "\u110A\u1165\u11BC" },
            new string [] { "\uC37E", "\uC37E", "\u110A\u1165\u11BD", "\uC37E", "\u110A\u1165\u11BD" },
            new string [] { "\uC37F", "\uC37F", "\u110A\u1165\u11BE", "\uC37F", "\u110A\u1165\u11BE" },
            new string [] { "\uC380", "\uC380", "\u110A\u1165\u11BF", "\uC380", "\u110A\u1165\u11BF" },
            new string [] { "\uC381", "\uC381", "\u110A\u1165\u11C0", "\uC381", "\u110A\u1165\u11C0" },
            new string [] { "\uC382", "\uC382", "\u110A\u1165\u11C1", "\uC382", "\u110A\u1165\u11C1" },
            new string [] { "\uC383", "\uC383", "\u110A\u1165\u11C2", "\uC383", "\u110A\u1165\u11C2" },
            new string [] { "\uC384", "\uC384", "\u110A\u1166", "\uC384", "\u110A\u1166" },
            new string [] { "\uC385", "\uC385", "\u110A\u1166\u11A8", "\uC385", "\u110A\u1166\u11A8" },
            new string [] { "\uC386", "\uC386", "\u110A\u1166\u11A9", "\uC386", "\u110A\u1166\u11A9" },
            new string [] { "\uC387", "\uC387", "\u110A\u1166\u11AA", "\uC387", "\u110A\u1166\u11AA" },
            new string [] { "\uC388", "\uC388", "\u110A\u1166\u11AB", "\uC388", "\u110A\u1166\u11AB" },
            new string [] { "\uC389", "\uC389", "\u110A\u1166\u11AC", "\uC389", "\u110A\u1166\u11AC" },
            new string [] { "\uC38A", "\uC38A", "\u110A\u1166\u11AD", "\uC38A", "\u110A\u1166\u11AD" },
            new string [] { "\uC38B", "\uC38B", "\u110A\u1166\u11AE", "\uC38B", "\u110A\u1166\u11AE" },
            new string [] { "\uC38C", "\uC38C", "\u110A\u1166\u11AF", "\uC38C", "\u110A\u1166\u11AF" },
            new string [] { "\uC38D", "\uC38D", "\u110A\u1166\u11B0", "\uC38D", "\u110A\u1166\u11B0" },
            new string [] { "\uC38E", "\uC38E", "\u110A\u1166\u11B1", "\uC38E", "\u110A\u1166\u11B1" },
            new string [] { "\uC38F", "\uC38F", "\u110A\u1166\u11B2", "\uC38F", "\u110A\u1166\u11B2" },
            new string [] { "\uC390", "\uC390", "\u110A\u1166\u11B3", "\uC390", "\u110A\u1166\u11B3" },
            new string [] { "\uC391", "\uC391", "\u110A\u1166\u11B4", "\uC391", "\u110A\u1166\u11B4" },
            new string [] { "\uC392", "\uC392", "\u110A\u1166\u11B5", "\uC392", "\u110A\u1166\u11B5" },
            new string [] { "\uC393", "\uC393", "\u110A\u1166\u11B6", "\uC393", "\u110A\u1166\u11B6" },
            new string [] { "\uC394", "\uC394", "\u110A\u1166\u11B7", "\uC394", "\u110A\u1166\u11B7" },
            new string [] { "\uC395", "\uC395", "\u110A\u1166\u11B8", "\uC395", "\u110A\u1166\u11B8" },
            new string [] { "\uC396", "\uC396", "\u110A\u1166\u11B9", "\uC396", "\u110A\u1166\u11B9" },
            new string [] { "\uC397", "\uC397", "\u110A\u1166\u11BA", "\uC397", "\u110A\u1166\u11BA" },
            new string [] { "\uC398", "\uC398", "\u110A\u1166\u11BB", "\uC398", "\u110A\u1166\u11BB" },
            new string [] { "\uC399", "\uC399", "\u110A\u1166\u11BC", "\uC399", "\u110A\u1166\u11BC" },
            new string [] { "\uC39A", "\uC39A", "\u110A\u1166\u11BD", "\uC39A", "\u110A\u1166\u11BD" },
            new string [] { "\uC39B", "\uC39B", "\u110A\u1166\u11BE", "\uC39B", "\u110A\u1166\u11BE" },
            new string [] { "\uC39C", "\uC39C", "\u110A\u1166\u11BF", "\uC39C", "\u110A\u1166\u11BF" },
            new string [] { "\uC39D", "\uC39D", "\u110A\u1166\u11C0", "\uC39D", "\u110A\u1166\u11C0" },
            new string [] { "\uC39E", "\uC39E", "\u110A\u1166\u11C1", "\uC39E", "\u110A\u1166\u11C1" },
            new string [] { "\uC39F", "\uC39F", "\u110A\u1166\u11C2", "\uC39F", "\u110A\u1166\u11C2" },
            new string [] { "\uC3A0", "\uC3A0", "\u110A\u1167", "\uC3A0", "\u110A\u1167" },
            new string [] { "\uC3A1", "\uC3A1", "\u110A\u1167\u11A8", "\uC3A1", "\u110A\u1167\u11A8" },
            new string [] { "\uC3A2", "\uC3A2", "\u110A\u1167\u11A9", "\uC3A2", "\u110A\u1167\u11A9" },
            new string [] { "\uC3A3", "\uC3A3", "\u110A\u1167\u11AA", "\uC3A3", "\u110A\u1167\u11AA" },
            new string [] { "\uC3A4", "\uC3A4", "\u110A\u1167\u11AB", "\uC3A4", "\u110A\u1167\u11AB" },
            new string [] { "\uC3A5", "\uC3A5", "\u110A\u1167\u11AC", "\uC3A5", "\u110A\u1167\u11AC" },
            new string [] { "\uC3A6", "\uC3A6", "\u110A\u1167\u11AD", "\uC3A6", "\u110A\u1167\u11AD" },
            new string [] { "\uC3A7", "\uC3A7", "\u110A\u1167\u11AE", "\uC3A7", "\u110A\u1167\u11AE" },
            new string [] { "\uC3A8", "\uC3A8", "\u110A\u1167\u11AF", "\uC3A8", "\u110A\u1167\u11AF" },
            new string [] { "\uC3A9", "\uC3A9", "\u110A\u1167\u11B0", "\uC3A9", "\u110A\u1167\u11B0" },
            new string [] { "\uC3AA", "\uC3AA", "\u110A\u1167\u11B1", "\uC3AA", "\u110A\u1167\u11B1" },
            new string [] { "\uC3AB", "\uC3AB", "\u110A\u1167\u11B2", "\uC3AB", "\u110A\u1167\u11B2" },
            new string [] { "\uC3AC", "\uC3AC", "\u110A\u1167\u11B3", "\uC3AC", "\u110A\u1167\u11B3" },
            new string [] { "\uC3AD", "\uC3AD", "\u110A\u1167\u11B4", "\uC3AD", "\u110A\u1167\u11B4" },
            new string [] { "\uC3AE", "\uC3AE", "\u110A\u1167\u11B5", "\uC3AE", "\u110A\u1167\u11B5" },
            new string [] { "\uC3AF", "\uC3AF", "\u110A\u1167\u11B6", "\uC3AF", "\u110A\u1167\u11B6" },
            new string [] { "\uC3B0", "\uC3B0", "\u110A\u1167\u11B7", "\uC3B0", "\u110A\u1167\u11B7" },
            new string [] { "\uC3B1", "\uC3B1", "\u110A\u1167\u11B8", "\uC3B1", "\u110A\u1167\u11B8" },
            new string [] { "\uC3B2", "\uC3B2", "\u110A\u1167\u11B9", "\uC3B2", "\u110A\u1167\u11B9" },
            new string [] { "\uC3B3", "\uC3B3", "\u110A\u1167\u11BA", "\uC3B3", "\u110A\u1167\u11BA" },
            new string [] { "\uC3B4", "\uC3B4", "\u110A\u1167\u11BB", "\uC3B4", "\u110A\u1167\u11BB" },
            new string [] { "\uC3B5", "\uC3B5", "\u110A\u1167\u11BC", "\uC3B5", "\u110A\u1167\u11BC" },
            new string [] { "\uC3B6", "\uC3B6", "\u110A\u1167\u11BD", "\uC3B6", "\u110A\u1167\u11BD" },
            new string [] { "\uC3B7", "\uC3B7", "\u110A\u1167\u11BE", "\uC3B7", "\u110A\u1167\u11BE" },
            new string [] { "\uC3B8", "\uC3B8", "\u110A\u1167\u11BF", "\uC3B8", "\u110A\u1167\u11BF" },
            new string [] { "\uC3B9", "\uC3B9", "\u110A\u1167\u11C0", "\uC3B9", "\u110A\u1167\u11C0" },
            new string [] { "\uC3BA", "\uC3BA", "\u110A\u1167\u11C1", "\uC3BA", "\u110A\u1167\u11C1" },
            new string [] { "\uC3BB", "\uC3BB", "\u110A\u1167\u11C2", "\uC3BB", "\u110A\u1167\u11C2" },
            new string [] { "\uC3BC", "\uC3BC", "\u110A\u1168", "\uC3BC", "\u110A\u1168" },
            new string [] { "\uC3BD", "\uC3BD", "\u110A\u1168\u11A8", "\uC3BD", "\u110A\u1168\u11A8" },
            new string [] { "\uC3BE", "\uC3BE", "\u110A\u1168\u11A9", "\uC3BE", "\u110A\u1168\u11A9" },
            new string [] { "\uC3BF", "\uC3BF", "\u110A\u1168\u11AA", "\uC3BF", "\u110A\u1168\u11AA" },
            new string [] { "\uC3C0", "\uC3C0", "\u110A\u1168\u11AB", "\uC3C0", "\u110A\u1168\u11AB" },
            new string [] { "\uC3C1", "\uC3C1", "\u110A\u1168\u11AC", "\uC3C1", "\u110A\u1168\u11AC" },
            new string [] { "\uC3C2", "\uC3C2", "\u110A\u1168\u11AD", "\uC3C2", "\u110A\u1168\u11AD" },
            new string [] { "\uC3C3", "\uC3C3", "\u110A\u1168\u11AE", "\uC3C3", "\u110A\u1168\u11AE" },
            new string [] { "\uC3C4", "\uC3C4", "\u110A\u1168\u11AF", "\uC3C4", "\u110A\u1168\u11AF" },
            new string [] { "\uC3C5", "\uC3C5", "\u110A\u1168\u11B0", "\uC3C5", "\u110A\u1168\u11B0" },
            new string [] { "\uC3C6", "\uC3C6", "\u110A\u1168\u11B1", "\uC3C6", "\u110A\u1168\u11B1" },
            new string [] { "\uC3C7", "\uC3C7", "\u110A\u1168\u11B2", "\uC3C7", "\u110A\u1168\u11B2" },
            new string [] { "\uC3C8", "\uC3C8", "\u110A\u1168\u11B3", "\uC3C8", "\u110A\u1168\u11B3" },
            new string [] { "\uC3C9", "\uC3C9", "\u110A\u1168\u11B4", "\uC3C9", "\u110A\u1168\u11B4" },
            new string [] { "\uC3CA", "\uC3CA", "\u110A\u1168\u11B5", "\uC3CA", "\u110A\u1168\u11B5" },
            new string [] { "\uC3CB", "\uC3CB", "\u110A\u1168\u11B6", "\uC3CB", "\u110A\u1168\u11B6" },
            new string [] { "\uC3CC", "\uC3CC", "\u110A\u1168\u11B7", "\uC3CC", "\u110A\u1168\u11B7" },
            new string [] { "\uC3CD", "\uC3CD", "\u110A\u1168\u11B8", "\uC3CD", "\u110A\u1168\u11B8" },
            new string [] { "\uC3CE", "\uC3CE", "\u110A\u1168\u11B9", "\uC3CE", "\u110A\u1168\u11B9" },
            new string [] { "\uC3CF", "\uC3CF", "\u110A\u1168\u11BA", "\uC3CF", "\u110A\u1168\u11BA" },
            new string [] { "\uC3D0", "\uC3D0", "\u110A\u1168\u11BB", "\uC3D0", "\u110A\u1168\u11BB" },
            new string [] { "\uC3D1", "\uC3D1", "\u110A\u1168\u11BC", "\uC3D1", "\u110A\u1168\u11BC" },
            new string [] { "\uC3D2", "\uC3D2", "\u110A\u1168\u11BD", "\uC3D2", "\u110A\u1168\u11BD" },
            new string [] { "\uC3D3", "\uC3D3", "\u110A\u1168\u11BE", "\uC3D3", "\u110A\u1168\u11BE" },
            new string [] { "\uC3D4", "\uC3D4", "\u110A\u1168\u11BF", "\uC3D4", "\u110A\u1168\u11BF" },
            new string [] { "\uC3D5", "\uC3D5", "\u110A\u1168\u11C0", "\uC3D5", "\u110A\u1168\u11C0" },
            new string [] { "\uC3D6", "\uC3D6", "\u110A\u1168\u11C1", "\uC3D6", "\u110A\u1168\u11C1" },
            new string [] { "\uC3D7", "\uC3D7", "\u110A\u1168\u11C2", "\uC3D7", "\u110A\u1168\u11C2" },
            new string [] { "\uC3D8", "\uC3D8", "\u110A\u1169", "\uC3D8", "\u110A\u1169" },
            new string [] { "\uC3D9", "\uC3D9", "\u110A\u1169\u11A8", "\uC3D9", "\u110A\u1169\u11A8" },
            new string [] { "\uC3DA", "\uC3DA", "\u110A\u1169\u11A9", "\uC3DA", "\u110A\u1169\u11A9" },
            new string [] { "\uC3DB", "\uC3DB", "\u110A\u1169\u11AA", "\uC3DB", "\u110A\u1169\u11AA" },
            new string [] { "\uC3DC", "\uC3DC", "\u110A\u1169\u11AB", "\uC3DC", "\u110A\u1169\u11AB" },
            new string [] { "\uC3DD", "\uC3DD", "\u110A\u1169\u11AC", "\uC3DD", "\u110A\u1169\u11AC" },
            new string [] { "\uC3DE", "\uC3DE", "\u110A\u1169\u11AD", "\uC3DE", "\u110A\u1169\u11AD" },
            new string [] { "\uC3DF", "\uC3DF", "\u110A\u1169\u11AE", "\uC3DF", "\u110A\u1169\u11AE" },
            new string [] { "\uC3E0", "\uC3E0", "\u110A\u1169\u11AF", "\uC3E0", "\u110A\u1169\u11AF" },
            new string [] { "\uC3E1", "\uC3E1", "\u110A\u1169\u11B0", "\uC3E1", "\u110A\u1169\u11B0" },
            new string [] { "\uC3E2", "\uC3E2", "\u110A\u1169\u11B1", "\uC3E2", "\u110A\u1169\u11B1" },
            new string [] { "\uC3E3", "\uC3E3", "\u110A\u1169\u11B2", "\uC3E3", "\u110A\u1169\u11B2" },
            new string [] { "\uC3E4", "\uC3E4", "\u110A\u1169\u11B3", "\uC3E4", "\u110A\u1169\u11B3" },
            new string [] { "\uC3E5", "\uC3E5", "\u110A\u1169\u11B4", "\uC3E5", "\u110A\u1169\u11B4" },
            new string [] { "\uC3E6", "\uC3E6", "\u110A\u1169\u11B5", "\uC3E6", "\u110A\u1169\u11B5" },
            new string [] { "\uC3E7", "\uC3E7", "\u110A\u1169\u11B6", "\uC3E7", "\u110A\u1169\u11B6" },
            new string [] { "\uC3E8", "\uC3E8", "\u110A\u1169\u11B7", "\uC3E8", "\u110A\u1169\u11B7" },
            new string [] { "\uC3E9", "\uC3E9", "\u110A\u1169\u11B8", "\uC3E9", "\u110A\u1169\u11B8" },
            new string [] { "\uC3EA", "\uC3EA", "\u110A\u1169\u11B9", "\uC3EA", "\u110A\u1169\u11B9" },
            new string [] { "\uC3EB", "\uC3EB", "\u110A\u1169\u11BA", "\uC3EB", "\u110A\u1169\u11BA" },
            new string [] { "\uC3EC", "\uC3EC", "\u110A\u1169\u11BB", "\uC3EC", "\u110A\u1169\u11BB" },
            new string [] { "\uC3ED", "\uC3ED", "\u110A\u1169\u11BC", "\uC3ED", "\u110A\u1169\u11BC" },
            new string [] { "\uC3EE", "\uC3EE", "\u110A\u1169\u11BD", "\uC3EE", "\u110A\u1169\u11BD" },
            new string [] { "\uC3EF", "\uC3EF", "\u110A\u1169\u11BE", "\uC3EF", "\u110A\u1169\u11BE" },
            new string [] { "\uC3F0", "\uC3F0", "\u110A\u1169\u11BF", "\uC3F0", "\u110A\u1169\u11BF" },
            new string [] { "\uC3F1", "\uC3F1", "\u110A\u1169\u11C0", "\uC3F1", "\u110A\u1169\u11C0" },
            new string [] { "\uC3F2", "\uC3F2", "\u110A\u1169\u11C1", "\uC3F2", "\u110A\u1169\u11C1" },
            new string [] { "\uC3F3", "\uC3F3", "\u110A\u1169\u11C2", "\uC3F3", "\u110A\u1169\u11C2" },
            new string [] { "\uC3F4", "\uC3F4", "\u110A\u116A", "\uC3F4", "\u110A\u116A" },
            new string [] { "\uC3F5", "\uC3F5", "\u110A\u116A\u11A8", "\uC3F5", "\u110A\u116A\u11A8" },
            new string [] { "\uC3F6", "\uC3F6", "\u110A\u116A\u11A9", "\uC3F6", "\u110A\u116A\u11A9" },
            new string [] { "\uC3F7", "\uC3F7", "\u110A\u116A\u11AA", "\uC3F7", "\u110A\u116A\u11AA" },
            new string [] { "\uC3F8", "\uC3F8", "\u110A\u116A\u11AB", "\uC3F8", "\u110A\u116A\u11AB" },
            new string [] { "\uC3F9", "\uC3F9", "\u110A\u116A\u11AC", "\uC3F9", "\u110A\u116A\u11AC" },
            new string [] { "\uC3FA", "\uC3FA", "\u110A\u116A\u11AD", "\uC3FA", "\u110A\u116A\u11AD" },
            new string [] { "\uC3FB", "\uC3FB", "\u110A\u116A\u11AE", "\uC3FB", "\u110A\u116A\u11AE" },
            new string [] { "\uC3FC", "\uC3FC", "\u110A\u116A\u11AF", "\uC3FC", "\u110A\u116A\u11AF" },
            new string [] { "\uC3FD", "\uC3FD", "\u110A\u116A\u11B0", "\uC3FD", "\u110A\u116A\u11B0" },
            new string [] { "\uC3FE", "\uC3FE", "\u110A\u116A\u11B1", "\uC3FE", "\u110A\u116A\u11B1" },
            new string [] { "\uC3FF", "\uC3FF", "\u110A\u116A\u11B2", "\uC3FF", "\u110A\u116A\u11B2" },
            new string [] { "\uC400", "\uC400", "\u110A\u116A\u11B3", "\uC400", "\u110A\u116A\u11B3" },
            new string [] { "\uC401", "\uC401", "\u110A\u116A\u11B4", "\uC401", "\u110A\u116A\u11B4" },
            new string [] { "\uC402", "\uC402", "\u110A\u116A\u11B5", "\uC402", "\u110A\u116A\u11B5" },
            new string [] { "\uC403", "\uC403", "\u110A\u116A\u11B6", "\uC403", "\u110A\u116A\u11B6" },
            new string [] { "\uC404", "\uC404", "\u110A\u116A\u11B7", "\uC404", "\u110A\u116A\u11B7" },
            new string [] { "\uC405", "\uC405", "\u110A\u116A\u11B8", "\uC405", "\u110A\u116A\u11B8" },
            new string [] { "\uC406", "\uC406", "\u110A\u116A\u11B9", "\uC406", "\u110A\u116A\u11B9" },
            new string [] { "\uC407", "\uC407", "\u110A\u116A\u11BA", "\uC407", "\u110A\u116A\u11BA" },
            new string [] { "\uC408", "\uC408", "\u110A\u116A\u11BB", "\uC408", "\u110A\u116A\u11BB" },
            new string [] { "\uC409", "\uC409", "\u110A\u116A\u11BC", "\uC409", "\u110A\u116A\u11BC" },
            new string [] { "\uC40A", "\uC40A", "\u110A\u116A\u11BD", "\uC40A", "\u110A\u116A\u11BD" },
            new string [] { "\uC40B", "\uC40B", "\u110A\u116A\u11BE", "\uC40B", "\u110A\u116A\u11BE" },
            new string [] { "\uC40C", "\uC40C", "\u110A\u116A\u11BF", "\uC40C", "\u110A\u116A\u11BF" },
            new string [] { "\uC40D", "\uC40D", "\u110A\u116A\u11C0", "\uC40D", "\u110A\u116A\u11C0" },
            new string [] { "\uC40E", "\uC40E", "\u110A\u116A\u11C1", "\uC40E", "\u110A\u116A\u11C1" },
            new string [] { "\uC40F", "\uC40F", "\u110A\u116A\u11C2", "\uC40F", "\u110A\u116A\u11C2" },
            new string [] { "\uC410", "\uC410", "\u110A\u116B", "\uC410", "\u110A\u116B" },
            new string [] { "\uC411", "\uC411", "\u110A\u116B\u11A8", "\uC411", "\u110A\u116B\u11A8" },
            new string [] { "\uC412", "\uC412", "\u110A\u116B\u11A9", "\uC412", "\u110A\u116B\u11A9" },
            new string [] { "\uC413", "\uC413", "\u110A\u116B\u11AA", "\uC413", "\u110A\u116B\u11AA" },
            new string [] { "\uC414", "\uC414", "\u110A\u116B\u11AB", "\uC414", "\u110A\u116B\u11AB" },
            new string [] { "\uC415", "\uC415", "\u110A\u116B\u11AC", "\uC415", "\u110A\u116B\u11AC" },
            new string [] { "\uC416", "\uC416", "\u110A\u116B\u11AD", "\uC416", "\u110A\u116B\u11AD" },
            new string [] { "\uC417", "\uC417", "\u110A\u116B\u11AE", "\uC417", "\u110A\u116B\u11AE" },
            new string [] { "\uC418", "\uC418", "\u110A\u116B\u11AF", "\uC418", "\u110A\u116B\u11AF" },
            new string [] { "\uC419", "\uC419", "\u110A\u116B\u11B0", "\uC419", "\u110A\u116B\u11B0" },
            new string [] { "\uC41A", "\uC41A", "\u110A\u116B\u11B1", "\uC41A", "\u110A\u116B\u11B1" },
            new string [] { "\uC41B", "\uC41B", "\u110A\u116B\u11B2", "\uC41B", "\u110A\u116B\u11B2" },
            new string [] { "\uC41C", "\uC41C", "\u110A\u116B\u11B3", "\uC41C", "\u110A\u116B\u11B3" },
            new string [] { "\uC41D", "\uC41D", "\u110A\u116B\u11B4", "\uC41D", "\u110A\u116B\u11B4" },
            new string [] { "\uC41E", "\uC41E", "\u110A\u116B\u11B5", "\uC41E", "\u110A\u116B\u11B5" },
            new string [] { "\uC41F", "\uC41F", "\u110A\u116B\u11B6", "\uC41F", "\u110A\u116B\u11B6" },
            new string [] { "\uC420", "\uC420", "\u110A\u116B\u11B7", "\uC420", "\u110A\u116B\u11B7" },
            new string [] { "\uC421", "\uC421", "\u110A\u116B\u11B8", "\uC421", "\u110A\u116B\u11B8" },
            new string [] { "\uC422", "\uC422", "\u110A\u116B\u11B9", "\uC422", "\u110A\u116B\u11B9" },
            new string [] { "\uC423", "\uC423", "\u110A\u116B\u11BA", "\uC423", "\u110A\u116B\u11BA" },
            new string [] { "\uC424", "\uC424", "\u110A\u116B\u11BB", "\uC424", "\u110A\u116B\u11BB" },
            new string [] { "\uC425", "\uC425", "\u110A\u116B\u11BC", "\uC425", "\u110A\u116B\u11BC" },
            new string [] { "\uC426", "\uC426", "\u110A\u116B\u11BD", "\uC426", "\u110A\u116B\u11BD" },
            new string [] { "\uC427", "\uC427", "\u110A\u116B\u11BE", "\uC427", "\u110A\u116B\u11BE" },
            new string [] { "\uC428", "\uC428", "\u110A\u116B\u11BF", "\uC428", "\u110A\u116B\u11BF" },
            new string [] { "\uC429", "\uC429", "\u110A\u116B\u11C0", "\uC429", "\u110A\u116B\u11C0" },
            new string [] { "\uC42A", "\uC42A", "\u110A\u116B\u11C1", "\uC42A", "\u110A\u116B\u11C1" },
            new string [] { "\uC42B", "\uC42B", "\u110A\u116B\u11C2", "\uC42B", "\u110A\u116B\u11C2" },
            new string [] { "\uC42C", "\uC42C", "\u110A\u116C", "\uC42C", "\u110A\u116C" },
            new string [] { "\uC42D", "\uC42D", "\u110A\u116C\u11A8", "\uC42D", "\u110A\u116C\u11A8" },
            new string [] { "\uC42E", "\uC42E", "\u110A\u116C\u11A9", "\uC42E", "\u110A\u116C\u11A9" },
            new string [] { "\uC42F", "\uC42F", "\u110A\u116C\u11AA", "\uC42F", "\u110A\u116C\u11AA" },
            new string [] { "\uC430", "\uC430", "\u110A\u116C\u11AB", "\uC430", "\u110A\u116C\u11AB" },
            new string [] { "\uC431", "\uC431", "\u110A\u116C\u11AC", "\uC431", "\u110A\u116C\u11AC" },
            new string [] { "\uC432", "\uC432", "\u110A\u116C\u11AD", "\uC432", "\u110A\u116C\u11AD" },
            new string [] { "\uC433", "\uC433", "\u110A\u116C\u11AE", "\uC433", "\u110A\u116C\u11AE" },
            new string [] { "\uC434", "\uC434", "\u110A\u116C\u11AF", "\uC434", "\u110A\u116C\u11AF" },
            new string [] { "\uC435", "\uC435", "\u110A\u116C\u11B0", "\uC435", "\u110A\u116C\u11B0" },
            new string [] { "\uC436", "\uC436", "\u110A\u116C\u11B1", "\uC436", "\u110A\u116C\u11B1" },
            new string [] { "\uC437", "\uC437", "\u110A\u116C\u11B2", "\uC437", "\u110A\u116C\u11B2" },
            new string [] { "\uC438", "\uC438", "\u110A\u116C\u11B3", "\uC438", "\u110A\u116C\u11B3" },
            new string [] { "\uC439", "\uC439", "\u110A\u116C\u11B4", "\uC439", "\u110A\u116C\u11B4" },
            new string [] { "\uC43A", "\uC43A", "\u110A\u116C\u11B5", "\uC43A", "\u110A\u116C\u11B5" },
            new string [] { "\uC43B", "\uC43B", "\u110A\u116C\u11B6", "\uC43B", "\u110A\u116C\u11B6" },
            new string [] { "\uC43C", "\uC43C", "\u110A\u116C\u11B7", "\uC43C", "\u110A\u116C\u11B7" },
            new string [] { "\uC43D", "\uC43D", "\u110A\u116C\u11B8", "\uC43D", "\u110A\u116C\u11B8" },
            new string [] { "\uC43E", "\uC43E", "\u110A\u116C\u11B9", "\uC43E", "\u110A\u116C\u11B9" },
            new string [] { "\uC43F", "\uC43F", "\u110A\u116C\u11BA", "\uC43F", "\u110A\u116C\u11BA" },
            new string [] { "\uC440", "\uC440", "\u110A\u116C\u11BB", "\uC440", "\u110A\u116C\u11BB" },
            new string [] { "\uC441", "\uC441", "\u110A\u116C\u11BC", "\uC441", "\u110A\u116C\u11BC" },
            new string [] { "\uC442", "\uC442", "\u110A\u116C\u11BD", "\uC442", "\u110A\u116C\u11BD" },
            new string [] { "\uC443", "\uC443", "\u110A\u116C\u11BE", "\uC443", "\u110A\u116C\u11BE" },
            new string [] { "\uC444", "\uC444", "\u110A\u116C\u11BF", "\uC444", "\u110A\u116C\u11BF" },
            new string [] { "\uC445", "\uC445", "\u110A\u116C\u11C0", "\uC445", "\u110A\u116C\u11C0" },
            new string [] { "\uC446", "\uC446", "\u110A\u116C\u11C1", "\uC446", "\u110A\u116C\u11C1" },
            new string [] { "\uC447", "\uC447", "\u110A\u116C\u11C2", "\uC447", "\u110A\u116C\u11C2" },
            new string [] { "\uC448", "\uC448", "\u110A\u116D", "\uC448", "\u110A\u116D" },
            new string [] { "\uC449", "\uC449", "\u110A\u116D\u11A8", "\uC449", "\u110A\u116D\u11A8" },
            new string [] { "\uC44A", "\uC44A", "\u110A\u116D\u11A9", "\uC44A", "\u110A\u116D\u11A9" },
            new string [] { "\uC44B", "\uC44B", "\u110A\u116D\u11AA", "\uC44B", "\u110A\u116D\u11AA" },
            new string [] { "\uC44C", "\uC44C", "\u110A\u116D\u11AB", "\uC44C", "\u110A\u116D\u11AB" },
            new string [] { "\uC44D", "\uC44D", "\u110A\u116D\u11AC", "\uC44D", "\u110A\u116D\u11AC" },
            new string [] { "\uC44E", "\uC44E", "\u110A\u116D\u11AD", "\uC44E", "\u110A\u116D\u11AD" },
            new string [] { "\uC44F", "\uC44F", "\u110A\u116D\u11AE", "\uC44F", "\u110A\u116D\u11AE" },
            new string [] { "\uC450", "\uC450", "\u110A\u116D\u11AF", "\uC450", "\u110A\u116D\u11AF" },
            new string [] { "\uC451", "\uC451", "\u110A\u116D\u11B0", "\uC451", "\u110A\u116D\u11B0" },
            new string [] { "\uC452", "\uC452", "\u110A\u116D\u11B1", "\uC452", "\u110A\u116D\u11B1" },
            new string [] { "\uC453", "\uC453", "\u110A\u116D\u11B2", "\uC453", "\u110A\u116D\u11B2" },
            new string [] { "\uC454", "\uC454", "\u110A\u116D\u11B3", "\uC454", "\u110A\u116D\u11B3" },
            new string [] { "\uC455", "\uC455", "\u110A\u116D\u11B4", "\uC455", "\u110A\u116D\u11B4" },
            new string [] { "\uC456", "\uC456", "\u110A\u116D\u11B5", "\uC456", "\u110A\u116D\u11B5" },
            new string [] { "\uC457", "\uC457", "\u110A\u116D\u11B6", "\uC457", "\u110A\u116D\u11B6" },
            new string [] { "\uC458", "\uC458", "\u110A\u116D\u11B7", "\uC458", "\u110A\u116D\u11B7" },
            new string [] { "\uC459", "\uC459", "\u110A\u116D\u11B8", "\uC459", "\u110A\u116D\u11B8" },
            new string [] { "\uC45A", "\uC45A", "\u110A\u116D\u11B9", "\uC45A", "\u110A\u116D\u11B9" },
            new string [] { "\uC45B", "\uC45B", "\u110A\u116D\u11BA", "\uC45B", "\u110A\u116D\u11BA" },
            new string [] { "\uC45C", "\uC45C", "\u110A\u116D\u11BB", "\uC45C", "\u110A\u116D\u11BB" },
            new string [] { "\uC45D", "\uC45D", "\u110A\u116D\u11BC", "\uC45D", "\u110A\u116D\u11BC" },
            new string [] { "\uC45E", "\uC45E", "\u110A\u116D\u11BD", "\uC45E", "\u110A\u116D\u11BD" },
            new string [] { "\uC45F", "\uC45F", "\u110A\u116D\u11BE", "\uC45F", "\u110A\u116D\u11BE" },
            new string [] { "\uC460", "\uC460", "\u110A\u116D\u11BF", "\uC460", "\u110A\u116D\u11BF" },
            new string [] { "\uC461", "\uC461", "\u110A\u116D\u11C0", "\uC461", "\u110A\u116D\u11C0" },
            new string [] { "\uC462", "\uC462", "\u110A\u116D\u11C1", "\uC462", "\u110A\u116D\u11C1" },
            new string [] { "\uC463", "\uC463", "\u110A\u116D\u11C2", "\uC463", "\u110A\u116D\u11C2" },
            new string [] { "\uC464", "\uC464", "\u110A\u116E", "\uC464", "\u110A\u116E" },
            new string [] { "\uC465", "\uC465", "\u110A\u116E\u11A8", "\uC465", "\u110A\u116E\u11A8" },
            new string [] { "\uC466", "\uC466", "\u110A\u116E\u11A9", "\uC466", "\u110A\u116E\u11A9" },
            new string [] { "\uC467", "\uC467", "\u110A\u116E\u11AA", "\uC467", "\u110A\u116E\u11AA" },
            new string [] { "\uC468", "\uC468", "\u110A\u116E\u11AB", "\uC468", "\u110A\u116E\u11AB" },
            new string [] { "\uC469", "\uC469", "\u110A\u116E\u11AC", "\uC469", "\u110A\u116E\u11AC" },
            new string [] { "\uC46A", "\uC46A", "\u110A\u116E\u11AD", "\uC46A", "\u110A\u116E\u11AD" },
            new string [] { "\uC46B", "\uC46B", "\u110A\u116E\u11AE", "\uC46B", "\u110A\u116E\u11AE" },
            new string [] { "\uC46C", "\uC46C", "\u110A\u116E\u11AF", "\uC46C", "\u110A\u116E\u11AF" },
            new string [] { "\uC46D", "\uC46D", "\u110A\u116E\u11B0", "\uC46D", "\u110A\u116E\u11B0" },
            new string [] { "\uC46E", "\uC46E", "\u110A\u116E\u11B1", "\uC46E", "\u110A\u116E\u11B1" },
            new string [] { "\uC46F", "\uC46F", "\u110A\u116E\u11B2", "\uC46F", "\u110A\u116E\u11B2" },
            new string [] { "\uC470", "\uC470", "\u110A\u116E\u11B3", "\uC470", "\u110A\u116E\u11B3" },
            new string [] { "\uC471", "\uC471", "\u110A\u116E\u11B4", "\uC471", "\u110A\u116E\u11B4" },
            new string [] { "\uC472", "\uC472", "\u110A\u116E\u11B5", "\uC472", "\u110A\u116E\u11B5" },
            new string [] { "\uC473", "\uC473", "\u110A\u116E\u11B6", "\uC473", "\u110A\u116E\u11B6" },
            new string [] { "\uC474", "\uC474", "\u110A\u116E\u11B7", "\uC474", "\u110A\u116E\u11B7" },
            new string [] { "\uC475", "\uC475", "\u110A\u116E\u11B8", "\uC475", "\u110A\u116E\u11B8" },
            new string [] { "\uC476", "\uC476", "\u110A\u116E\u11B9", "\uC476", "\u110A\u116E\u11B9" },
            new string [] { "\uC477", "\uC477", "\u110A\u116E\u11BA", "\uC477", "\u110A\u116E\u11BA" },
            new string [] { "\uC478", "\uC478", "\u110A\u116E\u11BB", "\uC478", "\u110A\u116E\u11BB" },
            new string [] { "\uC479", "\uC479", "\u110A\u116E\u11BC", "\uC479", "\u110A\u116E\u11BC" },
            new string [] { "\uC47A", "\uC47A", "\u110A\u116E\u11BD", "\uC47A", "\u110A\u116E\u11BD" },
            new string [] { "\uC47B", "\uC47B", "\u110A\u116E\u11BE", "\uC47B", "\u110A\u116E\u11BE" },
            new string [] { "\uC47C", "\uC47C", "\u110A\u116E\u11BF", "\uC47C", "\u110A\u116E\u11BF" },
            new string [] { "\uC47D", "\uC47D", "\u110A\u116E\u11C0", "\uC47D", "\u110A\u116E\u11C0" },
            new string [] { "\uC47E", "\uC47E", "\u110A\u116E\u11C1", "\uC47E", "\u110A\u116E\u11C1" },
            new string [] { "\uC47F", "\uC47F", "\u110A\u116E\u11C2", "\uC47F", "\u110A\u116E\u11C2" },
            new string [] { "\uC480", "\uC480", "\u110A\u116F", "\uC480", "\u110A\u116F" },
            new string [] { "\uC481", "\uC481", "\u110A\u116F\u11A8", "\uC481", "\u110A\u116F\u11A8" },
            new string [] { "\uC482", "\uC482", "\u110A\u116F\u11A9", "\uC482", "\u110A\u116F\u11A9" },
            new string [] { "\uC483", "\uC483", "\u110A\u116F\u11AA", "\uC483", "\u110A\u116F\u11AA" },
            new string [] { "\uC484", "\uC484", "\u110A\u116F\u11AB", "\uC484", "\u110A\u116F\u11AB" },
            new string [] { "\uC485", "\uC485", "\u110A\u116F\u11AC", "\uC485", "\u110A\u116F\u11AC" },
            new string [] { "\uC486", "\uC486", "\u110A\u116F\u11AD", "\uC486", "\u110A\u116F\u11AD" },
            new string [] { "\uC487", "\uC487", "\u110A\u116F\u11AE", "\uC487", "\u110A\u116F\u11AE" },
            new string [] { "\uC488", "\uC488", "\u110A\u116F\u11AF", "\uC488", "\u110A\u116F\u11AF" },
            new string [] { "\uC489", "\uC489", "\u110A\u116F\u11B0", "\uC489", "\u110A\u116F\u11B0" },
            new string [] { "\uC48A", "\uC48A", "\u110A\u116F\u11B1", "\uC48A", "\u110A\u116F\u11B1" },
            new string [] { "\uC48B", "\uC48B", "\u110A\u116F\u11B2", "\uC48B", "\u110A\u116F\u11B2" },
            new string [] { "\uC48C", "\uC48C", "\u110A\u116F\u11B3", "\uC48C", "\u110A\u116F\u11B3" },
            new string [] { "\uC48D", "\uC48D", "\u110A\u116F\u11B4", "\uC48D", "\u110A\u116F\u11B4" },
            new string [] { "\uC48E", "\uC48E", "\u110A\u116F\u11B5", "\uC48E", "\u110A\u116F\u11B5" },
            new string [] { "\uC48F", "\uC48F", "\u110A\u116F\u11B6", "\uC48F", "\u110A\u116F\u11B6" },
            new string [] { "\uC490", "\uC490", "\u110A\u116F\u11B7", "\uC490", "\u110A\u116F\u11B7" },
            new string [] { "\uC491", "\uC491", "\u110A\u116F\u11B8", "\uC491", "\u110A\u116F\u11B8" },
            new string [] { "\uC492", "\uC492", "\u110A\u116F\u11B9", "\uC492", "\u110A\u116F\u11B9" },
            new string [] { "\uC493", "\uC493", "\u110A\u116F\u11BA", "\uC493", "\u110A\u116F\u11BA" },
            new string [] { "\uC494", "\uC494", "\u110A\u116F\u11BB", "\uC494", "\u110A\u116F\u11BB" },
            new string [] { "\uC495", "\uC495", "\u110A\u116F\u11BC", "\uC495", "\u110A\u116F\u11BC" },
            new string [] { "\uC496", "\uC496", "\u110A\u116F\u11BD", "\uC496", "\u110A\u116F\u11BD" },
            new string [] { "\uC497", "\uC497", "\u110A\u116F\u11BE", "\uC497", "\u110A\u116F\u11BE" },
            new string [] { "\uC498", "\uC498", "\u110A\u116F\u11BF", "\uC498", "\u110A\u116F\u11BF" },
            new string [] { "\uC499", "\uC499", "\u110A\u116F\u11C0", "\uC499", "\u110A\u116F\u11C0" },
            new string [] { "\uC49A", "\uC49A", "\u110A\u116F\u11C1", "\uC49A", "\u110A\u116F\u11C1" },
            new string [] { "\uC49B", "\uC49B", "\u110A\u116F\u11C2", "\uC49B", "\u110A\u116F\u11C2" },
            new string [] { "\uC49C", "\uC49C", "\u110A\u1170", "\uC49C", "\u110A\u1170" },
            new string [] { "\uC49D", "\uC49D", "\u110A\u1170\u11A8", "\uC49D", "\u110A\u1170\u11A8" },
            new string [] { "\uC49E", "\uC49E", "\u110A\u1170\u11A9", "\uC49E", "\u110A\u1170\u11A9" },
            new string [] { "\uC49F", "\uC49F", "\u110A\u1170\u11AA", "\uC49F", "\u110A\u1170\u11AA" },
            new string [] { "\uC4A0", "\uC4A0", "\u110A\u1170\u11AB", "\uC4A0", "\u110A\u1170\u11AB" },
            new string [] { "\uC4A1", "\uC4A1", "\u110A\u1170\u11AC", "\uC4A1", "\u110A\u1170\u11AC" },
            new string [] { "\uC4A2", "\uC4A2", "\u110A\u1170\u11AD", "\uC4A2", "\u110A\u1170\u11AD" },
            new string [] { "\uC4A3", "\uC4A3", "\u110A\u1170\u11AE", "\uC4A3", "\u110A\u1170\u11AE" },
            new string [] { "\uC4A4", "\uC4A4", "\u110A\u1170\u11AF", "\uC4A4", "\u110A\u1170\u11AF" },
            new string [] { "\uC4A5", "\uC4A5", "\u110A\u1170\u11B0", "\uC4A5", "\u110A\u1170\u11B0" },
            new string [] { "\uC4A6", "\uC4A6", "\u110A\u1170\u11B1", "\uC4A6", "\u110A\u1170\u11B1" },
            new string [] { "\uC4A7", "\uC4A7", "\u110A\u1170\u11B2", "\uC4A7", "\u110A\u1170\u11B2" },
            new string [] { "\uC4A8", "\uC4A8", "\u110A\u1170\u11B3", "\uC4A8", "\u110A\u1170\u11B3" },
            new string [] { "\uC4A9", "\uC4A9", "\u110A\u1170\u11B4", "\uC4A9", "\u110A\u1170\u11B4" },
            new string [] { "\uC4AA", "\uC4AA", "\u110A\u1170\u11B5", "\uC4AA", "\u110A\u1170\u11B5" },
            new string [] { "\uC4AB", "\uC4AB", "\u110A\u1170\u11B6", "\uC4AB", "\u110A\u1170\u11B6" },
            new string [] { "\uC4AC", "\uC4AC", "\u110A\u1170\u11B7", "\uC4AC", "\u110A\u1170\u11B7" },
            new string [] { "\uC4AD", "\uC4AD", "\u110A\u1170\u11B8", "\uC4AD", "\u110A\u1170\u11B8" },
            new string [] { "\uC4AE", "\uC4AE", "\u110A\u1170\u11B9", "\uC4AE", "\u110A\u1170\u11B9" },
            new string [] { "\uC4AF", "\uC4AF", "\u110A\u1170\u11BA", "\uC4AF", "\u110A\u1170\u11BA" },
            new string [] { "\uC4B0", "\uC4B0", "\u110A\u1170\u11BB", "\uC4B0", "\u110A\u1170\u11BB" },
            new string [] { "\uC4B1", "\uC4B1", "\u110A\u1170\u11BC", "\uC4B1", "\u110A\u1170\u11BC" },
            new string [] { "\uC4B2", "\uC4B2", "\u110A\u1170\u11BD", "\uC4B2", "\u110A\u1170\u11BD" },
            new string [] { "\uC4B3", "\uC4B3", "\u110A\u1170\u11BE", "\uC4B3", "\u110A\u1170\u11BE" },
            new string [] { "\uC4B4", "\uC4B4", "\u110A\u1170\u11BF", "\uC4B4", "\u110A\u1170\u11BF" },
            new string [] { "\uC4B5", "\uC4B5", "\u110A\u1170\u11C0", "\uC4B5", "\u110A\u1170\u11C0" },
            new string [] { "\uC4B6", "\uC4B6", "\u110A\u1170\u11C1", "\uC4B6", "\u110A\u1170\u11C1" },
            new string [] { "\uC4B7", "\uC4B7", "\u110A\u1170\u11C2", "\uC4B7", "\u110A\u1170\u11C2" },
            new string [] { "\uC4B8", "\uC4B8", "\u110A\u1171", "\uC4B8", "\u110A\u1171" },
            new string [] { "\uC4B9", "\uC4B9", "\u110A\u1171\u11A8", "\uC4B9", "\u110A\u1171\u11A8" },
            new string [] { "\uC4BA", "\uC4BA", "\u110A\u1171\u11A9", "\uC4BA", "\u110A\u1171\u11A9" },
            new string [] { "\uC4BB", "\uC4BB", "\u110A\u1171\u11AA", "\uC4BB", "\u110A\u1171\u11AA" },
            new string [] { "\uC4BC", "\uC4BC", "\u110A\u1171\u11AB", "\uC4BC", "\u110A\u1171\u11AB" },
            new string [] { "\uC4BD", "\uC4BD", "\u110A\u1171\u11AC", "\uC4BD", "\u110A\u1171\u11AC" },
            new string [] { "\uC4BE", "\uC4BE", "\u110A\u1171\u11AD", "\uC4BE", "\u110A\u1171\u11AD" },
            new string [] { "\uC4BF", "\uC4BF", "\u110A\u1171\u11AE", "\uC4BF", "\u110A\u1171\u11AE" },
            new string [] { "\uC4C0", "\uC4C0", "\u110A\u1171\u11AF", "\uC4C0", "\u110A\u1171\u11AF" },
            new string [] { "\uC4C1", "\uC4C1", "\u110A\u1171\u11B0", "\uC4C1", "\u110A\u1171\u11B0" },
            new string [] { "\uC4C2", "\uC4C2", "\u110A\u1171\u11B1", "\uC4C2", "\u110A\u1171\u11B1" },
            new string [] { "\uC4C3", "\uC4C3", "\u110A\u1171\u11B2", "\uC4C3", "\u110A\u1171\u11B2" },
            new string [] { "\uC4C4", "\uC4C4", "\u110A\u1171\u11B3", "\uC4C4", "\u110A\u1171\u11B3" },
            new string [] { "\uC4C5", "\uC4C5", "\u110A\u1171\u11B4", "\uC4C5", "\u110A\u1171\u11B4" },
            new string [] { "\uC4C6", "\uC4C6", "\u110A\u1171\u11B5", "\uC4C6", "\u110A\u1171\u11B5" },
            new string [] { "\uC4C7", "\uC4C7", "\u110A\u1171\u11B6", "\uC4C7", "\u110A\u1171\u11B6" },
            new string [] { "\uC4C8", "\uC4C8", "\u110A\u1171\u11B7", "\uC4C8", "\u110A\u1171\u11B7" },
            new string [] { "\uC4C9", "\uC4C9", "\u110A\u1171\u11B8", "\uC4C9", "\u110A\u1171\u11B8" },
            new string [] { "\uC4CA", "\uC4CA", "\u110A\u1171\u11B9", "\uC4CA", "\u110A\u1171\u11B9" },
            new string [] { "\uC4CB", "\uC4CB", "\u110A\u1171\u11BA", "\uC4CB", "\u110A\u1171\u11BA" },
            new string [] { "\uC4CC", "\uC4CC", "\u110A\u1171\u11BB", "\uC4CC", "\u110A\u1171\u11BB" },
            new string [] { "\uC4CD", "\uC4CD", "\u110A\u1171\u11BC", "\uC4CD", "\u110A\u1171\u11BC" },
            new string [] { "\uC4CE", "\uC4CE", "\u110A\u1171\u11BD", "\uC4CE", "\u110A\u1171\u11BD" },
            new string [] { "\uC4CF", "\uC4CF", "\u110A\u1171\u11BE", "\uC4CF", "\u110A\u1171\u11BE" },
            new string [] { "\uC4D0", "\uC4D0", "\u110A\u1171\u11BF", "\uC4D0", "\u110A\u1171\u11BF" },
            new string [] { "\uC4D1", "\uC4D1", "\u110A\u1171\u11C0", "\uC4D1", "\u110A\u1171\u11C0" },
            new string [] { "\uC4D2", "\uC4D2", "\u110A\u1171\u11C1", "\uC4D2", "\u110A\u1171\u11C1" },
            new string [] { "\uC4D3", "\uC4D3", "\u110A\u1171\u11C2", "\uC4D3", "\u110A\u1171\u11C2" },
            new string [] { "\uC4D4", "\uC4D4", "\u110A\u1172", "\uC4D4", "\u110A\u1172" },
            new string [] { "\uC4D5", "\uC4D5", "\u110A\u1172\u11A8", "\uC4D5", "\u110A\u1172\u11A8" },
            new string [] { "\uC4D6", "\uC4D6", "\u110A\u1172\u11A9", "\uC4D6", "\u110A\u1172\u11A9" },
            new string [] { "\uC4D7", "\uC4D7", "\u110A\u1172\u11AA", "\uC4D7", "\u110A\u1172\u11AA" },
            new string [] { "\uC4D8", "\uC4D8", "\u110A\u1172\u11AB", "\uC4D8", "\u110A\u1172\u11AB" },
            new string [] { "\uC4D9", "\uC4D9", "\u110A\u1172\u11AC", "\uC4D9", "\u110A\u1172\u11AC" },
            new string [] { "\uC4DA", "\uC4DA", "\u110A\u1172\u11AD", "\uC4DA", "\u110A\u1172\u11AD" },
            new string [] { "\uC4DB", "\uC4DB", "\u110A\u1172\u11AE", "\uC4DB", "\u110A\u1172\u11AE" },
            new string [] { "\uC4DC", "\uC4DC", "\u110A\u1172\u11AF", "\uC4DC", "\u110A\u1172\u11AF" },
            new string [] { "\uC4DD", "\uC4DD", "\u110A\u1172\u11B0", "\uC4DD", "\u110A\u1172\u11B0" },
            new string [] { "\uC4DE", "\uC4DE", "\u110A\u1172\u11B1", "\uC4DE", "\u110A\u1172\u11B1" },
            new string [] { "\uC4DF", "\uC4DF", "\u110A\u1172\u11B2", "\uC4DF", "\u110A\u1172\u11B2" },
            new string [] { "\uC4E0", "\uC4E0", "\u110A\u1172\u11B3", "\uC4E0", "\u110A\u1172\u11B3" },
            new string [] { "\uC4E1", "\uC4E1", "\u110A\u1172\u11B4", "\uC4E1", "\u110A\u1172\u11B4" },
            new string [] { "\uC4E2", "\uC4E2", "\u110A\u1172\u11B5", "\uC4E2", "\u110A\u1172\u11B5" },
            new string [] { "\uC4E3", "\uC4E3", "\u110A\u1172\u11B6", "\uC4E3", "\u110A\u1172\u11B6" },
            new string [] { "\uC4E4", "\uC4E4", "\u110A\u1172\u11B7", "\uC4E4", "\u110A\u1172\u11B7" },
            new string [] { "\uC4E5", "\uC4E5", "\u110A\u1172\u11B8", "\uC4E5", "\u110A\u1172\u11B8" },
            new string [] { "\uC4E6", "\uC4E6", "\u110A\u1172\u11B9", "\uC4E6", "\u110A\u1172\u11B9" },
            new string [] { "\uC4E7", "\uC4E7", "\u110A\u1172\u11BA", "\uC4E7", "\u110A\u1172\u11BA" },
            new string [] { "\uC4E8", "\uC4E8", "\u110A\u1172\u11BB", "\uC4E8", "\u110A\u1172\u11BB" },
            new string [] { "\uC4E9", "\uC4E9", "\u110A\u1172\u11BC", "\uC4E9", "\u110A\u1172\u11BC" },
            new string [] { "\uC4EA", "\uC4EA", "\u110A\u1172\u11BD", "\uC4EA", "\u110A\u1172\u11BD" },
            new string [] { "\uC4EB", "\uC4EB", "\u110A\u1172\u11BE", "\uC4EB", "\u110A\u1172\u11BE" },
            new string [] { "\uC4EC", "\uC4EC", "\u110A\u1172\u11BF", "\uC4EC", "\u110A\u1172\u11BF" },
            new string [] { "\uC4ED", "\uC4ED", "\u110A\u1172\u11C0", "\uC4ED", "\u110A\u1172\u11C0" },
            new string [] { "\uC4EE", "\uC4EE", "\u110A\u1172\u11C1", "\uC4EE", "\u110A\u1172\u11C1" },
            new string [] { "\uC4EF", "\uC4EF", "\u110A\u1172\u11C2", "\uC4EF", "\u110A\u1172\u11C2" },
            new string [] { "\uC4F0", "\uC4F0", "\u110A\u1173", "\uC4F0", "\u110A\u1173" },
            new string [] { "\uC4F1", "\uC4F1", "\u110A\u1173\u11A8", "\uC4F1", "\u110A\u1173\u11A8" },
            new string [] { "\uC4F2", "\uC4F2", "\u110A\u1173\u11A9", "\uC4F2", "\u110A\u1173\u11A9" },
            new string [] { "\uC4F3", "\uC4F3", "\u110A\u1173\u11AA", "\uC4F3", "\u110A\u1173\u11AA" },
            new string [] { "\uC4F4", "\uC4F4", "\u110A\u1173\u11AB", "\uC4F4", "\u110A\u1173\u11AB" },
            new string [] { "\uC4F5", "\uC4F5", "\u110A\u1173\u11AC", "\uC4F5", "\u110A\u1173\u11AC" },
            new string [] { "\uC4F6", "\uC4F6", "\u110A\u1173\u11AD", "\uC4F6", "\u110A\u1173\u11AD" },
            new string [] { "\uC4F7", "\uC4F7", "\u110A\u1173\u11AE", "\uC4F7", "\u110A\u1173\u11AE" },
            new string [] { "\uC4F8", "\uC4F8", "\u110A\u1173\u11AF", "\uC4F8", "\u110A\u1173\u11AF" },
            new string [] { "\uC4F9", "\uC4F9", "\u110A\u1173\u11B0", "\uC4F9", "\u110A\u1173\u11B0" },
            new string [] { "\uC4FA", "\uC4FA", "\u110A\u1173\u11B1", "\uC4FA", "\u110A\u1173\u11B1" },
            new string [] { "\uC4FB", "\uC4FB", "\u110A\u1173\u11B2", "\uC4FB", "\u110A\u1173\u11B2" },
            new string [] { "\uC4FC", "\uC4FC", "\u110A\u1173\u11B3", "\uC4FC", "\u110A\u1173\u11B3" },
            new string [] { "\uC4FD", "\uC4FD", "\u110A\u1173\u11B4", "\uC4FD", "\u110A\u1173\u11B4" },
            new string [] { "\uC4FE", "\uC4FE", "\u110A\u1173\u11B5", "\uC4FE", "\u110A\u1173\u11B5" },
            new string [] { "\uC4FF", "\uC4FF", "\u110A\u1173\u11B6", "\uC4FF", "\u110A\u1173\u11B6" },
            new string [] { "\uC500", "\uC500", "\u110A\u1173\u11B7", "\uC500", "\u110A\u1173\u11B7" },
            new string [] { "\uC501", "\uC501", "\u110A\u1173\u11B8", "\uC501", "\u110A\u1173\u11B8" },
            new string [] { "\uC502", "\uC502", "\u110A\u1173\u11B9", "\uC502", "\u110A\u1173\u11B9" },
            new string [] { "\uC503", "\uC503", "\u110A\u1173\u11BA", "\uC503", "\u110A\u1173\u11BA" },
            new string [] { "\uC504", "\uC504", "\u110A\u1173\u11BB", "\uC504", "\u110A\u1173\u11BB" },
            new string [] { "\uC505", "\uC505", "\u110A\u1173\u11BC", "\uC505", "\u110A\u1173\u11BC" },
            new string [] { "\uC506", "\uC506", "\u110A\u1173\u11BD", "\uC506", "\u110A\u1173\u11BD" },
            new string [] { "\uC507", "\uC507", "\u110A\u1173\u11BE", "\uC507", "\u110A\u1173\u11BE" },
            new string [] { "\uC508", "\uC508", "\u110A\u1173\u11BF", "\uC508", "\u110A\u1173\u11BF" },
            new string [] { "\uC509", "\uC509", "\u110A\u1173\u11C0", "\uC509", "\u110A\u1173\u11C0" },
            new string [] { "\uC50A", "\uC50A", "\u110A\u1173\u11C1", "\uC50A", "\u110A\u1173\u11C1" },
            new string [] { "\uC50B", "\uC50B", "\u110A\u1173\u11C2", "\uC50B", "\u110A\u1173\u11C2" },
            new string [] { "\uC50C", "\uC50C", "\u110A\u1174", "\uC50C", "\u110A\u1174" },
            new string [] { "\uC50D", "\uC50D", "\u110A\u1174\u11A8", "\uC50D", "\u110A\u1174\u11A8" },
            new string [] { "\uC50E", "\uC50E", "\u110A\u1174\u11A9", "\uC50E", "\u110A\u1174\u11A9" },
            new string [] { "\uC50F", "\uC50F", "\u110A\u1174\u11AA", "\uC50F", "\u110A\u1174\u11AA" },
            new string [] { "\uC510", "\uC510", "\u110A\u1174\u11AB", "\uC510", "\u110A\u1174\u11AB" },
            new string [] { "\uC511", "\uC511", "\u110A\u1174\u11AC", "\uC511", "\u110A\u1174\u11AC" },
            new string [] { "\uC512", "\uC512", "\u110A\u1174\u11AD", "\uC512", "\u110A\u1174\u11AD" },
            new string [] { "\uC513", "\uC513", "\u110A\u1174\u11AE", "\uC513", "\u110A\u1174\u11AE" },
            new string [] { "\uC514", "\uC514", "\u110A\u1174\u11AF", "\uC514", "\u110A\u1174\u11AF" },
            new string [] { "\uC515", "\uC515", "\u110A\u1174\u11B0", "\uC515", "\u110A\u1174\u11B0" },
            new string [] { "\uC516", "\uC516", "\u110A\u1174\u11B1", "\uC516", "\u110A\u1174\u11B1" },
            new string [] { "\uC517", "\uC517", "\u110A\u1174\u11B2", "\uC517", "\u110A\u1174\u11B2" },
            new string [] { "\uC518", "\uC518", "\u110A\u1174\u11B3", "\uC518", "\u110A\u1174\u11B3" },
            new string [] { "\uC519", "\uC519", "\u110A\u1174\u11B4", "\uC519", "\u110A\u1174\u11B4" },
            new string [] { "\uC51A", "\uC51A", "\u110A\u1174\u11B5", "\uC51A", "\u110A\u1174\u11B5" },
            new string [] { "\uC51B", "\uC51B", "\u110A\u1174\u11B6", "\uC51B", "\u110A\u1174\u11B6" },
            new string [] { "\uC51C", "\uC51C", "\u110A\u1174\u11B7", "\uC51C", "\u110A\u1174\u11B7" },
            new string [] { "\uC51D", "\uC51D", "\u110A\u1174\u11B8", "\uC51D", "\u110A\u1174\u11B8" },
            new string [] { "\uC51E", "\uC51E", "\u110A\u1174\u11B9", "\uC51E", "\u110A\u1174\u11B9" },
            new string [] { "\uC51F", "\uC51F", "\u110A\u1174\u11BA", "\uC51F", "\u110A\u1174\u11BA" },
            new string [] { "\uC520", "\uC520", "\u110A\u1174\u11BB", "\uC520", "\u110A\u1174\u11BB" },
            new string [] { "\uC521", "\uC521", "\u110A\u1174\u11BC", "\uC521", "\u110A\u1174\u11BC" },
            new string [] { "\uC522", "\uC522", "\u110A\u1174\u11BD", "\uC522", "\u110A\u1174\u11BD" },
            new string [] { "\uC523", "\uC523", "\u110A\u1174\u11BE", "\uC523", "\u110A\u1174\u11BE" },
            new string [] { "\uC524", "\uC524", "\u110A\u1174\u11BF", "\uC524", "\u110A\u1174\u11BF" },
            new string [] { "\uC525", "\uC525", "\u110A\u1174\u11C0", "\uC525", "\u110A\u1174\u11C0" },
            new string [] { "\uC526", "\uC526", "\u110A\u1174\u11C1", "\uC526", "\u110A\u1174\u11C1" },
            new string [] { "\uC527", "\uC527", "\u110A\u1174\u11C2", "\uC527", "\u110A\u1174\u11C2" },
            new string [] { "\uC528", "\uC528", "\u110A\u1175", "\uC528", "\u110A\u1175" },
            new string [] { "\uC529", "\uC529", "\u110A\u1175\u11A8", "\uC529", "\u110A\u1175\u11A8" },
            new string [] { "\uC52A", "\uC52A", "\u110A\u1175\u11A9", "\uC52A", "\u110A\u1175\u11A9" },
            new string [] { "\uC52B", "\uC52B", "\u110A\u1175\u11AA", "\uC52B", "\u110A\u1175\u11AA" },
            new string [] { "\uC52C", "\uC52C", "\u110A\u1175\u11AB", "\uC52C", "\u110A\u1175\u11AB" },
            new string [] { "\uC52D", "\uC52D", "\u110A\u1175\u11AC", "\uC52D", "\u110A\u1175\u11AC" },
            new string [] { "\uC52E", "\uC52E", "\u110A\u1175\u11AD", "\uC52E", "\u110A\u1175\u11AD" },
            new string [] { "\uC52F", "\uC52F", "\u110A\u1175\u11AE", "\uC52F", "\u110A\u1175\u11AE" },
            new string [] { "\uC530", "\uC530", "\u110A\u1175\u11AF", "\uC530", "\u110A\u1175\u11AF" },
            new string [] { "\uC531", "\uC531", "\u110A\u1175\u11B0", "\uC531", "\u110A\u1175\u11B0" },
            new string [] { "\uC532", "\uC532", "\u110A\u1175\u11B1", "\uC532", "\u110A\u1175\u11B1" },
            new string [] { "\uC533", "\uC533", "\u110A\u1175\u11B2", "\uC533", "\u110A\u1175\u11B2" },
            new string [] { "\uC534", "\uC534", "\u110A\u1175\u11B3", "\uC534", "\u110A\u1175\u11B3" },
            new string [] { "\uC535", "\uC535", "\u110A\u1175\u11B4", "\uC535", "\u110A\u1175\u11B4" },
            new string [] { "\uC536", "\uC536", "\u110A\u1175\u11B5", "\uC536", "\u110A\u1175\u11B5" },
            new string [] { "\uC537", "\uC537", "\u110A\u1175\u11B6", "\uC537", "\u110A\u1175\u11B6" },
            new string [] { "\uC538", "\uC538", "\u110A\u1175\u11B7", "\uC538", "\u110A\u1175\u11B7" },
            new string [] { "\uC539", "\uC539", "\u110A\u1175\u11B8", "\uC539", "\u110A\u1175\u11B8" },
            new string [] { "\uC53A", "\uC53A", "\u110A\u1175\u11B9", "\uC53A", "\u110A\u1175\u11B9" },
            new string [] { "\uC53B", "\uC53B", "\u110A\u1175\u11BA", "\uC53B", "\u110A\u1175\u11BA" },
            new string [] { "\uC53C", "\uC53C", "\u110A\u1175\u11BB", "\uC53C", "\u110A\u1175\u11BB" },
            new string [] { "\uC53D", "\uC53D", "\u110A\u1175\u11BC", "\uC53D", "\u110A\u1175\u11BC" },
            new string [] { "\uC53E", "\uC53E", "\u110A\u1175\u11BD", "\uC53E", "\u110A\u1175\u11BD" },
            new string [] { "\uC53F", "\uC53F", "\u110A\u1175\u11BE", "\uC53F", "\u110A\u1175\u11BE" },
            new string [] { "\uC540", "\uC540", "\u110A\u1175\u11BF", "\uC540", "\u110A\u1175\u11BF" },
            new string [] { "\uC541", "\uC541", "\u110A\u1175\u11C0", "\uC541", "\u110A\u1175\u11C0" },
            new string [] { "\uC542", "\uC542", "\u110A\u1175\u11C1", "\uC542", "\u110A\u1175\u11C1" },
            new string [] { "\uC543", "\uC543", "\u110A\u1175\u11C2", "\uC543", "\u110A\u1175\u11C2" },
            new string [] { "\uC544", "\uC544", "\u110B\u1161", "\uC544", "\u110B\u1161" },
            new string [] { "\uC545", "\uC545", "\u110B\u1161\u11A8", "\uC545", "\u110B\u1161\u11A8" },
            new string [] { "\uC546", "\uC546", "\u110B\u1161\u11A9", "\uC546", "\u110B\u1161\u11A9" },
            new string [] { "\uC547", "\uC547", "\u110B\u1161\u11AA", "\uC547", "\u110B\u1161\u11AA" },
            new string [] { "\uC548", "\uC548", "\u110B\u1161\u11AB", "\uC548", "\u110B\u1161\u11AB" },
            new string [] { "\uC549", "\uC549", "\u110B\u1161\u11AC", "\uC549", "\u110B\u1161\u11AC" },
            new string [] { "\uC54A", "\uC54A", "\u110B\u1161\u11AD", "\uC54A", "\u110B\u1161\u11AD" },
            new string [] { "\uC54B", "\uC54B", "\u110B\u1161\u11AE", "\uC54B", "\u110B\u1161\u11AE" },
            new string [] { "\uC54C", "\uC54C", "\u110B\u1161\u11AF", "\uC54C", "\u110B\u1161\u11AF" },
            new string [] { "\uC54D", "\uC54D", "\u110B\u1161\u11B0", "\uC54D", "\u110B\u1161\u11B0" },
            new string [] { "\uC54E", "\uC54E", "\u110B\u1161\u11B1", "\uC54E", "\u110B\u1161\u11B1" },
            new string [] { "\uC54F", "\uC54F", "\u110B\u1161\u11B2", "\uC54F", "\u110B\u1161\u11B2" },
            new string [] { "\uC550", "\uC550", "\u110B\u1161\u11B3", "\uC550", "\u110B\u1161\u11B3" },
            new string [] { "\uC551", "\uC551", "\u110B\u1161\u11B4", "\uC551", "\u110B\u1161\u11B4" },
            new string [] { "\uC552", "\uC552", "\u110B\u1161\u11B5", "\uC552", "\u110B\u1161\u11B5" },
            new string [] { "\uC553", "\uC553", "\u110B\u1161\u11B6", "\uC553", "\u110B\u1161\u11B6" },
            new string [] { "\uC554", "\uC554", "\u110B\u1161\u11B7", "\uC554", "\u110B\u1161\u11B7" },
            new string [] { "\uC555", "\uC555", "\u110B\u1161\u11B8", "\uC555", "\u110B\u1161\u11B8" },
            new string [] { "\uC556", "\uC556", "\u110B\u1161\u11B9", "\uC556", "\u110B\u1161\u11B9" },
            new string [] { "\uC557", "\uC557", "\u110B\u1161\u11BA", "\uC557", "\u110B\u1161\u11BA" },
            new string [] { "\uC558", "\uC558", "\u110B\u1161\u11BB", "\uC558", "\u110B\u1161\u11BB" },
            new string [] { "\uC559", "\uC559", "\u110B\u1161\u11BC", "\uC559", "\u110B\u1161\u11BC" },
            new string [] { "\uC55A", "\uC55A", "\u110B\u1161\u11BD", "\uC55A", "\u110B\u1161\u11BD" },
            new string [] { "\uC55B", "\uC55B", "\u110B\u1161\u11BE", "\uC55B", "\u110B\u1161\u11BE" },
            new string [] { "\uC55C", "\uC55C", "\u110B\u1161\u11BF", "\uC55C", "\u110B\u1161\u11BF" },
            new string [] { "\uC55D", "\uC55D", "\u110B\u1161\u11C0", "\uC55D", "\u110B\u1161\u11C0" },
            new string [] { "\uC55E", "\uC55E", "\u110B\u1161\u11C1", "\uC55E", "\u110B\u1161\u11C1" },
            new string [] { "\uC55F", "\uC55F", "\u110B\u1161\u11C2", "\uC55F", "\u110B\u1161\u11C2" },
            new string [] { "\uC560", "\uC560", "\u110B\u1162", "\uC560", "\u110B\u1162" },
            new string [] { "\uC561", "\uC561", "\u110B\u1162\u11A8", "\uC561", "\u110B\u1162\u11A8" },
            new string [] { "\uC562", "\uC562", "\u110B\u1162\u11A9", "\uC562", "\u110B\u1162\u11A9" },
            new string [] { "\uC563", "\uC563", "\u110B\u1162\u11AA", "\uC563", "\u110B\u1162\u11AA" },
            new string [] { "\uC564", "\uC564", "\u110B\u1162\u11AB", "\uC564", "\u110B\u1162\u11AB" },
            new string [] { "\uC565", "\uC565", "\u110B\u1162\u11AC", "\uC565", "\u110B\u1162\u11AC" },
            new string [] { "\uC566", "\uC566", "\u110B\u1162\u11AD", "\uC566", "\u110B\u1162\u11AD" },
            new string [] { "\uC567", "\uC567", "\u110B\u1162\u11AE", "\uC567", "\u110B\u1162\u11AE" },
            new string [] { "\uC568", "\uC568", "\u110B\u1162\u11AF", "\uC568", "\u110B\u1162\u11AF" },
            new string [] { "\uC569", "\uC569", "\u110B\u1162\u11B0", "\uC569", "\u110B\u1162\u11B0" },
            new string [] { "\uC56A", "\uC56A", "\u110B\u1162\u11B1", "\uC56A", "\u110B\u1162\u11B1" },
            new string [] { "\uC56B", "\uC56B", "\u110B\u1162\u11B2", "\uC56B", "\u110B\u1162\u11B2" },
            new string [] { "\uC56C", "\uC56C", "\u110B\u1162\u11B3", "\uC56C", "\u110B\u1162\u11B3" },
            new string [] { "\uC56D", "\uC56D", "\u110B\u1162\u11B4", "\uC56D", "\u110B\u1162\u11B4" },
            new string [] { "\uC56E", "\uC56E", "\u110B\u1162\u11B5", "\uC56E", "\u110B\u1162\u11B5" },
            new string [] { "\uC56F", "\uC56F", "\u110B\u1162\u11B6", "\uC56F", "\u110B\u1162\u11B6" },
            new string [] { "\uC570", "\uC570", "\u110B\u1162\u11B7", "\uC570", "\u110B\u1162\u11B7" },
            new string [] { "\uC571", "\uC571", "\u110B\u1162\u11B8", "\uC571", "\u110B\u1162\u11B8" },
            new string [] { "\uC572", "\uC572", "\u110B\u1162\u11B9", "\uC572", "\u110B\u1162\u11B9" },
            new string [] { "\uC573", "\uC573", "\u110B\u1162\u11BA", "\uC573", "\u110B\u1162\u11BA" },
            new string [] { "\uC574", "\uC574", "\u110B\u1162\u11BB", "\uC574", "\u110B\u1162\u11BB" },
            new string [] { "\uC575", "\uC575", "\u110B\u1162\u11BC", "\uC575", "\u110B\u1162\u11BC" },
            new string [] { "\uC576", "\uC576", "\u110B\u1162\u11BD", "\uC576", "\u110B\u1162\u11BD" },
            new string [] { "\uC577", "\uC577", "\u110B\u1162\u11BE", "\uC577", "\u110B\u1162\u11BE" },
            new string [] { "\uC578", "\uC578", "\u110B\u1162\u11BF", "\uC578", "\u110B\u1162\u11BF" },
            new string [] { "\uC579", "\uC579", "\u110B\u1162\u11C0", "\uC579", "\u110B\u1162\u11C0" },
            new string [] { "\uC57A", "\uC57A", "\u110B\u1162\u11C1", "\uC57A", "\u110B\u1162\u11C1" },
            new string [] { "\uC57B", "\uC57B", "\u110B\u1162\u11C2", "\uC57B", "\u110B\u1162\u11C2" },
            new string [] { "\uC57C", "\uC57C", "\u110B\u1163", "\uC57C", "\u110B\u1163" },
            new string [] { "\uC57D", "\uC57D", "\u110B\u1163\u11A8", "\uC57D", "\u110B\u1163\u11A8" },
            new string [] { "\uC57E", "\uC57E", "\u110B\u1163\u11A9", "\uC57E", "\u110B\u1163\u11A9" },
            new string [] { "\uC57F", "\uC57F", "\u110B\u1163\u11AA", "\uC57F", "\u110B\u1163\u11AA" },
            new string [] { "\uC580", "\uC580", "\u110B\u1163\u11AB", "\uC580", "\u110B\u1163\u11AB" },
            new string [] { "\uC581", "\uC581", "\u110B\u1163\u11AC", "\uC581", "\u110B\u1163\u11AC" },
            new string [] { "\uC582", "\uC582", "\u110B\u1163\u11AD", "\uC582", "\u110B\u1163\u11AD" },
            new string [] { "\uC583", "\uC583", "\u110B\u1163\u11AE", "\uC583", "\u110B\u1163\u11AE" },
            new string [] { "\uC584", "\uC584", "\u110B\u1163\u11AF", "\uC584", "\u110B\u1163\u11AF" },
            new string [] { "\uC585", "\uC585", "\u110B\u1163\u11B0", "\uC585", "\u110B\u1163\u11B0" },
            new string [] { "\uC586", "\uC586", "\u110B\u1163\u11B1", "\uC586", "\u110B\u1163\u11B1" },
            new string [] { "\uC587", "\uC587", "\u110B\u1163\u11B2", "\uC587", "\u110B\u1163\u11B2" },
            new string [] { "\uC588", "\uC588", "\u110B\u1163\u11B3", "\uC588", "\u110B\u1163\u11B3" },
            new string [] { "\uC589", "\uC589", "\u110B\u1163\u11B4", "\uC589", "\u110B\u1163\u11B4" },
            new string [] { "\uC58A", "\uC58A", "\u110B\u1163\u11B5", "\uC58A", "\u110B\u1163\u11B5" },
            new string [] { "\uC58B", "\uC58B", "\u110B\u1163\u11B6", "\uC58B", "\u110B\u1163\u11B6" },
            new string [] { "\uC58C", "\uC58C", "\u110B\u1163\u11B7", "\uC58C", "\u110B\u1163\u11B7" },
            new string [] { "\uC58D", "\uC58D", "\u110B\u1163\u11B8", "\uC58D", "\u110B\u1163\u11B8" },
            new string [] { "\uC58E", "\uC58E", "\u110B\u1163\u11B9", "\uC58E", "\u110B\u1163\u11B9" },
            new string [] { "\uC58F", "\uC58F", "\u110B\u1163\u11BA", "\uC58F", "\u110B\u1163\u11BA" },
            new string [] { "\uC590", "\uC590", "\u110B\u1163\u11BB", "\uC590", "\u110B\u1163\u11BB" },
            new string [] { "\uC591", "\uC591", "\u110B\u1163\u11BC", "\uC591", "\u110B\u1163\u11BC" },
            new string [] { "\uC592", "\uC592", "\u110B\u1163\u11BD", "\uC592", "\u110B\u1163\u11BD" },
            new string [] { "\uC593", "\uC593", "\u110B\u1163\u11BE", "\uC593", "\u110B\u1163\u11BE" },
            new string [] { "\uC594", "\uC594", "\u110B\u1163\u11BF", "\uC594", "\u110B\u1163\u11BF" },
            new string [] { "\uC595", "\uC595", "\u110B\u1163\u11C0", "\uC595", "\u110B\u1163\u11C0" },
            new string [] { "\uC596", "\uC596", "\u110B\u1163\u11C1", "\uC596", "\u110B\u1163\u11C1" },
            new string [] { "\uC597", "\uC597", "\u110B\u1163\u11C2", "\uC597", "\u110B\u1163\u11C2" },
            new string [] { "\uC598", "\uC598", "\u110B\u1164", "\uC598", "\u110B\u1164" },
            new string [] { "\uC599", "\uC599", "\u110B\u1164\u11A8", "\uC599", "\u110B\u1164\u11A8" },
            new string [] { "\uC59A", "\uC59A", "\u110B\u1164\u11A9", "\uC59A", "\u110B\u1164\u11A9" },
            new string [] { "\uC59B", "\uC59B", "\u110B\u1164\u11AA", "\uC59B", "\u110B\u1164\u11AA" },
            new string [] { "\uC59C", "\uC59C", "\u110B\u1164\u11AB", "\uC59C", "\u110B\u1164\u11AB" },
            new string [] { "\uC59D", "\uC59D", "\u110B\u1164\u11AC", "\uC59D", "\u110B\u1164\u11AC" },
            new string [] { "\uC59E", "\uC59E", "\u110B\u1164\u11AD", "\uC59E", "\u110B\u1164\u11AD" },
            new string [] { "\uC59F", "\uC59F", "\u110B\u1164\u11AE", "\uC59F", "\u110B\u1164\u11AE" },
            new string [] { "\uC5A0", "\uC5A0", "\u110B\u1164\u11AF", "\uC5A0", "\u110B\u1164\u11AF" },
            new string [] { "\uC5A1", "\uC5A1", "\u110B\u1164\u11B0", "\uC5A1", "\u110B\u1164\u11B0" },
            new string [] { "\uC5A2", "\uC5A2", "\u110B\u1164\u11B1", "\uC5A2", "\u110B\u1164\u11B1" },
            new string [] { "\uC5A3", "\uC5A3", "\u110B\u1164\u11B2", "\uC5A3", "\u110B\u1164\u11B2" },
            new string [] { "\uC5A4", "\uC5A4", "\u110B\u1164\u11B3", "\uC5A4", "\u110B\u1164\u11B3" },
            new string [] { "\uC5A5", "\uC5A5", "\u110B\u1164\u11B4", "\uC5A5", "\u110B\u1164\u11B4" },
            new string [] { "\uC5A6", "\uC5A6", "\u110B\u1164\u11B5", "\uC5A6", "\u110B\u1164\u11B5" },
            new string [] { "\uC5A7", "\uC5A7", "\u110B\u1164\u11B6", "\uC5A7", "\u110B\u1164\u11B6" },
            new string [] { "\uC5A8", "\uC5A8", "\u110B\u1164\u11B7", "\uC5A8", "\u110B\u1164\u11B7" },
            new string [] { "\uC5A9", "\uC5A9", "\u110B\u1164\u11B8", "\uC5A9", "\u110B\u1164\u11B8" },
            new string [] { "\uC5AA", "\uC5AA", "\u110B\u1164\u11B9", "\uC5AA", "\u110B\u1164\u11B9" },
            new string [] { "\uC5AB", "\uC5AB", "\u110B\u1164\u11BA", "\uC5AB", "\u110B\u1164\u11BA" },
            new string [] { "\uC5AC", "\uC5AC", "\u110B\u1164\u11BB", "\uC5AC", "\u110B\u1164\u11BB" },
            new string [] { "\uC5AD", "\uC5AD", "\u110B\u1164\u11BC", "\uC5AD", "\u110B\u1164\u11BC" },
            new string [] { "\uC5AE", "\uC5AE", "\u110B\u1164\u11BD", "\uC5AE", "\u110B\u1164\u11BD" },
            new string [] { "\uC5AF", "\uC5AF", "\u110B\u1164\u11BE", "\uC5AF", "\u110B\u1164\u11BE" },
            new string [] { "\uC5B0", "\uC5B0", "\u110B\u1164\u11BF", "\uC5B0", "\u110B\u1164\u11BF" },
            new string [] { "\uC5B1", "\uC5B1", "\u110B\u1164\u11C0", "\uC5B1", "\u110B\u1164\u11C0" },
            new string [] { "\uC5B2", "\uC5B2", "\u110B\u1164\u11C1", "\uC5B2", "\u110B\u1164\u11C1" },
            new string [] { "\uC5B3", "\uC5B3", "\u110B\u1164\u11C2", "\uC5B3", "\u110B\u1164\u11C2" },
            new string [] { "\uC5B4", "\uC5B4", "\u110B\u1165", "\uC5B4", "\u110B\u1165" },
            new string [] { "\uC5B5", "\uC5B5", "\u110B\u1165\u11A8", "\uC5B5", "\u110B\u1165\u11A8" },
            new string [] { "\uC5B6", "\uC5B6", "\u110B\u1165\u11A9", "\uC5B6", "\u110B\u1165\u11A9" },
            new string [] { "\uC5B7", "\uC5B7", "\u110B\u1165\u11AA", "\uC5B7", "\u110B\u1165\u11AA" },
            new string [] { "\uC5B8", "\uC5B8", "\u110B\u1165\u11AB", "\uC5B8", "\u110B\u1165\u11AB" },
            new string [] { "\uC5B9", "\uC5B9", "\u110B\u1165\u11AC", "\uC5B9", "\u110B\u1165\u11AC" },
            new string [] { "\uC5BA", "\uC5BA", "\u110B\u1165\u11AD", "\uC5BA", "\u110B\u1165\u11AD" },
            new string [] { "\uC5BB", "\uC5BB", "\u110B\u1165\u11AE", "\uC5BB", "\u110B\u1165\u11AE" },
            new string [] { "\uC5BC", "\uC5BC", "\u110B\u1165\u11AF", "\uC5BC", "\u110B\u1165\u11AF" },
            new string [] { "\uC5BD", "\uC5BD", "\u110B\u1165\u11B0", "\uC5BD", "\u110B\u1165\u11B0" },
            new string [] { "\uC5BE", "\uC5BE", "\u110B\u1165\u11B1", "\uC5BE", "\u110B\u1165\u11B1" },
            new string [] { "\uC5BF", "\uC5BF", "\u110B\u1165\u11B2", "\uC5BF", "\u110B\u1165\u11B2" },
            new string [] { "\uC5C0", "\uC5C0", "\u110B\u1165\u11B3", "\uC5C0", "\u110B\u1165\u11B3" },
            new string [] { "\uC5C1", "\uC5C1", "\u110B\u1165\u11B4", "\uC5C1", "\u110B\u1165\u11B4" },
            new string [] { "\uC5C2", "\uC5C2", "\u110B\u1165\u11B5", "\uC5C2", "\u110B\u1165\u11B5" },
            new string [] { "\uC5C3", "\uC5C3", "\u110B\u1165\u11B6", "\uC5C3", "\u110B\u1165\u11B6" },
            new string [] { "\uC5C4", "\uC5C4", "\u110B\u1165\u11B7", "\uC5C4", "\u110B\u1165\u11B7" },
            new string [] { "\uC5C5", "\uC5C5", "\u110B\u1165\u11B8", "\uC5C5", "\u110B\u1165\u11B8" },
            new string [] { "\uC5C6", "\uC5C6", "\u110B\u1165\u11B9", "\uC5C6", "\u110B\u1165\u11B9" },
            new string [] { "\uC5C7", "\uC5C7", "\u110B\u1165\u11BA", "\uC5C7", "\u110B\u1165\u11BA" },
            new string [] { "\uC5C8", "\uC5C8", "\u110B\u1165\u11BB", "\uC5C8", "\u110B\u1165\u11BB" },
            new string [] { "\uC5C9", "\uC5C9", "\u110B\u1165\u11BC", "\uC5C9", "\u110B\u1165\u11BC" },
            new string [] { "\uC5CA", "\uC5CA", "\u110B\u1165\u11BD", "\uC5CA", "\u110B\u1165\u11BD" },
            new string [] { "\uC5CB", "\uC5CB", "\u110B\u1165\u11BE", "\uC5CB", "\u110B\u1165\u11BE" },
            new string [] { "\uC5CC", "\uC5CC", "\u110B\u1165\u11BF", "\uC5CC", "\u110B\u1165\u11BF" },
            new string [] { "\uC5CD", "\uC5CD", "\u110B\u1165\u11C0", "\uC5CD", "\u110B\u1165\u11C0" },
            new string [] { "\uC5CE", "\uC5CE", "\u110B\u1165\u11C1", "\uC5CE", "\u110B\u1165\u11C1" },
            new string [] { "\uC5CF", "\uC5CF", "\u110B\u1165\u11C2", "\uC5CF", "\u110B\u1165\u11C2" },
            new string [] { "\uC5D0", "\uC5D0", "\u110B\u1166", "\uC5D0", "\u110B\u1166" },
            new string [] { "\uC5D1", "\uC5D1", "\u110B\u1166\u11A8", "\uC5D1", "\u110B\u1166\u11A8" },
            new string [] { "\uC5D2", "\uC5D2", "\u110B\u1166\u11A9", "\uC5D2", "\u110B\u1166\u11A9" },
            new string [] { "\uC5D3", "\uC5D3", "\u110B\u1166\u11AA", "\uC5D3", "\u110B\u1166\u11AA" },
            new string [] { "\uC5D4", "\uC5D4", "\u110B\u1166\u11AB", "\uC5D4", "\u110B\u1166\u11AB" },
            new string [] { "\uC5D5", "\uC5D5", "\u110B\u1166\u11AC", "\uC5D5", "\u110B\u1166\u11AC" },
            new string [] { "\uC5D6", "\uC5D6", "\u110B\u1166\u11AD", "\uC5D6", "\u110B\u1166\u11AD" },
            new string [] { "\uC5D7", "\uC5D7", "\u110B\u1166\u11AE", "\uC5D7", "\u110B\u1166\u11AE" },
            new string [] { "\uC5D8", "\uC5D8", "\u110B\u1166\u11AF", "\uC5D8", "\u110B\u1166\u11AF" },
            new string [] { "\uC5D9", "\uC5D9", "\u110B\u1166\u11B0", "\uC5D9", "\u110B\u1166\u11B0" },
            new string [] { "\uC5DA", "\uC5DA", "\u110B\u1166\u11B1", "\uC5DA", "\u110B\u1166\u11B1" },
            new string [] { "\uC5DB", "\uC5DB", "\u110B\u1166\u11B2", "\uC5DB", "\u110B\u1166\u11B2" },
            new string [] { "\uC5DC", "\uC5DC", "\u110B\u1166\u11B3", "\uC5DC", "\u110B\u1166\u11B3" },
            new string [] { "\uC5DD", "\uC5DD", "\u110B\u1166\u11B4", "\uC5DD", "\u110B\u1166\u11B4" },
            new string [] { "\uC5DE", "\uC5DE", "\u110B\u1166\u11B5", "\uC5DE", "\u110B\u1166\u11B5" },
            new string [] { "\uC5DF", "\uC5DF", "\u110B\u1166\u11B6", "\uC5DF", "\u110B\u1166\u11B6" },
            new string [] { "\uC5E0", "\uC5E0", "\u110B\u1166\u11B7", "\uC5E0", "\u110B\u1166\u11B7" },
            new string [] { "\uC5E1", "\uC5E1", "\u110B\u1166\u11B8", "\uC5E1", "\u110B\u1166\u11B8" },
            new string [] { "\uC5E2", "\uC5E2", "\u110B\u1166\u11B9", "\uC5E2", "\u110B\u1166\u11B9" },
            new string [] { "\uC5E3", "\uC5E3", "\u110B\u1166\u11BA", "\uC5E3", "\u110B\u1166\u11BA" },
            new string [] { "\uC5E4", "\uC5E4", "\u110B\u1166\u11BB", "\uC5E4", "\u110B\u1166\u11BB" },
            new string [] { "\uC5E5", "\uC5E5", "\u110B\u1166\u11BC", "\uC5E5", "\u110B\u1166\u11BC" },
            new string [] { "\uC5E6", "\uC5E6", "\u110B\u1166\u11BD", "\uC5E6", "\u110B\u1166\u11BD" },
            new string [] { "\uC5E7", "\uC5E7", "\u110B\u1166\u11BE", "\uC5E7", "\u110B\u1166\u11BE" },
            new string [] { "\uC5E8", "\uC5E8", "\u110B\u1166\u11BF", "\uC5E8", "\u110B\u1166\u11BF" },
            new string [] { "\uC5E9", "\uC5E9", "\u110B\u1166\u11C0", "\uC5E9", "\u110B\u1166\u11C0" },
            new string [] { "\uC5EA", "\uC5EA", "\u110B\u1166\u11C1", "\uC5EA", "\u110B\u1166\u11C1" },
            new string [] { "\uC5EB", "\uC5EB", "\u110B\u1166\u11C2", "\uC5EB", "\u110B\u1166\u11C2" },
            new string [] { "\uC5EC", "\uC5EC", "\u110B\u1167", "\uC5EC", "\u110B\u1167" },
            new string [] { "\uC5ED", "\uC5ED", "\u110B\u1167\u11A8", "\uC5ED", "\u110B\u1167\u11A8" },
            new string [] { "\uC5EE", "\uC5EE", "\u110B\u1167\u11A9", "\uC5EE", "\u110B\u1167\u11A9" },
            new string [] { "\uC5EF", "\uC5EF", "\u110B\u1167\u11AA", "\uC5EF", "\u110B\u1167\u11AA" },
            new string [] { "\uC5F0", "\uC5F0", "\u110B\u1167\u11AB", "\uC5F0", "\u110B\u1167\u11AB" },
            new string [] { "\uC5F1", "\uC5F1", "\u110B\u1167\u11AC", "\uC5F1", "\u110B\u1167\u11AC" },
            new string [] { "\uC5F2", "\uC5F2", "\u110B\u1167\u11AD", "\uC5F2", "\u110B\u1167\u11AD" },
            new string [] { "\uC5F3", "\uC5F3", "\u110B\u1167\u11AE", "\uC5F3", "\u110B\u1167\u11AE" },
            new string [] { "\uC5F4", "\uC5F4", "\u110B\u1167\u11AF", "\uC5F4", "\u110B\u1167\u11AF" },
            new string [] { "\uC5F5", "\uC5F5", "\u110B\u1167\u11B0", "\uC5F5", "\u110B\u1167\u11B0" },
            new string [] { "\uC5F6", "\uC5F6", "\u110B\u1167\u11B1", "\uC5F6", "\u110B\u1167\u11B1" },
            new string [] { "\uC5F7", "\uC5F7", "\u110B\u1167\u11B2", "\uC5F7", "\u110B\u1167\u11B2" },
            new string [] { "\uC5F8", "\uC5F8", "\u110B\u1167\u11B3", "\uC5F8", "\u110B\u1167\u11B3" },
            new string [] { "\uC5F9", "\uC5F9", "\u110B\u1167\u11B4", "\uC5F9", "\u110B\u1167\u11B4" },
            new string [] { "\uC5FA", "\uC5FA", "\u110B\u1167\u11B5", "\uC5FA", "\u110B\u1167\u11B5" },
            new string [] { "\uC5FB", "\uC5FB", "\u110B\u1167\u11B6", "\uC5FB", "\u110B\u1167\u11B6" },
            new string [] { "\uC5FC", "\uC5FC", "\u110B\u1167\u11B7", "\uC5FC", "\u110B\u1167\u11B7" },
            new string [] { "\uC5FD", "\uC5FD", "\u110B\u1167\u11B8", "\uC5FD", "\u110B\u1167\u11B8" },
            new string [] { "\uC5FE", "\uC5FE", "\u110B\u1167\u11B9", "\uC5FE", "\u110B\u1167\u11B9" },
            new string [] { "\uC5FF", "\uC5FF", "\u110B\u1167\u11BA", "\uC5FF", "\u110B\u1167\u11BA" },
            new string [] { "\uC600", "\uC600", "\u110B\u1167\u11BB", "\uC600", "\u110B\u1167\u11BB" },
            new string [] { "\uC601", "\uC601", "\u110B\u1167\u11BC", "\uC601", "\u110B\u1167\u11BC" },
            new string [] { "\uC602", "\uC602", "\u110B\u1167\u11BD", "\uC602", "\u110B\u1167\u11BD" },
            new string [] { "\uC603", "\uC603", "\u110B\u1167\u11BE", "\uC603", "\u110B\u1167\u11BE" },
            new string [] { "\uC604", "\uC604", "\u110B\u1167\u11BF", "\uC604", "\u110B\u1167\u11BF" },
            new string [] { "\uC605", "\uC605", "\u110B\u1167\u11C0", "\uC605", "\u110B\u1167\u11C0" },
            new string [] { "\uC606", "\uC606", "\u110B\u1167\u11C1", "\uC606", "\u110B\u1167\u11C1" },
            new string [] { "\uC607", "\uC607", "\u110B\u1167\u11C2", "\uC607", "\u110B\u1167\u11C2" },
            new string [] { "\uC608", "\uC608", "\u110B\u1168", "\uC608", "\u110B\u1168" },
            new string [] { "\uC609", "\uC609", "\u110B\u1168\u11A8", "\uC609", "\u110B\u1168\u11A8" },
            new string [] { "\uC60A", "\uC60A", "\u110B\u1168\u11A9", "\uC60A", "\u110B\u1168\u11A9" },
            new string [] { "\uC60B", "\uC60B", "\u110B\u1168\u11AA", "\uC60B", "\u110B\u1168\u11AA" },
            new string [] { "\uC60C", "\uC60C", "\u110B\u1168\u11AB", "\uC60C", "\u110B\u1168\u11AB" },
            new string [] { "\uC60D", "\uC60D", "\u110B\u1168\u11AC", "\uC60D", "\u110B\u1168\u11AC" },
            new string [] { "\uC60E", "\uC60E", "\u110B\u1168\u11AD", "\uC60E", "\u110B\u1168\u11AD" },
            new string [] { "\uC60F", "\uC60F", "\u110B\u1168\u11AE", "\uC60F", "\u110B\u1168\u11AE" },
            new string [] { "\uC610", "\uC610", "\u110B\u1168\u11AF", "\uC610", "\u110B\u1168\u11AF" },
            new string [] { "\uC611", "\uC611", "\u110B\u1168\u11B0", "\uC611", "\u110B\u1168\u11B0" },
            new string [] { "\uC612", "\uC612", "\u110B\u1168\u11B1", "\uC612", "\u110B\u1168\u11B1" },
            new string [] { "\uC613", "\uC613", "\u110B\u1168\u11B2", "\uC613", "\u110B\u1168\u11B2" },
            new string [] { "\uC614", "\uC614", "\u110B\u1168\u11B3", "\uC614", "\u110B\u1168\u11B3" },
            new string [] { "\uC615", "\uC615", "\u110B\u1168\u11B4", "\uC615", "\u110B\u1168\u11B4" },
            new string [] { "\uC616", "\uC616", "\u110B\u1168\u11B5", "\uC616", "\u110B\u1168\u11B5" },
            new string [] { "\uC617", "\uC617", "\u110B\u1168\u11B6", "\uC617", "\u110B\u1168\u11B6" },
            new string [] { "\uC618", "\uC618", "\u110B\u1168\u11B7", "\uC618", "\u110B\u1168\u11B7" },
            new string [] { "\uC619", "\uC619", "\u110B\u1168\u11B8", "\uC619", "\u110B\u1168\u11B8" },
            new string [] { "\uC61A", "\uC61A", "\u110B\u1168\u11B9", "\uC61A", "\u110B\u1168\u11B9" },
            new string [] { "\uC61B", "\uC61B", "\u110B\u1168\u11BA", "\uC61B", "\u110B\u1168\u11BA" },
            new string [] { "\uC61C", "\uC61C", "\u110B\u1168\u11BB", "\uC61C", "\u110B\u1168\u11BB" },
            new string [] { "\uC61D", "\uC61D", "\u110B\u1168\u11BC", "\uC61D", "\u110B\u1168\u11BC" },
            new string [] { "\uC61E", "\uC61E", "\u110B\u1168\u11BD", "\uC61E", "\u110B\u1168\u11BD" },
            new string [] { "\uC61F", "\uC61F", "\u110B\u1168\u11BE", "\uC61F", "\u110B\u1168\u11BE" },
            new string [] { "\uC620", "\uC620", "\u110B\u1168\u11BF", "\uC620", "\u110B\u1168\u11BF" },
            new string [] { "\uC621", "\uC621", "\u110B\u1168\u11C0", "\uC621", "\u110B\u1168\u11C0" },
            new string [] { "\uC622", "\uC622", "\u110B\u1168\u11C1", "\uC622", "\u110B\u1168\u11C1" },
            new string [] { "\uC623", "\uC623", "\u110B\u1168\u11C2", "\uC623", "\u110B\u1168\u11C2" },
            new string [] { "\uC624", "\uC624", "\u110B\u1169", "\uC624", "\u110B\u1169" },
            new string [] { "\uC625", "\uC625", "\u110B\u1169\u11A8", "\uC625", "\u110B\u1169\u11A8" },
            new string [] { "\uC626", "\uC626", "\u110B\u1169\u11A9", "\uC626", "\u110B\u1169\u11A9" },
            new string [] { "\uC627", "\uC627", "\u110B\u1169\u11AA", "\uC627", "\u110B\u1169\u11AA" },
            new string [] { "\uC628", "\uC628", "\u110B\u1169\u11AB", "\uC628", "\u110B\u1169\u11AB" },
            new string [] { "\uC629", "\uC629", "\u110B\u1169\u11AC", "\uC629", "\u110B\u1169\u11AC" },
            new string [] { "\uC62A", "\uC62A", "\u110B\u1169\u11AD", "\uC62A", "\u110B\u1169\u11AD" },
            new string [] { "\uC62B", "\uC62B", "\u110B\u1169\u11AE", "\uC62B", "\u110B\u1169\u11AE" },
            new string [] { "\uC62C", "\uC62C", "\u110B\u1169\u11AF", "\uC62C", "\u110B\u1169\u11AF" },
            new string [] { "\uC62D", "\uC62D", "\u110B\u1169\u11B0", "\uC62D", "\u110B\u1169\u11B0" },
            new string [] { "\uC62E", "\uC62E", "\u110B\u1169\u11B1", "\uC62E", "\u110B\u1169\u11B1" },
            new string [] { "\uC62F", "\uC62F", "\u110B\u1169\u11B2", "\uC62F", "\u110B\u1169\u11B2" },
            new string [] { "\uC630", "\uC630", "\u110B\u1169\u11B3", "\uC630", "\u110B\u1169\u11B3" },
            new string [] { "\uC631", "\uC631", "\u110B\u1169\u11B4", "\uC631", "\u110B\u1169\u11B4" },
            new string [] { "\uC632", "\uC632", "\u110B\u1169\u11B5", "\uC632", "\u110B\u1169\u11B5" },
            new string [] { "\uC633", "\uC633", "\u110B\u1169\u11B6", "\uC633", "\u110B\u1169\u11B6" },
            new string [] { "\uC634", "\uC634", "\u110B\u1169\u11B7", "\uC634", "\u110B\u1169\u11B7" },
            new string [] { "\uC635", "\uC635", "\u110B\u1169\u11B8", "\uC635", "\u110B\u1169\u11B8" },
            new string [] { "\uC636", "\uC636", "\u110B\u1169\u11B9", "\uC636", "\u110B\u1169\u11B9" },
            new string [] { "\uC637", "\uC637", "\u110B\u1169\u11BA", "\uC637", "\u110B\u1169\u11BA" },
            new string [] { "\uC638", "\uC638", "\u110B\u1169\u11BB", "\uC638", "\u110B\u1169\u11BB" },
            new string [] { "\uC639", "\uC639", "\u110B\u1169\u11BC", "\uC639", "\u110B\u1169\u11BC" },
            new string [] { "\uC63A", "\uC63A", "\u110B\u1169\u11BD", "\uC63A", "\u110B\u1169\u11BD" },
            new string [] { "\uC63B", "\uC63B", "\u110B\u1169\u11BE", "\uC63B", "\u110B\u1169\u11BE" },
            new string [] { "\uC63C", "\uC63C", "\u110B\u1169\u11BF", "\uC63C", "\u110B\u1169\u11BF" },
            new string [] { "\uC63D", "\uC63D", "\u110B\u1169\u11C0", "\uC63D", "\u110B\u1169\u11C0" },
            new string [] { "\uC63E", "\uC63E", "\u110B\u1169\u11C1", "\uC63E", "\u110B\u1169\u11C1" },
            new string [] { "\uC63F", "\uC63F", "\u110B\u1169\u11C2", "\uC63F", "\u110B\u1169\u11C2" },
            new string [] { "\uC640", "\uC640", "\u110B\u116A", "\uC640", "\u110B\u116A" },
            new string [] { "\uC641", "\uC641", "\u110B\u116A\u11A8", "\uC641", "\u110B\u116A\u11A8" },
            new string [] { "\uC642", "\uC642", "\u110B\u116A\u11A9", "\uC642", "\u110B\u116A\u11A9" },
            new string [] { "\uC643", "\uC643", "\u110B\u116A\u11AA", "\uC643", "\u110B\u116A\u11AA" },
            new string [] { "\uC644", "\uC644", "\u110B\u116A\u11AB", "\uC644", "\u110B\u116A\u11AB" },
            new string [] { "\uC645", "\uC645", "\u110B\u116A\u11AC", "\uC645", "\u110B\u116A\u11AC" },
            new string [] { "\uC646", "\uC646", "\u110B\u116A\u11AD", "\uC646", "\u110B\u116A\u11AD" },
            new string [] { "\uC647", "\uC647", "\u110B\u116A\u11AE", "\uC647", "\u110B\u116A\u11AE" },
            new string [] { "\uC648", "\uC648", "\u110B\u116A\u11AF", "\uC648", "\u110B\u116A\u11AF" },
            new string [] { "\uC649", "\uC649", "\u110B\u116A\u11B0", "\uC649", "\u110B\u116A\u11B0" },
            new string [] { "\uC64A", "\uC64A", "\u110B\u116A\u11B1", "\uC64A", "\u110B\u116A\u11B1" },
            new string [] { "\uC64B", "\uC64B", "\u110B\u116A\u11B2", "\uC64B", "\u110B\u116A\u11B2" },
            new string [] { "\uC64C", "\uC64C", "\u110B\u116A\u11B3", "\uC64C", "\u110B\u116A\u11B3" },
            new string [] { "\uC64D", "\uC64D", "\u110B\u116A\u11B4", "\uC64D", "\u110B\u116A\u11B4" },
            new string [] { "\uC64E", "\uC64E", "\u110B\u116A\u11B5", "\uC64E", "\u110B\u116A\u11B5" },
            new string [] { "\uC64F", "\uC64F", "\u110B\u116A\u11B6", "\uC64F", "\u110B\u116A\u11B6" },
            new string [] { "\uC650", "\uC650", "\u110B\u116A\u11B7", "\uC650", "\u110B\u116A\u11B7" },
            new string [] { "\uC651", "\uC651", "\u110B\u116A\u11B8", "\uC651", "\u110B\u116A\u11B8" },
            new string [] { "\uC652", "\uC652", "\u110B\u116A\u11B9", "\uC652", "\u110B\u116A\u11B9" },
            new string [] { "\uC653", "\uC653", "\u110B\u116A\u11BA", "\uC653", "\u110B\u116A\u11BA" },
            new string [] { "\uC654", "\uC654", "\u110B\u116A\u11BB", "\uC654", "\u110B\u116A\u11BB" },
            new string [] { "\uC655", "\uC655", "\u110B\u116A\u11BC", "\uC655", "\u110B\u116A\u11BC" },
            new string [] { "\uC656", "\uC656", "\u110B\u116A\u11BD", "\uC656", "\u110B\u116A\u11BD" },
            new string [] { "\uC657", "\uC657", "\u110B\u116A\u11BE", "\uC657", "\u110B\u116A\u11BE" },
            new string [] { "\uC658", "\uC658", "\u110B\u116A\u11BF", "\uC658", "\u110B\u116A\u11BF" },
            new string [] { "\uC659", "\uC659", "\u110B\u116A\u11C0", "\uC659", "\u110B\u116A\u11C0" },
            new string [] { "\uC65A", "\uC65A", "\u110B\u116A\u11C1", "\uC65A", "\u110B\u116A\u11C1" },
            new string [] { "\uC65B", "\uC65B", "\u110B\u116A\u11C2", "\uC65B", "\u110B\u116A\u11C2" },
            new string [] { "\uC65C", "\uC65C", "\u110B\u116B", "\uC65C", "\u110B\u116B" },
            new string [] { "\uC65D", "\uC65D", "\u110B\u116B\u11A8", "\uC65D", "\u110B\u116B\u11A8" },
            new string [] { "\uC65E", "\uC65E", "\u110B\u116B\u11A9", "\uC65E", "\u110B\u116B\u11A9" },
            new string [] { "\uC65F", "\uC65F", "\u110B\u116B\u11AA", "\uC65F", "\u110B\u116B\u11AA" },
            new string [] { "\uC660", "\uC660", "\u110B\u116B\u11AB", "\uC660", "\u110B\u116B\u11AB" },
            new string [] { "\uC661", "\uC661", "\u110B\u116B\u11AC", "\uC661", "\u110B\u116B\u11AC" },
            new string [] { "\uC662", "\uC662", "\u110B\u116B\u11AD", "\uC662", "\u110B\u116B\u11AD" },
            new string [] { "\uC663", "\uC663", "\u110B\u116B\u11AE", "\uC663", "\u110B\u116B\u11AE" },
            new string [] { "\uC664", "\uC664", "\u110B\u116B\u11AF", "\uC664", "\u110B\u116B\u11AF" },
            new string [] { "\uC665", "\uC665", "\u110B\u116B\u11B0", "\uC665", "\u110B\u116B\u11B0" },
            new string [] { "\uC666", "\uC666", "\u110B\u116B\u11B1", "\uC666", "\u110B\u116B\u11B1" },
            new string [] { "\uC667", "\uC667", "\u110B\u116B\u11B2", "\uC667", "\u110B\u116B\u11B2" },
            new string [] { "\uC668", "\uC668", "\u110B\u116B\u11B3", "\uC668", "\u110B\u116B\u11B3" },
            new string [] { "\uC669", "\uC669", "\u110B\u116B\u11B4", "\uC669", "\u110B\u116B\u11B4" },
            new string [] { "\uC66A", "\uC66A", "\u110B\u116B\u11B5", "\uC66A", "\u110B\u116B\u11B5" },
            new string [] { "\uC66B", "\uC66B", "\u110B\u116B\u11B6", "\uC66B", "\u110B\u116B\u11B6" },
            new string [] { "\uC66C", "\uC66C", "\u110B\u116B\u11B7", "\uC66C", "\u110B\u116B\u11B7" },
            new string [] { "\uC66D", "\uC66D", "\u110B\u116B\u11B8", "\uC66D", "\u110B\u116B\u11B8" },
            new string [] { "\uC66E", "\uC66E", "\u110B\u116B\u11B9", "\uC66E", "\u110B\u116B\u11B9" },
            new string [] { "\uC66F", "\uC66F", "\u110B\u116B\u11BA", "\uC66F", "\u110B\u116B\u11BA" },
            new string [] { "\uC670", "\uC670", "\u110B\u116B\u11BB", "\uC670", "\u110B\u116B\u11BB" },
            new string [] { "\uC671", "\uC671", "\u110B\u116B\u11BC", "\uC671", "\u110B\u116B\u11BC" },
            new string [] { "\uC672", "\uC672", "\u110B\u116B\u11BD", "\uC672", "\u110B\u116B\u11BD" },
            new string [] { "\uC673", "\uC673", "\u110B\u116B\u11BE", "\uC673", "\u110B\u116B\u11BE" },
            new string [] { "\uC674", "\uC674", "\u110B\u116B\u11BF", "\uC674", "\u110B\u116B\u11BF" },
            new string [] { "\uC675", "\uC675", "\u110B\u116B\u11C0", "\uC675", "\u110B\u116B\u11C0" },
            new string [] { "\uC676", "\uC676", "\u110B\u116B\u11C1", "\uC676", "\u110B\u116B\u11C1" },
            new string [] { "\uC677", "\uC677", "\u110B\u116B\u11C2", "\uC677", "\u110B\u116B\u11C2" },
            new string [] { "\uC678", "\uC678", "\u110B\u116C", "\uC678", "\u110B\u116C" },
            new string [] { "\uC679", "\uC679", "\u110B\u116C\u11A8", "\uC679", "\u110B\u116C\u11A8" },
            new string [] { "\uC67A", "\uC67A", "\u110B\u116C\u11A9", "\uC67A", "\u110B\u116C\u11A9" },
            new string [] { "\uC67B", "\uC67B", "\u110B\u116C\u11AA", "\uC67B", "\u110B\u116C\u11AA" },
            new string [] { "\uC67C", "\uC67C", "\u110B\u116C\u11AB", "\uC67C", "\u110B\u116C\u11AB" },
            new string [] { "\uC67D", "\uC67D", "\u110B\u116C\u11AC", "\uC67D", "\u110B\u116C\u11AC" },
            new string [] { "\uC67E", "\uC67E", "\u110B\u116C\u11AD", "\uC67E", "\u110B\u116C\u11AD" },
            new string [] { "\uC67F", "\uC67F", "\u110B\u116C\u11AE", "\uC67F", "\u110B\u116C\u11AE" },
            new string [] { "\uC680", "\uC680", "\u110B\u116C\u11AF", "\uC680", "\u110B\u116C\u11AF" },
            new string [] { "\uC681", "\uC681", "\u110B\u116C\u11B0", "\uC681", "\u110B\u116C\u11B0" },
            new string [] { "\uC682", "\uC682", "\u110B\u116C\u11B1", "\uC682", "\u110B\u116C\u11B1" },
            new string [] { "\uC683", "\uC683", "\u110B\u116C\u11B2", "\uC683", "\u110B\u116C\u11B2" },
            new string [] { "\uC684", "\uC684", "\u110B\u116C\u11B3", "\uC684", "\u110B\u116C\u11B3" },
            new string [] { "\uC685", "\uC685", "\u110B\u116C\u11B4", "\uC685", "\u110B\u116C\u11B4" },
            new string [] { "\uC686", "\uC686", "\u110B\u116C\u11B5", "\uC686", "\u110B\u116C\u11B5" },
            new string [] { "\uC687", "\uC687", "\u110B\u116C\u11B6", "\uC687", "\u110B\u116C\u11B6" },
            new string [] { "\uC688", "\uC688", "\u110B\u116C\u11B7", "\uC688", "\u110B\u116C\u11B7" },
            new string [] { "\uC689", "\uC689", "\u110B\u116C\u11B8", "\uC689", "\u110B\u116C\u11B8" },
            new string [] { "\uC68A", "\uC68A", "\u110B\u116C\u11B9", "\uC68A", "\u110B\u116C\u11B9" },
            new string [] { "\uC68B", "\uC68B", "\u110B\u116C\u11BA", "\uC68B", "\u110B\u116C\u11BA" },
            new string [] { "\uC68C", "\uC68C", "\u110B\u116C\u11BB", "\uC68C", "\u110B\u116C\u11BB" },
            new string [] { "\uC68D", "\uC68D", "\u110B\u116C\u11BC", "\uC68D", "\u110B\u116C\u11BC" },
            new string [] { "\uC68E", "\uC68E", "\u110B\u116C\u11BD", "\uC68E", "\u110B\u116C\u11BD" },
            new string [] { "\uC68F", "\uC68F", "\u110B\u116C\u11BE", "\uC68F", "\u110B\u116C\u11BE" },
            new string [] { "\uC690", "\uC690", "\u110B\u116C\u11BF", "\uC690", "\u110B\u116C\u11BF" },
            new string [] { "\uC691", "\uC691", "\u110B\u116C\u11C0", "\uC691", "\u110B\u116C\u11C0" },
            new string [] { "\uC692", "\uC692", "\u110B\u116C\u11C1", "\uC692", "\u110B\u116C\u11C1" },
            new string [] { "\uC693", "\uC693", "\u110B\u116C\u11C2", "\uC693", "\u110B\u116C\u11C2" },
            new string [] { "\uC694", "\uC694", "\u110B\u116D", "\uC694", "\u110B\u116D" },
            new string [] { "\uC695", "\uC695", "\u110B\u116D\u11A8", "\uC695", "\u110B\u116D\u11A8" },
            new string [] { "\uC696", "\uC696", "\u110B\u116D\u11A9", "\uC696", "\u110B\u116D\u11A9" },
            new string [] { "\uC697", "\uC697", "\u110B\u116D\u11AA", "\uC697", "\u110B\u116D\u11AA" },
            new string [] { "\uC698", "\uC698", "\u110B\u116D\u11AB", "\uC698", "\u110B\u116D\u11AB" },
            new string [] { "\uC699", "\uC699", "\u110B\u116D\u11AC", "\uC699", "\u110B\u116D\u11AC" },
            new string [] { "\uC69A", "\uC69A", "\u110B\u116D\u11AD", "\uC69A", "\u110B\u116D\u11AD" },
            new string [] { "\uC69B", "\uC69B", "\u110B\u116D\u11AE", "\uC69B", "\u110B\u116D\u11AE" },
            new string [] { "\uC69C", "\uC69C", "\u110B\u116D\u11AF", "\uC69C", "\u110B\u116D\u11AF" },
            new string [] { "\uC69D", "\uC69D", "\u110B\u116D\u11B0", "\uC69D", "\u110B\u116D\u11B0" },
            new string [] { "\uC69E", "\uC69E", "\u110B\u116D\u11B1", "\uC69E", "\u110B\u116D\u11B1" },
            new string [] { "\uC69F", "\uC69F", "\u110B\u116D\u11B2", "\uC69F", "\u110B\u116D\u11B2" },
            new string [] { "\uC6A0", "\uC6A0", "\u110B\u116D\u11B3", "\uC6A0", "\u110B\u116D\u11B3" },
            new string [] { "\uC6A1", "\uC6A1", "\u110B\u116D\u11B4", "\uC6A1", "\u110B\u116D\u11B4" },
            new string [] { "\uC6A2", "\uC6A2", "\u110B\u116D\u11B5", "\uC6A2", "\u110B\u116D\u11B5" },
            new string [] { "\uC6A3", "\uC6A3", "\u110B\u116D\u11B6", "\uC6A3", "\u110B\u116D\u11B6" },
            new string [] { "\uC6A4", "\uC6A4", "\u110B\u116D\u11B7", "\uC6A4", "\u110B\u116D\u11B7" },
            new string [] { "\uC6A5", "\uC6A5", "\u110B\u116D\u11B8", "\uC6A5", "\u110B\u116D\u11B8" },
            new string [] { "\uC6A6", "\uC6A6", "\u110B\u116D\u11B9", "\uC6A6", "\u110B\u116D\u11B9" },
            new string [] { "\uC6A7", "\uC6A7", "\u110B\u116D\u11BA", "\uC6A7", "\u110B\u116D\u11BA" },
            new string [] { "\uC6A8", "\uC6A8", "\u110B\u116D\u11BB", "\uC6A8", "\u110B\u116D\u11BB" },
            new string [] { "\uC6A9", "\uC6A9", "\u110B\u116D\u11BC", "\uC6A9", "\u110B\u116D\u11BC" },
            new string [] { "\uC6AA", "\uC6AA", "\u110B\u116D\u11BD", "\uC6AA", "\u110B\u116D\u11BD" },
            new string [] { "\uC6AB", "\uC6AB", "\u110B\u116D\u11BE", "\uC6AB", "\u110B\u116D\u11BE" },
            new string [] { "\uC6AC", "\uC6AC", "\u110B\u116D\u11BF", "\uC6AC", "\u110B\u116D\u11BF" },
            new string [] { "\uC6AD", "\uC6AD", "\u110B\u116D\u11C0", "\uC6AD", "\u110B\u116D\u11C0" },
            new string [] { "\uC6AE", "\uC6AE", "\u110B\u116D\u11C1", "\uC6AE", "\u110B\u116D\u11C1" },
            new string [] { "\uC6AF", "\uC6AF", "\u110B\u116D\u11C2", "\uC6AF", "\u110B\u116D\u11C2" },
            new string [] { "\uC6B0", "\uC6B0", "\u110B\u116E", "\uC6B0", "\u110B\u116E" },
            new string [] { "\uC6B1", "\uC6B1", "\u110B\u116E\u11A8", "\uC6B1", "\u110B\u116E\u11A8" },
            new string [] { "\uC6B2", "\uC6B2", "\u110B\u116E\u11A9", "\uC6B2", "\u110B\u116E\u11A9" },
            new string [] { "\uC6B3", "\uC6B3", "\u110B\u116E\u11AA", "\uC6B3", "\u110B\u116E\u11AA" },
            new string [] { "\uC6B4", "\uC6B4", "\u110B\u116E\u11AB", "\uC6B4", "\u110B\u116E\u11AB" },
            new string [] { "\uC6B5", "\uC6B5", "\u110B\u116E\u11AC", "\uC6B5", "\u110B\u116E\u11AC" },
            new string [] { "\uC6B6", "\uC6B6", "\u110B\u116E\u11AD", "\uC6B6", "\u110B\u116E\u11AD" },
            new string [] { "\uC6B7", "\uC6B7", "\u110B\u116E\u11AE", "\uC6B7", "\u110B\u116E\u11AE" },
            new string [] { "\uC6B8", "\uC6B8", "\u110B\u116E\u11AF", "\uC6B8", "\u110B\u116E\u11AF" },
            new string [] { "\uC6B9", "\uC6B9", "\u110B\u116E\u11B0", "\uC6B9", "\u110B\u116E\u11B0" },
            new string [] { "\uC6BA", "\uC6BA", "\u110B\u116E\u11B1", "\uC6BA", "\u110B\u116E\u11B1" },
            new string [] { "\uC6BB", "\uC6BB", "\u110B\u116E\u11B2", "\uC6BB", "\u110B\u116E\u11B2" },
            new string [] { "\uC6BC", "\uC6BC", "\u110B\u116E\u11B3", "\uC6BC", "\u110B\u116E\u11B3" },
            new string [] { "\uC6BD", "\uC6BD", "\u110B\u116E\u11B4", "\uC6BD", "\u110B\u116E\u11B4" },
            new string [] { "\uC6BE", "\uC6BE", "\u110B\u116E\u11B5", "\uC6BE", "\u110B\u116E\u11B5" },
            new string [] { "\uC6BF", "\uC6BF", "\u110B\u116E\u11B6", "\uC6BF", "\u110B\u116E\u11B6" },
            new string [] { "\uC6C0", "\uC6C0", "\u110B\u116E\u11B7", "\uC6C0", "\u110B\u116E\u11B7" },
            new string [] { "\uC6C1", "\uC6C1", "\u110B\u116E\u11B8", "\uC6C1", "\u110B\u116E\u11B8" },
            new string [] { "\uC6C2", "\uC6C2", "\u110B\u116E\u11B9", "\uC6C2", "\u110B\u116E\u11B9" },
            new string [] { "\uC6C3", "\uC6C3", "\u110B\u116E\u11BA", "\uC6C3", "\u110B\u116E\u11BA" },
            new string [] { "\uC6C4", "\uC6C4", "\u110B\u116E\u11BB", "\uC6C4", "\u110B\u116E\u11BB" },
            new string [] { "\uC6C5", "\uC6C5", "\u110B\u116E\u11BC", "\uC6C5", "\u110B\u116E\u11BC" },
            new string [] { "\uC6C6", "\uC6C6", "\u110B\u116E\u11BD", "\uC6C6", "\u110B\u116E\u11BD" },
            new string [] { "\uC6C7", "\uC6C7", "\u110B\u116E\u11BE", "\uC6C7", "\u110B\u116E\u11BE" },
            new string [] { "\uC6C8", "\uC6C8", "\u110B\u116E\u11BF", "\uC6C8", "\u110B\u116E\u11BF" },
            new string [] { "\uC6C9", "\uC6C9", "\u110B\u116E\u11C0", "\uC6C9", "\u110B\u116E\u11C0" },
            new string [] { "\uC6CA", "\uC6CA", "\u110B\u116E\u11C1", "\uC6CA", "\u110B\u116E\u11C1" },
            new string [] { "\uC6CB", "\uC6CB", "\u110B\u116E\u11C2", "\uC6CB", "\u110B\u116E\u11C2" },
            new string [] { "\uC6CC", "\uC6CC", "\u110B\u116F", "\uC6CC", "\u110B\u116F" },
            new string [] { "\uC6CD", "\uC6CD", "\u110B\u116F\u11A8", "\uC6CD", "\u110B\u116F\u11A8" },
            new string [] { "\uC6CE", "\uC6CE", "\u110B\u116F\u11A9", "\uC6CE", "\u110B\u116F\u11A9" },
            new string [] { "\uC6CF", "\uC6CF", "\u110B\u116F\u11AA", "\uC6CF", "\u110B\u116F\u11AA" },
            new string [] { "\uC6D0", "\uC6D0", "\u110B\u116F\u11AB", "\uC6D0", "\u110B\u116F\u11AB" },
            new string [] { "\uC6D1", "\uC6D1", "\u110B\u116F\u11AC", "\uC6D1", "\u110B\u116F\u11AC" },
            new string [] { "\uC6D2", "\uC6D2", "\u110B\u116F\u11AD", "\uC6D2", "\u110B\u116F\u11AD" },
            new string [] { "\uC6D3", "\uC6D3", "\u110B\u116F\u11AE", "\uC6D3", "\u110B\u116F\u11AE" },
            new string [] { "\uC6D4", "\uC6D4", "\u110B\u116F\u11AF", "\uC6D4", "\u110B\u116F\u11AF" },
            new string [] { "\uC6D5", "\uC6D5", "\u110B\u116F\u11B0", "\uC6D5", "\u110B\u116F\u11B0" },
            new string [] { "\uC6D6", "\uC6D6", "\u110B\u116F\u11B1", "\uC6D6", "\u110B\u116F\u11B1" },
            new string [] { "\uC6D7", "\uC6D7", "\u110B\u116F\u11B2", "\uC6D7", "\u110B\u116F\u11B2" },
            new string [] { "\uC6D8", "\uC6D8", "\u110B\u116F\u11B3", "\uC6D8", "\u110B\u116F\u11B3" },
            new string [] { "\uC6D9", "\uC6D9", "\u110B\u116F\u11B4", "\uC6D9", "\u110B\u116F\u11B4" },
            new string [] { "\uC6DA", "\uC6DA", "\u110B\u116F\u11B5", "\uC6DA", "\u110B\u116F\u11B5" },
            new string [] { "\uC6DB", "\uC6DB", "\u110B\u116F\u11B6", "\uC6DB", "\u110B\u116F\u11B6" },
            new string [] { "\uC6DC", "\uC6DC", "\u110B\u116F\u11B7", "\uC6DC", "\u110B\u116F\u11B7" },
            new string [] { "\uC6DD", "\uC6DD", "\u110B\u116F\u11B8", "\uC6DD", "\u110B\u116F\u11B8" },
            new string [] { "\uC6DE", "\uC6DE", "\u110B\u116F\u11B9", "\uC6DE", "\u110B\u116F\u11B9" },
            new string [] { "\uC6DF", "\uC6DF", "\u110B\u116F\u11BA", "\uC6DF", "\u110B\u116F\u11BA" },
            new string [] { "\uC6E0", "\uC6E0", "\u110B\u116F\u11BB", "\uC6E0", "\u110B\u116F\u11BB" },
            new string [] { "\uC6E1", "\uC6E1", "\u110B\u116F\u11BC", "\uC6E1", "\u110B\u116F\u11BC" },
            new string [] { "\uC6E2", "\uC6E2", "\u110B\u116F\u11BD", "\uC6E2", "\u110B\u116F\u11BD" },
            new string [] { "\uC6E3", "\uC6E3", "\u110B\u116F\u11BE", "\uC6E3", "\u110B\u116F\u11BE" },
            new string [] { "\uC6E4", "\uC6E4", "\u110B\u116F\u11BF", "\uC6E4", "\u110B\u116F\u11BF" },
            new string [] { "\uC6E5", "\uC6E5", "\u110B\u116F\u11C0", "\uC6E5", "\u110B\u116F\u11C0" },
            new string [] { "\uC6E6", "\uC6E6", "\u110B\u116F\u11C1", "\uC6E6", "\u110B\u116F\u11C1" },
            new string [] { "\uC6E7", "\uC6E7", "\u110B\u116F\u11C2", "\uC6E7", "\u110B\u116F\u11C2" },
            new string [] { "\uC6E8", "\uC6E8", "\u110B\u1170", "\uC6E8", "\u110B\u1170" },
            new string [] { "\uC6E9", "\uC6E9", "\u110B\u1170\u11A8", "\uC6E9", "\u110B\u1170\u11A8" },
            new string [] { "\uC6EA", "\uC6EA", "\u110B\u1170\u11A9", "\uC6EA", "\u110B\u1170\u11A9" },
            new string [] { "\uC6EB", "\uC6EB", "\u110B\u1170\u11AA", "\uC6EB", "\u110B\u1170\u11AA" },
            new string [] { "\uC6EC", "\uC6EC", "\u110B\u1170\u11AB", "\uC6EC", "\u110B\u1170\u11AB" },
            new string [] { "\uC6ED", "\uC6ED", "\u110B\u1170\u11AC", "\uC6ED", "\u110B\u1170\u11AC" },
            new string [] { "\uC6EE", "\uC6EE", "\u110B\u1170\u11AD", "\uC6EE", "\u110B\u1170\u11AD" },
            new string [] { "\uC6EF", "\uC6EF", "\u110B\u1170\u11AE", "\uC6EF", "\u110B\u1170\u11AE" },
            new string [] { "\uC6F0", "\uC6F0", "\u110B\u1170\u11AF", "\uC6F0", "\u110B\u1170\u11AF" },
            new string [] { "\uC6F1", "\uC6F1", "\u110B\u1170\u11B0", "\uC6F1", "\u110B\u1170\u11B0" },
            new string [] { "\uC6F2", "\uC6F2", "\u110B\u1170\u11B1", "\uC6F2", "\u110B\u1170\u11B1" },
            new string [] { "\uC6F3", "\uC6F3", "\u110B\u1170\u11B2", "\uC6F3", "\u110B\u1170\u11B2" },
            new string [] { "\uC6F4", "\uC6F4", "\u110B\u1170\u11B3", "\uC6F4", "\u110B\u1170\u11B3" },
            new string [] { "\uC6F5", "\uC6F5", "\u110B\u1170\u11B4", "\uC6F5", "\u110B\u1170\u11B4" },
            new string [] { "\uC6F6", "\uC6F6", "\u110B\u1170\u11B5", "\uC6F6", "\u110B\u1170\u11B5" },
            new string [] { "\uC6F7", "\uC6F7", "\u110B\u1170\u11B6", "\uC6F7", "\u110B\u1170\u11B6" },
            new string [] { "\uC6F8", "\uC6F8", "\u110B\u1170\u11B7", "\uC6F8", "\u110B\u1170\u11B7" },
            new string [] { "\uC6F9", "\uC6F9", "\u110B\u1170\u11B8", "\uC6F9", "\u110B\u1170\u11B8" },
            new string [] { "\uC6FA", "\uC6FA", "\u110B\u1170\u11B9", "\uC6FA", "\u110B\u1170\u11B9" },
            new string [] { "\uC6FB", "\uC6FB", "\u110B\u1170\u11BA", "\uC6FB", "\u110B\u1170\u11BA" },
            new string [] { "\uC6FC", "\uC6FC", "\u110B\u1170\u11BB", "\uC6FC", "\u110B\u1170\u11BB" },
            new string [] { "\uC6FD", "\uC6FD", "\u110B\u1170\u11BC", "\uC6FD", "\u110B\u1170\u11BC" },
            new string [] { "\uC6FE", "\uC6FE", "\u110B\u1170\u11BD", "\uC6FE", "\u110B\u1170\u11BD" },
            new string [] { "\uC6FF", "\uC6FF", "\u110B\u1170\u11BE", "\uC6FF", "\u110B\u1170\u11BE" },
            new string [] { "\uC700", "\uC700", "\u110B\u1170\u11BF", "\uC700", "\u110B\u1170\u11BF" },
            new string [] { "\uC701", "\uC701", "\u110B\u1170\u11C0", "\uC701", "\u110B\u1170\u11C0" },
            new string [] { "\uC702", "\uC702", "\u110B\u1170\u11C1", "\uC702", "\u110B\u1170\u11C1" },
            new string [] { "\uC703", "\uC703", "\u110B\u1170\u11C2", "\uC703", "\u110B\u1170\u11C2" },
            new string [] { "\uC704", "\uC704", "\u110B\u1171", "\uC704", "\u110B\u1171" },
            new string [] { "\uC705", "\uC705", "\u110B\u1171\u11A8", "\uC705", "\u110B\u1171\u11A8" },
            new string [] { "\uC706", "\uC706", "\u110B\u1171\u11A9", "\uC706", "\u110B\u1171\u11A9" },
            new string [] { "\uC707", "\uC707", "\u110B\u1171\u11AA", "\uC707", "\u110B\u1171\u11AA" },
            new string [] { "\uC708", "\uC708", "\u110B\u1171\u11AB", "\uC708", "\u110B\u1171\u11AB" },
            new string [] { "\uC709", "\uC709", "\u110B\u1171\u11AC", "\uC709", "\u110B\u1171\u11AC" },
            new string [] { "\uC70A", "\uC70A", "\u110B\u1171\u11AD", "\uC70A", "\u110B\u1171\u11AD" },
            new string [] { "\uC70B", "\uC70B", "\u110B\u1171\u11AE", "\uC70B", "\u110B\u1171\u11AE" },
            new string [] { "\uC70C", "\uC70C", "\u110B\u1171\u11AF", "\uC70C", "\u110B\u1171\u11AF" },
            new string [] { "\uC70D", "\uC70D", "\u110B\u1171\u11B0", "\uC70D", "\u110B\u1171\u11B0" },
            new string [] { "\uC70E", "\uC70E", "\u110B\u1171\u11B1", "\uC70E", "\u110B\u1171\u11B1" },
            new string [] { "\uC70F", "\uC70F", "\u110B\u1171\u11B2", "\uC70F", "\u110B\u1171\u11B2" },
            new string [] { "\uC710", "\uC710", "\u110B\u1171\u11B3", "\uC710", "\u110B\u1171\u11B3" },
            new string [] { "\uC711", "\uC711", "\u110B\u1171\u11B4", "\uC711", "\u110B\u1171\u11B4" },
            new string [] { "\uC712", "\uC712", "\u110B\u1171\u11B5", "\uC712", "\u110B\u1171\u11B5" },
            new string [] { "\uC713", "\uC713", "\u110B\u1171\u11B6", "\uC713", "\u110B\u1171\u11B6" },
            new string [] { "\uC714", "\uC714", "\u110B\u1171\u11B7", "\uC714", "\u110B\u1171\u11B7" },
            new string [] { "\uC715", "\uC715", "\u110B\u1171\u11B8", "\uC715", "\u110B\u1171\u11B8" },
            new string [] { "\uC716", "\uC716", "\u110B\u1171\u11B9", "\uC716", "\u110B\u1171\u11B9" },
            new string [] { "\uC717", "\uC717", "\u110B\u1171\u11BA", "\uC717", "\u110B\u1171\u11BA" },
            new string [] { "\uC718", "\uC718", "\u110B\u1171\u11BB", "\uC718", "\u110B\u1171\u11BB" },
            new string [] { "\uC719", "\uC719", "\u110B\u1171\u11BC", "\uC719", "\u110B\u1171\u11BC" },
            new string [] { "\uC71A", "\uC71A", "\u110B\u1171\u11BD", "\uC71A", "\u110B\u1171\u11BD" },
            new string [] { "\uC71B", "\uC71B", "\u110B\u1171\u11BE", "\uC71B", "\u110B\u1171\u11BE" },
            new string [] { "\uC71C", "\uC71C", "\u110B\u1171\u11BF", "\uC71C", "\u110B\u1171\u11BF" },
            new string [] { "\uC71D", "\uC71D", "\u110B\u1171\u11C0", "\uC71D", "\u110B\u1171\u11C0" },
            new string [] { "\uC71E", "\uC71E", "\u110B\u1171\u11C1", "\uC71E", "\u110B\u1171\u11C1" },
            new string [] { "\uC71F", "\uC71F", "\u110B\u1171\u11C2", "\uC71F", "\u110B\u1171\u11C2" },
            new string [] { "\uC720", "\uC720", "\u110B\u1172", "\uC720", "\u110B\u1172" },
            new string [] { "\uC721", "\uC721", "\u110B\u1172\u11A8", "\uC721", "\u110B\u1172\u11A8" },
            new string [] { "\uC722", "\uC722", "\u110B\u1172\u11A9", "\uC722", "\u110B\u1172\u11A9" },
            new string [] { "\uC723", "\uC723", "\u110B\u1172\u11AA", "\uC723", "\u110B\u1172\u11AA" },
            new string [] { "\uC724", "\uC724", "\u110B\u1172\u11AB", "\uC724", "\u110B\u1172\u11AB" },
            new string [] { "\uC725", "\uC725", "\u110B\u1172\u11AC", "\uC725", "\u110B\u1172\u11AC" },
            new string [] { "\uC726", "\uC726", "\u110B\u1172\u11AD", "\uC726", "\u110B\u1172\u11AD" },
            new string [] { "\uC727", "\uC727", "\u110B\u1172\u11AE", "\uC727", "\u110B\u1172\u11AE" },
            new string [] { "\uC728", "\uC728", "\u110B\u1172\u11AF", "\uC728", "\u110B\u1172\u11AF" },
            new string [] { "\uC729", "\uC729", "\u110B\u1172\u11B0", "\uC729", "\u110B\u1172\u11B0" },
            new string [] { "\uC72A", "\uC72A", "\u110B\u1172\u11B1", "\uC72A", "\u110B\u1172\u11B1" },
            new string [] { "\uC72B", "\uC72B", "\u110B\u1172\u11B2", "\uC72B", "\u110B\u1172\u11B2" },
            new string [] { "\uC72C", "\uC72C", "\u110B\u1172\u11B3", "\uC72C", "\u110B\u1172\u11B3" },
            new string [] { "\uC72D", "\uC72D", "\u110B\u1172\u11B4", "\uC72D", "\u110B\u1172\u11B4" },
            new string [] { "\uC72E", "\uC72E", "\u110B\u1172\u11B5", "\uC72E", "\u110B\u1172\u11B5" },
            new string [] { "\uC72F", "\uC72F", "\u110B\u1172\u11B6", "\uC72F", "\u110B\u1172\u11B6" },
            new string [] { "\uC730", "\uC730", "\u110B\u1172\u11B7", "\uC730", "\u110B\u1172\u11B7" },
            new string [] { "\uC731", "\uC731", "\u110B\u1172\u11B8", "\uC731", "\u110B\u1172\u11B8" },
            new string [] { "\uC732", "\uC732", "\u110B\u1172\u11B9", "\uC732", "\u110B\u1172\u11B9" },
            new string [] { "\uC733", "\uC733", "\u110B\u1172\u11BA", "\uC733", "\u110B\u1172\u11BA" },
            new string [] { "\uC734", "\uC734", "\u110B\u1172\u11BB", "\uC734", "\u110B\u1172\u11BB" },
            new string [] { "\uC735", "\uC735", "\u110B\u1172\u11BC", "\uC735", "\u110B\u1172\u11BC" },
            new string [] { "\uC736", "\uC736", "\u110B\u1172\u11BD", "\uC736", "\u110B\u1172\u11BD" },
            new string [] { "\uC737", "\uC737", "\u110B\u1172\u11BE", "\uC737", "\u110B\u1172\u11BE" },
            new string [] { "\uC738", "\uC738", "\u110B\u1172\u11BF", "\uC738", "\u110B\u1172\u11BF" },
            new string [] { "\uC739", "\uC739", "\u110B\u1172\u11C0", "\uC739", "\u110B\u1172\u11C0" },
            new string [] { "\uC73A", "\uC73A", "\u110B\u1172\u11C1", "\uC73A", "\u110B\u1172\u11C1" },
            new string [] { "\uC73B", "\uC73B", "\u110B\u1172\u11C2", "\uC73B", "\u110B\u1172\u11C2" },
            new string [] { "\uC73C", "\uC73C", "\u110B\u1173", "\uC73C", "\u110B\u1173" },
            new string [] { "\uC73D", "\uC73D", "\u110B\u1173\u11A8", "\uC73D", "\u110B\u1173\u11A8" },
            new string [] { "\uC73E", "\uC73E", "\u110B\u1173\u11A9", "\uC73E", "\u110B\u1173\u11A9" },
            new string [] { "\uC73F", "\uC73F", "\u110B\u1173\u11AA", "\uC73F", "\u110B\u1173\u11AA" },
            new string [] { "\uC740", "\uC740", "\u110B\u1173\u11AB", "\uC740", "\u110B\u1173\u11AB" },
            new string [] { "\uC741", "\uC741", "\u110B\u1173\u11AC", "\uC741", "\u110B\u1173\u11AC" },
            new string [] { "\uC742", "\uC742", "\u110B\u1173\u11AD", "\uC742", "\u110B\u1173\u11AD" },
            new string [] { "\uC743", "\uC743", "\u110B\u1173\u11AE", "\uC743", "\u110B\u1173\u11AE" },
            new string [] { "\uC744", "\uC744", "\u110B\u1173\u11AF", "\uC744", "\u110B\u1173\u11AF" },
            new string [] { "\uC745", "\uC745", "\u110B\u1173\u11B0", "\uC745", "\u110B\u1173\u11B0" },
            new string [] { "\uC746", "\uC746", "\u110B\u1173\u11B1", "\uC746", "\u110B\u1173\u11B1" },
            new string [] { "\uC747", "\uC747", "\u110B\u1173\u11B2", "\uC747", "\u110B\u1173\u11B2" },
            new string [] { "\uC748", "\uC748", "\u110B\u1173\u11B3", "\uC748", "\u110B\u1173\u11B3" },
            new string [] { "\uC749", "\uC749", "\u110B\u1173\u11B4", "\uC749", "\u110B\u1173\u11B4" },
            new string [] { "\uC74A", "\uC74A", "\u110B\u1173\u11B5", "\uC74A", "\u110B\u1173\u11B5" },
            new string [] { "\uC74B", "\uC74B", "\u110B\u1173\u11B6", "\uC74B", "\u110B\u1173\u11B6" },
            new string [] { "\uC74C", "\uC74C", "\u110B\u1173\u11B7", "\uC74C", "\u110B\u1173\u11B7" },
            new string [] { "\uC74D", "\uC74D", "\u110B\u1173\u11B8", "\uC74D", "\u110B\u1173\u11B8" },
            new string [] { "\uC74E", "\uC74E", "\u110B\u1173\u11B9", "\uC74E", "\u110B\u1173\u11B9" },
            new string [] { "\uC74F", "\uC74F", "\u110B\u1173\u11BA", "\uC74F", "\u110B\u1173\u11BA" },
            new string [] { "\uC750", "\uC750", "\u110B\u1173\u11BB", "\uC750", "\u110B\u1173\u11BB" },
            new string [] { "\uC751", "\uC751", "\u110B\u1173\u11BC", "\uC751", "\u110B\u1173\u11BC" },
            new string [] { "\uC752", "\uC752", "\u110B\u1173\u11BD", "\uC752", "\u110B\u1173\u11BD" },
            new string [] { "\uC753", "\uC753", "\u110B\u1173\u11BE", "\uC753", "\u110B\u1173\u11BE" },
            new string [] { "\uC754", "\uC754", "\u110B\u1173\u11BF", "\uC754", "\u110B\u1173\u11BF" },
            new string [] { "\uC755", "\uC755", "\u110B\u1173\u11C0", "\uC755", "\u110B\u1173\u11C0" },
            new string [] { "\uC756", "\uC756", "\u110B\u1173\u11C1", "\uC756", "\u110B\u1173\u11C1" },
            new string [] { "\uC757", "\uC757", "\u110B\u1173\u11C2", "\uC757", "\u110B\u1173\u11C2" },
            new string [] { "\uC758", "\uC758", "\u110B\u1174", "\uC758", "\u110B\u1174" },
            new string [] { "\uC759", "\uC759", "\u110B\u1174\u11A8", "\uC759", "\u110B\u1174\u11A8" },
            new string [] { "\uC75A", "\uC75A", "\u110B\u1174\u11A9", "\uC75A", "\u110B\u1174\u11A9" },
            new string [] { "\uC75B", "\uC75B", "\u110B\u1174\u11AA", "\uC75B", "\u110B\u1174\u11AA" },
            new string [] { "\uC75C", "\uC75C", "\u110B\u1174\u11AB", "\uC75C", "\u110B\u1174\u11AB" },
            new string [] { "\uC75D", "\uC75D", "\u110B\u1174\u11AC", "\uC75D", "\u110B\u1174\u11AC" },
            new string [] { "\uC75E", "\uC75E", "\u110B\u1174\u11AD", "\uC75E", "\u110B\u1174\u11AD" },
            new string [] { "\uC75F", "\uC75F", "\u110B\u1174\u11AE", "\uC75F", "\u110B\u1174\u11AE" },
            new string [] { "\uC760", "\uC760", "\u110B\u1174\u11AF", "\uC760", "\u110B\u1174\u11AF" },
            new string [] { "\uC761", "\uC761", "\u110B\u1174\u11B0", "\uC761", "\u110B\u1174\u11B0" },
            new string [] { "\uC762", "\uC762", "\u110B\u1174\u11B1", "\uC762", "\u110B\u1174\u11B1" },
            new string [] { "\uC763", "\uC763", "\u110B\u1174\u11B2", "\uC763", "\u110B\u1174\u11B2" },
            new string [] { "\uC764", "\uC764", "\u110B\u1174\u11B3", "\uC764", "\u110B\u1174\u11B3" },
            new string [] { "\uC765", "\uC765", "\u110B\u1174\u11B4", "\uC765", "\u110B\u1174\u11B4" },
            new string [] { "\uC766", "\uC766", "\u110B\u1174\u11B5", "\uC766", "\u110B\u1174\u11B5" },
            new string [] { "\uC767", "\uC767", "\u110B\u1174\u11B6", "\uC767", "\u110B\u1174\u11B6" },
            new string [] { "\uC768", "\uC768", "\u110B\u1174\u11B7", "\uC768", "\u110B\u1174\u11B7" },
            new string [] { "\uC769", "\uC769", "\u110B\u1174\u11B8", "\uC769", "\u110B\u1174\u11B8" },
            new string [] { "\uC76A", "\uC76A", "\u110B\u1174\u11B9", "\uC76A", "\u110B\u1174\u11B9" },
            new string [] { "\uC76B", "\uC76B", "\u110B\u1174\u11BA", "\uC76B", "\u110B\u1174\u11BA" },
            new string [] { "\uC76C", "\uC76C", "\u110B\u1174\u11BB", "\uC76C", "\u110B\u1174\u11BB" },
            new string [] { "\uC76D", "\uC76D", "\u110B\u1174\u11BC", "\uC76D", "\u110B\u1174\u11BC" },
            new string [] { "\uC76E", "\uC76E", "\u110B\u1174\u11BD", "\uC76E", "\u110B\u1174\u11BD" },
            new string [] { "\uC76F", "\uC76F", "\u110B\u1174\u11BE", "\uC76F", "\u110B\u1174\u11BE" },
            new string [] { "\uC770", "\uC770", "\u110B\u1174\u11BF", "\uC770", "\u110B\u1174\u11BF" },
            new string [] { "\uC771", "\uC771", "\u110B\u1174\u11C0", "\uC771", "\u110B\u1174\u11C0" },
            new string [] { "\uC772", "\uC772", "\u110B\u1174\u11C1", "\uC772", "\u110B\u1174\u11C1" },
            new string [] { "\uC773", "\uC773", "\u110B\u1174\u11C2", "\uC773", "\u110B\u1174\u11C2" },
            new string [] { "\uC774", "\uC774", "\u110B\u1175", "\uC774", "\u110B\u1175" },
            new string [] { "\uC775", "\uC775", "\u110B\u1175\u11A8", "\uC775", "\u110B\u1175\u11A8" },
            new string [] { "\uC776", "\uC776", "\u110B\u1175\u11A9", "\uC776", "\u110B\u1175\u11A9" },
            new string [] { "\uC777", "\uC777", "\u110B\u1175\u11AA", "\uC777", "\u110B\u1175\u11AA" },
            new string [] { "\uC778", "\uC778", "\u110B\u1175\u11AB", "\uC778", "\u110B\u1175\u11AB" },
            new string [] { "\uC779", "\uC779", "\u110B\u1175\u11AC", "\uC779", "\u110B\u1175\u11AC" },
            new string [] { "\uC77A", "\uC77A", "\u110B\u1175\u11AD", "\uC77A", "\u110B\u1175\u11AD" },
            new string [] { "\uC77B", "\uC77B", "\u110B\u1175\u11AE", "\uC77B", "\u110B\u1175\u11AE" },
            new string [] { "\uC77C", "\uC77C", "\u110B\u1175\u11AF", "\uC77C", "\u110B\u1175\u11AF" },
            new string [] { "\uC77D", "\uC77D", "\u110B\u1175\u11B0", "\uC77D", "\u110B\u1175\u11B0" },
            new string [] { "\uC77E", "\uC77E", "\u110B\u1175\u11B1", "\uC77E", "\u110B\u1175\u11B1" },
            new string [] { "\uC77F", "\uC77F", "\u110B\u1175\u11B2", "\uC77F", "\u110B\u1175\u11B2" },
            new string [] { "\uC780", "\uC780", "\u110B\u1175\u11B3", "\uC780", "\u110B\u1175\u11B3" },
            new string [] { "\uC781", "\uC781", "\u110B\u1175\u11B4", "\uC781", "\u110B\u1175\u11B4" },
            new string [] { "\uC782", "\uC782", "\u110B\u1175\u11B5", "\uC782", "\u110B\u1175\u11B5" },
            new string [] { "\uC783", "\uC783", "\u110B\u1175\u11B6", "\uC783", "\u110B\u1175\u11B6" },
            new string [] { "\uC784", "\uC784", "\u110B\u1175\u11B7", "\uC784", "\u110B\u1175\u11B7" },
            new string [] { "\uC785", "\uC785", "\u110B\u1175\u11B8", "\uC785", "\u110B\u1175\u11B8" },
            new string [] { "\uC786", "\uC786", "\u110B\u1175\u11B9", "\uC786", "\u110B\u1175\u11B9" },
            new string [] { "\uC787", "\uC787", "\u110B\u1175\u11BA", "\uC787", "\u110B\u1175\u11BA" },
            new string [] { "\uC788", "\uC788", "\u110B\u1175\u11BB", "\uC788", "\u110B\u1175\u11BB" },
            new string [] { "\uC789", "\uC789", "\u110B\u1175\u11BC", "\uC789", "\u110B\u1175\u11BC" },
            new string [] { "\uC78A", "\uC78A", "\u110B\u1175\u11BD", "\uC78A", "\u110B\u1175\u11BD" },
            new string [] { "\uC78B", "\uC78B", "\u110B\u1175\u11BE", "\uC78B", "\u110B\u1175\u11BE" },
            new string [] { "\uC78C", "\uC78C", "\u110B\u1175\u11BF", "\uC78C", "\u110B\u1175\u11BF" },
            new string [] { "\uC78D", "\uC78D", "\u110B\u1175\u11C0", "\uC78D", "\u110B\u1175\u11C0" },
            new string [] { "\uC78E", "\uC78E", "\u110B\u1175\u11C1", "\uC78E", "\u110B\u1175\u11C1" },
            new string [] { "\uC78F", "\uC78F", "\u110B\u1175\u11C2", "\uC78F", "\u110B\u1175\u11C2" },
            new string [] { "\uC790", "\uC790", "\u110C\u1161", "\uC790", "\u110C\u1161" },
            new string [] { "\uC791", "\uC791", "\u110C\u1161\u11A8", "\uC791", "\u110C\u1161\u11A8" },
            new string [] { "\uC792", "\uC792", "\u110C\u1161\u11A9", "\uC792", "\u110C\u1161\u11A9" },
            new string [] { "\uC793", "\uC793", "\u110C\u1161\u11AA", "\uC793", "\u110C\u1161\u11AA" },
            new string [] { "\uC794", "\uC794", "\u110C\u1161\u11AB", "\uC794", "\u110C\u1161\u11AB" },
            new string [] { "\uC795", "\uC795", "\u110C\u1161\u11AC", "\uC795", "\u110C\u1161\u11AC" },
            new string [] { "\uC796", "\uC796", "\u110C\u1161\u11AD", "\uC796", "\u110C\u1161\u11AD" },
            new string [] { "\uC797", "\uC797", "\u110C\u1161\u11AE", "\uC797", "\u110C\u1161\u11AE" },
            new string [] { "\uC798", "\uC798", "\u110C\u1161\u11AF", "\uC798", "\u110C\u1161\u11AF" },
            new string [] { "\uC799", "\uC799", "\u110C\u1161\u11B0", "\uC799", "\u110C\u1161\u11B0" },
            new string [] { "\uC79A", "\uC79A", "\u110C\u1161\u11B1", "\uC79A", "\u110C\u1161\u11B1" },
            new string [] { "\uC79B", "\uC79B", "\u110C\u1161\u11B2", "\uC79B", "\u110C\u1161\u11B2" },
            new string [] { "\uC79C", "\uC79C", "\u110C\u1161\u11B3", "\uC79C", "\u110C\u1161\u11B3" },
            new string [] { "\uC79D", "\uC79D", "\u110C\u1161\u11B4", "\uC79D", "\u110C\u1161\u11B4" },
            new string [] { "\uC79E", "\uC79E", "\u110C\u1161\u11B5", "\uC79E", "\u110C\u1161\u11B5" },
            new string [] { "\uC79F", "\uC79F", "\u110C\u1161\u11B6", "\uC79F", "\u110C\u1161\u11B6" },
            new string [] { "\uC7A0", "\uC7A0", "\u110C\u1161\u11B7", "\uC7A0", "\u110C\u1161\u11B7" },
            new string [] { "\uC7A1", "\uC7A1", "\u110C\u1161\u11B8", "\uC7A1", "\u110C\u1161\u11B8" },
            new string [] { "\uC7A2", "\uC7A2", "\u110C\u1161\u11B9", "\uC7A2", "\u110C\u1161\u11B9" },
            new string [] { "\uC7A3", "\uC7A3", "\u110C\u1161\u11BA", "\uC7A3", "\u110C\u1161\u11BA" },
            new string [] { "\uC7A4", "\uC7A4", "\u110C\u1161\u11BB", "\uC7A4", "\u110C\u1161\u11BB" },
            new string [] { "\uC7A5", "\uC7A5", "\u110C\u1161\u11BC", "\uC7A5", "\u110C\u1161\u11BC" },
            new string [] { "\uC7A6", "\uC7A6", "\u110C\u1161\u11BD", "\uC7A6", "\u110C\u1161\u11BD" },
            new string [] { "\uC7A7", "\uC7A7", "\u110C\u1161\u11BE", "\uC7A7", "\u110C\u1161\u11BE" },
            new string [] { "\uC7A8", "\uC7A8", "\u110C\u1161\u11BF", "\uC7A8", "\u110C\u1161\u11BF" },
            new string [] { "\uC7A9", "\uC7A9", "\u110C\u1161\u11C0", "\uC7A9", "\u110C\u1161\u11C0" },
            new string [] { "\uC7AA", "\uC7AA", "\u110C\u1161\u11C1", "\uC7AA", "\u110C\u1161\u11C1" },
            new string [] { "\uC7AB", "\uC7AB", "\u110C\u1161\u11C2", "\uC7AB", "\u110C\u1161\u11C2" },
            new string [] { "\uC7AC", "\uC7AC", "\u110C\u1162", "\uC7AC", "\u110C\u1162" },
            new string [] { "\uC7AD", "\uC7AD", "\u110C\u1162\u11A8", "\uC7AD", "\u110C\u1162\u11A8" },
            new string [] { "\uC7AE", "\uC7AE", "\u110C\u1162\u11A9", "\uC7AE", "\u110C\u1162\u11A9" },
            new string [] { "\uC7AF", "\uC7AF", "\u110C\u1162\u11AA", "\uC7AF", "\u110C\u1162\u11AA" },
            new string [] { "\uC7B0", "\uC7B0", "\u110C\u1162\u11AB", "\uC7B0", "\u110C\u1162\u11AB" },
            new string [] { "\uC7B1", "\uC7B1", "\u110C\u1162\u11AC", "\uC7B1", "\u110C\u1162\u11AC" },
            new string [] { "\uC7B2", "\uC7B2", "\u110C\u1162\u11AD", "\uC7B2", "\u110C\u1162\u11AD" },
            new string [] { "\uC7B3", "\uC7B3", "\u110C\u1162\u11AE", "\uC7B3", "\u110C\u1162\u11AE" },
            new string [] { "\uC7B4", "\uC7B4", "\u110C\u1162\u11AF", "\uC7B4", "\u110C\u1162\u11AF" },
            new string [] { "\uC7B5", "\uC7B5", "\u110C\u1162\u11B0", "\uC7B5", "\u110C\u1162\u11B0" },
            new string [] { "\uC7B6", "\uC7B6", "\u110C\u1162\u11B1", "\uC7B6", "\u110C\u1162\u11B1" },
            new string [] { "\uC7B7", "\uC7B7", "\u110C\u1162\u11B2", "\uC7B7", "\u110C\u1162\u11B2" },
            new string [] { "\uC7B8", "\uC7B8", "\u110C\u1162\u11B3", "\uC7B8", "\u110C\u1162\u11B3" },
            new string [] { "\uC7B9", "\uC7B9", "\u110C\u1162\u11B4", "\uC7B9", "\u110C\u1162\u11B4" },
            new string [] { "\uC7BA", "\uC7BA", "\u110C\u1162\u11B5", "\uC7BA", "\u110C\u1162\u11B5" },
            new string [] { "\uC7BB", "\uC7BB", "\u110C\u1162\u11B6", "\uC7BB", "\u110C\u1162\u11B6" },
            new string [] { "\uC7BC", "\uC7BC", "\u110C\u1162\u11B7", "\uC7BC", "\u110C\u1162\u11B7" },
            new string [] { "\uC7BD", "\uC7BD", "\u110C\u1162\u11B8", "\uC7BD", "\u110C\u1162\u11B8" },
            new string [] { "\uC7BE", "\uC7BE", "\u110C\u1162\u11B9", "\uC7BE", "\u110C\u1162\u11B9" },
            new string [] { "\uC7BF", "\uC7BF", "\u110C\u1162\u11BA", "\uC7BF", "\u110C\u1162\u11BA" },
            new string [] { "\uC7C0", "\uC7C0", "\u110C\u1162\u11BB", "\uC7C0", "\u110C\u1162\u11BB" },
            new string [] { "\uC7C1", "\uC7C1", "\u110C\u1162\u11BC", "\uC7C1", "\u110C\u1162\u11BC" },
            new string [] { "\uC7C2", "\uC7C2", "\u110C\u1162\u11BD", "\uC7C2", "\u110C\u1162\u11BD" },
            new string [] { "\uC7C3", "\uC7C3", "\u110C\u1162\u11BE", "\uC7C3", "\u110C\u1162\u11BE" },
            new string [] { "\uC7C4", "\uC7C4", "\u110C\u1162\u11BF", "\uC7C4", "\u110C\u1162\u11BF" },
            new string [] { "\uC7C5", "\uC7C5", "\u110C\u1162\u11C0", "\uC7C5", "\u110C\u1162\u11C0" },
            new string [] { "\uC7C6", "\uC7C6", "\u110C\u1162\u11C1", "\uC7C6", "\u110C\u1162\u11C1" },
            new string [] { "\uC7C7", "\uC7C7", "\u110C\u1162\u11C2", "\uC7C7", "\u110C\u1162\u11C2" },
            new string [] { "\uC7C8", "\uC7C8", "\u110C\u1163", "\uC7C8", "\u110C\u1163" },
            new string [] { "\uC7C9", "\uC7C9", "\u110C\u1163\u11A8", "\uC7C9", "\u110C\u1163\u11A8" },
            new string [] { "\uC7CA", "\uC7CA", "\u110C\u1163\u11A9", "\uC7CA", "\u110C\u1163\u11A9" },
            new string [] { "\uC7CB", "\uC7CB", "\u110C\u1163\u11AA", "\uC7CB", "\u110C\u1163\u11AA" },
            new string [] { "\uC7CC", "\uC7CC", "\u110C\u1163\u11AB", "\uC7CC", "\u110C\u1163\u11AB" },
            new string [] { "\uC7CD", "\uC7CD", "\u110C\u1163\u11AC", "\uC7CD", "\u110C\u1163\u11AC" },
            new string [] { "\uC7CE", "\uC7CE", "\u110C\u1163\u11AD", "\uC7CE", "\u110C\u1163\u11AD" },
            new string [] { "\uC7CF", "\uC7CF", "\u110C\u1163\u11AE", "\uC7CF", "\u110C\u1163\u11AE" },
            new string [] { "\uC7D0", "\uC7D0", "\u110C\u1163\u11AF", "\uC7D0", "\u110C\u1163\u11AF" },
            new string [] { "\uC7D1", "\uC7D1", "\u110C\u1163\u11B0", "\uC7D1", "\u110C\u1163\u11B0" },
            new string [] { "\uC7D2", "\uC7D2", "\u110C\u1163\u11B1", "\uC7D2", "\u110C\u1163\u11B1" },
            new string [] { "\uC7D3", "\uC7D3", "\u110C\u1163\u11B2", "\uC7D3", "\u110C\u1163\u11B2" },
            new string [] { "\uC7D4", "\uC7D4", "\u110C\u1163\u11B3", "\uC7D4", "\u110C\u1163\u11B3" },
            new string [] { "\uC7D5", "\uC7D5", "\u110C\u1163\u11B4", "\uC7D5", "\u110C\u1163\u11B4" },
            new string [] { "\uC7D6", "\uC7D6", "\u110C\u1163\u11B5", "\uC7D6", "\u110C\u1163\u11B5" },
            new string [] { "\uC7D7", "\uC7D7", "\u110C\u1163\u11B6", "\uC7D7", "\u110C\u1163\u11B6" },
            new string [] { "\uC7D8", "\uC7D8", "\u110C\u1163\u11B7", "\uC7D8", "\u110C\u1163\u11B7" },
            new string [] { "\uC7D9", "\uC7D9", "\u110C\u1163\u11B8", "\uC7D9", "\u110C\u1163\u11B8" },
            new string [] { "\uC7DA", "\uC7DA", "\u110C\u1163\u11B9", "\uC7DA", "\u110C\u1163\u11B9" },
            new string [] { "\uC7DB", "\uC7DB", "\u110C\u1163\u11BA", "\uC7DB", "\u110C\u1163\u11BA" },
            new string [] { "\uC7DC", "\uC7DC", "\u110C\u1163\u11BB", "\uC7DC", "\u110C\u1163\u11BB" },
            new string [] { "\uC7DD", "\uC7DD", "\u110C\u1163\u11BC", "\uC7DD", "\u110C\u1163\u11BC" },
            new string [] { "\uC7DE", "\uC7DE", "\u110C\u1163\u11BD", "\uC7DE", "\u110C\u1163\u11BD" },
            new string [] { "\uC7DF", "\uC7DF", "\u110C\u1163\u11BE", "\uC7DF", "\u110C\u1163\u11BE" },
            new string [] { "\uC7E0", "\uC7E0", "\u110C\u1163\u11BF", "\uC7E0", "\u110C\u1163\u11BF" },
            new string [] { "\uC7E1", "\uC7E1", "\u110C\u1163\u11C0", "\uC7E1", "\u110C\u1163\u11C0" },
            new string [] { "\uC7E2", "\uC7E2", "\u110C\u1163\u11C1", "\uC7E2", "\u110C\u1163\u11C1" },
            new string [] { "\uC7E3", "\uC7E3", "\u110C\u1163\u11C2", "\uC7E3", "\u110C\u1163\u11C2" },
            new string [] { "\uC7E4", "\uC7E4", "\u110C\u1164", "\uC7E4", "\u110C\u1164" },
            new string [] { "\uC7E5", "\uC7E5", "\u110C\u1164\u11A8", "\uC7E5", "\u110C\u1164\u11A8" },
            new string [] { "\uC7E6", "\uC7E6", "\u110C\u1164\u11A9", "\uC7E6", "\u110C\u1164\u11A9" },
            new string [] { "\uC7E7", "\uC7E7", "\u110C\u1164\u11AA", "\uC7E7", "\u110C\u1164\u11AA" },
            new string [] { "\uC7E8", "\uC7E8", "\u110C\u1164\u11AB", "\uC7E8", "\u110C\u1164\u11AB" },
            new string [] { "\uC7E9", "\uC7E9", "\u110C\u1164\u11AC", "\uC7E9", "\u110C\u1164\u11AC" },
            new string [] { "\uC7EA", "\uC7EA", "\u110C\u1164\u11AD", "\uC7EA", "\u110C\u1164\u11AD" },
            new string [] { "\uC7EB", "\uC7EB", "\u110C\u1164\u11AE", "\uC7EB", "\u110C\u1164\u11AE" },
            new string [] { "\uC7EC", "\uC7EC", "\u110C\u1164\u11AF", "\uC7EC", "\u110C\u1164\u11AF" },
            new string [] { "\uC7ED", "\uC7ED", "\u110C\u1164\u11B0", "\uC7ED", "\u110C\u1164\u11B0" },
            new string [] { "\uC7EE", "\uC7EE", "\u110C\u1164\u11B1", "\uC7EE", "\u110C\u1164\u11B1" },
            new string [] { "\uC7EF", "\uC7EF", "\u110C\u1164\u11B2", "\uC7EF", "\u110C\u1164\u11B2" },
            new string [] { "\uC7F0", "\uC7F0", "\u110C\u1164\u11B3", "\uC7F0", "\u110C\u1164\u11B3" },
            new string [] { "\uC7F1", "\uC7F1", "\u110C\u1164\u11B4", "\uC7F1", "\u110C\u1164\u11B4" },
            new string [] { "\uC7F2", "\uC7F2", "\u110C\u1164\u11B5", "\uC7F2", "\u110C\u1164\u11B5" },
            new string [] { "\uC7F3", "\uC7F3", "\u110C\u1164\u11B6", "\uC7F3", "\u110C\u1164\u11B6" },
            new string [] { "\uC7F4", "\uC7F4", "\u110C\u1164\u11B7", "\uC7F4", "\u110C\u1164\u11B7" },
            new string [] { "\uC7F5", "\uC7F5", "\u110C\u1164\u11B8", "\uC7F5", "\u110C\u1164\u11B8" },
            new string [] { "\uC7F6", "\uC7F6", "\u110C\u1164\u11B9", "\uC7F6", "\u110C\u1164\u11B9" },
            new string [] { "\uC7F7", "\uC7F7", "\u110C\u1164\u11BA", "\uC7F7", "\u110C\u1164\u11BA" },
            new string [] { "\uC7F8", "\uC7F8", "\u110C\u1164\u11BB", "\uC7F8", "\u110C\u1164\u11BB" },
            new string [] { "\uC7F9", "\uC7F9", "\u110C\u1164\u11BC", "\uC7F9", "\u110C\u1164\u11BC" },
            new string [] { "\uC7FA", "\uC7FA", "\u110C\u1164\u11BD", "\uC7FA", "\u110C\u1164\u11BD" },
            new string [] { "\uC7FB", "\uC7FB", "\u110C\u1164\u11BE", "\uC7FB", "\u110C\u1164\u11BE" },
            new string [] { "\uC7FC", "\uC7FC", "\u110C\u1164\u11BF", "\uC7FC", "\u110C\u1164\u11BF" },
            new string [] { "\uC7FD", "\uC7FD", "\u110C\u1164\u11C0", "\uC7FD", "\u110C\u1164\u11C0" },
            new string [] { "\uC7FE", "\uC7FE", "\u110C\u1164\u11C1", "\uC7FE", "\u110C\u1164\u11C1" },
            new string [] { "\uC7FF", "\uC7FF", "\u110C\u1164\u11C2", "\uC7FF", "\u110C\u1164\u11C2" },
            new string [] { "\uC800", "\uC800", "\u110C\u1165", "\uC800", "\u110C\u1165" },
            new string [] { "\uC801", "\uC801", "\u110C\u1165\u11A8", "\uC801", "\u110C\u1165\u11A8" },
            new string [] { "\uC802", "\uC802", "\u110C\u1165\u11A9", "\uC802", "\u110C\u1165\u11A9" },
            new string [] { "\uC803", "\uC803", "\u110C\u1165\u11AA", "\uC803", "\u110C\u1165\u11AA" },
            new string [] { "\uC804", "\uC804", "\u110C\u1165\u11AB", "\uC804", "\u110C\u1165\u11AB" },
            new string [] { "\uC805", "\uC805", "\u110C\u1165\u11AC", "\uC805", "\u110C\u1165\u11AC" },
            new string [] { "\uC806", "\uC806", "\u110C\u1165\u11AD", "\uC806", "\u110C\u1165\u11AD" },
            new string [] { "\uC807", "\uC807", "\u110C\u1165\u11AE", "\uC807", "\u110C\u1165\u11AE" },
            new string [] { "\uC808", "\uC808", "\u110C\u1165\u11AF", "\uC808", "\u110C\u1165\u11AF" },
            new string [] { "\uC809", "\uC809", "\u110C\u1165\u11B0", "\uC809", "\u110C\u1165\u11B0" },
            new string [] { "\uC80A", "\uC80A", "\u110C\u1165\u11B1", "\uC80A", "\u110C\u1165\u11B1" },
            new string [] { "\uC80B", "\uC80B", "\u110C\u1165\u11B2", "\uC80B", "\u110C\u1165\u11B2" },
            new string [] { "\uC80C", "\uC80C", "\u110C\u1165\u11B3", "\uC80C", "\u110C\u1165\u11B3" },
            new string [] { "\uC80D", "\uC80D", "\u110C\u1165\u11B4", "\uC80D", "\u110C\u1165\u11B4" },
            new string [] { "\uC80E", "\uC80E", "\u110C\u1165\u11B5", "\uC80E", "\u110C\u1165\u11B5" },
            new string [] { "\uC80F", "\uC80F", "\u110C\u1165\u11B6", "\uC80F", "\u110C\u1165\u11B6" },
            new string [] { "\uC810", "\uC810", "\u110C\u1165\u11B7", "\uC810", "\u110C\u1165\u11B7" },
            new string [] { "\uC811", "\uC811", "\u110C\u1165\u11B8", "\uC811", "\u110C\u1165\u11B8" },
            new string [] { "\uC812", "\uC812", "\u110C\u1165\u11B9", "\uC812", "\u110C\u1165\u11B9" },
            new string [] { "\uC813", "\uC813", "\u110C\u1165\u11BA", "\uC813", "\u110C\u1165\u11BA" },
            new string [] { "\uC814", "\uC814", "\u110C\u1165\u11BB", "\uC814", "\u110C\u1165\u11BB" },
            new string [] { "\uC815", "\uC815", "\u110C\u1165\u11BC", "\uC815", "\u110C\u1165\u11BC" },
            new string [] { "\uC816", "\uC816", "\u110C\u1165\u11BD", "\uC816", "\u110C\u1165\u11BD" },
            new string [] { "\uC817", "\uC817", "\u110C\u1165\u11BE", "\uC817", "\u110C\u1165\u11BE" },
            new string [] { "\uC818", "\uC818", "\u110C\u1165\u11BF", "\uC818", "\u110C\u1165\u11BF" },
            new string [] { "\uC819", "\uC819", "\u110C\u1165\u11C0", "\uC819", "\u110C\u1165\u11C0" },
            new string [] { "\uC81A", "\uC81A", "\u110C\u1165\u11C1", "\uC81A", "\u110C\u1165\u11C1" },
            new string [] { "\uC81B", "\uC81B", "\u110C\u1165\u11C2", "\uC81B", "\u110C\u1165\u11C2" },
            new string [] { "\uC81C", "\uC81C", "\u110C\u1166", "\uC81C", "\u110C\u1166" },
            new string [] { "\uC81D", "\uC81D", "\u110C\u1166\u11A8", "\uC81D", "\u110C\u1166\u11A8" },
            new string [] { "\uC81E", "\uC81E", "\u110C\u1166\u11A9", "\uC81E", "\u110C\u1166\u11A9" },
            new string [] { "\uC81F", "\uC81F", "\u110C\u1166\u11AA", "\uC81F", "\u110C\u1166\u11AA" },
            new string [] { "\uC820", "\uC820", "\u110C\u1166\u11AB", "\uC820", "\u110C\u1166\u11AB" },
            new string [] { "\uC821", "\uC821", "\u110C\u1166\u11AC", "\uC821", "\u110C\u1166\u11AC" },
            new string [] { "\uC822", "\uC822", "\u110C\u1166\u11AD", "\uC822", "\u110C\u1166\u11AD" },
            new string [] { "\uC823", "\uC823", "\u110C\u1166\u11AE", "\uC823", "\u110C\u1166\u11AE" },
            new string [] { "\uC824", "\uC824", "\u110C\u1166\u11AF", "\uC824", "\u110C\u1166\u11AF" },
            new string [] { "\uC825", "\uC825", "\u110C\u1166\u11B0", "\uC825", "\u110C\u1166\u11B0" },
            new string [] { "\uC826", "\uC826", "\u110C\u1166\u11B1", "\uC826", "\u110C\u1166\u11B1" },
            new string [] { "\uC827", "\uC827", "\u110C\u1166\u11B2", "\uC827", "\u110C\u1166\u11B2" },
            new string [] { "\uC828", "\uC828", "\u110C\u1166\u11B3", "\uC828", "\u110C\u1166\u11B3" },
            new string [] { "\uC829", "\uC829", "\u110C\u1166\u11B4", "\uC829", "\u110C\u1166\u11B4" },
            new string [] { "\uC82A", "\uC82A", "\u110C\u1166\u11B5", "\uC82A", "\u110C\u1166\u11B5" },
            new string [] { "\uC82B", "\uC82B", "\u110C\u1166\u11B6", "\uC82B", "\u110C\u1166\u11B6" },
            new string [] { "\uC82C", "\uC82C", "\u110C\u1166\u11B7", "\uC82C", "\u110C\u1166\u11B7" },
            new string [] { "\uC82D", "\uC82D", "\u110C\u1166\u11B8", "\uC82D", "\u110C\u1166\u11B8" },
            new string [] { "\uC82E", "\uC82E", "\u110C\u1166\u11B9", "\uC82E", "\u110C\u1166\u11B9" },
            new string [] { "\uC82F", "\uC82F", "\u110C\u1166\u11BA", "\uC82F", "\u110C\u1166\u11BA" },
            new string [] { "\uC830", "\uC830", "\u110C\u1166\u11BB", "\uC830", "\u110C\u1166\u11BB" },
            new string [] { "\uC831", "\uC831", "\u110C\u1166\u11BC", "\uC831", "\u110C\u1166\u11BC" },
            new string [] { "\uC832", "\uC832", "\u110C\u1166\u11BD", "\uC832", "\u110C\u1166\u11BD" },
            new string [] { "\uC833", "\uC833", "\u110C\u1166\u11BE", "\uC833", "\u110C\u1166\u11BE" },
            new string [] { "\uC834", "\uC834", "\u110C\u1166\u11BF", "\uC834", "\u110C\u1166\u11BF" },
            new string [] { "\uC835", "\uC835", "\u110C\u1166\u11C0", "\uC835", "\u110C\u1166\u11C0" },
            new string [] { "\uC836", "\uC836", "\u110C\u1166\u11C1", "\uC836", "\u110C\u1166\u11C1" },
            new string [] { "\uC837", "\uC837", "\u110C\u1166\u11C2", "\uC837", "\u110C\u1166\u11C2" },
            new string [] { "\uC838", "\uC838", "\u110C\u1167", "\uC838", "\u110C\u1167" },
            new string [] { "\uC839", "\uC839", "\u110C\u1167\u11A8", "\uC839", "\u110C\u1167\u11A8" },
            new string [] { "\uC83A", "\uC83A", "\u110C\u1167\u11A9", "\uC83A", "\u110C\u1167\u11A9" },
            new string [] { "\uC83B", "\uC83B", "\u110C\u1167\u11AA", "\uC83B", "\u110C\u1167\u11AA" },
            new string [] { "\uC83C", "\uC83C", "\u110C\u1167\u11AB", "\uC83C", "\u110C\u1167\u11AB" },
            new string [] { "\uC83D", "\uC83D", "\u110C\u1167\u11AC", "\uC83D", "\u110C\u1167\u11AC" },
            new string [] { "\uC83E", "\uC83E", "\u110C\u1167\u11AD", "\uC83E", "\u110C\u1167\u11AD" },
            new string [] { "\uC83F", "\uC83F", "\u110C\u1167\u11AE", "\uC83F", "\u110C\u1167\u11AE" },
            new string [] { "\uC840", "\uC840", "\u110C\u1167\u11AF", "\uC840", "\u110C\u1167\u11AF" },
            new string [] { "\uC841", "\uC841", "\u110C\u1167\u11B0", "\uC841", "\u110C\u1167\u11B0" },
            new string [] { "\uC842", "\uC842", "\u110C\u1167\u11B1", "\uC842", "\u110C\u1167\u11B1" },
            new string [] { "\uC843", "\uC843", "\u110C\u1167\u11B2", "\uC843", "\u110C\u1167\u11B2" },
            new string [] { "\uC844", "\uC844", "\u110C\u1167\u11B3", "\uC844", "\u110C\u1167\u11B3" },
            new string [] { "\uC845", "\uC845", "\u110C\u1167\u11B4", "\uC845", "\u110C\u1167\u11B4" },
            new string [] { "\uC846", "\uC846", "\u110C\u1167\u11B5", "\uC846", "\u110C\u1167\u11B5" },
            new string [] { "\uC847", "\uC847", "\u110C\u1167\u11B6", "\uC847", "\u110C\u1167\u11B6" },
            new string [] { "\uC848", "\uC848", "\u110C\u1167\u11B7", "\uC848", "\u110C\u1167\u11B7" },
            new string [] { "\uC849", "\uC849", "\u110C\u1167\u11B8", "\uC849", "\u110C\u1167\u11B8" },
            new string [] { "\uC84A", "\uC84A", "\u110C\u1167\u11B9", "\uC84A", "\u110C\u1167\u11B9" },
            new string [] { "\uC84B", "\uC84B", "\u110C\u1167\u11BA", "\uC84B", "\u110C\u1167\u11BA" },
            new string [] { "\uC84C", "\uC84C", "\u110C\u1167\u11BB", "\uC84C", "\u110C\u1167\u11BB" },
            new string [] { "\uC84D", "\uC84D", "\u110C\u1167\u11BC", "\uC84D", "\u110C\u1167\u11BC" },
            new string [] { "\uC84E", "\uC84E", "\u110C\u1167\u11BD", "\uC84E", "\u110C\u1167\u11BD" },
            new string [] { "\uC84F", "\uC84F", "\u110C\u1167\u11BE", "\uC84F", "\u110C\u1167\u11BE" },
            new string [] { "\uC850", "\uC850", "\u110C\u1167\u11BF", "\uC850", "\u110C\u1167\u11BF" },
            new string [] { "\uC851", "\uC851", "\u110C\u1167\u11C0", "\uC851", "\u110C\u1167\u11C0" },
            new string [] { "\uC852", "\uC852", "\u110C\u1167\u11C1", "\uC852", "\u110C\u1167\u11C1" },
            new string [] { "\uC853", "\uC853", "\u110C\u1167\u11C2", "\uC853", "\u110C\u1167\u11C2" },
            new string [] { "\uC854", "\uC854", "\u110C\u1168", "\uC854", "\u110C\u1168" },
            new string [] { "\uC855", "\uC855", "\u110C\u1168\u11A8", "\uC855", "\u110C\u1168\u11A8" },
            new string [] { "\uC856", "\uC856", "\u110C\u1168\u11A9", "\uC856", "\u110C\u1168\u11A9" },
            new string [] { "\uC857", "\uC857", "\u110C\u1168\u11AA", "\uC857", "\u110C\u1168\u11AA" },
            new string [] { "\uC858", "\uC858", "\u110C\u1168\u11AB", "\uC858", "\u110C\u1168\u11AB" },
            new string [] { "\uC859", "\uC859", "\u110C\u1168\u11AC", "\uC859", "\u110C\u1168\u11AC" },
            new string [] { "\uC85A", "\uC85A", "\u110C\u1168\u11AD", "\uC85A", "\u110C\u1168\u11AD" },
            new string [] { "\uC85B", "\uC85B", "\u110C\u1168\u11AE", "\uC85B", "\u110C\u1168\u11AE" },
            new string [] { "\uC85C", "\uC85C", "\u110C\u1168\u11AF", "\uC85C", "\u110C\u1168\u11AF" },
            new string [] { "\uC85D", "\uC85D", "\u110C\u1168\u11B0", "\uC85D", "\u110C\u1168\u11B0" },
            new string [] { "\uC85E", "\uC85E", "\u110C\u1168\u11B1", "\uC85E", "\u110C\u1168\u11B1" },
            new string [] { "\uC85F", "\uC85F", "\u110C\u1168\u11B2", "\uC85F", "\u110C\u1168\u11B2" },
            new string [] { "\uC860", "\uC860", "\u110C\u1168\u11B3", "\uC860", "\u110C\u1168\u11B3" },
            new string [] { "\uC861", "\uC861", "\u110C\u1168\u11B4", "\uC861", "\u110C\u1168\u11B4" },
            new string [] { "\uC862", "\uC862", "\u110C\u1168\u11B5", "\uC862", "\u110C\u1168\u11B5" },
            new string [] { "\uC863", "\uC863", "\u110C\u1168\u11B6", "\uC863", "\u110C\u1168\u11B6" },
            new string [] { "\uC864", "\uC864", "\u110C\u1168\u11B7", "\uC864", "\u110C\u1168\u11B7" },
            new string [] { "\uC865", "\uC865", "\u110C\u1168\u11B8", "\uC865", "\u110C\u1168\u11B8" },
            new string [] { "\uC866", "\uC866", "\u110C\u1168\u11B9", "\uC866", "\u110C\u1168\u11B9" },
            new string [] { "\uC867", "\uC867", "\u110C\u1168\u11BA", "\uC867", "\u110C\u1168\u11BA" },
            new string [] { "\uC868", "\uC868", "\u110C\u1168\u11BB", "\uC868", "\u110C\u1168\u11BB" },
            new string [] { "\uC869", "\uC869", "\u110C\u1168\u11BC", "\uC869", "\u110C\u1168\u11BC" },
            new string [] { "\uC86A", "\uC86A", "\u110C\u1168\u11BD", "\uC86A", "\u110C\u1168\u11BD" },
            new string [] { "\uC86B", "\uC86B", "\u110C\u1168\u11BE", "\uC86B", "\u110C\u1168\u11BE" },
            new string [] { "\uC86C", "\uC86C", "\u110C\u1168\u11BF", "\uC86C", "\u110C\u1168\u11BF" },
            new string [] { "\uC86D", "\uC86D", "\u110C\u1168\u11C0", "\uC86D", "\u110C\u1168\u11C0" },
            new string [] { "\uC86E", "\uC86E", "\u110C\u1168\u11C1", "\uC86E", "\u110C\u1168\u11C1" },
            new string [] { "\uC86F", "\uC86F", "\u110C\u1168\u11C2", "\uC86F", "\u110C\u1168\u11C2" },
            new string [] { "\uC870", "\uC870", "\u110C\u1169", "\uC870", "\u110C\u1169" },
            new string [] { "\uC871", "\uC871", "\u110C\u1169\u11A8", "\uC871", "\u110C\u1169\u11A8" },
            new string [] { "\uC872", "\uC872", "\u110C\u1169\u11A9", "\uC872", "\u110C\u1169\u11A9" },
            new string [] { "\uC873", "\uC873", "\u110C\u1169\u11AA", "\uC873", "\u110C\u1169\u11AA" },
            new string [] { "\uC874", "\uC874", "\u110C\u1169\u11AB", "\uC874", "\u110C\u1169\u11AB" },
            new string [] { "\uC875", "\uC875", "\u110C\u1169\u11AC", "\uC875", "\u110C\u1169\u11AC" },
            new string [] { "\uC876", "\uC876", "\u110C\u1169\u11AD", "\uC876", "\u110C\u1169\u11AD" },
            new string [] { "\uC877", "\uC877", "\u110C\u1169\u11AE", "\uC877", "\u110C\u1169\u11AE" },
            new string [] { "\uC878", "\uC878", "\u110C\u1169\u11AF", "\uC878", "\u110C\u1169\u11AF" },
            new string [] { "\uC879", "\uC879", "\u110C\u1169\u11B0", "\uC879", "\u110C\u1169\u11B0" },
            new string [] { "\uC87A", "\uC87A", "\u110C\u1169\u11B1", "\uC87A", "\u110C\u1169\u11B1" },
            new string [] { "\uC87B", "\uC87B", "\u110C\u1169\u11B2", "\uC87B", "\u110C\u1169\u11B2" },
            new string [] { "\uC87C", "\uC87C", "\u110C\u1169\u11B3", "\uC87C", "\u110C\u1169\u11B3" },
            new string [] { "\uC87D", "\uC87D", "\u110C\u1169\u11B4", "\uC87D", "\u110C\u1169\u11B4" },
            new string [] { "\uC87E", "\uC87E", "\u110C\u1169\u11B5", "\uC87E", "\u110C\u1169\u11B5" },
            new string [] { "\uC87F", "\uC87F", "\u110C\u1169\u11B6", "\uC87F", "\u110C\u1169\u11B6" },
            new string [] { "\uC880", "\uC880", "\u110C\u1169\u11B7", "\uC880", "\u110C\u1169\u11B7" },
            new string [] { "\uC881", "\uC881", "\u110C\u1169\u11B8", "\uC881", "\u110C\u1169\u11B8" },
            new string [] { "\uC882", "\uC882", "\u110C\u1169\u11B9", "\uC882", "\u110C\u1169\u11B9" },
            new string [] { "\uC883", "\uC883", "\u110C\u1169\u11BA", "\uC883", "\u110C\u1169\u11BA" },
            new string [] { "\uC884", "\uC884", "\u110C\u1169\u11BB", "\uC884", "\u110C\u1169\u11BB" },
            new string [] { "\uC885", "\uC885", "\u110C\u1169\u11BC", "\uC885", "\u110C\u1169\u11BC" },
            new string [] { "\uC886", "\uC886", "\u110C\u1169\u11BD", "\uC886", "\u110C\u1169\u11BD" },
            new string [] { "\uC887", "\uC887", "\u110C\u1169\u11BE", "\uC887", "\u110C\u1169\u11BE" },
            new string [] { "\uC888", "\uC888", "\u110C\u1169\u11BF", "\uC888", "\u110C\u1169\u11BF" },
            new string [] { "\uC889", "\uC889", "\u110C\u1169\u11C0", "\uC889", "\u110C\u1169\u11C0" },
            new string [] { "\uC88A", "\uC88A", "\u110C\u1169\u11C1", "\uC88A", "\u110C\u1169\u11C1" },
            new string [] { "\uC88B", "\uC88B", "\u110C\u1169\u11C2", "\uC88B", "\u110C\u1169\u11C2" },
            new string [] { "\uC88C", "\uC88C", "\u110C\u116A", "\uC88C", "\u110C\u116A" },
            new string [] { "\uC88D", "\uC88D", "\u110C\u116A\u11A8", "\uC88D", "\u110C\u116A\u11A8" },
            new string [] { "\uC88E", "\uC88E", "\u110C\u116A\u11A9", "\uC88E", "\u110C\u116A\u11A9" },
            new string [] { "\uC88F", "\uC88F", "\u110C\u116A\u11AA", "\uC88F", "\u110C\u116A\u11AA" },
            new string [] { "\uC890", "\uC890", "\u110C\u116A\u11AB", "\uC890", "\u110C\u116A\u11AB" },
            new string [] { "\uC891", "\uC891", "\u110C\u116A\u11AC", "\uC891", "\u110C\u116A\u11AC" },
            new string [] { "\uC892", "\uC892", "\u110C\u116A\u11AD", "\uC892", "\u110C\u116A\u11AD" },
            new string [] { "\uC893", "\uC893", "\u110C\u116A\u11AE", "\uC893", "\u110C\u116A\u11AE" },
            new string [] { "\uC894", "\uC894", "\u110C\u116A\u11AF", "\uC894", "\u110C\u116A\u11AF" },
            new string [] { "\uC895", "\uC895", "\u110C\u116A\u11B0", "\uC895", "\u110C\u116A\u11B0" },
            new string [] { "\uC896", "\uC896", "\u110C\u116A\u11B1", "\uC896", "\u110C\u116A\u11B1" },
            new string [] { "\uC897", "\uC897", "\u110C\u116A\u11B2", "\uC897", "\u110C\u116A\u11B2" },
            new string [] { "\uC898", "\uC898", "\u110C\u116A\u11B3", "\uC898", "\u110C\u116A\u11B3" },
            new string [] { "\uC899", "\uC899", "\u110C\u116A\u11B4", "\uC899", "\u110C\u116A\u11B4" },
            new string [] { "\uC89A", "\uC89A", "\u110C\u116A\u11B5", "\uC89A", "\u110C\u116A\u11B5" },
            new string [] { "\uC89B", "\uC89B", "\u110C\u116A\u11B6", "\uC89B", "\u110C\u116A\u11B6" },
            new string [] { "\uC89C", "\uC89C", "\u110C\u116A\u11B7", "\uC89C", "\u110C\u116A\u11B7" },
            new string [] { "\uC89D", "\uC89D", "\u110C\u116A\u11B8", "\uC89D", "\u110C\u116A\u11B8" },
            new string [] { "\uC89E", "\uC89E", "\u110C\u116A\u11B9", "\uC89E", "\u110C\u116A\u11B9" },
            new string [] { "\uC89F", "\uC89F", "\u110C\u116A\u11BA", "\uC89F", "\u110C\u116A\u11BA" },
            new string [] { "\uC8A0", "\uC8A0", "\u110C\u116A\u11BB", "\uC8A0", "\u110C\u116A\u11BB" },
            new string [] { "\uC8A1", "\uC8A1", "\u110C\u116A\u11BC", "\uC8A1", "\u110C\u116A\u11BC" },
            new string [] { "\uC8A2", "\uC8A2", "\u110C\u116A\u11BD", "\uC8A2", "\u110C\u116A\u11BD" },
            new string [] { "\uC8A3", "\uC8A3", "\u110C\u116A\u11BE", "\uC8A3", "\u110C\u116A\u11BE" },
            new string [] { "\uC8A4", "\uC8A4", "\u110C\u116A\u11BF", "\uC8A4", "\u110C\u116A\u11BF" },
            new string [] { "\uC8A5", "\uC8A5", "\u110C\u116A\u11C0", "\uC8A5", "\u110C\u116A\u11C0" },
            new string [] { "\uC8A6", "\uC8A6", "\u110C\u116A\u11C1", "\uC8A6", "\u110C\u116A\u11C1" },
            new string [] { "\uC8A7", "\uC8A7", "\u110C\u116A\u11C2", "\uC8A7", "\u110C\u116A\u11C2" },
            new string [] { "\uC8A8", "\uC8A8", "\u110C\u116B", "\uC8A8", "\u110C\u116B" },
            new string [] { "\uC8A9", "\uC8A9", "\u110C\u116B\u11A8", "\uC8A9", "\u110C\u116B\u11A8" },
            new string [] { "\uC8AA", "\uC8AA", "\u110C\u116B\u11A9", "\uC8AA", "\u110C\u116B\u11A9" },
            new string [] { "\uC8AB", "\uC8AB", "\u110C\u116B\u11AA", "\uC8AB", "\u110C\u116B\u11AA" },
            new string [] { "\uC8AC", "\uC8AC", "\u110C\u116B\u11AB", "\uC8AC", "\u110C\u116B\u11AB" },
            new string [] { "\uC8AD", "\uC8AD", "\u110C\u116B\u11AC", "\uC8AD", "\u110C\u116B\u11AC" },
            new string [] { "\uC8AE", "\uC8AE", "\u110C\u116B\u11AD", "\uC8AE", "\u110C\u116B\u11AD" },
            new string [] { "\uC8AF", "\uC8AF", "\u110C\u116B\u11AE", "\uC8AF", "\u110C\u116B\u11AE" },
            new string [] { "\uC8B0", "\uC8B0", "\u110C\u116B\u11AF", "\uC8B0", "\u110C\u116B\u11AF" },
            new string [] { "\uC8B1", "\uC8B1", "\u110C\u116B\u11B0", "\uC8B1", "\u110C\u116B\u11B0" },
            new string [] { "\uC8B2", "\uC8B2", "\u110C\u116B\u11B1", "\uC8B2", "\u110C\u116B\u11B1" },
            new string [] { "\uC8B3", "\uC8B3", "\u110C\u116B\u11B2", "\uC8B3", "\u110C\u116B\u11B2" },
            new string [] { "\uC8B4", "\uC8B4", "\u110C\u116B\u11B3", "\uC8B4", "\u110C\u116B\u11B3" },
            new string [] { "\uC8B5", "\uC8B5", "\u110C\u116B\u11B4", "\uC8B5", "\u110C\u116B\u11B4" },
            new string [] { "\uC8B6", "\uC8B6", "\u110C\u116B\u11B5", "\uC8B6", "\u110C\u116B\u11B5" },
            new string [] { "\uC8B7", "\uC8B7", "\u110C\u116B\u11B6", "\uC8B7", "\u110C\u116B\u11B6" },
            new string [] { "\uC8B8", "\uC8B8", "\u110C\u116B\u11B7", "\uC8B8", "\u110C\u116B\u11B7" },
            new string [] { "\uC8B9", "\uC8B9", "\u110C\u116B\u11B8", "\uC8B9", "\u110C\u116B\u11B8" },
            new string [] { "\uC8BA", "\uC8BA", "\u110C\u116B\u11B9", "\uC8BA", "\u110C\u116B\u11B9" },
            new string [] { "\uC8BB", "\uC8BB", "\u110C\u116B\u11BA", "\uC8BB", "\u110C\u116B\u11BA" },
            new string [] { "\uC8BC", "\uC8BC", "\u110C\u116B\u11BB", "\uC8BC", "\u110C\u116B\u11BB" },
            new string [] { "\uC8BD", "\uC8BD", "\u110C\u116B\u11BC", "\uC8BD", "\u110C\u116B\u11BC" },
            new string [] { "\uC8BE", "\uC8BE", "\u110C\u116B\u11BD", "\uC8BE", "\u110C\u116B\u11BD" },
            new string [] { "\uC8BF", "\uC8BF", "\u110C\u116B\u11BE", "\uC8BF", "\u110C\u116B\u11BE" },
            new string [] { "\uC8C0", "\uC8C0", "\u110C\u116B\u11BF", "\uC8C0", "\u110C\u116B\u11BF" },
            new string [] { "\uC8C1", "\uC8C1", "\u110C\u116B\u11C0", "\uC8C1", "\u110C\u116B\u11C0" },
            new string [] { "\uC8C2", "\uC8C2", "\u110C\u116B\u11C1", "\uC8C2", "\u110C\u116B\u11C1" },
            new string [] { "\uC8C3", "\uC8C3", "\u110C\u116B\u11C2", "\uC8C3", "\u110C\u116B\u11C2" },
            new string [] { "\uC8C4", "\uC8C4", "\u110C\u116C", "\uC8C4", "\u110C\u116C" },
            new string [] { "\uC8C5", "\uC8C5", "\u110C\u116C\u11A8", "\uC8C5", "\u110C\u116C\u11A8" },
            new string [] { "\uC8C6", "\uC8C6", "\u110C\u116C\u11A9", "\uC8C6", "\u110C\u116C\u11A9" },
            new string [] { "\uC8C7", "\uC8C7", "\u110C\u116C\u11AA", "\uC8C7", "\u110C\u116C\u11AA" },
            new string [] { "\uC8C8", "\uC8C8", "\u110C\u116C\u11AB", "\uC8C8", "\u110C\u116C\u11AB" },
            new string [] { "\uC8C9", "\uC8C9", "\u110C\u116C\u11AC", "\uC8C9", "\u110C\u116C\u11AC" },
            new string [] { "\uC8CA", "\uC8CA", "\u110C\u116C\u11AD", "\uC8CA", "\u110C\u116C\u11AD" },
            new string [] { "\uC8CB", "\uC8CB", "\u110C\u116C\u11AE", "\uC8CB", "\u110C\u116C\u11AE" },
            new string [] { "\uC8CC", "\uC8CC", "\u110C\u116C\u11AF", "\uC8CC", "\u110C\u116C\u11AF" },
            new string [] { "\uC8CD", "\uC8CD", "\u110C\u116C\u11B0", "\uC8CD", "\u110C\u116C\u11B0" },
            new string [] { "\uC8CE", "\uC8CE", "\u110C\u116C\u11B1", "\uC8CE", "\u110C\u116C\u11B1" },
            new string [] { "\uC8CF", "\uC8CF", "\u110C\u116C\u11B2", "\uC8CF", "\u110C\u116C\u11B2" },
            new string [] { "\uC8D0", "\uC8D0", "\u110C\u116C\u11B3", "\uC8D0", "\u110C\u116C\u11B3" },
            new string [] { "\uC8D1", "\uC8D1", "\u110C\u116C\u11B4", "\uC8D1", "\u110C\u116C\u11B4" },
            new string [] { "\uC8D2", "\uC8D2", "\u110C\u116C\u11B5", "\uC8D2", "\u110C\u116C\u11B5" },
            new string [] { "\uC8D3", "\uC8D3", "\u110C\u116C\u11B6", "\uC8D3", "\u110C\u116C\u11B6" },
            new string [] { "\uC8D4", "\uC8D4", "\u110C\u116C\u11B7", "\uC8D4", "\u110C\u116C\u11B7" },
            new string [] { "\uC8D5", "\uC8D5", "\u110C\u116C\u11B8", "\uC8D5", "\u110C\u116C\u11B8" },
            new string [] { "\uC8D6", "\uC8D6", "\u110C\u116C\u11B9", "\uC8D6", "\u110C\u116C\u11B9" },
            new string [] { "\uC8D7", "\uC8D7", "\u110C\u116C\u11BA", "\uC8D7", "\u110C\u116C\u11BA" },
            new string [] { "\uC8D8", "\uC8D8", "\u110C\u116C\u11BB", "\uC8D8", "\u110C\u116C\u11BB" },
            new string [] { "\uC8D9", "\uC8D9", "\u110C\u116C\u11BC", "\uC8D9", "\u110C\u116C\u11BC" },
            new string [] { "\uC8DA", "\uC8DA", "\u110C\u116C\u11BD", "\uC8DA", "\u110C\u116C\u11BD" },
            new string [] { "\uC8DB", "\uC8DB", "\u110C\u116C\u11BE", "\uC8DB", "\u110C\u116C\u11BE" },
            new string [] { "\uC8DC", "\uC8DC", "\u110C\u116C\u11BF", "\uC8DC", "\u110C\u116C\u11BF" },
            new string [] { "\uC8DD", "\uC8DD", "\u110C\u116C\u11C0", "\uC8DD", "\u110C\u116C\u11C0" },
            new string [] { "\uC8DE", "\uC8DE", "\u110C\u116C\u11C1", "\uC8DE", "\u110C\u116C\u11C1" },
            new string [] { "\uC8DF", "\uC8DF", "\u110C\u116C\u11C2", "\uC8DF", "\u110C\u116C\u11C2" },
            new string [] { "\uC8E0", "\uC8E0", "\u110C\u116D", "\uC8E0", "\u110C\u116D" },
            new string [] { "\uC8E1", "\uC8E1", "\u110C\u116D\u11A8", "\uC8E1", "\u110C\u116D\u11A8" },
            new string [] { "\uC8E2", "\uC8E2", "\u110C\u116D\u11A9", "\uC8E2", "\u110C\u116D\u11A9" },
            new string [] { "\uC8E3", "\uC8E3", "\u110C\u116D\u11AA", "\uC8E3", "\u110C\u116D\u11AA" },
            new string [] { "\uC8E4", "\uC8E4", "\u110C\u116D\u11AB", "\uC8E4", "\u110C\u116D\u11AB" },
            new string [] { "\uC8E5", "\uC8E5", "\u110C\u116D\u11AC", "\uC8E5", "\u110C\u116D\u11AC" },
            new string [] { "\uC8E6", "\uC8E6", "\u110C\u116D\u11AD", "\uC8E6", "\u110C\u116D\u11AD" },
            new string [] { "\uC8E7", "\uC8E7", "\u110C\u116D\u11AE", "\uC8E7", "\u110C\u116D\u11AE" },
            new string [] { "\uC8E8", "\uC8E8", "\u110C\u116D\u11AF", "\uC8E8", "\u110C\u116D\u11AF" },
            new string [] { "\uC8E9", "\uC8E9", "\u110C\u116D\u11B0", "\uC8E9", "\u110C\u116D\u11B0" },
            new string [] { "\uC8EA", "\uC8EA", "\u110C\u116D\u11B1", "\uC8EA", "\u110C\u116D\u11B1" },
            new string [] { "\uC8EB", "\uC8EB", "\u110C\u116D\u11B2", "\uC8EB", "\u110C\u116D\u11B2" },
            new string [] { "\uC8EC", "\uC8EC", "\u110C\u116D\u11B3", "\uC8EC", "\u110C\u116D\u11B3" },
            new string [] { "\uC8ED", "\uC8ED", "\u110C\u116D\u11B4", "\uC8ED", "\u110C\u116D\u11B4" },
            new string [] { "\uC8EE", "\uC8EE", "\u110C\u116D\u11B5", "\uC8EE", "\u110C\u116D\u11B5" },
            new string [] { "\uC8EF", "\uC8EF", "\u110C\u116D\u11B6", "\uC8EF", "\u110C\u116D\u11B6" },
            new string [] { "\uC8F0", "\uC8F0", "\u110C\u116D\u11B7", "\uC8F0", "\u110C\u116D\u11B7" },
            new string [] { "\uC8F1", "\uC8F1", "\u110C\u116D\u11B8", "\uC8F1", "\u110C\u116D\u11B8" },
            new string [] { "\uC8F2", "\uC8F2", "\u110C\u116D\u11B9", "\uC8F2", "\u110C\u116D\u11B9" },
            new string [] { "\uC8F3", "\uC8F3", "\u110C\u116D\u11BA", "\uC8F3", "\u110C\u116D\u11BA" },
            new string [] { "\uC8F4", "\uC8F4", "\u110C\u116D\u11BB", "\uC8F4", "\u110C\u116D\u11BB" },
            new string [] { "\uC8F5", "\uC8F5", "\u110C\u116D\u11BC", "\uC8F5", "\u110C\u116D\u11BC" },
            new string [] { "\uC8F6", "\uC8F6", "\u110C\u116D\u11BD", "\uC8F6", "\u110C\u116D\u11BD" },
            new string [] { "\uC8F7", "\uC8F7", "\u110C\u116D\u11BE", "\uC8F7", "\u110C\u116D\u11BE" },
            new string [] { "\uC8F8", "\uC8F8", "\u110C\u116D\u11BF", "\uC8F8", "\u110C\u116D\u11BF" },
            new string [] { "\uC8F9", "\uC8F9", "\u110C\u116D\u11C0", "\uC8F9", "\u110C\u116D\u11C0" },
            new string [] { "\uC8FA", "\uC8FA", "\u110C\u116D\u11C1", "\uC8FA", "\u110C\u116D\u11C1" },
            new string [] { "\uC8FB", "\uC8FB", "\u110C\u116D\u11C2", "\uC8FB", "\u110C\u116D\u11C2" },
            new string [] { "\uC8FC", "\uC8FC", "\u110C\u116E", "\uC8FC", "\u110C\u116E" },
            new string [] { "\uC8FD", "\uC8FD", "\u110C\u116E\u11A8", "\uC8FD", "\u110C\u116E\u11A8" },
            new string [] { "\uC8FE", "\uC8FE", "\u110C\u116E\u11A9", "\uC8FE", "\u110C\u116E\u11A9" },
            new string [] { "\uC8FF", "\uC8FF", "\u110C\u116E\u11AA", "\uC8FF", "\u110C\u116E\u11AA" },
            new string [] { "\uC900", "\uC900", "\u110C\u116E\u11AB", "\uC900", "\u110C\u116E\u11AB" },
            new string [] { "\uC901", "\uC901", "\u110C\u116E\u11AC", "\uC901", "\u110C\u116E\u11AC" },
            new string [] { "\uC902", "\uC902", "\u110C\u116E\u11AD", "\uC902", "\u110C\u116E\u11AD" },
            new string [] { "\uC903", "\uC903", "\u110C\u116E\u11AE", "\uC903", "\u110C\u116E\u11AE" },
            new string [] { "\uC904", "\uC904", "\u110C\u116E\u11AF", "\uC904", "\u110C\u116E\u11AF" },
            new string [] { "\uC905", "\uC905", "\u110C\u116E\u11B0", "\uC905", "\u110C\u116E\u11B0" },
            new string [] { "\uC906", "\uC906", "\u110C\u116E\u11B1", "\uC906", "\u110C\u116E\u11B1" },
            new string [] { "\uC907", "\uC907", "\u110C\u116E\u11B2", "\uC907", "\u110C\u116E\u11B2" },
            new string [] { "\uC908", "\uC908", "\u110C\u116E\u11B3", "\uC908", "\u110C\u116E\u11B3" },
            new string [] { "\uC909", "\uC909", "\u110C\u116E\u11B4", "\uC909", "\u110C\u116E\u11B4" },
            new string [] { "\uC90A", "\uC90A", "\u110C\u116E\u11B5", "\uC90A", "\u110C\u116E\u11B5" },
            new string [] { "\uC90B", "\uC90B", "\u110C\u116E\u11B6", "\uC90B", "\u110C\u116E\u11B6" },
            new string [] { "\uC90C", "\uC90C", "\u110C\u116E\u11B7", "\uC90C", "\u110C\u116E\u11B7" },
            new string [] { "\uC90D", "\uC90D", "\u110C\u116E\u11B8", "\uC90D", "\u110C\u116E\u11B8" },
            new string [] { "\uC90E", "\uC90E", "\u110C\u116E\u11B9", "\uC90E", "\u110C\u116E\u11B9" },
            new string [] { "\uC90F", "\uC90F", "\u110C\u116E\u11BA", "\uC90F", "\u110C\u116E\u11BA" },
            new string [] { "\uC910", "\uC910", "\u110C\u116E\u11BB", "\uC910", "\u110C\u116E\u11BB" },
            new string [] { "\uC911", "\uC911", "\u110C\u116E\u11BC", "\uC911", "\u110C\u116E\u11BC" },
