// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

using Xunit;
using System;
using System.Reflection;
using System.Collections.Generic;

#pragma warning disable 0067

namespace System.Reflection.Tests
{
    public class EventInfoPropertyTests
    {
        // Verify AddMethod for EventInfo
        [Fact]
        public void VerifyAddMethod1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.AddMethod;
            Assert.NotNull(mi);
            Assert.Equal("add_EventPublic", mi.Name);
        }

        // Verify AddMethod for EventInfo
        [Fact]
        public void VerifyAddMethod2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicStatic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.AddMethod;
            Assert.NotNull(mi);
            Assert.Equal("add_EventPublicStatic", mi.Name);
        }

        // Verify AddMethod for EventInfo
        [Fact]
        public void VerifyAddMethod3()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.AddMethod;
            Assert.NotNull(mi);
            Assert.Equal("add_EventPublic", mi.Name);
        }

        // Verify AddMethod for EventInfo
        [Fact]
        public void VerifyAddMethod4()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublicNew");
            Assert.NotNull(ei);
            MethodInfo mi = ei.AddMethod;
            Assert.NotNull(mi);
            Assert.Equal("add_EventPublicNew", mi.Name);
        }

        // Verify RemoveMethod for EventInfo
        [Fact]
        public void VerifyRemoveMethod1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RemoveMethod;
            Assert.NotNull(mi);
            Assert.Equal("remove_EventPublic", mi.Name);
        }

        // Verify RemoveMethod for EventInfo
        [Fact]
        public void VerifyRemoveMethod2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicStatic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RemoveMethod;
            Assert.NotNull(mi);
            Assert.Equal("remove_EventPublicStatic", mi.Name);
        }

        // Verify RemoveMethod for EventInfo
        [Fact]
        public void VerifyRemoveMethod3()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RemoveMethod;
            Assert.NotNull(mi);
            Assert.Equal("remove_EventPublic", mi.Name);
        }

        // Verify RemoveMethod for EventInfo
        [Fact]
        public void VerifyRemoveMethod4()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublicNew");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RemoveMethod;
            Assert.NotNull(mi);
            Assert.Equal("remove_EventPublicNew", mi.Name);
        }

        // Verify RaiseMethod for EventInfo
        // Events generated by C# compiler does not have raise method associated with them 
        // so it always return NUll.
        [Fact]
        public void VerifyRaiseMethod1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RaiseMethod;
            Assert.Null(mi);
        }

        [Fact]
        public void VerifyRaiseMethod2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicStatic");
            Assert.NotNull(ei);
            MethodInfo mi = ei.RaiseMethod;
            Assert.Null(mi);
        }

        //Verify EventHandlerType
        [Fact]
        public void VerifyEventHandlerType1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            Type t = ei.EventHandlerType;
            Assert.NotNull(t);
            Assert.Equal(typeof(EventHandler), t);
        }

        //Verify EventHandlerType
        [Fact]
        public void VerifyEventHandlerType2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicStatic");
            Assert.NotNull(ei);
            Type t = ei.EventHandlerType;
            Assert.NotNull(t);
            Assert.Equal(typeof(EventHandler), t);
        }

        //Verify EventHandlerType
        [Fact]
        public void VerifyEventHandlerType3()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublic");
            Assert.NotNull(ei);
            Type t = ei.EventHandlerType;
            Assert.NotNull(t);
            Assert.Equal(typeof(EventHandler), t);
        }

        //Verify EventHandlerType
        [Fact]
        public void VerifyEventHandlerType4()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublicNew");
            Assert.NotNull(ei);
            Type t = ei.EventHandlerType;
            Assert.NotNull(t);
            Assert.Equal(typeof(EventHandler), t);
        }

        //Verify EventHandlerType
        [Fact]
        public void VerifyEventHandlerType5()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicVirtual");
            Assert.NotNull(ei);
            Type t = ei.EventHandlerType;
            Assert.NotNull(t);
            Assert.Equal(typeof(EventHandler), t);
        }

        //Verify Attributes
        [Fact]
        public void VerifyAttributes1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            EventAttributes attr = ei.Attributes;
            EventAttributes expectedAttr = EventAttributes.None;
            Assert.Equal((object)expectedAttr, (object)attr);
        }

        //Verify Attributes
        [Fact]
        public void VerifyAttributes2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicStatic");
            Assert.NotNull(ei);
            EventAttributes attr = ei.Attributes;
            EventAttributes expectedAttr = EventAttributes.None;
            Assert.Equal((object)expectedAttr, (object)attr);
        }

        //Verify Attributes
        [Fact]
        public void VerifyAttributes3()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublic");
            Assert.NotNull(ei);
            EventAttributes attr = ei.Attributes;
            EventAttributes expectedAttr = EventAttributes.None;
            Assert.Equal((object)expectedAttr, (object)attr);
        }

        //Verify Attributes
        [Fact]
        public void VerifyAttributes4()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertySubClass), "EventPublicNew");
            Assert.NotNull(ei);
            EventAttributes attr = ei.Attributes;
            EventAttributes expectedAttr = EventAttributes.None;
            Assert.Equal((object)expectedAttr, (object)attr);
        }

        //Verify Attributes
        [Fact]
        public void VerifyAttributes5()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicVirtual");
            Assert.NotNull(ei);
            EventAttributes attr = ei.Attributes;
            EventAttributes expectedAttr = EventAttributes.None;
            Assert.Equal((object)expectedAttr, (object)attr);
        }

        //Verify IsSpecialName Property
        [Fact]
        public void VerifyIsSpecialName1()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublic");
            Assert.NotNull(ei);
            Assert.False(ei.IsSpecialName);
        }

        //Verify IsSpecialName Property
        [Fact]
        public void VerifyIsSpecialName2()
        {
            EventInfo ei = GetEventInfo(typeof(EventInfoPropertyBaseClass), "EventPublicVirtual");
            Assert.NotNull(ei);
            Assert.False(ei.IsSpecialName);
        }

        //private helper methods
        private static EventInfo GetEventInfo(Type t, string eventName)
        {
            TypeInfo ti = t.GetTypeInfo();
            IEnumerator<EventInfo> allevents = ti.DeclaredEvents.GetEnumerator();
            EventInfo eventFound = null;
            while (allevents.MoveNext())
            {
                if (eventName.Equals(allevents.Current.Name))
                {
                    eventFound = allevents.Current;
                    break;
                }
            }

            return eventFound;
        }
    }

    //Metadata for Reflection
    public class EventInfoPropertyBaseClass
    {
        public event EventHandler EventPublic;					// inherited
        public static event EventHandler EventPublicStatic;
        public virtual event EventHandler EventPublicVirtual;
    }

    public class EventInfoPropertySubClass : EventInfoPropertyBaseClass
    {
        public new event EventHandler EventPublic;	//overrides event				
        public event EventHandler EventPublicNew;  // new event
    }

    internal class MyClass
    {
        public event EventHandler myevent
        {
            add { }
            remove { }
        }
    }
}
