__license__ = "GNU Affero General Public License http://www.gnu.org/licenses/agpl.html"
__copyright__ = "Copyright (C) 2021 The OctoPrint Project - Released under terms of the AGPLv3 License"

import datetime
import os
import re

import pytest

from octoprint.util.files import (
    m20_timestamp_to_unix_timestamp,
    sanitize_filename,
    search_through_file,
    search_through_file_python,
    unix_timestamp_to_m20_timestamp,
)


@pytest.mark.parametrize(
    "filename, expected, really_universal",
    (
        ("some_file.gcode", "some_file.gcode", False),
        ("NUL.gcode", "NUL_.gcode", False),
        ("LPT1", "LPT1_", False),
        (".test.gcode", "test.gcode", False),
        ("..test.gcode", "test.gcode", False),
        ("file with space.gcode", "file with space.gcode", False),
        ("Wölfe 🐺.gcode", "Wölfe 🐺.gcode", False),
        # really universal
        ("some_file.gcode", "some_file.gcode", True),
        ("NUL.gcode", "NUL_.gcode", True),
        ("LPT1", "LPT1_", True),
        (".test.gcode", "test.gcode", True),
        ("..test.gcode", "test.gcode", True),
        ("file with space.gcode", "file_with_space.gcode", True),
        ("Wölfe 🐺.gcode", "Wolfe_wolf.gcode", True),
        # invalid chars
        ("file\1file.gcode", "filefile.gcode", False),
        ("file;|&?$*<>file.gcode", "filefile.gcode", False),
        ("file\1file.gcode", "filefile.gcode", True),
        ("file;|&?$*<>file.gcode", "filefile.gcode", True),
    ),
)
def test_sanitize_filename(filename, expected, really_universal):
    actual = sanitize_filename(filename, really_universal=really_universal)
    assert actual == expected


@pytest.mark.parametrize(
    "filename", ("file/with/slash.gcode", "file\\with\\backslash.gcode")
)
def test_sanitize_filename_invalid(filename):
    try:
        sanitize_filename(filename)
        raise RuntimeError("expected ValueError")
    except ValueError as exc:
        assert str(exc) == "name must not contain / or \\"


@pytest.mark.parametrize(
    "term, regex, expected",
    (
        ("umlaut", False, True),
        ("BOM", False, True),
        (r"^[^#]*BOM", True, False),
    ),
)
def test_search_through_file(term, regex, expected):
    path = os.path.join(
        os.path.abspath(os.path.dirname(__file__)), "_files", "utf8_without_bom.txt"
    )
    actual = search_through_file(path, term, regex=regex)
    assert actual == expected


@pytest.mark.parametrize(
    "term, expected",
    (
        ("umlaut", True),
        ("BOM", True),
        (r"^[^#]*BOM", False),
    ),
)
def test_search_through_file_python(term, expected):
    compiled = re.compile(term)
    path = os.path.join(
        os.path.abspath(os.path.dirname(__file__)), "_files", "utf8_without_bom.txt"
    )
    actual = search_through_file_python(path, term, compiled)
    assert actual == expected


# based on https://github.com/nathanhi/pyfatfs/blob/master/tests/test_DosDateTime.py
m20_timestamp_tests = [
    ("0x210000", datetime.datetime(1980, 1, 1).timestamp()),
    ("0x21bf7d", datetime.datetime(1980, 1, 1, 23, 59, 58).timestamp()),
    ("0x549088aa", datetime.datetime(2022, 4, 16, 17, 5, 20).timestamp()),
    ("0x28210800", datetime.datetime(2000, 1, 1, 1, 0).timestamp()),
]

# 32bit time_t systems will fail with:
# "OverflowError: timestamp out of range for platform time_t"
# for this date.
try:
    m20_timestamp_tests.append(
        ("0xff9f0000", datetime.datetime(2107, 12, 31).timestamp())
    )
except OverflowError:
    pass


@pytest.mark.parametrize("val,expected", m20_timestamp_tests)
def test_m20_timestamp_to_unix_timestamp(val, expected):
    assert m20_timestamp_to_unix_timestamp(val) == expected


@pytest.mark.parametrize("expected,val", m20_timestamp_tests)
def test_unix_timestamp_to_m20_timestamp(expected, val):
    assert unix_timestamp_to_m20_timestamp(val) == expected
