package output

import (
	"encoding/json"
	"errors"
	"fmt"
	"strings"

	"github.com/OctopusDeploy/cli/pkg/constants"

	"github.com/OctopusDeploy/cli/pkg/usage"

	"github.com/spf13/cobra"
	"github.com/spf13/viper"
)

func PrintArray[T any](items []T, cmd *cobra.Command, mappers Mappers[T]) error {
	outputFormat, _ := cmd.Flags().GetString(constants.FlagOutputFormat)
	if outputFormat == "" {
		outputFormat = viper.GetString(constants.ConfigOutputFormat)
	}

	switch strings.ToLower(outputFormat) {
	case constants.OutputFormatJson:
		jsonMapper := mappers.Json
		if jsonMapper == nil {
			return errors.New("command does not support output in JSON format")
		}
		var outputJson []any
		for _, e := range items {
			outputJson = append(outputJson, jsonMapper(e))
		}

		data, _ := json.MarshalIndent(outputJson, "", "  ")
		cmd.Println(string(data))

	case constants.OutputFormatBasic:
		textMapper := mappers.Basic
		if textMapper == nil {
			return errors.New("command does not support output in plain text")
		}
		for _, e := range items {
			cmd.Println(textMapper(e))
		}

	case constants.OutputFormatTable, "": // table is the default of unspecified
		tableMapper := mappers.Table
		if tableMapper.Row == nil {
			return errors.New("command does not support output in table format")
		}

		t := NewTable(cmd.OutOrStdout())
		if tableMapper.Header != nil {
			for k, v := range tableMapper.Header {
				tableMapper.Header[k] = Bold(v)
			}
			t.AddRow(tableMapper.Header...)
		}

		for _, item := range items {
			t.AddRow(tableMapper.Row(item)...)
		}

		return t.Print()

	default:
		return usage.NewUsageError(
			fmt.Sprintf("unsupported output format %s. Valid values are 'json', 'table', 'basic'. Defaults to table", outputFormat),
			cmd)
	}
	return nil
}
