package keeper

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	sdk "github.com/cosmos/cosmos-sdk/types"
	sdkerrors "github.com/cosmos/cosmos-sdk/types/errors"

	"ollo/x/liquidity/amm"
	"ollo/x/liquidity/types"
)

func (k Keeper) PriceLimits(ctx sdk.Context, lastPrice sdk.Dec) (lowest, highest sdk.Dec) {
	return types.PriceLimits(lastPrice, k.GetMaxPriceLimitRatio(ctx), int(k.GetTickPrecision(ctx)))
}

// ValidateMsgLimitOrder validates types.MsgLimitOrder with state and returns
// calculated offer coin and price that is fit into ticks.
func (k Keeper) ValidateMsgLimitOrder(ctx sdk.Context, msg *types.MsgOrderLimit) (offerCoin sdk.Coin, price sdk.Dec, err error) {
	spendable := k.bankKeeper.SpendableCoins(ctx, msg.GetOrdererAddress())
	if spendableAmt := spendable.AmountOf(msg.OfferCoin.Denom); spendableAmt.LT(msg.OfferCoin.Amount) {
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
			sdkerrors.ErrInsufficientFunds, "%s is smaller than %s",
			sdk.NewCoin(msg.OfferCoin.Denom, spendableAmt), msg.OfferCoin)
	}

	tickPrec := k.GetTickPrecision(ctx)
	maxOrderLifespan := k.GetMaxOrderLifespan(ctx)

	if msg.OrderLifespan > maxOrderLifespan {
		return sdk.Coin{}, sdk.Dec{},
			sdkerrors.Wrapf(types.ErrTooLongOrderLifespan, "%s is longer than %s", msg.OrderLifespan, maxOrderLifespan)
	}

	pair, found := k.GetPair(ctx, msg.PairId)
	if !found {
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(sdkerrors.ErrNotFound, "pair %d not found", msg.PairId)
	}

	var upperPriceLimit, lowerPriceLimit sdk.Dec
	if pair.LastPrice != nil {
		lowerPriceLimit, upperPriceLimit = k.PriceLimits(ctx, *pair.LastPrice)
	} else {
		upperPriceLimit = amm.HighestTick(int(tickPrec))
		lowerPriceLimit = amm.LowestTick(int(tickPrec))
	}
	switch {
	case msg.Price.GT(upperPriceLimit):
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "%s is higher than %s", msg.Price, upperPriceLimit)
	case msg.Price.LT(lowerPriceLimit):
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "%s is lower than %s", msg.Price, lowerPriceLimit)
	}

	switch msg.Direction {
	case types.OrderDirectionBuy:
		if msg.OfferCoin.Denom != pair.QuoteDenom || msg.DemandCoinDenom != pair.BaseDenom {
			return sdk.Coin{}, sdk.Dec{},
				sdkerrors.Wrapf(types.ErrWrongPair, "denom pair (%s, %s) != (%s, %s)",
					msg.DemandCoinDenom, msg.OfferCoin.Denom, pair.BaseDenom, pair.QuoteDenom)
		}
		price = amm.PriceToDownTick(msg.Price, int(tickPrec))
		offerCoin = sdk.NewCoin(msg.OfferCoin.Denom, amm.OfferCoinAmount(amm.Buy, price, msg.Amount))
		if msg.OfferCoin.IsLT(offerCoin) {
			return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
				types.ErrInsufficientOfferCoin, "%s is smaller than %s", msg.OfferCoin, offerCoin)
		}
	case types.OrderDirectionSell:
		if msg.OfferCoin.Denom != pair.BaseDenom || msg.DemandCoinDenom != pair.QuoteDenom {
			return sdk.Coin{}, sdk.Dec{},
				sdkerrors.Wrapf(types.ErrWrongPair, "denom pair (%s, %s) != (%s, %s)",
					msg.OfferCoin.Denom, msg.DemandCoinDenom, pair.BaseDenom, pair.QuoteDenom)
		}
		price = amm.PriceToUpTick(msg.Price, int(tickPrec))
		offerCoin = sdk.NewCoin(msg.OfferCoin.Denom, msg.Amount)
		if msg.OfferCoin.Amount.LT(msg.Amount) {
			return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
				types.ErrInsufficientOfferCoin, "%s is smaller than %s", msg.OfferCoin, sdk.NewCoin(msg.OfferCoin.Denom, msg.Amount))
		}
	}
	if types.IsTooSmallOrderAmount(msg.Amount, price) {
		return sdk.Coin{}, sdk.Dec{}, types.ErrTooSmallOrder
	}

	return offerCoin, price, nil
}

// LimitOrder handles types.MsgLimitOrder and stores types.Order.
func (k Keeper) LimitOrder(ctx sdk.Context, msg *types.MsgOrderLimit) (types.Order, error) {
	offerCoin, price, err := k.ValidateMsgLimitOrder(ctx, msg)
	if err != nil {
		return types.Order{}, err
	}

	refundedCoin := msg.OfferCoin.Sub(offerCoin)
	pair, _ := k.GetPair(ctx, msg.PairId)
	if err := k.bankKeeper.SendCoins(ctx, msg.GetOrdererAddress(), pair.GetEscrowAddress(), sdk.NewCoins(offerCoin)); err != nil {
		return types.Order{}, err
	}

	requestId := k.getNextOrderIdWithUpdate(ctx, pair)
	expireAt := ctx.BlockTime().Add(msg.OrderLifespan)
	order := types.NewOrderForOrderLimit(msg, requestId, pair, offerCoin, price, expireAt, ctx.BlockHeight())
	k.SetOrder(ctx, order)
	k.SetOrderIndex(ctx, order)

	ctx.GasMeter().ConsumeGas(k.GetOrderExtraGas(ctx), "OrderExtraGas")

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeLimitOrder,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.Orderer),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(msg.PairId, 10)),
			sdk.NewAttribute(types.AttributeKeyOrderDirection, msg.Direction.String()),
			sdk.NewAttribute(types.AttributeKeyOfferCoin, offerCoin.String()),
			sdk.NewAttribute(types.AttributeKeyDemandCoinDenom, msg.DemandCoinDenom),
			sdk.NewAttribute(types.AttributeKeyPrice, price.String()),
			sdk.NewAttribute(types.AttributeKeyAmount, msg.Amount.String()),
			sdk.NewAttribute(types.AttributeKeyOrderId, strconv.FormatUint(order.Id, 10)),
			sdk.NewAttribute(types.AttributeKeyBatchId, strconv.FormatUint(order.BatchId, 10)),
			sdk.NewAttribute(types.AttributeKeyExpireAt, order.Expires.Format(time.RFC3339)),
			sdk.NewAttribute(types.AttributeKeyRefundedCoins, refundedCoin.String()),
		),
	})

	return order, nil
}

// ValidateMsgOrderMarket validates types.MsgOrderMarket with state and returns
// calculated offer coin and price.
func (k Keeper) ValidateMsgOrderMarket(ctx sdk.Context, msg *types.MsgOrderMarket) (offerCoin sdk.Coin, price sdk.Dec, err error) {
	spendable := k.bankKeeper.SpendableCoins(ctx, msg.GetOrdererAddress())
	if spendableAmt := spendable.AmountOf(msg.OfferCoin.Denom); spendableAmt.LT(msg.OfferCoin.Amount) {
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
			sdkerrors.ErrInsufficientFunds, "%s is smaller than %s",
			sdk.NewCoin(msg.OfferCoin.Denom, spendableAmt), msg.OfferCoin)
	}

	maxOrderLifespan := k.GetMaxOrderLifespan(ctx)
	maxPriceLimitRatio := k.GetMaxPriceLimitRatio(ctx)
	tickPrec := k.GetTickPrecision(ctx)

	if msg.OrderLifespan > maxOrderLifespan {
		return sdk.Coin{}, sdk.Dec{},
			sdkerrors.Wrapf(types.ErrTooLongOrderLifespan, "%s is longer than %s", msg.OrderLifespan, maxOrderLifespan)
	}

	pair, found := k.GetPair(ctx, msg.PairId)
	if !found {
		return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(sdkerrors.ErrNotFound, "pair %d not found", msg.PairId)
	}

	if pair.LastPrice == nil {
		return sdk.Coin{}, sdk.Dec{}, types.ErrNoLastPrice
	}
	lastPrice := *pair.LastPrice

	switch msg.Direction {
	case types.OrderDirectionBuy:
		if msg.OfferCoin.Denom != pair.QuoteDenom || msg.DemandCoinDenom != pair.BaseDenom {
			return sdk.Coin{}, sdk.Dec{},
				sdkerrors.Wrapf(types.ErrWrongPair, "denom pair (%s, %s) != (%s, %s)",
					msg.DemandCoinDenom, msg.OfferCoin.Denom, pair.BaseDenom, pair.QuoteDenom)
		}
		price = amm.PriceToDownTick(lastPrice.Mul(sdk.OneDec().Add(maxPriceLimitRatio)), int(tickPrec))
		offerCoin = sdk.NewCoin(msg.OfferCoin.Denom, amm.OfferCoinAmount(amm.Buy, price, msg.Amount))
		if msg.OfferCoin.IsLT(offerCoin) {
			return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
				types.ErrInsufficientOfferCoin, "%s is smaller than %s", msg.OfferCoin, offerCoin)
		}
	case types.OrderDirectionSell:
		if msg.OfferCoin.Denom != pair.BaseDenom || msg.DemandCoinDenom != pair.QuoteDenom {
			return sdk.Coin{}, sdk.Dec{},
				sdkerrors.Wrapf(types.ErrWrongPair, "denom pair (%s, %s) != (%s, %s)",
					msg.OfferCoin.Denom, msg.DemandCoinDenom, pair.BaseDenom, pair.QuoteDenom)
		}
		price = amm.PriceToUpTick(lastPrice.Mul(sdk.OneDec().Sub(maxPriceLimitRatio)), int(tickPrec))
		offerCoin = sdk.NewCoin(msg.OfferCoin.Denom, msg.Amount)
		if msg.OfferCoin.Amount.LT(msg.Amount) {
			return sdk.Coin{}, sdk.Dec{}, sdkerrors.Wrapf(
				types.ErrInsufficientOfferCoin, "%s is smaller than %s", msg.OfferCoin, sdk.NewCoin(msg.OfferCoin.Denom, msg.Amount))
		}
	}
	if types.IsTooSmallOrderAmount(msg.Amount, price) {
		return sdk.Coin{}, sdk.Dec{}, types.ErrTooSmallOrder
	}

	return offerCoin, price, nil
}

// OrderMarket handles types.MsgOrderMarket and stores types.Order.
func (k Keeper) OrderMarket(ctx sdk.Context, msg *types.MsgOrderMarket) (types.Order, error) {
	offerCoin, price, err := k.ValidateMsgOrderMarket(ctx, msg)
	if err != nil {
		return types.Order{}, err
	}

	refundedCoin := msg.OfferCoin.Sub(offerCoin)
	pair, _ := k.GetPair(ctx, msg.PairId)
	if err := k.bankKeeper.SendCoins(ctx, msg.GetOrdererAddress(), pair.GetEscrowAddress(), sdk.NewCoins(offerCoin)); err != nil {
		return types.Order{}, err
	}

	requestId := k.getNextOrderIdWithUpdate(ctx, pair)
	expireAt := ctx.BlockTime().Add(msg.OrderLifespan)
	order := types.NewOrderForOrderMarket(msg, requestId, pair, offerCoin, price, expireAt, ctx.BlockHeight())
	k.SetOrder(ctx, order)
	k.SetOrderIndex(ctx, order)

	ctx.GasMeter().ConsumeGas(k.GetOrderExtraGas(ctx), "OrderExtraGas")

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeMMOrder,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.Orderer),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(msg.PairId, 10)),
			sdk.NewAttribute(types.AttributeKeyOrderDirection, msg.Direction.String()),
			sdk.NewAttribute(types.AttributeKeyOfferCoin, offerCoin.String()),
			sdk.NewAttribute(types.AttributeKeyDemandCoinDenom, msg.DemandCoinDenom),
			sdk.NewAttribute(types.AttributeKeyPrice, price.String()),
			sdk.NewAttribute(types.AttributeKeyAmount, msg.Amount.String()),
			sdk.NewAttribute(types.AttributeKeyOrderId, strconv.FormatUint(order.Id, 10)),
			sdk.NewAttribute(types.AttributeKeyBatchId, strconv.FormatUint(order.BatchId, 10)),
			sdk.NewAttribute(types.AttributeKeyExpireAt, order.Expires.Format(time.RFC3339)),
			sdk.NewAttribute(types.AttributeKeyRefundedCoins, refundedCoin.String()),
		),
	})

	return order, nil
}

func (k Keeper) MMOrder(ctx sdk.Context, msg *types.MsgOrderMarketMaking) (orders []types.Order, err error) {
	tickPrec := int(k.GetTickPrecision(ctx))

	if msg.SellAmount.IsPositive() {
		if !amm.PriceToDownTick(msg.MinSellPrice, tickPrec).Equal(msg.MinSellPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceNotOnTicks, "min sell price is not on ticks")
		}
		if !amm.PriceToDownTick(msg.MaxSellPrice, tickPrec).Equal(msg.MaxSellPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceNotOnTicks, "max sell price is not on ticks")
		}
	}
	if msg.BuyAmount.IsPositive() {
		if !amm.PriceToDownTick(msg.MinBuyPrice, tickPrec).Equal(msg.MinBuyPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceNotOnTicks, "min buy price is not on ticks")
		}
		if !amm.PriceToDownTick(msg.MaxBuyPrice, tickPrec).Equal(msg.MaxBuyPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceNotOnTicks, "max buy price is not on ticks")
		}
	}

	pair, found := k.GetPair(ctx, msg.PairId)
	if !found {
		return nil, sdkerrors.Wrapf(sdkerrors.ErrNotFound, "pair %d not found", msg.PairId)
	}

	var lowestPrice, highestPrice sdk.Dec
	if pair.LastPrice != nil {
		lowestPrice, highestPrice = types.PriceLimits(*pair.LastPrice, k.GetMaxPriceLimitRatio(ctx), tickPrec)
	} else {
		lowestPrice = amm.LowestTick(tickPrec)
		highestPrice = amm.HighestTick(tickPrec)
	}

	if msg.SellAmount.IsPositive() {
		if msg.MinSellPrice.LT(lowestPrice) || msg.MinSellPrice.GT(highestPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "min sell price is out of range [%s, %s]", lowestPrice, highestPrice)
		}
		if msg.MaxSellPrice.LT(lowestPrice) || msg.MaxSellPrice.GT(highestPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "max sell price is out of range [%s, %s]", lowestPrice, highestPrice)
		}
	}
	if msg.BuyAmount.IsPositive() {
		if msg.MinBuyPrice.LT(lowestPrice) || msg.MinBuyPrice.GT(highestPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "min buy price is out of range [%s, %s]", lowestPrice, highestPrice)
		}
		if msg.MaxBuyPrice.LT(lowestPrice) || msg.MaxBuyPrice.GT(highestPrice) {
			return nil, sdkerrors.Wrapf(types.ErrPriceOutOfRange, "max buy price is out of range [%s, %s]", lowestPrice, highestPrice)
		}
	}

	maxNumTicks := int(k.GetMaxNumMarketMakingOrderTicks(ctx))

	var buyTicks, sellTicks []types.MMOrderTick
	offerBaseCoin := sdk.NewInt64Coin(pair.BaseDenom, 0)
	offerQuoteCoin := sdk.NewInt64Coin(pair.QuoteDenom, 0)
	if msg.BuyAmount.IsPositive() {
		buyTicks = types.MMOrderTicks(
			types.OrderDirectionBuy, msg.MinBuyPrice, msg.MaxBuyPrice, msg.BuyAmount, maxNumTicks, tickPrec)
		for _, tick := range buyTicks {
			offerQuoteCoin = offerQuoteCoin.AddAmount(tick.OfferCoinAmount)
		}
	}
	if msg.SellAmount.IsPositive() {
		sellTicks = types.MMOrderTicks(
			types.OrderDirectionSell, msg.MinSellPrice, msg.MaxSellPrice, msg.SellAmount, maxNumTicks, tickPrec)
		for _, tick := range sellTicks {
			offerBaseCoin = offerBaseCoin.AddAmount(tick.OfferCoinAmount)
		}
	}

	orderer := msg.GetOrderer()
	spendable := k.bankKeeper.SpendableCoins(ctx, sdk.AccAddress(orderer))
	if spendableAmt := spendable.AmountOf(pair.BaseDenom); spendableAmt.LT(offerBaseCoin.Amount) {
		return nil, sdkerrors.Wrapf(
			sdkerrors.ErrInsufficientFunds, "%s is smaller than %s",
			sdk.NewCoin(pair.BaseDenom, spendableAmt), offerBaseCoin)
	}
	if spendableAmt := spendable.AmountOf(pair.QuoteDenom); spendableAmt.LT(offerQuoteCoin.Amount) {
		return nil, sdkerrors.Wrapf(
			sdkerrors.ErrInsufficientFunds, "%s is smaller than %s",
			sdk.NewCoin(pair.QuoteDenom, spendableAmt), offerQuoteCoin)
	}

	maxOrderLifespan := k.GetMaxOrderLifespan(ctx)
	if msg.OrderLifespan > maxOrderLifespan {
		return nil, sdkerrors.Wrapf(
			types.ErrTooLongOrderLifespan, "%s is longer than %s", msg.OrderLifespan, maxOrderLifespan)
	}

	// First, cancel existing market making orders in the pair from the orderer.
	canceledOrderIds, err := k.cancelMMOrder(ctx, sdk.AccAddress(orderer), pair, true)
	if err != nil {
		return nil, err
	}

	if err := k.bankKeeper.SendCoins(ctx, sdk.AccAddress(orderer), pair.GetEscrowAddress(), sdk.NewCoins(offerBaseCoin, offerQuoteCoin)); err != nil {
		return nil, err
	}

	expireAt := ctx.BlockTime().Add(msg.OrderLifespan)
	lastOrderId := pair.LastOrderId

	var orderIds []uint64
	for _, tick := range buyTicks {
		lastOrderId++
		offerCoin := sdk.NewCoin(pair.QuoteDenom, tick.OfferCoinAmount)
		order := types.NewOrder(
			types.OrderTypeMarketMaking, lastOrderId, pair, sdk.AccAddress(orderer),
			offerCoin, tick.Price, tick.Amount, expireAt, ctx.BlockHeight())
		k.SetOrder(ctx, order)
		k.SetOrderIndex(ctx, order)
		orders = append(orders, order)
		orderIds = append(orderIds, order.Id)
	}
	for _, tick := range sellTicks {
		lastOrderId++
		offerCoin := sdk.NewCoin(pair.BaseDenom, tick.OfferCoinAmount)
		order := types.NewOrder(
			types.OrderTypeMarketMaking, lastOrderId, pair, sdk.AccAddress(orderer),
			offerCoin, tick.Price, tick.Amount, expireAt, ctx.BlockHeight())
		k.SetOrder(ctx, order)
		k.SetOrderIndex(ctx, order)
		orders = append(orders, order)
		orderIds = append(orderIds, order.Id)
	}

	pair.LastOrderId = lastOrderId
	k.SetPair(ctx, pair)

	k.SetMarketMakingOrderId(ctx, types.NewMMOrderIndex(sdk.AccAddress(orderer), pair.Id, orderIds))

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeMMOrder,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.Orderer),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(msg.PairId, 10)),
			sdk.NewAttribute(types.AttributeKeyBatchId, strconv.FormatUint(pair.CurrentBatchId, 10)),
			sdk.NewAttribute(types.AttributeKeyOrderIds, types.FormatUint64s(orderIds)),
			sdk.NewAttribute(types.AttributeKeyCanceledOrderIds, types.FormatUint64s(canceledOrderIds)),
		),
	})
	return
}

// ValidateMsgCancelOrder validates types.MsgCancelOrder and returns the order.
func (k Keeper) ValidateMsgCancelOrder(ctx sdk.Context, msg *types.MsgCancelOrder) (order types.Order, err error) {
	var found bool
	order, found = k.GetOrder(ctx, msg.PairId, msg.OrderId)
	if !found {
		return types.Order{},
			sdkerrors.Wrapf(sdkerrors.ErrNotFound, "order %d not found in pair %d", msg.OrderId, msg.PairId)
	}
	if msg.OrderAddr != order.CreatorAddr {
		return types.Order{}, sdkerrors.Wrap(sdkerrors.ErrUnauthorized, "mismatching orderer")
	}
	if order.Status == types.OrderStatusCanceled {
		return types.Order{}, types.ErrAlreadyCanceled
	}
	pair, _ := k.GetPair(ctx, msg.PairId)
	if order.BatchId == pair.CurrentBatchId {
		return types.Order{}, types.ErrSameBatch
	}
	return order, nil
}

// CancelOrder handles types.MsgCancelOrder and cancels an order.
func (k Keeper) CancelOrder(ctx sdk.Context, msg *types.MsgCancelOrder) error {
	order, err := k.ValidateMsgCancelOrder(ctx, msg)
	if err != nil {
		return err
	}

	if err := k.FinishOrder(ctx, order, types.OrderStatusCanceled); err != nil {
		return err
	}

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeCancelOrder,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.OrderAddr),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(msg.PairId, 10)),
			sdk.NewAttribute(types.AttributeKeyOrderId, strconv.FormatUint(msg.OrderId, 10)),
		),
	})

	return nil
}

// CancelAllOrders handles types.MsgCancelAllOrders and cancels all orders.
func (k Keeper) CancelAllOrders(ctx sdk.Context, msg *types.MsgCancelAllOrders) error {
	orderPairCache := map[uint64]types.Pair{} // maps order's pair id to pair, to cache the result
	pairIdSet := map[uint64]struct{}{}        // set of pairs where to cancel orders
	var pairIds []string                      // needed to emit an event
	for _, pairId := range msg.PairIds {
		pair, found := k.GetPair(ctx, pairId)
		if !found { // check if the pair exists
			return sdkerrors.Wrapf(sdkerrors.ErrNotFound, "pair %d not found", pairId)
		}
		pairIdSet[pairId] = struct{}{} // add pair id to the set
		pairIds = append(pairIds, strconv.FormatUint(pairId, 10))
		orderPairCache[pairId] = pair // also cache the pair to use at below
	}

	var canceledOrderIds []string
	if err := k.IterateOrdersByOrderer(ctx, msg.GetOrdererAddress(), func(order types.Order) (stop bool, err error) {
		_, ok := pairIdSet[order.PairId] // is the pair included in the pair set?
		if len(pairIdSet) == 0 || ok {   // pair ids not specified(cancel all), or the pair is in the set
			pair, ok := orderPairCache[order.PairId]
			if !ok {
				pair, _ = k.GetPair(ctx, order.PairId)
				orderPairCache[order.PairId] = pair
			}
			if order.Status != types.OrderStatusCanceled && order.BatchId < pair.CurrentBatchId {
				if err := k.FinishOrder(ctx, order, types.OrderStatusCanceled); err != nil {
					return false, err
				}
				canceledOrderIds = append(canceledOrderIds, strconv.FormatUint(order.Id, 10))
			}
		}
		return false, nil
	}); err != nil {
		return err
	}

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeCancelAllOrders,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.OrderAddr),
			sdk.NewAttribute(types.AttributeKeyPairIds, strings.Join(pairIds, ",")),
			sdk.NewAttribute(types.AttributeKeyCanceledOrderIds, strings.Join(canceledOrderIds, ",")),
		),
	})

	return nil
}

func (k Keeper) cancelMMOrder(ctx sdk.Context, orderer sdk.AccAddress, pair types.Pair, skipIfNotFound bool) (canceledOrderIds []uint64, err error) {
	index, found := k.GetMarketMakingOrderId(ctx, orderer, pair.Id)
	if found {
		for _, orderId := range index.OrderIds {
			order, found := k.GetOrder(ctx, pair.Id, orderId)
			if !found {
				// The order has already been deleted from store.
				continue
			}
			if order.BatchId == pair.CurrentBatchId {
				return nil, sdkerrors.Wrap(types.ErrSameBatch, "couldn't cancel previously placed orders")
			}
			if order.Status.CanBeCanceled() {
				if err := k.FinishOrder(ctx, order, types.OrderStatusCanceled); err != nil {
					return nil, err
				}
				canceledOrderIds = append(canceledOrderIds, order.Id)
			}
		}
		k.DeleteMarketMakingOrderId(ctx, index)
	} else if !skipIfNotFound {
		return nil, sdkerrors.Wrap(sdkerrors.ErrNotFound, "previous market making orders not found")
	}
	return
}

// CancelMMOrder handles types.MsgCancelMMOrder and cancels previous market making
// orders.
func (k Keeper) CancelMMOrder(ctx sdk.Context, msg *types.MsgCancelMarketMakingOrder) (canceledOrderIds []uint64, err error) {
	pair, found := k.GetPair(ctx, msg.PairId)
	if !found {
		return nil, sdkerrors.Wrapf(sdkerrors.ErrNotFound, "pair %d not found", msg.PairId)
	}

	canceledOrderIds, err = k.cancelMMOrder(ctx, msg.GetOrdererAddress(), pair, false)
	if err != nil {
		return
	}

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeCancelMMOrder,
			sdk.NewAttribute(types.AttributeKeyOrderer, msg.Orderer),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(pair.Id, 10)),
			sdk.NewAttribute(types.AttributeKeyCanceledOrderIds, types.FormatUint64s(canceledOrderIds)),
		),
	})

	return canceledOrderIds, nil
}

func (k Keeper) ExecuteMatching(ctx sdk.Context, pair types.Pair) error {
	ob := amm.NewOrderBook()

	if err := k.IterateOrdersByPair(ctx, pair.Id, func(order types.Order) (stop bool, err error) {
		switch order.Status {
		case types.OrderStatusMatching,
			types.OrderStatusNoMatch,
			types.OrderStatusPartialMatch:
			if order.Status != types.OrderStatusMatching && order.ExpiredAt(ctx.BlockTime()) {
				if err := k.FinishOrder(ctx, order, types.OrderStatusExpired); err != nil {
					return false, err
				}
				return false, nil
			}
			// TODO: add orders only when price is in the range?
			ob.AddOrder(types.NewUserOrder(order))
			if order.Status == types.OrderStatusMatching {
				order.SetStatus(types.OrderStatusNoMatch)
				k.SetOrder(ctx, order)
			}
		case types.OrderStatusCanceled:
		default:
			return false, fmt.Errorf("invalid order status: %s", order.Status)
		}
		return false, nil
	}); err != nil {
		return err
	}

	var pools []*types.PoolOrderer
	_ = k.IteratePoolsByPair(ctx, pair.Id, func(pool types.Pool) (stop bool, err error) {
		if pool.Disabled {
			return false, nil
		}
		rx, ry := k.getPoolBalances(ctx, pool, pair)
		ps := k.GetPoolCoinSupply(ctx, pool)
		ammPool := types.NewPoolOrderer(
			pool.AMMPool(rx.Amount, ry.Amount, ps),
			pool.Id, pool.GetReserveAddress(), pair.BaseDenom, pair.QuoteDenom)
		if ammPool.IsDepleted() {
			k.MarkPoolAsDisabled(ctx, pool)
			return false, nil
		}
		pools = append(pools, ammPool)
		return false, nil
	})

	matchPrice, quoteCoinDiff, matched := k.Match(ctx, ob, pools, pair.LastPrice)
	if matched {
		orders := ob.Orders()
		if err := k.ApplyMatchResult(ctx, pair, orders, quoteCoinDiff); err != nil {
			return err
		}
		pair.LastPrice = &matchPrice
	}

	pair.CurrentBatchId++
	k.SetPair(ctx, pair)

	return nil
}

func (k Keeper) Match(ctx sdk.Context, ob *amm.OrderBook, pools []*types.PoolOrderer, lastPrice *sdk.Dec) (matchPrice sdk.Dec, quoteCoinDiff sdk.Int, matched bool) {
	tickPrec := int(k.GetTickPrecision(ctx))
	if lastPrice == nil {
		ov := amm.MultipleOrderViews{ob.MakeView()}
		for _, pool := range pools {
			ov = append(ov, pool)
		}
		var found bool
		matchPrice, found = amm.FindMatchPrice(ov, tickPrec)
		if !found {
			return sdk.Dec{}, sdk.Int{}, false
		}
		for _, pool := range pools {
			buyAmt := pool.BuyAmountOver(matchPrice, true)
			if buyAmt.IsPositive() {
				ob.AddOrder(pool.Order(amm.Buy, matchPrice, buyAmt))
			}
			sellAmt := pool.SellAmountUnder(matchPrice, true)
			if sellAmt.IsPositive() {
				ob.AddOrder(pool.Order(amm.Sell, matchPrice, sellAmt))
			}
		}
		quoteCoinDiff, matched = ob.MatchAtSinglePrice(matchPrice)
	} else {
		lowestPrice, highestPrice := k.PriceLimits(ctx, *lastPrice)
		for _, pool := range pools {
			poolOrders := amm.PoolOrders(pool, pool, lowestPrice, highestPrice, tickPrec)
			ob.AddOrder(poolOrders...)
		}
		matchPrice, quoteCoinDiff, matched = ob.Match(*lastPrice)
	}
	return
}

func (k Keeper) ApplyMatchResult(ctx sdk.Context, pair types.Pair, orders []amm.Order, quoteCoinDiff sdk.Int) error {
	bulkOp := types.NewBulkSendCoinsOperation()
	for _, order := range orders { // TODO: need optimization to filter matched orders only
		order, ok := order.(*types.PoolOrder)
		if !ok {
			continue
		}
		if !order.IsMatched() {
			continue
		}
		paidCoin := sdk.NewCoin(order.OfferCoinDenom, order.PaidOfferCoinAmount)
		bulkOp.QueueSendCoins(order.ReserveAddress, pair.GetEscrowAddress(), sdk.NewCoins(paidCoin))
	}
	if err := bulkOp.Run(ctx, k.bankKeeper); err != nil {
		return err
	}
	bulkOp = types.NewBulkSendCoinsOperation()
	type PoolMatchResult struct {
		PoolId         uint64
		OrderDirection types.OrderDirection
		PaidCoin       sdk.Coin
		Received       sdk.Coin
		MatchedAmount  sdk.Int
	}
	poolMatchResultById := map[uint64]*PoolMatchResult{}
	var poolMatchResults []*PoolMatchResult
	for _, order := range orders {
		if !order.IsMatched() {
			continue
		}

		matchedAmt := order.GetAmount().Sub(order.GetOpenAmount())

		switch order := order.(type) {
		case *types.UserOrder:
			paidCoin := sdk.NewCoin(order.OfferCoinDenom, order.PaidOfferCoinAmount)
			receivedCoin := sdk.NewCoin(order.DemandCoinDenom, order.ReceivedDemandCoinAmount)

			o, _ := k.GetOrder(ctx, pair.Id, order.OrderId)
			o.OpenAmt = o.OpenAmt.Sub(matchedAmt)
			o.Remaining = o.Remaining.Sub(paidCoin)
			o.Received = o.Received.Add(receivedCoin)

			if o.OpenAmt.IsZero() {
				if err := k.FinishOrder(ctx, o, types.OrderStatusMatched); err != nil {
					return err
				}
			} else {
				o.SetStatus(types.OrderStatusPartialMatch)
				k.SetOrder(ctx, o)
			}
			bulkOp.QueueSendCoins(pair.GetEscrowAddress(), order.Orderer, sdk.NewCoins(receivedCoin))

			ctx.EventManager().EmitEvents(sdk.Events{
				sdk.NewEvent(
					types.EventTypeUserOrderMatched,
					sdk.NewAttribute(types.AttributeKeyOrderDirection, types.OrderDirectionFromAMM(order.Direction).String()),
					sdk.NewAttribute(types.AttributeKeyOrderer, order.Orderer.String()),
					sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(pair.Id, 10)),
					sdk.NewAttribute(types.AttributeKeyOrderId, strconv.FormatUint(order.OrderId, 10)),
					sdk.NewAttribute(types.AttributeKeyMatchedAmount, matchedAmt.String()),
					sdk.NewAttribute(types.AttributeKeyPaidCoin, paidCoin.String()),
					sdk.NewAttribute(types.AttributeKeyReceivedCoin, receivedCoin.String()),
				),
			})
		case *types.PoolOrder:
			paidCoin := sdk.NewCoin(order.OfferCoinDenom, order.PaidOfferCoinAmount)
			receivedCoin := sdk.NewCoin(order.DemandCoinDenom, order.ReceivedDemandCoinAmount)

			bulkOp.QueueSendCoins(pair.GetEscrowAddress(), order.ReserveAddress, sdk.NewCoins(receivedCoin))

			r, ok := poolMatchResultById[order.PoolId]
			if !ok {
				r = &PoolMatchResult{
					PoolId:         order.PoolId,
					OrderDirection: types.OrderDirectionFromAMM(order.Direction),
					PaidCoin:       sdk.NewCoin(paidCoin.Denom, sdk.ZeroInt()),
					Received:       sdk.NewCoin(receivedCoin.Denom, sdk.ZeroInt()),
					MatchedAmount:  sdk.ZeroInt(),
				}
				poolMatchResultById[order.PoolId] = r
				poolMatchResults = append(poolMatchResults, r)
			}
			dir := types.OrderDirectionFromAMM(order.Direction)
			if r.OrderDirection != dir {
				panic(fmt.Errorf("wrong order direction: %s != %s", dir, r.OrderDirection))
			}
			r.PaidCoin = r.PaidCoin.Add(paidCoin)
			r.Received = r.Received.Add(receivedCoin)
			r.MatchedAmount = r.MatchedAmount.Add(matchedAmt)
		default:
			panic(fmt.Errorf("invalid order type: %T", order))
		}
	}
	bulkOp.QueueSendCoins(pair.GetEscrowAddress(), k.GetDustCollector(ctx), sdk.NewCoins(sdk.NewCoin(pair.QuoteDenom, quoteCoinDiff)))
	if err := bulkOp.Run(ctx, k.bankKeeper); err != nil {
		return err
	}
	for _, r := range poolMatchResults {
		ctx.EventManager().EmitEvents(sdk.Events{
			sdk.NewEvent(
				types.EventTypePoolOrderMatched,
				sdk.NewAttribute(types.AttributeKeyOrderDirection, r.OrderDirection.String()),
				sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(pair.Id, 10)),
				sdk.NewAttribute(types.AttributeKeyPoolId, strconv.FormatUint(r.PoolId, 10)),
				sdk.NewAttribute(types.AttributeKeyMatchedAmount, r.MatchedAmount.String()),
				sdk.NewAttribute(types.AttributeKeyPaidCoin, r.PaidCoin.String()),
				sdk.NewAttribute(types.AttributeKeyReceivedCoin, r.Received.String()),
			),
		})
	}
	return nil
}

func (k Keeper) FinishOrder(ctx sdk.Context, order types.Order, status types.OrderStatus) error {
	if order.Status == types.OrderStatusMatched || order.Status.IsCanceledOrExpired() { // sanity check
		return nil
	}

	if order.Remaining.IsPositive() {
		pair, _ := k.GetPair(ctx, order.PairId)
		if err := k.bankKeeper.SendCoins(ctx, pair.GetEscrowAddress(), order.GetOrderer(), sdk.NewCoins(order.Remaining)); err != nil {
			return err
		}
	}

	order.SetStatus(status)
	k.SetOrder(ctx, order)

	ctx.EventManager().EmitEvents(sdk.Events{
		sdk.NewEvent(
			types.EventTypeOrderResult,
			sdk.NewAttribute(types.AttributeKeyOrderDirection, order.Direction.String()),
			sdk.NewAttribute(types.AttributeKeyOrderer, order.CreatorAddr),
			sdk.NewAttribute(types.AttributeKeyPairId, strconv.FormatUint(order.PairId, 10)),
			sdk.NewAttribute(types.AttributeKeyOrderId, strconv.FormatUint(order.Id, 10)),
			sdk.NewAttribute(types.AttributeKeyAmount, order.Amt.String()),
			sdk.NewAttribute(types.AttributeKeyOpenAmount, order.OpenAmt.String()),
			sdk.NewAttribute(types.AttributeKeyOfferCoin, order.Offer.String()),
			sdk.NewAttribute(types.AttributeKeyRemainingOfferCoin, order.Remaining.String()),
			sdk.NewAttribute(types.AttributeKeyReceivedCoin, order.Received.String()),
			sdk.NewAttribute(types.AttributeKeyStatus, order.Status.String()),
		),
	})

	return nil
}
