package types

import (
	"fmt"

	sdk "github.com/cosmos/cosmos-sdk/types"

	"ollo/x/liquidity/amm"
	"strings"
)

func IsOverflow(r interface{}) bool {
	switch r := r.(type) {
	case string:
		s := strings.ToLower(r)
		return strings.Contains(s, "overflow") || strings.HasSuffix(s, "out of bound")
	}
	return false
}

func SafeMath(f, onOverflow func()) {
	defer func() {
		if r := recover(); r != nil {
			if IsOverflow(r) {
				onOverflow()
			} else {
				panic(r)
			}
		}
	}()
	f()
}

// OrderDirectionFromAMM converts amm.OrderDirection to liquidity module's
// OrderDirection.
func OrderDirectionFromAMM(dir amm.OrderDirection) OrderDirection {
	switch dir {
	case amm.Buy:
		return OrderDirectionBuy
	case amm.Sell:
		return OrderDirectionSell
	default:
		panic(fmt.Errorf("invalid order direction: %s", dir))
	}
}

type UserOrder struct {
	*amm.BaseOrder
	Orderer                         sdk.AccAddress
	OrderId                         uint64
	BatchId                         uint64
	OfferCoinDenom, DemandCoinDenom string
}

// NewUserOrder returns a new user order.
func NewUserOrder(order Order) *UserOrder {
	var dir amm.OrderDirection
	var amt sdk.Int
	switch order.Direction {
	case OrderDirectionBuy:
		dir = amm.Buy
		SafeMath(func() {
			amt = sdk.MinInt(
				order.OpenAmt,
				order.Remaining.Amount.Quo(order.Price.TruncateInt()),
			)
		}, func() {
			amt = order.OpenAmt
		})
	case OrderDirectionSell:
		dir = amm.Sell
		amt = order.OpenAmt
	}
	return &UserOrder{
		BaseOrder:       amm.NewBaseOrder(dir, order.Price, amt, order.Remaining.Amount),
		Orderer:         order.GetOrderer(),
		OrderId:         order.Id,
		BatchId:         order.BatchId,
		OfferCoinDenom:  order.Offer.Denom,
		DemandCoinDenom: order.Received.Denom,
	}
}

func (order *UserOrder) GetBatchId() uint64 {
	return order.BatchId
}

func (order *UserOrder) HasPriority(other amm.Order) bool {
	if !order.Amount.Equal(other.GetAmount()) {
		return order.BaseOrder.HasPriority(other)
	}
	switch other := other.(type) {
	case *UserOrder:
		return order.OrderId < other.OrderId
	case *PoolOrder:
		return true
	default:
		panic(fmt.Errorf("invalid order type: %T", other))
	}
}

func (order *UserOrder) String() string {
	return fmt.Sprintf("UserOrder(%d,%d,%s,%s,%s)",
		order.OrderId, order.BatchId, order.Direction, order.Price, order.Amount)
}

type PoolOrder struct {
	*amm.BaseOrder
	PoolId                          uint64
	ReserveAddress                  sdk.AccAddress
	OfferCoinDenom, DemandCoinDenom string
}

func NewPoolOrder(
	poolId uint64, reserveAddr sdk.AccAddress, dir amm.OrderDirection, price sdk.Dec, amt sdk.Int,
	offerCoinDenom, demandCoinDenom string) *PoolOrder {
	return &PoolOrder{
		BaseOrder:       amm.NewBaseOrder(dir, price, amt, amm.OfferCoinAmount(dir, price, amt)),
		PoolId:          poolId,
		ReserveAddress:  reserveAddr,
		OfferCoinDenom:  offerCoinDenom,
		DemandCoinDenom: demandCoinDenom,
	}
}

func (order *Order) HasPriority(other amm.Order) bool {
	if !order.Amt.Equal(other.GetAmount()) {
		return order.HasPriority(other)
	}
	switch other := other.(type) {
	case *UserOrder:
		return false
	case *PoolOrder:
		return order.PairId < other.PoolId
	default:
		panic(fmt.Errorf("invalid order type: %T", other))
	}
}

func (order *PoolOrder) String() string {
	return fmt.Sprintf("PoolOrder(%d,%s,%s,%s)",
		order.PoolId, order.Direction, order.Price, order.Amount)
}
