import { Menu } from '@tauri-apps/api/menu';
import { TrayIcon, TrayIconOptions } from '@tauri-apps/api/tray';
import { defaultWindowIcon } from '@tauri-apps/api/app';
import { t } from '@/i18n';
import { getCurrentWebviewWindow } from '@tauri-apps/api/webviewWindow';
import { getCurrentWindow } from '@tauri-apps/api/window';
import { isProd } from '.';
import { openSettingsWindow } from './window';
import checkForUpdate from './updater';
import { message } from '@tauri-apps/plugin-dialog';
import { isFunction } from 'radash';

declare global {
  interface Window {
    __TRAY_INSTANCE?: TrayIcon;
  }
}

export async function createTrayMenu() {
  if (getCurrentWebviewWindow().label !== 'main') return;
  const menu = await Menu.new({
    items: [
      {
        id: 'open',
        text: t('tray.open'),
        action: async () => {
          await getCurrentWindow().show();
          await getCurrentWindow().setFocus();
        },
        accelerator: 'CmdOrCtrl+O',
      },
      {
        id: 'settings',
        text: t('tray.settings'),
        action: () => {
          openSettingsWindow();
        },
        accelerator: 'CmdOrCtrl+,',
      },
      {
        id: 'check_update',
        text: t('tray.check_update'),
        action: async () => {
          const updater = await checkForUpdate();
          if (!updater) {
            message(t('settings.about.version.no_update_available'), {
              title: t('tray.check_update'),
            });
          }
        },
        accelerator: 'CmdOrCtrl+U',
      },
      {
        id: 'quit',
        text: t('tray.quit'),
        action: () => {
          getCurrentWindow().destroy();
        },
        accelerator: 'CmdOrCtrl+Q',
      },
    ],
  });
  return menu;
}

export async function initTray() {
  if (getCurrentWebviewWindow().label !== 'main' || window.__TRAY_INSTANCE) return;

  const menu = await createTrayMenu();
  const icon = await defaultWindowIcon();
  const options: TrayIconOptions = {
    tooltip: 'PicSharp',
    icon,
    iconAsTemplate: true,
    menu,
    menuOnLeftClick: false,
    action: async (event) => {
      switch (event.type) {
        case 'Click':
          if (event.button === 'Right') return;
          await getCurrentWindow().show();
          await getCurrentWindow().setFocus();
          break;
      }
    },
  };

  window.__TRAY_INSTANCE = await TrayIcon.new(options);
}

export async function destroyTray() {
  if (isFunction(window.__TRAY_INSTANCE?.close)) {
    await window.__TRAY_INSTANCE.close();
    window.__TRAY_INSTANCE = null;
  }
}

if (isProd) {
  initTray();
}
