<?php

namespace ccxt;

// PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
// https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

use Exception; // a common import
use \ccxt\ExchangeError;
use \ccxt\AuthenticationError;
use \ccxt\AddressPending;
use \ccxt\InvalidOrder;
use \ccxt\OrderNotFound;
use \ccxt\DDoSProtection;

class bittrex extends Exchange {

    public function describe() {
        return $this->deep_extend(parent::describe (), array(
            'id' => 'bittrex',
            'name' => 'Bittrex',
            'countries' => array( 'US' ),
            'version' => 'v1.1',
            'rateLimit' => 1500,
            'certified' => true,
            'pro' => true,
            // new metainfo interface
            'has' => array(
                'cancelOrder' => true,
                'CORS' => false,
                'createMarketOrder' => true,
                'createOrder' => true,
                'fetchBalance' => true,
                'fetchDeposits' => true,
                'fetchDepositAddress' => true,
                'fetchClosedOrders' => true,
                'fetchCurrencies' => true,
                'fetchMarkets' => true,
                'fetchMyTrades' => 'emulated',
                'fetchOHLCV' => true,
                'fetchOrder' => true,
                'fetchOrderBook' => true,
                'fetchOpenOrders' => true,
                'fetchTicker' => true,
                'fetchTickers' => true,
                'fetchTime' => true,
                'fetchTrades' => true,
                'fetchTransactions' => false,
                'fetchWithdrawals' => true,
                'withdraw' => true,
            ),
            'timeframes' => array(
                '1m' => 'MINUTE_1',
                '5m' => 'MINUTE_5',
                '1h' => 'HOUR_1',
                '1d' => 'DAY_1',
            ),
            'hostname' => 'bittrex.com',
            'urls' => array(
                'logo' => 'https://user-images.githubusercontent.com/51840849/87153921-edf53180-c2c0-11ea-96b9-f2a9a95a455b.jpg',
                'api' => array(
                    'public' => 'https://{hostname}/api',
                    'account' => 'https://{hostname}/api',
                    'market' => 'https://{hostname}/api',
                    'v3' => 'https://api.bittrex.com/v3',
                    'v3public' => 'https://api.bittrex.com/v3',
                ),
                'www' => 'https://bittrex.com',
                'doc' => array(
                    'https://bittrex.github.io/api/',
                    'https://bittrex.github.io/api/v3',
                    'https://www.npmjs.com/package/bittrex-node',
                ),
                'fees' => array(
                    'https://bittrex.zendesk.com/hc/en-us/articles/115003684371-BITTREX-SERVICE-FEES-AND-WITHDRAWAL-LIMITATIONS',
                    'https://bittrex.zendesk.com/hc/en-us/articles/115000199651-What-fees-does-Bittrex-charge-',
                ),
                'referral' => 'https://bittrex.com/Account/Register?referralCode=1ZE-G0G-M3B',
            ),
            'api' => array(
                'v3' => array(
                    'get' => array(
                        'account',
                        'account/volume',
                        'addresses',
                        'addresses/{currencySymbol}',
                        'balances',
                        'balances/{currencySymbol}',
                        'currencies',
                        'currencies/{symbol}',
                        'deposits/open',
                        'deposits/closed',
                        'deposits/ByTxId/{txId}',
                        'deposits/{depositId}',
                        'orders/closed',
                        'orders/open',
                        'orders/{orderId}',
                        'orders/{orderId}/executions',
                        'ping',
                        'subaccounts/{subaccountId}',
                        'subaccounts',
                        'withdrawals/open',
                        'withdrawals/closed',
                        'withdrawals/ByTxId/{txId}',
                        'withdrawals/{withdrawalId}',
                        'withdrawals/whitelistAddresses',
                        'conditional-orders/{conditionalOrderId}',
                        'conditional-orders/closed',
                        'conditional-orders/open',
                        'transfers/sent',
                        'transfers/received',
                        'transfers/{transferId}',
                    ),
                    'post' => array(
                        'addresses',
                        'orders',
                        'subaccounts',
                        'withdrawals',
                        'conditional-orders',
                        'transfers',
                    ),
                    'delete' => array(
                        'orders/{orderId}',
                        'withdrawals/{withdrawalId}',
                        'conditional-orders/{conditionalOrderId}',
                    ),
                ),
                'v3public' => array(
                    'get' => array(
                        'markets',
                        'markets/summaries',
                        'markets/{marketSymbol}',
                        'markets/{marketSymbol}/summary',
                        'markets/{marketSymbol}/orderbook',
                        'markets/{marketSymbol}/trades',
                        'markets/{marketSymbol}/ticker',
                        'markets/{marketSymbol}/candles/{candleInterval}/recent',
                        'markets/{marketSymbol}/candles/{candleInterval}/historical/{year}/{month}/{day}',
                        'markets/{marketSymbol}/candles/{candleInterval}/historical/{year}/{month}',
                        'markets/{marketSymbol}/candles/{candleInterval}/historical/{year}',
                    ),
                ),
                'public' => array(
                    'get' => array(
                        'currencies',
                        'markethistory',
                        'markets',
                        'marketsummaries',
                        'marketsummary',
                        'orderbook',
                        'ticker',
                    ),
                ),
                'account' => array(
                    'get' => array(
                        'balance',
                        'balances',
                        'depositaddress',
                        'deposithistory',
                        'order',
                        'orders',
                        'orderhistory',
                        'withdrawalhistory',
                        'withdraw',
                    ),
                ),
                'market' => array(
                    'get' => array(
                        'buylimit',
                        'buymarket',
                        'cancel',
                        'openorders',
                        'selllimit',
                        'sellmarket',
                    ),
                ),
            ),
            'fees' => array(
                'trading' => array(
                    'tierBased' => false,
                    'percentage' => true,
                    'maker' => 0.0025,
                    'taker' => 0.0025,
                ),
                'funding' => array(
                    'tierBased' => false,
                    'percentage' => false,
                    'withdraw' => array(
                        'BTC' => 0.0005,
                        'LTC' => 0.01,
                        'DOGE' => 2,
                        'VTC' => 0.02,
                        'PPC' => 0.02,
                        'FTC' => 0.2,
                        'RDD' => 2,
                        'NXT' => 2,
                        'DASH' => 0.05,
                        'POT' => 0.002,
                        'BLK' => 0.02,
                        'EMC2' => 0.2,
                        'XMY' => 0.2,
                        'GLD' => 0.0002,
                        'SLR' => 0.2,
                        'GRS' => 0.2,
                    ),
                    'deposit' => array(
                        'BTC' => 0,
                        'LTC' => 0,
                        'DOGE' => 0,
                        'VTC' => 0,
                        'PPC' => 0,
                        'FTC' => 0,
                        'RDD' => 0,
                        'NXT' => 0,
                        'DASH' => 0,
                        'POT' => 0,
                        'BLK' => 0,
                        'EMC2' => 0,
                        'XMY' => 0,
                        'GLD' => 0,
                        'SLR' => 0,
                        'GRS' => 0,
                    ),
                ),
            ),
            'exceptions' => array(
                'exact' => array(
                    'BAD_REQUEST' => '\\ccxt\\BadRequest', // array("code":"BAD_REQUEST","detail":"Refer to the data field for specific field validation failures.","data":array("invalidRequestParameter":"day"))
                    'STARTDATE_OUT_OF_RANGE' => '\\ccxt\\BadRequest', // array("code":"STARTDATE_OUT_OF_RANGE")
                    // 'Call to Cancel was throttled. Try again in 60 seconds.' => '\\ccxt\\DDoSProtection',
                    // 'Call to GetBalances was throttled. Try again in 60 seconds.' => '\\ccxt\\DDoSProtection',
                    'APISIGN_NOT_PROVIDED' => '\\ccxt\\AuthenticationError',
                    'INVALID_SIGNATURE' => '\\ccxt\\AuthenticationError',
                    'INVALID_CURRENCY' => '\\ccxt\\ExchangeError',
                    'INVALID_PERMISSION' => '\\ccxt\\AuthenticationError',
                    'INSUFFICIENT_FUNDS' => '\\ccxt\\InsufficientFunds',
                    'INVALID_CEILING_MARKET_BUY' => '\\ccxt\\InvalidOrder',
                    'INVALID_FIAT_ACCOUNT' => '\\ccxt\\InvalidOrder',
                    'INVALID_ORDER_TYPE' => '\\ccxt\\InvalidOrder',
                    'QUANTITY_NOT_PROVIDED' => '\\ccxt\\InvalidOrder',
                    'MIN_TRADE_REQUIREMENT_NOT_MET' => '\\ccxt\\InvalidOrder',
                    'ORDER_NOT_OPEN' => '\\ccxt\\OrderNotFound',
                    'INVALID_ORDER' => '\\ccxt\\InvalidOrder',
                    'UUID_INVALID' => '\\ccxt\\OrderNotFound',
                    'RATE_NOT_PROVIDED' => '\\ccxt\\InvalidOrder', // createLimitBuyOrder ('ETH/BTC', 1, 0)
                    'INVALID_MARKET' => '\\ccxt\\BadSymbol', // array("success":false,"message":"INVALID_MARKET","result":null,"explanation":null)
                    'WHITELIST_VIOLATION_IP' => '\\ccxt\\PermissionDenied',
                    'DUST_TRADE_DISALLOWED_MIN_VALUE' => '\\ccxt\\InvalidOrder',
                    'RESTRICTED_MARKET' => '\\ccxt\\BadSymbol',
                    'We are down for scheduled maintenance, but we\u2019ll be back up shortly.' => '\\ccxt\\OnMaintenance', // array("success":false,"message":"We are down for scheduled maintenance, but we\u2019ll be back up shortly.","result":null,"explanation":null)
                ),
                'broad' => array(
                    'throttled' => '\\ccxt\\DDoSProtection',
                    'problem' => '\\ccxt\\ExchangeNotAvailable',
                ),
            ),
            'options' => array(
                'parseOrderStatus' => false,
                'hasAlreadyAuthenticatedSuccessfully' => false, // a workaround for APIKEY_INVALID
                'symbolSeparator' => '-',
                // With certain currencies, like
                // AEON, BTS, GXS, NXT, SBD, STEEM, STR, XEM, XLM, XMR, XRP
                // an additional tag / memo / payment id is usually required by exchanges.
                // With Bittrex some currencies imply the "base address . tag" logic.
                // The base address for depositing is stored on $this->currencies[code]
                // The base address identifies the exchange as the recipient
                // while the tag identifies the user account within the exchange
                // and the tag is retrieved with fetchDepositAddress.
                'tag' => array(
                    'NXT' => true, // NXT, BURST
                    'CRYPTO_NOTE_PAYMENTID' => true, // AEON, XMR
                    'BITSHAREX' => true, // BTS
                    'RIPPLE' => true, // XRP
                    'NEM' => true, // XEM
                    'STELLAR' => true, // XLM
                    'STEEM' => true, // SBD, GOLOS
                    // https://github.com/ccxt/ccxt/issues/4794
                    // 'LISK' => true, // LSK
                ),
                'subaccountId' => null,
                // see the implementation of fetchClosedOrdersV3 below
                'fetchClosedOrdersMethod' => 'fetch_closed_orders_v3',
                'fetchClosedOrdersFilterBySince' => true,
                // 'createOrderMethod' => 'create_order_v1',
            ),
            'commonCurrencies' => array(
                'BITS' => 'SWIFT',
            ),
        ));
    }

    public function cost_to_precision($symbol, $cost) {
        return $this->decimal_to_precision($cost, TRUNCATE, $this->markets[$symbol]['precision']['price'], DECIMAL_PLACES);
    }

    public function fee_to_precision($symbol, $fee) {
        return $this->decimal_to_precision($fee, TRUNCATE, $this->markets[$symbol]['precision']['price'], DECIMAL_PLACES);
    }

    public function fetch_markets($params = array ()) {
        $response = $this->v3publicGetMarkets ($params);
        //
        //     array(
        //         array(
        //             "$symbol":"LTC-BTC",
        //             "baseCurrencySymbol":"LTC",
        //             "quoteCurrencySymbol":"BTC",
        //             "minTradeSize":"0.01686767",
        //             "$precision":8,
        //             "$status":"ONLINE", // "OFFLINE"
        //             "createdAt":"2014-02-13T00:00:00Z"
        //         ),
        //         {
        //             "$symbol":"VDX-USDT",
        //             "baseCurrencySymbol":"VDX",
        //             "quoteCurrencySymbol":"USDT",
        //             "minTradeSize":"300.00000000",
        //             "$precision":8,
        //             "$status":"ONLINE", // "OFFLINE"
        //             "createdAt":"2019-05-23T00:41:21.843Z",
        //             "notice":"USDT has swapped to an ERC20-based token as of August 5, 2019."
        //         }
        //     )
        //
        $result = array();
        // $markets = $this->safe_value($response, 'result');
        for ($i = 0; $i < count($response); $i++) {
            $market = $response[$i];
            $baseId = $this->safe_string($market, 'baseCurrencySymbol');
            $quoteId = $this->safe_string($market, 'quoteCurrencySymbol');
            // bittrex v1 uses inverted pairs, v3 uses regular pairs
            // we use v3 for fetchMarkets and v1 throughout the rest of this implementation
            // therefore we swap the $base ←→ $quote here to be v1-compatible
            // https://github.com/ccxt/ccxt/issues/5634
            // $id = $this->safe_string($market, 'symbol');
            $id = $quoteId . $this->options['symbolSeparator'] . $baseId;
            $base = $this->safe_currency_code($baseId);
            $quote = $this->safe_currency_code($quoteId);
            $symbol = $base . '/' . $quote;
            $pricePrecision = $this->safe_integer($market, 'precision', 8);
            $precision = array(
                'amount' => 8,
                'price' => $pricePrecision,
            );
            $status = $this->safe_string($market, 'status');
            $active = ($status === 'ONLINE');
            $result[] = array(
                'id' => $id,
                'symbol' => $symbol,
                'base' => $base,
                'quote' => $quote,
                'baseId' => $baseId,
                'quoteId' => $quoteId,
                'active' => $active,
                'info' => $market,
                'precision' => $precision,
                'limits' => array(
                    'amount' => array(
                        'min' => $this->safe_float($market, 'minTradeSize'),
                        'max' => null,
                    ),
                    'price' => array(
                        'min' => pow(10, -$precision['price']),
                        'max' => null,
                    ),
                ),
            );
        }
        return $result;
    }

    public function fetch_balance($params = array ()) {
        $this->load_markets();
        $response = $this->accountGetBalances ($params);
        $balances = $this->safe_value($response, 'result');
        $result = array( 'info' => $balances );
        $indexed = $this->index_by($balances, 'Currency');
        $currencyIds = is_array($indexed) ? array_keys($indexed) : array();
        for ($i = 0; $i < count($currencyIds); $i++) {
            $currencyId = $currencyIds[$i];
            $code = $this->safe_currency_code($currencyId);
            $account = $this->account();
            $balance = $indexed[$currencyId];
            $account['free'] = $this->safe_float($balance, 'Available');
            $account['total'] = $this->safe_float($balance, 'Balance');
            $result[$code] = $account;
        }
        return $this->parse_balance($result);
    }

    public function fetch_order_book($symbol, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array(
            'market' => $this->market_id($symbol),
            'type' => 'both',
        );
        $response = $this->publicGetOrderbook (array_merge($request, $params));
        $orderbook = $response['result'];
        if (is_array($params) && array_key_exists('type', $params)) {
            if ($params['type'] === 'buy') {
                $orderbook = array(
                    'buy' => $response['result'],
                    'sell' => array(),
                );
            } else if ($params['type'] === 'sell') {
                $orderbook = array(
                    'buy' => array(),
                    'sell' => $response['result'],
                );
            }
        }
        return $this->parse_order_book($orderbook, null, 'buy', 'sell', 'Rate', 'Quantity');
    }

    public function fetch_currencies($params = array ()) {
        $response = $this->publicGetCurrencies ($params);
        //
        //     {
        //         "success" => true,
        //         "message" => "",
        //         "$result" => array(
        //             array(
        //                 "Currency" => "BTC",
        //                 "CurrencyLong":"Bitcoin",
        //                 "MinConfirmation":2,
        //                 "TxFee":0.00050000,
        //                 "IsActive":true,
        //                 "IsRestricted":false,
        //                 "CoinType":"BITCOIN",
        //                 "BaseAddress":"1N52wHoVR79PMDishab2XmRHsbekCdGquK",
        //                 "Notice":null
        //             ),
        //             ...,
        //         )
        //     }
        //
        $currencies = $this->safe_value($response, 'result', array());
        $result = array();
        for ($i = 0; $i < count($currencies); $i++) {
            $currency = $currencies[$i];
            $id = $this->safe_string($currency, 'Currency');
            $code = $this->safe_currency_code($id);
            $precision = 8; // default $precision, todo => fix "magic constants"
            $address = $this->safe_value($currency, 'BaseAddress');
            $fee = $this->safe_float($currency, 'TxFee'); // todo => redesign
            $result[$code] = array(
                'id' => $id,
                'code' => $code,
                'address' => $address,
                'info' => $currency,
                'type' => $this->safe_string($currency, 'CoinType'),
                'name' => $this->safe_string($currency, 'CurrencyLong'),
                'active' => $this->safe_value($currency, 'IsActive'),
                'fee' => $fee,
                'precision' => $precision,
                'limits' => array(
                    'amount' => array(
                        'min' => pow(10, -$precision),
                        'max' => null,
                    ),
                    'price' => array(
                        'min' => pow(10, -$precision),
                        'max' => null,
                    ),
                    'cost' => array(
                        'min' => null,
                        'max' => null,
                    ),
                    'withdraw' => array(
                        'min' => $fee,
                        'max' => null,
                    ),
                ),
            );
        }
        return $result;
    }

    public function parse_ticker($ticker, $market = null) {
        //
        //     {
        //         "MarketName":"BTC-ETH",
        //         "High":0.02127099,
        //         "Low":0.02035064,
        //         "Volume":10288.40271571,
        //         "Last":0.02070510,
        //         "BaseVolume":214.64663206,
        //         "TimeStamp":"2019-09-18T21:03:59.897",
        //         "Bid":0.02070509,
        //         "Ask":0.02070510,
        //         "OpenBuyOrders":1228,
        //         "OpenSellOrders":5899,
        //         "PrevDay":0.02082823,
        //         "Created":"2015-08-14T09:02:24.817"
        //     }
        //
        $timestamp = $this->parse8601($this->safe_string($ticker, 'TimeStamp'));
        $symbol = null;
        $marketId = $this->safe_string($ticker, 'MarketName');
        if ($marketId !== null) {
            if (is_array($this->markets_by_id) && array_key_exists($marketId, $this->markets_by_id)) {
                $market = $this->markets_by_id[$marketId];
            } else {
                $symbol = $this->parse_symbol($marketId);
            }
        }
        if (($symbol === null) && ($market !== null)) {
            $symbol = $market['symbol'];
        }
        $previous = $this->safe_float($ticker, 'PrevDay');
        $last = $this->safe_float($ticker, 'Last');
        $change = null;
        $percentage = null;
        if ($last !== null) {
            if ($previous !== null) {
                $change = $last - $previous;
                if ($previous > 0) {
                    $percentage = ($change / $previous) * 100;
                }
            }
        }
        return array(
            'symbol' => $symbol,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'high' => $this->safe_float($ticker, 'High'),
            'low' => $this->safe_float($ticker, 'Low'),
            'bid' => $this->safe_float($ticker, 'Bid'),
            'bidVolume' => null,
            'ask' => $this->safe_float($ticker, 'Ask'),
            'askVolume' => null,
            'vwap' => null,
            'open' => $previous,
            'close' => $last,
            'last' => $last,
            'previousClose' => null,
            'change' => $change,
            'percentage' => $percentage,
            'average' => null,
            'baseVolume' => $this->safe_float($ticker, 'Volume'),
            'quoteVolume' => $this->safe_float($ticker, 'BaseVolume'),
            'info' => $ticker,
        );
    }

    public function fetch_tickers($symbols = null, $params = array ()) {
        $this->load_markets();
        $response = $this->publicGetMarketsummaries ($params);
        $result = $this->safe_value($response, 'result');
        $tickers = array();
        for ($i = 0; $i < count($result); $i++) {
            $ticker = $this->parse_ticker($result[$i]);
            $tickers[] = $ticker;
        }
        return $this->filter_by_array($tickers, 'symbol', $symbols);
    }

    public function fetch_ticker($symbol, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market' => $market['id'],
        );
        $response = $this->publicGetMarketsummary (array_merge($request, $params));
        //
        //     {
        //         "success":true,
        //         "message":"",
        //         "result":array(
        //             {
        //                 "MarketName":"BTC-ETH",
        //                 "High":0.02127099,
        //                 "Low":0.02035064,
        //                 "Volume":10288.40271571,
        //                 "Last":0.02070510,
        //                 "BaseVolume":214.64663206,
        //                 "TimeStamp":"2019-09-18T21:03:59.897",
        //                 "Bid":0.02070509,
        //                 "Ask":0.02070510,
        //                 "OpenBuyOrders":1228,
        //                 "OpenSellOrders":5899,
        //                 "PrevDay":0.02082823,
        //                 "Created":"2015-08-14T09:02:24.817"
        //             }
        //         )
        //     }
        //
        $ticker = $response['result'][0];
        return $this->parse_ticker($ticker, $market);
    }

    public function parse_trade($trade, $market = null) {
        $timestamp = $this->parse8601($trade['TimeStamp'] . '+00:00');
        $side = null;
        if ($trade['OrderType'] === 'BUY') {
            $side = 'buy';
        } else if ($trade['OrderType'] === 'SELL') {
            $side = 'sell';
        }
        $id = $this->safe_string_2($trade, 'Id', 'ID');
        $symbol = null;
        if ($market !== null) {
            $symbol = $market['symbol'];
        }
        $cost = null;
        $price = $this->safe_float($trade, 'Price');
        $amount = $this->safe_float($trade, 'Quantity');
        if ($amount !== null) {
            if ($price !== null) {
                $cost = $price * $amount;
            }
        }
        return array(
            'info' => $trade,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'symbol' => $symbol,
            'id' => $id,
            'order' => null,
            'type' => 'limit',
            'takerOrMaker' => null,
            'side' => $side,
            'price' => $price,
            'amount' => $amount,
            'cost' => $cost,
            'fee' => null,
        );
    }

    public function fetch_time($params = array ()) {
        $response = $this->v3GetPing ($params);
        //
        //     {
        //         "serverTime" => 1594596023162
        //     }
        //
        return $this->safe_integer($response, 'serverTime');
    }

    public function fetch_trades($symbol, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market' => $market['id'],
        );
        $response = $this->publicGetMarkethistory (array_merge($request, $params));
        if (is_array($response) && array_key_exists('result', $response)) {
            if ($response['result'] !== null) {
                return $this->parse_trades($response['result'], $market, $since, $limit);
            }
        }
        throw new ExchangeError($this->id . ' fetchTrades() returned null response');
    }

    public function parse_ohlcv($ohlcv, $market = null) {
        //
        //     {
        //         "startsAt":"2020-06-12T02:35:00Z",
        //         "open":"0.02493753",
        //         "high":"0.02493753",
        //         "low":"0.02493753",
        //         "close":"0.02493753",
        //         "volume":"0.09590123",
        //         "quoteVolume":"0.00239153"
        //     }
        //
        return array(
            $this->parse8601($this->safe_string($ohlcv, 'startsAt')),
            $this->safe_float($ohlcv, 'open'),
            $this->safe_float($ohlcv, 'high'),
            $this->safe_float($ohlcv, 'low'),
            $this->safe_float($ohlcv, 'close'),
            $this->safe_float($ohlcv, 'volume'),
        );
    }

    public function fetch_ohlcv($symbol, $timeframe = '1m', $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $reverseId = $market['baseId'] . '-' . $market['quoteId'];
        $request = array(
            'candleInterval' => $this->timeframes[$timeframe],
            'marketSymbol' => $reverseId,
        );
        $method = 'v3publicGetMarketsMarketSymbolCandlesCandleIntervalRecent';
        if ($since !== null) {
            $now = $this->milliseconds();
            $difference = abs($now - $since);
            $sinceDate = $this->ymd($since);
            $parts = explode('-', $sinceDate);
            $sinceYear = $this->safe_integer($parts, 0);
            $sinceMonth = $this->safe_integer($parts, 1);
            $sinceDay = $this->safe_integer($parts, 2);
            if ($timeframe === '1d') {
                // if the $since argument is beyond one year into the past
                if ($difference > 31622400000) {
                    $method = 'v3publicGetMarketsMarketSymbolCandlesCandleIntervalHistoricalYear';
                    $request['year'] = $sinceYear;
                }
                // $request['year'] = year;
            } else if ($timeframe === '1h') {
                // if the $since argument is beyond 31 days into the past
                if ($difference > 2678400000) {
                    $method = 'v3publicGetMarketsMarketSymbolCandlesCandleIntervalHistoricalYearMonth';
                    $request['year'] = $sinceYear;
                    $request['month'] = $sinceMonth;
                }
            } else {
                // if the $since argument is beyond 1 day into the past
                if ($difference > 86400000) {
                    $method = 'v3publicGetMarketsMarketSymbolCandlesCandleIntervalHistoricalYearMonthDay';
                    $request['year'] = $sinceYear;
                    $request['month'] = $sinceMonth;
                    $request['day'] = $sinceDay;
                }
            }
        }
        $response = $this->$method (array_merge($request, $params));
        //
        //     array(
        //         array("startsAt":"2020-06-12T02:35:00Z","open":"0.02493753","high":"0.02493753","low":"0.02493753","close":"0.02493753","volume":"0.09590123","quoteVolume":"0.00239153"),
        //         array("startsAt":"2020-06-12T02:40:00Z","open":"0.02491874","high":"0.02491874","low":"0.02490970","close":"0.02490970","volume":"0.04515695","quoteVolume":"0.00112505"),
        //         array("startsAt":"2020-06-12T02:45:00Z","open":"0.02490753","high":"0.02493143","low":"0.02490753","close":"0.02493143","volume":"0.17769640","quoteVolume":"0.00442663")
        //     )
        //
        return $this->parse_ohlcvs($response, $market, $timeframe, $since, $limit);
    }

    public function fetch_open_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market'] = $market['id'];
        }
        $response = $this->marketGetOpenorders (array_merge($request, $params));
        $result = $this->safe_value($response, 'result', array());
        $orders = $this->parse_orders($result, $market, $since, $limit);
        return $this->filter_by_symbol($orders, $symbol);
    }

    public function create_order($symbol, $type, $side, $amount, $price = null, $params = array ()) {
        $uppercaseType = strtoupper($type);
        $isMarket = ($uppercaseType === 'MARKET');
        $isCeilingLimit = ($uppercaseType === 'CEILING_LIMIT');
        $isCeilingMarket = ($uppercaseType === 'CEILING_MARKET');
        $isV3 = $isMarket || $isCeilingLimit || $isCeilingMarket;
        $defaultMethod = $isV3 ? 'create_order_v3' : 'create_order_v1';
        $method = $this->safe_value($this->options, 'createOrderMethod', $defaultMethod);
        return $this->$method ($symbol, $type, $side, $amount, $price, $params);
    }

    public function create_order_v3($symbol, $type, $side, $amount, $price = null, $params = array ()) {
        // A ceiling order is a $market or limit order that allows you to specify
        // the $amount of quote currency you want to spend (or receive, if selling)
        // instead of the quantity of the $market currency (e.g. buy $100 USD of BTC
        // at the current $market BTC $price)
        $this->load_markets();
        $market = $this->market($symbol);
        $uppercaseType = strtoupper($type);
        $reverseId = $market['baseId'] . '-' . $market['quoteId'];
        $request = array(
            'marketSymbol' => $reverseId,
            'direction' => strtoupper($side),
            'type' => $uppercaseType, // LIMIT, MARKET, CEILING_LIMIT, CEILING_MARKET
            // 'quantity' => $this->amount_to_precision($symbol, $amount), // required for limit orders, excluded for ceiling orders
            // 'ceiling' => $this->price_to_precision($symbol, $price), // required for ceiling orders, excluded for non-ceiling orders
            // 'limit' => $this->price_to_precision($symbol, $price), // required for limit orders, excluded for $market orders
            // 'timeInForce' => 'GOOD_TIL_CANCELLED', // IMMEDIATE_OR_CANCEL, FILL_OR_KILL, POST_ONLY_GOOD_TIL_CANCELLED
            // 'useAwards' => false, // optional
        );
        $isCeilingLimit = ($uppercaseType === 'CEILING_LIMIT');
        $isCeilingMarket = ($uppercaseType === 'CEILING_MARKET');
        $isCeilingOrder = $isCeilingLimit || $isCeilingMarket;
        if ($isCeilingOrder) {
            $request['ceiling'] = $this->price_to_precision($symbol, $price);
            // bittrex only accepts IMMEDIATE_OR_CANCEL or FILL_OR_KILL for ceiling orders
            $request['timeInForce'] = 'IMMEDIATE_OR_CANCEL';
        } else {
            $request['quantity'] = $this->amount_to_precision($symbol, $amount);
            if ($uppercaseType === 'LIMIT') {
                $request['limit'] = $this->price_to_precision($symbol, $price);
                $request['timeInForce'] = 'GOOD_TIL_CANCELLED';
            } else {
                // bittrex does not allow GOOD_TIL_CANCELLED for $market orders
                $request['timeInForce'] = 'IMMEDIATE_OR_CANCEL';
            }
        }
        $response = $this->v3PostOrders (array_merge($request, $params));
        //
        //     {
        //         id => 'f03d5e98-b5ac-48fb-8647-dd4db828a297',
        //         marketSymbol => 'BTC-USDT',
        //         direction => 'SELL',
        //         $type => 'LIMIT',
        //         quantity => '0.01',
        //         limit => '6000',
        //         timeInForce => 'GOOD_TIL_CANCELLED',
        //         fillQuantity => '0.00000000',
        //         commission => '0.00000000',
        //         proceeds => '0.00000000',
        //         status => 'OPEN',
        //         createdAt => '2020-03-18T02:37:33.42Z',
        //         updatedAt => '2020-03-18T02:37:33.42Z'
        //       }
        //
        return $this->parse_order_v3($response, $market);
    }

    public function create_order_v1($symbol, $type, $side, $amount, $price = null, $params = array ()) {
        if ($type !== 'limit') {
            throw new ExchangeError($this->id . ' allows limit orders only');
        }
        $this->load_markets();
        $market = $this->market($symbol);
        $method = 'marketGet' . $this->capitalize($side) . $type;
        $request = array(
            'market' => $market['id'],
            'quantity' => $this->amount_to_precision($symbol, $amount),
            'rate' => $this->price_to_precision($symbol, $price),
        );
        // if ($type == 'limit')
        //     order['rate'] = $this->price_to_precision($symbol, $price);
        $response = $this->$method (array_merge($request, $params));
        $orderIdField = $this->get_order_id_field();
        $orderId = $this->safe_string($response['result'], $orderIdField);
        return array(
            'info' => $response,
            'id' => $orderId,
            'symbol' => $symbol,
            'type' => $type,
            'side' => $side,
            'status' => 'open',
        );
    }

    public function get_order_id_field() {
        return 'uuid';
    }

    public function cancel_order($id, $symbol = null, $params = array ()) {
        $this->load_markets();
        $orderIdField = $this->get_order_id_field();
        $request = array();
        $request[$orderIdField] = $id;
        $response = $this->marketGetCancel (array_merge($request, $params));
        //
        //     {
        //         "success" => true,
        //         "message" => "''",
        //         "$result" => {
        //             "uuid" => "614c34e4-8d71-11e3-94b5-425861b86ab6"
        //         }
        //     }
        //
        $result = $this->safe_value($response, 'result', array());
        return array_merge($this->parse_order($result), array(
            'id' => $id,
            'info' => $response,
            'status' => 'canceled',
        ));
    }

    public function fetch_deposits($code = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        // https://support.bittrex.com/hc/en-us/articles/115003723911
        $request = array();
        $currency = null;
        if ($code !== null) {
            $currency = $this->currency($code);
            $request['currency'] = $currency['id'];
        }
        $response = $this->accountGetDeposithistory (array_merge($request, $params));
        //
        //     { success =>    true,
        //       message =>   "",
        //        $result => array( {            Id =>  22578097,
        //                           Amount =>  0.3,
        //                         Currency => "ETH",
        //                    Confirmations =>  15,
        //                      LastUpdated => "2018-06-10T07:12:10.57",
        //                             TxId => "0xf50b5ba2ca5438b58f93516eaa523eaf35b4420ca0f24061003df1be7…",
        //                    CryptoAddress => "0xb25f281fa51f1635abd4a60b0870a62d2a7fa404"                    } ) }
        //
        // we cannot filter by `$since` timestamp, as it isn't set by Bittrex
        // see https://github.com/ccxt/ccxt/issues/4067
        $result = $this->safe_value($response, 'result', array());
        // return $this->parse_transactions($result, $currency, $since, $limit);
        return $this->parse_transactions($result, $currency, null, $limit);
    }

    public function fetch_withdrawals($code = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        // https://support.bittrex.com/hc/en-us/articles/115003723911
        $request = array();
        $currency = null;
        if ($code !== null) {
            $currency = $this->currency($code);
            $request['currency'] = $currency['id'];
        }
        $response = $this->accountGetWithdrawalhistory (array_merge($request, $params));
        //
        //     {
        //         "success" : true,
        //         "message" : "",
        //         "result" : [array(
        //                 "PaymentUuid" : "b32c7a5c-90c6-4c6e-835c-e16df12708b1",
        //                 "Currency" : "BTC",
        //                 "Amount" : 17.00000000,
        //                 "Address" : "1DfaaFBdbB5nrHj87x3NHS4onvw1GPNyAu",
        //                 "Opened" : "2014-07-09T04:24:47.217",
        //                 "Authorized" : true,
        //                 "PendingPayment" : false,
        //                 "TxCost" : 0.00020000,
        //                 "TxId" : null,
        //                 "Canceled" : true,
        //                 "InvalidAddress" : false
        //             ), {
        //                 "PaymentUuid" : "d193da98-788c-4188-a8f9-8ec2c33fdfcf",
        //                 "Currency" : "XC",
        //                 "Amount" : 7513.75121715,
        //                 "Address" : "TcnSMgAd7EonF2Dgc4c9K14L12RBaW5S5J",
        //                 "Opened" : "2014-07-08T23:13:31.83",
        //                 "Authorized" : true,
        //                 "PendingPayment" : false,
        //                 "TxCost" : 0.00002000,
        //                 "TxId" : "d8a575c2a71c7e56d02ab8e26bb1ef0a2f6cf2094f6ca2116476a569c1e84f6e",
        //                 "Canceled" : false,
        //                 "InvalidAddress" : false
        //             }
        //         ]
        //     }
        //
        return $this->parse_transactions($response['result'], $currency, $since, $limit);
    }

    public function parse_transaction($transaction, $currency = null) {
        //
        // fetchDeposits
        //
        //     {
        //         Id =>  72578097,
        //         Amount =>  0.3,
        //         Currency => "ETH",
        //         Confirmations =>  15,
        //         LastUpdated => "2018-06-17T07:12:14.57",
        //         TxId => "0xb31b5ba2ca5438b58f93516eaa523eaf35b4420ca0f24061003df1be7…",
        //         CryptoAddress => "0x2d5f281fa51f1635abd4a60b0870a62d2a7fa404"
        //     }
        //
        // fetchWithdrawals
        //
        //     {
        //         "PaymentUuid" : "e293da98-788c-4188-a8f9-8ec2c33fdfcf",
        //         "Currency" : "XC",
        //         "Amount" : 7513.75121715,
        //         "Address" : "EVnSMgAd7EonF2Dgc4c9K14L12RBaW5S5J",
        //         "Opened" : "2014-07-08T23:13:31.83",
        //         "Authorized" : true,
        //         "PendingPayment" : false,
        //         "TxCost" : 0.00002000,
        //         "TxId" : "b4a575c2a71c7e56d02ab8e26bb1ef0a2f6cf2094f6ca2116476a569c1e84f6e",
        //         "Canceled" : false,
        //         "InvalidAddress" : false
        //     }
        //
        $id = $this->safe_string_2($transaction, 'Id', 'PaymentUuid');
        $amount = $this->safe_float($transaction, 'Amount');
        $address = $this->safe_string_2($transaction, 'CryptoAddress', 'Address');
        $txid = $this->safe_string($transaction, 'TxId');
        $updated = $this->parse8601($this->safe_string($transaction, 'LastUpdated'));
        $opened = $this->parse8601($this->safe_string($transaction, 'Opened'));
        $timestamp = $opened ? $opened : $updated;
        $type = ($opened === null) ? 'deposit' : 'withdrawal';
        $currencyId = $this->safe_string($transaction, 'Currency');
        $code = $this->safe_currency_code($currencyId, $currency);
        $status = 'pending';
        if ($type === 'deposit') {
            //
            // deposits $numConfirmations never reach the $minConfirmations number
            // we set all of them to 'ok', otherwise they'd all be 'pending'
            //
            //     $numConfirmations = $this->safe_integer($transaction, 'Confirmations', 0);
            //     $minConfirmations = $this->safe_integer($currency['info'], 'MinConfirmation');
            //     if ($numConfirmations >= $minConfirmations) {
            //         $status = 'ok';
            //     }
            //
            $status = 'ok';
        } else {
            $authorized = $this->safe_value($transaction, 'Authorized', false);
            $pendingPayment = $this->safe_value($transaction, 'PendingPayment', false);
            $canceled = $this->safe_value($transaction, 'Canceled', false);
            $invalidAddress = $this->safe_value($transaction, 'InvalidAddress', false);
            if ($invalidAddress) {
                $status = 'failed';
            } else if ($canceled) {
                $status = 'canceled';
            } else if ($pendingPayment) {
                $status = 'pending';
            } else if ($authorized && ($txid !== null)) {
                $status = 'ok';
            }
        }
        $feeCost = $this->safe_float($transaction, 'TxCost');
        if ($feeCost === null) {
            if ($type === 'deposit') {
                // according to https://support.bittrex.com/hc/en-us/articles/115000199651-What-fees-does-Bittrex-charge-
                $feeCost = 0;
            }
        }
        return array(
            'info' => $transaction,
            'id' => $id,
            'currency' => $code,
            'amount' => $amount,
            'address' => $address,
            'tag' => null,
            'status' => $status,
            'type' => $type,
            'updated' => $updated,
            'txid' => $txid,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'fee' => array(
                'currency' => $code,
                'cost' => $feeCost,
            ),
        );
    }

    public function parse_symbol($id) {
        list($quoteId, $baseId) = explode($this->options['symbolSeparator'], $id);
        $base = $this->safe_currency_code($baseId);
        $quote = $this->safe_currency_code($quoteId);
        return $base . '/' . $quote;
    }

    public function parse_order($order, $market = null) {
        if (is_array($order) && array_key_exists('marketSymbol', $order)) {
            return $this->parse_order_v3($order, $market);
        } else {
            return $this->parse_order_v2($order, $market);
        }
    }

    public function parse_orders($orders, $market = null, $since = null, $limit = null, $params = array ()) {
        if ($this->options['fetchClosedOrdersFilterBySince']) {
            return parent::parse_orders($orders, $market, $since, $limit, $params);
        } else {
            return parent::parse_orders($orders, $market, null, $limit, $params);
        }
    }

    public function parse_order_status($status) {
        $statuses = array(
            'CLOSED' => 'closed',
            'OPEN' => 'open',
            'CANCELLED' => 'canceled',
            'CANCELED' => 'canceled',
        );
        return $this->safe_string($statuses, $status, $status);
    }

    public function parse_order_v3($order, $market = null) {
        //
        //     {
        //         id => '1be35109-b763-44ce-b6ea-05b6b0735c0c',
        //         $marketSymbol => 'LTC-ETH',
        //         $direction => 'BUY',
        //         $type => 'LIMIT',
        //         $quantity => '0.50000000',
        //         $limit => '0.17846699',
        //         timeInForce => 'GOOD_TIL_CANCELLED',
        //         $fillQuantity => '0.50000000',
        //         $commission => '0.00022286',
        //         $proceeds => '0.08914915',
        //         $status => 'CLOSED',
        //         $createdAt => '2018-06-23T13:14:28.613Z',
        //         $updatedAt => '2018-06-23T13:14:30.19Z',
        //         $closedAt => '2018-06-23T13:14:30.19Z'
        //     }
        //
        $marketSymbol = $this->safe_string($order, 'marketSymbol');
        $symbol = null;
        $feeCurrency = null;
        if ($marketSymbol !== null) {
            list($baseId, $quoteId) = explode('-', $marketSymbol);
            $base = $this->safe_currency_code($baseId);
            $quote = $this->safe_currency_code($quoteId);
            $symbol = $base . '/' . $quote;
            $feeCurrency = $quote;
        }
        $direction = $this->safe_string_lower($order, 'direction');
        $createdAt = $this->safe_string($order, 'createdAt');
        $updatedAt = $this->safe_string($order, 'updatedAt');
        $closedAt = $this->safe_string($order, 'closedAt');
        $lastTradeTimestamp = null;
        if ($closedAt !== null) {
            $lastTradeTimestamp = $this->parse8601($closedAt);
        } else if ($updatedAt) {
            $lastTradeTimestamp = $this->parse8601($updatedAt);
        }
        $timestamp = $this->parse8601($createdAt);
        $type = $this->safe_string_lower($order, 'type');
        $quantity = $this->safe_float($order, 'quantity');
        $limit = $this->safe_float($order, 'limit');
        $fillQuantity = $this->safe_float($order, 'fillQuantity');
        $commission = $this->safe_float($order, 'commission');
        $proceeds = $this->safe_float($order, 'proceeds');
        $status = $this->safe_string_lower($order, 'status');
        $average = null;
        $remaining = null;
        if ($fillQuantity !== null) {
            if ($proceeds !== null) {
                if ($fillQuantity > 0) {
                    $average = $proceeds / $fillQuantity;
                } else if ($proceeds === 0) {
                    $average = 0;
                }
            }
            if ($quantity !== null) {
                $remaining = $quantity - $fillQuantity;
            }
        }
        return array(
            'id' => $this->safe_string($order, 'id'),
            'clientOrderId' => null,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'lastTradeTimestamp' => $lastTradeTimestamp,
            'symbol' => $symbol,
            'type' => $type,
            'side' => $direction,
            'price' => $limit,
            'cost' => $proceeds,
            'average' => $average,
            'amount' => $quantity,
            'filled' => $fillQuantity,
            'remaining' => $remaining,
            'status' => $status,
            'fee' => array(
                'cost' => $commission,
                'currency' => $feeCurrency,
            ),
            'info' => $order,
            'trades' => null,
        );
    }

    public function parse_order_v2($order, $market = null) {
        //
        //     {
        //         "Uuid" => "string (uuid)",
        //         "OrderUuid" => "8925d746-bc9f-4684-b1aa-e507467aaa99",
        //         "Exchange" => "BTC-LTC",
        //         "OrderType" => "string",
        //         "Quantity" => 100000,
        //         "QuantityRemaining" => 100000,
        //         "Limit" => 1e-8,
        //         "CommissionPaid" => 0,
        //         "Price" => 0,
        //         "PricePerUnit" => null,
        //         "Opened" => "2014-07-09T03:55:48.583",
        //         "Closed" => null,
        //         "CancelInitiated" => "boolean",
        //         "ImmediateOrCancel" => "boolean",
        //         "IsConditional" => "boolean"
        //     }
        //
        $side = $this->safe_string_2($order, 'OrderType', 'Type');
        $isBuyOrder = ($side === 'LIMIT_BUY') || ($side === 'BUY') || ($side === 'MARKET_BUY');
        $isSellOrder = ($side === 'LIMIT_SELL') || ($side === 'SELL') || ($side === 'MARKET_SELL');
        $type = 'limit';
        if (($side === 'MARKET_BUY') || ($side === 'MARKET_SELL')) {
            $type = 'market';
        }
        if ($isBuyOrder) {
            $side = 'buy';
        }
        if ($isSellOrder) {
            $side = 'sell';
        }
        // We parse different fields in a very specific $order->
        // Order might well be $closed and then canceled.
        $status = null;
        if ((is_array($order) && array_key_exists('Opened', $order)) && $order['Opened']) {
            $status = 'open';
        }
        if ((is_array($order) && array_key_exists('Closed', $order)) && $order['Closed']) {
            $status = 'closed';
        }
        if ((is_array($order) && array_key_exists('CancelInitiated', $order)) && $order['CancelInitiated']) {
            $status = 'canceled';
        }
        if ((is_array($order) && array_key_exists('Status', $order)) && $this->options['parseOrderStatus']) {
            $status = $this->parse_order_status($this->safe_string($order, 'Status'));
        }
        $symbol = null;
        if (is_array($order) && array_key_exists('Exchange', $order)) {
            $marketId = $this->safe_string($order, 'Exchange');
            if ($marketId !== null) {
                if (is_array($this->markets_by_id) && array_key_exists($marketId, $this->markets_by_id)) {
                    $market = $this->markets_by_id[$marketId];
                    $symbol = $market['symbol'];
                } else {
                    $symbol = $this->parse_symbol($marketId);
                }
            }
        } else {
            if ($market !== null) {
                $symbol = $market['symbol'];
            }
        }
        $timestamp = null;
        $opened = $this->safe_string($order, 'Opened');
        if ($opened !== null) {
            $timestamp = $this->parse8601($opened . '+00:00');
        }
        $created = $this->safe_string($order, 'Created');
        if ($created !== null) {
            $timestamp = $this->parse8601($created . '+00:00');
        }
        $lastTradeTimestamp = null;
        $lastTimestamp = $this->safe_string($order, 'TimeStamp');
        if ($lastTimestamp !== null) {
            $lastTradeTimestamp = $this->parse8601($lastTimestamp . '+00:00');
        }
        $closed = $this->safe_string($order, 'Closed');
        if ($closed !== null) {
            $lastTradeTimestamp = $this->parse8601($closed . '+00:00');
        }
        if ($timestamp === null) {
            $timestamp = $lastTradeTimestamp;
        }
        $fee = null;
        $feeCost = $this->safe_float_2($order, 'Commission', 'CommissionPaid');
        if ($feeCost !== null) {
            $fee = array(
                'cost' => $feeCost,
            );
            if ($market !== null) {
                $fee['currency'] = $market['quote'];
            } else if ($symbol !== null) {
                $currencyIds = explode('/', $symbol);
                $quoteCurrencyId = $currencyIds[1];
                $fee['currency'] = $this->safe_currency_code($quoteCurrencyId);
            }
        }
        $price = $this->safe_float($order, 'Limit');
        $cost = $this->safe_float($order, 'Price');
        $amount = $this->safe_float($order, 'Quantity');
        $remaining = $this->safe_float($order, 'QuantityRemaining');
        $filled = null;
        if ($amount !== null && $remaining !== null) {
            $filled = $amount - $remaining;
            if (($status === 'closed') && ($remaining > 0)) {
                $status = 'canceled';
            }
        }
        if (!$cost) {
            if ($price && $filled) {
                $cost = $price * $filled;
            }
        }
        if (!$price) {
            if ($cost && $filled) {
                $price = $cost / $filled;
            }
        }
        $average = $this->safe_float($order, 'PricePerUnit');
        $id = $this->safe_string_2($order, 'OrderUuid', 'OrderId');
        return array(
            'info' => $order,
            'id' => $id,
            'clientOrderId' => null,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'lastTradeTimestamp' => $lastTradeTimestamp,
            'symbol' => $symbol,
            'type' => $type,
            'side' => $side,
            'price' => $price,
            'cost' => $cost,
            'average' => $average,
            'amount' => $amount,
            'filled' => $filled,
            'remaining' => $remaining,
            'status' => $status,
            'fee' => $fee,
            'trades' => null,
        );
    }

    public function fetch_order($id, $symbol = null, $params = array ()) {
        $this->load_markets();
        $response = null;
        try {
            $orderIdField = $this->get_order_id_field();
            $request = array();
            $request[$orderIdField] = $id;
            $response = $this->accountGetOrder (array_merge($request, $params));
        } catch (Exception $e) {
            if ($this->last_json_response) {
                $message = $this->safe_string($this->last_json_response, 'message');
                if ($message === 'UUID_INVALID') {
                    throw new OrderNotFound($this->id . ' fetchOrder() error => ' . $this->last_http_response);
                }
            }
            throw $e;
        }
        if (!$response['result']) {
            throw new OrderNotFound($this->id . ' order ' . $id . ' not found');
        }
        return $this->parse_order($response['result']);
    }

    public function order_to_trade($order) {
        // this entire method should be moved to the base class
        $timestamp = $this->safe_integer_2($order, 'lastTradeTimestamp', 'timestamp');
        return array(
            'id' => $this->safe_string($order, 'id'),
            'side' => $this->safe_string($order, 'side'),
            'order' => $this->safe_string($order, 'id'),
            'type' => $this->safe_string($order, 'type'),
            'price' => $this->safe_float($order, 'average'),
            'amount' => $this->safe_float($order, 'filled'),
            'cost' => $this->safe_float($order, 'cost'),
            'symbol' => $this->safe_string($order, 'symbol'),
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'fee' => $this->safe_value($order, 'fee'),
            'info' => $order,
            'takerOrMaker' => null,
        );
    }

    public function orders_to_trades($orders) {
        // this entire method should be moved to the base class
        $result = array();
        for ($i = 0; $i < count($orders); $i++) {
            $result[] = $this->order_to_trade($orders[$i]);
        }
        return $result;
    }

    public function fetch_my_trades_v2($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market'] = $market['id'];
        }
        $response = $this->accountGetOrderhistory (array_merge($request, $params));
        $result = $this->safe_value($response, 'result', array());
        $orders = $this->parse_orders($result, $market);
        $trades = $this->orders_to_trades($orders);
        if ($symbol !== null) {
            return $this->filter_by_since_limit($trades, $since, $limit);
        } else {
            return $this->filter_by_symbol_since_limit($trades, $symbol, $since, $limit);
        }
    }

    public function fetch_my_trades_v3($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        if ($limit !== null) {
            $request['pageSize'] = $limit;
        }
        if ($since !== null) {
            $request['startDate'] = $this->ymdhms($since, 'T') . 'Z';
        }
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            // because of this line we will have to rethink the entire v3
            // in other words, markets define all the rest of the API
            // and v3 $market ids are reversed in comparison to v1
            // v3 has to be a completely separate implementation
            // otherwise we will have to shuffle symbols and currencies everywhere
            // which is prone to errors, as was shown here
            // https://github.com/ccxt/ccxt/pull/5219#issuecomment-499646209
            $request['marketSymbol'] = $market['base'] . '-' . $market['quote'];
        }
        $response = $this->v3GetOrdersClosed (array_merge($request, $params));
        $orders = $this->parse_orders($response, $market);
        $trades = $this->orders_to_trades($orders);
        if ($symbol !== null) {
            return $this->filter_by_since_limit($trades, $since, $limit);
        } else {
            return $this->filter_by_symbol_since_limit($trades, $symbol, $since, $limit);
        }
    }

    public function fetch_my_trades($symbol = null, $since = null, $limit = null, $params = array ()) {
        $method = $this->safe_string($this->options, 'fetchMyTradesMethod', 'fetch_my_trades_v3');
        return $this->$method ($symbol, $since, $limit, $params);
    }

    public function fetch_closed_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $method = $this->safe_string($this->options, 'fetchClosedOrdersMethod', 'fetch_closed_orders_v3');
        return $this->$method ($symbol, $since, $limit, $params);
    }

    public function fetch_closed_orders_v2($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market'] = $market['id'];
        }
        $response = $this->accountGetOrderhistory (array_merge($request, $params));
        $result = $this->safe_value($response, 'result', array());
        $orders = $this->parse_orders($result, $market, $since, $limit);
        if ($symbol !== null) {
            return $this->filter_by_symbol($orders, $symbol);
        }
        return $orders;
    }

    public function fetch_closed_orders_v3($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        if ($limit !== null) {
            $request['pageSize'] = $limit;
        }
        if ($since !== null) {
            $request['startDate'] = $this->ymdhms($since, 'T') . 'Z';
        }
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            // because of this line we will have to rethink the entire v3
            // in other words, markets define all the rest of the API
            // and v3 $market ids are reversed in comparison to v1
            // v3 has to be a completely separate implementation
            // otherwise we will have to shuffle symbols and currencies everywhere
            // which is prone to errors, as was shown here
            // https://github.com/ccxt/ccxt/pull/5219#issuecomment-499646209
            $request['marketSymbol'] = $market['base'] . '-' . $market['quote'];
        }
        $response = $this->v3GetOrdersClosed (array_merge($request, $params));
        $orders = $this->parse_orders($response, $market, $since, $limit);
        if ($symbol !== null) {
            return $this->filter_by_symbol($orders, $symbol);
        }
        return $orders;
    }

    public function fetch_deposit_address($code, $params = array ()) {
        $this->load_markets();
        $currency = $this->currency($code);
        $request = array(
            'currency' => $currency['id'],
        );
        $response = $this->accountGetDepositaddress (array_merge($request, $params));
        //
        //     array( "success" => false, "$message" => "ADDRESS_GENERATING", "result" => null )
        //
        //     { success =>    true,
        //       $message =>   "",
        //        result => { Currency => "INCNT",
        //                   Address => "3PHvQt9bK21f7eVQVdJzrNPcsMzXabEA5Ha" } } }
        //
        $address = $this->safe_string($response['result'], 'Address');
        $message = $this->safe_string($response, 'message');
        if (!$address || $message === 'ADDRESS_GENERATING') {
            throw new AddressPending($this->id . ' the $address for ' . $code . ' is being generated (pending, not ready yet, retry again later)');
        }
        $tag = null;
        if (is_array($this->options['tag']) && array_key_exists($currency['type'], $this->options['tag'])) {
            $tag = $address;
            $address = $currency['address'];
        }
        $this->check_address($address);
        return array(
            'currency' => $code,
            'address' => $address,
            'tag' => $tag,
            'info' => $response,
        );
    }

    public function withdraw($code, $amount, $address, $tag = null, $params = array ()) {
        $this->check_address($address);
        $this->load_markets();
        $currency = $this->currency($code);
        $request = array(
            'currency' => $currency['id'],
            'quantity' => $amount,
            'address' => $address,
        );
        if ($tag !== null) {
            $request['paymentid'] = $tag;
        }
        $response = $this->accountGetWithdraw (array_merge($request, $params));
        $result = $this->safe_value($response, 'result', array());
        $id = $this->safe_string($result, 'uuid');
        return array(
            'info' => $response,
            'id' => $id,
        );
    }

    public function sign($path, $api = 'public', $method = 'GET', $params = array (), $headers = null, $body = null) {
        $url = $this->implode_params($this->urls['api'][$api], array(
            'hostname' => $this->hostname,
        )) . '/';
        if ($api !== 'v3' && $api !== 'v3public') {
            $url .= $this->version . '/';
        }
        if ($api === 'public') {
            $url .= $api . '/' . strtolower($method) . $this->implode_params($path, $params);
            $params = $this->omit($params, $this->extract_params($path));
            if ($params) {
                $url .= '?' . $this->urlencode($params);
            }
        } else if ($api === 'v3public') {
            $url .= $this->implode_params($path, $params);
            $params = $this->omit($params, $this->extract_params($path));
            if ($params) {
                $url .= '?' . $this->urlencode($params);
            }
        } else if ($api === 'v3') {
            $url .= $path;
            $hashString = '';
            if ($method === 'POST') {
                $body = $this->json($params);
                $hashString = $body;
            } else {
                if ($params) {
                    $url .= '?' . $this->rawencode($params);
                }
            }
            $contentHash = $this->hash($this->encode($hashString), 'sha512', 'hex');
            $timestamp = (string) $this->milliseconds();
            $auth = $timestamp . $url . $method . $contentHash;
            $subaccountId = $this->safe_value($this->options, 'subaccountId');
            if ($subaccountId !== null) {
                $auth .= $subaccountId;
            }
            $signature = $this->hmac($this->encode($auth), $this->encode($this->secret), 'sha512');
            $headers = array(
                'Api-Key' => $this->apiKey,
                'Api-Timestamp' => $timestamp,
                'Api-Content-Hash' => $contentHash,
                'Api-Signature' => $signature,
            );
            if ($subaccountId !== null) {
                $headers['Api-Subaccount-Id'] = $subaccountId;
            }
            if ($method === 'POST') {
                $headers['Content-Type'] = 'application/json';
            }
        } else {
            $this->check_required_credentials();
            $url .= $api . '/';
            if ((($api === 'account') && ($path !== 'withdraw')) || ($path === 'openorders')) {
                $url .= strtolower($method);
            }
            $request = array(
                'apikey' => $this->apiKey,
            );
            $disableNonce = $this->safe_value($this->options, 'disableNonce');
            if (($disableNonce === null) || !$disableNonce) {
                $request['nonce'] = $this->nonce();
            }
            $url .= $path . '?' . $this->urlencode(array_merge($request, $params));
            $signature = $this->hmac($this->encode($url), $this->encode($this->secret), 'sha512');
            $headers = array( 'apisign' => $signature );
        }
        return array( 'url' => $url, 'method' => $method, 'body' => $body, 'headers' => $headers );
    }

    public function handle_errors($code, $reason, $url, $method, $headers, $body, $response, $requestHeaders, $requestBody) {
        if ($response === null) {
            return; // fallback to default error handler
        }
        //
        //     array( $success => false, $message => "$message" )
        //
        if ($body[0] === '{') {
            $feedback = $this->id . ' ' . $body;
            $success = $this->safe_value($response, 'success');
            if ($success === null) {
                $code = $this->safe_string($response, 'code');
                if ($code !== null) {
                    $this->throw_exactly_matched_exception($this->exceptions['exact'], $code, $feedback);
                    if ($code !== null) {
                        $this->throw_broadly_matched_exception($this->exceptions['broad'], $code, $feedback);
                    }
                }
                // throw new ExchangeError($this->id . ' malformed $response ' . $this->json($response));
                return;
            }
            if (gettype($success) === 'string') {
                // bleutrade uses string instead of boolean
                $success = ($success === 'true') ? true : false;
            }
            if (!$success) {
                $message = $this->safe_string($response, 'message');
                if ($message === 'APIKEY_INVALID') {
                    if ($this->options['hasAlreadyAuthenticatedSuccessfully']) {
                        throw new DDoSProtection($feedback);
                    } else {
                        throw new AuthenticationError($feedback);
                    }
                }
                // https://github.com/ccxt/ccxt/issues/4932
                // the following two lines are now redundant, see line 171 in describe()
                //
                //     if ($message === 'DUST_TRADE_DISALLOWED_MIN_VALUE_50K_SAT')
                //         throw new InvalidOrder($this->id . ' order cost should be over 50k satoshi ' . $this->json($response));
                //
                if ($message === 'INVALID_ORDER') {
                    // Bittrex will return an ambiguous INVALID_ORDER $message
                    // upon canceling already-canceled and closed orders
                    // therefore this special case for cancelOrder
                    // $url = 'https://bittrex.com/api/v1.1/market/cancel?apikey=API_KEY&uuid=ORDER_UUID'
                    $cancel = 'cancel';
                    $indexOfCancel = mb_strpos($url, $cancel);
                    if ($indexOfCancel >= 0) {
                        $urlParts = explode('?', $url);
                        $numParts = is_array($urlParts) ? count($urlParts) : 0;
                        if ($numParts > 1) {
                            $query = $urlParts[1];
                            $params = explode('&', $query);
                            $numParams = is_array($params) ? count($params) : 0;
                            $orderId = null;
                            for ($i = 0; $i < $numParams; $i++) {
                                $param = $params[$i];
                                $keyValue = explode('=', $param);
                                if ($keyValue[0] === 'uuid') {
                                    $orderId = $keyValue[1];
                                    break;
                                }
                            }
                            if ($orderId !== null) {
                                throw new OrderNotFound($this->id . ' cancelOrder ' . $orderId . ' ' . $this->json($response));
                            } else {
                                throw new OrderNotFound($this->id . ' cancelOrder ' . $this->json($response));
                            }
                        }
                    }
                }
                $this->throw_exactly_matched_exception($this->exceptions['exact'], $message, $feedback);
                if ($message !== null) {
                    $this->throw_broadly_matched_exception($this->exceptions['broad'], $message, $feedback);
                }
                throw new ExchangeError($feedback);
            }
        }
    }

    public function request($path, $api = 'public', $method = 'GET', $params = array (), $headers = null, $body = null) {
        $response = $this->fetch2($path, $api, $method, $params, $headers, $body);
        // a workaround for APIKEY_INVALID
        if (($api === 'account') || ($api === 'market')) {
            $this->options['hasAlreadyAuthenticatedSuccessfully'] = true;
        }
        return $response;
    }
}
