<?php

namespace ccxt;

// PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
// https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

use Exception; // a common import
use \ccxt\ExchangeError;
use \ccxt\AuthenticationError;
use \ccxt\ArgumentsRequired;
use \ccxt\BadResponse;
use \ccxt\InvalidAddress;
use \ccxt\InvalidOrder;

class probit extends Exchange {

    public function describe() {
        return $this->deep_extend(parent::describe (), array(
            'id' => 'probit',
            'name' => 'ProBit',
            'countries' => array( 'SC', 'KR' ), // Seychelles, South Korea
            'rateLimit' => 250, // ms
            'has' => array(
                'CORS' => true,
                'fetchTime' => true,
                'fetchMarkets' => true,
                'fetchCurrencies' => true,
                'fetchTickers' => true,
                'fetchTicker' => true,
                'fetchOHLCV' => true,
                'fetchOrderBook' => true,
                'fetchTrades' => true,
                'fetchBalance' => true,
                'createOrder' => true,
                'createMarketOrder' => true,
                'cancelOrder' => true,
                'fetchOrder' => true,
                'fetchOpenOrders' => true,
                'fetchClosedOrders' => true,
                'fetchMyTrades' => true,
                'fetchDepositAddress' => true,
                'withdraw' => true,
                'signIn' => true,
            ),
            'timeframes' => array(
                '1m' => '1m',
                '3m' => '3m',
                '5m' => '5m',
                '10m' => '10m',
                '15m' => '15m',
                '30m' => '30m',
                '1h' => '1h',
                '4h' => '4h',
                '6h' => '6h',
                '12h' => '12h',
                '1d' => '1D',
                '1w' => '1W',
                '1M' => '1M',
            ),
            'version' => 'v1',
            'urls' => array(
                'logo' => 'https://user-images.githubusercontent.com/51840849/79268032-c4379480-7ea2-11ea-80b3-dd96bb29fd0d.jpg',
                'api' => array(
                    'accounts' => 'https://accounts.probit.com',
                    'public' => 'https://api.probit.com/api/exchange',
                    'private' => 'https://api.probit.com/api/exchange',
                ),
                'www' => 'https://www.probit.com',
                'doc' => array(
                    'https://docs-en.probit.com',
                    'https://docs-ko.probit.com',
                ),
                'fees' => 'https://support.probit.com/hc/en-us/articles/360020968611-Trading-Fees',
                'referral' => 'https://www.probit.com/r/34608773',
            ),
            'api' => array(
                'public' => array(
                    'get' => array(
                        'market',
                        'currency',
                        'currency_with_platform',
                        'time',
                        'ticker',
                        'order_book',
                        'trade',
                        'candle',
                    ),
                ),
                'private' => array(
                    'post' => array(
                        'new_order',
                        'cancel_order',
                        'withdrawal',
                    ),
                    'get' => array(
                        'balance',
                        'order',
                        'open_order',
                        'order_history',
                        'trade_history',
                        'deposit_address',
                    ),
                ),
                'accounts' => array(
                    'post' => array(
                        'token',
                    ),
                ),
            ),
            'fees' => array(
                'trading' => array(
                    'tierBased' => false,
                    'percentage' => true,
                    'maker' => 0.2 / 100,
                    'taker' => 0.2 / 100,
                ),
            ),
            'exceptions' => array(
                'exact' => array(
                    'UNAUTHORIZED' => '\\ccxt\\AuthenticationError',
                    'INVALID_ARGUMENT' => '\\ccxt\\BadRequest', // Parameters are not a valid format, parameters are empty, or out of range, or a parameter was sent when not required.
                    'TRADING_UNAVAILABLE' => '\\ccxt\\ExchangeNotAvailable',
                    'NOT_ENOUGH_BALANCE' => '\\ccxt\\InsufficientFunds',
                    'NOT_ALLOWED_COMBINATION' => '\\ccxt\\BadRequest',
                    'INVALID_ORDER' => '\\ccxt\\InvalidOrder', // Requested order does not exist, or it is not your order
                    'RATE_LIMIT_EXCEEDED' => '\\ccxt\\RateLimitExceeded', // You are sending requests too frequently. Please try it later.
                    'MARKET_UNAVAILABLE' => '\\ccxt\\ExchangeNotAvailable', // Market is closed today
                    'INVALID_MARKET' => '\\ccxt\\BadSymbol', // Requested market is not exist
                    'INVALID_CURRENCY' => '\\ccxt\\BadRequest', // Requested currency is not exist on ProBit system
                    'TOO_MANY_OPEN_ORDERS' => '\\ccxt\\DDoSProtection', // Too many open orders
                    'DUPLICATE_ADDRESS' => '\\ccxt\\InvalidAddress', // Address already exists in withdrawal address list
                ),
            ),
            'requiredCredentials' => array(
                'apiKey' => true,
                'secret' => true,
            ),
            'options' => array(
                'createMarketBuyOrderRequiresPrice' => true,
                'timeInForce' => array(
                    'limit' => 'gtc',
                    'market' => 'ioc',
                ),
            ),
            'commonCurrencies' => array(
                'BTCBEAR' => 'BEAR',
                'BTCBULL' => 'BULL',
            ),
        ));
    }

    public function fetch_markets($params = array ()) {
        $response = $this->publicGetMarket ($params);
        //
        //     {
        //         "data":array(
        //             array(
        //                 "$id":"MONA-USDT",
        //                 "base_currency_id":"MONA",
        //                 "quote_currency_id":"USDT",
        //                 "min_price":"0.001",
        //                 "max_price":"9999999999999999",
        //                 "price_increment":"0.001",
        //                 "min_quantity":"0.0001",
        //                 "max_quantity":"9999999999999999",
        //                 "quantity_precision":4,
        //                 "min_cost":"1",
        //                 "max_cost":"9999999999999999",
        //                 "cost_precision":8,
        //                 "taker_fee_rate":"0.2",
        //                 "maker_fee_rate":"0.2",
        //                 "show_in_ui":true,
        //                 "$closed":false
        //             ),
        //         )
        //     }
        //
        $markets = $this->safe_value($response, 'data', array());
        $result = array();
        for ($i = 0; $i < count($markets); $i++) {
            $market = $markets[$i];
            $id = $this->safe_string($market, 'id');
            $baseId = $this->safe_string($market, 'base_currency_id');
            $quoteId = $this->safe_string($market, 'quote_currency_id');
            $base = $this->safe_currency_code($baseId);
            $quote = $this->safe_currency_code($quoteId);
            $symbol = $base . '/' . $quote;
            $closed = $this->safe_value($market, 'closed', false);
            $active = !$closed;
            $priceIncrement = $this->safe_string($market, 'price_increment');
            $precision = array(
                'amount' => $this->safe_integer($market, 'quantity_precision'),
                'price' => $this->precision_from_string($priceIncrement),
                'cost' => $this->safe_integer($market, 'cost_precision'),
            );
            $takerFeeRate = $this->safe_float($market, 'taker_fee_rate');
            $makerFeeRate = $this->safe_float($market, 'maker_fee_rate');
            $result[] = array(
                'id' => $id,
                'info' => $market,
                'symbol' => $symbol,
                'base' => $base,
                'quote' => $quote,
                'baseId' => $baseId,
                'quoteId' => $quoteId,
                'active' => $active,
                'precision' => $precision,
                'taker' => $takerFeeRate / 100,
                'maker' => $makerFeeRate / 100,
                'limits' => array(
                    'amount' => array(
                        'min' => $this->safe_float($market, 'min_quantity'),
                        'max' => $this->safe_float($market, 'max_quantity'),
                    ),
                    'price' => array(
                        'min' => $this->safe_float($market, 'min_price'),
                        'max' => $this->safe_float($market, 'max_price'),
                    ),
                    'cost' => array(
                        'min' => $this->safe_float($market, 'min_cost'),
                        'max' => $this->safe_float($market, 'max_cost'),
                    ),
                ),
            );
        }
        return $result;
    }

    public function fetch_currencies($params = array ()) {
        $response = $this->publicGetCurrencyWithPlatform ($params);
        //
        //     {
        //         "data":array(
        //             {
        //                 "$id":"USDT",
        //                 "display_name":array("ko-kr":"테더","en-us":"Tether"),
        //                 "show_in_ui":true,
        //                 "$platform":[
        //                     array(
        //                         "$id":"ETH",
        //                         "priority":1,
        //                         "deposit":true,
        //                         "withdrawal":true,
        //                         "currency_id":"USDT",
        //                         "$precision":6,
        //                         "min_confirmation_count":15,
        //                         "require_destination_tag":false,
        //                         "display_name":array("$name":array("ko-kr":"ERC-20","en-us":"ERC-20")),
        //                         "min_deposit_amount":"0",
        //                         "min_withdrawal_amount":"1",
        //                         "withdrawal_fee":[
        //                             array("amount":"0.01","priority":2,"currency_id":"ETH"),
        //                             array("amount":"1.5","priority":1,"currency_id":"USDT"),
        //                         ),
        //                         "deposit_fee":array(),
        //                         "suspended_reason":"",
        //                         "deposit_suspended":false,
        //                         "withdrawal_suspended":false
        //                     ),
        //                     {
        //                         "$id":"OMNI",
        //                         "priority":2,
        //                         "deposit":true,
        //                         "withdrawal":true,
        //                         "currency_id":"USDT",
        //                         "$precision":6,
        //                         "min_confirmation_count":3,
        //                         "require_destination_tag":false,
        //                         "display_name":array("$name":array("ko-kr":"OMNI","en-us":"OMNI")),
        //                         "min_deposit_amount":"0",
        //                         "min_withdrawal_amount":"5",
        //                         "withdrawal_fee":[array("amount":"5","priority":1,"currency_id":"USDT")],
        //                         "deposit_fee":array(),
        //                         "suspended_reason":"wallet_maintenance",
        //                         "deposit_suspended":false,
        //                         "withdrawal_suspended":false
        //                     }
        //                 ],
        //                 "stakeable":false,
        //                 "unstakeable":false,
        //                 "auto_stake":false,
        //                 "auto_stake_amount":"0"
        //             }
        //         ]
        //     }
        //
        $currencies = $this->safe_value($response, 'data');
        $result = array();
        for ($i = 0; $i < count($currencies); $i++) {
            $currency = $currencies[$i];
            $id = $this->safe_string($currency, 'id');
            $code = $this->safe_currency_code($id);
            $displayName = $this->safe_value($currency, 'display_name');
            $name = $this->safe_string($displayName, 'en-us');
            $platforms = $this->safe_value($currency, 'platform', array());
            $platformsByPriority = $this->sort_by($platforms, 'priority');
            $platform = $this->safe_value($platformsByPriority, 0, array());
            $precision = $this->safe_integer($platform, 'precision');
            $depositSuspended = $this->safe_value($platform, 'deposit_suspended');
            $withdrawalSuspended = $this->safe_value($platform, 'withdrawal_suspended');
            $active = !($depositSuspended && $withdrawalSuspended);
            $withdrawalFees = $this->safe_value($platform, 'withdrawal_fee', array());
            $withdrawalFeesByPriority = $this->sort_by($withdrawalFees, 'priority');
            $withdrawalFee = $this->safe_value($withdrawalFeesByPriority, 0, array());
            $fee = $this->safe_float($withdrawalFee, 'amount');
            $result[$code] = array(
                'id' => $id,
                'code' => $code,
                'info' => $currency,
                'name' => $name,
                'active' => $active,
                'fee' => $fee,
                'precision' => $precision,
                'limits' => array(
                    'amount' => array(
                        'min' => pow(10, -$precision),
                        'max' => pow(10, $precision),
                    ),
                    'price' => array(
                        'min' => pow(10, -$precision),
                        'max' => pow(10, $precision),
                    ),
                    'cost' => array(
                        'min' => null,
                        'max' => null,
                    ),
                    'deposit' => array(
                        'min' => $this->safe_float($platform, 'min_deposit_amount'),
                        'max' => null,
                    ),
                    'withdraw' => array(
                        'min' => $this->safe_float($platform, 'min_withdrawal_amount'),
                        'max' => null,
                    ),
                ),
            );
        }
        return $result;
    }

    public function fetch_balance($params = array ()) {
        $this->load_markets();
        $response = $this->privateGetBalance ($params);
        //
        //     {
        //         $data => array(
        //             {
        //                 "currency_id":"XRP",
        //                 "total":"100",
        //                 "available":"0",
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data');
        $result = array( 'info' => $data );
        for ($i = 0; $i < count($data); $i++) {
            $balance = $data[$i];
            $currencyId = $this->safe_string($balance, 'currency_id');
            $code = $this->safe_currency_code($currencyId);
            $account = $this->account();
            $account['total'] = $this->safe_float($balance, 'total');
            $account['free'] = $this->safe_float($balance, 'available');
            $result[$code] = $account;
        }
        return $this->parse_balance($result);
    }

    public function fetch_order_book($symbol, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market_id' => $market['id'],
        );
        $response = $this->publicGetOrderBook (array_merge($request, $params));
        //
        //     {
        //         $data => array(
        //             array( side => 'buy', price => '0.000031', quantity => '10' ),
        //             array( side => 'buy', price => '0.00356007', quantity => '4.92156877' ),
        //             array( side => 'sell', price => '0.1857', quantity => '0.17' ),
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $dataBySide = $this->group_by($data, 'side');
        return $this->parse_order_book($dataBySide, null, 'buy', 'sell', 'price', 'quantity');
    }

    public function fetch_tickers($symbols = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        if ($symbols !== null) {
            $marketIds = $this->market_ids($symbols);
            $request['market_ids'] = implode(',', $marketIds);
        }
        $response = $this->publicGetTicker (array_merge($request, $params));
        //
        //     {
        //         "$data":array(
        //             {
        //                 "last":"0.022902",
        //                 "low":"0.021693",
        //                 "high":"0.024093",
        //                 "change":"-0.000047",
        //                 "base_volume":"15681.986",
        //                 "quote_volume":"360.514403624",
        //                 "market_id":"ETH-BTC",
        //                 "time":"2020-04-12T18:43:38.000Z"
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_tickers($data, $symbols);
    }

    public function parse_tickers($rawTickers, $symbols = null) {
        $tickers = array();
        for ($i = 0; $i < count($rawTickers); $i++) {
            $tickers[] = $this->parse_ticker($rawTickers[$i]);
        }
        return $this->filter_by_array($tickers, 'symbol', $symbols);
    }

    public function fetch_ticker($symbol, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market_ids' => $market['id'],
        );
        $response = $this->publicGetTicker (array_merge($request, $params));
        //
        //     {
        //         "$data":array(
        //             {
        //                 "last":"0.022902",
        //                 "low":"0.021693",
        //                 "high":"0.024093",
        //                 "change":"-0.000047",
        //                 "base_volume":"15681.986",
        //                 "quote_volume":"360.514403624",
        //                 "market_id":"ETH-BTC",
        //                 "time":"2020-04-12T18:43:38.000Z"
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $ticker = $this->safe_value($data, 0);
        if ($ticker === null) {
            throw new BadResponse($this->id . ' fetchTicker() returned an empty response');
        }
        return $this->parse_ticker($ticker, $market);
    }

    public function parse_ticker($ticker, $market = null) {
        //
        //     {
        //         "last":"0.022902",
        //         "low":"0.021693",
        //         "high":"0.024093",
        //         "$change":"-0.000047",
        //         "base_volume":"15681.986",
        //         "quote_volume":"360.514403624",
        //         "market_id":"ETH-BTC",
        //         "time":"2020-04-12T18:43:38.000Z"
        //     }
        //
        $timestamp = $this->parse8601($this->safe_string($ticker, 'time'));
        $symbol = null;
        $marketId = $this->safe_string($ticker, 'market_id');
        if ($marketId !== null) {
            if (is_array($this->markets_by_id) && array_key_exists($marketId, $this->markets_by_id)) {
                $market = $this->markets_by_id[$marketId];
            } else {
                list($baseId, $quoteId) = explode('-', $marketId);
                $base = $this->safe_currency_code($baseId);
                $quote = $this->safe_currency_code($quoteId);
                $symbol = $base . '/' . $quote;
            }
        }
        if (($symbol === null) && ($market !== null)) {
            $symbol = $market['symbol'];
        }
        $close = $this->safe_float($ticker, 'last');
        $change = $this->safe_float($ticker, 'change');
        $percentage = null;
        $open = null;
        if ($change !== null) {
            if ($close !== null) {
                $open = $close - $change;
                $percentage = ($change / $open) * 100;
            }
        }
        $baseVolume = $this->safe_float($ticker, 'base_volume');
        $quoteVolume = $this->safe_float($ticker, 'quote_volume');
        $vwap = null;
        if (($baseVolume !== null) && ($quoteVolume !== null)) {
            $vwap = $baseVolume / $quoteVolume;
        }
        return array(
            'symbol' => $symbol,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'high' => $this->safe_float($ticker, 'high'),
            'low' => $this->safe_float($ticker, 'low'),
            'bid' => null,
            'bidVolume' => null,
            'ask' => null,
            'askVolume' => null,
            'vwap' => $vwap,
            'open' => $open,
            'close' => $close,
            'last' => $close,
            'previousClose' => null, // previous day $close
            'change' => $change,
            'percentage' => $percentage,
            'average' => null,
            'baseVolume' => $baseVolume,
            'quoteVolume' => $quoteVolume,
            'info' => $ticker,
        );
    }

    public function fetch_my_trades($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = null;
        $request = array(
            'limit' => 100,
            'start_time' => $this->iso8601(0),
            'end_time' => $this->iso8601($this->milliseconds()),
        );
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market_id'] = $market['id'];
        }
        if ($since !== null) {
            $request['start_time'] = $this->iso8601($since);
        }
        if ($limit !== null) {
            $request['limit'] = $limit;
        }
        $response = $this->privateGetTradeHistory (array_merge($request, $params));
        //
        //     {
        //         $data => array(
        //             {
        //                 "id":"BTC-USDT:183566",
        //                 "order_id":"17209376",
        //                 "side":"sell",
        //                 "fee_amount":"0.657396569175",
        //                 "fee_currency_id":"USDT",
        //                 "status":"settled",
        //                 "price":"6573.96569175",
        //                 "quantity":"0.1",
        //                 "cost":"657.396569175",
        //                 "time":"2018-08-10T06:06:46.000Z",
        //                 "market_id":"BTC-USDT"
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_trades($data, $market, $since, $limit);
    }

    public function fetch_trades($symbol, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market_id' => $market['id'],
            'limit' => 100,
            'start_time' => '1970-01-01T00:00:00.000Z',
            'end_time' => $this->iso8601($this->milliseconds()),
        );
        if ($since !== null) {
            $request['start_time'] = $this->iso8601($since);
        }
        if ($limit !== null) {
            $request['limit'] = $limit;
        }
        $response = $this->publicGetTrade (array_merge($request, $params));
        //
        //     {
        //         "$data":array(
        //             array(
        //                 "id":"ETH-BTC:3331886",
        //                 "price":"0.022981",
        //                 "quantity":"12.337",
        //                 "time":"2020-04-12T20:55:42.371Z",
        //                 "side":"sell",
        //                 "tick_direction":"down"
        //             ),
        //             {
        //                 "id":"ETH-BTC:3331885",
        //                 "price":"0.022982",
        //                 "quantity":"6.472",
        //                 "time":"2020-04-12T20:55:39.652Z",
        //                 "side":"sell",
        //                 "tick_direction":"down"
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_trades($data, $market, $since, $limit);
    }

    public function parse_trade($trade, $market = null) {
        //
        // fetchTrades (public)
        //
        //     {
        //         "$id":"ETH-BTC:3331886",
        //         "$price":"0.022981",
        //         "quantity":"12.337",
        //         "time":"2020-04-12T20:55:42.371Z",
        //         "$side":"sell",
        //         "tick_direction":"down"
        //     }
        //
        // fetchMyTrades (private)
        //
        //     {
        //         "$id":"BTC-USDT:183566",
        //         "order_id":"17209376",
        //         "$side":"sell",
        //         "fee_amount":"0.657396569175",
        //         "fee_currency_id":"USDT",
        //         "status":"settled",
        //         "$price":"6573.96569175",
        //         "quantity":"0.1",
        //         "$cost":"657.396569175",
        //         "time":"2018-08-10T06:06:46.000Z",
        //         "market_id":"BTC-USDT"
        //     }
        //
        $timestamp = $this->parse8601($this->safe_string($trade, 'time'));
        $symbol = null;
        $id = $this->safe_string($trade, 'id');
        if ($id !== null) {
            $parts = explode(':', $id);
            $marketId = $this->safe_string($parts, 0);
            if ($marketId === null) {
                $marketId = $this->safe_string($trade, 'market_id');
            }
            if ($marketId !== null) {
                if (is_array($this->markets_by_id) && array_key_exists($marketId, $this->markets_by_id)) {
                    $market = $this->markets_by_id[$marketId];
                } else {
                    list($baseId, $quoteId) = explode('-', $marketId);
                    $base = $this->safe_currency_code($baseId);
                    $quote = $this->safe_currency_code($quoteId);
                    $symbol = $base . '/' . $quote;
                }
            }
        }
        if (($symbol === null) && ($market !== null)) {
            $symbol = $market['symbol'];
        }
        $side = $this->safe_string($trade, 'side');
        $price = $this->safe_float($trade, 'price');
        $amount = $this->safe_float($trade, 'quantity');
        $cost = null;
        if ($price !== null) {
            if ($amount !== null) {
                $cost = $price * $amount;
            }
        }
        $orderId = $this->safe_string($trade, 'order_id');
        $feeCost = $this->safe_float($trade, 'fee_amount');
        $fee = null;
        if ($feeCost !== null) {
            $feeCurrencyId = $this->safe_string($trade, 'fee_currency_id');
            $feeCurrencyCode = $this->safe_currency_code($feeCurrencyId);
            $fee = array(
                'cost' => $feeCost,
                'currency' => $feeCurrencyCode,
            );
        }
        return array(
            'id' => $id,
            'info' => $trade,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'symbol' => $symbol,
            'order' => $orderId,
            'type' => null,
            'side' => $side,
            'takerOrMaker' => null,
            'price' => $price,
            'amount' => $amount,
            'cost' => $cost,
            'fee' => $fee,
        );
    }

    public function fetch_time($params = array ()) {
        $response = $this->publicGetTime ($params);
        //
        //     array( "data":"2020-04-12T18:54:25.390Z" )
        //
        $timestamp = $this->parse8601($this->safe_string($response, 'data'));
        return $timestamp;
    }

    public function normalize_ohlcv_timestamp($timestamp, $timeframe, $after = false) {
        $duration = $this->parse_timeframe($timeframe);
        if ($timeframe === '1M') {
            $iso8601 = $this->iso8601($timestamp);
            $parts = explode('-', $iso8601);
            $year = $this->safe_string($parts, 0);
            $month = $this->safe_integer($parts, 1);
            if ($after) {
                $month = $this->sum($month, 1);
            }
            if ($month < 10) {
                $month = '0' . (string) $month;
            } else {
                $month = (string) $month;
            }
            return $year . '-' . $month . '-01T00:00:00.000Z';
        } else if ($timeframe === '1w') {
            $timestamp = intval ($timestamp / 1000);
            $firstSunday = 259200; // 1970-01-04T00:00:00.000Z
            $difference = $timestamp - $firstSunday;
            $numWeeks = $this->integer_divide($difference, $duration);
            $previousSunday = $this->sum($firstSunday, $numWeeks * $duration);
            if ($after) {
                $previousSunday = $this->sum($previousSunday, $duration);
            }
            return $this->iso8601($previousSunday * 1000);
        } else {
            $timestamp = intval ($timestamp / 1000);
            $difference = $this->integer_modulo($timestamp, $duration);
            $timestamp -= $difference;
            if ($after) {
                $timestamp = $this->sum($timestamp, $duration);
            }
            return $this->iso8601($timestamp * 1000);
        }
    }

    public function fetch_ohlcv($symbol, $timeframe = '1m', $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $interval = $this->timeframes[$timeframe];
        $limit = ($limit === null) ? 100 : $limit;
        $requestLimit = $this->sum($limit, 1);
        $requestLimit = min (1000, $requestLimit); // max 1000
        $request = array(
            'market_ids' => $market['id'],
            'interval' => $interval,
            'sort' => 'asc', // 'asc' will always include the start_time, 'desc' will always include end_time
            'limit' => $requestLimit, // max 1000
        );
        $now = $this->milliseconds();
        $duration = $this->parse_timeframe($timeframe);
        $startTime = $since;
        $endTime = $now;
        if ($since === null) {
            if ($limit === null) {
                throw new ArgumentsRequired($this->id . ' fetchOHLCV requires either a $since argument or a $limit argument');
            } else {
                $startTime = $now - $limit * $duration * 1000;
            }
        } else {
            if ($limit === null) {
                $endTime = $now;
            } else {
                $endTime = $this->sum($since, $this->sum($limit, 1) * $duration * 1000);
            }
        }
        $startTimeNormalized = $this->normalize_ohlcv_timestamp($startTime, $timeframe);
        $endTimeNormalized = $this->normalize_ohlcv_timestamp($endTime, $timeframe, true);
        $request['start_time'] = $startTimeNormalized;
        $request['end_time'] = $endTimeNormalized;
        $response = $this->publicGetCandle (array_merge($request, $params));
        //
        //     {
        //         "$data":array(
        //             array(
        //                 "market_id":"ETH-BTC",
        //                 "open":"0.02811",
        //                 "close":"0.02811",
        //                 "low":"0.02811",
        //                 "high":"0.02811",
        //                 "base_volume":"0.0005",
        //                 "quote_volume":"0.000014055",
        //                 "start_time":"2018-11-30T18:19:00.000Z",
        //                 "end_time":"2018-11-30T18:20:00.000Z"
        //             ),
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_ohlcvs($data, $market, $timeframe, $since, $limit);
    }

    public function parse_ohlcv($ohlcv, $market = null) {
        //
        //     {
        //         "market_id":"ETH-BTC",
        //         "open":"0.02811",
        //         "close":"0.02811",
        //         "low":"0.02811",
        //         "high":"0.02811",
        //         "base_volume":"0.0005",
        //         "quote_volume":"0.000014055",
        //         "start_time":"2018-11-30T18:19:00.000Z",
        //         "end_time":"2018-11-30T18:20:00.000Z"
        //     }
        //
        return array(
            $this->parse8601($this->safe_string($ohlcv, 'start_time')),
            $this->safe_float($ohlcv, 'open'),
            $this->safe_float($ohlcv, 'high'),
            $this->safe_float($ohlcv, 'low'),
            $this->safe_float($ohlcv, 'close'),
            $this->safe_float($ohlcv, 'base_volume'),
        );
    }

    public function fetch_open_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $since = $this->parse8601($since);
        $request = array();
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market_id'] = $market['id'];
        }
        $response = $this->privateGetOpenOrder (array_merge($request, $params));
        $data = $this->safe_value($response, 'data');
        return $this->parse_orders($data, $market, $since, $limit);
    }

    public function fetch_closed_orders($symbol = null, $since = null, $limit = null, $params = array ()) {
        $this->load_markets();
        $request = array(
            'start_time' => $this->iso8601(0),
            'end_time' => $this->iso8601($this->milliseconds()),
            'limit' => 100,
        );
        $market = null;
        if ($symbol !== null) {
            $market = $this->market($symbol);
            $request['market_id'] = $market['id'];
        }
        if ($since) {
            $request['start_time'] = $this->iso8601($since);
        }
        if ($limit) {
            $request['limit'] = $limit;
        }
        $response = $this->privateGetOrderHistory (array_merge($request, $params));
        $data = $this->safe_value($response, 'data');
        return $this->parse_orders($data, $market, $since, $limit);
    }

    public function fetch_order($id, $symbol = null, $params = array ()) {
        if ($symbol === null) {
            throw new ArgumentsRequired($this->id . ' fetchOrder requires a $symbol argument');
        }
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market_id' => $market['id'],
        );
        $clientOrderId = $this->safe_string_2($params, 'clientOrderId', 'client_order_id');
        if ($clientOrderId !== null) {
            $request['client_order_id'] = $clientOrderId;
        } else {
            $request['order_id'] = $id;
        }
        $query = $this->omit($params, array( 'clientOrderId', 'client_order_id' ));
        $response = $this->privateGetOrder (array_merge($request, $query));
        $data = $this->safe_value($response, 'data', array());
        $order = $this->safe_value($data, 0);
        return $this->parse_order($order, $market);
    }

    public function parse_order_status($status) {
        $statuses = array(
            'open' => 'open',
            'cancelled' => 'canceled',
            'filled' => 'closed',
        );
        return $this->safe_string($statuses, $status, $status);
    }

    public function parse_order($order, $market = null) {
        //
        //     {
        //         $id => string,
        //         user_id => string,
        //         market_id => string,
        //         $type => 'orderType',
        //         $side => 'side',
        //         quantity => string,
        //         limit_price => string,
        //         time_in_force => 'timeInForce',
        //         filled_cost => string,
        //         filled_quantity => string,
        //         open_quantity => string,
        //         cancelled_quantity => string,
        //         $status => 'orderStatus',
        //         time => 'date',
        //         client_order_id => string,
        //     }
        //
        $status = $this->parse_order_status($this->safe_string($order, 'status'));
        $id = $this->safe_string($order, 'id');
        $type = $this->safe_string($order, 'type');
        $side = $this->safe_string($order, 'side');
        $symbol = null;
        $marketId = $this->safe_string($order, 'market_id');
        if ($marketId !== null) {
            if (is_array($this->markets_by_id) && array_key_exists($marketId, $this->markets_by_id)) {
                $market = $this->markets_by_id[$marketId];
            } else {
                list($baseId, $quoteId) = explode('-', $marketId);
                $base = $this->safe_currency_code($baseId);
                $quote = $this->safe_currency_code($quoteId);
                $symbol = $base . '/' . $quote;
            }
        }
        if (($symbol === null) && ($market !== null)) {
            $symbol = $market['symbol'];
        }
        $timestamp = $this->parse8601($this->safe_string($order, 'time'));
        $price = $this->safe_float($order, 'limit_price');
        $filled = $this->safe_float($order, 'filled_quantity');
        $remaining = $this->safe_float($order, 'open_quantity');
        $canceledAmount = $this->safe_float($order, 'cancelled_quantity');
        if ($canceledAmount !== null) {
            $remaining = $this->sum($remaining, $canceledAmount);
        }
        $amount = $this->safe_float($order, 'quantity', $this->sum($filled, $remaining));
        $cost = $this->safe_float_2($order, 'filled_cost', 'cost');
        if ($type === 'market') {
            $price = null;
        }
        $average = null;
        if ($filled !== null) {
            if ($cost === null) {
                if ($price !== null) {
                    $cost = $price * $filled;
                }
            }
            if ($cost !== null) {
                if ($filled > 0) {
                    $average = $cost / $filled;
                }
            }
        }
        $clientOrderId = $this->safe_string($order, 'client_order_id');
        if ($clientOrderId === '') {
            $clientOrderId = null;
        }
        return array(
            'id' => $id,
            'info' => $order,
            'clientOrderId' => $clientOrderId,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'lastTradeTimestamp' => null,
            'symbol' => $symbol,
            'type' => $type,
            'side' => $side,
            'status' => $status,
            'price' => $price,
            'amount' => $amount,
            'filled' => $filled,
            'remaining' => $remaining,
            'average' => $average,
            'cost' => $cost,
            'fee' => null,
            'trades' => null,
        );
    }

    public function cost_to_precision($symbol, $cost) {
        return $this->decimal_to_precision($cost, TRUNCATE, $this->markets[$symbol]['precision']['cost'], $this->precisionMode);
    }

    public function create_order($symbol, $type, $side, $amount, $price = null, $params = array ()) {
        $this->load_markets();
        $market = $this->market($symbol);
        $options = $this->safe_value($this->options, 'timeInForce');
        $defaultTimeInForce = $this->safe_value($options, $type);
        $timeInForce = $this->safe_string_2($params, 'timeInForce', 'time_in_force', $defaultTimeInForce);
        $request = array(
            'market_id' => $market['id'],
            'type' => $type,
            'side' => $side,
            'time_in_force' => $timeInForce,
        );
        $clientOrderId = $this->safe_string_2($params, 'clientOrderId', 'client_order_id');
        if ($clientOrderId !== null) {
            $request['client_order_id'] = $clientOrderId;
        }
        $costToPrecision = null;
        if ($type === 'limit') {
            $request['limit_price'] = $this->price_to_precision($symbol, $price);
            $request['quantity'] = $this->amount_to_precision($symbol, $amount);
        } else if ($type === 'market') {
            // for $market buy it requires the $amount of quote currency to spend
            if ($side === 'buy') {
                $cost = $this->safe_float($params, 'cost');
                $createMarketBuyOrderRequiresPrice = $this->safe_value($this->options, 'createMarketBuyOrderRequiresPrice', true);
                if ($createMarketBuyOrderRequiresPrice) {
                    if ($price !== null) {
                        if ($cost === null) {
                            $cost = $amount * $price;
                        }
                    } else if ($cost === null) {
                        throw new InvalidOrder($this->id . " createOrder() requires the $price argument for $market buy orders to calculate total $order $cost ($amount to spend), where $cost = $amount * $price-> Supply a $price argument to createOrder() call if you want the $cost to be calculated for you from $price and $amount, or, alternatively, add .options['createMarketBuyOrderRequiresPrice'] = false and supply the total $cost value in the 'amount' argument or in the 'cost' extra parameter (the exchange-specific behaviour)");
                    }
                } else {
                    $cost = ($cost === null) ? $amount : $cost;
                }
                $costToPrecision = $this->cost_to_precision($symbol, $cost);
                $request['cost'] = $costToPrecision;
            } else {
                $request['quantity'] = $this->amount_to_precision($symbol, $amount);
            }
        }
        $query = $this->omit($params, array( 'timeInForce', 'time_in_force', 'clientOrderId', 'client_order_id' ));
        $response = $this->privatePostNewOrder (array_merge($request, $query));
        //
        //     {
        //         $data => {
        //             id => string,
        //             user_id => string,
        //             market_id => string,
        //             $type => 'orderType',
        //             $side => 'side',
        //             quantity => string,
        //             limit_price => string,
        //             time_in_force => 'timeInForce',
        //             filled_cost => string,
        //             filled_quantity => string,
        //             open_quantity => string,
        //             cancelled_quantity => string,
        //             status => 'orderStatus',
        //             time => 'date',
        //             client_order_id => string,
        //         }
        //     }
        //
        $data = $this->safe_value($response, 'data');
        $order = $this->parse_order($data, $market);
        // a workaround for incorrect huge amounts
        // returned by the exchange on $market buys
        if (($type === 'market') && ($side === 'buy')) {
            $order['amount'] = null;
            $order['cost'] = floatval ($costToPrecision);
            $order['remaining'] = null;
        }
        return $order;
    }

    public function cancel_order($id, $symbol = null, $params = array ()) {
        if ($symbol === null) {
            throw new ArgumentsRequired($this->id . ' cancelOrder requires a $symbol argument');
        }
        $this->load_markets();
        $market = $this->market($symbol);
        $request = array(
            'market_id' => $market['id'],
            'order_id' => $id,
        );
        $response = $this->privatePostCancelOrder (array_merge($request, $params));
        $data = $this->safe_value($response, 'data');
        return $this->parse_order($data);
    }

    public function parse_deposit_address($depositAddress, $currency = null) {
        $address = $this->safe_string($depositAddress, 'address');
        $tag = $this->safe_string($depositAddress, 'destination_tag');
        $currencyId = $this->safe_string($depositAddress, 'currency_id');
        $code = $this->safe_currency_code($currencyId);
        $this->check_address($address);
        return array(
            'currency' => $code,
            'address' => $address,
            'tag' => $tag,
            'info' => $depositAddress,
        );
    }

    public function fetch_deposit_address($code, $params = array ()) {
        $this->load_markets();
        $currency = $this->currency($code);
        $request = array(
            'currency_id' => $currency['id'],
        );
        $response = $this->privateGetDepositAddress (array_merge($request, $params));
        //
        //     {
        //         "$data":array(
        //             {
        //                 "currency_id":"ETH",
        //                 "address":"0x12e2caf3c4051ba1146e612f532901a423a9898a",
        //                 "destination_tag":null
        //             }
        //         )
        //     }
        //
        $data = $this->safe_value($response, 'data', array());
        $firstAddress = $this->safe_value($data, 0);
        if ($firstAddress === null) {
            throw new InvalidAddress($this->id . ' fetchDepositAddress returned an empty response');
        }
        return $this->parse_deposit_address($firstAddress, $currency);
    }

    public function fetch_deposit_addresses($codes = null, $params = array ()) {
        $this->load_markets();
        $request = array();
        if ($codes) {
            $currencyIds = array();
            for ($i = 0; $i < count($codes); $i++) {
                $currency = $this->currency($codes[$i]);
                $currencyIds[] = $currency['id'];
            }
            $request['currency_id'] = implode(',', $codes);
        }
        $response = $this->privateGetDepositAddress (array_merge($request, $params));
        $data = $this->safe_value($response, 'data', array());
        return $this->parse_deposit_addresses($data);
    }

    public function parse_deposit_addresses($addresses) {
        $result = array();
        for ($i = 0; $i < count($addresses); $i++) {
            $address = $this->parse_deposit_address($addresses[$i]);
            $code = $address['currency'];
            $result[$code] = $address;
        }
        return $result;
    }

    public function withdraw($code, $amount, $address, $tag = null, $params = array ()) {
        // In order to use this method
        // you need to allow API withdrawal from the API Settings Page, and
        // and register the list of withdrawal addresses and destination tags on the API Settings page
        // you can only withdraw to the registered addresses using the API
        $this->check_address($address);
        $this->load_markets();
        $currency = $this->currency($code);
        if ($tag === null) {
            $tag = '';
        }
        $request = array(
            'currency_id' => $currency['id'],
            // 'platform_id' => 'ETH', // if omitted it will use the default platform for the $currency
            'address' => $address,
            'destination_tag' => $tag,
            'amount' => $this->currency_to_precision($code, $amount),
            // which $currency to pay the withdrawal fees
            // only applicable for currencies that accepts multiple withdrawal fee options
            // 'fee_currency_id' => 'ETH', // if omitted it will use the default fee policy for each $currency
            // whether the $amount field includes fees
            // 'include_fee' => false, // makes sense only when fee_currency_id is equal to currency_id
        );
        $response = $this->privatePostWithdrawal (array_merge($request, $params));
        $data = $this->safe_value($response, 'data');
        return $this->parse_transaction($data, $currency);
    }

    public function parse_transaction($transaction, $currency = null) {
        $id = $this->safe_string($transaction, 'id');
        $amount = $this->safe_float($transaction, 'amount');
        $address = $this->safe_string($transaction, 'address');
        $tag = $this->safe_string($transaction, 'destination_tag');
        $txid = $this->safe_string($transaction, 'hash');
        $timestamp = $this->parse8601($this->safe_string($transaction, 'time'));
        $type = $this->safe_string($transaction, 'type');
        $currencyId = $this->safe_string($transaction, 'currency_id');
        $code = $this->safe_currency_code($currencyId);
        $status = $this->parse_transaction_status($this->safe_string($transaction, 'status'));
        $feeCost = $this->safe_float($transaction, 'fee');
        $fee = null;
        if ($feeCost !== null && $feeCost !== 0) {
            $fee = array(
                'currency' => $code,
                'cost' => $feeCost,
            );
        }
        return array(
            'id' => $id,
            'currency' => $code,
            'amount' => $amount,
            'addressFrom' => null,
            'address' => $address,
            'addressTo' => $address,
            'tagFrom' => null,
            'tag' => $tag,
            'tagTo' => $tag,
            'status' => $status,
            'type' => $type,
            'txid' => $txid,
            'timestamp' => $timestamp,
            'datetime' => $this->iso8601($timestamp),
            'fee' => $fee,
            'info' => $transaction,
        );
    }

    public function parse_transaction_status($status) {
        $statuses = array(
            'requested' => 'pending',
            'pending' => 'pending',
            'confirming' => 'pending',
            'confirmed' => 'pending',
            'applying' => 'pending',
            'done' => 'ok',
            'cancelled' => 'canceled',
            'cancelling' => 'canceled',
        );
        return $this->safe_string($statuses, $status, $status);
    }

    public function nonce() {
        return $this->milliseconds();
    }

    public function sign($path, $api = 'public', $method = 'GET', $params = array (), $headers = null, $body = null) {
        $url = $this->urls['api'][$api] . '/';
        $query = $this->omit($params, $this->extract_params($path));
        if ($api === 'accounts') {
            $this->check_required_credentials();
            $url .= $this->implode_params($path, $params);
            $auth = $this->apiKey . ':' . $this->secret;
            $auth64 = base64_encode($this->encode($auth));
            $headers = array(
                'Authorization' => 'Basic ' . $this->decode($auth64),
                'Content-Type' => 'application/json',
            );
            if ($query) {
                $body = $this->json($query);
            }
        } else {
            $url .= $this->version . '/';
            if ($api === 'public') {
                $url .= $this->implode_params($path, $params);
                if ($query) {
                    $url .= '?' . $this->urlencode($query);
                }
            } else if ($api === 'private') {
                $now = $this->milliseconds();
                $this->check_required_credentials();
                $expires = $this->safe_integer($this->options, 'expires');
                if (($expires === null) || ($expires < $now)) {
                    throw new AuthenticationError($this->id . ' $accessToken expired, call signIn() method');
                }
                $accessToken = $this->safe_string($this->options, 'accessToken');
                $headers = array(
                    'Authorization' => 'Bearer ' . $accessToken,
                );
                $url .= $this->implode_params($path, $params);
                if ($method === 'GET') {
                    if ($query) {
                        $url .= '?' . $this->urlencode($query);
                    }
                } else if ($query) {
                    $body = $this->json($query);
                    $headers['Content-Type'] = 'application/json';
                }
            }
        }
        return array( 'url' => $url, 'method' => $method, 'body' => $body, 'headers' => $headers );
    }

    public function sign_in($params = array ()) {
        $this->check_required_credentials();
        $request = array(
            'grant_type' => 'client_credentials', // the only supported value
        );
        $response = $this->accountsPostToken (array_merge($request, $params));
        //
        //     {
        //         access_token => '0ttDv/2hTTn3bLi8GP1gKaneiEQ6+0hOBenPrxNQt2s=',
        //         token_type => 'bearer',
        //         expires_in => 900
        //     }
        //
        $expiresIn = $this->safe_integer($response, 'expires_in');
        $accessToken = $this->safe_string($response, 'access_token');
        $this->options['accessToken'] = $accessToken;
        $this->options['expires'] = $this->sum($this->milliseconds(), $expiresIn * 1000);
        return $response;
    }

    public function handle_errors($code, $reason, $url, $method, $headers, $body, $response, $requestHeaders, $requestBody) {
        if ($response === null) {
            return; // fallback to default error handler
        }
        if (is_array($response) && array_key_exists('errorCode', $response)) {
            $errorCode = $this->safe_string($response, 'errorCode');
            $message = $this->safe_string($response, 'message');
            if ($errorCode !== null) {
                $feedback = $this->id . ' ' . $body;
                $this->throw_exactly_matched_exception($this->exceptions['exact'], $message, $feedback);
                $this->throw_broadly_matched_exception($this->exceptions['exact'], $errorCode, $feedback);
                throw new ExchangeError($feedback);
            }
        }
    }
}
