/*********************************************************************/
/* Copyright (c) 2016, EPFL/Blue Brain Project                       */
/*                     Raphael Dumusc <raphael.dumusc@epfl.ch>       */
/* All rights reserved.                                              */
/*                                                                   */
/* Redistribution and use in source and binary forms, with or        */
/* without modification, are permitted provided that the following   */
/* conditions are met:                                               */
/*                                                                   */
/*   1. Redistributions of source code must retain the above         */
/*      copyright notice, this list of conditions and the following  */
/*      disclaimer.                                                  */
/*                                                                   */
/*   2. Redistributions in binary form must reproduce the above      */
/*      copyright notice, this list of conditions and the following  */
/*      disclaimer in the documentation and/or other materials       */
/*      provided with the distribution.                              */
/*                                                                   */
/*    THIS  SOFTWARE IS PROVIDED  BY THE  UNIVERSITY OF  TEXAS AT    */
/*    AUSTIN  ``AS IS''  AND ANY  EXPRESS OR  IMPLIED WARRANTIES,    */
/*    INCLUDING, BUT  NOT LIMITED  TO, THE IMPLIED  WARRANTIES OF    */
/*    MERCHANTABILITY  AND FITNESS FOR  A PARTICULAR  PURPOSE ARE    */
/*    DISCLAIMED.  IN  NO EVENT SHALL THE UNIVERSITY  OF TEXAS AT    */
/*    AUSTIN OR CONTRIBUTORS BE  LIABLE FOR ANY DIRECT, INDIRECT,    */
/*    INCIDENTAL,  SPECIAL, EXEMPLARY,  OR  CONSEQUENTIAL DAMAGES    */
/*    (INCLUDING, BUT  NOT LIMITED TO,  PROCUREMENT OF SUBSTITUTE    */
/*    GOODS  OR  SERVICES; LOSS  OF  USE,  DATA,  OR PROFITS;  OR    */
/*    BUSINESS INTERRUPTION) HOWEVER CAUSED  AND ON ANY THEORY OF    */
/*    LIABILITY, WHETHER  IN CONTRACT, STRICT  LIABILITY, OR TORT    */
/*    (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY WAY OUT    */
/*    OF  THE  USE OF  THIS  SOFTWARE,  EVEN  IF ADVISED  OF  THE    */
/*    POSSIBILITY OF SUCH DAMAGE.                                    */
/*                                                                   */
/* The views and conclusions contained in the software and           */
/* documentation are those of the authors and should not be          */
/* interpreted as representing official policies, either expressed   */
/* or implied, of Ecole polytechnique federale de Lausanne.          */
/*********************************************************************/

#include "RestCommand.h"

#include "jsonschema.h"
#include "log.h"

#include <QJsonDocument>
#include <QJsonObject>
#include <QStringList>

namespace
{
const QString description = "Command '%1' of Tide application";
}

std::string _makeSchema( const std::string& name, const bool takesValue )
{
    const auto data = QString::fromStdString( name ).split( '/' );
    if( data.size() != 2 )
        return "";

    QJsonObject obj;
    if( takesValue )
        obj["uri"] = QString();

    return jsonschema::create( data[1], obj, description.arg( data[1] ));
}

RestCommand::RestCommand( const std::string& name, const bool takesValue )
    : _name{ name }
    , _schema{ _makeSchema( name, takesValue ) }
    , _takesValue( takesValue )
{}

std::string RestCommand::getTypeName() const
{
    return _name;
}

std::string RestCommand::getSchema() const
{
    return _schema;
}

bool RestCommand::_fromJSON( const std::string& string )
{
    const QByteArray input = QString::fromStdString( string ).toUtf8();
    const QJsonDocument doc = QJsonDocument::fromJson( input );
    if( doc.isNull() || !doc.isObject( ))
    {
        put_flog( LOG_INFO, "Error parsing JSON string: '%s'", string.c_str( ));
        return false;
    }

    if( !_takesValue )
    {
        emit received( "" );
        return true;
    }

    const QJsonValue value = doc.object()["uri"];
    if( !value.isString( ))
        return false;

    emit received( value.toString( ));
    return true;
}
