/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.v3.map.reader;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.mapsforge.v3.core.MercatorProjection;
import org.mapsforge.v3.core.Tag;
import org.mapsforge.v3.core.Tile;
import org.mapsforge.v3.map.reader.IndexCache;
import org.mapsforge.v3.map.reader.MapDatabaseCallback;
import org.mapsforge.v3.map.reader.QueryCalculations;
import org.mapsforge.v3.map.reader.QueryParameters;
import org.mapsforge.v3.map.reader.ReadBuffer;
import org.mapsforge.v3.map.reader.header.FileOpenResult;
import org.mapsforge.v3.map.reader.header.MapFileHeader;
import org.mapsforge.v3.map.reader.header.MapFileInfo;
import org.mapsforge.v3.map.reader.header.SubFileParameter;

public class MapDatabase {
    private static final long BITMASK_INDEX_OFFSET = 0x7FFFFFFFFFL;
    private static final long BITMASK_INDEX_WATER = 0x8000000000L;
    private static final String DEBUG_SIGNATURE_BLOCK = "block signature: ";
    private static final String DEBUG_SIGNATURE_POI = "POI signature: ";
    private static final String DEBUG_SIGNATURE_WAY = "way signature: ";
    private static final int INDEX_CACHE_SIZE = 64;
    private static final String INVALID_FIRST_WAY_OFFSET = "invalid first way offset: ";
    private static final Logger LOG = Logger.getLogger(MapDatabase.class.getName());
    private static final int MAXIMUM_WAY_NODES_SEQUENCE_LENGTH = 8192;
    private static final int MAXIMUM_ZOOM_TABLE_OBJECTS = 65536;
    private static final int POI_FEATURE_ELEVATION = 32;
    private static final int POI_FEATURE_HOUSE_NUMBER = 64;
    private static final int POI_FEATURE_NAME = 128;
    private static final int POI_LAYER_BITMASK = 240;
    private static final int POI_LAYER_SHIFT = 4;
    private static final int POI_NUMBER_OF_TAGS_BITMASK = 15;
    private static final String READ_ONLY_MODE = "r";
    private static final byte SIGNATURE_LENGTH_BLOCK = 32;
    private static final byte SIGNATURE_LENGTH_POI = 32;
    private static final byte SIGNATURE_LENGTH_WAY = 32;
    private static final String TAG_KEY_ELE = "ele";
    private static final String TAG_KEY_HOUSE_NUMBER = "addr:housenumber";
    private static final String TAG_KEY_NAME = "name";
    private static final String TAG_KEY_REF = "ref";
    private static final int WAY_FEATURE_DATA_BLOCKS_BYTE = 8;
    private static final int WAY_FEATURE_DOUBLE_DELTA_ENCODING = 4;
    private static final int WAY_FEATURE_HOUSE_NUMBER = 64;
    private static final int WAY_FEATURE_LABEL_POSITION = 16;
    private static final int WAY_FEATURE_NAME = 128;
    private static final int WAY_FEATURE_REF = 32;
    private static final int WAY_LAYER_BITMASK = 240;
    private static final int WAY_LAYER_SHIFT = 4;
    private static final int WAY_NUMBER_OF_TAGS_BITMASK = 15;
    private IndexCache databaseIndexCache;
    private long fileSize;
    private RandomAccessFile inputFile;
    private MapFileHeader mapFileHeader;
    private ReadBuffer readBuffer;
    private String signatureBlock;
    private String signaturePoi;
    private String signatureWay;
    private int tileLatitude;
    private int tileLongitude;

    public void closeFile() {
        try {
            this.mapFileHeader = null;
            if (this.databaseIndexCache != null) {
                this.databaseIndexCache.destroy();
                this.databaseIndexCache = null;
            }
            if (this.inputFile != null) {
                this.inputFile.close();
                this.inputFile = null;
            }
            this.readBuffer = null;
        }
        catch (IOException e) {
            LOG.log(Level.SEVERE, null, e);
        }
    }

    public void executeQuery(Tile tile, MapDatabaseCallback mapDatabaseCallback) {
        try {
            this.prepareExecution();
            if (this.mapFileHeader == null) {
                return;
            }
            QueryParameters queryParameters = new QueryParameters();
            queryParameters.queryZoomLevel = this.mapFileHeader.getQueryZoomLevel(tile.zoomLevel);
            SubFileParameter subFileParameter = this.mapFileHeader.getSubFileParameter(queryParameters.queryZoomLevel);
            if (subFileParameter == null) {
                LOG.warning("no sub-file for zoom level: " + queryParameters.queryZoomLevel);
                return;
            }
            QueryCalculations.calculateBaseTiles(queryParameters, tile, subFileParameter);
            QueryCalculations.calculateBlocks(queryParameters, subFileParameter);
            this.processBlocks(mapDatabaseCallback, queryParameters, subFileParameter);
        }
        catch (IOException e) {
            LOG.log(Level.SEVERE, null, e);
        }
    }

    public MapFileInfo getMapFileInfo() {
        if (this.mapFileHeader == null) {
            throw new IllegalStateException("no map file is currently opened");
        }
        return this.mapFileHeader.getMapFileInfo();
    }

    public boolean hasOpenFile() {
        return this.inputFile != null;
    }

    public FileOpenResult openFile(File mapFile) {
        try {
            if (mapFile == null) {
                throw new IllegalArgumentException("mapFile must not be null");
            }
            this.closeFile();
            if (!mapFile.exists()) {
                return new FileOpenResult("file does not exist: " + mapFile);
            }
            if (!mapFile.isFile()) {
                return new FileOpenResult("not a file: " + mapFile);
            }
            if (!mapFile.canRead()) {
                return new FileOpenResult("cannot read file: " + mapFile);
            }
            this.inputFile = new RandomAccessFile(mapFile, READ_ONLY_MODE);
            this.fileSize = this.inputFile.length();
            this.readBuffer = new ReadBuffer(this.inputFile);
            this.mapFileHeader = new MapFileHeader();
            FileOpenResult fileOpenResult = this.mapFileHeader.readHeader(this.readBuffer, this.fileSize);
            if (!fileOpenResult.isSuccess()) {
                this.closeFile();
                return fileOpenResult;
            }
            return FileOpenResult.SUCCESS;
        }
        catch (IOException e) {
            LOG.log(Level.SEVERE, null, e);
            this.closeFile();
            return new FileOpenResult(e.getMessage());
        }
    }

    private void decodeWayNodesDoubleDelta(float[] waySegment) {
        int wayNodeLatitude = this.tileLatitude + this.readBuffer.readSignedInt();
        int wayNodeLongitude = this.tileLongitude + this.readBuffer.readSignedInt();
        waySegment[1] = wayNodeLatitude;
        waySegment[0] = wayNodeLongitude;
        int previousSingleDeltaLatitude = 0;
        int previousSingleDeltaLongitude = 0;
        for (int wayNodesIndex = 2; wayNodesIndex < waySegment.length; wayNodesIndex += 2) {
            int doubleDeltaLatitude = this.readBuffer.readSignedInt();
            int doubleDeltaLongitude = this.readBuffer.readSignedInt();
            int singleDeltaLatitude = doubleDeltaLatitude + previousSingleDeltaLatitude;
            int singleDeltaLongitude = doubleDeltaLongitude + previousSingleDeltaLongitude;
            waySegment[wayNodesIndex + 1] = wayNodeLatitude += singleDeltaLatitude;
            waySegment[wayNodesIndex] = wayNodeLongitude += singleDeltaLongitude;
            previousSingleDeltaLatitude = singleDeltaLatitude;
            previousSingleDeltaLongitude = singleDeltaLongitude;
        }
    }

    private void decodeWayNodesSingleDelta(float[] waySegment) {
        int wayNodeLatitude = this.tileLatitude + this.readBuffer.readSignedInt();
        int wayNodeLongitude = this.tileLongitude + this.readBuffer.readSignedInt();
        waySegment[1] = wayNodeLatitude;
        waySegment[0] = wayNodeLongitude;
        for (int wayNodesIndex = 2; wayNodesIndex < waySegment.length; wayNodesIndex += 2) {
            waySegment[wayNodesIndex + 1] = wayNodeLatitude += this.readBuffer.readSignedInt();
            waySegment[wayNodesIndex] = wayNodeLongitude += this.readBuffer.readSignedInt();
        }
    }

    private void logDebugSignatures() {
        if (this.mapFileHeader.getMapFileInfo().debugFile) {
            LOG.warning(DEBUG_SIGNATURE_WAY + this.signatureWay);
            LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
        }
    }

    private void prepareExecution() {
        if (this.databaseIndexCache == null) {
            this.databaseIndexCache = new IndexCache(this.inputFile, 64);
        }
    }

    private void processBlock(QueryParameters queryParameters, SubFileParameter subFileParameter, MapDatabaseCallback mapDatabaseCallback) {
        if (!this.processBlockSignature()) {
            return;
        }
        int[][] zoomTable = this.readZoomTable(subFileParameter);
        if (zoomTable == null) {
            return;
        }
        int zoomTableRow = queryParameters.queryZoomLevel - subFileParameter.zoomLevelMin;
        int poisOnQueryZoomLevel = zoomTable[zoomTableRow][0];
        int waysOnQueryZoomLevel = zoomTable[zoomTableRow][1];
        int firstWayOffset = this.readBuffer.readUnsignedInt();
        if (firstWayOffset < 0) {
            LOG.warning(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            if (this.mapFileHeader.getMapFileInfo().debugFile) {
                LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
            }
            return;
        }
        if ((firstWayOffset += this.readBuffer.getBufferPosition()) > this.readBuffer.getBufferSize()) {
            LOG.warning(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            if (this.mapFileHeader.getMapFileInfo().debugFile) {
                LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
            }
            return;
        }
        if (!this.processPOIs(mapDatabaseCallback, poisOnQueryZoomLevel)) {
            return;
        }
        if (this.readBuffer.getBufferPosition() > firstWayOffset) {
            LOG.warning("invalid buffer position: " + this.readBuffer.getBufferPosition());
            if (this.mapFileHeader.getMapFileInfo().debugFile) {
                LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
            }
            return;
        }
        this.readBuffer.setBufferPosition(firstWayOffset);
        if (!this.processWays(queryParameters, mapDatabaseCallback, waysOnQueryZoomLevel)) {
            return;
        }
    }

    private void processBlocks(MapDatabaseCallback mapDatabaseCallback, QueryParameters queryParameters, SubFileParameter subFileParameter) throws IOException {
        boolean queryIsWater = true;
        boolean queryReadWaterInfo = false;
        for (long row = queryParameters.fromBlockY; row <= queryParameters.toBlockY; ++row) {
            for (long column = queryParameters.fromBlockX; column <= queryParameters.toBlockX; ++column) {
                long nextBlockPointer;
                long currentBlockPointer;
                long blockNumber = row * subFileParameter.blocksWidth + column;
                long currentBlockIndexEntry = this.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber);
                if (queryIsWater) {
                    queryIsWater &= (currentBlockIndexEntry & 0x8000000000L) != 0L;
                    queryReadWaterInfo = true;
                }
                if ((currentBlockPointer = currentBlockIndexEntry & 0x7FFFFFFFFFL) < 1L || currentBlockPointer > subFileParameter.subFileSize) {
                    LOG.warning("invalid current block pointer: " + currentBlockPointer);
                    LOG.warning("subFileSize: " + subFileParameter.subFileSize);
                    return;
                }
                if (blockNumber + 1L == subFileParameter.numberOfBlocks) {
                    nextBlockPointer = subFileParameter.subFileSize;
                } else {
                    nextBlockPointer = this.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber + 1L) & 0x7FFFFFFFFFL;
                    if (nextBlockPointer < 1L || nextBlockPointer > subFileParameter.subFileSize) {
                        LOG.warning("invalid next block pointer: " + nextBlockPointer);
                        LOG.warning("sub-file size: " + subFileParameter.subFileSize);
                        return;
                    }
                }
                int currentBlockSize = (int)(nextBlockPointer - currentBlockPointer);
                if (currentBlockSize < 0) {
                    LOG.warning("current block size must not be negative: " + currentBlockSize);
                    return;
                }
                if (currentBlockSize == 0) continue;
                if (currentBlockSize > 2500000) {
                    LOG.warning("current block size too large: " + currentBlockSize);
                    continue;
                }
                if (currentBlockPointer + (long)currentBlockSize > this.fileSize) {
                    LOG.warning("current block largher than file size: " + currentBlockSize);
                    return;
                }
                this.inputFile.seek(subFileParameter.startAddress + currentBlockPointer);
                if (!this.readBuffer.readFromFile(currentBlockSize)) {
                    LOG.warning("reading current block has failed: " + currentBlockSize);
                    return;
                }
                double tileLatitudeDeg = MercatorProjection.tileYToLatitude(subFileParameter.boundaryTileTop + row, subFileParameter.baseZoomLevel);
                double tileLongitudeDeg = MercatorProjection.tileXToLongitude(subFileParameter.boundaryTileLeft + column, subFileParameter.baseZoomLevel);
                this.tileLatitude = (int)(tileLatitudeDeg * 1000000.0);
                this.tileLongitude = (int)(tileLongitudeDeg * 1000000.0);
                try {
                    this.processBlock(queryParameters, subFileParameter, mapDatabaseCallback);
                    continue;
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    LOG.log(Level.SEVERE, null, e);
                }
            }
        }
        if (queryIsWater && queryReadWaterInfo) {
            mapDatabaseCallback.renderWaterBackground();
        }
    }

    private boolean processBlockSignature() {
        if (this.mapFileHeader.getMapFileInfo().debugFile) {
            this.signatureBlock = this.readBuffer.readUTF8EncodedString(32);
            if (!this.signatureBlock.startsWith("###TileStart")) {
                LOG.warning("invalid block signature: " + this.signatureBlock);
                return false;
            }
        }
        return true;
    }

    private boolean processPOIs(MapDatabaseCallback mapDatabaseCallback, int numberOfPois) {
        ArrayList<Tag> tags = new ArrayList<Tag>();
        Tag[] poiTags = this.mapFileHeader.getMapFileInfo().poiTags;
        for (int elementCounter = numberOfPois; elementCounter != 0; --elementCounter) {
            boolean featureElevation;
            if (this.mapFileHeader.getMapFileInfo().debugFile) {
                this.signaturePoi = this.readBuffer.readUTF8EncodedString(32);
                if (!this.signaturePoi.startsWith("***POIStart")) {
                    LOG.warning("invalid POI signature: " + this.signaturePoi);
                    LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                    return false;
                }
            }
            int latitude = this.tileLatitude + this.readBuffer.readSignedInt();
            int longitude = this.tileLongitude + this.readBuffer.readSignedInt();
            byte specialByte = this.readBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            tags.clear();
            for (byte tagIndex = numberOfTags; tagIndex != 0; tagIndex = (byte)(tagIndex - 1)) {
                int tagId = this.readBuffer.readUnsignedInt();
                if (tagId < 0 || tagId >= poiTags.length) {
                    LOG.warning("invalid POI tag ID: " + tagId);
                    if (this.mapFileHeader.getMapFileInfo().debugFile) {
                        LOG.warning(DEBUG_SIGNATURE_POI + this.signaturePoi);
                        LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                    }
                    return false;
                }
                tags.add(poiTags[tagId]);
            }
            byte featureByte = this.readBuffer.readByte();
            boolean featureName = (featureByte & 0x80) != 0;
            boolean featureHouseNumber = (featureByte & 0x40) != 0;
            boolean bl = featureElevation = (featureByte & 0x20) != 0;
            if (featureName) {
                tags.add(new Tag(TAG_KEY_NAME, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureHouseNumber) {
                tags.add(new Tag(TAG_KEY_HOUSE_NUMBER, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureElevation) {
                tags.add(new Tag(TAG_KEY_ELE, Integer.toString(this.readBuffer.readSignedInt())));
            }
            mapDatabaseCallback.renderPointOfInterest(layer, latitude, longitude, tags);
        }
        return true;
    }

    private float[][] processWayDataBlock(boolean doubleDeltaEncoding) {
        int numberOfWayCoordinateBlocks = this.readBuffer.readUnsignedInt();
        if (numberOfWayCoordinateBlocks < 1 || numberOfWayCoordinateBlocks > Short.MAX_VALUE) {
            LOG.warning("invalid number of way coordinate blocks: " + numberOfWayCoordinateBlocks);
            this.logDebugSignatures();
            return null;
        }
        float[][] wayCoordinates = new float[numberOfWayCoordinateBlocks][];
        for (int coordinateBlock = 0; coordinateBlock < numberOfWayCoordinateBlocks; ++coordinateBlock) {
            int numberOfWayNodes = this.readBuffer.readUnsignedInt();
            if (numberOfWayNodes < 2 || numberOfWayNodes > 8192) {
                LOG.warning("invalid number of way nodes: " + numberOfWayNodes);
                this.logDebugSignatures();
                return null;
            }
            int wayNodesSequenceLength = numberOfWayNodes * 2;
            float[] waySegment = new float[wayNodesSequenceLength];
            if (doubleDeltaEncoding) {
                this.decodeWayNodesDoubleDelta(waySegment);
            } else {
                this.decodeWayNodesSingleDelta(waySegment);
            }
            wayCoordinates[coordinateBlock] = waySegment;
        }
        return wayCoordinates;
    }

    private boolean processWays(QueryParameters queryParameters, MapDatabaseCallback mapDatabaseCallback, int numberOfWays) {
        ArrayList<Tag> tags = new ArrayList<Tag>();
        Tag[] wayTags = this.mapFileHeader.getMapFileInfo().wayTags;
        for (int elementCounter = numberOfWays; elementCounter != 0; --elementCounter) {
            boolean featureWayDoubleDeltaEncoding;
            int wayDataSize;
            if (this.mapFileHeader.getMapFileInfo().debugFile) {
                this.signatureWay = this.readBuffer.readUTF8EncodedString(32);
                if (!this.signatureWay.startsWith("---WayStart")) {
                    LOG.warning("invalid way signature: " + this.signatureWay);
                    LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                    return false;
                }
            }
            if ((wayDataSize = this.readBuffer.readUnsignedInt()) < 0) {
                LOG.warning("invalid way data size: " + wayDataSize);
                if (this.mapFileHeader.getMapFileInfo().debugFile) {
                    LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                }
                return false;
            }
            if (queryParameters.useTileBitmask) {
                int tileBitmask = this.readBuffer.readShort();
                if ((queryParameters.queryTileBitmask & tileBitmask) == 0) {
                    this.readBuffer.skipBytes(wayDataSize - 2);
                    continue;
                }
            } else {
                this.readBuffer.skipBytes(2);
            }
            byte specialByte = this.readBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            tags.clear();
            for (byte tagIndex = numberOfTags; tagIndex != 0; tagIndex = (byte)(tagIndex - 1)) {
                int tagId = this.readBuffer.readUnsignedInt();
                if (tagId < 0 || tagId >= wayTags.length) {
                    LOG.warning("invalid way tag ID: " + tagId);
                    this.logDebugSignatures();
                    return false;
                }
                tags.add(wayTags[tagId]);
            }
            byte featureByte = this.readBuffer.readByte();
            boolean featureName = (featureByte & 0x80) != 0;
            boolean featureHouseNumber = (featureByte & 0x40) != 0;
            boolean featureRef = (featureByte & 0x20) != 0;
            boolean featureLabelPosition = (featureByte & 0x10) != 0;
            boolean featureWayDataBlocksByte = (featureByte & 8) != 0;
            boolean bl = featureWayDoubleDeltaEncoding = (featureByte & 4) != 0;
            if (featureName) {
                tags.add(new Tag(TAG_KEY_NAME, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureHouseNumber) {
                tags.add(new Tag(TAG_KEY_HOUSE_NUMBER, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureRef) {
                tags.add(new Tag(TAG_KEY_REF, this.readBuffer.readUTF8EncodedString()));
            }
            float[] labelPosition = this.readOptionalLabelPosition(featureLabelPosition);
            int wayDataBlocks = this.readOptionalWayDataBlocksByte(featureWayDataBlocksByte);
            if (wayDataBlocks < 1) {
                LOG.warning("invalid number of way data blocks: " + wayDataBlocks);
                this.logDebugSignatures();
                return false;
            }
            for (int wayDataBlock = 0; wayDataBlock < wayDataBlocks; ++wayDataBlock) {
                float[][] wayNodes = this.processWayDataBlock(featureWayDoubleDeltaEncoding);
                if (wayNodes == null) {
                    return false;
                }
                mapDatabaseCallback.renderWay(layer, labelPosition, tags, wayNodes);
            }
        }
        return true;
    }

    private float[] readOptionalLabelPosition(boolean featureLabelPosition) {
        float[] labelPosition = null;
        if (featureLabelPosition) {
            labelPosition = new float[2];
            labelPosition[1] = this.tileLatitude + this.readBuffer.readSignedInt();
            labelPosition[0] = this.tileLongitude + this.readBuffer.readSignedInt();
        }
        return labelPosition;
    }

    private int readOptionalWayDataBlocksByte(boolean featureWayDataBlocksByte) {
        if (featureWayDataBlocksByte) {
            return this.readBuffer.readUnsignedInt();
        }
        return 1;
    }

    private int[][] readZoomTable(SubFileParameter subFileParameter) {
        int rows = subFileParameter.zoomLevelMax - subFileParameter.zoomLevelMin + 1;
        int[][] zoomTable = new int[rows][2];
        int cumulatedNumberOfPois = 0;
        int cumulatedNumberOfWays = 0;
        for (int row = 0; row < rows; ++row) {
            cumulatedNumberOfWays += this.readBuffer.readUnsignedInt();
            if ((cumulatedNumberOfPois += this.readBuffer.readUnsignedInt()) < 0 || cumulatedNumberOfPois > 65536) {
                LOG.warning("invalid cumulated number of POIs in row " + row + ' ' + cumulatedNumberOfPois);
                if (this.mapFileHeader.getMapFileInfo().debugFile) {
                    LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                }
                return null;
            }
            if (cumulatedNumberOfWays < 0 || cumulatedNumberOfWays > 65536) {
                LOG.warning("invalid cumulated number of ways in row " + row + ' ' + cumulatedNumberOfWays);
                if (this.mapFileHeader.getMapFileInfo().debugFile) {
                    LOG.warning(DEBUG_SIGNATURE_BLOCK + this.signatureBlock);
                }
                return null;
            }
            zoomTable[row][0] = cumulatedNumberOfPois;
            zoomTable[row][1] = cumulatedNumberOfWays;
        }
        return zoomTable;
    }
}

