﻿using System.Collections;
using Beatmap.Base;
using Beatmap.Enums;
using Beatmap.Shared;
using Beatmap.V2;
using Beatmap.V3;
using NUnit.Framework;
using SimpleJSON;
using Tests.Util;
using UnityEngine;
using UnityEngine.TestTools;

namespace Tests
{
    public class PaintTest
    {
        [UnityOneTimeSetUp]
        public IEnumerator LoadMap()
        {
            return TestUtils.LoadMap(3);
        }

        [OneTimeTearDown]
        public void FinalTearDown()
        {
            TestUtils.ReturnSettings();
        }

        [SetUp]
        public void SetUp()
        {
            Settings.Instance.MapVersion = 3;
        }

        [TearDown]
        public void ContainerCleanup()
        {
            BeatmapActionContainer.RemoveAllActionsOfType<BeatmapAction>();
            CleanupUtils.CleanupEvents();
        }

        [Test]
        public void PaintGradientUndo()
        {
            Settings.Instance.MapVersion = 2;

            var actionContainer = Object.FindObjectOfType<BeatmapActionContainer>();
            var colorPicker = Object.FindObjectOfType<ColorPicker>();
            var painter = Object.FindObjectOfType<PaintSelectedObjects>();

            var eventsContainer = BeatmapObjectContainerCollection.GetCollectionForType<EventGridContainer>(ObjectType.Event);
            var root = eventsContainer.transform.root;

            var selectionController = root.GetComponentInChildren<SelectionController>();
            var eventPlacement = root.GetComponentInChildren<EventPlacement>();

            var customData = new JSONObject();
            customData["_lightGradient"] = new ChromaLightGradient(Color.blue, Color.cyan).ToJson();
            BaseEvent baseEventA = new BaseEvent { JsonTime = 2, Type = 1, Value = 1, FloatValue = 1, CustomData = customData };
            PlaceUtils.PlaceEvent(eventPlacement, baseEventA);

            SelectionController.Select(baseEventA);

            colorPicker.CurrentColor = Color.red;
            painter.Paint();

            selectionController.ShiftSelection(1, 0);

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(2, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(Color.red, ((BaseEvent)eventsContainer.MapObjects[0]).CustomLightGradient.StartColor);

            // Undo move
            actionContainer.Undo();

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(1, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(Color.red, ((BaseEvent)eventsContainer.MapObjects[0]).CustomLightGradient.StartColor);

            // Undo paint
            actionContainer.Undo();

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(1, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(Color.blue, ((BaseEvent)eventsContainer.MapObjects[0]).CustomLightGradient.StartColor);
        }

        [Test]
        public void PaintUndo()
        {
            var actionContainer = Object.FindObjectOfType<BeatmapActionContainer>();
            var colorPicker = Object.FindObjectOfType<ColorPicker>();
            var painter = Object.FindObjectOfType<PaintSelectedObjects>();

            var eventsContainer = BeatmapObjectContainerCollection.GetCollectionForType<EventGridContainer>(ObjectType.Event);
            var root = eventsContainer.transform.root;

            var selectionController = root.GetComponentInChildren<SelectionController>();
            var eventPlacement = root.GetComponentInChildren<EventPlacement>();

            BaseEvent baseEventA = new BaseEvent { JsonTime = 2, Type = 1, Value = 1 };
            PlaceUtils.PlaceEvent(eventPlacement, baseEventA);

            SelectionController.Select(baseEventA);

            colorPicker.CurrentColor = Color.red;
            painter.Paint();

            selectionController.ShiftSelection(1, 0);

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(2, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(Color.red,
                eventsContainer.MapObjects[0].CustomData[baseEventA.CustomKeyColor].ReadColor());

            // Undo move
            actionContainer.Undo();

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(1, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(Color.red,
                eventsContainer.MapObjects[0].CustomData[baseEventA.CustomKeyColor].ReadColor());

            // Undo paint
            actionContainer.Undo();

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(1, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(true,
                eventsContainer.MapObjects[0].CustomData == null || !eventsContainer.MapObjects[0].CustomData
                    .HasKey(baseEventA.CustomKeyColor));
        }

        [Test]
        public void IgnoresOff()
        {
            var colorPicker = Object.FindObjectOfType<ColorPicker>();
            var painter = Object.FindObjectOfType<PaintSelectedObjects>();

            var eventsContainer = BeatmapObjectContainerCollection.GetCollectionForType<EventGridContainer>(ObjectType.Event);
            var root = eventsContainer.transform.root;
            var eventPlacement = root.GetComponentInChildren<EventPlacement>();

            BaseEvent baseEventA = new BaseEvent { JsonTime = 2, Type = 1, Value = 0 };
            PlaceUtils.PlaceEvent(eventPlacement, baseEventA);

            SelectionController.Select(baseEventA);

            colorPicker.CurrentColor = Color.red;
            painter.Paint();

            Assert.AreEqual(1, eventsContainer.MapObjects.Count);
            Assert.AreEqual(2, eventsContainer.MapObjects[0].JsonTime);
            Assert.AreEqual(1, ((BaseEvent)eventsContainer.MapObjects[0]).Type);
            Assert.AreEqual(true,
                eventsContainer.MapObjects[0].CustomData == null || !eventsContainer.MapObjects[0].CustomData
                    .HasKey(baseEventA.CustomKeyColor));
        }
    }
}