using System;
using System.Collections.Generic;
using Beatmap.Base;
using Beatmap.Base.Customs;
using Beatmap.Enums;
using Beatmap.Info;
using Beatmap.V2;
using Beatmap.V2.Customs;
using Beatmap.V3;
using Beatmap.V3.Customs;
using Beatmap.V4;
using SimpleJSON;
using UnityEngine;

namespace Beatmap.Helper
{
    public static class BeatmapFactory
    {
        // In the case of v4, we need the info and map info to read additional files
        public static BaseDifficulty GetDifficultyFromJson(JSONNode mainNode, string directoryAndFile,
            BaseInfo info, InfoDifficulty infoDifficulty)
        {
            BaseDifficulty difficulty;

            var v = PeekMapVersionFromJson(mainNode);

            switch (v[0])
            {
                case '4':
                    Settings.Instance.MapVersion = 4;
                    difficulty = V4Difficulty.GetFromJson(mainNode, directoryAndFile);
                    V4Difficulty.LoadBpmFromAudioData(difficulty, info);
                    V4Difficulty.LoadLightsFromLightshowFile(difficulty, info, infoDifficulty);
                    V4Difficulty.LoadBookmarksFromOfficialEditor(difficulty, info, infoDifficulty);
                    break;
                case '3':
                    Settings.Instance.MapVersion = 3;
                    difficulty = V3Difficulty.GetFromJson(mainNode, directoryAndFile);
                    break;
                case '2':
                    Settings.Instance.MapVersion = 2;
                    difficulty = V2Difficulty.GetFromJson(mainNode, directoryAndFile);
                    break;
                default:
                    return null;
            }

            difficulty.BootstrapBpmEvents(info.BeatsPerMinute);
            difficulty.RecomputeAllObjectSongBpmTimes();
            return difficulty;
        }

        private static string PeekMapVersionFromJson(JSONNode mainNode)
        {
            var nodeEnum = mainNode.GetEnumerator();
            while (nodeEnum.MoveNext())
            {
                var key = nodeEnum.Current.Key;
                var node = nodeEnum.Current.Value;
                if (key == "version" || key == "_version") return node.Value;
            }

            Debug.LogError("no version detected, return default version 2.0.0.");
            return "2.0.0";
        }

        public static TConcrete Clone<TConcrete>(TConcrete cloneable) where TConcrete : BaseItem
        {
            if (cloneable is null) throw new ArgumentException("cloneable is null.");
            return cloneable.Clone() as TConcrete;
        }

        // instantiate from JSON - Used for node editor
        public static BaseBpmEvent BpmEvent(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3BpmEvent.GetFromJson(node)
            : V2BpmEvent.GetFromJson(node);

        public static BaseNote Note(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? node.HasKey("c")
                ? V3ColorNote.GetFromJson(node)
                : V3BombNote.GetFromJson(node)
            : V2Note.GetFromJson(node);

        public static BaseNote Bomb(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3BombNote.GetFromJson(node)
            : V2Note.GetFromJson(node);

        public static BaseObstacle Obstacle(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3Obstacle.GetFromJson(node)
            : V2Obstacle.GetFromJson(node);

        public static BaseArc Arc(JSONNode node) => Settings.Instance.MapVersion is 3 or 4 
            ? V3Arc.GetFromJson(node) 
            : V2Arc.GetFromJson(node);

        public static BaseChain Chain(JSONNode node) => V3Chain.GetFromJson(node);

        public static BaseWaypoint Waypoint(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3Waypoint.GetFromJson(node)
            : V2Waypoint.GetFromJson(node);

        public static BaseEvent Event(JSONNode node)
        {
            if (Settings.Instance.MapVersion is 3 or 4)
            {
                if (node.HasKey("e") || node.HasKey("r"))
                    return V3RotationEvent.GetFromJson(node);
                if (node.HasKey("o"))
                    return V3ColorBoostEvent.GetFromJson(node);

                return V3BasicEvent.GetFromJson(node);
            }
            else
            {
                return V2Event.GetFromJson(node);
            }
        }
        
        public static BaseLightColorEventBoxGroup<BaseLightColorEventBox> LightColorEventBoxGroups(JSONNode node) =>
            V3LightColorEventBoxGroup.GetFromJson(node);

        public static BaseLightRotationEventBoxGroup<BaseLightRotationEventBox>
            LightRotationEventBoxGroups(JSONNode node) => V3LightRotationEventBoxGroup.GetFromJson(node);

        public static BaseLightTranslationEventBoxGroup<BaseLightTranslationEventBox>
            LightTranslationEventBoxGroups(JSONNode node) => V3LightTranslationEventBoxGroup.GetFromJson(node);

        public static BaseEventTypesWithKeywords EventTypesWithKeywords(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3BasicEventTypesWithKeywords.GetFromJson(node)
            : V2SpecialEventsKeywordFilters.GetFromJson(node);

        public static BaseBpmChange BpmChange(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3BpmChange.GetFromJson(node)
            : V2BpmChange.GetFromJson(node);

        public static BaseBookmark Bookmark(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3Bookmark.GetFromJson(node)
            : V2Bookmark.GetFromJson(node);

        public static BaseCustomEvent CustomEvent(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3CustomEvent.GetFromJson(node)
            : V2CustomEvent.GetFromJson(node);

        public static BaseEnvironmentEnhancement EnvironmentEnhancement(JSONNode node) => Settings.Instance.MapVersion is 3 or 4
            ? V3EnvironmentEnhancement.GetFromJson(node)
            : V2EnvironmentEnhancement.GetFromJson(node);
    }
}
