use std::collections::BTreeMap;

use anyhow::anyhow;
use anyhow::Result;
use gourd_lib::experiment::Experiment;
use gourd_lib::file_system::FileOperations;
use log::debug;

use crate::status::StatusProvider;

/// File status generated by slurm
#[derive(Debug, Clone, PartialEq)]
pub struct SlurmFileOutput {
    /// The stdout output of the slurm job
    pub stdout: String,
    /// The stderr output of the slurm job
    pub stderr: String,
}

/// Provide job status information based on the slurm output files.
#[derive(Debug, Clone, Copy, Default)]
pub struct SlurmFileStatus {}

impl<T> StatusProvider<T, SlurmFileOutput> for SlurmFileStatus
where
    T: FileOperations,
{
    fn get_statuses(
        connection: &T,
        experiment: &Experiment,
    ) -> Result<BTreeMap<usize, SlurmFileOutput>> {
        let mut run_id_to_status: BTreeMap<usize, SlurmFileOutput> = BTreeMap::new();

        for (run_id, run) in experiment.runs.iter().enumerate() {
            if let Some(slurm_id) = run.slurm_id.as_ref() {
                // in `src/gourd/slurm/interactor.rs` we specified the files to be saved as
                // `gourd_%A_%a.out` and `gourd_%A_%a.err`
                let out = match connection.read_utf8(
                    &experiment
                        .slurm_out(slurm_id)
                        .ok_or(anyhow!("Slurm config not found"))?,
                ) {
                    Ok(slurm_output) => slurm_output,
                    Err(e) => {
                        debug!("Failed to read slurm output for run {}: {}", run_id, e);
                        "File not created yet".to_string()
                    }
                };
                let err = match connection.read_utf8(
                    &experiment
                        .slurm_err(slurm_id)
                        .ok_or(anyhow!("Slurm config not found"))?,
                ) {
                    Ok(slurm_output) => slurm_output,
                    Err(e) => {
                        debug!("Failed to read slurm output for run {}: {}", run_id, e);
                        "File not created yet".to_string()
                    }
                };
                run_id_to_status.insert(
                    run_id,
                    SlurmFileOutput {
                        stdout: out,
                        stderr: err,
                    },
                );
            }
        }

        Ok(run_id_to_status)
    }
}
