/*
 * Copyright (C) 2009, Kevin Thompson <kevin.thompson@theautomaters.com>
 * Copyright (C) 2009, Henon <meinrad.recheis@gmail.com>
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above
 *   copyright notice, this list of conditions and the following
 *   disclaimer in the documentation and/or other materials provided
 *   with the distribution.
 *
 * - Neither the name of the Git Development Community nor the
 *   names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior
 *   written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 */

using System;
using System.Security.Cryptography;
using System.IO;

namespace GitSharp.Core.Util
{
    public class Sha1MessageDigest : MessageDigest
    {
        public Sha1MessageDigest()
        {
        }

        public Sha1MessageDigest(byte[] buffer)
            : base(buffer)
        {
        }

        #region ICloneable Members

        public override object Clone()
        {
            return new Sha1MessageDigest(Buffer);
        }

        #endregion

        public override byte[] Digest(byte[] input)
        {
            return new SHA1Managed().ComputeHash(input);
        }
    }

    public class Md5MessageDigest : MessageDigest
    {
        public Md5MessageDigest()
        {
        }

        public Md5MessageDigest(byte[] buffer)
            : base(buffer)
        {
        }

        public override byte[] Digest(byte[] input)
        {
            return new MD5CryptoServiceProvider().ComputeHash(input);
        }

        public override object Clone()
        {
            return new Md5MessageDigest(Buffer);
        }
    }

    public abstract class MessageDigest : ICloneable, IDisposable
    {
        private MemoryStream _stream;

        protected MessageDigest(byte[] buffer)
        {
            _stream = new MemoryStream(buffer, true);
        }

        protected MessageDigest()
        {
            Reset();
        }

        public static MessageDigest getInstance(string algorithm)
        {
            switch (algorithm.ToLower())
            {
                case "sha-1":
                    return new Sha1MessageDigest();
                case "md5":
                    return new Md5MessageDigest();
                default:
                    throw new NotSupportedException(string.Format("The requested algorithm \"{0}\" is not supported.", algorithm));
            }
        }

        protected byte[] Buffer
        {
            get { return _stream.ToArray(); }
        }

        public byte[] Digest()
        {
            var ret = Digest(_stream.ToArray());
            Reset();
            return ret;
        }

        public abstract byte[] Digest(byte[] input);
        public abstract object Clone();

        public void Reset()
        {
            _stream = new MemoryStream();
        }

        public void Update(byte input)
        {
            _stream.WriteByte(input);
        }

        public void Update(byte[] input)
        {
            _stream.Write(input, 0, input.Length);
        }

        public void Update(byte[] input, int index, int count)
        {
            _stream.Write(input, index, count);
        }

        public void Dispose()
        {
            _stream.Dispose();
        }
    }


}
