#include "fpmas.h"
#include "../../../mocks/model/mock_grid.h"
#include "../../../mocks/random/mock_random.h"

using namespace testing;
using namespace fpmas::model;

class RandomGridAgentMappingTest : public Test {
	protected:
		NiceMock<MockDistribution<DiscreteCoordinate>> mock_x;
		NiceMock<MockDistribution<DiscreteCoordinate>> mock_y;

		// Actual cells that might be used as parameters for the countAt method
		std::array<NiceMock<MockGridCell>, 5> cells;
		std::array<DiscreteCoordinate, 7> x_values {
			0, 4, 2, 0, 3, 4, 5
		};
		std::array<DiscreteCoordinate, 7> y_values {
			1, 1, 0, 3, 2, 7, 1
		};
		// A set of cells corresponding to the x_values and y_values,
		// used to check that the total count of agents generated by the
		// mapping is 10
		std::array<NiceMock<MockGridCell>, 7> count_cells;

		void SetUp() override {
			ON_CALL(mock_x, call)
				.WillByDefault(ReturnRoundRobin({
							0, 4, 2, 0, 0, 3, 4, 3, 5, 0
							}));
			ON_CALL(mock_y, call)
				.WillByDefault(ReturnRoundRobin({
							1, 1, 0, 3, 1, 2, 7, 2, 1, 3
							}));

			// 2 agents in this cell
			ON_CALL(cells[0], location)
				.WillByDefault(Return(DiscretePoint {0, 1}));
			// 1 agent
			ON_CALL(cells[1], location)
				.WillByDefault(Return(DiscretePoint {4, 7}));
			// 0 agent
			ON_CALL(cells[2], location)
				.WillByDefault(Return(DiscretePoint {6, 2}));
			// 2 agent
			ON_CALL(cells[3], location)
				.WillByDefault(Return(DiscretePoint {3, 2}));
			// 1 agent
			ON_CALL(cells[4], location)
				.WillByDefault(Return(DiscretePoint {5, 1}));

			for(std::size_t i = 0; i < 7; i++)
				ON_CALL(count_cells[i], location)
					.WillByDefault(Return(DiscretePoint
								{x_values[i], y_values[i]}
								));
		}

};

TEST_F(RandomGridAgentMappingTest, test) {
	RandomGridAgentMapping random_mapping(mock_x, mock_y, 10);

	ASSERT_EQ(random_mapping.countAt(&cells[0]), 2);
	ASSERT_EQ(random_mapping.countAt(&cells[1]), 1);
	ASSERT_EQ(random_mapping.countAt(&cells[2]), 0);
	ASSERT_EQ(random_mapping.countAt(&cells[3]), 2);
	ASSERT_EQ(random_mapping.countAt(&cells[4]), 1);

	std::size_t total_count = 0;
	for(auto& cell : count_cells)
		total_count+=random_mapping.countAt(&cell);
	ASSERT_EQ(total_count, 10);
}

class AgentMappingTest : public Test {
	protected:
		static constexpr DiscreteCoordinate grid_width {100};
		static constexpr DiscreteCoordinate grid_height {100};
		static const std::size_t num_agent;

		std::array<fpmas::model::GridCell, grid_width*grid_height> cells;

		AgentMappingTest() {
			fpmas::seed(fpmas::random::default_seed);
		}

		void SetUp() override {
			for(DiscreteCoordinate x = 0; x < grid_width; x++)
				for(DiscreteCoordinate y = 0; y < grid_height; y++)
					cells[grid_width*x+y] = {{x, y}};
		}
};
const std::size_t AgentMappingTest::num_agent = 4000;

class UniformGridAgentMappingTest : public AgentMappingTest {
	protected:
		UniformGridAgentMapping mapping {grid_width, grid_height, num_agent};
};

// Notice that an mpi test (in mpi/model/grid.cpp) checks the consistency of
// the map across processes. Then its enough to check reproducibility locally.
TEST_F(UniformGridAgentMappingTest, reproducibility) {
	fpmas::seed(fpmas::random::default_seed);
	UniformGridAgentMapping other_mapping(grid_width, grid_height, num_agent);
	for(auto& cell : cells)
		ASSERT_EQ(mapping.countAt(&cell), other_mapping.countAt(&cell));
}

class ConstrainedGridAgentMappingTest : public AgentMappingTest {
	protected:
		ConstrainedGridAgentMapping mapping {grid_width, grid_height, num_agent, 2};

};

TEST_F(ConstrainedGridAgentMappingTest, test) {
	std::size_t total_count = 0;
	for(auto& cell : cells) {
		std::size_t local_count = mapping.countAt(&cell);
		ASSERT_LE(local_count, 2);
		total_count += local_count;
	}
	ASSERT_EQ(total_count, num_agent);
}

// Notice that an mpi test (in mpi/model/grid.cpp) checks the consistency of
// the map across processes. Then its enough to check reproducibility locally.
TEST_F(ConstrainedGridAgentMappingTest, reproducibility) {
	fpmas::seed(fpmas::random::default_seed);
	ConstrainedGridAgentMapping other_mapping(grid_width, grid_height, num_agent, 2);
	for(auto& cell : cells)
		ASSERT_EQ(mapping.countAt(&cell), other_mapping.countAt(&cell));
}

